// The Module object: Our interface to the outside world. We import
// and export values on it, and do the work to get that through
// closure compiler if necessary. There are various ways Module can be used:
// 1. Not defined. We create it here
// 2. A function parameter, function(Module) { ..generated code.. }
// 3. pre-run appended it, var Module = {}; ..generated code..
// 4. External script tag defines var Module.
// We need to do an eval in order to handle the closure compiler
// case, where this code here is minified but Module was defined
// elsewhere (e.g. case 4 above). We also need to check if Module
// already exists (e.g. case 3 above).
// Note that if you want to run closure, and also to use Module
// after the generated code, you will need to define   var Module = {};
// before the code. Then that object will be used in the code, and you
// can continue to use Module afterwards as well.
var Module;
if (!Module) Module = (typeof Module !== 'undefined' ? Module : null) || {};

// Sometimes an existing Module object exists with properties
// meant to overwrite the default module functionality. Here
// we collect those properties and reapply _after_ we configure
// the current environment's defaults to avoid having to be so
// defensive during initialization.
var moduleOverrides = {};
for (var key in Module) {
  if (Module.hasOwnProperty(key)) {
    moduleOverrides[key] = Module[key];
  }
}

// The environment setup code below is customized to use Module.
// *** Environment setup code ***
var ENVIRONMENT_IS_WEB = false;
var ENVIRONMENT_IS_WORKER = false;
var ENVIRONMENT_IS_NODE = true;
var ENVIRONMENT_IS_SHELL = false;

// Three configurations we can be running in:
// 1) We could be the application main() thread running in the main JS UI thread. (ENVIRONMENT_IS_WORKER == false and ENVIRONMENT_IS_PTHREAD == false)
// 2) We could be the application main() thread proxied to worker. (with Emscripten -s PROXY_TO_WORKER=1) (ENVIRONMENT_IS_WORKER == true, ENVIRONMENT_IS_PTHREAD == false)
// 3) We could be an application pthread running in a worker. (ENVIRONMENT_IS_WORKER == true and ENVIRONMENT_IS_PTHREAD == true)

if (Module['ENVIRONMENT']) {
  if (Module['ENVIRONMENT'] === 'WEB') {
    ENVIRONMENT_IS_WEB = true;
  } else if (Module['ENVIRONMENT'] === 'WORKER') {
    ENVIRONMENT_IS_WORKER = true;
  } else if (Module['ENVIRONMENT'] === 'NODE') {
    ENVIRONMENT_IS_NODE = true;
  } else if (Module['ENVIRONMENT'] === 'SHELL') {
    ENVIRONMENT_IS_SHELL = true;
  } else {
    throw new Error('The provided Module[\'ENVIRONMENT\'] value is not valid. It must be one of: WEB|WORKER|NODE|SHELL.');
  }
} else {
  ENVIRONMENT_IS_WEB = typeof window === 'object';
  ENVIRONMENT_IS_WORKER = typeof importScripts === 'function';
  ENVIRONMENT_IS_NODE = typeof process === 'object' && typeof require === 'function' && !ENVIRONMENT_IS_WEB && !ENVIRONMENT_IS_WORKER;
  ENVIRONMENT_IS_SHELL = !ENVIRONMENT_IS_WEB && !ENVIRONMENT_IS_NODE && !ENVIRONMENT_IS_WORKER;
}

if (ENVIRONMENT_IS_NODE) {
  // Expose functionality in the same simple way that the shells work
  // Note that we pollute the global namespace here, otherwise we break in node
  if (!Module['print']) Module['print'] = console.log;
  if (!Module['printErr']) Module['printErr'] = console.warn;

  var nodeFS;
  var nodePath;

  Module['read'] = function shell_read(filename, binary) {
    if (!nodeFS) nodeFS = require('fs');
    if (!nodePath) nodePath = require('path');
    filename = nodePath['normalize'](filename);
    var ret = nodeFS['readFileSync'](filename);
    return binary ? ret : ret.toString();
  };

  Module['readBinary'] = function readBinary(filename) {
    var ret = Module['read'](filename, true);
    if (!ret.buffer) {
      ret = new Uint8Array(ret);
    }
    assert(ret.buffer);
    return ret;
  };

  Module['load'] = function load(f) {
    globalEval(read(f));
  };

  if (!Module['thisProgram']) {
    if (process['argv'].length > 1) {
      Module['thisProgram'] = process['argv'][1].replace(/\\/g, '/');
    } else {
      Module['thisProgram'] = 'unknown-program';
    }
  }

  Module['arguments'] = process['argv'].slice(2);

  if (typeof module !== 'undefined') {
    module['exports'] = Module;
  }

  process['on']('uncaughtException', function(ex) {
    // suppress ExitStatus exceptions from showing an error
    if (!(ex instanceof ExitStatus)) {
      throw ex;
    }
  });

  Module['inspect'] = function () { return '[Emscripten Module object]'; };
}
else if (ENVIRONMENT_IS_SHELL) {
  if (!Module['print']) Module['print'] = print;
  if (typeof printErr != 'undefined') Module['printErr'] = printErr; // not present in v8 or older sm

  if (typeof read != 'undefined') {
    Module['read'] = read;
  } else {
    Module['read'] = function shell_read() { throw 'no read() available' };
  }

  Module['readBinary'] = function readBinary(f) {
    if (typeof readbuffer === 'function') {
      return new Uint8Array(readbuffer(f));
    }
    var data = read(f, 'binary');
    assert(typeof data === 'object');
    return data;
  };

  if (typeof scriptArgs != 'undefined') {
    Module['arguments'] = scriptArgs;
  } else if (typeof arguments != 'undefined') {
    Module['arguments'] = arguments;
  }

  if (typeof quit === 'function') {
    Module['quit'] = function(status, toThrow) {
      quit(status);
    }
  }

}
else if (ENVIRONMENT_IS_WEB || ENVIRONMENT_IS_WORKER) {
  Module['read'] = function shell_read(url) {
    var xhr = new XMLHttpRequest();
    xhr.open('GET', url, false);
    xhr.send(null);
    return xhr.responseText;
  };

  if (ENVIRONMENT_IS_WORKER) {
    Module['readBinary'] = function readBinary(url) {
      var xhr = new XMLHttpRequest();
      xhr.open('GET', url, false);
      xhr.responseType = 'arraybuffer';
      xhr.send(null);
      return new Uint8Array(xhr.response);
    };
  }

  Module['readAsync'] = function readAsync(url, onload, onerror) {
    var xhr = new XMLHttpRequest();
    xhr.open('GET', url, true);
    xhr.responseType = 'arraybuffer';
    xhr.onload = function xhr_onload() {
      if (xhr.status == 200 || (xhr.status == 0 && xhr.response)) { // file URLs can return 0
        onload(xhr.response);
      } else {
        onerror();
      }
    };
    xhr.onerror = onerror;
    xhr.send(null);
  };

  if (typeof arguments != 'undefined') {
    Module['arguments'] = arguments;
  }

  if (typeof console !== 'undefined') {
    if (!Module['print']) Module['print'] = function shell_print(x) {
      console.log(x);
    };
    if (!Module['printErr']) Module['printErr'] = function shell_printErr(x) {
      console.warn(x);
    };
  } else {
    // Probably a worker, and without console.log. We can do very little here...
    var TRY_USE_DUMP = false;
    if (!Module['print']) Module['print'] = (TRY_USE_DUMP && (typeof(dump) !== "undefined") ? (function(x) {
      dump(x);
    }) : (function(x) {
      // self.postMessage(x); // enable this if you want stdout to be sent as messages
    }));
  }

  if (ENVIRONMENT_IS_WORKER) {
    Module['load'] = importScripts;
  }

  if (typeof Module['setWindowTitle'] === 'undefined') {
    Module['setWindowTitle'] = function(title) { document.title = title };
  }
}
else {
  // Unreachable because SHELL is dependant on the others
  throw 'Unknown runtime environment. Where are we?';
}

function globalEval(x) {
  eval.call(null, x);
}
if (!Module['load'] && Module['read']) {
  Module['load'] = function load(f) {
    globalEval(Module['read'](f));
  };
}
if (!Module['print']) {
  Module['print'] = function(){};
}
if (!Module['printErr']) {
  Module['printErr'] = Module['print'];
}
if (!Module['arguments']) {
  Module['arguments'] = [];
}
if (!Module['thisProgram']) {
  Module['thisProgram'] = './this.program';
}
if (!Module['quit']) {
  Module['quit'] = function(status, toThrow) {
    throw toThrow;
  }
}

// *** Environment setup code ***

// Closure helpers
Module.print = Module['print'];
Module.printErr = Module['printErr'];

// Callbacks
Module['preRun'] = [];
Module['postRun'] = [];

// Merge back in the overrides
for (var key in moduleOverrides) {
  if (moduleOverrides.hasOwnProperty(key)) {
    Module[key] = moduleOverrides[key];
  }
}
// Free the object hierarchy contained in the overrides, this lets the GC
// reclaim data used e.g. in memoryInitializerRequest, which is a large typed array.
moduleOverrides = undefined;



// {{PREAMBLE_ADDITIONS}}

// === Preamble library stuff ===

// Documentation for the public APIs defined in this file must be updated in:
//    site/source/docs/api_reference/preamble.js.rst
// A prebuilt local version of the documentation is available at:
//    site/build/text/docs/api_reference/preamble.js.txt
// You can also build docs locally as HTML or other formats in site/
// An online HTML version (which may be of a different version of Emscripten)
//    is up at http://kripken.github.io/emscripten-site/docs/api_reference/preamble.js.html

//========================================
// Runtime code shared with compiler
//========================================

var Runtime = {
  setTempRet0: function (value) {
    tempRet0 = value;
    return value;
  },
  getTempRet0: function () {
    return tempRet0;
  },
  stackSave: function () {
    return STACKTOP;
  },
  stackRestore: function (stackTop) {
    STACKTOP = stackTop;
  },
  getNativeTypeSize: function (type) {
    switch (type) {
      case 'i1': case 'i8': return 1;
      case 'i16': return 2;
      case 'i32': return 4;
      case 'i64': return 8;
      case 'float': return 4;
      case 'double': return 8;
      default: {
        if (type[type.length-1] === '*') {
          return Runtime.QUANTUM_SIZE; // A pointer
        } else if (type[0] === 'i') {
          var bits = parseInt(type.substr(1));
          assert(bits % 8 === 0);
          return bits/8;
        } else {
          return 0;
        }
      }
    }
  },
  getNativeFieldSize: function (type) {
    return Math.max(Runtime.getNativeTypeSize(type), Runtime.QUANTUM_SIZE);
  },
  STACK_ALIGN: 16,
  prepVararg: function (ptr, type) {
    if (type === 'double' || type === 'i64') {
      // move so the load is aligned
      if (ptr & 7) {
        assert((ptr & 7) === 4);
        ptr += 4;
      }
    } else {
      assert((ptr & 3) === 0);
    }
    return ptr;
  },
  getAlignSize: function (type, size, vararg) {
    // we align i64s and doubles on 64-bit boundaries, unlike x86
    if (!vararg && (type == 'i64' || type == 'double')) return 8;
    if (!type) return Math.min(size, 8); // align structures internally to 64 bits
    return Math.min(size || (type ? Runtime.getNativeFieldSize(type) : 0), Runtime.QUANTUM_SIZE);
  },
  dynCall: function (sig, ptr, args) {
    if (args && args.length) {
      return Module['dynCall_' + sig].apply(null, [ptr].concat(args));
    } else {
      return Module['dynCall_' + sig].call(null, ptr);
    }
  },
  functionPointers: [],
  addFunction: function (func) {
    for (var i = 0; i < Runtime.functionPointers.length; i++) {
      if (!Runtime.functionPointers[i]) {
        Runtime.functionPointers[i] = func;
        return 2*(1 + i);
      }
    }
    throw 'Finished up all reserved function pointers. Use a higher value for RESERVED_FUNCTION_POINTERS.';
  },
  removeFunction: function (index) {
    Runtime.functionPointers[(index-2)/2] = null;
  },
  warnOnce: function (text) {
    if (!Runtime.warnOnce.shown) Runtime.warnOnce.shown = {};
    if (!Runtime.warnOnce.shown[text]) {
      Runtime.warnOnce.shown[text] = 1;
      Module.printErr(text);
    }
  },
  funcWrappers: {},
  getFuncWrapper: function (func, sig) {
    assert(sig);
    if (!Runtime.funcWrappers[sig]) {
      Runtime.funcWrappers[sig] = {};
    }
    var sigCache = Runtime.funcWrappers[sig];
    if (!sigCache[func]) {
      // optimize away arguments usage in common cases
      if (sig.length === 1) {
        sigCache[func] = function dynCall_wrapper() {
          return Runtime.dynCall(sig, func);
        };
      } else if (sig.length === 2) {
        sigCache[func] = function dynCall_wrapper(arg) {
          return Runtime.dynCall(sig, func, [arg]);
        };
      } else {
        // general case
        sigCache[func] = function dynCall_wrapper() {
          return Runtime.dynCall(sig, func, Array.prototype.slice.call(arguments));
        };
      }
    }
    return sigCache[func];
  },
  getCompilerSetting: function (name) {
    throw 'You must build with -s RETAIN_COMPILER_SETTINGS=1 for Runtime.getCompilerSetting or emscripten_get_compiler_setting to work';
  },
  stackAlloc: function (size) { var ret = STACKTOP;STACKTOP = (STACKTOP + size)|0;STACKTOP = (((STACKTOP)+15)&-16); return ret; },
  staticAlloc: function (size) { var ret = STATICTOP;STATICTOP = (STATICTOP + size)|0;STATICTOP = (((STATICTOP)+15)&-16); return ret; },
  dynamicAlloc: function (size) { var ret = HEAP32[DYNAMICTOP_PTR>>2];var end = (((ret + size + 15)|0) & -16);HEAP32[DYNAMICTOP_PTR>>2] = end;if (end >= TOTAL_MEMORY) {var success = enlargeMemory();if (!success) {HEAP32[DYNAMICTOP_PTR>>2] = ret;return 0;}}return ret;},
  alignMemory: function (size,quantum) { var ret = size = Math.ceil((size)/(quantum ? quantum : 16))*(quantum ? quantum : 16); return ret; },
  makeBigInt: function (low,high,unsigned) { var ret = (unsigned ? ((+((low>>>0)))+((+((high>>>0)))*(+4294967296))) : ((+((low>>>0)))+((+((high|0)))*(+4294967296)))); return ret; },
  GLOBAL_BASE: 8,
  QUANTUM_SIZE: 4,
  __dummy__: 0
}



Module["Runtime"] = Runtime;



//========================================
// Runtime essentials
//========================================

var ABORT = 0; // whether we are quitting the application. no code should run after this. set in exit() and abort()
var EXITSTATUS = 0;

/** @type {function(*, string=)} */
function assert(condition, text) {
  if (!condition) {
    abort('Assertion failed: ' + text);
  }
}

var globalScope = this;

// Returns the C function with a specified identifier (for C++, you need to do manual name mangling)
function getCFunc(ident) {
  var func = Module['_' + ident]; // closure exported function
  if (!func) {
    try { func = eval('_' + ident); } catch(e) {}
  }
  assert(func, 'Cannot call unknown function ' + ident + ' (perhaps LLVM optimizations or closure removed it?)');
  return func;
}

var cwrap, ccall;
(function(){
  var JSfuncs = {
    // Helpers for cwrap -- it can't refer to Runtime directly because it might
    // be renamed by closure, instead it calls JSfuncs['stackSave'].body to find
    // out what the minified function name is.
    'stackSave': function() {
      Runtime.stackSave()
    },
    'stackRestore': function() {
      Runtime.stackRestore()
    },
    // type conversion from js to c
    'arrayToC' : function(arr) {
      var ret = Runtime.stackAlloc(arr.length);
      writeArrayToMemory(arr, ret);
      return ret;
    },
    'stringToC' : function(str) {
      var ret = 0;
      if (str !== null && str !== undefined && str !== 0) { // null string
        // at most 4 bytes per UTF-8 code point, +1 for the trailing '\0'
        var len = (str.length << 2) + 1;
        ret = Runtime.stackAlloc(len);
        stringToUTF8(str, ret, len);
      }
      return ret;
    }
  };
  // For fast lookup of conversion functions
  var toC = {'string' : JSfuncs['stringToC'], 'array' : JSfuncs['arrayToC']};

  // C calling interface.
  ccall = function ccallFunc(ident, returnType, argTypes, args, opts) {
    var func = getCFunc(ident);
    var cArgs = [];
    var stack = 0;
    if (args) {
      for (var i = 0; i < args.length; i++) {
        var converter = toC[argTypes[i]];
        if (converter) {
          if (stack === 0) stack = Runtime.stackSave();
          cArgs[i] = converter(args[i]);
        } else {
          cArgs[i] = args[i];
        }
      }
    }
    var ret = func.apply(null, cArgs);
    if (returnType === 'string') ret = Pointer_stringify(ret);
    if (stack !== 0) {
      if (opts && opts.async) {
        EmterpreterAsync.asyncFinalizers.push(function() {
          Runtime.stackRestore(stack);
        });
        return;
      }
      Runtime.stackRestore(stack);
    }
    return ret;
  }

  var sourceRegex = /^function\s*[a-zA-Z$_0-9]*\s*\(([^)]*)\)\s*{\s*([^*]*?)[\s;]*(?:return\s*(.*?)[;\s]*)?}$/;
  function parseJSFunc(jsfunc) {
    // Match the body and the return value of a javascript function source
    var parsed = jsfunc.toString().match(sourceRegex).slice(1);
    return {arguments : parsed[0], body : parsed[1], returnValue: parsed[2]}
  }

  // sources of useful functions. we create this lazily as it can trigger a source decompression on this entire file
  var JSsource = null;
  function ensureJSsource() {
    if (!JSsource) {
      JSsource = {};
      for (var fun in JSfuncs) {
        if (JSfuncs.hasOwnProperty(fun)) {
          // Elements of toCsource are arrays of three items:
          // the code, and the return value
          JSsource[fun] = parseJSFunc(JSfuncs[fun]);
        }
      }
    }
  }

  cwrap = function cwrap(ident, returnType, argTypes) {
    argTypes = argTypes || [];
    var cfunc = getCFunc(ident);
    // When the function takes numbers and returns a number, we can just return
    // the original function
    var numericArgs = argTypes.every(function(type){ return type === 'number'});
    var numericRet = (returnType !== 'string');
    if ( numericRet && numericArgs) {
      return cfunc;
    }
    // Creation of the arguments list (["$1","$2",...,"$nargs"])
    var argNames = argTypes.map(function(x,i){return '$'+i});
    var funcstr = "(function(" + argNames.join(',') + ") {";
    var nargs = argTypes.length;
    if (!numericArgs) {
      // Generate the code needed to convert the arguments from javascript
      // values to pointers
      ensureJSsource();
      funcstr += 'var stack = ' + JSsource['stackSave'].body + ';';
      for (var i = 0; i < nargs; i++) {
        var arg = argNames[i], type = argTypes[i];
        if (type === 'number') continue;
        var convertCode = JSsource[type + 'ToC']; // [code, return]
        funcstr += 'var ' + convertCode.arguments + ' = ' + arg + ';';
        funcstr += convertCode.body + ';';
        funcstr += arg + '=(' + convertCode.returnValue + ');';
      }
    }

    // When the code is compressed, the name of cfunc is not literally 'cfunc' anymore
    var cfuncname = parseJSFunc(function(){return cfunc}).returnValue;
    // Call the function
    funcstr += 'var ret = ' + cfuncname + '(' + argNames.join(',') + ');';
    if (!numericRet) { // Return type can only by 'string' or 'number'
      // Convert the result to a string
      var strgfy = parseJSFunc(function(){return Pointer_stringify}).returnValue;
      funcstr += 'ret = ' + strgfy + '(ret);';
    }
    if (!numericArgs) {
      // If we had a stack, restore it
      ensureJSsource();
      funcstr += JSsource['stackRestore'].body.replace('()', '(stack)') + ';';
    }
    funcstr += 'return ret})';
    return eval(funcstr);
  };
})();
Module["ccall"] = ccall;
Module["cwrap"] = cwrap;

/** @type {function(number, number, string, boolean=)} */
function setValue(ptr, value, type, noSafe) {
  type = type || 'i8';
  if (type.charAt(type.length-1) === '*') type = 'i32'; // pointers are 32-bit
    switch(type) {
      case 'i1': HEAP8[((ptr)>>0)]=value; break;
      case 'i8': HEAP8[((ptr)>>0)]=value; break;
      case 'i16': HEAP16[((ptr)>>1)]=value; break;
      case 'i32': HEAP32[((ptr)>>2)]=value; break;
      case 'i64': (tempI64 = [value>>>0,(tempDouble=value,(+(Math_abs(tempDouble))) >= (+1) ? (tempDouble > (+0) ? ((Math_min((+(Math_floor((tempDouble)/(+4294967296)))), (+4294967295)))|0)>>>0 : (~~((+(Math_ceil((tempDouble - +(((~~(tempDouble)))>>>0))/(+4294967296))))))>>>0) : 0)],HEAP32[((ptr)>>2)]=tempI64[0],HEAP32[(((ptr)+(4))>>2)]=tempI64[1]); break;
      case 'float': HEAPF32[((ptr)>>2)]=value; break;
      case 'double': HEAPF64[((ptr)>>3)]=value; break;
      default: abort('invalid type for setValue: ' + type);
    }
}
Module["setValue"] = setValue;

/** @type {function(number, string, boolean=)} */
function getValue(ptr, type, noSafe) {
  type = type || 'i8';
  if (type.charAt(type.length-1) === '*') type = 'i32'; // pointers are 32-bit
    switch(type) {
      case 'i1': return HEAP8[((ptr)>>0)];
      case 'i8': return HEAP8[((ptr)>>0)];
      case 'i16': return HEAP16[((ptr)>>1)];
      case 'i32': return HEAP32[((ptr)>>2)];
      case 'i64': return HEAP32[((ptr)>>2)];
      case 'float': return HEAPF32[((ptr)>>2)];
      case 'double': return HEAPF64[((ptr)>>3)];
      default: abort('invalid type for setValue: ' + type);
    }
  return null;
}
Module["getValue"] = getValue;

var ALLOC_NORMAL = 0; // Tries to use _malloc()
var ALLOC_STACK = 1; // Lives for the duration of the current function call
var ALLOC_STATIC = 2; // Cannot be freed
var ALLOC_DYNAMIC = 3; // Cannot be freed except through sbrk
var ALLOC_NONE = 4; // Do not allocate
Module["ALLOC_NORMAL"] = ALLOC_NORMAL;
Module["ALLOC_STACK"] = ALLOC_STACK;
Module["ALLOC_STATIC"] = ALLOC_STATIC;
Module["ALLOC_DYNAMIC"] = ALLOC_DYNAMIC;
Module["ALLOC_NONE"] = ALLOC_NONE;

// allocate(): This is for internal use. You can use it yourself as well, but the interface
//             is a little tricky (see docs right below). The reason is that it is optimized
//             for multiple syntaxes to save space in generated code. So you should
//             normally not use allocate(), and instead allocate memory using _malloc(),
//             initialize it with setValue(), and so forth.
// @slab: An array of data, or a number. If a number, then the size of the block to allocate,
//        in *bytes* (note that this is sometimes confusing: the next parameter does not
//        affect this!)
// @types: Either an array of types, one for each byte (or 0 if no type at that position),
//         or a single type which is used for the entire block. This only matters if there
//         is initial data - if @slab is a number, then this does not matter at all and is
//         ignored.
// @allocator: How to allocate memory, see ALLOC_*
/** @type {function((TypedArray|Array<number>|number), string, number, number=)} */
function allocate(slab, types, allocator, ptr) {
  var zeroinit, size;
  if (typeof slab === 'number') {
    zeroinit = true;
    size = slab;
  } else {
    zeroinit = false;
    size = slab.length;
  }

  var singleType = typeof types === 'string' ? types : null;

  var ret;
  if (allocator == ALLOC_NONE) {
    ret = ptr;
  } else {
    ret = [typeof _malloc === 'function' ? _malloc : Runtime.staticAlloc, Runtime.stackAlloc, Runtime.staticAlloc, Runtime.dynamicAlloc][allocator === undefined ? ALLOC_STATIC : allocator](Math.max(size, singleType ? 1 : types.length));
  }

  if (zeroinit) {
    var ptr = ret, stop;
    assert((ret & 3) == 0);
    stop = ret + (size & ~3);
    for (; ptr < stop; ptr += 4) {
      HEAP32[((ptr)>>2)]=0;
    }
    stop = ret + size;
    while (ptr < stop) {
      HEAP8[((ptr++)>>0)]=0;
    }
    return ret;
  }

  if (singleType === 'i8') {
    if (slab.subarray || slab.slice) {
      HEAPU8.set(/** @type {!Uint8Array} */ (slab), ret);
    } else {
      HEAPU8.set(new Uint8Array(slab), ret);
    }
    return ret;
  }

  var i = 0, type, typeSize, previousType;
  while (i < size) {
    var curr = slab[i];

    if (typeof curr === 'function') {
      curr = Runtime.getFunctionIndex(curr);
    }

    type = singleType || types[i];
    if (type === 0) {
      i++;
      continue;
    }

    if (type == 'i64') type = 'i32'; // special case: we have one i32 here, and one i32 later

    setValue(ret+i, curr, type);

    // no need to look up size unless type changes, so cache it
    if (previousType !== type) {
      typeSize = Runtime.getNativeTypeSize(type);
      previousType = type;
    }
    i += typeSize;
  }

  return ret;
}
Module["allocate"] = allocate;

// Allocate memory during any stage of startup - static memory early on, dynamic memory later, malloc when ready
function getMemory(size) {
  if (!staticSealed) return Runtime.staticAlloc(size);
  if (!runtimeInitialized) return Runtime.dynamicAlloc(size);
  return _malloc(size);
}
Module["getMemory"] = getMemory;

/** @type {function(number, number=)} */
function Pointer_stringify(ptr, length) {
  if (length === 0 || !ptr) return '';
  // TODO: use TextDecoder
  // Find the length, and check for UTF while doing so
  var hasUtf = 0;
  var t;
  var i = 0;
  while (1) {
    t = HEAPU8[(((ptr)+(i))>>0)];
    hasUtf |= t;
    if (t == 0 && !length) break;
    i++;
    if (length && i == length) break;
  }
  if (!length) length = i;

  var ret = '';

  if (hasUtf < 128) {
    var MAX_CHUNK = 1024; // split up into chunks, because .apply on a huge string can overflow the stack
    var curr;
    while (length > 0) {
      curr = String.fromCharCode.apply(String, HEAPU8.subarray(ptr, ptr + Math.min(length, MAX_CHUNK)));
      ret = ret ? ret + curr : curr;
      ptr += MAX_CHUNK;
      length -= MAX_CHUNK;
    }
    return ret;
  }
  return Module['UTF8ToString'](ptr);
}
Module["Pointer_stringify"] = Pointer_stringify;

// Given a pointer 'ptr' to a null-terminated ASCII-encoded string in the emscripten HEAP, returns
// a copy of that string as a Javascript String object.

function AsciiToString(ptr) {
  var str = '';
  while (1) {
    var ch = HEAP8[((ptr++)>>0)];
    if (!ch) return str;
    str += String.fromCharCode(ch);
  }
}
Module["AsciiToString"] = AsciiToString;

// Copies the given Javascript String object 'str' to the emscripten HEAP at address 'outPtr',
// null-terminated and encoded in ASCII form. The copy will require at most str.length+1 bytes of space in the HEAP.

function stringToAscii(str, outPtr) {
  return writeAsciiToMemory(str, outPtr, false);
}
Module["stringToAscii"] = stringToAscii;

// Given a pointer 'ptr' to a null-terminated UTF8-encoded string in the given array that contains uint8 values, returns
// a copy of that string as a Javascript String object.

var UTF8Decoder = typeof TextDecoder !== 'undefined' ? new TextDecoder('utf8') : undefined;
function UTF8ArrayToString(u8Array, idx) {
  var endPtr = idx;
  // TextDecoder needs to know the byte length in advance, it doesn't stop on null terminator by itself.
  // Also, use the length info to avoid running tiny strings through TextDecoder, since .subarray() allocates garbage.
  while (u8Array[endPtr]) ++endPtr;

  if (endPtr - idx > 16 && u8Array.subarray && UTF8Decoder) {
    return UTF8Decoder.decode(u8Array.subarray(idx, endPtr));
  } else {
    var u0, u1, u2, u3, u4, u5;

    var str = '';
    while (1) {
      // For UTF8 byte structure, see http://en.wikipedia.org/wiki/UTF-8#Description and https://www.ietf.org/rfc/rfc2279.txt and https://tools.ietf.org/html/rfc3629
      u0 = u8Array[idx++];
      if (!u0) return str;
      if (!(u0 & 0x80)) { str += String.fromCharCode(u0); continue; }
      u1 = u8Array[idx++] & 63;
      if ((u0 & 0xE0) == 0xC0) { str += String.fromCharCode(((u0 & 31) << 6) | u1); continue; }
      u2 = u8Array[idx++] & 63;
      if ((u0 & 0xF0) == 0xE0) {
        u0 = ((u0 & 15) << 12) | (u1 << 6) | u2;
      } else {
        u3 = u8Array[idx++] & 63;
        if ((u0 & 0xF8) == 0xF0) {
          u0 = ((u0 & 7) << 18) | (u1 << 12) | (u2 << 6) | u3;
        } else {
          u4 = u8Array[idx++] & 63;
          if ((u0 & 0xFC) == 0xF8) {
            u0 = ((u0 & 3) << 24) | (u1 << 18) | (u2 << 12) | (u3 << 6) | u4;
          } else {
            u5 = u8Array[idx++] & 63;
            u0 = ((u0 & 1) << 30) | (u1 << 24) | (u2 << 18) | (u3 << 12) | (u4 << 6) | u5;
          }
        }
      }
      if (u0 < 0x10000) {
        str += String.fromCharCode(u0);
      } else {
        var ch = u0 - 0x10000;
        str += String.fromCharCode(0xD800 | (ch >> 10), 0xDC00 | (ch & 0x3FF));
      }
    }
  }
}
Module["UTF8ArrayToString"] = UTF8ArrayToString;

// Given a pointer 'ptr' to a null-terminated UTF8-encoded string in the emscripten HEAP, returns
// a copy of that string as a Javascript String object.

function UTF8ToString(ptr) {
  return UTF8ArrayToString(HEAPU8,ptr);
}
Module["UTF8ToString"] = UTF8ToString;

// Copies the given Javascript String object 'str' to the given byte array at address 'outIdx',
// encoded in UTF8 form and null-terminated. The copy will require at most str.length*4+1 bytes of space in the HEAP.
// Use the function lengthBytesUTF8 to compute the exact number of bytes (excluding null terminator) that this function will write.
// Parameters:
//   str: the Javascript string to copy.
//   outU8Array: the array to copy to. Each index in this array is assumed to be one 8-byte element.
//   outIdx: The starting offset in the array to begin the copying.
//   maxBytesToWrite: The maximum number of bytes this function can write to the array. This count should include the null
//                    terminator, i.e. if maxBytesToWrite=1, only the null terminator will be written and nothing else.
//                    maxBytesToWrite=0 does not write any bytes to the output, not even the null terminator.
// Returns the number of bytes written, EXCLUDING the null terminator.

function stringToUTF8Array(str, outU8Array, outIdx, maxBytesToWrite) {
  if (!(maxBytesToWrite > 0)) // Parameter maxBytesToWrite is not optional. Negative values, 0, null, undefined and false each don't write out any bytes.
    return 0;

  var startIdx = outIdx;
  var endIdx = outIdx + maxBytesToWrite - 1; // -1 for string null terminator.
  for (var i = 0; i < str.length; ++i) {
    // Gotcha: charCodeAt returns a 16-bit word that is a UTF-16 encoded code unit, not a Unicode code point of the character! So decode UTF16->UTF32->UTF8.
    // See http://unicode.org/faq/utf_bom.html#utf16-3
    // For UTF8 byte structure, see http://en.wikipedia.org/wiki/UTF-8#Description and https://www.ietf.org/rfc/rfc2279.txt and https://tools.ietf.org/html/rfc3629
    var u = str.charCodeAt(i); // possibly a lead surrogate
    if (u >= 0xD800 && u <= 0xDFFF) u = 0x10000 + ((u & 0x3FF) << 10) | (str.charCodeAt(++i) & 0x3FF);
    if (u <= 0x7F) {
      if (outIdx >= endIdx) break;
      outU8Array[outIdx++] = u;
    } else if (u <= 0x7FF) {
      if (outIdx + 1 >= endIdx) break;
      outU8Array[outIdx++] = 0xC0 | (u >> 6);
      outU8Array[outIdx++] = 0x80 | (u & 63);
    } else if (u <= 0xFFFF) {
      if (outIdx + 2 >= endIdx) break;
      outU8Array[outIdx++] = 0xE0 | (u >> 12);
      outU8Array[outIdx++] = 0x80 | ((u >> 6) & 63);
      outU8Array[outIdx++] = 0x80 | (u & 63);
    } else if (u <= 0x1FFFFF) {
      if (outIdx + 3 >= endIdx) break;
      outU8Array[outIdx++] = 0xF0 | (u >> 18);
      outU8Array[outIdx++] = 0x80 | ((u >> 12) & 63);
      outU8Array[outIdx++] = 0x80 | ((u >> 6) & 63);
      outU8Array[outIdx++] = 0x80 | (u & 63);
    } else if (u <= 0x3FFFFFF) {
      if (outIdx + 4 >= endIdx) break;
      outU8Array[outIdx++] = 0xF8 | (u >> 24);
      outU8Array[outIdx++] = 0x80 | ((u >> 18) & 63);
      outU8Array[outIdx++] = 0x80 | ((u >> 12) & 63);
      outU8Array[outIdx++] = 0x80 | ((u >> 6) & 63);
      outU8Array[outIdx++] = 0x80 | (u & 63);
    } else {
      if (outIdx + 5 >= endIdx) break;
      outU8Array[outIdx++] = 0xFC | (u >> 30);
      outU8Array[outIdx++] = 0x80 | ((u >> 24) & 63);
      outU8Array[outIdx++] = 0x80 | ((u >> 18) & 63);
      outU8Array[outIdx++] = 0x80 | ((u >> 12) & 63);
      outU8Array[outIdx++] = 0x80 | ((u >> 6) & 63);
      outU8Array[outIdx++] = 0x80 | (u & 63);
    }
  }
  // Null-terminate the pointer to the buffer.
  outU8Array[outIdx] = 0;
  return outIdx - startIdx;
}
Module["stringToUTF8Array"] = stringToUTF8Array;

// Copies the given Javascript String object 'str' to the emscripten HEAP at address 'outPtr',
// null-terminated and encoded in UTF8 form. The copy will require at most str.length*4+1 bytes of space in the HEAP.
// Use the function lengthBytesUTF8 to compute the exact number of bytes (excluding null terminator) that this function will write.
// Returns the number of bytes written, EXCLUDING the null terminator.

function stringToUTF8(str, outPtr, maxBytesToWrite) {
  return stringToUTF8Array(str, HEAPU8,outPtr, maxBytesToWrite);
}
Module["stringToUTF8"] = stringToUTF8;

// Returns the number of bytes the given Javascript string takes if encoded as a UTF8 byte array, EXCLUDING the null terminator byte.

function lengthBytesUTF8(str) {
  var len = 0;
  for (var i = 0; i < str.length; ++i) {
    // Gotcha: charCodeAt returns a 16-bit word that is a UTF-16 encoded code unit, not a Unicode code point of the character! So decode UTF16->UTF32->UTF8.
    // See http://unicode.org/faq/utf_bom.html#utf16-3
    var u = str.charCodeAt(i); // possibly a lead surrogate
    if (u >= 0xD800 && u <= 0xDFFF) u = 0x10000 + ((u & 0x3FF) << 10) | (str.charCodeAt(++i) & 0x3FF);
    if (u <= 0x7F) {
      ++len;
    } else if (u <= 0x7FF) {
      len += 2;
    } else if (u <= 0xFFFF) {
      len += 3;
    } else if (u <= 0x1FFFFF) {
      len += 4;
    } else if (u <= 0x3FFFFFF) {
      len += 5;
    } else {
      len += 6;
    }
  }
  return len;
}
Module["lengthBytesUTF8"] = lengthBytesUTF8;

// Given a pointer 'ptr' to a null-terminated UTF16LE-encoded string in the emscripten HEAP, returns
// a copy of that string as a Javascript String object.

var UTF16Decoder = typeof TextDecoder !== 'undefined' ? new TextDecoder('utf-16le') : undefined;
function UTF16ToString(ptr) {
  var endPtr = ptr;
  // TextDecoder needs to know the byte length in advance, it doesn't stop on null terminator by itself.
  // Also, use the length info to avoid running tiny strings through TextDecoder, since .subarray() allocates garbage.
  var idx = endPtr >> 1;
  while (HEAP16[idx]) ++idx;
  endPtr = idx << 1;

  if (endPtr - ptr > 32 && UTF16Decoder) {
    return UTF16Decoder.decode(HEAPU8.subarray(ptr, endPtr));
  } else {
    var i = 0;

    var str = '';
    while (1) {
      var codeUnit = HEAP16[(((ptr)+(i*2))>>1)];
      if (codeUnit == 0) return str;
      ++i;
      // fromCharCode constructs a character from a UTF-16 code unit, so we can pass the UTF16 string right through.
      str += String.fromCharCode(codeUnit);
    }
  }
}


// Copies the given Javascript String object 'str' to the emscripten HEAP at address 'outPtr',
// null-terminated and encoded in UTF16 form. The copy will require at most str.length*4+2 bytes of space in the HEAP.
// Use the function lengthBytesUTF16() to compute the exact number of bytes (excluding null terminator) that this function will write.
// Parameters:
//   str: the Javascript string to copy.
//   outPtr: Byte address in Emscripten HEAP where to write the string to.
//   maxBytesToWrite: The maximum number of bytes this function can write to the array. This count should include the null
//                    terminator, i.e. if maxBytesToWrite=2, only the null terminator will be written and nothing else.
//                    maxBytesToWrite<2 does not write any bytes to the output, not even the null terminator.
// Returns the number of bytes written, EXCLUDING the null terminator.

function stringToUTF16(str, outPtr, maxBytesToWrite) {
  // Backwards compatibility: if max bytes is not specified, assume unsafe unbounded write is allowed.
  if (maxBytesToWrite === undefined) {
    maxBytesToWrite = 0x7FFFFFFF;
  }
  if (maxBytesToWrite < 2) return 0;
  maxBytesToWrite -= 2; // Null terminator.
  var startPtr = outPtr;
  var numCharsToWrite = (maxBytesToWrite < str.length*2) ? (maxBytesToWrite / 2) : str.length;
  for (var i = 0; i < numCharsToWrite; ++i) {
    // charCodeAt returns a UTF-16 encoded code unit, so it can be directly written to the HEAP.
    var codeUnit = str.charCodeAt(i); // possibly a lead surrogate
    HEAP16[((outPtr)>>1)]=codeUnit;
    outPtr += 2;
  }
  // Null-terminate the pointer to the HEAP.
  HEAP16[((outPtr)>>1)]=0;
  return outPtr - startPtr;
}


// Returns the number of bytes the given Javascript string takes if encoded as a UTF16 byte array, EXCLUDING the null terminator byte.

function lengthBytesUTF16(str) {
  return str.length*2;
}


function UTF32ToString(ptr) {
  var i = 0;

  var str = '';
  while (1) {
    var utf32 = HEAP32[(((ptr)+(i*4))>>2)];
    if (utf32 == 0)
      return str;
    ++i;
    // Gotcha: fromCharCode constructs a character from a UTF-16 encoded code (pair), not from a Unicode code point! So encode the code point to UTF-16 for constructing.
    // See http://unicode.org/faq/utf_bom.html#utf16-3
    if (utf32 >= 0x10000) {
      var ch = utf32 - 0x10000;
      str += String.fromCharCode(0xD800 | (ch >> 10), 0xDC00 | (ch & 0x3FF));
    } else {
      str += String.fromCharCode(utf32);
    }
  }
}


// Copies the given Javascript String object 'str' to the emscripten HEAP at address 'outPtr',
// null-terminated and encoded in UTF32 form. The copy will require at most str.length*4+4 bytes of space in the HEAP.
// Use the function lengthBytesUTF32() to compute the exact number of bytes (excluding null terminator) that this function will write.
// Parameters:
//   str: the Javascript string to copy.
//   outPtr: Byte address in Emscripten HEAP where to write the string to.
//   maxBytesToWrite: The maximum number of bytes this function can write to the array. This count should include the null
//                    terminator, i.e. if maxBytesToWrite=4, only the null terminator will be written and nothing else.
//                    maxBytesToWrite<4 does not write any bytes to the output, not even the null terminator.
// Returns the number of bytes written, EXCLUDING the null terminator.

function stringToUTF32(str, outPtr, maxBytesToWrite) {
  // Backwards compatibility: if max bytes is not specified, assume unsafe unbounded write is allowed.
  if (maxBytesToWrite === undefined) {
    maxBytesToWrite = 0x7FFFFFFF;
  }
  if (maxBytesToWrite < 4) return 0;
  var startPtr = outPtr;
  var endPtr = startPtr + maxBytesToWrite - 4;
  for (var i = 0; i < str.length; ++i) {
    // Gotcha: charCodeAt returns a 16-bit word that is a UTF-16 encoded code unit, not a Unicode code point of the character! We must decode the string to UTF-32 to the heap.
    // See http://unicode.org/faq/utf_bom.html#utf16-3
    var codeUnit = str.charCodeAt(i); // possibly a lead surrogate
    if (codeUnit >= 0xD800 && codeUnit <= 0xDFFF) {
      var trailSurrogate = str.charCodeAt(++i);
      codeUnit = 0x10000 + ((codeUnit & 0x3FF) << 10) | (trailSurrogate & 0x3FF);
    }
    HEAP32[((outPtr)>>2)]=codeUnit;
    outPtr += 4;
    if (outPtr + 4 > endPtr) break;
  }
  // Null-terminate the pointer to the HEAP.
  HEAP32[((outPtr)>>2)]=0;
  return outPtr - startPtr;
}


// Returns the number of bytes the given Javascript string takes if encoded as a UTF16 byte array, EXCLUDING the null terminator byte.

function lengthBytesUTF32(str) {
  var len = 0;
  for (var i = 0; i < str.length; ++i) {
    // Gotcha: charCodeAt returns a 16-bit word that is a UTF-16 encoded code unit, not a Unicode code point of the character! We must decode the string to UTF-32 to the heap.
    // See http://unicode.org/faq/utf_bom.html#utf16-3
    var codeUnit = str.charCodeAt(i);
    if (codeUnit >= 0xD800 && codeUnit <= 0xDFFF) ++i; // possibly a lead surrogate, so skip over the tail surrogate.
    len += 4;
  }

  return len;
}


function demangle(func) {
  var __cxa_demangle_func = Module['___cxa_demangle'] || Module['__cxa_demangle'];
  if (__cxa_demangle_func) {
    try {
      var s =
        func.substr(1);
      var len = lengthBytesUTF8(s)+1;
      var buf = _malloc(len);
      stringToUTF8(s, buf, len);
      var status = _malloc(4);
      var ret = __cxa_demangle_func(buf, 0, 0, status);
      if (getValue(status, 'i32') === 0 && ret) {
        return Pointer_stringify(ret);
      }
      // otherwise, libcxxabi failed
    } catch(e) {
      // ignore problems here
    } finally {
      if (buf) _free(buf);
      if (status) _free(status);
      if (ret) _free(ret);
    }
    // failure when using libcxxabi, don't demangle
    return func;
  }
  Runtime.warnOnce('warning: build with  -s DEMANGLE_SUPPORT=1  to link in libcxxabi demangling');
  return func;
}

function demangleAll(text) {
  var regex =
    /__Z[\w\d_]+/g;
  return text.replace(regex,
    function(x) {
      var y = demangle(x);
      return x === y ? x : (x + ' [' + y + ']');
    });
}

function jsStackTrace() {
  var err = new Error();
  if (!err.stack) {
    // IE10+ special cases: It does have callstack info, but it is only populated if an Error object is thrown,
    // so try that as a special-case.
    try {
      throw new Error(0);
    } catch(e) {
      err = e;
    }
    if (!err.stack) {
      return '(no stack trace available)';
    }
  }
  return err.stack.toString();
}

function stackTrace() {
  var js = jsStackTrace();
  if (Module['extraStackTrace']) js += '\n' + Module['extraStackTrace']();
  return demangleAll(js);
}
Module["stackTrace"] = stackTrace;

// Memory management

var PAGE_SIZE = 16384;
var WASM_PAGE_SIZE = 65536;
var ASMJS_PAGE_SIZE = 16777216;
var MIN_TOTAL_MEMORY = 16777216;

function alignUp(x, multiple) {
  if (x % multiple > 0) {
    x += multiple - (x % multiple);
  }
  return x;
}

var HEAP,
/** @type {ArrayBuffer} */
  buffer,
/** @type {Int8Array} */
  HEAP8,
/** @type {Uint8Array} */
  HEAPU8,
/** @type {Int16Array} */
  HEAP16,
/** @type {Uint16Array} */
  HEAPU16,
/** @type {Int32Array} */
  HEAP32,
/** @type {Uint32Array} */
  HEAPU32,
/** @type {Float32Array} */
  HEAPF32,
/** @type {Float64Array} */
  HEAPF64;

function updateGlobalBuffer(buf) {
  Module['buffer'] = buffer = buf;
}

function updateGlobalBufferViews() {
  Module['HEAP8'] = HEAP8 = new Int8Array(buffer);
  Module['HEAP16'] = HEAP16 = new Int16Array(buffer);
  Module['HEAP32'] = HEAP32 = new Int32Array(buffer);
  Module['HEAPU8'] = HEAPU8 = new Uint8Array(buffer);
  Module['HEAPU16'] = HEAPU16 = new Uint16Array(buffer);
  Module['HEAPU32'] = HEAPU32 = new Uint32Array(buffer);
  Module['HEAPF32'] = HEAPF32 = new Float32Array(buffer);
  Module['HEAPF64'] = HEAPF64 = new Float64Array(buffer);
}

var STATIC_BASE, STATICTOP, staticSealed; // static area
var STACK_BASE, STACKTOP, STACK_MAX; // stack area
var DYNAMIC_BASE, DYNAMICTOP_PTR; // dynamic area handled by sbrk

  STATIC_BASE = STATICTOP = STACK_BASE = STACKTOP = STACK_MAX = DYNAMIC_BASE = DYNAMICTOP_PTR = 0;
  staticSealed = false;



function abortOnCannotGrowMemory() {
  abort('Cannot enlarge memory arrays. Either (1) compile with  -s TOTAL_MEMORY=X  with X higher than the current value ' + TOTAL_MEMORY + ', (2) compile with  -s ALLOW_MEMORY_GROWTH=1  which allows increasing the size at runtime but prevents some optimizations, (3) set Module.TOTAL_MEMORY to a higher value before the program runs, or (4) if you want malloc to return NULL (0) instead of this abort, compile with  -s ABORTING_MALLOC=0 ');
}


function enlargeMemory() {
  abortOnCannotGrowMemory();
}


var TOTAL_STACK = Module['TOTAL_STACK'] || 5242880;
var TOTAL_MEMORY = Module['TOTAL_MEMORY'] || 536870912;
if (TOTAL_MEMORY < TOTAL_STACK) Module.printErr('TOTAL_MEMORY should be larger than TOTAL_STACK, was ' + TOTAL_MEMORY + '! (TOTAL_STACK=' + TOTAL_STACK + ')');

// Initialize the runtime's memory



// Use a provided buffer, if there is one, or else allocate a new one
if (Module['buffer']) {
  buffer = Module['buffer'];
} else {
  // Use a WebAssembly memory where available
  {
    buffer = new ArrayBuffer(TOTAL_MEMORY);
  }
}
updateGlobalBufferViews();


function getTotalMemory() {
  return TOTAL_MEMORY;
}

// Endianness check (note: assumes compiler arch was little-endian)
  HEAP32[0] = 0x63736d65; /* 'emsc' */
HEAP16[1] = 0x6373;
if (HEAPU8[2] !== 0x73 || HEAPU8[3] !== 0x63) throw 'Runtime error: expected the system to be little-endian!';

Module['HEAP'] = HEAP;
Module['buffer'] = buffer;
Module['HEAP8'] = HEAP8;
Module['HEAP16'] = HEAP16;
Module['HEAP32'] = HEAP32;
Module['HEAPU8'] = HEAPU8;
Module['HEAPU16'] = HEAPU16;
Module['HEAPU32'] = HEAPU32;
Module['HEAPF32'] = HEAPF32;
Module['HEAPF64'] = HEAPF64;

function callRuntimeCallbacks(callbacks) {
  while(callbacks.length > 0) {
    var callback = callbacks.shift();
    if (typeof callback == 'function') {
      callback();
      continue;
    }
    var func = callback.func;
    if (typeof func === 'number') {
      if (callback.arg === undefined) {
        Module['dynCall_v'](func);
      } else {
        Module['dynCall_vi'](func, callback.arg);
      }
    } else {
      func(callback.arg === undefined ? null : callback.arg);
    }
  }
}

var __ATPRERUN__  = []; // functions called before the runtime is initialized
var __ATINIT__    = []; // functions called during startup
var __ATMAIN__    = []; // functions called when main() is to be run
var __ATEXIT__    = []; // functions called during shutdown
var __ATPOSTRUN__ = []; // functions called after the runtime has exited

var runtimeInitialized = false;
var runtimeExited = false;


function preRun() {
  // compatibility - merge in anything from Module['preRun'] at this time
  if (Module['preRun']) {
    if (typeof Module['preRun'] == 'function') Module['preRun'] = [Module['preRun']];
    while (Module['preRun'].length) {
      addOnPreRun(Module['preRun'].shift());
    }
  }
  callRuntimeCallbacks(__ATPRERUN__);
}

function ensureInitRuntime() {
  if (runtimeInitialized) return;
  runtimeInitialized = true;
  callRuntimeCallbacks(__ATINIT__);
}

function preMain() {
  callRuntimeCallbacks(__ATMAIN__);
}

function exitRuntime() {
  callRuntimeCallbacks(__ATEXIT__);
  runtimeExited = true;
}

function postRun() {
  // compatibility - merge in anything from Module['postRun'] at this time
  if (Module['postRun']) {
    if (typeof Module['postRun'] == 'function') Module['postRun'] = [Module['postRun']];
    while (Module['postRun'].length) {
      addOnPostRun(Module['postRun'].shift());
    }
  }
  callRuntimeCallbacks(__ATPOSTRUN__);
}

function addOnPreRun(cb) {
  __ATPRERUN__.unshift(cb);
}
Module["addOnPreRun"] = addOnPreRun;

function addOnInit(cb) {
  __ATINIT__.unshift(cb);
}
Module["addOnInit"] = addOnInit;

function addOnPreMain(cb) {
  __ATMAIN__.unshift(cb);
}
Module["addOnPreMain"] = addOnPreMain;

function addOnExit(cb) {
  __ATEXIT__.unshift(cb);
}
Module["addOnExit"] = addOnExit;

function addOnPostRun(cb) {
  __ATPOSTRUN__.unshift(cb);
}
Module["addOnPostRun"] = addOnPostRun;

// Tools

/** @type {function(string, boolean=, number=)} */
function intArrayFromString(stringy, dontAddNull, length) {
  var len = length > 0 ? length : lengthBytesUTF8(stringy)+1;
  var u8array = new Array(len);
  var numBytesWritten = stringToUTF8Array(stringy, u8array, 0, u8array.length);
  if (dontAddNull) u8array.length = numBytesWritten;
  return u8array;
}
Module["intArrayFromString"] = intArrayFromString;

function intArrayToString(array) {
  var ret = [];
  for (var i = 0; i < array.length; i++) {
    var chr = array[i];
    if (chr > 0xFF) {
      chr &= 0xFF;
    }
    ret.push(String.fromCharCode(chr));
  }
  return ret.join('');
}
Module["intArrayToString"] = intArrayToString;

// Deprecated: This function should not be called because it is unsafe and does not provide
// a maximum length limit of how many bytes it is allowed to write. Prefer calling the
// function stringToUTF8Array() instead, which takes in a maximum length that can be used
// to be secure from out of bounds writes.
/** @deprecated */
function writeStringToMemory(string, buffer, dontAddNull) {
  Runtime.warnOnce('writeStringToMemory is deprecated and should not be called! Use stringToUTF8() instead!');

  var /** @type {number} */ lastChar, /** @type {number} */ end;
  if (dontAddNull) {
    // stringToUTF8Array always appends null. If we don't want to do that, remember the
    // character that existed at the location where the null will be placed, and restore
    // that after the write (below).
    end = buffer + lengthBytesUTF8(string);
    lastChar = HEAP8[end];
  }
  stringToUTF8(string, buffer, Infinity);
  if (dontAddNull) HEAP8[end] = lastChar; // Restore the value under the null character.
}
Module["writeStringToMemory"] = writeStringToMemory;

function writeArrayToMemory(array, buffer) {
  HEAP8.set(array, buffer);
}
Module["writeArrayToMemory"] = writeArrayToMemory;

function writeAsciiToMemory(str, buffer, dontAddNull) {
  for (var i = 0; i < str.length; ++i) {
    HEAP8[((buffer++)>>0)]=str.charCodeAt(i);
  }
  // Null-terminate the pointer to the HEAP.
  if (!dontAddNull) HEAP8[((buffer)>>0)]=0;
}
Module["writeAsciiToMemory"] = writeAsciiToMemory;

function unSign(value, bits, ignore) {
  if (value >= 0) {
    return value;
  }
  return bits <= 32 ? 2*Math.abs(1 << (bits-1)) + value // Need some trickery, since if bits == 32, we are right at the limit of the bits JS uses in bitshifts
                    : Math.pow(2, bits)         + value;
}
function reSign(value, bits, ignore) {
  if (value <= 0) {
    return value;
  }
  var half = bits <= 32 ? Math.abs(1 << (bits-1)) // abs is needed if bits == 32
                        : Math.pow(2, bits-1);
  if (value >= half && (bits <= 32 || value > half)) { // for huge values, we can hit the precision limit and always get true here. so don't do that
                                                       // but, in general there is no perfect solution here. With 64-bit ints, we get rounding and errors
                                                       // TODO: In i64 mode 1, resign the two parts separately and safely
    value = -2*half + value; // Cannot bitshift half, as it may be at the limit of the bits JS uses in bitshifts
  }
  return value;
}

// check for imul support, and also for correctness ( https://bugs.webkit.org/show_bug.cgi?id=126345 )
if (!Math['imul'] || Math['imul'](0xffffffff, 5) !== -5) Math['imul'] = function imul(a, b) {
  var ah  = a >>> 16;
  var al = a & 0xffff;
  var bh  = b >>> 16;
  var bl = b & 0xffff;
  return (al*bl + ((ah*bl + al*bh) << 16))|0;
};
Math.imul = Math['imul'];


if (!Math['clz32']) Math['clz32'] = function(x) {
  x = x >>> 0;
  for (var i = 0; i < 32; i++) {
    if (x & (1 << (31 - i))) return i;
  }
  return 32;
};
Math.clz32 = Math['clz32']

if (!Math['trunc']) Math['trunc'] = function(x) {
  return x < 0 ? Math.ceil(x) : Math.floor(x);
};
Math.trunc = Math['trunc'];

var Math_abs = Math.abs;
var Math_cos = Math.cos;
var Math_sin = Math.sin;
var Math_tan = Math.tan;
var Math_acos = Math.acos;
var Math_asin = Math.asin;
var Math_atan = Math.atan;
var Math_atan2 = Math.atan2;
var Math_exp = Math.exp;
var Math_log = Math.log;
var Math_sqrt = Math.sqrt;
var Math_ceil = Math.ceil;
var Math_floor = Math.floor;
var Math_pow = Math.pow;
var Math_imul = Math.imul;
var Math_fround = Math.fround;
var Math_round = Math.round;
var Math_min = Math.min;
var Math_clz32 = Math.clz32;
var Math_trunc = Math.trunc;

// A counter of dependencies for calling run(). If we need to
// do asynchronous work before running, increment this and
// decrement it. Incrementing must happen in a place like
// PRE_RUN_ADDITIONS (used by emcc to add file preloading).
// Note that you can add dependencies in preRun, even though
// it happens right before run - run will be postponed until
// the dependencies are met.
var runDependencies = 0;
var runDependencyWatcher = null;
var dependenciesFulfilled = null; // overridden to take different actions when all run dependencies are fulfilled

function getUniqueRunDependency(id) {
  return id;
}

function addRunDependency(id) {
  runDependencies++;
  if (Module['monitorRunDependencies']) {
    Module['monitorRunDependencies'](runDependencies);
  }
}
Module["addRunDependency"] = addRunDependency;

function removeRunDependency(id) {
  runDependencies--;
  if (Module['monitorRunDependencies']) {
    Module['monitorRunDependencies'](runDependencies);
  }
  if (runDependencies == 0) {
    if (runDependencyWatcher !== null) {
      clearInterval(runDependencyWatcher);
      runDependencyWatcher = null;
    }
    if (dependenciesFulfilled) {
      var callback = dependenciesFulfilled;
      dependenciesFulfilled = null;
      callback(); // can add another dependenciesFulfilled
    }
  }
}
Module["removeRunDependency"] = removeRunDependency;

Module["preloadedImages"] = {}; // maps url to image data
Module["preloadedAudios"] = {}; // maps url to audio data



var memoryInitializer = null;






// === Body ===

var ASM_CONSTS = [function($0) { Module.debugPointer($0); },
 function($0, $1, $2, $3, $4) { Module.blit($0, $1, $2, $3, $4); },
 function() { return Module.acquireInputLock(); },
 function() { return Module.getInputValue(Module.InputBufferAddresses.mouseButtonStateAddr); },
 function() { return Module.getInputValue(Module.InputBufferAddresses.mouseMoveFlagAddr); },
 function() { return Module.getInputValue(Module.InputBufferAddresses.mouseMoveXDeltaAddr); },
 function() { return Module.getInputValue(Module.InputBufferAddresses.mouseMoveYDeltaAddr); },
 function() { return Module.getInputValue(Module.InputBufferAddresses.keyEventFlagAddr); },
 function() { return Module.getInputValue(Module.InputBufferAddresses.keyCodeAddr); },
 function() { return Module.getInputValue(Module.InputBufferAddresses.keyStateAddr); },
 function() { Module.releaseInputLock(); },
 function($0, $1, $2, $3) { Module.openAudio($0, $1, $2, $3); },
 function($0, $1, $2) { return Module.enqueueAudio($0, $1, $2); }];

function _emscripten_asm_const_iiiii(code, a0, a1, a2, a3) {
  return ASM_CONSTS[code](a0, a1, a2, a3);
}

function _emscripten_asm_const_i(code) {
  return ASM_CONSTS[code]();
}

function _emscripten_asm_const_ii(code, a0) {
  return ASM_CONSTS[code](a0);
}

function _emscripten_asm_const_iiii(code, a0, a1, a2) {
  return ASM_CONSTS[code](a0, a1, a2);
}

function _emscripten_asm_const_iiiiii(code, a0, a1, a2, a3, a4) {
  return ASM_CONSTS[code](a0, a1, a2, a3, a4);
}



STATIC_BASE = Runtime.GLOBAL_BASE;

STATICTOP = STATIC_BASE + 660784;
/* global initializers */  __ATINIT__.push({ func: function() { __GLOBAL__sub_I_ether_cpp() } }, { func: function() { __GLOBAL__sub_I_sony_cpp() } }, { func: function() { __GLOBAL__sub_I_disk_cpp() } }, { func: function() { __GLOBAL__sub_I_cdrom_cpp() } }, { func: function() { __GLOBAL__sub_I_video_cpp() } }, { func: function() { __GLOBAL__sub_I_audio_cpp() } }, { func: function() { __GLOBAL__sub_I_video_sdl_cpp() } }, { func: function() { __GLOBAL__sub_I_prefs_unix_cpp() } });


memoryInitializer = "BasiliskII.js.mem";





/* no memory initializer */
var tempDoublePtr = STATICTOP; STATICTOP += 16;

function copyTempFloat(ptr) { // functions, because inlining this code increases code size too much

  HEAP8[tempDoublePtr] = HEAP8[ptr];

  HEAP8[tempDoublePtr+1] = HEAP8[ptr+1];

  HEAP8[tempDoublePtr+2] = HEAP8[ptr+2];

  HEAP8[tempDoublePtr+3] = HEAP8[ptr+3];

}

function copyTempDouble(ptr) {

  HEAP8[tempDoublePtr] = HEAP8[ptr];

  HEAP8[tempDoublePtr+1] = HEAP8[ptr+1];

  HEAP8[tempDoublePtr+2] = HEAP8[ptr+2];

  HEAP8[tempDoublePtr+3] = HEAP8[ptr+3];

  HEAP8[tempDoublePtr+4] = HEAP8[ptr+4];

  HEAP8[tempDoublePtr+5] = HEAP8[ptr+5];

  HEAP8[tempDoublePtr+6] = HEAP8[ptr+6];

  HEAP8[tempDoublePtr+7] = HEAP8[ptr+7];

}

// {{PRE_LIBRARY}}



  function _atexit(func, arg) {
      __ATEXIT__.unshift({ func: func, arg: arg });
    }function ___cxa_atexit() {
  return _atexit.apply(null, arguments)
  }


  var _tzname=STATICTOP; STATICTOP += 16;;

  var _daylight=STATICTOP; STATICTOP += 16;;

  var _timezone=STATICTOP; STATICTOP += 16;;function _tzset() {
      // TODO: Use (malleable) environment variables instead of system settings.
      if (_tzset.called) return;
      _tzset.called = true;

      HEAP32[((_timezone)>>2)]=-(new Date()).getTimezoneOffset() * 60;

      var winter = new Date(2000, 0, 1);
      var summer = new Date(2000, 6, 1);
      HEAP32[((_daylight)>>2)]=Number(winter.getTimezoneOffset() != summer.getTimezoneOffset());

      function extractZone(date) {
        var match = date.toTimeString().match(/\(([A-Za-z ]+)\)$/);
        return match ? match[1] : "GMT";
      };
      var winterName = extractZone(winter);
      var summerName = extractZone(summer);
      var winterNamePtr = allocate(intArrayFromString(winterName), 'i8', ALLOC_NORMAL);
      var summerNamePtr = allocate(intArrayFromString(summerName), 'i8', ALLOC_NORMAL);
      if (summer.getTimezoneOffset() < winter.getTimezoneOffset()) {
        // Northern hemisphere
        HEAP32[((_tzname)>>2)]=winterNamePtr;
        HEAP32[(((_tzname)+(4))>>2)]=summerNamePtr;
      } else {
        HEAP32[((_tzname)>>2)]=summerNamePtr;
        HEAP32[(((_tzname)+(4))>>2)]=winterNamePtr;
      }
    }





  function _sem_wait() {}

  function _sem_post() {}




  var ERRNO_CODES={EPERM:1,ENOENT:2,ESRCH:3,EINTR:4,EIO:5,ENXIO:6,E2BIG:7,ENOEXEC:8,EBADF:9,ECHILD:10,EAGAIN:11,EWOULDBLOCK:11,ENOMEM:12,EACCES:13,EFAULT:14,ENOTBLK:15,EBUSY:16,EEXIST:17,EXDEV:18,ENODEV:19,ENOTDIR:20,EISDIR:21,EINVAL:22,ENFILE:23,EMFILE:24,ENOTTY:25,ETXTBSY:26,EFBIG:27,ENOSPC:28,ESPIPE:29,EROFS:30,EMLINK:31,EPIPE:32,EDOM:33,ERANGE:34,ENOMSG:42,EIDRM:43,ECHRNG:44,EL2NSYNC:45,EL3HLT:46,EL3RST:47,ELNRNG:48,EUNATCH:49,ENOCSI:50,EL2HLT:51,EDEADLK:35,ENOLCK:37,EBADE:52,EBADR:53,EXFULL:54,ENOANO:55,EBADRQC:56,EBADSLT:57,EDEADLOCK:35,EBFONT:59,ENOSTR:60,ENODATA:61,ETIME:62,ENOSR:63,ENONET:64,ENOPKG:65,EREMOTE:66,ENOLINK:67,EADV:68,ESRMNT:69,ECOMM:70,EPROTO:71,EMULTIHOP:72,EDOTDOT:73,EBADMSG:74,ENOTUNIQ:76,EBADFD:77,EREMCHG:78,ELIBACC:79,ELIBBAD:80,ELIBSCN:81,ELIBMAX:82,ELIBEXEC:83,ENOSYS:38,ENOTEMPTY:39,ENAMETOOLONG:36,ELOOP:40,EOPNOTSUPP:95,EPFNOSUPPORT:96,ECONNRESET:104,ENOBUFS:105,EAFNOSUPPORT:97,EPROTOTYPE:91,ENOTSOCK:88,ENOPROTOOPT:92,ESHUTDOWN:108,ECONNREFUSED:111,EADDRINUSE:98,ECONNABORTED:103,ENETUNREACH:101,ENETDOWN:100,ETIMEDOUT:110,EHOSTDOWN:112,EHOSTUNREACH:113,EINPROGRESS:115,EALREADY:114,EDESTADDRREQ:89,EMSGSIZE:90,EPROTONOSUPPORT:93,ESOCKTNOSUPPORT:94,EADDRNOTAVAIL:99,ENETRESET:102,EISCONN:106,ENOTCONN:107,ETOOMANYREFS:109,EUSERS:87,EDQUOT:122,ESTALE:116,ENOTSUP:95,ENOMEDIUM:123,EILSEQ:84,EOVERFLOW:75,ECANCELED:125,ENOTRECOVERABLE:131,EOWNERDEAD:130,ESTRPIPE:86};

  var ERRNO_MESSAGES={0:"Success",1:"Not super-user",2:"No such file or directory",3:"No such process",4:"Interrupted system call",5:"I/O error",6:"No such device or address",7:"Arg list too long",8:"Exec format error",9:"Bad file number",10:"No children",11:"No more processes",12:"Not enough core",13:"Permission denied",14:"Bad address",15:"Block device required",16:"Mount device busy",17:"File exists",18:"Cross-device link",19:"No such device",20:"Not a directory",21:"Is a directory",22:"Invalid argument",23:"Too many open files in system",24:"Too many open files",25:"Not a typewriter",26:"Text file busy",27:"File too large",28:"No space left on device",29:"Illegal seek",30:"Read only file system",31:"Too many links",32:"Broken pipe",33:"Math arg out of domain of func",34:"Math result not representable",35:"File locking deadlock error",36:"File or path name too long",37:"No record locks available",38:"Function not implemented",39:"Directory not empty",40:"Too many symbolic links",42:"No message of desired type",43:"Identifier removed",44:"Channel number out of range",45:"Level 2 not synchronized",46:"Level 3 halted",47:"Level 3 reset",48:"Link number out of range",49:"Protocol driver not attached",50:"No CSI structure available",51:"Level 2 halted",52:"Invalid exchange",53:"Invalid request descriptor",54:"Exchange full",55:"No anode",56:"Invalid request code",57:"Invalid slot",59:"Bad font file fmt",60:"Device not a stream",61:"No data (for no delay io)",62:"Timer expired",63:"Out of streams resources",64:"Machine is not on the network",65:"Package not installed",66:"The object is remote",67:"The link has been severed",68:"Advertise error",69:"Srmount error",70:"Communication error on send",71:"Protocol error",72:"Multihop attempted",73:"Cross mount point (not really error)",74:"Trying to read unreadable message",75:"Value too large for defined data type",76:"Given log. name not unique",77:"f.d. invalid for this operation",78:"Remote address changed",79:"Can   access a needed shared lib",80:"Accessing a corrupted shared lib",81:".lib section in a.out corrupted",82:"Attempting to link in too many libs",83:"Attempting to exec a shared library",84:"Illegal byte sequence",86:"Streams pipe error",87:"Too many users",88:"Socket operation on non-socket",89:"Destination address required",90:"Message too long",91:"Protocol wrong type for socket",92:"Protocol not available",93:"Unknown protocol",94:"Socket type not supported",95:"Not supported",96:"Protocol family not supported",97:"Address family not supported by protocol family",98:"Address already in use",99:"Address not available",100:"Network interface is not configured",101:"Network is unreachable",102:"Connection reset by network",103:"Connection aborted",104:"Connection reset by peer",105:"No buffer space available",106:"Socket is already connected",107:"Socket is not connected",108:"Can't send after socket shutdown",109:"Too many references",110:"Connection timed out",111:"Connection refused",112:"Host is down",113:"Host is unreachable",114:"Socket already connected",115:"Connection already in progress",116:"Stale file handle",122:"Quota exceeded",123:"No medium (in tape drive)",125:"Operation canceled",130:"Previous owner died",131:"State not recoverable"};

  function ___setErrNo(value) {
      if (Module['___errno_location']) HEAP32[((Module['___errno_location']())>>2)]=value;
      return value;
    }

  var PATH={splitPath:function (filename) {
        var splitPathRe = /^(\/?|)([\s\S]*?)((?:\.{1,2}|[^\/]+?|)(\.[^.\/]*|))(?:[\/]*)$/;
        return splitPathRe.exec(filename).slice(1);
      },normalizeArray:function (parts, allowAboveRoot) {
        // if the path tries to go above the root, `up` ends up > 0
        var up = 0;
        for (var i = parts.length - 1; i >= 0; i--) {
          var last = parts[i];
          if (last === '.') {
            parts.splice(i, 1);
          } else if (last === '..') {
            parts.splice(i, 1);
            up++;
          } else if (up) {
            parts.splice(i, 1);
            up--;
          }
        }
        // if the path is allowed to go above the root, restore leading ..s
        if (allowAboveRoot) {
          for (; up; up--) {
            parts.unshift('..');
          }
        }
        return parts;
      },normalize:function (path) {
        var isAbsolute = path.charAt(0) === '/',
            trailingSlash = path.substr(-1) === '/';
        // Normalize the path
        path = PATH.normalizeArray(path.split('/').filter(function(p) {
          return !!p;
        }), !isAbsolute).join('/');
        if (!path && !isAbsolute) {
          path = '.';
        }
        if (path && trailingSlash) {
          path += '/';
        }
        return (isAbsolute ? '/' : '') + path;
      },dirname:function (path) {
        var result = PATH.splitPath(path),
            root = result[0],
            dir = result[1];
        if (!root && !dir) {
          // No dirname whatsoever
          return '.';
        }
        if (dir) {
          // It has a dirname, strip trailing slash
          dir = dir.substr(0, dir.length - 1);
        }
        return root + dir;
      },basename:function (path) {
        // EMSCRIPTEN return '/'' for '/', not an empty string
        if (path === '/') return '/';
        var lastSlash = path.lastIndexOf('/');
        if (lastSlash === -1) return path;
        return path.substr(lastSlash+1);
      },extname:function (path) {
        return PATH.splitPath(path)[3];
      },join:function () {
        var paths = Array.prototype.slice.call(arguments, 0);
        return PATH.normalize(paths.join('/'));
      },join2:function (l, r) {
        return PATH.normalize(l + '/' + r);
      },resolve:function () {
        var resolvedPath = '',
          resolvedAbsolute = false;
        for (var i = arguments.length - 1; i >= -1 && !resolvedAbsolute; i--) {
          var path = (i >= 0) ? arguments[i] : FS.cwd();
          // Skip empty and invalid entries
          if (typeof path !== 'string') {
            throw new TypeError('Arguments to path.resolve must be strings');
          } else if (!path) {
            return ''; // an invalid portion invalidates the whole thing
          }
          resolvedPath = path + '/' + resolvedPath;
          resolvedAbsolute = path.charAt(0) === '/';
        }
        // At this point the path should be resolved to a full absolute path, but
        // handle relative paths to be safe (might happen when process.cwd() fails)
        resolvedPath = PATH.normalizeArray(resolvedPath.split('/').filter(function(p) {
          return !!p;
        }), !resolvedAbsolute).join('/');
        return ((resolvedAbsolute ? '/' : '') + resolvedPath) || '.';
      },relative:function (from, to) {
        from = PATH.resolve(from).substr(1);
        to = PATH.resolve(to).substr(1);
        function trim(arr) {
          var start = 0;
          for (; start < arr.length; start++) {
            if (arr[start] !== '') break;
          }
          var end = arr.length - 1;
          for (; end >= 0; end--) {
            if (arr[end] !== '') break;
          }
          if (start > end) return [];
          return arr.slice(start, end - start + 1);
        }
        var fromParts = trim(from.split('/'));
        var toParts = trim(to.split('/'));
        var length = Math.min(fromParts.length, toParts.length);
        var samePartsLength = length;
        for (var i = 0; i < length; i++) {
          if (fromParts[i] !== toParts[i]) {
            samePartsLength = i;
            break;
          }
        }
        var outputParts = [];
        for (var i = samePartsLength; i < fromParts.length; i++) {
          outputParts.push('..');
        }
        outputParts = outputParts.concat(toParts.slice(samePartsLength));
        return outputParts.join('/');
      }};

  var TTY={ttys:[],init:function () {
        // https://github.com/kripken/emscripten/pull/1555
        // if (ENVIRONMENT_IS_NODE) {
        //   // currently, FS.init does not distinguish if process.stdin is a file or TTY
        //   // device, it always assumes it's a TTY device. because of this, we're forcing
        //   // process.stdin to UTF8 encoding to at least make stdin reading compatible
        //   // with text files until FS.init can be refactored.
        //   process['stdin']['setEncoding']('utf8');
        // }
      },shutdown:function () {
        // https://github.com/kripken/emscripten/pull/1555
        // if (ENVIRONMENT_IS_NODE) {
        //   // inolen: any idea as to why node -e 'process.stdin.read()' wouldn't exit immediately (with process.stdin being a tty)?
        //   // isaacs: because now it's reading from the stream, you've expressed interest in it, so that read() kicks off a _read() which creates a ReadReq operation
        //   // inolen: I thought read() in that case was a synchronous operation that just grabbed some amount of buffered data if it exists?
        //   // isaacs: it is. but it also triggers a _read() call, which calls readStart() on the handle
        //   // isaacs: do process.stdin.pause() and i'd think it'd probably close the pending call
        //   process['stdin']['pause']();
        // }
      },register:function (dev, ops) {
        TTY.ttys[dev] = { input: [], output: [], ops: ops };
        FS.registerDevice(dev, TTY.stream_ops);
      },stream_ops:{open:function (stream) {
          var tty = TTY.ttys[stream.node.rdev];
          if (!tty) {
            throw new FS.ErrnoError(ERRNO_CODES.ENODEV);
          }
          stream.tty = tty;
          stream.seekable = false;
        },close:function (stream) {
          // flush any pending line data
          stream.tty.ops.flush(stream.tty);
        },flush:function (stream) {
          stream.tty.ops.flush(stream.tty);
        },read:function (stream, buffer, offset, length, pos /* ignored */) {
          if (!stream.tty || !stream.tty.ops.get_char) {
            throw new FS.ErrnoError(ERRNO_CODES.ENXIO);
          }
          var bytesRead = 0;
          for (var i = 0; i < length; i++) {
            var result;
            try {
              result = stream.tty.ops.get_char(stream.tty);
            } catch (e) {
              throw new FS.ErrnoError(ERRNO_CODES.EIO);
            }
            if (result === undefined && bytesRead === 0) {
              throw new FS.ErrnoError(ERRNO_CODES.EAGAIN);
            }
            if (result === null || result === undefined) break;
            bytesRead++;
            buffer[offset+i] = result;
          }
          if (bytesRead) {
            stream.node.timestamp = Date.now();
          }
          return bytesRead;
        },write:function (stream, buffer, offset, length, pos) {
          if (!stream.tty || !stream.tty.ops.put_char) {
            throw new FS.ErrnoError(ERRNO_CODES.ENXIO);
          }
          for (var i = 0; i < length; i++) {
            try {
              stream.tty.ops.put_char(stream.tty, buffer[offset+i]);
            } catch (e) {
              throw new FS.ErrnoError(ERRNO_CODES.EIO);
            }
          }
          if (length) {
            stream.node.timestamp = Date.now();
          }
          return i;
        }},default_tty_ops:{get_char:function (tty) {
          if (!tty.input.length) {
            var result = null;
            if (ENVIRONMENT_IS_NODE) {
              // we will read data by chunks of BUFSIZE
              var BUFSIZE = 256;
              var buf = new Buffer(BUFSIZE);
              var bytesRead = 0;

              var isPosixPlatform = (process.platform != 'win32'); // Node doesn't offer a direct check, so test by exclusion

              var fd = process.stdin.fd;
              if (isPosixPlatform) {
                // Linux and Mac cannot use process.stdin.fd (which isn't set up as sync)
                var usingDevice = false;
                try {
                  fd = fs.openSync('/dev/stdin', 'r');
                  usingDevice = true;
                } catch (e) {}
              }

              try {
                bytesRead = fs.readSync(fd, buf, 0, BUFSIZE, null);
              } catch(e) {
                // Cross-platform differences: on Windows, reading EOF throws an exception, but on other OSes,
                // reading EOF returns 0. Uniformize behavior by treating the EOF exception to return 0.
                if (e.toString().indexOf('EOF') != -1) bytesRead = 0;
                else throw e;
              }

              if (usingDevice) { fs.closeSync(fd); }
              if (bytesRead > 0) {
                result = buf.slice(0, bytesRead).toString('utf-8');
              } else {
                result = null;
              }

            } else if (typeof window != 'undefined' &&
              typeof window.prompt == 'function') {
              // Browser.
              result = window.prompt('Input: ');  // returns null on cancel
              if (result !== null) {
                result += '\n';
              }
            } else if (typeof readline == 'function') {
              // Command line.
              result = readline();
              if (result !== null) {
                result += '\n';
              }
            }
            if (!result) {
              return null;
            }
            tty.input = intArrayFromString(result, true);
          }
          return tty.input.shift();
        },put_char:function (tty, val) {
          if (val === null || val === 10) {
            Module['print'](UTF8ArrayToString(tty.output, 0));
            tty.output = [];
          } else {
            if (val != 0) tty.output.push(val); // val == 0 would cut text output off in the middle.
          }
        },flush:function (tty) {
          if (tty.output && tty.output.length > 0) {
            Module['print'](UTF8ArrayToString(tty.output, 0));
            tty.output = [];
          }
        }},default_tty1_ops:{put_char:function (tty, val) {
          if (val === null || val === 10) {
            Module['printErr'](UTF8ArrayToString(tty.output, 0));
            tty.output = [];
          } else {
            if (val != 0) tty.output.push(val);
          }
        },flush:function (tty) {
          if (tty.output && tty.output.length > 0) {
            Module['printErr'](UTF8ArrayToString(tty.output, 0));
            tty.output = [];
          }
        }}};

  var MEMFS={ops_table:null,mount:function (mount) {
        return MEMFS.createNode(null, '/', 16384 | 511 /* 0777 */, 0);
      },createNode:function (parent, name, mode, dev) {
        if (FS.isBlkdev(mode) || FS.isFIFO(mode)) {
          // no supported
          throw new FS.ErrnoError(ERRNO_CODES.EPERM);
        }
        if (!MEMFS.ops_table) {
          MEMFS.ops_table = {
            dir: {
              node: {
                getattr: MEMFS.node_ops.getattr,
                setattr: MEMFS.node_ops.setattr,
                lookup: MEMFS.node_ops.lookup,
                mknod: MEMFS.node_ops.mknod,
                rename: MEMFS.node_ops.rename,
                unlink: MEMFS.node_ops.unlink,
                rmdir: MEMFS.node_ops.rmdir,
                readdir: MEMFS.node_ops.readdir,
                symlink: MEMFS.node_ops.symlink
              },
              stream: {
                llseek: MEMFS.stream_ops.llseek
              }
            },
            file: {
              node: {
                getattr: MEMFS.node_ops.getattr,
                setattr: MEMFS.node_ops.setattr
              },
              stream: {
                llseek: MEMFS.stream_ops.llseek,
                read: MEMFS.stream_ops.read,
                write: MEMFS.stream_ops.write,
                allocate: MEMFS.stream_ops.allocate,
                mmap: MEMFS.stream_ops.mmap,
                msync: MEMFS.stream_ops.msync
              }
            },
            link: {
              node: {
                getattr: MEMFS.node_ops.getattr,
                setattr: MEMFS.node_ops.setattr,
                readlink: MEMFS.node_ops.readlink
              },
              stream: {}
            },
            chrdev: {
              node: {
                getattr: MEMFS.node_ops.getattr,
                setattr: MEMFS.node_ops.setattr
              },
              stream: FS.chrdev_stream_ops
            }
          };
        }
        var node = FS.createNode(parent, name, mode, dev);
        if (FS.isDir(node.mode)) {
          node.node_ops = MEMFS.ops_table.dir.node;
          node.stream_ops = MEMFS.ops_table.dir.stream;
          node.contents = {};
        } else if (FS.isFile(node.mode)) {
          node.node_ops = MEMFS.ops_table.file.node;
          node.stream_ops = MEMFS.ops_table.file.stream;
          node.usedBytes = 0; // The actual number of bytes used in the typed array, as opposed to contents.length which gives the whole capacity.
          // When the byte data of the file is populated, this will point to either a typed array, or a normal JS array. Typed arrays are preferred
          // for performance, and used by default. However, typed arrays are not resizable like normal JS arrays are, so there is a small disk size
          // penalty involved for appending file writes that continuously grow a file similar to std::vector capacity vs used -scheme.
          node.contents = null;
        } else if (FS.isLink(node.mode)) {
          node.node_ops = MEMFS.ops_table.link.node;
          node.stream_ops = MEMFS.ops_table.link.stream;
        } else if (FS.isChrdev(node.mode)) {
          node.node_ops = MEMFS.ops_table.chrdev.node;
          node.stream_ops = MEMFS.ops_table.chrdev.stream;
        }
        node.timestamp = Date.now();
        // add the new node to the parent
        if (parent) {
          parent.contents[name] = node;
        }
        return node;
      },getFileDataAsRegularArray:function (node) {
        if (node.contents && node.contents.subarray) {
          var arr = [];
          for (var i = 0; i < node.usedBytes; ++i) arr.push(node.contents[i]);
          return arr; // Returns a copy of the original data.
        }
        return node.contents; // No-op, the file contents are already in a JS array. Return as-is.
      },getFileDataAsTypedArray:function (node) {
        if (!node.contents) return new Uint8Array;
        if (node.contents.subarray) return node.contents.subarray(0, node.usedBytes); // Make sure to not return excess unused bytes.
        return new Uint8Array(node.contents);
      },expandFileStorage:function (node, newCapacity) {
        // If we are asked to expand the size of a file that already exists, revert to using a standard JS array to store the file
        // instead of a typed array. This makes resizing the array more flexible because we can just .push() elements at the back to
        // increase the size.
        if (node.contents && node.contents.subarray && newCapacity > node.contents.length) {
          node.contents = MEMFS.getFileDataAsRegularArray(node);
          node.usedBytes = node.contents.length; // We might be writing to a lazy-loaded file which had overridden this property, so force-reset it.
        }

        if (!node.contents || node.contents.subarray) { // Keep using a typed array if creating a new storage, or if old one was a typed array as well.
          var prevCapacity = node.contents ? node.contents.length : 0;
          if (prevCapacity >= newCapacity) return; // No need to expand, the storage was already large enough.
          // Don't expand strictly to the given requested limit if it's only a very small increase, but instead geometrically grow capacity.
          // For small filesizes (<1MB), perform size*2 geometric increase, but for large sizes, do a much more conservative size*1.125 increase to
          // avoid overshooting the allocation cap by a very large margin.
          var CAPACITY_DOUBLING_MAX = 1024 * 1024;
          newCapacity = Math.max(newCapacity, (prevCapacity * (prevCapacity < CAPACITY_DOUBLING_MAX ? 2.0 : 1.125)) | 0);
          if (prevCapacity != 0) newCapacity = Math.max(newCapacity, 256); // At minimum allocate 256b for each file when expanding.
          var oldContents = node.contents;
          node.contents = new Uint8Array(newCapacity); // Allocate new storage.
          if (node.usedBytes > 0) node.contents.set(oldContents.subarray(0, node.usedBytes), 0); // Copy old data over to the new storage.
          return;
        }
        // Not using a typed array to back the file storage. Use a standard JS array instead.
        if (!node.contents && newCapacity > 0) node.contents = [];
        while (node.contents.length < newCapacity) node.contents.push(0);
      },resizeFileStorage:function (node, newSize) {
        if (node.usedBytes == newSize) return;
        if (newSize == 0) {
          node.contents = null; // Fully decommit when requesting a resize to zero.
          node.usedBytes = 0;
          return;
        }
        if (!node.contents || node.contents.subarray) { // Resize a typed array if that is being used as the backing store.
          var oldContents = node.contents;
          node.contents = new Uint8Array(new ArrayBuffer(newSize)); // Allocate new storage.
          if (oldContents) {
            node.contents.set(oldContents.subarray(0, Math.min(newSize, node.usedBytes))); // Copy old data over to the new storage.
          }
          node.usedBytes = newSize;
          return;
        }
        // Backing with a JS array.
        if (!node.contents) node.contents = [];
        if (node.contents.length > newSize) node.contents.length = newSize;
        else while (node.contents.length < newSize) node.contents.push(0);
        node.usedBytes = newSize;
      },node_ops:{getattr:function (node) {
          var attr = {};
          // device numbers reuse inode numbers.
          attr.dev = FS.isChrdev(node.mode) ? node.id : 1;
          attr.ino = node.id;
          attr.mode = node.mode;
          attr.nlink = 1;
          attr.uid = 0;
          attr.gid = 0;
          attr.rdev = node.rdev;
          if (FS.isDir(node.mode)) {
            attr.size = 4096;
          } else if (FS.isFile(node.mode)) {
            attr.size = node.usedBytes;
          } else if (FS.isLink(node.mode)) {
            attr.size = node.link.length;
          } else {
            attr.size = 0;
          }
          attr.atime = new Date(node.timestamp);
          attr.mtime = new Date(node.timestamp);
          attr.ctime = new Date(node.timestamp);
          // NOTE: In our implementation, st_blocks = Math.ceil(st_size/st_blksize),
          //       but this is not required by the standard.
          attr.blksize = 4096;
          attr.blocks = Math.ceil(attr.size / attr.blksize);
          return attr;
        },setattr:function (node, attr) {
          if (attr.mode !== undefined) {
            node.mode = attr.mode;
          }
          if (attr.timestamp !== undefined) {
            node.timestamp = attr.timestamp;
          }
          if (attr.size !== undefined) {
            MEMFS.resizeFileStorage(node, attr.size);
          }
        },lookup:function (parent, name) {
          throw FS.genericErrors[ERRNO_CODES.ENOENT];
        },mknod:function (parent, name, mode, dev) {
          return MEMFS.createNode(parent, name, mode, dev);
        },rename:function (old_node, new_dir, new_name) {
          // if we're overwriting a directory at new_name, make sure it's empty.
          if (FS.isDir(old_node.mode)) {
            var new_node;
            try {
              new_node = FS.lookupNode(new_dir, new_name);
            } catch (e) {
            }
            if (new_node) {
              for (var i in new_node.contents) {
                throw new FS.ErrnoError(ERRNO_CODES.ENOTEMPTY);
              }
            }
          }
          // do the internal rewiring
          delete old_node.parent.contents[old_node.name];
          old_node.name = new_name;
          new_dir.contents[new_name] = old_node;
          old_node.parent = new_dir;
        },unlink:function (parent, name) {
          delete parent.contents[name];
        },rmdir:function (parent, name) {
          var node = FS.lookupNode(parent, name);
          for (var i in node.contents) {
            throw new FS.ErrnoError(ERRNO_CODES.ENOTEMPTY);
          }
          delete parent.contents[name];
        },readdir:function (node) {
          var entries = ['.', '..']
          for (var key in node.contents) {
            if (!node.contents.hasOwnProperty(key)) {
              continue;
            }
            entries.push(key);
          }
          return entries;
        },symlink:function (parent, newname, oldpath) {
          var node = MEMFS.createNode(parent, newname, 511 /* 0777 */ | 40960, 0);
          node.link = oldpath;
          return node;
        },readlink:function (node) {
          if (!FS.isLink(node.mode)) {
            throw new FS.ErrnoError(ERRNO_CODES.EINVAL);
          }
          return node.link;
        }},stream_ops:{read:function (stream, buffer, offset, length, position) {
          var contents = stream.node.contents;
          if (position >= stream.node.usedBytes) return 0;
          var size = Math.min(stream.node.usedBytes - position, length);
          assert(size >= 0);
          if (size > 8 && contents.subarray) { // non-trivial, and typed array
            buffer.set(contents.subarray(position, position + size), offset);
          } else {
            for (var i = 0; i < size; i++) buffer[offset + i] = contents[position + i];
          }
          return size;
        },write:function (stream, buffer, offset, length, position, canOwn) {
          if (!length) return 0;
          var node = stream.node;
          node.timestamp = Date.now();

          if (buffer.subarray && (!node.contents || node.contents.subarray)) { // This write is from a typed array to a typed array?
            if (canOwn) {
              node.contents = buffer.subarray(offset, offset + length);
              node.usedBytes = length;
              return length;
            } else if (node.usedBytes === 0 && position === 0) { // If this is a simple first write to an empty file, do a fast set since we don't need to care about old data.
              node.contents = new Uint8Array(buffer.subarray(offset, offset + length));
              node.usedBytes = length;
              return length;
            } else if (position + length <= node.usedBytes) { // Writing to an already allocated and used subrange of the file?
              node.contents.set(buffer.subarray(offset, offset + length), position);
              return length;
            }
          }

          // Appending to an existing file and we need to reallocate, or source data did not come as a typed array.
          MEMFS.expandFileStorage(node, position+length);
          if (node.contents.subarray && buffer.subarray) node.contents.set(buffer.subarray(offset, offset + length), position); // Use typed array write if available.
          else {
            for (var i = 0; i < length; i++) {
             node.contents[position + i] = buffer[offset + i]; // Or fall back to manual write if not.
            }
          }
          node.usedBytes = Math.max(node.usedBytes, position+length);
          return length;
        },llseek:function (stream, offset, whence) {
          var position = offset;
          if (whence === 1) {  // SEEK_CUR.
            position += stream.position;
          } else if (whence === 2) {  // SEEK_END.
            if (FS.isFile(stream.node.mode)) {
              position += stream.node.usedBytes;
            }
          }
          if (position < 0) {
            throw new FS.ErrnoError(ERRNO_CODES.EINVAL);
          }
          return position;
        },allocate:function (stream, offset, length) {
          MEMFS.expandFileStorage(stream.node, offset + length);
          stream.node.usedBytes = Math.max(stream.node.usedBytes, offset + length);
        },mmap:function (stream, buffer, offset, length, position, prot, flags) {
          if (!FS.isFile(stream.node.mode)) {
            throw new FS.ErrnoError(ERRNO_CODES.ENODEV);
          }
          var ptr;
          var allocated;
          var contents = stream.node.contents;
          // Only make a new copy when MAP_PRIVATE is specified.
          if ( !(flags & 2) &&
                (contents.buffer === buffer || contents.buffer === buffer.buffer) ) {
            // We can't emulate MAP_SHARED when the file is not backed by the buffer
            // we're mapping to (e.g. the HEAP buffer).
            allocated = false;
            ptr = contents.byteOffset;
          } else {
            // Try to avoid unnecessary slices.
            if (position > 0 || position + length < stream.node.usedBytes) {
              if (contents.subarray) {
                contents = contents.subarray(position, position + length);
              } else {
                contents = Array.prototype.slice.call(contents, position, position + length);
              }
            }
            allocated = true;
            ptr = _malloc(length);
            if (!ptr) {
              throw new FS.ErrnoError(ERRNO_CODES.ENOMEM);
            }
            buffer.set(contents, ptr);
          }
          return { ptr: ptr, allocated: allocated };
        },msync:function (stream, buffer, offset, length, mmapFlags) {
          if (!FS.isFile(stream.node.mode)) {
            throw new FS.ErrnoError(ERRNO_CODES.ENODEV);
          }
          if (mmapFlags & 2) {
            // MAP_PRIVATE calls need not to be synced back to underlying fs
            return 0;
          }

          var bytesWritten = MEMFS.stream_ops.write(stream, buffer, 0, length, offset, false);
          // should we check if bytesWritten and length are the same?
          return 0;
        }}};

  var IDBFS={dbs:{},indexedDB:function () {
        if (typeof indexedDB !== 'undefined') return indexedDB;
        var ret = null;
        if (typeof window === 'object') ret = window.indexedDB || window.mozIndexedDB || window.webkitIndexedDB || window.msIndexedDB;
        assert(ret, 'IDBFS used, but indexedDB not supported');
        return ret;
      },DB_VERSION:21,DB_STORE_NAME:"FILE_DATA",mount:function (mount) {
        // reuse all of the core MEMFS functionality
        return MEMFS.mount.apply(null, arguments);
      },syncfs:function (mount, populate, callback) {
        IDBFS.getLocalSet(mount, function(err, local) {
          if (err) return callback(err);

          IDBFS.getRemoteSet(mount, function(err, remote) {
            if (err) return callback(err);

            var src = populate ? remote : local;
            var dst = populate ? local : remote;

            IDBFS.reconcile(src, dst, callback);
          });
        });
      },getDB:function (name, callback) {
        // check the cache first
        var db = IDBFS.dbs[name];
        if (db) {
          return callback(null, db);
        }

        var req;
        try {
          req = IDBFS.indexedDB().open(name, IDBFS.DB_VERSION);
        } catch (e) {
          return callback(e);
        }
        if (!req) {
          return callback("Unable to connect to IndexedDB");
        }
        req.onupgradeneeded = function(e) {
          var db = e.target.result;
          var transaction = e.target.transaction;

          var fileStore;

          if (db.objectStoreNames.contains(IDBFS.DB_STORE_NAME)) {
            fileStore = transaction.objectStore(IDBFS.DB_STORE_NAME);
          } else {
            fileStore = db.createObjectStore(IDBFS.DB_STORE_NAME);
          }

          if (!fileStore.indexNames.contains('timestamp')) {
            fileStore.createIndex('timestamp', 'timestamp', { unique: false });
          }
        };
        req.onsuccess = function() {
          db = req.result;

          // add to the cache
          IDBFS.dbs[name] = db;
          callback(null, db);
        };
        req.onerror = function(e) {
          callback(this.error);
          e.preventDefault();
        };
      },getLocalSet:function (mount, callback) {
        var entries = {};

        function isRealDir(p) {
          return p !== '.' && p !== '..';
        };
        function toAbsolute(root) {
          return function(p) {
            return PATH.join2(root, p);
          }
        };

        var check = FS.readdir(mount.mountpoint).filter(isRealDir).map(toAbsolute(mount.mountpoint));

        while (check.length) {
          var path = check.pop();
          var stat;

          try {
            stat = FS.stat(path);
          } catch (e) {
            return callback(e);
          }

          if (FS.isDir(stat.mode)) {
            check.push.apply(check, FS.readdir(path).filter(isRealDir).map(toAbsolute(path)));
          }

          entries[path] = { timestamp: stat.mtime };
        }

        return callback(null, { type: 'local', entries: entries });
      },getRemoteSet:function (mount, callback) {
        var entries = {};

        IDBFS.getDB(mount.mountpoint, function(err, db) {
          if (err) return callback(err);

          var transaction = db.transaction([IDBFS.DB_STORE_NAME], 'readonly');
          transaction.onerror = function(e) {
            callback(this.error);
            e.preventDefault();
          };

          var store = transaction.objectStore(IDBFS.DB_STORE_NAME);
          var index = store.index('timestamp');

          index.openKeyCursor().onsuccess = function(event) {
            var cursor = event.target.result;

            if (!cursor) {
              return callback(null, { type: 'remote', db: db, entries: entries });
            }

            entries[cursor.primaryKey] = { timestamp: cursor.key };

            cursor.continue();
          };
        });
      },loadLocalEntry:function (path, callback) {
        var stat, node;

        try {
          var lookup = FS.lookupPath(path);
          node = lookup.node;
          stat = FS.stat(path);
        } catch (e) {
          return callback(e);
        }

        if (FS.isDir(stat.mode)) {
          return callback(null, { timestamp: stat.mtime, mode: stat.mode });
        } else if (FS.isFile(stat.mode)) {
          // Performance consideration: storing a normal JavaScript array to a IndexedDB is much slower than storing a typed array.
          // Therefore always convert the file contents to a typed array first before writing the data to IndexedDB.
          node.contents = MEMFS.getFileDataAsTypedArray(node);
          return callback(null, { timestamp: stat.mtime, mode: stat.mode, contents: node.contents });
        } else {
          return callback(new Error('node type not supported'));
        }
      },storeLocalEntry:function (path, entry, callback) {
        try {
          if (FS.isDir(entry.mode)) {
            FS.mkdir(path, entry.mode);
          } else if (FS.isFile(entry.mode)) {
            FS.writeFile(path, entry.contents, { encoding: 'binary', canOwn: true });
          } else {
            return callback(new Error('node type not supported'));
          }

          FS.chmod(path, entry.mode);
          FS.utime(path, entry.timestamp, entry.timestamp);
        } catch (e) {
          return callback(e);
        }

        callback(null);
      },removeLocalEntry:function (path, callback) {
        try {
          var lookup = FS.lookupPath(path);
          var stat = FS.stat(path);

          if (FS.isDir(stat.mode)) {
            FS.rmdir(path);
          } else if (FS.isFile(stat.mode)) {
            FS.unlink(path);
          }
        } catch (e) {
          return callback(e);
        }

        callback(null);
      },loadRemoteEntry:function (store, path, callback) {
        var req = store.get(path);
        req.onsuccess = function(event) { callback(null, event.target.result); };
        req.onerror = function(e) {
          callback(this.error);
          e.preventDefault();
        };
      },storeRemoteEntry:function (store, path, entry, callback) {
        var req = store.put(entry, path);
        req.onsuccess = function() { callback(null); };
        req.onerror = function(e) {
          callback(this.error);
          e.preventDefault();
        };
      },removeRemoteEntry:function (store, path, callback) {
        var req = store.delete(path);
        req.onsuccess = function() { callback(null); };
        req.onerror = function(e) {
          callback(this.error);
          e.preventDefault();
        };
      },reconcile:function (src, dst, callback) {
        var total = 0;

        var create = [];
        Object.keys(src.entries).forEach(function (key) {
          var e = src.entries[key];
          var e2 = dst.entries[key];
          if (!e2 || e.timestamp > e2.timestamp) {
            create.push(key);
            total++;
          }
        });

        var remove = [];
        Object.keys(dst.entries).forEach(function (key) {
          var e = dst.entries[key];
          var e2 = src.entries[key];
          if (!e2) {
            remove.push(key);
            total++;
          }
        });

        if (!total) {
          return callback(null);
        }

        var errored = false;
        var completed = 0;
        var db = src.type === 'remote' ? src.db : dst.db;
        var transaction = db.transaction([IDBFS.DB_STORE_NAME], 'readwrite');
        var store = transaction.objectStore(IDBFS.DB_STORE_NAME);

        function done(err) {
          if (err) {
            if (!done.errored) {
              done.errored = true;
              return callback(err);
            }
            return;
          }
          if (++completed >= total) {
            return callback(null);
          }
        };

        transaction.onerror = function(e) {
          done(this.error);
          e.preventDefault();
        };

        // sort paths in ascending order so directory entries are created
        // before the files inside them
        create.sort().forEach(function (path) {
          if (dst.type === 'local') {
            IDBFS.loadRemoteEntry(store, path, function (err, entry) {
              if (err) return done(err);
              IDBFS.storeLocalEntry(path, entry, done);
            });
          } else {
            IDBFS.loadLocalEntry(path, function (err, entry) {
              if (err) return done(err);
              IDBFS.storeRemoteEntry(store, path, entry, done);
            });
          }
        });

        // sort paths in descending order so files are deleted before their
        // parent directories
        remove.sort().reverse().forEach(function(path) {
          if (dst.type === 'local') {
            IDBFS.removeLocalEntry(path, done);
          } else {
            IDBFS.removeRemoteEntry(store, path, done);
          }
        });
      }};

  var NODEFS={isWindows:false,staticInit:function () {
        NODEFS.isWindows = !!process.platform.match(/^win/);
      },mount:function (mount) {
        assert(ENVIRONMENT_IS_NODE);
        return NODEFS.createNode(null, '/', NODEFS.getMode(mount.opts.root), 0);
      },createNode:function (parent, name, mode, dev) {
        if (!FS.isDir(mode) && !FS.isFile(mode) && !FS.isLink(mode)) {
          throw new FS.ErrnoError(ERRNO_CODES.EINVAL);
        }
        var node = FS.createNode(parent, name, mode);
        node.node_ops = NODEFS.node_ops;
        node.stream_ops = NODEFS.stream_ops;
        return node;
      },getMode:function (path) {
        var stat;
        try {
          stat = fs.lstatSync(path);
          if (NODEFS.isWindows) {
            // On Windows, directories return permission bits 'rw-rw-rw-', even though they have 'rwxrwxrwx', so
            // propagate write bits to execute bits.
            stat.mode = stat.mode | ((stat.mode & 146) >> 1);
          }
        } catch (e) {
          if (!e.code) throw e;
          throw new FS.ErrnoError(ERRNO_CODES[e.code]);
        }
        return stat.mode;
      },realPath:function (node) {
        var parts = [];
        while (node.parent !== node) {
          parts.push(node.name);
          node = node.parent;
        }
        parts.push(node.mount.opts.root);
        parts.reverse();
        return PATH.join.apply(null, parts);
      },flagsToPermissionStringMap:{0:"r",1:"r+",2:"r+",64:"r",65:"r+",66:"r+",129:"rx+",193:"rx+",514:"w+",577:"w",578:"w+",705:"wx",706:"wx+",1024:"a",1025:"a",1026:"a+",1089:"a",1090:"a+",1153:"ax",1154:"ax+",1217:"ax",1218:"ax+",4096:"rs",4098:"rs+"},flagsToPermissionString:function (flags) {
        flags &= ~0x200000 /*O_PATH*/; // Ignore this flag from musl, otherwise node.js fails to open the file.
        flags &= ~0x800 /*O_NONBLOCK*/; // Ignore this flag from musl, otherwise node.js fails to open the file.
        flags &= ~0x8000 /*O_LARGEFILE*/; // Ignore this flag from musl, otherwise node.js fails to open the file.
        flags &= ~0x80000 /*O_CLOEXEC*/; // Some applications may pass it; it makes no sense for a single process.
        if (flags in NODEFS.flagsToPermissionStringMap) {
          return NODEFS.flagsToPermissionStringMap[flags];
        } else {
          throw new FS.ErrnoError(ERRNO_CODES.EINVAL);
        }
      },node_ops:{getattr:function (node) {
          var path = NODEFS.realPath(node);
          var stat;
          try {
            stat = fs.lstatSync(path);
          } catch (e) {
            if (!e.code) throw e;
            throw new FS.ErrnoError(ERRNO_CODES[e.code]);
          }
          // node.js v0.10.20 doesn't report blksize and blocks on Windows. Fake them with default blksize of 4096.
          // See http://support.microsoft.com/kb/140365
          if (NODEFS.isWindows && !stat.blksize) {
            stat.blksize = 4096;
          }
          if (NODEFS.isWindows && !stat.blocks) {
            stat.blocks = (stat.size+stat.blksize-1)/stat.blksize|0;
          }
          return {
            dev: stat.dev,
            ino: stat.ino,
            mode: stat.mode,
            nlink: stat.nlink,
            uid: stat.uid,
            gid: stat.gid,
            rdev: stat.rdev,
            size: stat.size,
            atime: stat.atime,
            mtime: stat.mtime,
            ctime: stat.ctime,
            blksize: stat.blksize,
            blocks: stat.blocks
          };
        },setattr:function (node, attr) {
          var path = NODEFS.realPath(node);
          try {
            if (attr.mode !== undefined) {
              fs.chmodSync(path, attr.mode);
              // update the common node structure mode as well
              node.mode = attr.mode;
            }
            if (attr.timestamp !== undefined) {
              var date = new Date(attr.timestamp);
              fs.utimesSync(path, date, date);
            }
            if (attr.size !== undefined) {
              fs.truncateSync(path, attr.size);
            }
          } catch (e) {
            if (!e.code) throw e;
            throw new FS.ErrnoError(ERRNO_CODES[e.code]);
          }
        },lookup:function (parent, name) {
          var path = PATH.join2(NODEFS.realPath(parent), name);
          var mode = NODEFS.getMode(path);
          return NODEFS.createNode(parent, name, mode);
        },mknod:function (parent, name, mode, dev) {
          var node = NODEFS.createNode(parent, name, mode, dev);
          // create the backing node for this in the fs root as well
          var path = NODEFS.realPath(node);
          try {
            if (FS.isDir(node.mode)) {
              fs.mkdirSync(path, node.mode);
            } else {
              fs.writeFileSync(path, '', { mode: node.mode });
            }
          } catch (e) {
            if (!e.code) throw e;
            throw new FS.ErrnoError(ERRNO_CODES[e.code]);
          }
          return node;
        },rename:function (oldNode, newDir, newName) {
          var oldPath = NODEFS.realPath(oldNode);
          var newPath = PATH.join2(NODEFS.realPath(newDir), newName);
          try {
            fs.renameSync(oldPath, newPath);
          } catch (e) {
            if (!e.code) throw e;
            throw new FS.ErrnoError(ERRNO_CODES[e.code]);
          }
        },unlink:function (parent, name) {
          var path = PATH.join2(NODEFS.realPath(parent), name);
          try {
            fs.unlinkSync(path);
          } catch (e) {
            if (!e.code) throw e;
            throw new FS.ErrnoError(ERRNO_CODES[e.code]);
          }
        },rmdir:function (parent, name) {
          var path = PATH.join2(NODEFS.realPath(parent), name);
          try {
            fs.rmdirSync(path);
          } catch (e) {
            if (!e.code) throw e;
            throw new FS.ErrnoError(ERRNO_CODES[e.code]);
          }
        },readdir:function (node) {
          var path = NODEFS.realPath(node);
          try {
            return fs.readdirSync(path);
          } catch (e) {
            if (!e.code) throw e;
            throw new FS.ErrnoError(ERRNO_CODES[e.code]);
          }
        },symlink:function (parent, newName, oldPath) {
          var newPath = PATH.join2(NODEFS.realPath(parent), newName);
          try {
            fs.symlinkSync(oldPath, newPath);
          } catch (e) {
            if (!e.code) throw e;
            throw new FS.ErrnoError(ERRNO_CODES[e.code]);
          }
        },readlink:function (node) {
          var path = NODEFS.realPath(node);
          try {
            path = fs.readlinkSync(path);
            path = NODEJS_PATH.relative(NODEJS_PATH.resolve(node.mount.opts.root), path);
            return path;
          } catch (e) {
            if (!e.code) throw e;
            throw new FS.ErrnoError(ERRNO_CODES[e.code]);
          }
        }},stream_ops:{open:function (stream) {
          var path = NODEFS.realPath(stream.node);
          try {
            if (FS.isFile(stream.node.mode)) {
              stream.nfd = fs.openSync(path, NODEFS.flagsToPermissionString(stream.flags));
            }
          } catch (e) {
            if (!e.code) throw e;
            throw new FS.ErrnoError(ERRNO_CODES[e.code]);
          }
        },close:function (stream) {
          try {
            if (FS.isFile(stream.node.mode) && stream.nfd) {
              fs.closeSync(stream.nfd);
            }
          } catch (e) {
            if (!e.code) throw e;
            throw new FS.ErrnoError(ERRNO_CODES[e.code]);
          }
        },read:function (stream, buffer, offset, length, position) {
          if (length === 0) return 0; // node errors on 0 length reads
          // FIXME this is terrible.
          var nbuffer = new Buffer(length);
          var res;
          try {
            res = fs.readSync(stream.nfd, nbuffer, 0, length, position);
          } catch (e) {
            throw new FS.ErrnoError(ERRNO_CODES[e.code]);
          }
          if (res > 0) {
            for (var i = 0; i < res; i++) {
              buffer[offset + i] = nbuffer[i];
            }
          }
          return res;
        },write:function (stream, buffer, offset, length, position) {
          // FIXME this is terrible.
          var nbuffer = new Buffer(buffer.subarray(offset, offset + length));
          var res;
          try {
            res = fs.writeSync(stream.nfd, nbuffer, 0, length, position);
          } catch (e) {
            throw new FS.ErrnoError(ERRNO_CODES[e.code]);
          }
          return res;
        },llseek:function (stream, offset, whence) {
          var position = offset;
          if (whence === 1) {  // SEEK_CUR.
            position += stream.position;
          } else if (whence === 2) {  // SEEK_END.
            if (FS.isFile(stream.node.mode)) {
              try {
                var stat = fs.fstatSync(stream.nfd);
                position += stat.size;
              } catch (e) {
                throw new FS.ErrnoError(ERRNO_CODES[e.code]);
              }
            }
          }

          if (position < 0) {
            throw new FS.ErrnoError(ERRNO_CODES.EINVAL);
          }

          return position;
        }}};

  var WORKERFS={DIR_MODE:16895,FILE_MODE:33279,reader:null,mount:function (mount) {
        assert(ENVIRONMENT_IS_WORKER);
        if (!WORKERFS.reader) WORKERFS.reader = new FileReaderSync();
        var root = WORKERFS.createNode(null, '/', WORKERFS.DIR_MODE, 0);
        var createdParents = {};
        function ensureParent(path) {
          // return the parent node, creating subdirs as necessary
          var parts = path.split('/');
          var parent = root;
          for (var i = 0; i < parts.length-1; i++) {
            var curr = parts.slice(0, i+1).join('/');
            // Issue 4254: Using curr as a node name will prevent the node
            // from being found in FS.nameTable when FS.open is called on
            // a path which holds a child of this node,
            // given that all FS functions assume node names
            // are just their corresponding parts within their given path,
            // rather than incremental aggregates which include their parent's
            // directories.
            if (!createdParents[curr]) {
              createdParents[curr] = WORKERFS.createNode(parent, parts[i], WORKERFS.DIR_MODE, 0);
            }
            parent = createdParents[curr];
          }
          return parent;
        }
        function base(path) {
          var parts = path.split('/');
          return parts[parts.length-1];
        }
        // We also accept FileList here, by using Array.prototype
        Array.prototype.forEach.call(mount.opts["files"] || [], function(file) {
          WORKERFS.createNode(ensureParent(file.name), base(file.name), WORKERFS.FILE_MODE, 0, file, file.lastModifiedDate);
        });
        (mount.opts["blobs"] || []).forEach(function(obj) {
          WORKERFS.createNode(ensureParent(obj["name"]), base(obj["name"]), WORKERFS.FILE_MODE, 0, obj["data"]);
        });
        (mount.opts["packages"] || []).forEach(function(pack) {
          pack['metadata'].files.forEach(function(file) {
            var name = file.filename.substr(1); // remove initial slash
            WORKERFS.createNode(ensureParent(name), base(name), WORKERFS.FILE_MODE, 0, pack['blob'].slice(file.start, file.end));
          });
        });
        return root;
      },createNode:function (parent, name, mode, dev, contents, mtime) {
        var node = FS.createNode(parent, name, mode);
        node.mode = mode;
        node.node_ops = WORKERFS.node_ops;
        node.stream_ops = WORKERFS.stream_ops;
        node.timestamp = (mtime || new Date).getTime();
        assert(WORKERFS.FILE_MODE !== WORKERFS.DIR_MODE);
        if (mode === WORKERFS.FILE_MODE) {
          node.size = contents.size;
          node.contents = contents;
        } else {
          node.size = 4096;
          node.contents = {};
        }
        if (parent) {
          parent.contents[name] = node;
        }
        return node;
      },node_ops:{getattr:function (node) {
          return {
            dev: 1,
            ino: undefined,
            mode: node.mode,
            nlink: 1,
            uid: 0,
            gid: 0,
            rdev: undefined,
            size: node.size,
            atime: new Date(node.timestamp),
            mtime: new Date(node.timestamp),
            ctime: new Date(node.timestamp),
            blksize: 4096,
            blocks: Math.ceil(node.size / 4096),
          };
        },setattr:function (node, attr) {
          if (attr.mode !== undefined) {
            node.mode = attr.mode;
          }
          if (attr.timestamp !== undefined) {
            node.timestamp = attr.timestamp;
          }
        },lookup:function (parent, name) {
          throw new FS.ErrnoError(ERRNO_CODES.ENOENT);
        },mknod:function (parent, name, mode, dev) {
          throw new FS.ErrnoError(ERRNO_CODES.EPERM);
        },rename:function (oldNode, newDir, newName) {
          throw new FS.ErrnoError(ERRNO_CODES.EPERM);
        },unlink:function (parent, name) {
          throw new FS.ErrnoError(ERRNO_CODES.EPERM);
        },rmdir:function (parent, name) {
          throw new FS.ErrnoError(ERRNO_CODES.EPERM);
        },readdir:function (node) {
          var entries = ['.', '..'];
          for (var key in node.contents) {
            if (!node.contents.hasOwnProperty(key)) {
              continue;
            }
            entries.push(key);
          }
          return entries;
        },symlink:function (parent, newName, oldPath) {
          throw new FS.ErrnoError(ERRNO_CODES.EPERM);
        },readlink:function (node) {
          throw new FS.ErrnoError(ERRNO_CODES.EPERM);
        }},stream_ops:{read:function (stream, buffer, offset, length, position) {
          if (position >= stream.node.size) return 0;
          var chunk = stream.node.contents.slice(position, position + length);
          var ab = WORKERFS.reader.readAsArrayBuffer(chunk);
          buffer.set(new Uint8Array(ab), offset);
          return chunk.size;
        },write:function (stream, buffer, offset, length, position) {
          throw new FS.ErrnoError(ERRNO_CODES.EIO);
        },llseek:function (stream, offset, whence) {
          var position = offset;
          if (whence === 1) {  // SEEK_CUR.
            position += stream.position;
          } else if (whence === 2) {  // SEEK_END.
            if (FS.isFile(stream.node.mode)) {
              position += stream.node.size;
            }
          }
          if (position < 0) {
            throw new FS.ErrnoError(ERRNO_CODES.EINVAL);
          }
          return position;
        }}};

  var _stdin=STATICTOP; STATICTOP += 16;;

  var _stdout=STATICTOP; STATICTOP += 16;;

  var _stderr=STATICTOP; STATICTOP += 16;;var FS={root:null,mounts:[],devices:[null],streams:[],nextInode:1,nameTable:null,currentPath:"/",initialized:false,ignorePermissions:true,trackingDelegate:{},tracking:{openFlags:{READ:1,WRITE:2}},ErrnoError:null,genericErrors:{},filesystems:null,syncFSRequests:0,handleFSError:function (e) {
        if (!(e instanceof FS.ErrnoError)) throw e + ' : ' + stackTrace();
        return ___setErrNo(e.errno);
      },lookupPath:function (path, opts) {
        path = PATH.resolve(FS.cwd(), path);
        opts = opts || {};

        if (!path) return { path: '', node: null };

        var defaults = {
          follow_mount: true,
          recurse_count: 0
        };
        for (var key in defaults) {
          if (opts[key] === undefined) {
            opts[key] = defaults[key];
          }
        }

        if (opts.recurse_count > 8) {  // max recursive lookup of 8
          throw new FS.ErrnoError(ERRNO_CODES.ELOOP);
        }

        // split the path
        var parts = PATH.normalizeArray(path.split('/').filter(function(p) {
          return !!p;
        }), false);

        // start at the root
        var current = FS.root;
        var current_path = '/';

        for (var i = 0; i < parts.length; i++) {
          var islast = (i === parts.length-1);
          if (islast && opts.parent) {
            // stop resolving
            break;
          }

          current = FS.lookupNode(current, parts[i]);
          current_path = PATH.join2(current_path, parts[i]);

          // jump to the mount's root node if this is a mountpoint
          if (FS.isMountpoint(current)) {
            if (!islast || (islast && opts.follow_mount)) {
              current = current.mounted.root;
            }
          }

          // by default, lookupPath will not follow a symlink if it is the final path component.
          // setting opts.follow = true will override this behavior.
          if (!islast || opts.follow) {
            var count = 0;
            while (FS.isLink(current.mode)) {
              var link = FS.readlink(current_path);
              current_path = PATH.resolve(PATH.dirname(current_path), link);

              var lookup = FS.lookupPath(current_path, { recurse_count: opts.recurse_count });
              current = lookup.node;

              if (count++ > 40) {  // limit max consecutive symlinks to 40 (SYMLOOP_MAX).
                throw new FS.ErrnoError(ERRNO_CODES.ELOOP);
              }
            }
          }
        }

        return { path: current_path, node: current };
      },getPath:function (node) {
        var path;
        while (true) {
          if (FS.isRoot(node)) {
            var mount = node.mount.mountpoint;
            if (!path) return mount;
            return mount[mount.length-1] !== '/' ? mount + '/' + path : mount + path;
          }
          path = path ? node.name + '/' + path : node.name;
          node = node.parent;
        }
      },hashName:function (parentid, name) {
        var hash = 0;


        for (var i = 0; i < name.length; i++) {
          hash = ((hash << 5) - hash + name.charCodeAt(i)) | 0;
        }
        return ((parentid + hash) >>> 0) % FS.nameTable.length;
      },hashAddNode:function (node) {
        var hash = FS.hashName(node.parent.id, node.name);
        node.name_next = FS.nameTable[hash];
        FS.nameTable[hash] = node;
      },hashRemoveNode:function (node) {
        var hash = FS.hashName(node.parent.id, node.name);
        if (FS.nameTable[hash] === node) {
          FS.nameTable[hash] = node.name_next;
        } else {
          var current = FS.nameTable[hash];
          while (current) {
            if (current.name_next === node) {
              current.name_next = node.name_next;
              break;
            }
            current = current.name_next;
          }
        }
      },lookupNode:function (parent, name) {
        var err = FS.mayLookup(parent);
        if (err) {
          throw new FS.ErrnoError(err, parent);
        }
        var hash = FS.hashName(parent.id, name);
        for (var node = FS.nameTable[hash]; node; node = node.name_next) {
          var nodeName = node.name;
          if (node.parent.id === parent.id && nodeName === name) {
            return node;
          }
        }
        // if we failed to find it in the cache, call into the VFS
        return FS.lookup(parent, name);
      },createNode:function (parent, name, mode, rdev) {
        if (!FS.FSNode) {
          FS.FSNode = function(parent, name, mode, rdev) {
            if (!parent) {
              parent = this;  // root node sets parent to itself
            }
            this.parent = parent;
            this.mount = parent.mount;
            this.mounted = null;
            this.id = FS.nextInode++;
            this.name = name;
            this.mode = mode;
            this.node_ops = {};
            this.stream_ops = {};
            this.rdev = rdev;
          };

          FS.FSNode.prototype = {};

          // compatibility
          var readMode = 292 | 73;
          var writeMode = 146;

          // NOTE we must use Object.defineProperties instead of individual calls to
          // Object.defineProperty in order to make closure compiler happy
          Object.defineProperties(FS.FSNode.prototype, {
            read: {
              get: function() { return (this.mode & readMode) === readMode; },
              set: function(val) { val ? this.mode |= readMode : this.mode &= ~readMode; }
            },
            write: {
              get: function() { return (this.mode & writeMode) === writeMode; },
              set: function(val) { val ? this.mode |= writeMode : this.mode &= ~writeMode; }
            },
            isFolder: {
              get: function() { return FS.isDir(this.mode); }
            },
            isDevice: {
              get: function() { return FS.isChrdev(this.mode); }
            }
          });
        }

        var node = new FS.FSNode(parent, name, mode, rdev);

        FS.hashAddNode(node);

        return node;
      },destroyNode:function (node) {
        FS.hashRemoveNode(node);
      },isRoot:function (node) {
        return node === node.parent;
      },isMountpoint:function (node) {
        return !!node.mounted;
      },isFile:function (mode) {
        return (mode & 61440) === 32768;
      },isDir:function (mode) {
        return (mode & 61440) === 16384;
      },isLink:function (mode) {
        return (mode & 61440) === 40960;
      },isChrdev:function (mode) {
        return (mode & 61440) === 8192;
      },isBlkdev:function (mode) {
        return (mode & 61440) === 24576;
      },isFIFO:function (mode) {
        return (mode & 61440) === 4096;
      },isSocket:function (mode) {
        return (mode & 49152) === 49152;
      },flagModes:{"r":0,"rs":1052672,"r+":2,"w":577,"wx":705,"xw":705,"w+":578,"wx+":706,"xw+":706,"a":1089,"ax":1217,"xa":1217,"a+":1090,"ax+":1218,"xa+":1218},modeStringToFlags:function (str) {
        var flags = FS.flagModes[str];
        if (typeof flags === 'undefined') {
          throw new Error('Unknown file open mode: ' + str);
        }
        return flags;
      },flagsToPermissionString:function (flag) {
        var perms = ['r', 'w', 'rw'][flag & 3];
        if ((flag & 512)) {
          perms += 'w';
        }
        return perms;
      },nodePermissions:function (node, perms) {
        if (FS.ignorePermissions) {
          return 0;
        }
        // return 0 if any user, group or owner bits are set.
        if (perms.indexOf('r') !== -1 && !(node.mode & 292)) {
          return ERRNO_CODES.EACCES;
        } else if (perms.indexOf('w') !== -1 && !(node.mode & 146)) {
          return ERRNO_CODES.EACCES;
        } else if (perms.indexOf('x') !== -1 && !(node.mode & 73)) {
          return ERRNO_CODES.EACCES;
        }
        return 0;
      },mayLookup:function (dir) {
        var err = FS.nodePermissions(dir, 'x');
        if (err) return err;
        if (!dir.node_ops.lookup) return ERRNO_CODES.EACCES;
        return 0;
      },mayCreate:function (dir, name) {
        try {
          var node = FS.lookupNode(dir, name);
          return ERRNO_CODES.EEXIST;
        } catch (e) {
        }
        return FS.nodePermissions(dir, 'wx');
      },mayDelete:function (dir, name, isdir) {
        var node;
        try {
          node = FS.lookupNode(dir, name);
        } catch (e) {
          return e.errno;
        }
        var err = FS.nodePermissions(dir, 'wx');
        if (err) {
          return err;
        }
        if (isdir) {
          if (!FS.isDir(node.mode)) {
            return ERRNO_CODES.ENOTDIR;
          }
          if (FS.isRoot(node) || FS.getPath(node) === FS.cwd()) {
            return ERRNO_CODES.EBUSY;
          }
        } else {
          if (FS.isDir(node.mode)) {
            return ERRNO_CODES.EISDIR;
          }
        }
        return 0;
      },mayOpen:function (node, flags) {
        if (!node) {
          return ERRNO_CODES.ENOENT;
        }
        if (FS.isLink(node.mode)) {
          return ERRNO_CODES.ELOOP;
        } else if (FS.isDir(node.mode)) {
          if (FS.flagsToPermissionString(flags) !== 'r' || // opening for write
              (flags & 512)) { // TODO: check for O_SEARCH? (== search for dir only)
            return ERRNO_CODES.EISDIR;
          }
        }
        return FS.nodePermissions(node, FS.flagsToPermissionString(flags));
      },MAX_OPEN_FDS:4096,nextfd:function (fd_start, fd_end) {
        fd_start = fd_start || 0;
        fd_end = fd_end || FS.MAX_OPEN_FDS;
        for (var fd = fd_start; fd <= fd_end; fd++) {
          if (!FS.streams[fd]) {
            return fd;
          }
        }
        throw new FS.ErrnoError(ERRNO_CODES.EMFILE);
      },getStream:function (fd) {
        return FS.streams[fd];
      },createStream:function (stream, fd_start, fd_end) {
        if (!FS.FSStream) {
          FS.FSStream = function(){};
          FS.FSStream.prototype = {};
          // compatibility
          Object.defineProperties(FS.FSStream.prototype, {
            object: {
              get: function() { return this.node; },
              set: function(val) { this.node = val; }
            },
            isRead: {
              get: function() { return (this.flags & 2097155) !== 1; }
            },
            isWrite: {
              get: function() { return (this.flags & 2097155) !== 0; }
            },
            isAppend: {
              get: function() { return (this.flags & 1024); }
            }
          });
        }
        // clone it, so we can return an instance of FSStream
        var newStream = new FS.FSStream();
        for (var p in stream) {
          newStream[p] = stream[p];
        }
        stream = newStream;
        var fd = FS.nextfd(fd_start, fd_end);
        stream.fd = fd;
        FS.streams[fd] = stream;
        return stream;
      },closeStream:function (fd) {
        FS.streams[fd] = null;
      },chrdev_stream_ops:{open:function (stream) {
          var device = FS.getDevice(stream.node.rdev);
          // override node's stream ops with the device's
          stream.stream_ops = device.stream_ops;
          // forward the open call
          if (stream.stream_ops.open) {
            stream.stream_ops.open(stream);
          }
        },llseek:function () {
          throw new FS.ErrnoError(ERRNO_CODES.ESPIPE);
        }},major:function (dev) {
        return ((dev) >> 8);
      },minor:function (dev) {
        return ((dev) & 0xff);
      },makedev:function (ma, mi) {
        return ((ma) << 8 | (mi));
      },registerDevice:function (dev, ops) {
        FS.devices[dev] = { stream_ops: ops };
      },getDevice:function (dev) {
        return FS.devices[dev];
      },getMounts:function (mount) {
        var mounts = [];
        var check = [mount];

        while (check.length) {
          var m = check.pop();

          mounts.push(m);

          check.push.apply(check, m.mounts);
        }

        return mounts;
      },syncfs:function (populate, callback) {
        if (typeof(populate) === 'function') {
          callback = populate;
          populate = false;
        }

        FS.syncFSRequests++;

        if (FS.syncFSRequests > 1) {
          console.log('warning: ' + FS.syncFSRequests + ' FS.syncfs operations in flight at once, probably just doing extra work');
        }

        var mounts = FS.getMounts(FS.root.mount);
        var completed = 0;

        function doCallback(err) {
          assert(FS.syncFSRequests > 0);
          FS.syncFSRequests--;
          return callback(err);
        }

        function done(err) {
          if (err) {
            if (!done.errored) {
              done.errored = true;
              return doCallback(err);
            }
            return;
          }
          if (++completed >= mounts.length) {
            doCallback(null);
          }
        };

        // sync all mounts
        mounts.forEach(function (mount) {
          if (!mount.type.syncfs) {
            return done(null);
          }
          mount.type.syncfs(mount, populate, done);
        });
      },mount:function (type, opts, mountpoint) {
        var root = mountpoint === '/';
        var pseudo = !mountpoint;
        var node;

        if (root && FS.root) {
          throw new FS.ErrnoError(ERRNO_CODES.EBUSY);
        } else if (!root && !pseudo) {
          var lookup = FS.lookupPath(mountpoint, { follow_mount: false });

          mountpoint = lookup.path;  // use the absolute path
          node = lookup.node;

          if (FS.isMountpoint(node)) {
            throw new FS.ErrnoError(ERRNO_CODES.EBUSY);
          }

          if (!FS.isDir(node.mode)) {
            throw new FS.ErrnoError(ERRNO_CODES.ENOTDIR);
          }
        }

        var mount = {
          type: type,
          opts: opts,
          mountpoint: mountpoint,
          mounts: []
        };

        // create a root node for the fs
        var mountRoot = type.mount(mount);
        mountRoot.mount = mount;
        mount.root = mountRoot;

        if (root) {
          FS.root = mountRoot;
        } else if (node) {
          // set as a mountpoint
          node.mounted = mount;

          // add the new mount to the current mount's children
          if (node.mount) {
            node.mount.mounts.push(mount);
          }
        }

        return mountRoot;
      },unmount:function (mountpoint) {
        var lookup = FS.lookupPath(mountpoint, { follow_mount: false });

        if (!FS.isMountpoint(lookup.node)) {
          throw new FS.ErrnoError(ERRNO_CODES.EINVAL);
        }

        // destroy the nodes for this mount, and all its child mounts
        var node = lookup.node;
        var mount = node.mounted;
        var mounts = FS.getMounts(mount);

        Object.keys(FS.nameTable).forEach(function (hash) {
          var current = FS.nameTable[hash];

          while (current) {
            var next = current.name_next;

            if (mounts.indexOf(current.mount) !== -1) {
              FS.destroyNode(current);
            }

            current = next;
          }
        });

        // no longer a mountpoint
        node.mounted = null;

        // remove this mount from the child mounts
        var idx = node.mount.mounts.indexOf(mount);
        assert(idx !== -1);
        node.mount.mounts.splice(idx, 1);
      },lookup:function (parent, name) {
        return parent.node_ops.lookup(parent, name);
      },mknod:function (path, mode, dev) {
        var lookup = FS.lookupPath(path, { parent: true });
        var parent = lookup.node;
        var name = PATH.basename(path);
        if (!name || name === '.' || name === '..') {
          throw new FS.ErrnoError(ERRNO_CODES.EINVAL);
        }
        var err = FS.mayCreate(parent, name);
        if (err) {
          throw new FS.ErrnoError(err);
        }
        if (!parent.node_ops.mknod) {
          throw new FS.ErrnoError(ERRNO_CODES.EPERM);
        }
        return parent.node_ops.mknod(parent, name, mode, dev);
      },create:function (path, mode) {
        mode = mode !== undefined ? mode : 438 /* 0666 */;
        mode &= 4095;
        mode |= 32768;
        return FS.mknod(path, mode, 0);
      },mkdir:function (path, mode) {
        mode = mode !== undefined ? mode : 511 /* 0777 */;
        mode &= 511 | 512;
        mode |= 16384;
        return FS.mknod(path, mode, 0);
      },mkdirTree:function (path, mode) {
        var dirs = path.split('/');
        var d = '';
        for (var i = 0; i < dirs.length; ++i) {
          if (!dirs[i]) continue;
          d += '/' + dirs[i];
          try {
            FS.mkdir(d, mode);
          } catch(e) {
            if (e.errno != ERRNO_CODES.EEXIST) throw e;
          }
        }
      },mkdev:function (path, mode, dev) {
        if (typeof(dev) === 'undefined') {
          dev = mode;
          mode = 438 /* 0666 */;
        }
        mode |= 8192;
        return FS.mknod(path, mode, dev);
      },symlink:function (oldpath, newpath) {
        if (!PATH.resolve(oldpath)) {
          throw new FS.ErrnoError(ERRNO_CODES.ENOENT);
        }
        var lookup = FS.lookupPath(newpath, { parent: true });
        var parent = lookup.node;
        if (!parent) {
          throw new FS.ErrnoError(ERRNO_CODES.ENOENT);
        }
        var newname = PATH.basename(newpath);
        var err = FS.mayCreate(parent, newname);
        if (err) {
          throw new FS.ErrnoError(err);
        }
        if (!parent.node_ops.symlink) {
          throw new FS.ErrnoError(ERRNO_CODES.EPERM);
        }
        return parent.node_ops.symlink(parent, newname, oldpath);
      },rename:function (old_path, new_path) {
        var old_dirname = PATH.dirname(old_path);
        var new_dirname = PATH.dirname(new_path);
        var old_name = PATH.basename(old_path);
        var new_name = PATH.basename(new_path);
        // parents must exist
        var lookup, old_dir, new_dir;
        try {
          lookup = FS.lookupPath(old_path, { parent: true });
          old_dir = lookup.node;
          lookup = FS.lookupPath(new_path, { parent: true });
          new_dir = lookup.node;
        } catch (e) {
          throw new FS.ErrnoError(ERRNO_CODES.EBUSY);
        }
        if (!old_dir || !new_dir) throw new FS.ErrnoError(ERRNO_CODES.ENOENT);
        // need to be part of the same mount
        if (old_dir.mount !== new_dir.mount) {
          throw new FS.ErrnoError(ERRNO_CODES.EXDEV);
        }
        // source must exist
        var old_node = FS.lookupNode(old_dir, old_name);
        // old path should not be an ancestor of the new path
        var relative = PATH.relative(old_path, new_dirname);
        if (relative.charAt(0) !== '.') {
          throw new FS.ErrnoError(ERRNO_CODES.EINVAL);
        }
        // new path should not be an ancestor of the old path
        relative = PATH.relative(new_path, old_dirname);
        if (relative.charAt(0) !== '.') {
          throw new FS.ErrnoError(ERRNO_CODES.ENOTEMPTY);
        }
        // see if the new path already exists
        var new_node;
        try {
          new_node = FS.lookupNode(new_dir, new_name);
        } catch (e) {
          // not fatal
        }
        // early out if nothing needs to change
        if (old_node === new_node) {
          return;
        }
        // we'll need to delete the old entry
        var isdir = FS.isDir(old_node.mode);
        var err = FS.mayDelete(old_dir, old_name, isdir);
        if (err) {
          throw new FS.ErrnoError(err);
        }
        // need delete permissions if we'll be overwriting.
        // need create permissions if new doesn't already exist.
        err = new_node ?
          FS.mayDelete(new_dir, new_name, isdir) :
          FS.mayCreate(new_dir, new_name);
        if (err) {
          throw new FS.ErrnoError(err);
        }
        if (!old_dir.node_ops.rename) {
          throw new FS.ErrnoError(ERRNO_CODES.EPERM);
        }
        if (FS.isMountpoint(old_node) || (new_node && FS.isMountpoint(new_node))) {
          throw new FS.ErrnoError(ERRNO_CODES.EBUSY);
        }
        // if we are going to change the parent, check write permissions
        if (new_dir !== old_dir) {
          err = FS.nodePermissions(old_dir, 'w');
          if (err) {
            throw new FS.ErrnoError(err);
          }
        }
        try {
          if (FS.trackingDelegate['willMovePath']) {
            FS.trackingDelegate['willMovePath'](old_path, new_path);
          }
        } catch(e) {
          console.log("FS.trackingDelegate['willMovePath']('"+old_path+"', '"+new_path+"') threw an exception: " + e.message);
        }
        // remove the node from the lookup hash
        FS.hashRemoveNode(old_node);
        // do the underlying fs rename
        try {
          old_dir.node_ops.rename(old_node, new_dir, new_name);
        } catch (e) {
          throw e;
        } finally {
          // add the node back to the hash (in case node_ops.rename
          // changed its name)
          FS.hashAddNode(old_node);
        }
        try {
          if (FS.trackingDelegate['onMovePath']) FS.trackingDelegate['onMovePath'](old_path, new_path);
        } catch(e) {
          console.log("FS.trackingDelegate['onMovePath']('"+old_path+"', '"+new_path+"') threw an exception: " + e.message);
        }
      },rmdir:function (path) {
        var lookup = FS.lookupPath(path, { parent: true });
        var parent = lookup.node;
        var name = PATH.basename(path);
        var node = FS.lookupNode(parent, name);
        var err = FS.mayDelete(parent, name, true);
        if (err) {
          throw new FS.ErrnoError(err);
        }
        if (!parent.node_ops.rmdir) {
          throw new FS.ErrnoError(ERRNO_CODES.EPERM);
        }
        if (FS.isMountpoint(node)) {
          throw new FS.ErrnoError(ERRNO_CODES.EBUSY);
        }
        try {
          if (FS.trackingDelegate['willDeletePath']) {
            FS.trackingDelegate['willDeletePath'](path);
          }
        } catch(e) {
          console.log("FS.trackingDelegate['willDeletePath']('"+path+"') threw an exception: " + e.message);
        }
        parent.node_ops.rmdir(parent, name);
        FS.destroyNode(node);
        try {
          if (FS.trackingDelegate['onDeletePath']) FS.trackingDelegate['onDeletePath'](path);
        } catch(e) {
          console.log("FS.trackingDelegate['onDeletePath']('"+path+"') threw an exception: " + e.message);
        }
      },readdir:function (path) {
        var lookup = FS.lookupPath(path, { follow: true });
        var node = lookup.node;
        if (!node.node_ops.readdir) {
          throw new FS.ErrnoError(ERRNO_CODES.ENOTDIR);
        }
        return node.node_ops.readdir(node);
      },unlink:function (path) {
        var lookup = FS.lookupPath(path, { parent: true });
        var parent = lookup.node;
        var name = PATH.basename(path);
        var node = FS.lookupNode(parent, name);
        var err = FS.mayDelete(parent, name, false);
        if (err) {
          // According to POSIX, we should map EISDIR to EPERM, but
          // we instead do what Linux does (and we must, as we use
          // the musl linux libc).
          throw new FS.ErrnoError(err);
        }
        if (!parent.node_ops.unlink) {
          throw new FS.ErrnoError(ERRNO_CODES.EPERM);
        }
        if (FS.isMountpoint(node)) {
          throw new FS.ErrnoError(ERRNO_CODES.EBUSY);
        }
        try {
          if (FS.trackingDelegate['willDeletePath']) {
            FS.trackingDelegate['willDeletePath'](path);
          }
        } catch(e) {
          console.log("FS.trackingDelegate['willDeletePath']('"+path+"') threw an exception: " + e.message);
        }
        parent.node_ops.unlink(parent, name);
        FS.destroyNode(node);
        try {
          if (FS.trackingDelegate['onDeletePath']) FS.trackingDelegate['onDeletePath'](path);
        } catch(e) {
          console.log("FS.trackingDelegate['onDeletePath']('"+path+"') threw an exception: " + e.message);
        }
      },readlink:function (path) {
        var lookup = FS.lookupPath(path);
        var link = lookup.node;
        if (!link) {
          throw new FS.ErrnoError(ERRNO_CODES.ENOENT);
        }
        if (!link.node_ops.readlink) {
          throw new FS.ErrnoError(ERRNO_CODES.EINVAL);
        }
        return PATH.resolve(FS.getPath(link.parent), link.node_ops.readlink(link));
      },stat:function (path, dontFollow) {
        var lookup = FS.lookupPath(path, { follow: !dontFollow });
        var node = lookup.node;
        if (!node) {
          throw new FS.ErrnoError(ERRNO_CODES.ENOENT);
        }
        if (!node.node_ops.getattr) {
          throw new FS.ErrnoError(ERRNO_CODES.EPERM);
        }
        return node.node_ops.getattr(node);
      },lstat:function (path) {
        return FS.stat(path, true);
      },chmod:function (path, mode, dontFollow) {
        var node;
        if (typeof path === 'string') {
          var lookup = FS.lookupPath(path, { follow: !dontFollow });
          node = lookup.node;
        } else {
          node = path;
        }
        if (!node.node_ops.setattr) {
          throw new FS.ErrnoError(ERRNO_CODES.EPERM);
        }
        node.node_ops.setattr(node, {
          mode: (mode & 4095) | (node.mode & ~4095),
          timestamp: Date.now()
        });
      },lchmod:function (path, mode) {
        FS.chmod(path, mode, true);
      },fchmod:function (fd, mode) {
        var stream = FS.getStream(fd);
        if (!stream) {
          throw new FS.ErrnoError(ERRNO_CODES.EBADF);
        }
        FS.chmod(stream.node, mode);
      },chown:function (path, uid, gid, dontFollow) {
        var node;
        if (typeof path === 'string') {
          var lookup = FS.lookupPath(path, { follow: !dontFollow });
          node = lookup.node;
        } else {
          node = path;
        }
        if (!node.node_ops.setattr) {
          throw new FS.ErrnoError(ERRNO_CODES.EPERM);
        }
        node.node_ops.setattr(node, {
          timestamp: Date.now()
          // we ignore the uid / gid for now
        });
      },lchown:function (path, uid, gid) {
        FS.chown(path, uid, gid, true);
      },fchown:function (fd, uid, gid) {
        var stream = FS.getStream(fd);
        if (!stream) {
          throw new FS.ErrnoError(ERRNO_CODES.EBADF);
        }
        FS.chown(stream.node, uid, gid);
      },truncate:function (path, len) {
        if (len < 0) {
          throw new FS.ErrnoError(ERRNO_CODES.EINVAL);
        }
        var node;
        if (typeof path === 'string') {
          var lookup = FS.lookupPath(path, { follow: true });
          node = lookup.node;
        } else {
          node = path;
        }
        if (!node.node_ops.setattr) {
          throw new FS.ErrnoError(ERRNO_CODES.EPERM);
        }
        if (FS.isDir(node.mode)) {
          throw new FS.ErrnoError(ERRNO_CODES.EISDIR);
        }
        if (!FS.isFile(node.mode)) {
          throw new FS.ErrnoError(ERRNO_CODES.EINVAL);
        }
        var err = FS.nodePermissions(node, 'w');
        if (err) {
          throw new FS.ErrnoError(err);
        }
        node.node_ops.setattr(node, {
          size: len,
          timestamp: Date.now()
        });
      },ftruncate:function (fd, len) {
        var stream = FS.getStream(fd);
        if (!stream) {
          throw new FS.ErrnoError(ERRNO_CODES.EBADF);
        }
        if ((stream.flags & 2097155) === 0) {
          throw new FS.ErrnoError(ERRNO_CODES.EINVAL);
        }
        FS.truncate(stream.node, len);
      },utime:function (path, atime, mtime) {
        var lookup = FS.lookupPath(path, { follow: true });
        var node = lookup.node;
        node.node_ops.setattr(node, {
          timestamp: Math.max(atime, mtime)
        });
      },open:function (path, flags, mode, fd_start, fd_end) {
        if (path === "") {
          throw new FS.ErrnoError(ERRNO_CODES.ENOENT);
        }
        flags = typeof flags === 'string' ? FS.modeStringToFlags(flags) : flags;
        mode = typeof mode === 'undefined' ? 438 /* 0666 */ : mode;
        if ((flags & 64)) {
          mode = (mode & 4095) | 32768;
        } else {
          mode = 0;
        }
        var node;
        if (typeof path === 'object') {
          node = path;
        } else {
          path = PATH.normalize(path);
          try {
            var lookup = FS.lookupPath(path, {
              follow: !(flags & 131072)
            });
            node = lookup.node;
          } catch (e) {
            // ignore
          }
        }
        // perhaps we need to create the node
        var created = false;
        if ((flags & 64)) {
          if (node) {
            // if O_CREAT and O_EXCL are set, error out if the node already exists
            if ((flags & 128)) {
              throw new FS.ErrnoError(ERRNO_CODES.EEXIST);
            }
          } else {
            // node doesn't exist, try to create it
            node = FS.mknod(path, mode, 0);
            created = true;
          }
        }
        if (!node) {
          throw new FS.ErrnoError(ERRNO_CODES.ENOENT);
        }
        // can't truncate a device
        if (FS.isChrdev(node.mode)) {
          flags &= ~512;
        }
        // if asked only for a directory, then this must be one
        if ((flags & 65536) && !FS.isDir(node.mode)) {
          throw new FS.ErrnoError(ERRNO_CODES.ENOTDIR);
        }
        // check permissions, if this is not a file we just created now (it is ok to
        // create and write to a file with read-only permissions; it is read-only
        // for later use)
        if (!created) {
          var err = FS.mayOpen(node, flags);
          if (err) {
            throw new FS.ErrnoError(err);
          }
        }
        // do truncation if necessary
        if ((flags & 512)) {
          FS.truncate(node, 0);
        }
        // we've already handled these, don't pass down to the underlying vfs
        flags &= ~(128 | 512);

        // register the stream with the filesystem
        var stream = FS.createStream({
          node: node,
          path: FS.getPath(node),  // we want the absolute path to the node
          flags: flags,
          seekable: true,
          position: 0,
          stream_ops: node.stream_ops,
          // used by the file family libc calls (fopen, fwrite, ferror, etc.)
          ungotten: [],
          error: false
        }, fd_start, fd_end);
        // call the new stream's open function
        if (stream.stream_ops.open) {
          stream.stream_ops.open(stream);
        }
        if (Module['logReadFiles'] && !(flags & 1)) {
          if (!FS.readFiles) FS.readFiles = {};
          if (!(path in FS.readFiles)) {
            FS.readFiles[path] = 1;
            Module['printErr']('read file: ' + path);
          }
        }
        try {
          if (FS.trackingDelegate['onOpenFile']) {
            var trackingFlags = 0;
            if ((flags & 2097155) !== 1) {
              trackingFlags |= FS.tracking.openFlags.READ;
            }
            if ((flags & 2097155) !== 0) {
              trackingFlags |= FS.tracking.openFlags.WRITE;
            }
            FS.trackingDelegate['onOpenFile'](path, trackingFlags);
          }
        } catch(e) {
          console.log("FS.trackingDelegate['onOpenFile']('"+path+"', flags) threw an exception: " + e.message);
        }
        return stream;
      },close:function (stream) {
        if (stream.getdents) stream.getdents = null; // free readdir state
        try {
          if (stream.stream_ops.close) {
            stream.stream_ops.close(stream);
          }
        } catch (e) {
          throw e;
        } finally {
          FS.closeStream(stream.fd);
        }
      },llseek:function (stream, offset, whence) {
        if (!stream.seekable || !stream.stream_ops.llseek) {
          throw new FS.ErrnoError(ERRNO_CODES.ESPIPE);
        }
        stream.position = stream.stream_ops.llseek(stream, offset, whence);
        stream.ungotten = [];
        return stream.position;
      },read:function (stream, buffer, offset, length, position) {
        if (length < 0 || position < 0) {
          throw new FS.ErrnoError(ERRNO_CODES.EINVAL);
        }
        if ((stream.flags & 2097155) === 1) {
          throw new FS.ErrnoError(ERRNO_CODES.EBADF);
        }
        if (FS.isDir(stream.node.mode)) {
          throw new FS.ErrnoError(ERRNO_CODES.EISDIR);
        }
        if (!stream.stream_ops.read) {
          throw new FS.ErrnoError(ERRNO_CODES.EINVAL);
        }
        var seeking = true;
        if (typeof position === 'undefined') {
          position = stream.position;
          seeking = false;
        } else if (!stream.seekable) {
          throw new FS.ErrnoError(ERRNO_CODES.ESPIPE);
        }
        var bytesRead = stream.stream_ops.read(stream, buffer, offset, length, position);
        if (!seeking) stream.position += bytesRead;
        return bytesRead;
      },write:function (stream, buffer, offset, length, position, canOwn) {
        if (length < 0 || position < 0) {
          throw new FS.ErrnoError(ERRNO_CODES.EINVAL);
        }
        if ((stream.flags & 2097155) === 0) {
          throw new FS.ErrnoError(ERRNO_CODES.EBADF);
        }
        if (FS.isDir(stream.node.mode)) {
          throw new FS.ErrnoError(ERRNO_CODES.EISDIR);
        }
        if (!stream.stream_ops.write) {
          throw new FS.ErrnoError(ERRNO_CODES.EINVAL);
        }
        if (stream.flags & 1024) {
          // seek to the end before writing in append mode
          FS.llseek(stream, 0, 2);
        }
        var seeking = true;
        if (typeof position === 'undefined') {
          position = stream.position;
          seeking = false;
        } else if (!stream.seekable) {
          throw new FS.ErrnoError(ERRNO_CODES.ESPIPE);
        }
        var bytesWritten = stream.stream_ops.write(stream, buffer, offset, length, position, canOwn);
        if (!seeking) stream.position += bytesWritten;
        try {
          if (stream.path && FS.trackingDelegate['onWriteToFile']) FS.trackingDelegate['onWriteToFile'](stream.path);
        } catch(e) {
          console.log("FS.trackingDelegate['onWriteToFile']('"+path+"') threw an exception: " + e.message);
        }
        return bytesWritten;
      },allocate:function (stream, offset, length) {
        if (offset < 0 || length <= 0) {
          throw new FS.ErrnoError(ERRNO_CODES.EINVAL);
        }
        if ((stream.flags & 2097155) === 0) {
          throw new FS.ErrnoError(ERRNO_CODES.EBADF);
        }
        if (!FS.isFile(stream.node.mode) && !FS.isDir(stream.node.mode)) {
          throw new FS.ErrnoError(ERRNO_CODES.ENODEV);
        }
        if (!stream.stream_ops.allocate) {
          throw new FS.ErrnoError(ERRNO_CODES.EOPNOTSUPP);
        }
        stream.stream_ops.allocate(stream, offset, length);
      },mmap:function (stream, buffer, offset, length, position, prot, flags) {
        // TODO if PROT is PROT_WRITE, make sure we have write access
        if ((stream.flags & 2097155) === 1) {
          throw new FS.ErrnoError(ERRNO_CODES.EACCES);
        }
        if (!stream.stream_ops.mmap) {
          throw new FS.ErrnoError(ERRNO_CODES.ENODEV);
        }
        return stream.stream_ops.mmap(stream, buffer, offset, length, position, prot, flags);
      },msync:function (stream, buffer, offset, length, mmapFlags) {
        if (!stream || !stream.stream_ops.msync) {
          return 0;
        }
        return stream.stream_ops.msync(stream, buffer, offset, length, mmapFlags);
      },munmap:function (stream) {
        return 0;
      },ioctl:function (stream, cmd, arg) {
        if (!stream.stream_ops.ioctl) {
          throw new FS.ErrnoError(ERRNO_CODES.ENOTTY);
        }
        return stream.stream_ops.ioctl(stream, cmd, arg);
      },readFile:function (path, opts) {
        opts = opts || {};
        opts.flags = opts.flags || 'r';
        opts.encoding = opts.encoding || 'binary';
        if (opts.encoding !== 'utf8' && opts.encoding !== 'binary') {
          throw new Error('Invalid encoding type "' + opts.encoding + '"');
        }
        var ret;
        var stream = FS.open(path, opts.flags);
        var stat = FS.stat(path);
        var length = stat.size;
        var buf = new Uint8Array(length);
        FS.read(stream, buf, 0, length, 0);
        if (opts.encoding === 'utf8') {
          ret = UTF8ArrayToString(buf, 0);
        } else if (opts.encoding === 'binary') {
          ret = buf;
        }
        FS.close(stream);
        return ret;
      },writeFile:function (path, data, opts) {
        opts = opts || {};
        opts.flags = opts.flags || 'w';
        opts.encoding = opts.encoding || 'utf8';
        if (opts.encoding !== 'utf8' && opts.encoding !== 'binary') {
          throw new Error('Invalid encoding type "' + opts.encoding + '"');
        }
        var stream = FS.open(path, opts.flags, opts.mode);
        if (opts.encoding === 'utf8') {
          var buf = new Uint8Array(lengthBytesUTF8(data)+1);
          var actualNumBytes = stringToUTF8Array(data, buf, 0, buf.length);
          FS.write(stream, buf, 0, actualNumBytes, 0, opts.canOwn);
        } else if (opts.encoding === 'binary') {
          FS.write(stream, data, 0, data.length, 0, opts.canOwn);
        }
        FS.close(stream);
      },cwd:function () {
        return FS.currentPath;
      },chdir:function (path) {
        var lookup = FS.lookupPath(path, { follow: true });
        if (lookup.node === null) {
          throw new FS.ErrnoError(ERRNO_CODES.ENOENT);
        }
        if (!FS.isDir(lookup.node.mode)) {
          throw new FS.ErrnoError(ERRNO_CODES.ENOTDIR);
        }
        var err = FS.nodePermissions(lookup.node, 'x');
        if (err) {
          throw new FS.ErrnoError(err);
        }
        FS.currentPath = lookup.path;
      },createDefaultDirectories:function () {
        FS.mkdir('/tmp');
        FS.mkdir('/home');
        FS.mkdir('/home/electron');
        FS.mkdir('/macintosh.js')
      },createDefaultDevices:function () {
        // create /dev
        FS.mkdir('/dev');
        // setup /dev/null
        FS.registerDevice(FS.makedev(1, 3), {
          read: function() { return 0; },
          write: function(stream, buffer, offset, length, pos) { return length; }
        });
        FS.mkdev('/dev/null', FS.makedev(1, 3));
        // setup /dev/tty and /dev/tty1
        // stderr needs to print output using Module['printErr']
        // so we register a second tty just for it.
        TTY.register(FS.makedev(5, 0), TTY.default_tty_ops);
        TTY.register(FS.makedev(6, 0), TTY.default_tty1_ops);
        FS.mkdev('/dev/tty', FS.makedev(5, 0));
        FS.mkdev('/dev/tty1', FS.makedev(6, 0));
        // setup /dev/[u]random
        var random_device;
        if (typeof crypto !== 'undefined') {
          // for modern web browsers
          var randomBuffer = new Uint8Array(1);
          random_device = function() { crypto.getRandomValues(randomBuffer); return randomBuffer[0]; };
        } else if (ENVIRONMENT_IS_NODE) {
          // for nodejs
          random_device = function() { return require('crypto').randomBytes(1)[0]; };
        } else {
          // default for ES5 platforms
          random_device = function() { return (Math.random()*256)|0; };
        }
        FS.createDevice('/dev', 'random', random_device);
        FS.createDevice('/dev', 'urandom', random_device);
        // we're not going to emulate the actual shm device,
        // just create the tmp dirs that reside in it commonly
        FS.mkdir('/dev/shm');
        FS.mkdir('/dev/shm/tmp');
      },createSpecialDirectories:function () {
        // create /proc/self/fd which allows /proc/self/fd/6 => readlink gives the name of the stream for fd 6 (see test_unistd_ttyname)
        FS.mkdir('/proc');
        FS.mkdir('/proc/self');
        FS.mkdir('/proc/self/fd');
        FS.mount({
          mount: function() {
            var node = FS.createNode('/proc/self', 'fd', 16384 | 511 /* 0777 */, 73);
            node.node_ops = {
              lookup: function(parent, name) {
                var fd = +name;
                var stream = FS.getStream(fd);
                if (!stream) throw new FS.ErrnoError(ERRNO_CODES.EBADF);
                var ret = {
                  parent: null,
                  mount: { mountpoint: 'fake' },
                  node_ops: { readlink: function() { return stream.path } }
                };
                ret.parent = ret; // make it look like a simple root node
                return ret;
              }
            };
            return node;
          }
        }, {}, '/proc/self/fd');
      },createStandardStreams:function () {
        // TODO deprecate the old functionality of a single
        // input / output callback and that utilizes FS.createDevice
        // and instead require a unique set of stream ops

        // by default, we symlink the standard streams to the
        // default tty devices. however, if the standard streams
        // have been overwritten we create a unique device for
        // them instead.
        if (Module['stdin']) {
          FS.createDevice('/dev', 'stdin', Module['stdin']);
        } else {
          FS.symlink('/dev/tty', '/dev/stdin');
        }
        if (Module['stdout']) {
          FS.createDevice('/dev', 'stdout', null, Module['stdout']);
        } else {
          FS.symlink('/dev/tty', '/dev/stdout');
        }
        if (Module['stderr']) {
          FS.createDevice('/dev', 'stderr', null, Module['stderr']);
        } else {
          FS.symlink('/dev/tty1', '/dev/stderr');
        }

        // open default streams for the stdin, stdout and stderr devices
        var stdin = FS.open('/dev/stdin', 'r');
        assert(stdin.fd === 0, 'invalid handle for stdin (' + stdin.fd + ')');

        var stdout = FS.open('/dev/stdout', 'w');
        assert(stdout.fd === 1, 'invalid handle for stdout (' + stdout.fd + ')');

        var stderr = FS.open('/dev/stderr', 'w');
        assert(stderr.fd === 2, 'invalid handle for stderr (' + stderr.fd + ')');
      },ensureErrnoError:function () {
        if (FS.ErrnoError) return;
        FS.ErrnoError = function ErrnoError(errno, node) {
          //Module.printErr(stackTrace()); // useful for debugging
          this.node = node;
          this.setErrno = function(errno) {
            this.errno = errno;
            for (var key in ERRNO_CODES) {
              if (ERRNO_CODES[key] === errno) {
                this.code = key;
                break;
              }
            }
          };
          this.setErrno(errno);
          this.message = ERRNO_MESSAGES[errno];
        };
        FS.ErrnoError.prototype = new Error();
        FS.ErrnoError.prototype.constructor = FS.ErrnoError;
        // Some errors may happen quite a bit, to avoid overhead we reuse them (and suffer a lack of stack info)
        [ERRNO_CODES.ENOENT].forEach(function(code) {
          FS.genericErrors[code] = new FS.ErrnoError(code);
          FS.genericErrors[code].stack = '<generic error, no stack>';
        });
      },staticInit:function () {
        FS.ensureErrnoError();

        FS.nameTable = new Array(4096);

        FS.mount(MEMFS, {}, '/');

        FS.createDefaultDirectories();
        FS.createDefaultDevices();
        FS.createSpecialDirectories();

        // Begin surgery
        //if (ENVIRONMENT_IS_NODE) { var fs = require("fs"); var NODEJS_PATH = require("path"); NODEFS.staticInit(); };
        //FS.mount(NODEFS, { root: '.' }, '/home/electron');
        // onmessage(() => {
        //   console.log('Message received from main script');
        // });

        // End surgery

        FS.filesystems = {
          'MEMFS': MEMFS,
          'IDBFS': IDBFS,
          'NODEFS': NODEFS,
          'WORKERFS': WORKERFS,
        };
      },init:function (input, output, error) {
        assert(!FS.init.initialized, 'FS.init was previously called. If you want to initialize later with custom parameters, remove any earlier calls (note that one is automatically added to the generated code)');
        FS.init.initialized = true;

        FS.ensureErrnoError();

        // Allow Module.stdin etc. to provide defaults, if none explicitly passed to us here
        Module['stdin'] = input || Module['stdin'];
        Module['stdout'] = output || Module['stdout'];
        Module['stderr'] = error || Module['stderr'];

        FS.createStandardStreams();
      },quit:function () {
        FS.init.initialized = false;
        // force-flush all streams, so we get musl std streams printed out
        var fflush = Module['_fflush'];
        if (fflush) fflush(0);
        // close all of our streams
        for (var i = 0; i < FS.streams.length; i++) {
          var stream = FS.streams[i];
          if (!stream) {
            continue;
          }
          FS.close(stream);
        }
      },getMode:function (canRead, canWrite) {
        var mode = 0;
        if (canRead) mode |= 292 | 73;
        if (canWrite) mode |= 146;
        return mode;
      },joinPath:function (parts, forceRelative) {
        var path = PATH.join.apply(null, parts);
        if (forceRelative && path[0] == '/') path = path.substr(1);
        return path;
      },absolutePath:function (relative, base) {
        return PATH.resolve(base, relative);
      },standardizePath:function (path) {
        return PATH.normalize(path);
      },findObject:function (path, dontResolveLastLink) {
        var ret = FS.analyzePath(path, dontResolveLastLink);
        if (ret.exists) {
          return ret.object;
        } else {
          ___setErrNo(ret.error);
          return null;
        }
      },analyzePath:function (path, dontResolveLastLink) {
        // operate from within the context of the symlink's target
        try {
          var lookup = FS.lookupPath(path, { follow: !dontResolveLastLink });
          path = lookup.path;
        } catch (e) {
        }
        var ret = {
          isRoot: false, exists: false, error: 0, name: null, path: null, object: null,
          parentExists: false, parentPath: null, parentObject: null
        };
        try {
          var lookup = FS.lookupPath(path, { parent: true });
          ret.parentExists = true;
          ret.parentPath = lookup.path;
          ret.parentObject = lookup.node;
          ret.name = PATH.basename(path);
          lookup = FS.lookupPath(path, { follow: !dontResolveLastLink });
          ret.exists = true;
          ret.path = lookup.path;
          ret.object = lookup.node;
          ret.name = lookup.node.name;
          ret.isRoot = lookup.path === '/';
        } catch (e) {
          ret.error = e.errno;
        };
        return ret;
      },createFolder:function (parent, name, canRead, canWrite) {
        var path = PATH.join2(typeof parent === 'string' ? parent : FS.getPath(parent), name);
        var mode = FS.getMode(canRead, canWrite);
        return FS.mkdir(path, mode);
      },createPath:function (parent, path, canRead, canWrite) {
        parent = typeof parent === 'string' ? parent : FS.getPath(parent);
        var parts = path.split('/').reverse();
        while (parts.length) {
          var part = parts.pop();
          if (!part) continue;
          var current = PATH.join2(parent, part);
          try {
            FS.mkdir(current);
          } catch (e) {
            // ignore EEXIST
          }
          parent = current;
        }
        return current;
      },createFile:function (parent, name, properties, canRead, canWrite) {
        var path = PATH.join2(typeof parent === 'string' ? parent : FS.getPath(parent), name);
        var mode = FS.getMode(canRead, canWrite);
        return FS.create(path, mode);
      },createDataFile:function (parent, name, data, canRead, canWrite, canOwn) {
        var path = name ? PATH.join2(typeof parent === 'string' ? parent : FS.getPath(parent), name) : parent;
        var mode = FS.getMode(canRead, canWrite);
        var node = FS.create(path, mode);
        if (data) {
          if (typeof data === 'string') {
            var arr = new Array(data.length);
            for (var i = 0, len = data.length; i < len; ++i) arr[i] = data.charCodeAt(i);
            data = arr;
          }
          // make sure we can write to the file
          FS.chmod(node, mode | 146);
          var stream = FS.open(node, 'w');
          FS.write(stream, data, 0, data.length, 0, canOwn);
          FS.close(stream);
          FS.chmod(node, mode);
        }
        return node;
      },createDevice:function (parent, name, input, output) {
        var path = PATH.join2(typeof parent === 'string' ? parent : FS.getPath(parent), name);
        var mode = FS.getMode(!!input, !!output);
        if (!FS.createDevice.major) FS.createDevice.major = 64;
        var dev = FS.makedev(FS.createDevice.major++, 0);
        // Create a fake device that a set of stream ops to emulate
        // the old behavior.
        FS.registerDevice(dev, {
          open: function(stream) {
            stream.seekable = false;
          },
          close: function(stream) {
            // flush any pending line data
            if (output && output.buffer && output.buffer.length) {
              output(10);
            }
          },
          read: function(stream, buffer, offset, length, pos /* ignored */) {
            var bytesRead = 0;
            for (var i = 0; i < length; i++) {
              var result;
              try {
                result = input();
              } catch (e) {
                throw new FS.ErrnoError(ERRNO_CODES.EIO);
              }
              if (result === undefined && bytesRead === 0) {
                throw new FS.ErrnoError(ERRNO_CODES.EAGAIN);
              }
              if (result === null || result === undefined) break;
              bytesRead++;
              buffer[offset+i] = result;
            }
            if (bytesRead) {
              stream.node.timestamp = Date.now();
            }
            return bytesRead;
          },
          write: function(stream, buffer, offset, length, pos) {
            for (var i = 0; i < length; i++) {
              try {
                output(buffer[offset+i]);
              } catch (e) {
                throw new FS.ErrnoError(ERRNO_CODES.EIO);
              }
            }
            if (length) {
              stream.node.timestamp = Date.now();
            }
            return i;
          }
        });
        return FS.mkdev(path, mode, dev);
      },createLink:function (parent, name, target, canRead, canWrite) {
        var path = PATH.join2(typeof parent === 'string' ? parent : FS.getPath(parent), name);
        return FS.symlink(target, path);
      },forceLoadFile:function (obj) {
        if (obj.isDevice || obj.isFolder || obj.link || obj.contents) return true;
        var success = true;
        if (typeof XMLHttpRequest !== 'undefined') {
          throw new Error("Lazy loading should have been performed (contents set) in createLazyFile, but it was not. Lazy loading only works in web workers. Use --embed-file or --preload-file in emcc on the main thread.");
        } else if (Module['read']) {
          // Command-line.
          try {
            // WARNING: Can't read binary files in V8's d8 or tracemonkey's js, as
            //          read() will try to parse UTF8.
            obj.contents = intArrayFromString(Module['read'](obj.url), true);
            obj.usedBytes = obj.contents.length;
          } catch (e) {
            success = false;
          }
        } else {
          throw new Error('Cannot load without read() or XMLHttpRequest.');
        }
        if (!success) ___setErrNo(ERRNO_CODES.EIO);
        return success;
      },createLazyFile:function (parent, name, url, canRead, canWrite) {
        // Lazy chunked Uint8Array (implements get and length from Uint8Array). Actual getting is abstracted away for eventual reuse.
        function LazyUint8Array() {
          this.lengthKnown = false;
          this.chunks = []; // Loaded chunks. Index is the chunk number
        }
        LazyUint8Array.prototype.get = function LazyUint8Array_get(idx) {
          if (idx > this.length-1 || idx < 0) {
            return undefined;
          }
          var chunkOffset = idx % this.chunkSize;
          var chunkNum = (idx / this.chunkSize)|0;
          return this.getter(chunkNum)[chunkOffset];
        }
        LazyUint8Array.prototype.setDataGetter = function LazyUint8Array_setDataGetter(getter) {
          this.getter = getter;
        }
        LazyUint8Array.prototype.cacheLength = function LazyUint8Array_cacheLength() {
          // Find length
          var xhr = new XMLHttpRequest();
          xhr.open('HEAD', url, false);
          xhr.send(null);
          if (!(xhr.status >= 200 && xhr.status < 300 || xhr.status === 304)) throw new Error("Couldn't load " + url + ". Status: " + xhr.status);
          var datalength = Number(xhr.getResponseHeader("Content-length"));
          var header;
          var hasByteServing = (header = xhr.getResponseHeader("Accept-Ranges")) && header === "bytes";
          var usesGzip = (header = xhr.getResponseHeader("Content-Encoding")) && header === "gzip";

          var chunkSize = 1024*1024; // Chunk size in bytes

          if (!hasByteServing) chunkSize = datalength;

          // Function to get a range from the remote URL.
          var doXHR = (function(from, to) {
            if (from > to) throw new Error("invalid range (" + from + ", " + to + ") or no bytes requested!");
            if (to > datalength-1) throw new Error("only " + datalength + " bytes available! programmer error!");

            // TODO: Use mozResponseArrayBuffer, responseStream, etc. if available.
            var xhr = new XMLHttpRequest();
            xhr.open('GET', url, false);
            if (datalength !== chunkSize) xhr.setRequestHeader("Range", "bytes=" + from + "-" + to);

            // Some hints to the browser that we want binary data.
            if (typeof Uint8Array != 'undefined') xhr.responseType = 'arraybuffer';
            if (xhr.overrideMimeType) {
              xhr.overrideMimeType('text/plain; charset=x-user-defined');
            }

            xhr.send(null);
            if (!(xhr.status >= 200 && xhr.status < 300 || xhr.status === 304)) throw new Error("Couldn't load " + url + ". Status: " + xhr.status);
            if (xhr.response !== undefined) {
              return new Uint8Array(xhr.response || []);
            } else {
              return intArrayFromString(xhr.responseText || '', true);
            }
          });
          var lazyArray = this;
          lazyArray.setDataGetter(function(chunkNum) {
            var start = chunkNum * chunkSize;
            var end = (chunkNum+1) * chunkSize - 1; // including this byte
            end = Math.min(end, datalength-1); // if datalength-1 is selected, this is the last block
            if (typeof(lazyArray.chunks[chunkNum]) === "undefined") {
              lazyArray.chunks[chunkNum] = doXHR(start, end);
            }
            if (typeof(lazyArray.chunks[chunkNum]) === "undefined") throw new Error("doXHR failed!");
            return lazyArray.chunks[chunkNum];
          });

          if (usesGzip || !datalength) {
            // if the server uses gzip or doesn't supply the length, we have to download the whole file to get the (uncompressed) length
            chunkSize = datalength = 1; // this will force getter(0)/doXHR do download the whole file
            datalength = this.getter(0).length;
            chunkSize = datalength;
            console.log("LazyFiles on gzip forces download of the whole file when length is accessed");
          }

          this._length = datalength;
          this._chunkSize = chunkSize;
          this.lengthKnown = true;
        }
        if (typeof XMLHttpRequest !== 'undefined') {
          if (!ENVIRONMENT_IS_WORKER) throw 'Cannot do synchronous binary XHRs outside webworkers in modern browsers. Use --embed-file or --preload-file in emcc';
          var lazyArray = new LazyUint8Array();
          Object.defineProperties(lazyArray, {
            length: {
              get: function() {
                if(!this.lengthKnown) {
                  this.cacheLength();
                }
                return this._length;
              }
            },
            chunkSize: {
              get: function() {
                if(!this.lengthKnown) {
                  this.cacheLength();
                }
                return this._chunkSize;
              }
            }
          });

          var properties = { isDevice: false, contents: lazyArray };
        } else {
          var properties = { isDevice: false, url: url };
        }

        var node = FS.createFile(parent, name, properties, canRead, canWrite);
        // This is a total hack, but I want to get this lazy file code out of the
        // core of MEMFS. If we want to keep this lazy file concept I feel it should
        // be its own thin LAZYFS proxying calls to MEMFS.
        if (properties.contents) {
          node.contents = properties.contents;
        } else if (properties.url) {
          node.contents = null;
          node.url = properties.url;
        }
        // Add a function that defers querying the file size until it is asked the first time.
        Object.defineProperties(node, {
          usedBytes: {
            get: function() { return this.contents.length; }
          }
        });
        // override each stream op with one that tries to force load the lazy file first
        var stream_ops = {};
        var keys = Object.keys(node.stream_ops);
        keys.forEach(function(key) {
          var fn = node.stream_ops[key];
          stream_ops[key] = function forceLoadLazyFile() {
            if (!FS.forceLoadFile(node)) {
              throw new FS.ErrnoError(ERRNO_CODES.EIO);
            }
            return fn.apply(null, arguments);
          };
        });
        // use a custom read function
        stream_ops.read = function stream_ops_read(stream, buffer, offset, length, position) {
          if (!FS.forceLoadFile(node)) {
            throw new FS.ErrnoError(ERRNO_CODES.EIO);
          }
          var contents = stream.node.contents;
          if (position >= contents.length)
            return 0;
          var size = Math.min(contents.length - position, length);
          assert(size >= 0);
          if (contents.slice) { // normal array
            for (var i = 0; i < size; i++) {
              buffer[offset + i] = contents[position + i];
            }
          } else {
            for (var i = 0; i < size; i++) { // LazyUint8Array from sync binary XHR
              buffer[offset + i] = contents.get(position + i);
            }
          }
          return size;
        };
        node.stream_ops = stream_ops;
        return node;
      },createPreloadedFile:function (parent, name, url, canRead, canWrite, onload, onerror, dontCreateFile, canOwn, preFinish) {
        Browser.init(); // XXX perhaps this method should move onto Browser?
        // TODO we should allow people to just pass in a complete filename instead
        // of parent and name being that we just join them anyways
        var fullname = name ? PATH.resolve(PATH.join2(parent, name)) : parent;
        var dep = getUniqueRunDependency('cp ' + fullname); // might have several active requests for the same fullname
        function processData(byteArray) {
          function finish(byteArray) {
            if (preFinish) preFinish();
            if (!dontCreateFile) {
              FS.createDataFile(parent, name, byteArray, canRead, canWrite, canOwn);
            }
            if (onload) onload();
            removeRunDependency(dep);
          }
          var handled = false;
          Module['preloadPlugins'].forEach(function(plugin) {
            if (handled) return;
            if (plugin['canHandle'](fullname)) {
              plugin['handle'](byteArray, fullname, finish, function() {
                if (onerror) onerror();
                removeRunDependency(dep);
              });
              handled = true;
            }
          });
          if (!handled) finish(byteArray);
        }
        addRunDependency(dep);
        if (typeof url == 'string') {
          Browser.asyncLoad(url, function(byteArray) {
            processData(byteArray);
          }, onerror);
        } else {
          processData(url);
        }
      },indexedDB:function () {
        return window.indexedDB || window.mozIndexedDB || window.webkitIndexedDB || window.msIndexedDB;
      },DB_NAME:function () {
        return 'EM_FS_' + window.location.pathname;
      },DB_VERSION:20,DB_STORE_NAME:"FILE_DATA",saveFilesToDB:function (paths, onload, onerror) {
        onload = onload || function(){};
        onerror = onerror || function(){};
        var indexedDB = FS.indexedDB();
        try {
          var openRequest = indexedDB.open(FS.DB_NAME(), FS.DB_VERSION);
        } catch (e) {
          return onerror(e);
        }
        openRequest.onupgradeneeded = function openRequest_onupgradeneeded() {
          console.log('creating db');
          var db = openRequest.result;
          db.createObjectStore(FS.DB_STORE_NAME);
        };
        openRequest.onsuccess = function openRequest_onsuccess() {
          var db = openRequest.result;
          var transaction = db.transaction([FS.DB_STORE_NAME], 'readwrite');
          var files = transaction.objectStore(FS.DB_STORE_NAME);
          var ok = 0, fail = 0, total = paths.length;
          function finish() {
            if (fail == 0) onload(); else onerror();
          }
          paths.forEach(function(path) {
            var putRequest = files.put(FS.analyzePath(path).object.contents, path);
            putRequest.onsuccess = function putRequest_onsuccess() { ok++; if (ok + fail == total) finish() };
            putRequest.onerror = function putRequest_onerror() { fail++; if (ok + fail == total) finish() };
          });
          transaction.onerror = onerror;
        };
        openRequest.onerror = onerror;
      },loadFilesFromDB:function (paths, onload, onerror) {
        onload = onload || function(){};
        onerror = onerror || function(){};
        var indexedDB = FS.indexedDB();
        try {
          var openRequest = indexedDB.open(FS.DB_NAME(), FS.DB_VERSION);
        } catch (e) {
          return onerror(e);
        }
        openRequest.onupgradeneeded = onerror; // no database to load from
        openRequest.onsuccess = function openRequest_onsuccess() {
          var db = openRequest.result;
          try {
            var transaction = db.transaction([FS.DB_STORE_NAME], 'readonly');
          } catch(e) {
            onerror(e);
            return;
          }
          var files = transaction.objectStore(FS.DB_STORE_NAME);
          var ok = 0, fail = 0, total = paths.length;
          function finish() {
            if (fail == 0) onload(); else onerror();
          }
          paths.forEach(function(path) {
            var getRequest = files.get(path);
            getRequest.onsuccess = function getRequest_onsuccess() {
              if (FS.analyzePath(path).exists) {
                FS.unlink(path);
              }
              FS.createDataFile(PATH.dirname(path), PATH.basename(path), getRequest.result, true, true, true);
              ok++;
              if (ok + fail == total) finish();
            };
            getRequest.onerror = function getRequest_onerror() { fail++; if (ok + fail == total) finish() };
          });
          transaction.onerror = onerror;
        };
        openRequest.onerror = onerror;
      }};var SYSCALLS={DEFAULT_POLLMASK:5,mappings:{},umask:511,calculateAt:function (dirfd, path) {
        if (path[0] !== '/') {
          // relative path
          var dir;
          if (dirfd === -100) {
            dir = FS.cwd();
          } else {
            var dirstream = FS.getStream(dirfd);
            if (!dirstream) throw new FS.ErrnoError(ERRNO_CODES.EBADF);
            dir = dirstream.path;
          }
          path = PATH.join2(dir, path);
        }
        return path;
      },doStat:function (func, path, buf) {
        try {
          var stat = func(path);
        } catch (e) {
          if (e && e.node && PATH.normalize(path) !== PATH.normalize(FS.getPath(e.node))) {
            // an error occurred while trying to look up the path; we should just report ENOTDIR
            return -ERRNO_CODES.ENOTDIR;
          }
          throw e;
        }
        HEAP32[((buf)>>2)]=stat.dev;
        HEAP32[(((buf)+(4))>>2)]=0;
        HEAP32[(((buf)+(8))>>2)]=stat.ino;
        HEAP32[(((buf)+(12))>>2)]=stat.mode;
        HEAP32[(((buf)+(16))>>2)]=stat.nlink;
        HEAP32[(((buf)+(20))>>2)]=stat.uid;
        HEAP32[(((buf)+(24))>>2)]=stat.gid;
        HEAP32[(((buf)+(28))>>2)]=stat.rdev;
        HEAP32[(((buf)+(32))>>2)]=0;
        HEAP32[(((buf)+(36))>>2)]=stat.size;
        HEAP32[(((buf)+(40))>>2)]=4096;
        HEAP32[(((buf)+(44))>>2)]=stat.blocks;
        HEAP32[(((buf)+(48))>>2)]=(stat.atime.getTime() / 1000)|0;
        HEAP32[(((buf)+(52))>>2)]=0;
        HEAP32[(((buf)+(56))>>2)]=(stat.mtime.getTime() / 1000)|0;
        HEAP32[(((buf)+(60))>>2)]=0;
        HEAP32[(((buf)+(64))>>2)]=(stat.ctime.getTime() / 1000)|0;
        HEAP32[(((buf)+(68))>>2)]=0;
        HEAP32[(((buf)+(72))>>2)]=stat.ino;
        return 0;
      },doMsync:function (addr, stream, len, flags) {
        var buffer = new Uint8Array(HEAPU8.subarray(addr, addr + len));
        FS.msync(stream, buffer, 0, len, flags);
      },doMkdir:function (path, mode) {
        // remove a trailing slash, if one - /a/b/ has basename of '', but
        // we want to create b in the context of this function
        path = PATH.normalize(path);
        if (path[path.length-1] === '/') path = path.substr(0, path.length-1);
        FS.mkdir(path, mode, 0);
        return 0;
      },doMknod:function (path, mode, dev) {
        // we don't want this in the JS API as it uses mknod to create all nodes.
        switch (mode & 61440) {
          case 32768:
          case 8192:
          case 24576:
          case 4096:
          case 49152:
            break;
          default: return -ERRNO_CODES.EINVAL;
        }
        FS.mknod(path, mode, dev);
        return 0;
      },doReadlink:function (path, buf, bufsize) {
        if (bufsize <= 0) return -ERRNO_CODES.EINVAL;
        var ret = FS.readlink(path);

        var len = Math.min(bufsize, lengthBytesUTF8(ret));
        var endChar = HEAP8[buf+len];
        stringToUTF8(ret, buf, bufsize+1);
        // readlink is one of the rare functions that write out a C string, but does never append a null to the output buffer(!)
        // stringToUTF8() always appends a null byte, so restore the character under the null byte after the write.
        HEAP8[buf+len] = endChar;

        return len;
      },doAccess:function (path, amode) {
        if (amode & ~7) {
          // need a valid mode
          return -ERRNO_CODES.EINVAL;
        }
        var node;
        var lookup = FS.lookupPath(path, { follow: true });
        node = lookup.node;
        var perms = '';
        if (amode & 4) perms += 'r';
        if (amode & 2) perms += 'w';
        if (amode & 1) perms += 'x';
        if (perms /* otherwise, they've just passed F_OK */ && FS.nodePermissions(node, perms)) {
          return -ERRNO_CODES.EACCES;
        }
        return 0;
      },doDup:function (path, flags, suggestFD) {
        var suggest = FS.getStream(suggestFD);
        if (suggest) FS.close(suggest);
        return FS.open(path, flags, 0, suggestFD, suggestFD).fd;
      },doReadv:function (stream, iov, iovcnt, offset) {
        var ret = 0;
        for (var i = 0; i < iovcnt; i++) {
          var ptr = HEAP32[(((iov)+(i*8))>>2)];
          var len = HEAP32[(((iov)+(i*8 + 4))>>2)];
          var curr = FS.read(stream, HEAP8,ptr, len, offset);
          if (curr < 0) return -1;
          ret += curr;
          if (curr < len) break; // nothing more to read
        }
        return ret;
      },doWritev:function (stream, iov, iovcnt, offset) {
        var ret = 0;
        for (var i = 0; i < iovcnt; i++) {
          var ptr = HEAP32[(((iov)+(i*8))>>2)];
          var len = HEAP32[(((iov)+(i*8 + 4))>>2)];
          var curr = FS.write(stream, HEAP8,ptr, len, offset);
          if (curr < 0) return -1;
          ret += curr;
        }
        return ret;
      },varargs:0,get:function (varargs) {
        SYSCALLS.varargs += 4;
        var ret = HEAP32[(((SYSCALLS.varargs)-(4))>>2)];
        return ret;
      },getStr:function () {
        var ret = Pointer_stringify(SYSCALLS.get());
        return ret;
      },getStreamFromFD:function () {
        var stream = FS.getStream(SYSCALLS.get());
        if (!stream) throw new FS.ErrnoError(ERRNO_CODES.EBADF);
        return stream;
      },getSocketFromFD:function () {
        var socket = SOCKFS.getSocket(SYSCALLS.get());
        if (!socket) throw new FS.ErrnoError(ERRNO_CODES.EBADF);
        return socket;
      },getSocketAddress:function (allowNull) {
        var addrp = SYSCALLS.get(), addrlen = SYSCALLS.get();
        if (allowNull && addrp === 0) return null;
        var info = __read_sockaddr(addrp, addrlen);
        if (info.errno) throw new FS.ErrnoError(info.errno);
        info.addr = DNS.lookup_addr(info.addr) || info.addr;
        return info;
      },get64:function () {
        var low = SYSCALLS.get(), high = SYSCALLS.get();
        if (low >= 0) assert(high === 0);
        else assert(high === -1);
        return low;
      },getZero:function () {
        assert(SYSCALLS.get() === 0);
      }};function ___syscall195(which, varargs) {SYSCALLS.varargs = varargs;
  try {
   // SYS_stat64
      var path = SYSCALLS.getStr(), buf = SYSCALLS.get();
      return SYSCALLS.doStat(FS.stat, path, buf);
    } catch (e) {
    if (typeof FS === 'undefined' || !(e instanceof FS.ErrnoError)) abort(e);
    return -e.errno;
  }
  }

  function ___syscall194(which, varargs) {SYSCALLS.varargs = varargs;
  try {
   // ftruncate64
      var fd = SYSCALLS.get(), zero = SYSCALLS.getZero(), length = SYSCALLS.get64();
      FS.ftruncate(fd, length);
      return 0;
    } catch (e) {
    if (typeof FS === 'undefined' || !(e instanceof FS.ErrnoError)) abort(e);
    return -e.errno;
  }
  }

  function ___syscall197(which, varargs) {SYSCALLS.varargs = varargs;
  try {
   // SYS_fstat64
      var stream = SYSCALLS.getStreamFromFD(), buf = SYSCALLS.get();
      return SYSCALLS.doStat(FS.stat, stream.path, buf);
    } catch (e) {
    if (typeof FS === 'undefined' || !(e instanceof FS.ErrnoError)) abort(e);
    return -e.errno;
  }
  }


  var GL={counter:1,lastError:0,buffers:[],mappedBuffers:{},programs:[],framebuffers:[],renderbuffers:[],textures:[],uniforms:[],shaders:[],vaos:[],contexts:[],currentContext:null,offscreenCanvases:{},timerQueriesEXT:[],byteSizeByTypeRoot:5120,byteSizeByType:[1,1,2,2,4,4,4,2,3,4,8],programInfos:{},stringCache:{},tempFixedLengthArray:[],packAlignment:4,unpackAlignment:4,init:function () {
        GL.miniTempBuffer = new Float32Array(GL.MINI_TEMP_BUFFER_SIZE);
        for (var i = 0; i < GL.MINI_TEMP_BUFFER_SIZE; i++) {
          GL.miniTempBufferViews[i] = GL.miniTempBuffer.subarray(0, i+1);
        }

        // For functions such as glDrawBuffers, glInvalidateFramebuffer and glInvalidateSubFramebuffer that need to pass a short array to the WebGL API,
        // create a set of short fixed-length arrays to avoid having to generate any garbage when calling those functions.
        for (var i = 0; i < 32; i++) {
          GL.tempFixedLengthArray.push(new Array(i));
        }
      },recordError:function recordError(errorCode) {
        if (!GL.lastError) {
          GL.lastError = errorCode;
        }
      },getNewId:function (table) {
        var ret = GL.counter++;
        for (var i = table.length; i < ret; i++) {
          table[i] = null;
        }
        return ret;
      },MINI_TEMP_BUFFER_SIZE:256,miniTempBuffer:null,miniTempBufferViews:[0],getSource:function (shader, count, string, length) {
        var source = '';
        for (var i = 0; i < count; ++i) {
          var frag;
          if (length) {
            var len = HEAP32[(((length)+(i*4))>>2)];
            if (len < 0) {
              frag = Pointer_stringify(HEAP32[(((string)+(i*4))>>2)]);
            } else {
              frag = Pointer_stringify(HEAP32[(((string)+(i*4))>>2)], len);
            }
          } else {
            frag = Pointer_stringify(HEAP32[(((string)+(i*4))>>2)]);
          }
          source += frag;
        }
        return source;
      },createContext:function (canvas, webGLContextAttributes) {
        if (typeof webGLContextAttributes['majorVersion'] === 'undefined' && typeof webGLContextAttributes['minorVersion'] === 'undefined') {
          webGLContextAttributes['majorVersion'] = 1;
          webGLContextAttributes['minorVersion'] = 0;
        }


        var ctx;
        var errorInfo = '?';
        function onContextCreationError(event) {
          errorInfo = event.statusMessage || errorInfo;
        }
        try {
          canvas.addEventListener('webglcontextcreationerror', onContextCreationError, false);
          try {
            if (webGLContextAttributes['majorVersion'] == 1 && webGLContextAttributes['minorVersion'] == 0) {
              ctx = canvas.getContext("webgl", webGLContextAttributes) || canvas.getContext("experimental-webgl", webGLContextAttributes);
            } else if (webGLContextAttributes['majorVersion'] == 2 && webGLContextAttributes['minorVersion'] == 0) {
              ctx = canvas.getContext("webgl2", webGLContextAttributes);
            } else {
              throw 'Unsupported WebGL context version ' + majorVersion + '.' + minorVersion + '!'
            }
          } finally {
            canvas.removeEventListener('webglcontextcreationerror', onContextCreationError, false);
          }
          if (!ctx) throw ':(';
        } catch (e) {
          Module.print('Could not create canvas: ' + [errorInfo, e, JSON.stringify(webGLContextAttributes)]);
          return 0;
        }
        // possible GL_DEBUG entry point: ctx = wrapDebugGL(ctx);

        if (!ctx) return 0;
        return GL.registerContext(ctx, webGLContextAttributes);
      },registerContext:function (ctx, webGLContextAttributes) {
        var handle = GL.getNewId(GL.contexts);
        var context = {
          handle: handle,
          attributes: webGLContextAttributes,
          version: webGLContextAttributes['majorVersion'],
          GLctx: ctx
        };


        // Store the created context object so that we can access the context given a canvas without having to pass the parameters again.
        if (ctx.canvas) ctx.canvas.GLctxObject = context;
        GL.contexts[handle] = context;
        if (typeof webGLContextAttributes['enableExtensionsByDefault'] === 'undefined' || webGLContextAttributes['enableExtensionsByDefault']) {
          GL.initExtensions(context);
        }
        return handle;
      },makeContextCurrent:function (contextHandle) {
        var context = GL.contexts[contextHandle];
        if (!context) return false;
        GLctx = Module.ctx = context.GLctx; // Active WebGL context object.
        GL.currentContext = context; // Active Emscripten GL layer context object.
        return true;
      },getContext:function (contextHandle) {
        return GL.contexts[contextHandle];
      },deleteContext:function (contextHandle) {
        if (GL.currentContext === GL.contexts[contextHandle]) GL.currentContext = null;
        if (typeof JSEvents === 'object') JSEvents.removeAllHandlersOnTarget(GL.contexts[contextHandle].GLctx.canvas); // Release all JS event handlers on the DOM element that the GL context is associated with since the context is now deleted.
        if (GL.contexts[contextHandle] && GL.contexts[contextHandle].GLctx.canvas) GL.contexts[contextHandle].GLctx.canvas.GLctxObject = undefined; // Make sure the canvas object no longer refers to the context object so there are no GC surprises.
        GL.contexts[contextHandle] = null;
      },initExtensions:function (context) {
        // If this function is called without a specific context object, init the extensions of the currently active context.
        if (!context) context = GL.currentContext;

        if (context.initExtensionsDone) return;
        context.initExtensionsDone = true;

        var GLctx = context.GLctx;

        context.maxVertexAttribs = GLctx.getParameter(GLctx.MAX_VERTEX_ATTRIBS);

        // Detect the presence of a few extensions manually, this GL interop layer itself will need to know if they exist.

        if (context.version < 2) {
          // Extension available from Firefox 26 and Google Chrome 30
          var instancedArraysExt = GLctx.getExtension('ANGLE_instanced_arrays');
          if (instancedArraysExt) {
            GLctx['vertexAttribDivisor'] = function(index, divisor) { instancedArraysExt['vertexAttribDivisorANGLE'](index, divisor); };
            GLctx['drawArraysInstanced'] = function(mode, first, count, primcount) { instancedArraysExt['drawArraysInstancedANGLE'](mode, first, count, primcount); };
            GLctx['drawElementsInstanced'] = function(mode, count, type, indices, primcount) { instancedArraysExt['drawElementsInstancedANGLE'](mode, count, type, indices, primcount); };
          }

          // Extension available from Firefox 25 and WebKit
          var vaoExt = GLctx.getExtension('OES_vertex_array_object');
          if (vaoExt) {
            GLctx['createVertexArray'] = function() { return vaoExt['createVertexArrayOES'](); };
            GLctx['deleteVertexArray'] = function(vao) { vaoExt['deleteVertexArrayOES'](vao); };
            GLctx['bindVertexArray'] = function(vao) { vaoExt['bindVertexArrayOES'](vao); };
            GLctx['isVertexArray'] = function(vao) { return vaoExt['isVertexArrayOES'](vao); };
          }

          var drawBuffersExt = GLctx.getExtension('WEBGL_draw_buffers');
          if (drawBuffersExt) {
            GLctx['drawBuffers'] = function(n, bufs) { drawBuffersExt['drawBuffersWEBGL'](n, bufs); };
          }
        }

        GLctx.disjointTimerQueryExt = GLctx.getExtension("EXT_disjoint_timer_query");

        // These are the 'safe' feature-enabling extensions that don't add any performance impact related to e.g. debugging, and
        // should be enabled by default so that client GLES2/GL code will not need to go through extra hoops to get its stuff working.
        // As new extensions are ratified at http://www.khronos.org/registry/webgl/extensions/ , feel free to add your new extensions
        // here, as long as they don't produce a performance impact for users that might not be using those extensions.
        // E.g. debugging-related extensions should probably be off by default.
        var automaticallyEnabledExtensions = [ "OES_texture_float", "OES_texture_half_float", "OES_standard_derivatives",
                                               "OES_vertex_array_object", "WEBGL_compressed_texture_s3tc", "WEBGL_depth_texture",
                                               "OES_element_index_uint", "EXT_texture_filter_anisotropic", "ANGLE_instanced_arrays",
                                               "OES_texture_float_linear", "OES_texture_half_float_linear", "WEBGL_compressed_texture_atc",
                                               "WEBGL_compressed_texture_pvrtc", "EXT_color_buffer_half_float", "WEBGL_color_buffer_float",
                                               "EXT_frag_depth", "EXT_sRGB", "WEBGL_draw_buffers", "WEBGL_shared_resources",
                                               "EXT_shader_texture_lod", "EXT_color_buffer_float"];

        function shouldEnableAutomatically(extension) {
          var ret = false;
          automaticallyEnabledExtensions.forEach(function(include) {
            if (ext.indexOf(include) != -1) {
              ret = true;
            }
          });
          return ret;
        }

        var exts = GLctx.getSupportedExtensions();
        if (exts && exts.length > 0) {
          GLctx.getSupportedExtensions().forEach(function(ext) {
            if (automaticallyEnabledExtensions.indexOf(ext) != -1) {
              GLctx.getExtension(ext); // Calling .getExtension enables that extension permanently, no need to store the return value to be enabled.
            }
          });
        }
      },populateUniformTable:function (program) {
        var p = GL.programs[program];
        GL.programInfos[program] = {
          uniforms: {},
          maxUniformLength: 0, // This is eagerly computed below, since we already enumerate all uniforms anyway.
          maxAttributeLength: -1, // This is lazily computed and cached, computed when/if first asked, "-1" meaning not computed yet.
          maxUniformBlockNameLength: -1 // Lazily computed as well
        };

        var ptable = GL.programInfos[program];
        var utable = ptable.uniforms;
        // A program's uniform table maps the string name of an uniform to an integer location of that uniform.
        // The global GL.uniforms map maps integer locations to WebGLUniformLocations.
        var numUniforms = GLctx.getProgramParameter(p, GLctx.ACTIVE_UNIFORMS);
        for (var i = 0; i < numUniforms; ++i) {
          var u = GLctx.getActiveUniform(p, i);

          var name = u.name;
          ptable.maxUniformLength = Math.max(ptable.maxUniformLength, name.length+1);

          // Strip off any trailing array specifier we might have got, e.g. "[0]".
          if (name.indexOf(']', name.length-1) !== -1) {
            var ls = name.lastIndexOf('[');
            name = name.slice(0, ls);
          }

          // Optimize memory usage slightly: If we have an array of uniforms, e.g. 'vec3 colors[3];', then
          // only store the string 'colors' in utable, and 'colors[0]', 'colors[1]' and 'colors[2]' will be parsed as 'colors'+i.
          // Note that for the GL.uniforms table, we still need to fetch the all WebGLUniformLocations for all the indices.
          var loc = GLctx.getUniformLocation(p, name);
          if (loc != null)
          {
            var id = GL.getNewId(GL.uniforms);
            utable[name] = [u.size, id];
            GL.uniforms[id] = loc;

            for (var j = 1; j < u.size; ++j) {
              var n = name + '['+j+']';
              loc = GLctx.getUniformLocation(p, n);
              id = GL.getNewId(GL.uniforms);

              GL.uniforms[id] = loc;
            }
          }
        }
      }};




  function _emscripten_set_main_loop_timing(mode, value) {
      Browser.mainLoop.timingMode = mode;
      Browser.mainLoop.timingValue = value;

      if (!Browser.mainLoop.func) {
        return 1; // Return non-zero on failure, can't set timing mode when there is no main loop.
      }

      if (mode == 0 /*EM_TIMING_SETTIMEOUT*/) {
        Browser.mainLoop.scheduler = function Browser_mainLoop_scheduler_setTimeout() {
          var timeUntilNextTick = Math.max(0, Browser.mainLoop.tickStartTime + value - _emscripten_get_now())|0;
          setTimeout(Browser.mainLoop.runner, timeUntilNextTick); // doing this each time means that on exception, we stop
        };
        Browser.mainLoop.method = 'timeout';
      } else if (mode == 1 /*EM_TIMING_RAF*/) {
        Browser.mainLoop.scheduler = function Browser_mainLoop_scheduler_rAF() {
          Browser.requestAnimationFrame(Browser.mainLoop.runner);
        };
        Browser.mainLoop.method = 'rAF';
      } else if (mode == 2 /*EM_TIMING_SETIMMEDIATE*/) {
        if (!window['setImmediate']) {
          // Emulate setImmediate. (note: not a complete polyfill, we don't emulate clearImmediate() to keep code size to minimum, since not needed)
          var setImmediates = [];
          var emscriptenMainLoopMessageId = 'setimmediate';
          function Browser_setImmediate_messageHandler(event) {
            if (event.source === window && event.data === emscriptenMainLoopMessageId) {
              event.stopPropagation();
              setImmediates.shift()();
            }
          }
          window.addEventListener("message", Browser_setImmediate_messageHandler, true);
          window['setImmediate'] = function Browser_emulated_setImmediate(func) {
            setImmediates.push(func);
            if (ENVIRONMENT_IS_WORKER) {
              if (Module['setImmediates'] === undefined) Module['setImmediates'] = [];
              Module['setImmediates'].push(func);
              window.postMessage({target: emscriptenMainLoopMessageId}); // In --proxy-to-worker, route the message via proxyClient.js
            } else window.postMessage(emscriptenMainLoopMessageId, "*"); // On the main thread, can just send the message to itself.
          }
        }
        Browser.mainLoop.scheduler = function Browser_mainLoop_scheduler_setImmediate() {
          window['setImmediate'](Browser.mainLoop.runner);
        };
        Browser.mainLoop.method = 'immediate';
      }
      return 0;
    }

  function _emscripten_get_now() { abort() }

  function _emscripten_set_main_loop(func, fps, simulateInfiniteLoop, arg, noSetTiming) {
      Module['noExitRuntime'] = true;

      Module.print('Hiii')

      assert(!Browser.mainLoop.func, 'emscripten_set_main_loop: there can only be one main loop function at once: call emscripten_cancel_main_loop to cancel the previous one before setting a new one with different parameters.');

      Browser.mainLoop.func = func;
      Browser.mainLoop.arg = arg;

      var browserIterationFunc;
      if (typeof arg !== 'undefined') {
        browserIterationFunc = function() {
          Module['dynCall_vi'](func, arg);
        };
      } else {
        browserIterationFunc = function() {
          Module['dynCall_v'](func);
        };
      }

      var thisMainLoopId = Browser.mainLoop.currentlyRunningMainloop;

      Browser.mainLoop.runner = function Browser_mainLoop_runner() {
        Module.print("runner");

        if (ABORT) return;
        if (Browser.mainLoop.queue.length > 0) {
          var start = Date.now();
          var blocker = Browser.mainLoop.queue.shift();
          blocker.func(blocker.arg);
          if (Browser.mainLoop.remainingBlockers) {
            var remaining = Browser.mainLoop.remainingBlockers;
            var next = remaining%1 == 0 ? remaining-1 : Math.floor(remaining);
            if (blocker.counted) {
              Browser.mainLoop.remainingBlockers = next;
            } else {
              // not counted, but move the progress along a tiny bit
              next = next + 0.5; // do not steal all the next one's progress
              Browser.mainLoop.remainingBlockers = (8*remaining + next)/9;
            }
          }
          console.log('main loop blocker "' + blocker.name + '" took ' + (Date.now() - start) + ' ms'); //, left: ' + Browser.mainLoop.remainingBlockers);
          Browser.mainLoop.updateStatus();

          // catches pause/resume main loop from blocker execution
          if (thisMainLoopId < Browser.mainLoop.currentlyRunningMainloop) return;

          setTimeout(Browser.mainLoop.runner, 0);
          return;
        }

        // catch pauses from non-main loop sources
        if (thisMainLoopId < Browser.mainLoop.currentlyRunningMainloop) return;

        // Implement very basic swap interval control
        Browser.mainLoop.currentFrameNumber = Browser.mainLoop.currentFrameNumber + 1 | 0;
        if (Browser.mainLoop.timingMode == 1/*EM_TIMING_RAF*/ && Browser.mainLoop.timingValue > 1 && Browser.mainLoop.currentFrameNumber % Browser.mainLoop.timingValue != 0) {
          // Not the scheduled time to render this frame - skip.
          Browser.mainLoop.scheduler();
          return;
        } else if (Browser.mainLoop.timingMode == 0/*EM_TIMING_SETTIMEOUT*/) {
          Browser.mainLoop.tickStartTime = _emscripten_get_now();
        }

        // Signal GL rendering layer that processing of a new frame is about to start. This helps it optimize
        // VBO double-buffering and reduce GPU stalls.


        if (Browser.mainLoop.method === 'timeout' && Module.ctx) {
          Module.printErr('Looks like you are rendering without using requestAnimationFrame for the main loop. You should use 0 for the frame rate in emscripten_set_main_loop in order to use requestAnimationFrame, as that can greatly improve your frame rates!');
          Browser.mainLoop.method = ''; // just warn once per call to set main loop
        }

        Browser.mainLoop.runIter(browserIterationFunc);


        // catch pauses from the main loop itself
        if (thisMainLoopId < Browser.mainLoop.currentlyRunningMainloop) return;

        // Queue new audio data. This is important to be right after the main loop invocation, so that we will immediately be able
        // to queue the newest produced audio samples.
        // TODO: Consider adding pre- and post- rAF callbacks so that GL.newRenderingFrameStarted() and SDL.audio.queueNewAudioData()
        //       do not need to be hardcoded into this function, but can be more generic.
        if (typeof SDL === 'object' && SDL.audio && SDL.audio.queueNewAudioData) SDL.audio.queueNewAudioData();

        Browser.mainLoop.scheduler();
      }

      if (!noSetTiming) {
        if (fps && fps > 0) _emscripten_set_main_loop_timing(0/*EM_TIMING_SETTIMEOUT*/, 1000.0 / fps);
        else _emscripten_set_main_loop_timing(1/*EM_TIMING_RAF*/, 1); // Do rAF by rendering each frame (no decimating)

        Browser.mainLoop.scheduler();
      }

      if (simulateInfiniteLoop) {
        throw 'SimulateInfiniteLoop';
      }
    }var Browser={mainLoop:{scheduler:null,method:"",currentlyRunningMainloop:0,func:null,arg:0,timingMode:0,timingValue:0,currentFrameNumber:0,queue:[],pause:function () {
          Browser.mainLoop.scheduler = null;
          Browser.mainLoop.currentlyRunningMainloop++; // Incrementing this signals the previous main loop that it's now become old, and it must return.
        },resume:function () {
          Browser.mainLoop.currentlyRunningMainloop++;
          var timingMode = Browser.mainLoop.timingMode;
          var timingValue = Browser.mainLoop.timingValue;
          var func = Browser.mainLoop.func;
          Browser.mainLoop.func = null;
          _emscripten_set_main_loop(func, 0, false, Browser.mainLoop.arg, true /* do not set timing and call scheduler, we will do it on the next lines */);
          _emscripten_set_main_loop_timing(timingMode, timingValue);
          Browser.mainLoop.scheduler();
        },updateStatus:function () {
          if (Module['setStatus']) {
            var message = Module['statusMessage'] || 'Please wait...';
            var remaining = Browser.mainLoop.remainingBlockers;
            var expected = Browser.mainLoop.expectedBlockers;
            if (remaining) {
              if (remaining < expected) {
                Module['setStatus'](message + ' (' + (expected - remaining) + '/' + expected + ')');
              } else {
                Module['setStatus'](message);
              }
            } else {
              Module['setStatus']('');
            }
          }
        },runIter:function (func) {
          if (ABORT) return;
          if (Module['preMainLoop']) {
            var preRet = Module['preMainLoop']();
            if (preRet === false) {
              return; // |return false| skips a frame
            }
          }
          try {
            func();
          } catch (e) {
            if (e instanceof ExitStatus) {
              return;
            } else {
              if (e && typeof e === 'object' && e.stack) Module.printErr('exception thrown: ' + [e, e.stack]);
              throw e;
            }
          }
          if (Module['postMainLoop']) Module['postMainLoop']();
        }},isFullscreen:false,pointerLock:false,moduleContextCreatedCallbacks:[],workers:[],init:function () {
        if (!Module["preloadPlugins"]) Module["preloadPlugins"] = []; // needs to exist even in workers

        if (Browser.initted) return;
        Browser.initted = true;

        try {
          new Blob();
          Browser.hasBlobConstructor = true;
        } catch(e) {
          Browser.hasBlobConstructor = false;
          console.log("warning: no blob constructor, cannot create blobs with mimetypes");
        }
        Browser.BlobBuilder = typeof MozBlobBuilder != "undefined" ? MozBlobBuilder : (typeof WebKitBlobBuilder != "undefined" ? WebKitBlobBuilder : (!Browser.hasBlobConstructor ? console.log("warning: no BlobBuilder") : null));
        Browser.URLObject = typeof window != "undefined" ? (window.URL ? window.URL : window.webkitURL) : undefined;
        if (!Module.noImageDecoding && typeof Browser.URLObject === 'undefined') {
          console.log("warning: Browser does not support creating object URLs. Built-in browser image decoding will not be available.");
          Module.noImageDecoding = true;
        }

        // Support for plugins that can process preloaded files. You can add more of these to
        // your app by creating and appending to Module.preloadPlugins.
        //
        // Each plugin is asked if it can handle a file based on the file's name. If it can,
        // it is given the file's raw data. When it is done, it calls a callback with the file's
        // (possibly modified) data. For example, a plugin might decompress a file, or it
        // might create some side data structure for use later (like an Image element, etc.).

        var imagePlugin = {};
        imagePlugin['canHandle'] = function imagePlugin_canHandle(name) {
          return !Module.noImageDecoding && /\.(jpg|jpeg|png|bmp)$/i.test(name);
        };
        imagePlugin['handle'] = function imagePlugin_handle(byteArray, name, onload, onerror) {
          var b = null;
          if (Browser.hasBlobConstructor) {
            try {
              b = new Blob([byteArray], { type: Browser.getMimetype(name) });
              if (b.size !== byteArray.length) { // Safari bug #118630
                // Safari's Blob can only take an ArrayBuffer
                b = new Blob([(new Uint8Array(byteArray)).buffer], { type: Browser.getMimetype(name) });
              }
            } catch(e) {
              Runtime.warnOnce('Blob constructor present but fails: ' + e + '; falling back to blob builder');
            }
          }
          if (!b) {
            var bb = new Browser.BlobBuilder();
            bb.append((new Uint8Array(byteArray)).buffer); // we need to pass a buffer, and must copy the array to get the right data range
            b = bb.getBlob();
          }
          var url = Browser.URLObject.createObjectURL(b);
          var img = new Image();
          img.onload = function img_onload() {
            assert(img.complete, 'Image ' + name + ' could not be decoded');
            var canvas = document.createElement('canvas');
            canvas.width = img.width;
            canvas.height = img.height;
            var ctx = canvas.getContext('2d');
            ctx.drawImage(img, 0, 0);
            Module["preloadedImages"][name] = canvas;
            Browser.URLObject.revokeObjectURL(url);
            if (onload) onload(byteArray);
          };
          img.onerror = function img_onerror(event) {
            console.log('Image ' + url + ' could not be decoded');
            if (onerror) onerror();
          };
          img.src = url;
        };
        Module['preloadPlugins'].push(imagePlugin);

        var audioPlugin = {};
        audioPlugin['canHandle'] = function audioPlugin_canHandle(name) {
          return !Module.noAudioDecoding && name.substr(-4) in { '.ogg': 1, '.wav': 1, '.mp3': 1 };
        };
        audioPlugin['handle'] = function audioPlugin_handle(byteArray, name, onload, onerror) {
          var done = false;
          function finish(audio) {
            if (done) return;
            done = true;
            Module["preloadedAudios"][name] = audio;
            if (onload) onload(byteArray);
          }
          function fail() {
            if (done) return;
            done = true;
            Module["preloadedAudios"][name] = new Audio(); // empty shim
            if (onerror) onerror();
          }
          if (Browser.hasBlobConstructor) {
            try {
              var b = new Blob([byteArray], { type: Browser.getMimetype(name) });
            } catch(e) {
              return fail();
            }
            var url = Browser.URLObject.createObjectURL(b); // XXX we never revoke this!
            var audio = new Audio();
            audio.addEventListener('canplaythrough', function() { finish(audio) }, false); // use addEventListener due to chromium bug 124926
            audio.onerror = function audio_onerror(event) {
              if (done) return;
              console.log('warning: browser could not fully decode audio ' + name + ', trying slower base64 approach');
              function encode64(data) {
                var BASE = 'ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/';
                var PAD = '=';
                var ret = '';
                var leftchar = 0;
                var leftbits = 0;
                for (var i = 0; i < data.length; i++) {
                  leftchar = (leftchar << 8) | data[i];
                  leftbits += 8;
                  while (leftbits >= 6) {
                    var curr = (leftchar >> (leftbits-6)) & 0x3f;
                    leftbits -= 6;
                    ret += BASE[curr];
                  }
                }
                if (leftbits == 2) {
                  ret += BASE[(leftchar&3) << 4];
                  ret += PAD + PAD;
                } else if (leftbits == 4) {
                  ret += BASE[(leftchar&0xf) << 2];
                  ret += PAD;
                }
                return ret;
              }
              audio.src = 'data:audio/x-' + name.substr(-3) + ';base64,' + encode64(byteArray);
              finish(audio); // we don't wait for confirmation this worked - but it's worth trying
            };
            audio.src = url;
            // workaround for chrome bug 124926 - we do not always get oncanplaythrough or onerror
            Browser.safeSetTimeout(function() {
              finish(audio); // try to use it even though it is not necessarily ready to play
            }, 10000);
          } else {
            return fail();
          }
        };
        Module['preloadPlugins'].push(audioPlugin);

        // Canvas event setup

        function pointerLockChange() {
          Browser.pointerLock = document['pointerLockElement'] === Module['canvas'] ||
                                document['mozPointerLockElement'] === Module['canvas'] ||
                                document['webkitPointerLockElement'] === Module['canvas'] ||
                                document['msPointerLockElement'] === Module['canvas'];
        }
        var canvas = Module['canvas'];
        if (canvas) {
          // forced aspect ratio can be enabled by defining 'forcedAspectRatio' on Module
          // Module['forcedAspectRatio'] = 4 / 3;

          canvas.requestPointerLock = canvas['requestPointerLock'] ||
                                      canvas['mozRequestPointerLock'] ||
                                      canvas['webkitRequestPointerLock'] ||
                                      canvas['msRequestPointerLock'] ||
                                      function(){};
          canvas.exitPointerLock = document['exitPointerLock'] ||
                                   document['mozExitPointerLock'] ||
                                   document['webkitExitPointerLock'] ||
                                   document['msExitPointerLock'] ||
                                   function(){}; // no-op if function does not exist
          canvas.exitPointerLock = canvas.exitPointerLock.bind(document);

          document.addEventListener('pointerlockchange', pointerLockChange, false);
          document.addEventListener('mozpointerlockchange', pointerLockChange, false);
          document.addEventListener('webkitpointerlockchange', pointerLockChange, false);
          document.addEventListener('mspointerlockchange', pointerLockChange, false);

          if (Module['elementPointerLock']) {
            canvas.addEventListener("click", function(ev) {
              if (!Browser.pointerLock && Module['canvas'].requestPointerLock) {
                Module['canvas'].requestPointerLock();
                ev.preventDefault();
              }
            }, false);
          }
        }
      },createContext:function (canvas, useWebGL, setInModule, webGLContextAttributes) {
        if (useWebGL && Module.ctx && canvas == Module.canvas) return Module.ctx; // no need to recreate GL context if it's already been created for this canvas.

        var ctx;
        var contextHandle;
        if (useWebGL) {
          // For GLES2/desktop GL compatibility, adjust a few defaults to be different to WebGL defaults, so that they align better with the desktop defaults.
          var contextAttributes = {
            antialias: false,
            alpha: false
          };

          if (webGLContextAttributes) {
            for (var attribute in webGLContextAttributes) {
              contextAttributes[attribute] = webGLContextAttributes[attribute];
            }
          }

          contextHandle = GL.createContext(canvas, contextAttributes);
          if (contextHandle) {
            ctx = GL.getContext(contextHandle).GLctx;
          }
        } else {
          ctx = canvas.getContext('2d');
        }

        if (!ctx) return null;

        if (setInModule) {
          if (!useWebGL) assert(typeof GLctx === 'undefined', 'cannot set in module if GLctx is used, but we are a non-GL context that would replace it');

          Module.ctx = ctx;
          if (useWebGL) GL.makeContextCurrent(contextHandle);
          Module.useWebGL = useWebGL;
          Browser.moduleContextCreatedCallbacks.forEach(function(callback) { callback() });
          Browser.init();
        }
        return ctx;
      },destroyContext:function (canvas, useWebGL, setInModule) {},fullscreenHandlersInstalled:false,lockPointer:undefined,resizeCanvas:undefined,requestFullscreen:function (lockPointer, resizeCanvas, vrDevice) {
        Browser.lockPointer = lockPointer;
        Browser.resizeCanvas = resizeCanvas;
        Browser.vrDevice = vrDevice;
        if (typeof Browser.lockPointer === 'undefined') Browser.lockPointer = true;
        if (typeof Browser.resizeCanvas === 'undefined') Browser.resizeCanvas = false;
        if (typeof Browser.vrDevice === 'undefined') Browser.vrDevice = null;

        var canvas = Module['canvas'];
        function fullscreenChange() {
          Browser.isFullscreen = false;
          var canvasContainer = canvas.parentNode;
          if ((document['fullscreenElement'] || document['mozFullScreenElement'] ||
               document['msFullscreenElement'] || document['webkitFullscreenElement'] ||
               document['webkitCurrentFullScreenElement']) === canvasContainer) {
            canvas.exitFullscreen = document['exitFullscreen'] ||
                                    document['cancelFullScreen'] ||
                                    document['mozCancelFullScreen'] ||
                                    document['msExitFullscreen'] ||
                                    document['webkitCancelFullScreen'] ||
                                    function() {};
            canvas.exitFullscreen = canvas.exitFullscreen.bind(document);
            if (Browser.lockPointer) canvas.requestPointerLock();
            Browser.isFullscreen = true;
            if (Browser.resizeCanvas) Browser.setFullscreenCanvasSize();
          } else {

            // remove the full screen specific parent of the canvas again to restore the HTML structure from before going full screen
            canvasContainer.parentNode.insertBefore(canvas, canvasContainer);
            canvasContainer.parentNode.removeChild(canvasContainer);

            if (Browser.resizeCanvas) Browser.setWindowedCanvasSize();
          }
          if (Module['onFullScreen']) Module['onFullScreen'](Browser.isFullscreen);
          if (Module['onFullscreen']) Module['onFullscreen'](Browser.isFullscreen);
          Browser.updateCanvasDimensions(canvas);
        }

        if (!Browser.fullscreenHandlersInstalled) {
          Browser.fullscreenHandlersInstalled = true;
          document.addEventListener('fullscreenchange', fullscreenChange, false);
          document.addEventListener('mozfullscreenchange', fullscreenChange, false);
          document.addEventListener('webkitfullscreenchange', fullscreenChange, false);
          document.addEventListener('MSFullscreenChange', fullscreenChange, false);
        }

        // create a new parent to ensure the canvas has no siblings. this allows browsers to optimize full screen performance when its parent is the full screen root
        var canvasContainer = document.createElement("div");
        canvas.parentNode.insertBefore(canvasContainer, canvas);
        canvasContainer.appendChild(canvas);

        // use parent of canvas as full screen root to allow aspect ratio correction (Firefox stretches the root to screen size)
        canvasContainer.requestFullscreen = canvasContainer['requestFullscreen'] ||
                                            canvasContainer['mozRequestFullScreen'] ||
                                            canvasContainer['msRequestFullscreen'] ||
                                           (canvasContainer['webkitRequestFullscreen'] ? function() { canvasContainer['webkitRequestFullscreen'](Element['ALLOW_KEYBOARD_INPUT']) } : null) ||
                                           (canvasContainer['webkitRequestFullScreen'] ? function() { canvasContainer['webkitRequestFullScreen'](Element['ALLOW_KEYBOARD_INPUT']) } : null);

        if (vrDevice) {
          canvasContainer.requestFullscreen({ vrDisplay: vrDevice });
        } else {
          canvasContainer.requestFullscreen();
        }
      },requestFullScreen:function (lockPointer, resizeCanvas, vrDevice) {
          Module.printErr('Browser.requestFullScreen() is deprecated. Please call Browser.requestFullscreen instead.');
          Browser.requestFullScreen = function(lockPointer, resizeCanvas, vrDevice) {
            return Browser.requestFullscreen(lockPointer, resizeCanvas, vrDevice);
          }
          return Browser.requestFullscreen(lockPointer, resizeCanvas, vrDevice);
      },nextRAF:0,fakeRequestAnimationFrame:function (func) {
        // try to keep 60fps between calls to here
        var now = Date.now();
        if (Browser.nextRAF === 0) {
          Browser.nextRAF = now + 1000/60;
        } else {
          while (now + 2 >= Browser.nextRAF) { // fudge a little, to avoid timer jitter causing us to do lots of delay:0
            Browser.nextRAF += 1000/60;
          }
        }
        var delay = Math.max(Browser.nextRAF - now, 0);
        setTimeout(func, delay);
      },requestAnimationFrame:function requestAnimationFrame(func) {
        if (typeof window === 'undefined') { // Provide fallback to setTimeout if window is undefined (e.g. in Node.js)
          Browser.fakeRequestAnimationFrame(func);
        } else {
          if (!window.requestAnimationFrame) {
            window.requestAnimationFrame = window['requestAnimationFrame'] ||
                                           window['mozRequestAnimationFrame'] ||
                                           window['webkitRequestAnimationFrame'] ||
                                           window['msRequestAnimationFrame'] ||
                                           window['oRequestAnimationFrame'] ||
                                           Browser.fakeRequestAnimationFrame;
          }
          window.requestAnimationFrame(func);
        }
      },safeCallback:function (func) {
        return function() {
          if (!ABORT) return func.apply(null, arguments);
        };
      },allowAsyncCallbacks:true,queuedAsyncCallbacks:[],pauseAsyncCallbacks:function () {
        Browser.allowAsyncCallbacks = false;
      },resumeAsyncCallbacks:function () { // marks future callbacks as ok to execute, and synchronously runs any remaining ones right now
        Browser.allowAsyncCallbacks = true;
        if (Browser.queuedAsyncCallbacks.length > 0) {
          var callbacks = Browser.queuedAsyncCallbacks;
          Browser.queuedAsyncCallbacks = [];
          callbacks.forEach(function(func) {
            func();
          });
        }
      },safeRequestAnimationFrame:function (func) {
        return Browser.requestAnimationFrame(function() {
          if (ABORT) return;
          if (Browser.allowAsyncCallbacks) {
            func();
          } else {
            Browser.queuedAsyncCallbacks.push(func);
          }
        });
      },safeSetTimeout:function (func, timeout) {
        Module['noExitRuntime'] = true;
        return setTimeout(function() {
          if (ABORT) return;
          if (Browser.allowAsyncCallbacks) {
            func();
          } else {
            Browser.queuedAsyncCallbacks.push(func);
          }
        }, timeout);
      },safeSetInterval:function (func, timeout) {
        Module['noExitRuntime'] = true;
        return setInterval(function() {
          if (ABORT) return;
          if (Browser.allowAsyncCallbacks) {
            func();
          } // drop it on the floor otherwise, next interval will kick in
        }, timeout);
      },getMimetype:function (name) {
        return {
          'jpg': 'image/jpeg',
          'jpeg': 'image/jpeg',
          'png': 'image/png',
          'bmp': 'image/bmp',
          'ogg': 'audio/ogg',
          'wav': 'audio/wav',
          'mp3': 'audio/mpeg'
        }[name.substr(name.lastIndexOf('.')+1)];
      },getUserMedia:function (func) {
        if(!window.getUserMedia) {
          window.getUserMedia = navigator['getUserMedia'] ||
                                navigator['mozGetUserMedia'];
        }
        window.getUserMedia(func);
      },getMovementX:function (event) {
        return event['movementX'] ||
               event['mozMovementX'] ||
               event['webkitMovementX'] ||
               0;
      },getMovementY:function (event) {
        return event['movementY'] ||
               event['mozMovementY'] ||
               event['webkitMovementY'] ||
               0;
      },getMouseWheelDelta:function (event) {
        var delta = 0;
        switch (event.type) {
          case 'DOMMouseScroll':
            delta = event.detail;
            break;
          case 'mousewheel':
            delta = event.wheelDelta;
            break;
          case 'wheel':
            delta = event['deltaY'];
            break;
          default:
            throw 'unrecognized mouse wheel event: ' + event.type;
        }
        return delta;
      },mouseX:0,mouseY:0,mouseMovementX:0,mouseMovementY:0,touches:{},lastTouches:{},calculateMouseEvent:function (event) { // event should be mousemove, mousedown or mouseup
        if (Browser.pointerLock) {
          // When the pointer is locked, calculate the coordinates
          // based on the movement of the mouse.
          // Workaround for Firefox bug 764498
          if (event.type != 'mousemove' &&
              ('mozMovementX' in event)) {
            Browser.mouseMovementX = Browser.mouseMovementY = 0;
          } else {
            Browser.mouseMovementX = Browser.getMovementX(event);
            Browser.mouseMovementY = Browser.getMovementY(event);
          }

          // check if SDL is available
          if (typeof SDL != "undefined") {
          	Browser.mouseX = SDL.mouseX + Browser.mouseMovementX;
          	Browser.mouseY = SDL.mouseY + Browser.mouseMovementY;
          } else {
          	// just add the mouse delta to the current absolut mouse position
          	// FIXME: ideally this should be clamped against the canvas size and zero
          	Browser.mouseX += Browser.mouseMovementX;
          	Browser.mouseY += Browser.mouseMovementY;
          }
        } else {
          // Otherwise, calculate the movement based on the changes
          // in the coordinates.
          var rect = Module["canvas"].getBoundingClientRect();
          var cw = Module["canvas"].width;
          var ch = Module["canvas"].height;

          // Neither .scrollX or .pageXOffset are defined in a spec, but
          // we prefer .scrollX because it is currently in a spec draft.
          // (see: http://www.w3.org/TR/2013/WD-cssom-view-20131217/)
          var scrollX = ((typeof window.scrollX !== 'undefined') ? window.scrollX : window.pageXOffset);
          var scrollY = ((typeof window.scrollY !== 'undefined') ? window.scrollY : window.pageYOffset);

          if (event.type === 'touchstart' || event.type === 'touchend' || event.type === 'touchmove') {
            var touch = event.touch;
            if (touch === undefined) {
              return; // the "touch" property is only defined in SDL

            }
            var adjustedX = touch.pageX - (scrollX + rect.left);
            var adjustedY = touch.pageY - (scrollY + rect.top);

            adjustedX = adjustedX * (cw / rect.width);
            adjustedY = adjustedY * (ch / rect.height);

            var coords = { x: adjustedX, y: adjustedY };

            if (event.type === 'touchstart') {
              Browser.lastTouches[touch.identifier] = coords;
              Browser.touches[touch.identifier] = coords;
            } else if (event.type === 'touchend' || event.type === 'touchmove') {
              var last = Browser.touches[touch.identifier];
              if (!last) last = coords;
              Browser.lastTouches[touch.identifier] = last;
              Browser.touches[touch.identifier] = coords;
            }
            return;
          }

          var x = event.pageX - (scrollX + rect.left);
          var y = event.pageY - (scrollY + rect.top);

          // the canvas might be CSS-scaled compared to its backbuffer;
          // SDL-using content will want mouse coordinates in terms
          // of backbuffer units.
          x = x * (cw / rect.width);
          y = y * (ch / rect.height);

          Browser.mouseMovementX = x - Browser.mouseX;
          Browser.mouseMovementY = y - Browser.mouseY;
          Browser.mouseX = x;
          Browser.mouseY = y;
        }
      },asyncLoad:function (url, onload, onerror, noRunDep) {
        var dep = !noRunDep ? getUniqueRunDependency('al ' + url) : '';
        Module['readAsync'](url, function(arrayBuffer) {
          assert(arrayBuffer, 'Loading data file "' + url + '" failed (no arrayBuffer).');
          onload(new Uint8Array(arrayBuffer));
          if (dep) removeRunDependency(dep);
        }, function(event) {
          if (onerror) {
            onerror();
          } else {
            throw 'Loading data file "' + url + '" failed.';
          }
        });
        if (dep) addRunDependency(dep);
      },resizeListeners:[],updateResizeListeners:function () {
        var canvas = Module['canvas'];
        Browser.resizeListeners.forEach(function(listener) {
          listener(canvas.width, canvas.height);
        });
      },setCanvasSize:function (width, height, noUpdates) {
        var canvas = Module['canvas'];
        Browser.updateCanvasDimensions(canvas, width, height);
        if (!noUpdates) Browser.updateResizeListeners();
      },windowedWidth:0,windowedHeight:0,setFullscreenCanvasSize:function () {
        // check if SDL is available
        if (typeof SDL != "undefined") {
        	var flags = HEAPU32[((SDL.screen+Runtime.QUANTUM_SIZE*0)>>2)];
        	flags = flags | 0x00800000; // set SDL_FULLSCREEN flag
        	HEAP32[((SDL.screen+Runtime.QUANTUM_SIZE*0)>>2)]=flags
        }
        Browser.updateResizeListeners();
      },setWindowedCanvasSize:function () {
        // check if SDL is available
        if (typeof SDL != "undefined") {
        	var flags = HEAPU32[((SDL.screen+Runtime.QUANTUM_SIZE*0)>>2)];
        	flags = flags & ~0x00800000; // clear SDL_FULLSCREEN flag
        	HEAP32[((SDL.screen+Runtime.QUANTUM_SIZE*0)>>2)]=flags
        }
        Browser.updateResizeListeners();
      },updateCanvasDimensions:function (canvas, wNative, hNative) {
        if (wNative && hNative) {
          canvas.widthNative = wNative;
          canvas.heightNative = hNative;
        } else {
          wNative = canvas.widthNative;
          hNative = canvas.heightNative;
        }
        var w = wNative;
        var h = hNative;
        if (Module['forcedAspectRatio'] && Module['forcedAspectRatio'] > 0) {
          if (w/h < Module['forcedAspectRatio']) {
            w = Math.round(h * Module['forcedAspectRatio']);
          } else {
            h = Math.round(w / Module['forcedAspectRatio']);
          }
        }
        if (((document['fullscreenElement'] || document['mozFullScreenElement'] ||
             document['msFullscreenElement'] || document['webkitFullscreenElement'] ||
             document['webkitCurrentFullScreenElement']) === canvas.parentNode) && (typeof screen != 'undefined')) {
           var factor = Math.min(screen.width / w, screen.height / h);
           w = Math.round(w * factor);
           h = Math.round(h * factor);
        }
        if (Browser.resizeCanvas) {
          if (canvas.width  != w) canvas.width  = w;
          if (canvas.height != h) canvas.height = h;
          if (typeof canvas.style != 'undefined') {
            canvas.style.removeProperty( "width");
            canvas.style.removeProperty("height");
          }
        } else {
          if (canvas.width  != wNative) canvas.width  = wNative;
          if (canvas.height != hNative) canvas.height = hNative;
          if (typeof canvas.style != 'undefined') {
            if (w != wNative || h != hNative) {
              canvas.style.setProperty( "width", w + "px", "important");
              canvas.style.setProperty("height", h + "px", "important");
            } else {
              canvas.style.removeProperty( "width");
              canvas.style.removeProperty("height");
            }
          }
        }
      },wgetRequests:{},nextWgetRequestHandle:0,getNextWgetRequestHandle:function () {
        var handle = Browser.nextWgetRequestHandle;
        Browser.nextWgetRequestHandle++;
        return handle;
      }};





















  var _environ=STATICTOP; STATICTOP += 16;;var ___environ=_environ;function ___buildEnvironment(env) {
      // WARNING: Arbitrary limit!
      var MAX_ENV_VALUES = 64;
      var TOTAL_ENV_SIZE = 1024;

      // Statically allocate memory for the environment.
      var poolPtr;
      var envPtr;
      if (!___buildEnvironment.called) {
        ___buildEnvironment.called = true;
        // Set default values. Use string keys for Closure Compiler compatibility.
        ENV['USER'] = ENV['LOGNAME'] = 'web_user';
        ENV['PATH'] = '/';
        ENV['PWD'] = '/';
        ENV['HOME'] = '/home/web_user';
        ENV['LANG'] = 'C';
        ENV['_'] = Module['thisProgram'];
        // Allocate memory.
        poolPtr = allocate(TOTAL_ENV_SIZE, 'i8', ALLOC_STATIC);
        envPtr = allocate(MAX_ENV_VALUES * 4,
                          'i8*', ALLOC_STATIC);
        HEAP32[((envPtr)>>2)]=poolPtr;
        HEAP32[((_environ)>>2)]=envPtr;
      } else {
        envPtr = HEAP32[((_environ)>>2)];
        poolPtr = HEAP32[((envPtr)>>2)];
      }

      // Collect key=value lines.
      var strings = [];
      var totalSize = 0;
      for (var key in env) {
        if (typeof env[key] === 'string') {
          var line = key + '=' + env[key];
          strings.push(line);
          totalSize += line.length;
        }
      }
      if (totalSize > TOTAL_ENV_SIZE) {
        throw new Error('Environment size exceeded TOTAL_ENV_SIZE!');
      }

      // Make new.
      var ptrSize = 4;
      for (var i = 0; i < strings.length; i++) {
        var line = strings[i];
        writeAsciiToMemory(line, poolPtr);
        HEAP32[(((envPtr)+(i * ptrSize))>>2)]=poolPtr;
        poolPtr += line.length + 1;
      }
      HEAP32[(((envPtr)+(strings.length * ptrSize))>>2)]=0;
    }var ENV={};function _getenv(name) {
      // char *getenv(const char *name);
      // http://pubs.opengroup.org/onlinepubs/009695399/functions/getenv.html
      if (name === 0) return 0;
      name = Pointer_stringify(name);
      if (!ENV.hasOwnProperty(name)) return 0;

      if (_getenv.ret) _free(_getenv.ret);
      _getenv.ret = allocate(intArrayFromString(ENV[name]), 'i8', ALLOC_NORMAL);
      return _getenv.ret;
    }

  function _putenv(string) {
      // int putenv(char *string);
      // http://pubs.opengroup.org/onlinepubs/009695399/functions/putenv.html
      // WARNING: According to the standard (and the glibc implementation), the
      //          string is taken by reference so future changes are reflected.
      //          We copy it instead, possibly breaking some uses.
      if (string === 0) {
        ___setErrNo(ERRNO_CODES.EINVAL);
        return -1;
      }
      string = Pointer_stringify(string);
      var splitPoint = string.indexOf('=')
      if (string === '' || string.indexOf('=') === -1) {
        ___setErrNo(ERRNO_CODES.EINVAL);
        return -1;
      }
      var name = string.slice(0, splitPoint);
      var value = string.slice(splitPoint + 1);
      if (!(name in ENV) || ENV[name] !== value) {
        ENV[name] = value;
        ___buildEnvironment(ENV);
      }
      return 0;
    }

  function _SDL_RWFromConstMem(mem, size) {
      var id = SDL.rwops.length; // TODO: recycle ids when they are null
      SDL.rwops.push({ bytes: mem, count: size });
      return id;
    }function _TTF_FontHeight(font) {
      var fontData = SDL.fonts[font];
      return fontData.size;
    }function _TTF_SizeText(font, text, w, h) {
      var fontData = SDL.fonts[font];
      if (w) {
        HEAP32[((w)>>2)]=SDL.estimateTextWidth(fontData, Pointer_stringify(text));
      }
      if (h) {
        HEAP32[((h)>>2)]=fontData.size;
      }
      return 0;
    }function _TTF_RenderText_Solid(font, text, color) {
      // XXX the font and color are ignored
      text = Pointer_stringify(text) || ' '; // if given an empty string, still return a valid surface
      var fontData = SDL.fonts[font];
      var w = SDL.estimateTextWidth(fontData, text);
      var h = fontData.size;
      var color = SDL.loadColorToCSSRGB(color); // XXX alpha breaks fonts?
      var fontString = SDL.makeFontString(h, fontData.name);
      var surf = SDL.makeSurface(w, h, 0, false, 'text:' + text); // bogus numbers..
      var surfData = SDL.surfaces[surf];
      surfData.ctx.save();
      surfData.ctx.fillStyle = color;
      surfData.ctx.font = fontString;
      // use bottom alligment, because it works
      // same in all browsers, more info here:
      // https://bugzilla.mozilla.org/show_bug.cgi?id=737852
      surfData.ctx.textBaseline = 'bottom';
      surfData.ctx.fillText(text, 0, h|0);
      surfData.ctx.restore();
      return surf;
    }function _Mix_HaltMusic() {
      var audio = SDL.music.audio;
      if (audio) {
        audio.src = audio.src; // rewind <media> element
        audio.currentPosition = 0; // rewind Web Audio graph playback.
        audio.pause();
      }
      SDL.music.audio = null;
      if (SDL.hookMusicFinished) {
        Module['dynCall_v'](SDL.hookMusicFinished);
      }
      return 0;
    }function _Mix_PlayMusic(id, loops) {
      // Pause old music if it exists.
      if (SDL.music.audio) {
        if (!SDL.music.audio.paused) Module.printErr('Music is already playing. ' + SDL.music.source);
        SDL.music.audio.pause();
      }
      var info = SDL.audios[id];
      var audio;
      if (info.webAudio) { // Play via Web Audio API
        // Create an instance of the WebAudio object.
        audio = {};
        audio.resource = info; // This new webAudio object is an instance that refers to this existing resource.
        audio.paused = false;
        audio.currentPosition = 0;
        audio.play = function() { SDL.playWebAudio(this); }
        audio.pause = function() { SDL.pauseWebAudio(this); }
      } else if (info.audio) { // Play via the <audio> element
        audio = info.audio;
      }
      audio['onended'] = function() { if (SDL.music.audio == this) _Mix_HaltMusic(); } // will send callback
      audio.loop = loops != 0; // TODO: handle N loops for finite N
      audio.volume = SDL.music.volume;
      SDL.music.audio = audio;
      audio.play();
      return 0;
    }function _Mix_FreeChunk(id) {
      SDL.audios[id] = null;
    }function _Mix_LoadWAV_RW(rwopsID, freesrc) {
      var rwops = SDL.rwops[rwopsID];


      if (rwops === undefined)
        return 0;

      var filename = '';
      var audio;
      var webAudio;
      var bytes;

      if (rwops.filename !== undefined) {
        filename = PATH.resolve(rwops.filename);
        var raw = Module["preloadedAudios"][filename];
        if (!raw) {
          if (raw === null) Module.printErr('Trying to reuse preloaded audio, but freePreloadedMediaOnUse is set!');
          if (!Module.noAudioDecoding) Runtime.warnOnce('Cannot find preloaded audio ' + filename);

          // see if we can read the file-contents from the in-memory FS
          try {
            bytes = FS.readFile(filename);
          } catch (e) {
            Module.printErr('Couldn\'t find file for: ' + filename);
            return 0;
          }
        }
        if (Module['freePreloadedMediaOnUse']) {
          Module["preloadedAudios"][filename] = null;
        }
        audio = raw;
      }
      else if (rwops.bytes !== undefined) {
        // For Web Audio context buffer decoding, we must make a clone of the audio data, but for <media> element,
        // a view to existing data is sufficient.
        if (SDL.webAudioAvailable()) bytes = HEAPU8.buffer.slice(rwops.bytes, rwops.bytes + rwops.count);
        else bytes = HEAPU8.subarray(rwops.bytes, rwops.bytes + rwops.count);
      }
      else {
        return 0;
      }

      var arrayBuffer = bytes ? bytes.buffer || bytes : bytes;

      // To allow user code to work around browser bugs with audio playback on <audio> elements an Web Audio, enable
      // the user code to hook in a callback to decide on a file basis whether each file should use Web Audio or <audio> for decoding and playback.
      // In particular, see https://bugzilla.mozilla.org/show_bug.cgi?id=654787 and ?id=1012801 for tradeoffs.
      var canPlayWithWebAudio = Module['SDL_canPlayWithWebAudio'] === undefined || Module['SDL_canPlayWithWebAudio'](filename, arrayBuffer);

      if (bytes !== undefined && SDL.webAudioAvailable() && canPlayWithWebAudio) {
        audio = undefined;
        webAudio = {};
        // The audio decoding process is asynchronous, which gives trouble if user code plays the audio data back immediately
        // after loading. Therefore prepare an array of callback handlers to run when this audio decoding is complete, which
        // will then start the playback (with some delay).
        webAudio.onDecodeComplete = []; // While this member array exists, decoding hasn't finished yet.
        function onDecodeComplete(data) {
          webAudio.decodedBuffer = data;
          // Call all handlers that were waiting for this decode to finish, and clear the handler list.
          webAudio.onDecodeComplete.forEach(function(e) { e(); });
          webAudio.onDecodeComplete = undefined; // Don't allow more callback handlers since audio has finished decoding.
        }

        SDL.audioContext['decodeAudioData'](arrayBuffer, onDecodeComplete);
      } else if (audio === undefined && bytes) {
        // Here, we didn't find a preloaded audio but we either were passed a filepath for
        // which we loaded bytes, or we were passed some bytes
        var blob = new Blob([bytes], {type: rwops.mimetype});
        var url = URL.createObjectURL(blob);
        audio = new Audio();
        audio.src = url;
        audio.mozAudioChannelType = 'content'; // bugzilla 910340
      }

      var id = SDL.audios.length;
      // Keep the loaded audio in the audio arrays, ready for playback
      SDL.audios.push({
        source: filename,
        audio: audio, // Points to the <audio> element, if loaded
        webAudio: webAudio // Points to a Web Audio -specific resource object, if loaded
      });
      return id;
    }function _Mix_PlayChannel(channel, id, loops) {
      // TODO: handle fixed amount of N loops. Currently loops either 0 or infinite times.

      // Get the audio element associated with the ID
      var info = SDL.audios[id];
      if (!info) return -1;
      if (!info.audio && !info.webAudio) return -1;

      // If the user asks us to allocate a channel automatically, get the first
      // free one.
      if (channel == -1) {
        for (var i = SDL.channelMinimumNumber; i < SDL.numChannels; i++) {
          if (!SDL.channels[i].audio) {
            channel = i;
            break;
          }
        }
        if (channel == -1) {
          Module.printErr('All ' + SDL.numChannels + ' channels in use!');
          return -1;
        }
      }
      var channelInfo = SDL.channels[channel];
      var audio;
      if (info.webAudio) {
        // Create an instance of the WebAudio object.
        audio = {};
        audio.resource = info; // This new object is an instance that refers to this existing resource.
        audio.paused = false;
        audio.currentPosition = 0;
        // Make our instance look similar to the instance of a <media> to make api simple.
        audio.play = function() { SDL.playWebAudio(this); }
        audio.pause = function() { SDL.pauseWebAudio(this); }
      } else {
        // We clone the audio node to utilize the preloaded audio buffer, since
        // the browser has already preloaded the audio file.
        audio = info.audio.cloneNode(true);
        audio.numChannels = info.audio.numChannels;
        audio.frequency = info.audio.frequency;
      }
      audio['onended'] = function SDL_audio_onended() { // TODO: cache these
        if (channelInfo.audio == this) { channelInfo.audio.paused = true; channelInfo.audio = null; }
        if (SDL.channelFinished) Runtime.getFuncWrapper(SDL.channelFinished, 'vi')(channel);
      }
      channelInfo.audio = audio;
      // TODO: handle N loops. Behavior matches Mix_PlayMusic
      audio.loop = loops != 0;
      audio.volume = channelInfo.volume;
      audio.play();
      return channel;
    }function _SDL_PauseAudio(pauseOn) {
      if (!SDL.audio) {
        return;
      }
      if (pauseOn) {
        if (SDL.audio.timer !== undefined) {
          clearTimeout(SDL.audio.timer);
          SDL.audio.numAudioTimersPending = 0;
          SDL.audio.timer = undefined;
        }
      } else if (!SDL.audio.timer) {
        // Start the audio playback timer callback loop.
        SDL.audio.numAudioTimersPending = 1;
        SDL.audio.timer = Browser.safeSetTimeout(SDL.audio.caller, 1);
      }
      SDL.audio.paused = pauseOn;
    }function _SDL_CloseAudio() {
      if (SDL.audio) {
        _SDL_PauseAudio(1);
        _free(SDL.audio.buffer);
        SDL.audio = null;
        SDL.allocateChannels(0);
      }
    }function _SDL_LockSurface(surf) {
      var surfData = SDL.surfaces[surf];

      surfData.locked++;
      if (surfData.locked > 1) return 0;

      if (!surfData.buffer) {
        surfData.buffer = _malloc(surfData.width * surfData.height * 4);
        HEAP32[(((surf)+(20))>>2)]=surfData.buffer;
      }

      // Mark in C/C++-accessible SDL structure
      // SDL_Surface has the following fields: Uint32 flags, SDL_PixelFormat *format; int w, h; Uint16 pitch; void *pixels; ...
      // So we have fields all of the same size, and 5 of them before us.
      // TODO: Use macros like in library.js
      HEAP32[(((surf)+(20))>>2)]=surfData.buffer;

      if (surf == SDL.screen && Module.screenIsReadOnly && surfData.image) return 0;

      if (SDL.defaults.discardOnLock) {
        if (!surfData.image) {
          surfData.image = surfData.ctx.createImageData(surfData.width, surfData.height);
        }
        if (!SDL.defaults.opaqueFrontBuffer) return;
      } else {
        surfData.image = surfData.ctx.getImageData(0, 0, surfData.width, surfData.height);
      }

      // Emulate desktop behavior and kill alpha values on the locked surface. (very costly!) Set SDL.defaults.opaqueFrontBuffer = false
      // if you don't want this.
      if (surf == SDL.screen && SDL.defaults.opaqueFrontBuffer) {
        var data = surfData.image.data;
        var num = data.length;
        for (var i = 0; i < num/4; i++) {
          data[i*4+3] = 255; // opacity, as canvases blend alpha
        }
      }

      if (SDL.defaults.copyOnLock && !SDL.defaults.discardOnLock) {
        // Copy pixel data to somewhere accessible to 'C/C++'
        if (surfData.isFlagSet(0x00200000 /* SDL_HWPALETTE */)) {
          // If this is neaded then
          // we should compact the data from 32bpp to 8bpp index.
          // I think best way to implement this is use
          // additional colorMap hash (color->index).
          // Something like this:
          //
          // var size = surfData.width * surfData.height;
          // var data = '';
          // for (var i = 0; i<size; i++) {
          //   var color = SDL.translateRGBAToColor(
          //     surfData.image.data[i*4   ],
          //     surfData.image.data[i*4 +1],
          //     surfData.image.data[i*4 +2],
          //     255);
          //   var index = surfData.colorMap[color];
          //   HEAP8[(((surfData.buffer)+(i))>>0)]=index;
          // }
          throw 'CopyOnLock is not supported for SDL_LockSurface with SDL_HWPALETTE flag set' + new Error().stack;
        } else {
          HEAPU8.set(surfData.image.data, surfData.buffer);
        }
      }

      return 0;
    }

  function _SDL_FreeRW(rwopsID) {
      SDL.rwops[rwopsID] = null;
      while (SDL.rwops.length > 0 && SDL.rwops[SDL.rwops.length-1] === null) {
        SDL.rwops.pop();
      }
    }function _IMG_Load_RW(rwopsID, freeSrc) {
      try {
        // stb_image integration support
        var cleanup = function() {
          if (rwops && freeSrc) _SDL_FreeRW(rwopsID);
        }
        var addCleanup = function(func) {
          var old = cleanup;
          cleanup = function added_cleanup() {
            old();
            func();
          }
        }
        var callStbImage = function(func, params) {
          var x = Module['_malloc'](4);
          var y = Module['_malloc'](4);
          var comp = Module['_malloc'](4);
          addCleanup(function() {
            Module['_free'](x);
            Module['_free'](y);
            Module['_free'](comp);
            if (data) Module['_stbi_image_free'](data);
          });
          var data = Module['_' + func].apply(null, params.concat([x, y, comp, 0]));
          if (!data) return null;
          return {
            rawData: true,
            data: data,
            width: HEAP32[((x)>>2)],
            height: HEAP32[((y)>>2)],
            size: HEAP32[((x)>>2)] * HEAP32[((y)>>2)] * HEAP32[((comp)>>2)],
            bpp: HEAP32[((comp)>>2)]
          };
        }

        var rwops = SDL.rwops[rwopsID];
        if (rwops === undefined) {
          return 0;
        }

        var filename = rwops.filename;
        if (filename === undefined) {
          Runtime.warnOnce('Only file names that have been preloaded are supported for IMG_Load_RW. Consider using STB_IMAGE=1 if you want synchronous image decoding (see settings.js), or package files with --use-preload-plugins');
          return 0;
        }

        if (!raw) {
          filename = PATH.resolve(filename);
          var raw = Module["preloadedImages"][filename];
          if (!raw) {
            if (raw === null) Module.printErr('Trying to reuse preloaded image, but freePreloadedMediaOnUse is set!');
            Runtime.warnOnce('Cannot find preloaded image ' + filename);
            Runtime.warnOnce('Cannot find preloaded image ' + filename + '. Consider using STB_IMAGE=1 if you want synchronous image decoding (see settings.js), or package files with --use-preload-plugins');
            return 0;
          } else if (Module['freePreloadedMediaOnUse']) {
            Module["preloadedImages"][filename] = null;
          }
        }

        var surf = SDL.makeSurface(raw.width, raw.height, 0, false, 'load:' + filename);
        var surfData = SDL.surfaces[surf];
        surfData.ctx.globalCompositeOperation = "copy";
        if (!raw.rawData) {
          surfData.ctx.drawImage(raw, 0, 0, raw.width, raw.height, 0, 0, raw.width, raw.height);
        } else {
          var imageData = surfData.ctx.getImageData(0, 0, surfData.width, surfData.height);
          if (raw.bpp == 4) {
            // rgba
            imageData.data.set(HEAPU8.subarray((raw.data),(raw.data+raw.size)));
          } else if (raw.bpp == 3) {
            // rgb
            var pixels = raw.size/3;
            var data = imageData.data;
            var sourcePtr = raw.data;
            var destPtr = 0;
            for (var i = 0; i < pixels; i++) {
              data[destPtr++] = HEAPU8[((sourcePtr++)>>0)];
              data[destPtr++] = HEAPU8[((sourcePtr++)>>0)];
              data[destPtr++] = HEAPU8[((sourcePtr++)>>0)];
              data[destPtr++] = 255;
            }
          } else if (raw.bpp == 2) {
            // grayscale + alpha
            var pixels = raw.size;
            var data = imageData.data;
            var sourcePtr = raw.data;
            var destPtr = 0;
            for (var i = 0; i < pixels; i++) {
              var gray = HEAPU8[((sourcePtr++)>>0)];
              var alpha = HEAPU8[((sourcePtr++)>>0)];
              data[destPtr++] = gray;
              data[destPtr++] = gray;
              data[destPtr++] = gray;
              data[destPtr++] = alpha;
            }
          } else if (raw.bpp == 1) {
            // grayscale
            var pixels = raw.size;
            var data = imageData.data;
            var sourcePtr = raw.data;
            var destPtr = 0;
            for (var i = 0; i < pixels; i++) {
              var value = HEAPU8[((sourcePtr++)>>0)];
              data[destPtr++] = value;
              data[destPtr++] = value;
              data[destPtr++] = value;
              data[destPtr++] = 255;
            }
          } else {
            Module.printErr('cannot handle bpp ' + raw.bpp);
            return 0;
          }
          surfData.ctx.putImageData(imageData, 0, 0);
        }
        surfData.ctx.globalCompositeOperation = "source-over";
        // XXX SDL does not specify that loaded images must have available pixel data, in fact
        //     there are cases where you just want to blit them, so you just need the hardware
        //     accelerated version. However, code everywhere seems to assume that the pixels
        //     are in fact available, so we retrieve it here. This does add overhead though.
        _SDL_LockSurface(surf);
        surfData.locked--; // The surface is not actually locked in this hack
        if (SDL.GL) {
          // After getting the pixel data, we can free the canvas and context if we do not need to do 2D canvas blitting
          surfData.canvas = surfData.ctx = null;
        }
        return surf;
      } finally {
        cleanup();
      }
    }

  function _SDL_RWFromFile(_name, mode) {
      var id = SDL.rwops.length; // TODO: recycle ids when they are null
      var name = Pointer_stringify(_name)
      SDL.rwops.push({ filename: name, mimetype: Browser.getMimetype(name) });
      return id;
    }function _IMG_Load(filename){
      var rwops = _SDL_RWFromFile(filename);
      var result = _IMG_Load_RW(rwops, 1);
      return result;
    }function _SDL_UpperBlitScaled(src, srcrect, dst, dstrect) {
      return SDL.blitSurface(src, srcrect, dst, dstrect, true);
    }function _SDL_UpperBlit(src, srcrect, dst, dstrect) {
      return SDL.blitSurface(src, srcrect, dst, dstrect, false);
    }function _SDL_GetTicks() {
      return (Date.now() - SDL.startTime)|0;
    }var SDL={defaults:{width:320,height:200,copyOnLock:true,discardOnLock:false,opaqueFrontBuffer:true},version:null,surfaces:{},canvasPool:[],events:[],fonts:[null],audios:[null],rwops:[null],music:{audio:null,volume:1},mixerFrequency:22050,mixerFormat:32784,mixerNumChannels:2,mixerChunkSize:1024,channelMinimumNumber:0,GL:false,glAttributes:{0:3,1:3,2:2,3:0,4:0,5:1,6:16,7:0,8:0,9:0,10:0,11:0,12:0,13:0,14:0,15:1,16:0,17:0,18:0},keyboardState:null,keyboardMap:{},canRequestFullscreen:false,isRequestingFullscreen:false,textInput:false,startTime:null,initFlags:0,buttonState:0,modState:0,DOMButtons:[0,0,0],DOMEventToSDLEvent:{},TOUCH_DEFAULT_ID:0,eventHandler:null,eventHandlerContext:null,eventHandlerTemp:0,keyCodes:{16:1249,17:1248,18:1250,20:1081,33:1099,34:1102,35:1101,36:1098,37:1104,38:1106,39:1103,40:1105,44:316,45:1097,46:127,91:1251,93:1125,96:1122,97:1113,98:1114,99:1115,100:1116,101:1117,102:1118,103:1119,104:1120,105:1121,106:1109,107:1111,109:1110,110:1123,111:1108,112:1082,113:1083,114:1084,115:1085,116:1086,117:1087,118:1088,119:1089,120:1090,121:1091,122:1092,123:1093,124:1128,125:1129,126:1130,127:1131,128:1132,129:1133,130:1134,131:1135,132:1136,133:1137,134:1138,135:1139,144:1107,160:94,161:33,162:34,163:35,164:36,165:37,166:38,167:95,168:40,169:41,170:42,171:43,172:124,173:45,174:123,175:125,176:126,181:127,182:129,183:128,188:44,190:46,191:47,192:96,219:91,220:92,221:93,222:39,224:1251},scanCodes:{8:42,9:43,13:40,27:41,32:44,35:204,39:53,44:54,46:55,47:56,48:39,49:30,50:31,51:32,52:33,53:34,54:35,55:36,56:37,57:38,58:203,59:51,61:46,91:47,92:49,93:48,96:52,97:4,98:5,99:6,100:7,101:8,102:9,103:10,104:11,105:12,106:13,107:14,108:15,109:16,110:17,111:18,112:19,113:20,114:21,115:22,116:23,117:24,118:25,119:26,120:27,121:28,122:29,127:76,305:224,308:226,316:70},loadRect:function (rect) {
        return {
          x: HEAP32[((rect + 0)>>2)],
          y: HEAP32[((rect + 4)>>2)],
          w: HEAP32[((rect + 8)>>2)],
          h: HEAP32[((rect + 12)>>2)]
        };
      },updateRect:function (rect, r) {
        HEAP32[((rect)>>2)]=r.x;
        HEAP32[(((rect)+(4))>>2)]=r.y;
        HEAP32[(((rect)+(8))>>2)]=r.w;
        HEAP32[(((rect)+(12))>>2)]=r.h;
      },intersectionOfRects:function (first, second) {
        var leftX = Math.max(first.x, second.x);
        var leftY = Math.max(first.y, second.y);
        var rightX = Math.min(first.x + first.w, second.x + second.w);
        var rightY = Math.min(first.y + first.h, second.y + second.h);

        return {
          x: leftX,
          y: leftY,
          w: Math.max(leftX, rightX) - leftX,
          h: Math.max(leftY, rightY) - leftY
        }
      },checkPixelFormat:function (fmt) {
      },loadColorToCSSRGB:function (color) {
        var rgba = HEAP32[((color)>>2)];
        return 'rgb(' + (rgba&255) + ',' + ((rgba >> 8)&255) + ',' + ((rgba >> 16)&255) + ')';
      },loadColorToCSSRGBA:function (color) {
        var rgba = HEAP32[((color)>>2)];
        return 'rgba(' + (rgba&255) + ',' + ((rgba >> 8)&255) + ',' + ((rgba >> 16)&255) + ',' + (((rgba >> 24)&255)/255) + ')';
      },translateColorToCSSRGBA:function (rgba) {
        return 'rgba(' + (rgba&0xff) + ',' + (rgba>>8 & 0xff) + ',' + (rgba>>16 & 0xff) + ',' + (rgba>>>24)/0xff + ')';
      },translateRGBAToCSSRGBA:function (r, g, b, a) {
        return 'rgba(' + (r&0xff) + ',' + (g&0xff) + ',' + (b&0xff) + ',' + (a&0xff)/255 + ')';
      },translateRGBAToColor:function (r, g, b, a) {
        return r | g << 8 | b << 16 | a << 24;
      },makeSurface:function (width, height, flags, usePageCanvas, source, rmask, gmask, bmask, amask) {
        flags = flags || 0;
        var is_SDL_HWSURFACE = flags & 0x00000001;
        var is_SDL_HWPALETTE = flags & 0x00200000;
        var is_SDL_OPENGL = flags & 0x04000000;

        var surf = _malloc(60);
        var pixelFormat = _malloc(44);
        //surface with SDL_HWPALETTE flag is 8bpp surface (1 byte)
        var bpp = is_SDL_HWPALETTE ? 1 : 4;
        var buffer = 0;

        // preemptively initialize this for software surfaces,
        // otherwise it will be lazily initialized inside of SDL_LockSurface
        if (!is_SDL_HWSURFACE && !is_SDL_OPENGL) {
          buffer = _malloc(width * height * 4);
        }

        HEAP32[((surf)>>2)]=flags;
        HEAP32[(((surf)+(4))>>2)]=pixelFormat;
        HEAP32[(((surf)+(8))>>2)]=width;
        HEAP32[(((surf)+(12))>>2)]=height;
        HEAP32[(((surf)+(16))>>2)]=width * bpp;  // assuming RGBA or indexed for now,
                                                                                          // since that is what ImageData gives us in browsers
        HEAP32[(((surf)+(20))>>2)]=buffer;

        HEAP32[(((surf)+(36))>>2)]=0;
        HEAP32[(((surf)+(40))>>2)]=0;
        HEAP32[(((surf)+(44))>>2)]=Module["canvas"].width;
        HEAP32[(((surf)+(48))>>2)]=Module["canvas"].height;

        HEAP32[(((surf)+(56))>>2)]=1;

        HEAP32[((pixelFormat)>>2)]=-2042224636;
        HEAP32[(((pixelFormat)+(4))>>2)]=0;// TODO
        HEAP8[(((pixelFormat)+(8))>>0)]=bpp * 8;
        HEAP8[(((pixelFormat)+(9))>>0)]=bpp;

        HEAP32[(((pixelFormat)+(12))>>2)]=rmask || 0x000000ff;
        HEAP32[(((pixelFormat)+(16))>>2)]=gmask || 0x0000ff00;
        HEAP32[(((pixelFormat)+(20))>>2)]=bmask || 0x00ff0000;
        HEAP32[(((pixelFormat)+(24))>>2)]=amask || 0xff000000;

        // Decide if we want to use WebGL or not
        SDL.GL = SDL.GL || is_SDL_OPENGL;
        var canvas;
        if (!usePageCanvas) {
          if (SDL.canvasPool.length > 0) {
            canvas = SDL.canvasPool.pop();
          } else {
            canvas = document.createElement('canvas');
          }
          canvas.width = width;
          canvas.height = height;
        } else {
          canvas = Module['canvas'];
        }

        var webGLContextAttributes = {
          antialias: ((SDL.glAttributes[13 /*SDL_GL_MULTISAMPLEBUFFERS*/] != 0) && (SDL.glAttributes[14 /*SDL_GL_MULTISAMPLESAMPLES*/] > 1)),
          depth: (SDL.glAttributes[6 /*SDL_GL_DEPTH_SIZE*/] > 0),
          stencil: (SDL.glAttributes[7 /*SDL_GL_STENCIL_SIZE*/] > 0),
          alpha: (SDL.glAttributes[3 /*SDL_GL_ALPHA_SIZE*/] > 0)
        };

        var ctx = Browser.createContext(canvas, is_SDL_OPENGL, usePageCanvas, webGLContextAttributes);

        SDL.surfaces[surf] = {
          width: width,
          height: height,
          canvas: canvas,
          ctx: ctx,
          surf: surf,
          buffer: buffer,
          pixelFormat: pixelFormat,
          alpha: 255,
          flags: flags,
          locked: 0,
          usePageCanvas: usePageCanvas,
          source: source,

          isFlagSet: function(flag) {
            return flags & flag;
          }
        };

        return surf;
      },copyIndexedColorData:function (surfData, rX, rY, rW, rH) {
        // HWPALETTE works with palette
        // setted by SDL_SetColors
        if (!surfData.colors) {
          return;
        }

        var fullWidth  = Module['canvas'].width;
        var fullHeight = Module['canvas'].height;

        var startX  = rX || 0;
        var startY  = rY || 0;
        var endX    = (rW || (fullWidth - startX)) + startX;
        var endY    = (rH || (fullHeight - startY)) + startY;

        var buffer  = surfData.buffer;

        if (!surfData.image.data32) {
          surfData.image.data32 = new Uint32Array(surfData.image.data.buffer);
        }
        var data32   = surfData.image.data32;

        var colors32 = surfData.colors32;

        for (var y = startY; y < endY; ++y) {
          var base = y * fullWidth;
          for (var x = startX; x < endX; ++x) {
            data32[base + x] = colors32[HEAPU8[((buffer + base + x)>>0)]];
          }
        }
      },freeSurface:function (surf) {
        var refcountPointer = surf + 56;
        var refcount = HEAP32[((refcountPointer)>>2)];
        if (refcount > 1) {
          HEAP32[((refcountPointer)>>2)]=refcount - 1;
          return;
        }

        var info = SDL.surfaces[surf];
        if (!info.usePageCanvas && info.canvas) SDL.canvasPool.push(info.canvas);
        if (info.buffer) _free(info.buffer);
        _free(info.pixelFormat);
        _free(surf);
        SDL.surfaces[surf] = null;

        if (surf === SDL.screen) {
          SDL.screen = null;
        }
      },blitSurface__deps:["SDL_LockSurface"],blitSurface:function (src, srcrect, dst, dstrect, scale) {
        var srcData = SDL.surfaces[src];
        var dstData = SDL.surfaces[dst];
        var sr, dr;
        if (srcrect) {
          sr = SDL.loadRect(srcrect);
        } else {
          sr = { x: 0, y: 0, w: srcData.width, h: srcData.height };
        }
        if (dstrect) {
          dr = SDL.loadRect(dstrect);
        } else {
          dr = { x: 0, y: 0, w: srcData.width, h: srcData.height };
        }
        if (dstData.clipRect) {
          var widthScale = (!scale || sr.w === 0) ? 1 : sr.w / dr.w;
          var heightScale = (!scale || sr.h === 0) ? 1 : sr.h / dr.h;

          dr = SDL.intersectionOfRects(dstData.clipRect, dr);

          sr.w = dr.w * widthScale;
          sr.h = dr.h * heightScale;

          if (dstrect) {
            SDL.updateRect(dstrect, dr);
          }
        }
        var blitw, blith;
        if (scale) {
          blitw = dr.w; blith = dr.h;
        } else {
          blitw = sr.w; blith = sr.h;
        }
        if (sr.w === 0 || sr.h === 0 || blitw === 0 || blith === 0) {
          return 0;
        }
        var oldAlpha = dstData.ctx.globalAlpha;
        dstData.ctx.globalAlpha = srcData.alpha/255;
        dstData.ctx.drawImage(srcData.canvas, sr.x, sr.y, sr.w, sr.h, dr.x, dr.y, blitw, blith);
        dstData.ctx.globalAlpha = oldAlpha;
        if (dst != SDL.screen) {
          // XXX As in IMG_Load, for compatibility we write out |pixels|
          Runtime.warnOnce('WARNING: copying canvas data to memory for compatibility');
          _SDL_LockSurface(dst);
          dstData.locked--; // The surface is not actually locked in this hack
        }
        return 0;
      },downFingers:{},savedKeydown:null,receiveEvent:function (event) {
        function unpressAllPressedKeys() {
          // Un-press all pressed keys: TODO
          for (var code in SDL.keyboardMap) {
            SDL.events.push({
              type: 'keyup',
              keyCode: SDL.keyboardMap[code]
            });
          }
        };
        switch(event.type) {
          case 'touchstart': case 'touchmove': {
            event.preventDefault();

            var touches = [];

            // Clear out any touchstart events that we've already processed
            if (event.type === 'touchstart') {
              for (var i = 0; i < event.touches.length; i++) {
                var touch = event.touches[i];
                if (SDL.downFingers[touch.identifier] != true) {
                  SDL.downFingers[touch.identifier] = true;
                  touches.push(touch);
                }
              }
            } else {
              touches = event.touches;
            }

            var firstTouch = touches[0];
            if (firstTouch) {
              if (event.type == 'touchstart') {
                SDL.DOMButtons[0] = 1;
              }
              var mouseEventType;
              switch(event.type) {
                case 'touchstart': mouseEventType = 'mousedown'; break;
                case 'touchmove': mouseEventType = 'mousemove'; break;
              }
              var mouseEvent = {
                type: mouseEventType,
                button: 0,
                pageX: firstTouch.clientX,
                pageY: firstTouch.clientY
              };
              SDL.events.push(mouseEvent);
            }

            for (var i = 0; i < touches.length; i++) {
              var touch = touches[i];
              SDL.events.push({
                type: event.type,
                touch: touch
              });
            };
            break;
          }
          case 'touchend': {
            event.preventDefault();

            // Remove the entry in the SDL.downFingers hash
            // because the finger is no longer down.
            for(var i = 0; i < event.changedTouches.length; i++) {
              var touch = event.changedTouches[i];
              if (SDL.downFingers[touch.identifier] === true) {
                delete SDL.downFingers[touch.identifier];
              }
            }

            var mouseEvent = {
              type: 'mouseup',
              button: 0,
              pageX: event.changedTouches[0].clientX,
              pageY: event.changedTouches[0].clientY
            };
            SDL.DOMButtons[0] = 0;
            SDL.events.push(mouseEvent);

            for (var i = 0; i < event.changedTouches.length; i++) {
              var touch = event.changedTouches[i];
              SDL.events.push({
                type: 'touchend',
                touch: touch
              });
            };
            break;
          }
          case 'DOMMouseScroll': case 'mousewheel': case 'wheel':
            var delta = -Browser.getMouseWheelDelta(event); // Flip the wheel direction to translate from browser wheel direction (+:down) to SDL direction (+:up)
            delta = (delta == 0) ? 0 : (delta > 0 ? Math.max(delta, 1) : Math.min(delta, -1)); // Quantize to integer so that minimum scroll is at least +/- 1.

            // Simulate old-style SDL events representing mouse wheel input as buttons
            var button = delta > 0 ? 3 /*SDL_BUTTON_WHEELUP-1*/ : 4 /*SDL_BUTTON_WHEELDOWN-1*/; // Subtract one since JS->C marshalling is defined to add one back.
            SDL.events.push({ type: 'mousedown', button: button, pageX: event.pageX, pageY: event.pageY });
            SDL.events.push({ type: 'mouseup', button: button, pageX: event.pageX, pageY: event.pageY });

            // Pass a delta motion event.
            SDL.events.push({ type: 'wheel', deltaX: 0, deltaY: delta });
            event.preventDefault(); // If we don't prevent this, then 'wheel' event will be sent again by the browser as 'DOMMouseScroll' and we will receive this same event the second time.
            break;
          case 'mousemove':
            if (SDL.DOMButtons[0] === 1) {
              SDL.events.push({
                type: 'touchmove',
                touch: {
                  identifier: 0,
                  deviceID: -1,
                  pageX: event.pageX,
                  pageY: event.pageY
                }
              });
            }
            if (Browser.pointerLock) {
              // workaround for firefox bug 750111
              if ('mozMovementX' in event) {
                event['movementX'] = event['mozMovementX'];
                event['movementY'] = event['mozMovementY'];
              }
              // workaround for Firefox bug 782777
              if (event['movementX'] == 0 && event['movementY'] == 0) {
                // ignore a mousemove event if it doesn't contain any movement info
                // (without pointer lock, we infer movement from pageX/pageY, so this check is unnecessary)
                event.preventDefault();
                return;
              }
            }
            // fall through
          case 'keydown': case 'keyup': case 'keypress': case 'mousedown': case 'mouseup':
            // If we preventDefault on keydown events, the subsequent keypress events
            // won't fire. However, it's fine (and in some cases necessary) to
            // preventDefault for keys that don't generate a character. Otherwise,
            // preventDefault is the right thing to do in general.
            if (event.type !== 'keydown' || (!SDL.unicode && !SDL.textInput) || (event.keyCode === 8 /* backspace */ || event.keyCode === 9 /* tab */)) {
              event.preventDefault();
            }

            if (event.type == 'mousedown') {
              SDL.DOMButtons[event.button] = 1;
              SDL.events.push({
                type: 'touchstart',
                touch: {
                  identifier: 0,
                  deviceID: -1,
                  pageX: event.pageX,
                  pageY: event.pageY
                }
              });
            } else if (event.type == 'mouseup') {
              // ignore extra ups, can happen if we leave the canvas while pressing down, then return,
              // since we add a mouseup in that case
              if (!SDL.DOMButtons[event.button]) {
                return;
              }

              SDL.events.push({
                type: 'touchend',
                touch: {
                  identifier: 0,
                  deviceID: -1,
                  pageX: event.pageX,
                  pageY: event.pageY
                }
              });
              SDL.DOMButtons[event.button] = 0;
            }

            // We can only request fullscreen as the result of user input.
            // Due to this limitation, we toggle a boolean on keydown which
            // SDL_WM_ToggleFullScreen will check and subsequently set another
            // flag indicating for us to request fullscreen on the following
            // keyup. This isn't perfect, but it enables SDL_WM_ToggleFullScreen
            // to work as the result of a keypress (which is an extremely
            // common use case).
            if (event.type === 'keydown' || event.type === 'mousedown') {
              SDL.canRequestFullscreen = true;
            } else if (event.type === 'keyup' || event.type === 'mouseup') {
              if (SDL.isRequestingFullscreen) {
                Module['requestFullscreen'](/*lockPointer=*/true, /*resizeCanvas=*/true);
                SDL.isRequestingFullscreen = false;
              }
              SDL.canRequestFullscreen = false;
            }

            // SDL expects a unicode character to be passed to its keydown events.
            // Unfortunately, the browser APIs only provide a charCode property on
            // keypress events, so we must backfill in keydown events with their
            // subsequent keypress event's charCode.
            if (event.type === 'keypress' && SDL.savedKeydown) {
              // charCode is read-only
              SDL.savedKeydown.keypressCharCode = event.charCode;
              SDL.savedKeydown = null;
            } else if (event.type === 'keydown') {
              SDL.savedKeydown = event;
            }

            // Don't push keypress events unless SDL_StartTextInput has been called.
            if (event.type !== 'keypress' || SDL.textInput) {
              SDL.events.push(event);
            }
            break;
          case 'mouseout':
            // Un-press all pressed mouse buttons, because we might miss the release outside of the canvas
            for (var i = 0; i < 3; i++) {
              if (SDL.DOMButtons[i]) {
                SDL.events.push({
                  type: 'mouseup',
                  button: i,
                  pageX: event.pageX,
                  pageY: event.pageY
                });
                SDL.DOMButtons[i] = 0;
              }
            }
            event.preventDefault();
            break;
          case 'focus':
            SDL.events.push(event);
            event.preventDefault();
            break;
          case 'blur':
            SDL.events.push(event);
            unpressAllPressedKeys();
            event.preventDefault();
            break;
          case 'visibilitychange':
            SDL.events.push({
              type: 'visibilitychange',
              visible: !document.hidden
            });
            unpressAllPressedKeys();
            event.preventDefault();
            break;
          case 'unload':
            if (Browser.mainLoop.runner) {
              SDL.events.push(event);
              // Force-run a main event loop, since otherwise this event will never be caught!
              Browser.mainLoop.runner();
            }
            return;
          case 'resize':
            SDL.events.push(event);
            // manually triggered resize event doesn't have a preventDefault member
            if (event.preventDefault) {
              event.preventDefault();
            }
            break;
        }
        if (SDL.events.length >= 10000) {
          Module.printErr('SDL event queue full, dropping events');
          SDL.events = SDL.events.slice(0, 10000);
        }
        // If we have a handler installed, this will push the events to the app
        // instead of the app polling for them.
        SDL.flushEventsToHandler();
        return;
      },lookupKeyCodeForEvent:function (event) {
          var code = event.keyCode;
          if (code >= 65 && code <= 90) {
            code += 32; // make lowercase for SDL
          } else {
            code = SDL.keyCodes[event.keyCode] || event.keyCode;
            // If this is one of the modifier keys (224 | 1<<10 - 227 | 1<<10), and the event specifies that it is
            // a right key, add 4 to get the right key SDL key code.
            if (event.location === KeyboardEvent.DOM_KEY_LOCATION_RIGHT && code >= (224 | 1<<10) && code <= (227 | 1<<10)) {
              code += 4;
            }
          }
          return code;
      },handleEvent:function (event) {
        if (event.handled) return;
        event.handled = true;

        switch (event.type) {
          case 'touchstart': case 'touchend': case 'touchmove': {
            Browser.calculateMouseEvent(event);
            break;
          }
          case 'keydown': case 'keyup': {
            var down = event.type === 'keydown';
            var code = SDL.lookupKeyCodeForEvent(event);
            HEAP8[(((SDL.keyboardState)+(code))>>0)]=down;
            // TODO: lmeta, rmeta, numlock, capslock, KMOD_MODE, KMOD_RESERVED
            SDL.modState = (HEAP8[(((SDL.keyboardState)+(1248))>>0)] ? 0x0040 : 0) | // KMOD_LCTRL
              (HEAP8[(((SDL.keyboardState)+(1249))>>0)] ? 0x0001 : 0) | // KMOD_LSHIFT
              (HEAP8[(((SDL.keyboardState)+(1250))>>0)] ? 0x0100 : 0) | // KMOD_LALT
              (HEAP8[(((SDL.keyboardState)+(1252))>>0)] ? 0x0080 : 0) | // KMOD_RCTRL
              (HEAP8[(((SDL.keyboardState)+(1253))>>0)] ? 0x0002 : 0) | // KMOD_RSHIFT
              (HEAP8[(((SDL.keyboardState)+(1254))>>0)] ? 0x0200 : 0); //  KMOD_RALT
            if (down) {
              SDL.keyboardMap[code] = event.keyCode; // save the DOM input, which we can use to unpress it during blur
            } else {
              delete SDL.keyboardMap[code];
            }

            break;
          }
          case 'mousedown': case 'mouseup':
            if (event.type == 'mousedown') {
              // SDL_BUTTON(x) is defined as (1 << ((x)-1)).  SDL buttons are 1-3,
              // and DOM buttons are 0-2, so this means that the below formula is
              // correct.
              SDL.buttonState |= 1 << event.button;
            } else if (event.type == 'mouseup') {
              SDL.buttonState &= ~(1 << event.button);
            }
            // fall through
          case 'mousemove': {
            Browser.calculateMouseEvent(event);
            break;
          }
        }
      },flushEventsToHandler:function () {
        if (!SDL.eventHandler) return;

        while (SDL.pollEvent(SDL.eventHandlerTemp)) {
          Module['dynCall_iii'](SDL.eventHandler, SDL.eventHandlerContext, SDL.eventHandlerTemp);
        }
      },pollEvent:function (ptr) {
        if (SDL.initFlags & 0x200 && SDL.joystickEventState) {
          // If SDL_INIT_JOYSTICK was supplied AND the joystick system is configured
          // to automatically query for events, query for joystick events.
          SDL.queryJoysticks();
        }
        if (ptr) {
          while (SDL.events.length > 0) {
            if (SDL.makeCEvent(SDL.events.shift(), ptr) !== false) return 1;
          }
          return 0;
        } else {
          // XXX: somewhat risky in that we do not check if the event is real or not (makeCEvent returns false) if no pointer supplied
          return SDL.events.length > 0;
        }
      },makeCEvent:function (event, ptr) {
        if (typeof event === 'number') {
          // This is a pointer to a copy of a native C event that was SDL_PushEvent'ed
          _memcpy(ptr, event, 28);
          _free(event); // the copy is no longer needed
          return;
        }

        SDL.handleEvent(event);

        switch (event.type) {
          case 'keydown': case 'keyup': {
            var down = event.type === 'keydown';
            //Module.print('Received key event: ' + event.keyCode);
            var key = SDL.lookupKeyCodeForEvent(event);
            var scan;
            if (key >= 1024) {
              scan = key - 1024;
            } else {
              scan = SDL.scanCodes[key] || key;
            }

            HEAP32[((ptr)>>2)]=SDL.DOMEventToSDLEvent[event.type];
            HEAP8[(((ptr)+(8))>>0)]=down ? 1 : 0;
            HEAP8[(((ptr)+(9))>>0)]=0; // TODO
            HEAP32[(((ptr)+(12))>>2)]=scan;
            HEAP32[(((ptr)+(16))>>2)]=key;
            HEAP16[(((ptr)+(20))>>1)]=SDL.modState;
            // some non-character keys (e.g. backspace and tab) won't have keypressCharCode set, fill in with the keyCode.
            HEAP32[(((ptr)+(24))>>2)]=event.keypressCharCode || key;

            break;
          }
          case 'keypress': {
            HEAP32[((ptr)>>2)]=SDL.DOMEventToSDLEvent[event.type];
            // Not filling in windowID for now
            var cStr = intArrayFromString(String.fromCharCode(event.charCode));
            for (var i = 0; i < cStr.length; ++i) {
              HEAP8[(((ptr)+(8 + i))>>0)]=cStr[i];
            }
            break;
          }
          case 'mousedown': case 'mouseup': case 'mousemove': {
            if (event.type != 'mousemove') {
              var down = event.type === 'mousedown';
              HEAP32[((ptr)>>2)]=SDL.DOMEventToSDLEvent[event.type];
              HEAP32[(((ptr)+(4))>>2)]=0;
              HEAP32[(((ptr)+(8))>>2)]=0;
              HEAP32[(((ptr)+(12))>>2)]=0;
              HEAP8[(((ptr)+(16))>>0)]=event.button+1; // DOM buttons are 0-2, SDL 1-3
              HEAP8[(((ptr)+(17))>>0)]=down ? 1 : 0;
              HEAP32[(((ptr)+(20))>>2)]=Browser.mouseX;
              HEAP32[(((ptr)+(24))>>2)]=Browser.mouseY;
            } else {
              HEAP32[((ptr)>>2)]=SDL.DOMEventToSDLEvent[event.type];
              HEAP32[(((ptr)+(4))>>2)]=0;
              HEAP32[(((ptr)+(8))>>2)]=0;
              HEAP32[(((ptr)+(12))>>2)]=0;
              HEAP32[(((ptr)+(16))>>2)]=SDL.buttonState;
              HEAP32[(((ptr)+(20))>>2)]=Browser.mouseX;
              HEAP32[(((ptr)+(24))>>2)]=Browser.mouseY;
              HEAP32[(((ptr)+(28))>>2)]=Browser.mouseMovementX;
              HEAP32[(((ptr)+(32))>>2)]=Browser.mouseMovementY;
            }
            break;
          }
          case 'wheel': {
            HEAP32[((ptr)>>2)]=SDL.DOMEventToSDLEvent[event.type];
            HEAP32[(((ptr)+(16))>>2)]=event.deltaX;
            HEAP32[(((ptr)+(20))>>2)]=event.deltaY;
            break;
          }
          case 'touchstart': case 'touchend': case 'touchmove': {
            var touch = event.touch;
            if (!Browser.touches[touch.identifier]) break;
            var w = Module['canvas'].width;
            var h = Module['canvas'].height;
            var x = Browser.touches[touch.identifier].x / w;
            var y = Browser.touches[touch.identifier].y / h;
            var lx = Browser.lastTouches[touch.identifier].x / w;
            var ly = Browser.lastTouches[touch.identifier].y / h;
            var dx = x - lx;
            var dy = y - ly;
            if (touch['deviceID'] === undefined) touch.deviceID = SDL.TOUCH_DEFAULT_ID;
            if (dx === 0 && dy === 0 && event.type === 'touchmove') return false; // don't send these if nothing happened
            HEAP32[((ptr)>>2)]=SDL.DOMEventToSDLEvent[event.type];
            HEAP32[(((ptr)+(4))>>2)]=_SDL_GetTicks();
            (tempI64 = [touch.deviceID>>>0,(tempDouble=touch.deviceID,(+(Math_abs(tempDouble))) >= (+1) ? (tempDouble > (+0) ? ((Math_min((+(Math_floor((tempDouble)/(+4294967296)))), (+4294967295)))|0)>>>0 : (~~((+(Math_ceil((tempDouble - +(((~~(tempDouble)))>>>0))/(+4294967296))))))>>>0) : 0)],HEAP32[(((ptr)+(8))>>2)]=tempI64[0],HEAP32[(((ptr)+(12))>>2)]=tempI64[1]);
            (tempI64 = [touch.identifier>>>0,(tempDouble=touch.identifier,(+(Math_abs(tempDouble))) >= (+1) ? (tempDouble > (+0) ? ((Math_min((+(Math_floor((tempDouble)/(+4294967296)))), (+4294967295)))|0)>>>0 : (~~((+(Math_ceil((tempDouble - +(((~~(tempDouble)))>>>0))/(+4294967296))))))>>>0) : 0)],HEAP32[(((ptr)+(16))>>2)]=tempI64[0],HEAP32[(((ptr)+(20))>>2)]=tempI64[1]);
            HEAPF32[(((ptr)+(24))>>2)]=x;
            HEAPF32[(((ptr)+(28))>>2)]=y;
            HEAPF32[(((ptr)+(32))>>2)]=dx;
            HEAPF32[(((ptr)+(36))>>2)]=dy;
            if (touch.force !== undefined) {
              HEAPF32[(((ptr)+(40))>>2)]=touch.force;
            } else { // No pressure data, send a digital 0/1 pressure.
              HEAPF32[(((ptr)+(40))>>2)]=event.type == "touchend" ? 0 : 1;
            }
            break;
          }
          case 'unload': {
            HEAP32[((ptr)>>2)]=SDL.DOMEventToSDLEvent[event.type];
            break;
          }
          case 'resize': {
            HEAP32[((ptr)>>2)]=SDL.DOMEventToSDLEvent[event.type];
            HEAP32[(((ptr)+(4))>>2)]=event.w;
            HEAP32[(((ptr)+(8))>>2)]=event.h;
            break;
          }
          case 'joystick_button_up': case 'joystick_button_down': {
            var state = event.type === 'joystick_button_up' ? 0 : 1;
            HEAP32[((ptr)>>2)]=SDL.DOMEventToSDLEvent[event.type];
            HEAP8[(((ptr)+(4))>>0)]=event.index;
            HEAP8[(((ptr)+(5))>>0)]=event.button;
            HEAP8[(((ptr)+(6))>>0)]=state;
            break;
          }
          case 'joystick_axis_motion': {
            HEAP32[((ptr)>>2)]=SDL.DOMEventToSDLEvent[event.type];
            HEAP8[(((ptr)+(4))>>0)]=event.index;
            HEAP8[(((ptr)+(5))>>0)]=event.axis;
            HEAP32[(((ptr)+(8))>>2)]=SDL.joystickAxisValueConversion(event.value);
            break;
          }
          case 'focus': {
            var SDL_WINDOWEVENT_FOCUS_GAINED = 12 /* SDL_WINDOWEVENT_FOCUS_GAINED */;
            HEAP32[((ptr)>>2)]=SDL.DOMEventToSDLEvent[event.type];
            HEAP32[(((ptr)+(4))>>2)]=0;
            HEAP8[(((ptr)+(8))>>0)]=SDL_WINDOWEVENT_FOCUS_GAINED;
            break;
          }
          case 'blur': {
            var SDL_WINDOWEVENT_FOCUS_LOST = 13 /* SDL_WINDOWEVENT_FOCUS_LOST */;
            HEAP32[((ptr)>>2)]=SDL.DOMEventToSDLEvent[event.type];
            HEAP32[(((ptr)+(4))>>2)]=0;
            HEAP8[(((ptr)+(8))>>0)]=SDL_WINDOWEVENT_FOCUS_LOST;
            break;
          }
          case 'visibilitychange': {
            var SDL_WINDOWEVENT_SHOWN  = 1 /* SDL_WINDOWEVENT_SHOWN */;
            var SDL_WINDOWEVENT_HIDDEN = 2 /* SDL_WINDOWEVENT_HIDDEN */;
            var visibilityEventID = event.visible ? SDL_WINDOWEVENT_SHOWN : SDL_WINDOWEVENT_HIDDEN;
            HEAP32[((ptr)>>2)]=SDL.DOMEventToSDLEvent[event.type];
            HEAP32[(((ptr)+(4))>>2)]=0;
            HEAP8[(((ptr)+(8))>>0)]=visibilityEventID;
            break;
          }
          default: throw 'Unhandled SDL event: ' + event.type;
        }
      },makeFontString:function (height, fontName) {
        if (fontName.charAt(0) != "'" && fontName.charAt(0) != '"') {
          // https://developer.mozilla.org/ru/docs/Web/CSS/font-family
          // Font family names containing whitespace should be quoted.
          // BTW, quote all font names is easier than searching spaces
          fontName = '"' + fontName + '"';
        }
        return height + 'px ' + fontName + ', serif';
      },estimateTextWidth:function (fontData, text) {
        var h = fontData.size;
        var fontString = SDL.makeFontString(h, fontData.name);
        var tempCtx = SDL.ttfContext;
        tempCtx.save();
        tempCtx.font = fontString;
        var ret = tempCtx.measureText(text).width | 0;
        tempCtx.restore();
        return ret;
      },allocateChannels:function (num) { // called from Mix_AllocateChannels and init
        if (SDL.numChannels && SDL.numChannels >= num && num != 0) return;
        SDL.numChannels = num;
        SDL.channels = [];
        for (var i = 0; i < num; i++) {
          SDL.channels[i] = {
            audio: null,
            volume: 1.0
          };
        }
      },setGetVolume:function (info, volume) {
        if (!info) return 0;
        var ret = info.volume * 128; // MIX_MAX_VOLUME
        if (volume != -1) {
          info.volume = Math.min(Math.max(volume, 0), 128) / 128;
          if (info.audio) {
            try {
              info.audio.volume = info.volume; // For <audio> element
              if (info.audio.webAudioGainNode) info.audio.webAudioGainNode['gain']['value'] = info.volume; // For WebAudio playback
            } catch(e) {
              Module.printErr('setGetVolume failed to set audio volume: ' + e);
            }
          }
        }
        return ret;
      },setPannerPosition:function (info, x, y, z) {
        if (!info) return;
        if (info.audio) {
          if (info.audio.webAudioPannerNode) {
            info.audio.webAudioPannerNode['setPosition'](x, y, z);
          }
        }
      },playWebAudio:function (audio) {
        if (!audio) return;
        if (audio.webAudioNode) return; // This instance is already playing, don't start again.
        if (!SDL.webAudioAvailable()) return;
        try {
          var webAudio = audio.resource.webAudio;
          audio.paused = false;
          if (!webAudio.decodedBuffer) {
            if (webAudio.onDecodeComplete === undefined) abort("Cannot play back audio object that was not loaded");
            webAudio.onDecodeComplete.push(function() { if (!audio.paused) SDL.playWebAudio(audio); });
            return;
          }
          audio.webAudioNode = SDL.audioContext['createBufferSource']();
          audio.webAudioNode['buffer'] = webAudio.decodedBuffer;
          audio.webAudioNode['loop'] = audio.loop;
          audio.webAudioNode['onended'] = function() { audio['onended'](); } // For <media> element compatibility, route the onended signal to the instance.

          audio.webAudioPannerNode = SDL.audioContext['createPanner']();
          // avoid Chrome bug
          // If posz = 0, the sound will come from only the right.
          // By posz = -0.5 (slightly ahead), the sound will come from right and left correctly.
          audio.webAudioPannerNode["setPosition"](0, 0, -.5);
          audio.webAudioPannerNode['panningModel'] = 'equalpower';

          // Add an intermediate gain node to control volume.
          audio.webAudioGainNode = SDL.audioContext['createGain']();
          audio.webAudioGainNode['gain']['value'] = audio.volume;

          audio.webAudioNode['connect'](audio.webAudioPannerNode);
          audio.webAudioPannerNode['connect'](audio.webAudioGainNode);
          audio.webAudioGainNode['connect'](SDL.audioContext['destination']);

          audio.webAudioNode['start'](0, audio.currentPosition);
          audio.startTime = SDL.audioContext['currentTime'] - audio.currentPosition;
        } catch(e) {
          Module.printErr('playWebAudio failed: ' + e);
        }
      },pauseWebAudio:function (audio) {
        if (!audio) return;
        if (audio.webAudioNode) {
          try {
            // Remember where we left off, so that if/when we resume, we can restart the playback at a proper place.
            audio.currentPosition = (SDL.audioContext['currentTime'] - audio.startTime) % audio.resource.webAudio.decodedBuffer.duration;
            // Important: When we reach here, the audio playback is stopped by the user. But when calling .stop() below, the Web Audio
            // graph will send the onended signal, but we don't want to process that, since pausing should not clear/destroy the audio
            // channel.
            audio.webAudioNode['onended'] = undefined;
            audio.webAudioNode.stop(0); // 0 is a default parameter, but WebKit is confused by it #3861
            audio.webAudioNode = undefined;
          } catch(e) {
            Module.printErr('pauseWebAudio failed: ' + e);
          }
        }
        audio.paused = true;
      },openAudioContext:function () {
        // Initialize Web Audio API if we haven't done so yet. Note: Only initialize Web Audio context ever once on the web page,
        // since initializing multiple times fails on Chrome saying 'audio resources have been exhausted'.
        if (!SDL.audioContext) {
          if (typeof(AudioContext) !== 'undefined') SDL.audioContext = new AudioContext();
          else if (typeof(webkitAudioContext) !== 'undefined') SDL.audioContext = new webkitAudioContext();
        }
      },webAudioAvailable:function () { return !!SDL.audioContext; },fillWebAudioBufferFromHeap:function (heapPtr, sizeSamplesPerChannel, dstAudioBuffer) {
        // The input audio data is interleaved across the channels, i.e. [L, R, L, R, L, R, ...] and is either 8-bit or 16-bit as
        // supported by the SDL API. The output audio wave data for Web Audio API must be in planar buffers of [-1,1]-normalized Float32 data,
        // so perform a buffer conversion for the data.
        var numChannels = SDL.audio.channels;
        for(var c = 0; c < numChannels; ++c) {
          var channelData = dstAudioBuffer['getChannelData'](c);
          if (channelData.length != sizeSamplesPerChannel) {
            throw 'Web Audio output buffer length mismatch! Destination size: ' + channelData.length + ' samples vs expected ' + sizeSamplesPerChannel + ' samples!';
          }
          if (SDL.audio.format == 0x8010 /*AUDIO_S16LSB*/) {
            for(var j = 0; j < sizeSamplesPerChannel; ++j) {
              channelData[j] = (HEAP16[(((heapPtr)+((j*numChannels + c)*2))>>1)]) / 0x8000;
            }
          } else if (SDL.audio.format == 0x0008 /*AUDIO_U8*/) {
            for(var j = 0; j < sizeSamplesPerChannel; ++j) {
              var v = (HEAP8[(((heapPtr)+(j*numChannels + c))>>0)]);
              channelData[j] = ((v >= 0) ? v-128 : v+128) /128;
            }
          }
        }
      },debugSurface:function (surfData) {
        console.log('dumping surface ' + [surfData.surf, surfData.source, surfData.width, surfData.height]);
        var image = surfData.ctx.getImageData(0, 0, surfData.width, surfData.height);
        var data = image.data;
        var num = Math.min(surfData.width, surfData.height);
        for (var i = 0; i < num; i++) {
          console.log('   diagonal ' + i + ':' + [data[i*surfData.width*4 + i*4 + 0], data[i*surfData.width*4 + i*4 + 1], data[i*surfData.width*4 + i*4 + 2], data[i*surfData.width*4 + i*4 + 3]]);
        }
      },joystickEventState:1,lastJoystickState:{},joystickNamePool:{},recordJoystickState:function (joystick, state) {
        // Standardize button state.
        var buttons = new Array(state.buttons.length);
        for (var i = 0; i < state.buttons.length; i++) {
          buttons[i] = SDL.getJoystickButtonState(state.buttons[i]);
        }

        SDL.lastJoystickState[joystick] = {
          buttons: buttons,
          axes: state.axes.slice(0),
          timestamp: state.timestamp,
          index: state.index,
          id: state.id
        };
      },getJoystickButtonState:function (button) {
        if (typeof button === 'object') {
          // Current gamepad API editor's draft (Firefox Nightly)
          // https://dvcs.w3.org/hg/gamepad/raw-file/default/gamepad.html#idl-def-GamepadButton
          return button['pressed'];
        } else {
          // Current gamepad API working draft (Firefox / Chrome Stable)
          // http://www.w3.org/TR/2012/WD-gamepad-20120529/#gamepad-interface
          return button > 0;
        }
      },queryJoysticks:function () {
        for (var joystick in SDL.lastJoystickState) {
          var state = SDL.getGamepad(joystick - 1);
          var prevState = SDL.lastJoystickState[joystick];
          // If joystick was removed, state returns null.
          if (typeof state === 'undefined') return;
          // Check only if the timestamp has differed.
          // NOTE: Timestamp is not available in Firefox.
          if (typeof state.timestamp !== 'number' || state.timestamp !== prevState.timestamp) {
            var i;
            for (i = 0; i < state.buttons.length; i++) {
              var buttonState = SDL.getJoystickButtonState(state.buttons[i]);
              // NOTE: The previous state already has a boolean representation of
              //       its button, so no need to standardize its button state here.
              if (buttonState !== prevState.buttons[i]) {
                // Insert button-press event.
                SDL.events.push({
                  type: buttonState ? 'joystick_button_down' : 'joystick_button_up',
                  joystick: joystick,
                  index: joystick - 1,
                  button: i
                });
              }
            }
            for (i = 0; i < state.axes.length; i++) {
              if (state.axes[i] !== prevState.axes[i]) {
                // Insert axes-change event.
                SDL.events.push({
                  type: 'joystick_axis_motion',
                  joystick: joystick,
                  index: joystick - 1,
                  axis: i,
                  value: state.axes[i]
                });
              }
            }

            SDL.recordJoystickState(joystick, state);
          }
        }
      },joystickAxisValueConversion:function (value) {
        // Make sure value is properly clamped
        value = Math.min(1, Math.max(value, -1));
        // Ensures that 0 is 0, 1 is 32767, and -1 is 32768.
        return Math.ceil(((value+1) * 32767.5) - 32768);
      },getGamepads:function () {
        var fcn = navigator.getGamepads || navigator.webkitGamepads || navigator.mozGamepads || navigator.gamepads || navigator.webkitGetGamepads;
        if (fcn !== undefined) {
          // The function must be applied on the navigator object.
          return fcn.apply(navigator);
        } else {
          return [];
        }
      },getGamepad:function (deviceIndex) {
        var gamepads = SDL.getGamepads();
        if (gamepads.length > deviceIndex && deviceIndex >= 0) {
          return gamepads[deviceIndex];
        }
        return null;
      }};function _SDL_SetVideoMode(width, height, depth, flags) {
      ['touchstart', 'touchend', 'touchmove', 'mousedown', 'mouseup', 'mousemove', 'DOMMouseScroll', 'mousewheel', 'wheel', 'mouseout'].forEach(function(event) {
        Module['canvas'].addEventListener(event, SDL.receiveEvent, true);
      });

      var canvas = Module['canvas'];

      // (0,0) means 'use fullscreen' in native; in Emscripten, use the current canvas size.
      if (width == 0 && height == 0) {
        width = canvas.width;
        height = canvas.height;
      }

      if (!SDL.addedResizeListener) {
        SDL.addedResizeListener = true;
        Browser.resizeListeners.push(function(w, h) {
          if (!SDL.settingVideoMode) {
            SDL.receiveEvent({
              type: 'resize',
              w: w,
              h: h
            });
          }
        });
      }

      if (width !== canvas.width || height !== canvas.height) {
        SDL.settingVideoMode = true; // SetVideoMode itself should not trigger resize events
        Browser.setCanvasSize(width, height);
        SDL.settingVideoMode = false;
      }

      // Free the old surface first if there is one
      if (SDL.screen) {
        SDL.freeSurface(SDL.screen);
        assert(!SDL.screen);
      }

      if (SDL.GL) flags = flags | 0x04000000; // SDL_OPENGL - if we are using GL, then later calls to SetVideoMode may not mention GL, but we do need it. Once in GL mode, we never leave it.

      SDL.screen = SDL.makeSurface(width, height, flags, true, 'screen');

      return SDL.screen;
    }




  var cttz_i8 = allocate([8,0,1,0,2,0,1,0,3,0,1,0,2,0,1,0,4,0,1,0,2,0,1,0,3,0,1,0,2,0,1,0,5,0,1,0,2,0,1,0,3,0,1,0,2,0,1,0,4,0,1,0,2,0,1,0,3,0,1,0,2,0,1,0,6,0,1,0,2,0,1,0,3,0,1,0,2,0,1,0,4,0,1,0,2,0,1,0,3,0,1,0,2,0,1,0,5,0,1,0,2,0,1,0,3,0,1,0,2,0,1,0,4,0,1,0,2,0,1,0,3,0,1,0,2,0,1,0,7,0,1,0,2,0,1,0,3,0,1,0,2,0,1,0,4,0,1,0,2,0,1,0,3,0,1,0,2,0,1,0,5,0,1,0,2,0,1,0,3,0,1,0,2,0,1,0,4,0,1,0,2,0,1,0,3,0,1,0,2,0,1,0,6,0,1,0,2,0,1,0,3,0,1,0,2,0,1,0,4,0,1,0,2,0,1,0,3,0,1,0,2,0,1,0,5,0,1,0,2,0,1,0,3,0,1,0,2,0,1,0,4,0,1,0,2,0,1,0,3,0,1,0,2,0,1,0], "i8", ALLOC_STATIC);

  function _usleep(useconds) {
      // int usleep(useconds_t useconds);
      // http://pubs.opengroup.org/onlinepubs/000095399/functions/usleep.html
      // We're single-threaded, so use a busy loop. Super-ugly.
      var msec = useconds / 1000;
      if ((ENVIRONMENT_IS_WEB || ENVIRONMENT_IS_WORKER) && self['performance'] && self['performance']['now']) {
        var start = self['performance']['now']();
        while (self['performance']['now']() - start < msec) {
          // Do nothing.
        }
      } else {
        var start = Date.now();
        while (Date.now() - start < msec) {
          // Do nothing.
        }
      }
      return 0;
    }
  Module["_usleep"] = _usleep;

  function _SDL_ShowCursor(toggle) {
      switch (toggle) {
        case 0: // SDL_DISABLE
          if (Browser.isFullscreen) { // only try to lock the pointer when in full screen mode
            Module['canvas'].requestPointerLock();
            return 0;
          } else { // else return SDL_ENABLE to indicate the failure
            return 1;
          }
          break;
        case 1: // SDL_ENABLE
          Module['canvas'].exitPointerLock();
          return 1;
          break;
        case -1: // SDL_QUERY
          return !Browser.pointerLock;
          break;
        default:
          console.log( "SDL_ShowCursor called with unknown toggle parameter value: " + toggle + "." );
          break;
      }
    }

  function _SDL_UnlockSurface(surf) {
      assert(!SDL.GL); // in GL mode we do not keep around 2D canvases and contexts

      var surfData = SDL.surfaces[surf];

      if (!surfData.locked || --surfData.locked > 0) {
        return;
      }

      // Copy pixel data to image
      if (surfData.isFlagSet(0x00200000 /* SDL_HWPALETTE */)) {
        SDL.copyIndexedColorData(surfData);
      } else if (!surfData.colors) {
        var data = surfData.image.data;
        var buffer = surfData.buffer;
        assert(buffer % 4 == 0, 'Invalid buffer offset: ' + buffer);
        var src = buffer >> 2;
        var dst = 0;
        var isScreen = surf == SDL.screen;
        var num;
        if (typeof CanvasPixelArray !== 'undefined' && data instanceof CanvasPixelArray) {
          // IE10/IE11: ImageData objects are backed by the deprecated CanvasPixelArray,
          // not UInt8ClampedArray. These don't have buffers, so we need to revert
          // to copying a byte at a time. We do the undefined check because modern
          // browsers do not define CanvasPixelArray anymore.
          num = data.length;
          while (dst < num) {
            var val = HEAP32[src]; // This is optimized. Instead, we could do HEAP32[(((buffer)+(dst))>>2)];
            data[dst  ] = val & 0xff;
            data[dst+1] = (val >> 8) & 0xff;
            data[dst+2] = (val >> 16) & 0xff;
            data[dst+3] = isScreen ? 0xff : ((val >> 24) & 0xff);
            src++;
            dst += 4;
          }
        } else {
          var data32 = new Uint32Array(data.buffer);
          if (isScreen && SDL.defaults.opaqueFrontBuffer) {
            num = data32.length;
            // logically we need to do
            //      while (dst < num) {
            //          data32[dst++] = HEAP32[src++] | 0xff000000
            //      }
            // the following code is faster though, because
            // .set() is almost free - easily 10x faster due to
            // native memcpy efficiencies, and the remaining loop
            // just stores, not load + store, so it is faster
            data32.set(HEAP32.subarray(src, src + num));
            var data8 = new Uint8Array(data.buffer);
            var i = 3;
            var j = i + 4*num;
            if (num % 8 == 0) {
              // unrolling gives big speedups
              while (i < j) {
                data8[i] = 0xff;
                i = i + 4 | 0;
                data8[i] = 0xff;
                i = i + 4 | 0;
                data8[i] = 0xff;
                i = i + 4 | 0;
                data8[i] = 0xff;
                i = i + 4 | 0;
                data8[i] = 0xff;
                i = i + 4 | 0;
                data8[i] = 0xff;
                i = i + 4 | 0;
                data8[i] = 0xff;
                i = i + 4 | 0;
                data8[i] = 0xff;
                i = i + 4 | 0;
              }
             } else {
              while (i < j) {
                data8[i] = 0xff;
                i = i + 4 | 0;
              }
            }
          } else {
            data32.set(HEAP32.subarray(src, src + data32.length));
          }
        }
      } else {
        var width = Module['canvas'].width;
        var height = Module['canvas'].height;
        var s = surfData.buffer;
        var data = surfData.image.data;
        var colors = surfData.colors; // TODO: optimize using colors32
        for (var y = 0; y < height; y++) {
          var base = y*width*4;
          for (var x = 0; x < width; x++) {
            // See comment above about signs
            var val = HEAPU8[((s++)>>0)] * 4;
            var start = base + x*4;
            data[start]   = colors[val];
            data[start+1] = colors[val+1];
            data[start+2] = colors[val+2];
          }
          s += width*3;
        }
      }
      // Copy to canvas
      surfData.ctx.putImageData(surfData.image, 0, 0);
      // Note that we save the image, so future writes are fast. But, memory is not yet released
    }

  function _SDL_GetVideoInfo() {
      // %struct.SDL_VideoInfo = type { i32, i32, %struct.SDL_PixelFormat*, i32, i32 } - 5 fields of quantum size
      var ret = _malloc(5*Runtime.QUANTUM_SIZE);
      HEAP32[((ret+Runtime.QUANTUM_SIZE*0)>>2)]=0; // TODO
      HEAP32[((ret+Runtime.QUANTUM_SIZE*1)>>2)]=0; // TODO
      HEAP32[((ret+Runtime.QUANTUM_SIZE*2)>>2)]=0;
      HEAP32[((ret+Runtime.QUANTUM_SIZE*3)>>2)]=Module["canvas"].width;
      HEAP32[((ret+Runtime.QUANTUM_SIZE*4)>>2)]=Module["canvas"].height;
      return ret;
    }

  function _sem_init() {}

  function ___syscall54(which, varargs) {SYSCALLS.varargs = varargs;
  try {
   // ioctl
      var stream = SYSCALLS.getStreamFromFD(), op = SYSCALLS.get();
      switch (op) {
        case 21505: {
          if (!stream.tty) return -ERRNO_CODES.ENOTTY;
          return 0;
        }
        case 21506: {
          if (!stream.tty) return -ERRNO_CODES.ENOTTY;
          return 0; // no-op, not actually adjusting terminal settings
        }
        case 21519: {
          if (!stream.tty) return -ERRNO_CODES.ENOTTY;
          var argp = SYSCALLS.get();
          HEAP32[((argp)>>2)]=0;
          return 0;
        }
        case 21520: {
          if (!stream.tty) return -ERRNO_CODES.ENOTTY;
          return -ERRNO_CODES.EINVAL; // not supported
        }
        case 21531: {
          var argp = SYSCALLS.get();
          return FS.ioctl(stream, op, argp);
        }
        case 21523: {
          // TODO: in theory we should write to the winsize struct that gets
          // passed in, but for now musl doesn't read anything on it
          if (!stream.tty) return -ERRNO_CODES.ENOTTY;
          return 0;
        }
        default: abort('bad ioctl syscall ' + op);
      }
    } catch (e) {
    if (typeof FS === 'undefined' || !(e instanceof FS.ErrnoError)) abort(e);
    return -e.errno;
  }
  }


  var ___tm_current=STATICTOP; STATICTOP += 48;;


  var ___tm_timezone=allocate(intArrayFromString("GMT"), "i8", ALLOC_STATIC);function _localtime_r(time, tmPtr) {
      _tzset();
      var date = new Date(HEAP32[((time)>>2)]*1000);
      HEAP32[((tmPtr)>>2)]=date.getSeconds();
      HEAP32[(((tmPtr)+(4))>>2)]=date.getMinutes();
      HEAP32[(((tmPtr)+(8))>>2)]=date.getHours();
      HEAP32[(((tmPtr)+(12))>>2)]=date.getDate();
      HEAP32[(((tmPtr)+(16))>>2)]=date.getMonth();
      HEAP32[(((tmPtr)+(20))>>2)]=date.getFullYear()-1900;
      HEAP32[(((tmPtr)+(24))>>2)]=date.getDay();

      var start = new Date(date.getFullYear(), 0, 1);
      var yday = ((date.getTime() - start.getTime()) / (1000 * 60 * 60 * 24))|0;
      HEAP32[(((tmPtr)+(28))>>2)]=yday;
      HEAP32[(((tmPtr)+(36))>>2)]=-(date.getTimezoneOffset() * 60);

      // DST is in December in South
      var summerOffset = new Date(2000, 6, 1).getTimezoneOffset();
      var winterOffset = start.getTimezoneOffset();
      var dst = (date.getTimezoneOffset() == Math.min(winterOffset, summerOffset))|0;
      HEAP32[(((tmPtr)+(32))>>2)]=dst;

      var zonePtr = HEAP32[(((_tzname)+(dst ? Runtime.QUANTUM_SIZE : 0))>>2)];
      HEAP32[(((tmPtr)+(40))>>2)]=zonePtr;

      return tmPtr;
    }function _localtime(time) {
      return _localtime_r(time, ___tm_current);
    }



  function ___syscall38(which, varargs) {SYSCALLS.varargs = varargs;
  try {
   // rename
      var old_path = SYSCALLS.getStr(), new_path = SYSCALLS.getStr();
      FS.rename(old_path, new_path);
      return 0;
    } catch (e) {
    if (typeof FS === 'undefined' || !(e instanceof FS.ErrnoError)) abort(e);
    return -e.errno;
  }
  }


  function _llvm_exp2_f32(x) {
      return Math.pow(2, x);
    }function _llvm_exp2_f64() {
  return _llvm_exp2_f32.apply(null, arguments)
  }

  function ___syscall33(which, varargs) {SYSCALLS.varargs = varargs;
  try {
   // access
      var path = SYSCALLS.getStr(), amode = SYSCALLS.get();
      return SYSCALLS.doAccess(path, amode);
    } catch (e) {
    if (typeof FS === 'undefined' || !(e instanceof FS.ErrnoError)) abort(e);
    return -e.errno;
  }
  }


  function _gettimeofday(ptr) {
      var now = Date.now();
      HEAP32[((ptr)>>2)]=(now/1000)|0; // seconds
      HEAP32[(((ptr)+(4))>>2)]=((now % 1000)*1000)|0; // microseconds
      return 0;
    }

  function _SDL_PumpEvents(){
      SDL.events.forEach(function(event) {
        SDL.handleEvent(event);
      });
    }

  function _SDL_WM_SetCaption(title, icon) {
      if (title && typeof Module['setWindowTitle'] !== 'undefined') {
        Module['setWindowTitle'](Pointer_stringify(title));
      }
      icon = icon && Pointer_stringify(icon);
    }


  function _emscripten_memcpy_big(dest, src, num) {
      HEAPU8.set(HEAPU8.subarray(src, src+num), dest);
      return dest;
    }

  function _SDL_mutexV() { return 0 }

  function _SDL_mutexP() { return 0 }

  var _llvm_pow_f64=Math_pow;







  function ___syscall122(which, varargs) {SYSCALLS.varargs = varargs;
  try {
   // uname
      var buf = SYSCALLS.get();
      if (!buf) return -ERRNO_CODES.EFAULT
      var layout = {"sysname":0,"nodename":65,"domainname":325,"machine":260,"version":195,"release":130,"__size__":390};
      function copyString(element, value) {
        var offset = layout[element];
        writeAsciiToMemory(value, buf + offset);
      }
      copyString('sysname', 'Emscripten');
      copyString('nodename', 'emscripten');
      copyString('release', '1.0');
      copyString('version', '#1');
      copyString('machine', 'x86-JS');
      return 0;
    } catch (e) {
    if (typeof FS === 'undefined' || !(e instanceof FS.ErrnoError)) abort(e);
    return -e.errno;
  }
  }



  function ___syscall40(which, varargs) {SYSCALLS.varargs = varargs;
  try {
   // rmdir
      var path = SYSCALLS.getStr();
      FS.rmdir(path);
      return 0;
    } catch (e) {
    if (typeof FS === 'undefined' || !(e instanceof FS.ErrnoError)) abort(e);
    return -e.errno;
  }
  }

  function _SDL_QuitSubSystem(flags) {
      Module.print('SDL_QuitSubSystem called (and ignored)');
    }

  function ___assert_fail(condition, filename, line, func) {
      ABORT = true;
      throw 'Assertion failed: ' + Pointer_stringify(condition) + ', at: ' + [filename ? Pointer_stringify(filename) : 'unknown filename', line, func ? Pointer_stringify(func) : 'unknown function'] + ' at ' + stackTrace();
    }

  function _SDL_MapRGB(fmt, r, g, b) {
      SDL.checkPixelFormat(fmt);
      // We assume the machine is little-endian.
      return r&0xff|(g&0xff)<<8|(b&0xff)<<16|0xff000000;
    }



  function ___syscall39(which, varargs) {SYSCALLS.varargs = varargs;
  try {
   // mkdir
      var path = SYSCALLS.getStr(), mode = SYSCALLS.get();
      return SYSCALLS.doMkdir(path, mode);
    } catch (e) {
    if (typeof FS === 'undefined' || !(e instanceof FS.ErrnoError)) abort(e);
    return -e.errno;
  }
  }



  function _abort() {
      Module['abort']();
    }



  function ___lock() {}

  function ___unlock() {}

  function _SDL_VideoModeOK(width, height, depth, flags) {
      // SDL_VideoModeOK returns 0 if the requested mode is not supported under any bit depth, or returns the
      // bits-per-pixel of the closest available mode with the given width, height and requested surface flags
      return depth; // all modes are ok.
    }

  var _emscripten_asm_const_int=true;


  function __exit(status) {
      // void _exit(int status);
      // http://pubs.opengroup.org/onlinepubs/000095399/functions/exit.html
      Module['exit'](status);
    }function _exit(status) {
      __exit(status);
    }

  function _SDL_UpdateRect(surf, x, y, w, h) {
      // We actually do the whole screen in Unlock...
    }

  function ___cxa_pure_virtual() {
      ABORT = true;
      throw 'Pure virtual function called!';
    }

  function _SDL_FreeSurface(surf) {
      if (surf) SDL.freeSurface(surf);
    }

  function _sem_destroy() {}

  function _SDL_VideoDriverName(buf, max_size) {
      if (SDL.startTime === null) {
        return 0; //return NULL
      }
      //driverName - emscripten_sdl_driver
      var driverName = [101, 109, 115, 99, 114, 105, 112, 116, 101,
        110, 95, 115, 100, 108, 95, 100, 114, 105, 118, 101, 114];

      var index = 0;
      var size  = driverName.length;

      if (max_size <= size) {
        size = max_size - 1; //-1 cause null-terminator
      }

      while (index < size) {
          var value = driverName[index];
          HEAP8[(((buf)+(index))>>0)]=value;
          index++;
      }

      HEAP8[(((buf)+(index))>>0)]=0;
      return buf;
    }

  function ___syscall10(which, varargs) {SYSCALLS.varargs = varargs;
  try {
   // unlink
      var path = SYSCALLS.getStr();
      FS.unlink(path);
      return 0;
    } catch (e) {
    if (typeof FS === 'undefined' || !(e instanceof FS.ErrnoError)) abort(e);
    return -e.errno;
  }
  }



  function __inet_pton4_raw(str) {
      var b = str.split('.');
      for (var i = 0; i < 4; i++) {
        var tmp = Number(b[i]);
        if (isNaN(tmp)) return null;
        b[i] = tmp;
      }
      return (b[0] | (b[1] << 8) | (b[2] << 16) | (b[3] << 24)) >>> 0;
    }

  function __inet_pton6_raw(str) {
      var words;
      var w, offset, z, i;
      /* http://home.deds.nl/~aeron/regex/ */
      var valid6regx = /^((?=.*::)(?!.*::.+::)(::)?([\dA-F]{1,4}:(:|\b)|){5}|([\dA-F]{1,4}:){6})((([\dA-F]{1,4}((?!\3)::|:\b|$))|(?!\2\3)){2}|(((2[0-4]|1\d|[1-9])?\d|25[0-5])\.?\b){4})$/i
      var parts = [];
      if (!valid6regx.test(str)) {
        return null;
      }
      if (str === "::") {
        return [0, 0, 0, 0, 0, 0, 0, 0];
      }
      // Z placeholder to keep track of zeros when splitting the string on ":"
      if (str.indexOf("::") === 0) {
        str = str.replace("::", "Z:"); // leading zeros case
      } else {
        str = str.replace("::", ":Z:");
      }

      if (str.indexOf(".") > 0) {
        // parse IPv4 embedded stress
        str = str.replace(new RegExp('[.]', 'g'), ":");
        words = str.split(":");
        words[words.length-4] = parseInt(words[words.length-4]) + parseInt(words[words.length-3])*256;
        words[words.length-3] = parseInt(words[words.length-2]) + parseInt(words[words.length-1])*256;
        words = words.slice(0, words.length-2);
      } else {
        words = str.split(":");
      }

      offset = 0; z = 0;
      for (w=0; w < words.length; w++) {
        if (typeof words[w] === 'string') {
          if (words[w] === 'Z') {
            // compressed zeros - write appropriate number of zero words
            for (z = 0; z < (8 - words.length+1); z++) {
              parts[w+z] = 0;
            }
            offset = z-1;
          } else {
            // parse hex to field to 16-bit value and write it in network byte-order
            parts[w+offset] = _htons(parseInt(words[w],16));
          }
        } else {
          // parsed IPv4 words
          parts[w+offset] = words[w];
        }
      }
      return [
        (parts[1] << 16) | parts[0],
        (parts[3] << 16) | parts[2],
        (parts[5] << 16) | parts[4],
        (parts[7] << 16) | parts[6]
      ];
    }var DNS={address_map:{id:1,addrs:{},names:{}},lookup_name:function (name) {
        // If the name is already a valid ipv4 / ipv6 address, don't generate a fake one.
        var res = __inet_pton4_raw(name);
        if (res !== null) {
          return name;
        }
        res = __inet_pton6_raw(name);
        if (res !== null) {
          return name;
        }

        // See if this name is already mapped.
        var addr;

        if (DNS.address_map.addrs[name]) {
          addr = DNS.address_map.addrs[name];
        } else {
          var id = DNS.address_map.id++;
          assert(id < 65535, 'exceeded max address mappings of 65535');

          addr = '172.29.' + (id & 0xff) + '.' + (id & 0xff00);

          DNS.address_map.names[addr] = name;
          DNS.address_map.addrs[name] = addr;
        }

        return addr;
      },lookup_addr:function (addr) {
        if (DNS.address_map.names[addr]) {
          return DNS.address_map.names[addr];
        }

        return null;
      }};function _gethostbyname(name) {
      name = Pointer_stringify(name);

      // generate hostent
      var ret = _malloc(20); // XXX possibly leaked, as are others here
      var nameBuf = _malloc(name.length+1);
      stringToUTF8(name, nameBuf, name.length+1);
      HEAP32[((ret)>>2)]=nameBuf;
      var aliasesBuf = _malloc(4);
      HEAP32[((aliasesBuf)>>2)]=0;
      HEAP32[(((ret)+(4))>>2)]=aliasesBuf;
      var afinet = 2;
      HEAP32[(((ret)+(8))>>2)]=afinet;
      HEAP32[(((ret)+(12))>>2)]=4;
      var addrListBuf = _malloc(12);
      HEAP32[((addrListBuf)>>2)]=addrListBuf+8;
      HEAP32[(((addrListBuf)+(4))>>2)]=0;
      HEAP32[(((addrListBuf)+(8))>>2)]=__inet_pton4_raw(DNS.lookup_name(name));
      HEAP32[(((ret)+(16))>>2)]=addrListBuf;
      return ret;
    }

  function _SDL_WM_GrabInput() {}

  function ___syscall3(which, varargs) {SYSCALLS.varargs = varargs;
  try {
   // read
      var stream = SYSCALLS.getStreamFromFD(), buf = SYSCALLS.get(), count = SYSCALLS.get();
      return FS.read(stream, HEAP8,buf, count);
    } catch (e) {
    if (typeof FS === 'undefined' || !(e instanceof FS.ErrnoError)) abort(e);
    return -e.errno;
  }
  }

  function ___syscall5(which, varargs) {SYSCALLS.varargs = varargs;
  try {
   // open
      var pathname = SYSCALLS.getStr(), flags = SYSCALLS.get(), mode = SYSCALLS.get() // optional TODO
      var stream = FS.open(pathname, flags, mode);
      return stream.fd;
    } catch (e) {
    if (typeof FS === 'undefined' || !(e instanceof FS.ErrnoError)) abort(e);
    return -e.errno;
  }
  }

  function ___syscall4(which, varargs) {SYSCALLS.varargs = varargs;
  try {
   // write
      var stream = SYSCALLS.getStreamFromFD(), buf = SYSCALLS.get(), count = SYSCALLS.get();
      return FS.write(stream, HEAP8,buf, count);
    } catch (e) {
    if (typeof FS === 'undefined' || !(e instanceof FS.ErrnoError)) abort(e);
    return -e.errno;
  }
  }

  function ___syscall6(which, varargs) {SYSCALLS.varargs = varargs;
  try {
   // close
      var stream = SYSCALLS.getStreamFromFD();
      FS.close(stream);
      return 0;
    } catch (e) {
    if (typeof FS === 'undefined' || !(e instanceof FS.ErrnoError)) abort(e);
    return -e.errno;
  }
  }







  var SOCKFS={mount:function (mount) {
        // If Module['websocket'] has already been defined (e.g. for configuring
        // the subprotocol/url) use that, if not initialise it to a new object.
        Module['websocket'] = (Module['websocket'] &&
                               ('object' === typeof Module['websocket'])) ? Module['websocket'] : {};

        // Add the Event registration mechanism to the exported websocket configuration
        // object so we can register network callbacks from native JavaScript too.
        // For more documentation see system/include/emscripten/emscripten.h
        Module['websocket']._callbacks = {};
        Module['websocket']['on'] = function(event, callback) {
  	    if ('function' === typeof callback) {
  		  this._callbacks[event] = callback;
          }
  	    return this;
        };

        Module['websocket'].emit = function(event, param) {
  	    if ('function' === typeof this._callbacks[event]) {
  		  this._callbacks[event].call(this, param);
          }
        };

        // If debug is enabled register simple default logging callbacks for each Event.

        return FS.createNode(null, '/', 16384 | 511 /* 0777 */, 0);
      },createSocket:function (family, type, protocol) {
        var streaming = type == 1;
        if (protocol) {
          assert(streaming == (protocol == 6)); // if SOCK_STREAM, must be tcp
        }

        // create our internal socket structure
        var sock = {
          family: family,
          type: type,
          protocol: protocol,
          server: null,
          error: null, // Used in getsockopt for SOL_SOCKET/SO_ERROR test
          peers: {},
          pending: [],
          recv_queue: [],
          sock_ops: SOCKFS.websocket_sock_ops
        };

        // create the filesystem node to store the socket structure
        var name = SOCKFS.nextname();
        var node = FS.createNode(SOCKFS.root, name, 49152, 0);
        node.sock = sock;

        // and the wrapping stream that enables library functions such
        // as read and write to indirectly interact with the socket
        var stream = FS.createStream({
          path: name,
          node: node,
          flags: FS.modeStringToFlags('r+'),
          seekable: false,
          stream_ops: SOCKFS.stream_ops
        });

        // map the new stream to the socket structure (sockets have a 1:1
        // relationship with a stream)
        sock.stream = stream;

        return sock;
      },getSocket:function (fd) {
        var stream = FS.getStream(fd);
        if (!stream || !FS.isSocket(stream.node.mode)) {
          return null;
        }
        return stream.node.sock;
      },stream_ops:{poll:function (stream) {
          var sock = stream.node.sock;
          return sock.sock_ops.poll(sock);
        },ioctl:function (stream, request, varargs) {
          var sock = stream.node.sock;
          return sock.sock_ops.ioctl(sock, request, varargs);
        },read:function (stream, buffer, offset, length, position /* ignored */) {
          var sock = stream.node.sock;
          var msg = sock.sock_ops.recvmsg(sock, length);
          if (!msg) {
            // socket is closed
            return 0;
          }
          buffer.set(msg.buffer, offset);
          return msg.buffer.length;
        },write:function (stream, buffer, offset, length, position /* ignored */) {
          var sock = stream.node.sock;
          return sock.sock_ops.sendmsg(sock, buffer, offset, length);
        },close:function (stream) {
          var sock = stream.node.sock;
          sock.sock_ops.close(sock);
        }},nextname:function () {
        if (!SOCKFS.nextname.current) {
          SOCKFS.nextname.current = 0;
        }
        return 'socket[' + (SOCKFS.nextname.current++) + ']';
      },websocket_sock_ops:{createPeer:function (sock, addr, port) {
          var ws;

          if (typeof addr === 'object') {
            ws = addr;
            addr = null;
            port = null;
          }

          if (ws) {
            // for sockets that've already connected (e.g. we're the server)
            // we can inspect the _socket property for the address
            if (ws._socket) {
              addr = ws._socket.remoteAddress;
              port = ws._socket.remotePort;
            }
            // if we're just now initializing a connection to the remote,
            // inspect the url property
            else {
              var result = /ws[s]?:\/\/([^:]+):(\d+)/.exec(ws.url);
              if (!result) {
                throw new Error('WebSocket URL must be in the format ws(s)://address:port');
              }
              addr = result[1];
              port = parseInt(result[2], 10);
            }
          } else {
            // create the actual websocket object and connect
            try {
              // runtimeConfig gets set to true if WebSocket runtime configuration is available.
              var runtimeConfig = (Module['websocket'] && ('object' === typeof Module['websocket']));

              // The default value is 'ws://' the replace is needed because the compiler replaces '//' comments with '#'
              // comments without checking context, so we'd end up with ws:#, the replace swaps the '#' for '//' again.
              var url = 'ws:#'.replace('#', '//');

              if (runtimeConfig) {
                if ('string' === typeof Module['websocket']['url']) {
                  url = Module['websocket']['url']; // Fetch runtime WebSocket URL config.
                }
              }

              if (url === 'ws://' || url === 'wss://') { // Is the supplied URL config just a prefix, if so complete it.
                var parts = addr.split('/');
                url = url + parts[0] + ":" + port + "/" + parts.slice(1).join('/');
              }

              // Make the WebSocket subprotocol (Sec-WebSocket-Protocol) default to binary if no configuration is set.
              var subProtocols = 'binary'; // The default value is 'binary'

              if (runtimeConfig) {
                if ('string' === typeof Module['websocket']['subprotocol']) {
                  subProtocols = Module['websocket']['subprotocol']; // Fetch runtime WebSocket subprotocol config.
                }
              }

              // The regex trims the string (removes spaces at the beginning and end, then splits the string by
              // <any space>,<any space> into an Array. Whitespace removal is important for Websockify and ws.
              subProtocols = subProtocols.replace(/^ +| +$/g,"").split(/ *, */);

              // The node ws library API for specifying optional subprotocol is slightly different than the browser's.
              var opts = ENVIRONMENT_IS_NODE ? {'protocol': subProtocols.toString()} : subProtocols;

              // If node we use the ws library.
              var WebSocketConstructor;
              if (ENVIRONMENT_IS_NODE) {
                WebSocketConstructor = require('ws');
              } else if (ENVIRONMENT_IS_WEB) {
                WebSocketConstructor = window['WebSocket'];
              } else {
                WebSocketConstructor = WebSocket;
              }
              ws = new WebSocketConstructor(url, opts);
              ws.binaryType = 'arraybuffer';
            } catch (e) {
              throw new FS.ErrnoError(ERRNO_CODES.EHOSTUNREACH);
            }
          }


          var peer = {
            addr: addr,
            port: port,
            socket: ws,
            dgram_send_queue: []
          };

          SOCKFS.websocket_sock_ops.addPeer(sock, peer);
          SOCKFS.websocket_sock_ops.handlePeerEvents(sock, peer);

          // if this is a bound dgram socket, send the port number first to allow
          // us to override the ephemeral port reported to us by remotePort on the
          // remote end.
          if (sock.type === 2 && typeof sock.sport !== 'undefined') {
            peer.dgram_send_queue.push(new Uint8Array([
                255, 255, 255, 255,
                'p'.charCodeAt(0), 'o'.charCodeAt(0), 'r'.charCodeAt(0), 't'.charCodeAt(0),
                ((sock.sport & 0xff00) >> 8) , (sock.sport & 0xff)
            ]));
          }

          return peer;
        },getPeer:function (sock, addr, port) {
          return sock.peers[addr + ':' + port];
        },addPeer:function (sock, peer) {
          sock.peers[peer.addr + ':' + peer.port] = peer;
        },removePeer:function (sock, peer) {
          delete sock.peers[peer.addr + ':' + peer.port];
        },handlePeerEvents:function (sock, peer) {
          var first = true;

          var handleOpen = function () {

            Module['websocket'].emit('open', sock.stream.fd);

            try {
              var queued = peer.dgram_send_queue.shift();
              while (queued) {
                peer.socket.send(queued);
                queued = peer.dgram_send_queue.shift();
              }
            } catch (e) {
              // not much we can do here in the way of proper error handling as we've already
              // lied and said this data was sent. shut it down.
              peer.socket.close();
            }
          };

          function handleMessage(data) {
            assert(typeof data !== 'string' && data.byteLength !== undefined);  // must receive an ArrayBuffer

            // An empty ArrayBuffer will emit a pseudo disconnect event
            // as recv/recvmsg will return zero which indicates that a socket
            // has performed a shutdown although the connection has not been disconnected yet.
            if (data.byteLength == 0) {
              return;
            }
            data = new Uint8Array(data);  // make a typed array view on the array buffer


            // if this is the port message, override the peer's port with it
            var wasfirst = first;
            first = false;
            if (wasfirst &&
                data.length === 10 &&
                data[0] === 255 && data[1] === 255 && data[2] === 255 && data[3] === 255 &&
                data[4] === 'p'.charCodeAt(0) && data[5] === 'o'.charCodeAt(0) && data[6] === 'r'.charCodeAt(0) && data[7] === 't'.charCodeAt(0)) {
              // update the peer's port and it's key in the peer map
              var newport = ((data[8] << 8) | data[9]);
              SOCKFS.websocket_sock_ops.removePeer(sock, peer);
              peer.port = newport;
              SOCKFS.websocket_sock_ops.addPeer(sock, peer);
              return;
            }

            sock.recv_queue.push({ addr: peer.addr, port: peer.port, data: data });
            Module['websocket'].emit('message', sock.stream.fd);
          };

          if (ENVIRONMENT_IS_NODE) {
            peer.socket.on('open', handleOpen);
            peer.socket.on('message', function(data, flags) {
              if (!flags.binary) {
                return;
              }
              handleMessage((new Uint8Array(data)).buffer);  // copy from node Buffer -> ArrayBuffer
            });
            peer.socket.on('close', function() {
              Module['websocket'].emit('close', sock.stream.fd);
            });
            peer.socket.on('error', function(error) {
              // Although the ws library may pass errors that may be more descriptive than
              // ECONNREFUSED they are not necessarily the expected error code e.g.
              // ENOTFOUND on getaddrinfo seems to be node.js specific, so using ECONNREFUSED
              // is still probably the most useful thing to do.
              sock.error = ERRNO_CODES.ECONNREFUSED; // Used in getsockopt for SOL_SOCKET/SO_ERROR test.
              Module['websocket'].emit('error', [sock.stream.fd, sock.error, 'ECONNREFUSED: Connection refused']);
              // don't throw
            });
          } else {
            peer.socket.onopen = handleOpen;
            peer.socket.onclose = function() {
              Module['websocket'].emit('close', sock.stream.fd);
            };
            peer.socket.onmessage = function peer_socket_onmessage(event) {
              handleMessage(event.data);
            };
            peer.socket.onerror = function(error) {
              // The WebSocket spec only allows a 'simple event' to be thrown on error,
              // so we only really know as much as ECONNREFUSED.
              sock.error = ERRNO_CODES.ECONNREFUSED; // Used in getsockopt for SOL_SOCKET/SO_ERROR test.
              Module['websocket'].emit('error', [sock.stream.fd, sock.error, 'ECONNREFUSED: Connection refused']);
            };
          }
        },poll:function (sock) {
          if (sock.type === 1 && sock.server) {
            // listen sockets should only say they're available for reading
            // if there are pending clients.
            return sock.pending.length ? (64 | 1) : 0;
          }

          var mask = 0;
          var dest = sock.type === 1 ?  // we only care about the socket state for connection-based sockets
            SOCKFS.websocket_sock_ops.getPeer(sock, sock.daddr, sock.dport) :
            null;

          if (sock.recv_queue.length ||
              !dest ||  // connection-less sockets are always ready to read
              (dest && dest.socket.readyState === dest.socket.CLOSING) ||
              (dest && dest.socket.readyState === dest.socket.CLOSED)) {  // let recv return 0 once closed
            mask |= (64 | 1);
          }

          if (!dest ||  // connection-less sockets are always ready to write
              (dest && dest.socket.readyState === dest.socket.OPEN)) {
            mask |= 4;
          }

          if ((dest && dest.socket.readyState === dest.socket.CLOSING) ||
              (dest && dest.socket.readyState === dest.socket.CLOSED)) {
            mask |= 16;
          }

          return mask;
        },ioctl:function (sock, request, arg) {
          switch (request) {
            case 21531:
              var bytes = 0;
              if (sock.recv_queue.length) {
                bytes = sock.recv_queue[0].data.length;
              }
              HEAP32[((arg)>>2)]=bytes;
              return 0;
            default:
              return ERRNO_CODES.EINVAL;
          }
        },close:function (sock) {
          // if we've spawned a listen server, close it
          if (sock.server) {
            try {
              sock.server.close();
            } catch (e) {
            }
            sock.server = null;
          }
          // close any peer connections
          var peers = Object.keys(sock.peers);
          for (var i = 0; i < peers.length; i++) {
            var peer = sock.peers[peers[i]];
            try {
              peer.socket.close();
            } catch (e) {
            }
            SOCKFS.websocket_sock_ops.removePeer(sock, peer);
          }
          return 0;
        },bind:function (sock, addr, port) {
          if (typeof sock.saddr !== 'undefined' || typeof sock.sport !== 'undefined') {
            throw new FS.ErrnoError(ERRNO_CODES.EINVAL);  // already bound
          }
          sock.saddr = addr;
          sock.sport = port;
          // in order to emulate dgram sockets, we need to launch a listen server when
          // binding on a connection-less socket
          // note: this is only required on the server side
          if (sock.type === 2) {
            // close the existing server if it exists
            if (sock.server) {
              sock.server.close();
              sock.server = null;
            }
            // swallow error operation not supported error that occurs when binding in the
            // browser where this isn't supported
            try {
              sock.sock_ops.listen(sock, 0);
            } catch (e) {
              if (!(e instanceof FS.ErrnoError)) throw e;
              if (e.errno !== ERRNO_CODES.EOPNOTSUPP) throw e;
            }
          }
        },connect:function (sock, addr, port) {
          if (sock.server) {
            throw new FS.ErrnoError(ERRNO_CODES.EOPNOTSUPP);
          }

          // TODO autobind
          // if (!sock.addr && sock.type == 2) {
          // }

          // early out if we're already connected / in the middle of connecting
          if (typeof sock.daddr !== 'undefined' && typeof sock.dport !== 'undefined') {
            var dest = SOCKFS.websocket_sock_ops.getPeer(sock, sock.daddr, sock.dport);
            if (dest) {
              if (dest.socket.readyState === dest.socket.CONNECTING) {
                throw new FS.ErrnoError(ERRNO_CODES.EALREADY);
              } else {
                throw new FS.ErrnoError(ERRNO_CODES.EISCONN);
              }
            }
          }

          // add the socket to our peer list and set our
          // destination address / port to match
          var peer = SOCKFS.websocket_sock_ops.createPeer(sock, addr, port);
          sock.daddr = peer.addr;
          sock.dport = peer.port;

          // always "fail" in non-blocking mode
          throw new FS.ErrnoError(ERRNO_CODES.EINPROGRESS);
        },listen:function (sock, backlog) {
          if (!ENVIRONMENT_IS_NODE) {
            throw new FS.ErrnoError(ERRNO_CODES.EOPNOTSUPP);
          }
          if (sock.server) {
             throw new FS.ErrnoError(ERRNO_CODES.EINVAL);  // already listening
          }
          var WebSocketServer = require('ws').Server;
          var host = sock.saddr;
          sock.server = new WebSocketServer({
            host: host,
            port: sock.sport
            // TODO support backlog
          });
          Module['websocket'].emit('listen', sock.stream.fd); // Send Event with listen fd.

          sock.server.on('connection', function(ws) {
            if (sock.type === 1) {
              var newsock = SOCKFS.createSocket(sock.family, sock.type, sock.protocol);

              // create a peer on the new socket
              var peer = SOCKFS.websocket_sock_ops.createPeer(newsock, ws);
              newsock.daddr = peer.addr;
              newsock.dport = peer.port;

              // push to queue for accept to pick up
              sock.pending.push(newsock);
              Module['websocket'].emit('connection', newsock.stream.fd);
            } else {
              // create a peer on the listen socket so calling sendto
              // with the listen socket and an address will resolve
              // to the correct client
              SOCKFS.websocket_sock_ops.createPeer(sock, ws);
              Module['websocket'].emit('connection', sock.stream.fd);
            }
          });
          sock.server.on('closed', function() {
            Module['websocket'].emit('close', sock.stream.fd);
            sock.server = null;
          });
          sock.server.on('error', function(error) {
            // Although the ws library may pass errors that may be more descriptive than
            // ECONNREFUSED they are not necessarily the expected error code e.g.
            // ENOTFOUND on getaddrinfo seems to be node.js specific, so using EHOSTUNREACH
            // is still probably the most useful thing to do. This error shouldn't
            // occur in a well written app as errors should get trapped in the compiled
            // app's own getaddrinfo call.
            sock.error = ERRNO_CODES.EHOSTUNREACH; // Used in getsockopt for SOL_SOCKET/SO_ERROR test.
            Module['websocket'].emit('error', [sock.stream.fd, sock.error, 'EHOSTUNREACH: Host is unreachable']);
            // don't throw
          });
        },accept:function (listensock) {
          if (!listensock.server) {
            throw new FS.ErrnoError(ERRNO_CODES.EINVAL);
          }
          var newsock = listensock.pending.shift();
          newsock.stream.flags = listensock.stream.flags;
          return newsock;
        },getname:function (sock, peer) {
          var addr, port;
          if (peer) {
            if (sock.daddr === undefined || sock.dport === undefined) {
              throw new FS.ErrnoError(ERRNO_CODES.ENOTCONN);
            }
            addr = sock.daddr;
            port = sock.dport;
          } else {
            // TODO saddr and sport will be set for bind()'d UDP sockets, but what
            // should we be returning for TCP sockets that've been connect()'d?
            addr = sock.saddr || 0;
            port = sock.sport || 0;
          }
          return { addr: addr, port: port };
        },sendmsg:function (sock, buffer, offset, length, addr, port) {
          if (sock.type === 2) {
            // connection-less sockets will honor the message address,
            // and otherwise fall back to the bound destination address
            if (addr === undefined || port === undefined) {
              addr = sock.daddr;
              port = sock.dport;
            }
            // if there was no address to fall back to, error out
            if (addr === undefined || port === undefined) {
              throw new FS.ErrnoError(ERRNO_CODES.EDESTADDRREQ);
            }
          } else {
            // connection-based sockets will only use the bound
            addr = sock.daddr;
            port = sock.dport;
          }

          // find the peer for the destination address
          var dest = SOCKFS.websocket_sock_ops.getPeer(sock, addr, port);

          // early out if not connected with a connection-based socket
          if (sock.type === 1) {
            if (!dest || dest.socket.readyState === dest.socket.CLOSING || dest.socket.readyState === dest.socket.CLOSED) {
              throw new FS.ErrnoError(ERRNO_CODES.ENOTCONN);
            } else if (dest.socket.readyState === dest.socket.CONNECTING) {
              throw new FS.ErrnoError(ERRNO_CODES.EAGAIN);
            }
          }

          // create a copy of the incoming data to send, as the WebSocket API
          // doesn't work entirely with an ArrayBufferView, it'll just send
          // the entire underlying buffer
          var data;
          if (buffer instanceof Array || buffer instanceof ArrayBuffer) {
            data = buffer.slice(offset, offset + length);
          } else {  // ArrayBufferView
            data = buffer.buffer.slice(buffer.byteOffset + offset, buffer.byteOffset + offset + length);
          }

          // if we're emulating a connection-less dgram socket and don't have
          // a cached connection, queue the buffer to send upon connect and
          // lie, saying the data was sent now.
          if (sock.type === 2) {
            if (!dest || dest.socket.readyState !== dest.socket.OPEN) {
              // if we're not connected, open a new connection
              if (!dest || dest.socket.readyState === dest.socket.CLOSING || dest.socket.readyState === dest.socket.CLOSED) {
                dest = SOCKFS.websocket_sock_ops.createPeer(sock, addr, port);
              }
              dest.dgram_send_queue.push(data);
              return length;
            }
          }

          try {
            // send the actual data
            dest.socket.send(data);
            return length;
          } catch (e) {
            throw new FS.ErrnoError(ERRNO_CODES.EINVAL);
          }
        },recvmsg:function (sock, length) {
          // http://pubs.opengroup.org/onlinepubs/7908799/xns/recvmsg.html
          if (sock.type === 1 && sock.server) {
            // tcp servers should not be recv()'ing on the listen socket
            throw new FS.ErrnoError(ERRNO_CODES.ENOTCONN);
          }

          var queued = sock.recv_queue.shift();
          if (!queued) {
            if (sock.type === 1) {
              var dest = SOCKFS.websocket_sock_ops.getPeer(sock, sock.daddr, sock.dport);

              if (!dest) {
                // if we have a destination address but are not connected, error out
                throw new FS.ErrnoError(ERRNO_CODES.ENOTCONN);
              }
              else if (dest.socket.readyState === dest.socket.CLOSING || dest.socket.readyState === dest.socket.CLOSED) {
                // return null if the socket has closed
                return null;
              }
              else {
                // else, our socket is in a valid state but truly has nothing available
                throw new FS.ErrnoError(ERRNO_CODES.EAGAIN);
              }
            } else {
              throw new FS.ErrnoError(ERRNO_CODES.EAGAIN);
            }
          }

          // queued.data will be an ArrayBuffer if it's unadulterated, but if it's
          // requeued TCP data it'll be an ArrayBufferView
          var queuedLength = queued.data.byteLength || queued.data.length;
          var queuedOffset = queued.data.byteOffset || 0;
          var queuedBuffer = queued.data.buffer || queued.data;
          var bytesRead = Math.min(length, queuedLength);
          var res = {
            buffer: new Uint8Array(queuedBuffer, queuedOffset, bytesRead),
            addr: queued.addr,
            port: queued.port
          };


          // push back any unread data for TCP connections
          if (sock.type === 1 && bytesRead < queuedLength) {
            var bytesRemaining = queuedLength - bytesRead;
            queued.data = new Uint8Array(queuedBuffer, queuedOffset + bytesRead, bytesRemaining);
            sock.recv_queue.unshift(queued);
          }

          return res;
        }}};


  var Sockets={BUFFER_SIZE:10240,MAX_BUFFER_SIZE:10485760,nextFd:1,fds:{},nextport:1,maxport:65535,peer:null,connections:{},portmap:{},localAddr:4261412874,addrPool:[33554442,50331658,67108874,83886090,100663306,117440522,134217738,150994954,167772170,184549386,201326602,218103818,234881034]};

  function __inet_ntop4_raw(addr) {
      return (addr & 0xff) + '.' + ((addr >> 8) & 0xff) + '.' + ((addr >> 16) & 0xff) + '.' + ((addr >> 24) & 0xff)
    }

  function __inet_ntop6_raw(ints) {
      //  ref:  http://www.ietf.org/rfc/rfc2373.txt - section 2.5.4
      //  Format for IPv4 compatible and mapped  128-bit IPv6 Addresses
      //  128-bits are split into eight 16-bit words
      //  stored in network byte order (big-endian)
      //  |                80 bits               | 16 |      32 bits        |
      //  +-----------------------------------------------------------------+
      //  |               10 bytes               |  2 |      4 bytes        |
      //  +--------------------------------------+--------------------------+
      //  +               5 words                |  1 |      2 words        |
      //  +--------------------------------------+--------------------------+
      //  |0000..............................0000|0000|    IPv4 ADDRESS     | (compatible)
      //  +--------------------------------------+----+---------------------+
      //  |0000..............................0000|FFFF|    IPv4 ADDRESS     | (mapped)
      //  +--------------------------------------+----+---------------------+
      var str = "";
      var word = 0;
      var longest = 0;
      var lastzero = 0;
      var zstart = 0;
      var len = 0;
      var i = 0;
      var parts = [
        ints[0] & 0xffff,
        (ints[0] >> 16),
        ints[1] & 0xffff,
        (ints[1] >> 16),
        ints[2] & 0xffff,
        (ints[2] >> 16),
        ints[3] & 0xffff,
        (ints[3] >> 16)
      ];

      // Handle IPv4-compatible, IPv4-mapped, loopback and any/unspecified addresses

      var hasipv4 = true;
      var v4part = "";
      // check if the 10 high-order bytes are all zeros (first 5 words)
      for (i = 0; i < 5; i++) {
        if (parts[i] !== 0) { hasipv4 = false; break; }
      }

      if (hasipv4) {
        // low-order 32-bits store an IPv4 address (bytes 13 to 16) (last 2 words)
        v4part = __inet_ntop4_raw(parts[6] | (parts[7] << 16));
        // IPv4-mapped IPv6 address if 16-bit value (bytes 11 and 12) == 0xFFFF (6th word)
        if (parts[5] === -1) {
          str = "::ffff:";
          str += v4part;
          return str;
        }
        // IPv4-compatible IPv6 address if 16-bit value (bytes 11 and 12) == 0x0000 (6th word)
        if (parts[5] === 0) {
          str = "::";
          //special case IPv6 addresses
          if(v4part === "0.0.0.0") v4part = ""; // any/unspecified address
          if(v4part === "0.0.0.1") v4part = "1";// loopback address
          str += v4part;
          return str;
        }
      }

      // Handle all other IPv6 addresses

      // first run to find the longest contiguous zero words
      for (word = 0; word < 8; word++) {
        if (parts[word] === 0) {
          if (word - lastzero > 1) {
            len = 0;
          }
          lastzero = word;
          len++;
        }
        if (len > longest) {
          longest = len;
          zstart = word - longest + 1;
        }
      }

      for (word = 0; word < 8; word++) {
        if (longest > 1) {
          // compress contiguous zeros - to produce "::"
          if (parts[word] === 0 && word >= zstart && word < (zstart + longest) ) {
            if (word === zstart) {
              str += ":";
              if (zstart === 0) str += ":"; //leading zeros case
            }
            continue;
          }
        }
        // converts 16-bit words from big-endian to little-endian before converting to hex string
        str += Number(_ntohs(parts[word] & 0xffff)).toString(16);
        str += word < 7 ? ":" : "";
      }
      return str;
    }function __read_sockaddr(sa, salen) {
      // family / port offsets are common to both sockaddr_in and sockaddr_in6
      var family = HEAP16[((sa)>>1)];
      var port = _ntohs(HEAP16[(((sa)+(2))>>1)]);
      var addr;

      switch (family) {
        case 2:
          if (salen !== 16) {
            return { errno: ERRNO_CODES.EINVAL };
          }
          addr = HEAP32[(((sa)+(4))>>2)];
          addr = __inet_ntop4_raw(addr);
          break;
        case 10:
          if (salen !== 28) {
            return { errno: ERRNO_CODES.EINVAL };
          }
          addr = [
            HEAP32[(((sa)+(8))>>2)],
            HEAP32[(((sa)+(12))>>2)],
            HEAP32[(((sa)+(16))>>2)],
            HEAP32[(((sa)+(20))>>2)]
          ];
          addr = __inet_ntop6_raw(addr);
          break;
        default:
          return { errno: ERRNO_CODES.EAFNOSUPPORT };
      }

      return { family: family, addr: addr, port: port };
    }

  function __write_sockaddr(sa, family, addr, port) {
      switch (family) {
        case 2:
          addr = __inet_pton4_raw(addr);
          HEAP16[((sa)>>1)]=family;
          HEAP32[(((sa)+(4))>>2)]=addr;
          HEAP16[(((sa)+(2))>>1)]=_htons(port);
          break;
        case 10:
          addr = __inet_pton6_raw(addr);
          HEAP32[((sa)>>2)]=family;
          HEAP32[(((sa)+(8))>>2)]=addr[0];
          HEAP32[(((sa)+(12))>>2)]=addr[1];
          HEAP32[(((sa)+(16))>>2)]=addr[2];
          HEAP32[(((sa)+(20))>>2)]=addr[3];
          HEAP16[(((sa)+(2))>>1)]=_htons(port);
          HEAP32[(((sa)+(4))>>2)]=0;
          HEAP32[(((sa)+(24))>>2)]=0;
          break;
        default:
          return { errno: ERRNO_CODES.EAFNOSUPPORT };
      }
      // kind of lame, but let's match _read_sockaddr's interface
      return {};
    }function ___syscall102(which, varargs) {SYSCALLS.varargs = varargs;
  try {
   // socketcall
      var call = SYSCALLS.get(), socketvararg = SYSCALLS.get();
      // socketcalls pass the rest of the arguments in a struct
      SYSCALLS.varargs = socketvararg;
      switch (call) {
        case 1: { // socket
          var domain = SYSCALLS.get(), type = SYSCALLS.get(), protocol = SYSCALLS.get();
          var sock = SOCKFS.createSocket(domain, type, protocol);
          assert(sock.stream.fd < 64); // XXX ? select() assumes socket fd values are in 0..63
          return sock.stream.fd;
        }
        case 2: { // bind
          var sock = SYSCALLS.getSocketFromFD(), info = SYSCALLS.getSocketAddress();
          sock.sock_ops.bind(sock, info.addr, info.port);
          return 0;
        }
        case 3: { // connect
          var sock = SYSCALLS.getSocketFromFD(), info = SYSCALLS.getSocketAddress();
          sock.sock_ops.connect(sock, info.addr, info.port);
          return 0;
        }
        case 4: { // listen
          var sock = SYSCALLS.getSocketFromFD(), backlog = SYSCALLS.get();
          sock.sock_ops.listen(sock, backlog);
          return 0;
        }
        case 5: { // accept
          var sock = SYSCALLS.getSocketFromFD(), addr = SYSCALLS.get(), addrlen = SYSCALLS.get();
          var newsock = sock.sock_ops.accept(sock);
          if (addr) {
            var res = __write_sockaddr(addr, newsock.family, DNS.lookup_name(newsock.daddr), newsock.dport);
            assert(!res.errno);
          }
          return newsock.stream.fd;
        }
        case 6: { // getsockname
          var sock = SYSCALLS.getSocketFromFD(), addr = SYSCALLS.get(), addrlen = SYSCALLS.get();
          // TODO: sock.saddr should never be undefined, see TODO in websocket_sock_ops.getname
          var res = __write_sockaddr(addr, sock.family, DNS.lookup_name(sock.saddr || '0.0.0.0'), sock.sport);
          assert(!res.errno);
          return 0;
        }
        case 7: { // getpeername
          var sock = SYSCALLS.getSocketFromFD(), addr = SYSCALLS.get(), addrlen = SYSCALLS.get();
          if (!sock.daddr) {
            return -ERRNO_CODES.ENOTCONN; // The socket is not connected.
          }
          var res = __write_sockaddr(addr, sock.family, DNS.lookup_name(sock.daddr), sock.dport);
          assert(!res.errno);
          return 0;
        }
        case 11: { // sendto
          var sock = SYSCALLS.getSocketFromFD(), message = SYSCALLS.get(), length = SYSCALLS.get(), flags = SYSCALLS.get(), dest = SYSCALLS.getSocketAddress(true);
          if (!dest) {
            // send, no address provided
            return FS.write(sock.stream, HEAP8,message, length);
          } else {
            // sendto an address
            return sock.sock_ops.sendmsg(sock, HEAP8,message, length, dest.addr, dest.port);
          }
        }
        case 12: { // recvfrom
          var sock = SYSCALLS.getSocketFromFD(), buf = SYSCALLS.get(), len = SYSCALLS.get(), flags = SYSCALLS.get(), addr = SYSCALLS.get(), addrlen = SYSCALLS.get();
          var msg = sock.sock_ops.recvmsg(sock, len);
          if (!msg) return 0; // socket is closed
          if (addr) {
            var res = __write_sockaddr(addr, sock.family, DNS.lookup_name(msg.addr), msg.port);
            assert(!res.errno);
          }
          HEAPU8.set(msg.buffer, buf);
          return msg.buffer.byteLength;
        }
        case 14: { // setsockopt
          return -ERRNO_CODES.ENOPROTOOPT; // The option is unknown at the level indicated.
        }
        case 15: { // getsockopt
          var sock = SYSCALLS.getSocketFromFD(), level = SYSCALLS.get(), optname = SYSCALLS.get(), optval = SYSCALLS.get(), optlen = SYSCALLS.get();
          // Minimal getsockopt aimed at resolving https://github.com/kripken/emscripten/issues/2211
          // so only supports SOL_SOCKET with SO_ERROR.
          if (level === 1) {
            if (optname === 4) {
              HEAP32[((optval)>>2)]=sock.error;
              HEAP32[((optlen)>>2)]=4;
              sock.error = null; // Clear the error (The SO_ERROR option obtains and then clears this field).
              return 0;
            }
          }
          return -ERRNO_CODES.ENOPROTOOPT; // The option is unknown at the level indicated.
        }
        case 16: { // sendmsg
          var sock = SYSCALLS.getSocketFromFD(), message = SYSCALLS.get(), flags = SYSCALLS.get();
          var iov = HEAP32[(((message)+(8))>>2)];
          var num = HEAP32[(((message)+(12))>>2)];
          // read the address and port to send to
          var addr, port;
          var name = HEAP32[((message)>>2)];
          var namelen = HEAP32[(((message)+(4))>>2)];
          if (name) {
            var info = __read_sockaddr(name, namelen);
            if (info.errno) return -info.errno;
            port = info.port;
            addr = DNS.lookup_addr(info.addr) || info.addr;
          }
          // concatenate scatter-gather arrays into one message buffer
          var total = 0;
          for (var i = 0; i < num; i++) {
            total += HEAP32[(((iov)+((8 * i) + 4))>>2)];
          }
          var view = new Uint8Array(total);
          var offset = 0;
          for (var i = 0; i < num; i++) {
            var iovbase = HEAP32[(((iov)+((8 * i) + 0))>>2)];
            var iovlen = HEAP32[(((iov)+((8 * i) + 4))>>2)];
            for (var j = 0; j < iovlen; j++) {
              view[offset++] = HEAP8[(((iovbase)+(j))>>0)];
            }
          }
          // write the buffer
          return sock.sock_ops.sendmsg(sock, view, 0, total, addr, port);
        }
        case 17: { // recvmsg
          var sock = SYSCALLS.getSocketFromFD(), message = SYSCALLS.get(), flags = SYSCALLS.get();
          var iov = HEAP32[(((message)+(8))>>2)];
          var num = HEAP32[(((message)+(12))>>2)];
          // get the total amount of data we can read across all arrays
          var total = 0;
          for (var i = 0; i < num; i++) {
            total += HEAP32[(((iov)+((8 * i) + 4))>>2)];
          }
          // try to read total data
          var msg = sock.sock_ops.recvmsg(sock, total);
          if (!msg) return 0; // socket is closed

          // TODO honor flags:
          // MSG_OOB
          // Requests out-of-band data. The significance and semantics of out-of-band data are protocol-specific.
          // MSG_PEEK
          // Peeks at the incoming message.
          // MSG_WAITALL
          // Requests that the function block until the full amount of data requested can be returned. The function may return a smaller amount of data if a signal is caught, if the connection is terminated, if MSG_PEEK was specified, or if an error is pending for the socket.

          // write the source address out
          var name = HEAP32[((message)>>2)];
          if (name) {
            var res = __write_sockaddr(name, sock.family, DNS.lookup_name(msg.addr), msg.port);
            assert(!res.errno);
          }
          // write the buffer out to the scatter-gather arrays
          var bytesRead = 0;
          var bytesRemaining = msg.buffer.byteLength;
          for (var i = 0; bytesRemaining > 0 && i < num; i++) {
            var iovbase = HEAP32[(((iov)+((8 * i) + 0))>>2)];
            var iovlen = HEAP32[(((iov)+((8 * i) + 4))>>2)];
            if (!iovlen) {
              continue;
            }
            var length = Math.min(iovlen, bytesRemaining);
            var buf = msg.buffer.subarray(bytesRead, bytesRead + length);
            HEAPU8.set(buf, iovbase + bytesRead);
            bytesRead += length;
            bytesRemaining -= length;
          }

          // TODO set msghdr.msg_flags
          // MSG_EOR
          // End of record was received (if supported by the protocol).
          // MSG_OOB
          // Out-of-band data was received.
          // MSG_TRUNC
          // Normal data was truncated.
          // MSG_CTRUNC

          return bytesRead;
        }
        default: abort('unsupported socketcall syscall ' + call);
      }
    } catch (e) {
    if (typeof FS === 'undefined' || !(e instanceof FS.ErrnoError)) abort(e);
    return -e.errno;
  }
  }



  function _time(ptr) {
      var ret = (Date.now()/1000)|0;
      if (ptr) {
        HEAP32[((ptr)>>2)]=ret;
      }
      return ret;
    }

  function ___syscall142(which, varargs) {SYSCALLS.varargs = varargs;
  try {
   // newselect
      // readfds are supported,
      // writefds checks socket open status
      // exceptfds not supported
      // timeout is always 0 - fully async
      var nfds = SYSCALLS.get(), readfds = SYSCALLS.get(), writefds = SYSCALLS.get(), exceptfds = SYSCALLS.get(), timeout = SYSCALLS.get();

      assert(nfds <= 64, 'nfds must be less than or equal to 64');  // fd sets have 64 bits // TODO: this could be 1024 based on current musl headers
      assert(!exceptfds, 'exceptfds not supported');

      var total = 0;

      var srcReadLow = (readfds ? HEAP32[((readfds)>>2)] : 0),
          srcReadHigh = (readfds ? HEAP32[(((readfds)+(4))>>2)] : 0);
      var srcWriteLow = (writefds ? HEAP32[((writefds)>>2)] : 0),
          srcWriteHigh = (writefds ? HEAP32[(((writefds)+(4))>>2)] : 0);
      var srcExceptLow = (exceptfds ? HEAP32[((exceptfds)>>2)] : 0),
          srcExceptHigh = (exceptfds ? HEAP32[(((exceptfds)+(4))>>2)] : 0);

      var dstReadLow = 0,
          dstReadHigh = 0;
      var dstWriteLow = 0,
          dstWriteHigh = 0;
      var dstExceptLow = 0,
          dstExceptHigh = 0;

      var allLow = (readfds ? HEAP32[((readfds)>>2)] : 0) |
                   (writefds ? HEAP32[((writefds)>>2)] : 0) |
                   (exceptfds ? HEAP32[((exceptfds)>>2)] : 0);
      var allHigh = (readfds ? HEAP32[(((readfds)+(4))>>2)] : 0) |
                    (writefds ? HEAP32[(((writefds)+(4))>>2)] : 0) |
                    (exceptfds ? HEAP32[(((exceptfds)+(4))>>2)] : 0);

      function check(fd, low, high, val) {
        return (fd < 32 ? (low & val) : (high & val));
      }

      for (var fd = 0; fd < nfds; fd++) {
        var mask = 1 << (fd % 32);
        if (!(check(fd, allLow, allHigh, mask))) {
          continue;  // index isn't in the set
        }

        var stream = FS.getStream(fd);
        if (!stream) throw new FS.ErrnoError(ERRNO_CODES.EBADF);

        var flags = SYSCALLS.DEFAULT_POLLMASK;

        if (stream.stream_ops.poll) {
          flags = stream.stream_ops.poll(stream);
        }

        if ((flags & 1) && check(fd, srcReadLow, srcReadHigh, mask)) {
          fd < 32 ? (dstReadLow = dstReadLow | mask) : (dstReadHigh = dstReadHigh | mask);
          total++;
        }
        if ((flags & 4) && check(fd, srcWriteLow, srcWriteHigh, mask)) {
          fd < 32 ? (dstWriteLow = dstWriteLow | mask) : (dstWriteHigh = dstWriteHigh | mask);
          total++;
        }
        if ((flags & 2) && check(fd, srcExceptLow, srcExceptHigh, mask)) {
          fd < 32 ? (dstExceptLow = dstExceptLow | mask) : (dstExceptHigh = dstExceptHigh | mask);
          total++;
        }
      }

      if (readfds) {
        HEAP32[((readfds)>>2)]=dstReadLow;
        HEAP32[(((readfds)+(4))>>2)]=dstReadHigh;
      }
      if (writefds) {
        HEAP32[((writefds)>>2)]=dstWriteLow;
        HEAP32[(((writefds)+(4))>>2)]=dstWriteHigh;
      }
      if (exceptfds) {
        HEAP32[((exceptfds)>>2)]=dstExceptLow;
        HEAP32[(((exceptfds)+(4))>>2)]=dstExceptHigh;
      }

      return total;
    } catch (e) {
    if (typeof FS === 'undefined' || !(e instanceof FS.ErrnoError)) abort(e);
    return -e.errno;
  }
  }

  function ___syscall140(which, varargs) {SYSCALLS.varargs = varargs;
  try {
   // llseek
      var stream = SYSCALLS.getStreamFromFD(), offset_high = SYSCALLS.get(), offset_low = SYSCALLS.get(), result = SYSCALLS.get(), whence = SYSCALLS.get();
      // NOTE: offset_high is unused - Emscripten's off_t is 32-bit
      var offset = offset_low;
      FS.llseek(stream, offset, whence);
      HEAP32[((result)>>2)]=stream.position;
      if (stream.getdents && offset === 0 && whence === 0) stream.getdents = null; // reset readdir state
      return 0;
    } catch (e) {
    if (typeof FS === 'undefined' || !(e instanceof FS.ErrnoError)) abort(e);
    return -e.errno;
  }
  }

  function ___syscall145(which, varargs) {SYSCALLS.varargs = varargs;
  try {
   // readv
      var stream = SYSCALLS.getStreamFromFD(), iov = SYSCALLS.get(), iovcnt = SYSCALLS.get();
      return SYSCALLS.doReadv(stream, iov, iovcnt);
    } catch (e) {
    if (typeof FS === 'undefined' || !(e instanceof FS.ErrnoError)) abort(e);
    return -e.errno;
  }
  }

  function ___syscall146(which, varargs) {SYSCALLS.varargs = varargs;
  try {
   // writev
      var stream = SYSCALLS.getStreamFromFD(), iov = SYSCALLS.get(), iovcnt = SYSCALLS.get();
      return SYSCALLS.doWritev(stream, iov, iovcnt);
    } catch (e) {
    if (typeof FS === 'undefined' || !(e instanceof FS.ErrnoError)) abort(e);
    return -e.errno;
  }
  }

  function ___syscall221(which, varargs) {SYSCALLS.varargs = varargs;
  try {
   // fcntl64
      var stream = SYSCALLS.getStreamFromFD(), cmd = SYSCALLS.get();
      switch (cmd) {
        case 0: {
          var arg = SYSCALLS.get();
          if (arg < 0) {
            return -ERRNO_CODES.EINVAL;
          }
          var newStream;
          newStream = FS.open(stream.path, stream.flags, 0, arg);
          return newStream.fd;
        }
        case 1:
        case 2:
          return 0;  // FD_CLOEXEC makes no sense for a single process.
        case 3:
          return stream.flags;
        case 4: {
          var arg = SYSCALLS.get();
          stream.flags |= arg;
          return 0;
        }
        case 12:
        case 12: {
          var arg = SYSCALLS.get();
          var offset = 0;
          // We're always unlocked.
          HEAP16[(((arg)+(offset))>>1)]=2;
          return 0;
        }
        case 13:
        case 14:
        case 13:
        case 14:
          return 0; // Pretend that the locking is successful.
        case 16:
        case 8:
          return -ERRNO_CODES.EINVAL; // These are for sockets. We don't have them fully implemented yet.
        case 9:
          // musl trusts getown return values, due to a bug where they must be, as they overlap with errors. just return -1 here, so fnctl() returns that, and we set errno ourselves.
          ___setErrNo(ERRNO_CODES.EINVAL);
          return -1;
        default: {
          return -ERRNO_CODES.EINVAL;
        }
      }
    } catch (e) {
    if (typeof FS === 'undefined' || !(e instanceof FS.ErrnoError)) abort(e);
    return -e.errno;
  }
  }

  function ___syscall220(which, varargs) {SYSCALLS.varargs = varargs;
  try {
   // SYS_getdents64
      var stream = SYSCALLS.getStreamFromFD(), dirp = SYSCALLS.get(), count = SYSCALLS.get();
      if (!stream.getdents) {
        stream.getdents = FS.readdir(stream.path);
      }
      var pos = 0;
      while (stream.getdents.length > 0 && pos + 268 <= count) {
        var id;
        var type;
        var name = stream.getdents.pop();
        assert(name.length < 256); // limit of dirent struct
        if (name[0] === '.') {
          id = 1;
          type = 4; // DT_DIR
        } else {
          var child = FS.lookupNode(stream.node, name);
          id = child.id;
          type = FS.isChrdev(child.mode) ? 2 :  // DT_CHR, character device.
                 FS.isDir(child.mode) ? 4 :     // DT_DIR, directory.
                 FS.isLink(child.mode) ? 10 :   // DT_LNK, symbolic link.
                 8;                             // DT_REG, regular file.
        }
        HEAP32[((dirp + pos)>>2)]=id;
        HEAP32[(((dirp + pos)+(4))>>2)]=stream.position;
        HEAP16[(((dirp + pos)+(8))>>1)]=268;
        HEAP8[(((dirp + pos)+(10))>>0)]=type;
        for (var i = 0; i < name.length; i++) {
          HEAP8[(((dirp + pos)+(11 + i))>>0)]=name.charCodeAt(i);
        }
        HEAP8[(((dirp + pos)+(11 + i))>>0)]=0;
        pos += 268;
      }
      return pos;
    } catch (e) {
    if (typeof FS === 'undefined' || !(e instanceof FS.ErrnoError)) abort(e);
    return -e.errno;
  }
  }

  var ___dso_handle=STATICTOP; STATICTOP += 16;;
FS.staticInit();__ATINIT__.unshift(function() { if (!Module["noFSInit"] && !FS.init.initialized) FS.init() });__ATMAIN__.push(function() { FS.ignorePermissions = false });__ATEXIT__.push(function() { FS.quit() });Module["FS"]=FS;Module["FS_createFolder"] = FS.createFolder;Module["FS_createPath"] = FS.createPath;Module["FS_createDataFile"] = FS.createDataFile;Module["FS_createPreloadedFile"] = FS.createPreloadedFile;Module["FS_stat"] = FS.stat;Module["FS_createLazyFile"] = FS.createLazyFile;Module["FS_createLink"] = FS.createLink;Module["FS_createDevice"] = FS.createDevice;Module["FS_unlink"] = FS.unlink;;
__ATINIT__.unshift(function() { TTY.init() });__ATEXIT__.push(function() { TTY.shutdown() });;
var GLctx; GL.init();
Module["requestFullScreen"] = function Module_requestFullScreen(lockPointer, resizeCanvas, vrDevice) { Module.printErr("Module.requestFullScreen is deprecated. Please call Module.requestFullscreen instead."); Module["requestFullScreen"] = Module["requestFullscreen"]; Browser.requestFullScreen(lockPointer, resizeCanvas, vrDevice) };
  Module["requestFullscreen"] = function Module_requestFullscreen(lockPointer, resizeCanvas, vrDevice) { Browser.requestFullscreen(lockPointer, resizeCanvas, vrDevice) };
  Module["requestAnimationFrame"] = function Module_requestAnimationFrame(func) { Browser.requestAnimationFrame(func) };
  Module["setCanvasSize"] = function Module_setCanvasSize(width, height, noUpdates) { Browser.setCanvasSize(width, height, noUpdates) };
  Module["pauseMainLoop"] = function Module_pauseMainLoop() { Browser.mainLoop.pause() };
  Module["resumeMainLoop"] = function Module_resumeMainLoop() { Browser.mainLoop.resume() };
  Module["getUserMedia"] = function Module_getUserMedia() { Browser.getUserMedia() }
  Module["createContext"] = function Module_createContext(canvas, useWebGL, setInModule, webGLContextAttributes) { return Browser.createContext(canvas, useWebGL, setInModule, webGLContextAttributes) };
if (ENVIRONMENT_IS_NODE) {
    _emscripten_get_now = function _emscripten_get_now_actual() {
      var t = process['hrtime']();
      return t[0] * 1e3 + t[1] / 1e6;
    };
  } else if (typeof dateNow !== 'undefined') {
    _emscripten_get_now = dateNow;
  } else if (typeof self === 'object' && self['performance'] && typeof self['performance']['now'] === 'function') {
    _emscripten_get_now = function() { return self['performance']['now'](); };
  } else if (typeof performance === 'object' && typeof performance['now'] === 'function') {
    _emscripten_get_now = function() { return performance['now'](); };
  } else {
    _emscripten_get_now = Date.now;
  };
___buildEnvironment(ENV);;
__ATINIT__.push(function() { SOCKFS.root = FS.mount(SOCKFS, {}, null); });;
DYNAMICTOP_PTR = allocate(1, "i32", ALLOC_STATIC);

STACK_BASE = STACKTOP = Runtime.alignMemory(STATICTOP);

STACK_MAX = STACK_BASE + TOTAL_STACK;

DYNAMIC_BASE = Runtime.alignMemory(STACK_MAX);

HEAP32[DYNAMICTOP_PTR>>2] = DYNAMIC_BASE;

staticSealed = true; // seal the static portion of memory


function invoke_iiii(index,a1,a2,a3) {
  try {
    return Module["dynCall_iiii"](index,a1,a2,a3);
  } catch(e) {
    if (typeof e !== 'number' && e !== 'longjmp') throw e;
    Module["setThrew"](1, 0);
  }
}

function invoke_viiiii(index,a1,a2,a3,a4,a5) {
  try {
    Module["dynCall_viiiii"](index,a1,a2,a3,a4,a5);
  } catch(e) {
    if (typeof e !== 'number' && e !== 'longjmp') throw e;
    Module["setThrew"](1, 0);
  }
}

function invoke_vi(index,a1) {
  try {
    Module["dynCall_vi"](index,a1);
  } catch(e) {
    if (typeof e !== 'number' && e !== 'longjmp') throw e;
    Module["setThrew"](1, 0);
  }
}

function invoke_vii(index,a1,a2) {
  try {
    Module["dynCall_vii"](index,a1,a2);
  } catch(e) {
    if (typeof e !== 'number' && e !== 'longjmp') throw e;
    Module["setThrew"](1, 0);
  }
}

function invoke_ii(index,a1) {
  try {
    return Module["dynCall_ii"](index,a1);
  } catch(e) {
    if (typeof e !== 'number' && e !== 'longjmp') throw e;
    Module["setThrew"](1, 0);
  }
}

function invoke_viii(index,a1,a2,a3) {
  try {
    Module["dynCall_viii"](index,a1,a2,a3);
  } catch(e) {
    if (typeof e !== 'number' && e !== 'longjmp') throw e;
    Module["setThrew"](1, 0);
  }
}

function invoke_v(index) {
  try {
    Module["dynCall_v"](index);
  } catch(e) {
    if (typeof e !== 'number' && e !== 'longjmp') throw e;
    Module["setThrew"](1, 0);
  }
}

function invoke_iiiii(index,a1,a2,a3,a4) {
  try {
    return Module["dynCall_iiiii"](index,a1,a2,a3,a4);
  } catch(e) {
    if (typeof e !== 'number' && e !== 'longjmp') throw e;
    Module["setThrew"](1, 0);
  }
}

function invoke_viiiiii(index,a1,a2,a3,a4,a5,a6) {
  try {
    Module["dynCall_viiiiii"](index,a1,a2,a3,a4,a5,a6);
  } catch(e) {
    if (typeof e !== 'number' && e !== 'longjmp') throw e;
    Module["setThrew"](1, 0);
  }
}

function invoke_iii(index,a1,a2) {
  try {
    return Module["dynCall_iii"](index,a1,a2);
  } catch(e) {
    if (typeof e !== 'number' && e !== 'longjmp') throw e;
    Module["setThrew"](1, 0);
  }
}

function invoke_viiii(index,a1,a2,a3,a4) {
  try {
    Module["dynCall_viiii"](index,a1,a2,a3,a4);
  } catch(e) {
    if (typeof e !== 'number' && e !== 'longjmp') throw e;
    Module["setThrew"](1, 0);
  }
}

Module.asmGlobalArg = { "Math": Math, "Int8Array": Int8Array, "Int16Array": Int16Array, "Int32Array": Int32Array, "Uint8Array": Uint8Array, "Uint16Array": Uint16Array, "Uint32Array": Uint32Array, "Float32Array": Float32Array, "Float64Array": Float64Array, "NaN": NaN, "Infinity": Infinity };

Module.asmLibraryArg = { "abort": abort, "assert": assert, "enlargeMemory": enlargeMemory, "getTotalMemory": getTotalMemory, "abortOnCannotGrowMemory": abortOnCannotGrowMemory, "invoke_iiii": invoke_iiii, "invoke_viiiii": invoke_viiiii, "invoke_vi": invoke_vi, "invoke_vii": invoke_vii, "invoke_ii": invoke_ii, "invoke_viii": invoke_viii, "invoke_v": invoke_v, "invoke_iiiii": invoke_iiiii, "invoke_viiiiii": invoke_viiiiii, "invoke_iii": invoke_iii, "invoke_viiii": invoke_viiii, "___syscall221": ___syscall221, "___syscall220": ___syscall220, "__inet_ntop6_raw": __inet_ntop6_raw, "_SDL_RWFromFile": _SDL_RWFromFile, "___assert_fail": ___assert_fail, "_emscripten_asm_const_ii": _emscripten_asm_const_ii, "_llvm_exp2_f64": _llvm_exp2_f64, "_SDL_ShowCursor": _SDL_ShowCursor, "_emscripten_set_main_loop_timing": _emscripten_set_main_loop_timing, "_SDL_FreeSurface": _SDL_FreeSurface, "_Mix_PlayChannel": _Mix_PlayChannel, "_emscripten_memcpy_big": _emscripten_memcpy_big, "_Mix_PlayMusic": _Mix_PlayMusic, "_Mix_LoadWAV_RW": _Mix_LoadWAV_RW, "__write_sockaddr": __write_sockaddr, "___cxa_atexit": ___cxa_atexit, "__exit": __exit, "_Mix_HaltMusic": _Mix_HaltMusic, "_SDL_UpdateRect": _SDL_UpdateRect, "___syscall140": ___syscall140, "___syscall142": ___syscall142, "___syscall145": ___syscall145, "___syscall146": ___syscall146, "_emscripten_asm_const_iiiii": _emscripten_asm_const_iiiii, "_SDL_QuitSubSystem": _SDL_QuitSubSystem, "___syscall122": ___syscall122, "__inet_ntop4_raw": __inet_ntop4_raw, "_SDL_GetTicks": _SDL_GetTicks, "_sem_init": _sem_init, "_SDL_VideoDriverName": _SDL_VideoDriverName, "_emscripten_asm_const_i": _emscripten_asm_const_i, "_SDL_LockSurface": _SDL_LockSurface, "___setErrNo": ___setErrNo, "_emscripten_get_now": _emscripten_get_now, "___syscall10": ___syscall10, "_SDL_MapRGB": _SDL_MapRGB, "___syscall3": ___syscall3, "___lock": ___lock, "_SDL_SetVideoMode": _SDL_SetVideoMode, "___syscall6": ___syscall6, "___syscall5": ___syscall5, "___syscall4": ___syscall4, "_time": _time, "_gettimeofday": _gettimeofday, "_SDL_UpperBlitScaled": _SDL_UpperBlitScaled, "_exit": _exit, "__inet_pton4_raw": __inet_pton4_raw, "_putenv": _putenv, "___syscall102": ___syscall102, "_llvm_pow_f64": _llvm_pow_f64, "_IMG_Load": _IMG_Load, "_TTF_FontHeight": _TTF_FontHeight, "___buildEnvironment": ___buildEnvironment, "_SDL_PumpEvents": _SDL_PumpEvents, "_localtime_r": _localtime_r, "_tzset": _tzset, "_SDL_WM_GrabInput": _SDL_WM_GrabInput, "___syscall197": ___syscall197, "___syscall195": ___syscall195, "___syscall194": ___syscall194, "_sem_wait": _sem_wait, "_TTF_RenderText_Solid": _TTF_RenderText_Solid, "_getenv": _getenv, "___syscall33": ___syscall33, "___syscall39": ___syscall39, "___syscall38": ___syscall38, "_emscripten_asm_const_iiiiii": _emscripten_asm_const_iiiiii, "_gethostbyname": _gethostbyname, "_SDL_UnlockSurface": _SDL_UnlockSurface, "_abort": _abort, "_localtime": _localtime, "___cxa_pure_virtual": ___cxa_pure_virtual, "_llvm_exp2_f32": _llvm_exp2_f32, "___syscall40": ___syscall40, "_sem_destroy": _sem_destroy, "_SDL_CloseAudio": _SDL_CloseAudio, "__inet_pton6_raw": __inet_pton6_raw, "__read_sockaddr": __read_sockaddr, "_usleep": _usleep, "_SDL_GetVideoInfo": _SDL_GetVideoInfo, "_SDL_FreeRW": _SDL_FreeRW, "_sem_post": _sem_post, "_SDL_PauseAudio": _SDL_PauseAudio, "_Mix_FreeChunk": _Mix_FreeChunk, "_SDL_WM_SetCaption": _SDL_WM_SetCaption, "_IMG_Load_RW": _IMG_Load_RW, "_TTF_SizeText": _TTF_SizeText, "___syscall54": ___syscall54, "___unlock": ___unlock, "_SDL_mutexP": _SDL_mutexP, "_emscripten_set_main_loop": _emscripten_set_main_loop, "_SDL_mutexV": _SDL_mutexV, "_emscripten_asm_const_iiii": _emscripten_asm_const_iiii, "_SDL_VideoModeOK": _SDL_VideoModeOK, "_atexit": _atexit, "_SDL_UpperBlit": _SDL_UpperBlit, "_SDL_RWFromConstMem": _SDL_RWFromConstMem, "DYNAMICTOP_PTR": DYNAMICTOP_PTR, "tempDoublePtr": tempDoublePtr, "ABORT": ABORT, "STACKTOP": STACKTOP, "STACK_MAX": STACK_MAX, "cttz_i8": cttz_i8, "___dso_handle": ___dso_handle };
// EMSCRIPTEN_START_ASM
var asm = (function(global, env, buffer) {
'use asm';


  var HEAP8 = new global.Int8Array(buffer);
  var HEAP16 = new global.Int16Array(buffer);
  var HEAP32 = new global.Int32Array(buffer);
  var HEAPU8 = new global.Uint8Array(buffer);
  var HEAPU16 = new global.Uint16Array(buffer);
  var HEAPU32 = new global.Uint32Array(buffer);
  var HEAPF32 = new global.Float32Array(buffer);
  var HEAPF64 = new global.Float64Array(buffer);

  var DYNAMICTOP_PTR=env.DYNAMICTOP_PTR|0;
  var tempDoublePtr=env.tempDoublePtr|0;
  var ABORT=env.ABORT|0;
  var STACKTOP=env.STACKTOP|0;
  var STACK_MAX=env.STACK_MAX|0;
  var cttz_i8=env.cttz_i8|0;
  var ___dso_handle=env.___dso_handle|0;

  var __THREW__ = 0;
  var threwValue = 0;
  var setjmpId = 0;
  var undef = 0;
  var nan = global.NaN, inf = global.Infinity;
  var tempInt = 0, tempBigInt = 0, tempBigIntS = 0, tempValue = 0, tempDouble = 0.0;
  var tempRet0 = 0;

  var Math_floor=global.Math.floor;
  var Math_abs=global.Math.abs;
  var Math_sqrt=global.Math.sqrt;
  var Math_pow=global.Math.pow;
  var Math_cos=global.Math.cos;
  var Math_sin=global.Math.sin;
  var Math_tan=global.Math.tan;
  var Math_acos=global.Math.acos;
  var Math_asin=global.Math.asin;
  var Math_atan=global.Math.atan;
  var Math_atan2=global.Math.atan2;
  var Math_exp=global.Math.exp;
  var Math_log=global.Math.log;
  var Math_ceil=global.Math.ceil;
  var Math_imul=global.Math.imul;
  var Math_min=global.Math.min;
  var Math_max=global.Math.max;
  var Math_clz32=global.Math.clz32;
  var abort=env.abort;
  var assert=env.assert;
  var enlargeMemory=env.enlargeMemory;
  var getTotalMemory=env.getTotalMemory;
  var abortOnCannotGrowMemory=env.abortOnCannotGrowMemory;
  var invoke_iiii=env.invoke_iiii;
  var invoke_viiiii=env.invoke_viiiii;
  var invoke_vi=env.invoke_vi;
  var invoke_vii=env.invoke_vii;
  var invoke_ii=env.invoke_ii;
  var invoke_viii=env.invoke_viii;
  var invoke_v=env.invoke_v;
  var invoke_iiiii=env.invoke_iiiii;
  var invoke_viiiiii=env.invoke_viiiiii;
  var invoke_iii=env.invoke_iii;
  var invoke_viiii=env.invoke_viiii;
  var ___syscall221=env.___syscall221;
  var ___syscall220=env.___syscall220;
  var __inet_ntop6_raw=env.__inet_ntop6_raw;
  var _SDL_RWFromFile=env._SDL_RWFromFile;
  var ___assert_fail=env.___assert_fail;
  var _emscripten_asm_const_ii=env._emscripten_asm_const_ii;
  var _llvm_exp2_f64=env._llvm_exp2_f64;
  var _SDL_ShowCursor=env._SDL_ShowCursor;
  var _emscripten_set_main_loop_timing=env._emscripten_set_main_loop_timing;
  var _SDL_FreeSurface=env._SDL_FreeSurface;
  var _Mix_PlayChannel=env._Mix_PlayChannel;
  var _emscripten_memcpy_big=env._emscripten_memcpy_big;
  var _Mix_PlayMusic=env._Mix_PlayMusic;
  var _Mix_LoadWAV_RW=env._Mix_LoadWAV_RW;
  var __write_sockaddr=env.__write_sockaddr;
  var ___cxa_atexit=env.___cxa_atexit;
  var __exit=env.__exit;
  var _Mix_HaltMusic=env._Mix_HaltMusic;
  var _SDL_UpdateRect=env._SDL_UpdateRect;
  var ___syscall140=env.___syscall140;
  var ___syscall142=env.___syscall142;
  var ___syscall145=env.___syscall145;
  var ___syscall146=env.___syscall146;
  var _emscripten_asm_const_iiiii=env._emscripten_asm_const_iiiii;
  var _SDL_QuitSubSystem=env._SDL_QuitSubSystem;
  var ___syscall122=env.___syscall122;
  var __inet_ntop4_raw=env.__inet_ntop4_raw;
  var _SDL_GetTicks=env._SDL_GetTicks;
  var _sem_init=env._sem_init;
  var _SDL_VideoDriverName=env._SDL_VideoDriverName;
  var _emscripten_asm_const_i=env._emscripten_asm_const_i;
  var _SDL_LockSurface=env._SDL_LockSurface;
  var ___setErrNo=env.___setErrNo;
  var _emscripten_get_now=env._emscripten_get_now;
  var ___syscall10=env.___syscall10;
  var _SDL_MapRGB=env._SDL_MapRGB;
  var ___syscall3=env.___syscall3;
  var ___lock=env.___lock;
  var _SDL_SetVideoMode=env._SDL_SetVideoMode;
  var ___syscall6=env.___syscall6;
  var ___syscall5=env.___syscall5;
  var ___syscall4=env.___syscall4;
  var _time=env._time;
  var _gettimeofday=env._gettimeofday;
  var _SDL_UpperBlitScaled=env._SDL_UpperBlitScaled;
  var _exit=env._exit;
  var __inet_pton4_raw=env.__inet_pton4_raw;
  var _putenv=env._putenv;
  var ___syscall102=env.___syscall102;
  var _llvm_pow_f64=env._llvm_pow_f64;
  var _IMG_Load=env._IMG_Load;
  var _TTF_FontHeight=env._TTF_FontHeight;
  var ___buildEnvironment=env.___buildEnvironment;
  var _SDL_PumpEvents=env._SDL_PumpEvents;
  var _localtime_r=env._localtime_r;
  var _tzset=env._tzset;
  var _SDL_WM_GrabInput=env._SDL_WM_GrabInput;
  var ___syscall197=env.___syscall197;
  var ___syscall195=env.___syscall195;
  var ___syscall194=env.___syscall194;
  var _sem_wait=env._sem_wait;
  var _TTF_RenderText_Solid=env._TTF_RenderText_Solid;
  var _getenv=env._getenv;
  var ___syscall33=env.___syscall33;
  var ___syscall39=env.___syscall39;
  var ___syscall38=env.___syscall38;
  var _emscripten_asm_const_iiiiii=env._emscripten_asm_const_iiiiii;
  var _gethostbyname=env._gethostbyname;
  var _SDL_UnlockSurface=env._SDL_UnlockSurface;
  var _abort=env._abort;
  var _localtime=env._localtime;
  var ___cxa_pure_virtual=env.___cxa_pure_virtual;
  var _llvm_exp2_f32=env._llvm_exp2_f32;
  var ___syscall40=env.___syscall40;
  var _sem_destroy=env._sem_destroy;
  var _SDL_CloseAudio=env._SDL_CloseAudio;
  var __inet_pton6_raw=env.__inet_pton6_raw;
  var __read_sockaddr=env.__read_sockaddr;
  var _usleep=env._usleep;
  var _SDL_GetVideoInfo=env._SDL_GetVideoInfo;
  var _SDL_FreeRW=env._SDL_FreeRW;
  var _sem_post=env._sem_post;
  var _SDL_PauseAudio=env._SDL_PauseAudio;
  var _Mix_FreeChunk=env._Mix_FreeChunk;
  var _SDL_WM_SetCaption=env._SDL_WM_SetCaption;
  var _IMG_Load_RW=env._IMG_Load_RW;
  var _TTF_SizeText=env._TTF_SizeText;
  var ___syscall54=env.___syscall54;
  var ___unlock=env.___unlock;
  var _SDL_mutexP=env._SDL_mutexP;
  var _emscripten_set_main_loop=env._emscripten_set_main_loop;
  var _SDL_mutexV=env._SDL_mutexV;
  var _emscripten_asm_const_iiii=env._emscripten_asm_const_iiii;
  var _SDL_VideoModeOK=env._SDL_VideoModeOK;
  var _atexit=env._atexit;
  var _SDL_UpperBlit=env._SDL_UpperBlit;
  var _SDL_RWFromConstMem=env._SDL_RWFromConstMem;
  var tempFloat = 0.0;

// EMSCRIPTEN_START_FUNCS
function __Z8ExtFSHFSjtjjs($0, $1, $2, $3, $4) {
 $0 = $0 | 0;
 $1 = $1 | 0;
 $2 = $2 | 0;
 $3 = $3 | 0;
 $4 = $4 | 0;
 var $$$i156 = 0, $$0 = 0, $$0$i = 0, $$0$i$i = 0, $$0$i$i140 = 0, $$0$i$i62 = 0, $$0$i$i67 = 0, $$0$i$i82 = 0, $$0$i104$i = 0, $$0$i110$i = 0, $$0$i118 = 0, $$0$i121 = 0, $$0$i153 = 0, $$0$i72 = 0, $$0$i75 = 0, $$0$i76$i = 0, $$0$i85 = 0, $$0$i92 = 0, $$010$i$i = 0, $$010$i$i131 = 0, $$0100122$i = 0, $$012$i$i = 0, $$012$i$i112 = 0, $$013$i$i = 0, $$013$i$i113 = 0, $$03440$i = 0, $$05$i108$i = 0, $$05$i74$i = 0, $$0511$pn$i$i = 0, $$0511$pn$i$i130 = 0, $$06$i$i = 0, $$06$i$i137 = 0, $$06$i$i79 = 0, $$06$i101$i = 0, $$06$i109$i = 0, $$06$i75$i = 0, $$0612$pn$i$i = 0, $$0612$pn$i$i129 = 0, $$06584$i = 0, $$066$lcssa$i = 0, $$07$i$i = 0, $$07$i$i138 = 0, $$07$i$i80 = 0, $$07$i102$i = 0, $$094$lcssa$i = 0, $$096$$i = 0, $$096$lcssa$i = 0, $$096121$i = 0, $$099$off0$i = 0, $$1$i = 0, $$1$i120 = 0, $$1$i124 = 0, $$1$i155 = 0, $$1$i158 = 0, $$1$i77 = 0, $$1$i87 = 0, $$137$i = 0, $$197$i = 0, $$2$i = 0, $$2$i101 = 0, $$2$i126 = 0, $$2$i97 = 0, $$222$i = 0, $$225$i = 0, $$298$i = 0, $$4$ph$i = 0, $$4$ph$i142 = 0, $$5$i = 0, $$5$i150 = 0, $$6$i = 0, $$6$i152 = 0, $$pn$i$i = 0, $$pn$i$i144 = 0, $$pre$i108 = 0, $$pre$phi$i110Z2D = 0, $$pre$phi$iZ2D = 0, $$pre$phi28$iZ2D = 0, $$sink$i = 0, $$sink$i$i = 0, $$sink$i$i115 = 0, $1000 = 0, $1001 = 0, $1008 = 0, $1019 = 0, $103 = 0, $1034 = 0, $1040 = 0, $1046 = 0, $1052 = 0, $1060 = 0, $1065 = 0, $1070 = 0, $108 = 0, $1081 = 0, $1086 = 0, $1088 = 0, $1097 = 0, $11 = 0, $1103 = 0, $1104 = 0, $1110 = 0, $1116 = 0, $1124 = 0, $1129 = 0, $1134 = 0, $1145 = 0, $1150 = 0, $1152 = 0, $1161 = 0, $1167 = 0, $1174 = 0, $1176 = 0, $1182 = 0, $1184 = 0, $119 = 0, $1190 = 0, $1192 = 0, $1198 = 0, $1199 = 0, $12 = 0, $1213 = 0, $1224 = 0, $1230 = 0, $1233 = 0, $1239 = 0, $124 = 0, $1247 = 0, $1248 = 0, $1253 = 0, $126 = 0, $1260 = 0, $1261 = 0, $1267 = 0, $1269 = 0, $1270 = 0, $1279 = 0, $1281 = 0, $1282 = 0, $1283 = 0, $1284 = 0, $1286 = 0, $1287 = 0, $1291 = 0, $1293 = 0, $1297 = 0, $1311 = 0, $1313 = 0, $1319 = 0, $1320 = 0, $1322 = 0, $1326 = 0, $1327 = 0, $1329 = 0, $1340 = 0, $1344 = 0, $135 = 0, $1350 = 0, $1352 = 0, $1359 = 0, $1360 = 0, $1366 = 0, $1373 = 0, $1380 = 0, $1394 = 0, $1399 = 0, $1404 = 0, $1405 = 0, $1409 = 0, $141 = 0, $1414 = 0, $1418 = 0, $1420 = 0, $1424 = 0, $1430 = 0, $1436 = 0, $1438 = 0, $1445 = 0, $1449 = 0, $1455 = 0, $1462 = 0, $1463 = 0, $1468 = 0, $1474 = 0, $1476 = 0, $148 = 0, $1481 = 0, $1482 = 0, $1485 = 0, $1490 = 0, $1495 = 0, $1500 = 0, $1511 = 0, $1512 = 0, $1513 = 0, $1520 = 0, $1524 = 0, $1529 = 0, $1530 = 0, $1535 = 0, $1537 = 0, $1544 = 0, $1550 = 0, $1551 = 0, $1557 = 0, $1558 = 0, $156 = 0, $1564 = 0, $1569 = 0, $1575 = 0, $1576 = 0, $1582 = 0, $1588 = 0, $1589 = 0, $1595 = 0, $1601 = 0, $1602 = 0, $1607 = 0, $1613 = 0, $1614 = 0, $1619 = 0, $1625 = 0, $1626 = 0, $1631 = 0, $1637 = 0, $1638 = 0, $164 = 0, $1644 = 0, $1650 = 0, $1656 = 0, $1657 = 0, $1662 = 0, $1668 = 0, $1669 = 0, $1674 = 0, $1680 = 0, $1687 = 0, $1689 = 0, $1694 = 0, $1701 = 0, $1707 = 0, $1709 = 0, $1715 = 0, $1717 = 0, $172 = 0, $1723 = 0, $1726 = 0, $1733 = 0, $1744 = 0, $1746 = 0, $1753 = 0, $1759 = 0, $1761 = 0, $1764 = 0, $1765 = 0, $1766 = 0, $177 = 0, $1772 = 0, $1776 = 0, $1780 = 0, $1786 = 0, $1792 = 0, $1793 = 0, $18 = 0, $1801 = 0, $1809 = 0, $1816 = 0, $1817 = 0, $1820 = 0, $1821 = 0, $1827 = 0, $1837 = 0, $1839 = 0, $1842 = 0, $1848 = 0, $1854 = 0, $1860 = 0, $1862 = 0, $1868 = 0, $187 = 0, $1876 = 0, $1882 = 0, $1883 = 0, $1889 = 0, $1897 = 0, $190 = 0, $1903 = 0, $1909 = 0, $1921 = 0, $1926 = 0, $1928 = 0, $1934 = 0, $1939 = 0, $1947 = 0, $1954 = 0, $1961 = 0, $1968 = 0, $1975 = 0, $198 = 0, $1981 = 0, $1983 = 0, $1985 = 0, $1988 = 0, $199 = 0, $1993 = 0, $1994 = 0, $1998 = 0, $20 = 0, $2004 = 0, $2010 = 0, $2012 = 0, $2018 = 0, $2024 = 0, $2030 = 0, $2036 = 0, $2042 = 0, $2048 = 0, $205 = 0, $2054 = 0, $2061 = 0, $2067 = 0, $2074 = 0, $2080 = 0, $2086 = 0, $2093 = 0, $2097 = 0, $2102 = 0, $2103 = 0, $2108 = 0, $214 = 0, $217 = 0, $219 = 0, $22 = 0, $224 = 0, $232 = 0, $237 = 0, $242 = 0, $253 = 0, $258 = 0, $260 = 0, $269 = 0, $272 = 0, $278 = 0, $28 = 0, $285 = 0, $293 = 0, $301 = 0, $309 = 0, $314 = 0, $324 = 0, $327 = 0, $33 = 0, $335 = 0, $336 = 0, $342 = 0, $351 = 0, $352 = 0, $354 = 0, $361 = 0, $367 = 0, $368 = 0, $374 = 0, $38 = 0, $380 = 0, $382 = 0, $388 = 0, $394 = 0, $400 = 0, $406 = 0, $412 = 0, $418 = 0, $424 = 0, $430 = 0, $436 = 0, $44 = 0, $443 = 0, $449 = 0, $456 = 0, $458 = 0, $464 = 0, $470 = 0, $476 = 0, $482 = 0, $488 = 0, $49 = 0, $494 = 0, $5 = 0, $500 = 0, $506 = 0, $51 = 0, $513 = 0, $519 = 0, $520 = 0, $524 = 0, $528 = 0, $534 = 0, $535 = 0, $54 = 0, $540 = 0, $546 = 0, $548 = 0, $55 = 0, $554 = 0, $555 = 0, $558 = 0, $564 = 0, $572 = 0, $573 = 0, $578 = 0, $586 = 0, $594 = 0, $595 = 0, $597 = 0, $598 = 0, $6 = 0, $600 = 0, $601 = 0, $605 = 0, $611 = 0, $618 = 0, $620 = 0, $621 = 0, $623 = 0, $625 = 0, $631 = 0, $634 = 0, $641 = 0, $649 = 0, $651 = 0, $658 = 0, $66 = 0, $664 = 0, $666 = 0, $669 = 0, $670 = 0, $671 = 0, $677 = 0, $679 = 0, $685 = 0, $693 = 0, $699 = 0, $7 = 0, $702 = 0, $711 = 0, $717 = 0, $719 = 0, $725 = 0, $733 = 0, $739 = 0, $740 = 0, $746 = 0, $754 = 0, $760 = 0, $768 = 0, $774 = 0, $780 = 0, $781 = 0, $799 = 0, $8 = 0, $806 = 0, $809 = 0, $810 = 0, $815 = 0, $821 = 0, $823 = 0, $829 = 0, $83 = 0, $834 = 0, $839 = 0, $85 = 0, $850 = 0, $855 = 0, $857 = 0, $866 = 0, $876 = 0, $878 = 0, $884 = 0, $892 = 0, $90 = 0, $904 = 0, $906 = 0, $912 = 0, $917 = 0, $922 = 0, $933 = 0, $938 = 0, $940 = 0, $948 = 0, $949 = 0, $954 = 0, $959 = 0, $965 = 0, $98 = 0, $984 = 0, $990 = 0, $sext$i136 = 0, $sext$i78 = 0, $trunc = 0, dest = 0, label = 0, sp = 0, src = 0, stop = 0;
 sp = STACKTOP; //@line 30229
 STACKTOP = STACKTOP + 1184 | 0; //@line 30230
 $5 = sp + 1100 | 0; //@line 30231
 $6 = sp + 1024 | 0; //@line 30232
 $7 = sp; //@line 30233
 $8 = $1 & 65535; //@line 30234
 $11 = ($8 & 512 | 0) != 0; //@line 30237
 $trunc = $8 & 61695; //@line 30238
 L1 : do {
  if ($trunc << 16 >> 16 < -24555) {
   do {
    switch ($trunc << 16 >> 16) {
    case -24576:
     {
      $12 = $2 + 48 | 0; //@line 30245
      if ($11) {
       $18 = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($12 >>> 16 << 2) >> 2] >> 2] & 127]($12) | 0; //@line 30252
      } else {
       $18 = 0; //@line 30254
      }
      $$0 = __ZL7fs_openjjjb($2, $18, $0, 0) | 0; //@line 30257
      STACKTOP = sp; //@line 30258
      return $$0 | 0; //@line 30258
     }
    case -24575:
     {
      $20 = $2 + 24 | 0; //@line 30262
      $22 = 123608 + ($20 >>> 16 << 2) | 0; //@line 30264
      HEAP32[$5 >> 2] = (FUNCTION_TABLE_ii[HEAP32[(HEAP32[$22 >> 2] | 0) + 4 >> 2] & 127]($20) | 0) << 16 >> 16; //@line 30271
      $28 = HEAP32[30848] | 0; //@line 30272
      HEAP32[$5 + 32 >> 2] = $28 + 306; //@line 30275
      _Execute68k($28 + 752 | 0, $5); //@line 30277
      $33 = (HEAP32[30848] | 0) + 306 | 0; //@line 30279
      $38 = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($33 >>> 16 << 2) >> 2] >> 2] & 127]($33) | 0; //@line 30284
      if (($38 | 0) == 0 | (HEAP32[$5 >> 2] & 65535 | 0) != 0) {
       $$0$i = -51; //@line 30291
      } else {
       $44 = 123608 + ($38 >>> 16 << 2) | 0; //@line 30294
       if (!(FUNCTION_TABLE_ii[HEAP32[HEAP32[$44 >> 2] >> 2] & 127]($38) | 0)) {
        $$0$i = -38; //@line 30300
       } else {
        $49 = $38 + 54 | 0; //@line 30302
        $51 = 123608 + ($49 >>> 16 << 2) | 0; //@line 30304
        $54 = FUNCTION_TABLE_ii[HEAP32[HEAP32[$51 >> 2] >> 2] & 127]($49) | 0; //@line 30307
        $55 = $38 + 4 | 0; //@line 30308
        L13 : do {
         if (!((FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($55 >>> 16 << 2) >> 2] | 0) + 8 >> 2] & 127]($55) | 0) & 2)) {
          _close($54) | 0; //@line 30319
         } else {
          $66 = FUNCTION_TABLE_ii[HEAP32[HEAP32[$44 >> 2] >> 2] & 127]($38) | 0; //@line 30323
          $$06$i$i = HEAP32[30826] | 0; //@line 30324
          if ($$06$i$i | 0) {
           $$07$i$i = $$06$i$i; //@line 30327
           while (1) {
            if ((HEAP32[$$07$i$i + 4 >> 2] | 0) == ($66 | 0)) {
             break;
            }
            $$0$i$i = HEAP32[$$07$i$i >> 2] | 0; //@line 30335
            if (!$$0$i$i) {
             break L13;
            } else {
             $$07$i$i = $$0$i$i; //@line 30340
            }
           }
           __ZL19get_path_for_fsitemP6FSItem($$07$i$i); //@line 30343
           __Z11close_rforkPKci(657423, $54); //@line 30344
          }
         }
        } while (0);
        FUNCTION_TABLE_vii[HEAP32[(HEAP32[$51 >> 2] | 0) + 12 >> 2] & 31]($49, -1); //@line 30351
        HEAP32[$6 >> 2] = FUNCTION_TABLE_ii[HEAP32[(HEAP32[$22 >> 2] | 0) + 4 >> 2] & 127]($20) | 0; //@line 30356
        _Execute68k((HEAP32[30848] | 0) + 724 | 0, $6); //@line 30359
        $$0$i = HEAP32[$6 >> 2] & 65535; //@line 30362
       }
      }
      $$0 = $$0$i; //@line 30365
      STACKTOP = sp; //@line 30366
      return $$0 | 0; //@line 30366
     }
    case -24574:
     {
      $83 = $2 + 36 | 0; //@line 30370
      $85 = 123608 + ($83 >>> 16 << 2) | 0; //@line 30372
      if ((FUNCTION_TABLE_ii[HEAP32[HEAP32[$85 >> 2] >> 2] & 127]($83) | 0) < 0) {
       $$0 = -50; //@line 30378
       STACKTOP = sp; //@line 30379
       return $$0 | 0; //@line 30379
      }
      $90 = $2 + 24 | 0; //@line 30381
      HEAP32[$5 >> 2] = (FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($90 >>> 16 << 2) >> 2] | 0) + 4 >> 2] & 127]($90) | 0) << 16 >> 16; //@line 30390
      $98 = HEAP32[30848] | 0; //@line 30391
      HEAP32[$5 + 32 >> 2] = $98 + 306; //@line 30394
      _Execute68k($98 + 752 | 0, $5); //@line 30396
      $103 = (HEAP32[30848] | 0) + 306 | 0; //@line 30398
      $108 = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($103 >>> 16 << 2) >> 2] >> 2] & 127]($103) | 0; //@line 30403
      if (($108 | 0) == 0 | (HEAP32[$5 >> 2] & 65535 | 0) != 0) {
       $$0 = -51; //@line 30410
       STACKTOP = sp; //@line 30411
       return $$0 | 0; //@line 30411
      }
      if (!(FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($108 >>> 16 << 2) >> 2] >> 2] & 127]($108) | 0)) {
       $$0 = -38; //@line 30420
       STACKTOP = sp; //@line 30421
       return $$0 | 0; //@line 30421
      }
      $119 = $108 + 54 | 0; //@line 30423
      $124 = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($119 >>> 16 << 2) >> 2] >> 2] & 127]($119) | 0; //@line 30428
      if (($124 | 0) < 0) {
       $126 = $108 + 4 | 0; //@line 30431
       $135 = $2 + 40 | 0; //@line 30440
       if (!((FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($126 >>> 16 << 2) >> 2] | 0) + 8 >> 2] & 127]($126) | 0) & 2)) {
        $$0 = -38; //@line 30442
        STACKTOP = sp; //@line 30443
        return $$0 | 0; //@line 30443
       }
       FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($135 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($135, 0); //@line 30450
       $$0 = -39; //@line 30451
       STACKTOP = sp; //@line 30452
       return $$0 | 0; //@line 30452
      }
      $141 = $2 + 44 | 0; //@line 30454
      switch ((FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($141 >>> 16 << 2) >> 2] | 0) + 4 >> 2] & 127]($141) | 0) & 3) {
      case 1:
       {
        $148 = $2 + 46 | 0; //@line 30465
        if ((_lseek($124, FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($148 >>> 16 << 2) >> 2] >> 2] & 127]($148) | 0, 0) | 0) < 0) {
         $$0 = -40; //@line 30474
         STACKTOP = sp; //@line 30475
         return $$0 | 0; //@line 30475
        }
        break;
       }
      case 2:
       {
        $156 = $2 + 46 | 0; //@line 30480
        if ((_lseek($124, FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($156 >>> 16 << 2) >> 2] >> 2] & 127]($156) | 0, 2) | 0) < 0) {
         $$0 = -40; //@line 30489
         STACKTOP = sp; //@line 30490
         return $$0 | 0; //@line 30490
        }
        break;
       }
      case 3:
       {
        $164 = $2 + 46 | 0; //@line 30495
        if ((_lseek($124, FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($164 >>> 16 << 2) >> 2] >> 2] & 127]($164) | 0, 1) | 0) < 0) {
         $$0 = -40; //@line 30504
         STACKTOP = sp; //@line 30505
         return $$0 | 0; //@line 30505
        }
        break;
       }
      default:
       {}
      }
      $172 = $2 + 32 | 0; //@line 30512
      $177 = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($172 >>> 16 << 2) >> 2] >> 2] & 127]($172) | 0; //@line 30517
      $187 = __Z10extfs_readiPvj($124, FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($177 >>> 16 << 2) >> 2] | 0) + 24 >> 2] & 127]($177) | 0, FUNCTION_TABLE_ii[HEAP32[HEAP32[$85 >> 2] >> 2] & 127]($83) | 0) | 0; //@line 30527
      switch (HEAP32[(___errno_location() | 0) >> 2] | 0) {
      case 0:
       {
        $$0$i$i62 = 0; //@line 30532
        break;
       }
      case 21:
      case 2:
       {
        $$0$i$i62 = -43; //@line 30536
        break;
       }
      case 1:
      case 13:
       {
        $$0$i$i62 = -54; //@line 30540
        break;
       }
      case 17:
       {
        $$0$i$i62 = -48; //@line 30544
        break;
       }
      case 39:
      case 16:
       {
        $$0$i$i62 = -47; //@line 30548
        break;
       }
      case 28:
       {
        $$0$i$i62 = -34; //@line 30552
        break;
       }
      case 30:
       {
        $$0$i$i62 = -44; //@line 30556
        break;
       }
      case 24:
       {
        $$0$i$i62 = -42; //@line 30560
        break;
       }
      case 12:
       {
        $$0$i$i62 = -108; //@line 30564
        break;
       }
      default:
       {
        $$0$i$i62 = -36; //@line 30568
       }
      }
      $190 = $2 + 40 | 0; //@line 30571
      FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($190 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($190, ($187 | 0) > 0 ? $187 : 0); //@line 30579
      $198 = _lseek($124, 0, 1) | 0; //@line 30580
      $199 = $108 + 16 | 0; //@line 30581
      FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($199 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($199, $198); //@line 30587
      $205 = $2 + 46 | 0; //@line 30588
      FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($205 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($205, $198); //@line 30594
      $214 = ($187 | 0) == (FUNCTION_TABLE_ii[HEAP32[HEAP32[$85 >> 2] >> 2] & 127]($83) | 0); //@line 30598
      $$0 = $214 ? 0 : ($187 | 0) < 0 ? $$0$i$i62 : -39; //@line 30602
      STACKTOP = sp; //@line 30603
      return $$0 | 0; //@line 30603
     }
    case -24573:
     {
      $217 = $2 + 36 | 0; //@line 30607
      $219 = 123608 + ($217 >>> 16 << 2) | 0; //@line 30609
      if ((FUNCTION_TABLE_ii[HEAP32[HEAP32[$219 >> 2] >> 2] & 127]($217) | 0) < 0) {
       $$0 = -50; //@line 30615
       STACKTOP = sp; //@line 30616
       return $$0 | 0; //@line 30616
      }
      $224 = $2 + 24 | 0; //@line 30618
      HEAP32[$5 >> 2] = (FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($224 >>> 16 << 2) >> 2] | 0) + 4 >> 2] & 127]($224) | 0) << 16 >> 16; //@line 30627
      $232 = HEAP32[30848] | 0; //@line 30628
      HEAP32[$5 + 32 >> 2] = $232 + 306; //@line 30631
      _Execute68k($232 + 752 | 0, $5); //@line 30633
      $237 = (HEAP32[30848] | 0) + 306 | 0; //@line 30635
      $242 = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($237 >>> 16 << 2) >> 2] >> 2] & 127]($237) | 0; //@line 30640
      if (($242 | 0) == 0 | (HEAP32[$5 >> 2] & 65535 | 0) != 0) {
       $$0 = -51; //@line 30647
       STACKTOP = sp; //@line 30648
       return $$0 | 0; //@line 30648
      }
      if (!(FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($242 >>> 16 << 2) >> 2] >> 2] & 127]($242) | 0)) {
       $$0 = -38; //@line 30657
       STACKTOP = sp; //@line 30658
       return $$0 | 0; //@line 30658
      }
      $253 = $242 + 54 | 0; //@line 30660
      $258 = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($253 >>> 16 << 2) >> 2] >> 2] & 127]($253) | 0; //@line 30665
      if (($258 | 0) < 0) {
       $260 = $242 + 4 | 0; //@line 30668
       if (!((FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($260 >>> 16 << 2) >> 2] | 0) + 8 >> 2] & 127]($260) | 0) & 2)) {
        $$0 = -38; //@line 30678
        STACKTOP = sp; //@line 30679
        return $$0 | 0; //@line 30679
       }
       $269 = $2 + 40 | 0; //@line 30681
       $272 = FUNCTION_TABLE_ii[HEAP32[HEAP32[$219 >> 2] >> 2] & 127]($217) | 0; //@line 30684
       FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($269 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($269, $272); //@line 30690
       $$0 = 0; //@line 30691
       STACKTOP = sp; //@line 30692
       return $$0 | 0; //@line 30692
      }
      $278 = $2 + 44 | 0; //@line 30694
      switch ((FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($278 >>> 16 << 2) >> 2] | 0) + 4 >> 2] & 127]($278) | 0) & 3) {
      case 1:
       {
        $285 = $2 + 46 | 0; //@line 30705
        if ((_lseek($258, FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($285 >>> 16 << 2) >> 2] >> 2] & 127]($285) | 0, 0) | 0) < 0) {
         $$0 = -40; //@line 30714
         STACKTOP = sp; //@line 30715
         return $$0 | 0; //@line 30715
        }
        break;
       }
      case 2:
       {
        $293 = $2 + 46 | 0; //@line 30720
        if ((_lseek($258, FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($293 >>> 16 << 2) >> 2] >> 2] & 127]($293) | 0, 2) | 0) < 0) {
         $$0 = -40; //@line 30729
         STACKTOP = sp; //@line 30730
         return $$0 | 0; //@line 30730
        }
        break;
       }
      case 3:
       {
        $301 = $2 + 46 | 0; //@line 30735
        if ((_lseek($258, FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($301 >>> 16 << 2) >> 2] >> 2] & 127]($301) | 0, 1) | 0) < 0) {
         $$0 = -40; //@line 30744
         STACKTOP = sp; //@line 30745
         return $$0 | 0; //@line 30745
        }
        break;
       }
      default:
       {}
      }
      $309 = $2 + 32 | 0; //@line 30752
      $314 = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($309 >>> 16 << 2) >> 2] >> 2] & 127]($309) | 0; //@line 30757
      $324 = __Z11extfs_writeiPvj($258, FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($314 >>> 16 << 2) >> 2] | 0) + 24 >> 2] & 127]($314) | 0, FUNCTION_TABLE_ii[HEAP32[HEAP32[$219 >> 2] >> 2] & 127]($217) | 0) | 0; //@line 30767
      switch (HEAP32[(___errno_location() | 0) >> 2] | 0) {
      case 0:
       {
        $$0$i$i67 = 0; //@line 30772
        break;
       }
      case 21:
      case 2:
       {
        $$0$i$i67 = -43; //@line 30776
        break;
       }
      case 1:
      case 13:
       {
        $$0$i$i67 = -54; //@line 30780
        break;
       }
      case 17:
       {
        $$0$i$i67 = -48; //@line 30784
        break;
       }
      case 39:
      case 16:
       {
        $$0$i$i67 = -47; //@line 30788
        break;
       }
      case 28:
       {
        $$0$i$i67 = -34; //@line 30792
        break;
       }
      case 30:
       {
        $$0$i$i67 = -44; //@line 30796
        break;
       }
      case 24:
       {
        $$0$i$i67 = -42; //@line 30800
        break;
       }
      case 12:
       {
        $$0$i$i67 = -108; //@line 30804
        break;
       }
      default:
       {
        $$0$i$i67 = -36; //@line 30808
       }
      }
      $327 = $2 + 40 | 0; //@line 30811
      FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($327 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($327, ($324 | 0) > 0 ? $324 : 0); //@line 30819
      $335 = _lseek($258, 0, 1) | 0; //@line 30820
      $336 = $242 + 16 | 0; //@line 30821
      FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($336 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($336, $335); //@line 30827
      $342 = $2 + 46 | 0; //@line 30828
      FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($342 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($342, $335); //@line 30834
      $351 = ($324 | 0) == (FUNCTION_TABLE_ii[HEAP32[HEAP32[$219 >> 2] >> 2] & 127]($217) | 0); //@line 30838
      $$0 = $351 ? 0 : $$0$i$i67; //@line 30840
      STACKTOP = sp; //@line 30841
      return $$0 | 0; //@line 30841
     }
    case -24569:
     {
      $352 = $2 + 18 | 0; //@line 30845
      $354 = 123608 + ($352 >>> 16 << 2) | 0; //@line 30847
      if (FUNCTION_TABLE_ii[HEAP32[HEAP32[$354 >> 2] >> 2] & 127]($352) | 0) {
       $361 = FUNCTION_TABLE_ii[HEAP32[HEAP32[$354 >> 2] >> 2] & 127]($352) | 0; //@line 30855
       $367 = FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($361 >>> 16 << 2) >> 2] | 0) + 24 >> 2] & 127]($361) | 0; //@line 30861
       $368 = HEAP8[657390] | 0; //@line 30862
       HEAP8[$367 >> 0] = $368; //@line 30863
       if ($368 << 24 >> 24) {
        $$010$i$i = $368 << 24 >> 24; //@line 30867
        $$0511$pn$i$i = 657390; //@line 30867
        $$0612$pn$i$i = $367; //@line 30867
        do {
         $$0511$pn$i$i = $$0511$pn$i$i + 1 | 0; //@line 30869
         $$0612$pn$i$i = $$0612$pn$i$i + 1 | 0; //@line 30870
         $$010$i$i = $$010$i$i + -1 | 0; //@line 30871
         HEAP8[$$0612$pn$i$i >> 0] = HEAP8[$$0511$pn$i$i >> 0] | 0; //@line 30873
        } while (($$010$i$i | 0) != 0);
       }
      }
      $374 = $2 + 30 | 0; //@line 30883
      FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($374 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($374, 0); //@line 30889
      $380 = $2 + 34 | 0; //@line 30890
      $382 = __Z13TimeToMacTimel(HEAP32[30843] | 0) | 0; //@line 30892
      FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($380 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($380, $382); //@line 30898
      $388 = $2 + 38 | 0; //@line 30899
      FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($388 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($388, 0); //@line 30905
      $394 = $2 + 40 | 0; //@line 30906
      FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($394 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($394, 1); //@line 30912
      $400 = $2 + 42 | 0; //@line 30913
      FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($400 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($400, 0); //@line 30919
      $406 = $2 + 44 | 0; //@line 30920
      FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($406 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($406, 0); //@line 30926
      $412 = $2 + 46 | 0; //@line 30927
      FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($412 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($412, 65535); //@line 30933
      $418 = $2 + 48 | 0; //@line 30934
      FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($418 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($418, 16384); //@line 30940
      $424 = $2 + 52 | 0; //@line 30941
      FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($424 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($424, 16384); //@line 30947
      $430 = $2 + 56 | 0; //@line 30948
      FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($430 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($430, 0); //@line 30954
      $436 = $2 + 58 | 0; //@line 30955
      FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($436 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($436, HEAP32[383] | 0); //@line 30962
      $443 = $2 + 62 | 0; //@line 30963
      FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($443 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($443, 65535); //@line 30969
      if (!$11) {
       $$0 = 0; //@line 30971
       STACKTOP = sp; //@line 30972
       return $$0 | 0; //@line 30972
      }
      $449 = $2 + 66 | 0; //@line 30974
      FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($449 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($449, HEAP32[30849] | 0); //@line 30981
      $456 = $2 + 68 | 0; //@line 30982
      $458 = (HEAP32[30848] | 0) + 26 | 0; //@line 30984
      $464 = FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($458 >>> 16 << 2) >> 2] | 0) + 4 >> 2] & 127]($458) | 0; //@line 30990
      FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($456 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($456, $464); //@line 30996
      $470 = $2 + 70 | 0; //@line 30997
      FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($470 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($470, 25185); //@line 31003
      $476 = $2 + 72 | 0; //@line 31004
      FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($476 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($476, 0); //@line 31010
      $482 = $2 + 76 | 0; //@line 31011
      FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($482 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($482, 0); //@line 31017
      $488 = $2 + 78 | 0; //@line 31018
      FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($488 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($488, 0); //@line 31024
      $494 = $2 + 82 | 0; //@line 31025
      FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($494 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($494, 1); //@line 31031
      $500 = $2 + 86 | 0; //@line 31032
      FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($500 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($500, 1); //@line 31038
      $506 = $2 + 90 | 0; //@line 31039
      dest = FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($506 >>> 16 << 2) >> 2] | 0) + 24 >> 2] & 127]($506) | 0; //@line 31046
      stop = dest + 32 | 0; //@line 31046
      do {
       HEAP8[dest >> 0] = 0; //@line 31046
       dest = dest + 1 | 0; //@line 31046
      } while ((dest | 0) < (stop | 0));
      $$0 = 0; //@line 31047
      STACKTOP = sp; //@line 31048
      return $$0 | 0; //@line 31048
     }
    case -24568:
     {
      $513 = $2 + 48 | 0; //@line 31052
      if ($11) {
       $519 = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($513 >>> 16 << 2) >> 2] >> 2] & 127]($513) | 0; //@line 31059
      } else {
       $519 = 0; //@line 31061
      }
      $520 = __ZL17get_item_and_pathjjRP6FSItemb($2, $519, $5) | 0; //@line 31063
      L112 : do {
       if (!($520 << 16 >> 16)) {
        if (!(_access(657423, 0) | 0)) {
         $$1$i = -48; //@line 31070
        } else {
         $524 = _creat(657423, 438) | 0; //@line 31072
         if (($524 | 0) >= 0) {
          _close($524) | 0; //@line 31075
          $$1$i = 0; //@line 31076
          break;
         }
         switch (HEAP32[(___errno_location() | 0) >> 2] | 0) {
         case 0:
          {
           $$1$i = 0; //@line 31083
           break L112;
           break;
          }
         case 21:
         case 2:
          {
           $$1$i = -43; //@line 31088
           break L112;
           break;
          }
         case 1:
         case 13:
          {
           $$1$i = -54; //@line 31093
           break L112;
           break;
          }
         case 17:
          {
           $$1$i = -48; //@line 31098
           break L112;
           break;
          }
         case 39:
         case 16:
          {
           $$1$i = -47; //@line 31103
           break L112;
           break;
          }
         case 28:
          {
           $$1$i = -34; //@line 31108
           break L112;
           break;
          }
         case 30:
          {
           $$1$i = -44; //@line 31113
           break L112;
           break;
          }
         case 24:
          {
           $$1$i = -42; //@line 31118
           break L112;
           break;
          }
         case 12:
          {
           $$1$i = -108; //@line 31123
           break L112;
           break;
          }
         default:
          {
           $$1$i = -36; //@line 31128
           break L112;
          }
         }
        }
       } else {
        $$1$i = $520; //@line 31134
       }
      } while (0);
      $$0 = $$1$i; //@line 31137
      STACKTOP = sp; //@line 31138
      return $$0 | 0; //@line 31138
     }
    case -24567:
     {
      $528 = $2 + 48 | 0; //@line 31142
      if ($11) {
       $534 = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($528 >>> 16 << 2) >> 2] >> 2] & 127]($528) | 0; //@line 31149
      } else {
       $534 = 0; //@line 31151
      }
      $535 = __ZL17get_item_and_pathjjRP6FSItemb($2, $534, $5) | 0; //@line 31153
      L134 : do {
       if (!($535 << 16 >> 16)) {
        if (__Z12extfs_removePKc(657423) | 0) {
         $$0$i72 = 0; //@line 31159
        } else {
         switch (HEAP32[(___errno_location() | 0) >> 2] | 0) {
         case 0:
          {
           $$0$i72 = 0; //@line 31165
           break L134;
           break;
          }
         case 21:
         case 2:
          {
           $$0$i72 = -43; //@line 31170
           break L134;
           break;
          }
         case 1:
         case 13:
          {
           $$0$i72 = -54; //@line 31175
           break L134;
           break;
          }
         case 17:
          {
           $$0$i72 = -48; //@line 31180
           break L134;
           break;
          }
         case 39:
         case 16:
          {
           $$0$i72 = -47; //@line 31185
           break L134;
           break;
          }
         case 28:
          {
           $$0$i72 = -34; //@line 31190
           break L134;
           break;
          }
         case 30:
          {
           $$0$i72 = -44; //@line 31195
           break L134;
           break;
          }
         case 24:
          {
           $$0$i72 = -42; //@line 31200
           break L134;
           break;
          }
         case 12:
          {
           $$0$i72 = -108; //@line 31205
           break L134;
           break;
          }
         default:
          {
           $$0$i72 = -36; //@line 31210
           break L134;
          }
         }
        }
       } else {
        $$0$i72 = $535; //@line 31216
       }
      } while (0);
      $$0 = $$0$i72; //@line 31219
      STACKTOP = sp; //@line 31220
      return $$0 | 0; //@line 31220
     }
    case -24566:
     {
      $540 = $2 + 48 | 0; //@line 31224
      if ($11) {
       $546 = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($540 >>> 16 << 2) >> 2] >> 2] & 127]($540) | 0; //@line 31231
      } else {
       $546 = 0; //@line 31233
      }
      $$0 = __ZL7fs_openjjjb($2, $546, $0, 1) | 0; //@line 31236
      STACKTOP = sp; //@line 31237
      return $$0 | 0; //@line 31237
     }
    case -24565:
     {
      $548 = $2 + 48 | 0; //@line 31241
      if ($11) {
       $554 = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($548 >>> 16 << 2) >> 2] >> 2] & 127]($548) | 0; //@line 31248
      } else {
       $554 = 0; //@line 31250
      }
      $555 = __ZL17get_item_and_pathjjRP6FSItemb($2, $554, $5) | 0; //@line 31252
      if (!($555 << 16 >> 16)) {
       _strcpy($7, 657423) | 0; //@line 31255
       $558 = (HEAP32[30848] | 0) + 238 | 0; //@line 31257
       $564 = FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($558 >>> 16 << 2) >> 2] | 0) + 24 >> 2] & 127]($558) | 0; //@line 31263
       dest = $564; //@line 31270
       src = FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($2 >>> 16 << 2) >> 2] | 0) + 24 >> 2] & 127]($2) | 0; //@line 31270
       stop = dest + 50 | 0; //@line 31270
       do {
        HEAP8[dest >> 0] = HEAP8[src >> 0] | 0; //@line 31270
        dest = dest + 1 | 0; //@line 31270
        src = src + 1 | 0; //@line 31270
       } while ((dest | 0) < (stop | 0));
       $572 = (HEAP32[30848] | 0) + 256 | 0; //@line 31272
       $573 = $2 + 28 | 0; //@line 31273
       $578 = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($573 >>> 16 << 2) >> 2] >> 2] & 127]($573) | 0; //@line 31278
       FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($572 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($572, $578); //@line 31284
       $586 = __ZL17get_item_and_pathjjRP6FSItemb((HEAP32[30848] | 0) + 238 | 0, $554, $6) | 0; //@line 31287
       L160 : do {
        if (!($586 << 16 >> 16)) {
         if (!(_access(657423, 0) | 0)) {
          $$0$i75 = -48; //@line 31294
         } else {
          if (__Z12extfs_renamePKcS0_($7, 657423) | 0) {
           $594 = (HEAP32[$5 >> 2] | 0) + 4 | 0; //@line 31299
           $595 = HEAP32[$594 >> 2] | 0; //@line 31300
           $597 = (HEAP32[$6 >> 2] | 0) + 4 | 0; //@line 31302
           $598 = HEAP32[$597 >> 2] | 0; //@line 31303
           $$012$i$i = HEAP32[30826] | 0; //@line 31304
           if ($$012$i$i | 0) {
            $$013$i$i = $$012$i$i; //@line 31307
            do {
             $600 = $$013$i$i + 8 | 0; //@line 31309
             $601 = HEAP32[$600 >> 2] | 0; //@line 31310
             if (($601 | 0) == ($595 | 0)) {
              $$sink$i$i = $598; //@line 31313
              label = 117; //@line 31314
             } else {
              if (($601 | 0) == ($598 | 0)) {
               $$sink$i$i = $595; //@line 31318
               label = 117; //@line 31319
              }
             }
             if ((label | 0) == 117) {
              label = 0; //@line 31323
              HEAP32[$600 >> 2] = $$sink$i$i; //@line 31324
             }
             $$013$i$i = HEAP32[$$013$i$i >> 2] | 0; //@line 31326
            } while (($$013$i$i | 0) != 0);
           }
           HEAP32[$594 >> 2] = $598; //@line 31335
           HEAP32[$597 >> 2] = $595; //@line 31336
           $$0$i75 = 0; //@line 31337
           break;
          }
          switch (HEAP32[(___errno_location() | 0) >> 2] | 0) {
          case 0:
           {
            $$0$i75 = 0; //@line 31344
            break L160;
            break;
           }
          case 21:
          case 2:
           {
            $$0$i75 = -43; //@line 31349
            break L160;
            break;
           }
          case 1:
          case 13:
           {
            $$0$i75 = -54; //@line 31354
            break L160;
            break;
           }
          case 17:
           {
            $$0$i75 = -48; //@line 31359
            break L160;
            break;
           }
          case 39:
          case 16:
           {
            $$0$i75 = -47; //@line 31364
            break L160;
            break;
           }
          case 28:
           {
            $$0$i75 = -34; //@line 31369
            break L160;
            break;
           }
          case 30:
           {
            $$0$i75 = -44; //@line 31374
            break L160;
            break;
           }
          case 24:
           {
            $$0$i75 = -42; //@line 31379
            break L160;
            break;
           }
          case 12:
           {
            $$0$i75 = -108; //@line 31384
            break L160;
            break;
           }
          default:
           {
            $$0$i75 = -36; //@line 31389
            break L160;
           }
          }
         }
        } else {
         $$0$i75 = $586; //@line 31395
        }
       } while (0);
       $$1$i77 = $$0$i75; //@line 31398
      } else {
       $$1$i77 = $555; //@line 31400
      }
      $$0 = $$1$i77; //@line 31402
      STACKTOP = sp; //@line 31403
      return $$0 | 0; //@line 31403
     }
    case -24564:
     {
      $605 = $2 + 48 | 0; //@line 31407
      if ($11) {
       $620 = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($605 >>> 16 << 2) >> 2] >> 2] & 127]($605) | 0; //@line 31414
      } else {
       $620 = 0; //@line 31416
      }
      $611 = $2 + 28 | 0; //@line 31418
      $sext$i78 = (FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($611 >>> 16 << 2) >> 2] | 0) + 4 >> 2] & 127]($611) | 0) << 16; //@line 31425
      $618 = $sext$i78 >> 16; //@line 31426
      L192 : do {
       if (($sext$i78 | 0) < 65536) {
        $621 = __ZL17get_item_and_pathjjRP6FSItemb($2, $620, $5) | 0; //@line 31430
        if (!($621 << 16 >> 16)) {
         label = 139; //@line 31433
        } else {
         $$6$i = $621; //@line 31435
        }
       } else {
        $623 = __ZL15get_current_dirjjRjb($2, $620, $6, 1) | 0; //@line 31438
        L195 : do {
         if (!($623 << 16 >> 16)) {
          $625 = HEAP32[$6 >> 2] | 0; //@line 31442
          $$06$i$i79 = HEAP32[30826] | 0; //@line 31443
          if (!$$06$i$i79) {
           $$4$ph$i = -120; //@line 31446
          } else {
           $$07$i$i80 = $$06$i$i79; //@line 31448
           while (1) {
            if ((HEAP32[$$07$i$i80 + 4 >> 2] | 0) == ($625 | 0)) {
             break;
            }
            $$0$i$i82 = HEAP32[$$07$i$i80 >> 2] | 0; //@line 31456
            if (!$$0$i$i82) {
             $$4$ph$i = -120; //@line 31459
             break L195;
            } else {
             $$07$i$i80 = $$0$i$i82; //@line 31462
            }
           }
           __ZL19get_path_for_fsitemP6FSItem($$07$i$i80); //@line 31465
           $631 = _opendir(657423) | 0; //@line 31466
           if (!$631) {
            $$4$ph$i = -120; //@line 31469
           } else {
            L202 : do {
             if (($sext$i78 | 0) > 0) {
              $$06584$i = 0; //@line 31474
              L203 : while (1) {
               do {
                $634 = _readdir($631) | 0; //@line 31477
                if (!$634) {
                 break L203;
                }
               } while ((HEAP8[$634 + 11 >> 0] | 0) == 46);
               $$06584$i = $$06584$i + 1 | 0; //@line 31489
               if (($$06584$i | 0) >= ($618 | 0)) {
                $$066$lcssa$i = $634; //@line 31494
                break L202;
               }
              }
              _closedir($631) | 0; //@line 31498
              $$4$ph$i = -43; //@line 31499
              break L195;
             } else {
              $$066$lcssa$i = 0; //@line 31502
             }
            } while (0);
            $641 = $$066$lcssa$i + 11 | 0; //@line 31505
            __Z18add_path_componentPcPKc(657423, $641); //@line 31506
            HEAP32[$5 >> 2] = __ZL11find_fsitemPKcP6FSItem($641, $$07$i$i80) | 0; //@line 31508
            _closedir($631) | 0; //@line 31509
            label = 139; //@line 31510
            break L192;
           }
          }
         } else {
          $$4$ph$i = $623; //@line 31515
         }
        } while (0);
        $$6$i = $$4$ph$i; //@line 31518
       }
      } while (0);
      if ((label | 0) == 139) {
       if (!(_stat(657423, $6) | 0)) {
        if ((HEAP32[$6 + 12 >> 2] & 61440 | 0) == 16384) {
         $$5$i = -43; //@line 31530
        } else {
         $649 = $2 + 18 | 0; //@line 31532
         $651 = 123608 + ($649 >>> 16 << 2) | 0; //@line 31534
         L217 : do {
          if (FUNCTION_TABLE_ii[HEAP32[HEAP32[$651 >> 2] >> 2] & 127]($649) | 0) {
           $658 = FUNCTION_TABLE_ii[HEAP32[HEAP32[$651 >> 2] >> 2] & 127]($649) | 0; //@line 31543
           $664 = FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($658 >>> 16 << 2) >> 2] | 0) + 24 >> 2] & 127]($658) | 0; //@line 31549
           $666 = (HEAP32[$5 >> 2] | 0) + 20 | 0; //@line 31551
           HEAP8[$664 >> 0] = _strlen($666) | 0; //@line 31554
           $$05$i74$i = $666; //@line 31555
           $$pn$i$i = $664; //@line 31555
           while (1) {
            $$06$i75$i = $$pn$i$i + 1 | 0; //@line 31557
            $669 = $$05$i74$i + 1 | 0; //@line 31558
            $670 = HEAP8[$$05$i74$i >> 0] | 0; //@line 31559
            switch ($670 << 24 >> 24) {
            case 0:
             {
              break L217;
              break;
             }
            case 58:
             {
              $$0$i76$i = 47; //@line 31566
              break;
             }
            default:
             {
              $$0$i76$i = $670; //@line 31570
             }
            }
            HEAP8[$$06$i75$i >> 0] = $$0$i76$i; //@line 31573
            $$05$i74$i = $669; //@line 31574
            $$pn$i$i = $$06$i75$i; //@line 31574
           }
          }
         } while (0);
         $671 = $2 + 24 | 0; //@line 31578
         FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($671 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($671, 0); //@line 31584
         $677 = $2 + 30 | 0; //@line 31585
         $679 = (_access(657423, 2) | 0) != 0 & 1; //@line 31588
         FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($677 >>> 16 << 2) >> 2] | 0) + 20 >> 2] & 31]($677, $679); //@line 31594
         $685 = HEAP32[$5 >> 2] | 0; //@line 31595
         FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($605 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($605, HEAP32[$685 + 4 >> 2] | 0); //@line 31603
         $693 = $2 + 72 | 0; //@line 31604
         FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($693 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($693, 0); //@line 31610
         $699 = $2 + 76 | 0; //@line 31611
         $702 = __Z13TimeToMacTimel(HEAP32[$6 + 56 >> 2] | 0) | 0; //@line 31614
         FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($699 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($699, $702); //@line 31620
         __Z9get_finfoPKcjjb(657423, $2 + 32 | 0, $11 ? $2 + 84 | 0 : 0, 0); //@line 31624
         $711 = $2 + 52 | 0; //@line 31625
         FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($711 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($711, 0); //@line 31631
         $717 = $2 + 54 | 0; //@line 31632
         $719 = HEAP32[$6 + 36 >> 2] | 0; //@line 31634
         FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($717 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($717, $719); //@line 31640
         $725 = $2 + 58 | 0; //@line 31641
         FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($725 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($725, ($719 | 16383) + 1 | 0); //@line 31649
         $733 = $2 + 62 | 0; //@line 31650
         FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($733 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($733, 0); //@line 31656
         $739 = __Z14get_rfork_sizePKc(657423) | 0; //@line 31657
         $740 = $2 + 64 | 0; //@line 31658
         FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($740 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($740, $739); //@line 31664
         $746 = $2 + 68 | 0; //@line 31665
         FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($746 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($746, ($739 | 16383) + 1 | 0); //@line 31673
         if ($11) {
          $754 = $2 + 80 | 0; //@line 31675
          FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($754 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($754, 0); //@line 31681
          $760 = $2 + 100 | 0; //@line 31682
          FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($760 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($760, HEAP32[$685 + 8 >> 2] | 0); //@line 31690
          $768 = $2 + 104 | 0; //@line 31691
          FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($768 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($768, 0); //@line 31697
          $$5$i = 0; //@line 31698
         } else {
          $$5$i = 0; //@line 31700
         }
        }
       } else {
        $$5$i = -43; //@line 31704
       }
       $$6$i = $$5$i; //@line 31706
      }
      $$0 = $$6$i; //@line 31708
      STACKTOP = sp; //@line 31709
      return $$0 | 0; //@line 31709
     }
    case -24563:
     {
      $774 = $2 + 48 | 0; //@line 31713
      if ($11) {
       $780 = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($774 >>> 16 << 2) >> 2] >> 2] & 127]($774) | 0; //@line 31720
      } else {
       $780 = 0; //@line 31722
      }
      $781 = __ZL17get_item_and_pathjjRP6FSItemb($2, $780, $5) | 0; //@line 31724
      if (!($781 << 16 >> 16)) {
       L235 : do {
        if ((_stat(657423, $6) | 0) < 0) {
         switch (HEAP32[(___errno_location() | 0) >> 2] | 0) {
         case 0:
          {
           $$0$i85 = 0; //@line 31735
           break L235;
           break;
          }
         case 21:
         case 2:
          {
           $$0$i85 = -43; //@line 31740
           break L235;
           break;
          }
         case 1:
         case 13:
          {
           $$0$i85 = -54; //@line 31745
           break L235;
           break;
          }
         case 17:
          {
           $$0$i85 = -48; //@line 31750
           break L235;
           break;
          }
         case 39:
         case 16:
          {
           $$0$i85 = -47; //@line 31755
           break L235;
           break;
          }
         case 28:
          {
           $$0$i85 = -34; //@line 31760
           break L235;
           break;
          }
         case 30:
          {
           $$0$i85 = -44; //@line 31765
           break L235;
           break;
          }
         case 24:
          {
           $$0$i85 = -42; //@line 31770
           break L235;
           break;
          }
         case 12:
          {
           $$0$i85 = -108; //@line 31775
           break L235;
           break;
          }
         default:
          {
           $$0$i85 = -36; //@line 31780
           break L235;
          }
         }
        } else {
         if ((HEAP32[$6 + 12 >> 2] & 61440 | 0) == 16384) {
          $$0$i85 = -43; //@line 31790
         } else {
          __Z9set_finfoPKcjjb(657423, $2 + 32 | 0, $11 ? $2 + 84 | 0 : 0, 0); //@line 31795
          $$0$i85 = 0; //@line 31796
         }
        }
       } while (0);
       $$1$i87 = $$0$i85; //@line 31800
      } else {
       $$1$i87 = $781; //@line 31802
      }
      $$0 = $$1$i87; //@line 31804
      STACKTOP = sp; //@line 31805
      return $$0 | 0; //@line 31805
     }
    case -24562:
     {
      HEAP32[$5 + 32 >> 2] = $0; //@line 31810
      _Execute68k((HEAP32[30848] | 0) + 670 | 0, $5); //@line 31813
      $$0 = HEAP32[$5 >> 2] & 65535; //@line 31816
      STACKTOP = sp; //@line 31817
      return $$0 | 0; //@line 31817
     }
    case -24561:
     {
      $799 = $2 + 22 | 0; //@line 31821
      $806 = (FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($799 >>> 16 << 2) >> 2] | 0) + 4 >> 2] & 127]($799) | 0) << 16 >> 16; //@line 31829
      $$0 = ($806 | 0) == (HEAP32[30849] | 0) ? 0 : -58; //@line 31833
      STACKTOP = sp; //@line 31834
      return $$0 | 0; //@line 31834
     }
    case -24560:
     {
      $809 = $2 + 40 | 0; //@line 31838
      $810 = $2 + 36 | 0; //@line 31839
      $815 = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($810 >>> 16 << 2) >> 2] >> 2] & 127]($810) | 0; //@line 31844
      FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($809 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($809, $815); //@line 31850
      $$0 = 0; //@line 31851
      STACKTOP = sp; //@line 31852
      return $$0 | 0; //@line 31852
     }
    case -24559:
     {
      $821 = $2 + 24 | 0; //@line 31856
      $823 = 123608 + ($821 >>> 16 << 2) | 0; //@line 31858
      HEAP32[$5 >> 2] = (FUNCTION_TABLE_ii[HEAP32[(HEAP32[$823 >> 2] | 0) + 4 >> 2] & 127]($821) | 0) << 16 >> 16; //@line 31865
      $829 = HEAP32[30848] | 0; //@line 31866
      HEAP32[$5 + 32 >> 2] = $829 + 306; //@line 31869
      _Execute68k($829 + 752 | 0, $5); //@line 31871
      $834 = (HEAP32[30848] | 0) + 306 | 0; //@line 31873
      $839 = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($834 >>> 16 << 2) >> 2] >> 2] & 127]($834) | 0; //@line 31878
      do {
       if (($839 | 0) == 0 | (HEAP32[$5 >> 2] & 65535 | 0) != 0) {
        $$2$i = -51; //@line 31886
       } else {
        if (!(FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($839 >>> 16 << 2) >> 2] >> 2] & 127]($839) | 0)) {
         $$2$i = -38; //@line 31895
        } else {
         $850 = $839 + 54 | 0; //@line 31897
         $855 = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($850 >>> 16 << 2) >> 2] >> 2] & 127]($850) | 0; //@line 31902
         if (($855 | 0) < 0) {
          $857 = $839 + 4 | 0; //@line 31905
          $866 = $2 + 28 | 0; //@line 31914
          if (!((FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($857 >>> 16 << 2) >> 2] | 0) + 8 >> 2] & 127]($857) | 0) & 2)) {
           $$2$i = -38; //@line 31916
           break;
          }
          FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($866 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($866, 0); //@line 31924
          $$2$i = 0; //@line 31925
          break;
         }
         L266 : do {
          if ((_fstat($855, $5) | 0) < 0) {
           switch (HEAP32[(___errno_location() | 0) >> 2] | 0) {
           case 0:
            {
             $$0$i92 = 0; //@line 31936
             break L266;
             break;
            }
           case 21:
           case 2:
            {
             $$0$i92 = -43; //@line 31941
             break L266;
             break;
            }
           case 1:
           case 13:
            {
             $$0$i92 = -54; //@line 31946
             break L266;
             break;
            }
           case 17:
            {
             $$0$i92 = -48; //@line 31951
             break L266;
             break;
            }
           case 39:
           case 16:
            {
             $$0$i92 = -47; //@line 31956
             break L266;
             break;
            }
           case 28:
            {
             $$0$i92 = -34; //@line 31961
             break L266;
             break;
            }
           case 30:
            {
             $$0$i92 = -44; //@line 31966
             break L266;
             break;
            }
           case 24:
            {
             $$0$i92 = -42; //@line 31971
             break L266;
             break;
            }
           case 12:
            {
             $$0$i92 = -108; //@line 31976
             break L266;
             break;
            }
           default:
            {
             $$0$i92 = -36; //@line 31981
             break L266;
            }
           }
          } else {
           $876 = $839 + 8 | 0; //@line 31986
           $878 = HEAP32[$5 + 36 >> 2] | 0; //@line 31988
           FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($876 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($876, $878); //@line 31994
           $884 = $839 + 12 | 0; //@line 31995
           FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($884 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($884, ($878 | 16383) + 1 | 0); //@line 32003
           $892 = $2 + 28 | 0; //@line 32004
           FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($892 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($892, $878); //@line 32010
           HEAP32[$6 >> 2] = FUNCTION_TABLE_ii[HEAP32[(HEAP32[$823 >> 2] | 0) + 4 >> 2] & 127]($821) | 0; //@line 32015
           _Execute68k((HEAP32[30848] | 0) + 766 | 0, $6); //@line 32018
           $$0$i92 = 0; //@line 32019
          }
         } while (0);
         $$2$i = $$0$i92; //@line 32022
        }
       }
      } while (0);
      $$0 = $$2$i; //@line 32026
      STACKTOP = sp; //@line 32027
      return $$0 | 0; //@line 32027
     }
    case -24558:
     {
      $904 = $2 + 24 | 0; //@line 32031
      $906 = 123608 + ($904 >>> 16 << 2) | 0; //@line 32033
      HEAP32[$5 >> 2] = (FUNCTION_TABLE_ii[HEAP32[(HEAP32[$906 >> 2] | 0) + 4 >> 2] & 127]($904) | 0) << 16 >> 16; //@line 32040
      $912 = HEAP32[30848] | 0; //@line 32041
      HEAP32[$5 + 32 >> 2] = $912 + 306; //@line 32044
      _Execute68k($912 + 752 | 0, $5); //@line 32046
      $917 = (HEAP32[30848] | 0) + 306 | 0; //@line 32048
      $922 = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($917 >>> 16 << 2) >> 2] >> 2] & 127]($917) | 0; //@line 32053
      L283 : do {
       if (($922 | 0) == 0 | (HEAP32[$5 >> 2] & 65535 | 0) != 0) {
        $$2$i97 = -51; //@line 32061
       } else {
        if (!(FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($922 >>> 16 << 2) >> 2] >> 2] & 127]($922) | 0)) {
         $$2$i97 = -38; //@line 32070
        } else {
         $933 = $922 + 54 | 0; //@line 32072
         $938 = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($933 >>> 16 << 2) >> 2] >> 2] & 127]($933) | 0; //@line 32077
         if (($938 | 0) < 0) {
          $940 = $922 + 4 | 0; //@line 32080
          $948 = ((FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($940 >>> 16 << 2) >> 2] | 0) + 8 >> 2] & 127]($940) | 0) & 2 | 0) == 0; //@line 32088
          $$2$i97 = $948 ? -38 : 0; //@line 32090
          break;
         }
         $949 = $2 + 28 | 0; //@line 32093
         $954 = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($949 >>> 16 << 2) >> 2] >> 2] & 127]($949) | 0; //@line 32098
         if ((_ftruncate($938, $954) | 0) >= 0) {
          $959 = $922 + 8 | 0; //@line 32102
          FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($959 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($959, $954); //@line 32108
          $965 = $922 + 12 | 0; //@line 32109
          FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($965 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($965, ($954 | 16383) + 1 | 0); //@line 32117
          HEAP32[$6 >> 2] = FUNCTION_TABLE_ii[HEAP32[(HEAP32[$906 >> 2] | 0) + 4 >> 2] & 127]($904) | 0; //@line 32122
          _Execute68k((HEAP32[30848] | 0) + 766 | 0, $6); //@line 32125
          $$2$i97 = 0; //@line 32126
          break;
         }
         switch (HEAP32[(___errno_location() | 0) >> 2] | 0) {
         case 0:
          {
           $$2$i97 = 0; //@line 32133
           break L283;
           break;
          }
         case 21:
         case 2:
          {
           $$2$i97 = -43; //@line 32138
           break L283;
           break;
          }
         case 1:
         case 13:
          {
           $$2$i97 = -54; //@line 32143
           break L283;
           break;
          }
         case 17:
          {
           $$2$i97 = -48; //@line 32148
           break L283;
           break;
          }
         case 39:
         case 16:
          {
           $$2$i97 = -47; //@line 32153
           break L283;
           break;
          }
         case 28:
          {
           $$2$i97 = -34; //@line 32158
           break L283;
           break;
          }
         case 30:
          {
           $$2$i97 = -44; //@line 32163
           break L283;
           break;
          }
         case 24:
          {
           $$2$i97 = -42; //@line 32168
           break L283;
           break;
          }
         case 12:
          {
           $$2$i97 = -108; //@line 32173
           break L283;
           break;
          }
         default:
          {
           $$2$i97 = -36; //@line 32178
           break L283;
          }
         }
        }
       }
      } while (0);
      $$0 = $$2$i97; //@line 32185
      STACKTOP = sp; //@line 32186
      return $$0 | 0; //@line 32186
     }
    case -24556:
     {
      HEAP32[$5 + 32 >> 2] = $2; //@line 32191
      _Execute68k((HEAP32[30848] | 0) + 632 | 0, $5); //@line 32194
      $$0 = HEAP32[$5 >> 2] & 65535; //@line 32197
      STACKTOP = sp; //@line 32198
      return $$0 | 0; //@line 32198
     }
    case -24557:
     {
      $$0 = 0; //@line 32202
      STACKTOP = sp; //@line 32203
      return $$0 | 0; //@line 32203
     }
    default:
     {
      break L1;
     }
    }
   } while (0);
  } else {
   if ($trunc << 16 >> 16 < 2) {
    if ($trunc << 16 >> 16 < -24511) {
     switch ($trunc << 16 >> 16) {
     case -24555:
      {
       do {
        if ($11) {
         $984 = $2 + 48 | 0; //@line 32220
         $990 = __ZL17get_item_and_pathjjRP6FSItemb($2, FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($984 >>> 16 << 2) >> 2] >> 2] & 127]($984) | 0, $6) | 0; //@line 32226
         if ($990 << 16 >> 16) {
          $$2$i101 = $990; //@line 32229
          break;
         }
         if (!(_stat(657423, $7) | 0)) {
          if ((HEAP32[$7 + 12 >> 2] & 61440 | 0) == 16384) {
           $1000 = HEAP32[(HEAP32[$6 >> 2] | 0) + 4 >> 2] | 0; //@line 32242
           $1001 = $0 + 78 | 0; //@line 32243
           $$222$i = FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($1001 >>> 16 << 2) >> 2] | 0) + 4 >> 2] & 127]($1001) | 0; //@line 32250
           $$225$i = $1000; //@line 32250
           $$pre$phi$iZ2D = $5; //@line 32250
           label = 216; //@line 32251
           break;
          }
         }
         $$2$i101 = -120; //@line 32255
        } else {
         $1008 = $2 + 22 | 0; //@line 32257
         HEAP32[$5 >> 2] = FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($1008 >>> 16 << 2) >> 2] | 0) + 4 >> 2] & 127]($1008) | 0; //@line 32264
         _Execute68k((HEAP32[30848] | 0) + 682 | 0, $5); //@line 32267
         $1019 = (HEAP32[$5 >> 2] & 65535 | 0) == 0; //@line 32270
         $$sink$i = $1019 ? $1008 : $0 + 78 | 0; //@line 32272
         $$222$i = FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($$sink$i >>> 16 << 2) >> 2] | 0) + 4 >> 2] & 127]($$sink$i) | 0; //@line 32280
         $$225$i = $1019 ? 0 : 2; //@line 32280
         $$pre$phi$iZ2D = $5; //@line 32280
         label = 216; //@line 32281
        }
       } while (0);
       if ((label | 0) == 216) {
        HEAP32[$$pre$phi$iZ2D >> 2] = 0; //@line 32285
        HEAP32[$5 + 4 >> 2] = $$225$i; //@line 32287
        HEAP32[$5 + 8 >> 2] = $$222$i << 16 >> 16; //@line 32291
        _Execute68k((HEAP32[30848] | 0) + 694 | 0, $5); //@line 32294
        $$2$i101 = HEAP32[$$pre$phi$iZ2D >> 2] & 65535; //@line 32297
       }
       $$0 = $$2$i101; //@line 32299
       STACKTOP = sp; //@line 32300
       return $$0 | 0; //@line 32300
      }
     case -24552:
      {
       $1034 = $2 + 36 | 0; //@line 32304
       FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($1034 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($1034, 0); //@line 32310
       $1040 = $2 + 40 | 0; //@line 32311
       FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($1040 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($1040, 0); //@line 32317
       $1046 = $2 + 44 | 0; //@line 32318
       FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($1046 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($1046, 0); //@line 32324
       $1052 = $2 + 24 | 0; //@line 32325
       HEAP32[$5 >> 2] = (FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($1052 >>> 16 << 2) >> 2] | 0) + 4 >> 2] & 127]($1052) | 0) << 16 >> 16; //@line 32334
       $1060 = HEAP32[30848] | 0; //@line 32335
       HEAP32[$5 + 32 >> 2] = $1060 + 306; //@line 32338
       _Execute68k($1060 + 752 | 0, $5); //@line 32340
       $1065 = (HEAP32[30848] | 0) + 306 | 0; //@line 32342
       $1070 = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($1065 >>> 16 << 2) >> 2] >> 2] & 127]($1065) | 0; //@line 32347
       if (($1070 | 0) == 0 | (HEAP32[$5 >> 2] & 65535 | 0) != 0) {
        $$0 = -51; //@line 32354
        STACKTOP = sp; //@line 32355
        return $$0 | 0; //@line 32355
       }
       if (!(FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($1070 >>> 16 << 2) >> 2] >> 2] & 127]($1070) | 0)) {
        $$0 = -38; //@line 32364
        STACKTOP = sp; //@line 32365
        return $$0 | 0; //@line 32365
       }
       $1081 = $1070 + 54 | 0; //@line 32367
       $1086 = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($1081 >>> 16 << 2) >> 2] >> 2] & 127]($1081) | 0; //@line 32372
       if (($1086 | 0) >= 0) {
        $1103 = _lseek($1086, 0, 1) | 0; //@line 32375
        $1104 = $1070 + 16 | 0; //@line 32376
        FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($1104 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($1104, $1103); //@line 32382
        $1110 = $2 + 46 | 0; //@line 32383
        FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($1110 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($1110, $1103); //@line 32389
        $$0 = 0; //@line 32390
        STACKTOP = sp; //@line 32391
        return $$0 | 0; //@line 32391
       }
       $1088 = $1070 + 4 | 0; //@line 32393
       $1097 = $2 + 46 | 0; //@line 32402
       if (!((FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($1088 >>> 16 << 2) >> 2] | 0) + 8 >> 2] & 127]($1088) | 0) & 2)) {
        $$0 = -38; //@line 32404
        STACKTOP = sp; //@line 32405
        return $$0 | 0; //@line 32405
       }
       FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($1097 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($1097, 0); //@line 32412
       $$0 = 0; //@line 32413
       STACKTOP = sp; //@line 32414
       return $$0 | 0; //@line 32414
      }
     case -24523:
     case -24553:
      {
       $$0 = 0; //@line 32418
       STACKTOP = sp; //@line 32419
       return $$0 | 0; //@line 32419
      }
     default:
      {
       break L1;
      }
     }
    }
    if ($trunc << 16 >> 16 < -24508) {
     switch ($trunc << 16 >> 16) {
     case -24510:
     case -24511:
      {
       $$0 = 0; //@line 32431
       break;
      }
     default:
      {
       break L1;
      }
     }
     STACKTOP = sp; //@line 32438
     return $$0 | 0; //@line 32438
    }
    if ($trunc << 16 >> 16 >= -24507) {
     if ($trunc << 16 >> 16 < 1) {
      switch ($trunc << 16 >> 16) {
      case -24507:
       {
        $$0 = 0; //@line 32446
        break;
       }
      default:
       {
        break L1;
       }
      }
      STACKTOP = sp; //@line 32453
      return $$0 | 0; //@line 32453
     }
     switch ($trunc << 16 >> 16) {
     case 1:
      {
       break;
      }
     default:
      {
       break L1;
      }
     }
     HEAP32[$5 + 32 >> 2] = $2; //@line 32464
     _Execute68k((HEAP32[30848] | 0) + 778 | 0, $5); //@line 32467
     $$0 = HEAP32[$5 >> 2] & 65535; //@line 32470
     STACKTOP = sp; //@line 32471
     return $$0 | 0; //@line 32471
    }
    switch ($trunc << 16 >> 16) {
    case -24508:
     {
      break;
     }
    default:
     {
      break L1;
     }
    }
    $1116 = $2 + 24 | 0; //@line 32481
    HEAP32[$5 >> 2] = (FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($1116 >>> 16 << 2) >> 2] | 0) + 4 >> 2] & 127]($1116) | 0) << 16 >> 16; //@line 32490
    $1124 = HEAP32[30848] | 0; //@line 32491
    HEAP32[$5 + 32 >> 2] = $1124 + 306; //@line 32494
    _Execute68k($1124 + 752 | 0, $5); //@line 32496
    $1129 = (HEAP32[30848] | 0) + 306 | 0; //@line 32498
    $1134 = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($1129 >>> 16 << 2) >> 2] >> 2] & 127]($1129) | 0; //@line 32503
    if (($1134 | 0) == 0 | (HEAP32[$5 >> 2] & 65535 | 0) != 0) {
     $$0 = -51; //@line 32510
     STACKTOP = sp; //@line 32511
     return $$0 | 0; //@line 32511
    }
    if (!(FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($1134 >>> 16 << 2) >> 2] >> 2] & 127]($1134) | 0)) {
     $$0 = -38; //@line 32520
     STACKTOP = sp; //@line 32521
     return $$0 | 0; //@line 32521
    }
    $1145 = $1134 + 54 | 0; //@line 32523
    $1150 = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($1145 >>> 16 << 2) >> 2] >> 2] & 127]($1145) | 0; //@line 32528
    if (($1150 | 0) < 0) {
     $1152 = $1134 + 4 | 0; //@line 32531
     $1161 = $2 + 46 | 0; //@line 32540
     if (!((FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($1152 >>> 16 << 2) >> 2] | 0) + 8 >> 2] & 127]($1152) | 0) & 2)) {
      $$0 = -38; //@line 32542
      STACKTOP = sp; //@line 32543
      return $$0 | 0; //@line 32543
     }
     FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($1161 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($1161, 0); //@line 32550
     $$0 = 0; //@line 32551
     STACKTOP = sp; //@line 32552
     return $$0 | 0; //@line 32552
    }
    $1167 = $2 + 44 | 0; //@line 32554
    switch ((FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($1167 >>> 16 << 2) >> 2] | 0) + 4 >> 2] & 127]($1167) | 0) & 3) {
    case 1:
     {
      $1174 = $2 + 46 | 0; //@line 32565
      $1176 = 123608 + ($1174 >>> 16 << 2) | 0; //@line 32567
      if ((_lseek($1150, FUNCTION_TABLE_ii[HEAP32[HEAP32[$1176 >> 2] >> 2] & 127]($1174) | 0, 0) | 0) < 0) {
       $$0 = -40; //@line 32574
       STACKTOP = sp; //@line 32575
       return $$0 | 0; //@line 32575
      } else {
       $$pre$phi$i110Z2D = $1174; //@line 32577
       $$pre$phi28$iZ2D = $1176; //@line 32577
      }
      break;
     }
    case 2:
     {
      $1182 = $2 + 46 | 0; //@line 32582
      $1184 = 123608 + ($1182 >>> 16 << 2) | 0; //@line 32584
      if ((_lseek($1150, FUNCTION_TABLE_ii[HEAP32[HEAP32[$1184 >> 2] >> 2] & 127]($1182) | 0, 2) | 0) < 0) {
       $$0 = -40; //@line 32591
       STACKTOP = sp; //@line 32592
       return $$0 | 0; //@line 32592
      } else {
       $$pre$phi$i110Z2D = $1182; //@line 32594
       $$pre$phi28$iZ2D = $1184; //@line 32594
      }
      break;
     }
    case 3:
     {
      $1190 = $2 + 46 | 0; //@line 32599
      $1192 = 123608 + ($1190 >>> 16 << 2) | 0; //@line 32601
      if ((_lseek($1150, FUNCTION_TABLE_ii[HEAP32[HEAP32[$1192 >> 2] >> 2] & 127]($1190) | 0, 1) | 0) < 0) {
       $$0 = -40; //@line 32608
       STACKTOP = sp; //@line 32609
       return $$0 | 0; //@line 32609
      } else {
       $$pre$phi$i110Z2D = $1190; //@line 32611
       $$pre$phi28$iZ2D = $1192; //@line 32611
      }
      break;
     }
    default:
     {
      $$pre$i108 = $2 + 46 | 0; //@line 32616
      $$pre$phi$i110Z2D = $$pre$i108; //@line 32619
      $$pre$phi28$iZ2D = 123608 + ($$pre$i108 >>> 16 << 2) | 0; //@line 32619
     }
    }
    $1198 = _lseek($1150, 0, 1) | 0; //@line 32622
    $1199 = $1134 + 16 | 0; //@line 32623
    FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($1199 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($1199, $1198); //@line 32629
    FUNCTION_TABLE_vii[HEAP32[(HEAP32[$$pre$phi28$iZ2D >> 2] | 0) + 12 >> 2] & 31]($$pre$phi$i110Z2D, $1198); //@line 32633
    $$0 = 0; //@line 32634
    STACKTOP = sp; //@line 32635
    return $$0 | 0; //@line 32635
   }
   do {
    switch ($trunc << 16 >> 16) {
    case 2:
     {
      $1213 = $2 + 22 | 0; //@line 32640
      HEAP32[$5 >> 2] = FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($1213 >>> 16 << 2) >> 2] | 0) + 4 >> 2] & 127]($1213) | 0; //@line 32647
      _Execute68k((HEAP32[30848] | 0) + 790 | 0, $5); //@line 32650
      $$0 = HEAP32[$5 >> 2] & 65535; //@line 32653
      STACKTOP = sp; //@line 32654
      return $$0 | 0; //@line 32654
     }
    case 5:
     {
      $1224 = $2 + 48 | 0; //@line 32658
      $1230 = __ZL17get_item_and_pathjjRP6FSItemb($2, FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($1224 >>> 16 << 2) >> 2] >> 2] & 127]($1224) | 0, $5) | 0; //@line 32664
      if (!($1230 << 16 >> 16)) {
       _strcpy($7, 657423) | 0; //@line 32667
       $1233 = (HEAP32[30848] | 0) + 238 | 0; //@line 32669
       $1239 = FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($1233 >>> 16 << 2) >> 2] | 0) + 24 >> 2] & 127]($1233) | 0; //@line 32675
       dest = $1239; //@line 32682
       src = FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($2 >>> 16 << 2) >> 2] | 0) + 24 >> 2] & 127]($2) | 0; //@line 32682
       stop = dest + 50 | 0; //@line 32682
       do {
        HEAP8[dest >> 0] = HEAP8[src >> 0] | 0; //@line 32682
        dest = dest + 1 | 0; //@line 32682
        src = src + 1 | 0; //@line 32682
       } while ((dest | 0) < (stop | 0));
       $1247 = (HEAP32[30848] | 0) + 256 | 0; //@line 32684
       $1248 = $2 + 28 | 0; //@line 32685
       $1253 = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($1248 >>> 16 << 2) >> 2] >> 2] & 127]($1248) | 0; //@line 32690
       FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($1247 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($1247, $1253); //@line 32696
       $1260 = (HEAP32[30848] | 0) + 238 | 0; //@line 32698
       $1261 = $2 + 36 | 0; //@line 32699
       $1267 = __ZL17get_item_and_pathjjRP6FSItemb($1260, FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($1261 >>> 16 << 2) >> 2] >> 2] & 127]($1261) | 0, $6) | 0; //@line 32705
       L389 : do {
        if (!($1267 << 16 >> 16)) {
         $1269 = HEAP32[$5 >> 2] | 0; //@line 32709
         $1270 = $1269 + 16 | 0; //@line 32710
         __Z18add_path_componentPcPKc(657423, HEAP32[$1270 >> 2] | 0); //@line 32712
         if (!(_access(657423, 0) | 0)) {
          $$0$i118 = -48; //@line 32716
         } else {
          if (__Z12extfs_renamePKcS0_($7, 657423) | 0) {
           $1279 = __ZL11find_fsitemPKcP6FSItem(HEAP32[$1270 >> 2] | 0, HEAP32[$6 >> 2] | 0) | 0; //@line 32722
           if (!$1279) {
            $$0$i118 = 0; //@line 32725
            break;
           }
           $1281 = $1269 + 4 | 0; //@line 32728
           $1282 = HEAP32[$1281 >> 2] | 0; //@line 32729
           $1283 = $1279 + 4 | 0; //@line 32730
           $1284 = HEAP32[$1283 >> 2] | 0; //@line 32731
           $$012$i$i112 = HEAP32[30826] | 0; //@line 32732
           if ($$012$i$i112 | 0) {
            $$013$i$i113 = $$012$i$i112; //@line 32735
            do {
             $1286 = $$013$i$i113 + 8 | 0; //@line 32737
             $1287 = HEAP32[$1286 >> 2] | 0; //@line 32738
             if (($1287 | 0) == ($1282 | 0)) {
              $$sink$i$i115 = $1284; //@line 32741
              label = 255; //@line 32742
             } else {
              if (($1287 | 0) == ($1284 | 0)) {
               $$sink$i$i115 = $1282; //@line 32746
               label = 255; //@line 32747
              }
             }
             if ((label | 0) == 255) {
              label = 0; //@line 32751
              HEAP32[$1286 >> 2] = $$sink$i$i115; //@line 32752
             }
             $$013$i$i113 = HEAP32[$$013$i$i113 >> 2] | 0; //@line 32754
            } while (($$013$i$i113 | 0) != 0);
           }
           HEAP32[$1281 >> 2] = $1284; //@line 32763
           HEAP32[$1283 >> 2] = $1282; //@line 32764
           $$0$i118 = 0; //@line 32765
           break;
          }
          switch (HEAP32[(___errno_location() | 0) >> 2] | 0) {
          case 0:
           {
            $$0$i118 = 0; //@line 32772
            break L389;
            break;
           }
          case 21:
          case 2:
           {
            $$0$i118 = -43; //@line 32777
            break L389;
            break;
           }
          case 1:
          case 13:
           {
            $$0$i118 = -54; //@line 32782
            break L389;
            break;
           }
          case 17:
           {
            $$0$i118 = -48; //@line 32787
            break L389;
            break;
           }
          case 39:
          case 16:
           {
            $$0$i118 = -47; //@line 32792
            break L389;
            break;
           }
          case 28:
           {
            $$0$i118 = -34; //@line 32797
            break L389;
            break;
           }
          case 30:
           {
            $$0$i118 = -44; //@line 32802
            break L389;
            break;
           }
          case 24:
           {
            $$0$i118 = -42; //@line 32807
            break L389;
            break;
           }
          case 12:
           {
            $$0$i118 = -108; //@line 32812
            break L389;
            break;
           }
          default:
           {
            $$0$i118 = -36; //@line 32817
            break L389;
           }
          }
         }
        } else {
         $$0$i118 = $1267; //@line 32823
        }
       } while (0);
       $$1$i120 = $$0$i118; //@line 32826
      } else {
       $$1$i120 = $1230; //@line 32828
      }
      $$0 = $$1$i120; //@line 32830
      STACKTOP = sp; //@line 32831
      return $$0 | 0; //@line 32831
     }
    case 6:
     {
      $1291 = $2 + 48 | 0; //@line 32835
      $1293 = 123608 + ($1291 >>> 16 << 2) | 0; //@line 32837
      $1297 = __ZL17get_item_and_pathjjRP6FSItemb($2, FUNCTION_TABLE_ii[HEAP32[HEAP32[$1293 >> 2] >> 2] & 127]($1291) | 0, $5) | 0; //@line 32841
      L419 : do {
       if (!($1297 << 16 >> 16)) {
        if (!(_access(657423, 0) | 0)) {
         $$0$i121 = -48; //@line 32848
        } else {
         if ((_mkdir(657423, 511) | 0) >= 0) {
          FUNCTION_TABLE_vii[HEAP32[(HEAP32[$1293 >> 2] | 0) + 12 >> 2] & 31]($1291, HEAP32[(HEAP32[$5 >> 2] | 0) + 4 >> 2] | 0); //@line 32859
          $$0$i121 = 0; //@line 32860
          break;
         }
         switch (HEAP32[(___errno_location() | 0) >> 2] | 0) {
         case 0:
          {
           $$0$i121 = 0; //@line 32867
           break L419;
           break;
          }
         case 21:
         case 2:
          {
           $$0$i121 = -43; //@line 32872
           break L419;
           break;
          }
         case 1:
         case 13:
          {
           $$0$i121 = -54; //@line 32877
           break L419;
           break;
          }
         case 17:
          {
           $$0$i121 = -48; //@line 32882
           break L419;
           break;
          }
         case 39:
         case 16:
          {
           $$0$i121 = -47; //@line 32887
           break L419;
           break;
          }
         case 28:
          {
           $$0$i121 = -34; //@line 32892
           break L419;
           break;
          }
         case 30:
          {
           $$0$i121 = -44; //@line 32897
           break L419;
           break;
          }
         case 24:
          {
           $$0$i121 = -42; //@line 32902
           break L419;
           break;
          }
         case 12:
          {
           $$0$i121 = -108; //@line 32907
           break L419;
           break;
          }
         default:
          {
           $$0$i121 = -36; //@line 32912
           break L419;
          }
         }
        }
       } else {
        $$0$i121 = $1297; //@line 32918
       }
      } while (0);
      $$0 = $$0$i121; //@line 32921
      STACKTOP = sp; //@line 32922
      return $$0 | 0; //@line 32922
     }
    case 7:
     {
      $1311 = $2 + 26 | 0; //@line 32926
      $1313 = 123608 + ($1311 >>> 16 << 2) | 0; //@line 32928
      $1319 = $2 + 28 | 0; //@line 32934
      $1320 = $2 + 22 | 0; //@line 32935
      $1322 = 123608 + ($1320 >>> 16 << 2) | 0; //@line 32937
      if (!(FUNCTION_TABLE_ii[HEAP32[(HEAP32[$1313 >> 2] | 0) + 4 >> 2] & 127]($1311) | 0)) {
       $1326 = FUNCTION_TABLE_ii[HEAP32[(HEAP32[$1322 >> 2] | 0) + 4 >> 2] & 127]($1320) | 0; //@line 32942
       $1327 = $0 + 78 | 0; //@line 32943
       $1329 = 123608 + ($1327 >>> 16 << 2) | 0; //@line 32945
       if (($1326 | 0) == (FUNCTION_TABLE_ii[HEAP32[(HEAP32[$1329 >> 2] | 0) + 4 >> 2] & 127]($1327) | 0)) {
        FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($1319 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($1319, 0); //@line 32957
        $1340 = $2 + 32 | 0; //@line 32958
        $1344 = FUNCTION_TABLE_ii[HEAP32[(HEAP32[$1329 >> 2] | 0) + 4 >> 2] & 127]($1327) | 0; //@line 32962
        FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($1340 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($1340, $1344); //@line 32968
        $1350 = $2 + 18 | 0; //@line 32969
        $1352 = 123608 + ($1350 >>> 16 << 2) | 0; //@line 32971
        if (FUNCTION_TABLE_ii[HEAP32[HEAP32[$1352 >> 2] >> 2] & 127]($1350) | 0) {
         $1359 = FUNCTION_TABLE_ii[HEAP32[HEAP32[$1352 >> 2] >> 2] & 127]($1350) | 0; //@line 32979
         $1360 = $0 + 44 | 0; //@line 32980
         $1366 = FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($1359 >>> 16 << 2) >> 2] | 0) + 24 >> 2] & 127]($1359) | 0; //@line 32986
         dest = $1366; //@line 32993
         src = FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($1360 >>> 16 << 2) >> 2] | 0) + 24 >> 2] & 127]($1360) | 0; //@line 32993
         stop = dest + 28 | 0; //@line 32993
         do {
          HEAP8[dest >> 0] = HEAP8[src >> 0] | 0; //@line 32993
          dest = dest + 1 | 0; //@line 32993
          src = src + 1 | 0; //@line 32993
         } while ((dest | 0) < (stop | 0));
        }
        $1373 = $2 + 48 | 0; //@line 32995
        FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($1373 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($1373, 2); //@line 33001
        $$1$i124 = 0; //@line 33002
       } else {
        label = 280; //@line 33004
       }
      } else {
       label = 280; //@line 33007
      }
      do {
       if ((label | 0) == 280) {
        $1380 = 123608 + ($1319 >>> 16 << 2) | 0; //@line 33012
        HEAP32[$5 >> 2] = FUNCTION_TABLE_ii[HEAP32[HEAP32[$1380 >> 2] >> 2] & 127]($1319) | 0; //@line 33016
        HEAP32[$5 + 4 >> 2] = FUNCTION_TABLE_ii[HEAP32[(HEAP32[$1313 >> 2] | 0) + 4 >> 2] & 127]($1311) | 0; //@line 33022
        HEAP32[$5 + 8 >> 2] = FUNCTION_TABLE_ii[HEAP32[(HEAP32[$1322 >> 2] | 0) + 4 >> 2] & 127]($1320) | 0; //@line 33028
        $1394 = HEAP32[30848] | 0; //@line 33029
        HEAP32[$5 + 32 >> 2] = $1394 + 306; //@line 33032
        _Execute68k($1394 + 614 | 0, $5); //@line 33034
        $1399 = (HEAP32[30848] | 0) + 306 | 0; //@line 33036
        $1404 = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($1399 >>> 16 << 2) >> 2] >> 2] & 127]($1399) | 0; //@line 33041
        $1405 = HEAP32[$5 >> 2] | 0; //@line 33042
        if ($1405 & 65535 | 0) {
         $$1$i124 = $1405 & 65535; //@line 33047
         break;
        }
        $1409 = $1404 + 12 | 0; //@line 33050
        $1414 = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($1409 >>> 16 << 2) >> 2] >> 2] & 127]($1409) | 0; //@line 33055
        FUNCTION_TABLE_vii[HEAP32[(HEAP32[$1380 >> 2] | 0) + 12 >> 2] & 31]($1319, $1414); //@line 33059
        $1418 = $2 + 32 | 0; //@line 33060
        $1420 = 123608 + ($1404 >>> 16 << 2) | 0; //@line 33062
        $1424 = (FUNCTION_TABLE_ii[HEAP32[HEAP32[$1420 >> 2] >> 2] & 127]($1404) | 0) + 78 | 0; //@line 33066
        $1430 = FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($1424 >>> 16 << 2) >> 2] | 0) + 4 >> 2] & 127]($1424) | 0; //@line 33072
        FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($1418 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($1418, $1430); //@line 33078
        $1436 = $2 + 18 | 0; //@line 33079
        $1438 = 123608 + ($1436 >>> 16 << 2) | 0; //@line 33081
        if (FUNCTION_TABLE_ii[HEAP32[HEAP32[$1438 >> 2] >> 2] & 127]($1436) | 0) {
         $1445 = FUNCTION_TABLE_ii[HEAP32[HEAP32[$1438 >> 2] >> 2] & 127]($1436) | 0; //@line 33089
         $1449 = (FUNCTION_TABLE_ii[HEAP32[HEAP32[$1420 >> 2] >> 2] & 127]($1404) | 0) + 44 | 0; //@line 33093
         $1455 = FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($1445 >>> 16 << 2) >> 2] | 0) + 24 >> 2] & 127]($1445) | 0; //@line 33099
         dest = $1455; //@line 33106
         src = FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($1449 >>> 16 << 2) >> 2] | 0) + 24 >> 2] & 127]($1449) | 0; //@line 33106
         stop = dest + 28 | 0; //@line 33106
         do {
          HEAP8[dest >> 0] = HEAP8[src >> 0] | 0; //@line 33106
          dest = dest + 1 | 0; //@line 33106
          src = src + 1 | 0; //@line 33106
         } while ((dest | 0) < (stop | 0));
        }
        $1462 = $2 + 48 | 0; //@line 33108
        $1463 = $1404 + 4 | 0; //@line 33109
        $1468 = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($1463 >>> 16 << 2) >> 2] >> 2] & 127]($1463) | 0; //@line 33114
        FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($1462 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($1462, $1468); //@line 33120
        $$1$i124 = 0; //@line 33121
       }
      } while (0);
      $$0 = $$1$i124; //@line 33124
      STACKTOP = sp; //@line 33125
      return $$0 | 0; //@line 33125
     }
    case 8:
     {
      $1474 = $2 + 28 | 0; //@line 33129
      $1476 = 123608 + ($1474 >>> 16 << 2) | 0; //@line 33131
      $1481 = (FUNCTION_TABLE_ii[HEAP32[(HEAP32[$1476 >> 2] | 0) + 4 >> 2] & 127]($1474) | 0) == 0; //@line 33136
      $1482 = $2 + 24 | 0; //@line 33137
      $1485 = HEAP32[123608 + ($1482 >>> 16 << 2) >> 2] | 0; //@line 33140
      L455 : do {
       if ($1481) {
        HEAP32[$5 >> 2] = (FUNCTION_TABLE_ii[HEAP32[$1485 + 4 >> 2] & 127]($1482) | 0) << 16 >> 16; //@line 33148
        $1490 = HEAP32[30848] | 0; //@line 33149
        HEAP32[$5 + 32 >> 2] = $1490 + 306; //@line 33152
        _Execute68k($1490 + 752 | 0, $5); //@line 33154
        $1495 = (HEAP32[30848] | 0) + 306 | 0; //@line 33156
        $1500 = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($1495 >>> 16 << 2) >> 2] >> 2] & 127]($1495) | 0; //@line 33161
        if (!(HEAP32[$5 >> 2] & 65535)) {
         $$2$i126 = $1500; //@line 33166
         label = 293; //@line 33167
        } else {
         $$137$i = -51; //@line 33169
        }
       } else {
        FUNCTION_TABLE_vii[HEAP32[$1485 + 16 >> 2] & 31]($1482, 0); //@line 33174
        if ((FUNCTION_TABLE_ii[HEAP32[(HEAP32[$1476 >> 2] | 0) + 4 >> 2] & 127]($1474) | 0) > 0) {
         $1511 = $6 + 32 | 0; //@line 33181
         $1512 = $6 + 36 | 0; //@line 33182
         $1513 = $6 + 40 | 0; //@line 33183
         $$03440$i = 0; //@line 33184
         while (1) {
          HEAP32[$1511 >> 2] = $0; //@line 33186
          HEAP32[$1512 >> 2] = $1482; //@line 33187
          $1520 = HEAP32[30848] | 0; //@line 33188
          HEAP32[$1513 >> 2] = $1520 + 306; //@line 33190
          _Execute68k($1520 + 736 | 0, $6); //@line 33192
          $1524 = (HEAP32[30848] | 0) + 306 | 0; //@line 33194
          $1529 = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($1524 >>> 16 << 2) >> 2] >> 2] & 127]($1524) | 0; //@line 33199
          $1530 = HEAP32[$6 >> 2] | 0; //@line 33200
          $$03440$i = $$03440$i + 1 | 0; //@line 33203
          if ($1530 & 65535 | 0) {
           break;
          }
          if (($$03440$i | 0) >= (FUNCTION_TABLE_ii[HEAP32[(HEAP32[$1476 >> 2] | 0) + 4 >> 2] & 127]($1474) | 0)) {
           $$2$i126 = $1529; //@line 33215
           label = 293; //@line 33216
           break L455;
          }
         }
         $$137$i = $1530 & 65535; //@line 33221
        } else {
         $$137$i = -51; //@line 33223
        }
       }
      } while (0);
      if ((label | 0) == 293) {
       if (!$$2$i126) {
        $$137$i = -51; //@line 33230
       } else {
        $1535 = $2 + 18 | 0; //@line 33232
        $1537 = 123608 + ($1535 >>> 16 << 2) | 0; //@line 33234
        if (FUNCTION_TABLE_ii[HEAP32[HEAP32[$1537 >> 2] >> 2] & 127]($1535) | 0) {
         $1544 = FUNCTION_TABLE_ii[HEAP32[HEAP32[$1537 >> 2] >> 2] & 127]($1535) | 0; //@line 33242
         $1550 = FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($1544 >>> 16 << 2) >> 2] | 0) + 24 >> 2] & 127]($1544) | 0; //@line 33248
         $1551 = $$2$i126 + 62 | 0; //@line 33249
         $1557 = FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($1551 >>> 16 << 2) >> 2] | 0) + 24 >> 2] & 127]($1551) | 0; //@line 33255
         $1558 = HEAP8[$1557 >> 0] | 0; //@line 33256
         HEAP8[$1550 >> 0] = $1558; //@line 33257
         if ($1558 << 24 >> 24) {
          $$010$i$i131 = $1558 << 24 >> 24; //@line 33261
          $$0511$pn$i$i130 = $1557; //@line 33261
          $$0612$pn$i$i129 = $1550; //@line 33261
          do {
           $$0511$pn$i$i130 = $$0511$pn$i$i130 + 1 | 0; //@line 33263
           $$0612$pn$i$i129 = $$0612$pn$i$i129 + 1 | 0; //@line 33264
           $$010$i$i131 = $$010$i$i131 + -1 | 0; //@line 33265
           HEAP8[$$0612$pn$i$i129 >> 0] = HEAP8[$$0511$pn$i$i130 >> 0] | 0; //@line 33267
          } while (($$010$i$i131 | 0) != 0);
         }
        }
        $1564 = $2 + 32 | 0; //@line 33277
        $1569 = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($$2$i126 >>> 16 << 2) >> 2] >> 2] & 127]($$2$i126) | 0; //@line 33282
        FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($1564 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($1564, $1569); //@line 33288
        $1575 = $2 + 36 | 0; //@line 33289
        $1576 = $$2$i126 + 4 | 0; //@line 33290
        $1582 = FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($1576 >>> 16 << 2) >> 2] | 0) + 8 >> 2] & 127]($1576) | 0; //@line 33296
        FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($1575 >>> 16 << 2) >> 2] | 0) + 20 >> 2] & 31]($1575, $1582); //@line 33302
        $1588 = $2 + 38 | 0; //@line 33303
        $1589 = $$2$i126 + 6 | 0; //@line 33304
        $1595 = FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($1589 >>> 16 << 2) >> 2] | 0) + 4 >> 2] & 127]($1589) | 0; //@line 33310
        FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($1588 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($1588, $1595); //@line 33316
        $1601 = $2 + 40 | 0; //@line 33317
        $1602 = $$2$i126 + 8 | 0; //@line 33318
        $1607 = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($1602 >>> 16 << 2) >> 2] >> 2] & 127]($1602) | 0; //@line 33323
        FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($1601 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($1601, $1607); //@line 33329
        $1613 = $2 + 44 | 0; //@line 33330
        $1614 = $$2$i126 + 12 | 0; //@line 33331
        $1619 = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($1614 >>> 16 << 2) >> 2] >> 2] & 127]($1614) | 0; //@line 33336
        FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($1613 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($1613, $1619); //@line 33342
        $1625 = $2 + 48 | 0; //@line 33343
        $1626 = $$2$i126 + 16 | 0; //@line 33344
        $1631 = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($1626 >>> 16 << 2) >> 2] >> 2] & 127]($1626) | 0; //@line 33349
        FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($1625 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($1625, $1631); //@line 33355
        $1637 = $2 + 52 | 0; //@line 33356
        $1638 = $$2$i126 + 20 | 0; //@line 33357
        $1644 = (FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($1638 >>> 16 << 2) >> 2] >> 2] & 127]($1638) | 0) + 78 | 0; //@line 33363
        $1650 = FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($1644 >>> 16 << 2) >> 2] | 0) + 4 >> 2] & 127]($1644) | 0; //@line 33369
        FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($1637 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($1637, $1650); //@line 33375
        $1656 = $2 + 54 | 0; //@line 33376
        $1657 = $$2$i126 + 30 | 0; //@line 33377
        $1662 = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($1657 >>> 16 << 2) >> 2] >> 2] & 127]($1657) | 0; //@line 33382
        FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($1656 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($1656, $1662); //@line 33388
        $1668 = $2 + 58 | 0; //@line 33389
        $1669 = $$2$i126 + 58 | 0; //@line 33390
        $1674 = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($1669 >>> 16 << 2) >> 2] >> 2] & 127]($1669) | 0; //@line 33395
        FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($1668 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($1668, $1674); //@line 33401
        $$137$i = 0; //@line 33402
       }
      }
      $$0 = $$137$i; //@line 33405
      STACKTOP = sp; //@line 33406
      return $$0 | 0; //@line 33406
     }
    case 9:
     {
      $1680 = $2 + 28 | 0; //@line 33410
      $sext$i136 = (FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($1680 >>> 16 << 2) >> 2] | 0) + 4 >> 2] & 127]($1680) | 0) << 16; //@line 33417
      $1687 = $sext$i136 >> 16; //@line 33418
      L475 : do {
       if (($sext$i136 | 0) < 0) {
        $1689 = $2 + 48 | 0; //@line 33422
        $1694 = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($1689 >>> 16 << 2) >> 2] >> 2] & 127]($1689) | 0; //@line 33427
        $$06$i$i137 = HEAP32[30826] | 0; //@line 33428
        L477 : do {
         if ($$06$i$i137 | 0) {
          $$07$i$i138 = $$06$i$i137; //@line 33432
          while (1) {
           if ((HEAP32[$$07$i$i138 + 4 >> 2] | 0) == ($1694 | 0)) {
            break;
           }
           $$0$i$i140 = HEAP32[$$07$i$i138 >> 2] | 0; //@line 33440
           if (!$$0$i$i140) {
            break L477;
           } else {
            $$07$i$i138 = $$0$i$i140; //@line 33445
           }
          }
          HEAP32[$5 >> 2] = $$07$i$i138; //@line 33448
          __ZL19get_path_for_fsitemP6FSItem($$07$i$i138); //@line 33449
          label = 321; //@line 33450
          break L475;
         }
        } while (0);
        HEAP32[$5 >> 2] = 0; //@line 33454
        $$6$i152 = -120; //@line 33455
       } else {
        if (!$1687) {
         $1701 = $2 + 48 | 0; //@line 33459
         $1707 = __ZL17get_item_and_pathjjRP6FSItemb($2, FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($1701 >>> 16 << 2) >> 2] >> 2] & 127]($1701) | 0, $5) | 0; //@line 33465
         if (!($1707 << 16 >> 16)) {
          label = 321; //@line 33468
          break;
         } else {
          $$6$i152 = $1707; //@line 33471
          break;
         }
        }
        $1709 = $2 + 48 | 0; //@line 33475
        $1715 = __ZL15get_current_dirjjRjb($2, FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($1709 >>> 16 << 2) >> 2] >> 2] & 127]($1709) | 0, $6, 1) | 0; //@line 33481
        L487 : do {
         if (!($1715 << 16 >> 16)) {
          $1717 = HEAP32[$6 >> 2] | 0; //@line 33485
          $$06$i101$i = HEAP32[30826] | 0; //@line 33486
          if (!$$06$i101$i) {
           $$4$ph$i142 = -120; //@line 33489
          } else {
           $$07$i102$i = $$06$i101$i; //@line 33491
           while (1) {
            if ((HEAP32[$$07$i102$i + 4 >> 2] | 0) == ($1717 | 0)) {
             break;
            }
            $$0$i104$i = HEAP32[$$07$i102$i >> 2] | 0; //@line 33499
            if (!$$0$i104$i) {
             $$4$ph$i142 = -120; //@line 33502
             break L487;
            } else {
             $$07$i102$i = $$0$i104$i; //@line 33505
            }
           }
           __ZL19get_path_for_fsitemP6FSItem($$07$i102$i); //@line 33508
           $1723 = _opendir(657423) | 0; //@line 33509
           if (!$1723) {
            $$4$ph$i142 = -120; //@line 33512
           } else {
            L494 : do {
             if (($sext$i136 | 0) > 0) {
              $$0100122$i = 0; //@line 33517
              L495 : while (1) {
               do {
                $1726 = _readdir($1723) | 0; //@line 33520
                if (!$1726) {
                 break L495;
                }
               } while ((HEAP8[$1726 + 11 >> 0] | 0) == 46);
               $$0100122$i = $$0100122$i + 1 | 0; //@line 33532
               if (($$0100122$i | 0) >= ($1687 | 0)) {
                $$094$lcssa$i = $1726; //@line 33537
                break L494;
               }
              }
              _closedir($1723) | 0; //@line 33541
              $$4$ph$i142 = -43; //@line 33542
              break L487;
             } else {
              $$094$lcssa$i = 0; //@line 33545
             }
            } while (0);
            $1733 = $$094$lcssa$i + 11 | 0; //@line 33548
            __Z18add_path_componentPcPKc(657423, $1733); //@line 33549
            HEAP32[$5 >> 2] = __ZL11find_fsitemPKcP6FSItem($1733, $$07$i102$i) | 0; //@line 33551
            _closedir($1723) | 0; //@line 33552
            label = 321; //@line 33553
            break L475;
           }
          }
         } else {
          $$4$ph$i142 = $1715; //@line 33558
         }
        } while (0);
        $$6$i152 = $$4$ph$i142; //@line 33561
       }
      } while (0);
      if ((label | 0) == 321) {
       L506 : do {
        if ((_stat(657423, $6) | 0) < 0) {
         switch (HEAP32[(___errno_location() | 0) >> 2] | 0) {
         case 0:
          {
           $$5$i150 = 0; //@line 33573
           break L506;
           break;
          }
         case 21:
         case 2:
          {
           $$5$i150 = -43; //@line 33578
           break L506;
           break;
          }
         case 1:
         case 13:
          {
           $$5$i150 = -54; //@line 33583
           break L506;
           break;
          }
         case 17:
          {
           $$5$i150 = -48; //@line 33588
           break L506;
           break;
          }
         case 39:
         case 16:
          {
           $$5$i150 = -47; //@line 33593
           break L506;
           break;
          }
         case 28:
          {
           $$5$i150 = -34; //@line 33598
           break L506;
           break;
          }
         case 30:
          {
           $$5$i150 = -44; //@line 33603
           break L506;
           break;
          }
         case 24:
          {
           $$5$i150 = -42; //@line 33608
           break L506;
           break;
          }
         case 12:
          {
           $$5$i150 = -108; //@line 33613
           break L506;
           break;
          }
         default:
          {
           $$5$i150 = -36; //@line 33618
           break L506;
          }
         }
        } else {
         if (($1687 | 0) == -1) {
          if ((HEAP32[$6 + 12 >> 2] & 61440 | 0) != 16384) {
           $$5$i150 = -120; //@line 33630
           break;
          }
         }
         $1744 = $2 + 18 | 0; //@line 33634
         $1746 = 123608 + ($1744 >>> 16 << 2) | 0; //@line 33636
         L522 : do {
          if (FUNCTION_TABLE_ii[HEAP32[HEAP32[$1746 >> 2] >> 2] & 127]($1744) | 0) {
           $1753 = FUNCTION_TABLE_ii[HEAP32[HEAP32[$1746 >> 2] >> 2] & 127]($1744) | 0; //@line 33645
           $1759 = FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($1753 >>> 16 << 2) >> 2] | 0) + 24 >> 2] & 127]($1753) | 0; //@line 33651
           $1761 = (HEAP32[$5 >> 2] | 0) + 20 | 0; //@line 33653
           HEAP8[$1759 >> 0] = _strlen($1761) | 0; //@line 33656
           $$05$i108$i = $1761; //@line 33657
           $$pn$i$i144 = $1759; //@line 33657
           while (1) {
            $$06$i109$i = $$pn$i$i144 + 1 | 0; //@line 33659
            $1764 = $$05$i108$i + 1 | 0; //@line 33660
            $1765 = HEAP8[$$05$i108$i >> 0] | 0; //@line 33661
            switch ($1765 << 24 >> 24) {
            case 0:
             {
              break L522;
              break;
             }
            case 58:
             {
              $$0$i110$i = 47; //@line 33668
              break;
             }
            default:
             {
              $$0$i110$i = $1765; //@line 33672
             }
            }
            HEAP8[$$06$i109$i >> 0] = $$0$i110$i; //@line 33675
            $$05$i108$i = $1764; //@line 33676
            $$pn$i$i144 = $$06$i109$i; //@line 33676
           }
          }
         } while (0);
         $1766 = $2 + 24 | 0; //@line 33680
         FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($1766 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($1766, 0); //@line 33686
         $1772 = $2 + 30 | 0; //@line 33687
         $1776 = (HEAP32[$6 + 12 >> 2] & 61440 | 0) == 16384; //@line 33691
         $1780 = ($1776 ? 16 : 0) | (_access(657423, 2) | 0) != 0; //@line 33696
         FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($1772 >>> 16 << 2) >> 2] | 0) + 20 >> 2] & 31]($1772, $1780); //@line 33702
         $1786 = $2 + 31 | 0; //@line 33703
         FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($1786 >>> 16 << 2) >> 2] | 0) + 20 >> 2] & 31]($1786, 0); //@line 33709
         $1792 = $2 + 48 | 0; //@line 33710
         $1793 = HEAP32[$5 >> 2] | 0; //@line 33711
         FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($1792 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($1792, HEAP32[$1793 + 4 >> 2] | 0); //@line 33719
         $1801 = $2 + 100 | 0; //@line 33720
         FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($1801 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($1801, HEAP32[$1793 + 8 >> 2] | 0); //@line 33728
         $1809 = $2 + 72 | 0; //@line 33729
         FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($1809 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($1809, 0); //@line 33735
         $1816 = HEAP32[$6 + 56 >> 2] | 0; //@line 33737
         $1817 = $1793 + 52 | 0; //@line 33738
         if (($1816 | 0) > (HEAP32[$1817 >> 2] | 0)) {
          HEAP32[$1817 >> 2] = $1816; //@line 33742
          $$099$off0$i = 0; //@line 33743
         } else {
          $$099$off0$i = 1; //@line 33745
         }
         $1820 = $2 + 76 | 0; //@line 33747
         $1821 = __Z13TimeToMacTimel($1816) | 0; //@line 33748
         FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($1820 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($1820, $1821); //@line 33754
         $1827 = $2 + 80 | 0; //@line 33755
         FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($1827 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($1827, 0); //@line 33761
         __Z9get_finfoPKcjjb(657423, $2 + 32 | 0, $2 + 84 | 0, $1776); //@line 33764
         if (!$1776) {
          $1854 = $2 + 52 | 0; //@line 33766
          FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($1854 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($1854, 0); //@line 33772
          $1860 = $2 + 54 | 0; //@line 33773
          $1862 = HEAP32[$6 + 36 >> 2] | 0; //@line 33775
          FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($1860 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($1860, $1862); //@line 33781
          $1868 = $2 + 58 | 0; //@line 33782
          FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($1868 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($1868, ($1862 | 16383) + 1 | 0); //@line 33790
          $1876 = $2 + 62 | 0; //@line 33791
          FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($1876 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($1876, 0); //@line 33797
          $1882 = __Z14get_rfork_sizePKc(657423) | 0; //@line 33798
          $1883 = $2 + 64 | 0; //@line 33799
          FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($1883 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($1883, $1882); //@line 33805
          $1889 = $2 + 68 | 0; //@line 33806
          FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($1889 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($1889, ($1882 | 16383) + 1 | 0); //@line 33814
          $1897 = $2 + 104 | 0; //@line 33815
          FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($1897 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($1897, 0); //@line 33821
          $$5$i150 = 0; //@line 33822
          break;
         }
         if ($$099$off0$i) {
          $$298$i = HEAP32[$1793 + 56 >> 2] | 0; //@line 33828
         } else {
          $1837 = _opendir(657423) | 0; //@line 33830
          if (!$1837) {
           $$197$i = 0; //@line 33833
          } else {
           $1839 = _readdir($1837) | 0; //@line 33835
           if (!$1839) {
            $$096$lcssa$i = 0; //@line 33838
           } else {
            $$096121$i = 0; //@line 33840
            $1842 = $1839; //@line 33840
            while (1) {
             $$096$$i = ((HEAP8[$1842 + 11 >> 0] | 0) != 46 & 1) + $$096121$i | 0; //@line 33846
             $1842 = _readdir($1837) | 0; //@line 33847
             if (!$1842) {
              $$096$lcssa$i = $$096$$i; //@line 33850
              break;
             } else {
              $$096121$i = $$096$$i; //@line 33853
             }
            }
           }
           _closedir($1837) | 0; //@line 33857
           $$197$i = $$096$lcssa$i; //@line 33858
          }
          HEAP32[$1793 + 56 >> 2] = $$197$i; //@line 33861
          $$298$i = $$197$i; //@line 33862
         }
         $1848 = $2 + 52 | 0; //@line 33864
         FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($1848 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($1848, $$298$i); //@line 33870
         $$5$i150 = 0; //@line 33871
        }
       } while (0);
       $$6$i152 = $$5$i150; //@line 33874
      }
      $$0 = $$6$i152; //@line 33876
      STACKTOP = sp; //@line 33877
      return $$0 | 0; //@line 33877
     }
    case 10:
     {
      $1903 = $2 + 48 | 0; //@line 33881
      $1909 = __ZL17get_item_and_pathjjRP6FSItemb($2, FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($1903 >>> 16 << 2) >> 2] >> 2] & 127]($1903) | 0, $5) | 0; //@line 33887
      if (!($1909 << 16 >> 16)) {
       L553 : do {
        if ((_stat(657423, $6) | 0) < 0) {
         switch (HEAP32[(___errno_location() | 0) >> 2] | 0) {
         case 0:
          {
           $$0$i153 = 0; //@line 33898
           break L553;
           break;
          }
         case 21:
         case 2:
          {
           $$0$i153 = -43; //@line 33903
           break L553;
           break;
          }
         case 1:
         case 13:
          {
           $$0$i153 = -54; //@line 33908
           break L553;
           break;
          }
         case 17:
          {
           $$0$i153 = -48; //@line 33913
           break L553;
           break;
          }
         case 39:
         case 16:
          {
           $$0$i153 = -47; //@line 33918
           break L553;
           break;
          }
         case 28:
          {
           $$0$i153 = -34; //@line 33923
           break L553;
           break;
          }
         case 30:
          {
           $$0$i153 = -44; //@line 33928
           break L553;
           break;
          }
         case 24:
          {
           $$0$i153 = -42; //@line 33933
           break L553;
           break;
          }
         case 12:
          {
           $$0$i153 = -108; //@line 33938
           break L553;
           break;
          }
         default:
          {
           $$0$i153 = -36; //@line 33943
           break L553;
          }
         }
        } else {
         __Z9set_finfoPKcjjb(657423, $2 + 32 | 0, $2 + 84 | 0, (HEAP32[$6 + 12 >> 2] & 61440 | 0) == 16384); //@line 33954
         $$0$i153 = 0; //@line 33955
        }
       } while (0);
       $$1$i155 = $$0$i153; //@line 33958
      } else {
       $$1$i155 = $1909; //@line 33960
      }
      $$0 = $$1$i155; //@line 33962
      STACKTOP = sp; //@line 33963
      return $$0 | 0; //@line 33963
     }
    case 48:
     {
      $1921 = $2 + 36 | 0; //@line 33967
      $1926 = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($1921 >>> 16 << 2) >> 2] >> 2] & 127]($1921) | 0; //@line 33972
      $$$i156 = $1926 >>> 0 < 20 ? $1926 : 20; //@line 33974
      $1928 = $2 + 40 | 0; //@line 33975
      FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($1928 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($1928, $$$i156); //@line 33981
      $1934 = $2 + 32 | 0; //@line 33982
      $1939 = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($1934 >>> 16 << 2) >> 2] >> 2] & 127]($1934) | 0; //@line 33987
      if (!$$$i156) {
       $$0 = 0; //@line 33990
       STACKTOP = sp; //@line 33991
       return $$0 | 0; //@line 33991
      }
      FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($1939 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($1939, 2); //@line 33998
      $1947 = $1939 + 2 | 0; //@line 34000
      if ($$$i156 >>> 0 <= 2) {
       $$0 = 0; //@line 34002
       STACKTOP = sp; //@line 34003
       return $$0 | 0; //@line 34003
      }
      FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($1947 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($1947, 1040908288); //@line 34010
      $1954 = $1939 + 6 | 0; //@line 34012
      if ($$$i156 >>> 0 <= 6) {
       $$0 = 0; //@line 34014
       STACKTOP = sp; //@line 34015
       return $$0 | 0; //@line 34015
      }
      FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($1954 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($1954, 0); //@line 34022
      $1961 = $1939 + 10 | 0; //@line 34024
      if ($$$i156 >>> 0 <= 10) {
       $$0 = 0; //@line 34026
       STACKTOP = sp; //@line 34027
       return $$0 | 0; //@line 34027
      }
      FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($1961 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($1961, 0); //@line 34034
      $1968 = $1939 + 14 | 0; //@line 34036
      if ($$$i156 >>> 0 <= 14) {
       $$0 = 0; //@line 34038
       STACKTOP = sp; //@line 34039
       return $$0 | 0; //@line 34039
      }
      FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($1968 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($1968, 0); //@line 34046
      $1975 = $1939 + 18 | 0; //@line 34048
      if ($$$i156 >>> 0 <= 18) {
       $$0 = 0; //@line 34050
       STACKTOP = sp; //@line 34051
       return $$0 | 0; //@line 34051
      }
      FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($1975 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($1975, 0); //@line 34058
      $$0 = 0; //@line 34059
      STACKTOP = sp; //@line 34060
      return $$0 | 0; //@line 34060
     }
    case 65:
     {
      $1981 = HEAP32[30848] | 0; //@line 34064
      $1983 = $5 + 32 | 0; //@line 34066
      HEAP32[$1983 >> 2] = $1981 + 306; //@line 34067
      $1985 = $5 + 36 | 0; //@line 34069
      HEAP32[$1985 >> 2] = $1981 + 308; //@line 34070
      _Execute68k($1981 + 562 | 0, $5); //@line 34072
      $1988 = (HEAP32[30848] | 0) + 308 | 0; //@line 34074
      $1993 = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($1988 >>> 16 << 2) >> 2] >> 2] & 127]($1988) | 0; //@line 34079
      $1994 = HEAP32[$5 >> 2] | 0; //@line 34080
      do {
       if (!($1994 & 65535)) {
        $1998 = $1993 + 8 | 0; //@line 34085
        FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($1998 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($1998, 16964); //@line 34091
        $2004 = $1993 + 10 | 0; //@line 34092
        FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($2004 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($2004, 0); //@line 34098
        $2010 = $1993 + 14 | 0; //@line 34099
        $2012 = __Z13TimeToMacTimel(HEAP32[30843] | 0) | 0; //@line 34101
        FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($2010 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($2010, $2012); //@line 34107
        $2018 = $1993 + 94 | 0; //@line 34108
        FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($2018 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($2018, 0); //@line 34114
        $2024 = $1993 + 20 | 0; //@line 34115
        FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($2024 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($2024, 1); //@line 34121
        $2030 = $1993 + 112 | 0; //@line 34122
        FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($2030 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($2030, 1); //@line 34128
        $2036 = $1993 + 26 | 0; //@line 34129
        FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($2036 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($2036, 65535); //@line 34135
        $2042 = $1993 + 28 | 0; //@line 34136
        FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($2042 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($2042, 16384); //@line 34142
        $2048 = $1993 + 32 | 0; //@line 34143
        FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($2048 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($2048, 16384); //@line 34149
        $2054 = $1993 + 38 | 0; //@line 34150
        FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($2054 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($2054, HEAP32[383] | 0); //@line 34157
        $2061 = $1993 + 42 | 0; //@line 34158
        FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($2061 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($2061, 65535); //@line 34164
        $2067 = $1993 + 44 | 0; //@line 34165
        dest = FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($2067 >>> 16 << 2) >> 2] | 0) + 24 >> 2] & 127]($2067) | 0; //@line 34172
        src = 657390; //@line 34172
        stop = dest + 28 | 0; //@line 34172
        do {
         HEAP8[dest >> 0] = HEAP8[src >> 0] | 0; //@line 34172
         dest = dest + 1 | 0; //@line 34172
         src = src + 1 | 0; //@line 34172
        } while ((dest | 0) < (stop | 0));
        $2074 = $1993 + 76 | 0; //@line 34173
        FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($2074 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($2074, 25185); //@line 34179
        $2080 = $1993 + 114 | 0; //@line 34180
        FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($2080 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($2080, 1); //@line 34186
        $2086 = $1993 + 118 | 0; //@line 34187
        FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($2086 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($2086, 1); //@line 34193
        HEAP32[$5 >> 2] = HEAP32[30849]; //@line 34195
        $2093 = HEAP32[30848] | 0; //@line 34196
        HEAP32[$1983 >> 2] = $2093 + 306; //@line 34198
        HEAP32[$1985 >> 2] = $1993; //@line 34199
        _Execute68k($2093 + 578 | 0, $5); //@line 34201
        $2097 = (HEAP32[30848] | 0) + 306 | 0; //@line 34203
        $2102 = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($2097 >>> 16 << 2) >> 2] >> 2] & 127]($2097) | 0; //@line 34208
        $2103 = HEAP32[$5 >> 2] | 0; //@line 34209
        if (!($2103 & 65535)) {
         HEAP32[$5 >> 2] = HEAP32[30849]; //@line 34214
         HEAP32[$1983 >> 2] = 7; //@line 34215
         _Execute68kTrap(-24529, $5); //@line 34216
         $2108 = $2 + 22 | 0; //@line 34217
         FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($2108 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($2108, $2102 << 16 >> 16); //@line 34225
         $$1$i158 = 0; //@line 34226
         break;
        } else {
         $$1$i158 = $2103 & 65535; //@line 34230
         break;
        }
       } else {
        $$1$i158 = $1994 & 65535; //@line 34235
       }
      } while (0);
      $$0 = $$1$i158; //@line 34238
      STACKTOP = sp; //@line 34239
      return $$0 | 0; //@line 34239
     }
    case 11:
     {
      $$0 = 0; //@line 34243
      STACKTOP = sp; //@line 34244
      return $$0 | 0; //@line 34244
     }
    default:
     {
      break L1;
     }
    }
   } while (0);
  }
 } while (0);
 $$0 = -50; //@line 34254
 STACKTOP = sp; //@line 34255
 return $$0 | 0; //@line 34255
}
function __Z16fpuop_arithmeticjj($0, $1) {
 $0 = $0 | 0;
 $1 = $1 | 0;
 var $$$0$i$i = 0, $$$047$1$i$i = 0, $$$047$i$i = 0, $$$4$2$i$i = 0, $$$4$4$i$i = 0, $$$4$6$i$i = 0, $$$4$i$i = 0, $$$6$2$i$i = 0, $$$6$4$i$i = 0, $$$6$i$i = 0, $$$7$i$i = 0, $$$9$i$i = 0, $$$i479 = 0, $$$i482 = 0, $$$i485 = 0, $$$i488 = 0, $$$i502 = 0, $$$i505 = 0, $$0$i$i = 0.0, $$0$i$i457 = 0, $$0$i$i459 = 0, $$0$i$i468 = 0, $$0$i$i514 = 0, $$0$i101$i = 0.0, $$0$i104$i = 0.0, $$0$i107$i = 0, $$0$i492 = 0.0, $$0$i92$i = 0.0, $$0$i95$i = 0.0, $$0$i98$i = 0.0, $$0134$i = 0, $$0135$i = 0, $$0136$i = 0, $$0138$i = 0, $$0139$i = 0, $$0372642 = 0, $$0382$in = 0, $$0382$in$ph = 0, $$03963$i$i = 0, $$040$i$i = 0, $$0405058$i$i = 0, $$04051$i$i = 0, $$041$i$i = 0, $$0414859$i$i = 0, $$04149$i$i = 0, $$0603 = 0, $$0604 = 0, $$0605 = 0, $$0606 = 0, $$0607 = 0, $$0608 = 0, $$080$i = 0, $$098$i = 0, $$1$i$i = 0, $$1$i105$i = 0, $$1376 = 0, $$1379 = 0, $$1379$ = 0, $$1383641 = 0, $$145$i = 0, $$1644 = 0, $$2$i$i = 0, $$2377 = 0, $$2384643 = 0, $$2638 = 0, $$3381 = 0, $$3385$in = 0, $$3385$in$ph = 0, $$3640 = 0, $$4$2$i$i = 0, $$4$4$i$i = 0, $$4$6$i$i = 0, $$4$i$i = 0, $$4637 = 0, $$5639 = 0, $$6$2$i$i = 0, $$6$4$i$i = 0, $$6$6$i$i = 0, $$6$7$i$i = 0, $$6$i$i = 0, $$7$i$i = 0, $$cast$i = 0, $$cast$i509 = 0, $$cast107$i = 0, $$cast146$i = 0, $$pre = 0, $$sink = 0.0, $$sink$i = 0, $$sink104$i$sink = 0.0, $$val410 = 0.0, $$val413 = 0.0, $$val414 = 0.0, $$val439 = 0.0, $$val454 = 0.0, $10 = 0, $100 = 0, $1004 = 0, $1009 = 0, $1015 = 0, $1036 = 0, $1042 = 0, $105 = 0, $1054 = 0, $1055 = 0, $106 = 0, $1060 = 0, $1061 = 0, $1066 = 0, $1067 = 0, $1072 = 0, $1073 = 0, $1087 = 0, $1088 = 0, $1092 = 0, $1093 = 0, $1094 = 0, $1095 = 0, $1098 = 0, $11 = 0, $1103 = 0, $1105 = 0, $1116 = 0, $1126 = 0, $113 = 0, $1130 = 0, $1131 = 0, $1136 = 0, $1137 = 0, $1142 = 0, $1143 = 0, $1144 = 0, $1158 = 0, $1159 = 0, $116 = 0, $1163 = 0, $1164 = 0, $1165 = 0, $1166 = 0, $1169 = 0, $1174 = 0, $1176 = 0, $1187 = 0, $12 = 0.0, $1201 = 0, $1227 = 0, $123 = 0, $1233 = 0, $1236 = 0, $1237 = 0, $1241 = 0, $1258 = 0, $1259 = 0, $126 = 0, $1267 = 0, $1269 = 0, $1271 = 0, $1272 = 0, $1279 = 0, $1284 = 0, $1285 = 0, $1292 = 0, $1295 = 0, $1302 = 0, $1305 = 0, $1322 = 0, $1325 = 0, $1328 = 0, $1335 = 0, $1340 = 0, $1343 = 0, $1350 = 0, $1357 = 0, $1360 = 0, $1366 = 0, $1370 = 0, $1383 = 0, $1388 = 0, $1389 = 0, $1394 = 0, $1395 = 0, $1400 = 0, $1405 = 0, $1407 = 0, $1416 = 0, $1417 = 0, $1418 = 0, $1419 = 0, $1426 = 0, $143 = 0, $1433 = 0, $1435 = 0, $1438 = 0, $1440 = 0, $1443 = 0, $1451 = 0, $1452 = 0, $1457 = 0, $1458 = 0, $146 = 0, $1463 = 0, $149 = 0, $15 = 0, $1547 = 0, $156 = 0, $1575 = 0, $1576 = 0, $1581 = 0, $1594 = 0, $1600 = 0, $1601 = 0, $1602 = 0, $1605 = 0, $1607 = 0.0, $161 = 0, $1612 = 0.0, $1614 = 0.0, $1618 = 0.0, $1620 = 0.0, $1624 = 0.0, $1626 = 0.0, $1628 = 0, $1632 = 0.0, $1633 = 0, $1635 = 0.0, $1636 = 0, $164 = 0, $1640 = 0.0, $1641 = 0, $1643 = 0.0, $1644 = 0, $1648 = 0.0, $1649 = 0, $1651 = 0.0, $1652 = 0, $1653 = 0, $1655 = 0, $1658 = 0, $1659 = 0, $1664 = 0, $1666 = 0, $1675 = 0, $1681 = 0, $1685 = 0, $1690 = 0, $1692 = 0, $1701 = 0, $1707 = 0, $171 = 0, $1713 = 0.0, $1717 = 0, $1718 = 0, $1724 = 0, $1725 = 0, $1726 = 0, $1727 = 0, $1728 = 0, $1733 = 0, $1734 = 0, $1752 = 0.0, $1754 = 0.0, $1757 = 0.0, $1759 = 0.0, $1762 = 0.0, $1765 = 0.0, $1767 = 0.0, $1769 = 0.0, $1771 = 0.0, $1773 = 0.0, $1775 = 0.0, $1777 = 0.0, $1779 = 0.0, $178 = 0, $1782 = 0.0, $1784 = 0.0, $1786 = 0.0, $1789 = 0.0, $1791 = 0.0, $1793 = 0.0, $1795 = 0.0, $1797 = 0.0, $1799 = 0.0, $181 = 0, $1811 = 0, $1818 = 0, $1819 = 0, $1824 = 0, $1828 = 0.0, $1842 = 0, $1849 = 0, $1850 = 0, $1856 = 0, $1857 = 0, $1862 = 0, $1864 = 0.0, $1865 = 0, $1866 = 0.0, $1869 = 0.0, $187 = 0, $1874 = 0, $1876 = 0.0, $1882 = 0, $1883 = 0, $1885 = 0, $1888 = 0, $1889 = 0, $1894 = 0, $1896 = 0, $1905 = 0, $1911 = 0, $1915 = 0, $1920 = 0, $1922 = 0, $1931 = 0, $1937 = 0, $1942 = 0.0, $1944 = 0, $1945 = 0, $1951 = 0, $1952 = 0, $1953 = 0, $1954 = 0, $1955 = 0, $1960 = 0, $1961 = 0, $1966 = 0, $1970 = 0.0, $1971 = 0, $1972 = 0.0, $1976 = 0.0, $1981 = 0, $1983 = 0.0, $1989 = 0, $1990 = 0, $1992 = 0, $1995 = 0, $2 = 0, $2001 = 0, $2003 = 0, $2012 = 0, $2018 = 0, $2027 = 0, $2029 = 0, $2038 = 0, $2044 = 0, $2049 = 0, $2052 = 0, $2054 = 0, $2057 = 0, $2063 = 0, $2065 = 0, $2067 = 0, $2069 = 0, $2072 = 0, $2073 = 0, $2075 = 0, $2078 = 0, $2079 = 0, $2083 = 0.0, $2084 = 0, $2086 = 0.0, $2087 = 0, $2089 = 0.0, $2093 = 0.0, $2103 = 0, $2109 = 0, $2115 = 0, $2121 = 0, $2122 = 0, $2123 = 0, $2124 = 0, $2125 = 0, $2126 = 0, $2127 = 0, $216 = 0.0, $218 = 0, $219 = 0, $220 = 0, $222 = 0, $227 = 0, $239 = 0, $245 = 0, $251 = 0, $252 = 0, $253 = 0, $258 = 0, $262 = 0, $266 = 0, $268 = 0, $271 = 0, $272 = 0, $277 = 0, $279 = 0, $280 = 0, $287 = 0, $288 = 0, $294 = 0, $295 = 0, $3 = 0, $302 = 0, $303 = 0, $309 = 0, $310 = 0, $317 = 0, $318 = 0, $323 = 0, $324 = 0, $329 = 0, $330 = 0, $336 = 0, $337 = 0, $344 = 0, $345 = 0, $351 = 0, $352 = 0, $359 = 0, $360 = 0, $366 = 0, $367 = 0, $373 = 0, $374 = 0, $376 = 0, $377 = 0, $378 = 0, $380 = 0, $383 = 0, $385 = 0, $387 = 0, $390 = 0, $392 = 0, $397 = 0, $405 = 0, $409 = 0, $415 = 0, $421 = 0, $449 = 0, $451 = 0, $460 = 0, $493 = 0, $499 = 0, $500 = 0, $501 = 0, $504 = 0, $513 = 0, $515 = 0, $518 = 0, $519 = 0, $525 = 0.0, $553 = 0, $560 = 0, $569 = 0, $570 = 0, $588 = 0, $597 = 0, $599 = 0, $602 = 0, $603 = 0, $609 = 0.0, $637 = 0, $644 = 0, $653 = 0, $654 = 0, $672 = 0, $675 = 0, $678 = 0, $684 = 0, $687 = 0, $691 = 0, $695 = 0, $698 = 0, $7 = 0, $708 = 0, $709 = 0, $719 = 0, $736 = 0, $745 = 0, $751 = 0, $752 = 0, $754 = 0, $762 = 0, $774 = 0, $777 = 0, $779 = 0, $782 = 0, $783 = 0, $789 = 0.0, $79 = 0, $80 = 0, $809 = 0, $814 = 0, $819 = 0, $824 = 0, $825 = 0, $826 = 0, $839 = 0, $845 = 0, $846 = 0, $848 = 0, $859 = 0, $864 = 0, $870 = 0, $875 = 0, $88 = 0, $884 = 0, $885 = 0, $899 = 0, $90 = 0, $917 = 0, $92 = 0, $923 = 0, $929 = 0, $93 = 0, $942 = 0.0, $944 = 0, $945 = 0, $946 = 0, $948 = 0, $953 = 0, $960 = 0, $961 = 0, $967 = 0, $973 = 0, $985 = 0.0, $987 = 0, $988 = 0, $989 = 0, $991 = 0, $996 = 0, $exp2 = 0.0, $or$cond398 = 0, $or$cond401 = 0, $storemerge = 0, $storemerge671 = 0, $vararg_buffer = 0, $vararg_buffer1 = 0, label = 0, sp = 0;
 sp = STACKTOP; //@line 123
 STACKTOP = STACKTOP + 128 | 0; //@line 124
 $vararg_buffer1 = sp + 16 | 0; //@line 125
 $vararg_buffer = sp + 8 | 0; //@line 126
 $2 = sp; //@line 127
 $3 = sp + 20 | 0; //@line 128
 switch ($1 >>> 13 & 7) {
 case 3:
  {
   $7 = 112672 + (($1 >>> 7 & 7) << 3) | 0; //@line 136
   $10 = $1 >>> 10; //@line 139
   $11 = $10 & 7; //@line 140
   if (!($1 & 16384)) {
    $12 = +HEAPF64[$7 >> 3]; //@line 142
    HEAPF64[112672 + ($11 << 3) >> 3] = $12; //@line 144
    HEAPF64[14092] = $12; //@line 145
    STACKTOP = sp; //@line 146
    return;
   }
   $15 = $0 & 7; //@line 149
   L9 : do {
    switch ($0 >>> 3 & 7) {
    case 0:
     {
      switch ($10 & 7) {
      case 6:
       {
        switch ((HEAP32[28188] | HEAP32[28187]) >>> 4 & 3) {
        case 1:
         {
          $$0$i$i = +(~~+HEAPF64[$7 >> 3] | 0); //@line 170
          break;
         }
        case 2:
         {
          $$0$i$i = +Math_floor(+(+HEAPF64[$7 >> 3])); //@line 176
          break;
         }
        case 0:
         {
          $$0$i$i = +(~~(+HEAPF64[$7 >> 3] + .5) | 0); //@line 184
          break;
         }
        case 3:
         {
          $$0$i$i = +Math_ceil(+(+HEAPF64[$7 >> 3])); //@line 190
          break;
         }
        default:
         {}
        }
        $$sink$i = ~~$$0$i$i & 255 | HEAP32[651008 + ($15 << 2) >> 2] & -256; //@line 203
        break;
       }
      case 4:
       {
        switch ((HEAP32[28188] | HEAP32[28187]) >>> 4 & 3) {
        case 1:
         {
          $$0$i92$i = +(~~+HEAPF64[$7 >> 3] | 0); //@line 218
          break;
         }
        case 2:
         {
          $$0$i92$i = +Math_floor(+(+HEAPF64[$7 >> 3])); //@line 224
          break;
         }
        case 0:
         {
          $$0$i92$i = +(~~(+HEAPF64[$7 >> 3] + .5) | 0); //@line 232
          break;
         }
        case 3:
         {
          $$0$i92$i = +Math_ceil(+(+HEAPF64[$7 >> 3])); //@line 238
          break;
         }
        default:
         {}
        }
        $$sink$i = ~~$$0$i92$i & 65535 | HEAP32[651008 + ($15 << 2) >> 2] & -65536; //@line 251
        break;
       }
      case 0:
       {
        switch ((HEAP32[28188] | HEAP32[28187]) >>> 4 & 3) {
        case 1:
         {
          $$0$i95$i = +(~~+HEAPF64[$7 >> 3] | 0); //@line 266
          break;
         }
        case 2:
         {
          $$0$i95$i = +Math_floor(+(+HEAPF64[$7 >> 3])); //@line 272
          break;
         }
        case 0:
         {
          $$0$i95$i = +(~~(+HEAPF64[$7 >> 3] + .5) | 0); //@line 280
          break;
         }
        case 3:
         {
          $$0$i95$i = +Math_ceil(+(+HEAPF64[$7 >> 3])); //@line 286
          break;
         }
        default:
         {}
        }
        $$sink$i = ~~$$0$i95$i; //@line 294
        break;
       }
      case 1:
       {
        $$sink$i = (HEAPF32[tempDoublePtr >> 2] = +HEAPF64[$7 >> 3], HEAP32[tempDoublePtr >> 2] | 0); //@line 301
        break;
       }
      default:
       {
        break L9;
       }
      }
      HEAP32[651008 + ($15 << 2) >> 2] = $$sink$i; //@line 309
      STACKTOP = sp; //@line 310
      return;
     }
    case 1:
     {
      break;
     }
    case 2:
     {
      $$080$i = HEAP32[651040 + ($15 << 2) >> 2] | 0; //@line 319
      label = 41; //@line 320
      break;
     }
    case 3:
     {
      $79 = 651040 + ($15 << 2) | 0; //@line 324
      $80 = HEAP32[$79 >> 2] | 0; //@line 325
      HEAP32[$79 >> 2] = (HEAP32[(($15 | 0) == 7 ? 6704 : 6736) + ($11 << 2) >> 2] | 0) + $80; //@line 331
      $$080$i = $80; //@line 332
      label = 41; //@line 333
      break;
     }
    case 4:
     {
      $88 = 651040 + ($15 << 2) | 0; //@line 341
      $90 = (HEAP32[$88 >> 2] | 0) - (HEAP32[(($15 | 0) == 7 ? 6704 : 6736) + ($11 << 2) >> 2] | 0) | 0; //@line 343
      HEAP32[$88 >> 2] = $90; //@line 344
      $$080$i = $90; //@line 345
      label = 41; //@line 346
      break;
     }
    case 5:
     {
      $92 = HEAP32[651040 + ($15 << 2) >> 2] | 0; //@line 351
      $93 = HEAP32[162769] | 0; //@line 352
      $100 = HEAPU8[$93 >> 0] << 8 | HEAPU8[$93 + 1 >> 0]; //@line 359
      HEAP32[162769] = $93 + 2; //@line 361
      $$080$i = ($100 << 16 >> 16) + $92 | 0; //@line 365
      label = 41; //@line 366
      break;
     }
    case 6:
     {
      $105 = HEAP32[651040 + ($15 << 2) >> 2] | 0; //@line 371
      $106 = HEAP32[162769] | 0; //@line 372
      $113 = HEAPU8[$106 >> 0] << 8 | HEAPU8[$106 + 1 >> 0]; //@line 379
      HEAP32[162769] = $106 + 2; //@line 381
      $$080$i = __Z15get_disp_ea_020jj($105, $113) | 0; //@line 383
      label = 41; //@line 384
      break;
     }
    case 7:
     {
      switch ($0 & 7) {
      case 0:
       {
        $116 = HEAP32[162769] | 0; //@line 392
        $123 = HEAPU8[$116 >> 0] << 8 | HEAPU8[$116 + 1 >> 0]; //@line 399
        HEAP32[162769] = $116 + 2; //@line 401
        $$080$i = $123 << 16 >> 16; //@line 404
        label = 41; //@line 405
        break L9;
        break;
       }
      case 1:
       {
        $126 = HEAP32[162769] | 0; //@line 410
        $143 = HEAPU8[$126 + 1 >> 0] << 16 | HEAPU8[$126 >> 0] << 24 | HEAPU8[$126 + 2 >> 0] << 8 | HEAPU8[$126 + 3 >> 0]; //@line 427
        HEAP32[162769] = $126 + 4; //@line 429
        $$080$i = $143; //@line 430
        label = 41; //@line 431
        break L9;
        break;
       }
      case 2:
       {
        $146 = HEAP32[162769] | 0; //@line 437
        $149 = $146 + (HEAP32[162768] | 0) - (HEAP32[162770] | 0) | 0; //@line 440
        $$cast146$i = $146; //@line 441
        $156 = HEAPU8[$$cast146$i >> 0] << 8 | HEAPU8[$$cast146$i + 1 >> 0]; //@line 448
        HEAP32[162769] = $$cast146$i + 2; //@line 450
        $$080$i = $149 + ($156 << 16 >> 16) | 0; //@line 454
        label = 41; //@line 455
        break L9;
        break;
       }
      case 3:
       {
        $161 = HEAP32[162769] | 0; //@line 461
        $164 = $161 + (HEAP32[162768] | 0) - (HEAP32[162770] | 0) | 0; //@line 464
        $$cast$i = $161; //@line 465
        $171 = HEAPU8[$$cast$i >> 0] << 8 | HEAPU8[$$cast$i + 1 >> 0]; //@line 472
        HEAP32[162769] = $$cast$i + 2; //@line 474
        $$080$i = __Z15get_disp_ea_020jj($164, $171) | 0; //@line 476
        label = 41; //@line 477
        break L9;
        break;
       }
      case 4:
       {
        $178 = (HEAP32[162769] | 0) + (HEAP32[162768] | 0) - (HEAP32[162770] | 0) | 0; //@line 486
        $181 = $178 + (HEAP32[6704 + ($11 << 2) >> 2] | 0) | 0; //@line 489
        $187 = FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($181 >>> 16 << 2) >> 2] | 0) + 24 >> 2] & 127]($181) | 0; //@line 495
        HEAP32[162770] = $187; //@line 496
        HEAP32[162769] = $187; //@line 497
        HEAP32[162768] = $181; //@line 498
        $$080$i = $178; //@line 499
        label = 41; //@line 500
        break L9;
        break;
       }
      default:
       {
        break L9;
       }
      }
      break;
     }
    default:
     {}
    }
   } while (0);
   L52 : do {
    if ((label | 0) == 41) {
     switch ($10 & 7) {
     case 0:
      {
       switch ((HEAP32[28188] | HEAP32[28187]) >>> 4 & 3) {
       case 1:
        {
         $$0$i104$i = +(~~+HEAPF64[$7 >> 3] | 0); //@line 532
         break;
        }
       case 2:
        {
         $$0$i104$i = +Math_floor(+(+HEAPF64[$7 >> 3])); //@line 538
         break;
        }
       case 0:
        {
         $$0$i104$i = +(~~(+HEAPF64[$7 >> 3] + .5) | 0); //@line 546
         break;
        }
       case 3:
        {
         $$0$i104$i = +Math_ceil(+(+HEAPF64[$7 >> 3])); //@line 552
         break;
        }
       default:
        {}
       }
       FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($$080$i >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($$080$i, ~~$$0$i104$i); //@line 565
       STACKTOP = sp; //@line 566
       return;
      }
     case 1:
      {
       FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($$080$i >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($$080$i, (HEAPF32[tempDoublePtr >> 2] = +HEAPF64[$7 >> 3], HEAP32[tempDoublePtr >> 2] | 0)); //@line 578
       STACKTOP = sp; //@line 579
       return;
      }
     case 2:
      {
       $216 = +HEAPF64[$7 >> 3]; //@line 583
       HEAPF64[tempDoublePtr >> 3] = $216; //@line 585
       $218 = HEAP32[tempDoublePtr >> 2] | 0; //@line 585
       $219 = HEAP32[tempDoublePtr + 4 >> 2] | 0; //@line 586
       if ($216 == 0.0) {
        $$0134$i = 0; //@line 588
        $$0135$i = 0; //@line 588
        $$0139$i = 0; //@line 588
       } else {
        $220 = _bitshift64Lshr($218 | 0, $219 | 0, 52) | 0; //@line 590
        $222 = $220 & 2047; //@line 592
        $227 = _bitshift64Lshr($218 | 0, $219 | 0, 21) | 0; //@line 599
        $$0134$i = (($222 | 0) == 2047 ? 2147418112 : ($222 << 16) + 1006632960 | 0) | $219 & -2147483648; //@line 606
        $$0135$i = $218 >>> 21 | $227 & 2147481600 | -2147483648; //@line 606
        $$0139$i = $218 << 11; //@line 606
       }
       FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($$080$i >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($$080$i, $$0134$i); //@line 613
       $239 = $$080$i + 4 | 0; //@line 614
       FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($239 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($239, $$0135$i); //@line 620
       $245 = $$080$i + 8 | 0; //@line 621
       FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($245 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($245, $$0139$i); //@line 627
       STACKTOP = sp; //@line 628
       return;
      }
     case 3:
      {
       HEAPF64[$vararg_buffer >> 3] = +HEAPF64[$7 >> 3]; //@line 633
       _sprintf($3, 106240, $vararg_buffer) | 0; //@line 634
       $251 = HEAP8[$3 >> 0] | 0; //@line 635
       $252 = $251 << 24 >> 24 == 45; //@line 636
       $253 = $3 + 1 | 0; //@line 637
       $$0$i107$i = $252 ? $253 : $3; //@line 641
       $$$0$i$i = ($252 ? HEAP8[$253 >> 0] | 0 : $251) << 24 >> 24 == 43 ? $$0$i107$i + 1 | 0 : $$0$i107$i; //@line 644
       $258 = $$$0$i$i + 1 | 0; //@line 645
       $262 = (HEAP8[$$$0$i$i >> 0] | 0) + -48 | ($252 ? -2147483648 : 0); //@line 649
       $$2$i$i = (HEAP8[$258 >> 0] | 0) == 46 ? $$$0$i$i + 2 | 0 : $258; //@line 653
       $266 = HEAP8[$$2$i$i >> 0] | 0; //@line 654
       $268 = $$2$i$i + 1 | 0; //@line 657
       if (($266 + -48 & 255) < 10) {
        $$4$i$i = $268; //@line 663
        $271 = HEAP8[$268 >> 0] | 0; //@line 663
        $277 = ($266 << 24 >> 24 << 4) + -768 | 0; //@line 663
       } else {
        $$4$i$i = $$2$i$i; //@line 665
        $271 = $266; //@line 665
        $277 = 0; //@line 665
       }
       $272 = ($271 + -48 & 255) < 10; //@line 668
       $$$4$i$i = $272 ? $$4$i$i + 1 | 0 : $$4$i$i; //@line 674
       $279 = HEAP8[$$$4$i$i >> 0] | 0; //@line 676
       $280 = ($279 + -48 & 255) < 10; //@line 678
       $$4$2$i$i = $280 ? $$$4$i$i + 1 | 0 : $$$4$i$i; //@line 684
       $287 = HEAP8[$$4$2$i$i >> 0] | 0; //@line 686
       $288 = ($287 + -48 & 255) < 10; //@line 688
       $$$4$2$i$i = $288 ? $$4$2$i$i + 1 | 0 : $$4$2$i$i; //@line 694
       $294 = HEAP8[$$$4$2$i$i >> 0] | 0; //@line 696
       $295 = ($294 + -48 & 255) < 10; //@line 698
       $$4$4$i$i = $295 ? $$$4$2$i$i + 1 | 0 : $$$4$2$i$i; //@line 704
       $302 = HEAP8[$$4$4$i$i >> 0] | 0; //@line 706
       $303 = ($302 + -48 & 255) < 10; //@line 708
       $$$4$4$i$i = $303 ? $$4$4$i$i + 1 | 0 : $$4$4$i$i; //@line 714
       $309 = HEAP8[$$$4$4$i$i >> 0] | 0; //@line 716
       $310 = ($309 + -48 & 255) < 10; //@line 718
       $$4$6$i$i = $310 ? $$$4$4$i$i + 1 | 0 : $$$4$4$i$i; //@line 724
       $317 = HEAP8[$$4$6$i$i >> 0] | 0; //@line 726
       $318 = ($317 + -48 & 255) < 10; //@line 728
       $$145$i = (((((($280 ? ($279 << 24 >> 24) + 16777168 | 0 : 0) | (($272 ? ($271 << 24 >> 24) + 16777168 | 0 : 0) | $277) << 4) << 4 | ($288 ? ($287 << 24 >> 24) + 16777168 | 0 : 0)) << 4 | ($295 ? ($294 << 24 >> 24) + 16777168 | 0 : 0)) << 4 | ($303 ? ($302 << 24 >> 24) + 16777168 | 0 : 0)) << 4 | ($310 ? ($309 << 24 >> 24) + 268435408 | 0 : 0)) << 4 | ($318 ? ($317 << 24 >> 24) + -48 | 0 : 0); //@line 733
       $$$4$6$i$i = $318 ? $$4$6$i$i + 1 | 0 : $$4$6$i$i; //@line 734
       $323 = HEAP8[$$$4$6$i$i >> 0] | 0; //@line 735
       $324 = ($323 + -48 & 255) < 10; //@line 737
       $$6$i$i = $324 ? $$$4$6$i$i + 1 | 0 : $$$4$6$i$i; //@line 740
       $329 = HEAP8[$$6$i$i >> 0] | 0; //@line 744
       $330 = ($329 + -48 & 255) < 10; //@line 746
       $$$6$i$i = $330 ? $$6$i$i + 1 | 0 : $$6$i$i; //@line 752
       $336 = HEAP8[$$$6$i$i >> 0] | 0; //@line 754
       $337 = ($336 + -48 & 255) < 10; //@line 756
       $$6$2$i$i = $337 ? $$$6$i$i + 1 | 0 : $$$6$i$i; //@line 762
       $344 = HEAP8[$$6$2$i$i >> 0] | 0; //@line 764
       $345 = ($344 + -48 & 255) < 10; //@line 766
       $$$6$2$i$i = $345 ? $$6$2$i$i + 1 | 0 : $$6$2$i$i; //@line 772
       $351 = HEAP8[$$$6$2$i$i >> 0] | 0; //@line 774
       $352 = ($351 + -48 & 255) < 10; //@line 776
       $$6$4$i$i = $352 ? $$$6$2$i$i + 1 | 0 : $$$6$2$i$i; //@line 782
       $359 = HEAP8[$$6$4$i$i >> 0] | 0; //@line 784
       $360 = ($359 + -48 & 255) < 10; //@line 786
       $$$6$4$i$i = $360 ? $$6$4$i$i + 1 | 0 : $$6$4$i$i; //@line 792
       $366 = HEAP8[$$$6$4$i$i >> 0] | 0; //@line 794
       $367 = ($366 + -48 & 255) < 10; //@line 796
       $$6$6$i$i = $367 ? $$$6$4$i$i + 1 | 0 : $$$6$4$i$i; //@line 802
       $373 = ((((((($324 ? ($323 << 24 >> 24 << 4) + 16776448 | 0 : 0) | ($330 ? ($329 << 24 >> 24) + 16777168 | 0 : 0)) << 4 | ($337 ? ($336 << 24 >> 24) + 16777168 | 0 : 0)) << 4 | ($345 ? ($344 << 24 >> 24) + 16777168 | 0 : 0)) << 4 | ($352 ? ($351 << 24 >> 24) + 16777168 | 0 : 0)) << 4 | ($360 ? ($359 << 24 >> 24) + 16777168 | 0 : 0)) << 4 | ($367 ? ($366 << 24 >> 24) + 268435408 | 0 : 0)) << 4; //@line 803
       $374 = HEAP8[$$6$6$i$i >> 0] | 0; //@line 804
       $376 = $$6$6$i$i + 1 | 0; //@line 807
       if (($374 + -48 & 255) < 10) {
        $$0136$i = $373 | ($374 << 24 >> 24) + -48; //@line 813
        $$6$7$i$i = $376; //@line 813
        $409 = HEAP8[$376 >> 0] | 0; //@line 813
       } else {
        $$0136$i = $373; //@line 815
        $$6$7$i$i = $$6$6$i$i; //@line 815
        $409 = $374; //@line 815
       }
       switch ($409 << 24 >> 24) {
       case 69:
       case 101:
        {
         $377 = $$6$7$i$i + 1 | 0; //@line 819
         $378 = HEAP8[$377 >> 0] | 0; //@line 820
         $380 = $$6$7$i$i + 2 | 0; //@line 822
         if ($378 << 24 >> 24 == 45) {
          $$7$i$i = $380; //@line 826
          $383 = HEAP8[$380 >> 0] | 0; //@line 826
          $405 = $262 | 1073741824; //@line 826
         } else {
          $$7$i$i = $377; //@line 828
          $383 = $378; //@line 828
          $405 = $262; //@line 828
         }
         $$$7$i$i = $383 << 24 >> 24 == 43 ? $$7$i$i + 1 | 0 : $$7$i$i; //@line 832
         $385 = HEAP8[$$$7$i$i >> 0] | 0; //@line 833
         $387 = ($385 + -48 & 255) < 10; //@line 836
         $$$047$i$i = $387 ? ($385 << 24 >> 24) + -48 | 0 : 0; //@line 839
         $$$9$i$i = $387 ? $$$7$i$i + 1 | 0 : $$$7$i$i; //@line 840
         $390 = HEAP8[$$$9$i$i >> 0] | 0; //@line 841
         $392 = ($390 + -48 & 255) < 10; //@line 844
         $$$047$1$i$i = $392 ? $$$047$i$i << 4 | ($390 << 24 >> 24) + -48 : $$$047$i$i; //@line 849
         $397 = HEAP8[($392 ? $$$9$i$i + 1 | 0 : $$$9$i$i) >> 0] | 0; //@line 851
         $$0138$i = (($397 + -48 & 255) < 10 ? $$$047$1$i$i << 4 | ($397 << 24 >> 24) + 65488 : $$$047$1$i$i) << 16 | $405; //@line 861
         break;
        }
       default:
        {
         $$0138$i = $262; //@line 865
        }
       }
       FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($$080$i >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($$080$i, $$0138$i); //@line 873
       $415 = $$080$i + 4 | 0; //@line 874
       FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($415 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($415, $$145$i); //@line 880
       $421 = $$080$i + 8 | 0; //@line 881
       FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($421 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($421, $$0136$i); //@line 887
       STACKTOP = sp; //@line 888
       return;
      }
     case 4:
      {
       switch ((HEAP32[28188] | HEAP32[28187]) >>> 4 & 3) {
       case 1:
        {
         $$0$i101$i = +(~~+HEAPF64[$7 >> 3] | 0); //@line 903
         break;
        }
       case 2:
        {
         $$0$i101$i = +Math_floor(+(+HEAPF64[$7 >> 3])); //@line 909
         break;
        }
       case 0:
        {
         $$0$i101$i = +(~~(+HEAPF64[$7 >> 3] + .5) | 0); //@line 917
         break;
        }
       case 3:
        {
         $$0$i101$i = +Math_ceil(+(+HEAPF64[$7 >> 3])); //@line 923
         break;
        }
       default:
        {}
       }
       FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($$080$i >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($$080$i, ~~$$0$i101$i << 16 >> 16); //@line 938
       STACKTOP = sp; //@line 939
       return;
      }
     case 5:
      {
       $449 = $7; //@line 943
       $451 = HEAP32[$449 >> 2] | 0; //@line 945
       FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($$080$i >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($$080$i, HEAP32[$449 + 4 >> 2] | 0); //@line 954
       $460 = $$080$i + 4 | 0; //@line 955
       FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($460 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($460, $451); //@line 961
       STACKTOP = sp; //@line 962
       return;
      }
     case 6:
      {
       switch ((HEAP32[28188] | HEAP32[28187]) >>> 4 & 3) {
       case 1:
        {
         $$0$i98$i = +(~~+HEAPF64[$7 >> 3] | 0); //@line 977
         break;
        }
       case 2:
        {
         $$0$i98$i = +Math_floor(+(+HEAPF64[$7 >> 3])); //@line 983
         break;
        }
       case 0:
        {
         $$0$i98$i = +(~~(+HEAPF64[$7 >> 3] + .5) | 0); //@line 991
         break;
        }
       case 3:
        {
         $$0$i98$i = +Math_ceil(+(+HEAPF64[$7 >> 3])); //@line 997
         break;
        }
       default:
        {}
       }
       FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($$080$i >>> 16 << 2) >> 2] | 0) + 20 >> 2] & 31]($$080$i, ~~$$0$i98$i << 24 >> 24); //@line 1012
       STACKTOP = sp; //@line 1013
       return;
      }
     default:
      {
       break L52;
      }
     }
    }
   } while (0);
   $493 = (HEAP32[162768] | 0) + -4 + (HEAP32[162769] | 0) - (HEAP32[162770] | 0) | 0; //@line 1027
   $499 = FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($493 >>> 16 << 2) >> 2] | 0) + 24 >> 2] & 127]($493) | 0; //@line 1033
   HEAP32[162770] = $499; //@line 1034
   HEAP32[162769] = $499; //@line 1035
   HEAP32[162768] = $493; //@line 1036
   __Z7op_illgj($0); //@line 1037
   STACKTOP = sp; //@line 1038
   return;
  }
 case 5:
 case 4:
  {
   $500 = $0 & 56; //@line 1042
   $501 = $1 & 8192; //@line 1043
   switch ($500 & 63) {
   case 0:
    {
     $504 = ($1 & 4096 | 0) != 0; //@line 1050
     if (!$501) {
      if ($504) {
       $553 = HEAP32[651008 + (($0 & 7) << 2) >> 2] | 0; //@line 1055
       HEAP32[28187] = $553 & 192; //@line 1057
       HEAP32[28188] = $553 & 48; //@line 1059
      }
      if ($1 & 2048 | 0) {
       $560 = HEAP32[651008 + (($0 & 7) << 2) >> 2] | 0; //@line 1066
       if (!($560 & 16777216)) {
        HEAPF64[14092] = ($560 & 67108864 | 0) == 0 ? $560 & 134217728 | 0 ? -1.0 : 1.0 : 0.0; //@line 1076
       } else {
        $569 = HEAP32[28185] | 2147483647; //@line 1084
        $570 = 112736; //@line 1085
        HEAP32[$570 >> 2] = -1; //@line 1087
        HEAP32[$570 + 4 >> 2] = $569; //@line 1090
       }
       HEAP32[28191] = $560 & 65280; //@line 1093
       HEAP32[28192] = $560 & 255; //@line 1095
       HEAP32[28190] = $560 & 16711680; //@line 1097
      }
      if (!($1 & 1024)) {
       STACKTOP = sp; //@line 1102
       return;
      }
      HEAP32[28193] = HEAP32[651008 + (($0 & 7) << 2) >> 2]; //@line 1107
      STACKTOP = sp; //@line 1108
      return;
     }
     if ($504) {
      HEAP32[651008 + (($0 & 7) << 2) >> 2] = (HEAP32[28188] | HEAP32[28187]) & 65535; //@line 1117
     }
     if ($1 & 2048 | 0) {
      $513 = 112736; //@line 1122
      $515 = HEAP32[$513 >> 2] | 0; //@line 1124
      $518 = HEAP32[$513 + 4 >> 2] | 0; //@line 1127
      $519 = $518 & 2147483647; //@line 1128
      HEAP32[tempDoublePtr >> 2] = $515; //@line 1134
      HEAP32[tempDoublePtr + 4 >> 2] = $518; //@line 1134
      $525 = +HEAPF64[tempDoublePtr >> 3]; //@line 1134
      if (($519 | ($515 | 0 - $515) >>> 31) >>> 0 > 2146435072) {
       $$0$i$i457 = 16777216; //@line 1136
      } else {
       if ($525 == 0.0) {
        $$0$i$i457 = 67108864; //@line 1140
       } else {
        if ($525 < 0.0) {
         $$0$i$i457 = 134217728; //@line 1144
        } else {
         $$0$i$i457 = 0; //@line 1146
        }
       }
      }
      HEAP32[651008 + (($0 & 7) << 2) >> 2] = ($518 >> 30 & ~(($515 | 0 - ($519 ^ 2146435072 | $515)) >> 31) | 0 ? $$0$i$i457 | 33554432 : $$0$i$i457) | HEAP32[28191] | HEAP32[28192] | HEAP32[28190]; //@line 1169
     }
     if (!($1 & 1024)) {
      STACKTOP = sp; //@line 1174
      return;
     }
     HEAP32[651008 + (($0 & 7) << 2) >> 2] = HEAP32[28193]; //@line 1179
     STACKTOP = sp; //@line 1180
     return;
    }
   case 8:
    {
     $588 = ($1 & 4096 | 0) != 0; //@line 1186
     if (!$501) {
      if ($588) {
       $637 = HEAP32[651040 + (($0 & 7) << 2) >> 2] | 0; //@line 1191
       HEAP32[28187] = $637 & 192; //@line 1193
       HEAP32[28188] = $637 & 48; //@line 1195
      }
      if ($1 & 2048 | 0) {
       $644 = HEAP32[651040 + (($0 & 7) << 2) >> 2] | 0; //@line 1202
       if (!($644 & 16777216)) {
        HEAPF64[14092] = ($644 & 67108864 | 0) == 0 ? $644 & 134217728 | 0 ? -1.0 : 1.0 : 0.0; //@line 1212
       } else {
        $653 = HEAP32[28185] | 2147483647; //@line 1220
        $654 = 112736; //@line 1221
        HEAP32[$654 >> 2] = -1; //@line 1223
        HEAP32[$654 + 4 >> 2] = $653; //@line 1226
       }
       HEAP32[28191] = $644 & 65280; //@line 1229
       HEAP32[28192] = $644 & 255; //@line 1231
       HEAP32[28190] = $644 & 16711680; //@line 1233
      }
      if (!($1 & 1024)) {
       STACKTOP = sp; //@line 1238
       return;
      }
      HEAP32[28193] = HEAP32[651040 + (($0 & 7) << 2) >> 2]; //@line 1243
      STACKTOP = sp; //@line 1244
      return;
     }
     if ($588) {
      HEAP32[651040 + (($0 & 7) << 2) >> 2] = (HEAP32[28188] | HEAP32[28187]) & 65535; //@line 1253
     }
     if ($1 & 2048 | 0) {
      $597 = 112736; //@line 1258
      $599 = HEAP32[$597 >> 2] | 0; //@line 1260
      $602 = HEAP32[$597 + 4 >> 2] | 0; //@line 1263
      $603 = $602 & 2147483647; //@line 1264
      HEAP32[tempDoublePtr >> 2] = $599; //@line 1270
      HEAP32[tempDoublePtr + 4 >> 2] = $602; //@line 1270
      $609 = +HEAPF64[tempDoublePtr >> 3]; //@line 1270
      if (($603 | ($599 | 0 - $599) >>> 31) >>> 0 > 2146435072) {
       $$0$i$i459 = 16777216; //@line 1272
      } else {
       if ($609 == 0.0) {
        $$0$i$i459 = 67108864; //@line 1276
       } else {
        if ($609 < 0.0) {
         $$0$i$i459 = 134217728; //@line 1280
        } else {
         $$0$i$i459 = 0; //@line 1282
        }
       }
      }
      HEAP32[651040 + (($0 & 7) << 2) >> 2] = ($602 >> 30 & ~(($599 | 0 - ($603 ^ 2146435072 | $599)) >> 31) | 0 ? $$0$i$i459 | 33554432 : $$0$i$i459) | HEAP32[28191] | HEAP32[28192] | HEAP32[28190]; //@line 1305
     }
     if (!($1 & 1024)) {
      STACKTOP = sp; //@line 1310
      return;
     }
     HEAP32[651040 + (($0 & 7) << 2) >> 2] = HEAP32[28193]; //@line 1315
     STACKTOP = sp; //@line 1316
     return;
    }
   default:
    {
     $672 = ($501 | 0) == 0; //@line 1322
     if (($0 & 63 | 0) == 60) {
      if (!$672) {
       STACKTOP = sp; //@line 1325
       return;
      }
      if ($1 & 4096 | 0) {
       $675 = HEAP32[162769] | 0; //@line 1330
       $678 = HEAPU8[$675 + 3 >> 0] | 0; //@line 1333
       HEAP32[162769] = $675 + 4; //@line 1335
       HEAP32[28187] = $678 & 192; //@line 1337
       HEAP32[28188] = $678 & 48; //@line 1339
      }
      if ($1 & 2048 | 0) {
       $684 = HEAP32[162769] | 0; //@line 1344
       $687 = HEAPU8[$684 >> 0] << 24; //@line 1347
       $691 = HEAPU8[$684 + 1 >> 0] << 16; //@line 1351
       $695 = HEAPU8[$684 + 2 >> 0] << 8; //@line 1355
       $698 = HEAPU8[$684 + 3 >> 0] | 0; //@line 1358
       HEAP32[162769] = $684 + 4; //@line 1360
       if (!($687 & 16777216)) {
        HEAPF64[14092] = ($687 & 67108864 | 0) == 0 ? $687 & 134217728 | 0 ? -1.0 : 1.0 : 0.0; //@line 1370
       } else {
        $708 = HEAP32[28185] | 2147483647; //@line 1378
        $709 = 112736; //@line 1379
        HEAP32[$709 >> 2] = -1; //@line 1381
        HEAP32[$709 + 4 >> 2] = $708; //@line 1384
       }
       HEAP32[28191] = $695; //@line 1386
       HEAP32[28192] = $698; //@line 1387
       HEAP32[28190] = $691; //@line 1388
      }
      if (!($1 & 1024)) {
       STACKTOP = sp; //@line 1393
       return;
      }
      $719 = HEAP32[162769] | 0; //@line 1395
      $736 = HEAPU8[$719 + 1 >> 0] << 16 | HEAPU8[$719 >> 0] << 24 | HEAPU8[$719 + 2 >> 0] << 8 | HEAPU8[$719 + 3 >> 0]; //@line 1412
      HEAP32[162769] = $719 + 4; //@line 1414
      HEAP32[28193] = $736; //@line 1415
      STACKTOP = sp; //@line 1416
      return;
     }
     if ($672) {
      if (!(__ZL9get_fp_adjPj($0, $2) | 0)) {
       $839 = (HEAP32[162768] | 0) + -4 + (HEAP32[162769] | 0) - (HEAP32[162770] | 0) | 0; //@line 1427
       $845 = FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($839 >>> 16 << 2) >> 2] | 0) + 24 >> 2] & 127]($839) | 0; //@line 1433
       HEAP32[162770] = $845; //@line 1434
       HEAP32[162769] = $845; //@line 1435
       HEAP32[162768] = $839; //@line 1436
       __Z7op_illgj($0); //@line 1437
       STACKTOP = sp; //@line 1438
       return;
      }
      $846 = ($500 | 0) == 32; //@line 1440
      if ($846) {
       $848 = $1 >>> 10 & 4; //@line 1443
       $$1379 = ($1 & 2048 | 0) == 0 ? $848 : $848 + 4 | 0; //@line 1447
       $$1379$ = ($1 & 1024 | 0) == 0 ? $$1379 : $$1379 + 4 | 0; //@line 1451
       HEAP32[$2 >> 2] = (HEAP32[$2 >> 2] | 0) - $$1379$; //@line 1454
       $$3381 = $$1379$; //@line 1455
      } else {
       $$3381 = 0; //@line 1457
      }
      if ($1 & 4096 | 0) {
       $859 = HEAP32[$2 >> 2] | 0; //@line 1462
       $864 = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($859 >>> 16 << 2) >> 2] >> 2] & 127]($859) | 0; //@line 1467
       HEAP32[28187] = $864 & 192; //@line 1469
       HEAP32[28188] = $864 & 48; //@line 1471
       HEAP32[$2 >> 2] = $859 + 4; //@line 1473
      }
      if ($1 & 2048 | 0) {
       $870 = HEAP32[$2 >> 2] | 0; //@line 1478
       $875 = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($870 >>> 16 << 2) >> 2] >> 2] & 127]($870) | 0; //@line 1483
       if (!($875 & 16777216)) {
        HEAPF64[14092] = ($875 & 67108864 | 0) == 0 ? $875 & 134217728 | 0 ? -1.0 : 1.0 : 0.0; //@line 1493
       } else {
        $884 = HEAP32[28185] | 2147483647; //@line 1501
        $885 = 112736; //@line 1502
        HEAP32[$885 >> 2] = -1; //@line 1504
        HEAP32[$885 + 4 >> 2] = $884; //@line 1507
       }
       HEAP32[28191] = $875 & 65280; //@line 1510
       HEAP32[28192] = $875 & 255; //@line 1512
       HEAP32[28190] = $875 & 16711680; //@line 1514
       HEAP32[$2 >> 2] = $870 + 4; //@line 1516
      }
      if ($1 & 1024 | 0) {
       $899 = HEAP32[$2 >> 2] | 0; //@line 1521
       HEAP32[28193] = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($899 >>> 16 << 2) >> 2] >> 2] & 127]($899) | 0; //@line 1527
       HEAP32[$2 >> 2] = $899 + 4; //@line 1529
      }
      if (($500 | 0) == 24) {
       HEAP32[651040 + (($0 & 7) << 2) >> 2] = HEAP32[$2 >> 2]; //@line 1536
      }
      if ($846) {
       HEAP32[651040 + (($0 & 7) << 2) >> 2] = (HEAP32[$2 >> 2] | 0) - $$3381; //@line 1543
      }
      STACKTOP = sp; //@line 1545
      return;
     }
     if (!(__ZL9get_fp_adjPj($0, $2) | 0)) {
      $745 = (HEAP32[162768] | 0) + -4 + (HEAP32[162769] | 0) - (HEAP32[162770] | 0) | 0; //@line 1555
      $751 = FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($745 >>> 16 << 2) >> 2] | 0) + 24 >> 2] & 127]($745) | 0; //@line 1561
      HEAP32[162770] = $751; //@line 1562
      HEAP32[162769] = $751; //@line 1563
      HEAP32[162768] = $745; //@line 1564
      __Z7op_illgj($0); //@line 1565
      STACKTOP = sp; //@line 1566
      return;
     }
     $752 = ($500 | 0) == 32; //@line 1568
     if ($752) {
      $754 = $1 >>> 10 & 4; //@line 1571
      $$1376 = ($1 & 2048 | 0) == 0 ? $754 : $754 + 4 | 0; //@line 1575
      $$2377 = ($1 & 1024 | 0) == 0 ? $$1376 : $$1376 + 4 | 0; //@line 1580
     } else {
      $$2377 = 0; //@line 1582
     }
     $762 = (HEAP32[$2 >> 2] | 0) - $$2377 | 0; //@line 1585
     HEAP32[$2 >> 2] = $762; //@line 1586
     if (!($1 & 4096)) {
      $809 = $762; //@line 1590
     } else {
      FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($762 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($762, (HEAP32[28188] | HEAP32[28187]) & 65535); //@line 1601
      $774 = $762 + 4 | 0; //@line 1602
      HEAP32[$2 >> 2] = $774; //@line 1603
      $809 = $774; //@line 1604
     }
     if (!($1 & 2048)) {
      $819 = $809; //@line 1609
     } else {
      $777 = 112736; //@line 1611
      $779 = HEAP32[$777 >> 2] | 0; //@line 1613
      $782 = HEAP32[$777 + 4 >> 2] | 0; //@line 1616
      $783 = $782 & 2147483647; //@line 1617
      HEAP32[tempDoublePtr >> 2] = $779; //@line 1623
      HEAP32[tempDoublePtr + 4 >> 2] = $782; //@line 1623
      $789 = +HEAPF64[tempDoublePtr >> 3]; //@line 1623
      if (($783 | ($779 | 0 - $779) >>> 31) >>> 0 > 2146435072) {
       $$0$i$i468 = 16777216; //@line 1625
      } else {
       if ($789 == 0.0) {
        $$0$i$i468 = 67108864; //@line 1629
       } else {
        if ($789 < 0.0) {
         $$0$i$i468 = 134217728; //@line 1633
        } else {
         $$0$i$i468 = 0; //@line 1635
        }
       }
      }
      FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($809 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($809, ($782 >> 30 & ~(($779 | 0 - ($783 ^ 2146435072 | $779)) >> 31) | 0 ? $$0$i$i468 | 33554432 : $$0$i$i468) | HEAP32[28191] | HEAP32[28192] | HEAP32[28190]); //@line 1661
      $814 = $809 + 4 | 0; //@line 1662
      HEAP32[$2 >> 2] = $814; //@line 1663
      $819 = $814; //@line 1664
     }
     if (!($1 & 1024)) {
      $826 = $819; //@line 1669
     } else {
      FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($819 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($819, HEAP32[28193] | 0); //@line 1677
      $824 = $819 + 4 | 0; //@line 1678
      HEAP32[$2 >> 2] = $824; //@line 1679
      $826 = $824; //@line 1680
     }
     $825 = $826 - $$2377 | 0; //@line 1682
     HEAP32[$2 >> 2] = $825; //@line 1683
     if (($500 | 0) == 24) {
      HEAP32[651040 + (($0 & 7) << 2) >> 2] = $825; //@line 1688
     }
     if ($752) {
      HEAP32[651040 + (($0 & 7) << 2) >> 2] = $825; //@line 1693
     }
     STACKTOP = sp; //@line 1695
     return;
    }
   }
   break;
  }
 case 7:
 case 6:
  {
   $917 = (__ZL9get_fp_adjPj($0, $2) | 0) == 0; //@line 1704
   L255 : do {
    if (!($1 & 8192)) {
     if ($917) {
      $1036 = (HEAP32[162768] | 0) + -4 + (HEAP32[162769] | 0) - (HEAP32[162770] | 0) | 0; //@line 1713
      $1042 = FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($1036 >>> 16 << 2) >> 2] | 0) + 24 >> 2] & 127]($1036) | 0; //@line 1719
      HEAP32[162770] = $1042; //@line 1720
      HEAP32[162769] = $1042; //@line 1721
      HEAP32[162768] = $1036; //@line 1722
      __Z7op_illgj($0); //@line 1723
      break;
     }
     switch ($1 >>> 11 & 3) {
     case 0:
      {
       $$3385$in$ph = $1; //@line 1731
       label = 196; //@line 1732
       break;
      }
     case 1:
      {
       $$3385$in$ph = HEAP32[651008 + (($1 >>> 4 & 3) << 2) >> 2] | 0; //@line 1740
       label = 196; //@line 1741
       break;
      }
     case 2:
      {
       $$3385$in = $1; //@line 1745
       label = 197; //@line 1746
       break;
      }
     case 3:
      {
       $$3385$in = HEAP32[651008 + (($1 >>> 4 & 3) << 2) >> 2] | 0; //@line 1754
       label = 197; //@line 1755
       break;
      }
     default:
      {}
     }
     if ((label | 0) == 196) {
      $$2638 = 7; //@line 1765
      $$4637 = $$3385$in$ph & 255; //@line 1765
      $1055 = HEAP32[$2 >> 2] | 0; //@line 1765
      while (1) {
       do {
        if (!($$4637 & 128)) {
         $2124 = $1055; //@line 1771
        } else {
         $1054 = $1055 + -4 | 0; //@line 1773
         $1060 = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($1054 >>> 16 << 2) >> 2] >> 2] & 127]($1054) | 0; //@line 1778
         $1061 = $1055 + -8 | 0; //@line 1779
         $1066 = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($1061 >>> 16 << 2) >> 2] >> 2] & 127]($1061) | 0; //@line 1784
         $1067 = $1055 + -12 | 0; //@line 1785
         $1072 = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($1067 >>> 16 << 2) >> 2] >> 2] & 127]($1067) | 0; //@line 1790
         $1073 = 112672 + ($$2638 << 3) | 0; //@line 1791
         if (!($1066 | $1060 | $1072)) {
          HEAPF64[$1073 >> 3] = 0.0; //@line 1796
          $2124 = $1067; //@line 1797
          break;
         }
         if (($1060 | 0) != 0 & (($1066 | 0) != 0 & ($1072 & 2147418112 | 0) == 2147418112)) {
          $1087 = HEAP32[$1073 + 4 >> 2] | 2147483647; //@line 1813
          $1088 = $1073; //@line 1814
          HEAP32[$1088 >> 2] = -1; //@line 1816
          HEAP32[$1088 + 4 >> 2] = $1087; //@line 1819
          $2124 = $1067; //@line 1820
          break;
         } else {
          $1092 = $1072 >>> 16; //@line 1823
          $1093 = $1092 & 32767; //@line 1824
          $1094 = $1093 >>> 0 < 15360; //@line 1825
          $1095 = $1093 >>> 0 > 17406; //@line 1826
          $1098 = _bitshift64Shl($1092 + 1024 & 2047 | 0, 0, 52) | 0; //@line 1829
          $1103 = $1094 ? 0 : $1095 ? 2146435072 : tempRet0; //@line 1834
          $1105 = _bitshift64Shl($1072 >>> 31 | 0, 0, 63) | 0; //@line 1836
          $1116 = $1073; //@line 1847
          HEAP32[$1116 >> 2] = $1066 << 21 | $1060 >>> 11 | $1105 | ($1094 ? 0 : $1095 ? 0 : $1098); //@line 1849
          HEAP32[$1116 + 4 >> 2] = $1066 >>> 11 & 1048575 | tempRet0 | $1103; //@line 1852
          $2124 = $1067; //@line 1853
          break;
         }
        }
       } while (0);
       if (($$2638 | 0) > 0) {
        $$2638 = $$2638 + -1 | 0; //@line 1862
        $$4637 = $$4637 << 1; //@line 1862
        $1055 = $2124; //@line 1862
       } else {
        $storemerge671 = $2124; //@line 1864
        break;
       }
      }
     } else if ((label | 0) == 197) {
      $$3640 = 0; //@line 1872
      $$5639 = $$3385$in & 255; //@line 1872
      $1126 = HEAP32[$2 >> 2] | 0; //@line 1872
      while (1) {
       do {
        if (!($$5639 & 128)) {
         $2125 = $1126; //@line 1878
        } else {
         $1130 = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($1126 >>> 16 << 2) >> 2] >> 2] & 127]($1126) | 0; //@line 1884
         $1131 = $1126 + 4 | 0; //@line 1885
         $1136 = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($1131 >>> 16 << 2) >> 2] >> 2] & 127]($1131) | 0; //@line 1890
         $1137 = $1126 + 8 | 0; //@line 1891
         $1142 = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($1137 >>> 16 << 2) >> 2] >> 2] & 127]($1137) | 0; //@line 1896
         $1143 = $1126 + 12 | 0; //@line 1897
         $1144 = 112672 + ($$3640 << 3) | 0; //@line 1898
         if (!($1136 | $1130 | $1142)) {
          HEAPF64[$1144 >> 3] = 0.0; //@line 1903
          $2125 = $1143; //@line 1904
          break;
         }
         if (($1130 & 2147418112 | 0) == 2147418112 & ($1136 | 0) != 0 & ($1142 | 0) != 0) {
          $1158 = HEAP32[$1144 + 4 >> 2] | 2147483647; //@line 1920
          $1159 = $1144; //@line 1921
          HEAP32[$1159 >> 2] = -1; //@line 1923
          HEAP32[$1159 + 4 >> 2] = $1158; //@line 1926
          $2125 = $1143; //@line 1927
          break;
         } else {
          $1163 = $1130 >>> 16; //@line 1930
          $1164 = $1163 & 32767; //@line 1931
          $1165 = $1164 >>> 0 < 15360; //@line 1932
          $1166 = $1164 >>> 0 > 17406; //@line 1933
          $1169 = _bitshift64Shl($1163 + 1024 & 2047 | 0, 0, 52) | 0; //@line 1936
          $1174 = $1165 ? 0 : $1166 ? 2146435072 : tempRet0; //@line 1941
          $1176 = _bitshift64Shl($1130 >>> 31 | 0, 0, 63) | 0; //@line 1943
          $1187 = $1144; //@line 1954
          HEAP32[$1187 >> 2] = ($1165 ? 0 : $1166 ? 0 : $1169) | $1176 | ($1142 >>> 11 | $1136 << 21); //@line 1956
          HEAP32[$1187 + 4 >> 2] = $1174 | tempRet0 | $1136 >>> 11 & 1048575; //@line 1959
          $2125 = $1143; //@line 1960
          break;
         }
        }
       } while (0);
       $$3640 = $$3640 + 1 | 0; //@line 1966
       if (($$3640 | 0) == 8) {
        $storemerge671 = $2125; //@line 1969
        break;
       } else {
        $$5639 = $$5639 << 1; //@line 1972
        $1126 = $2125; //@line 1972
       }
      }
     }
     HEAP32[$2 >> 2] = $storemerge671; //@line 1976
     switch ($0 & 56) {
     case 24:
      {
       HEAP32[651040 + (($0 & 7) << 2) >> 2] = HEAP32[$2 >> 2]; //@line 1985
       break L255;
       break;
      }
     case 32:
      {
       HEAP32[651040 + (($0 & 7) << 2) >> 2] = HEAP32[$2 >> 2]; //@line 1993
       break L255;
       break;
      }
     default:
      {
       break L255;
      }
     }
    } else {
     if ($917) {
      $923 = (HEAP32[162768] | 0) + -4 + (HEAP32[162769] | 0) - (HEAP32[162770] | 0) | 0; //@line 2008
      $929 = FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($923 >>> 16 << 2) >> 2] | 0) + 24 >> 2] & 127]($923) | 0; //@line 2014
      HEAP32[162770] = $929; //@line 2015
      HEAP32[162769] = $929; //@line 2016
      HEAP32[162768] = $923; //@line 2017
      __Z7op_illgj($0); //@line 2018
      break;
     }
     switch ($1 >>> 11 & 3) {
     case 0:
      {
       $$0382$in$ph = $1; //@line 2026
       label = 175; //@line 2027
       break;
      }
     case 1:
      {
       $$0382$in$ph = HEAP32[651008 + (($1 >>> 4 & 3) << 2) >> 2] | 0; //@line 2035
       label = 175; //@line 2036
       break;
      }
     case 2:
      {
       $$0382$in = $1; //@line 2040
       label = 176; //@line 2041
       break;
      }
     case 3:
      {
       $$0382$in = HEAP32[651008 + (($1 >>> 4 & 3) << 2) >> 2] | 0; //@line 2049
       label = 176; //@line 2050
       break;
      }
     default:
      {}
     }
     if ((label | 0) == 175) {
      $$0372642 = 7; //@line 2060
      $$1383641 = $$0382$in$ph & 255; //@line 2060
      $961 = HEAP32[$2 >> 2] | 0; //@line 2060
      while (1) {
       if (!($$1383641 & 128)) {
        $2122 = $961; //@line 2065
       } else {
        $942 = +HEAPF64[112672 + ($$0372642 << 3) >> 3]; //@line 2068
        HEAPF64[tempDoublePtr >> 3] = $942; //@line 2070
        $944 = HEAP32[tempDoublePtr >> 2] | 0; //@line 2070
        $945 = HEAP32[tempDoublePtr + 4 >> 2] | 0; //@line 2071
        if ($942 == 0.0) {
         $$0603 = 0; //@line 2073
         $$0604 = 0; //@line 2073
         $$0608 = 0; //@line 2073
        } else {
         $946 = _bitshift64Lshr($944 | 0, $945 | 0, 52) | 0; //@line 2075
         $948 = $946 & 2047; //@line 2077
         $953 = _bitshift64Lshr($944 | 0, $945 | 0, 21) | 0; //@line 2084
         $$0603 = (($948 | 0) == 2047 ? 2147418112 : ($948 << 16) + 1006632960 | 0) | $945 & -2147483648; //@line 2091
         $$0604 = $944 >>> 21 | $953 & 2147481600 | -2147483648; //@line 2091
         $$0608 = $944 << 11; //@line 2091
        }
        $960 = $961 + -4 | 0; //@line 2093
        FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($960 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($960, $$0608); //@line 2099
        $967 = $961 + -8 | 0; //@line 2100
        FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($967 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($967, $$0604); //@line 2106
        $973 = $961 + -12 | 0; //@line 2107
        FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($973 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($973, $$0603); //@line 2113
        $2122 = $973; //@line 2114
       }
       if (($$0372642 | 0) > 0) {
        $$0372642 = $$0372642 + -1 | 0; //@line 2120
        $$1383641 = $$1383641 << 1; //@line 2120
        $961 = $2122; //@line 2120
       } else {
        $storemerge = $2122; //@line 2122
        break;
       }
      }
     } else if ((label | 0) == 176) {
      $$1644 = 0; //@line 2130
      $$2384643 = $$0382$in & 255; //@line 2130
      $1004 = HEAP32[$2 >> 2] | 0; //@line 2130
      while (1) {
       if (!($$2384643 & 128)) {
        $2123 = $1004; //@line 2135
       } else {
        $985 = +HEAPF64[112672 + ($$1644 << 3) >> 3]; //@line 2138
        HEAPF64[tempDoublePtr >> 3] = $985; //@line 2140
        $987 = HEAP32[tempDoublePtr >> 2] | 0; //@line 2140
        $988 = HEAP32[tempDoublePtr + 4 >> 2] | 0; //@line 2141
        if ($985 == 0.0) {
         $$0605 = 0; //@line 2143
         $$0606 = 0; //@line 2143
         $$0607 = 0; //@line 2143
        } else {
         $989 = _bitshift64Lshr($987 | 0, $988 | 0, 52) | 0; //@line 2145
         $991 = $989 & 2047; //@line 2147
         $996 = _bitshift64Lshr($987 | 0, $988 | 0, 21) | 0; //@line 2154
         $$0605 = $987 << 11; //@line 2161
         $$0606 = $987 >>> 21 | $996 & 2147481600 | -2147483648; //@line 2161
         $$0607 = (($991 | 0) == 2047 ? 2147418112 : ($991 << 16) + 1006632960 | 0) | $988 & -2147483648; //@line 2161
        }
        FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($1004 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($1004, $$0607); //@line 2168
        $1009 = $1004 + 4 | 0; //@line 2169
        FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($1009 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($1009, $$0606); //@line 2175
        $1015 = $1004 + 8 | 0; //@line 2176
        FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($1015 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($1015, $$0605); //@line 2182
        $2123 = $1004 + 12 | 0; //@line 2184
       }
       $$1644 = $$1644 + 1 | 0; //@line 2187
       if (($$1644 | 0) == 8) {
        $storemerge = $2123; //@line 2190
        break;
       } else {
        $$2384643 = $$2384643 << 1; //@line 2193
        $1004 = $2123; //@line 2193
       }
      }
     }
     HEAP32[$2 >> 2] = $storemerge; //@line 2197
     switch ($0 & 56) {
     case 24:
      {
       HEAP32[651040 + (($0 & 7) << 2) >> 2] = HEAP32[$2 >> 2]; //@line 2206
       break L255;
       break;
      }
     case 32:
      {
       HEAP32[651040 + (($0 & 7) << 2) >> 2] = HEAP32[$2 >> 2]; //@line 2214
       break L255;
       break;
      }
     default:
      {
       break L255;
      }
     }
    }
   } while (0);
   STACKTOP = sp; //@line 2224
   return;
  }
 case 2:
 case 0:
  {
   $1201 = $1 >>> 7 & 7; //@line 2229
   if (($1 & 64512 | 0) == 23552) {
    do {
     switch ($1 & 127) {
     case 0:
      {
       HEAPF64[112672 + ($1201 << 3) >> 3] = 3.141592653589793; //@line 2239
       $$val454 = 3.141592653589793; //@line 2240
       break;
      }
     case 11:
      {
       HEAPF64[112672 + ($1201 << 3) >> 3] = .3010299956639812; //@line 2245
       $$val454 = .3010299956639812; //@line 2246
       break;
      }
     case 12:
      {
       HEAPF64[112672 + ($1201 << 3) >> 3] = 2.718281828459045; //@line 2251
       $$val454 = 2.718281828459045; //@line 2252
       break;
      }
     case 13:
      {
       HEAPF64[112672 + ($1201 << 3) >> 3] = 1.4426950408889634; //@line 2257
       $$val454 = 1.4426950408889634; //@line 2258
       break;
      }
     case 14:
      {
       HEAPF64[112672 + ($1201 << 3) >> 3] = .4342944819032518; //@line 2263
       $$val454 = .4342944819032518; //@line 2264
       break;
      }
     case 15:
      {
       HEAPF64[112672 + ($1201 << 3) >> 3] = 0.0; //@line 2269
       $$val454 = 0.0; //@line 2270
       break;
      }
     case 48:
      {
       HEAPF64[112672 + ($1201 << 3) >> 3] = .6931471805599453; //@line 2275
       $$val454 = .6931471805599453; //@line 2276
       break;
      }
     case 49:
      {
       HEAPF64[112672 + ($1201 << 3) >> 3] = 2.302585092994046; //@line 2281
       $$val454 = 2.302585092994046; //@line 2282
       break;
      }
     case 50:
      {
       HEAPF64[112672 + ($1201 << 3) >> 3] = 1.0; //@line 2287
       $$val454 = 1.0; //@line 2288
       break;
      }
     case 51:
      {
       HEAPF64[112672 + ($1201 << 3) >> 3] = 10.0; //@line 2293
       $$val454 = 10.0; //@line 2294
       break;
      }
     case 52:
      {
       HEAPF64[112672 + ($1201 << 3) >> 3] = 100.0; //@line 2299
       $$val454 = 100.0; //@line 2300
       break;
      }
     case 53:
      {
       HEAPF64[112672 + ($1201 << 3) >> 3] = 1.0e4; //@line 2305
       $$val454 = 1.0e4; //@line 2306
       break;
      }
     case 54:
      {
       HEAPF64[112672 + ($1201 << 3) >> 3] = 1.0e8; //@line 2311
       $$val454 = 1.0e8; //@line 2312
       break;
      }
     case 55:
      {
       HEAPF64[112672 + ($1201 << 3) >> 3] = 1.0e16; //@line 2317
       $$val454 = 1.0e16; //@line 2318
       break;
      }
     case 56:
      {
       HEAPF64[112672 + ($1201 << 3) >> 3] = 1.0e+32; //@line 2323
       $$val454 = 1.0e+32; //@line 2324
       break;
      }
     case 57:
      {
       HEAPF64[112672 + ($1201 << 3) >> 3] = 1.0e+64; //@line 2329
       $$val454 = 1.0e+64; //@line 2330
       break;
      }
     case 58:
      {
       HEAPF64[112672 + ($1201 << 3) >> 3] = 1.0e+128; //@line 2335
       $$val454 = 1.0e+128; //@line 2336
       break;
      }
     case 59:
      {
       HEAPF64[112672 + ($1201 << 3) >> 3] = 1.0e+256; //@line 2341
       $$val454 = 1.0e+256; //@line 2342
       break;
      }
     default:
      {
       $1227 = (HEAP32[162768] | 0) + -4 + (HEAP32[162769] | 0) - (HEAP32[162770] | 0) | 0; //@line 2351
       $1233 = FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($1227 >>> 16 << 2) >> 2] | 0) + 24 >> 2] & 127]($1227) | 0; //@line 2357
       HEAP32[162770] = $1233; //@line 2358
       HEAP32[162769] = $1233; //@line 2359
       HEAP32[162768] = $1227; //@line 2360
       __Z7op_illgj($0); //@line 2361
       $$val454 = +HEAPF64[112672 + ($1201 << 3) >> 3]; //@line 2364
      }
     }
    } while (0);
    HEAPF64[14092] = $$val454; //@line 2368
    STACKTOP = sp; //@line 2369
    return;
   }
   $1236 = $1 >>> 10; //@line 2373
   $1237 = $1236 & 7; //@line 2374
   L352 : do {
    if (!($1 & 16384)) {
     $$sink104$i$sink = +HEAPF64[112672 + ($1237 << 3) >> 3]; //@line 2379
    } else {
     $1241 = $0 & 7; //@line 2382
     L355 : do {
      switch ($0 >>> 3 & 7) {
      case 0:
       {
        switch ($1236 & 7) {
        case 6:
         {
          $$sink104$i$sink = +((HEAP32[651008 + ($1241 << 2) >> 2] & 255) << 24 >> 24); //@line 2396
          break L352;
          break;
         }
        case 4:
         {
          $$sink104$i$sink = +((HEAP32[651008 + ($1241 << 2) >> 2] & 65535) << 16 >> 16); //@line 2405
          break L352;
          break;
         }
        case 0:
         {
          $$sink104$i$sink = +(HEAP32[651008 + ($1241 << 2) >> 2] | 0); //@line 2413
          break L352;
          break;
         }
        case 1:
         {
          $$sink104$i$sink = +HEAPF32[651008 + ($1241 << 2) >> 2]; //@line 2421
          break L352;
          break;
         }
        default:
         {
          break L355;
         }
        }
        break;
       }
      case 1:
       {
        break;
       }
      case 2:
       {
        $$098$i = HEAP32[651040 + ($1241 << 2) >> 2] | 0; //@line 2437
        label = 258; //@line 2438
        break;
       }
      case 3:
       {
        $1258 = 651040 + ($1241 << 2) | 0; //@line 2442
        $1259 = HEAP32[$1258 >> 2] | 0; //@line 2443
        HEAP32[$1258 >> 2] = (HEAP32[(($1241 | 0) == 7 ? 6704 : 6736) + ($1237 << 2) >> 2] | 0) + $1259; //@line 2449
        $$098$i = $1259; //@line 2450
        label = 258; //@line 2451
        break;
       }
      case 4:
       {
        $1267 = 651040 + ($1241 << 2) | 0; //@line 2459
        $1269 = (HEAP32[$1267 >> 2] | 0) - (HEAP32[(($1241 | 0) == 7 ? 6704 : 6736) + ($1237 << 2) >> 2] | 0) | 0; //@line 2461
        HEAP32[$1267 >> 2] = $1269; //@line 2462
        $$098$i = $1269; //@line 2463
        label = 258; //@line 2464
        break;
       }
      case 5:
       {
        $1271 = HEAP32[651040 + ($1241 << 2) >> 2] | 0; //@line 2469
        $1272 = HEAP32[162769] | 0; //@line 2470
        $1279 = HEAPU8[$1272 >> 0] << 8 | HEAPU8[$1272 + 1 >> 0]; //@line 2477
        HEAP32[162769] = $1272 + 2; //@line 2479
        $$098$i = ($1279 << 16 >> 16) + $1271 | 0; //@line 2483
        label = 258; //@line 2484
        break;
       }
      case 6:
       {
        $1284 = HEAP32[651040 + ($1241 << 2) >> 2] | 0; //@line 2489
        $1285 = HEAP32[162769] | 0; //@line 2490
        $1292 = HEAPU8[$1285 >> 0] << 8 | HEAPU8[$1285 + 1 >> 0]; //@line 2497
        HEAP32[162769] = $1285 + 2; //@line 2499
        $$098$i = __Z15get_disp_ea_020jj($1284, $1292) | 0; //@line 2501
        label = 258; //@line 2502
        break;
       }
      case 7:
       {
        switch ($0 & 7) {
        case 0:
         {
          $1295 = HEAP32[162769] | 0; //@line 2510
          $1302 = HEAPU8[$1295 >> 0] << 8 | HEAPU8[$1295 + 1 >> 0]; //@line 2517
          HEAP32[162769] = $1295 + 2; //@line 2519
          $$098$i = $1302 << 16 >> 16; //@line 2522
          label = 258; //@line 2523
          break L355;
          break;
         }
        case 1:
         {
          $1305 = HEAP32[162769] | 0; //@line 2528
          $1322 = HEAPU8[$1305 + 1 >> 0] << 16 | HEAPU8[$1305 >> 0] << 24 | HEAPU8[$1305 + 2 >> 0] << 8 | HEAPU8[$1305 + 3 >> 0]; //@line 2545
          HEAP32[162769] = $1305 + 4; //@line 2547
          $$098$i = $1322; //@line 2548
          label = 258; //@line 2549
          break L355;
          break;
         }
        case 2:
         {
          $1325 = HEAP32[162769] | 0; //@line 2555
          $1328 = $1325 + (HEAP32[162768] | 0) - (HEAP32[162770] | 0) | 0; //@line 2558
          $$cast107$i = $1325; //@line 2559
          $1335 = HEAPU8[$$cast107$i >> 0] << 8 | HEAPU8[$$cast107$i + 1 >> 0]; //@line 2566
          HEAP32[162769] = $$cast107$i + 2; //@line 2568
          $$098$i = $1328 + ($1335 << 16 >> 16) | 0; //@line 2572
          label = 258; //@line 2573
          break L355;
          break;
         }
        case 3:
         {
          $1340 = HEAP32[162769] | 0; //@line 2579
          $1343 = $1340 + (HEAP32[162768] | 0) - (HEAP32[162770] | 0) | 0; //@line 2582
          $$cast$i509 = $1340; //@line 2583
          $1350 = HEAPU8[$$cast$i509 >> 0] << 8 | HEAPU8[$$cast$i509 + 1 >> 0]; //@line 2590
          HEAP32[162769] = $$cast$i509 + 2; //@line 2592
          $$098$i = __Z15get_disp_ea_020jj($1343, $1350) | 0; //@line 2594
          label = 258; //@line 2595
          break L355;
          break;
         }
        case 4:
         {
          $1357 = (HEAP32[162769] | 0) + (HEAP32[162768] | 0) - (HEAP32[162770] | 0) | 0; //@line 2604
          $1360 = $1357 + (HEAP32[6704 + ($1237 << 2) >> 2] | 0) | 0; //@line 2607
          $1366 = FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($1360 >>> 16 << 2) >> 2] | 0) + 24 >> 2] & 127]($1360) | 0; //@line 2613
          HEAP32[162770] = $1366; //@line 2614
          HEAP32[162769] = $1366; //@line 2615
          HEAP32[162768] = $1360; //@line 2616
          $$098$i = $1357 + (($1237 | 0) == 6 & 1) | 0; //@line 2620
          label = 258; //@line 2621
          break L355;
          break;
         }
        default:
         {
          break L355;
         }
        }
        break;
       }
      default:
       {}
      }
     } while (0);
     L375 : do {
      if ((label | 0) == 258) {
       $1370 = 123608 + ($$098$i >>> 16 << 2) | 0; //@line 2639
       FUNCTION_TABLE_ii[HEAP32[(HEAP32[$1370 >> 2] | 0) + 24 >> 2] & 127]($$098$i) | 0; //@line 2643
       FUNCTION_TABLE_ii[HEAP32[(HEAP32[$1370 >> 2] | 0) + 24 >> 2] & 127]($$098$i) | 0; //@line 2647
       switch ($1236 & 7) {
       case 0:
        {
         $$sink104$i$sink = +(FUNCTION_TABLE_ii[HEAP32[HEAP32[$1370 >> 2] >> 2] & 127]($$098$i) | 0); //@line 2656
         break L352;
         break;
        }
       case 1:
        {
         $1383 = FUNCTION_TABLE_ii[HEAP32[HEAP32[$1370 >> 2] >> 2] & 127]($$098$i) | 0; //@line 2663
         $$sink104$i$sink = (HEAP32[tempDoublePtr >> 2] = $1383, +HEAPF32[tempDoublePtr >> 2]); //@line 2666
         break L352;
         break;
        }
       case 2:
        {
         $1388 = FUNCTION_TABLE_ii[HEAP32[HEAP32[$1370 >> 2] >> 2] & 127]($$098$i) | 0; //@line 2673
         $1389 = $$098$i + 4 | 0; //@line 2674
         $1394 = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($1389 >>> 16 << 2) >> 2] >> 2] & 127]($1389) | 0; //@line 2679
         $1395 = $$098$i + 8 | 0; //@line 2680
         $1400 = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($1395 >>> 16 << 2) >> 2] >> 2] & 127]($1395) | 0; //@line 2685
         if (!($1394 | $1388 & 2147418112 | $1400)) {
          $$sink104$i$sink = 0.0; //@line 2691
          break L352;
         }
         $1405 = $1388 >>> 31; //@line 2694
         $1407 = $1388 >>> 16 & 32767; //@line 2696
         do {
          if (($1394 | 0) > -1) {
           if (!($1400 | $1394)) {
            if (($1407 | 0) == 32767) {
             $$04051$i$i = $1394; //@line 2705
             $$04149$i$i = $1400; //@line 2705
             $1438 = 0; //@line 2705
             $1440 = 2146435072; //@line 2705
             break;
            } else {
             $$0405058$i$i = $1394; //@line 2708
             $$0414859$i$i = $1400; //@line 2708
             $2126 = 0; //@line 2708
             $2127 = 0; //@line 2708
             label = 270; //@line 2709
             break;
            }
           }
           if (($1407 | 0) != 0 & (($1394 | 0) > -1 | ($1394 | 0) == -1 & $1400 >>> 0 > 4294967295)) {
            $$03963$i$i = $1407; //@line 2721
            $1416 = $1400; //@line 2721
            $1417 = $1394; //@line 2721
            while (1) {
             $1418 = _bitshift64Shl($1416 | 0, $1417 | 0, 1) | 0; //@line 2723
             $1419 = tempRet0; //@line 2724
             $$03963$i$i = $$03963$i$i + -1 | 0; //@line 2725
             if (!((($1419 | 0) > -1 | ($1419 | 0) == -1 & $1418 >>> 0 > 4294967295) & ($$03963$i$i | 0) != 0)) {
              break;
             } else {
              $1416 = $1418; //@line 2734
              $1417 = $1419; //@line 2734
             }
            }
            $1426 = _bitshift64Lshr($1416 | 0, $1417 | 0, 31) | 0; //@line 2739
            $$040$i$i = $1426; //@line 2741
            $$041$i$i = $1418; //@line 2741
            $$1$i105$i = $$03963$i$i; //@line 2741
            label = 268; //@line 2742
           } else {
            $$040$i$i = $1394; //@line 2744
            $$041$i$i = $1400; //@line 2744
            $$1$i105$i = $1407; //@line 2744
            label = 268; //@line 2745
           }
          } else {
           $$040$i$i = $1394; //@line 2748
           $$041$i$i = $1400; //@line 2748
           $$1$i105$i = $1407; //@line 2748
           label = 268; //@line 2749
          }
         } while (0);
         if ((label | 0) == 268) {
          if ($$1$i105$i >>> 0 < 15360) {
           $$04051$i$i = $$040$i$i; //@line 2755
           $$04149$i$i = $$041$i$i; //@line 2755
           $1438 = 0; //@line 2755
           $1440 = 0; //@line 2755
          } else {
           $1433 = _bitshift64Shl($$1$i105$i + 1024 & 2047 | 0, 0, 52) | 0; //@line 2760
           if ($$1$i105$i >>> 0 > 17406) {
            $$04051$i$i = $$040$i$i; //@line 2763
            $$04149$i$i = $$041$i$i; //@line 2763
            $1438 = 0; //@line 2763
            $1440 = 2146435072; //@line 2763
           } else {
            $$0405058$i$i = $$040$i$i; //@line 2765
            $$0414859$i$i = $$041$i$i; //@line 2765
            $2126 = $1433; //@line 2765
            $2127 = tempRet0; //@line 2765
            label = 270; //@line 2766
           }
          }
         }
         if ((label | 0) == 270) {
          $$04051$i$i = $$0405058$i$i; //@line 2771
          $$04149$i$i = $$0414859$i$i; //@line 2771
          $1438 = $2126; //@line 2771
          $1440 = $2127; //@line 2771
         }
         $1435 = _bitshift64Shl($1405 | 0, 0, 63) | 0; //@line 2773
         $1443 = $1440 | tempRet0 | $$04051$i$i >>> 11 & 1048575; //@line 2779
         HEAP32[tempDoublePtr >> 2] = $1438 | $1435 | ($$04149$i$i >>> 11 | $$04051$i$i << 21); //@line 2784
         HEAP32[tempDoublePtr + 4 >> 2] = $1443; //@line 2784
         $$sink104$i$sink = +HEAPF64[tempDoublePtr >> 3]; //@line 2785
         break L352;
         break;
        }
       case 3:
        {
         $1451 = FUNCTION_TABLE_ii[HEAP32[HEAP32[$1370 >> 2] >> 2] & 127]($$098$i) | 0; //@line 2792
         $1452 = $$098$i + 4 | 0; //@line 2793
         $1457 = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($1452 >>> 16 << 2) >> 2] >> 2] & 127]($1452) | 0; //@line 2798
         $1458 = $$098$i + 8 | 0; //@line 2799
         $1463 = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($1458 >>> 16 << 2) >> 2] >> 2] & 127]($1458) | 0; //@line 2804
         if (($1451 | 0) < 0) {
          HEAP8[$3 >> 0] = 45; //@line 2808
          $$0$i$i514 = $3 + 1 | 0; //@line 2809
         } else {
          $$0$i$i514 = $3; //@line 2811
         }
         HEAP8[$$0$i$i514 >> 0] = $1451 & 15 | 48; //@line 2817
         HEAP8[$$0$i$i514 + 1 >> 0] = 46; //@line 2819
         HEAP8[$$0$i$i514 + 2 >> 0] = $1457 >>> 28 | 48; //@line 2824
         HEAP8[$$0$i$i514 + 3 >> 0] = $1457 >>> 24 & 15 | 48; //@line 2830
         HEAP8[$$0$i$i514 + 4 >> 0] = $1457 >>> 20 & 15 | 48; //@line 2836
         HEAP8[$$0$i$i514 + 5 >> 0] = $1457 >>> 16 & 15 | 48; //@line 2842
         HEAP8[$$0$i$i514 + 6 >> 0] = $1457 >>> 12 & 15 | 48; //@line 2848
         HEAP8[$$0$i$i514 + 7 >> 0] = $1457 >>> 8 & 15 | 48; //@line 2854
         HEAP8[$$0$i$i514 + 8 >> 0] = $1457 >>> 4 & 15 | 48; //@line 2860
         HEAP8[$$0$i$i514 + 9 >> 0] = $1457 & 15 | 48; //@line 2865
         HEAP8[$$0$i$i514 + 10 >> 0] = $1463 >>> 28 | 48; //@line 2870
         HEAP8[$$0$i$i514 + 11 >> 0] = $1463 >>> 24 & 15 | 48; //@line 2876
         HEAP8[$$0$i$i514 + 12 >> 0] = $1463 >>> 20 & 15 | 48; //@line 2882
         HEAP8[$$0$i$i514 + 13 >> 0] = $1463 >>> 16 & 15 | 48; //@line 2888
         HEAP8[$$0$i$i514 + 14 >> 0] = $1463 >>> 12 & 15 | 48; //@line 2894
         HEAP8[$$0$i$i514 + 15 >> 0] = $1463 >>> 8 & 15 | 48; //@line 2900
         HEAP8[$$0$i$i514 + 16 >> 0] = $1463 >>> 4 & 15 | 48; //@line 2906
         HEAP8[$$0$i$i514 + 17 >> 0] = $1463 & 15 | 48; //@line 2911
         $1547 = $$0$i$i514 + 19 | 0; //@line 2912
         HEAP8[$$0$i$i514 + 18 >> 0] = 69; //@line 2913
         if (!($1451 & 1073741824)) {
          $$1$i$i = $1547; //@line 2917
         } else {
          HEAP8[$1547 >> 0] = 45; //@line 2920
          $$1$i$i = $$0$i$i514 + 20 | 0; //@line 2921
         }
         HEAP8[$$1$i$i >> 0] = $1451 >>> 24 & 15 | 48; //@line 2928
         HEAP8[$$1$i$i + 1 >> 0] = $1451 >>> 20 & 15 | 48; //@line 2934
         HEAP8[$$1$i$i + 2 >> 0] = $1451 >>> 16 & 15 | 48; //@line 2940
         HEAP8[$$1$i$i + 3 >> 0] = 0; //@line 2941
         HEAP32[$vararg_buffer1 >> 2] = $2; //@line 2942
         _sscanf($3, 106246, $vararg_buffer1) | 0; //@line 2943
         $$sink104$i$sink = +HEAPF64[$2 >> 3]; //@line 2945
         break L352;
         break;
        }
       case 4:
        {
         $$sink104$i$sink = +(((FUNCTION_TABLE_ii[HEAP32[(HEAP32[$1370 >> 2] | 0) + 4 >> 2] & 127]($$098$i) | 0) & 65535) << 16 >> 16); //@line 2956
         break L352;
         break;
        }
       case 5:
        {
         $1575 = FUNCTION_TABLE_ii[HEAP32[HEAP32[$1370 >> 2] >> 2] & 127]($$098$i) | 0; //@line 2963
         $1576 = $$098$i + 4 | 0; //@line 2964
         $1581 = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($1576 >>> 16 << 2) >> 2] >> 2] & 127]($1576) | 0; //@line 2969
         HEAP32[tempDoublePtr >> 2] = $1581; //@line 2970
         HEAP32[tempDoublePtr + 4 >> 2] = $1575; //@line 2970
         $$sink104$i$sink = +HEAPF64[tempDoublePtr >> 3]; //@line 2971
         break L352;
         break;
        }
       case 6:
        {
         $$sink104$i$sink = +(((FUNCTION_TABLE_ii[HEAP32[(HEAP32[$1370 >> 2] | 0) + 8 >> 2] & 127]($$098$i) | 0) & 255) << 24 >> 24); //@line 2982
         break L352;
         break;
        }
       default:
        {
         break L375;
        }
       }
      }
     } while (0);
     $1594 = (HEAP32[162768] | 0) + -4 + (HEAP32[162769] | 0) - (HEAP32[162770] | 0) | 0; //@line 2997
     $1600 = FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($1594 >>> 16 << 2) >> 2] | 0) + 24 >> 2] & 127]($1594) | 0; //@line 3003
     HEAP32[162770] = $1600; //@line 3004
     HEAP32[162769] = $1600; //@line 3005
     HEAP32[162768] = $1594; //@line 3006
     __Z7op_illgj($0); //@line 3007
     STACKTOP = sp; //@line 3008
     return;
    }
   } while (0);
   HEAPF64[tempDoublePtr >> 3] = $$sink104$i$sink; //@line 3011
   $1601 = HEAP32[tempDoublePtr >> 2] | 0; //@line 3011
   $1602 = HEAP32[tempDoublePtr + 4 >> 2] | 0; //@line 3012
   $1605 = $1 & 127; //@line 3015
   $$pre = $1 & 255; //@line 3016
   L409 : do {
    if (!(HEAP8[112776] | 0)) {
     switch ($$pre & 127) {
     case 0:
      {
       label = 310; //@line 3022
       break;
      }
     case 1:
      {
       label = 311; //@line 3026
       break;
      }
     case 2:
      {
       label = 318; //@line 3030
       break;
      }
     case 3:
      {
       label = 319; //@line 3034
       break;
      }
     case 4:
      {
       label = 320; //@line 3038
       break;
      }
     case 6:
      {
       label = 321; //@line 3042
       break;
      }
     case 8:
      {
       label = 322; //@line 3046
       break;
      }
     case 9:
      {
       label = 323; //@line 3050
       break;
      }
     case 10:
      {
       label = 324; //@line 3054
       break;
      }
     case 12:
      {
       label = 325; //@line 3058
       break;
      }
     case 13:
      {
       label = 326; //@line 3062
       break;
      }
     case 14:
      {
       label = 327; //@line 3066
       break;
      }
     case 15:
      {
       label = 328; //@line 3070
       break;
      }
     case 16:
      {
       label = 329; //@line 3074
       break;
      }
     case 17:
      {
       label = 330; //@line 3078
       break;
      }
     case 18:
      {
       label = 331; //@line 3082
       break;
      }
     case 20:
      {
       label = 332; //@line 3086
       break;
      }
     case 21:
      {
       label = 333; //@line 3090
       break;
      }
     case 22:
      {
       label = 334; //@line 3094
       break;
      }
     case 24:
      {
       label = 335; //@line 3098
       break;
      }
     case 25:
      {
       label = 336; //@line 3102
       break;
      }
     case 26:
      {
       label = 337; //@line 3106
       break;
      }
     case 28:
      {
       label = 338; //@line 3110
       break;
      }
     case 29:
      {
       label = 339; //@line 3114
       break;
      }
     case 30:
      {
       label = 340; //@line 3118
       break;
      }
     case 31:
      {
       label = 344; //@line 3122
       break;
      }
     case 32:
      {
       label = 350; //@line 3126
       break;
      }
     case 33:
      {
       label = 351; //@line 3130
       break;
      }
     case 35:
      {
       label = 352; //@line 3134
       break;
      }
     case 36:
      {
       label = 364; //@line 3138
       break;
      }
     case 37:
      {
       label = 365; //@line 3142
       break;
      }
     case 38:
      {
       label = 366; //@line 3146
       break;
      }
     case 39:
      {
       label = 374; //@line 3150
       break;
      }
     case 40:
      {
       label = 375; //@line 3154
       break;
      }
     case 34:
      {
       label = 376; //@line 3158
       break;
      }
     case 55:
     case 54:
     case 53:
     case 52:
     case 51:
     case 50:
     case 49:
     case 48:
      {
       label = 377; //@line 3162
       break;
      }
     case 56:
      {
       label = 378; //@line 3166
       break;
      }
     case 58:
      {
       label = 379; //@line 3170
       break;
      }
     default:
      {
       label = 380; //@line 3174
      }
     }
    } else {
     do {
      switch ($$pre & 127) {
      case 0:
       {
        label = 310; //@line 3182
        break L409;
        break;
       }
      case 1:
       {
        label = 311; //@line 3187
        break L409;
        break;
       }
      case 2:
       {
        label = 318; //@line 3192
        break L409;
        break;
       }
      case 3:
       {
        label = 319; //@line 3197
        break L409;
        break;
       }
      case 4:
       {
        label = 320; //@line 3202
        break L409;
        break;
       }
      case 6:
       {
        label = 321; //@line 3207
        break L409;
        break;
       }
      case 8:
       {
        label = 322; //@line 3212
        break L409;
        break;
       }
      case 9:
       {
        label = 323; //@line 3217
        break L409;
        break;
       }
      case 10:
       {
        label = 324; //@line 3222
        break L409;
        break;
       }
      case 12:
       {
        label = 325; //@line 3227
        break L409;
        break;
       }
      case 13:
       {
        label = 326; //@line 3232
        break L409;
        break;
       }
      case 14:
       {
        label = 327; //@line 3237
        break L409;
        break;
       }
      case 15:
       {
        label = 328; //@line 3242
        break L409;
        break;
       }
      case 16:
       {
        label = 329; //@line 3247
        break L409;
        break;
       }
      case 17:
       {
        label = 330; //@line 3252
        break L409;
        break;
       }
      case 18:
       {
        label = 331; //@line 3257
        break L409;
        break;
       }
      case 20:
       {
        label = 332; //@line 3262
        break L409;
        break;
       }
      case 21:
       {
        label = 333; //@line 3267
        break L409;
        break;
       }
      case 22:
       {
        label = 334; //@line 3272
        break L409;
        break;
       }
      case 24:
       {
        label = 335; //@line 3277
        break L409;
        break;
       }
      case 25:
       {
        label = 336; //@line 3282
        break L409;
        break;
       }
      case 26:
       {
        label = 337; //@line 3287
        break L409;
        break;
       }
      case 28:
       {
        label = 338; //@line 3292
        break L409;
        break;
       }
      case 29:
       {
        label = 339; //@line 3297
        break L409;
        break;
       }
      case 30:
       {
        label = 340; //@line 3302
        break L409;
        break;
       }
      case 31:
       {
        label = 344; //@line 3307
        break L409;
        break;
       }
      case 32:
       {
        label = 350; //@line 3312
        break L409;
        break;
       }
      case 33:
       {
        label = 351; //@line 3317
        break L409;
        break;
       }
      case 35:
       {
        label = 352; //@line 3322
        break L409;
        break;
       }
      case 36:
       {
        label = 364; //@line 3327
        break L409;
        break;
       }
      case 37:
       {
        label = 365; //@line 3332
        break L409;
        break;
       }
      case 38:
       {
        label = 366; //@line 3337
        break L409;
        break;
       }
      case 39:
       {
        label = 374; //@line 3342
        break L409;
        break;
       }
      case 40:
       {
        label = 375; //@line 3347
        break L409;
        break;
       }
      case 34:
       {
        label = 376; //@line 3352
        break L409;
        break;
       }
      case 55:
      case 54:
      case 53:
      case 52:
      case 51:
      case 50:
      case 49:
      case 48:
       {
        label = 377; //@line 3357
        break L409;
        break;
       }
      case 56:
       {
        label = 378; //@line 3362
        break L409;
        break;
       }
      case 58:
       {
        label = 379; //@line 3367
        break L409;
        break;
       }
      case 64:
       {
        $1607 = $$sink104$i$sink; //@line 3373
        HEAPF64[112672 + ($1201 << 3) >> 3] = $1607; //@line 3375
        HEAPF64[14092] = $1607; //@line 3376
        STACKTOP = sp; //@line 3377
        return;
       }
      case 68:
       {
        HEAPF64[112672 + ($1201 << 3) >> 3] = $$sink104$i$sink; //@line 3382
        HEAPF64[14092] = $$sink104$i$sink; //@line 3383
        STACKTOP = sp; //@line 3384
        return;
       }
      case 65:
       {
        $1612 = +Math_sqrt(+$$sink104$i$sink); //@line 3390
        HEAPF64[112672 + ($1201 << 3) >> 3] = $1612; //@line 3392
        HEAPF64[14092] = $1612; //@line 3393
        STACKTOP = sp; //@line 3394
        return;
       }
      case 69:
       {
        $1614 = +Math_sqrt(+$$sink104$i$sink); //@line 3398
        HEAPF64[112672 + ($1201 << 3) >> 3] = $1614; //@line 3400
        HEAPF64[14092] = $1614; //@line 3401
        STACKTOP = sp; //@line 3402
        return;
       }
      case 88:
       {
        $1618 = +Math_abs(+$$sink104$i$sink); //@line 3408
        HEAPF64[112672 + ($1201 << 3) >> 3] = $1618; //@line 3410
        HEAPF64[14092] = $1618; //@line 3411
        STACKTOP = sp; //@line 3412
        return;
       }
      case 92:
       {
        $1620 = +Math_abs(+$$sink104$i$sink); //@line 3416
        HEAPF64[112672 + ($1201 << 3) >> 3] = $1620; //@line 3418
        HEAPF64[14092] = $1620; //@line 3419
        STACKTOP = sp; //@line 3420
        return;
       }
      case 90:
       {
        $1624 = -$$sink104$i$sink; //@line 3426
        HEAPF64[112672 + ($1201 << 3) >> 3] = $1624; //@line 3428
        HEAPF64[14092] = $1624; //@line 3429
        STACKTOP = sp; //@line 3430
        return;
       }
      case 94:
       {
        $1626 = -$$sink104$i$sink; //@line 3434
        HEAPF64[112672 + ($1201 << 3) >> 3] = $1626; //@line 3436
        HEAPF64[14092] = $1626; //@line 3437
        STACKTOP = sp; //@line 3438
        return;
       }
      case 96:
       {
        $1628 = 112672 + ($1201 << 3) | 0; //@line 3442
        $1632 = +HEAPF64[$1628 >> 3] / $$sink104$i$sink; //@line 3446
        HEAPF64[$1628 >> 3] = $1632; //@line 3447
        HEAPF64[14092] = $1632; //@line 3448
        STACKTOP = sp; //@line 3449
        return;
       }
      case 100:
       {
        $1633 = 112672 + ($1201 << 3) | 0; //@line 3453
        $1635 = +HEAPF64[$1633 >> 3] / $$sink104$i$sink; //@line 3455
        HEAPF64[$1633 >> 3] = $1635; //@line 3456
        HEAPF64[14092] = $1635; //@line 3457
        STACKTOP = sp; //@line 3458
        return;
       }
      case 98:
       {
        $1636 = 112672 + ($1201 << 3) | 0; //@line 3462
        $1640 = $$sink104$i$sink + +HEAPF64[$1636 >> 3]; //@line 3466
        HEAPF64[$1636 >> 3] = $1640; //@line 3467
        HEAPF64[14092] = $1640; //@line 3468
        STACKTOP = sp; //@line 3469
        return;
       }
      case 102:
       {
        $1641 = 112672 + ($1201 << 3) | 0; //@line 3473
        $1643 = $$sink104$i$sink + +HEAPF64[$1641 >> 3]; //@line 3475
        HEAPF64[$1641 >> 3] = $1643; //@line 3476
        HEAPF64[14092] = $1643; //@line 3477
        STACKTOP = sp; //@line 3478
        return;
       }
      case 104:
       {
        $1644 = 112672 + ($1201 << 3) | 0; //@line 3482
        $1648 = +HEAPF64[$1644 >> 3] - $$sink104$i$sink; //@line 3486
        HEAPF64[$1644 >> 3] = $1648; //@line 3487
        HEAPF64[14092] = $1648; //@line 3488
        STACKTOP = sp; //@line 3489
        return;
       }
      case 108:
       {
        $1649 = 112672 + ($1201 << 3) | 0; //@line 3493
        $1651 = +HEAPF64[$1649 >> 3] - $$sink104$i$sink; //@line 3495
        HEAPF64[$1649 >> 3] = $1651; //@line 3496
        HEAPF64[14092] = $1651; //@line 3497
        STACKTOP = sp; //@line 3498
        return;
       }
      case 103:
      case 99:
       {
        $1652 = 112672 + ($1201 << 3) | 0; //@line 3502
        $1653 = $1652; //@line 3503
        $1655 = HEAP32[$1653 >> 2] | 0; //@line 3505
        $1658 = HEAP32[$1653 + 4 >> 2] | 0; //@line 3508
        $1659 = _bitshift64Lshr($1655 | 0, $1658 | 0, 63) | 0; //@line 3509
        HEAP32[162825] = $1659; //@line 3511
        $1664 = ($1655 | 0) == 0 & ($1658 & 2147483647 | 0) == 0; //@line 3515
        HEAP32[162826] = $1664 & 1; //@line 3517
        $1666 = $1658 & 2147483647; //@line 3518
        $1675 = ($1658 >> 30 & ~(($1655 | 0 - ($1666 ^ 2146435072 | $1655)) >> 31) | 0) != 0; //@line 3527
        HEAP32[162827] = $1675 & 1; //@line 3529
        $1681 = ($1666 | ($1655 | 0 - $1655) >>> 31) >>> 0 > 2146435072; //@line 3534
        HEAP32[162828] = $1681 & 1; //@line 3536
        $$$i505 = ($1681 | ($1664 | $1675)) ^ 1; //@line 3539
        HEAP32[162829] = $$$i505 & 1; //@line 3541
        $1685 = _bitshift64Lshr($1601 | 0, $1602 | 0, 63) | 0; //@line 3542
        HEAP32[162830] = $1685; //@line 3544
        $1690 = ($1601 | 0) == 0 & ($1602 & 2147483647 | 0) == 0; //@line 3548
        HEAP32[162831] = $1690 & 1; //@line 3550
        $1692 = $1602 & 2147483647; //@line 3551
        $1701 = ($1602 >> 30 & ~(($1601 | 0 - ($1692 ^ 2146435072 | $1601)) >> 31) | 0) != 0; //@line 3560
        HEAP32[162832] = $1701 & 1; //@line 3562
        $1707 = ($1692 | ($1601 | 0 - $1601) >>> 31) >>> 0 > 2146435072; //@line 3567
        HEAP32[162833] = $1707 & 1; //@line 3569
        $$$i502 = ($1707 | ($1690 | $1701)) ^ 1; //@line 3572
        HEAP32[162834] = $$$i502 & 1; //@line 3574
        do {
         if ($$$i505 & $$$i502) {
          HEAP32[tempDoublePtr >> 2] = $1655; //@line 3578
          HEAP32[tempDoublePtr + 4 >> 2] = $1658; //@line 3578
          $1713 = $$sink104$i$sink * +HEAPF64[tempDoublePtr >> 3]; //@line 3580
          $$sink = ($1605 | 0) == 99 ? $1713 : $1713; //@line 3583
          HEAPF64[$1652 >> 3] = $$sink; //@line 3584
          $$val439 = $$sink; //@line 3585
         } else {
          if ($1707 | $1681 | $1664 & $1701 | $1675 & $1690) {
           $1717 = $1658 | 2147483647; //@line 3593
           $1718 = $1652; //@line 3594
           HEAP32[$1718 >> 2] = -1; //@line 3596
           HEAP32[$1718 + 4 >> 2] = $1717; //@line 3599
           HEAP32[tempDoublePtr >> 2] = -1; //@line 3600
           HEAP32[tempDoublePtr + 4 >> 2] = $1717; //@line 3600
           $$val439 = +HEAPF64[tempDoublePtr >> 3]; //@line 3601
           break;
          }
          $1724 = ($1659 | 0) == 0; //@line 3605
          $1725 = ($1685 | 0) != 0; //@line 3606
          $or$cond398 = $1724 & $1725 | ($1724 | $1725) ^ 1; //@line 3610
          if ($1690 | $1664) {
           if ($or$cond398) {
            HEAPF64[$1652 >> 3] = -0.0; //@line 3613
            $$val439 = -0.0; //@line 3614
            break;
           } else {
            HEAPF64[$1652 >> 3] = 0.0; //@line 3617
            $$val439 = 0.0; //@line 3618
            break;
           }
          } else {
           $1726 = $1658 & 524288; //@line 3622
           if ($or$cond398) {
            $1727 = $1726 | -1048576; //@line 3624
            $1728 = $1652; //@line 3625
            HEAP32[$1728 >> 2] = 0; //@line 3627
            HEAP32[$1728 + 4 >> 2] = $1727; //@line 3630
            HEAP32[tempDoublePtr >> 2] = 0; //@line 3631
            HEAP32[tempDoublePtr + 4 >> 2] = $1727; //@line 3631
            $$val439 = +HEAPF64[tempDoublePtr >> 3]; //@line 3632
            break;
           } else {
            $1733 = $1726 | 2146435072; //@line 3635
            $1734 = $1652; //@line 3636
            HEAP32[$1734 >> 2] = 0; //@line 3638
            HEAP32[$1734 + 4 >> 2] = $1733; //@line 3641
            HEAP32[tempDoublePtr >> 2] = 0; //@line 3642
            HEAP32[tempDoublePtr + 4 >> 2] = $1733; //@line 3642
            $$val439 = +HEAPF64[tempDoublePtr >> 3]; //@line 3643
            break;
           }
          }
         }
        } while (0);
        HEAPF64[14092] = $$val439; //@line 3649
        STACKTOP = sp; //@line 3650
        return;
       }
      default:
       {
        label = 380; //@line 3654
        break L409;
       }
      }
     } while (0);
    }
   } while (0);
   switch (label | 0) {
   case 310:
    {
     HEAPF64[112672 + ($1201 << 3) >> 3] = $$sink104$i$sink; //@line 3664
     HEAPF64[14092] = $$sink104$i$sink; //@line 3665
     STACKTOP = sp; //@line 3666
     return;
    }
   case 311:
    {
     switch ((HEAP32[28188] | HEAP32[28187]) >>> 4 & 3) {
     case 1:
      {
       $$0$i492 = +(~~$$sink104$i$sink | 0); //@line 3680
       break;
      }
     case 2:
      {
       $$0$i492 = +Math_floor(+$$sink104$i$sink); //@line 3685
       break;
      }
     case 0:
      {
       $$0$i492 = +(~~($$sink104$i$sink + .5) | 0); //@line 3692
       break;
      }
     case 3:
      {
       $$0$i492 = +Math_ceil(+$$sink104$i$sink); //@line 3697
       break;
      }
     default:
      {}
     }
     $1752 = +(~~$$0$i492 | 0); //@line 3705
     HEAPF64[112672 + ($1201 << 3) >> 3] = $1752; //@line 3707
     HEAPF64[14092] = $1752; //@line 3708
     STACKTOP = sp; //@line 3709
     return;
    }
   case 318:
    {
     $1754 = +_sinh($$sink104$i$sink); //@line 3713
     HEAPF64[112672 + ($1201 << 3) >> 3] = $1754; //@line 3715
     HEAPF64[14092] = $1754; //@line 3716
     STACKTOP = sp; //@line 3717
     return;
    }
   case 319:
    {
     $1757 = +(~~$$sink104$i$sink | 0); //@line 3722
     HEAPF64[112672 + ($1201 << 3) >> 3] = $1757; //@line 3724
     HEAPF64[14092] = $1757; //@line 3725
     STACKTOP = sp; //@line 3726
     return;
    }
   case 320:
    {
     $1759 = +Math_sqrt(+$$sink104$i$sink); //@line 3730
     HEAPF64[112672 + ($1201 << 3) >> 3] = $1759; //@line 3732
     HEAPF64[14092] = $1759; //@line 3733
     STACKTOP = sp; //@line 3734
     return;
    }
   case 321:
    {
     $1762 = +Math_log(+($$sink104$i$sink + 1.0)); //@line 3739
     HEAPF64[112672 + ($1201 << 3) >> 3] = $1762; //@line 3741
     HEAPF64[14092] = $1762; //@line 3742
     STACKTOP = sp; //@line 3743
     return;
    }
   case 322:
    {
     $1765 = +Math_exp(+$$sink104$i$sink) + -1.0; //@line 3748
     HEAPF64[112672 + ($1201 << 3) >> 3] = $1765; //@line 3750
     HEAPF64[14092] = $1765; //@line 3751
     STACKTOP = sp; //@line 3752
     return;
    }
   case 323:
    {
     $1767 = +_tanh($$sink104$i$sink); //@line 3756
     HEAPF64[112672 + ($1201 << 3) >> 3] = $1767; //@line 3758
     HEAPF64[14092] = $1767; //@line 3759
     STACKTOP = sp; //@line 3760
     return;
    }
   case 324:
    {
     $1769 = +Math_atan(+$$sink104$i$sink); //@line 3764
     HEAPF64[112672 + ($1201 << 3) >> 3] = $1769; //@line 3766
     HEAPF64[14092] = $1769; //@line 3767
     STACKTOP = sp; //@line 3768
     return;
    }
   case 325:
    {
     $1771 = +Math_asin(+$$sink104$i$sink); //@line 3772
     HEAPF64[112672 + ($1201 << 3) >> 3] = $1771; //@line 3774
     HEAPF64[14092] = $1771; //@line 3775
     STACKTOP = sp; //@line 3776
     return;
    }
   case 326:
    {
     $1773 = +_atanh($$sink104$i$sink); //@line 3780
     HEAPF64[112672 + ($1201 << 3) >> 3] = $1773; //@line 3782
     HEAPF64[14092] = $1773; //@line 3783
     STACKTOP = sp; //@line 3784
     return;
    }
   case 327:
    {
     $1775 = +Math_sin(+$$sink104$i$sink); //@line 3788
     HEAPF64[112672 + ($1201 << 3) >> 3] = $1775; //@line 3790
     HEAPF64[14092] = $1775; //@line 3791
     STACKTOP = sp; //@line 3792
     return;
    }
   case 328:
    {
     $1777 = +Math_tan(+$$sink104$i$sink); //@line 3796
     HEAPF64[112672 + ($1201 << 3) >> 3] = $1777; //@line 3798
     HEAPF64[14092] = $1777; //@line 3799
     STACKTOP = sp; //@line 3800
     return;
    }
   case 329:
    {
     $1779 = +Math_exp(+$$sink104$i$sink); //@line 3804
     HEAPF64[112672 + ($1201 << 3) >> 3] = $1779; //@line 3806
     HEAPF64[14092] = $1779; //@line 3807
     STACKTOP = sp; //@line 3808
     return;
    }
   case 330:
    {
     $exp2 = +_llvm_exp2_f64(+$$sink104$i$sink); //@line 3812
     HEAPF64[112672 + ($1201 << 3) >> 3] = $exp2; //@line 3814
     HEAPF64[14092] = $exp2; //@line 3815
     STACKTOP = sp; //@line 3816
     return;
    }
   case 331:
    {
     $1782 = +Math_pow(10.0, +$$sink104$i$sink); //@line 3820
     HEAPF64[112672 + ($1201 << 3) >> 3] = $1782; //@line 3822
     HEAPF64[14092] = $1782; //@line 3823
     STACKTOP = sp; //@line 3824
     return;
    }
   case 332:
    {
     $1784 = +Math_log(+$$sink104$i$sink); //@line 3828
     HEAPF64[112672 + ($1201 << 3) >> 3] = $1784; //@line 3830
     HEAPF64[14092] = $1784; //@line 3831
     STACKTOP = sp; //@line 3832
     return;
    }
   case 333:
    {
     $1786 = +_log10($$sink104$i$sink); //@line 3836
     HEAPF64[112672 + ($1201 << 3) >> 3] = $1786; //@line 3838
     HEAPF64[14092] = $1786; //@line 3839
     STACKTOP = sp; //@line 3840
     return;
    }
   case 334:
    {
     $1789 = +Math_log(+$$sink104$i$sink) / .6931471805599453; //@line 3845
     HEAPF64[112672 + ($1201 << 3) >> 3] = $1789; //@line 3847
     HEAPF64[14092] = $1789; //@line 3848
     STACKTOP = sp; //@line 3849
     return;
    }
   case 335:
    {
     $1791 = +Math_abs(+$$sink104$i$sink); //@line 3853
     HEAPF64[112672 + ($1201 << 3) >> 3] = $1791; //@line 3855
     HEAPF64[14092] = $1791; //@line 3856
     STACKTOP = sp; //@line 3857
     return;
    }
   case 336:
    {
     $1793 = +_cosh($$sink104$i$sink); //@line 3861
     HEAPF64[112672 + ($1201 << 3) >> 3] = $1793; //@line 3863
     HEAPF64[14092] = $1793; //@line 3864
     STACKTOP = sp; //@line 3865
     return;
    }
   case 337:
    {
     $1795 = -$$sink104$i$sink; //@line 3869
     HEAPF64[112672 + ($1201 << 3) >> 3] = $1795; //@line 3871
     HEAPF64[14092] = $1795; //@line 3872
     STACKTOP = sp; //@line 3873
     return;
    }
   case 338:
    {
     $1797 = +Math_acos(+$$sink104$i$sink); //@line 3877
     HEAPF64[112672 + ($1201 << 3) >> 3] = $1797; //@line 3879
     HEAPF64[14092] = $1797; //@line 3880
     STACKTOP = sp; //@line 3881
     return;
    }
   case 339:
    {
     $1799 = +Math_cos(+$$sink104$i$sink); //@line 3885
     HEAPF64[112672 + ($1201 << 3) >> 3] = $1799; //@line 3887
     HEAPF64[14092] = $1799; //@line 3888
     STACKTOP = sp; //@line 3889
     return;
    }
   case 340:
    {
     if (!($1602 >> 30 & ~(($1601 | 0 - ($1602 & 2147483647 ^ 2146435072 | $1601)) >> 31))) {
      $1824 = _bitshift64Lshr($1601 | 0, $1602 | 0, 52) | 0; //@line 3904
      $1828 = +(($1824 & 2047) + -1023 | 0); //@line 3908
      HEAPF64[112672 + ($1201 << 3) >> 3] = $1828; //@line 3910
      $$val414 = $1828; //@line 3911
     } else {
      $1811 = 112672 + ($1201 << 3) | 0; //@line 3913
      $1818 = HEAP32[$1811 + 4 >> 2] | 2147483647; //@line 3920
      $1819 = $1811; //@line 3921
      HEAP32[$1819 >> 2] = -1; //@line 3923
      HEAP32[$1819 + 4 >> 2] = $1818; //@line 3926
      HEAP32[tempDoublePtr >> 2] = -1; //@line 3927
      HEAP32[tempDoublePtr + 4 >> 2] = $1818; //@line 3927
      $$val414 = +HEAPF64[tempDoublePtr >> 3]; //@line 3928
     }
     HEAPF64[14092] = $$val414; //@line 3930
     STACKTOP = sp; //@line 3931
     return;
    }
   case 344:
    {
     do {
      if ($$sink104$i$sink == 0.0) {
       HEAPF64[112672 + ($1201 << 3) >> 3] = 0.0; //@line 3939
       $$val413 = 0.0; //@line 3940
      } else {
       $1842 = 112672 + ($1201 << 3) | 0; //@line 3952
       if (!($1602 >> 30 & ~(($1601 | 0 - ($1602 & 2147483647 ^ 2146435072 | $1601)) >> 31))) {
        $1856 = $1602 & -2146435073 | 1072693248; //@line 3955
        $1857 = $1842; //@line 3956
        HEAP32[$1857 >> 2] = $1601; //@line 3958
        HEAP32[$1857 + 4 >> 2] = $1856; //@line 3961
        HEAP32[tempDoublePtr >> 2] = $1601; //@line 3962
        HEAP32[tempDoublePtr + 4 >> 2] = $1856; //@line 3962
        $$val413 = +HEAPF64[tempDoublePtr >> 3]; //@line 3963
        break;
       } else {
        $1849 = HEAP32[$1842 + 4 >> 2] | 2147483647; //@line 3972
        $1850 = $1842; //@line 3973
        HEAP32[$1850 >> 2] = -1; //@line 3975
        HEAP32[$1850 + 4 >> 2] = $1849; //@line 3978
        HEAP32[tempDoublePtr >> 2] = -1; //@line 3979
        HEAP32[tempDoublePtr + 4 >> 2] = $1849; //@line 3979
        $$val413 = +HEAPF64[tempDoublePtr >> 3]; //@line 3980
        break;
       }
      }
     } while (0);
     HEAPF64[14092] = $$val413; //@line 3985
     STACKTOP = sp; //@line 3986
     return;
    }
   case 350:
    {
     $1862 = 112672 + ($1201 << 3) | 0; //@line 3990
     $1864 = +HEAPF64[$1862 >> 3] / $$sink104$i$sink; //@line 3992
     HEAPF64[$1862 >> 3] = $1864; //@line 3993
     HEAPF64[14092] = $1864; //@line 3994
     STACKTOP = sp; //@line 3995
     return;
    }
   case 351:
    {
     $1865 = 112672 + ($1201 << 3) | 0; //@line 3999
     $1866 = +HEAPF64[$1865 >> 3]; //@line 4000
     $1869 = +(~~($1866 / $$sink104$i$sink) | 0); //@line 4003
     HEAPF64[tempDoublePtr >> 3] = $1866; //@line 4004
     HEAP32[tempDoublePtr >> 2] ^ $1601; //@line 4006
     $1874 = (HEAP32[tempDoublePtr + 4 >> 2] ^ $1602 | 0) < 0 ? 8388608 : 0; //@line 4009
     $1876 = $1866 - $$sink104$i$sink * $1869; //@line 4011
     HEAPF64[$1865 >> 3] = $1876; //@line 4012
     HEAPF64[14092] = $1876; //@line 4013
     HEAP32[28190] = ~~+Math_abs(+$1869) >>> 0 << 16 & 8323072 | $1874; //@line 4019
     STACKTOP = sp; //@line 4020
     return;
    }
   case 352:
    {
     $1882 = 112672 + ($1201 << 3) | 0; //@line 4024
     $1883 = $1882; //@line 4025
     $1885 = HEAP32[$1883 >> 2] | 0; //@line 4027
     $1888 = HEAP32[$1883 + 4 >> 2] | 0; //@line 4030
     $1889 = _bitshift64Lshr($1885 | 0, $1888 | 0, 63) | 0; //@line 4031
     HEAP32[162825] = $1889; //@line 4033
     $1894 = ($1885 | 0) == 0 & ($1888 & 2147483647 | 0) == 0; //@line 4037
     HEAP32[162826] = $1894 & 1; //@line 4039
     $1896 = $1888 & 2147483647; //@line 4040
     $1905 = ($1888 >> 30 & ~(($1885 | 0 - ($1896 ^ 2146435072 | $1885)) >> 31) | 0) != 0; //@line 4049
     HEAP32[162827] = $1905 & 1; //@line 4051
     $1911 = ($1896 | ($1885 | 0 - $1885) >>> 31) >>> 0 > 2146435072; //@line 4056
     HEAP32[162828] = $1911 & 1; //@line 4058
     $$$i488 = ($1911 | ($1894 | $1905)) ^ 1; //@line 4061
     HEAP32[162829] = $$$i488 & 1; //@line 4063
     $1915 = _bitshift64Lshr($1601 | 0, $1602 | 0, 63) | 0; //@line 4064
     HEAP32[162830] = $1915; //@line 4066
     $1920 = ($1601 | 0) == 0 & ($1602 & 2147483647 | 0) == 0; //@line 4070
     HEAP32[162831] = $1920 & 1; //@line 4072
     $1922 = $1602 & 2147483647; //@line 4073
     $1931 = ($1602 >> 30 & ~(($1601 | 0 - ($1922 ^ 2146435072 | $1601)) >> 31) | 0) != 0; //@line 4082
     HEAP32[162832] = $1931 & 1; //@line 4084
     $1937 = ($1922 | ($1601 | 0 - $1601) >>> 31) >>> 0 > 2146435072; //@line 4089
     HEAP32[162833] = $1937 & 1; //@line 4091
     $$$i485 = ($1937 | ($1920 | $1931)) ^ 1; //@line 4094
     HEAP32[162834] = $$$i485 & 1; //@line 4096
     do {
      if ($$$i488 & $$$i485) {
       HEAP32[tempDoublePtr >> 2] = $1885; //@line 4100
       HEAP32[tempDoublePtr + 4 >> 2] = $1888; //@line 4100
       $1942 = $$sink104$i$sink * +HEAPF64[tempDoublePtr >> 3]; //@line 4101
       HEAPF64[$1882 >> 3] = $1942; //@line 4102
       $$val410 = $1942; //@line 4103
      } else {
       if ($1937 | $1911 | $1894 & $1931 | $1905 & $1920) {
        $1944 = $1888 | 2147483647; //@line 4111
        $1945 = $1882; //@line 4112
        HEAP32[$1945 >> 2] = -1; //@line 4114
        HEAP32[$1945 + 4 >> 2] = $1944; //@line 4117
        HEAP32[tempDoublePtr >> 2] = -1; //@line 4118
        HEAP32[tempDoublePtr + 4 >> 2] = $1944; //@line 4118
        $$val410 = +HEAPF64[tempDoublePtr >> 3]; //@line 4119
        break;
       }
       $1951 = ($1889 | 0) == 0; //@line 4123
       $1952 = ($1915 | 0) != 0; //@line 4124
       $or$cond401 = $1951 & $1952 | ($1951 | $1952) ^ 1; //@line 4128
       if ($1920 | $1894) {
        if ($or$cond401) {
         HEAPF64[$1882 >> 3] = -0.0; //@line 4131
         $$val410 = -0.0; //@line 4132
         break;
        } else {
         HEAPF64[$1882 >> 3] = 0.0; //@line 4135
         $$val410 = 0.0; //@line 4136
         break;
        }
       } else {
        $1953 = $1888 & 524288; //@line 4140
        if ($or$cond401) {
         $1954 = $1953 | -1048576; //@line 4142
         $1955 = $1882; //@line 4143
         HEAP32[$1955 >> 2] = 0; //@line 4145
         HEAP32[$1955 + 4 >> 2] = $1954; //@line 4148
         HEAP32[tempDoublePtr >> 2] = 0; //@line 4149
         HEAP32[tempDoublePtr + 4 >> 2] = $1954; //@line 4149
         $$val410 = +HEAPF64[tempDoublePtr >> 3]; //@line 4150
         break;
        } else {
         $1960 = $1953 | 2146435072; //@line 4153
         $1961 = $1882; //@line 4154
         HEAP32[$1961 >> 2] = 0; //@line 4156
         HEAP32[$1961 + 4 >> 2] = $1960; //@line 4159
         HEAP32[tempDoublePtr >> 2] = 0; //@line 4160
         HEAP32[tempDoublePtr + 4 >> 2] = $1960; //@line 4160
         $$val410 = +HEAPF64[tempDoublePtr >> 3]; //@line 4161
         break;
        }
       }
      }
     } while (0);
     HEAPF64[14092] = $$val410; //@line 4167
     STACKTOP = sp; //@line 4168
     return;
    }
   case 364:
    {
     $1966 = 112672 + ($1201 << 3) | 0; //@line 4172
     $1970 = +HEAPF64[$1966 >> 3] / $$sink104$i$sink; //@line 4176
     HEAPF64[$1966 >> 3] = $1970; //@line 4177
     HEAPF64[14092] = $1970; //@line 4178
     STACKTOP = sp; //@line 4179
     return;
    }
   case 365:
    {
     $1971 = 112672 + ($1201 << 3) | 0; //@line 4183
     $1972 = +HEAPF64[$1971 >> 3]; //@line 4184
     $1976 = +(~~($1972 / $$sink104$i$sink + .5) | 0); //@line 4188
     HEAPF64[tempDoublePtr >> 3] = $1972; //@line 4189
     HEAP32[tempDoublePtr >> 2] ^ $1601; //@line 4191
     $1981 = (HEAP32[tempDoublePtr + 4 >> 2] ^ $1602 | 0) < 0 ? 8388608 : 0; //@line 4194
     $1983 = $1972 - $$sink104$i$sink * $1976; //@line 4196
     HEAPF64[$1971 >> 3] = $1983; //@line 4197
     HEAPF64[14092] = $1983; //@line 4198
     HEAP32[28190] = ~~+Math_abs(+$1976) >>> 0 << 16 & 8323072 | $1981; //@line 4204
     STACKTOP = sp; //@line 4205
     return;
    }
   case 366:
    {
     $1989 = 112672 + ($1201 << 3) | 0; //@line 4209
     $1990 = $1989; //@line 4210
     $1992 = HEAP32[$1990 >> 2] | 0; //@line 4212
     $1995 = HEAP32[$1990 + 4 >> 2] | 0; //@line 4215
     HEAP32[162825] = _bitshift64Lshr($1992 | 0, $1995 | 0, 63) | 0; //@line 4218
     $2001 = ($1992 | 0) == 0 & ($1995 & 2147483647 | 0) == 0; //@line 4222
     HEAP32[162826] = $2001 & 1; //@line 4224
     $2003 = $1995 & 2147483647; //@line 4225
     $2012 = ($1995 >> 30 & ~(($1992 | 0 - ($2003 ^ 2146435072 | $1992)) >> 31) | 0) != 0; //@line 4234
     HEAP32[162827] = $2012 & 1; //@line 4236
     $2018 = ($2003 | ($1992 | 0 - $1992) >>> 31) >>> 0 > 2146435072; //@line 4241
     HEAP32[162828] = $2018 & 1; //@line 4243
     $$$i482 = ($2018 | ($2001 | $2012)) ^ 1; //@line 4246
     HEAP32[162829] = $$$i482 & 1; //@line 4248
     HEAP32[162830] = _bitshift64Lshr($1601 | 0, $1602 | 0, 63) | 0; //@line 4251
     $2027 = ($1601 | 0) == 0 & ($1602 & 2147483647 | 0) == 0; //@line 4255
     HEAP32[162831] = $2027 & 1; //@line 4257
     $2029 = $1602 & 2147483647; //@line 4258
     $2038 = ($1602 >> 30 & ~(($1601 | 0 - ($2029 ^ 2146435072 | $1601)) >> 31) | 0) != 0; //@line 4267
     HEAP32[162832] = $2038 & 1; //@line 4269
     $2044 = ($2029 | ($1601 | 0 - $1601) >>> 31) >>> 0 > 2146435072; //@line 4274
     HEAP32[162833] = $2044 & 1; //@line 4276
     $$$i479 = ($2044 | ($2027 | $2038)) ^ 1; //@line 4279
     HEAP32[162834] = $$$i479 & 1; //@line 4281
     if ($$$i482 & $$$i479) {
      $2049 = _bitshift64Lshr($1992 | 0, $1995 | 0, 52) | 0; //@line 4285
      $2052 = ($2049 & 2047) + ~~$$sink104$i$sink | 0; //@line 4288
      if ($2052 >>> 0 < 15360) {
       $2063 = 0; //@line 4291
       $2065 = 0; //@line 4291
      } else {
       $2054 = $2052 >>> 0 > 17406; //@line 4293
       $2057 = _bitshift64Shl($2052 + 1024 & 2047 | 0, 0, 52) | 0; //@line 4296
       $2063 = $2054 ? 0 : $2057; //@line 4300
       $2065 = $2054 ? 2146435072 : tempRet0; //@line 4300
      }
      $2069 = $1992 | $2063; //@line 4305
      $2072 = $1995 & -2146435073 | $2065; //@line 4305
      label = 372; //@line 4306
     } else {
      if ($2038) {
       $2069 = -1; //@line 4310
       $2072 = $1995 | 2147483647; //@line 4310
       label = 372; //@line 4311
      } else {
       $2075 = $1992; //@line 4313
       $2078 = $1995; //@line 4313
      }
     }
     if ((label | 0) == 372) {
      $2067 = $1989; //@line 4317
      HEAP32[$2067 >> 2] = $2069; //@line 4319
      HEAP32[$2067 + 4 >> 2] = $2072; //@line 4322
      $2075 = $2069; //@line 4323
      $2078 = $2072; //@line 4323
     }
     $2073 = 112736; //@line 4325
     HEAP32[$2073 >> 2] = $2075; //@line 4327
     HEAP32[$2073 + 4 >> 2] = $2078; //@line 4330
     STACKTOP = sp; //@line 4331
     return;
    }
   case 374:
    {
     $2079 = 112672 + ($1201 << 3) | 0; //@line 4335
     $2083 = $$sink104$i$sink * +HEAPF64[$2079 >> 3]; //@line 4339
     HEAPF64[$2079 >> 3] = $2083; //@line 4340
     HEAPF64[14092] = $2083; //@line 4341
     STACKTOP = sp; //@line 4342
     return;
    }
   case 375:
    {
     $2084 = 112672 + ($1201 << 3) | 0; //@line 4346
     $2086 = +HEAPF64[$2084 >> 3] - $$sink104$i$sink; //@line 4348
     HEAPF64[$2084 >> 3] = $2086; //@line 4349
     HEAPF64[14092] = $2086; //@line 4350
     STACKTOP = sp; //@line 4351
     return;
    }
   case 376:
    {
     $2087 = 112672 + ($1201 << 3) | 0; //@line 4355
     $2089 = $$sink104$i$sink + +HEAPF64[$2087 >> 3]; //@line 4357
     HEAPF64[$2087 >> 3] = $2089; //@line 4358
     HEAPF64[14092] = $2089; //@line 4359
     STACKTOP = sp; //@line 4360
     return;
    }
   case 377:
    {
     HEAPF64[112672 + (($1 & 7) << 3) >> 3] = +Math_cos(+$$sink104$i$sink); //@line 4367
     $2093 = +Math_sin(+$$sink104$i$sink); //@line 4368
     HEAPF64[112672 + ($1201 << 3) >> 3] = $2093; //@line 4370
     HEAPF64[14092] = $2093; //@line 4371
     STACKTOP = sp; //@line 4372
     return;
    }
   case 378:
    {
     HEAPF64[14092] = 1.0; //@line 4376
     HEAP32[28191] = 0; //@line 4377
     HEAP32[28192] = 0; //@line 4378
     HEAP32[28190] = 0; //@line 4379
     HEAPF64[14092] = +HEAPF64[112672 + ($1201 << 3) >> 3] - $$sink104$i$sink; //@line 4383
     STACKTOP = sp; //@line 4384
     return;
    }
   case 379:
    {
     HEAP32[28191] = 0; //@line 4388
     HEAP32[28192] = 0; //@line 4389
     HEAP32[28190] = 0; //@line 4390
     HEAPF64[14092] = $$sink104$i$sink; //@line 4391
     STACKTOP = sp; //@line 4392
     return;
    }
   case 380:
    {
     $2103 = (HEAP32[162768] | 0) + -4 + (HEAP32[162769] | 0) - (HEAP32[162770] | 0) | 0; //@line 4401
     $2109 = FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($2103 >>> 16 << 2) >> 2] | 0) + 24 >> 2] & 127]($2103) | 0; //@line 4407
     HEAP32[162770] = $2109; //@line 4408
     HEAP32[162769] = $2109; //@line 4409
     HEAP32[162768] = $2103; //@line 4410
     __Z7op_illgj($0); //@line 4411
     STACKTOP = sp; //@line 4412
     return;
    }
   }
   break;
  }
 default:
  {
   $2115 = (HEAP32[162768] | 0) + -4 + (HEAP32[162769] | 0) - (HEAP32[162770] | 0) | 0; //@line 4424
   $2121 = FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($2115 >>> 16 << 2) >> 2] | 0) + 24 >> 2] & 127]($2115) | 0; //@line 4430
   HEAP32[162770] = $2121; //@line 4431
   HEAP32[162769] = $2121; //@line 4432
   HEAP32[162768] = $2115; //@line 4433
   __Z7op_illgj($0); //@line 4434
   STACKTOP = sp; //@line 4435
   return;
  }
 }
}
function __Z8PatchROMv() {
 var $$0 = 0, $$018$i$i$i = 0, $$020$i$i$i = 0, $$02235$i$i = 0, $$02334$i$i = 0, $$0292$i = 0, $$030$lcssa$i$i = 0, $$03034$i$i = 0, $$031$i$i = 0, $$032$i$i = 0, $$043$i = 0, $$043$i$i = 0, $$043$i22 = 0, $$043$i392$i = 0, $$043$i407$i = 0, $$043$i422$i = 0, $$043$i437$i = 0, $$043$i452$i = 0, $$043$i7 = 0, $$048$i = 0, $$048$i$i = 0, $$048$i21 = 0, $$048$i391$i = 0, $$048$i406$i = 0, $$048$i421$i = 0, $$048$i436$i = 0, $$048$i451$i = 0, $$048$i6 = 0, $$051$off0$i = 0, $$051$off0$i$i = 0, $$051$off0$i15 = 0, $$051$off0$i30 = 0, $$051$off0$i400$i = 0, $$051$off0$i415$i = 0, $$051$off0$i430$i = 0, $$051$off0$i445$i = 0, $$051$off0$i460$i = 0, $$05673$i = 0, $$05673$i$i = 0, $$05673$i11 = 0, $$05673$i26 = 0, $$05673$i396$i = 0, $$05673$i411$i = 0, $$05673$i426$i = 0, $$05673$i441$i = 0, $$05673$i456$i = 0, $$057$i = 0, $$057$i$i = 0, $$057$i20 = 0, $$057$i390$i = 0, $$057$i405$i = 0, $$057$i420$i = 0, $$057$i435$i = 0, $$057$i450$i = 0, $$057$i5 = 0, $$09$i$i = 0, $$09$i297$i = 0, $$09$i300$i = 0, $$09$i303$i = 0, $$09$i306$i = 0, $$09$i309$i = 0, $$09$i312$i = 0, $$09$i315$i = 0, $$09$i318$i = 0, $$09$i321$i = 0, $$09$i324$i = 0, $$09$i327$i = 0, $$09$i330$i = 0, $$09$i333$i = 0, $$09$i336$i = 0, $$09$i339$i = 0, $$09$i342$i = 0, $$09$i345$lcssa$i = 0, $$09$i351$i = 0, $$09$i354$lcssa$i = 0, $$09$i357$i = 0, $$09$i360$i = 0, $$09$i363$i = 0, $$09$i366$i = 0, $$09$i369$i = 0, $$09$i373$i = 0, $$09$i379$i = 0, $$1293$ph$i = 0, $$131$i$i = 0, $$14475$i = 0, $$14475$i$i = 0, $$14475$i24 = 0, $$14475$i394$i = 0, $$14475$i409$i = 0, $$14475$i424$i = 0, $$14475$i439$i = 0, $$14475$i454$i = 0, $$14475$i9 = 0, $$14974$i = 0, $$14974$i$i = 0, $$14974$i10 = 0, $$14974$i25 = 0, $$14974$i395$i = 0, $$14974$i410$i = 0, $$14974$i425$i = 0, $$14974$i440$i = 0, $$14974$i455$i = 0, $$15872$i = 0, $$15872$i$i = 0, $$15872$i12 = 0, $$15872$i27 = 0, $$15872$i397$i = 0, $$15872$i412$i = 0, $$15872$i427$i = 0, $$15872$i442$i = 0, $$15872$i457$i = 0, $$2$i = 0, $$2$i383$i = 0, $$2$i387$i = 0, $$2$i41 = 0, $$245$i = 0, $$245$i$i = 0, $$245$i16 = 0, $$245$i31 = 0, $$245$i401$i = 0, $$245$i416$i = 0, $$245$i431$i = 0, $$245$i446$i = 0, $$245$i461$i = 0, $$3$i = 0, $$4$i = 0, $$461$i = 0, $$461$i$i = 0, $$461$i14 = 0, $$461$i29 = 0, $$461$i399$i = 0, $$461$i414$i = 0, $$461$i429$i = 0, $$461$i444$i = 0, $$461$i459$i = 0, $$5$ph$i = 0, $$5$ph$i$i = 0, $$5$ph$i18 = 0, $$5$ph$i33 = 0, $$5$ph$i403$i = 0, $$5$ph$i418$i = 0, $$5$ph$i433$i = 0, $$5$ph$i448$i = 0, $$5$ph$i463$i = 0, $$sink = 0, $$sink$i = 0, $$sink295$in$i = 0, $$sink296$i = 0, $$sink45$in = 0, $$sink46 = 0, $1000 = 0, $1003 = 0, $1005 = 0, $1007 = 0, $1009 = 0, $1013 = 0, $1014 = 0, $102 = 0, $1021 = 0, $1022 = 0, $1028 = 0, $103 = 0, $1033 = 0, $1035 = 0, $1036 = 0, $1037 = 0, $1042 = 0, $1043 = 0, $1048 = 0, $1049 = 0, $1058 = 0, $1066 = 0, $1068 = 0, $1069 = 0, $1072 = 0, $1075 = 0, $1077 = 0, $1078 = 0, $1084 = 0, $1089 = 0, $109 = 0, $1091 = 0, $1092 = 0, $1093 = 0, $1098 = 0, $1099 = 0, $11 = 0, $1104 = 0, $1105 = 0, $1114 = 0, $112 = 0, $1153 = 0, $1154 = 0, $1160 = 0, $1165 = 0, $1167 = 0, $1168 = 0, $1169 = 0, $117 = 0, $1174 = 0, $1175 = 0, $118 = 0, $1180 = 0, $1181 = 0, $1190 = 0, $1199 = 0, $1204 = 0, $1216 = 0, $1218 = 0, $1225 = 0, $1226 = 0, $1227 = 0, $1244 = 0, $1249 = 0, $125 = 0, $1251 = 0, $1256 = 0, $1261 = 0, $1264 = 0, $1267 = 0, $1269 = 0, $1276 = 0, $1277 = 0, $1278 = 0, $1295 = 0, $1300 = 0, $1302 = 0, $1307 = 0, $1312 = 0, $1315 = 0, $1317 = 0, $132 = 0, $1321 = 0, $1323 = 0, $1330 = 0, $1331 = 0, $1332 = 0, $1349 = 0, $135 = 0, $1354 = 0, $1356 = 0, $1361 = 0, $1366 = 0, $1369 = 0, $1371 = 0, $1373 = 0, $1375 = 0, $1377 = 0, $1379 = 0, $1383 = 0, $1385 = 0, $1392 = 0, $1393 = 0, $1394 = 0, $1411 = 0, $1416 = 0, $1418 = 0, $1423 = 0, $1428 = 0, $143 = 0, $1431 = 0, $1433 = 0, $1435 = 0, $1437 = 0, $1439 = 0, $1441 = 0, $1443 = 0, $1445 = 0, $1450 = 0, $1452 = 0, $1459 = 0, $146 = 0, $1461 = 0, $1468 = 0, $1469 = 0, $1470 = 0, $1487 = 0, $1492 = 0, $1494 = 0, $1499 = 0, $1504 = 0, $1507 = 0, $1509 = 0, $151 = 0, $1511 = 0, $1515 = 0, $1536 = 0, $1538 = 0, $1540 = 0, $1542 = 0, $1544 = 0, $1546 = 0, $155 = 0, $1550 = 0, $1552 = 0, $1559 = 0, $156 = 0, $1560 = 0, $1561 = 0, $157 = 0, $1578 = 0, $1583 = 0, $1585 = 0, $1590 = 0, $1595 = 0, $1598 = 0, $1602 = 0, $1606 = 0, $1607 = 0, $1614 = 0, $1615 = 0, $1616 = 0, $163 = 0, $1633 = 0, $1638 = 0, $164 = 0, $1640 = 0, $1645 = 0, $1650 = 0, $1653 = 0, $1655 = 0, $1659 = 0, $1661 = 0, $1663 = 0, $1670 = 0, $1674 = 0, $1675 = 0, $1682 = 0, $1683 = 0, $1684 = 0, $170 = 0, $1701 = 0, $1706 = 0, $1708 = 0, $171 = 0, $1713 = 0, $1718 = 0, $1721 = 0, $1723 = 0, $1727 = 0, $1729 = 0, $1731 = 0, $1738 = 0, $1739 = 0, $1745 = 0, $1750 = 0, $1752 = 0, $1753 = 0, $1754 = 0, $1759 = 0, $1760 = 0, $1765 = 0, $1766 = 0, $177 = 0, $1775 = 0, $178 = 0, $1783 = 0, $1784 = 0, $179 = 0, $1790 = 0, $1798 = 0, $1803 = 0, $1805 = 0, $1806 = 0, $1807 = 0, $1812 = 0, $1813 = 0, $1818 = 0, $1819 = 0, $1828 = 0, $185 = 0, $1859 = 0, $1870 = 0, $1881 = 0, $1884 = 0, $19 = 0, $193 = 0, $2 = 0, $203 = 0, $210 = 0, $215 = 0, $223 = 0, $228 = 0, $233 = 0, $245 = 0, $249 = 0, $26 = 0, $263 = 0, $275 = 0, $281 = 0, $294 = 0, $307 = 0, $314 = 0, $32 = 0, $321 = 0, $328 = 0, $34 = 0, $349 = 0, $35 = 0, $351 = 0, $353 = 0, $355 = 0, $357 = 0, $359 = 0, $363 = 0, $364 = 0, $367 = 0, $370 = 0, $372 = 0, $374 = 0, $381 = 0, $400 = 0, $41 = 0, $410 = 0, $414 = 0, $416 = 0, $420 = 0, $425 = 0, $432 = 0, $441 = 0, $457 = 0, $462 = 0, $48 = 0, $49 = 0, $493 = 0, $501 = 0, $502 = 0, $507 = 0, $514 = 0, $524 = 0, $526 = 0, $528 = 0, $530 = 0, $533 = 0, $537 = 0, $539 = 0, $54 = 0, $55 = 0, $553 = 0, $555 = 0, $568 = 0, $569 = 0, $577 = 0, $578 = 0, $588 = 0, $589 = 0, $597 = 0, $599 = 0, $606 = 0, $607 = 0, $608 = 0, $61 = 0, $625 = 0, $63 = 0, $630 = 0, $632 = 0, $637 = 0, $64 = 0, $642 = 0, $645 = 0, $651 = 0, $652 = 0, $658 = 0, $662 = 0, $667 = 0, $668 = 0, $674 = 0, $675 = 0, $676 = 0, $680 = 0, $692 = 0, $695 = 0, $697 = 0, $698 = 0, $705 = 0, $710 = 0, $712 = 0, $739 = 0, $746 = 0, $747 = 0, $75 = 0, $752 = 0, $754 = 0, $756 = 0, $760 = 0, $761 = 0, $766 = 0, $768 = 0, $773 = 0, $774 = 0, $779 = 0, $781 = 0, $786 = 0, $792 = 0, $794 = 0, $796 = 0, $798 = 0, $802 = 0, $803 = 0, $809 = 0, $81 = 0, $810 = 0, $812 = 0, $814 = 0, $816 = 0, $82 = 0, $820 = 0, $845 = 0, $846 = 0, $851 = 0, $856 = 0, $858 = 0, $869 = 0, $870 = 0, $879 = 0, $88 = 0, $89 = 0, $912 = 0, $927 = 0, $934 = 0, $936 = 0, $938 = 0, $940 = 0, $944 = 0, $95 = 0, $955 = 0, $956 = 0, $96 = 0, $964 = 0, $965 = 0, $972 = 0, $973 = 0, $980 = 0, $982 = 0, $983 = 0, $988 = 0, $990 = 0, $995 = 0, $996 = 0, $997 = 0, $vararg_buffer = 0, $vararg_buffer1 = 0, $vararg_buffer10 = 0, $vararg_buffer13 = 0, $vararg_buffer23 = 0, $vararg_buffer26 = 0, $vararg_buffer4 = 0, $vararg_buffer7 = 0, dest = 0, label = 0, sp = 0, src = 0, stop = 0;
 sp = STACKTOP; //@line 2907
 STACKTOP = STACKTOP + 128 | 0; //@line 2908
 $vararg_buffer26 = sp + 104 | 0; //@line 2909
 $vararg_buffer23 = sp + 96 | 0; //@line 2910
 $vararg_buffer13 = sp + 64 | 0; //@line 2911
 $vararg_buffer10 = sp + 32 | 0; //@line 2912
 $vararg_buffer7 = sp + 24 | 0; //@line 2913
 $vararg_buffer4 = sp + 16 | 0; //@line 2914
 $vararg_buffer1 = sp + 8 | 0; //@line 2915
 $vararg_buffer = sp; //@line 2916
 if (!(HEAP8[651928] | 0)) {
  label = 21; //@line 2920
 } else {
  _puts(90716) | 0; //@line 2922
  $2 = HEAP32[30896] | 0; //@line 2923
  HEAP32[$vararg_buffer >> 2] = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($2 >>> 16 << 2) >> 2] >> 2] & 127]($2) | 0; //@line 2929
  _printf(90727, $vararg_buffer) | 0; //@line 2930
  HEAP32[$vararg_buffer1 >> 2] = HEAPU16[325960]; //@line 2933
  _printf(90747, $vararg_buffer1) | 0; //@line 2934
  $11 = (HEAP32[30896] | 0) + 18 | 0; //@line 2936
  HEAP32[$vararg_buffer4 >> 2] = FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($11 >>> 16 << 2) >> 2] | 0) + 4 >> 2] & 127]($11) | 0; //@line 2943
  _printf(90767, $vararg_buffer4) | 0; //@line 2944
  $19 = (HEAP32[30896] | 0) + 26 | 0; //@line 2946
  HEAP32[$vararg_buffer7 >> 2] = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($19 >>> 16 << 2) >> 2] >> 2] & 127]($19) | 0; //@line 2952
  _printf(90787, $vararg_buffer7) | 0; //@line 2953
  $26 = (HEAP32[30896] | 0) + 34 | 0; //@line 2955
  HEAP32[$vararg_buffer10 >> 2] = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($26 >>> 16 << 2) >> 2] >> 2] & 127]($26) | 0; //@line 2961
  _printf(90807, $vararg_buffer10) | 0; //@line 2962
  $32 = HEAP16[325960] | 0; //@line 2963
  if ($32 << 16 >> 16 == 1660) {
   _puts(90828) | 0; //@line 2966
   _puts(90843) | 0; //@line 2967
   _puts(90869) | 0; //@line 2968
   $34 = HEAP32[30896] | 0; //@line 2969
   $35 = $34 + 26 | 0; //@line 2970
   $41 = (FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($35 >>> 16 << 2) >> 2] >> 2] & 127]($35) | 0) + $34 | 0; //@line 2976
   $$031$i$i = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($41 >>> 16 << 2) >> 2] >> 2] & 127]($41) | 0; //@line 2982
   do {
    $48 = (HEAP32[30896] | 0) + $$031$i$i | 0; //@line 2985
    $49 = $48 + 12 | 0; //@line 2986
    $54 = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($49 >>> 16 << 2) >> 2] >> 2] & 127]($49) | 0; //@line 2991
    $55 = $48 + 23 | 0; //@line 2992
    $61 = FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($55 >>> 16 << 2) >> 2] | 0) + 8 >> 2] & 127]($55) | 0; //@line 2998
    if (($61 | 0) > 0) {
     $63 = $48 + 24 | 0; //@line 3001
     $$03034$i$i = 0; //@line 3002
     do {
      $64 = $63 + $$03034$i$i | 0; //@line 3004
      HEAP8[$vararg_buffer10 + $$03034$i$i >> 0] = FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($64 >>> 16 << 2) >> 2] | 0) + 8 >> 2] & 127]($64) | 0; //@line 3013
      $$03034$i$i = $$03034$i$i + 1 | 0; //@line 3014
     } while (($$03034$i$i | 0) != ($61 | 0));
     $$030$lcssa$i$i = $61; //@line 3017
    } else {
     $$030$lcssa$i$i = 0; //@line 3024
    }
    HEAP8[$vararg_buffer10 + $$030$lcssa$i$i >> 0] = 0; //@line 3027
    $75 = $48 + 16 | 0; //@line 3028
    $81 = FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($75 >>> 16 << 2) >> 2] | 0) + 8 >> 2] & 127]($75) | 0; //@line 3034
    $82 = $48 + 17 | 0; //@line 3035
    $88 = FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($82 >>> 16 << 2) >> 2] | 0) + 8 >> 2] & 127]($82) | 0; //@line 3041
    $89 = $48 + 18 | 0; //@line 3042
    $95 = FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($89 >>> 16 << 2) >> 2] | 0) + 8 >> 2] & 127]($89) | 0; //@line 3048
    $96 = $48 + 19 | 0; //@line 3049
    $102 = FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($96 >>> 16 << 2) >> 2] | 0) + 8 >> 2] & 127]($96) | 0; //@line 3055
    $103 = $48 + 20 | 0; //@line 3056
    $109 = FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($103 >>> 16 << 2) >> 2] | 0) + 4 >> 2] & 127]($103) | 0; //@line 3062
    $112 = $54 + -8 + (HEAP32[30896] | 0) | 0; //@line 3065
    $117 = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($112 >>> 16 << 2) >> 2] >> 2] & 127]($112) | 0; //@line 3070
    HEAP32[$vararg_buffer13 >> 2] = $54; //@line 3071
    HEAP32[$vararg_buffer13 + 4 >> 2] = $81; //@line 3073
    HEAP32[$vararg_buffer13 + 8 >> 2] = $88; //@line 3075
    HEAP32[$vararg_buffer13 + 12 >> 2] = $95; //@line 3077
    HEAP32[$vararg_buffer13 + 16 >> 2] = $102; //@line 3079
    HEAP32[$vararg_buffer13 + 20 >> 2] = $109; //@line 3081
    HEAP32[$vararg_buffer13 + 24 >> 2] = $117; //@line 3083
    HEAP32[$vararg_buffer13 + 28 >> 2] = $vararg_buffer10; //@line 3085
    _printf(90918, $vararg_buffer13) | 0; //@line 3086
    $118 = $48 + 8 | 0; //@line 3087
    $$031$i$i = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($118 >>> 16 << 2) >> 2] >> 2] & 127]($118) | 0; //@line 3092
   } while (($$031$i$i | 0) != 0);
   _putchar(10) | 0; //@line 3100
   $$02235$i$i = 0; //@line 3101
   $$02334$i$i = 12288; //@line 3101
   while (1) {
    $125 = (HEAP32[30896] | 0) + $$02334$i$i | 0; //@line 3104
    if ((FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($125 >>> 16 << 2) >> 2] >> 2] & 127]($125) | 0) == -603978491) {
     label = 10; //@line 3112
     break;
    }
    $$02235$i$i = $$02235$i$i + 2 | 0; //@line 3115
    if (($$02235$i$i | 0) >= 8192) {
     break;
    } else {
     $$02334$i$i = $$02334$i$i + 2 | 0; //@line 3119
    }
   }
   L15 : do {
    if ((label | 0) == 10) {
     $132 = $$02334$i$i + -16 | 0; //@line 3126
     if ($132 | 0) {
      $$032$i$i = $132; //@line 3129
      while (1) {
       $135 = (HEAP32[30896] | 0) + $$032$i$i | 0; //@line 3132
       if ((FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($135 >>> 16 << 2) >> 2] >> 2] & 127]($135) | 0) == ($132 - $$032$i$i | 0)) {
        break;
       }
       $143 = $$032$i$i + -4 | 0; //@line 3143
       if (!$143) {
        break L15;
       } else {
        $$032$i$i = $143; //@line 3148
       }
      }
      HEAP32[$vararg_buffer23 >> 2] = $$032$i$i; //@line 3151
      _printf(90942, $vararg_buffer23) | 0; //@line 3152
      _puts(90968) | 0; //@line 3153
      _puts(90869) | 0; //@line 3154
      $146 = (HEAP32[30896] | 0) + $$032$i$i | 0; //@line 3156
      $151 = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($146 >>> 16 << 2) >> 2] >> 2] & 127]($146) | 0; //@line 3161
      if ($151 | 0) {
       $$131$i$i = $$032$i$i; //@line 3164
       $155 = $151; //@line 3164
       do {
        $156 = $$131$i$i + $155 + (HEAP32[30896] | 0) | 0; //@line 3168
        $157 = $156 + 18 | 0; //@line 3169
        $163 = FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($157 >>> 16 << 2) >> 2] | 0) + 8 >> 2] & 127]($157) | 0; //@line 3175
        $164 = $156 + 16 | 0; //@line 3176
        $170 = FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($164 >>> 16 << 2) >> 2] | 0) + 4 >> 2] & 127]($164) | 0; //@line 3182
        $171 = $156 + 20 | 0; //@line 3183
        $177 = FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($171 >>> 16 << 2) >> 2] | 0) + 4 >> 2] & 127]($171) | 0; //@line 3189
        $178 = $163 & 255; //@line 3190
        $179 = $178 + 6 | 0; //@line 3191
        $$020$i$i$i = 0; //@line 3192
        while (1) {
         if ((HEAP32[1e3 + ($$020$i$i$i << 3) + 4 >> 2] | 0) == ($179 | 0)) {
          label = 16; //@line 3198
          break;
         }
         $185 = $$020$i$i$i + 1 | 0; //@line 3201
         if (($185 | 0) == 59) {
          $$018$i$i$i = 90997; //@line 3204
          break;
         } else {
          $$020$i$i$i = $185; //@line 3207
         }
        }
        if ((label | 0) == 16) {
         label = 0; //@line 3211
         $$018$i$i$i = HEAP32[1e3 + ($$020$i$i$i << 3) >> 2] | 0; //@line 3214
        }
        HEAP32[$vararg_buffer26 >> 2] = $156 - (HEAP32[30896] | 0); //@line 3220
        HEAP32[$vararg_buffer26 + 4 >> 2] = $178; //@line 3222
        HEAP32[$vararg_buffer26 + 8 >> 2] = $170 & 65535; //@line 3224
        HEAP32[$vararg_buffer26 + 12 >> 2] = $177 & 65535; //@line 3226
        HEAP32[$vararg_buffer26 + 16 >> 2] = $$018$i$i$i; //@line 3228
        _printf(91005, $vararg_buffer26) | 0; //@line 3229
        $$131$i$i = $$131$i$i + 4 | 0; //@line 3230
        $193 = (HEAP32[30896] | 0) + $$131$i$i | 0; //@line 3232
        $155 = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($193 >>> 16 << 2) >> 2] >> 2] & 127]($193) | 0; //@line 3237
       } while (($155 | 0) != 0);
      }
     }
    }
   } while (0);
   _putchar(10) | 0; //@line 3249
   label = 21; //@line 3250
  } else {
   $203 = $32; //@line 3252
  }
 }
 if ((label | 0) == 21) {
  $203 = HEAP16[325960] | 0; //@line 3257
 }
 switch ($203 << 16 >> 16) {
 case 630:
  {
   HEAP16[(HEAP32[30897] | 0) + 7232 >> 1] = _htons(24606) | 0; //@line 3264
   HEAP16[(HEAP32[30897] | 0) + 7276 >> 1] = _htons(31744) | 0; //@line 3268
   $210 = HEAP32[30897] | 0; //@line 3269
   HEAP16[$210 + 80 >> 1] = _htons(20081) | 0; //@line 3273
   HEAP16[$210 + 82 >> 1] = _htons(20081) | 0; //@line 3275
   $215 = HEAP32[30897] | 0; //@line 3276
   HEAP16[$215 + 106 >> 1] = _htons(20081) | 0; //@line 3280
   HEAP16[$215 + 108 >> 1] = _htons(20081) | 0; //@line 3282
   HEAP16[(HEAP32[30897] | 0) + 13156 >> 1] = _htons(20081) | 0; //@line 3286
   $223 = HEAP32[30897] | 0; //@line 3287
   HEAP16[$223 + 41664 >> 1] = _htons(28932) | 0; //@line 3291
   HEAP16[$223 + 41666 >> 1] = _htons(20181) | 0; //@line 3293
   $228 = HEAP32[30897] | 0; //@line 3294
   HEAP16[$228 + 286 >> 1] = _htons(20081) | 0; //@line 3298
   HEAP16[$228 + 288 >> 1] = _htons(20081) | 0; //@line 3300
   $233 = HEAP32[30897] | 0; //@line 3301
   HEAP16[$233 + 260138 >> 1] = _htons(28938) | 0; //@line 3305
   HEAP16[$233 + 260140 >> 1] = _htons(20081) | 0; //@line 3308
   HEAP16[$233 + 260142 >> 1] = _htons(20081) | 0; //@line 3311
   HEAP16[$233 + 260144 >> 1] = _htons(20081) | 0; //@line 3313
   HEAP16[(HEAP32[30897] | 0) + 3418 >> 1] = _htons(24606) | 0; //@line 3317
   HEAP32[28207] = 214656; //@line 3318
   $245 = HEAP32[30897] | 0; //@line 3319
   dest = $245 + 214656 | 0; //@line 3321
   src = 91029; //@line 3321
   stop = dest + 86 | 0; //@line 3321
   do {
    HEAP8[dest >> 0] = HEAP8[src >> 0] | 0; //@line 3321
    dest = dest + 1 | 0; //@line 3321
    src = src + 1 | 0; //@line 3321
   } while ((dest | 0) < (stop | 0));
   dest = $245 + 214912 | 0; //@line 3323
   src = 91115; //@line 3323
   stop = dest + 86 | 0; //@line 3323
   do {
    HEAP8[dest >> 0] = HEAP8[src >> 0] | 0; //@line 3323
    dest = dest + 1 | 0; //@line 3323
    src = src + 1 | 0; //@line 3323
   } while ((dest | 0) < (stop | 0));
   dest = $245 + 215168 | 0; //@line 3325
   src = 91201; //@line 3325
   stop = dest + 90 | 0; //@line 3325
   do {
    HEAP8[dest >> 0] = HEAP8[src >> 0] | 0; //@line 3325
    dest = dest + 1 | 0; //@line 3325
    src = src + 1 | 0; //@line 3325
   } while ((dest | 0) < (stop | 0));
   $249 = HEAP32[30896] | 0; //@line 3326
   HEAP32[28483] = $249 + 215680; //@line 3328
   _memcpy($245 + 215680 | 0, 93682, 258) | 0; //@line 3330
   HEAP32[28484] = $249 + 216192; //@line 3332
   _memcpy($245 + 216192 | 0, 93940, 258) | 0; //@line 3334
   HEAP32[28488] = $249 + 216704; //@line 3336
   _memcpy($245 + 216704 | 0, 94272, 258) | 0; //@line 3338
   HEAP32[28492] = $249 + 217216; //@line 3340
   _memcpy($245 + 217216 | 0, 94883, 258) | 0; //@line 3342
   HEAP32[28208] = 203694; //@line 3343
   HEAP16[$245 + 203706 >> 1] = _htons(28939) | 0; //@line 3347
   HEAP16[$245 + 203708 >> 1] = _htons(20085) | 0; //@line 3349
   $263 = HEAP32[28208] | 0; //@line 3351
   dest = (HEAP32[30897] | 0) + $263 + 256 | 0; //@line 3354
   src = 91291; //@line 3354
   stop = dest + 98 | 0; //@line 3354
   do {
    HEAP8[dest >> 0] = HEAP8[src >> 0] | 0; //@line 3354
    dest = dest + 1 | 0; //@line 3354
    src = src + 1 | 0; //@line 3354
   } while ((dest | 0) < (stop | 0));
   dest = (HEAP32[30897] | 0) + $263 + 512 | 0; //@line 3358
   src = 91389; //@line 3358
   stop = dest + 100 | 0; //@line 3358
   do {
    HEAP8[dest >> 0] = HEAP8[src >> 0] | 0; //@line 3358
    dest = dest + 1 | 0; //@line 3358
    src = src + 1 | 0; //@line 3358
   } while ((dest | 0) < (stop | 0));
   dest = (HEAP32[30897] | 0) + $263 + 768 | 0; //@line 3362
   src = 91489; //@line 3362
   stop = dest + 98 | 0; //@line 3362
   do {
    HEAP8[dest >> 0] = HEAP8[src >> 0] | 0; //@line 3362
    dest = dest + 1 | 0; //@line 3362
    src = src + 1 | 0; //@line 3362
   } while ((dest | 0) < (stop | 0));
   dest = (HEAP32[30897] | 0) + $263 + 1024 | 0; //@line 3366
   src = 91587; //@line 3366
   stop = dest + 100 | 0; //@line 3366
   do {
    HEAP8[dest >> 0] = HEAP8[src >> 0] | 0; //@line 3366
    dest = dest + 1 | 0; //@line 3366
    src = src + 1 | 0; //@line 3366
   } while ((dest | 0) < (stop | 0));
   $275 = HEAP32[30897] | 0; //@line 3367
   dest = $275 + 14464 | 0; //@line 3369
   src = 91687; //@line 3369
   stop = dest + 54 | 0; //@line 3369
   do {
    HEAP8[dest >> 0] = HEAP8[src >> 0] | 0; //@line 3369
    dest = dest + 1 | 0; //@line 3369
    src = src + 1 | 0; //@line 3369
   } while ((dest | 0) < (stop | 0));
   HEAP16[$275 + 108892 >> 1] = _htons(28964) | 0; //@line 3373
   HEAP16[$275 + 108894 >> 1] = _htons(20085) | 0; //@line 3375
   $281 = HEAP32[30897] | 0; //@line 3376
   HEAP16[$281 + 108906 >> 1] = _htons(16615) | 0; //@line 3380
   HEAP16[$281 + 108908 >> 1] = _htons(124) | 0; //@line 3383
   HEAP16[$281 + 108910 >> 1] = _htons(1792) | 0; //@line 3386
   HEAP16[$281 + 108912 >> 1] = _htons(28965) | 0; //@line 3389
   HEAP16[$281 + 108914 >> 1] = _htons(18143) | 0; //@line 3392
   HEAP16[$281 + 108916 >> 1] = _htons(20085) | 0; //@line 3394
   $294 = HEAP32[30897] | 0; //@line 3395
   HEAP16[$294 + 108932 >> 1] = _htons(16615) | 0; //@line 3399
   HEAP16[$294 + 108934 >> 1] = _htons(124) | 0; //@line 3402
   HEAP16[$294 + 108936 >> 1] = _htons(1792) | 0; //@line 3405
   HEAP16[$294 + 108938 >> 1] = _htons(28966) | 0; //@line 3408
   HEAP16[$294 + 108940 >> 1] = _htons(18143) | 0; //@line 3411
   $307 = $294 + 108944 | 0; //@line 3413
   HEAP16[$294 + 108942 >> 1] = _htons(20085) | 0; //@line 3414
   HEAP32[28205] = $307 - (HEAP32[30897] | 0); //@line 3418
   HEAP16[$307 >> 1] = _htons(28967) | 0; //@line 3421
   $314 = $294 + 108948 | 0; //@line 3423
   HEAP16[$294 + 108946 >> 1] = _htons(20085) | 0; //@line 3424
   HEAP32[28206] = $314 - (HEAP32[30897] | 0); //@line 3428
   HEAP16[$314 >> 1] = _htons(28982) | 0; //@line 3431
   HEAP16[$294 + 108950 >> 1] = _htons(20085) | 0; //@line 3433
   $321 = HEAP32[30897] | 0; //@line 3434
   HEAP16[$321 + 107014 >> 1] = _htons(28968) | 0; //@line 3438
   HEAP16[$321 + 107016 >> 1] = _htons(11849) | 0; //@line 3441
   HEAP16[$321 + 107018 >> 1] = _htons(20176) | 0; //@line 3443
   $328 = HEAP32[30897] | 0; //@line 3444
   HEAP16[$328 + 59200 >> 1] = _htons(20217) | 0; //@line 3448
   HEAP16[$328 + 59202 >> 1] = _htons(((HEAP32[30896] | 0) + 768 + (HEAP32[28207] | 0) | 0) >>> 16 & 65535) | 0; //@line 3457
   HEAP16[$328 + 59204 >> 1] = _htons((HEAP32[30896] | 0) + 768 + (HEAP32[28207] | 0) & 65535) | 0; //@line 3464
   $349 = (HEAP32[30897] | 0) + (HEAP32[28207] | 0) + 768 | 0; //@line 3468
   $351 = $349 + 2 | 0; //@line 3470
   HEAP16[$349 >> 1] = _htons(12035) | 0; //@line 3471
   $353 = $351 + 2 | 0; //@line 3473
   HEAP16[$351 >> 1] = _htons(8312) | 0; //@line 3474
   $355 = $353 + 2 | 0; //@line 3476
   HEAP16[$353 >> 1] = _htons(2032) | 0; //@line 3477
   $357 = $355 + 2 | 0; //@line 3479
   HEAP16[$355 >> 1] = _htons(20112) | 0; //@line 3480
   $359 = $357 + 2 | 0; //@line 3482
   HEAP16[$357 >> 1] = _htons(8735) | 0; //@line 3483
   HEAP16[$359 >> 1] = _htons(28972) | 0; //@line 3486
   HEAP16[$359 + 2 >> 1] = _htons(20085) | 0; //@line 3488
   $363 = HEAP32[30896] | 0; //@line 3489
   $364 = HEAP32[28207] | 0; //@line 3490
   HEAP32[28202] = $363 + 3072 + $364; //@line 3493
   $367 = $363 + 75668 | 0; //@line 3494
   $370 = (HEAP32[30897] | 0) + $364 + 3072 | 0; //@line 3497
   $372 = $370 + 2 | 0; //@line 3499
   HEAP16[$370 >> 1] = _htons(28970) | 0; //@line 3500
   $374 = $372 + 2 | 0; //@line 3502
   HEAP16[$372 >> 1] = _htons(20217) | 0; //@line 3503
   HEAP16[$374 >> 1] = _htons($367 >>> 16 & 65535) | 0; //@line 3508
   HEAP16[$374 + 2 >> 1] = _htons($367 & 65535) | 0; //@line 3511
   $381 = HEAP32[30897] | 0; //@line 3512
   HEAP16[$381 + 11066 >> 1] = _htons(22664) | 0; //@line 3516
   HEAP16[$381 + 11068 >> 1] = _htons(22664) | 0; //@line 3519
   HEAP16[$381 + 11070 >> 1] = _htons(20081) | 0; //@line 3522
   HEAP16[$381 + 11072 >> 1] = _htons(20081) | 0; //@line 3525
   HEAP16[$381 + 11074 >> 1] = _htons(20081) | 0; //@line 3528
   HEAP16[$381 + 11076 >> 1] = _htons(20081) | 0; //@line 3531
   HEAP16[$381 + 11078 >> 1] = _htons(20081) | 0; //@line 3534
   HEAP16[$381 + 11080 >> 1] = _htons(20081) | 0; //@line 3537
   HEAP16[$381 + 11082 >> 1] = _htons(20081) | 0; //@line 3539
   $400 = HEAP32[30897] | 0; //@line 3540
   HEAP16[$400 + 11236 >> 1] = _htons(20081) | 0; //@line 3544
   HEAP16[$400 + 11238 >> 1] = _htons(20081) | 0; //@line 3547
   HEAP16[$400 + 11240 >> 1] = _htons(28969) | 0; //@line 3550
   $$sink = _htons(19072) | 0; //@line 3553
   $$sink45$in = $400 + 11242 | 0; //@line 3553
   $$sink46 = $400 + 11244 | 0; //@line 3553
   break;
  }
 case 1660:
  {
   $410 = HEAP32[30897] | 0; //@line 3557
   $$09$i$i = 13312; //@line 3558
   while (1) {
    if (!(_memcmp($410 + $$09$i$i | 0, 91741, 8) | 0)) {
     break;
    }
    $414 = $$09$i$i + 1 | 0; //@line 3566
    if ($414 >>> 0 < 15360) {
     $$09$i$i = $414; //@line 3569
    } else {
     $$0 = 0; //@line 3571
     label = 246; //@line 3572
     break;
    }
   }
   if ((label | 0) == 246) {
    STACKTOP = sp; //@line 3577
    return $$0 | 0; //@line 3577
   }
   $416 = $$09$i$i + -16 | 0; //@line 3579
   HEAP32[28201] = $416; //@line 3580
   $420 = $$09$i$i + -4 + (HEAP32[30896] | 0) | 0; //@line 3584
   $425 = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($420 >>> 16 << 2) >> 2] >> 2] & 127]($420) | 0; //@line 3589
   HEAP8[$410 + $416 + $425 >> 0] = 3; //@line 3591
   dest = $410 + ($$09$i$i + -15 + $425) | 0; //@line 3595
   stop = dest + 15 | 0; //@line 3595
   do {
    HEAP8[dest >> 0] = 8; //@line 3595
    dest = dest + 1 | 0; //@line 3595
   } while ((dest | 0) < (stop | 0));
   $432 = (HEAP32[30897] | 0) + (HEAP32[28201] | 0) + 18 | 0; //@line 3599
   HEAP8[$432 >> 0] = __Z14PrefsFindInt32PKc(91749) | 0; //@line 3602
   if (!(HEAP32[30881] | 0)) {
    HEAP8[(HEAP32[30897] | 0) + (HEAP32[28201] | 0) + 22 >> 0] = 4; //@line 3610
   }
   $441 = HEAP32[30897] | 0; //@line 3612
   HEAP16[$441 + 140 >> 1] = _htons(28931) | 0; //@line 3616
   HEAP16[$441 + 142 >> 1] = _htons(20217) | 0; //@line 3619
   HEAP16[$441 + 144 >> 1] = _htons(((HEAP32[30896] | 0) + 186 | 0) >>> 16 & 65535) | 0; //@line 3626
   HEAP16[$441 + 146 >> 1] = _htons((HEAP32[30896] | 0) + 186 & 65535) | 0; //@line 3631
   $457 = HEAP32[30897] | 0; //@line 3632
   HEAP16[$457 + 194 >> 1] = _htons(20081) | 0; //@line 3636
   HEAP16[$457 + 196 >> 1] = _htons(20081) | 0; //@line 3638
   $462 = HEAP32[30897] | 0; //@line 3639
   HEAP16[$462 + 198 >> 1] = _htons(20081) | 0; //@line 3643
   HEAP16[$462 + 200 >> 1] = _htons(20081) | 0; //@line 3646
   HEAP16[$462 + 202 >> 1] = _htons(20081) | 0; //@line 3649
   HEAP16[$462 + 204 >> 1] = _htons(20081) | 0; //@line 3652
   HEAP16[$462 + 206 >> 1] = _htons(20081) | 0; //@line 3655
   HEAP16[$462 + 208 >> 1] = _htons(20081) | 0; //@line 3658
   HEAP16[$462 + 210 >> 1] = _htons(20081) | 0; //@line 3661
   HEAP16[$462 + 212 >> 1] = _htons(20081) | 0; //@line 3664
   HEAP16[$462 + 214 >> 1] = _htons(20081) | 0; //@line 3667
   HEAP16[$462 + 216 >> 1] = _htons(20081) | 0; //@line 3670
   HEAP16[$462 + 218 >> 1] = _htons(20081) | 0; //@line 3673
   HEAP16[$462 + 220 >> 1] = _htons(20081) | 0; //@line 3676
   HEAP16[$462 + 222 >> 1] = _htons(20081) | 0; //@line 3679
   HEAP16[$462 + 224 >> 1] = _htons(20081) | 0; //@line 3682
   HEAP16[$462 + 226 >> 1] = _htons(20081) | 0; //@line 3684
   $493 = HEAP32[30897] | 0; //@line 3685
   HEAP16[$493 + 1984 >> 1] = _htons((HEAP32[30880] | 0) + 32256 & 65535) | 0; //@line 3692
   HEAP16[$493 + 1986 >> 1] = _htons(20085) | 0; //@line 3694
   $501 = HEAP32[30897] | 0; //@line 3695
   $$09$i297$i = 2560; //@line 3696
   while (1) {
    $502 = $501 + $$09$i297$i | 0; //@line 3698
    if (!(_memcmp($502, 91757, 6) | 0)) {
     label = 32; //@line 3702
     break;
    }
    $$09$i297$i = $$09$i297$i + 1 | 0; //@line 3705
    if ($$09$i297$i >>> 0 >= 2816) {
     $514 = $501; //@line 3710
     break;
    }
   }
   if ((label | 0) == 32) {
    $507 = $502 + 2 | 0; //@line 3715
    HEAP16[$507 >> 1] = _htons(20081) | 0; //@line 3718
    HEAP16[$507 + 2 >> 1] = _htons(20081) | 0; //@line 3720
    $514 = HEAP32[30897] | 0; //@line 3722
   }
   L51 : do {
    if ((HEAP32[30898] | 0) >>> 0 < 524289) {
     $$09$i300$i = 16384; //@line 3728
     while (1) {
      if (!(_memcmp($514 + $$09$i300$i | 0, 91763, 7) | 0)) {
       $$0292$i = $$09$i300$i; //@line 3734
       break L51;
      }
      $$09$i300$i = $$09$i300$i + 1 | 0; //@line 3737
      if ($$09$i300$i >>> 0 >= 327680) {
       $$0 = 0; //@line 3742
       break;
      }
     }
     STACKTOP = sp; //@line 3746
     return $$0 | 0; //@line 3746
    } else {
     $$09$i303$i = 524288; //@line 3748
     while (1) {
      if (!(_memcmp($514 + $$09$i303$i | 0, 91770, 7) | 0)) {
       $$0292$i = $$09$i303$i; //@line 3754
       break L51;
      }
      $$09$i303$i = $$09$i303$i + 1 | 0; //@line 3757
      if ($$09$i303$i >>> 0 >= 589824) {
       $$0 = 0; //@line 3762
       break;
      }
     }
     STACKTOP = sp; //@line 3766
     return $$0 | 0; //@line 3766
    }
   } while (0);
   $524 = $514 + $$0292$i | 0; //@line 3769
   $526 = $524 + 2 | 0; //@line 3771
   HEAP16[$524 >> 1] = _htons(20081) | 0; //@line 3772
   $528 = $526 + 2 | 0; //@line 3774
   HEAP16[$526 >> 1] = _htons(20081) | 0; //@line 3775
   $530 = $528 + 2 | 0; //@line 3777
   HEAP16[$528 >> 1] = _htons(20081) | 0; //@line 3778
   HEAP16[$530 >> 1] = _htons(20081) | 0; //@line 3781
   $533 = $530 + 2 + 2 | 0; //@line 3782
   HEAP16[$533 >> 1] = _htons(28672) | 0; //@line 3785
   HEAP16[$533 + 2 >> 1] = _htons(20081) | 0; //@line 3787
   $537 = HEAP32[30898] | 0; //@line 3788
   $539 = HEAP32[30897] | 0; //@line 3790
   L61 : do {
    if ($537 >>> 0 < 524289) {
     $$09$i306$i = 16384; //@line 3793
     while (1) {
      if (!(_memcmp($539 + $$09$i306$i | 0, 91777, 5) | 0)) {
       $$1293$ph$i = $$09$i306$i; //@line 3799
       label = 43; //@line 3800
       break L61;
      }
      $$09$i306$i = $$09$i306$i + 1 | 0; //@line 3803
      if ($$09$i306$i >>> 0 >= 327680) {
       $553 = $537; //@line 3808
       $555 = $539; //@line 3808
       break;
      }
     }
    } else {
     $$09$i309$i = 524288; //@line 3813
     while (1) {
      if (!(_memcmp($539 + $$09$i309$i | 0, 91777, 5) | 0)) {
       $$1293$ph$i = $$09$i309$i; //@line 3819
       label = 43; //@line 3820
       break L61;
      }
      $$09$i309$i = $$09$i309$i + 1 | 0; //@line 3823
      if ($$09$i309$i >>> 0 >= 589824) {
       $553 = $537; //@line 3828
       $555 = $539; //@line 3828
       break;
      }
     }
    }
   } while (0);
   if ((label | 0) == 43) {
    HEAP8[$539 + $$1293$ph$i + 4 >> 0] = 96; //@line 3837
    $553 = HEAP32[30898] | 0; //@line 3840
    $555 = HEAP32[30897] | 0; //@line 3840
   }
   L71 : do {
    if ($553 >>> 0 < 524289) {
     $$09$i312$i = 16384; //@line 3845
     while (1) {
      if (!(_memcmp($555 + $$09$i312$i | 0, 91782, 14) | 0)) {
       $$2$i = $$09$i312$i; //@line 3851
       break L71;
      }
      $$09$i312$i = $$09$i312$i + 1 | 0; //@line 3854
      if ($$09$i312$i >>> 0 >= 327680) {
       $$0 = 0; //@line 3859
       break;
      }
     }
     STACKTOP = sp; //@line 3863
     return $$0 | 0; //@line 3863
    } else {
     $$09$i315$i = 524288; //@line 3865
     while (1) {
      if (!(_memcmp($555 + $$09$i315$i | 0, 91782, 14) | 0)) {
       $$2$i = $$09$i315$i; //@line 3871
       break L71;
      }
      $$09$i315$i = $$09$i315$i + 1 | 0; //@line 3874
      if ($$09$i315$i >>> 0 >= 589824) {
       $$0 = 0; //@line 3879
       break;
      }
     }
     STACKTOP = sp; //@line 3883
     return $$0 | 0; //@line 3883
    }
   } while (0);
   HEAP16[$555 + $$2$i + 6 >> 1] = _htons(20081) | 0; //@line 3889
   $568 = HEAP32[30897] | 0; //@line 3890
   $$09$i318$i = 262144; //@line 3891
   while (1) {
    $569 = $568 + $$09$i318$i | 0; //@line 3893
    if (!(_memcmp($569, 91796, 12) | 0)) {
     label = 52; //@line 3897
     break;
    }
    $$09$i318$i = $$09$i318$i + 1 | 0; //@line 3900
    if ($$09$i318$i >>> 0 >= 327680) {
     $578 = $568; //@line 3905
     break;
    }
   }
   if ((label | 0) == 52) {
    HEAP16[$569 >> 1] = _htons(28933) | 0; //@line 3912
    HEAP16[$569 + 2 >> 1] = _htons(20182) | 0; //@line 3914
    $578 = HEAP32[30897] | 0; //@line 3916
   }
   $$09$i321$i = 262144; //@line 3918
   do {
    $577 = $578 + $$09$i321$i | 0; //@line 3920
    if (!(_memcmp($577, 91808, 16) | 0)) {
     label = 56; //@line 3924
     break;
    }
    $$09$i321$i = $$09$i321$i + 1 | 0; //@line 3927
   } while ($$09$i321$i >>> 0 < 327680);
   if ((label | 0) == 56) {
    HEAP16[$577 >> 1] = _htons(28933) | 0; //@line 3938
    HEAP16[$577 + 2 >> 1] = _htons(20182) | 0; //@line 3940
   }
   L93 : do {
    if ((HEAP32[30898] | 0) >>> 0 > 524288) {
     $588 = HEAP32[30897] | 0; //@line 3946
     $$09$i324$i = 524288; //@line 3947
     while (1) {
      $589 = $588 + $$09$i324$i | 0; //@line 3949
      if (!(_memcmp($589, 91824, 12) | 0)) {
       break;
      }
      $$09$i324$i = $$09$i324$i + 1 | 0; //@line 3955
      if ($$09$i324$i >>> 0 >= 589824) {
       break L93;
      }
     }
     HEAP16[$589 >> 1] = _htons(28934) | 0; //@line 3965
     HEAP16[$589 + 2 >> 1] = _htons(20085) | 0; //@line 3967
    }
   } while (0);
   $597 = HEAP32[30897] | 0; //@line 3970
   $599 = (HEAP32[30896] | 0) + 34 | 0; //@line 3972
   $$043$i$i = $597 + (FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($599 >>> 16 << 2) >> 2] >> 2] & 127]($599) | 0) | 0; //@line 3979
   $$048$i$i = -22528; //@line 3979
   $$057$i$i = 0; //@line 3979
   L100 : while (1) {
    $$05673$i$i = 0; //@line 3981
    $$14475$i$i = $$043$i$i; //@line 3981
    $$14974$i$i = $$048$i$i; //@line 3981
    $$15872$i$i = $$057$i$i; //@line 3981
    while (1) {
     $606 = $$14475$i$i + 1 | 0; //@line 3983
     $607 = HEAP8[$$14475$i$i >> 0] | 0; //@line 3984
     $608 = $607 & 255; //@line 3985
     L104 : do {
      switch ($607 << 24 >> 24) {
      case -128:
       {
        $$051$off0$i$i = 0; //@line 3989
        $$245$i$i = $606; //@line 3989
        $$461$i$i = $$15872$i$i; //@line 3989
        break;
       }
      case -1:
       {
        $625 = HEAPU8[$$14475$i$i + 2 >> 0] << 16 | HEAPU8[$606 >> 0] << 24 | HEAPU8[$$14475$i$i + 3 >> 0] << 8 | HEAPU8[$$14475$i$i + 4 >> 0]; //@line 4009
        $$051$off0$i$i = $625; //@line 4011
        $$245$i$i = $$14475$i$i + 5 | 0; //@line 4011
        $$461$i$i = $625; //@line 4011
        break;
       }
      default:
       {
        if (!($608 & 128)) {
         $637 = HEAPU8[$606 >> 0] | $608 << 8; //@line 4022
         $642 = ($637 << 17 >> 16) + $$15872$i$i | 0; //@line 4028
         if (!(($637 << 1 & 65535) << 16 >> 16)) {
          $$5$ph$i$i = 0; //@line 4030
          break L100;
         } else {
          $$051$off0$i$i = $642; //@line 4033
          $$245$i$i = $$14475$i$i + 2 | 0; //@line 4033
          $$461$i$i = $642; //@line 4033
          break L104;
         }
        } else {
         $630 = $608 << 1 & 254; //@line 4038
         $632 = $630 + $$15872$i$i | 0; //@line 4040
         if (!$630) {
          $$5$ph$i$i = 0; //@line 4042
          break L100;
         } else {
          $$051$off0$i$i = $632; //@line 4045
          $$245$i$i = $606; //@line 4045
          $$461$i$i = $632; //@line 4045
          break L104;
         }
        }
       }
      }
     } while (0);
     if ($$14974$i$i << 16 >> 16 == -24493) {
      $$5$ph$i$i = $$051$off0$i$i; //@line 4054
      break L100;
     }
     $645 = $$05673$i$i + 1 | 0; //@line 4058
     if (($645 | 0) < 1024) {
      $$05673$i$i = $645; //@line 4061
      $$14475$i$i = $$245$i$i; //@line 4061
      $$14974$i$i = $$14974$i$i + 1 << 16 >> 16; //@line 4061
      $$15872$i$i = $$461$i$i; //@line 4061
     } else {
      $$043$i$i = $$245$i$i; //@line 4063
      $$048$i$i = -24576; //@line 4063
      $$057$i$i = $$461$i$i; //@line 4063
      continue L100;
     }
    }
   }
   $651 = (_ntohs(HEAP16[(HEAP32[30897] | 0) + $$5$ph$i$i >> 1] | 0) | 0) << 16 >> 16 == 20181; //@line 4072
   $652 = HEAP32[30897] | 0; //@line 4073
   L113 : do {
    if ($651) {
     $$09$i327$i = 720896; //@line 4076
     while (1) {
      if (!(_memcmp($652 + $$09$i327$i | 0, 91836, 8) | 0)) {
       $$3$i = $$09$i327$i; //@line 4082
       break L113;
      }
      $$09$i327$i = $$09$i327$i + 1 | 0; //@line 4085
      if ($$09$i327$i >>> 0 >= 753664) {
       $$0 = 0; //@line 4090
       break;
      }
     }
     STACKTOP = sp; //@line 4094
     return $$0 | 0; //@line 4094
    } else {
     $$3$i = $$5$ph$i$i; //@line 4096
    }
   } while (0);
   $658 = $652 + $$3$i | 0; //@line 4099
   HEAP16[$658 >> 1] = _htons(28932) | 0; //@line 4102
   HEAP16[$658 + 2 >> 1] = _htons(20181) | 0; //@line 4104
   $662 = HEAP32[30897] | 0; //@line 4105
   HEAP16[$662 + 270 >> 1] = _htons(28935) | 0; //@line 4109
   HEAP16[$662 + 272 >> 1] = _htons(20081) | 0; //@line 4111
   $667 = HEAP32[30897] | 0; //@line 4112
   $$09$i330$i = 2560; //@line 4113
   while (1) {
    $668 = $667 + $$09$i330$i | 0; //@line 4115
    if (!(_memcmp($668, 91844, 8) | 0)) {
     break;
    }
    $$09$i330$i = $$09$i330$i + 1 | 0; //@line 4121
    if ($$09$i330$i >>> 0 >= 2688) {
     $$0 = 0; //@line 4126
     label = 246; //@line 4127
     break;
    }
   }
   if ((label | 0) == 246) {
    STACKTOP = sp; //@line 4132
    return $$0 | 0; //@line 4132
   }
   HEAP16[$668 >> 1] = _htons(20085) | 0; //@line 4135
   $674 = HEAP32[30897] | 0; //@line 4136
   $675 = $674 + 16946 | 0; //@line 4137
   $676 = $674 + 16948 | 0; //@line 4138
   if ((_ntohs(HEAP16[$676 >> 1] | 0) | 0) << 16 >> 16 == 20721) {
    $680 = $674 + 16950 | 0; //@line 4143
    if ((_ntohs(HEAP16[$680 >> 1] | 0) | 0) << 16 >> 16 == -24319) {
     HEAP16[$675 >> 1] = _htons(20081) | 0; //@line 4149
     HEAP16[$676 >> 1] = _htons(20081) | 0; //@line 4151
     HEAP16[$680 >> 1] = _htons(20081) | 0; //@line 4154
     HEAP16[$674 + 16952 >> 1] = _htons(20081) | 0; //@line 4157
     HEAP16[$674 + 16954 >> 1] = _htons(20081) | 0; //@line 4159
    }
   }
   $692 = (HEAP32[30897] | 0) + 2496 | 0; //@line 4163
   HEAP16[$692 >> 1] = _htons(20085) | 0; //@line 4165
   $695 = (HEAP32[30897] | 0) + 2464 | 0; //@line 4167
   HEAP16[$695 >> 1] = _htons(20085) | 0; //@line 4169
   $697 = HEAP32[30897] | 0; //@line 4170
   $$09$i333$i = 16384; //@line 4171
   while (1) {
    $698 = $697 + $$09$i333$i | 0; //@line 4173
    if (!(_memcmp($698, 91852, 8) | 0)) {
     label = 83; //@line 4177
     break;
    }
    $$09$i333$i = $$09$i333$i + 1 | 0; //@line 4180
    if ($$09$i333$i >>> 0 >= 20480) {
     $705 = $697; //@line 4185
     break;
    }
   }
   if ((label | 0) == 83) {
    HEAP16[$698 >> 1] = _htons(20182) | 0; //@line 4191
    $705 = HEAP32[30897] | 0; //@line 4193
   }
   HEAP16[$705 + 400 >> 1] = _htons(20081) | 0; //@line 4198
   HEAP16[$705 + 402 >> 1] = _htons(20081) | 0; //@line 4200
   $710 = (HEAP32[30897] | 0) + 40780 | 0; //@line 4202
   HEAP16[$710 >> 1] = _htons(20085) | 0; //@line 4204
   $712 = HEAP32[30897] | 0; //@line 4205
   HEAP16[$712 + 2048 >> 1] = _htons(12796) | 0; //@line 4209
   HEAP16[$712 + 2050 >> 1] = _htons(1e4) | 0; //@line 4212
   HEAP16[$712 + 2052 >> 1] = _htons(3328) | 0; //@line 4215
   HEAP16[$712 + 2054 >> 1] = _htons(12796) | 0; //@line 4218
   HEAP16[$712 + 2056 >> 1] = _htons(1e4) | 0; //@line 4221
   HEAP16[$712 + 2058 >> 1] = _htons(3330) | 0; //@line 4224
   HEAP16[$712 + 2060 >> 1] = _htons(12796) | 0; //@line 4227
   HEAP16[$712 + 2062 >> 1] = _htons(1e4) | 0; //@line 4230
   HEAP16[$712 + 2064 >> 1] = _htons(2852) | 0; //@line 4233
   HEAP16[$712 + 2066 >> 1] = _htons(12796) | 0; //@line 4236
   HEAP16[$712 + 2068 >> 1] = _htons(1e4) | 0; //@line 4239
   HEAP16[$712 + 2070 >> 1] = _htons(3306) | 0; //@line 4242
   HEAP16[$712 + 2072 >> 1] = _htons(20085) | 0; //@line 4244
   $739 = HEAP32[30897] | 0; //@line 4245
   HEAP16[$739 + 45282 >> 1] = _htons(19679) | 0; //@line 4249
   HEAP16[$739 + 45284 >> 1] = _htons(7999) | 0; //@line 4252
   HEAP16[$739 + 45286 >> 1] = _htons(20085) | 0; //@line 4254
   $746 = HEAP32[30897] | 0; //@line 4255
   $$09$i336$i = 262144; //@line 4256
   while (1) {
    $747 = $746 + $$09$i336$i | 0; //@line 4258
    if (!(_memcmp($747, 91860, 10) | 0)) {
     label = 87; //@line 4262
     break;
    }
    $$09$i336$i = $$09$i336$i + 1 | 0; //@line 4265
    if ($$09$i336$i >>> 0 >= 327680) {
     $761 = $746; //@line 4270
     break;
    }
   }
   if ((label | 0) == 87) {
    $752 = $747 + 8 | 0; //@line 4275
    $754 = $752 + 2 | 0; //@line 4277
    HEAP16[$752 >> 1] = _htons(20081) | 0; //@line 4278
    $756 = $754 + 2 | 0; //@line 4280
    HEAP16[$754 >> 1] = _htons(20081) | 0; //@line 4281
    HEAP16[$756 >> 1] = _htons(20081) | 0; //@line 4284
    HEAP16[$756 + 2 >> 1] = _htons(20081) | 0; //@line 4286
    $761 = HEAP32[30897] | 0; //@line 4288
   }
   $$09$i339$i = 16384; //@line 4290
   do {
    $760 = $761 + $$09$i339$i | 0; //@line 4292
    if (!(_memcmp($760, 91870, 8) | 0)) {
     label = 91; //@line 4296
     break;
    }
    $$09$i339$i = $$09$i339$i + 1 | 0; //@line 4299
   } while ($$09$i339$i >>> 0 < 20480);
   if ((label | 0) == 91) {
    $766 = $760 + 6 | 0; //@line 4308
    $768 = $766 + 2 | 0; //@line 4310
    HEAP16[$766 >> 1] = _htons(28672) | 0; //@line 4311
    HEAP16[$768 >> 1] = _htons(-20416) | 0; //@line 4314
    HEAP16[$768 + 2 >> 1] = _htons(20182) | 0; //@line 4316
   }
   if (!(__Z14InstallSlotROMv() | 0)) {
    $$0 = 0; //@line 4320
    STACKTOP = sp; //@line 4321
    return $$0 | 0; //@line 4321
   }
   $773 = HEAP32[30897] | 0; //@line 4323
   $$09$i342$i = 20480; //@line 4324
   while (1) {
    $774 = $773 + $$09$i342$i | 0; //@line 4326
    if (!(_memcmp($774, 91878, 8) | 0)) {
     label = 96; //@line 4330
     break;
    }
    $$09$i342$i = $$09$i342$i + 1 | 0; //@line 4333
    if ($$09$i342$i >>> 0 >= 24576) {
     $786 = $773; //@line 4338
     break;
    }
   }
   if ((label | 0) == 96) {
    $779 = $774 + 6 | 0; //@line 4343
    $781 = $779 + 2 | 0; //@line 4345
    HEAP16[$779 >> 1] = _htons(20081) | 0; //@line 4346
    HEAP16[$781 >> 1] = _htons(20081) | 0; //@line 4349
    HEAP16[$781 + 2 >> 1] = _htons(20081) | 0; //@line 4351
    $786 = HEAP32[30897] | 0; //@line 4353
   }
   if (!(_memcmp($786 + 550 | 0, 91886, 2) | 0)) {
    $$09$i345$lcssa$i = 550; //@line 4359
   } else {
    if (!(_memcmp($786 + 551 | 0, 91886, 2) | 0)) {
     $$09$i345$lcssa$i = 551; //@line 4365
    } else {
     if (!(_memcmp($786 + 552 | 0, 91886, 2) | 0)) {
      $$09$i345$lcssa$i = 552; //@line 4371
     } else {
      if (!(_memcmp($786 + 553 | 0, 91886, 2) | 0)) {
       $$09$i345$lcssa$i = 553; //@line 4377
      } else {
       $$0 = 0; //@line 4379
       STACKTOP = sp; //@line 4380
       return $$0 | 0; //@line 4380
      }
     }
    }
   }
   $792 = $786 + $$09$i345$lcssa$i | 0; //@line 4385
   $794 = $792 + 2 | 0; //@line 4387
   HEAP16[$792 >> 1] = _htons(20081) | 0; //@line 4388
   $796 = $794 + 2 | 0; //@line 4390
   HEAP16[$794 >> 1] = _htons(20081) | 0; //@line 4391
   $798 = $796 + 2 | 0; //@line 4393
   HEAP16[$796 >> 1] = _htons(20081) | 0; //@line 4394
   HEAP16[$798 >> 1] = _htons(20081) | 0; //@line 4397
   HEAP16[$798 + 2 >> 1] = _htons(20081) | 0; //@line 4399
   $802 = HEAP32[30897] | 0; //@line 4400
   $803 = $802 + 560 | 0; //@line 4401
   do {
    if (!(_memcmp($803, 91886, 2) | 0)) {
     $$4$i = 560; //@line 4406
     $810 = $802; //@line 4406
    } else {
     if (!(_memcmp($802 + 561 | 0, 91886, 2) | 0)) {
      $$4$i = 561; //@line 4412
      $810 = $802; //@line 4412
     } else {
      if (!(_memcmp($802 + 562 | 0, 91886, 2) | 0)) {
       $$4$i = 562; //@line 4418
       $810 = $802; //@line 4418
      } else {
       if (!(_memcmp($802 + 563 | 0, 91886, 2) | 0)) {
        $$4$i = 563; //@line 4424
        $810 = $802; //@line 4424
       } else {
        if ((_ntohs(HEAP16[$803 >> 1] | 0) | 0) << 16 >> 16 == 24832) {
         $$4$i = 560; //@line 4431
         $810 = HEAP32[30897] | 0; //@line 4431
         break;
        } else {
         $$0 = 0; //@line 4434
         STACKTOP = sp; //@line 4435
         return $$0 | 0; //@line 4435
        }
       }
      }
     }
    }
   } while (0);
   $809 = $810 + $$4$i | 0; //@line 4442
   $812 = $809 + 2 | 0; //@line 4444
   HEAP16[$809 >> 1] = _htons(20081) | 0; //@line 4445
   $814 = $812 + 2 | 0; //@line 4447
   HEAP16[$812 >> 1] = _htons(20081) | 0; //@line 4448
   $816 = $814 + 2 | 0; //@line 4450
   HEAP16[$814 >> 1] = _htons(20081) | 0; //@line 4451
   HEAP16[$816 >> 1] = _htons(20081) | 0; //@line 4454
   HEAP16[$816 + 2 >> 1] = _htons(20081) | 0; //@line 4456
   $820 = HEAP32[30897] | 0; //@line 4457
   HEAP16[$820 + 1168 >> 1] = _htons(8248) | 0; //@line 4461
   HEAP16[$820 + 1170 >> 1] = _htons(268) | 0; //@line 4464
   HEAP16[$820 + 1172 >> 1] = _htons(-12104) | 0; //@line 4467
   HEAP16[$820 + 1174 >> 1] = _htons(678) | 0; //@line 4470
   HEAP16[$820 + 1176 >> 1] = _htons(-7544) | 0; //@line 4473
   HEAP16[$820 + 1178 >> 1] = _htons(2176) | 0; //@line 4476
   HEAP16[$820 + 1180 >> 1] = _htons(0) | 0; //@line 4479
   HEAP16[$820 + 1182 >> 1] = _htons(1088) | 0; //@line 4482
   HEAP16[$820 + 1184 >> 1] = _htons(1024) | 0; //@line 4485
   HEAP16[$820 + 1186 >> 1] = _htons(8256) | 0; //@line 4488
   HEAP16[$820 + 1188 >> 1] = _htons(28937) | 0; //@line 4491
   HEAP16[$820 + 1190 >> 1] = _htons(20085) | 0; //@line 4493
   $845 = HEAP32[30897] | 0; //@line 4494
   $$09$i351$i = 311296; //@line 4495
   while (1) {
    $846 = $845 + $$09$i351$i | 0; //@line 4497
    if (!(_memcmp($846, 91888, 20) | 0)) {
     label = 104; //@line 4501
     break;
    }
    $$09$i351$i = $$09$i351$i + 1 | 0; //@line 4504
    if ($$09$i351$i >>> 0 >= 311424) {
     $856 = $845; //@line 4509
     break;
    }
   }
   if ((label | 0) == 104) {
    $851 = $846 + 16 | 0; //@line 4514
    HEAP16[$851 >> 1] = _htons(20081) | 0; //@line 4517
    HEAP16[$851 + 2 >> 1] = _htons(20081) | 0; //@line 4519
    $856 = HEAP32[30897] | 0; //@line 4521
   }
   HEAP16[$856 + 4418 >> 1] = _htons(28938) | 0; //@line 4525
   $858 = HEAP32[30897] | 0; //@line 4526
   HEAP16[$858 + 4420 >> 1] = _htons(20081) | 0; //@line 4530
   HEAP16[$858 + 4422 >> 1] = _htons(20081) | 0; //@line 4533
   HEAP16[$858 + 4424 >> 1] = _htons(20081) | 0; //@line 4536
   HEAP16[$858 + 4426 >> 1] = _htons(20081) | 0; //@line 4538
   HEAP16[$858 + 4432 >> 1] = _htons(20081) | 0; //@line 4541
   $869 = HEAP32[30897] | 0; //@line 4542
   $870 = $869 + 43176 | 0; //@line 4543
   if (!(HEAP16[$870 >> 1] | 0)) {
    HEAP16[$869 + 732266 >> 1] = _htons(20081) | 0; //@line 4550
    HEAP16[$869 + 732268 >> 1] = _htons(20081) | 0; //@line 4553
    HEAP16[$869 + 732270 >> 1] = _htons(20081) | 0; //@line 4555
    $879 = HEAP32[30897] | 0; //@line 4556
    HEAP16[$879 + 732462 >> 1] = _htons(20081) | 0; //@line 4560
    HEAP16[$879 + 732464 >> 1] = _htons(20081) | 0; //@line 4563
    HEAP16[$879 + 732466 >> 1] = _htons(20081) | 0; //@line 4566
    HEAP16[$879 + 732468 >> 1] = _htons(20081) | 0; //@line 4569
    HEAP16[$879 + 732470 >> 1] = _htons(20081) | 0; //@line 4572
    HEAP16[$879 + 732472 >> 1] = _htons(20081) | 0; //@line 4575
    HEAP16[$879 + 732474 >> 1] = _htons(20081) | 0; //@line 4578
    HEAP16[$879 + 732476 >> 1] = _htons(20081) | 0; //@line 4581
    HEAP16[$879 + 732478 >> 1] = _htons(20081) | 0; //@line 4584
    HEAP16[$879 + 732480 >> 1] = _htons(20081) | 0; //@line 4587
    HEAP16[$879 + 732482 >> 1] = _htons(20081) | 0; //@line 4590
    HEAP16[$879 + 732484 >> 1] = _htons(20081) | 0; //@line 4592
    $$sink$i = _htons(20081) | 0; //@line 4596
    $$sink295$in$i = $879 + 732490 | 0; //@line 4596
    $$sink296$i = $879 + 732492 | 0; //@line 4596
   } else {
    HEAP16[$870 >> 1] = _htons(20081) | 0; //@line 4600
    HEAP16[$869 + 43178 >> 1] = _htons(20081) | 0; //@line 4603
    HEAP16[$869 + 43180 >> 1] = _htons(20081) | 0; //@line 4605
    $912 = HEAP32[30897] | 0; //@line 4606
    HEAP16[$912 + 42594 >> 1] = _htons(20081) | 0; //@line 4610
    HEAP16[$912 + 42596 >> 1] = _htons(20081) | 0; //@line 4613
    HEAP16[$912 + 42598 >> 1] = _htons(20081) | 0; //@line 4616
    HEAP16[$912 + 42600 >> 1] = _htons(20081) | 0; //@line 4619
    HEAP16[$912 + 42602 >> 1] = _htons(20081) | 0; //@line 4621
    $$sink$i = _htons(20081) | 0; //@line 4625
    $$sink295$in$i = $912 + 42608 | 0; //@line 4625
    $$sink296$i = $912 + 42610 | 0; //@line 4625
   }
   HEAP16[$$sink295$in$i >> 1] = $$sink$i; //@line 4627
   HEAP16[$$sink296$i >> 1] = _htons(20081) | 0; //@line 4629
   $927 = HEAP32[30897] | 0; //@line 4630
   if (!(_memcmp($927 + 750 | 0, 91886, 2) | 0)) {
    $$09$i354$lcssa$i = 750; //@line 4635
   } else {
    if (!(_memcmp($927 + 751 | 0, 91886, 2) | 0)) {
     $$09$i354$lcssa$i = 751; //@line 4641
    } else {
     if (!(_memcmp($927 + 752 | 0, 91886, 2) | 0)) {
      $$09$i354$lcssa$i = 752; //@line 4647
     } else {
      if (!(_memcmp($927 + 753 | 0, 91886, 2) | 0)) {
       $$09$i354$lcssa$i = 753; //@line 4653
      } else {
       $$0 = 0; //@line 4655
       STACKTOP = sp; //@line 4656
       return $$0 | 0; //@line 4656
      }
     }
    }
   }
   $934 = $927 + $$09$i354$lcssa$i | 0; //@line 4661
   $936 = $934 + 2 | 0; //@line 4663
   HEAP16[$934 >> 1] = _htons(20081) | 0; //@line 4664
   $938 = $936 + 2 | 0; //@line 4666
   HEAP16[$936 >> 1] = _htons(20081) | 0; //@line 4667
   $940 = $938 + 2 | 0; //@line 4669
   HEAP16[$938 >> 1] = _htons(20081) | 0; //@line 4670
   HEAP16[$940 >> 1] = _htons(20081) | 0; //@line 4673
   HEAP16[$940 + 2 >> 1] = _htons(20081) | 0; //@line 4675
   $944 = HEAP32[30897] | 0; //@line 4676
   HEAP16[$944 + 23416 >> 1] = _htons(20081) | 0; //@line 4680
   HEAP16[$944 + 23418 >> 1] = _htons(20081) | 0; //@line 4683
   HEAP16[$944 + 23420 >> 1] = _htons(9217) | 0; //@line 4686
   HEAP16[$944 + 23422 >> 1] = _htons(24670) | 0; //@line 4688
   L187 : do {
    if ((HEAP32[30898] | 0) >>> 0 > 524288) {
     $955 = HEAP32[30897] | 0; //@line 4693
     $$09$i357$i = 573440; //@line 4694
     while (1) {
      $956 = $955 + $$09$i357$i | 0; //@line 4696
      if (!(_memcmp($956, 91908, 8) | 0)) {
       break;
      }
      $$09$i357$i = $$09$i357$i + 1 | 0; //@line 4702
      if ($$09$i357$i >>> 0 >= 577536) {
       break L187;
      }
     }
     HEAP16[$956 >> 1] = _htons(9217) | 0; //@line 4712
     HEAP16[$956 + 2 >> 1] = _htons(20085) | 0; //@line 4714
    }
   } while (0);
   $964 = HEAP32[30897] | 0; //@line 4717
   $$09$i360$i = 40960; //@line 4718
   while (1) {
    $965 = $964 + $$09$i360$i | 0; //@line 4720
    if (!(_memcmp($965, 91916, 8) | 0)) {
     break;
    }
    $$09$i360$i = $$09$i360$i + 1 | 0; //@line 4726
    if ($$09$i360$i >>> 0 >= 41984) {
     $$0 = 0; //@line 4731
     label = 246; //@line 4732
     break;
    }
   }
   if ((label | 0) == 246) {
    STACKTOP = sp; //@line 4737
    return $$0 | 0; //@line 4737
   }
   HEAP16[$965 + 4 >> 1] = _htons(20085) | 0; //@line 4741
   $972 = HEAP32[30897] | 0; //@line 4742
   $$09$i363$i = 262144; //@line 4743
   do {
    $973 = $972 + $$09$i363$i | 0; //@line 4745
    if (!(_memcmp($973, 91924, 12) | 0)) {
     label = 121; //@line 4749
     break;
    }
    $$09$i363$i = $$09$i363$i + 1 | 0; //@line 4752
   } while ($$09$i363$i >>> 0 < 278528);
   if ((label | 0) == 121) {
    HEAP16[$973 + 4 >> 1] = _htons(20085) | 0; //@line 4763
   }
   $980 = HEAP32[30898] | 0; //@line 4765
   L206 : do {
    if ($980 >>> 0 > 524288) {
     $982 = HEAP32[30897] | 0; //@line 4769
     $$09$i366$i = 552960; //@line 4770
     while (1) {
      $983 = $982 + $$09$i366$i | 0; //@line 4772
      if (!(_memcmp($983, 91936, 10) | 0)) {
       label = 126; //@line 4776
       break;
      }
      $$09$i366$i = $$09$i366$i + 1 | 0; //@line 4779
      if ($$09$i366$i >>> 0 >= 555008) {
       $995 = $980; //@line 4784
       break;
      }
     }
     if ((label | 0) == 126) {
      $988 = $983 + 4 | 0; //@line 4789
      $990 = $988 + 2 | 0; //@line 4791
      HEAP16[$988 >> 1] = _htons(28976) | 0; //@line 4792
      HEAP16[$990 >> 1] = _htons(28672) | 0; //@line 4795
      HEAP16[$990 + 2 >> 1] = _htons(20085) | 0; //@line 4797
      $995 = HEAP32[30898] | 0; //@line 4799
     }
     if ($995 | 0) {
      $996 = HEAP32[30897] | 0; //@line 4803
      $$09$i369$i = 0; //@line 4804
      while (1) {
       $997 = $996 + $$09$i369$i | 0; //@line 4806
       if (!(_memcmp($997, 91946, 12) | 0)) {
        break;
       }
       $1000 = $$09$i369$i + 1 | 0; //@line 4812
       if ($1000 >>> 0 < $995 >>> 0) {
        $$09$i369$i = $1000; //@line 4815
       } else {
        break L206;
       }
      }
      if ($$09$i369$i | 0) {
       $1003 = $997 + 8 | 0; //@line 4822
       $1005 = $1003 + 2 | 0; //@line 4824
       HEAP16[$1003 >> 1] = _htons(20081) | 0; //@line 4825
       $1007 = $1005 + 2 | 0; //@line 4827
       HEAP16[$1005 >> 1] = _htons(-2824) | 0; //@line 4828
       $1009 = $1007 + 2 | 0; //@line 4830
       HEAP16[$1007 >> 1] = _htons(20081) | 0; //@line 4831
       HEAP16[$1009 >> 1] = _htons(28672) | 0; //@line 4834
       HEAP16[$1009 + 2 >> 1] = _htons(20085) | 0; //@line 4836
      }
     }
    }
   } while (0);
   $1013 = HEAP32[30897] | 0; //@line 4841
   $$09$i373$i = 323840; //@line 4842
   do {
    $1014 = $1013 + $$09$i373$i | 0; //@line 4844
    if (!(_memcmp($1014, 91958, 12) | 0)) {
     label = 136; //@line 4848
     break;
    }
    $$09$i373$i = $$09$i373$i + 1 | 0; //@line 4851
   } while ($$09$i373$i >>> 0 < 323968);
   if ((label | 0) == 136) {
    HEAP16[$1014 + 10 >> 1] = _htons(20081) | 0; //@line 4862
   }
   $1021 = HEAP32[30896] | 0; //@line 4864
   $1022 = $1021 + 26 | 0; //@line 4865
   $1028 = (FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($1022 >>> 16 << 2) >> 2] >> 2] & 127]($1022) | 0) + $1021 | 0; //@line 4871
   $1033 = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($1028 >>> 16 << 2) >> 2] >> 2] & 127]($1028) | 0; //@line 4876
   HEAP32[28209] = $1033; //@line 4877
   $1036 = $1033; //@line 4878
   do {
    $1035 = (HEAP32[30896] | 0) + $1036 | 0; //@line 4881
    $1037 = $1035 + 12 | 0; //@line 4882
    $1042 = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($1037 >>> 16 << 2) >> 2] >> 2] & 127]($1037) | 0; //@line 4887
    $1043 = $1035 + 16 | 0; //@line 4888
    $1048 = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($1043 >>> 16 << 2) >> 2] >> 2] & 127]($1043) | 0; //@line 4893
    $1049 = $1035 + 20 | 0; //@line 4894
    if (($1048 | 0) == 1146246738 & ((FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($1049 >>> 16 << 2) >> 2] | 0) + 4 >> 2] & 127]($1049) | 0) & 65535 | 0) == 51) {
     label = 140; //@line 4906
     break;
    }
    $1058 = $1035 + 8 | 0; //@line 4909
    $1036 = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($1058 >>> 16 << 2) >> 2] >> 2] & 127]($1058) | 0; //@line 4914
    HEAP32[28209] = $1036; //@line 4915
   } while (($1036 | 0) != 0);
   L230 : do {
    if ((label | 0) == 140) {
     if ($1042 | 0) {
      $1066 = $1042 + 65536 | 0; //@line 4927
      if ($1042 >>> 0 < 4294901760) {
       $1068 = HEAP32[30897] | 0; //@line 4930
       $$09$i379$i = $1042; //@line 4931
       while (1) {
        $1069 = $1068 + $$09$i379$i | 0; //@line 4933
        if (!(_memcmp($1069, 91970, 12) | 0)) {
         break;
        }
        $1072 = $$09$i379$i + 1 | 0; //@line 4939
        if ($1072 >>> 0 < $1066 >>> 0) {
         $$09$i379$i = $1072; //@line 4942
        } else {
         break L230;
        }
       }
       if ($$09$i379$i | 0) {
        $1075 = $1069 + 8 | 0; //@line 4949
        HEAP16[$1075 >> 1] = 0; //@line 4951
        HEAP16[$1075 + 2 >> 1] = 0; //@line 4952
       }
      }
     }
    }
   } while (0);
   $1077 = HEAP32[30896] | 0; //@line 4958
   $1078 = $1077 + 26 | 0; //@line 4959
   $1084 = (FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($1078 >>> 16 << 2) >> 2] >> 2] & 127]($1078) | 0) + $1077 | 0; //@line 4965
   $1089 = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($1084 >>> 16 << 2) >> 2] >> 2] & 127]($1084) | 0; //@line 4970
   HEAP32[28209] = $1089; //@line 4971
   $1092 = $1089; //@line 4972
   while (1) {
    $1091 = (HEAP32[30896] | 0) + $1092 | 0; //@line 4975
    $1093 = $1091 + 12 | 0; //@line 4976
    $1098 = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($1093 >>> 16 << 2) >> 2] >> 2] & 127]($1093) | 0; //@line 4981
    $1099 = $1091 + 16 | 0; //@line 4982
    $1104 = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($1099 >>> 16 << 2) >> 2] >> 2] & 127]($1099) | 0; //@line 4987
    $1105 = $1091 + 20 | 0; //@line 4988
    if (($1104 | 0) == 1146246738 & ((FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($1105 >>> 16 << 2) >> 2] | 0) + 4 >> 2] & 127]($1105) | 0) & 65535 | 0) == 4) {
     $$2$i383$i = $1098; //@line 5000
     break;
    }
    $1114 = $1091 + 8 | 0; //@line 5003
    $1092 = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($1114 >>> 16 << 2) >> 2] >> 2] & 127]($1114) | 0; //@line 5008
    HEAP32[28209] = $1092; //@line 5009
    if (!$1092) {
     $$2$i383$i = 0; //@line 5012
     break;
    }
   }
   HEAP32[28207] = $$2$i383$i; //@line 5018
   dest = (HEAP32[30897] | 0) + $$2$i383$i | 0; //@line 5021
   src = 91029; //@line 5021
   stop = dest + 86 | 0; //@line 5021
   do {
    HEAP8[dest >> 0] = HEAP8[src >> 0] | 0; //@line 5021
    dest = dest + 1 | 0; //@line 5021
    src = src + 1 | 0; //@line 5021
   } while ((dest | 0) < (stop | 0));
   dest = (HEAP32[30897] | 0) + $$2$i383$i + 256 | 0; //@line 5025
   src = 91115; //@line 5025
   stop = dest + 86 | 0; //@line 5025
   do {
    HEAP8[dest >> 0] = HEAP8[src >> 0] | 0; //@line 5025
    dest = dest + 1 | 0; //@line 5025
    src = src + 1 | 0; //@line 5025
   } while ((dest | 0) < (stop | 0));
   dest = (HEAP32[30897] | 0) + $$2$i383$i + 512 | 0; //@line 5029
   src = 91201; //@line 5029
   stop = dest + 90 | 0; //@line 5029
   do {
    HEAP8[dest >> 0] = HEAP8[src >> 0] | 0; //@line 5029
    dest = dest + 1 | 0; //@line 5029
    src = src + 1 | 0; //@line 5029
   } while ((dest | 0) < (stop | 0));
   HEAP32[28483] = $$2$i383$i + 1024 + (HEAP32[30896] | 0); //@line 5033
   _memcpy((HEAP32[30897] | 0) + $$2$i383$i + 1024 | 0, 93682, 258) | 0; //@line 5037
   HEAP32[28484] = $$2$i383$i + 1536 + (HEAP32[30896] | 0); //@line 5041
   _memcpy((HEAP32[30897] | 0) + $$2$i383$i + 1536 | 0, 93940, 258) | 0; //@line 5045
   HEAP32[28488] = $$2$i383$i + 2048 + (HEAP32[30896] | 0); //@line 5049
   _memcpy((HEAP32[30897] | 0) + $$2$i383$i + 2048 | 0, 94272, 258) | 0; //@line 5053
   HEAP32[28492] = $$2$i383$i + 2560 + (HEAP32[30896] | 0); //@line 5057
   _memcpy((HEAP32[30897] | 0) + $$2$i383$i + 2560 | 0, 94883, 258) | 0; //@line 5061
   $1153 = HEAP32[30896] | 0; //@line 5062
   $1154 = $1153 + 26 | 0; //@line 5063
   $1160 = (FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($1154 >>> 16 << 2) >> 2] >> 2] & 127]($1154) | 0) + $1153 | 0; //@line 5069
   $1165 = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($1160 >>> 16 << 2) >> 2] >> 2] & 127]($1160) | 0; //@line 5074
   HEAP32[28209] = $1165; //@line 5075
   $1168 = $1165; //@line 5076
   while (1) {
    $1167 = (HEAP32[30896] | 0) + $1168 | 0; //@line 5079
    $1169 = $1167 + 12 | 0; //@line 5080
    $1174 = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($1169 >>> 16 << 2) >> 2] >> 2] & 127]($1169) | 0; //@line 5085
    $1175 = $1167 + 16 | 0; //@line 5086
    $1180 = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($1175 >>> 16 << 2) >> 2] >> 2] & 127]($1175) | 0; //@line 5091
    $1181 = $1167 + 20 | 0; //@line 5092
    if (($1180 | 0) == 1397051972 & ((FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($1181 >>> 16 << 2) >> 2] | 0) + 4 >> 2] & 127]($1181) | 0) & 65535 | 0) == 0) {
     $$2$i387$i = $1174; //@line 5104
     break;
    }
    $1190 = $1167 + 8 | 0; //@line 5107
    $1168 = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($1190 >>> 16 << 2) >> 2] >> 2] & 127]($1190) | 0; //@line 5112
    HEAP32[28209] = $1168; //@line 5113
    if (!$1168) {
     $$2$i387$i = 0; //@line 5116
     break;
    }
   }
   HEAP32[28208] = $$2$i387$i; //@line 5122
   $1199 = (HEAP32[30897] | 0) + $$2$i387$i + 12 | 0; //@line 5125
   HEAP16[$1199 >> 1] = _htons(28939) | 0; //@line 5128
   HEAP16[$1199 + 2 >> 1] = _htons(20085) | 0; //@line 5130
   $1204 = HEAP32[28208] | 0; //@line 5132
   dest = (HEAP32[30897] | 0) + $1204 + 256 | 0; //@line 5135
   src = 91291; //@line 5135
   stop = dest + 98 | 0; //@line 5135
   do {
    HEAP8[dest >> 0] = HEAP8[src >> 0] | 0; //@line 5135
    dest = dest + 1 | 0; //@line 5135
    src = src + 1 | 0; //@line 5135
   } while ((dest | 0) < (stop | 0));
   dest = (HEAP32[30897] | 0) + $1204 + 512 | 0; //@line 5139
   src = 91389; //@line 5139
   stop = dest + 100 | 0; //@line 5139
   do {
    HEAP8[dest >> 0] = HEAP8[src >> 0] | 0; //@line 5139
    dest = dest + 1 | 0; //@line 5139
    src = src + 1 | 0; //@line 5139
   } while ((dest | 0) < (stop | 0));
   dest = (HEAP32[30897] | 0) + $1204 + 768 | 0; //@line 5143
   src = 91489; //@line 5143
   stop = dest + 98 | 0; //@line 5143
   do {
    HEAP8[dest >> 0] = HEAP8[src >> 0] | 0; //@line 5143
    dest = dest + 1 | 0; //@line 5143
    src = src + 1 | 0; //@line 5143
   } while ((dest | 0) < (stop | 0));
   dest = (HEAP32[30897] | 0) + $1204 + 1024 | 0; //@line 5147
   src = 91587; //@line 5147
   stop = dest + 100 | 0; //@line 5147
   do {
    HEAP8[dest >> 0] = HEAP8[src >> 0] | 0; //@line 5147
    dest = dest + 1 | 0; //@line 5147
    src = src + 1 | 0; //@line 5147
   } while ((dest | 0) < (stop | 0));
   $1216 = HEAP32[30897] | 0; //@line 5148
   $1218 = (HEAP32[30896] | 0) + 34 | 0; //@line 5150
   $$043$i392$i = $1216 + (FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($1218 >>> 16 << 2) >> 2] >> 2] & 127]($1218) | 0) | 0; //@line 5157
   $$048$i391$i = -22528; //@line 5157
   $$057$i390$i = 0; //@line 5157
   L248 : while (1) {
    $$05673$i396$i = 0; //@line 5159
    $$14475$i394$i = $$043$i392$i; //@line 5159
    $$14974$i395$i = $$048$i391$i; //@line 5159
    $$15872$i397$i = $$057$i390$i; //@line 5159
    while (1) {
     $1225 = $$14475$i394$i + 1 | 0; //@line 5161
     $1226 = HEAP8[$$14475$i394$i >> 0] | 0; //@line 5162
     $1227 = $1226 & 255; //@line 5163
     L252 : do {
      switch ($1226 << 24 >> 24) {
      case -128:
       {
        $$051$off0$i400$i = 0; //@line 5167
        $$245$i401$i = $1225; //@line 5167
        $$461$i399$i = $$15872$i397$i; //@line 5167
        break;
       }
      case -1:
       {
        $1244 = HEAPU8[$$14475$i394$i + 2 >> 0] << 16 | HEAPU8[$1225 >> 0] << 24 | HEAPU8[$$14475$i394$i + 3 >> 0] << 8 | HEAPU8[$$14475$i394$i + 4 >> 0]; //@line 5187
        $$051$off0$i400$i = $1244; //@line 5189
        $$245$i401$i = $$14475$i394$i + 5 | 0; //@line 5189
        $$461$i399$i = $1244; //@line 5189
        break;
       }
      default:
       {
        if (!($1227 & 128)) {
         $1256 = HEAPU8[$1225 >> 0] | $1227 << 8; //@line 5200
         $1261 = ($1256 << 17 >> 16) + $$15872$i397$i | 0; //@line 5206
         if (!(($1256 << 1 & 65535) << 16 >> 16)) {
          $$5$ph$i403$i = 0; //@line 5208
          break L248;
         } else {
          $$051$off0$i400$i = $1261; //@line 5211
          $$245$i401$i = $$14475$i394$i + 2 | 0; //@line 5211
          $$461$i399$i = $1261; //@line 5211
          break L252;
         }
        } else {
         $1249 = $1227 << 1 & 254; //@line 5216
         $1251 = $1249 + $$15872$i397$i | 0; //@line 5218
         if (!$1249) {
          $$5$ph$i403$i = 0; //@line 5220
          break L248;
         } else {
          $$051$off0$i400$i = $1251; //@line 5223
          $$245$i401$i = $1225; //@line 5223
          $$461$i399$i = $1251; //@line 5223
          break L252;
         }
        }
       }
      }
     } while (0);
     if ($$14974$i395$i << 16 >> 16 == -24452) {
      $$5$ph$i403$i = $$051$off0$i400$i; //@line 5232
      break L248;
     }
     $1264 = $$05673$i396$i + 1 | 0; //@line 5236
     if (($1264 | 0) < 1024) {
      $$05673$i396$i = $1264; //@line 5239
      $$14475$i394$i = $$245$i401$i; //@line 5239
      $$14974$i395$i = $$14974$i395$i + 1 << 16 >> 16; //@line 5239
      $$15872$i397$i = $$461$i399$i; //@line 5239
     } else {
      $$043$i392$i = $$245$i401$i; //@line 5241
      $$048$i391$i = -24576; //@line 5241
      $$057$i390$i = $$461$i399$i; //@line 5241
      continue L248;
     }
    }
   }
   dest = $1216 + $$5$ph$i403$i | 0; //@line 5247
   src = 91687; //@line 5247
   stop = dest + 54 | 0; //@line 5247
   do {
    HEAP8[dest >> 0] = HEAP8[src >> 0] | 0; //@line 5247
    dest = dest + 1 | 0; //@line 5247
    src = src + 1 | 0; //@line 5247
   } while ((dest | 0) < (stop | 0));
   $1267 = HEAP32[30897] | 0; //@line 5248
   $1269 = (HEAP32[30896] | 0) + 34 | 0; //@line 5250
   $$043$i407$i = $1267 + (FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($1269 >>> 16 << 2) >> 2] >> 2] & 127]($1269) | 0) | 0; //@line 5257
   $$048$i406$i = -22528; //@line 5257
   $$057$i405$i = 0; //@line 5257
   L261 : while (1) {
    $$05673$i411$i = 0; //@line 5259
    $$14475$i409$i = $$043$i407$i; //@line 5259
    $$14974$i410$i = $$048$i406$i; //@line 5259
    $$15872$i412$i = $$057$i405$i; //@line 5259
    while (1) {
     $1276 = $$14475$i409$i + 1 | 0; //@line 5261
     $1277 = HEAP8[$$14475$i409$i >> 0] | 0; //@line 5262
     $1278 = $1277 & 255; //@line 5263
     L265 : do {
      switch ($1277 << 24 >> 24) {
      case -128:
       {
        $$051$off0$i415$i = 0; //@line 5267
        $$245$i416$i = $1276; //@line 5267
        $$461$i414$i = $$15872$i412$i; //@line 5267
        break;
       }
      case -1:
       {
        $1295 = HEAPU8[$$14475$i409$i + 2 >> 0] << 16 | HEAPU8[$1276 >> 0] << 24 | HEAPU8[$$14475$i409$i + 3 >> 0] << 8 | HEAPU8[$$14475$i409$i + 4 >> 0]; //@line 5287
        $$051$off0$i415$i = $1295; //@line 5289
        $$245$i416$i = $$14475$i409$i + 5 | 0; //@line 5289
        $$461$i414$i = $1295; //@line 5289
        break;
       }
      default:
       {
        if (!($1278 & 128)) {
         $1307 = HEAPU8[$1276 >> 0] | $1278 << 8; //@line 5300
         $1312 = ($1307 << 17 >> 16) + $$15872$i412$i | 0; //@line 5306
         if (!(($1307 << 1 & 65535) << 16 >> 16)) {
          $$5$ph$i418$i = 0; //@line 5308
          break L261;
         } else {
          $$051$off0$i415$i = $1312; //@line 5311
          $$245$i416$i = $$14475$i409$i + 2 | 0; //@line 5311
          $$461$i414$i = $1312; //@line 5311
          break L265;
         }
        } else {
         $1300 = $1278 << 1 & 254; //@line 5316
         $1302 = $1300 + $$15872$i412$i | 0; //@line 5318
         if (!$1300) {
          $$5$ph$i418$i = 0; //@line 5320
          break L261;
         } else {
          $$051$off0$i415$i = $1302; //@line 5323
          $$245$i416$i = $1276; //@line 5323
          $$461$i414$i = $1302; //@line 5323
          break L265;
         }
        }
       }
      }
     } while (0);
     if ($$14974$i410$i << 16 >> 16 == -24488) {
      $$5$ph$i418$i = $$051$off0$i415$i; //@line 5332
      break L261;
     }
     $1315 = $$05673$i411$i + 1 | 0; //@line 5336
     if (($1315 | 0) < 1024) {
      $$05673$i411$i = $1315; //@line 5339
      $$14475$i409$i = $$245$i416$i; //@line 5339
      $$14974$i410$i = $$14974$i410$i + 1 << 16 >> 16; //@line 5339
      $$15872$i412$i = $$461$i414$i; //@line 5339
     } else {
      $$043$i407$i = $$245$i416$i; //@line 5341
      $$048$i406$i = -24576; //@line 5341
      $$057$i405$i = $$461$i414$i; //@line 5341
      continue L261;
     }
    }
   }
   $1317 = $1267 + $$5$ph$i418$i | 0; //@line 5346
   HEAP16[$1317 >> 1] = _htons(28964) | 0; //@line 5349
   HEAP16[$1317 + 2 >> 1] = _htons(20085) | 0; //@line 5351
   $1321 = HEAP32[30897] | 0; //@line 5352
   $1323 = (HEAP32[30896] | 0) + 34 | 0; //@line 5354
   $$043$i422$i = $1321 + (FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($1323 >>> 16 << 2) >> 2] >> 2] & 127]($1323) | 0) | 0; //@line 5361
   $$048$i421$i = -22528; //@line 5361
   $$057$i420$i = 0; //@line 5361
   L274 : while (1) {
    $$05673$i426$i = 0; //@line 5363
    $$14475$i424$i = $$043$i422$i; //@line 5363
    $$14974$i425$i = $$048$i421$i; //@line 5363
    $$15872$i427$i = $$057$i420$i; //@line 5363
    while (1) {
     $1330 = $$14475$i424$i + 1 | 0; //@line 5365
     $1331 = HEAP8[$$14475$i424$i >> 0] | 0; //@line 5366
     $1332 = $1331 & 255; //@line 5367
     L278 : do {
      switch ($1331 << 24 >> 24) {
      case -128:
       {
        $$051$off0$i430$i = 0; //@line 5371
        $$245$i431$i = $1330; //@line 5371
        $$461$i429$i = $$15872$i427$i; //@line 5371
        break;
       }
      case -1:
       {
        $1349 = HEAPU8[$$14475$i424$i + 2 >> 0] << 16 | HEAPU8[$1330 >> 0] << 24 | HEAPU8[$$14475$i424$i + 3 >> 0] << 8 | HEAPU8[$$14475$i424$i + 4 >> 0]; //@line 5391
        $$051$off0$i430$i = $1349; //@line 5393
        $$245$i431$i = $$14475$i424$i + 5 | 0; //@line 5393
        $$461$i429$i = $1349; //@line 5393
        break;
       }
      default:
       {
        if (!($1332 & 128)) {
         $1361 = HEAPU8[$1330 >> 0] | $1332 << 8; //@line 5404
         $1366 = ($1361 << 17 >> 16) + $$15872$i427$i | 0; //@line 5410
         if (!(($1361 << 1 & 65535) << 16 >> 16)) {
          $$5$ph$i433$i = 0; //@line 5412
          break L274;
         } else {
          $$051$off0$i430$i = $1366; //@line 5415
          $$245$i431$i = $$14475$i424$i + 2 | 0; //@line 5415
          $$461$i429$i = $1366; //@line 5415
          break L278;
         }
        } else {
         $1354 = $1332 << 1 & 254; //@line 5420
         $1356 = $1354 + $$15872$i427$i | 0; //@line 5422
         if (!$1354) {
          $$5$ph$i433$i = 0; //@line 5424
          break L274;
         } else {
          $$051$off0$i430$i = $1356; //@line 5427
          $$245$i431$i = $1330; //@line 5427
          $$461$i429$i = $1356; //@line 5427
          break L278;
         }
        }
       }
      }
     } while (0);
     if ($$14974$i425$i << 16 >> 16 == -24487) {
      $$5$ph$i433$i = $$051$off0$i430$i; //@line 5436
      break L274;
     }
     $1369 = $$05673$i426$i + 1 | 0; //@line 5440
     if (($1369 | 0) < 1024) {
      $$05673$i426$i = $1369; //@line 5443
      $$14475$i424$i = $$245$i431$i; //@line 5443
      $$14974$i425$i = $$14974$i425$i + 1 << 16 >> 16; //@line 5443
      $$15872$i427$i = $$461$i429$i; //@line 5443
     } else {
      $$043$i422$i = $$245$i431$i; //@line 5445
      $$048$i421$i = -24576; //@line 5445
      $$057$i420$i = $$461$i429$i; //@line 5445
      continue L274;
     }
    }
   }
   $1371 = $1321 + $$5$ph$i433$i | 0; //@line 5450
   $1373 = $1371 + 2 | 0; //@line 5452
   HEAP16[$1371 >> 1] = _htons(16615) | 0; //@line 5453
   $1375 = $1373 + 2 | 0; //@line 5455
   HEAP16[$1373 >> 1] = _htons(124) | 0; //@line 5456
   $1377 = $1375 + 2 | 0; //@line 5458
   HEAP16[$1375 >> 1] = _htons(1792) | 0; //@line 5459
   $1379 = $1377 + 2 | 0; //@line 5461
   HEAP16[$1377 >> 1] = _htons(28965) | 0; //@line 5462
   HEAP16[$1379 >> 1] = _htons(18143) | 0; //@line 5465
   HEAP16[$1379 + 2 >> 1] = _htons(20085) | 0; //@line 5467
   $1383 = HEAP32[30897] | 0; //@line 5468
   $1385 = (HEAP32[30896] | 0) + 34 | 0; //@line 5470
   $$043$i437$i = $1383 + (FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($1385 >>> 16 << 2) >> 2] >> 2] & 127]($1385) | 0) | 0; //@line 5477
   $$048$i436$i = -22528; //@line 5477
   $$057$i435$i = 0; //@line 5477
   L287 : while (1) {
    $$05673$i441$i = 0; //@line 5479
    $$14475$i439$i = $$043$i437$i; //@line 5479
    $$14974$i440$i = $$048$i436$i; //@line 5479
    $$15872$i442$i = $$057$i435$i; //@line 5479
    while (1) {
     $1392 = $$14475$i439$i + 1 | 0; //@line 5481
     $1393 = HEAP8[$$14475$i439$i >> 0] | 0; //@line 5482
     $1394 = $1393 & 255; //@line 5483
     L291 : do {
      switch ($1393 << 24 >> 24) {
      case -128:
       {
        $$051$off0$i445$i = 0; //@line 5487
        $$245$i446$i = $1392; //@line 5487
        $$461$i444$i = $$15872$i442$i; //@line 5487
        break;
       }
      case -1:
       {
        $1411 = HEAPU8[$$14475$i439$i + 2 >> 0] << 16 | HEAPU8[$1392 >> 0] << 24 | HEAPU8[$$14475$i439$i + 3 >> 0] << 8 | HEAPU8[$$14475$i439$i + 4 >> 0]; //@line 5507
        $$051$off0$i445$i = $1411; //@line 5509
        $$245$i446$i = $$14475$i439$i + 5 | 0; //@line 5509
        $$461$i444$i = $1411; //@line 5509
        break;
       }
      default:
       {
        if (!($1394 & 128)) {
         $1423 = HEAPU8[$1392 >> 0] | $1394 << 8; //@line 5520
         $1428 = ($1423 << 17 >> 16) + $$15872$i442$i | 0; //@line 5526
         if (!(($1423 << 1 & 65535) << 16 >> 16)) {
          $$5$ph$i448$i = 0; //@line 5528
          break L287;
         } else {
          $$051$off0$i445$i = $1428; //@line 5531
          $$245$i446$i = $$14475$i439$i + 2 | 0; //@line 5531
          $$461$i444$i = $1428; //@line 5531
          break L291;
         }
        } else {
         $1416 = $1394 << 1 & 254; //@line 5536
         $1418 = $1416 + $$15872$i442$i | 0; //@line 5538
         if (!$1416) {
          $$5$ph$i448$i = 0; //@line 5540
          break L287;
         } else {
          $$051$off0$i445$i = $1418; //@line 5543
          $$245$i446$i = $1392; //@line 5543
          $$461$i444$i = $1418; //@line 5543
          break L291;
         }
        }
       }
      }
     } while (0);
     if ($$14974$i440$i << 16 >> 16 == -24486) {
      $$5$ph$i448$i = $$051$off0$i445$i; //@line 5552
      break L287;
     }
     $1431 = $$05673$i441$i + 1 | 0; //@line 5556
     if (($1431 | 0) < 1024) {
      $$05673$i441$i = $1431; //@line 5559
      $$14475$i439$i = $$245$i446$i; //@line 5559
      $$14974$i440$i = $$14974$i440$i + 1 << 16 >> 16; //@line 5559
      $$15872$i442$i = $$461$i444$i; //@line 5559
     } else {
      $$043$i437$i = $$245$i446$i; //@line 5561
      $$048$i436$i = -24576; //@line 5561
      $$057$i435$i = $$461$i444$i; //@line 5561
      continue L287;
     }
    }
   }
   $1433 = $1383 + $$5$ph$i448$i | 0; //@line 5566
   $1435 = $1433 + 2 | 0; //@line 5568
   HEAP16[$1433 >> 1] = _htons(16615) | 0; //@line 5569
   $1437 = $1435 + 2 | 0; //@line 5571
   HEAP16[$1435 >> 1] = _htons(124) | 0; //@line 5572
   $1439 = $1437 + 2 | 0; //@line 5574
   HEAP16[$1437 >> 1] = _htons(1792) | 0; //@line 5575
   $1441 = $1439 + 2 | 0; //@line 5577
   HEAP16[$1439 >> 1] = _htons(28966) | 0; //@line 5578
   $1443 = $1441 + 2 | 0; //@line 5580
   HEAP16[$1441 >> 1] = _htons(18143) | 0; //@line 5581
   $1445 = $1443 + 2 | 0; //@line 5583
   HEAP16[$1443 >> 1] = _htons(20085) | 0; //@line 5584
   HEAP32[28205] = $1445 - (HEAP32[30897] | 0); //@line 5588
   $1450 = $1445 + 2 | 0; //@line 5590
   HEAP16[$1445 >> 1] = _htons(28967) | 0; //@line 5591
   $1452 = $1450 + 2 | 0; //@line 5593
   HEAP16[$1450 >> 1] = _htons(20085) | 0; //@line 5594
   HEAP32[28206] = $1452 - (HEAP32[30897] | 0); //@line 5598
   HEAP16[$1452 >> 1] = _htons(28982) | 0; //@line 5601
   HEAP16[$1452 + 2 >> 1] = _htons(20085) | 0; //@line 5603
   $1459 = HEAP32[30897] | 0; //@line 5604
   $1461 = (HEAP32[30896] | 0) + 34 | 0; //@line 5606
   $$043$i452$i = $1459 + (FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($1461 >>> 16 << 2) >> 2] >> 2] & 127]($1461) | 0) | 0; //@line 5613
   $$048$i451$i = -22528; //@line 5613
   $$057$i450$i = 0; //@line 5613
   L300 : while (1) {
    $$05673$i456$i = 0; //@line 5615
    $$14475$i454$i = $$043$i452$i; //@line 5615
    $$14974$i455$i = $$048$i451$i; //@line 5615
    $$15872$i457$i = $$057$i450$i; //@line 5615
    while (1) {
     $1468 = $$14475$i454$i + 1 | 0; //@line 5617
     $1469 = HEAP8[$$14475$i454$i >> 0] | 0; //@line 5618
     $1470 = $1469 & 255; //@line 5619
     L304 : do {
      switch ($1469 << 24 >> 24) {
      case -128:
       {
        $$051$off0$i460$i = 0; //@line 5623
        $$245$i461$i = $1468; //@line 5623
        $$461$i459$i = $$15872$i457$i; //@line 5623
        break;
       }
      case -1:
       {
        $1487 = HEAPU8[$$14475$i454$i + 2 >> 0] << 16 | HEAPU8[$1468 >> 0] << 24 | HEAPU8[$$14475$i454$i + 3 >> 0] << 8 | HEAPU8[$$14475$i454$i + 4 >> 0]; //@line 5643
        $$051$off0$i460$i = $1487; //@line 5645
        $$245$i461$i = $$14475$i454$i + 5 | 0; //@line 5645
        $$461$i459$i = $1487; //@line 5645
        break;
       }
      default:
       {
        if (!($1470 & 128)) {
         $1499 = HEAPU8[$1468 >> 0] | $1470 << 8; //@line 5656
         $1504 = ($1499 << 17 >> 16) + $$15872$i457$i | 0; //@line 5662
         if (!(($1499 << 1 & 65535) << 16 >> 16)) {
          $$5$ph$i463$i = 0; //@line 5664
          break L300;
         } else {
          $$051$off0$i460$i = $1504; //@line 5667
          $$245$i461$i = $$14475$i454$i + 2 | 0; //@line 5667
          $$461$i459$i = $1504; //@line 5667
          break L304;
         }
        } else {
         $1492 = $1470 << 1 & 254; //@line 5672
         $1494 = $1492 + $$15872$i457$i | 0; //@line 5674
         if (!$1492) {
          $$5$ph$i463$i = 0; //@line 5676
          break L300;
         } else {
          $$051$off0$i460$i = $1494; //@line 5679
          $$245$i461$i = $1468; //@line 5679
          $$461$i459$i = $1494; //@line 5679
          break L304;
         }
        }
       }
      }
     } while (0);
     if ($$14974$i455$i << 16 >> 16 == -22507) {
      $$5$ph$i463$i = $$051$off0$i460$i; //@line 5688
      break L300;
     }
     $1507 = $$05673$i456$i + 1 | 0; //@line 5692
     if (($1507 | 0) < 1024) {
      $$05673$i456$i = $1507; //@line 5695
      $$14475$i454$i = $$245$i461$i; //@line 5695
      $$14974$i455$i = $$14974$i455$i + 1 << 16 >> 16; //@line 5695
      $$15872$i457$i = $$461$i459$i; //@line 5695
     } else {
      $$043$i452$i = $$245$i461$i; //@line 5697
      $$048$i451$i = -24576; //@line 5697
      $$057$i450$i = $$461$i459$i; //@line 5697
      continue L300;
     }
    }
   }
   $1509 = $1459 + $$5$ph$i463$i | 0; //@line 5702
   $1511 = $1509 + 2 | 0; //@line 5704
   HEAP16[$1509 >> 1] = _htons(28968) | 0; //@line 5705
   HEAP16[$1511 >> 1] = _htons(11849) | 0; //@line 5708
   HEAP16[$1511 + 2 >> 1] = _htons(20176) | 0; //@line 5710
   $1515 = HEAP32[30897] | 0; //@line 5711
   HEAP16[$1515 + 112884 >> 1] = _htons(20217) | 0; //@line 5715
   HEAP16[$1515 + 112886 >> 1] = _htons(((HEAP32[30896] | 0) + 768 + (HEAP32[28207] | 0) | 0) >>> 16 & 65535) | 0; //@line 5724
   HEAP16[$1515 + 112888 >> 1] = _htons((HEAP32[30896] | 0) + 768 + (HEAP32[28207] | 0) & 65535) | 0; //@line 5731
   $1536 = (HEAP32[30897] | 0) + (HEAP32[28207] | 0) + 768 | 0; //@line 5735
   $1538 = $1536 + 2 | 0; //@line 5737
   HEAP16[$1536 >> 1] = _htons(12035) | 0; //@line 5738
   $1540 = $1538 + 2 | 0; //@line 5740
   HEAP16[$1538 >> 1] = _htons(8312) | 0; //@line 5741
   $1542 = $1540 + 2 | 0; //@line 5743
   HEAP16[$1540 >> 1] = _htons(2032) | 0; //@line 5744
   $1544 = $1542 + 2 | 0; //@line 5746
   HEAP16[$1542 >> 1] = _htons(20112) | 0; //@line 5747
   $1546 = $1544 + 2 | 0; //@line 5749
   HEAP16[$1544 >> 1] = _htons(8735) | 0; //@line 5750
   HEAP16[$1546 >> 1] = _htons(28972) | 0; //@line 5753
   HEAP16[$1546 + 2 >> 1] = _htons(20085) | 0; //@line 5755
   $1550 = HEAP32[30897] | 0; //@line 5756
   $1552 = (HEAP32[30896] | 0) + 34 | 0; //@line 5758
   $$043$i = $1550 + (FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($1552 >>> 16 << 2) >> 2] >> 2] & 127]($1552) | 0) | 0; //@line 5765
   $$048$i = -22528; //@line 5765
   $$057$i = 0; //@line 5765
   L313 : while (1) {
    $$05673$i = 0; //@line 5767
    $$14475$i = $$043$i; //@line 5767
    $$14974$i = $$048$i; //@line 5767
    $$15872$i = $$057$i; //@line 5767
    while (1) {
     $1559 = $$14475$i + 1 | 0; //@line 5769
     $1560 = HEAP8[$$14475$i >> 0] | 0; //@line 5770
     $1561 = $1560 & 255; //@line 5771
     L317 : do {
      switch ($1560 << 24 >> 24) {
      case -128:
       {
        $$051$off0$i = 0; //@line 5775
        $$245$i = $1559; //@line 5775
        $$461$i = $$15872$i; //@line 5775
        break;
       }
      case -1:
       {
        $1578 = HEAPU8[$$14475$i + 2 >> 0] << 16 | HEAPU8[$1559 >> 0] << 24 | HEAPU8[$$14475$i + 3 >> 0] << 8 | HEAPU8[$$14475$i + 4 >> 0]; //@line 5795
        $$051$off0$i = $1578; //@line 5797
        $$245$i = $$14475$i + 5 | 0; //@line 5797
        $$461$i = $1578; //@line 5797
        break;
       }
      default:
       {
        if (!($1561 & 128)) {
         $1590 = HEAPU8[$1559 >> 0] | $1561 << 8; //@line 5808
         $1595 = ($1590 << 17 >> 16) + $$15872$i | 0; //@line 5814
         if (!(($1590 << 1 & 65535) << 16 >> 16)) {
          $$5$ph$i = 0; //@line 5816
          break L313;
         } else {
          $$051$off0$i = $1595; //@line 5819
          $$245$i = $$14475$i + 2 | 0; //@line 5819
          $$461$i = $1595; //@line 5819
          break L317;
         }
        } else {
         $1583 = $1561 << 1 & 254; //@line 5824
         $1585 = $1583 + $$15872$i | 0; //@line 5826
         if (!$1583) {
          $$5$ph$i = 0; //@line 5828
          break L313;
         } else {
          $$051$off0$i = $1585; //@line 5831
          $$245$i = $1559; //@line 5831
          $$461$i = $1585; //@line 5831
          break L317;
         }
        }
       }
      }
     } while (0);
     if ($$14974$i << 16 >> 16 == -24485) {
      $$5$ph$i = $$051$off0$i; //@line 5840
      break L313;
     }
     $1598 = $$05673$i + 1 | 0; //@line 5844
     if (($1598 | 0) < 1024) {
      $$05673$i = $1598; //@line 5847
      $$14475$i = $$245$i; //@line 5847
      $$14974$i = $$14974$i + 1 << 16 >> 16; //@line 5847
      $$15872$i = $$461$i; //@line 5847
     } else {
      $$043$i = $$245$i; //@line 5849
      $$048$i = -24576; //@line 5849
      $$057$i = $$461$i; //@line 5849
      continue L313;
     }
    }
   }
   HEAP16[$1550 + $$5$ph$i >> 1] = _htons(28930) | 0; //@line 5856
   $1602 = HEAP32[30896] | 0; //@line 5857
   HEAP32[28202] = $1602 + 3072 + (HEAP32[28207] | 0); //@line 5861
   $1606 = HEAP32[30897] | 0; //@line 5862
   $1607 = $1602 + 34 | 0; //@line 5863
   $$043$i7 = $1606 + (FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($1607 >>> 16 << 2) >> 2] >> 2] & 127]($1607) | 0) | 0; //@line 5870
   $$048$i6 = -22528; //@line 5870
   $$057$i5 = 0; //@line 5870
   L326 : while (1) {
    $$05673$i11 = 0; //@line 5872
    $$14475$i9 = $$043$i7; //@line 5872
    $$14974$i10 = $$048$i6; //@line 5872
    $$15872$i12 = $$057$i5; //@line 5872
    while (1) {
     $1614 = $$14475$i9 + 1 | 0; //@line 5874
     $1615 = HEAP8[$$14475$i9 >> 0] | 0; //@line 5875
     $1616 = $1615 & 255; //@line 5876
     L330 : do {
      switch ($1615 << 24 >> 24) {
      case -128:
       {
        $$051$off0$i15 = 0; //@line 5880
        $$245$i16 = $1614; //@line 5880
        $$461$i14 = $$15872$i12; //@line 5880
        break;
       }
      case -1:
       {
        $1633 = HEAPU8[$$14475$i9 + 2 >> 0] << 16 | HEAPU8[$1614 >> 0] << 24 | HEAPU8[$$14475$i9 + 3 >> 0] << 8 | HEAPU8[$$14475$i9 + 4 >> 0]; //@line 5900
        $$051$off0$i15 = $1633; //@line 5902
        $$245$i16 = $$14475$i9 + 5 | 0; //@line 5902
        $$461$i14 = $1633; //@line 5902
        break;
       }
      default:
       {
        if (!($1616 & 128)) {
         $1645 = HEAPU8[$1614 >> 0] | $1616 << 8; //@line 5913
         $1650 = ($1645 << 17 >> 16) + $$15872$i12 | 0; //@line 5919
         if (!(($1645 << 1 & 65535) << 16 >> 16)) {
          $$5$ph$i18 = 0; //@line 5921
          break L326;
         } else {
          $$051$off0$i15 = $1650; //@line 5924
          $$245$i16 = $$14475$i9 + 2 | 0; //@line 5924
          $$461$i14 = $1650; //@line 5924
          break L330;
         }
        } else {
         $1638 = $1616 << 1 & 254; //@line 5929
         $1640 = $1638 + $$15872$i12 | 0; //@line 5931
         if (!$1638) {
          $$5$ph$i18 = 0; //@line 5933
          break L326;
         } else {
          $$051$off0$i15 = $1640; //@line 5936
          $$245$i16 = $1614; //@line 5936
          $$461$i14 = $1640; //@line 5936
          break L330;
         }
        }
       }
      }
     } while (0);
     if ($$14974$i10 << 16 >> 16 == -22018) {
      $$5$ph$i18 = $$051$off0$i15; //@line 5945
      break L326;
     }
     $1653 = $$05673$i11 + 1 | 0; //@line 5949
     if (($1653 | 0) < 1024) {
      $$05673$i11 = $1653; //@line 5952
      $$14475$i9 = $$245$i16; //@line 5952
      $$14974$i10 = $$14974$i10 + 1 << 16 >> 16; //@line 5952
      $$15872$i12 = $$461$i14; //@line 5952
     } else {
      $$043$i7 = $$245$i16; //@line 5954
      $$048$i6 = -24576; //@line 5954
      $$057$i5 = $$461$i14; //@line 5954
      continue L326;
     }
    }
   }
   $1655 = $$5$ph$i18 + $1602 | 0; //@line 5959
   $1659 = (HEAP32[30897] | 0) + (HEAP32[28207] | 0) + 3072 | 0; //@line 5963
   $1661 = $1659 + 2 | 0; //@line 5965
   HEAP16[$1659 >> 1] = _htons(28970) | 0; //@line 5966
   $1663 = $1661 + 2 | 0; //@line 5968
   HEAP16[$1661 >> 1] = _htons(20217) | 0; //@line 5969
   HEAP16[$1663 >> 1] = _htons($1655 >>> 16 & 65535) | 0; //@line 5974
   HEAP16[$1663 + 2 >> 1] = _htons($1655 & 65535) | 0; //@line 5977
   $1670 = HEAP32[30896] | 0; //@line 5978
   HEAP32[28203] = $1670 + 3328 + (HEAP32[28207] | 0); //@line 5982
   $1674 = HEAP32[30897] | 0; //@line 5983
   $1675 = $1670 + 34 | 0; //@line 5984
   $$043$i22 = $1674 + (FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($1675 >>> 16 << 2) >> 2] >> 2] & 127]($1675) | 0) | 0; //@line 5991
   $$048$i21 = -22528; //@line 5991
   $$057$i20 = 0; //@line 5991
   L339 : while (1) {
    $$05673$i26 = 0; //@line 5993
    $$14475$i24 = $$043$i22; //@line 5993
    $$14974$i25 = $$048$i21; //@line 5993
    $$15872$i27 = $$057$i20; //@line 5993
    while (1) {
     $1682 = $$14475$i24 + 1 | 0; //@line 5995
     $1683 = HEAP8[$$14475$i24 >> 0] | 0; //@line 5996
     $1684 = $1683 & 255; //@line 5997
     L343 : do {
      switch ($1683 << 24 >> 24) {
      case -128:
       {
        $$051$off0$i30 = 0; //@line 6001
        $$245$i31 = $1682; //@line 6001
        $$461$i29 = $$15872$i27; //@line 6001
        break;
       }
      case -1:
       {
        $1701 = HEAPU8[$$14475$i24 + 2 >> 0] << 16 | HEAPU8[$1682 >> 0] << 24 | HEAPU8[$$14475$i24 + 3 >> 0] << 8 | HEAPU8[$$14475$i24 + 4 >> 0]; //@line 6021
        $$051$off0$i30 = $1701; //@line 6023
        $$245$i31 = $$14475$i24 + 5 | 0; //@line 6023
        $$461$i29 = $1701; //@line 6023
        break;
       }
      default:
       {
        if (!($1684 & 128)) {
         $1713 = HEAPU8[$1682 >> 0] | $1684 << 8; //@line 6034
         $1718 = ($1713 << 17 >> 16) + $$15872$i27 | 0; //@line 6040
         if (!(($1713 << 1 & 65535) << 16 >> 16)) {
          $$5$ph$i33 = 0; //@line 6042
          break L339;
         } else {
          $$051$off0$i30 = $1718; //@line 6045
          $$245$i31 = $$14475$i24 + 2 | 0; //@line 6045
          $$461$i29 = $1718; //@line 6045
          break L343;
         }
        } else {
         $1706 = $1684 << 1 & 254; //@line 6050
         $1708 = $1706 + $$15872$i27 | 0; //@line 6052
         if (!$1706) {
          $$5$ph$i33 = 0; //@line 6054
          break L339;
         } else {
          $$051$off0$i30 = $1708; //@line 6057
          $$245$i31 = $1682; //@line 6057
          $$461$i29 = $1708; //@line 6057
          break L343;
         }
        }
       }
      }
     } while (0);
     if ($$14974$i25 << 16 >> 16 == -22019) {
      $$5$ph$i33 = $$051$off0$i30; //@line 6066
      break L339;
     }
     $1721 = $$05673$i26 + 1 | 0; //@line 6070
     if (($1721 | 0) < 1024) {
      $$05673$i26 = $1721; //@line 6073
      $$14475$i24 = $$245$i31; //@line 6073
      $$14974$i25 = $$14974$i25 + 1 << 16 >> 16; //@line 6073
      $$15872$i27 = $$461$i29; //@line 6073
     } else {
      $$043$i22 = $$245$i31; //@line 6075
      $$048$i21 = -24576; //@line 6075
      $$057$i20 = $$461$i29; //@line 6075
      continue L339;
     }
    }
   }
   $1723 = $$5$ph$i33 + $1670 | 0; //@line 6080
   $1727 = (HEAP32[30897] | 0) + (HEAP32[28207] | 0) + 3328 | 0; //@line 6084
   $1729 = $1727 + 2 | 0; //@line 6086
   HEAP16[$1727 >> 1] = _htons(28971) | 0; //@line 6087
   $1731 = $1729 + 2 | 0; //@line 6089
   HEAP16[$1729 >> 1] = _htons(20217) | 0; //@line 6090
   HEAP16[$1731 >> 1] = _htons($1723 >>> 16 & 65535) | 0; //@line 6095
   HEAP16[$1731 + 2 >> 1] = _htons($1723 & 65535) | 0; //@line 6098
   $1738 = HEAP32[30896] | 0; //@line 6099
   $1739 = $1738 + 26 | 0; //@line 6100
   $1745 = (FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($1739 >>> 16 << 2) >> 2] >> 2] & 127]($1739) | 0) + $1738 | 0; //@line 6106
   $1750 = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($1745 >>> 16 << 2) >> 2] >> 2] & 127]($1745) | 0; //@line 6111
   HEAP32[28209] = $1750; //@line 6112
   $1753 = $1750; //@line 6113
   while (1) {
    $1752 = (HEAP32[30896] | 0) + $1753 | 0; //@line 6116
    $1754 = $1752 + 12 | 0; //@line 6117
    $1759 = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($1754 >>> 16 << 2) >> 2] >> 2] & 127]($1754) | 0; //@line 6122
    $1760 = $1752 + 16 | 0; //@line 6123
    $1765 = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($1760 >>> 16 << 2) >> 2] >> 2] & 127]($1760) | 0; //@line 6128
    $1766 = $1752 + 20 | 0; //@line 6129
    if (($1765 | 0) == 1346454347 & ((FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($1766 >>> 16 << 2) >> 2] | 0) + 4 >> 2] & 127]($1766) | 0) & 65535 | 0) == 4) {
     break;
    }
    $1775 = $1752 + 8 | 0; //@line 6143
    $1753 = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($1775 >>> 16 << 2) >> 2] >> 2] & 127]($1775) | 0; //@line 6148
    HEAP32[28209] = $1753; //@line 6149
    if (!$1753) {
     $$0 = 0; //@line 6152
     label = 246; //@line 6153
     break;
    }
   }
   if ((label | 0) == 246) {
    STACKTOP = sp; //@line 6160
    return $$0 | 0; //@line 6160
   }
   if (!$1759) {
    $$0 = 0; //@line 6164
    STACKTOP = sp; //@line 6165
    return $$0 | 0; //@line 6165
   }
   $1783 = HEAP32[30896] | 0; //@line 6167
   $1784 = $1783 + 26 | 0; //@line 6168
   $1790 = (FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($1784 >>> 16 << 2) >> 2] >> 2] & 127]($1784) | 0) + $1783 | 0; //@line 6174
   FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($1790 >>> 16 << 2) >> 2] >> 2] & 127]($1790) | 0; //@line 6179
   $1798 = (HEAP32[30896] | 0) + 8 + (HEAP32[28209] | 0) | 0; //@line 6183
   $1803 = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($1798 >>> 16 << 2) >> 2] >> 2] & 127]($1798) | 0; //@line 6188
   HEAP32[28209] = $1803; //@line 6189
   $1806 = $1803; //@line 6190
   while (1) {
    $1805 = (HEAP32[30896] | 0) + $1806 | 0; //@line 6193
    $1807 = $1805 + 12 | 0; //@line 6194
    $1812 = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($1807 >>> 16 << 2) >> 2] >> 2] & 127]($1807) | 0; //@line 6199
    $1813 = $1805 + 16 | 0; //@line 6200
    $1818 = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($1813 >>> 16 << 2) >> 2] >> 2] & 127]($1813) | 0; //@line 6205
    $1819 = $1805 + 20 | 0; //@line 6206
    if (($1818 | 0) == 1346454347 & ((FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($1819 >>> 16 << 2) >> 2] | 0) + 4 >> 2] & 127]($1819) | 0) & 65535 | 0) == 4) {
     $$2$i41 = $1812; //@line 6218
     break;
    }
    $1828 = $1805 + 8 | 0; //@line 6221
    $1806 = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($1828 >>> 16 << 2) >> 2] >> 2] & 127]($1828) | 0; //@line 6226
    HEAP32[28209] = $1806; //@line 6227
    if (!$1806) {
     $$2$i41 = 0; //@line 6230
     break;
    }
   }
   if (!(HEAP32[30881] | $$2$i41)) {
    _puts(91982) | 0; //@line 6240
   }
   $1859 = HEAP32[30897] | 0; //@line 6242
   HEAP16[$1859 + 39876 >> 1] = _htons(28674) | 0; //@line 6246
   HEAP16[$1859 + 39878 >> 1] = _htons(20081) | 0; //@line 6249
   HEAP16[$1859 + 39880 >> 1] = _htons(20081) | 0; //@line 6252
   HEAP16[$1859 + 39882 >> 1] = _htons(20081) | 0; //@line 6255
   HEAP16[$1859 + 39884 >> 1] = _htons(20081) | 0; //@line 6257
   $1870 = HEAP32[30897] | 0; //@line 6258
   HEAP16[$1870 + 41622 >> 1] = _htons(20081) | 0; //@line 6262
   HEAP16[$1870 + 41624 >> 1] = _htons(20081) | 0; //@line 6265
   HEAP16[$1870 + 41626 >> 1] = _htons(28969) | 0; //@line 6268
   $$sink = _htons(19072) | 0; //@line 6271
   $$sink45$in = $1870 + 41628 | 0; //@line 6271
   $$sink46 = $1870 + 41630 | 0; //@line 6271
   break;
  }
 default:
  {
   $$0 = 0; //@line 6275
   STACKTOP = sp; //@line 6276
   return $$0 | 0; //@line 6276
  }
 }
 HEAP16[$$sink45$in >> 1] = $$sink; //@line 6279
 HEAP16[$$sink46 >> 1] = _htons(26612) | 0; //@line 6281
 $1881 = HEAP32[28204] | 0; //@line 6282
 if ($1881 | 0) {
  $1884 = (HEAP32[30897] | 0) + $1881 | 0; //@line 6286
  HEAP16[$1884 >> 1] = _htons(28929) | 0; //@line 6288
 }
 $$0 = 1; //@line 6290
 STACKTOP = sp; //@line 6291
 return $$0 | 0; //@line 6291
}
function _read_table68k() {
 var $$0$i607615$i = 0, $$0$i609643$i = 0, $$0$i611645$i = 0, $$0$i613$i = 0, $$0501795$i = 0, $$0502794$i = 0, $$0510$i = 0, $$0513$i = 0, $$0527618$ph$i = 0, $$0527618636$i = 0, $$0544$lcssa$i = 0, $$0544755$i = 0, $$0544755$i$us = 0, $$0547752$i = 0, $$0549$lcssa$i = 0, $$0549754$i = 0, $$0549754$i$us = 0, $$0551$lcssa$i = 0, $$0551753$i = 0, $$0551753$i$us = 0, $$0559751$i = 0, $$0561750$i = 0, $$0562749$i = 0, $$074 = 0, $$1$lcssa$i = 0, $$10$i = 0, $$10617$i = 0, $$10617$ph$i = 0, $$10617637$i = 0, $$10626$i = 0, $$11$i = 0, $$12774$i = 0, $$13$i = 0, $$14$i = 0, $$1504642$i = 0, $$1508$i = 0, $$1508621$i = 0, $$1508621$ph$i = 0, $$1508621633$i = 0, $$1508630$i = 0, $$1511$i = 0, $$1511$ph$i = 0, $$1514$i = 0, $$1514620$i = 0, $$1514620$ph$i = 0, $$1514620634$i = 0, $$1514629$i = 0, $$1517641$i = 0, $$1523$i = 0, $$1528$i = 0, $$1532640$i = 0, $$1539$i = 0, $$1545$i = 0, $$1545$i$us = 0, $$1548$i = 0, $$1550$i = 0, $$1550$i$us = 0, $$1552$i = 0, $$1552$i$us = 0, $$1560$i = 0, $$1566748$i = 0, $$1568$1$i = 0, $$1568$2$i = 0, $$1568$3$i = 0, $$1568$4$i = 0, $$1568$i = 0, $$1574$1$i = 0, $$1574$2$i = 0, $$1574$3$i = 0, $$1574$4$i = 0, $$1574$i = 0, $$15771$i = 0, $$16$i = 0, $$173 = 0, $$1756$i = 0, $$1756$i$us = 0, $$2$i = 0, $$2$i$us = 0, $$2505$i = 0, $$2505$ph$i = 0, $$2509$i = 0, $$2512$i = 0, $$2515$i = 0, $$2518$i = 0, $$2524$i = 0, $$2533$i = 0, $$2540$i = 0, $$2553$i = 0, $$2564$i = 0, $$2569780$i = 0, $$3$i = 0, $$3$i$us = 0, $$3506$i = 0, $$3519$i = 0, $$3525$i = 0, $$3525619$i = 0, $$3525619$ph$i = 0, $$3525619635$i = 0, $$3525628$i = 0, $$3534$i = 0, $$3541$i = 0, $$3554$i = 0, $$3570$i = 0, $$4520$i = 0, $$4520$ph$i = 0, $$4526$i = 0, $$4535$i = 0, $$4535$ph$i = 0, $$4542$i = 0, $$4542638$i = 0, $$4555768$i = 0, $$4777$i = 0, $$5$i = 0, $$5521$i = 0, $$5543$i = 0, $$5556$i = 0, $$5572$i = 0, $$6537$i = 0, $$6557$i = 0, $$7$i = 0, $$7558$i = 0, $$8765$i = 0, $$9$i = 0, $$hi = 0, $$hi13 = 0, $$hi18 = 0, $$hi24 = 0, $$hi30 = 0, $$hi38 = 0, $$hi8 = 0, $$sink$i = 0, $$sink$i$us = 0, $$sink104$i = 0, $$sink598$i = 0, $$sink602$i = 0, $$sink656$i = 0, $0 = 0, $1 = 0, $10 = 0, $101 = 0, $104 = 0, $11 = 0, $110 = 0, $112 = 0, $114 = 0, $117 = 0, $12 = 0, $121 = 0, $127 = 0, $128 = 0, $13 = 0, $131 = 0, $135 = 0, $138 = 0, $14 = 0, $140 = 0, $144 = 0, $149 = 0, $15 = 0, $153 = 0, $157 = 0, $16 = 0, $160 = 0, $163 = 0, $169 = 0, $17 = 0, $170 = 0, $173 = 0, $174 = 0, $18 = 0, $182 = 0, $187 = 0, $189 = 0, $19 = 0, $195 = 0, $2 = 0, $20 = 0, $21 = 0, $212 = 0, $213 = 0, $218 = 0, $219 = 0, $22 = 0, $226 = 0, $23 = 0, $234 = 0, $236 = 0, $24 = 0, $244 = 0, $25 = 0, $251 = 0, $252 = 0, $253 = 0, $258 = 0, $259 = 0, $26 = 0, $266 = 0, $27 = 0, $274 = 0, $276 = 0, $28 = 0, $284 = 0, $29 = 0, $291 = 0, $295 = 0, $3 = 0, $30 = 0, $300 = 0, $31 = 0, $310 = 0, $312 = 0, $319 = 0, $32 = 0, $33 = 0, $332 = 0, $333 = 0, $338 = 0, $339 = 0, $34 = 0, $35 = 0, $353 = 0, $355 = 0, $36 = 0, $362 = 0, $37 = 0, $370 = 0, $371 = 0, $376 = 0, $377 = 0, $38 = 0, $39 = 0, $391 = 0, $393 = 0, $4 = 0, $40 = 0, $400 = 0, $409 = 0, $41 = 0, $411 = 0, $412 = 0, $418 = 0, $42 = 0, $421 = 0, $423 = 0, $428 = 0, $43 = 0, $430 = 0, $432 = 0, $433 = 0, $434 = 0, $435 = 0, $439 = 0, $44 = 0, $45 = 0, $452 = 0, $453 = 0, $46 = 0, $463 = 0, $467 = 0, $468 = 0, $47 = 0, $470 = 0, $471 = 0, $477 = 0, $478 = 0, $479 = 0, $48 = 0, $481 = 0, $482 = 0, $483 = 0, $484 = 0, $488 = 0, $49 = 0, $498 = 0, $5 = 0, $50 = 0, $502 = 0, $503 = 0, $504 = 0, $505 = 0, $51 = 0, $52 = 0, $521 = 0, $522 = 0, $523 = 0, $53 = 0, $54 = 0, $55 = 0, $56 = 0, $57 = 0, $6 = 0, $63 = 0, $64 = 0, $65 = 0, $67 = 0, $68 = 0, $71 = 0, $74 = 0, $75 = 0, $79 = 0, $8 = 0, $80 = 0, $81 = 0, $82 = 0, $83 = 0, $84 = 0, $9 = 0, $90 = 0, $91 = 0, $94 = 0, $trunc594$1$i = 0, $trunc594$2$i = 0, $trunc594$3$i = 0, $trunc594$4$i = 0, $trunc594$i = 0, $vararg_buffer = 0, $vararg_buffer1 = 0, dest = 0, label = 0, sp = 0, src = 0, stop = 0;
 sp = STACKTOP; //@line 52818
 STACKTOP = STACKTOP + 384 | 0; //@line 52819
 $vararg_buffer1 = sp + 8 | 0; //@line 52820
 $vararg_buffer = sp; //@line 52821
 $0 = sp + 256 | 0; //@line 52822
 $1 = sp + 176 | 0; //@line 52823
 $2 = sp + 96 | 0; //@line 52824
 $3 = sp + 16 | 0; //@line 52825
 $4 = sp + 320 | 0; //@line 52826
 $5 = _malloc(1048576) | 0; //@line 52827
 HEAP32[162823] = $5; //@line 52828
 $$074 = 0; //@line 52829
 do {
  $53 = $5 + ($$074 << 4) + 9 | 0; //@line 52831
  $$hi = $53 + 4 | 0; //@line 52833
  $54 = (HEAPU8[$53 >> 0] | HEAPU8[$53 + 1 >> 0] << 8 | HEAPU8[$53 + 2 >> 0] << 16 | HEAPU8[$53 + 3 >> 0] << 24) & -16711681; //@line 52841
  $55 = ((HEAPU8[$$hi >> 0] | HEAPU8[$$hi + 1 >> 0] << 8) & 65535 | HEAPU8[$$hi + 2 >> 0] << 16) & 16777215; //@line 52842
  HEAP8[$53 >> 0] = $54; //@line 52843
  HEAP8[$53 + 1 >> 0] = $54 >> 8; //@line 52843
  HEAP8[$53 + 2 >> 0] = $54 >> 16; //@line 52843
  HEAP8[$53 + 3 >> 0] = $54 >> 24; //@line 52843
  $$hi8 = $53 + 4 | 0; //@line 52844
  $56 = $55 & 65535; //@line 52845
  HEAP8[$$hi8 >> 0] = $56; //@line 52846
  HEAP8[$$hi8 + 1 >> 0] = $56 >> 8; //@line 52846
  $57 = _bitshift64Lshr($55 | 0, 0, 16) | 0; //@line 52847
  HEAP8[$$hi8 + 2 >> 0] = $57; //@line 52851
  HEAP32[$5 + ($$074 << 4) >> 2] = -1; //@line 52853
  $$074 = $$074 + 1 | 0; //@line 52854
 } while (($$074 | 0) != 65536);
 $6 = HEAP32[21986] | 0; //@line 52862
 if (($6 | 0) <= 0) {
  STACKTOP = sp; //@line 52865
  return;
 }
 $8 = $0 + 56 | 0; //@line 52867
 $9 = $0 + 36 | 0; //@line 52868
 $10 = $0 + 40 | 0; //@line 52869
 $11 = $0 + 44 | 0; //@line 52870
 $12 = $0 + 48 | 0; //@line 52871
 $13 = $0 + 52 | 0; //@line 52872
 $14 = $0 + 60 | 0; //@line 52873
 $15 = $0 + 4 | 0; //@line 52874
 $16 = $0 + 24 | 0; //@line 52875
 $17 = $2 + 28 | 0; //@line 52876
 $18 = $2 + 68 | 0; //@line 52877
 $19 = $2 + 12 | 0; //@line 52878
 $20 = $1 + 24 | 0; //@line 52879
 $21 = $1 + 72 | 0; //@line 52880
 $22 = $1 + 12 | 0; //@line 52881
 $23 = $2 + 8 | 0; //@line 52882
 $24 = $0 + 32 | 0; //@line 52883
 $25 = $0 + 28 | 0; //@line 52884
 $26 = HEAP32[22038] | 0; //@line 52885
 $27 = $2 + 60 | 0; //@line 52886
 $28 = $3 + 60 | 0; //@line 52887
 $29 = $2 + 64 | 0; //@line 52888
 $30 = $3 + 64 | 0; //@line 52889
 $31 = $2 + 20 | 0; //@line 52890
 $32 = $2 + 32 | 0; //@line 52891
 $33 = $2 + 36 | 0; //@line 52892
 $34 = $2 + 40 | 0; //@line 52893
 $35 = $2 + 56 | 0; //@line 52894
 $36 = $2 + 52 | 0; //@line 52895
 $37 = $3 + 56 | 0; //@line 52896
 $38 = $2 + 48 | 0; //@line 52897
 $39 = $2 + 44 | 0; //@line 52898
 $40 = $3 + 48 | 0; //@line 52899
 $41 = $3 + 20 | 0; //@line 52900
 $42 = $3 + 28 | 0; //@line 52901
 $43 = $3 + 32 | 0; //@line 52902
 $44 = $3 + 36 | 0; //@line 52903
 $45 = $3 + 40 | 0; //@line 52904
 $46 = $2 + 72 | 0; //@line 52905
 $47 = $3 + 72 | 0; //@line 52906
 $48 = $2 + 76 | 0; //@line 52907
 $49 = $3 + 76 | 0; //@line 52908
 $50 = $2 + 16 | 0; //@line 52909
 $51 = $2 + 24 | 0; //@line 52910
 $52 = $3 + 24 | 0; //@line 52911
 $$173 = 0; //@line 52912
 $522 = $6; //@line 52912
 L7 : while (1) {
  dest = $0; //@line 52915
  src = 76360 + ($$173 << 6) | 0; //@line 52915
  stop = dest + 64 | 0; //@line 52915
  do {
   HEAP32[dest >> 2] = HEAP32[src >> 2]; //@line 52915
   dest = dest + 4 | 0; //@line 52915
   src = src + 4 | 0; //@line 52915
  } while ((dest | 0) < (stop | 0));
  $63 = HEAP8[$8 >> 0] | 0; //@line 52916
  $trunc594$i = HEAP8[$9 >> 0] | 0; //@line 52917
  switch (($trunc594$i & 255) >>> 3 & 7) {
  case 6:
  case 1:
   {
    $$1574$i = 0; //@line 52922
    break;
   }
  default:
   {
    $$1574$i = 16; //@line 52926
   }
  }
  switch ($trunc594$i & 7) {
  case 4:
  case 1:
   {
    $$1568$i = 0; //@line 52932
    break;
   }
  default:
   {
    $$1568$i = 16; //@line 52936
   }
  }
  $trunc594$1$i = HEAP8[$10 >> 0] | 0; //@line 52939
  switch (($trunc594$1$i & 255) >>> 3 & 7) {
  case 6:
  case 1:
   {
    $$1574$1$i = $$1574$i; //@line 52944
    break;
   }
  default:
   {
    $$1574$1$i = $$1574$i | 8; //@line 52949
   }
  }
  switch ($trunc594$1$i & 7) {
  case 4:
  case 1:
   {
    $$1568$1$i = $$1568$i; //@line 52955
    break;
   }
  default:
   {
    $$1568$1$i = $$1568$i | 8; //@line 52960
   }
  }
  $trunc594$2$i = HEAP8[$11 >> 0] | 0; //@line 52963
  switch (($trunc594$2$i & 255) >>> 3 & 7) {
  case 6:
  case 1:
   {
    $$1574$2$i = $$1574$1$i; //@line 52968
    break;
   }
  default:
   {
    $$1574$2$i = $$1574$1$i | 4; //@line 52973
   }
  }
  switch ($trunc594$2$i & 7) {
  case 4:
  case 1:
   {
    $$1568$2$i = $$1568$1$i; //@line 52979
    break;
   }
  default:
   {
    $$1568$2$i = $$1568$1$i | 4; //@line 52984
   }
  }
  $trunc594$3$i = HEAP8[$12 >> 0] | 0; //@line 52987
  switch (($trunc594$3$i & 255) >>> 3 & 7) {
  case 6:
  case 1:
   {
    $$1574$3$i = $$1574$2$i; //@line 52992
    break;
   }
  default:
   {
    $$1574$3$i = $$1574$2$i | 2; //@line 52997
   }
  }
  switch ($trunc594$3$i & 7) {
  case 4:
  case 1:
   {
    $$1568$3$i = $$1568$2$i; //@line 53003
    break;
   }
  default:
   {
    $$1568$3$i = $$1568$2$i | 2; //@line 53008
   }
  }
  $trunc594$4$i = HEAP8[$13 >> 0] | 0; //@line 53011
  switch (($trunc594$4$i & 255) >>> 3 & 7) {
  case 6:
  case 1:
   {
    $$1574$4$i = $$1574$3$i; //@line 53016
    break;
   }
  default:
   {
    $$1574$4$i = $$1574$3$i | 1; //@line 53021
   }
  }
  switch ($trunc594$4$i & 7) {
  case 4:
  case 1:
   {
    $$1568$4$i = $$1568$3$i; //@line 53027
    break;
   }
  default:
   {
    $$1568$4$i = $$1568$3$i | 1; //@line 53032
   }
  }
  $128 = HEAP32[$14 >> 2] | 0; //@line 53035
  $65 = HEAP32[$15 >> 2] | 0; //@line 53036
  if (($65 | 0) == 31) {
   $521 = $522; //@line 53039
  } else {
   $64 = 1 << $65; //@line 53041
   $67 = HEAP32[$0 >> 2] & 65535; //@line 53043
   $68 = HEAP32[$16 >> 2] | 0; //@line 53044
   $71 = HEAP32[$24 >> 2] << 28 & 805306368; //@line 53047
   $74 = _bitshift64Shl(HEAP32[$25 >> 2] & 7 | 0, 0, 48) | 0; //@line 53050
   $75 = tempRet0; //@line 53051
   $79 = _bitshift64Shl($63 & 7 | 0, 0, 51) | 0; //@line 53055
   $80 = tempRet0; //@line 53056
   $81 = $$1574$4$i & 255 | $79; //@line 53057
   $82 = $64 & 65535; //@line 53058
   $83 = $74 | $71; //@line 53059
   $$0501795$i = -1; //@line 53060
   $$0502794$i = 0; //@line 53060
   $$2569780$i = $$1568$4$i; //@line 53060
   while (1) {
    dest = $2; //@line 53062
    stop = dest + 80 | 0; //@line 53062
    do {
     HEAP32[dest >> 2] = 0; //@line 53062
     dest = dest + 4 | 0; //@line 53062
    } while ((dest | 0) < (stop | 0));
    dest = $1; //@line 53063
    stop = dest + 80 | 0; //@line 53063
    do {
     HEAP32[dest >> 2] = 0; //@line 53063
     dest = dest + 4 | 0; //@line 53063
    } while ((dest | 0) < (stop | 0));
    $$0547752$i = 0; //@line 53064
    $$0559751$i = $82; //@line 53064
    $$0561750$i = 32768; //@line 53064
    $$0562749$i = $67; //@line 53064
    $$1566748$i = 0; //@line 53064
    while (1) {
     $84 = $$0561750$i & 65535; //@line 53066
     if (!($84 & $68)) {
      $90 = HEAP8[$0 + 8 + $$0547752$i >> 0] | 0; //@line 53073
      $91 = ($$0559751$i & 65535) >>> 1; //@line 53074
      $94 = ($91 & 65535 & $$0502794$i | 0) != 0; //@line 53077
      HEAP32[$3 + ($90 << 2) >> 2] = 15 - $$1566748$i; //@line 53085
      $101 = $1 + ($90 << 2) | 0; //@line 53086
      HEAP32[$101 >> 2] = (HEAP32[$101 >> 2] | 0) + 1; //@line 53089
      $104 = $2 + ($90 << 2) | 0; //@line 53090
      HEAP32[$104 >> 2] = HEAP32[$104 >> 2] << 1 | $94 & 1; //@line 53094
      $$1548$i = $$0547752$i + 1 | 0; //@line 53095
      $$1560$i = $91; //@line 53095
      $$2564$i = $94 ? ($$0562749$i & 65535 | $$0561750$i) & 65535 : $$0562749$i; //@line 53095
     } else {
      $$1548$i = $$0547752$i; //@line 53097
      $$1560$i = $$0559751$i; //@line 53097
      $$2564$i = $$0562749$i; //@line 53097
     }
     $$1566748$i = $$1566748$i + 1 | 0; //@line 53099
     if (($$1566748$i | 0) == 16) {
      break;
     } else {
      $$0547752$i = $$1548$i; //@line 53105
      $$0559751$i = $$1560$i; //@line 53105
      $$0561750$i = $84 >>> 1; //@line 53105
      $$0562749$i = $$2564$i; //@line 53105
     }
    }
    $110 = HEAP32[$17 >> 2] | 0; //@line 53108
    if (!$110) {
     HEAP32[$17 >> 2] = 8; //@line 53111
     $523 = 8; //@line 53112
    } else {
     $523 = $110; //@line 53114
    }
    $112 = HEAP32[$18 >> 2] | 0; //@line 53116
    $114 = HEAP32[$19 >> 2] | 0; //@line 53118
    L52 : do {
     if (($112 | 0) == 3 | ($114 | 0) == 1) {
      $$5572$i = $$2569780$i; //@line 53123
      $$7$i = $$0501795$i; //@line 53123
     } else {
      $117 = (HEAP32[$20 >> 2] | 0) == 0; //@line 53126
      if (!$117) {
       switch (HEAP32[$51 >> 2] | 0) {
       case 255:
       case 0:
        {
         $$5572$i = $$2569780$i; //@line 53131
         $$7$i = $$0501795$i; //@line 53131
         break L52;
         break;
        }
       default:
        {}
       }
      }
      $121 = HEAP32[$46 >> 2] | 0; //@line 53141
      if ((HEAP32[$21 >> 2] | 0) != 0 & ($121 | 0) == 0) {
       $$5572$i = $$2569780$i; //@line 53145
       $$7$i = $$0501795$i; //@line 53145
      } else {
       if (!$117) {
        HEAP32[$31 >> 2] = HEAP32[$51 >> 2]; //@line 53149
        HEAP32[$41 >> 2] = HEAP32[$52 >> 2]; //@line 53151
       }
       if (HEAP32[$22 >> 2] | 0) {
        HEAP32[$23 >> 2] = $114; //@line 53156
       }
       $127 = HEAP8[$128 >> 0] | 0; //@line 53158
       L64 : do {
        if (!($127 << 24 >> 24)) {
         $$0544$lcssa$i = 2; //@line 53162
         $$0549$lcssa$i = 0; //@line 53162
         $$0551$lcssa$i = 0; //@line 53162
         $$1$lcssa$i = $$0501795$i; //@line 53162
        } else {
         if ($112 >>> 0 < 3) {
          $$0544755$i$us = 2; //@line 53166
          $$0549754$i$us = 0; //@line 53166
          $$0551753$i$us = 0; //@line 53166
          $$1756$i$us = $$0501795$i; //@line 53166
          $131 = $127; //@line 53166
         } else {
          $$0544755$i = 2; //@line 53168
          $$0549754$i = 0; //@line 53168
          $$0551753$i = 0; //@line 53168
          $$1756$i = $$0501795$i; //@line 53168
          $149 = $127; //@line 53168
          while (1) {
           if (_isspace($149 << 24 >> 24) | 0) {
            $$0544$lcssa$i = $$0544755$i; //@line 53174
            $$0549$lcssa$i = $$0549754$i; //@line 53174
            $$0551$lcssa$i = $$0551753$i; //@line 53174
            $$1$lcssa$i = $$1756$i; //@line 53174
            break L64;
           }
           L70 : do {
            if ($149 << 24 >> 24 == 46) {
             $153 = $$0551753$i + 1 | 0; //@line 53180
             switch (HEAP8[$128 + $153 >> 0] | 0) {
             case 122:
              {
               label = 44; //@line 53186
               break L7;
               break;
              }
             case 66:
              {
               $$1545$i = 0; //@line 53191
               $$1550$i = $$0549754$i; //@line 53191
               $$1552$i = $153; //@line 53191
               $$3$i = $$1756$i; //@line 53191
               break L70;
               break;
              }
             case 87:
              {
               $$1545$i = 1; //@line 53196
               $$1550$i = $$0549754$i; //@line 53196
               $$1552$i = $153; //@line 53196
               $$3$i = $$1756$i; //@line 53196
               break L70;
               break;
              }
             case 76:
              {
               $$1545$i = 2; //@line 53201
               $$1550$i = $$0549754$i; //@line 53201
               $$1552$i = $153; //@line 53201
               $$3$i = $$1756$i; //@line 53201
               break L70;
               break;
              }
             default:
              {
               label = 45; //@line 53206
               break L7;
              }
             }
            } else {
             $157 = $4 + $$0549754$i | 0; //@line 53211
             HEAP8[$157 >> 0] = $149; //@line 53212
             if ($149 << 24 >> 24 == 102) {
              switch (HEAP32[$50 >> 2] | 0) {
              case 0:
               {
                $$sink$i = 82; //@line 53218
                break;
               }
              case 1:
               {
                $$sink$i = 76; //@line 53222
                break;
               }
              default:
               {
                label = 49; //@line 53226
                break L7;
               }
              }
              HEAP8[$157 >> 0] = $$sink$i; //@line 53230
              $$2$i = -1; //@line 53231
             } else {
              $$2$i = $$1756$i; //@line 53233
             }
             $160 = $$0549754$i + 1 | 0; //@line 53235
             if ($160 >>> 0 > 62) {
              label = 52; //@line 53238
              break L7;
             } else {
              $$1545$i = $$0544755$i; //@line 53241
              $$1550$i = $160; //@line 53241
              $$1552$i = $$0551753$i; //@line 53241
              $$3$i = $$2$i; //@line 53241
             }
            }
           } while (0);
           $163 = $$1552$i + 1 | 0; //@line 53245
           $149 = HEAP8[$128 + $163 >> 0] | 0; //@line 53247
           if (!($149 << 24 >> 24)) {
            $$0544$lcssa$i = $$1545$i; //@line 53250
            $$0549$lcssa$i = $$1550$i; //@line 53250
            $$0551$lcssa$i = $163; //@line 53250
            $$1$lcssa$i = $$3$i; //@line 53250
            break L64;
           } else {
            $$0544755$i = $$1545$i; //@line 53253
            $$0549754$i = $$1550$i; //@line 53253
            $$0551753$i = $163; //@line 53253
            $$1756$i = $$3$i; //@line 53253
           }
          }
         }
         while (1) {
          if (_isspace($131 << 24 >> 24) | 0) {
           $$0544$lcssa$i = $$0544755$i$us; //@line 53262
           $$0549$lcssa$i = $$0549754$i$us; //@line 53262
           $$0551$lcssa$i = $$0551753$i$us; //@line 53262
           $$1$lcssa$i = $$1756$i$us; //@line 53262
           break L64;
          }
          L86 : do {
           if ($131 << 24 >> 24 == 46) {
            $140 = $$0551753$i$us + 1 | 0; //@line 53268
            switch (HEAP8[$128 + $140 >> 0] | 0) {
            case 66:
             {
              $$1545$i$us = 0; //@line 53274
              $$1550$i$us = $$0549754$i$us; //@line 53274
              $$1552$i$us = $140; //@line 53274
              $$3$i$us = $$1756$i$us; //@line 53274
              break L86;
              break;
             }
            case 87:
             {
              $$1545$i$us = 1; //@line 53279
              $$1550$i$us = $$0549754$i$us; //@line 53279
              $$1552$i$us = $140; //@line 53279
              $$3$i$us = $$1756$i$us; //@line 53279
              break L86;
              break;
             }
            case 76:
             {
              $$1545$i$us = 2; //@line 53284
              $$1550$i$us = $$0549754$i$us; //@line 53284
              $$1552$i$us = $140; //@line 53284
              $$3$i$us = $$1756$i$us; //@line 53284
              break L86;
              break;
             }
            case 122:
             {
              $$1545$i$us = $112; //@line 53289
              $$1550$i$us = $$0549754$i$us; //@line 53289
              $$1552$i$us = $140; //@line 53289
              $$3$i$us = $$1756$i$us; //@line 53289
              break L86;
              break;
             }
            default:
             {
              label = 45; //@line 53294
              break L7;
             }
            }
           } else {
            $135 = $4 + $$0549754$i$us | 0; //@line 53299
            HEAP8[$135 >> 0] = $131; //@line 53300
            if ($131 << 24 >> 24 == 102) {
             switch (HEAP32[$50 >> 2] | 0) {
             case 0:
              {
               $$sink$i$us = 82; //@line 53306
               break;
              }
             case 1:
              {
               $$sink$i$us = 76; //@line 53310
               break;
              }
             default:
              {
               label = 49; //@line 53314
               break L7;
              }
             }
             HEAP8[$135 >> 0] = $$sink$i$us; //@line 53318
             $$2$i$us = -1; //@line 53319
            } else {
             $$2$i$us = $$1756$i$us; //@line 53321
            }
            $138 = $$0549754$i$us + 1 | 0; //@line 53323
            if ($138 >>> 0 > 62) {
             label = 52; //@line 53326
             break L7;
            } else {
             $$1545$i$us = $$0544755$i$us; //@line 53329
             $$1550$i$us = $138; //@line 53329
             $$1552$i$us = $$0551753$i$us; //@line 53329
             $$3$i$us = $$2$i$us; //@line 53329
            }
           }
          } while (0);
          $144 = $$1552$i$us + 1 | 0; //@line 53333
          $131 = HEAP8[$128 + $144 >> 0] | 0; //@line 53335
          if (!($131 << 24 >> 24)) {
           $$0544$lcssa$i = $$1545$i$us; //@line 53338
           $$0549$lcssa$i = $$1550$i$us; //@line 53338
           $$0551$lcssa$i = $144; //@line 53338
           $$1$lcssa$i = $$3$i$us; //@line 53338
           break;
          } else {
           $$0544755$i$us = $$1545$i$us; //@line 53341
           $$0549754$i$us = $$1550$i$us; //@line 53341
           $$0551753$i$us = $144; //@line 53341
           $$1756$i$us = $$3$i$us; //@line 53341
          }
         }
        }
       } while (0);
       HEAP8[$4 + $$0549$lcssa$i >> 0] = 0; //@line 53347
       $$2553$i = $$0551$lcssa$i; //@line 53348
       while (1) {
        $169 = HEAP8[$128 + $$2553$i >> 0] | 0; //@line 53351
        $170 = $169 << 24 >> 24; //@line 53352
        if (!($169 << 24 >> 24)) {
         $$1528$i = 0; //@line 53355
         $$2509$i = -1; //@line 53355
         $$2512$i = 16; //@line 53355
         $$2515$i = 16; //@line 53355
         $$3506$i = -1; //@line 53355
         $$4526$i = -1; //@line 53355
         $$5521$i = -1; //@line 53355
         $$5543$i = 0; //@line 53355
         $$6537$i = 0; //@line 53355
         $467 = 0; //@line 53355
         $468 = 0; //@line 53355
         $470 = 0; //@line 53355
         $471 = 0; //@line 53355
         break;
        }
        $173 = $$2553$i + 1 | 0; //@line 53360
        if (!(_isspace($170) | 0)) {
         label = 57; //@line 53362
         break;
        } else {
         $$2553$i = $173; //@line 53365
        }
       }
       do {
        if ((label | 0) == 57) {
         label = 0; //@line 53370
         L106 : do {
          switch ($170 | 0) {
          case 68:
           {
            $174 = $$2553$i + 2 | 0; //@line 53374
            switch (HEAP8[$128 + $173 >> 0] | 0) {
            case 114:
             {
              $$0527618$ph$i = 0; //@line 53382
              $$10617$ph$i = $174; //@line 53382
              $$1508621$ph$i = HEAP32[$27 >> 2] | 0; //@line 53382
              $$1514620$ph$i = 0; //@line 53382
              $$3525619$ph$i = HEAP32[$28 >> 2] | 0; //@line 53382
              label = 124; //@line 53383
              break L106;
              break;
             }
            case 82:
             {
              $$0527618$ph$i = 0; //@line 53390
              $$10617$ph$i = $174; //@line 53390
              $$1508621$ph$i = HEAP32[$29 >> 2] | 0; //@line 53390
              $$1514620$ph$i = 0; //@line 53390
              $$3525619$ph$i = HEAP32[$30 >> 2] | 0; //@line 53390
              label = 124; //@line 53391
              break L106;
              break;
             }
            default:
             {
              label = 61; //@line 53396
              break L7;
             }
            }
            break;
           }
          case 65:
           {
            $182 = $$2553$i + 2 | 0; //@line 53403
            switch (HEAP8[$128 + $173 >> 0] | 0) {
            case 114:
             {
              $$sink598$i = 15; //@line 53409
              break;
             }
            case 82:
             {
              $$sink598$i = 16; //@line 53413
              break;
             }
            default:
             {
              label = 64; //@line 53417
              break L7;
             }
            }
            $187 = HEAP32[$2 + ($$sink598$i << 2) >> 2] | 0; //@line 53422
            $189 = HEAP32[$3 + ($$sink598$i << 2) >> 2] | 0; //@line 53424
            switch (HEAP8[$128 + $182 >> 0] | 0) {
            case 112:
             {
              $$0527618$ph$i = 0; //@line 53431
              $$10617$ph$i = $$2553$i + 3 | 0; //@line 53431
              $$1508621$ph$i = $187; //@line 53431
              $$1514620$ph$i = 4; //@line 53431
              $$3525619$ph$i = $189; //@line 53431
              label = 124; //@line 53432
              break L106;
              break;
             }
            case 80:
             {
              $$0527618$ph$i = 0; //@line 53438
              $$10617$ph$i = $$2553$i + 3 | 0; //@line 53438
              $$1508621$ph$i = $187; //@line 53438
              $$1514620$ph$i = 3; //@line 53438
              $$3525619$ph$i = $189; //@line 53438
              label = 124; //@line 53439
              break L106;
              break;
             }
            default:
             {
              $$10617$i = $182; //@line 53444
              $$1508621$i = $187; //@line 53444
              $$1514620$i = 1; //@line 53444
              $$3525619$i = $189; //@line 53444
              $$4542$i = 1; //@line 53444
              label = 125; //@line 53445
              break L106;
             }
            }
            break;
           }
          case 76:
           {
            $$10626$i = $173; //@line 53452
            $$1508630$i = -1; //@line 53452
            $$1514629$i = 8; //@line 53452
            $$3525628$i = -1; //@line 53452
            label = 123; //@line 53453
            break;
           }
          case 35:
           {
            $195 = $$2553$i + 2 | 0; //@line 53457
            do {
             switch (HEAP8[$128 + $173 >> 0] | 0) {
             case 122:
              {
               $$10626$i = $195; //@line 53464
               $$1508630$i = -1; //@line 53464
               $$1514629$i = 11; //@line 53464
               $$3525628$i = -1; //@line 53464
               label = 123; //@line 53465
               break L106;
               break;
              }
             case 48:
              {
               $$10626$i = $195; //@line 53470
               $$1508630$i = -1; //@line 53470
               $$1514629$i = 12; //@line 53470
               $$3525628$i = -1; //@line 53470
               label = 123; //@line 53471
               break L106;
               break;
              }
             case 49:
              {
               $$10626$i = $195; //@line 53476
               $$1508630$i = -1; //@line 53476
               $$1514629$i = 13; //@line 53476
               $$3525628$i = -1; //@line 53476
               label = 123; //@line 53477
               break L106;
               break;
              }
             case 50:
              {
               $$10626$i = $195; //@line 53482
               $$1508630$i = -1; //@line 53482
               $$1514629$i = 14; //@line 53482
               $$3525628$i = -1; //@line 53482
               label = 123; //@line 53483
               break L106;
               break;
              }
             case 105:
              {
               $$0527618$ph$i = 1; //@line 53492
               $$10617$ph$i = $195; //@line 53492
               $$1508621$ph$i = HEAP32[$31 >> 2] << 24 >> 24; //@line 53492
               $$1514620$ph$i = 15; //@line 53492
               $$3525619$ph$i = HEAP32[$41 >> 2] | 0; //@line 53492
               label = 124; //@line 53493
               break L106;
               break;
              }
             case 106:
              {
               $$0527618$ph$i = 3; //@line 53499
               $$10617$ph$i = $195; //@line 53499
               $$1508621$ph$i = $523; //@line 53499
               $$1514620$ph$i = 15; //@line 53499
               $$3525619$ph$i = HEAP32[$42 >> 2] | 0; //@line 53499
               label = 124; //@line 53500
               break L106;
               break;
              }
             case 74:
              {
               $$0527618$ph$i = 2; //@line 53507
               $$10617$ph$i = $195; //@line 53507
               $$1508621$ph$i = HEAP32[$32 >> 2] | 0; //@line 53507
               $$1514620$ph$i = 15; //@line 53507
               $$3525619$ph$i = HEAP32[$43 >> 2] | 0; //@line 53507
               label = 124; //@line 53508
               break L106;
               break;
              }
             case 107:
              {
               $$0527618$ph$i = 4; //@line 53515
               $$10617$ph$i = $195; //@line 53515
               $$1508621$ph$i = HEAP32[$33 >> 2] | 0; //@line 53515
               $$1514620$ph$i = 15; //@line 53515
               $$3525619$ph$i = HEAP32[$44 >> 2] | 0; //@line 53515
               label = 124; //@line 53516
               break L106;
               break;
              }
             case 75:
              {
               $$0527618$ph$i = 5; //@line 53523
               $$10617$ph$i = $195; //@line 53523
               $$1508621$ph$i = HEAP32[$34 >> 2] | 0; //@line 53523
               $$1514620$ph$i = 15; //@line 53523
               $$3525619$ph$i = HEAP32[$45 >> 2] | 0; //@line 53523
               label = 124; //@line 53524
               break L106;
               break;
              }
             case 69:
              {
               $$0527618$ph$i = 6; //@line 53530
               $$10617$ph$i = $195; //@line 53530
               $$1508621$ph$i = $121; //@line 53530
               $$1514620$ph$i = 15; //@line 53530
               $$3525619$ph$i = HEAP32[$47 >> 2] | 0; //@line 53530
               label = 124; //@line 53531
               break L106;
               break;
              }
             case 112:
              {
               $$0527618$ph$i = 7; //@line 53538
               $$10617$ph$i = $195; //@line 53538
               $$1508621$ph$i = HEAP32[$48 >> 2] | 0; //@line 53538
               $$1514620$ph$i = 15; //@line 53538
               $$3525619$ph$i = HEAP32[$49 >> 2] | 0; //@line 53538
               label = 124; //@line 53539
               break L106;
               break;
              }
             default:
              {
               label = 79; //@line 53544
               break L7;
              }
             }
            } while (0);
            break;
           }
          case 100:
           {
            $212 = HEAP32[$35 >> 2] | 0; //@line 53552
            $213 = HEAP32[$36 >> 2] | 0; //@line 53553
            L131 : do {
             switch ($213 | 0) {
             case 7:
              {
               switch ($212 | 0) {
               case 7:
               case 6:
               case 5:
                {
                 $$5572$i = $$2569780$i; //@line 53559
                 $$7$i = $$1$lcssa$i; //@line 53559
                 break L52;
                 break;
                }
               case 0:
                {
                 $$0$i613$i = 7; //@line 53564
                 $$1523$i = -1; //@line 53564
                 $$1539$i = 0; //@line 53564
                 break L131;
                 break;
                }
               case 1:
                {
                 $$0$i613$i = 8; //@line 53569
                 $$1523$i = -1; //@line 53569
                 $$1539$i = 0; //@line 53569
                 break L131;
                 break;
                }
               case 2:
                {
                 $$0$i613$i = 9; //@line 53574
                 $$1523$i = -1; //@line 53574
                 $$1539$i = 0; //@line 53574
                 break L131;
                 break;
                }
               case 3:
                {
                 $$0$i613$i = 10; //@line 53579
                 $$1523$i = -1; //@line 53579
                 $$1539$i = 0; //@line 53579
                 break L131;
                 break;
                }
               case 4:
                {
                 $$0$i613$i = 11; //@line 53584
                 $$1523$i = -1; //@line 53584
                 $$1539$i = 0; //@line 53584
                 break L131;
                 break;
                }
               default:
                {
                 label = 86; //@line 53589
                 break L7;
                }
               }
               break;
              }
             case 1:
             case 2:
             case 3:
             case 4:
             case 5:
             case 6:
             case 0:
              {
               $$0$i613$i = $213; //@line 53597
               $$1523$i = HEAP32[$37 >> 2] | 0; //@line 53597
               $$1539$i = 1; //@line 53597
               break;
              }
             default:
              {
               label = 86; //@line 53601
               break L7;
              }
             }
            } while (0);
            do {
             if ((HEAP8[$128 + $173 >> 0] | 0) == 91) {
              $218 = $$2553$i + 2 | 0; //@line 53611
              $219 = $128 + $218 | 0; //@line 53612
              if ((HEAP8[$219 >> 0] | 0) == 33) {
               $$3554$i = $218; //@line 53616
               while (1) {
                if ((__ZL13mode_from_strPKc($128 + ($$3554$i + 1) | 0) | 0) == ($$0$i613$i | 0)) {
                 $$5572$i = $$2569780$i; //@line 53623
                 $$7$i = $$1$lcssa$i; //@line 53623
                 break L52;
                }
                $226 = $$3554$i + 5 | 0; //@line 53626
                if ((HEAP8[$128 + $226 >> 0] | 0) == 44) {
                 $$3554$i = $226; //@line 53631
                } else {
                 break;
                }
               }
               $$0513$i = $$0$i613$i; //@line 53637
               $$6557$i = $$3554$i + 6 | 0; //@line 53637
               break;
              }
              $234 = (HEAP8[$128 + ($$2553$i + 6) >> 0] | 0) == 45; //@line 53643
              $236 = (__ZL13mode_from_strPKc($219) | 0) == ($$0$i613$i | 0); //@line 53645
              if ($234) {
               if (!$236) {
                $$5572$i = $$2569780$i; //@line 53648
                $$7$i = $$1$lcssa$i; //@line 53648
                break L52;
               }
               $$0513$i = __ZL13mode_from_strPKc($219 + 5 | 0) | 0; //@line 53654
               $$6557$i = $$2553$i + 12 | 0; //@line 53654
               break;
              }
              if ($236) {
               $$5556$i = $218; //@line 53658
              } else {
               $$4555768$i = $218; //@line 53660
               while (1) {
                if ((HEAP8[$128 + ($$4555768$i + 4) >> 0] | 0) == 93) {
                 $$5572$i = $$2569780$i; //@line 53667
                 $$7$i = $$1$lcssa$i; //@line 53667
                 break L52;
                }
                $244 = $$4555768$i + 5 | 0; //@line 53670
                if ((__ZL13mode_from_strPKc($128 + $244 | 0) | 0) == ($$0$i613$i | 0)) {
                 $$5556$i = $244; //@line 53675
                 break;
                } else {
                 $$4555768$i = $244; //@line 53678
                }
               }
              }
              while (1) {
               $251 = $$5556$i + 1 | 0; //@line 53686
               if ((HEAP8[$128 + $$5556$i >> 0] | 0) == 93) {
                $$10$i = $251; //@line 53688
                $$1508$i = $212; //@line 53688
                $$1514$i = $$0$i613$i; //@line 53688
                $$3525$i = $$1523$i; //@line 53688
                $$3541$i = $$1539$i; //@line 53688
                label = 122; //@line 53689
                break L106;
               } else {
                $$5556$i = $251; //@line 53692
               }
              }
             } else {
              $$0513$i = $$0$i613$i; //@line 53696
              $$6557$i = $173; //@line 53696
             }
            } while (0);
            if (($$0513$i + -9 | 0) >>> 0 < 3) {
             $$5572$i = $$2569780$i; //@line 53702
             $$7$i = $$1$lcssa$i; //@line 53702
             break L52;
            } else {
             $$10$i = $$6557$i; //@line 53705
             $$1508$i = $212; //@line 53705
             $$1514$i = $$0513$i; //@line 53705
             $$3525$i = $$1523$i; //@line 53705
             $$3541$i = $$1539$i; //@line 53705
             label = 122; //@line 53706
            }
            break;
           }
          case 115:
           {
            $252 = HEAP32[$38 >> 2] | 0; //@line 53711
            $253 = HEAP32[$39 >> 2] | 0; //@line 53712
            L160 : do {
             switch ($253 | 0) {
             case 7:
              {
               switch ($252 | 0) {
               case 7:
               case 6:
               case 5:
                {
                 $$5572$i = $$2569780$i; //@line 53718
                 $$7$i = $$1$lcssa$i; //@line 53718
                 break L52;
                 break;
                }
               case 0:
                {
                 $$0$i607615$i = 7; //@line 53723
                 $$2524$i = -1; //@line 53723
                 $$2540$i = 0; //@line 53723
                 break L160;
                 break;
                }
               case 1:
                {
                 $$0$i607615$i = 8; //@line 53728
                 $$2524$i = -1; //@line 53728
                 $$2540$i = 0; //@line 53728
                 break L160;
                 break;
                }
               case 2:
                {
                 $$0$i607615$i = 9; //@line 53733
                 $$2524$i = -1; //@line 53733
                 $$2540$i = 0; //@line 53733
                 break L160;
                 break;
                }
               case 3:
                {
                 $$0$i607615$i = 10; //@line 53738
                 $$2524$i = -1; //@line 53738
                 $$2540$i = 0; //@line 53738
                 break L160;
                 break;
                }
               case 4:
                {
                 $$0$i607615$i = 11; //@line 53743
                 $$2524$i = -1; //@line 53743
                 $$2540$i = 0; //@line 53743
                 break L160;
                 break;
                }
               default:
                {
                 label = 107; //@line 53748
                 break L7;
                }
               }
               break;
              }
             case 1:
             case 2:
             case 3:
             case 4:
             case 5:
             case 6:
             case 0:
              {
               $$0$i607615$i = $253; //@line 53756
               $$2524$i = HEAP32[$40 >> 2] | 0; //@line 53756
               $$2540$i = 1; //@line 53756
               break;
              }
             default:
              {
               label = 107; //@line 53760
               break L7;
              }
             }
            } while (0);
            if ((HEAP8[$128 + $173 >> 0] | 0) == 91) {
             $258 = $$2553$i + 2 | 0; //@line 53769
             $259 = $128 + $258 | 0; //@line 53770
             if ((HEAP8[$259 >> 0] | 0) == 33) {
              $$7558$i = $258; //@line 53774
              while (1) {
               if ((__ZL13mode_from_strPKc($128 + ($$7558$i + 1) | 0) | 0) == ($$0$i607615$i | 0)) {
                $$5572$i = $$2569780$i; //@line 53781
                $$7$i = $$1$lcssa$i; //@line 53781
                break L52;
               }
               $266 = $$7558$i + 5 | 0; //@line 53784
               if ((HEAP8[$128 + $266 >> 0] | 0) == 44) {
                $$7558$i = $266; //@line 53789
               } else {
                break;
               }
              }
              $$10$i = $$7558$i + 6 | 0; //@line 53795
              $$1508$i = $252; //@line 53795
              $$1514$i = $$0$i607615$i; //@line 53795
              $$3525$i = $$2524$i; //@line 53795
              $$3541$i = $$2540$i; //@line 53795
              label = 122; //@line 53796
              break L106;
             }
             $274 = (HEAP8[$128 + ($$2553$i + 6) >> 0] | 0) == 45; //@line 53802
             $276 = (__ZL13mode_from_strPKc($259) | 0) == ($$0$i607615$i | 0); //@line 53804
             if ($274) {
              if (!$276) {
               $$5572$i = $$2569780$i; //@line 53807
               $$7$i = $$1$lcssa$i; //@line 53807
               break L52;
              }
              $$10$i = $$2553$i + 12 | 0; //@line 53813
              $$1508$i = $252; //@line 53813
              $$1514$i = __ZL13mode_from_strPKc($259 + 5 | 0) | 0; //@line 53813
              $$3525$i = $$2524$i; //@line 53813
              $$3541$i = $$2540$i; //@line 53813
              label = 122; //@line 53814
              break L106;
             }
             if ($276) {
              $$9$i = $258; //@line 53818
             } else {
              $$8765$i = $258; //@line 53820
              while (1) {
               if ((HEAP8[$128 + ($$8765$i + 4) >> 0] | 0) == 93) {
                $$5572$i = $$2569780$i; //@line 53827
                $$7$i = $$1$lcssa$i; //@line 53827
                break L52;
               }
               $284 = $$8765$i + 5 | 0; //@line 53830
               if ((__ZL13mode_from_strPKc($128 + $284 | 0) | 0) == ($$0$i607615$i | 0)) {
                $$9$i = $284; //@line 53835
                break;
               } else {
                $$8765$i = $284; //@line 53838
               }
              }
             }
             while (1) {
              $291 = $$9$i + 1 | 0; //@line 53846
              if ((HEAP8[$128 + $$9$i >> 0] | 0) == 93) {
               $$10$i = $291; //@line 53848
               $$1508$i = $252; //@line 53848
               $$1514$i = $$0$i607615$i; //@line 53848
               $$3525$i = $$2524$i; //@line 53848
               $$3541$i = $$2540$i; //@line 53848
               label = 122; //@line 53849
               break;
              } else {
               $$9$i = $291; //@line 53852
              }
             }
            } else {
             $$10$i = $173; //@line 53856
             $$1508$i = $252; //@line 53856
             $$1514$i = $$0$i607615$i; //@line 53856
             $$3525$i = $$2524$i; //@line 53856
             $$3541$i = $$2540$i; //@line 53856
             label = 122; //@line 53857
            }
            break;
           }
          default:
           {
            label = 121; //@line 53862
            break L7;
           }
          }
         } while (0);
         if ((label | 0) == 122) {
          label = 0; //@line 53868
          switch ($$1514$i | 0) {
          case 0:
          case 1:
          case 2:
          case 3:
          case 4:
          case 5:
          case 6:
          case 15:
           {
            $$10617$i = $$10$i; //@line 53871
            $$1508621$i = $$1508$i; //@line 53871
            $$1514620$i = $$1514$i; //@line 53871
            $$3525619$i = $$3525$i; //@line 53871
            $$4542$i = $$3541$i; //@line 53871
            label = 125; //@line 53872
            break;
           }
          default:
           {
            $$10626$i = $$10$i; //@line 53876
            $$1508630$i = $$1508$i; //@line 53876
            $$1514629$i = $$1514$i; //@line 53876
            $$3525628$i = $$3525$i; //@line 53876
            label = 123; //@line 53877
           }
          }
         } else if ((label | 0) == 124) {
          label = 0; //@line 53882
          $$0527618636$i = $$0527618$ph$i; //@line 53884
          $$10617637$i = $$10617$ph$i; //@line 53884
          $$1508621633$i = $$1508621$ph$i; //@line 53884
          $$1514620634$i = $$1514620$ph$i; //@line 53884
          $$3525619635$i = $$3525619$ph$i; //@line 53884
          $$4542638$i = 1; //@line 53884
          $409 = ($$0544$lcssa$i | 0) == 0; //@line 53884
         }
         if ((label | 0) == 123) {
          label = 0; //@line 53887
          $$0527618636$i = 0; //@line 53889
          $$10617637$i = $$10626$i; //@line 53889
          $$1508621633$i = $$1508630$i; //@line 53889
          $$1514620634$i = $$1514629$i; //@line 53889
          $$3525619635$i = $$3525628$i; //@line 53889
          $$4542638$i = 0; //@line 53889
          $409 = ($$0544$lcssa$i | 0) == 0; //@line 53889
         } else if ((label | 0) == 125) {
          label = 0; //@line 53892
          $295 = ($$0544$lcssa$i | 0) == 0; //@line 53894
          if ($295 & ($$1514620$i | 0) == 1) {
           $$5572$i = $$2569780$i; //@line 53897
           $$7$i = $$1$lcssa$i; //@line 53897
           break L52;
          } else {
           $$0527618636$i = 0; //@line 53900
           $$10617637$i = $$10617$i; //@line 53900
           $$1508621633$i = $$1508621$i; //@line 53900
           $$1514620634$i = $$1514620$i; //@line 53900
           $$3525619635$i = $$3525619$i; //@line 53900
           $$4542638$i = $$4542$i; //@line 53900
           $409 = $295; //@line 53900
          }
         }
         if ((HEAP8[$128 + $$10617637$i >> 0] | 0) == 44) {
          $300 = $$10617637$i + 2 | 0; //@line 53908
          L194 : do {
           switch (HEAP8[$128 + ($$10617637$i + 1) >> 0] | 0) {
           case 68:
            {
             switch (HEAP8[$128 + $300 >> 0] | 0) {
             case 114:
              {
               $$sink602$i = 15; //@line 53920
               break;
              }
             case 82:
              {
               $$sink602$i = 16; //@line 53924
               break;
              }
             default:
              {
               label = 130; //@line 53928
               break L7;
              }
             }
             $310 = HEAP32[$3 + ($$sink602$i << 2) >> 2] | 0; //@line 53935
             if ($310 >>> 0 > 31) {
              label = 132; //@line 53938
              break L7;
             } else {
              $$1511$ph$i = 0; //@line 53941
              $$2505$ph$i = HEAP32[$2 + ($$sink602$i << 2) >> 2] | 0; //@line 53941
              $$4520$ph$i = $310; //@line 53941
              $$4535$ph$i = 1; //@line 53941
              label = 190; //@line 53942
             }
             break;
            }
           case 65:
            {
             $312 = $$10617637$i + 3 | 0; //@line 53947
             switch (HEAP8[$128 + $300 >> 0] | 0) {
             case 114:
              {
               $$sink656$i = 15; //@line 53953
               label = 136; //@line 53954
               break;
              }
             case 82:
              {
               $$sink656$i = 16; //@line 53958
               label = 136; //@line 53959
               break;
              }
             case 120:
              {
               $$1504642$i = 0; //@line 53963
               $$1517641$i = 0; //@line 53963
               $$1532640$i = 0; //@line 53963
               break;
              }
             default:
              {
               label = 135; //@line 53967
               break L7;
              }
             }
             if ((label | 0) == 136) {
              label = 0; //@line 53972
              $319 = HEAP32[$3 + ($$sink656$i << 2) >> 2] | 0; //@line 53976
              if ($319 >>> 0 > 31) {
               label = 137; //@line 53979
               break L7;
              } else {
               $$1504642$i = HEAP32[$2 + ($$sink656$i << 2) >> 2] | 0; //@line 53982
               $$1517641$i = $319; //@line 53982
               $$1532640$i = 1; //@line 53982
              }
             }
             switch (HEAP8[$128 + $312 >> 0] | 0) {
             case 112:
              {
               $$1511$ph$i = 4; //@line 53990
               $$2505$ph$i = $$1504642$i; //@line 53990
               $$4520$ph$i = $$1517641$i; //@line 53990
               $$4535$ph$i = $$1532640$i; //@line 53990
               label = 190; //@line 53991
               break L194;
               break;
              }
             case 80:
              {
               break;
              }
             default:
              {
               $$1511$i = 1; //@line 53999
               $$2505$i = $$1504642$i; //@line 53999
               $$4520$i = $$1517641$i; //@line 53999
               $$4535$i = $$1532640$i; //@line 53999
               label = 191; //@line 54000
               break L194;
              }
             }
             $$1511$ph$i = 3; //@line 54004
             $$2505$ph$i = $$1504642$i; //@line 54004
             $$4520$ph$i = $$1517641$i; //@line 54004
             $$4535$ph$i = $$1532640$i; //@line 54004
             label = 190; //@line 54005
             break;
            }
           case 76:
            {
             $$1511$ph$i = 8; //@line 54009
             $$2505$ph$i = -1; //@line 54009
             $$4520$ph$i = -1; //@line 54009
             $$4535$ph$i = 0; //@line 54009
             label = 190; //@line 54010
             break;
            }
           case 35:
            {
             switch (HEAP8[$128 + $300 >> 0] | 0) {
             case 122:
              {
               $$1511$ph$i = 11; //@line 54019
               $$2505$ph$i = -1; //@line 54019
               $$4520$ph$i = -1; //@line 54019
               $$4535$ph$i = 0; //@line 54019
               label = 190; //@line 54020
               break L194;
               break;
              }
             case 48:
              {
               $$1511$ph$i = 12; //@line 54025
               $$2505$ph$i = -1; //@line 54025
               $$4520$ph$i = -1; //@line 54025
               $$4535$ph$i = 0; //@line 54025
               label = 190; //@line 54026
               break L194;
               break;
              }
             case 49:
              {
               $$1511$ph$i = 13; //@line 54031
               $$2505$ph$i = -1; //@line 54031
               $$4520$ph$i = -1; //@line 54031
               $$4535$ph$i = 0; //@line 54031
               label = 190; //@line 54032
               break L194;
               break;
              }
             case 50:
              {
               $$1511$ph$i = 14; //@line 54037
               $$2505$ph$i = -1; //@line 54037
               $$4520$ph$i = -1; //@line 54037
               $$4535$ph$i = 0; //@line 54037
               label = 190; //@line 54038
               break L194;
               break;
              }
             case 105:
              {
               $$1511$ph$i = 15; //@line 54046
               $$2505$ph$i = HEAP32[$31 >> 2] << 24 >> 24; //@line 54046
               $$4520$ph$i = -1; //@line 54046
               $$4535$ph$i = 0; //@line 54046
               label = 190; //@line 54047
               break L194;
               break;
              }
             case 106:
              {
               $$1511$ph$i = 15; //@line 54052
               $$2505$ph$i = $523; //@line 54052
               $$4520$ph$i = -1; //@line 54052
               $$4535$ph$i = 0; //@line 54052
               label = 190; //@line 54053
               break L194;
               break;
              }
             case 74:
              {
               $$1511$ph$i = 15; //@line 54059
               $$2505$ph$i = HEAP32[$32 >> 2] | 0; //@line 54059
               $$4520$ph$i = -1; //@line 54059
               $$4535$ph$i = 0; //@line 54059
               label = 190; //@line 54060
               break L194;
               break;
              }
             case 107:
              {
               $$1511$ph$i = 15; //@line 54066
               $$2505$ph$i = HEAP32[$33 >> 2] | 0; //@line 54066
               $$4520$ph$i = -1; //@line 54066
               $$4535$ph$i = 0; //@line 54066
               label = 190; //@line 54067
               break L194;
               break;
              }
             case 75:
              {
               $$1511$ph$i = 15; //@line 54073
               $$2505$ph$i = HEAP32[$34 >> 2] | 0; //@line 54073
               $$4520$ph$i = -1; //@line 54073
               $$4535$ph$i = 0; //@line 54073
               label = 190; //@line 54074
               break L194;
               break;
              }
             default:
              {
               label = 149; //@line 54079
               break L7;
              }
             }
             break;
            }
           case 100:
            {
             $332 = HEAP32[$35 >> 2] | 0; //@line 54086
             $333 = HEAP32[$36 >> 2] | 0; //@line 54087
             L217 : do {
              switch ($333 | 0) {
              case 7:
               {
                switch ($332 | 0) {
                case 7:
                case 6:
                case 5:
                 {
                  $$5572$i = $$2569780$i; //@line 54093
                  $$7$i = $$1$lcssa$i; //@line 54093
                  break L52;
                  break;
                 }
                case 0:
                 {
                  $$0$i609643$i = 7; //@line 54098
                  $$2518$i = -1; //@line 54098
                  $$2533$i = 0; //@line 54098
                  break L217;
                  break;
                 }
                case 1:
                 {
                  $$0$i609643$i = 8; //@line 54103
                  $$2518$i = -1; //@line 54103
                  $$2533$i = 0; //@line 54103
                  break L217;
                  break;
                 }
                case 2:
                 {
                  $$0$i609643$i = 9; //@line 54108
                  $$2518$i = -1; //@line 54108
                  $$2533$i = 0; //@line 54108
                  break L217;
                  break;
                 }
                case 3:
                 {
                  $$0$i609643$i = 10; //@line 54113
                  $$2518$i = -1; //@line 54113
                  $$2533$i = 0; //@line 54113
                  break L217;
                  break;
                 }
                case 4:
                 {
                  $$0$i609643$i = 11; //@line 54118
                  $$2518$i = -1; //@line 54118
                  $$2533$i = 0; //@line 54118
                  break L217;
                  break;
                 }
                default:
                 {
                  label = 156; //@line 54123
                  break L7;
                 }
                }
                break;
               }
              case 1:
              case 2:
              case 3:
              case 4:
              case 5:
              case 6:
              case 0:
               {
                $$0$i609643$i = $333; //@line 54131
                $$2518$i = HEAP32[$37 >> 2] | 0; //@line 54131
                $$2533$i = 1; //@line 54131
                break;
               }
              default:
               {
                label = 156; //@line 54135
                break L7;
               }
              }
             } while (0);
             L226 : do {
              if ((HEAP8[$128 + $300 >> 0] | 0) == 91) {
               $338 = $$10617637$i + 3 | 0; //@line 54145
               $339 = $128 + $338 | 0; //@line 54146
               if ((HEAP8[$339 >> 0] | 0) == 33) {
                $$11$i = $338; //@line 54150
                while (1) {
                 if ((__ZL13mode_from_strPKc($128 + ($$11$i + 1) | 0) | 0) == ($$0$i609643$i | 0)) {
                  $$5572$i = $$2569780$i; //@line 54157
                  $$7$i = $$1$lcssa$i; //@line 54157
                  break L52;
                 }
                 $$11$i = $$11$i + 5 | 0; //@line 54160
                 if ((HEAP8[$128 + $$11$i >> 0] | 0) != 44) {
                  $$0510$i = $$0$i609643$i; //@line 54167
                  break L226;
                 }
                }
               }
               $353 = (HEAP8[$128 + ($$10617637$i + 7) >> 0] | 0) == 45; //@line 54175
               $355 = (__ZL13mode_from_strPKc($339) | 0) == ($$0$i609643$i | 0); //@line 54177
               if ($353) {
                if (!$355) {
                 $$5572$i = $$2569780$i; //@line 54180
                 $$7$i = $$1$lcssa$i; //@line 54180
                 break L52;
                }
                $$0510$i = __ZL13mode_from_strPKc($339 + 5 | 0) | 0; //@line 54185
                break;
               }
               if ($355) {
                $$13$i = $338; //@line 54189
               } else {
                $$12774$i = $338; //@line 54191
                while (1) {
                 if ((HEAP8[$128 + ($$12774$i + 4) >> 0] | 0) == 93) {
                  $$5572$i = $$2569780$i; //@line 54198
                  $$7$i = $$1$lcssa$i; //@line 54198
                  break L52;
                 }
                 $362 = $$12774$i + 5 | 0; //@line 54201
                 if ((__ZL13mode_from_strPKc($128 + $362 | 0) | 0) == ($$0$i609643$i | 0)) {
                  $$13$i = $362; //@line 54206
                  break;
                 } else {
                  $$12774$i = $362; //@line 54209
                 }
                }
               }
               while (1) {
                if ((HEAP8[$128 + $$13$i >> 0] | 0) == 93) {
                 $$1511$i = $$0$i609643$i; //@line 54219
                 $$2505$i = $332; //@line 54219
                 $$4520$i = $$2518$i; //@line 54219
                 $$4535$i = $$2533$i; //@line 54219
                 label = 191; //@line 54220
                 break L194;
                } else {
                 $$13$i = $$13$i + 1 | 0; //@line 54223
                }
               }
              } else {
               $$0510$i = $$0$i609643$i; //@line 54227
              }
             } while (0);
             if (($$0510$i + -9 | 0) >>> 0 < 3) {
              $$5572$i = $$2569780$i; //@line 54233
              $$7$i = $$1$lcssa$i; //@line 54233
              break L52;
             } else {
              $$1511$i = $$0510$i; //@line 54236
              $$2505$i = $332; //@line 54236
              $$4520$i = $$2518$i; //@line 54236
              $$4535$i = $$2533$i; //@line 54236
              label = 191; //@line 54237
             }
             break;
            }
           case 115:
            {
             $370 = HEAP32[$38 >> 2] | 0; //@line 54242
             $371 = HEAP32[$39 >> 2] | 0; //@line 54243
             L245 : do {
              switch ($371 | 0) {
              case 7:
               {
                switch ($370 | 0) {
                case 7:
                case 6:
                case 5:
                 {
                  $$5572$i = $$2569780$i; //@line 54249
                  $$7$i = $$1$lcssa$i; //@line 54249
                  break L52;
                  break;
                 }
                case 0:
                 {
                  $$0$i611645$i = 7; //@line 54254
                  $$3519$i = -1; //@line 54254
                  $$3534$i = 0; //@line 54254
                  break L245;
                  break;
                 }
                case 1:
                 {
                  $$0$i611645$i = 8; //@line 54259
                  $$3519$i = -1; //@line 54259
                  $$3534$i = 0; //@line 54259
                  break L245;
                  break;
                 }
                case 2:
                 {
                  $$0$i611645$i = 9; //@line 54264
                  $$3519$i = -1; //@line 54264
                  $$3534$i = 0; //@line 54264
                  break L245;
                  break;
                 }
                case 3:
                 {
                  $$0$i611645$i = 10; //@line 54269
                  $$3519$i = -1; //@line 54269
                  $$3534$i = 0; //@line 54269
                  break L245;
                  break;
                 }
                case 4:
                 {
                  $$0$i611645$i = 11; //@line 54274
                  $$3519$i = -1; //@line 54274
                  $$3534$i = 0; //@line 54274
                  break L245;
                  break;
                 }
                default:
                 {
                  label = 176; //@line 54279
                  break L7;
                 }
                }
                break;
               }
              case 1:
              case 2:
              case 3:
              case 4:
              case 5:
              case 6:
              case 0:
               {
                $$0$i611645$i = $371; //@line 54287
                $$3519$i = HEAP32[$40 >> 2] | 0; //@line 54287
                $$3534$i = 1; //@line 54287
                break;
               }
              default:
               {
                label = 176; //@line 54291
                break L7;
               }
              }
             } while (0);
             if ((HEAP8[$128 + $300 >> 0] | 0) == 91) {
              $376 = $$10617637$i + 3 | 0; //@line 54300
              $377 = $128 + $376 | 0; //@line 54301
              if ((HEAP8[$377 >> 0] | 0) == 33) {
               $$14$i = $376; //@line 54305
               while (1) {
                if ((__ZL13mode_from_strPKc($128 + ($$14$i + 1) | 0) | 0) == ($$0$i611645$i | 0)) {
                 $$5572$i = $$2569780$i; //@line 54312
                 $$7$i = $$1$lcssa$i; //@line 54312
                 break L52;
                }
                $$14$i = $$14$i + 5 | 0; //@line 54315
                if ((HEAP8[$128 + $$14$i >> 0] | 0) != 44) {
                 $$1511$i = $$0$i611645$i; //@line 54322
                 $$2505$i = $370; //@line 54322
                 $$4520$i = $$3519$i; //@line 54322
                 $$4535$i = $$3534$i; //@line 54322
                 label = 191; //@line 54323
                 break L194;
                }
               }
              }
              $391 = (HEAP8[$128 + ($$10617637$i + 7) >> 0] | 0) == 45; //@line 54331
              $393 = (__ZL13mode_from_strPKc($377) | 0) == ($$0$i611645$i | 0); //@line 54333
              if ($391) {
               if (!$393) {
                $$5572$i = $$2569780$i; //@line 54336
                $$7$i = $$1$lcssa$i; //@line 54336
                break L52;
               }
               $$1511$i = __ZL13mode_from_strPKc($377 + 5 | 0) | 0; //@line 54341
               $$2505$i = $370; //@line 54341
               $$4520$i = $$3519$i; //@line 54341
               $$4535$i = $$3534$i; //@line 54341
               label = 191; //@line 54342
               break L194;
              }
              if ($393) {
               $$16$i = $376; //@line 54346
              } else {
               $$15771$i = $376; //@line 54348
               while (1) {
                if ((HEAP8[$128 + ($$15771$i + 4) >> 0] | 0) == 93) {
                 $$5572$i = $$2569780$i; //@line 54355
                 $$7$i = $$1$lcssa$i; //@line 54355
                 break L52;
                }
                $400 = $$15771$i + 5 | 0; //@line 54358
                if ((__ZL13mode_from_strPKc($128 + $400 | 0) | 0) == ($$0$i611645$i | 0)) {
                 $$16$i = $400; //@line 54363
                 break;
                } else {
                 $$15771$i = $400; //@line 54366
                }
               }
              }
              while (1) {
               if ((HEAP8[$128 + $$16$i >> 0] | 0) == 93) {
                $$1511$i = $$0$i611645$i; //@line 54376
                $$2505$i = $370; //@line 54376
                $$4520$i = $$3519$i; //@line 54376
                $$4535$i = $$3534$i; //@line 54376
                label = 191; //@line 54377
                break;
               } else {
                $$16$i = $$16$i + 1 | 0; //@line 54380
               }
              }
             } else {
              $$1511$i = $$0$i611645$i; //@line 54384
              $$2505$i = $370; //@line 54384
              $$4520$i = $$3519$i; //@line 54384
              $$4535$i = $$3534$i; //@line 54384
              label = 191; //@line 54385
             }
             break;
            }
           default:
            {
             label = 189; //@line 54390
             break L7;
            }
           }
          } while (0);
          if ((label | 0) == 190) {
           label = 0; //@line 54396
           $$1528$i = $$0527618636$i; //@line 54399
           $$2509$i = $$1508621633$i; //@line 54399
           $$2512$i = $$1511$ph$i; //@line 54399
           $$2515$i = $$1514620634$i; //@line 54399
           $$3506$i = $$2505$ph$i; //@line 54399
           $$4526$i = $$3525619635$i; //@line 54399
           $$5521$i = $$4520$ph$i; //@line 54399
           $$5543$i = $$4542638$i; //@line 54399
           $$6537$i = $$1511$ph$i >>> 0 < 7 ? $$4535$ph$i : 0; //@line 54399
           $467 = 0; //@line 54399
           $468 = 0; //@line 54399
           $470 = 16384; //@line 54399
           $471 = 8192; //@line 54399
           break;
          } else if ((label | 0) == 191) {
           label = 0; //@line 54403
           if ($409 & ($$1511$i | 0) == 1) {
            $$5572$i = $$2569780$i; //@line 54409
            $$7$i = $$1$lcssa$i; //@line 54409
            break L52;
           } else {
            $$1528$i = $$0527618636$i; //@line 54412
            $$2509$i = $$1508621633$i; //@line 54412
            $$2512$i = $$1511$i; //@line 54412
            $$2515$i = $$1514620634$i; //@line 54412
            $$3506$i = $$2505$i; //@line 54412
            $$4526$i = $$3525619635$i; //@line 54412
            $$5521$i = $$4520$i; //@line 54412
            $$5543$i = $$4542638$i; //@line 54412
            $$6537$i = $$1511$i >>> 0 < 7 ? $$4535$i : 0; //@line 54412
            $467 = 0; //@line 54412
            $468 = 0; //@line 54412
            $470 = 16384; //@line 54412
            $471 = 8192; //@line 54412
            break;
           }
          }
         } else {
          $$1528$i = $$0527618636$i; //@line 54417
          $$2509$i = $$1508621633$i; //@line 54417
          $$2512$i = 16; //@line 54417
          $$2515$i = $$1514620634$i; //@line 54417
          $$3506$i = -1; //@line 54417
          $$4526$i = $$3525619635$i; //@line 54417
          $$5521$i = -1; //@line 54417
          $$5543$i = $$4542638$i; //@line 54417
          $$6537$i = 0; //@line 54417
          $467 = 0; //@line 54417
          $468 = 0; //@line 54417
          $470 = 0; //@line 54417
          $471 = 8192; //@line 54417
         }
        }
       } while (0);
       $411 = $$2564$i & 65535; //@line 54422
       $412 = (HEAP32[162823] | 0) + ($411 << 4) + 9 | 0; //@line 54423
       $$hi13 = $412 + 4 | 0; //@line 54425
       (HEAPU8[$$hi13 >> 0] | HEAPU8[$$hi13 + 1 >> 0] << 8) & 65535 | HEAPU8[$$hi13 + 2 >> 0] << 16; //@line 54432
       if (!(0 == 0 ? ((HEAPU8[$412 >> 0] | HEAPU8[$412 + 1 >> 0] << 8 | HEAPU8[$412 + 2 >> 0] << 16 | HEAPU8[$412 + 3 >> 0] << 24) & 16711680 | 0) == 0 : 0)) {
        HEAP32[$vararg_buffer1 >> 2] = $411; //@line 54438
        HEAP32[$vararg_buffer1 + 4 >> 2] = $128; //@line 54440
        _fprintf($26, 106158, $vararg_buffer1) | 0; //@line 54441
       }
       if (($$1$lcssa$i | 0) == -1) {
        $418 = HEAP32[1431] | 0; //@line 54445
        if (!(_strcmp($4, $418) | 0)) {
         $$5$i = 0; //@line 54449
        } else {
         $$4777$i = 0; //@line 54451
         $421 = $418; //@line 54451
         while (1) {
          if (!(HEAP8[$421 >> 0] | 0)) {
           label = 197; //@line 54456
           break L7;
          }
          $423 = $$4777$i + 1 | 0; //@line 54459
          $421 = HEAP32[5720 + ($423 << 3) + 4 >> 2] | 0; //@line 54461
          if (!(_strcmp($4, $421) | 0)) {
           $$5$i = $423; //@line 54465
           break;
          } else {
           $$4777$i = $423; //@line 54468
          }
         }
        }
       } else {
        $$5$i = $$1$lcssa$i; //@line 54473
       }
       $$sink104$i = HEAP32[162823] | 0; //@line 54477
       $428 = $$sink104$i + ($411 << 4) + 9 | 0; //@line 54478
       $$hi18 = $428 + 4 | 0; //@line 54480
       $430 = HEAP32[5720 + ($$5$i << 3) >> 2] << 16 & 16711680; //@line 54489
       $432 = ((HEAPU8[$$hi18 >> 0] | HEAPU8[$$hi18 + 1 >> 0] << 8) & 65535 | HEAPU8[$$hi18 + 2 >> 0] << 16) & 16777215; //@line 54491
       $433 = (HEAPU8[$428 >> 0] | HEAPU8[$428 + 1 >> 0] << 8 | HEAPU8[$428 + 2 >> 0] << 16 | HEAPU8[$428 + 3 >> 0] << 24) & -16711681 | $430; //@line 54492
       HEAP8[$428 >> 0] = $433; //@line 54493
       HEAP8[$428 + 1 >> 0] = $433 >> 8; //@line 54493
       HEAP8[$428 + 2 >> 0] = $433 >> 16; //@line 54493
       HEAP8[$428 + 3 >> 0] = $433 >> 24; //@line 54493
       $$hi24 = $428 + 4 | 0; //@line 54494
       $434 = $432 & 65535; //@line 54495
       HEAP8[$$hi24 >> 0] = $434; //@line 54496
       HEAP8[$$hi24 + 1 >> 0] = $434 >> 8; //@line 54496
       $435 = _bitshift64Lshr($432 | 0, 0, 16) | 0; //@line 54497
       HEAP8[$$hi24 + 2 >> 0] = $435; //@line 54501
       $439 = HEAP32[$23 >> 2] << 24; //@line 54503
       HEAP8[$$sink104$i + ($411 << 4) + 5 >> 0] = $$2509$i; //@line 54515
       HEAP8[$$sink104$i + ($411 << 4) + 4 >> 0] = $$3506$i; //@line 54518
       $452 = _bitshift64Shl($$2512$i & 31 | 0, 0, 40) | 0; //@line 54521
       $453 = tempRet0; //@line 54522
       HEAP8[$$sink104$i + ($411 << 4) + 7 >> 0] = $$5543$i | 0 ? $$4526$i & 255 : -1; //@line 54527
       HEAP8[$$sink104$i + ($411 << 4) + 6 >> 0] = $$6537$i | 0 ? $$5521$i & 255 : -1; //@line 54532
       $463 = _bitshift64Shl($$1528$i & 7 | 0, 0, 37) | 0; //@line 54534
       $477 = $467 | $468 | $463 | $452 | $433 & 16777215; //@line 54544
       $478 = $470 | $471 | tempRet0 | $$2515$i & 31 | $453 | $432 & 16285696; //@line 54545
       $479 = $477 | $439 & 251658240; //@line 54546
       $481 = $83 | ($479 | ((($430 >>> 16) + -21 | 0) >>> 0 < 4 ? ($$2512$i | 0) == 0 ? 2 : 0 : $$0544$lcssa$i) << 30); //@line 54548
       $482 = $75 | $478; //@line 54549
       HEAP8[$428 >> 0] = $481; //@line 54550
       HEAP8[$428 + 1 >> 0] = $481 >> 8; //@line 54550
       HEAP8[$428 + 2 >> 0] = $481 >> 16; //@line 54550
       HEAP8[$428 + 3 >> 0] = $481 >> 24; //@line 54550
       $$hi30 = $428 + 4 | 0; //@line 54551
       $483 = $482 & 65535; //@line 54552
       HEAP8[$$hi30 >> 0] = $483; //@line 54553
       HEAP8[$$hi30 + 1 >> 0] = $483 >> 8; //@line 54553
       $484 = _bitshift64Lshr($482 | 0, 0, 16) | 0; //@line 54554
       HEAP8[$$hi30 + 2 >> 0] = $484; //@line 54558
       $488 = _bitshift64Lshr($477 | 0, $478 & 16777215 | 0, 16) | 0; //@line 54560
       L283 : do {
        switch (($488 & 255) << 24 >> 24) {
        case 102:
        case 58:
        case 55:
        case 59:
         {
          do {
           switch ((_bitshift64Lshr($479 | 0, $478 & 16777215 | 0, 24) | 0) & 15) {
           case 1:
           case 0:
            {
             $$3570$i = 0; //@line 54574
             break L283;
             break;
            }
           case 2:
            {
             $$3570$i = 5; //@line 54579
             break L283;
             break;
            }
           case 3:
            {
             $$3570$i = 5; //@line 54584
             break L283;
             break;
            }
           case 4:
            {
             $$3570$i = 1; //@line 54589
             break L283;
             break;
            }
           case 5:
            {
             $$3570$i = 1; //@line 54594
             break L283;
             break;
            }
           case 6:
            {
             $$3570$i = 4; //@line 54599
             break L283;
             break;
            }
           case 7:
            {
             $$3570$i = 4; //@line 54604
             break L283;
             break;
            }
           case 8:
            {
             $$3570$i = 2; //@line 54609
             break L283;
             break;
            }
           case 9:
            {
             $$3570$i = 2; //@line 54614
             break L283;
             break;
            }
           case 10:
            {
             $$3570$i = 8; //@line 54619
             break L283;
             break;
            }
           case 11:
            {
             $$3570$i = 8; //@line 54624
             break L283;
             break;
            }
           case 12:
            {
             $$3570$i = 10; //@line 54629
             break L283;
             break;
            }
           case 13:
            {
             $$3570$i = 10; //@line 54634
             break L283;
             break;
            }
           case 14:
            {
             $$3570$i = 14; //@line 54639
             break L283;
             break;
            }
           case 15:
            {
             $$3570$i = 14; //@line 54644
             break L283;
             break;
            }
           default:
            {
             label = 215; //@line 54649
             break L7;
            }
           }
          } while (0);
          break;
         }
        default:
         {
          $$3570$i = $$2569780$i; //@line 54657
         }
        }
       } while (0);
       $498 = _bitshift64Shl($$3570$i & 255 | 0, 0, 8) | 0; //@line 54664
       $502 = $81 | $481 & -65536 | $498; //@line 54668
       $503 = $80 | $482 & 13107199 | tempRet0; //@line 54669
       HEAP8[$428 >> 0] = $502; //@line 54670
       HEAP8[$428 + 1 >> 0] = $502 >> 8; //@line 54670
       HEAP8[$428 + 2 >> 0] = $502 >> 16; //@line 54670
       HEAP8[$428 + 3 >> 0] = $502 >> 24; //@line 54670
       $$hi38 = $428 + 4 | 0; //@line 54671
       $504 = $503 & 65535; //@line 54672
       HEAP8[$$hi38 >> 0] = $504; //@line 54673
       HEAP8[$$hi38 + 1 >> 0] = $504 >> 8; //@line 54673
       $505 = _bitshift64Lshr($503 | 0, 0, 16) | 0; //@line 54674
       HEAP8[$$hi38 + 2 >> 0] = $505; //@line 54678
       $$5572$i = $$3570$i; //@line 54679
       $$7$i = $$5$i; //@line 54679
      }
     }
    } while (0);
    $$0502794$i = $$0502794$i + 1 | 0; //@line 54683
    if (($$0502794$i | 0) >= ($64 | 0)) {
     break;
    } else {
     $$0501795$i = $$7$i; //@line 54686
     $$2569780$i = $$5572$i; //@line 54686
    }
   }
   $521 = HEAP32[21986] | 0; //@line 54692
  }
  $$173 = $$173 + 1 | 0; //@line 54694
  if (($$173 | 0) >= ($521 | 0)) {
   label = 236; //@line 54699
   break;
  } else {
   $522 = $521; //@line 54697
  }
 }
 switch (label | 0) {
 case 44:
  {
   _abort(); //@line 54705
   break;
  }
 case 45:
  {
   _abort(); //@line 54710
   break;
  }
 case 49:
  {
   _abort(); //@line 54715
   break;
  }
 case 52:
  {
   HEAP8[$4 + 63 >> 0] = 0; //@line 54721
   HEAP32[$vararg_buffer >> 2] = $4; //@line 54722
   _fprintf($26, 106133, $vararg_buffer) | 0; //@line 54723
   _abort(); //@line 54724
   break;
  }
 case 61:
  {
   _abort(); //@line 54729
   break;
  }
 case 64:
  {
   _abort(); //@line 54734
   break;
  }
 case 79:
  {
   _abort(); //@line 54739
   break;
  }
 case 86:
  {
   _abort(); //@line 54744
   break;
  }
 case 107:
  {
   _abort(); //@line 54749
   break;
  }
 case 121:
  {
   _abort(); //@line 54754
   break;
  }
 case 130:
  {
   _abort(); //@line 54759
   break;
  }
 case 132:
  {
   _abort(); //@line 54764
   break;
  }
 case 135:
  {
   _abort(); //@line 54769
   break;
  }
 case 137:
  {
   _abort(); //@line 54774
   break;
  }
 case 149:
  {
   _abort(); //@line 54779
   break;
  }
 case 156:
  {
   _abort(); //@line 54784
   break;
  }
 case 176:
  {
   _abort(); //@line 54789
   break;
  }
 case 189:
  {
   _abort(); //@line 54794
   break;
  }
 case 197:
  {
   _abort(); //@line 54799
   break;
  }
 case 215:
  {
   break;
  }
 case 236:
  {
   STACKTOP = sp; //@line 54808
   return;
  }
 }
}
function __Z14InstallSlotROMv() {
 var $$0$i93 = 0, $$0158 = 0, $$023$lcssa$i = 0, $$02324$i = 0, $$025$i = 0, $$pre218 = 0, $$pre219 = 0, $$sink159 = 0, $0 = 0, $1 = 0, $100 = 0, $1118 = 0, $112 = 0, $1120 = 0, $114 = 0, $1140 = 0, $1142 = 0, $1144 = 0, $1145 = 0, $1147 = 0, $1151 = 0, $1162 = 0, $1172 = 0, $1173 = 0, $1174 = 0, $1175 = 0, $1176 = 0, $1177 = 0, $1178 = 0, $1179 = 0, $1215 = 0, $1217 = 0, $1253 = 0, $1255 = 0, $1291 = 0, $1293 = 0, $13 = 0, $1329 = 0, $1331 = 0, $134 = 0, $136 = 0, $1367 = 0, $1369 = 0, $1405 = 0, $1407 = 0, $1419 = 0, $1421 = 0, $1433 = 0, $1435 = 0, $1455 = 0, $1457 = 0, $1469 = 0, $1471 = 0, $1483 = 0, $1485 = 0, $150 = 0, $1511 = 0, $1512 = 0, $1530 = 0, $1531 = 0, $1549 = 0, $1550 = 0, $1568 = 0, $1569 = 0, $1587 = 0, $1588 = 0, $16 = 0, $1606 = 0, $1607 = 0, $1619 = 0, $1622 = 0, $1630 = 0, $1637 = 0, $1651 = 0, $1672 = 0, $1675 = 0, $1678 = 0, $1708 = 0, $1710 = 0, $1722 = 0, $1724 = 0, $1760 = 0, $1776 = 0, $1845 = 0, $1848 = 0, $1851 = 0, $186 = 0, $1897 = 0, $1909 = 0, $1913 = 0, $1927 = 0, $1943 = 0, $1946 = 0, $1947 = 0, $1949 = 0, $2 = 0, $202 = 0, $2021 = 0, $2025 = 0, $2143 = 0, $2145 = 0, $2165 = 0, $2167 = 0, $2179 = 0, $2181 = 0, $2225 = 0, $2227 = 0, $2239 = 0, $2241 = 0, $2244 = 0, $2249 = 0, $2250 = 0, $2262 = 0, $2267 = 0, $2268 = 0, $2280 = 0, $2282 = 0, $2302 = 0, $2304 = 0, $2318 = 0, $2366 = 0, $2368 = 0, $2370 = 0, $2372 = 0, $2374 = 0, $2382 = 0, $2391 = 0, $2393 = 0, $261 = 0, $264 = 0, $267 = 0, $3 = 0, $313 = 0, $37 = 0, $373 = 0, $377 = 0, $40 = 0, $43 = 0, $503 = 0, $505 = 0, $525 = 0, $578 = 0, $581 = 0, $584 = 0, $6 = 0, $66 = 0, $69 = 0, $7 = 0, $71 = 0, $72 = 0, $8 = 0, $86 = 0, $98 = 0, $vararg_buffer = 0, sp = 0;
 sp = STACKTOP; //@line 6416
 STACKTOP = STACKTOP + 288 | 0; //@line 6417
 $vararg_buffer = sp; //@line 6418
 $0 = sp + 8 | 0; //@line 6419
 $1 = sp + 24 | 0; //@line 6420
 $2 = HEAP32[30549] | 0; //@line 6421
 HEAP32[$0 >> 2] = 0; //@line 6422
 $3 = $0 + 4 | 0; //@line 6423
 HEAP32[$3 >> 2] = 0; //@line 6424
 HEAP32[$0 + 8 >> 2] = 0; //@line 6426
 HEAP8[651929] = 0; //@line 6427
 HEAP8[651930] = 1; //@line 6428
 HEAP8[651931] = 0; //@line 6429
 HEAP8[651932] = 0; //@line 6430
 HEAP8[651933] = 0; //@line 6431
 HEAP8[651934] = 0; //@line 6432
 HEAP8[651935] = 0; //@line 6433
 HEAP8[651936] = 0; //@line 6434
 HEAP8[651937] = 66; //@line 6435
 HEAP8[651938] = 97; //@line 6436
 HEAP8[651939] = 115; //@line 6437
 HEAP8[651940] = 105; //@line 6438
 HEAP8[651941] = 108; //@line 6439
 HEAP8[651942] = 105; //@line 6440
 HEAP8[651943] = 115; //@line 6441
 HEAP8[651944] = 107; //@line 6442
 HEAP8[651945] = 32; //@line 6443
 HEAP8[651946] = 73; //@line 6444
 HEAP8[651947] = 73; //@line 6445
 HEAP8[651948] = 32; //@line 6446
 HEAP8[651949] = 83; //@line 6447
 HEAP8[651950] = 108; //@line 6448
 HEAP8[651951] = 111; //@line 6449
 HEAP8[651952] = 116; //@line 6450
 HEAP8[651953] = 32; //@line 6451
 HEAP8[651954] = 82; //@line 6452
 HEAP8[651955] = 79; //@line 6453
 HEAP8[651956] = 77; //@line 6454
 HEAP8[651957] = 0; //@line 6455
 HEAP8[651958] = 0; //@line 6456
 HEAP8[651959] = 67; //@line 6457
 HEAP8[651960] = 104; //@line 6458
 HEAP8[651961] = 114; //@line 6459
 HEAP8[651962] = 105; //@line 6460
 HEAP8[651963] = 115; //@line 6461
 HEAP8[651964] = 116; //@line 6462
 HEAP8[651965] = 105; //@line 6463
 HEAP8[651966] = 97; //@line 6464
 HEAP8[651967] = 110; //@line 6465
 HEAP8[651968] = 32; //@line 6466
 HEAP8[651969] = 66; //@line 6467
 HEAP8[651970] = 97; //@line 6468
 HEAP8[651971] = 117; //@line 6469
 HEAP8[651972] = 101; //@line 6470
 HEAP8[651973] = 114; //@line 6471
 HEAP8[651974] = 0; //@line 6472
 HEAP32[28210] = 46; //@line 6473
 HEAP32[$vararg_buffer >> 2] = 1; //@line 6474
 HEAP32[$vararg_buffer + 4 >> 2] = 0; //@line 6476
 _sprintf($1, 92795, $vararg_buffer) | 0; //@line 6477
 $$0$i93 = $1; //@line 6479
 $8 = HEAP32[28210] | 0; //@line 6479
 while (1) {
  $6 = HEAP8[$$0$i93 >> 0] | 0; //@line 6482
  $7 = $8 + 1 | 0; //@line 6483
  HEAP8[651929 + $8 >> 0] = $6; //@line 6485
  if (!($6 << 24 >> 24)) {
   break;
  } else {
   $$0$i93 = $$0$i93 + 1 | 0; //@line 6490
   $8 = $7; //@line 6490
  }
 }
 HEAP32[28210] = $7; //@line 6493
 if (!($7 & 1)) {
  $16 = $7; //@line 6497
 } else {
  $13 = $8 + 2 | 0; //@line 6499
  HEAP32[28210] = $13; //@line 6501
  HEAP8[651929 + $7 >> 0] = 0; //@line 6502
  $16 = $13; //@line 6503
 }
 HEAP8[651929 + $16 >> 0] = 66; //@line 6507
 HEAP8[651929 + ($16 + 1) >> 0] = 97; //@line 6510
 HEAP8[651929 + ($16 + 2) >> 0] = 115; //@line 6513
 HEAP8[651929 + ($16 + 3) >> 0] = 105; //@line 6516
 HEAP8[651929 + ($16 + 4) >> 0] = 108; //@line 6519
 HEAP8[651929 + ($16 + 5) >> 0] = 105; //@line 6522
 HEAP8[651929 + ($16 + 6) >> 0] = 115; //@line 6525
 HEAP8[651929 + ($16 + 7) >> 0] = 107; //@line 6528
 HEAP8[651929 + ($16 + 8) >> 0] = 73; //@line 6531
 HEAP8[651929 + ($16 + 9) >> 0] = 73; //@line 6534
 HEAP8[651929 + ($16 + 10) >> 0] = 0; //@line 6536
 $37 = $16 + 11 | 0; //@line 6537
 HEAP32[28210] = $37; //@line 6538
 if (!($37 & 1)) {
  $43 = $37; //@line 6542
 } else {
  $40 = $16 + 12 | 0; //@line 6544
  HEAP32[28210] = $40; //@line 6546
  HEAP8[651929 + $37 >> 0] = 0; //@line 6547
  $43 = $40; //@line 6548
 }
 HEAP8[651929 + $43 >> 0] = 79; //@line 6552
 HEAP8[651929 + ($43 + 1) >> 0] = 99; //@line 6555
 HEAP8[651929 + ($43 + 2) >> 0] = 116; //@line 6558
 HEAP8[651929 + ($43 + 3) >> 0] = 32; //@line 6561
 HEAP8[651929 + ($43 + 4) >> 0] = 32; //@line 6564
 HEAP8[651929 + ($43 + 5) >> 0] = 56; //@line 6567
 HEAP8[651929 + ($43 + 6) >> 0] = 32; //@line 6570
 HEAP8[651929 + ($43 + 7) >> 0] = 50; //@line 6573
 HEAP8[651929 + ($43 + 8) >> 0] = 48; //@line 6576
 HEAP8[651929 + ($43 + 9) >> 0] = 49; //@line 6579
 HEAP8[651929 + ($43 + 10) >> 0] = 55; //@line 6582
 HEAP8[651929 + ($43 + 11) >> 0] = 0; //@line 6584
 $66 = $43 + 12 | 0; //@line 6585
 HEAP32[28210] = $66; //@line 6586
 if (!($66 & 1)) {
  $72 = $66; //@line 6590
 } else {
  $69 = $43 + 13 | 0; //@line 6592
  HEAP32[28210] = $69; //@line 6594
  HEAP8[651929 + $66 >> 0] = 0; //@line 6595
  $72 = $69; //@line 6596
 }
 $71 = 30 - $72 | 0; //@line 6598
 HEAP8[651929 + $72 >> 0] = 1; //@line 6601
 HEAP8[651929 + ($72 + 1) >> 0] = $71 >>> 16; //@line 6606
 HEAP8[651929 + ($72 + 2) >> 0] = $71 >>> 8; //@line 6611
 HEAP8[651929 + ($72 + 3) >> 0] = $71; //@line 6615
 $86 = 42 - $72 | 0; //@line 6616
 HEAP8[651929 + ($72 + 4) >> 0] = 3; //@line 6619
 HEAP8[651929 + ($72 + 5) >> 0] = $86 >>> 16; //@line 6624
 HEAP8[651929 + ($72 + 6) >> 0] = $86 >>> 8; //@line 6629
 $98 = $72 + 8 | 0; //@line 6631
 HEAP8[651929 + ($72 + 7) >> 0] = $86; //@line 6633
 $100 = $16 - $98 | 0; //@line 6634
 HEAP8[651929 + $98 >> 0] = 4; //@line 6637
 HEAP8[651929 + ($72 + 9) >> 0] = $100 >>> 16; //@line 6642
 HEAP8[651929 + ($72 + 10) >> 0] = $100 >>> 8; //@line 6647
 $112 = $72 + 12 | 0; //@line 6649
 HEAP8[651929 + ($72 + 11) >> 0] = $100; //@line 6651
 $114 = $43 - $112 | 0; //@line 6652
 HEAP8[651929 + $112 >> 0] = 5; //@line 6655
 HEAP8[651929 + ($72 + 13) >> 0] = $114 >>> 16; //@line 6660
 HEAP8[651929 + ($72 + 14) >> 0] = $114 >>> 8; //@line 6665
 HEAP8[651929 + ($72 + 15) >> 0] = $114; //@line 6669
 HEAP8[651929 + ($72 + 16) >> 0] = -1; //@line 6672
 HEAP8[651929 + ($72 + 17) >> 0] = 0; //@line 6675
 HEAP8[651929 + ($72 + 18) >> 0] = 0; //@line 6678
 $134 = $72 + 20 | 0; //@line 6679
 HEAP8[651929 + ($72 + 19) >> 0] = 0; //@line 6681
 $136 = -20 - $72 | 0; //@line 6682
 HEAP8[651929 + $134 >> 0] = 1; //@line 6685
 HEAP8[651929 + ($72 + 21) >> 0] = $136 >>> 16; //@line 6690
 HEAP8[651929 + ($72 + 22) >> 0] = $136 >>> 8; //@line 6695
 HEAP8[651929 + ($72 + 23) >> 0] = $136; //@line 6699
 $150 = -16 - $72 | 0; //@line 6700
 HEAP8[651929 + ($72 + 24) >> 0] = 2; //@line 6703
 HEAP8[651929 + ($72 + 25) >> 0] = $150 >>> 16; //@line 6708
 HEAP8[651929 + ($72 + 26) >> 0] = $150 >>> 8; //@line 6713
 HEAP8[651929 + ($72 + 27) >> 0] = $150; //@line 6717
 HEAP8[651929 + ($72 + 28) >> 0] = 32; //@line 6720
 HEAP8[651929 + ($72 + 29) >> 0] = 0; //@line 6723
 HEAP8[651929 + ($72 + 30) >> 0] = 66; //@line 6726
 HEAP8[651929 + ($72 + 31) >> 0] = 50; //@line 6729
 HEAP8[651929 + ($72 + 32) >> 0] = 36; //@line 6732
 HEAP8[651929 + ($72 + 33) >> 0] = -1; //@line 6735
 HEAP8[651929 + ($72 + 34) >> 0] = -1; //@line 6738
 HEAP8[651929 + ($72 + 35) >> 0] = -32; //@line 6741
 HEAP8[651929 + ($72 + 36) >> 0] = -1; //@line 6744
 HEAP8[651929 + ($72 + 37) >> 0] = 0; //@line 6747
 HEAP8[651929 + ($72 + 38) >> 0] = 0; //@line 6750
 $186 = $72 + 40 | 0; //@line 6751
 HEAP8[651929 + ($72 + 39) >> 0] = 0; //@line 6753
 HEAP8[651929 + $186 >> 0] = 0; //@line 6756
 HEAP8[651929 + ($72 + 41) >> 0] = 3; //@line 6759
 HEAP8[651929 + ($72 + 42) >> 0] = 0; //@line 6762
 HEAP8[651929 + ($72 + 43) >> 0] = 1; //@line 6765
 HEAP8[651929 + ($72 + 44) >> 0] = 0; //@line 6768
 HEAP8[651929 + ($72 + 45) >> 0] = 1; //@line 6771
 HEAP8[651929 + ($72 + 46) >> 0] = 66; //@line 6774
 $202 = $72 + 48 | 0; //@line 6775
 HEAP8[651929 + ($72 + 47) >> 0] = 50; //@line 6777
 HEAP8[651929 + $202 >> 0] = 68; //@line 6780
 HEAP8[651929 + ($72 + 49) >> 0] = 105; //@line 6783
 HEAP8[651929 + ($72 + 50) >> 0] = 115; //@line 6786
 HEAP8[651929 + ($72 + 51) >> 0] = 112; //@line 6789
 HEAP8[651929 + ($72 + 52) >> 0] = 108; //@line 6792
 HEAP8[651929 + ($72 + 53) >> 0] = 97; //@line 6795
 HEAP8[651929 + ($72 + 54) >> 0] = 121; //@line 6798
 HEAP8[651929 + ($72 + 55) >> 0] = 95; //@line 6801
 HEAP8[651929 + ($72 + 56) >> 0] = 86; //@line 6804
 HEAP8[651929 + ($72 + 57) >> 0] = 105; //@line 6807
 HEAP8[651929 + ($72 + 58) >> 0] = 100; //@line 6810
 HEAP8[651929 + ($72 + 59) >> 0] = 101; //@line 6813
 HEAP8[651929 + ($72 + 60) >> 0] = 111; //@line 6816
 HEAP8[651929 + ($72 + 61) >> 0] = 95; //@line 6819
 HEAP8[651929 + ($72 + 62) >> 0] = 65; //@line 6822
 HEAP8[651929 + ($72 + 63) >> 0] = 112; //@line 6825
 HEAP8[651929 + ($72 + 64) >> 0] = 112; //@line 6828
 HEAP8[651929 + ($72 + 65) >> 0] = 108; //@line 6831
 HEAP8[651929 + ($72 + 66) >> 0] = 101; //@line 6834
 HEAP8[651929 + ($72 + 67) >> 0] = 95; //@line 6837
 HEAP8[651929 + ($72 + 68) >> 0] = 66; //@line 6840
 HEAP8[651929 + ($72 + 69) >> 0] = 97; //@line 6843
 HEAP8[651929 + ($72 + 70) >> 0] = 115; //@line 6846
 HEAP8[651929 + ($72 + 71) >> 0] = 105; //@line 6849
 HEAP8[651929 + ($72 + 72) >> 0] = 108; //@line 6852
 HEAP8[651929 + ($72 + 73) >> 0] = 105; //@line 6855
 HEAP8[651929 + ($72 + 74) >> 0] = 115; //@line 6858
 HEAP8[651929 + ($72 + 75) >> 0] = 107; //@line 6861
 HEAP8[651929 + ($72 + 76) >> 0] = 0; //@line 6863
 $261 = $72 + 77 | 0; //@line 6864
 HEAP32[28210] = $261; //@line 6865
 if (!($261 & 1)) {
  $267 = $261; //@line 6869
 } else {
  $264 = $72 + 78 | 0; //@line 6871
  HEAP32[28210] = $264; //@line 6873
  HEAP8[651929 + $261 >> 0] = 0; //@line 6874
  $267 = $264; //@line 6875
 }
 HEAP8[651929 + $267 >> 0] = 0; //@line 6879
 HEAP8[651929 + ($267 + 1) >> 0] = 0; //@line 6882
 HEAP8[651929 + ($267 + 2) >> 0] = 0; //@line 6885
 HEAP8[651929 + ($267 + 3) >> 0] = 114; //@line 6888
 HEAP8[651929 + ($267 + 4) >> 0] = 76; //@line 6891
 HEAP8[651929 + ($267 + 5) >> 0] = 0; //@line 6894
 HEAP8[651929 + ($267 + 6) >> 0] = 0; //@line 6897
 HEAP8[651929 + ($267 + 7) >> 0] = 0; //@line 6900
 HEAP8[651929 + ($267 + 8) >> 0] = 0; //@line 6903
 HEAP8[651929 + ($267 + 9) >> 0] = 0; //@line 6906
 HEAP8[651929 + ($267 + 10) >> 0] = 0; //@line 6909
 HEAP8[651929 + ($267 + 11) >> 0] = 0; //@line 6912
 HEAP8[651929 + ($267 + 12) >> 0] = 0; //@line 6915
 HEAP8[651929 + ($267 + 13) >> 0] = 50; //@line 6918
 HEAP8[651929 + ($267 + 14) >> 0] = 0; //@line 6921
 HEAP8[651929 + ($267 + 15) >> 0] = 54; //@line 6924
 HEAP8[651929 + ($267 + 16) >> 0] = 0; //@line 6927
 HEAP8[651929 + ($267 + 17) >> 0] = 58; //@line 6930
 HEAP8[651929 + ($267 + 18) >> 0] = 0; //@line 6933
 HEAP8[651929 + ($267 + 19) >> 0] = 70; //@line 6936
 HEAP8[651929 + ($267 + 20) >> 0] = 0; //@line 6939
 HEAP8[651929 + ($267 + 21) >> 0] = 108; //@line 6942
 HEAP8[651929 + ($267 + 22) >> 0] = 29; //@line 6945
 $313 = $267 + 53 | 0; //@line 6946
 HEAP8[651929 + ($267 + 23) >> 0] = 46; //@line 6949
 HEAP8[651929 + ($267 + 24) >> 0] = 68; //@line 6952
 HEAP8[651929 + ($267 + 25) >> 0] = 105; //@line 6955
 HEAP8[651929 + ($267 + 26) >> 0] = 115; //@line 6958
 HEAP8[651929 + ($267 + 27) >> 0] = 112; //@line 6961
 HEAP8[651929 + ($267 + 28) >> 0] = 108; //@line 6964
 HEAP8[651929 + ($267 + 29) >> 0] = 97; //@line 6967
 HEAP8[651929 + ($267 + 30) >> 0] = 121; //@line 6970
 HEAP8[651929 + ($267 + 31) >> 0] = 95; //@line 6973
 HEAP8[651929 + ($267 + 32) >> 0] = 86; //@line 6976
 HEAP8[651929 + ($267 + 33) >> 0] = 105; //@line 6979
 HEAP8[651929 + ($267 + 34) >> 0] = 100; //@line 6982
 HEAP8[651929 + ($267 + 35) >> 0] = 101; //@line 6985
 HEAP8[651929 + ($267 + 36) >> 0] = 111; //@line 6988
 HEAP8[651929 + ($267 + 37) >> 0] = 95; //@line 6991
 HEAP8[651929 + ($267 + 38) >> 0] = 65; //@line 6994
 HEAP8[651929 + ($267 + 39) >> 0] = 112; //@line 6997
 HEAP8[651929 + ($267 + 40) >> 0] = 112; //@line 7000
 HEAP8[651929 + ($267 + 41) >> 0] = 108; //@line 7003
 HEAP8[651929 + ($267 + 42) >> 0] = 101; //@line 7006
 HEAP8[651929 + ($267 + 43) >> 0] = 95; //@line 7009
 HEAP8[651929 + ($267 + 44) >> 0] = 66; //@line 7012
 HEAP8[651929 + ($267 + 45) >> 0] = 97; //@line 7015
 HEAP8[651929 + ($267 + 46) >> 0] = 115; //@line 7018
 HEAP8[651929 + ($267 + 47) >> 0] = 105; //@line 7021
 HEAP8[651929 + ($267 + 48) >> 0] = 108; //@line 7024
 HEAP8[651929 + ($267 + 49) >> 0] = 105; //@line 7027
 HEAP8[651929 + ($267 + 50) >> 0] = 115; //@line 7030
 HEAP8[651929 + ($267 + 51) >> 0] = 107; //@line 7033
 HEAP8[651929 + ($267 + 52) >> 0] = 0; //@line 7035
 $373 = $267 + 52 | 0; //@line 7036
 HEAP32[28210] = $373; //@line 7037
 if (!($373 & 1)) {
  $377 = $373; //@line 7041
 } else {
  HEAP32[28210] = $313; //@line 7044
  HEAP8[651929 + $373 >> 0] = 0; //@line 7045
  $377 = $313; //@line 7046
 }
 HEAP8[651929 + $377 >> 0] = 0; //@line 7050
 HEAP8[651929 + ($377 + 1) >> 0] = 1; //@line 7053
 HEAP8[651929 + ($377 + 2) >> 0] = 113; //@line 7056
 HEAP8[651929 + ($377 + 3) >> 0] = 24; //@line 7059
 HEAP8[651929 + ($377 + 4) >> 0] = 78; //@line 7062
 HEAP8[651929 + ($377 + 5) >> 0] = 117; //@line 7065
 HEAP8[651929 + ($377 + 6) >> 0] = 112; //@line 7068
 HEAP8[651929 + ($377 + 7) >> 0] = -1; //@line 7071
 HEAP8[651929 + ($377 + 8) >> 0] = 96; //@line 7074
 HEAP8[651929 + ($377 + 9) >> 0] = 14; //@line 7077
 HEAP8[651929 + ($377 + 10) >> 0] = 113; //@line 7080
 HEAP8[651929 + ($377 + 11) >> 0] = 25; //@line 7083
 HEAP8[651929 + ($377 + 12) >> 0] = 12; //@line 7086
 HEAP8[651929 + ($377 + 13) >> 0] = 104; //@line 7089
 HEAP8[651929 + ($377 + 14) >> 0] = 0; //@line 7092
 HEAP8[651929 + ($377 + 15) >> 0] = 1; //@line 7095
 HEAP8[651929 + ($377 + 16) >> 0] = 0; //@line 7098
 HEAP8[651929 + ($377 + 17) >> 0] = 26; //@line 7101
 HEAP8[651929 + ($377 + 18) >> 0] = 102; //@line 7104
 HEAP8[651929 + ($377 + 19) >> 0] = 4; //@line 7107
 HEAP8[651929 + ($377 + 20) >> 0] = 78; //@line 7110
 HEAP8[651929 + ($377 + 21) >> 0] = 117; //@line 7113
 HEAP8[651929 + ($377 + 22) >> 0] = 113; //@line 7116
 HEAP8[651929 + ($377 + 23) >> 0] = 26; //@line 7119
 HEAP8[651929 + ($377 + 24) >> 0] = 50; //@line 7122
 HEAP8[651929 + ($377 + 25) >> 0] = 40; //@line 7125
 HEAP8[651929 + ($377 + 26) >> 0] = 0; //@line 7128
 HEAP8[651929 + ($377 + 27) >> 0] = 6; //@line 7131
 HEAP8[651929 + ($377 + 28) >> 0] = 8; //@line 7134
 HEAP8[651929 + ($377 + 29) >> 0] = 1; //@line 7137
 HEAP8[651929 + ($377 + 30) >> 0] = 0; //@line 7140
 HEAP8[651929 + ($377 + 31) >> 0] = 9; //@line 7143
 HEAP8[651929 + ($377 + 32) >> 0] = 103; //@line 7146
 HEAP8[651929 + ($377 + 33) >> 0] = 12; //@line 7149
 HEAP8[651929 + ($377 + 34) >> 0] = 74; //@line 7152
 HEAP8[651929 + ($377 + 35) >> 0] = 64; //@line 7155
 HEAP8[651929 + ($377 + 36) >> 0] = 111; //@line 7158
 HEAP8[651929 + ($377 + 37) >> 0] = 2; //@line 7161
 HEAP8[651929 + ($377 + 38) >> 0] = 66; //@line 7164
 HEAP8[651929 + ($377 + 39) >> 0] = 64; //@line 7167
 HEAP8[651929 + ($377 + 40) >> 0] = 49; //@line 7170
 HEAP8[651929 + ($377 + 41) >> 0] = 64; //@line 7173
 HEAP8[651929 + ($377 + 42) >> 0] = 0; //@line 7176
 HEAP8[651929 + ($377 + 43) >> 0] = 16; //@line 7179
 HEAP8[651929 + ($377 + 44) >> 0] = 78; //@line 7182
 HEAP8[651929 + ($377 + 45) >> 0] = 117; //@line 7185
 HEAP8[651929 + ($377 + 46) >> 0] = 74; //@line 7188
 HEAP8[651929 + ($377 + 47) >> 0] = 64; //@line 7191
 HEAP8[651929 + ($377 + 48) >> 0] = 111; //@line 7194
 HEAP8[651929 + ($377 + 49) >> 0] = 4; //@line 7197
 HEAP8[651929 + ($377 + 50) >> 0] = 66; //@line 7200
 HEAP8[651929 + ($377 + 51) >> 0] = 64; //@line 7203
 HEAP8[651929 + ($377 + 52) >> 0] = 78; //@line 7206
 HEAP8[651929 + ($377 + 53) >> 0] = 117; //@line 7209
 HEAP8[651929 + ($377 + 54) >> 0] = 47; //@line 7212
 HEAP8[651929 + ($377 + 55) >> 0] = 56; //@line 7215
 HEAP8[651929 + ($377 + 56) >> 0] = 8; //@line 7218
 HEAP8[651929 + ($377 + 57) >> 0] = -4; //@line 7221
 HEAP8[651929 + ($377 + 58) >> 0] = 78; //@line 7224
 HEAP8[651929 + ($377 + 59) >> 0] = 117; //@line 7227
 HEAP8[651929 + ($377 + 60) >> 0] = 112; //@line 7230
 HEAP8[651929 + ($377 + 61) >> 0] = -24; //@line 7233
 HEAP8[651929 + ($377 + 62) >> 0] = 78; //@line 7236
 $503 = $377 + 64 | 0; //@line 7237
 HEAP8[651929 + ($377 + 63) >> 0] = 117; //@line 7239
 $505 = $267 - $503 | 0; //@line 7240
 HEAP8[651929 + $503 >> 0] = 2; //@line 7243
 HEAP8[651929 + ($377 + 65) >> 0] = $505 >>> 16; //@line 7248
 HEAP8[651929 + ($377 + 66) >> 0] = $505 >>> 8; //@line 7253
 HEAP8[651929 + ($377 + 67) >> 0] = $505; //@line 7257
 HEAP8[651929 + ($377 + 68) >> 0] = -1; //@line 7260
 HEAP8[651929 + ($377 + 69) >> 0] = 0; //@line 7263
 HEAP8[651929 + ($377 + 70) >> 0] = 0; //@line 7266
 $525 = $377 + 72 | 0; //@line 7267
 HEAP8[651929 + ($377 + 71) >> 0] = 0; //@line 7269
 HEAP8[651929 + $525 >> 0] = 0; //@line 7272
 HEAP8[651929 + ($377 + 73) >> 0] = 0; //@line 7275
 HEAP8[651929 + ($377 + 74) >> 0] = 1; //@line 7278
 HEAP8[651929 + ($377 + 75) >> 0] = 38; //@line 7281
 HEAP8[651929 + ($377 + 76) >> 0] = 32; //@line 7284
 HEAP8[651929 + ($377 + 77) >> 0] = 0; //@line 7287
 HEAP8[651929 + ($377 + 78) >> 0] = 77; //@line 7290
 HEAP8[651929 + ($377 + 79) >> 0] = 97; //@line 7293
 HEAP8[651929 + ($377 + 80) >> 0] = 99; //@line 7296
 HEAP8[651929 + ($377 + 81) >> 0] = 32; //@line 7299
 HEAP8[651929 + ($377 + 82) >> 0] = 72; //@line 7302
 HEAP8[651929 + ($377 + 83) >> 0] = 105; //@line 7305
 HEAP8[651929 + ($377 + 84) >> 0] = 82; //@line 7308
 HEAP8[651929 + ($377 + 85) >> 0] = 101; //@line 7311
 HEAP8[651929 + ($377 + 86) >> 0] = 115; //@line 7314
 HEAP8[651929 + ($377 + 87) >> 0] = 32; //@line 7317
 HEAP8[651929 + ($377 + 88) >> 0] = 83; //@line 7320
 HEAP8[651929 + ($377 + 89) >> 0] = 116; //@line 7323
 HEAP8[651929 + ($377 + 90) >> 0] = 100; //@line 7326
 HEAP8[651929 + ($377 + 91) >> 0] = 32; //@line 7329
 HEAP8[651929 + ($377 + 92) >> 0] = 71; //@line 7332
 HEAP8[651929 + ($377 + 93) >> 0] = 97; //@line 7335
 HEAP8[651929 + ($377 + 94) >> 0] = 109; //@line 7338
 HEAP8[651929 + ($377 + 95) >> 0] = 109; //@line 7341
 HEAP8[651929 + ($377 + 96) >> 0] = 97; //@line 7344
 HEAP8[651929 + ($377 + 97) >> 0] = 0; //@line 7346
 $578 = $377 + 98 | 0; //@line 7347
 HEAP32[28210] = $578; //@line 7348
 if (!($578 & 1)) {
  $584 = $578; //@line 7352
 } else {
  $581 = $377 + 99 | 0; //@line 7354
  HEAP32[28210] = $581; //@line 7356
  HEAP8[651929 + $578 >> 0] = 0; //@line 7357
  $584 = $581; //@line 7358
 }
 HEAP8[651929 + $584 >> 0] = 0; //@line 7362
 HEAP8[651929 + ($584 + 1) >> 0] = 0; //@line 7365
 HEAP8[651929 + ($584 + 2) >> 0] = 0; //@line 7368
 HEAP8[651929 + ($584 + 3) >> 0] = 0; //@line 7371
 HEAP8[651929 + ($584 + 4) >> 0] = 0; //@line 7374
 HEAP8[651929 + ($584 + 5) >> 0] = 0; //@line 7377
 HEAP8[651929 + ($584 + 6) >> 0] = 0; //@line 7380
 HEAP8[651929 + ($584 + 7) >> 0] = 1; //@line 7383
 HEAP8[651929 + ($584 + 8) >> 0] = 1; //@line 7386
 HEAP8[651929 + ($584 + 9) >> 0] = 0; //@line 7389
 HEAP8[651929 + ($584 + 10) >> 0] = 0; //@line 7392
 HEAP8[651929 + ($584 + 11) >> 0] = 8; //@line 7395
 HEAP8[651929 + ($584 + 12) >> 0] = 0; //@line 7398
 HEAP8[651929 + ($584 + 13) >> 0] = 5; //@line 7401
 HEAP8[651929 + ($584 + 14) >> 0] = 9; //@line 7404
 HEAP8[651929 + ($584 + 15) >> 0] = 11; //@line 7407
 HEAP8[651929 + ($584 + 16) >> 0] = 14; //@line 7410
 HEAP8[651929 + ($584 + 17) >> 0] = 16; //@line 7413
 HEAP8[651929 + ($584 + 18) >> 0] = 19; //@line 7416
 HEAP8[651929 + ($584 + 19) >> 0] = 21; //@line 7419
 HEAP8[651929 + ($584 + 20) >> 0] = 23; //@line 7422
 HEAP8[651929 + ($584 + 21) >> 0] = 25; //@line 7425
 HEAP8[651929 + ($584 + 22) >> 0] = 27; //@line 7428
 HEAP8[651929 + ($584 + 23) >> 0] = 29; //@line 7431
 HEAP8[651929 + ($584 + 24) >> 0] = 30; //@line 7434
 HEAP8[651929 + ($584 + 25) >> 0] = 32; //@line 7437
 HEAP8[651929 + ($584 + 26) >> 0] = 34; //@line 7440
 HEAP8[651929 + ($584 + 27) >> 0] = 36; //@line 7443
 HEAP8[651929 + ($584 + 28) >> 0] = 37; //@line 7446
 HEAP8[651929 + ($584 + 29) >> 0] = 39; //@line 7449
 HEAP8[651929 + ($584 + 30) >> 0] = 40; //@line 7452
 HEAP8[651929 + ($584 + 31) >> 0] = 42; //@line 7455
 HEAP8[651929 + ($584 + 32) >> 0] = 44; //@line 7458
 HEAP8[651929 + ($584 + 33) >> 0] = 45; //@line 7461
 HEAP8[651929 + ($584 + 34) >> 0] = 47; //@line 7464
 HEAP8[651929 + ($584 + 35) >> 0] = 48; //@line 7467
 HEAP8[651929 + ($584 + 36) >> 0] = 49; //@line 7470
 HEAP8[651929 + ($584 + 37) >> 0] = 51; //@line 7473
 HEAP8[651929 + ($584 + 38) >> 0] = 52; //@line 7476
 HEAP8[651929 + ($584 + 39) >> 0] = 54; //@line 7479
 HEAP8[651929 + ($584 + 40) >> 0] = 55; //@line 7482
 HEAP8[651929 + ($584 + 41) >> 0] = 56; //@line 7485
 HEAP8[651929 + ($584 + 42) >> 0] = 58; //@line 7488
 HEAP8[651929 + ($584 + 43) >> 0] = 59; //@line 7491
 HEAP8[651929 + ($584 + 44) >> 0] = 60; //@line 7494
 HEAP8[651929 + ($584 + 45) >> 0] = 62; //@line 7497
 HEAP8[651929 + ($584 + 46) >> 0] = 63; //@line 7500
 HEAP8[651929 + ($584 + 47) >> 0] = 64; //@line 7503
 HEAP8[651929 + ($584 + 48) >> 0] = 66; //@line 7506
 HEAP8[651929 + ($584 + 49) >> 0] = 67; //@line 7509
 HEAP8[651929 + ($584 + 50) >> 0] = 68; //@line 7512
 HEAP8[651929 + ($584 + 51) >> 0] = 69; //@line 7515
 HEAP8[651929 + ($584 + 52) >> 0] = 71; //@line 7518
 HEAP8[651929 + ($584 + 53) >> 0] = 72; //@line 7521
 HEAP8[651929 + ($584 + 54) >> 0] = 73; //@line 7524
 HEAP8[651929 + ($584 + 55) >> 0] = 74; //@line 7527
 HEAP8[651929 + ($584 + 56) >> 0] = 75; //@line 7530
 HEAP8[651929 + ($584 + 57) >> 0] = 77; //@line 7533
 HEAP8[651929 + ($584 + 58) >> 0] = 78; //@line 7536
 HEAP8[651929 + ($584 + 59) >> 0] = 79; //@line 7539
 HEAP8[651929 + ($584 + 60) >> 0] = 80; //@line 7542
 HEAP8[651929 + ($584 + 61) >> 0] = 81; //@line 7545
 HEAP8[651929 + ($584 + 62) >> 0] = 82; //@line 7548
 HEAP8[651929 + ($584 + 63) >> 0] = 84; //@line 7551
 HEAP8[651929 + ($584 + 64) >> 0] = 85; //@line 7554
 HEAP8[651929 + ($584 + 65) >> 0] = 86; //@line 7557
 HEAP8[651929 + ($584 + 66) >> 0] = 87; //@line 7560
 HEAP8[651929 + ($584 + 67) >> 0] = 88; //@line 7563
 HEAP8[651929 + ($584 + 68) >> 0] = 89; //@line 7566
 HEAP8[651929 + ($584 + 69) >> 0] = 90; //@line 7569
 HEAP8[651929 + ($584 + 70) >> 0] = 91; //@line 7572
 HEAP8[651929 + ($584 + 71) >> 0] = 92; //@line 7575
 HEAP8[651929 + ($584 + 72) >> 0] = 94; //@line 7578
 HEAP8[651929 + ($584 + 73) >> 0] = 95; //@line 7581
 HEAP8[651929 + ($584 + 74) >> 0] = 96; //@line 7584
 HEAP8[651929 + ($584 + 75) >> 0] = 97; //@line 7587
 HEAP8[651929 + ($584 + 76) >> 0] = 98; //@line 7590
 HEAP8[651929 + ($584 + 77) >> 0] = 99; //@line 7593
 HEAP8[651929 + ($584 + 78) >> 0] = 100; //@line 7596
 HEAP8[651929 + ($584 + 79) >> 0] = 101; //@line 7599
 HEAP8[651929 + ($584 + 80) >> 0] = 102; //@line 7602
 HEAP8[651929 + ($584 + 81) >> 0] = 103; //@line 7605
 HEAP8[651929 + ($584 + 82) >> 0] = 104; //@line 7608
 HEAP8[651929 + ($584 + 83) >> 0] = 105; //@line 7611
 HEAP8[651929 + ($584 + 84) >> 0] = 106; //@line 7614
 HEAP8[651929 + ($584 + 85) >> 0] = 107; //@line 7617
 HEAP8[651929 + ($584 + 86) >> 0] = 108; //@line 7620
 HEAP8[651929 + ($584 + 87) >> 0] = 109; //@line 7623
 HEAP8[651929 + ($584 + 88) >> 0] = 110; //@line 7626
 HEAP8[651929 + ($584 + 89) >> 0] = 111; //@line 7629
 HEAP8[651929 + ($584 + 90) >> 0] = 112; //@line 7632
 HEAP8[651929 + ($584 + 91) >> 0] = 113; //@line 7635
 HEAP8[651929 + ($584 + 92) >> 0] = 114; //@line 7638
 HEAP8[651929 + ($584 + 93) >> 0] = 115; //@line 7641
 HEAP8[651929 + ($584 + 94) >> 0] = 116; //@line 7644
 HEAP8[651929 + ($584 + 95) >> 0] = 117; //@line 7647
 HEAP8[651929 + ($584 + 96) >> 0] = 118; //@line 7650
 HEAP8[651929 + ($584 + 97) >> 0] = 119; //@line 7653
 HEAP8[651929 + ($584 + 98) >> 0] = 120; //@line 7656
 HEAP8[651929 + ($584 + 99) >> 0] = 121; //@line 7659
 HEAP8[651929 + ($584 + 100) >> 0] = 122; //@line 7662
 HEAP8[651929 + ($584 + 101) >> 0] = 123; //@line 7665
 HEAP8[651929 + ($584 + 102) >> 0] = 124; //@line 7668
 HEAP8[651929 + ($584 + 103) >> 0] = 125; //@line 7671
 HEAP8[651929 + ($584 + 104) >> 0] = 126; //@line 7674
 HEAP8[651929 + ($584 + 105) >> 0] = 127; //@line 7677
 HEAP8[651929 + ($584 + 106) >> 0] = -128; //@line 7680
 HEAP8[651929 + ($584 + 107) >> 0] = -127; //@line 7683
 HEAP8[651929 + ($584 + 108) >> 0] = -127; //@line 7686
 HEAP8[651929 + ($584 + 109) >> 0] = -126; //@line 7689
 HEAP8[651929 + ($584 + 110) >> 0] = -125; //@line 7692
 HEAP8[651929 + ($584 + 111) >> 0] = -124; //@line 7695
 HEAP8[651929 + ($584 + 112) >> 0] = -123; //@line 7698
 HEAP8[651929 + ($584 + 113) >> 0] = -122; //@line 7701
 HEAP8[651929 + ($584 + 114) >> 0] = -121; //@line 7704
 HEAP8[651929 + ($584 + 115) >> 0] = -120; //@line 7707
 HEAP8[651929 + ($584 + 116) >> 0] = -119; //@line 7710
 HEAP8[651929 + ($584 + 117) >> 0] = -118; //@line 7713
 HEAP8[651929 + ($584 + 118) >> 0] = -117; //@line 7716
 HEAP8[651929 + ($584 + 119) >> 0] = -116; //@line 7719
 HEAP8[651929 + ($584 + 120) >> 0] = -116; //@line 7722
 HEAP8[651929 + ($584 + 121) >> 0] = -115; //@line 7725
 HEAP8[651929 + ($584 + 122) >> 0] = -114; //@line 7728
 HEAP8[651929 + ($584 + 123) >> 0] = -113; //@line 7731
 HEAP8[651929 + ($584 + 124) >> 0] = -112; //@line 7734
 HEAP8[651929 + ($584 + 125) >> 0] = -111; //@line 7737
 HEAP8[651929 + ($584 + 126) >> 0] = -110; //@line 7740
 HEAP8[651929 + ($584 + 127) >> 0] = -109; //@line 7743
 HEAP8[651929 + ($584 + 128) >> 0] = -108; //@line 7746
 HEAP8[651929 + ($584 + 129) >> 0] = -107; //@line 7749
 HEAP8[651929 + ($584 + 130) >> 0] = -107; //@line 7752
 HEAP8[651929 + ($584 + 131) >> 0] = -106; //@line 7755
 HEAP8[651929 + ($584 + 132) >> 0] = -105; //@line 7758
 HEAP8[651929 + ($584 + 133) >> 0] = -104; //@line 7761
 HEAP8[651929 + ($584 + 134) >> 0] = -103; //@line 7764
 HEAP8[651929 + ($584 + 135) >> 0] = -102; //@line 7767
 HEAP8[651929 + ($584 + 136) >> 0] = -101; //@line 7770
 HEAP8[651929 + ($584 + 137) >> 0] = -101; //@line 7773
 HEAP8[651929 + ($584 + 138) >> 0] = -100; //@line 7776
 HEAP8[651929 + ($584 + 139) >> 0] = -99; //@line 7779
 HEAP8[651929 + ($584 + 140) >> 0] = -98; //@line 7782
 HEAP8[651929 + ($584 + 141) >> 0] = -97; //@line 7785
 HEAP8[651929 + ($584 + 142) >> 0] = -96; //@line 7788
 HEAP8[651929 + ($584 + 143) >> 0] = -95; //@line 7791
 HEAP8[651929 + ($584 + 144) >> 0] = -95; //@line 7794
 HEAP8[651929 + ($584 + 145) >> 0] = -94; //@line 7797
 HEAP8[651929 + ($584 + 146) >> 0] = -93; //@line 7800
 HEAP8[651929 + ($584 + 147) >> 0] = -92; //@line 7803
 HEAP8[651929 + ($584 + 148) >> 0] = -91; //@line 7806
 HEAP8[651929 + ($584 + 149) >> 0] = -90; //@line 7809
 HEAP8[651929 + ($584 + 150) >> 0] = -90; //@line 7812
 HEAP8[651929 + ($584 + 151) >> 0] = -89; //@line 7815
 HEAP8[651929 + ($584 + 152) >> 0] = -88; //@line 7818
 HEAP8[651929 + ($584 + 153) >> 0] = -87; //@line 7821
 HEAP8[651929 + ($584 + 154) >> 0] = -86; //@line 7824
 HEAP8[651929 + ($584 + 155) >> 0] = -85; //@line 7827
 HEAP8[651929 + ($584 + 156) >> 0] = -85; //@line 7830
 HEAP8[651929 + ($584 + 157) >> 0] = -84; //@line 7833
 HEAP8[651929 + ($584 + 158) >> 0] = -83; //@line 7836
 HEAP8[651929 + ($584 + 159) >> 0] = -82; //@line 7839
 HEAP8[651929 + ($584 + 160) >> 0] = -81; //@line 7842
 HEAP8[651929 + ($584 + 161) >> 0] = -80; //@line 7845
 HEAP8[651929 + ($584 + 162) >> 0] = -80; //@line 7848
 HEAP8[651929 + ($584 + 163) >> 0] = -79; //@line 7851
 HEAP8[651929 + ($584 + 164) >> 0] = -78; //@line 7854
 HEAP8[651929 + ($584 + 165) >> 0] = -77; //@line 7857
 HEAP8[651929 + ($584 + 166) >> 0] = -76; //@line 7860
 HEAP8[651929 + ($584 + 167) >> 0] = -76; //@line 7863
 HEAP8[651929 + ($584 + 168) >> 0] = -75; //@line 7866
 HEAP8[651929 + ($584 + 169) >> 0] = -74; //@line 7869
 HEAP8[651929 + ($584 + 170) >> 0] = -73; //@line 7872
 HEAP8[651929 + ($584 + 171) >> 0] = -72; //@line 7875
 HEAP8[651929 + ($584 + 172) >> 0] = -72; //@line 7878
 HEAP8[651929 + ($584 + 173) >> 0] = -71; //@line 7881
 HEAP8[651929 + ($584 + 174) >> 0] = -70; //@line 7884
 HEAP8[651929 + ($584 + 175) >> 0] = -69; //@line 7887
 HEAP8[651929 + ($584 + 176) >> 0] = -68; //@line 7890
 HEAP8[651929 + ($584 + 177) >> 0] = -68; //@line 7893
 HEAP8[651929 + ($584 + 178) >> 0] = -67; //@line 7896
 HEAP8[651929 + ($584 + 179) >> 0] = -66; //@line 7899
 HEAP8[651929 + ($584 + 180) >> 0] = -65; //@line 7902
 HEAP8[651929 + ($584 + 181) >> 0] = -64; //@line 7905
 HEAP8[651929 + ($584 + 182) >> 0] = -64; //@line 7908
 HEAP8[651929 + ($584 + 183) >> 0] = -63; //@line 7911
 HEAP8[651929 + ($584 + 184) >> 0] = -62; //@line 7914
 HEAP8[651929 + ($584 + 185) >> 0] = -61; //@line 7917
 HEAP8[651929 + ($584 + 186) >> 0] = -61; //@line 7920
 HEAP8[651929 + ($584 + 187) >> 0] = -60; //@line 7923
 HEAP8[651929 + ($584 + 188) >> 0] = -59; //@line 7926
 HEAP8[651929 + ($584 + 189) >> 0] = -58; //@line 7929
 HEAP8[651929 + ($584 + 190) >> 0] = -57; //@line 7932
 HEAP8[651929 + ($584 + 191) >> 0] = -57; //@line 7935
 HEAP8[651929 + ($584 + 192) >> 0] = -56; //@line 7938
 HEAP8[651929 + ($584 + 193) >> 0] = -55; //@line 7941
 HEAP8[651929 + ($584 + 194) >> 0] = -54; //@line 7944
 HEAP8[651929 + ($584 + 195) >> 0] = -54; //@line 7947
 HEAP8[651929 + ($584 + 196) >> 0] = -53; //@line 7950
 HEAP8[651929 + ($584 + 197) >> 0] = -52; //@line 7953
 HEAP8[651929 + ($584 + 198) >> 0] = -51; //@line 7956
 HEAP8[651929 + ($584 + 199) >> 0] = -51; //@line 7959
 HEAP8[651929 + ($584 + 200) >> 0] = -50; //@line 7962
 HEAP8[651929 + ($584 + 201) >> 0] = -49; //@line 7965
 HEAP8[651929 + ($584 + 202) >> 0] = -48; //@line 7968
 HEAP8[651929 + ($584 + 203) >> 0] = -48; //@line 7971
 HEAP8[651929 + ($584 + 204) >> 0] = -47; //@line 7974
 HEAP8[651929 + ($584 + 205) >> 0] = -46; //@line 7977
 HEAP8[651929 + ($584 + 206) >> 0] = -45; //@line 7980
 HEAP8[651929 + ($584 + 207) >> 0] = -45; //@line 7983
 HEAP8[651929 + ($584 + 208) >> 0] = -44; //@line 7986
 HEAP8[651929 + ($584 + 209) >> 0] = -43; //@line 7989
 HEAP8[651929 + ($584 + 210) >> 0] = -42; //@line 7992
 HEAP8[651929 + ($584 + 211) >> 0] = -42; //@line 7995
 HEAP8[651929 + ($584 + 212) >> 0] = -41; //@line 7998
 HEAP8[651929 + ($584 + 213) >> 0] = -40; //@line 8001
 HEAP8[651929 + ($584 + 214) >> 0] = -39; //@line 8004
 HEAP8[651929 + ($584 + 215) >> 0] = -39; //@line 8007
 HEAP8[651929 + ($584 + 216) >> 0] = -38; //@line 8010
 HEAP8[651929 + ($584 + 217) >> 0] = -37; //@line 8013
 HEAP8[651929 + ($584 + 218) >> 0] = -36; //@line 8016
 HEAP8[651929 + ($584 + 219) >> 0] = -36; //@line 8019
 HEAP8[651929 + ($584 + 220) >> 0] = -35; //@line 8022
 HEAP8[651929 + ($584 + 221) >> 0] = -34; //@line 8025
 HEAP8[651929 + ($584 + 222) >> 0] = -33; //@line 8028
 HEAP8[651929 + ($584 + 223) >> 0] = -33; //@line 8031
 HEAP8[651929 + ($584 + 224) >> 0] = -32; //@line 8034
 HEAP8[651929 + ($584 + 225) >> 0] = -31; //@line 8037
 HEAP8[651929 + ($584 + 226) >> 0] = -31; //@line 8040
 HEAP8[651929 + ($584 + 227) >> 0] = -30; //@line 8043
 HEAP8[651929 + ($584 + 228) >> 0] = -29; //@line 8046
 HEAP8[651929 + ($584 + 229) >> 0] = -28; //@line 8049
 HEAP8[651929 + ($584 + 230) >> 0] = -28; //@line 8052
 HEAP8[651929 + ($584 + 231) >> 0] = -27; //@line 8055
 HEAP8[651929 + ($584 + 232) >> 0] = -26; //@line 8058
 HEAP8[651929 + ($584 + 233) >> 0] = -25; //@line 8061
 HEAP8[651929 + ($584 + 234) >> 0] = -25; //@line 8064
 HEAP8[651929 + ($584 + 235) >> 0] = -24; //@line 8067
 HEAP8[651929 + ($584 + 236) >> 0] = -23; //@line 8070
 HEAP8[651929 + ($584 + 237) >> 0] = -23; //@line 8073
 HEAP8[651929 + ($584 + 238) >> 0] = -22; //@line 8076
 HEAP8[651929 + ($584 + 239) >> 0] = -21; //@line 8079
 HEAP8[651929 + ($584 + 240) >> 0] = -20; //@line 8082
 HEAP8[651929 + ($584 + 241) >> 0] = -20; //@line 8085
 HEAP8[651929 + ($584 + 242) >> 0] = -19; //@line 8088
 HEAP8[651929 + ($584 + 243) >> 0] = -18; //@line 8091
 HEAP8[651929 + ($584 + 244) >> 0] = -18; //@line 8094
 HEAP8[651929 + ($584 + 245) >> 0] = -17; //@line 8097
 HEAP8[651929 + ($584 + 246) >> 0] = -16; //@line 8100
 HEAP8[651929 + ($584 + 247) >> 0] = -15; //@line 8103
 HEAP8[651929 + ($584 + 248) >> 0] = -15; //@line 8106
 HEAP8[651929 + ($584 + 249) >> 0] = -14; //@line 8109
 HEAP8[651929 + ($584 + 250) >> 0] = -13; //@line 8112
 HEAP8[651929 + ($584 + 251) >> 0] = -13; //@line 8115
 HEAP8[651929 + ($584 + 252) >> 0] = -12; //@line 8118
 HEAP8[651929 + ($584 + 253) >> 0] = -11; //@line 8121
 HEAP8[651929 + ($584 + 254) >> 0] = -11; //@line 8124
 HEAP8[651929 + ($584 + 255) >> 0] = -10; //@line 8127
 HEAP8[651929 + ($584 + 256) >> 0] = -9; //@line 8130
 HEAP8[651929 + ($584 + 257) >> 0] = -8; //@line 8133
 HEAP8[651929 + ($584 + 258) >> 0] = -8; //@line 8136
 HEAP8[651929 + ($584 + 259) >> 0] = -7; //@line 8139
 HEAP8[651929 + ($584 + 260) >> 0] = -6; //@line 8142
 HEAP8[651929 + ($584 + 261) >> 0] = -6; //@line 8145
 HEAP8[651929 + ($584 + 262) >> 0] = -5; //@line 8148
 HEAP8[651929 + ($584 + 263) >> 0] = -4; //@line 8151
 HEAP8[651929 + ($584 + 264) >> 0] = -4; //@line 8154
 HEAP8[651929 + ($584 + 265) >> 0] = -3; //@line 8157
 HEAP8[651929 + ($584 + 266) >> 0] = -2; //@line 8160
 $1118 = $584 + 268 | 0; //@line 8161
 HEAP8[651929 + ($584 + 267) >> 0] = -1; //@line 8163
 $1120 = $525 - $1118 | 0; //@line 8164
 HEAP8[651929 + $1118 >> 0] = -128; //@line 8167
 HEAP8[651929 + ($584 + 269) >> 0] = $1120 >>> 16; //@line 8172
 HEAP8[651929 + ($584 + 270) >> 0] = $1120 >>> 8; //@line 8177
 HEAP8[651929 + ($584 + 271) >> 0] = $1120; //@line 8181
 HEAP8[651929 + ($584 + 272) >> 0] = -1; //@line 8184
 HEAP8[651929 + ($584 + 273) >> 0] = 0; //@line 8187
 HEAP8[651929 + ($584 + 274) >> 0] = 0; //@line 8190
 $1140 = $584 + 276 | 0; //@line 8191
 HEAP32[28210] = $1140; //@line 8192
 HEAP8[651929 + ($584 + 275) >> 0] = 0; //@line 8194
 $1142 = HEAP32[30548] | 0; //@line 8195
 if (($1142 | 0) == ($2 | 0)) {
  $1637 = $1140; //@line 8198
 } else {
  $1144 = $0 + 8 | 0; //@line 8200
  $$sink159 = $1142; //@line 8201
  $1151 = $1140; //@line 8201
  while (1) {
   $1145 = HEAP32[$$sink159 >> 2] | 0; //@line 8203
   $1147 = HEAP32[$1145 + 20 >> 2] | 0; //@line 8205
   HEAP8[651929 + $1151 >> 0] = $1147 >>> 24; //@line 8210
   HEAP8[651929 + ($1151 + 1) >> 0] = $1147 >>> 16; //@line 8215
   HEAP8[651929 + ($1151 + 2) >> 0] = $1147 >>> 8; //@line 8220
   $1162 = $1151 + 4 | 0; //@line 8222
   HEAP8[651929 + ($1151 + 3) >> 0] = $1147; //@line 8224
   HEAP8[651929 + $1162 >> 0] = 0; //@line 8227
   HEAP8[651929 + ($1151 + 5) >> 0] = 0; //@line 8230
   HEAP8[651929 + ($1151 + 6) >> 0] = 0; //@line 8233
   HEAP32[28210] = $1151 + 8; //@line 8235
   HEAP8[651929 + ($1151 + 7) >> 0] = 0; //@line 8237
   $1172 = __ZL10VModeParmsRK12monitor_desc11video_depth($1145, 0) | 0; //@line 8238
   $1173 = __ZL10VModeParmsRK12monitor_desc11video_depth($1145, 1) | 0; //@line 8239
   $1174 = __ZL10VModeParmsRK12monitor_desc11video_depth($1145, 2) | 0; //@line 8240
   $1175 = __ZL10VModeParmsRK12monitor_desc11video_depth($1145, 3) | 0; //@line 8241
   $1176 = __ZL10VModeParmsRK12monitor_desc11video_depth($1145, 4) | 0; //@line 8242
   $1177 = __ZL10VModeParmsRK12monitor_desc11video_depth($1145, 5) | 0; //@line 8243
   $1178 = HEAP32[28210] | 0; //@line 8244
   $1179 = $1172 - $1178 | 0; //@line 8245
   HEAP8[651929 + $1178 >> 0] = 1; //@line 8248
   HEAP8[651929 + ($1178 + 1) >> 0] = $1179 >>> 16; //@line 8253
   HEAP8[651929 + ($1178 + 2) >> 0] = $1179 >>> 8; //@line 8258
   HEAP8[651929 + ($1178 + 3) >> 0] = $1179; //@line 8262
   HEAP8[651929 + ($1178 + 4) >> 0] = 3; //@line 8265
   HEAP8[651929 + ($1178 + 5) >> 0] = 0; //@line 8268
   HEAP8[651929 + ($1178 + 6) >> 0] = 0; //@line 8271
   HEAP8[651929 + ($1178 + 7) >> 0] = 1; //@line 8274
   HEAP8[651929 + ($1178 + 8) >> 0] = 4; //@line 8277
   HEAP8[651929 + ($1178 + 9) >> 0] = 0; //@line 8280
   HEAP8[651929 + ($1178 + 10) >> 0] = 0; //@line 8283
   HEAP8[651929 + ($1178 + 11) >> 0] = 0; //@line 8286
   HEAP8[651929 + ($1178 + 12) >> 0] = -1; //@line 8289
   HEAP8[651929 + ($1178 + 13) >> 0] = 0; //@line 8292
   HEAP8[651929 + ($1178 + 14) >> 0] = 0; //@line 8295
   $1215 = $1178 + 16 | 0; //@line 8296
   HEAP8[651929 + ($1178 + 15) >> 0] = 0; //@line 8298
   $1217 = $1173 - $1215 | 0; //@line 8299
   HEAP8[651929 + $1215 >> 0] = 1; //@line 8302
   HEAP8[651929 + ($1178 + 17) >> 0] = $1217 >>> 16; //@line 8307
   HEAP8[651929 + ($1178 + 18) >> 0] = $1217 >>> 8; //@line 8312
   HEAP8[651929 + ($1178 + 19) >> 0] = $1217; //@line 8316
   HEAP8[651929 + ($1178 + 20) >> 0] = 3; //@line 8319
   HEAP8[651929 + ($1178 + 21) >> 0] = 0; //@line 8322
   HEAP8[651929 + ($1178 + 22) >> 0] = 0; //@line 8325
   HEAP8[651929 + ($1178 + 23) >> 0] = 1; //@line 8328
   HEAP8[651929 + ($1178 + 24) >> 0] = 4; //@line 8331
   HEAP8[651929 + ($1178 + 25) >> 0] = 0; //@line 8334
   HEAP8[651929 + ($1178 + 26) >> 0] = 0; //@line 8337
   HEAP8[651929 + ($1178 + 27) >> 0] = 0; //@line 8340
   HEAP8[651929 + ($1178 + 28) >> 0] = -1; //@line 8343
   HEAP8[651929 + ($1178 + 29) >> 0] = 0; //@line 8346
   HEAP8[651929 + ($1178 + 30) >> 0] = 0; //@line 8349
   $1253 = $1178 + 32 | 0; //@line 8350
   HEAP8[651929 + ($1178 + 31) >> 0] = 0; //@line 8352
   $1255 = $1174 - $1253 | 0; //@line 8353
   HEAP8[651929 + $1253 >> 0] = 1; //@line 8356
   HEAP8[651929 + ($1178 + 33) >> 0] = $1255 >>> 16; //@line 8361
   HEAP8[651929 + ($1178 + 34) >> 0] = $1255 >>> 8; //@line 8366
   HEAP8[651929 + ($1178 + 35) >> 0] = $1255; //@line 8370
   HEAP8[651929 + ($1178 + 36) >> 0] = 3; //@line 8373
   HEAP8[651929 + ($1178 + 37) >> 0] = 0; //@line 8376
   HEAP8[651929 + ($1178 + 38) >> 0] = 0; //@line 8379
   HEAP8[651929 + ($1178 + 39) >> 0] = 1; //@line 8382
   HEAP8[651929 + ($1178 + 40) >> 0] = 4; //@line 8385
   HEAP8[651929 + ($1178 + 41) >> 0] = 0; //@line 8388
   HEAP8[651929 + ($1178 + 42) >> 0] = 0; //@line 8391
   HEAP8[651929 + ($1178 + 43) >> 0] = 0; //@line 8394
   HEAP8[651929 + ($1178 + 44) >> 0] = -1; //@line 8397
   HEAP8[651929 + ($1178 + 45) >> 0] = 0; //@line 8400
   HEAP8[651929 + ($1178 + 46) >> 0] = 0; //@line 8403
   $1291 = $1178 + 48 | 0; //@line 8404
   HEAP8[651929 + ($1178 + 47) >> 0] = 0; //@line 8406
   $1293 = $1175 - $1291 | 0; //@line 8407
   HEAP8[651929 + $1291 >> 0] = 1; //@line 8410
   HEAP8[651929 + ($1178 + 49) >> 0] = $1293 >>> 16; //@line 8415
   HEAP8[651929 + ($1178 + 50) >> 0] = $1293 >>> 8; //@line 8420
   HEAP8[651929 + ($1178 + 51) >> 0] = $1293; //@line 8424
   HEAP8[651929 + ($1178 + 52) >> 0] = 3; //@line 8427
   HEAP8[651929 + ($1178 + 53) >> 0] = 0; //@line 8430
   HEAP8[651929 + ($1178 + 54) >> 0] = 0; //@line 8433
   HEAP8[651929 + ($1178 + 55) >> 0] = 1; //@line 8436
   HEAP8[651929 + ($1178 + 56) >> 0] = 4; //@line 8439
   HEAP8[651929 + ($1178 + 57) >> 0] = 0; //@line 8442
   HEAP8[651929 + ($1178 + 58) >> 0] = 0; //@line 8445
   HEAP8[651929 + ($1178 + 59) >> 0] = 0; //@line 8448
   HEAP8[651929 + ($1178 + 60) >> 0] = -1; //@line 8451
   HEAP8[651929 + ($1178 + 61) >> 0] = 0; //@line 8454
   HEAP8[651929 + ($1178 + 62) >> 0] = 0; //@line 8457
   $1329 = $1178 + 64 | 0; //@line 8458
   HEAP8[651929 + ($1178 + 63) >> 0] = 0; //@line 8460
   $1331 = $1176 - $1329 | 0; //@line 8461
   HEAP8[651929 + $1329 >> 0] = 1; //@line 8464
   HEAP8[651929 + ($1178 + 65) >> 0] = $1331 >>> 16; //@line 8469
   HEAP8[651929 + ($1178 + 66) >> 0] = $1331 >>> 8; //@line 8474
   HEAP8[651929 + ($1178 + 67) >> 0] = $1331; //@line 8478
   HEAP8[651929 + ($1178 + 68) >> 0] = 3; //@line 8481
   HEAP8[651929 + ($1178 + 69) >> 0] = 0; //@line 8484
   HEAP8[651929 + ($1178 + 70) >> 0] = 0; //@line 8487
   HEAP8[651929 + ($1178 + 71) >> 0] = 1; //@line 8490
   HEAP8[651929 + ($1178 + 72) >> 0] = 4; //@line 8493
   HEAP8[651929 + ($1178 + 73) >> 0] = 0; //@line 8496
   HEAP8[651929 + ($1178 + 74) >> 0] = 0; //@line 8499
   HEAP8[651929 + ($1178 + 75) >> 0] = 2; //@line 8502
   HEAP8[651929 + ($1178 + 76) >> 0] = -1; //@line 8505
   HEAP8[651929 + ($1178 + 77) >> 0] = 0; //@line 8508
   HEAP8[651929 + ($1178 + 78) >> 0] = 0; //@line 8511
   $1367 = $1178 + 80 | 0; //@line 8512
   HEAP8[651929 + ($1178 + 79) >> 0] = 0; //@line 8514
   $1369 = $1177 - $1367 | 0; //@line 8515
   HEAP8[651929 + $1367 >> 0] = 1; //@line 8518
   HEAP8[651929 + ($1178 + 81) >> 0] = $1369 >>> 16; //@line 8523
   HEAP8[651929 + ($1178 + 82) >> 0] = $1369 >>> 8; //@line 8528
   HEAP8[651929 + ($1178 + 83) >> 0] = $1369; //@line 8532
   HEAP8[651929 + ($1178 + 84) >> 0] = 3; //@line 8535
   HEAP8[651929 + ($1178 + 85) >> 0] = 0; //@line 8538
   HEAP8[651929 + ($1178 + 86) >> 0] = 0; //@line 8541
   HEAP8[651929 + ($1178 + 87) >> 0] = 1; //@line 8544
   HEAP8[651929 + ($1178 + 88) >> 0] = 4; //@line 8547
   HEAP8[651929 + ($1178 + 89) >> 0] = 0; //@line 8550
   HEAP8[651929 + ($1178 + 90) >> 0] = 0; //@line 8553
   HEAP8[651929 + ($1178 + 91) >> 0] = 2; //@line 8556
   HEAP8[651929 + ($1178 + 92) >> 0] = -1; //@line 8559
   HEAP8[651929 + ($1178 + 93) >> 0] = 0; //@line 8562
   HEAP8[651929 + ($1178 + 94) >> 0] = 0; //@line 8565
   $1405 = $1178 + 96 | 0; //@line 8566
   HEAP8[651929 + ($1178 + 95) >> 0] = 0; //@line 8568
   $1407 = $186 - $1405 | 0; //@line 8569
   HEAP8[651929 + $1405 >> 0] = 1; //@line 8572
   HEAP8[651929 + ($1178 + 97) >> 0] = $1407 >>> 16; //@line 8577
   HEAP8[651929 + ($1178 + 98) >> 0] = $1407 >>> 8; //@line 8582
   $1419 = $1178 + 100 | 0; //@line 8584
   HEAP8[651929 + ($1178 + 99) >> 0] = $1407; //@line 8586
   $1421 = $202 - $1419 | 0; //@line 8587
   HEAP8[651929 + $1419 >> 0] = 2; //@line 8590
   HEAP8[651929 + ($1178 + 101) >> 0] = $1421 >>> 16; //@line 8595
   HEAP8[651929 + ($1178 + 102) >> 0] = $1421 >>> 8; //@line 8600
   $1433 = $1178 + 104 | 0; //@line 8602
   HEAP8[651929 + ($1178 + 103) >> 0] = $1421; //@line 8604
   $1435 = $503 - $1433 | 0; //@line 8605
   HEAP8[651929 + $1433 >> 0] = 4; //@line 8608
   HEAP8[651929 + ($1178 + 105) >> 0] = $1435 >>> 16; //@line 8613
   HEAP8[651929 + ($1178 + 106) >> 0] = $1435 >>> 8; //@line 8618
   HEAP8[651929 + ($1178 + 107) >> 0] = $1435; //@line 8622
   HEAP8[651929 + ($1178 + 108) >> 0] = 8; //@line 8625
   HEAP8[651929 + ($1178 + 109) >> 0] = 0; //@line 8628
   HEAP8[651929 + ($1178 + 110) >> 0] = 66; //@line 8631
   $1455 = $1178 + 112 | 0; //@line 8632
   HEAP8[651929 + ($1178 + 111) >> 0] = 50; //@line 8634
   $1457 = $1151 - $1455 | 0; //@line 8635
   HEAP8[651929 + $1455 >> 0] = 10; //@line 8638
   HEAP8[651929 + ($1178 + 113) >> 0] = $1457 >>> 16; //@line 8643
   HEAP8[651929 + ($1178 + 114) >> 0] = $1457 >>> 8; //@line 8648
   $1469 = $1178 + 116 | 0; //@line 8650
   HEAP8[651929 + ($1178 + 115) >> 0] = $1457; //@line 8652
   $1471 = $1162 - $1469 | 0; //@line 8653
   HEAP8[651929 + $1469 >> 0] = 11; //@line 8656
   HEAP8[651929 + ($1178 + 117) >> 0] = $1471 >>> 16; //@line 8661
   HEAP8[651929 + ($1178 + 118) >> 0] = $1471 >>> 8; //@line 8666
   $1483 = $1178 + 120 | 0; //@line 8668
   HEAP8[651929 + ($1178 + 119) >> 0] = $1471; //@line 8670
   $1485 = $1118 - $1483 | 0; //@line 8671
   HEAP8[651929 + $1483 >> 0] = 64; //@line 8674
   HEAP8[651929 + ($1178 + 121) >> 0] = $1485 >>> 16; //@line 8679
   HEAP8[651929 + ($1178 + 122) >> 0] = $1485 >>> 8; //@line 8684
   HEAP8[651929 + ($1178 + 123) >> 0] = $1485; //@line 8688
   HEAP8[651929 + ($1178 + 124) >> 0] = 125; //@line 8691
   HEAP8[651929 + ($1178 + 125) >> 0] = 0; //@line 8694
   HEAP8[651929 + ($1178 + 126) >> 0] = 0; //@line 8697
   HEAP32[28210] = $1178 + 128; //@line 8699
   HEAP8[651929 + ($1178 + 127) >> 0] = 6; //@line 8701
   if (__ZNK12monitor_desc9has_depthE11video_depth($1145, 0) | 0) {
    $1511 = HEAP32[28210] | 0; //@line 8707
    $1512 = $1178 - $1511 | 0; //@line 8708
    HEAP8[651929 + $1511 >> 0] = HEAP16[$1145 + 824 >> 1]; //@line 8711
    HEAP8[651929 + ($1511 + 1) >> 0] = $1512 >>> 16; //@line 8716
    HEAP8[651929 + ($1511 + 2) >> 0] = $1512 >>> 8; //@line 8721
    HEAP32[28210] = $1511 + 4; //@line 8724
    HEAP8[651929 + ($1511 + 3) >> 0] = $1512; //@line 8726
   }
   if (__ZNK12monitor_desc9has_depthE11video_depth($1145, 1) | 0) {
    $1530 = HEAP32[28210] | 0; //@line 8733
    $1531 = $1215 - $1530 | 0; //@line 8734
    HEAP8[651929 + $1530 >> 0] = HEAP16[$1145 + 826 >> 1]; //@line 8737
    HEAP8[651929 + ($1530 + 1) >> 0] = $1531 >>> 16; //@line 8742
    HEAP8[651929 + ($1530 + 2) >> 0] = $1531 >>> 8; //@line 8747
    HEAP32[28210] = $1530 + 4; //@line 8750
    HEAP8[651929 + ($1530 + 3) >> 0] = $1531; //@line 8752
   }
   if (__ZNK12monitor_desc9has_depthE11video_depth($1145, 2) | 0) {
    $1549 = HEAP32[28210] | 0; //@line 8759
    $1550 = $1253 - $1549 | 0; //@line 8760
    HEAP8[651929 + $1549 >> 0] = HEAP16[$1145 + 828 >> 1]; //@line 8763
    HEAP8[651929 + ($1549 + 1) >> 0] = $1550 >>> 16; //@line 8768
    HEAP8[651929 + ($1549 + 2) >> 0] = $1550 >>> 8; //@line 8773
    HEAP32[28210] = $1549 + 4; //@line 8776
    HEAP8[651929 + ($1549 + 3) >> 0] = $1550; //@line 8778
   }
   if (__ZNK12monitor_desc9has_depthE11video_depth($1145, 3) | 0) {
    $1568 = HEAP32[28210] | 0; //@line 8785
    $1569 = $1291 - $1568 | 0; //@line 8786
    HEAP8[651929 + $1568 >> 0] = HEAP16[$1145 + 830 >> 1]; //@line 8789
    HEAP8[651929 + ($1568 + 1) >> 0] = $1569 >>> 16; //@line 8794
    HEAP8[651929 + ($1568 + 2) >> 0] = $1569 >>> 8; //@line 8799
    HEAP32[28210] = $1568 + 4; //@line 8802
    HEAP8[651929 + ($1568 + 3) >> 0] = $1569; //@line 8804
   }
   if (__ZNK12monitor_desc9has_depthE11video_depth($1145, 4) | 0) {
    $1587 = HEAP32[28210] | 0; //@line 8811
    $1588 = $1329 - $1587 | 0; //@line 8812
    HEAP8[651929 + $1587 >> 0] = HEAP16[$1145 + 832 >> 1]; //@line 8815
    HEAP8[651929 + ($1587 + 1) >> 0] = $1588 >>> 16; //@line 8820
    HEAP8[651929 + ($1587 + 2) >> 0] = $1588 >>> 8; //@line 8825
    HEAP32[28210] = $1587 + 4; //@line 8828
    HEAP8[651929 + ($1587 + 3) >> 0] = $1588; //@line 8830
   }
   if (__ZNK12monitor_desc9has_depthE11video_depth($1145, 5) | 0) {
    $1606 = HEAP32[28210] | 0; //@line 8837
    $1607 = $1367 - $1606 | 0; //@line 8838
    HEAP8[651929 + $1606 >> 0] = HEAP16[$1145 + 834 >> 1]; //@line 8841
    HEAP8[651929 + ($1606 + 1) >> 0] = $1607 >>> 16; //@line 8846
    HEAP8[651929 + ($1606 + 2) >> 0] = $1607 >>> 8; //@line 8851
    $1619 = $1606 + 4 | 0; //@line 8853
    HEAP32[28210] = $1619; //@line 8854
    HEAP8[651929 + ($1606 + 3) >> 0] = $1607; //@line 8856
    $1622 = $1619; //@line 8857
   } else {
    $1622 = HEAP32[28210] | 0; //@line 8860
   }
   HEAP8[651929 + $1622 >> 0] = -1; //@line 8864
   HEAP8[651929 + ($1622 + 1) >> 0] = 0; //@line 8867
   HEAP8[651929 + ($1622 + 2) >> 0] = 0; //@line 8870
   HEAP32[28210] = $1622 + 4; //@line 8872
   HEAP8[651929 + ($1622 + 3) >> 0] = 0; //@line 8874
   HEAP32[$vararg_buffer >> 2] = $1405; //@line 8875
   $1630 = HEAP32[$3 >> 2] | 0; //@line 8876
   if (($1630 | 0) == (HEAP32[$1144 >> 2] | 0)) {
    __ZNSt3__26vectorIjNS_9allocatorIjEEE21__push_back_slow_pathIKjEEvRT_($0, $vararg_buffer); //@line 8880
   } else {
    HEAP32[$1630 >> 2] = $1405; //@line 8882
    HEAP32[$3 >> 2] = $1630 + 4; //@line 8884
   }
   $$sink159 = $$sink159 + 4 | 0; //@line 8886
   $$pre218 = HEAP32[28210] | 0; //@line 8888
   if (($$sink159 | 0) == ($2 | 0)) {
    $1637 = $$pre218; //@line 8890
    break;
   } else {
    $1151 = $$pre218; //@line 8893
   }
  }
 }
 HEAP8[651929 + $1637 >> 0] = 0; //@line 8899
 HEAP8[651929 + ($1637 + 1) >> 0] = 10; //@line 8902
 HEAP8[651929 + ($1637 + 2) >> 0] = 0; //@line 8905
 HEAP8[651929 + ($1637 + 3) >> 0] = 3; //@line 8908
 HEAP8[651929 + ($1637 + 4) >> 0] = 0; //@line 8911
 HEAP8[651929 + ($1637 + 5) >> 0] = 0; //@line 8914
 HEAP8[651929 + ($1637 + 6) >> 0] = 0; //@line 8917
 $1651 = $1637 + 8 | 0; //@line 8918
 HEAP8[651929 + ($1637 + 7) >> 0] = 24; //@line 8920
 HEAP8[651929 + $1651 >> 0] = 67; //@line 8923
 HEAP8[651929 + ($1637 + 9) >> 0] = 80; //@line 8926
 HEAP8[651929 + ($1637 + 10) >> 0] = 85; //@line 8929
 HEAP8[651929 + ($1637 + 11) >> 0] = 95; //@line 8932
 HEAP8[651929 + ($1637 + 12) >> 0] = 54; //@line 8935
 HEAP8[651929 + ($1637 + 13) >> 0] = 56; //@line 8938
 HEAP8[651929 + ($1637 + 14) >> 0] = 48; //@line 8941
 HEAP8[651929 + ($1637 + 15) >> 0] = 50; //@line 8944
 HEAP8[651929 + ($1637 + 16) >> 0] = 48; //@line 8947
 HEAP8[651929 + ($1637 + 17) >> 0] = 0; //@line 8949
 $1672 = $1637 + 18 | 0; //@line 8950
 HEAP32[28210] = $1672; //@line 8951
 if (!($1672 & 1)) {
  $1678 = $1672; //@line 8955
 } else {
  $1675 = $1637 + 19 | 0; //@line 8957
  HEAP32[28210] = $1675; //@line 8959
  HEAP8[651929 + $1672 >> 0] = 0; //@line 8960
  $1678 = $1675; //@line 8961
 }
 HEAP8[651929 + $1678 >> 0] = 0; //@line 8965
 HEAP8[651929 + ($1678 + 1) >> 0] = 0; //@line 8968
 HEAP8[651929 + ($1678 + 2) >> 0] = 0; //@line 8971
 HEAP8[651929 + ($1678 + 3) >> 0] = 0; //@line 8974
 HEAP8[651929 + ($1678 + 4) >> 0] = 127; //@line 8977
 HEAP8[651929 + ($1678 + 5) >> 0] = -1; //@line 8980
 HEAP8[651929 + ($1678 + 6) >> 0] = -1; //@line 8983
 HEAP8[651929 + ($1678 + 7) >> 0] = -1; //@line 8986
 HEAP8[651929 + ($1678 + 8) >> 0] = -16; //@line 8989
 HEAP8[651929 + ($1678 + 9) >> 0] = -128; //@line 8992
 HEAP8[651929 + ($1678 + 10) >> 0] = 0; //@line 8995
 HEAP8[651929 + ($1678 + 11) >> 0] = 0; //@line 8998
 HEAP8[651929 + ($1678 + 12) >> 0] = -16; //@line 9001
 HEAP8[651929 + ($1678 + 13) >> 0] = -1; //@line 9004
 HEAP8[651929 + ($1678 + 14) >> 0] = -1; //@line 9007
 $1708 = $1678 + 16 | 0; //@line 9008
 HEAP8[651929 + ($1678 + 15) >> 0] = -1; //@line 9010
 $1710 = $1637 - $1708 | 0; //@line 9011
 HEAP8[651929 + $1708 >> 0] = 1; //@line 9014
 HEAP8[651929 + ($1678 + 17) >> 0] = $1710 >>> 16; //@line 9019
 HEAP8[651929 + ($1678 + 18) >> 0] = $1710 >>> 8; //@line 9024
 $1722 = $1678 + 20 | 0; //@line 9026
 HEAP8[651929 + ($1678 + 19) >> 0] = $1710; //@line 9028
 $1724 = $1651 - $1722 | 0; //@line 9029
 HEAP8[651929 + $1722 >> 0] = 2; //@line 9032
 HEAP8[651929 + ($1678 + 21) >> 0] = $1724 >>> 16; //@line 9037
 HEAP8[651929 + ($1678 + 22) >> 0] = $1724 >>> 8; //@line 9042
 HEAP8[651929 + ($1678 + 23) >> 0] = $1724; //@line 9046
 HEAP8[651929 + ($1678 + 24) >> 0] = -127; //@line 9049
 HEAP8[651929 + ($1678 + 25) >> 0] = -1; //@line 9052
 HEAP8[651929 + ($1678 + 26) >> 0] = -1; //@line 9055
 HEAP8[651929 + ($1678 + 27) >> 0] = -24; //@line 9058
 HEAP8[651929 + ($1678 + 28) >> 0] = -126; //@line 9061
 HEAP8[651929 + ($1678 + 29) >> 0] = -1; //@line 9064
 HEAP8[651929 + ($1678 + 30) >> 0] = -1; //@line 9067
 HEAP8[651929 + ($1678 + 31) >> 0] = -20; //@line 9070
 HEAP8[651929 + ($1678 + 32) >> 0] = -1; //@line 9073
 HEAP8[651929 + ($1678 + 33) >> 0] = 0; //@line 9076
 HEAP8[651929 + ($1678 + 34) >> 0] = 0; //@line 9079
 $1760 = $1678 + 36 | 0; //@line 9080
 HEAP8[651929 + ($1678 + 35) >> 0] = 0; //@line 9082
 HEAP8[651929 + $1760 >> 0] = 0; //@line 9085
 HEAP8[651929 + ($1678 + 37) >> 0] = 4; //@line 9088
 HEAP8[651929 + ($1678 + 38) >> 0] = 0; //@line 9091
 HEAP8[651929 + ($1678 + 39) >> 0] = 1; //@line 9094
 HEAP8[651929 + ($1678 + 40) >> 0] = 0; //@line 9097
 HEAP8[651929 + ($1678 + 41) >> 0] = 1; //@line 9100
 HEAP8[651929 + ($1678 + 42) >> 0] = 66; //@line 9103
 $1776 = $1678 + 44 | 0; //@line 9104
 HEAP32[28210] = $1776; //@line 9105
 HEAP8[651929 + ($1678 + 43) >> 0] = 50; //@line 9107
 HEAP8[651929 + $1776 >> 0] = 78; //@line 9110
 HEAP8[651929 + ($1678 + 45) >> 0] = 101; //@line 9113
 HEAP8[651929 + ($1678 + 46) >> 0] = 116; //@line 9116
 HEAP8[651929 + ($1678 + 47) >> 0] = 119; //@line 9119
 HEAP8[651929 + ($1678 + 48) >> 0] = 111; //@line 9122
 HEAP8[651929 + ($1678 + 49) >> 0] = 114; //@line 9125
 HEAP8[651929 + ($1678 + 50) >> 0] = 107; //@line 9128
 HEAP8[651929 + ($1678 + 51) >> 0] = 95; //@line 9131
 HEAP8[651929 + ($1678 + 52) >> 0] = 69; //@line 9134
 HEAP8[651929 + ($1678 + 53) >> 0] = 116; //@line 9137
 HEAP8[651929 + ($1678 + 54) >> 0] = 104; //@line 9140
 HEAP8[651929 + ($1678 + 55) >> 0] = 101; //@line 9143
 HEAP8[651929 + ($1678 + 56) >> 0] = 114; //@line 9146
 HEAP8[651929 + ($1678 + 57) >> 0] = 110; //@line 9149
 HEAP8[651929 + ($1678 + 58) >> 0] = 101; //@line 9152
 HEAP8[651929 + ($1678 + 59) >> 0] = 116; //@line 9155
 HEAP8[651929 + ($1678 + 60) >> 0] = 95; //@line 9158
 HEAP8[651929 + ($1678 + 61) >> 0] = 65; //@line 9161
 HEAP8[651929 + ($1678 + 62) >> 0] = 112; //@line 9164
 HEAP8[651929 + ($1678 + 63) >> 0] = 112; //@line 9167
 HEAP8[651929 + ($1678 + 64) >> 0] = 108; //@line 9170
 HEAP8[651929 + ($1678 + 65) >> 0] = 101; //@line 9173
 HEAP8[651929 + ($1678 + 66) >> 0] = 95; //@line 9176
 HEAP8[651929 + ($1678 + 67) >> 0] = 66; //@line 9179
 HEAP8[651929 + ($1678 + 68) >> 0] = 97; //@line 9182
 HEAP8[651929 + ($1678 + 69) >> 0] = 115; //@line 9185
 HEAP8[651929 + ($1678 + 70) >> 0] = 105; //@line 9188
 HEAP8[651929 + ($1678 + 71) >> 0] = 108; //@line 9191
 HEAP8[651929 + ($1678 + 72) >> 0] = 105; //@line 9194
 HEAP8[651929 + ($1678 + 73) >> 0] = 115; //@line 9197
 HEAP8[651929 + ($1678 + 74) >> 0] = 107; //@line 9200
 HEAP8[651929 + ($1678 + 75) >> 0] = 73; //@line 9203
 HEAP8[651929 + ($1678 + 76) >> 0] = 73; //@line 9206
 HEAP8[651929 + ($1678 + 77) >> 0] = 0; //@line 9208
 $1845 = $1678 + 78 | 0; //@line 9209
 HEAP32[28210] = $1845; //@line 9210
 if (!($1845 & 1)) {
  $1851 = $1845; //@line 9214
 } else {
  $1848 = $1678 + 79 | 0; //@line 9216
  HEAP32[28210] = $1848; //@line 9218
  HEAP8[651929 + $1845 >> 0] = 0; //@line 9219
  $1851 = $1848; //@line 9220
 }
 HEAP8[651929 + $1851 >> 0] = 0; //@line 9224
 HEAP8[651929 + ($1851 + 1) >> 0] = 0; //@line 9227
 HEAP8[651929 + ($1851 + 2) >> 0] = 0; //@line 9230
 HEAP8[651929 + ($1851 + 3) >> 0] = -120; //@line 9233
 HEAP8[651929 + ($1851 + 4) >> 0] = 68; //@line 9236
 HEAP8[651929 + ($1851 + 5) >> 0] = 0; //@line 9239
 HEAP8[651929 + ($1851 + 6) >> 0] = 0; //@line 9242
 HEAP8[651929 + ($1851 + 7) >> 0] = 0; //@line 9245
 HEAP8[651929 + ($1851 + 8) >> 0] = 0; //@line 9248
 HEAP8[651929 + ($1851 + 9) >> 0] = 0; //@line 9251
 HEAP8[651929 + ($1851 + 10) >> 0] = 0; //@line 9254
 HEAP8[651929 + ($1851 + 11) >> 0] = 0; //@line 9257
 HEAP8[651929 + ($1851 + 12) >> 0] = 0; //@line 9260
 HEAP8[651929 + ($1851 + 13) >> 0] = 74; //@line 9263
 HEAP8[651929 + ($1851 + 14) >> 0] = 0; //@line 9266
 HEAP8[651929 + ($1851 + 15) >> 0] = 78; //@line 9269
 HEAP8[651929 + ($1851 + 16) >> 0] = 0; //@line 9272
 HEAP8[651929 + ($1851 + 17) >> 0] = 82; //@line 9275
 HEAP8[651929 + ($1851 + 18) >> 0] = 0; //@line 9278
 HEAP8[651929 + ($1851 + 19) >> 0] = 78; //@line 9281
 HEAP8[651929 + ($1851 + 20) >> 0] = 0; //@line 9284
 HEAP8[651929 + ($1851 + 21) >> 0] = -126; //@line 9287
 HEAP8[651929 + ($1851 + 22) >> 0] = 5; //@line 9290
 $1897 = $1851 + 29 | 0; //@line 9291
 HEAP8[651929 + ($1851 + 23) >> 0] = 46; //@line 9294
 HEAP8[651929 + ($1851 + 24) >> 0] = 69; //@line 9297
 HEAP8[651929 + ($1851 + 25) >> 0] = 78; //@line 9300
 HEAP8[651929 + ($1851 + 26) >> 0] = 69; //@line 9303
 HEAP8[651929 + ($1851 + 27) >> 0] = 84; //@line 9306
 HEAP8[651929 + ($1851 + 28) >> 0] = 0; //@line 9308
 $1909 = $1851 + 28 | 0; //@line 9309
 HEAP32[28210] = $1909; //@line 9310
 if (!($1909 & 1)) {
  $1913 = $1909; //@line 9314
 } else {
  HEAP32[28210] = $1897; //@line 9317
  HEAP8[651929 + $1909 >> 0] = 0; //@line 9318
  $1913 = $1897; //@line 9319
 }
 HEAP8[651929 + $1913 >> 0] = 1; //@line 9323
 HEAP8[651929 + ($1913 + 1) >> 0] = 17; //@line 9326
 HEAP8[651929 + ($1913 + 2) >> 0] = -128; //@line 9329
 HEAP8[651929 + ($1913 + 3) >> 0] = 0; //@line 9332
 HEAP8[651929 + ($1913 + 4) >> 0] = 0; //@line 9335
 HEAP8[651929 + ($1913 + 5) >> 0] = 0; //@line 9338
 HEAP8[651929 + ($1913 + 6) >> 0] = 7; //@line 9341
 $1927 = $1913 + 15 | 0; //@line 9342
 HEAP8[651929 + ($1913 + 7) >> 0] = 49; //@line 9345
 HEAP8[651929 + ($1913 + 8) >> 0] = 46; //@line 9348
 HEAP8[651929 + ($1913 + 9) >> 0] = 49; //@line 9351
 HEAP8[651929 + ($1913 + 10) >> 0] = 46; //@line 9354
 HEAP8[651929 + ($1913 + 11) >> 0] = 49; //@line 9357
 HEAP8[651929 + ($1913 + 12) >> 0] = 32; //@line 9360
 HEAP8[651929 + ($1913 + 13) >> 0] = 32; //@line 9363
 HEAP8[651929 + ($1913 + 14) >> 0] = 0; //@line 9365
 $1943 = $1913 + 14 | 0; //@line 9366
 HEAP32[28210] = $1943; //@line 9367
 if (!($1943 & 1)) {
  $1947 = $1943; //@line 9371
 } else {
  HEAP32[28210] = $1927; //@line 9374
  HEAP8[651929 + $1943 >> 0] = 0; //@line 9375
  $1947 = $1927; //@line 9376
 }
 $1946 = $1947 + 1 | 0; //@line 9378
 HEAP32[28210] = $1946; //@line 9379
 HEAP8[651929 + $1947 >> 0] = 35; //@line 9381
 $1949 = $1947 + 37 | 0; //@line 9382
 HEAP8[651929 + $1946 >> 0] = 66; //@line 9385
 HEAP8[651929 + ($1947 + 2) >> 0] = 97; //@line 9388
 HEAP8[651929 + ($1947 + 3) >> 0] = 115; //@line 9391
 HEAP8[651929 + ($1947 + 4) >> 0] = 105; //@line 9394
 HEAP8[651929 + ($1947 + 5) >> 0] = 108; //@line 9397
 HEAP8[651929 + ($1947 + 6) >> 0] = 105; //@line 9400
 HEAP8[651929 + ($1947 + 7) >> 0] = 115; //@line 9403
 HEAP8[651929 + ($1947 + 8) >> 0] = 107; //@line 9406
 HEAP8[651929 + ($1947 + 9) >> 0] = 32; //@line 9409
 HEAP8[651929 + ($1947 + 10) >> 0] = 73; //@line 9412
 HEAP8[651929 + ($1947 + 11) >> 0] = 73; //@line 9415
 HEAP8[651929 + ($1947 + 12) >> 0] = 32; //@line 9418
 HEAP8[651929 + ($1947 + 13) >> 0] = 69; //@line 9421
 HEAP8[651929 + ($1947 + 14) >> 0] = 116; //@line 9424
 HEAP8[651929 + ($1947 + 15) >> 0] = 104; //@line 9427
 HEAP8[651929 + ($1947 + 16) >> 0] = 101; //@line 9430
 HEAP8[651929 + ($1947 + 17) >> 0] = 114; //@line 9433
 HEAP8[651929 + ($1947 + 18) >> 0] = 110; //@line 9436
 HEAP8[651929 + ($1947 + 19) >> 0] = 101; //@line 9439
 HEAP8[651929 + ($1947 + 20) >> 0] = 116; //@line 9442
 HEAP8[651929 + ($1947 + 21) >> 0] = 32; //@line 9445
 HEAP8[651929 + ($1947 + 22) >> 0] = 78; //@line 9448
 HEAP8[651929 + ($1947 + 23) >> 0] = 101; //@line 9451
 HEAP8[651929 + ($1947 + 24) >> 0] = 116; //@line 9454
 HEAP8[651929 + ($1947 + 25) >> 0] = 119; //@line 9457
 HEAP8[651929 + ($1947 + 26) >> 0] = 111; //@line 9460
 HEAP8[651929 + ($1947 + 27) >> 0] = 114; //@line 9463
 HEAP8[651929 + ($1947 + 28) >> 0] = 107; //@line 9466
 HEAP8[651929 + ($1947 + 29) >> 0] = 32; //@line 9469
 HEAP8[651929 + ($1947 + 30) >> 0] = 68; //@line 9472
 HEAP8[651929 + ($1947 + 31) >> 0] = 114; //@line 9475
 HEAP8[651929 + ($1947 + 32) >> 0] = 105; //@line 9478
 HEAP8[651929 + ($1947 + 33) >> 0] = 118; //@line 9481
 HEAP8[651929 + ($1947 + 34) >> 0] = 101; //@line 9484
 HEAP8[651929 + ($1947 + 35) >> 0] = 114; //@line 9487
 HEAP8[651929 + ($1947 + 36) >> 0] = 0; //@line 9489
 $2021 = $1947 + 36 | 0; //@line 9490
 HEAP32[28210] = $2021; //@line 9491
 if (!($2021 & 1)) {
  $2025 = $2021; //@line 9495
 } else {
  HEAP32[28210] = $1949; //@line 9498
  HEAP8[651929 + $2021 >> 0] = 0; //@line 9499
  $2025 = $1949; //@line 9500
 }
 HEAP8[651929 + $2025 >> 0] = 113; //@line 9504
 HEAP8[651929 + ($2025 + 1) >> 0] = 32; //@line 9507
 HEAP8[651929 + ($2025 + 2) >> 0] = 78; //@line 9510
 HEAP8[651929 + ($2025 + 3) >> 0] = 117; //@line 9513
 HEAP8[651929 + ($2025 + 4) >> 0] = 112; //@line 9516
 HEAP8[651929 + ($2025 + 5) >> 0] = -17; //@line 9519
 HEAP8[651929 + ($2025 + 6) >> 0] = 96; //@line 9522
 HEAP8[651929 + ($2025 + 7) >> 0] = 12; //@line 9525
 HEAP8[651929 + ($2025 + 8) >> 0] = 113; //@line 9528
 HEAP8[651929 + ($2025 + 9) >> 0] = 33; //@line 9531
 HEAP8[651929 + ($2025 + 10) >> 0] = 12; //@line 9534
 HEAP8[651929 + ($2025 + 11) >> 0] = 104; //@line 9537
 HEAP8[651929 + ($2025 + 12) >> 0] = 0; //@line 9540
 HEAP8[651929 + ($2025 + 13) >> 0] = 1; //@line 9543
 HEAP8[651929 + ($2025 + 14) >> 0] = 0; //@line 9546
 HEAP8[651929 + ($2025 + 15) >> 0] = 26; //@line 9549
 HEAP8[651929 + ($2025 + 16) >> 0] = 102; //@line 9552
 HEAP8[651929 + ($2025 + 17) >> 0] = 2; //@line 9555
 HEAP8[651929 + ($2025 + 18) >> 0] = 78; //@line 9558
 HEAP8[651929 + ($2025 + 19) >> 0] = 117; //@line 9561
 HEAP8[651929 + ($2025 + 20) >> 0] = 50; //@line 9564
 HEAP8[651929 + ($2025 + 21) >> 0] = 40; //@line 9567
 HEAP8[651929 + ($2025 + 22) >> 0] = 0; //@line 9570
 HEAP8[651929 + ($2025 + 23) >> 0] = 6; //@line 9573
 HEAP8[651929 + ($2025 + 24) >> 0] = 8; //@line 9576
 HEAP8[651929 + ($2025 + 25) >> 0] = 1; //@line 9579
 HEAP8[651929 + ($2025 + 26) >> 0] = 0; //@line 9582
 HEAP8[651929 + ($2025 + 27) >> 0] = 9; //@line 9585
 HEAP8[651929 + ($2025 + 28) >> 0] = 103; //@line 9588
 HEAP8[651929 + ($2025 + 29) >> 0] = 12; //@line 9591
 HEAP8[651929 + ($2025 + 30) >> 0] = 74; //@line 9594
 HEAP8[651929 + ($2025 + 31) >> 0] = 64; //@line 9597
 HEAP8[651929 + ($2025 + 32) >> 0] = 111; //@line 9600
 HEAP8[651929 + ($2025 + 33) >> 0] = 2; //@line 9603
 HEAP8[651929 + ($2025 + 34) >> 0] = 66; //@line 9606
 HEAP8[651929 + ($2025 + 35) >> 0] = 64; //@line 9609
 HEAP8[651929 + ($2025 + 36) >> 0] = 49; //@line 9612
 HEAP8[651929 + ($2025 + 37) >> 0] = 64; //@line 9615
 HEAP8[651929 + ($2025 + 38) >> 0] = 0; //@line 9618
 HEAP8[651929 + ($2025 + 39) >> 0] = 16; //@line 9621
 HEAP8[651929 + ($2025 + 40) >> 0] = 78; //@line 9624
 HEAP8[651929 + ($2025 + 41) >> 0] = 117; //@line 9627
 HEAP8[651929 + ($2025 + 42) >> 0] = 74; //@line 9630
 HEAP8[651929 + ($2025 + 43) >> 0] = 64; //@line 9633
 HEAP8[651929 + ($2025 + 44) >> 0] = 111; //@line 9636
 HEAP8[651929 + ($2025 + 45) >> 0] = 4; //@line 9639
 HEAP8[651929 + ($2025 + 46) >> 0] = 66; //@line 9642
 HEAP8[651929 + ($2025 + 47) >> 0] = 64; //@line 9645
 HEAP8[651929 + ($2025 + 48) >> 0] = 78; //@line 9648
 HEAP8[651929 + ($2025 + 49) >> 0] = 117; //@line 9651
 HEAP8[651929 + ($2025 + 50) >> 0] = 47; //@line 9654
 HEAP8[651929 + ($2025 + 51) >> 0] = 56; //@line 9657
 HEAP8[651929 + ($2025 + 52) >> 0] = 8; //@line 9660
 HEAP8[651929 + ($2025 + 53) >> 0] = -4; //@line 9663
 HEAP8[651929 + ($2025 + 54) >> 0] = 78; //@line 9666
 HEAP8[651929 + ($2025 + 55) >> 0] = 117; //@line 9669
 HEAP8[651929 + ($2025 + 56) >> 0] = 112; //@line 9672
 HEAP8[651929 + ($2025 + 57) >> 0] = -24; //@line 9675
 HEAP8[651929 + ($2025 + 58) >> 0] = 78; //@line 9678
 $2143 = $2025 + 60 | 0; //@line 9679
 HEAP8[651929 + ($2025 + 59) >> 0] = 117; //@line 9681
 $2145 = $1851 - $2143 | 0; //@line 9682
 HEAP8[651929 + $2143 >> 0] = 2; //@line 9685
 HEAP8[651929 + ($2025 + 61) >> 0] = $2145 >>> 16; //@line 9690
 HEAP8[651929 + ($2025 + 62) >> 0] = $2145 >>> 8; //@line 9695
 HEAP8[651929 + ($2025 + 63) >> 0] = $2145; //@line 9699
 HEAP8[651929 + ($2025 + 64) >> 0] = -1; //@line 9702
 HEAP8[651929 + ($2025 + 65) >> 0] = 0; //@line 9705
 HEAP8[651929 + ($2025 + 66) >> 0] = 0; //@line 9708
 $2165 = $2025 + 68 | 0; //@line 9709
 HEAP8[651929 + ($2025 + 67) >> 0] = 0; //@line 9711
 $2167 = $1760 - $2165 | 0; //@line 9712
 HEAP8[651929 + $2165 >> 0] = 1; //@line 9715
 HEAP8[651929 + ($2025 + 69) >> 0] = $2167 >>> 16; //@line 9720
 HEAP8[651929 + ($2025 + 70) >> 0] = $2167 >>> 8; //@line 9725
 $2179 = $2025 + 72 | 0; //@line 9727
 HEAP8[651929 + ($2025 + 71) >> 0] = $2167; //@line 9729
 $2181 = $1776 - $2179 | 0; //@line 9730
 HEAP8[651929 + $2179 >> 0] = 2; //@line 9733
 HEAP8[651929 + ($2025 + 73) >> 0] = $2181 >>> 16; //@line 9738
 HEAP8[651929 + ($2025 + 74) >> 0] = $2181 >>> 8; //@line 9743
 HEAP8[651929 + ($2025 + 75) >> 0] = $2181; //@line 9747
 HEAP8[651929 + ($2025 + 76) >> 0] = 4; //@line 9750
 HEAP8[651929 + ($2025 + 77) >> 0] = -1; //@line 9753
 HEAP8[651929 + ($2025 + 78) >> 0] = -1; //@line 9756
 HEAP8[651929 + ($2025 + 79) >> 0] = -16; //@line 9759
 HEAP8[651929 + ($2025 + 80) >> 0] = 7; //@line 9762
 HEAP8[651929 + ($2025 + 81) >> 0] = 0; //@line 9765
 HEAP8[651929 + ($2025 + 82) >> 0] = 0; //@line 9768
 HEAP8[651929 + ($2025 + 83) >> 0] = 2; //@line 9771
 HEAP8[651929 + ($2025 + 84) >> 0] = 8; //@line 9774
 HEAP8[651929 + ($2025 + 85) >> 0] = 0; //@line 9777
 HEAP8[651929 + ($2025 + 86) >> 0] = 66; //@line 9780
 HEAP8[651929 + ($2025 + 87) >> 0] = 50; //@line 9783
 HEAP8[651929 + ($2025 + 88) >> 0] = -1; //@line 9786
 HEAP8[651929 + ($2025 + 89) >> 0] = 0; //@line 9789
 HEAP8[651929 + ($2025 + 90) >> 0] = 0; //@line 9792
 $2225 = $2025 + 92 | 0; //@line 9793
 HEAP8[651929 + ($2025 + 91) >> 0] = 0; //@line 9795
 $2227 = $134 - $2225 | 0; //@line 9796
 HEAP8[651929 + $2225 >> 0] = 1; //@line 9799
 HEAP8[651929 + ($2025 + 93) >> 0] = $2227 >>> 16; //@line 9804
 HEAP8[651929 + ($2025 + 94) >> 0] = $2227 >>> 8; //@line 9809
 $2239 = $2025 + 96 | 0; //@line 9811
 HEAP32[28210] = $2239; //@line 9812
 HEAP8[651929 + ($2025 + 95) >> 0] = $2227; //@line 9814
 $2241 = HEAP32[30548] | 0; //@line 9815
 if (($2241 | 0) == ($2 | 0)) {
  $2268 = $2239; //@line 9818
 } else {
  $$pre219 = HEAP32[$0 >> 2] | 0; //@line 9820
  $$0158 = 0; //@line 9821
  $2244 = $2241; //@line 9821
  $2250 = $2239; //@line 9821
  while (1) {
   $2249 = (HEAP32[$$pre219 + ($$0158 << 2) >> 2] | 0) - $2250 | 0; //@line 9828
   HEAP8[651929 + $2250 >> 0] = HEAP8[(HEAP32[$2244 >> 2] | 0) + 24 >> 0] | 0; //@line 9831
   HEAP8[651929 + ($2250 + 1) >> 0] = $2249 >>> 16; //@line 9836
   HEAP8[651929 + ($2250 + 2) >> 0] = $2249 >>> 8; //@line 9841
   $2262 = $2250 + 4 | 0; //@line 9843
   HEAP32[28210] = $2262; //@line 9844
   HEAP8[651929 + ($2250 + 3) >> 0] = $2249; //@line 9846
   $2244 = $2244 + 4 | 0; //@line 9847
   if (($2244 | 0) == ($2 | 0)) {
    $2268 = $2262; //@line 9851
    break;
   } else {
    $$0158 = $$0158 + 1 | 0; //@line 9854
    $2250 = $2262; //@line 9854
   }
  }
 }
 $2267 = $1708 - $2268 | 0; //@line 9858
 HEAP8[651929 + $2268 >> 0] = -16; //@line 9861
 HEAP8[651929 + ($2268 + 1) >> 0] = $2267 >>> 16; //@line 9866
 HEAP8[651929 + ($2268 + 2) >> 0] = $2267 >>> 8; //@line 9871
 $2280 = $2268 + 4 | 0; //@line 9873
 HEAP8[651929 + ($2268 + 3) >> 0] = $2267; //@line 9875
 $2282 = $2165 - $2280 | 0; //@line 9876
 HEAP8[651929 + $2280 >> 0] = -15; //@line 9879
 HEAP8[651929 + ($2268 + 5) >> 0] = $2282 >>> 16; //@line 9884
 HEAP8[651929 + ($2268 + 6) >> 0] = $2282 >>> 8; //@line 9889
 HEAP8[651929 + ($2268 + 7) >> 0] = $2282; //@line 9893
 HEAP8[651929 + ($2268 + 8) >> 0] = -1; //@line 9896
 HEAP8[651929 + ($2268 + 9) >> 0] = 0; //@line 9899
 HEAP8[651929 + ($2268 + 10) >> 0] = 0; //@line 9902
 $2302 = $2268 + 12 | 0; //@line 9903
 HEAP8[651929 + ($2268 + 11) >> 0] = 0; //@line 9905
 $2304 = $2225 - $2302 | 0; //@line 9906
 HEAP8[651929 + $2302 >> 0] = 0; //@line 9909
 HEAP8[651929 + ($2268 + 13) >> 0] = $2304 >>> 16; //@line 9914
 HEAP8[651929 + ($2268 + 14) >> 0] = $2304 >>> 8; //@line 9919
 HEAP8[651929 + ($2268 + 15) >> 0] = $2304; //@line 9923
 $2318 = $2268 + 32 | 0; //@line 9924
 HEAP8[651929 + ($2268 + 16) >> 0] = $2318 >>> 24; //@line 9929
 HEAP8[651929 + ($2268 + 17) >> 0] = $2318 >>> 16; //@line 9934
 HEAP8[651929 + ($2268 + 18) >> 0] = $2318 >>> 8; //@line 9939
 HEAP8[651929 + ($2268 + 19) >> 0] = $2318; //@line 9943
 HEAP8[651929 + ($2268 + 20) >> 0] = 0; //@line 9946
 HEAP8[651929 + ($2268 + 21) >> 0] = 0; //@line 9949
 HEAP8[651929 + ($2268 + 22) >> 0] = 0; //@line 9952
 HEAP8[651929 + ($2268 + 23) >> 0] = 0; //@line 9955
 HEAP8[651929 + ($2268 + 24) >> 0] = 1; //@line 9958
 HEAP8[651929 + ($2268 + 25) >> 0] = 1; //@line 9961
 HEAP8[651929 + ($2268 + 26) >> 0] = 90; //@line 9964
 HEAP8[651929 + ($2268 + 27) >> 0] = -109; //@line 9967
 HEAP8[651929 + ($2268 + 28) >> 0] = 43; //@line 9970
 HEAP8[651929 + ($2268 + 29) >> 0] = -57; //@line 9973
 HEAP8[651929 + ($2268 + 30) >> 0] = 0; //@line 9976
 HEAP32[28210] = $2318; //@line 9977
 HEAP8[651929 + ($2268 + 31) >> 0] = 15; //@line 9979
 HEAP32[28211] = $2318; //@line 9980
 _memcpy((HEAP32[30897] | 0) + (HEAP32[30898] | 0) + (-32 - $2268) | 0, 651929, $2318 | 0) | 0; //@line 9986
 $2366 = (HEAP32[30897] | 0) + (HEAP32[30898] | 0) + (-32 - $2268) | 0; //@line 9991
 $2368 = $2366 + ($2268 + 20) | 0; //@line 9993
 HEAP8[$2368 >> 0] = 0; //@line 9994
 $2370 = $2366 + ($2268 + 21) | 0; //@line 9996
 HEAP8[$2370 >> 0] = 0; //@line 9997
 $2372 = $2366 + ($2268 + 22) | 0; //@line 9999
 HEAP8[$2372 >> 0] = 0; //@line 10000
 $2374 = $2366 + ($2268 + 23) | 0; //@line 10002
 HEAP8[$2374 >> 0] = 0; //@line 10003
 if (($2318 | 0) > 0) {
  $$02324$i = 0; //@line 10006
  $$025$i = 0; //@line 10006
  while (1) {
   $2382 = (HEAPU8[$2366 + $$025$i >> 0] | 0) + ($$02324$i << 1 | $$02324$i >>> 31) | 0; //@line 10014
   $$025$i = $$025$i + 1 | 0; //@line 10015
   if (($$025$i | 0) == ($2318 | 0)) {
    $$023$lcssa$i = $2382; //@line 10018
    break;
   } else {
    $$02324$i = $2382; //@line 10021
   }
  }
 } else {
  $$023$lcssa$i = 0; //@line 10025
 }
 HEAP8[$2368 >> 0] = $$023$lcssa$i >>> 24; //@line 10029
 HEAP8[$2370 >> 0] = $$023$lcssa$i >>> 16; //@line 10032
 HEAP8[$2372 >> 0] = $$023$lcssa$i >>> 8; //@line 10035
 HEAP8[$2374 >> 0] = $$023$lcssa$i; //@line 10037
 $2391 = HEAP32[$0 >> 2] | 0; //@line 10038
 if (!$2391) {
  STACKTOP = sp; //@line 10041
  return 1; //@line 10041
 }
 $2393 = HEAP32[$3 >> 2] | 0; //@line 10043
 if (($2393 | 0) != ($2391 | 0)) {
  HEAP32[$3 >> 2] = $2393 + (~(($2393 + -4 - $2391 | 0) >>> 2) << 2); //@line 10053
 }
 __ZdlPv($2391); //@line 10055
 STACKTOP = sp; //@line 10056
 return 1; //@line 10056
}
function _malloc($0) {
 $0 = $0 | 0;
 var $$$0192$i = 0, $$$0193$i = 0, $$$4351$i = 0, $$$i = 0, $$0 = 0, $$0$i$i = 0, $$0$i$i$i = 0, $$0$i18$i = 0, $$01$i$i = 0, $$0189$i = 0, $$0192$lcssa$i = 0, $$01928$i = 0, $$0193$lcssa$i = 0, $$01937$i = 0, $$0197 = 0, $$0199 = 0, $$0206$i$i = 0, $$0207$i$i = 0, $$0211$i$i = 0, $$0212$i$i = 0, $$024371$i = 0, $$0287$i$i = 0, $$0288$i$i = 0, $$0289$i$i = 0, $$0295$i$i = 0, $$0296$i$i = 0, $$0342$i = 0, $$0344$i = 0, $$0345$i = 0, $$0347$i = 0, $$0353$i = 0, $$0358$i = 0, $$0359$i = 0, $$0361$i = 0, $$0362$i = 0, $$0368$i = 0, $$1196$i = 0, $$1198$i = 0, $$124470$i = 0, $$1291$i$i = 0, $$1293$i$i = 0, $$1343$i = 0, $$1348$i = 0, $$1363$i = 0, $$1370$i = 0, $$1374$i = 0, $$2234253237$i = 0, $$2247$ph$i = 0, $$2253$ph$i = 0, $$2355$i = 0, $$3$i = 0, $$3$i$i = 0, $$3$i201 = 0, $$3350$i = 0, $$3372$i = 0, $$4$lcssa$i = 0, $$4$ph$i = 0, $$415$i = 0, $$4236$i = 0, $$4351$lcssa$i = 0, $$435114$i = 0, $$4357$$4$i = 0, $$4357$ph$i = 0, $$435713$i = 0, $$723948$i = 0, $$749$i = 0, $$pre$phi$i$iZ2D = 0, $$pre$phi$i20$iZ2D = 0, $$pre$phi$i211Z2D = 0, $$pre$phi$iZ2D = 0, $$pre$phi11$i$iZ2D = 0, $$pre$phiZ2D = 0, $1 = 0, $1001 = 0, $1007 = 0, $101 = 0, $1010 = 0, $1011 = 0, $102 = 0, $1029 = 0, $1031 = 0, $1038 = 0, $1039 = 0, $1040 = 0, $1048 = 0, $1050 = 0, $1051 = 0, $1052 = 0, $108 = 0, $112 = 0, $114 = 0, $115 = 0, $117 = 0, $119 = 0, $121 = 0, $123 = 0, $125 = 0, $127 = 0, $129 = 0, $134 = 0, $138 = 0, $14 = 0, $142 = 0, $145 = 0, $148 = 0, $149 = 0, $155 = 0, $157 = 0, $16 = 0, $160 = 0, $162 = 0, $165 = 0, $167 = 0, $17 = 0, $170 = 0, $173 = 0, $174 = 0, $176 = 0, $177 = 0, $179 = 0, $18 = 0, $180 = 0, $182 = 0, $183 = 0, $188 = 0, $189 = 0, $19 = 0, $20 = 0, $201 = 0, $205 = 0, $211 = 0, $218 = 0, $222 = 0, $231 = 0, $232 = 0, $234 = 0, $235 = 0, $239 = 0, $240 = 0, $248 = 0, $249 = 0, $250 = 0, $252 = 0, $253 = 0, $258 = 0, $259 = 0, $262 = 0, $264 = 0, $267 = 0, $27 = 0, $272 = 0, $279 = 0, $289 = 0, $293 = 0, $299 = 0, $30 = 0, $303 = 0, $306 = 0, $310 = 0, $312 = 0, $313 = 0, $315 = 0, $317 = 0, $319 = 0, $321 = 0, $323 = 0, $325 = 0, $327 = 0, $337 = 0, $338 = 0, $34 = 0, $348 = 0, $350 = 0, $353 = 0, $355 = 0, $358 = 0, $360 = 0, $363 = 0, $366 = 0, $367 = 0, $369 = 0, $37 = 0, $370 = 0, $372 = 0, $373 = 0, $375 = 0, $376 = 0, $381 = 0, $382 = 0, $387 = 0, $394 = 0, $398 = 0, $404 = 0, $41 = 0, $411 = 0, $415 = 0, $423 = 0, $426 = 0, $427 = 0, $428 = 0, $432 = 0, $433 = 0, $439 = 0, $44 = 0, $444 = 0, $445 = 0, $448 = 0, $450 = 0, $453 = 0, $458 = 0, $464 = 0, $466 = 0, $468 = 0, $47 = 0, $470 = 0, $487 = 0, $489 = 0, $49 = 0, $496 = 0, $497 = 0, $498 = 0, $50 = 0, $506 = 0, $508 = 0, $509 = 0, $511 = 0, $52 = 0, $520 = 0, $524 = 0, $526 = 0, $527 = 0, $528 = 0, $538 = 0, $539 = 0, $54 = 0, $540 = 0, $541 = 0, $542 = 0, $543 = 0, $544 = 0, $546 = 0, $548 = 0, $549 = 0, $555 = 0, $557 = 0, $559 = 0, $56 = 0, $564 = 0, $566 = 0, $568 = 0, $569 = 0, $570 = 0, $578 = 0, $579 = 0, $58 = 0, $582 = 0, $586 = 0, $589 = 0, $591 = 0, $597 = 0, $6 = 0, $60 = 0, $601 = 0, $605 = 0, $614 = 0, $615 = 0, $62 = 0, $621 = 0, $623 = 0, $627 = 0, $630 = 0, $632 = 0, $637 = 0, $64 = 0, $643 = 0, $648 = 0, $649 = 0, $650 = 0, $656 = 0, $657 = 0, $658 = 0, $662 = 0, $67 = 0, $673 = 0, $678 = 0, $679 = 0, $681 = 0, $687 = 0, $689 = 0, $69 = 0, $693 = 0, $699 = 0, $7 = 0, $70 = 0, $703 = 0, $709 = 0, $71 = 0, $711 = 0, $717 = 0, $72 = 0, $721 = 0, $722 = 0, $727 = 0, $73 = 0, $733 = 0, $738 = 0, $741 = 0, $742 = 0, $745 = 0, $747 = 0, $749 = 0, $752 = 0, $763 = 0, $768 = 0, $77 = 0, $770 = 0, $773 = 0, $775 = 0, $778 = 0, $781 = 0, $782 = 0, $783 = 0, $785 = 0, $787 = 0, $788 = 0, $790 = 0, $791 = 0, $796 = 0, $797 = 0, $8 = 0, $80 = 0, $810 = 0, $813 = 0, $814 = 0, $820 = 0, $828 = 0, $834 = 0, $837 = 0, $838 = 0, $839 = 0, $84 = 0, $843 = 0, $844 = 0, $850 = 0, $855 = 0, $856 = 0, $859 = 0, $861 = 0, $864 = 0, $869 = 0, $87 = 0, $875 = 0, $877 = 0, $879 = 0, $880 = 0, $898 = 0, $9 = 0, $900 = 0, $907 = 0, $908 = 0, $909 = 0, $916 = 0, $92 = 0, $920 = 0, $924 = 0, $926 = 0, $93 = 0, $932 = 0, $933 = 0, $935 = 0, $936 = 0, $937 = 0, $940 = 0, $945 = 0, $946 = 0, $947 = 0, $95 = 0, $953 = 0, $955 = 0, $96 = 0, $961 = 0, $966 = 0, $969 = 0, $970 = 0, $971 = 0, $975 = 0, $976 = 0, $98 = 0, $982 = 0, $987 = 0, $988 = 0, $991 = 0, $993 = 0, $996 = 0, label = 0, sp = 0, $955$looptemp = 0;
 sp = STACKTOP; //@line 50262
 STACKTOP = STACKTOP + 16 | 0; //@line 50263
 $1 = sp; //@line 50264
 do {
  if ($0 >>> 0 < 245) {
   $6 = $0 >>> 0 < 11 ? 16 : $0 + 11 & -8; //@line 50271
   $7 = $6 >>> 3; //@line 50272
   $8 = HEAP32[162835] | 0; //@line 50273
   $9 = $8 >>> $7; //@line 50274
   if ($9 & 3 | 0) {
    $14 = ($9 & 1 ^ 1) + $7 | 0; //@line 50280
    $16 = 651380 + ($14 << 1 << 2) | 0; //@line 50282
    $17 = $16 + 8 | 0; //@line 50283
    $18 = HEAP32[$17 >> 2] | 0; //@line 50284
    $19 = $18 + 8 | 0; //@line 50285
    $20 = HEAP32[$19 >> 2] | 0; //@line 50286
    do {
     if (($16 | 0) == ($20 | 0)) {
      HEAP32[162835] = $8 & ~(1 << $14); //@line 50293
     } else {
      if ($20 >>> 0 < (HEAP32[162839] | 0) >>> 0) {
       _abort(); //@line 50298
      }
      $27 = $20 + 12 | 0; //@line 50301
      if ((HEAP32[$27 >> 2] | 0) == ($18 | 0)) {
       HEAP32[$27 >> 2] = $16; //@line 50305
       HEAP32[$17 >> 2] = $20; //@line 50306
       break;
      } else {
       _abort(); //@line 50309
      }
     }
    } while (0);
    $30 = $14 << 3; //@line 50314
    HEAP32[$18 + 4 >> 2] = $30 | 3; //@line 50317
    $34 = $18 + $30 + 4 | 0; //@line 50319
    HEAP32[$34 >> 2] = HEAP32[$34 >> 2] | 1; //@line 50322
    $$0 = $19; //@line 50323
    STACKTOP = sp; //@line 50324
    return $$0 | 0; //@line 50324
   }
   $37 = HEAP32[162837] | 0; //@line 50326
   if ($6 >>> 0 > $37 >>> 0) {
    if ($9 | 0) {
     $41 = 2 << $7; //@line 50332
     $44 = $9 << $7 & ($41 | 0 - $41); //@line 50335
     $47 = ($44 & 0 - $44) + -1 | 0; //@line 50338
     $49 = $47 >>> 12 & 16; //@line 50340
     $50 = $47 >>> $49; //@line 50341
     $52 = $50 >>> 5 & 8; //@line 50343
     $54 = $50 >>> $52; //@line 50345
     $56 = $54 >>> 2 & 4; //@line 50347
     $58 = $54 >>> $56; //@line 50349
     $60 = $58 >>> 1 & 2; //@line 50351
     $62 = $58 >>> $60; //@line 50353
     $64 = $62 >>> 1 & 1; //@line 50355
     $67 = ($52 | $49 | $56 | $60 | $64) + ($62 >>> $64) | 0; //@line 50358
     $69 = 651380 + ($67 << 1 << 2) | 0; //@line 50360
     $70 = $69 + 8 | 0; //@line 50361
     $71 = HEAP32[$70 >> 2] | 0; //@line 50362
     $72 = $71 + 8 | 0; //@line 50363
     $73 = HEAP32[$72 >> 2] | 0; //@line 50364
     do {
      if (($69 | 0) == ($73 | 0)) {
       $77 = $8 & ~(1 << $67); //@line 50370
       HEAP32[162835] = $77; //@line 50371
       $98 = $77; //@line 50372
      } else {
       if ($73 >>> 0 < (HEAP32[162839] | 0) >>> 0) {
        _abort(); //@line 50377
       }
       $80 = $73 + 12 | 0; //@line 50380
       if ((HEAP32[$80 >> 2] | 0) == ($71 | 0)) {
        HEAP32[$80 >> 2] = $69; //@line 50384
        HEAP32[$70 >> 2] = $73; //@line 50385
        $98 = $8; //@line 50386
        break;
       } else {
        _abort(); //@line 50389
       }
      }
     } while (0);
     $84 = ($67 << 3) - $6 | 0; //@line 50395
     HEAP32[$71 + 4 >> 2] = $6 | 3; //@line 50398
     $87 = $71 + $6 | 0; //@line 50399
     HEAP32[$87 + 4 >> 2] = $84 | 1; //@line 50402
     HEAP32[$87 + $84 >> 2] = $84; //@line 50404
     if ($37 | 0) {
      $92 = HEAP32[162840] | 0; //@line 50407
      $93 = $37 >>> 3; //@line 50408
      $95 = 651380 + ($93 << 1 << 2) | 0; //@line 50410
      $96 = 1 << $93; //@line 50411
      if (!($98 & $96)) {
       HEAP32[162835] = $98 | $96; //@line 50416
       $$0199 = $95; //@line 50418
       $$pre$phiZ2D = $95 + 8 | 0; //@line 50418
      } else {
       $101 = $95 + 8 | 0; //@line 50420
       $102 = HEAP32[$101 >> 2] | 0; //@line 50421
       if ($102 >>> 0 < (HEAP32[162839] | 0) >>> 0) {
        _abort(); //@line 50425
       } else {
        $$0199 = $102; //@line 50428
        $$pre$phiZ2D = $101; //@line 50428
       }
      }
      HEAP32[$$pre$phiZ2D >> 2] = $92; //@line 50431
      HEAP32[$$0199 + 12 >> 2] = $92; //@line 50433
      HEAP32[$92 + 8 >> 2] = $$0199; //@line 50435
      HEAP32[$92 + 12 >> 2] = $95; //@line 50437
     }
     HEAP32[162837] = $84; //@line 50439
     HEAP32[162840] = $87; //@line 50440
     $$0 = $72; //@line 50441
     STACKTOP = sp; //@line 50442
     return $$0 | 0; //@line 50442
    }
    $108 = HEAP32[162836] | 0; //@line 50444
    if (!$108) {
     $$0197 = $6; //@line 50447
    } else {
     $112 = ($108 & 0 - $108) + -1 | 0; //@line 50451
     $114 = $112 >>> 12 & 16; //@line 50453
     $115 = $112 >>> $114; //@line 50454
     $117 = $115 >>> 5 & 8; //@line 50456
     $119 = $115 >>> $117; //@line 50458
     $121 = $119 >>> 2 & 4; //@line 50460
     $123 = $119 >>> $121; //@line 50462
     $125 = $123 >>> 1 & 2; //@line 50464
     $127 = $123 >>> $125; //@line 50466
     $129 = $127 >>> 1 & 1; //@line 50468
     $134 = HEAP32[651644 + (($117 | $114 | $121 | $125 | $129) + ($127 >>> $129) << 2) >> 2] | 0; //@line 50473
     $138 = (HEAP32[$134 + 4 >> 2] & -8) - $6 | 0; //@line 50477
     $142 = HEAP32[$134 + 16 + (((HEAP32[$134 + 16 >> 2] | 0) == 0 & 1) << 2) >> 2] | 0; //@line 50483
     if (!$142) {
      $$0192$lcssa$i = $134; //@line 50486
      $$0193$lcssa$i = $138; //@line 50486
     } else {
      $$01928$i = $134; //@line 50488
      $$01937$i = $138; //@line 50488
      $145 = $142; //@line 50488
      while (1) {
       $148 = (HEAP32[$145 + 4 >> 2] & -8) - $6 | 0; //@line 50493
       $149 = $148 >>> 0 < $$01937$i >>> 0; //@line 50494
       $$$0193$i = $149 ? $148 : $$01937$i; //@line 50495
       $$$0192$i = $149 ? $145 : $$01928$i; //@line 50496
       $145 = HEAP32[$145 + 16 + (((HEAP32[$145 + 16 >> 2] | 0) == 0 & 1) << 2) >> 2] | 0; //@line 50502
       if (!$145) {
        $$0192$lcssa$i = $$$0192$i; //@line 50505
        $$0193$lcssa$i = $$$0193$i; //@line 50505
        break;
       } else {
        $$01928$i = $$$0192$i; //@line 50508
        $$01937$i = $$$0193$i; //@line 50508
       }
      }
     }
     $155 = HEAP32[162839] | 0; //@line 50512
     if ($$0192$lcssa$i >>> 0 < $155 >>> 0) {
      _abort(); //@line 50515
     }
     $157 = $$0192$lcssa$i + $6 | 0; //@line 50518
     if ($$0192$lcssa$i >>> 0 >= $157 >>> 0) {
      _abort(); //@line 50521
     }
     $160 = HEAP32[$$0192$lcssa$i + 24 >> 2] | 0; //@line 50525
     $162 = HEAP32[$$0192$lcssa$i + 12 >> 2] | 0; //@line 50527
     do {
      if (($162 | 0) == ($$0192$lcssa$i | 0)) {
       $173 = $$0192$lcssa$i + 20 | 0; //@line 50531
       $174 = HEAP32[$173 >> 2] | 0; //@line 50532
       if (!$174) {
        $176 = $$0192$lcssa$i + 16 | 0; //@line 50535
        $177 = HEAP32[$176 >> 2] | 0; //@line 50536
        if (!$177) {
         $$3$i = 0; //@line 50539
         break;
        } else {
         $$1196$i = $177; //@line 50542
         $$1198$i = $176; //@line 50542
        }
       } else {
        $$1196$i = $174; //@line 50545
        $$1198$i = $173; //@line 50545
       }
       while (1) {
        $179 = $$1196$i + 20 | 0; //@line 50548
        $180 = HEAP32[$179 >> 2] | 0; //@line 50549
        if ($180 | 0) {
         $$1196$i = $180; //@line 50552
         $$1198$i = $179; //@line 50552
         continue;
        }
        $182 = $$1196$i + 16 | 0; //@line 50555
        $183 = HEAP32[$182 >> 2] | 0; //@line 50556
        if (!$183) {
         break;
        } else {
         $$1196$i = $183; //@line 50561
         $$1198$i = $182; //@line 50561
        }
       }
       if ($$1198$i >>> 0 < $155 >>> 0) {
        _abort(); //@line 50566
       } else {
        HEAP32[$$1198$i >> 2] = 0; //@line 50569
        $$3$i = $$1196$i; //@line 50570
        break;
       }
      } else {
       $165 = HEAP32[$$0192$lcssa$i + 8 >> 2] | 0; //@line 50575
       if ($165 >>> 0 < $155 >>> 0) {
        _abort(); //@line 50578
       }
       $167 = $165 + 12 | 0; //@line 50581
       if ((HEAP32[$167 >> 2] | 0) != ($$0192$lcssa$i | 0)) {
        _abort(); //@line 50585
       }
       $170 = $162 + 8 | 0; //@line 50588
       if ((HEAP32[$170 >> 2] | 0) == ($$0192$lcssa$i | 0)) {
        HEAP32[$167 >> 2] = $162; //@line 50592
        HEAP32[$170 >> 2] = $165; //@line 50593
        $$3$i = $162; //@line 50594
        break;
       } else {
        _abort(); //@line 50597
       }
      }
     } while (0);
     L73 : do {
      if ($160 | 0) {
       $188 = HEAP32[$$0192$lcssa$i + 28 >> 2] | 0; //@line 50606
       $189 = 651644 + ($188 << 2) | 0; //@line 50607
       do {
        if (($$0192$lcssa$i | 0) == (HEAP32[$189 >> 2] | 0)) {
         HEAP32[$189 >> 2] = $$3$i; //@line 50612
         if (!$$3$i) {
          HEAP32[162836] = $108 & ~(1 << $188); //@line 50618
          break L73;
         }
        } else {
         if ($160 >>> 0 < (HEAP32[162839] | 0) >>> 0) {
          _abort(); //@line 50625
         } else {
          HEAP32[$160 + 16 + (((HEAP32[$160 + 16 >> 2] | 0) != ($$0192$lcssa$i | 0) & 1) << 2) >> 2] = $$3$i; //@line 50633
          if (!$$3$i) {
           break L73;
          } else {
           break;
          }
         }
        }
       } while (0);
       $201 = HEAP32[162839] | 0; //@line 50643
       if ($$3$i >>> 0 < $201 >>> 0) {
        _abort(); //@line 50646
       }
       HEAP32[$$3$i + 24 >> 2] = $160; //@line 50650
       $205 = HEAP32[$$0192$lcssa$i + 16 >> 2] | 0; //@line 50652
       do {
        if ($205 | 0) {
         if ($205 >>> 0 < $201 >>> 0) {
          _abort(); //@line 50658
         } else {
          HEAP32[$$3$i + 16 >> 2] = $205; //@line 50662
          HEAP32[$205 + 24 >> 2] = $$3$i; //@line 50664
          break;
         }
        }
       } while (0);
       $211 = HEAP32[$$0192$lcssa$i + 20 >> 2] | 0; //@line 50670
       if ($211 | 0) {
        if ($211 >>> 0 < (HEAP32[162839] | 0) >>> 0) {
         _abort(); //@line 50676
        } else {
         HEAP32[$$3$i + 20 >> 2] = $211; //@line 50680
         HEAP32[$211 + 24 >> 2] = $$3$i; //@line 50682
         break;
        }
       }
      }
     } while (0);
     if ($$0193$lcssa$i >>> 0 < 16) {
      $218 = $$0193$lcssa$i + $6 | 0; //@line 50690
      HEAP32[$$0192$lcssa$i + 4 >> 2] = $218 | 3; //@line 50693
      $222 = $$0192$lcssa$i + $218 + 4 | 0; //@line 50695
      HEAP32[$222 >> 2] = HEAP32[$222 >> 2] | 1; //@line 50698
     } else {
      HEAP32[$$0192$lcssa$i + 4 >> 2] = $6 | 3; //@line 50702
      HEAP32[$157 + 4 >> 2] = $$0193$lcssa$i | 1; //@line 50705
      HEAP32[$157 + $$0193$lcssa$i >> 2] = $$0193$lcssa$i; //@line 50707
      if ($37 | 0) {
       $231 = HEAP32[162840] | 0; //@line 50710
       $232 = $37 >>> 3; //@line 50711
       $234 = 651380 + ($232 << 1 << 2) | 0; //@line 50713
       $235 = 1 << $232; //@line 50714
       if (!($8 & $235)) {
        HEAP32[162835] = $8 | $235; //@line 50719
        $$0189$i = $234; //@line 50721
        $$pre$phi$iZ2D = $234 + 8 | 0; //@line 50721
       } else {
        $239 = $234 + 8 | 0; //@line 50723
        $240 = HEAP32[$239 >> 2] | 0; //@line 50724
        if ($240 >>> 0 < (HEAP32[162839] | 0) >>> 0) {
         _abort(); //@line 50728
        } else {
         $$0189$i = $240; //@line 50731
         $$pre$phi$iZ2D = $239; //@line 50731
        }
       }
       HEAP32[$$pre$phi$iZ2D >> 2] = $231; //@line 50734
       HEAP32[$$0189$i + 12 >> 2] = $231; //@line 50736
       HEAP32[$231 + 8 >> 2] = $$0189$i; //@line 50738
       HEAP32[$231 + 12 >> 2] = $234; //@line 50740
      }
      HEAP32[162837] = $$0193$lcssa$i; //@line 50742
      HEAP32[162840] = $157; //@line 50743
     }
     $$0 = $$0192$lcssa$i + 8 | 0; //@line 50746
     STACKTOP = sp; //@line 50747
     return $$0 | 0; //@line 50747
    }
   } else {
    $$0197 = $6; //@line 50750
   }
  } else {
   if ($0 >>> 0 > 4294967231) {
    $$0197 = -1; //@line 50755
   } else {
    $248 = $0 + 11 | 0; //@line 50757
    $249 = $248 & -8; //@line 50758
    $250 = HEAP32[162836] | 0; //@line 50759
    if (!$250) {
     $$0197 = $249; //@line 50762
    } else {
     $252 = 0 - $249 | 0; //@line 50764
     $253 = $248 >>> 8; //@line 50765
     if (!$253) {
      $$0358$i = 0; //@line 50768
     } else {
      if ($249 >>> 0 > 16777215) {
       $$0358$i = 31; //@line 50772
      } else {
       $258 = ($253 + 1048320 | 0) >>> 16 & 8; //@line 50776
       $259 = $253 << $258; //@line 50777
       $262 = ($259 + 520192 | 0) >>> 16 & 4; //@line 50780
       $264 = $259 << $262; //@line 50782
       $267 = ($264 + 245760 | 0) >>> 16 & 2; //@line 50785
       $272 = 14 - ($262 | $258 | $267) + ($264 << $267 >>> 15) | 0; //@line 50790
       $$0358$i = $249 >>> ($272 + 7 | 0) & 1 | $272 << 1; //@line 50796
      }
     }
     $279 = HEAP32[651644 + ($$0358$i << 2) >> 2] | 0; //@line 50800
     L117 : do {
      if (!$279) {
       $$2355$i = 0; //@line 50804
       $$3$i201 = 0; //@line 50804
       $$3350$i = $252; //@line 50804
       label = 81; //@line 50805
      } else {
       $$0342$i = 0; //@line 50812
       $$0347$i = $252; //@line 50812
       $$0353$i = $279; //@line 50812
       $$0359$i = $249 << (($$0358$i | 0) == 31 ? 0 : 25 - ($$0358$i >>> 1) | 0); //@line 50812
       $$0362$i = 0; //@line 50812
       while (1) {
        $289 = (HEAP32[$$0353$i + 4 >> 2] & -8) - $249 | 0; //@line 50817
        if ($289 >>> 0 < $$0347$i >>> 0) {
         if (!$289) {
          $$415$i = $$0353$i; //@line 50822
          $$435114$i = 0; //@line 50822
          $$435713$i = $$0353$i; //@line 50822
          label = 85; //@line 50823
          break L117;
         } else {
          $$1343$i = $$0353$i; //@line 50826
          $$1348$i = $289; //@line 50826
         }
        } else {
         $$1343$i = $$0342$i; //@line 50829
         $$1348$i = $$0347$i; //@line 50829
        }
        $293 = HEAP32[$$0353$i + 20 >> 2] | 0; //@line 50832
        $$0353$i = HEAP32[$$0353$i + 16 + ($$0359$i >>> 31 << 2) >> 2] | 0; //@line 50835
        $$1363$i = ($293 | 0) == 0 | ($293 | 0) == ($$0353$i | 0) ? $$0362$i : $293; //@line 50839
        $299 = ($$0353$i | 0) == 0; //@line 50840
        if ($299) {
         $$2355$i = $$1363$i; //@line 50845
         $$3$i201 = $$1343$i; //@line 50845
         $$3350$i = $$1348$i; //@line 50845
         label = 81; //@line 50846
         break;
        } else {
         $$0342$i = $$1343$i; //@line 50849
         $$0347$i = $$1348$i; //@line 50849
         $$0359$i = $$0359$i << (($299 ^ 1) & 1); //@line 50849
         $$0362$i = $$1363$i; //@line 50849
        }
       }
      }
     } while (0);
     if ((label | 0) == 81) {
      if (($$2355$i | 0) == 0 & ($$3$i201 | 0) == 0) {
       $303 = 2 << $$0358$i; //@line 50859
       $306 = $250 & ($303 | 0 - $303); //@line 50862
       if (!$306) {
        $$0197 = $249; //@line 50865
        break;
       }
       $310 = ($306 & 0 - $306) + -1 | 0; //@line 50870
       $312 = $310 >>> 12 & 16; //@line 50872
       $313 = $310 >>> $312; //@line 50873
       $315 = $313 >>> 5 & 8; //@line 50875
       $317 = $313 >>> $315; //@line 50877
       $319 = $317 >>> 2 & 4; //@line 50879
       $321 = $317 >>> $319; //@line 50881
       $323 = $321 >>> 1 & 2; //@line 50883
       $325 = $321 >>> $323; //@line 50885
       $327 = $325 >>> 1 & 1; //@line 50887
       $$4$ph$i = 0; //@line 50893
       $$4357$ph$i = HEAP32[651644 + (($315 | $312 | $319 | $323 | $327) + ($325 >>> $327) << 2) >> 2] | 0; //@line 50893
      } else {
       $$4$ph$i = $$3$i201; //@line 50895
       $$4357$ph$i = $$2355$i; //@line 50895
      }
      if (!$$4357$ph$i) {
       $$4$lcssa$i = $$4$ph$i; //@line 50899
       $$4351$lcssa$i = $$3350$i; //@line 50899
      } else {
       $$415$i = $$4$ph$i; //@line 50901
       $$435114$i = $$3350$i; //@line 50901
       $$435713$i = $$4357$ph$i; //@line 50901
       label = 85; //@line 50902
      }
     }
     if ((label | 0) == 85) {
      while (1) {
       label = 0; //@line 50907
       $337 = (HEAP32[$$435713$i + 4 >> 2] & -8) - $249 | 0; //@line 50911
       $338 = $337 >>> 0 < $$435114$i >>> 0; //@line 50912
       $$$4351$i = $338 ? $337 : $$435114$i; //@line 50913
       $$4357$$4$i = $338 ? $$435713$i : $$415$i; //@line 50914
       $$435713$i = HEAP32[$$435713$i + 16 + (((HEAP32[$$435713$i + 16 >> 2] | 0) == 0 & 1) << 2) >> 2] | 0; //@line 50920
       if (!$$435713$i) {
        $$4$lcssa$i = $$4357$$4$i; //@line 50923
        $$4351$lcssa$i = $$$4351$i; //@line 50923
        break;
       } else {
        $$415$i = $$4357$$4$i; //@line 50926
        $$435114$i = $$$4351$i; //@line 50926
        label = 85; //@line 50927
       }
      }
     }
     if (!$$4$lcssa$i) {
      $$0197 = $249; //@line 50933
     } else {
      if ($$4351$lcssa$i >>> 0 < ((HEAP32[162837] | 0) - $249 | 0) >>> 0) {
       $348 = HEAP32[162839] | 0; //@line 50939
       if ($$4$lcssa$i >>> 0 < $348 >>> 0) {
        _abort(); //@line 50942
       }
       $350 = $$4$lcssa$i + $249 | 0; //@line 50945
       if ($$4$lcssa$i >>> 0 >= $350 >>> 0) {
        _abort(); //@line 50948
       }
       $353 = HEAP32[$$4$lcssa$i + 24 >> 2] | 0; //@line 50952
       $355 = HEAP32[$$4$lcssa$i + 12 >> 2] | 0; //@line 50954
       do {
        if (($355 | 0) == ($$4$lcssa$i | 0)) {
         $366 = $$4$lcssa$i + 20 | 0; //@line 50958
         $367 = HEAP32[$366 >> 2] | 0; //@line 50959
         if (!$367) {
          $369 = $$4$lcssa$i + 16 | 0; //@line 50962
          $370 = HEAP32[$369 >> 2] | 0; //@line 50963
          if (!$370) {
           $$3372$i = 0; //@line 50966
           break;
          } else {
           $$1370$i = $370; //@line 50969
           $$1374$i = $369; //@line 50969
          }
         } else {
          $$1370$i = $367; //@line 50972
          $$1374$i = $366; //@line 50972
         }
         while (1) {
          $372 = $$1370$i + 20 | 0; //@line 50975
          $373 = HEAP32[$372 >> 2] | 0; //@line 50976
          if ($373 | 0) {
           $$1370$i = $373; //@line 50979
           $$1374$i = $372; //@line 50979
           continue;
          }
          $375 = $$1370$i + 16 | 0; //@line 50982
          $376 = HEAP32[$375 >> 2] | 0; //@line 50983
          if (!$376) {
           break;
          } else {
           $$1370$i = $376; //@line 50988
           $$1374$i = $375; //@line 50988
          }
         }
         if ($$1374$i >>> 0 < $348 >>> 0) {
          _abort(); //@line 50993
         } else {
          HEAP32[$$1374$i >> 2] = 0; //@line 50996
          $$3372$i = $$1370$i; //@line 50997
          break;
         }
        } else {
         $358 = HEAP32[$$4$lcssa$i + 8 >> 2] | 0; //@line 51002
         if ($358 >>> 0 < $348 >>> 0) {
          _abort(); //@line 51005
         }
         $360 = $358 + 12 | 0; //@line 51008
         if ((HEAP32[$360 >> 2] | 0) != ($$4$lcssa$i | 0)) {
          _abort(); //@line 51012
         }
         $363 = $355 + 8 | 0; //@line 51015
         if ((HEAP32[$363 >> 2] | 0) == ($$4$lcssa$i | 0)) {
          HEAP32[$360 >> 2] = $355; //@line 51019
          HEAP32[$363 >> 2] = $358; //@line 51020
          $$3372$i = $355; //@line 51021
          break;
         } else {
          _abort(); //@line 51024
         }
        }
       } while (0);
       L164 : do {
        if (!$353) {
         $470 = $250; //@line 51032
        } else {
         $381 = HEAP32[$$4$lcssa$i + 28 >> 2] | 0; //@line 51035
         $382 = 651644 + ($381 << 2) | 0; //@line 51036
         do {
          if (($$4$lcssa$i | 0) == (HEAP32[$382 >> 2] | 0)) {
           HEAP32[$382 >> 2] = $$3372$i; //@line 51041
           if (!$$3372$i) {
            $387 = $250 & ~(1 << $381); //@line 51046
            HEAP32[162836] = $387; //@line 51047
            $470 = $387; //@line 51048
            break L164;
           }
          } else {
           if ($353 >>> 0 < (HEAP32[162839] | 0) >>> 0) {
            _abort(); //@line 51055
           } else {
            HEAP32[$353 + 16 + (((HEAP32[$353 + 16 >> 2] | 0) != ($$4$lcssa$i | 0) & 1) << 2) >> 2] = $$3372$i; //@line 51063
            if (!$$3372$i) {
             $470 = $250; //@line 51066
             break L164;
            } else {
             break;
            }
           }
          }
         } while (0);
         $394 = HEAP32[162839] | 0; //@line 51074
         if ($$3372$i >>> 0 < $394 >>> 0) {
          _abort(); //@line 51077
         }
         HEAP32[$$3372$i + 24 >> 2] = $353; //@line 51081
         $398 = HEAP32[$$4$lcssa$i + 16 >> 2] | 0; //@line 51083
         do {
          if ($398 | 0) {
           if ($398 >>> 0 < $394 >>> 0) {
            _abort(); //@line 51089
           } else {
            HEAP32[$$3372$i + 16 >> 2] = $398; //@line 51093
            HEAP32[$398 + 24 >> 2] = $$3372$i; //@line 51095
            break;
           }
          }
         } while (0);
         $404 = HEAP32[$$4$lcssa$i + 20 >> 2] | 0; //@line 51101
         if (!$404) {
          $470 = $250; //@line 51104
         } else {
          if ($404 >>> 0 < (HEAP32[162839] | 0) >>> 0) {
           _abort(); //@line 51109
          } else {
           HEAP32[$$3372$i + 20 >> 2] = $404; //@line 51113
           HEAP32[$404 + 24 >> 2] = $$3372$i; //@line 51115
           $470 = $250; //@line 51116
           break;
          }
         }
        }
       } while (0);
       do {
        if ($$4351$lcssa$i >>> 0 < 16) {
         $411 = $$4351$lcssa$i + $249 | 0; //@line 51125
         HEAP32[$$4$lcssa$i + 4 >> 2] = $411 | 3; //@line 51128
         $415 = $$4$lcssa$i + $411 + 4 | 0; //@line 51130
         HEAP32[$415 >> 2] = HEAP32[$415 >> 2] | 1; //@line 51133
        } else {
         HEAP32[$$4$lcssa$i + 4 >> 2] = $249 | 3; //@line 51137
         HEAP32[$350 + 4 >> 2] = $$4351$lcssa$i | 1; //@line 51140
         HEAP32[$350 + $$4351$lcssa$i >> 2] = $$4351$lcssa$i; //@line 51142
         $423 = $$4351$lcssa$i >>> 3; //@line 51143
         if ($$4351$lcssa$i >>> 0 < 256) {
          $426 = 651380 + ($423 << 1 << 2) | 0; //@line 51147
          $427 = HEAP32[162835] | 0; //@line 51148
          $428 = 1 << $423; //@line 51149
          if (!($427 & $428)) {
           HEAP32[162835] = $427 | $428; //@line 51154
           $$0368$i = $426; //@line 51156
           $$pre$phi$i211Z2D = $426 + 8 | 0; //@line 51156
          } else {
           $432 = $426 + 8 | 0; //@line 51158
           $433 = HEAP32[$432 >> 2] | 0; //@line 51159
           if ($433 >>> 0 < (HEAP32[162839] | 0) >>> 0) {
            _abort(); //@line 51163
           } else {
            $$0368$i = $433; //@line 51166
            $$pre$phi$i211Z2D = $432; //@line 51166
           }
          }
          HEAP32[$$pre$phi$i211Z2D >> 2] = $350; //@line 51169
          HEAP32[$$0368$i + 12 >> 2] = $350; //@line 51171
          HEAP32[$350 + 8 >> 2] = $$0368$i; //@line 51173
          HEAP32[$350 + 12 >> 2] = $426; //@line 51175
          break;
         }
         $439 = $$4351$lcssa$i >>> 8; //@line 51178
         if (!$439) {
          $$0361$i = 0; //@line 51181
         } else {
          if ($$4351$lcssa$i >>> 0 > 16777215) {
           $$0361$i = 31; //@line 51185
          } else {
           $444 = ($439 + 1048320 | 0) >>> 16 & 8; //@line 51189
           $445 = $439 << $444; //@line 51190
           $448 = ($445 + 520192 | 0) >>> 16 & 4; //@line 51193
           $450 = $445 << $448; //@line 51195
           $453 = ($450 + 245760 | 0) >>> 16 & 2; //@line 51198
           $458 = 14 - ($448 | $444 | $453) + ($450 << $453 >>> 15) | 0; //@line 51203
           $$0361$i = $$4351$lcssa$i >>> ($458 + 7 | 0) & 1 | $458 << 1; //@line 51209
          }
         }
         $464 = 651644 + ($$0361$i << 2) | 0; //@line 51212
         HEAP32[$350 + 28 >> 2] = $$0361$i; //@line 51214
         $466 = $350 + 16 | 0; //@line 51215
         HEAP32[$466 + 4 >> 2] = 0; //@line 51217
         HEAP32[$466 >> 2] = 0; //@line 51218
         $468 = 1 << $$0361$i; //@line 51219
         if (!($470 & $468)) {
          HEAP32[162836] = $470 | $468; //@line 51224
          HEAP32[$464 >> 2] = $350; //@line 51225
          HEAP32[$350 + 24 >> 2] = $464; //@line 51227
          HEAP32[$350 + 12 >> 2] = $350; //@line 51229
          HEAP32[$350 + 8 >> 2] = $350; //@line 51231
          break;
         }
         $$0344$i = $$4351$lcssa$i << (($$0361$i | 0) == 31 ? 0 : 25 - ($$0361$i >>> 1) | 0); //@line 51240
         $$0345$i = HEAP32[$464 >> 2] | 0; //@line 51240
         while (1) {
          if ((HEAP32[$$0345$i + 4 >> 2] & -8 | 0) == ($$4351$lcssa$i | 0)) {
           label = 139; //@line 51247
           break;
          }
          $487 = $$0345$i + 16 + ($$0344$i >>> 31 << 2) | 0; //@line 51251
          $489 = HEAP32[$487 >> 2] | 0; //@line 51253
          if (!$489) {
           label = 136; //@line 51256
           break;
          } else {
           $$0344$i = $$0344$i << 1; //@line 51259
           $$0345$i = $489; //@line 51259
          }
         }
         if ((label | 0) == 136) {
          if ($487 >>> 0 < (HEAP32[162839] | 0) >>> 0) {
           _abort(); //@line 51266
          } else {
           HEAP32[$487 >> 2] = $350; //@line 51269
           HEAP32[$350 + 24 >> 2] = $$0345$i; //@line 51271
           HEAP32[$350 + 12 >> 2] = $350; //@line 51273
           HEAP32[$350 + 8 >> 2] = $350; //@line 51275
           break;
          }
         } else if ((label | 0) == 139) {
          $496 = $$0345$i + 8 | 0; //@line 51280
          $497 = HEAP32[$496 >> 2] | 0; //@line 51281
          $498 = HEAP32[162839] | 0; //@line 51282
          if ($497 >>> 0 >= $498 >>> 0 & $$0345$i >>> 0 >= $498 >>> 0) {
           HEAP32[$497 + 12 >> 2] = $350; //@line 51288
           HEAP32[$496 >> 2] = $350; //@line 51289
           HEAP32[$350 + 8 >> 2] = $497; //@line 51291
           HEAP32[$350 + 12 >> 2] = $$0345$i; //@line 51293
           HEAP32[$350 + 24 >> 2] = 0; //@line 51295
           break;
          } else {
           _abort(); //@line 51298
          }
         }
        }
       } while (0);
       $$0 = $$4$lcssa$i + 8 | 0; //@line 51305
       STACKTOP = sp; //@line 51306
       return $$0 | 0; //@line 51306
      } else {
       $$0197 = $249; //@line 51308
      }
     }
    }
   }
  }
 } while (0);
 $506 = HEAP32[162837] | 0; //@line 51315
 if ($506 >>> 0 >= $$0197 >>> 0) {
  $508 = $506 - $$0197 | 0; //@line 51318
  $509 = HEAP32[162840] | 0; //@line 51319
  if ($508 >>> 0 > 15) {
   $511 = $509 + $$0197 | 0; //@line 51322
   HEAP32[162840] = $511; //@line 51323
   HEAP32[162837] = $508; //@line 51324
   HEAP32[$511 + 4 >> 2] = $508 | 1; //@line 51327
   HEAP32[$511 + $508 >> 2] = $508; //@line 51329
   HEAP32[$509 + 4 >> 2] = $$0197 | 3; //@line 51332
  } else {
   HEAP32[162837] = 0; //@line 51334
   HEAP32[162840] = 0; //@line 51335
   HEAP32[$509 + 4 >> 2] = $506 | 3; //@line 51338
   $520 = $509 + $506 + 4 | 0; //@line 51340
   HEAP32[$520 >> 2] = HEAP32[$520 >> 2] | 1; //@line 51343
  }
  $$0 = $509 + 8 | 0; //@line 51346
  STACKTOP = sp; //@line 51347
  return $$0 | 0; //@line 51347
 }
 $524 = HEAP32[162838] | 0; //@line 51349
 if ($524 >>> 0 > $$0197 >>> 0) {
  $526 = $524 - $$0197 | 0; //@line 51352
  HEAP32[162838] = $526; //@line 51353
  $527 = HEAP32[162841] | 0; //@line 51354
  $528 = $527 + $$0197 | 0; //@line 51355
  HEAP32[162841] = $528; //@line 51356
  HEAP32[$528 + 4 >> 2] = $526 | 1; //@line 51359
  HEAP32[$527 + 4 >> 2] = $$0197 | 3; //@line 51362
  $$0 = $527 + 8 | 0; //@line 51364
  STACKTOP = sp; //@line 51365
  return $$0 | 0; //@line 51365
 }
 if (!(HEAP32[162953] | 0)) {
  HEAP32[162955] = 4096; //@line 51370
  HEAP32[162954] = 4096; //@line 51371
  HEAP32[162956] = -1; //@line 51372
  HEAP32[162957] = -1; //@line 51373
  HEAP32[162958] = 0; //@line 51374
  HEAP32[162946] = 0; //@line 51375
  $538 = $1 & -16 ^ 1431655768; //@line 51378
  HEAP32[$1 >> 2] = $538; //@line 51379
  HEAP32[162953] = $538; //@line 51380
  $542 = 4096; //@line 51381
 } else {
  $542 = HEAP32[162955] | 0; //@line 51384
 }
 $539 = $$0197 + 48 | 0; //@line 51386
 $540 = $$0197 + 47 | 0; //@line 51387
 $541 = $542 + $540 | 0; //@line 51388
 $543 = 0 - $542 | 0; //@line 51389
 $544 = $541 & $543; //@line 51390
 if ($544 >>> 0 <= $$0197 >>> 0) {
  $$0 = 0; //@line 51393
  STACKTOP = sp; //@line 51394
  return $$0 | 0; //@line 51394
 }
 $546 = HEAP32[162945] | 0; //@line 51396
 if ($546 | 0) {
  $548 = HEAP32[162943] | 0; //@line 51399
  $549 = $548 + $544 | 0; //@line 51400
  if ($549 >>> 0 <= $548 >>> 0 | $549 >>> 0 > $546 >>> 0) {
   $$0 = 0; //@line 51405
   STACKTOP = sp; //@line 51406
   return $$0 | 0; //@line 51406
  }
 }
 L244 : do {
  if (!(HEAP32[162946] & 4)) {
   $555 = HEAP32[162841] | 0; //@line 51414
   L246 : do {
    if (!$555) {
     label = 163; //@line 51418
    } else {
     $$0$i$i = 651788; //@line 51420
     while (1) {
      $557 = HEAP32[$$0$i$i >> 2] | 0; //@line 51422
      if ($557 >>> 0 <= $555 >>> 0) {
       $559 = $$0$i$i + 4 | 0; //@line 51425
       if (($557 + (HEAP32[$559 >> 2] | 0) | 0) >>> 0 > $555 >>> 0) {
        break;
       }
      }
      $564 = HEAP32[$$0$i$i + 8 >> 2] | 0; //@line 51434
      if (!$564) {
       label = 163; //@line 51437
       break L246;
      } else {
       $$0$i$i = $564; //@line 51440
      }
     }
     $589 = $541 - $524 & $543; //@line 51444
     if ($589 >>> 0 < 2147483647) {
      $591 = _sbrk($589 | 0) | 0; //@line 51447
      if (($591 | 0) == ((HEAP32[$$0$i$i >> 2] | 0) + (HEAP32[$559 >> 2] | 0) | 0)) {
       if (($591 | 0) == (-1 | 0)) {
        $$2234253237$i = $589; //@line 51455
       } else {
        $$723948$i = $589; //@line 51457
        $$749$i = $591; //@line 51457
        label = 180; //@line 51458
        break L244;
       }
      } else {
       $$2247$ph$i = $591; //@line 51462
       $$2253$ph$i = $589; //@line 51462
       label = 171; //@line 51463
      }
     } else {
      $$2234253237$i = 0; //@line 51466
     }
    }
   } while (0);
   do {
    if ((label | 0) == 163) {
     $566 = _sbrk(0) | 0; //@line 51472
     if (($566 | 0) == (-1 | 0)) {
      $$2234253237$i = 0; //@line 51475
     } else {
      $568 = $566; //@line 51477
      $569 = HEAP32[162954] | 0; //@line 51478
      $570 = $569 + -1 | 0; //@line 51479
      $$$i = (($570 & $568 | 0) == 0 ? 0 : ($570 + $568 & 0 - $569) - $568 | 0) + $544 | 0; //@line 51487
      $578 = HEAP32[162943] | 0; //@line 51488
      $579 = $$$i + $578 | 0; //@line 51489
      if ($$$i >>> 0 > $$0197 >>> 0 & $$$i >>> 0 < 2147483647) {
       $582 = HEAP32[162945] | 0; //@line 51494
       if ($582 | 0) {
        if ($579 >>> 0 <= $578 >>> 0 | $579 >>> 0 > $582 >>> 0) {
         $$2234253237$i = 0; //@line 51501
         break;
        }
       }
       $586 = _sbrk($$$i | 0) | 0; //@line 51505
       if (($586 | 0) == ($566 | 0)) {
        $$723948$i = $$$i; //@line 51508
        $$749$i = $566; //@line 51508
        label = 180; //@line 51509
        break L244;
       } else {
        $$2247$ph$i = $586; //@line 51512
        $$2253$ph$i = $$$i; //@line 51512
        label = 171; //@line 51513
       }
      } else {
       $$2234253237$i = 0; //@line 51516
      }
     }
    }
   } while (0);
   do {
    if ((label | 0) == 171) {
     $597 = 0 - $$2253$ph$i | 0; //@line 51523
     if (!($539 >>> 0 > $$2253$ph$i >>> 0 & ($$2253$ph$i >>> 0 < 2147483647 & ($$2247$ph$i | 0) != (-1 | 0)))) {
      if (($$2247$ph$i | 0) == (-1 | 0)) {
       $$2234253237$i = 0; //@line 51532
       break;
      } else {
       $$723948$i = $$2253$ph$i; //@line 51535
       $$749$i = $$2247$ph$i; //@line 51535
       label = 180; //@line 51536
       break L244;
      }
     }
     $601 = HEAP32[162955] | 0; //@line 51540
     $605 = $540 - $$2253$ph$i + $601 & 0 - $601; //@line 51544
     if ($605 >>> 0 >= 2147483647) {
      $$723948$i = $$2253$ph$i; //@line 51547
      $$749$i = $$2247$ph$i; //@line 51547
      label = 180; //@line 51548
      break L244;
     }
     if ((_sbrk($605 | 0) | 0) == (-1 | 0)) {
      _sbrk($597 | 0) | 0; //@line 51554
      $$2234253237$i = 0; //@line 51555
      break;
     } else {
      $$723948$i = $605 + $$2253$ph$i | 0; //@line 51559
      $$749$i = $$2247$ph$i; //@line 51559
      label = 180; //@line 51560
      break L244;
     }
    }
   } while (0);
   HEAP32[162946] = HEAP32[162946] | 4; //@line 51567
   $$4236$i = $$2234253237$i; //@line 51568
   label = 178; //@line 51569
  } else {
   $$4236$i = 0; //@line 51571
   label = 178; //@line 51572
  }
 } while (0);
 if ((label | 0) == 178) {
  if ($544 >>> 0 < 2147483647) {
   $614 = _sbrk($544 | 0) | 0; //@line 51578
   $615 = _sbrk(0) | 0; //@line 51579
   $621 = $615 - $614 | 0; //@line 51587
   $623 = $621 >>> 0 > ($$0197 + 40 | 0) >>> 0; //@line 51589
   if (!(($614 | 0) == (-1 | 0) | $623 ^ 1 | $614 >>> 0 < $615 >>> 0 & (($614 | 0) != (-1 | 0) & ($615 | 0) != (-1 | 0)) ^ 1)) {
    $$723948$i = $623 ? $621 : $$4236$i; //@line 51597
    $$749$i = $614; //@line 51597
    label = 180; //@line 51598
   }
  }
 }
 if ((label | 0) == 180) {
  $627 = (HEAP32[162943] | 0) + $$723948$i | 0; //@line 51604
  HEAP32[162943] = $627; //@line 51605
  if ($627 >>> 0 > (HEAP32[162944] | 0) >>> 0) {
   HEAP32[162944] = $627; //@line 51609
  }
  $630 = HEAP32[162841] | 0; //@line 51611
  do {
   if (!$630) {
    $632 = HEAP32[162839] | 0; //@line 51615
    if (($632 | 0) == 0 | $$749$i >>> 0 < $632 >>> 0) {
     HEAP32[162839] = $$749$i; //@line 51620
    }
    HEAP32[162947] = $$749$i; //@line 51622
    HEAP32[162948] = $$723948$i; //@line 51623
    HEAP32[162950] = 0; //@line 51624
    HEAP32[162844] = HEAP32[162953]; //@line 51626
    HEAP32[162843] = -1; //@line 51627
    $$01$i$i = 0; //@line 51628
    do {
     $637 = 651380 + ($$01$i$i << 1 << 2) | 0; //@line 51631
     HEAP32[$637 + 12 >> 2] = $637; //@line 51633
     HEAP32[$637 + 8 >> 2] = $637; //@line 51635
     $$01$i$i = $$01$i$i + 1 | 0; //@line 51636
    } while (($$01$i$i | 0) != 32);
    $643 = $$749$i + 8 | 0; //@line 51646
    $648 = ($643 & 7 | 0) == 0 ? 0 : 0 - $643 & 7; //@line 51651
    $649 = $$749$i + $648 | 0; //@line 51652
    $650 = $$723948$i + -40 - $648 | 0; //@line 51653
    HEAP32[162841] = $649; //@line 51654
    HEAP32[162838] = $650; //@line 51655
    HEAP32[$649 + 4 >> 2] = $650 | 1; //@line 51658
    HEAP32[$649 + $650 + 4 >> 2] = 40; //@line 51661
    HEAP32[162842] = HEAP32[162957]; //@line 51663
   } else {
    $$024371$i = 651788; //@line 51665
    while (1) {
     $656 = HEAP32[$$024371$i >> 2] | 0; //@line 51667
     $657 = $$024371$i + 4 | 0; //@line 51668
     $658 = HEAP32[$657 >> 2] | 0; //@line 51669
     if (($$749$i | 0) == ($656 + $658 | 0)) {
      label = 190; //@line 51673
      break;
     }
     $662 = HEAP32[$$024371$i + 8 >> 2] | 0; //@line 51677
     if (!$662) {
      break;
     } else {
      $$024371$i = $662; //@line 51682
     }
    }
    if ((label | 0) == 190) {
     if (!(HEAP32[$$024371$i + 12 >> 2] & 8)) {
      if ($630 >>> 0 < $$749$i >>> 0 & $630 >>> 0 >= $656 >>> 0) {
       HEAP32[$657 >> 2] = $658 + $$723948$i; //@line 51696
       $673 = $630 + 8 | 0; //@line 51699
       $678 = ($673 & 7 | 0) == 0 ? 0 : 0 - $673 & 7; //@line 51704
       $679 = $630 + $678 | 0; //@line 51705
       $681 = (HEAP32[162838] | 0) + ($$723948$i - $678) | 0; //@line 51707
       HEAP32[162841] = $679; //@line 51708
       HEAP32[162838] = $681; //@line 51709
       HEAP32[$679 + 4 >> 2] = $681 | 1; //@line 51712
       HEAP32[$679 + $681 + 4 >> 2] = 40; //@line 51715
       HEAP32[162842] = HEAP32[162957]; //@line 51717
       break;
      }
     }
    }
    $687 = HEAP32[162839] | 0; //@line 51722
    if ($$749$i >>> 0 < $687 >>> 0) {
     HEAP32[162839] = $$749$i; //@line 51725
     $752 = $$749$i; //@line 51726
    } else {
     $752 = $687; //@line 51728
    }
    $689 = $$749$i + $$723948$i | 0; //@line 51730
    $$124470$i = 651788; //@line 51731
    while (1) {
     if ((HEAP32[$$124470$i >> 2] | 0) == ($689 | 0)) {
      label = 198; //@line 51736
      break;
     }
     $693 = HEAP32[$$124470$i + 8 >> 2] | 0; //@line 51740
     if (!$693) {
      break;
     } else {
      $$124470$i = $693; //@line 51745
     }
    }
    if ((label | 0) == 198) {
     if (!(HEAP32[$$124470$i + 12 >> 2] & 8)) {
      HEAP32[$$124470$i >> 2] = $$749$i; //@line 51754
      $699 = $$124470$i + 4 | 0; //@line 51755
      HEAP32[$699 >> 2] = (HEAP32[$699 >> 2] | 0) + $$723948$i; //@line 51758
      $703 = $$749$i + 8 | 0; //@line 51760
      $709 = $$749$i + (($703 & 7 | 0) == 0 ? 0 : 0 - $703 & 7) | 0; //@line 51766
      $711 = $689 + 8 | 0; //@line 51768
      $717 = $689 + (($711 & 7 | 0) == 0 ? 0 : 0 - $711 & 7) | 0; //@line 51774
      $721 = $709 + $$0197 | 0; //@line 51778
      $722 = $717 - $709 - $$0197 | 0; //@line 51779
      HEAP32[$709 + 4 >> 2] = $$0197 | 3; //@line 51782
      do {
       if (($717 | 0) == ($630 | 0)) {
        $727 = (HEAP32[162838] | 0) + $722 | 0; //@line 51787
        HEAP32[162838] = $727; //@line 51788
        HEAP32[162841] = $721; //@line 51789
        HEAP32[$721 + 4 >> 2] = $727 | 1; //@line 51792
       } else {
        if (($717 | 0) == (HEAP32[162840] | 0)) {
         $733 = (HEAP32[162837] | 0) + $722 | 0; //@line 51798
         HEAP32[162837] = $733; //@line 51799
         HEAP32[162840] = $721; //@line 51800
         HEAP32[$721 + 4 >> 2] = $733 | 1; //@line 51803
         HEAP32[$721 + $733 >> 2] = $733; //@line 51805
         break;
        }
        $738 = HEAP32[$717 + 4 >> 2] | 0; //@line 51809
        if (($738 & 3 | 0) == 1) {
         $741 = $738 & -8; //@line 51813
         $742 = $738 >>> 3; //@line 51814
         L314 : do {
          if ($738 >>> 0 < 256) {
           $745 = HEAP32[$717 + 8 >> 2] | 0; //@line 51819
           $747 = HEAP32[$717 + 12 >> 2] | 0; //@line 51821
           $749 = 651380 + ($742 << 1 << 2) | 0; //@line 51823
           do {
            if (($745 | 0) != ($749 | 0)) {
             if ($745 >>> 0 < $752 >>> 0) {
              _abort(); //@line 51829
             }
             if ((HEAP32[$745 + 12 >> 2] | 0) == ($717 | 0)) {
              break;
             }
             _abort(); //@line 51838
            }
           } while (0);
           if (($747 | 0) == ($745 | 0)) {
            HEAP32[162835] = HEAP32[162835] & ~(1 << $742); //@line 51848
            break;
           }
           do {
            if (($747 | 0) == ($749 | 0)) {
             $$pre$phi11$i$iZ2D = $747 + 8 | 0; //@line 51855
            } else {
             if ($747 >>> 0 < $752 >>> 0) {
              _abort(); //@line 51859
             }
             $763 = $747 + 8 | 0; //@line 51862
             if ((HEAP32[$763 >> 2] | 0) == ($717 | 0)) {
              $$pre$phi11$i$iZ2D = $763; //@line 51866
              break;
             }
             _abort(); //@line 51869
            }
           } while (0);
           HEAP32[$745 + 12 >> 2] = $747; //@line 51874
           HEAP32[$$pre$phi11$i$iZ2D >> 2] = $745; //@line 51875
          } else {
           $768 = HEAP32[$717 + 24 >> 2] | 0; //@line 51878
           $770 = HEAP32[$717 + 12 >> 2] | 0; //@line 51880
           do {
            if (($770 | 0) == ($717 | 0)) {
             $781 = $717 + 16 | 0; //@line 51884
             $782 = $781 + 4 | 0; //@line 51885
             $783 = HEAP32[$782 >> 2] | 0; //@line 51886
             if (!$783) {
              $785 = HEAP32[$781 >> 2] | 0; //@line 51889
              if (!$785) {
               $$3$i$i = 0; //@line 51892
               break;
              } else {
               $$1291$i$i = $785; //@line 51895
               $$1293$i$i = $781; //@line 51895
              }
             } else {
              $$1291$i$i = $783; //@line 51898
              $$1293$i$i = $782; //@line 51898
             }
             while (1) {
              $787 = $$1291$i$i + 20 | 0; //@line 51901
              $788 = HEAP32[$787 >> 2] | 0; //@line 51902
              if ($788 | 0) {
               $$1291$i$i = $788; //@line 51905
               $$1293$i$i = $787; //@line 51905
               continue;
              }
              $790 = $$1291$i$i + 16 | 0; //@line 51908
              $791 = HEAP32[$790 >> 2] | 0; //@line 51909
              if (!$791) {
               break;
              } else {
               $$1291$i$i = $791; //@line 51914
               $$1293$i$i = $790; //@line 51914
              }
             }
             if ($$1293$i$i >>> 0 < $752 >>> 0) {
              _abort(); //@line 51919
             } else {
              HEAP32[$$1293$i$i >> 2] = 0; //@line 51922
              $$3$i$i = $$1291$i$i; //@line 51923
              break;
             }
            } else {
             $773 = HEAP32[$717 + 8 >> 2] | 0; //@line 51928
             if ($773 >>> 0 < $752 >>> 0) {
              _abort(); //@line 51931
             }
             $775 = $773 + 12 | 0; //@line 51934
             if ((HEAP32[$775 >> 2] | 0) != ($717 | 0)) {
              _abort(); //@line 51938
             }
             $778 = $770 + 8 | 0; //@line 51941
             if ((HEAP32[$778 >> 2] | 0) == ($717 | 0)) {
              HEAP32[$775 >> 2] = $770; //@line 51945
              HEAP32[$778 >> 2] = $773; //@line 51946
              $$3$i$i = $770; //@line 51947
              break;
             } else {
              _abort(); //@line 51950
             }
            }
           } while (0);
           if (!$768) {
            break;
           }
           $796 = HEAP32[$717 + 28 >> 2] | 0; //@line 51960
           $797 = 651644 + ($796 << 2) | 0; //@line 51961
           do {
            if (($717 | 0) == (HEAP32[$797 >> 2] | 0)) {
             HEAP32[$797 >> 2] = $$3$i$i; //@line 51966
             if ($$3$i$i | 0) {
              break;
             }
             HEAP32[162836] = HEAP32[162836] & ~(1 << $796); //@line 51975
             break L314;
            } else {
             if ($768 >>> 0 < (HEAP32[162839] | 0) >>> 0) {
              _abort(); //@line 51981
             } else {
              HEAP32[$768 + 16 + (((HEAP32[$768 + 16 >> 2] | 0) != ($717 | 0) & 1) << 2) >> 2] = $$3$i$i; //@line 51989
              if (!$$3$i$i) {
               break L314;
              } else {
               break;
              }
             }
            }
           } while (0);
           $810 = HEAP32[162839] | 0; //@line 51999
           if ($$3$i$i >>> 0 < $810 >>> 0) {
            _abort(); //@line 52002
           }
           HEAP32[$$3$i$i + 24 >> 2] = $768; //@line 52006
           $813 = $717 + 16 | 0; //@line 52007
           $814 = HEAP32[$813 >> 2] | 0; //@line 52008
           do {
            if ($814 | 0) {
             if ($814 >>> 0 < $810 >>> 0) {
              _abort(); //@line 52014
             } else {
              HEAP32[$$3$i$i + 16 >> 2] = $814; //@line 52018
              HEAP32[$814 + 24 >> 2] = $$3$i$i; //@line 52020
              break;
             }
            }
           } while (0);
           $820 = HEAP32[$813 + 4 >> 2] | 0; //@line 52026
           if (!$820) {
            break;
           }
           if ($820 >>> 0 < (HEAP32[162839] | 0) >>> 0) {
            _abort(); //@line 52034
           } else {
            HEAP32[$$3$i$i + 20 >> 2] = $820; //@line 52038
            HEAP32[$820 + 24 >> 2] = $$3$i$i; //@line 52040
            break;
           }
          }
         } while (0);
         $$0$i18$i = $717 + $741 | 0; //@line 52047
         $$0287$i$i = $741 + $722 | 0; //@line 52047
        } else {
         $$0$i18$i = $717; //@line 52049
         $$0287$i$i = $722; //@line 52049
        }
        $828 = $$0$i18$i + 4 | 0; //@line 52051
        HEAP32[$828 >> 2] = HEAP32[$828 >> 2] & -2; //@line 52054
        HEAP32[$721 + 4 >> 2] = $$0287$i$i | 1; //@line 52057
        HEAP32[$721 + $$0287$i$i >> 2] = $$0287$i$i; //@line 52059
        $834 = $$0287$i$i >>> 3; //@line 52060
        if ($$0287$i$i >>> 0 < 256) {
         $837 = 651380 + ($834 << 1 << 2) | 0; //@line 52064
         $838 = HEAP32[162835] | 0; //@line 52065
         $839 = 1 << $834; //@line 52066
         do {
          if (!($838 & $839)) {
           HEAP32[162835] = $838 | $839; //@line 52072
           $$0295$i$i = $837; //@line 52074
           $$pre$phi$i20$iZ2D = $837 + 8 | 0; //@line 52074
          } else {
           $843 = $837 + 8 | 0; //@line 52076
           $844 = HEAP32[$843 >> 2] | 0; //@line 52077
           if ($844 >>> 0 >= (HEAP32[162839] | 0) >>> 0) {
            $$0295$i$i = $844; //@line 52081
            $$pre$phi$i20$iZ2D = $843; //@line 52081
            break;
           }
           _abort(); //@line 52084
          }
         } while (0);
         HEAP32[$$pre$phi$i20$iZ2D >> 2] = $721; //@line 52088
         HEAP32[$$0295$i$i + 12 >> 2] = $721; //@line 52090
         HEAP32[$721 + 8 >> 2] = $$0295$i$i; //@line 52092
         HEAP32[$721 + 12 >> 2] = $837; //@line 52094
         break;
        }
        $850 = $$0287$i$i >>> 8; //@line 52097
        do {
         if (!$850) {
          $$0296$i$i = 0; //@line 52101
         } else {
          if ($$0287$i$i >>> 0 > 16777215) {
           $$0296$i$i = 31; //@line 52105
           break;
          }
          $855 = ($850 + 1048320 | 0) >>> 16 & 8; //@line 52110
          $856 = $850 << $855; //@line 52111
          $859 = ($856 + 520192 | 0) >>> 16 & 4; //@line 52114
          $861 = $856 << $859; //@line 52116
          $864 = ($861 + 245760 | 0) >>> 16 & 2; //@line 52119
          $869 = 14 - ($859 | $855 | $864) + ($861 << $864 >>> 15) | 0; //@line 52124
          $$0296$i$i = $$0287$i$i >>> ($869 + 7 | 0) & 1 | $869 << 1; //@line 52130
         }
        } while (0);
        $875 = 651644 + ($$0296$i$i << 2) | 0; //@line 52133
        HEAP32[$721 + 28 >> 2] = $$0296$i$i; //@line 52135
        $877 = $721 + 16 | 0; //@line 52136
        HEAP32[$877 + 4 >> 2] = 0; //@line 52138
        HEAP32[$877 >> 2] = 0; //@line 52139
        $879 = HEAP32[162836] | 0; //@line 52140
        $880 = 1 << $$0296$i$i; //@line 52141
        if (!($879 & $880)) {
         HEAP32[162836] = $879 | $880; //@line 52146
         HEAP32[$875 >> 2] = $721; //@line 52147
         HEAP32[$721 + 24 >> 2] = $875; //@line 52149
         HEAP32[$721 + 12 >> 2] = $721; //@line 52151
         HEAP32[$721 + 8 >> 2] = $721; //@line 52153
         break;
        }
        $$0288$i$i = $$0287$i$i << (($$0296$i$i | 0) == 31 ? 0 : 25 - ($$0296$i$i >>> 1) | 0); //@line 52162
        $$0289$i$i = HEAP32[$875 >> 2] | 0; //@line 52162
        while (1) {
         if ((HEAP32[$$0289$i$i + 4 >> 2] & -8 | 0) == ($$0287$i$i | 0)) {
          label = 265; //@line 52169
          break;
         }
         $898 = $$0289$i$i + 16 + ($$0288$i$i >>> 31 << 2) | 0; //@line 52173
         $900 = HEAP32[$898 >> 2] | 0; //@line 52175
         if (!$900) {
          label = 262; //@line 52178
          break;
         } else {
          $$0288$i$i = $$0288$i$i << 1; //@line 52181
          $$0289$i$i = $900; //@line 52181
         }
        }
        if ((label | 0) == 262) {
         if ($898 >>> 0 < (HEAP32[162839] | 0) >>> 0) {
          _abort(); //@line 52188
         } else {
          HEAP32[$898 >> 2] = $721; //@line 52191
          HEAP32[$721 + 24 >> 2] = $$0289$i$i; //@line 52193
          HEAP32[$721 + 12 >> 2] = $721; //@line 52195
          HEAP32[$721 + 8 >> 2] = $721; //@line 52197
          break;
         }
        } else if ((label | 0) == 265) {
         $907 = $$0289$i$i + 8 | 0; //@line 52202
         $908 = HEAP32[$907 >> 2] | 0; //@line 52203
         $909 = HEAP32[162839] | 0; //@line 52204
         if ($908 >>> 0 >= $909 >>> 0 & $$0289$i$i >>> 0 >= $909 >>> 0) {
          HEAP32[$908 + 12 >> 2] = $721; //@line 52210
          HEAP32[$907 >> 2] = $721; //@line 52211
          HEAP32[$721 + 8 >> 2] = $908; //@line 52213
          HEAP32[$721 + 12 >> 2] = $$0289$i$i; //@line 52215
          HEAP32[$721 + 24 >> 2] = 0; //@line 52217
          break;
         } else {
          _abort(); //@line 52220
         }
        }
       }
      } while (0);
      $$0 = $709 + 8 | 0; //@line 52227
      STACKTOP = sp; //@line 52228
      return $$0 | 0; //@line 52228
     }
    }
    $$0$i$i$i = 651788; //@line 52231
    while (1) {
     $916 = HEAP32[$$0$i$i$i >> 2] | 0; //@line 52233
     if ($916 >>> 0 <= $630 >>> 0) {
      $920 = $916 + (HEAP32[$$0$i$i$i + 4 >> 2] | 0) | 0; //@line 52238
      if ($920 >>> 0 > $630 >>> 0) {
       break;
      }
     }
     $$0$i$i$i = HEAP32[$$0$i$i$i + 8 >> 2] | 0; //@line 52246
    }
    $924 = $920 + -47 | 0; //@line 52248
    $926 = $924 + 8 | 0; //@line 52250
    $932 = $924 + (($926 & 7 | 0) == 0 ? 0 : 0 - $926 & 7) | 0; //@line 52256
    $933 = $630 + 16 | 0; //@line 52257
    $935 = $932 >>> 0 < $933 >>> 0 ? $630 : $932; //@line 52259
    $936 = $935 + 8 | 0; //@line 52260
    $937 = $935 + 24 | 0; //@line 52261
    $940 = $$749$i + 8 | 0; //@line 52264
    $945 = ($940 & 7 | 0) == 0 ? 0 : 0 - $940 & 7; //@line 52269
    $946 = $$749$i + $945 | 0; //@line 52270
    $947 = $$723948$i + -40 - $945 | 0; //@line 52271
    HEAP32[162841] = $946; //@line 52272
    HEAP32[162838] = $947; //@line 52273
    HEAP32[$946 + 4 >> 2] = $947 | 1; //@line 52276
    HEAP32[$946 + $947 + 4 >> 2] = 40; //@line 52279
    HEAP32[162842] = HEAP32[162957]; //@line 52281
    $953 = $935 + 4 | 0; //@line 52282
    HEAP32[$953 >> 2] = 27; //@line 52283
    HEAP32[$936 >> 2] = HEAP32[162947]; //@line 52284
    HEAP32[$936 + 4 >> 2] = HEAP32[162948]; //@line 52284
    HEAP32[$936 + 8 >> 2] = HEAP32[162949]; //@line 52284
    HEAP32[$936 + 12 >> 2] = HEAP32[162950]; //@line 52284
    HEAP32[162947] = $$749$i; //@line 52285
    HEAP32[162948] = $$723948$i; //@line 52286
    HEAP32[162950] = 0; //@line 52287
    HEAP32[162949] = $936; //@line 52288
    $955 = $937; //@line 52289
    do {
     $955$looptemp = $955;
     $955 = $955 + 4 | 0; //@line 52291
     HEAP32[$955 >> 2] = 7; //@line 52292
    } while (($955$looptemp + 8 | 0) >>> 0 < $920 >>> 0);
    if (($935 | 0) != ($630 | 0)) {
     $961 = $935 - $630 | 0; //@line 52305
     HEAP32[$953 >> 2] = HEAP32[$953 >> 2] & -2; //@line 52308
     HEAP32[$630 + 4 >> 2] = $961 | 1; //@line 52311
     HEAP32[$935 >> 2] = $961; //@line 52312
     $966 = $961 >>> 3; //@line 52313
     if ($961 >>> 0 < 256) {
      $969 = 651380 + ($966 << 1 << 2) | 0; //@line 52317
      $970 = HEAP32[162835] | 0; //@line 52318
      $971 = 1 << $966; //@line 52319
      if (!($970 & $971)) {
       HEAP32[162835] = $970 | $971; //@line 52324
       $$0211$i$i = $969; //@line 52326
       $$pre$phi$i$iZ2D = $969 + 8 | 0; //@line 52326
      } else {
       $975 = $969 + 8 | 0; //@line 52328
       $976 = HEAP32[$975 >> 2] | 0; //@line 52329
       if ($976 >>> 0 < (HEAP32[162839] | 0) >>> 0) {
        _abort(); //@line 52333
       } else {
        $$0211$i$i = $976; //@line 52336
        $$pre$phi$i$iZ2D = $975; //@line 52336
       }
      }
      HEAP32[$$pre$phi$i$iZ2D >> 2] = $630; //@line 52339
      HEAP32[$$0211$i$i + 12 >> 2] = $630; //@line 52341
      HEAP32[$630 + 8 >> 2] = $$0211$i$i; //@line 52343
      HEAP32[$630 + 12 >> 2] = $969; //@line 52345
      break;
     }
     $982 = $961 >>> 8; //@line 52348
     if (!$982) {
      $$0212$i$i = 0; //@line 52351
     } else {
      if ($961 >>> 0 > 16777215) {
       $$0212$i$i = 31; //@line 52355
      } else {
       $987 = ($982 + 1048320 | 0) >>> 16 & 8; //@line 52359
       $988 = $982 << $987; //@line 52360
       $991 = ($988 + 520192 | 0) >>> 16 & 4; //@line 52363
       $993 = $988 << $991; //@line 52365
       $996 = ($993 + 245760 | 0) >>> 16 & 2; //@line 52368
       $1001 = 14 - ($991 | $987 | $996) + ($993 << $996 >>> 15) | 0; //@line 52373
       $$0212$i$i = $961 >>> ($1001 + 7 | 0) & 1 | $1001 << 1; //@line 52379
      }
     }
     $1007 = 651644 + ($$0212$i$i << 2) | 0; //@line 52382
     HEAP32[$630 + 28 >> 2] = $$0212$i$i; //@line 52384
     HEAP32[$630 + 20 >> 2] = 0; //@line 52386
     HEAP32[$933 >> 2] = 0; //@line 52387
     $1010 = HEAP32[162836] | 0; //@line 52388
     $1011 = 1 << $$0212$i$i; //@line 52389
     if (!($1010 & $1011)) {
      HEAP32[162836] = $1010 | $1011; //@line 52394
      HEAP32[$1007 >> 2] = $630; //@line 52395
      HEAP32[$630 + 24 >> 2] = $1007; //@line 52397
      HEAP32[$630 + 12 >> 2] = $630; //@line 52399
      HEAP32[$630 + 8 >> 2] = $630; //@line 52401
      break;
     }
     $$0206$i$i = $961 << (($$0212$i$i | 0) == 31 ? 0 : 25 - ($$0212$i$i >>> 1) | 0); //@line 52410
     $$0207$i$i = HEAP32[$1007 >> 2] | 0; //@line 52410
     while (1) {
      if ((HEAP32[$$0207$i$i + 4 >> 2] & -8 | 0) == ($961 | 0)) {
       label = 292; //@line 52417
       break;
      }
      $1029 = $$0207$i$i + 16 + ($$0206$i$i >>> 31 << 2) | 0; //@line 52421
      $1031 = HEAP32[$1029 >> 2] | 0; //@line 52423
      if (!$1031) {
       label = 289; //@line 52426
       break;
      } else {
       $$0206$i$i = $$0206$i$i << 1; //@line 52429
       $$0207$i$i = $1031; //@line 52429
      }
     }
     if ((label | 0) == 289) {
      if ($1029 >>> 0 < (HEAP32[162839] | 0) >>> 0) {
       _abort(); //@line 52436
      } else {
       HEAP32[$1029 >> 2] = $630; //@line 52439
       HEAP32[$630 + 24 >> 2] = $$0207$i$i; //@line 52441
       HEAP32[$630 + 12 >> 2] = $630; //@line 52443
       HEAP32[$630 + 8 >> 2] = $630; //@line 52445
       break;
      }
     } else if ((label | 0) == 292) {
      $1038 = $$0207$i$i + 8 | 0; //@line 52450
      $1039 = HEAP32[$1038 >> 2] | 0; //@line 52451
      $1040 = HEAP32[162839] | 0; //@line 52452
      if ($1039 >>> 0 >= $1040 >>> 0 & $$0207$i$i >>> 0 >= $1040 >>> 0) {
       HEAP32[$1039 + 12 >> 2] = $630; //@line 52458
       HEAP32[$1038 >> 2] = $630; //@line 52459
       HEAP32[$630 + 8 >> 2] = $1039; //@line 52461
       HEAP32[$630 + 12 >> 2] = $$0207$i$i; //@line 52463
       HEAP32[$630 + 24 >> 2] = 0; //@line 52465
       break;
      } else {
       _abort(); //@line 52468
      }
     }
    }
   }
  } while (0);
  $1048 = HEAP32[162838] | 0; //@line 52475
  if ($1048 >>> 0 > $$0197 >>> 0) {
   $1050 = $1048 - $$0197 | 0; //@line 52478
   HEAP32[162838] = $1050; //@line 52479
   $1051 = HEAP32[162841] | 0; //@line 52480
   $1052 = $1051 + $$0197 | 0; //@line 52481
   HEAP32[162841] = $1052; //@line 52482
   HEAP32[$1052 + 4 >> 2] = $1050 | 1; //@line 52485
   HEAP32[$1051 + 4 >> 2] = $$0197 | 3; //@line 52488
   $$0 = $1051 + 8 | 0; //@line 52490
   STACKTOP = sp; //@line 52491
   return $$0 | 0; //@line 52491
  }
 }
 HEAP32[(___errno_location() | 0) >> 2] = 12; //@line 52495
 $$0 = 0; //@line 52496
 STACKTOP = sp; //@line 52497
 return $$0 | 0; //@line 52497
}
function __Z12CDROMControljj($0, $1) {
 $0 = $0 | 0;
 $1 = $1 | 0;
 var $$$i = 0, $$$i169 = 0, $$0$ph = 0, $$0142283 = 0, $$0142291 = 0, $$0146289 = 0, $$0258$ph = 0, $$0259$ph = 0, $$0260$ph = 0, $$0261$ph = 0, $$0262$ph = 0, $$0263$ph = 0, $$0264$ph = 0, $$0265$ph = 0, $$03439$i = 0, $$03439$i161 = 0, $$03439$i165 = 0, $$03439$i170 = 0, $$03439$i174 = 0, $$1143290 = 0, $$3 = 0, $$6 = 0, $$9 = 0, $$not = 0, $$sink = 0, $$sink275 = 0, $$sroa$0187$0 = 0, $$sroa$03$012$i = 0, $$sroa$03$07$i = 0, $10 = 0, $1001 = 0, $1013 = 0, $1014 = 0, $102 = 0, $1022 = 0, $1023 = 0, $1029 = 0, $1037 = 0, $1045 = 0, $1053 = 0, $1059 = 0, $109 = 0, $11 = 0, $115 = 0, $12 = 0, $121 = 0, $126 = 0, $13 = 0, $136 = 0, $142 = 0, $147 = 0, $15 = 0, $155 = 0, $165 = 0, $173 = 0, $175 = 0, $18 = 0, $189 = 0, $2 = 0, $201 = 0, $216 = 0, $228 = 0, $240 = 0, $246 = 0, $251 = 0, $252 = 0, $258 = 0, $259 = 0, $26 = 0, $267 = 0, $275 = 0, $287 = 0, $3 = 0, $30 = 0, $300 = 0, $313 = 0, $332 = 0, $339 = 0, $345 = 0, $357 = 0, $36 = 0, $367 = 0, $379 = 0, $38 = 0, $391 = 0, $4 = 0, $405 = 0, $413 = 0, $423 = 0, $43 = 0, $433 = 0, $445 = 0, $457 = 0, $469 = 0, $47 = 0, $48 = 0, $481 = 0, $493 = 0, $505 = 0, $517 = 0, $529 = 0, $537 = 0, $54 = 0, $545 = 0, $552 = 0, $553 = 0, $558 = 0, $579 = 0, $582 = 0, $585 = 0, $596 = 0, $60 = 0, $615 = 0, $625 = 0, $633 = 0, $634 = 0, $641 = 0, $648 = 0, $649 = 0, $654 = 0, $655 = 0, $656 = 0, $657 = 0, $66 = 0, $67 = 0, $68 = 0, $681 = 0, $684 = 0, $69 = 0, $715 = 0, $718 = 0, $72 = 0, $721 = 0, $732 = 0, $753 = 0, $76 = 0, $761 = 0, $775 = 0, $783 = 0, $785 = 0, $791 = 0, $793 = 0, $811 = 0, $814 = 0, $815 = 0, $816 = 0, $817 = 0, $83 = 0, $84 = 0, $841 = 0, $844 = 0, $857 = 0, $86 = 0, $868 = 0, $896 = 0, $905 = 0, $915 = 0, $927 = 0, $939 = 0, $95 = 0, $953 = 0, $961 = 0, $968 = 0, $969 = 0, $974 = 0, $995 = 0, $998 = 0, $trunc = 0, $vararg_buffer = 0, dest = 0, label = 0, sp = 0, stop = 0;
 sp = STACKTOP; //@line 20400
 STACKTOP = STACKTOP + 80 | 0; //@line 20401
 $vararg_buffer = sp; //@line 20402
 $2 = sp + 4 | 0; //@line 20403
 $3 = sp + 72 | 0; //@line 20404
 $4 = $0 + 26 | 0; //@line 20405
 $10 = FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($4 >>> 16 << 2) >> 2] | 0) + 4 >> 2] & 127]($4) | 0; //@line 20411
 $11 = $10 & 65535; //@line 20412
 $trunc = $10 & 65535; //@line 20413
 switch ($trunc << 16 >> 16) {
 case 65:
  {
   $12 = HEAP32[28494] | 0; //@line 20416
   $13 = HEAP32[28493] | 0; //@line 20417
   if (($13 | 0) != ($12 | 0)) {
    $15 = $2 + 32 | 0; //@line 20420
    $$sroa$03$012$i = $13; //@line 20421
    do {
     $18 = (HEAP32[$$sroa$03$012$i + 836 >> 2] | 0) + 3 | 0; //@line 20425
     if (!(FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($18 >>> 16 << 2) >> 2] | 0) + 8 >> 2] & 127]($18) | 0)) {
      $26 = $$sroa$03$012$i + 4 | 0; //@line 20434
      if (__Z17SysIsDiskInsertedPv(HEAP32[$26 >> 2] | 0) | 0) {
       __Z16CDROMMountVolumePv(HEAP32[$26 >> 2] | 0) | 0; //@line 20439
      }
     }
     $30 = $$sroa$03$012$i + 20 | 0; //@line 20442
     if (HEAP8[$30 >> 0] | 0) {
      HEAP32[$2 >> 2] = HEAP32[$$sroa$03$012$i >> 2]; //@line 20447
      HEAP32[$15 >> 2] = 7; //@line 20448
      _Execute68kTrap(-24529, $2); //@line 20449
      HEAP8[$30 >> 0] = 0; //@line 20450
     }
     $$sroa$03$012$i = $$sroa$03$012$i + 840 | 0; //@line 20452
    } while (($$sroa$03$012$i | 0) != ($12 | 0));
   }
   $36 = $1 + 4 | 0; //@line 20461
   $38 = 123608 + ($36 >>> 16 << 2) | 0; //@line 20463
   $43 = (FUNCTION_TABLE_ii[HEAP32[(HEAP32[$38 >> 2] | 0) + 4 >> 2] & 127]($36) | 0) & -8193; //@line 20468
   FUNCTION_TABLE_vii[HEAP32[(HEAP32[$38 >> 2] | 0) + 16 >> 2] & 31]($36, $43); //@line 20472
   HEAP8[657354] = 1; //@line 20473
   $$9 = 0; //@line 20474
   STACKTOP = sp; //@line 20475
   return $$9 | 0; //@line 20475
  }
 case 81:
  {
   $47 = $1 + 34 | 0; //@line 20479
   $48 = $0 + 28 | 0; //@line 20480
   $54 = FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($48 >>> 16 << 2) >> 2] | 0) + 4 >> 2] & 127]($48) | 0; //@line 20486
   FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($47 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($47, $54); //@line 20492
   $$9 = 0; //@line 20493
   STACKTOP = sp; //@line 20494
   return $$9 | 0; //@line 20494
  }
 case 1:
  {
   $$9 = 0; //@line 20498
   STACKTOP = sp; //@line 20499
   return $$9 | 0; //@line 20499
  }
 default:
  {
   $60 = $0 + 22 | 0; //@line 20503
   $66 = FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($60 >>> 16 << 2) >> 2] | 0) + 4 >> 2] & 127]($60) | 0; //@line 20509
   $67 = HEAP32[28494] | 0; //@line 20510
   $68 = HEAP32[28493] | 0; //@line 20511
   $69 = ($68 | 0) == ($67 | 0); //@line 20512
   if ($69) {
    $$9 = -56; //@line 20514
    STACKTOP = sp; //@line 20515
    return $$9 | 0; //@line 20515
   } else {
    $$sroa$03$07$i = $68; //@line 20517
   }
   while (1) {
    if ((HEAP32[$$sroa$03$07$i >> 2] | 0) == ($66 | 0)) {
     label = 15; //@line 20523
     break;
    }
    $72 = $$sroa$03$07$i + 840 | 0; //@line 20526
    if (($72 | 0) == ($67 | 0)) {
     label = 16; //@line 20529
     break;
    } else {
     $$sroa$03$07$i = $72; //@line 20532
    }
   }
   if ((label | 0) == 15) {
    $$not = ($$sroa$03$07$i | 0) != ($67 | 0); //@line 20536
    if ($$not | $69 ^ 1) {
     $$sroa$0187$0 = $$not ? $$sroa$03$07$i : $68; //@line 20541
    } else {
     $$9 = -56; //@line 20543
     STACKTOP = sp; //@line 20544
     return $$9 | 0; //@line 20544
    }
   } else if ((label | 0) == 16) {
    if ($69) {
     $$9 = -56; //@line 20549
     STACKTOP = sp; //@line 20550
     return $$9 | 0; //@line 20550
    } else {
     $$sroa$0187$0 = $68; //@line 20552
    }
   }
   do {
    switch ($trunc << 16 >> 16) {
    case 5:
     {
      $76 = (HEAP32[$$sroa$0187$0 + 836 >> 2] | 0) + 3 | 0; //@line 20560
      $83 = (FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($76 >>> 16 << 2) >> 2] | 0) + 8 >> 2] & 127]($76) | 0) == 0; //@line 20567
      $$9 = $83 ? -65 : 0; //@line 20569
      STACKTOP = sp; //@line 20570
      return $$9 | 0; //@line 20570
     }
    case 7:
     {
      $84 = $$sroa$0187$0 + 836 | 0; //@line 20574
      $86 = (HEAP32[$84 >> 2] | 0) + 3 | 0; //@line 20576
      if (!(FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($86 >>> 16 << 2) >> 2] | 0) + 8 >> 2] & 127]($86) | 0)) {
       $$9 = 0; //@line 20585
       STACKTOP = sp; //@line 20586
       return $$9 | 0; //@line 20586
      }
      $95 = (HEAP32[$84 >> 2] | 0) + 3 | 0; //@line 20589
      FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($95 >>> 16 << 2) >> 2] | 0) + 20 >> 2] & 31]($95, 0); //@line 20595
      HEAP32[$$sroa$0187$0 + 12 >> 2] = -1; //@line 20597
      $$9 = 0; //@line 20598
      STACKTOP = sp; //@line 20599
      return $$9 | 0; //@line 20599
     }
    case 22:
    case 21:
     {
      $102 = $0 + 28 | 0; //@line 20603
      FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($102 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($102, HEAP32[28492] | 0); //@line 20610
      $$9 = 0; //@line 20611
      STACKTOP = sp; //@line 20612
      return $$9 | 0; //@line 20612
     }
    case 23:
     {
      $109 = $0 + 28 | 0; //@line 20616
      FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($109 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($109, 2817); //@line 20622
      $$9 = 0; //@line 20623
      STACKTOP = sp; //@line 20624
      return $$9 | 0; //@line 20624
     }
    case 70:
     {
      $115 = $0 + 28 | 0; //@line 20628
      $121 = FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($115 >>> 16 << 2) >> 2] | 0) + 8 >> 2] & 127]($115) | 0; //@line 20634
      if (($121 & 252) >>> 0 > 3) {
       $$9 = -50; //@line 20638
       STACKTOP = sp; //@line 20639
       return $$9 | 0; //@line 20639
      }
      HEAP8[$$sroa$0187$0 + 833 >> 0] = $121; //@line 20643
      $$9 = 0; //@line 20644
      STACKTOP = sp; //@line 20645
      return $$9 | 0; //@line 20645
     }
    case 76:
     {
      $126 = $0 + 28 | 0; //@line 20649
      HEAP8[$$sroa$0187$0 + 21 >> 0] = (FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($126 >>> 16 << 2) >> 2] | 0) + 4 >> 2] & 127]($126) | 0) != 0 & 1; //@line 20659
      $$9 = 0; //@line 20660
      STACKTOP = sp; //@line 20661
      return $$9 | 0; //@line 20661
     }
    case 79:
     {
      $136 = $0 + 28 | 0; //@line 20665
      $142 = FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($136 >>> 16 << 2) >> 2] | 0) + 4 >> 2] & 127]($136) | 0; //@line 20671
      switch (($142 & 65535) << 16 >> 16) {
      case 2048:
      case 512:
       {
        break;
       }
      default:
       {
        $$9 = -50; //@line 20678
        STACKTOP = sp; //@line 20679
        return $$9 | 0; //@line 20679
       }
      }
      HEAP32[$$sroa$0187$0 + 8 >> 2] = $142 & 65535; //@line 20684
      $$9 = 0; //@line 20685
      STACKTOP = sp; //@line 20686
      return $$9 | 0; //@line 20686
     }
    case 80:
     {
      $147 = (HEAP32[$$sroa$0187$0 + 836 >> 2] | 0) + 3 | 0; //@line 20692
      if (!(FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($147 >>> 16 << 2) >> 2] | 0) + 8 >> 2] & 127]($147) | 0)) {
       $$9 = -65; //@line 20701
       STACKTOP = sp; //@line 20702
       return $$9 | 0; //@line 20702
      }
      $155 = $0 + 28 | 0; //@line 20704
      if ((FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($155 >>> 16 << 2) >> 2] | 0) + 4 >> 2] & 127]($155) | 0) == 1) {
       $$9 = 0; //@line 20713
       STACKTOP = sp; //@line 20714
       return $$9 | 0; //@line 20714
      } else {
       $$9 = 0; //@line 20716
       STACKTOP = sp; //@line 20717
       return $$9 | 0; //@line 20717
      }
      break;
     }
    case 100:
     {
      $165 = (HEAP32[$$sroa$0187$0 + 836 >> 2] | 0) + 3 | 0; //@line 20724
      if (!(FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($165 >>> 16 << 2) >> 2] | 0) + 8 >> 2] & 127]($165) | 0)) {
       $$9 = -65; //@line 20733
       STACKTOP = sp; //@line 20734
       return $$9 | 0; //@line 20734
      }
      $173 = $0 + 28 | 0; //@line 20736
      $175 = 123608 + ($173 >>> 16 << 2) | 0; //@line 20738
      switch (FUNCTION_TABLE_ii[HEAP32[(HEAP32[$175 >> 2] | 0) + 4 >> 2] & 127]($173) | 0) {
      case 1:
       {
        FUNCTION_TABLE_vii[HEAP32[(HEAP32[$175 >> 2] | 0) + 20 >> 2] & 31]($173, HEAPU8[95157 + (HEAPU8[$$sroa$0187$0 + 24 >> 0] | 0) >> 0] | 0); //@line 20754
        $189 = $0 + 29 | 0; //@line 20755
        FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($189 >>> 16 << 2) >> 2] | 0) + 20 >> 2] & 31]($189, HEAPU8[95157 + (HEAPU8[$$sroa$0187$0 + 25 >> 0] | 0) >> 0] | 0); //@line 20767
        $201 = $0 + 30 | 0; //@line 20768
        FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($201 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($201, 0); //@line 20774
        $$9 = 0; //@line 20775
        STACKTOP = sp; //@line 20776
        return $$9 | 0; //@line 20776
       }
      case 2:
       {
        FUNCTION_TABLE_vii[HEAP32[(HEAP32[$175 >> 2] | 0) + 20 >> 2] & 31]($173, HEAPU8[95157 + (HEAPU8[$$sroa$0187$0 + 826 >> 0] | 0) >> 0] | 0); //@line 20789
        $216 = $0 + 29 | 0; //@line 20790
        FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($216 >>> 16 << 2) >> 2] | 0) + 20 >> 2] & 31]($216, HEAPU8[95157 + (HEAPU8[$$sroa$0187$0 + 827 >> 0] | 0) >> 0] | 0); //@line 20802
        $228 = $0 + 30 | 0; //@line 20803
        FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($228 >>> 16 << 2) >> 2] | 0) + 20 >> 2] & 31]($228, HEAPU8[95157 + (HEAPU8[$$sroa$0187$0 + 828 >> 0] | 0) >> 0] | 0); //@line 20815
        $240 = $0 + 31 | 0; //@line 20816
        FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($240 >>> 16 << 2) >> 2] | 0) + 20 >> 2] & 31]($240, 0); //@line 20822
        $$9 = 0; //@line 20823
        STACKTOP = sp; //@line 20824
        return $$9 | 0; //@line 20824
       }
      case 3:
       {
        $246 = $0 + 30 | 0; //@line 20828
        $251 = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($246 >>> 16 << 2) >> 2] >> 2] & 127]($246) | 0; //@line 20833
        $252 = $0 + 34 | 0; //@line 20834
        $258 = FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($252 >>> 16 << 2) >> 2] | 0) + 4 >> 2] & 127]($252) | 0; //@line 20840
        $259 = $0 + 36 | 0; //@line 20841
        $267 = HEAP8[95413 + (FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($259 >>> 16 << 2) >> 2] | 0) + 8 >> 2] & 127]($259) | 0) >> 0] | 0; //@line 20849
        $$0142291 = 4; //@line 20850
        while (1) {
         if ((HEAP8[($$0142291 | 2) + ($$sroa$0187$0 + 22) >> 0] | 0) == $267 << 24 >> 24) {
          $$0142283 = $$0142291; //@line 20857
          break;
         }
         $$0142291 = $$0142291 + 8 | 0; //@line 20860
         if (($$0142291 | 0) >= 804) {
          label = 39; //@line 20865
          break;
         }
        }
        if ((label | 0) == 39) {
         if (($$0142291 | 0) == 804) {
          $$9 = 0; //@line 20872
          STACKTOP = sp; //@line 20873
          return $$9 | 0; //@line 20873
         } else {
          $$0142283 = $$0142291; //@line 20875
         }
        }
        $275 = $258 & 65535; //@line 20878
        if (!$275) {
         $$9 = 0; //@line 20881
         STACKTOP = sp; //@line 20882
         return $$9 | 0; //@line 20882
        } else {
         $$0146289 = $251; //@line 20884
         $$1143290 = $$0142283; //@line 20884
         $332 = $275; //@line 20884
        }
        while (1) {
         FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($$0146289 >>> 16 << 2) >> 2] | 0) + 20 >> 2] & 31]($$0146289, HEAP8[($$1143290 | 1) + ($$sroa$0187$0 + 22) >> 0] & 15); //@line 20897
         $287 = $$0146289 + 1 | 0; //@line 20898
         FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($287 >>> 16 << 2) >> 2] | 0) + 20 >> 2] & 31]($287, HEAPU8[95157 + (HEAPU8[$$1143290 + 5 + ($$sroa$0187$0 + 22) >> 0] | 0) >> 0] | 0); //@line 20911
         $300 = $$0146289 + 2 | 0; //@line 20912
         FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($300 >>> 16 << 2) >> 2] | 0) + 20 >> 2] & 31]($300, HEAPU8[95157 + (HEAPU8[$$1143290 + 6 + ($$sroa$0187$0 + 22) >> 0] | 0) >> 0] | 0); //@line 20925
         $313 = $$0146289 + 3 | 0; //@line 20926
         FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($313 >>> 16 << 2) >> 2] | 0) + 20 >> 2] & 31]($313, HEAPU8[95157 + (HEAPU8[$$1143290 + 7 + ($$sroa$0187$0 + 22) >> 0] | 0) >> 0] | 0); //@line 20939
         if ((HEAP8[($$1143290 | 2) + ($$sroa$0187$0 + 22) >> 0] | 0) == -86) {
          $$9 = 0; //@line 20945
          label = 132; //@line 20946
          break;
         }
         $332 = $332 + 65532 & 65535; //@line 20952
         if (!$332) {
          $$9 = 0; //@line 20955
          label = 132; //@line 20956
          break;
         } else {
          $$0146289 = $$0146289 + 4 | 0; //@line 20959
          $$1143290 = $$1143290 + 8 | 0; //@line 20959
         }
        }
        if ((label | 0) == 132) {
         STACKTOP = sp; //@line 20963
         return $$9 | 0; //@line 20963
        }
        break;
       }
      case 5:
       {
        FUNCTION_TABLE_vii[HEAP32[(HEAP32[$175 >> 2] | 0) + 16 >> 2] & 31]($173, 1); //@line 20971
        $339 = $0 + 30 | 0; //@line 20972
        FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($339 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($339, 1); //@line 20978
        $345 = $0 + 32 | 0; //@line 20979
        FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($345 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($345, HEAPU8[95157 + (HEAPU8[$$sroa$0187$0 + 24 >> 0] | 0) >> 0] | 0); //@line 20991
        $357 = $0 + 34 | 0; //@line 20992
        FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($357 >>> 16 << 2) >> 2] | 0) + 20 >> 2] & 31]($357, HEAP8[$$sroa$0187$0 + 27 >> 0] & 15); //@line 21002
        $367 = $0 + 35 | 0; //@line 21003
        FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($367 >>> 16 << 2) >> 2] | 0) + 20 >> 2] & 31]($367, HEAPU8[95157 + (HEAPU8[$$sroa$0187$0 + 31 >> 0] | 0) >> 0] | 0); //@line 21015
        $379 = $0 + 36 | 0; //@line 21016
        FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($379 >>> 16 << 2) >> 2] | 0) + 20 >> 2] & 31]($379, HEAPU8[95157 + (HEAPU8[$$sroa$0187$0 + 32 >> 0] | 0) >> 0] | 0); //@line 21028
        $391 = $0 + 37 | 0; //@line 21029
        FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($391 >>> 16 << 2) >> 2] | 0) + 20 >> 2] & 31]($391, HEAPU8[95157 + (HEAPU8[$$sroa$0187$0 + 33 >> 0] | 0) >> 0] | 0); //@line 21041
        $$9 = 0; //@line 21042
        STACKTOP = sp; //@line 21043
        return $$9 | 0; //@line 21043
       }
      default:
       {
        _puts(95669) | 0; //@line 21047
        $$9 = -50; //@line 21048
        STACKTOP = sp; //@line 21049
        return $$9 | 0; //@line 21049
       }
      }
      break;
     }
    case 101:
     {
      $405 = (HEAP32[$$sroa$0187$0 + 836 >> 2] | 0) + 3 | 0; //@line 21057
      if (!(FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($405 >>> 16 << 2) >> 2] | 0) + 8 >> 2] & 127]($405) | 0)) {
       $413 = $0 + 28 | 0; //@line 21066
       dest = FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($413 >>> 16 << 2) >> 2] | 0) + 24 >> 2] & 127]($413) | 0; //@line 21073
       stop = dest + 10 | 0; //@line 21073
       do {
        HEAP8[dest >> 0] = 0; //@line 21073
        dest = dest + 1 | 0; //@line 21073
       } while ((dest | 0) < (stop | 0));
       $$9 = -65; //@line 21074
       STACKTOP = sp; //@line 21075
       return $$9 | 0; //@line 21075
      }
      if (__Z16SysCDGetPositionPvPh(HEAP32[$$sroa$0187$0 + 4 >> 2] | 0, $2) | 0) {
       $423 = $0 + 28 | 0; //@line 21081
       FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($423 >>> 16 << 2) >> 2] | 0) + 20 >> 2] & 31]($423, HEAP8[$2 + 5 >> 0] & 15); //@line 21091
       $433 = $0 + 29 | 0; //@line 21092
       FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($433 >>> 16 << 2) >> 2] | 0) + 20 >> 2] & 31]($433, HEAPU8[95157 + (HEAPU8[$2 + 6 >> 0] | 0) >> 0] | 0); //@line 21104
       $445 = $0 + 30 | 0; //@line 21105
       FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($445 >>> 16 << 2) >> 2] | 0) + 20 >> 2] & 31]($445, HEAPU8[95157 + (HEAPU8[$2 + 7 >> 0] | 0) >> 0] | 0); //@line 21117
       $457 = $0 + 31 | 0; //@line 21118
       FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($457 >>> 16 << 2) >> 2] | 0) + 20 >> 2] & 31]($457, HEAPU8[95157 + (HEAPU8[$2 + 13 >> 0] | 0) >> 0] | 0); //@line 21130
       $469 = $0 + 32 | 0; //@line 21131
       FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($469 >>> 16 << 2) >> 2] | 0) + 20 >> 2] & 31]($469, HEAPU8[95157 + (HEAPU8[$2 + 14 >> 0] | 0) >> 0] | 0); //@line 21143
       $481 = $0 + 33 | 0; //@line 21144
       FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($481 >>> 16 << 2) >> 2] | 0) + 20 >> 2] & 31]($481, HEAPU8[95157 + (HEAPU8[$2 + 15 >> 0] | 0) >> 0] | 0); //@line 21156
       $493 = $0 + 34 | 0; //@line 21157
       FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($493 >>> 16 << 2) >> 2] | 0) + 20 >> 2] & 31]($493, HEAPU8[95157 + (HEAPU8[$2 + 9 >> 0] | 0) >> 0] | 0); //@line 21169
       $505 = $0 + 35 | 0; //@line 21170
       FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($505 >>> 16 << 2) >> 2] | 0) + 20 >> 2] & 31]($505, HEAPU8[95157 + (HEAPU8[$2 + 10 >> 0] | 0) >> 0] | 0); //@line 21182
       $517 = $0 + 36 | 0; //@line 21183
       FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($517 >>> 16 << 2) >> 2] | 0) + 20 >> 2] & 31]($517, HEAPU8[95157 + (HEAPU8[$2 + 11 >> 0] | 0) >> 0] | 0); //@line 21195
       $529 = $0 + 37 | 0; //@line 21196
       FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($529 >>> 16 << 2) >> 2] | 0) + 20 >> 2] & 31]($529, 0); //@line 21202
       $$3 = 0; //@line 21203
      } else {
       $$3 = -36; //@line 21205
      }
      $$9 = $$3; //@line 21207
      STACKTOP = sp; //@line 21208
      return $$9 | 0; //@line 21208
     }
    case 102:
     {
      _puts(95722) | 0; //@line 21212
      $$9 = -17; //@line 21213
      STACKTOP = sp; //@line 21214
      return $$9 | 0; //@line 21214
     }
    case 103:
     {
      $537 = (HEAP32[$$sroa$0187$0 + 836 >> 2] | 0) + 3 | 0; //@line 21220
      if (!(FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($537 >>> 16 << 2) >> 2] | 0) + 8 >> 2] & 127]($537) | 0)) {
       $$9 = -65; //@line 21229
       STACKTOP = sp; //@line 21230
       return $$9 | 0; //@line 21230
      }
      $545 = $0 + 28 | 0; //@line 21232
      $552 = (FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($545 >>> 16 << 2) >> 2] | 0) + 4 >> 2] & 127]($545) | 0) & 65535; //@line 21239
      $553 = $0 + 30 | 0; //@line 21240
      $558 = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($553 >>> 16 << 2) >> 2] >> 2] & 127]($553) | 0; //@line 21245
      L93 : do {
       switch ($552 << 16 >> 16) {
       case 0:
        {
         $$0263$ph = (($558 >>> 0) % 75 | 0) & 255; //@line 21256
         $$0264$ph = (((($558 >>> 0) / 75 | 0) >>> 0) % 60 | 0) & 255; //@line 21256
         $$0265$ph = (($558 >>> 0) / 4500 | 0) & 255; //@line 21256
         break;
        }
       case 1:
        {
         $$0263$ph = HEAP8[95413 + ($558 & 255) >> 0] | 0; //@line 21271
         $$0264$ph = HEAP8[95413 + ($558 >>> 8 & 255) >> 0] | 0; //@line 21271
         $$0265$ph = HEAP8[95413 + ($558 >>> 16 & 255) >> 0] | 0; //@line 21271
         break;
        }
       case 2:
        {
         $579 = HEAP8[95413 + ($558 & 255) >> 0] | 0; //@line 21277
         $$03439$i174 = 4; //@line 21278
         while (1) {
          $582 = HEAP8[($$03439$i174 | 2) + ($$sroa$0187$0 + 22) >> 0] | 0; //@line 21282
          if ($582 << 24 >> 24 == -86 ? 1 : $582 << 24 >> 24 == $579 << 24 >> 24) {
           label = 58; //@line 21287
           break;
          }
          $585 = $$03439$i174 + 8 | 0; //@line 21290
          if (($585 | 0) < 804) {
           $$03439$i174 = $585; //@line 21293
          } else {
           $$9 = -50; //@line 21295
           label = 132; //@line 21296
           break;
          }
         }
         if ((label | 0) == 58) {
          $$0263$ph = HEAP8[$$03439$i174 + 7 + ($$sroa$0187$0 + 22) >> 0] | 0; //@line 21310
          $$0264$ph = HEAP8[$$03439$i174 + 6 + ($$sroa$0187$0 + 22) >> 0] | 0; //@line 21310
          $$0265$ph = HEAP8[$$03439$i174 + 5 + ($$sroa$0187$0 + 22) >> 0] | 0; //@line 21310
          break L93;
         } else if ((label | 0) == 132) {
          STACKTOP = sp; //@line 21314
          return $$9 | 0; //@line 21314
         }
         break;
        }
       default:
        {
         $$9 = -50; //@line 21319
         STACKTOP = sp; //@line 21320
         return $$9 | 0; //@line 21320
        }
       }
      } while (0);
      $596 = $0 + 37 | 0; //@line 21324
      HEAP8[$$sroa$0187$0 + 832 >> 0] = (FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($596 >>> 16 << 2) >> 2] | 0) + 8 >> 2] & 127]($596) | 0) & 15; //@line 21334
      if (!(__Z9SysCDPlayPvhhhhhh(HEAP32[$$sroa$0187$0 + 4 >> 2] | 0, $$0265$ph, $$0264$ph, $$0263$ph, HEAP8[$$sroa$0187$0 + 829 >> 0] | 0, HEAP8[$$sroa$0187$0 + 830 >> 0] | 0, HEAP8[$$sroa$0187$0 + 831 >> 0] | 0) | 0)) {
       $$9 = -50; //@line 21345
       STACKTOP = sp; //@line 21346
       return $$9 | 0; //@line 21346
      }
      $615 = $0 + 34 | 0; //@line 21348
      if (FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($615 >>> 16 << 2) >> 2] | 0) + 4 >> 2] & 127]($615) | 0) {
       $$9 = 0; //@line 21357
       STACKTOP = sp; //@line 21358
       return $$9 | 0; //@line 21358
      }
      $$9 = 0; //@line 21360
      STACKTOP = sp; //@line 21361
      return $$9 | 0; //@line 21361
     }
    case 104:
     {
      $625 = (HEAP32[$$sroa$0187$0 + 836 >> 2] | 0) + 3 | 0; //@line 21367
      if (!(FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($625 >>> 16 << 2) >> 2] | 0) + 8 >> 2] & 127]($625) | 0)) {
       $$9 = -65; //@line 21376
       STACKTOP = sp; //@line 21377
       return $$9 | 0; //@line 21377
      }
      $633 = $0 + 28 | 0; //@line 21379
      $634 = $0 + 34 | 0; //@line 21380
      $641 = (FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($634 >>> 16 << 2) >> 2] | 0) + 4 >> 2] & 127]($634) | 0) == 0; //@line 21387
      $648 = (FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($633 >>> 16 << 2) >> 2] | 0) + 4 >> 2] & 127]($633) | 0) & 65535; //@line 21394
      $649 = $0 + 30 | 0; //@line 21395
      $654 = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($649 >>> 16 << 2) >> 2] >> 2] & 127]($649) | 0; //@line 21400
      if ($641) {
       L118 : do {
        switch ($648 << 16 >> 16) {
        case 0:
         {
          $$0260$ph = (($654 >>> 0) % 75 | 0) & 255; //@line 21412
          $$0261$ph = (((($654 >>> 0) / 75 | 0) >>> 0) % 60 | 0) & 255; //@line 21412
          $$0262$ph = (($654 >>> 0) / 4500 | 0) & 255; //@line 21412
          break;
         }
        case 1:
         {
          $$0260$ph = HEAP8[95413 + ($654 & 255) >> 0] | 0; //@line 21427
          $$0261$ph = HEAP8[95413 + ($654 >>> 8 & 255) >> 0] | 0; //@line 21427
          $$0262$ph = HEAP8[95413 + ($654 >>> 16 & 255) >> 0] | 0; //@line 21427
          break;
         }
        case 2:
         {
          $715 = HEAP8[95413 + ($654 & 255) >> 0] | 0; //@line 21433
          $$03439$i165 = 4; //@line 21434
          while (1) {
           $718 = HEAP8[($$03439$i165 | 2) + ($$sroa$0187$0 + 22) >> 0] | 0; //@line 21438
           if ($718 << 24 >> 24 == -86 ? 1 : $718 << 24 >> 24 == $715 << 24 >> 24) {
            label = 77; //@line 21443
            break;
           }
           $721 = $$03439$i165 + 8 | 0; //@line 21446
           if (($721 | 0) < 804) {
            $$03439$i165 = $721; //@line 21449
           } else {
            $$9 = -50; //@line 21451
            label = 132; //@line 21452
            break;
           }
          }
          if ((label | 0) == 77) {
           $$0260$ph = HEAP8[$$03439$i165 + 7 + ($$sroa$0187$0 + 22) >> 0] | 0; //@line 21466
           $$0261$ph = HEAP8[$$03439$i165 + 6 + ($$sroa$0187$0 + 22) >> 0] | 0; //@line 21466
           $$0262$ph = HEAP8[$$03439$i165 + 5 + ($$sroa$0187$0 + 22) >> 0] | 0; //@line 21466
           break L118;
          } else if ((label | 0) == 132) {
           STACKTOP = sp; //@line 21470
           return $$9 | 0; //@line 21470
          }
          break;
         }
        default:
         {
          $$9 = -50; //@line 21475
          STACKTOP = sp; //@line 21476
          return $$9 | 0; //@line 21476
         }
        }
       } while (0);
       $732 = $0 + 37 | 0; //@line 21480
       HEAP8[$$sroa$0187$0 + 832 >> 0] = (FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($732 >>> 16 << 2) >> 2] | 0) + 8 >> 2] & 127]($732) | 0) & 15; //@line 21490
       if (!(__Z9SysCDPlayPvhhhhhh(HEAP32[$$sroa$0187$0 + 4 >> 2] | 0, $$0262$ph, $$0261$ph, $$0260$ph, HEAP8[$$sroa$0187$0 + 829 >> 0] | 0, HEAP8[$$sroa$0187$0 + 830 >> 0] | 0, HEAP8[$$sroa$0187$0 + 831 >> 0] | 0) | 0)) {
        $$9 = -50; //@line 21501
        STACKTOP = sp; //@line 21502
        return $$9 | 0; //@line 21502
       }
      } else {
       $655 = $$sroa$0187$0 + 829 | 0; //@line 21505
       $656 = $$sroa$0187$0 + 830 | 0; //@line 21506
       $657 = $$sroa$0187$0 + 831 | 0; //@line 21507
       L132 : do {
        switch ($648 << 16 >> 16) {
        case 0:
         {
          HEAP8[$655 >> 0] = ($654 >>> 0) / 4500 | 0; //@line 21513
          HEAP8[$656 >> 0] = ((($654 >>> 0) / 75 | 0) >>> 0) % 60 | 0; //@line 21517
          $$sink = (($654 >>> 0) % 75 | 0) & 255; //@line 21520
          break;
         }
        case 1:
         {
          HEAP8[$655 >> 0] = HEAP8[95413 + ($654 >>> 16 & 255) >> 0] | 0; //@line 21528
          HEAP8[$656 >> 0] = HEAP8[95413 + ($654 >>> 8 & 255) >> 0] | 0; //@line 21533
          $$sink = HEAP8[95413 + ($654 & 255) >> 0] | 0; //@line 21537
          break;
         }
        case 2:
         {
          $$$i169 = (HEAP8[95413 + ($654 & 255) >> 0] | 0) + 1 << 24 >> 24; //@line 21544
          $$03439$i170 = 4; //@line 21545
          while (1) {
           $681 = HEAP8[($$03439$i170 | 2) + ($$sroa$0187$0 + 22) >> 0] | 0; //@line 21549
           if ($681 << 24 >> 24 == -86 ? 1 : $681 << 24 >> 24 == $$$i169 << 24 >> 24) {
            label = 70; //@line 21554
            break;
           }
           $684 = $$03439$i170 + 8 | 0; //@line 21557
           if (($684 | 0) < 804) {
            $$03439$i170 = $684; //@line 21560
           } else {
            $$9 = -50; //@line 21562
            label = 132; //@line 21563
            break;
           }
          }
          if ((label | 0) == 70) {
           HEAP8[$655 >> 0] = HEAP8[$$03439$i170 + 5 + ($$sroa$0187$0 + 22) >> 0] | 0; //@line 21571
           HEAP8[$656 >> 0] = HEAP8[$$03439$i170 + 6 + ($$sroa$0187$0 + 22) >> 0] | 0; //@line 21575
           $$sink = HEAP8[$$03439$i170 + 7 + ($$sroa$0187$0 + 22) >> 0] | 0; //@line 21579
           break L132;
          } else if ((label | 0) == 132) {
           STACKTOP = sp; //@line 21583
           return $$9 | 0; //@line 21583
          }
          break;
         }
        default:
         {
          $$9 = -50; //@line 21588
          STACKTOP = sp; //@line 21589
          return $$9 | 0; //@line 21589
         }
        }
       } while (0);
       HEAP8[$657 >> 0] = $$sink; //@line 21593
      }
      $$9 = 0; //@line 21595
      STACKTOP = sp; //@line 21596
      return $$9 | 0; //@line 21596
     }
    case 105:
     {
      $753 = (HEAP32[$$sroa$0187$0 + 836 >> 2] | 0) + 3 | 0; //@line 21602
      if (!(FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($753 >>> 16 << 2) >> 2] | 0) + 8 >> 2] & 127]($753) | 0)) {
       $$9 = -65; //@line 21611
       STACKTOP = sp; //@line 21612
       return $$9 | 0; //@line 21612
      }
      $761 = $0 + 28 | 0; //@line 21614
      switch (FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($761 >>> 16 << 2) >> 2] >> 2] & 127]($761) | 0) {
      case 0:
       {
        if (!(__Z11SysCDResumePv(HEAP32[$$sroa$0187$0 + 4 >> 2] | 0) | 0)) {
         $$9 = -50; //@line 21626
         STACKTOP = sp; //@line 21627
         return $$9 | 0; //@line 21627
        }
        break;
       }
      case 1:
       {
        if (!(__Z10SysCDPausePv(HEAP32[$$sroa$0187$0 + 4 >> 2] | 0) | 0)) {
         $$9 = -50; //@line 21636
         STACKTOP = sp; //@line 21637
         return $$9 | 0; //@line 21637
        }
        break;
       }
      default:
       {
        $$9 = -50; //@line 21642
        STACKTOP = sp; //@line 21643
        return $$9 | 0; //@line 21643
       }
      }
      $$9 = 0; //@line 21646
      STACKTOP = sp; //@line 21647
      return $$9 | 0; //@line 21647
     }
    case 106:
     {
      $775 = (HEAP32[$$sroa$0187$0 + 836 >> 2] | 0) + 3 | 0; //@line 21653
      if (!(FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($775 >>> 16 << 2) >> 2] | 0) + 8 >> 2] & 127]($775) | 0)) {
       $$9 = -65; //@line 21662
       STACKTOP = sp; //@line 21663
       return $$9 | 0; //@line 21663
      }
      $783 = $0 + 28 | 0; //@line 21665
      $785 = 123608 + ($783 >>> 16 << 2) | 0; //@line 21667
      $791 = $0 + 30 | 0; //@line 21673
      $793 = 123608 + ($791 >>> 16 << 2) | 0; //@line 21675
      if (!(FUNCTION_TABLE_ii[HEAP32[(HEAP32[$785 >> 2] | 0) + 4 >> 2] & 127]($783) | 0)) {
       if (!(FUNCTION_TABLE_ii[HEAP32[HEAP32[$793 >> 2] >> 2] & 127]($791) | 0)) {
        if (!(__Z9SysCDStopPvhhh(HEAP32[$$sroa$0187$0 + 4 >> 2] | 0, HEAP8[$$sroa$0187$0 + 826 >> 0] | 0, HEAP8[$$sroa$0187$0 + 827 >> 0] | 0, HEAP8[$$sroa$0187$0 + 828 >> 0] | 0) | 0)) {
         $$9 = -50; //@line 21692
         STACKTOP = sp; //@line 21693
         return $$9 | 0; //@line 21693
        }
       } else {
        label = 90; //@line 21696
       }
      } else {
       label = 90; //@line 21699
      }
      if ((label | 0) == 90) {
       $811 = (FUNCTION_TABLE_ii[HEAP32[(HEAP32[$785 >> 2] | 0) + 4 >> 2] & 127]($783) | 0) & 65535; //@line 21706
       $814 = FUNCTION_TABLE_ii[HEAP32[HEAP32[$793 >> 2] >> 2] & 127]($791) | 0; //@line 21709
       $815 = $$sroa$0187$0 + 829 | 0; //@line 21710
       $816 = $$sroa$0187$0 + 830 | 0; //@line 21711
       $817 = $$sroa$0187$0 + 831 | 0; //@line 21712
       L168 : do {
        switch ($811 << 16 >> 16) {
        case 0:
         {
          HEAP8[$815 >> 0] = ($814 >>> 0) / 4500 | 0; //@line 21718
          HEAP8[$816 >> 0] = ((($814 >>> 0) / 75 | 0) >>> 0) % 60 | 0; //@line 21722
          $$sink275 = (($814 >>> 0) % 75 | 0) & 255; //@line 21725
          break;
         }
        case 1:
         {
          HEAP8[$815 >> 0] = HEAP8[95413 + ($814 >>> 16 & 255) >> 0] | 0; //@line 21733
          HEAP8[$816 >> 0] = HEAP8[95413 + ($814 >>> 8 & 255) >> 0] | 0; //@line 21738
          $$sink275 = HEAP8[95413 + ($814 & 255) >> 0] | 0; //@line 21742
          break;
         }
        case 2:
         {
          $$$i = (HEAP8[95413 + ($814 & 255) >> 0] | 0) + 1 << 24 >> 24; //@line 21749
          $$03439$i161 = 4; //@line 21750
          while (1) {
           $841 = HEAP8[($$03439$i161 | 2) + ($$sroa$0187$0 + 22) >> 0] | 0; //@line 21754
           if ($841 << 24 >> 24 == -86 ? 1 : $841 << 24 >> 24 == $$$i << 24 >> 24) {
            label = 96; //@line 21759
            break;
           }
           $844 = $$03439$i161 + 8 | 0; //@line 21762
           if (($844 | 0) < 804) {
            $$03439$i161 = $844; //@line 21765
           } else {
            $$9 = -50; //@line 21767
            label = 132; //@line 21768
            break;
           }
          }
          if ((label | 0) == 96) {
           HEAP8[$815 >> 0] = HEAP8[$$03439$i161 + 5 + ($$sroa$0187$0 + 22) >> 0] | 0; //@line 21776
           HEAP8[$816 >> 0] = HEAP8[$$03439$i161 + 6 + ($$sroa$0187$0 + 22) >> 0] | 0; //@line 21780
           $$sink275 = HEAP8[$$03439$i161 + 7 + ($$sroa$0187$0 + 22) >> 0] | 0; //@line 21784
           break L168;
          } else if ((label | 0) == 132) {
           STACKTOP = sp; //@line 21788
           return $$9 | 0; //@line 21788
          }
          break;
         }
        default:
         {
          $$9 = -50; //@line 21793
          STACKTOP = sp; //@line 21794
          return $$9 | 0; //@line 21794
         }
        }
       } while (0);
       HEAP8[$817 >> 0] = $$sink275; //@line 21798
      }
      $$9 = 0; //@line 21800
      STACKTOP = sp; //@line 21801
      return $$9 | 0; //@line 21801
     }
    case 107:
     {
      $857 = (HEAP32[$$sroa$0187$0 + 836 >> 2] | 0) + 3 | 0; //@line 21807
      if (!(FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($857 >>> 16 << 2) >> 2] | 0) + 8 >> 2] & 127]($857) | 0)) {
       $$9 = -65; //@line 21816
       STACKTOP = sp; //@line 21817
       return $$9 | 0; //@line 21817
      }
      if (__Z16SysCDGetPositionPvPh(HEAP32[$$sroa$0187$0 + 4 >> 2] | 0, $2) | 0) {
       $868 = $0 + 28 | 0; //@line 21823
       switch (HEAP8[$2 + 1 >> 0] | 0) {
       case 17:
        {
         FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($868 >>> 16 << 2) >> 2] | 0) + 20 >> 2] & 31]($868, 0); //@line 21833
         break;
        }
       case 18:
        {
         FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($868 >>> 16 << 2) >> 2] | 0) + 20 >> 2] & 31]($868, 1); //@line 21842
         break;
        }
       case 19:
        {
         FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($868 >>> 16 << 2) >> 2] | 0) + 20 >> 2] & 31]($868, 3); //@line 21851
         break;
        }
       case 20:
        {
         FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($868 >>> 16 << 2) >> 2] | 0) + 20 >> 2] & 31]($868, 4); //@line 21860
         break;
        }
       default:
        {
         FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($868 >>> 16 << 2) >> 2] | 0) + 20 >> 2] & 31]($868, 5); //@line 21869
        }
       }
       $896 = $0 + 29 | 0; //@line 21872
       FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($896 >>> 16 << 2) >> 2] | 0) + 20 >> 2] & 31]($896, HEAPU8[$$sroa$0187$0 + 832 >> 0] | 0); //@line 21881
       $905 = $0 + 30 | 0; //@line 21882
       FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($905 >>> 16 << 2) >> 2] | 0) + 20 >> 2] & 31]($905, HEAP8[$2 + 5 >> 0] & 15); //@line 21892
       $915 = $0 + 31 | 0; //@line 21893
       FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($915 >>> 16 << 2) >> 2] | 0) + 20 >> 2] & 31]($915, HEAPU8[95157 + (HEAPU8[$2 + 9 >> 0] | 0) >> 0] | 0); //@line 21905
       $927 = $0 + 32 | 0; //@line 21906
       FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($927 >>> 16 << 2) >> 2] | 0) + 20 >> 2] & 31]($927, HEAPU8[95157 + (HEAPU8[$2 + 10 >> 0] | 0) >> 0] | 0); //@line 21918
       $939 = $0 + 33 | 0; //@line 21919
       FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($939 >>> 16 << 2) >> 2] | 0) + 20 >> 2] & 31]($939, HEAPU8[95157 + (HEAPU8[$2 + 11 >> 0] | 0) >> 0] | 0); //@line 21931
       $$6 = 0; //@line 21932
      } else {
       $$6 = -50; //@line 21934
      }
      $$9 = $$6; //@line 21936
      STACKTOP = sp; //@line 21937
      return $$9 | 0; //@line 21937
     }
    case 108:
     {
      $953 = (HEAP32[$$sroa$0187$0 + 836 >> 2] | 0) + 3 | 0; //@line 21943
      if (!(FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($953 >>> 16 << 2) >> 2] | 0) + 8 >> 2] & 127]($953) | 0)) {
       $$9 = -65; //@line 21952
       STACKTOP = sp; //@line 21953
       return $$9 | 0; //@line 21953
      }
      $961 = $0 + 28 | 0; //@line 21955
      $968 = (FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($961 >>> 16 << 2) >> 2] | 0) + 4 >> 2] & 127]($961) | 0) & 65535; //@line 21962
      $969 = $0 + 30 | 0; //@line 21963
      $974 = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($969 >>> 16 << 2) >> 2] >> 2] & 127]($969) | 0; //@line 21968
      L201 : do {
       switch ($968 << 16 >> 16) {
       case 0:
        {
         $$0$ph = (($974 >>> 0) % 75 | 0) & 255; //@line 21979
         $$0258$ph = (((($974 >>> 0) / 75 | 0) >>> 0) % 60 | 0) & 255; //@line 21979
         $$0259$ph = (($974 >>> 0) / 4500 | 0) & 255; //@line 21979
         break;
        }
       case 1:
        {
         $$0$ph = HEAP8[95413 + ($974 & 255) >> 0] | 0; //@line 21994
         $$0258$ph = HEAP8[95413 + ($974 >>> 8 & 255) >> 0] | 0; //@line 21994
         $$0259$ph = HEAP8[95413 + ($974 >>> 16 & 255) >> 0] | 0; //@line 21994
         break;
        }
       case 2:
        {
         $995 = HEAP8[95413 + ($974 & 255) >> 0] | 0; //@line 22000
         $$03439$i = 4; //@line 22001
         while (1) {
          $998 = HEAP8[($$03439$i | 2) + ($$sroa$0187$0 + 22) >> 0] | 0; //@line 22005
          if ($998 << 24 >> 24 == -86 ? 1 : $998 << 24 >> 24 == $995 << 24 >> 24) {
           label = 116; //@line 22010
           break;
          }
          $1001 = $$03439$i + 8 | 0; //@line 22013
          if (($1001 | 0) < 804) {
           $$03439$i = $1001; //@line 22016
          } else {
           $$9 = -50; //@line 22018
           label = 132; //@line 22019
           break;
          }
         }
         if ((label | 0) == 116) {
          $$0$ph = HEAP8[$$03439$i + 7 + ($$sroa$0187$0 + 22) >> 0] | 0; //@line 22033
          $$0258$ph = HEAP8[$$03439$i + 6 + ($$sroa$0187$0 + 22) >> 0] | 0; //@line 22033
          $$0259$ph = HEAP8[$$03439$i + 5 + ($$sroa$0187$0 + 22) >> 0] | 0; //@line 22033
          break L201;
         } else if ((label | 0) == 132) {
          STACKTOP = sp; //@line 22037
          return $$9 | 0; //@line 22037
         }
         break;
        }
       default:
        {
         $$9 = -50; //@line 22042
         STACKTOP = sp; //@line 22043
         return $$9 | 0; //@line 22043
        }
       }
      } while (0);
      $1013 = HEAP32[$$sroa$0187$0 + 4 >> 2] | 0; //@line 22048
      $1014 = $0 + 34 | 0; //@line 22049
      $1022 = __Z9SysCDScanPvhhhb($1013, $$0259$ph, $$0258$ph, $$0$ph, (FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($1014 >>> 16 << 2) >> 2] | 0) + 4 >> 2] & 127]($1014) | 0) != 0) | 0; //@line 22057
      $$9 = $1022 ? 0 : -50; //@line 22059
      STACKTOP = sp; //@line 22060
      return $$9 | 0; //@line 22060
     }
    case 109:
     {
      $1023 = $0 + 28 | 0; //@line 22064
      FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($1023 >>> 16 << 2) >> 2] | 0) + 8 >> 2] & 127]($1023) | 0; //@line 22070
      $1029 = $0 + 29 | 0; //@line 22071
      FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($1029 >>> 16 << 2) >> 2] | 0) + 8 >> 2] & 127]($1029) | 0; //@line 22077
      $$9 = 0; //@line 22078
      STACKTOP = sp; //@line 22079
      return $$9 | 0; //@line 22079
     }
    case 110:
     {
      _puts(95771) | 0; //@line 22083
      $$9 = -17; //@line 22084
      STACKTOP = sp; //@line 22085
      return $$9 | 0; //@line 22085
     }
    case 111:
     {
      _puts(95820) | 0; //@line 22089
      $$9 = -17; //@line 22090
      STACKTOP = sp; //@line 22091
      return $$9 | 0; //@line 22091
     }
    case 112:
     {
      HEAP8[$2 >> 0] = 0; //@line 22095
      HEAP8[$3 >> 0] = 0; //@line 22096
      __Z14SysCDGetVolumePvRhS0_(HEAP32[$$sroa$0187$0 + 4 >> 2] | 0, $2, $3); //@line 22099
      $1037 = $0 + 28 | 0; //@line 22100
      FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($1037 >>> 16 << 2) >> 2] | 0) + 20 >> 2] & 31]($1037, HEAPU8[$2 >> 0] | 0); //@line 22108
      $1045 = $0 + 29 | 0; //@line 22109
      FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($1045 >>> 16 << 2) >> 2] | 0) + 20 >> 2] & 31]($1045, HEAPU8[$3 >> 0] | 0); //@line 22117
      $$9 = 0; //@line 22118
      STACKTOP = sp; //@line 22119
      return $$9 | 0; //@line 22119
     }
    case 113:
     {
      $1053 = $0 + 28 | 0; //@line 22123
      FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($1053 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($1053, 255); //@line 22129
      $$9 = 0; //@line 22130
      STACKTOP = sp; //@line 22131
      return $$9 | 0; //@line 22131
     }
    case 114:
     {
      $$9 = 0; //@line 22135
      STACKTOP = sp; //@line 22136
      return $$9 | 0; //@line 22136
     }
    case 115:
     {
      _puts(95869) | 0; //@line 22140
      $$9 = -17; //@line 22141
      STACKTOP = sp; //@line 22142
      return $$9 | 0; //@line 22142
     }
    case 116:
     {
      _puts(95918) | 0; //@line 22146
      $$9 = -17; //@line 22147
      STACKTOP = sp; //@line 22148
      return $$9 | 0; //@line 22148
     }
    case 122:
     {
      _puts(95967) | 0; //@line 22152
      $$9 = -17; //@line 22153
      STACKTOP = sp; //@line 22154
      return $$9 | 0; //@line 22154
     }
    case 123:
     {
      _puts(96016) | 0; //@line 22158
      $$9 = -17; //@line 22159
      STACKTOP = sp; //@line 22160
      return $$9 | 0; //@line 22160
     }
    case 124:
     {
      _puts(96065) | 0; //@line 22164
      $$9 = -17; //@line 22165
      STACKTOP = sp; //@line 22166
      return $$9 | 0; //@line 22166
     }
    case 125:
     {
      _puts(96114) | 0; //@line 22170
      $$9 = -17; //@line 22171
      STACKTOP = sp; //@line 22172
      return $$9 | 0; //@line 22172
     }
    case 126:
     {
      $1059 = $0 + 28 | 0; //@line 22176
      FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($1059 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($1059, 0); //@line 22182
      $$9 = 0; //@line 22183
      STACKTOP = sp; //@line 22184
      return $$9 | 0; //@line 22184
     }
    case 6:
     {
      $$9 = -20; //@line 22188
      STACKTOP = sp; //@line 22189
      return $$9 | 0; //@line 22189
     }
    default:
     {
      HEAP32[$vararg_buffer >> 2] = $11; //@line 22193
      _printf(96163, $vararg_buffer) | 0; //@line 22194
      $$9 = -17; //@line 22195
      STACKTOP = sp; //@line 22196
      return $$9 | 0; //@line 22196
     }
    }
   } while (0);
  }
 }
 return 0; //@line 22202
}
function __Z13AudioDispatchjj($0, $1) {
 $0 = $0 | 0;
 $1 = $1 | 0;
 var $$0187 = 0, $$03450$i = 0, $$03549$i = 0, $$052$i = 0, $$07885$i = 0, $$07984$i = 0, $$087$i = 0, $$3$i = 0, $$3$i190 = 0, $$in$i = 0, $$in91$i = 0, $$in92$i = 0, $1004 = 0, $1011 = 0, $105 = 0, $111 = 0, $117 = 0, $123 = 0, $132 = 0, $138 = 0, $144 = 0, $150 = 0, $156 = 0, $162 = 0, $168 = 0, $17 = 0, $174 = 0, $183 = 0, $189 = 0, $195 = 0, $2 = 0, $201 = 0, $207 = 0, $213 = 0, $219 = 0, $225 = 0, $231 = 0, $237 = 0, $24 = 0, $243 = 0, $252 = 0, $258 = 0, $264 = 0, $270 = 0, $276 = 0, $282 = 0, $288 = 0, $294 = 0, $3 = 0, $30 = 0, $300 = 0, $306 = 0, $312 = 0, $321 = 0, $327 = 0, $333 = 0, $339 = 0, $345 = 0, $351 = 0, $357 = 0, $36 = 0, $363 = 0, $369 = 0, $375 = 0, $381 = 0, $390 = 0, $396 = 0, $4 = 0, $402 = 0, $408 = 0, $414 = 0, $42 = 0, $420 = 0, $426 = 0, $432 = 0, $438 = 0, $447 = 0, $453 = 0, $459 = 0, $465 = 0, $471 = 0, $477 = 0, $48 = 0, $483 = 0, $489 = 0, $495 = 0, $5 = 0, $505 = 0, $508 = 0, $510 = 0, $512 = 0, $514 = 0, $516 = 0, $531 = 0, $538 = 0, $54 = 0, $548 = 0, $556 = 0, $564 = 0, $572 = 0, $580 = 0, $587 = 0, $594 = 0, $6 = 0, $60 = 0, $600 = 0, $607 = 0, $618 = 0, $633 = 0, $634 = 0, $639 = 0, $640 = 0, $645 = 0, $656 = 0, $667 = 0, $677 = 0, $679 = 0, $683 = 0, $69 = 0, $708 = 0, $719 = 0, $729 = 0, $731 = 0, $735 = 0, $75 = 0, $760 = 0, $771 = 0, $781 = 0, $783 = 0, $787 = 0, $802 = 0, $808 = 0, $81 = 0, $815 = 0, $821 = 0, $845 = 0, $854 = 0, $861 = 0, $868 = 0, $87 = 0, $875 = 0, $882 = 0, $888 = 0, $894 = 0, $906 = 0, $907 = 0, $912 = 0, $913 = 0, $918 = 0, $923 = 0, $924 = 0, $926 = 0, $928 = 0, $93 = 0, $933 = 0, $935 = 0, $940 = 0, $941 = 0, $943 = 0, $945 = 0, $949 = 0, $951 = 0, $956 = 0, $957 = 0, $959 = 0, $961 = 0, $966 = 0, $968 = 0, $969 = 0, $977 = 0, $99 = 0, $sext = 0, label = 0, sp = 0;
 sp = STACKTOP; //@line 26693
 STACKTOP = STACKTOP + 208 | 0; //@line 26694
 $2 = sp + 136 | 0; //@line 26695
 $3 = sp + 68 | 0; //@line 26696
 $4 = sp; //@line 26697
 $5 = $0 + 4 | 0; //@line 26698
 $6 = $0 + 2 | 0; //@line 26699
 $sext = (FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($6 >>> 16 << 2) >> 2] | 0) + 4 >> 2] & 127]($6) | 0) << 16; //@line 26706
 do {
  switch ($sext >> 16 | 0) {
  case -1:
   {
    if (!(HEAP32[30824] | 0)) {
     HEAP32[$4 >> 2] = 204; //@line 26714
     _Execute68kTrap(-24512, $4); //@line 26715
     HEAP32[$4 >> 2] = 204; //@line 26716
     _Execute68kTrap(-23778, $4); //@line 26717
     $17 = HEAP32[$4 + 32 >> 2] | 0; //@line 26719
     if (!$17) {
      $$0187 = -108; //@line 26722
      STACKTOP = sp; //@line 26723
      return $$0187 | 0; //@line 26723
     }
     HEAP32[30824] = $17; //@line 26725
     FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($17 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($17, 22927); //@line 26731
     $24 = $17 + 2 | 0; //@line 26732
     FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($24 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($24, 12041); //@line 26738
     $30 = $17 + 4 | 0; //@line 26739
     FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($30 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($30, 12040); //@line 26745
     $36 = $17 + 6 | 0; //@line 26746
     FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($36 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($36, 28708); //@line 26752
     $42 = $17 + 8 | 0; //@line 26753
     FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($42 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($42, 43050); //@line 26759
     $48 = $17 + 10 | 0; //@line 26760
     FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($48 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($48, 8223); //@line 26766
     $54 = $17 + 12 | 0; //@line 26767
     FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($54 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($54, 20085); //@line 26773
     $60 = $17 + 14 | 0; //@line 26774
     if (($60 - (HEAP32[30824] | 0) | 0) != 14) {
      _puts(96528) | 0; //@line 26779
      __Z12QuitEmulatorv(); //@line 26780
     }
     FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($60 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($60, 21903); //@line 26788
     $69 = $17 + 16 | 0; //@line 26789
     FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($69 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($69, 12041); //@line 26795
     $75 = $17 + 18 | 0; //@line 26796
     FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($75 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($75, 12032); //@line 26802
     $81 = $17 + 20 | 0; //@line 26803
     FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($81 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($81, 12040); //@line 26809
     $87 = $17 + 22 | 0; //@line 26810
     FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($87 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($87, 8252); //@line 26816
     $93 = $17 + 24 | 0; //@line 26817
     FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($93 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($93, 101974040); //@line 26823
     $99 = $17 + 28 | 0; //@line 26824
     FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($99 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($99, 43008); //@line 26830
     $105 = $17 + 30 | 0; //@line 26831
     FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($105 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($105, 12319); //@line 26837
     $111 = $17 + 32 | 0; //@line 26838
     FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($111 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($111, 18624); //@line 26844
     $117 = $17 + 34 | 0; //@line 26845
     FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($117 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($117, 20085); //@line 26851
     $123 = $17 + 36 | 0; //@line 26852
     if (($123 - (HEAP32[30824] | 0) | 0) != 36) {
      _puts(96528) | 0; //@line 26857
      __Z12QuitEmulatorv(); //@line 26858
     }
     FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($123 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($123, 21903); //@line 26866
     $132 = $17 + 38 | 0; //@line 26867
     FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($132 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($132, 12040); //@line 26873
     $138 = $17 + 40 | 0; //@line 26874
     FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($138 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($138, 8252); //@line 26880
     $144 = $17 + 42 | 0; //@line 26881
     FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($144 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($144, 35127320); //@line 26887
     $150 = $17 + 46 | 0; //@line 26888
     FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($150 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($150, 43008); //@line 26894
     $156 = $17 + 48 | 0; //@line 26895
     FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($156 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($156, 12319); //@line 26901
     $162 = $17 + 50 | 0; //@line 26902
     FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($162 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($162, 18624); //@line 26908
     $168 = $17 + 52 | 0; //@line 26909
     FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($168 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($168, 20085); //@line 26915
     $174 = $17 + 54 | 0; //@line 26916
     if (($174 - (HEAP32[30824] | 0) | 0) != 54) {
      _puts(96528) | 0; //@line 26921
      __Z12QuitEmulatorv(); //@line 26922
     }
     FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($174 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($174, 22927); //@line 26930
     $183 = $17 + 56 | 0; //@line 26931
     FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($183 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($183, 12042); //@line 26937
     $189 = $17 + 58 | 0; //@line 26938
     FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($189 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($189, 12041); //@line 26944
     $195 = $17 + 60 | 0; //@line 26945
     FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($195 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($195, 12032); //@line 26951
     $201 = $17 + 62 | 0; //@line 26952
     FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($201 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($201, 12040); //@line 26958
     $207 = $17 + 64 | 0; //@line 26959
     FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($207 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($207, 12092); //@line 26965
     $213 = $17 + 66 | 0; //@line 26966
     FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($213 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($213, 786691); //@line 26972
     $219 = $17 + 70 | 0; //@line 26973
     FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($219 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($219, 28672); //@line 26979
     $225 = $17 + 72 | 0; //@line 26980
     FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($225 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($225, 43050); //@line 26986
     $231 = $17 + 74 | 0; //@line 26987
     FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($231 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($231, 8223); //@line 26993
     $237 = $17 + 76 | 0; //@line 26994
     FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($237 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($237, 20085); //@line 27000
     $243 = $17 + 78 | 0; //@line 27001
     if (($243 - (HEAP32[30824] | 0) | 0) != 78) {
      _puts(96528) | 0; //@line 27006
      __Z12QuitEmulatorv(); //@line 27007
     }
     FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($243 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($243, 22927); //@line 27015
     $252 = $17 + 80 | 0; //@line 27016
     FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($252 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($252, 12042); //@line 27022
     $258 = $17 + 82 | 0; //@line 27023
     FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($258 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($258, 12041); //@line 27029
     $264 = $17 + 84 | 0; //@line 27030
     FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($264 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($264, 12032); //@line 27036
     $270 = $17 + 86 | 0; //@line 27037
     FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($270 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($270, 12040); //@line 27043
     $276 = $17 + 88 | 0; //@line 27044
     FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($276 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($276, 12092); //@line 27050
     $282 = $17 + 90 | 0; //@line 27051
     FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($282 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($282, 786692); //@line 27057
     $288 = $17 + 94 | 0; //@line 27058
     FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($288 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($288, 28672); //@line 27064
     $294 = $17 + 96 | 0; //@line 27065
     FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($294 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($294, 43050); //@line 27071
     $300 = $17 + 98 | 0; //@line 27072
     FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($300 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($300, 8223); //@line 27078
     $306 = $17 + 100 | 0; //@line 27079
     FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($306 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($306, 20085); //@line 27085
     $312 = $17 + 102 | 0; //@line 27086
     if (($312 - (HEAP32[30824] | 0) | 0) != 102) {
      _puts(96528) | 0; //@line 27091
      __Z12QuitEmulatorv(); //@line 27092
     }
     FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($312 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($312, 22927); //@line 27100
     $321 = $17 + 104 | 0; //@line 27101
     FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($321 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($321, 12042); //@line 27107
     $327 = $17 + 106 | 0; //@line 27108
     FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($327 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($327, 12041); //@line 27114
     $333 = $17 + 108 | 0; //@line 27115
     FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($333 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($333, 12040); //@line 27121
     $339 = $17 + 110 | 0; //@line 27122
     FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($339 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($339, 12032); //@line 27128
     $345 = $17 + 112 | 0; //@line 27129
     FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($345 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($345, 12092); //@line 27135
     $351 = $17 + 114 | 0; //@line 27136
     FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($351 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($351, 786696); //@line 27142
     $357 = $17 + 118 | 0; //@line 27143
     FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($357 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($357, 28672); //@line 27149
     $363 = $17 + 120 | 0; //@line 27150
     FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($363 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($363, 43050); //@line 27156
     $369 = $17 + 122 | 0; //@line 27157
     FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($369 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($369, 8223); //@line 27163
     $375 = $17 + 124 | 0; //@line 27164
     FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($375 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($375, 20085); //@line 27170
     $381 = $17 + 126 | 0; //@line 27171
     if (($381 - (HEAP32[30824] | 0) | 0) != 126) {
      _puts(96528) | 0; //@line 27176
      __Z12QuitEmulatorv(); //@line 27177
     }
     FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($381 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($381, 22927); //@line 27185
     $390 = $17 + 128 | 0; //@line 27186
     FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($390 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($390, 12041); //@line 27192
     $396 = $17 + 130 | 0; //@line 27193
     FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($396 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($396, 12040); //@line 27199
     $402 = $17 + 132 | 0; //@line 27200
     FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($402 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($402, 12092); //@line 27206
     $408 = $17 + 134 | 0; //@line 27207
     FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($408 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($408, 262148); //@line 27213
     $414 = $17 + 138 | 0; //@line 27214
     FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($414 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($414, 28672); //@line 27220
     $420 = $17 + 140 | 0; //@line 27221
     FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($420 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($420, 43050); //@line 27227
     $426 = $17 + 142 | 0; //@line 27228
     FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($426 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($426, 8223); //@line 27234
     $432 = $17 + 144 | 0; //@line 27235
     FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($432 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($432, 20085); //@line 27241
     $438 = $17 + 146 | 0; //@line 27242
     if (($438 - (HEAP32[30824] | 0) | 0) != 146) {
      _puts(96528) | 0; //@line 27247
      __Z12QuitEmulatorv(); //@line 27248
     }
     FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($438 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($438, 22927); //@line 27256
     $447 = $17 + 148 | 0; //@line 27257
     FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($447 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($447, 12041); //@line 27263
     $453 = $17 + 150 | 0; //@line 27264
     FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($453 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($453, 16128); //@line 27270
     $459 = $17 + 152 | 0; //@line 27271
     FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($459 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($459, 12040); //@line 27277
     $465 = $17 + 154 | 0; //@line 27278
     FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($465 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($465, 12092); //@line 27284
     $471 = $17 + 156 | 0; //@line 27285
     FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($471 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($471, 393477); //@line 27291
     $477 = $17 + 160 | 0; //@line 27292
     FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($477 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($477, 28672); //@line 27298
     $483 = $17 + 162 | 0; //@line 27299
     FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($483 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($483, 43050); //@line 27305
     $489 = $17 + 164 | 0; //@line 27306
     FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($489 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($489, 8223); //@line 27312
     $495 = $17 + 166 | 0; //@line 27313
     FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($495 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($495, 20085); //@line 27319
     if (($17 + 168 - (HEAP32[30824] | 0) | 0) != 168) {
      _puts(96528) | 0; //@line 27325
      __Z12QuitEmulatorv(); //@line 27326
     }
    }
    HEAP8[657357] = 1; //@line 27330
    $505 = HEAP32[30825] | 0; //@line 27331
    if (!$505) {
     $508 = HEAP32[30825] | 0; //@line 27335
    } else {
     $508 = $505; //@line 27337
    }
    HEAP32[30825] = $508 + 1; //@line 27340
    $$0187 = 0; //@line 27341
    STACKTOP = sp; //@line 27342
    return $$0187 | 0; //@line 27342
   }
  case -2:
   {
    $510 = (HEAP32[30825] | 0) + -1 | 0; //@line 27347
    HEAP32[30825] = $510; //@line 27348
    if ($510 | 0) {
     $$0187 = 0; //@line 27351
     STACKTOP = sp; //@line 27352
     return $$0187 | 0; //@line 27352
    }
    $512 = HEAP32[30824] | 0; //@line 27354
    do {
     if ($512 | 0) {
      $514 = HEAP32[30820] | 0; //@line 27358
      $516 = $4 + 32 | 0; //@line 27360
      if (!$514) {
       HEAP32[$516 >> 2] = $512; //@line 27362
       _Execute68kTrap(-24545, $4); //@line 27363
       HEAP32[30824] = 0; //@line 27364
       break;
      }
      HEAP32[$516 >> 2] = $514; //@line 27367
      _Execute68k($512 + 36 | 0, $4); //@line 27369
      HEAP32[30820] = 0; //@line 27370
      $$0187 = HEAP32[$4 >> 2] | 0; //@line 27372
      STACKTOP = sp; //@line 27373
      return $$0187 | 0; //@line 27373
     }
    } while (0);
    HEAP32[30821] = 0; //@line 27376
    $$0187 = 0; //@line 27377
    STACKTOP = sp; //@line 27378
    return $$0187 | 0; //@line 27378
   }
  case -3:
   {
    switch ((FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($5 >>> 16 << 2) >> 2] | 0) + 4 >> 2] & 127]($5) | 0) << 16 >> 16 | 0) {
    case 261:
    case 260:
    case 259:
    case 3:
    case 1:
    case -5:
    case -4:
    case -3:
    case -2:
    case -1:
     {
      $$0187 = 1; //@line 27392
      STACKTOP = sp; //@line 27393
      return $$0187 | 0; //@line 27393
     }
    default:
     {}
    }
    $$0187 = 0; //@line 27399
    STACKTOP = sp; //@line 27400
    return $$0187 | 0; //@line 27400
   }
  case -5:
   {
    $$0187 = 0; //@line 27404
    STACKTOP = sp; //@line 27405
    return $$0187 | 0; //@line 27405
   }
  case 1:
   {
    if (!(HEAP8[657356] | 0)) {
     $$0187 = -200; //@line 27412
     STACKTOP = sp; //@line 27413
     return $$0187 | 0; //@line 27413
    }
    if (HEAP32[30820] | 0) {
     $$0187 = 0; //@line 27418
     STACKTOP = sp; //@line 27419
     return $$0187 | 0; //@line 27419
    }
    $531 = (HEAP32[30824] | 0) + 168 | 0; //@line 27422
    FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($531 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($531, 0); //@line 27428
    $538 = (HEAP32[30824] | 0) + 172 | 0; //@line 27430
    FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($538 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($538, (HEAP32[30818] | 0) == 16 ? 1953984371 : 1918990112); //@line 27439
    $548 = (HEAP32[30824] | 0) + 176 | 0; //@line 27441
    FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($548 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($548, HEAP32[30819] | 0); //@line 27448
    $556 = (HEAP32[30824] | 0) + 178 | 0; //@line 27450
    FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($556 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($556, HEAP32[30818] | 0); //@line 27457
    $564 = (HEAP32[30824] | 0) + 180 | 0; //@line 27459
    FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($564 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($564, HEAP32[30817] | 0); //@line 27466
    $572 = (HEAP32[30824] | 0) + 184 | 0; //@line 27468
    FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($572 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($572, HEAP32[30822] | 0); //@line 27475
    $580 = (HEAP32[30824] | 0) + 188 | 0; //@line 27477
    FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($580 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($580, 0); //@line 27483
    $587 = (HEAP32[30824] | 0) + 192 | 0; //@line 27485
    FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($587 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($587, 0); //@line 27491
    $594 = (HEAP32[30824] | 0) + 200 | 0; //@line 27493
    FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($594 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($594, 0); //@line 27499
    $600 = HEAP32[30824] | 0; //@line 27500
    HEAP32[$4 + 32 >> 2] = $600 + 196; //@line 27503
    HEAP32[$4 >> 2] = 0; //@line 27504
    HEAP32[$4 + 36 >> 2] = $600 + 168; //@line 27507
    _Execute68k($600 + 14 | 0, $4); //@line 27509
    $607 = (HEAP32[30824] | 0) + 196 | 0; //@line 27511
    HEAP32[30820] = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($607 >>> 16 << 2) >> 2] >> 2] & 127]($607) | 0; //@line 27517
    $$0187 = HEAP32[$4 >> 2] | 0; //@line 27519
    STACKTOP = sp; //@line 27520
    return $$0187 | 0; //@line 27520
   }
  case 3:
   {
    $618 = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($5 >>> 16 << 2) >> 2] >> 2] & 127]($5) | 0; //@line 27528
    FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($618 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($618, HEAP32[30820] | 0); //@line 27535
    $$0187 = 0; //@line 27536
    STACKTOP = sp; //@line 27537
    return $$0187 | 0; //@line 27537
   }
  case 257:
   {
    HEAP32[30821] = (HEAP32[30821] | 0) + 1; //@line 27543
    break;
   }
  case 258:
   {
    HEAP32[30821] = (HEAP32[30821] | 0) + -1; //@line 27549
    break;
   }
  case 259:
   {
    $633 = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($5 >>> 16 << 2) >> 2] >> 2] & 127]($5) | 0; //@line 27557
    $634 = $0 + 8 | 0; //@line 27558
    $639 = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($634 >>> 16 << 2) >> 2] >> 2] & 127]($634) | 0; //@line 27563
    $640 = $0 + 12 | 0; //@line 27564
    $645 = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($640 >>> 16 << 2) >> 2] >> 2] & 127]($640) | 0; //@line 27569
    L69 : do {
     if (($639 | 0) < 1752655469) {
      if (($639 | 0) >= 1752396912) {
       if (($639 | 0) < 1752592236) {
        switch ($639 | 0) {
        case 1752396912:
         {
          break;
         }
        default:
         {
          label = 51; //@line 27582
          break L69;
         }
        }
        FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($633 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($633, 7); //@line 27591
        $$3$i = 0; //@line 27592
        break;
       }
       if (($639 | 0) < 1752654451) {
        switch ($639 | 0) {
        case 1752592236:
         {
          break;
         }
        default:
         {
          label = 51; //@line 27602
          break L69;
         }
        }
        $821 = __Z21audio_get_main_volumev() | 0; //@line 27606
        FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($633 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($633, $821); //@line 27612
        $$3$i = 0; //@line 27613
        break;
       } else {
        switch ($639 | 0) {
        case 1752654451:
         {
          break;
         }
        default:
         {
          label = 51; //@line 27621
          break L69;
         }
        }
        FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($633 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($633, (HEAP32[30821] | 0) != 0 & 1); //@line 27633
        $$3$i = 0; //@line 27634
        break;
       }
      }
      if (($639 | 0) < 1667785078) {
       switch ($639 | 0) {
       case 1667785070:
        {
         break;
        }
       default:
        {
         label = 51; //@line 27645
         break L69;
        }
       }
       FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($633 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($633, HEAP32[30819] | 0); //@line 27655
       $$3$i = 0; //@line 27656
       break;
      }
      if (($639 | 0) >= 1752003956) {
       switch ($639 | 0) {
       case 1752003956:
        {
         break;
        }
       default:
        {
         label = 51; //@line 27666
         break L69;
        }
       }
       $815 = (__Z19audio_get_main_mutev() | 0) & 1; //@line 27671
       FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($633 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($633, $815); //@line 27677
       $$3$i = 0; //@line 27678
       break;
      }
      switch ($639 | 0) {
      case 1667785078:
       {
        break;
       }
      default:
       {
        label = 51; //@line 27686
        break L69;
       }
      }
      HEAP32[$2 >> 2] = (HEAP32[30815] | 0) - (HEAP32[30814] | 0); //@line 27693
      _Execute68kTrap(-24286, $2); //@line 27694
      $708 = HEAP32[$2 + 32 >> 2] | 0; //@line 27696
      if (!$708) {
       $$3$i = -108; //@line 27699
      } else {
       FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($633 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($633, (HEAP32[30815] | 0) - (HEAP32[30814] | 0) >> 1); //@line 27710
       $719 = $633 + 2 | 0; //@line 27711
       FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($719 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($719, $708); //@line 27717
       $729 = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($708 >>> 16 << 2) >> 2] >> 2] & 127]($708) | 0; //@line 27722
       $731 = HEAP32[30814] | 0; //@line 27724
       if ((HEAP32[30815] | 0) == ($731 | 0)) {
        $$3$i = 0; //@line 27727
       } else {
        $$07885$i = 0; //@line 27729
        $$in91$i = $731; //@line 27729
        do {
         $735 = ($$07885$i << 1) + $729 | 0; //@line 27733
         FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($735 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($735, HEAPU16[$$in91$i + ($$07885$i << 1) >> 1] | 0); //@line 27742
         $$07885$i = $$07885$i + 1 | 0; //@line 27743
         $$in91$i = HEAP32[30814] | 0; //@line 27745
        } while ($$07885$i >>> 0 < (HEAP32[30815] | 0) - $$in91$i >> 1 >>> 0);
        $$3$i = 0; //@line 27752
       }
      }
     } else {
      if (($639 | 0) < 1936875894) {
       if (($639 | 0) < 1936553332) {
        switch ($639 | 0) {
        case 1752655469:
         {
          break;
         }
        default:
         {
          label = 51; //@line 27768
          break L69;
         }
        }
        FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($633 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($633, 0); //@line 27777
        $845 = $633 + 4 | 0; //@line 27778
        FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($845 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($845, (HEAP32[30818] | 0) == 16 ? 1953984371 : 1918990112); //@line 27787
        $854 = $633 + 8 | 0; //@line 27788
        FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($854 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($854, HEAP32[30819] | 0); //@line 27795
        $861 = $633 + 10 | 0; //@line 27796
        FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($861 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($861, HEAP32[30818] | 0); //@line 27803
        $868 = $633 + 12 | 0; //@line 27804
        FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($868 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($868, HEAP32[30817] | 0); //@line 27811
        $875 = $633 + 16 | 0; //@line 27812
        FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($875 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($875, HEAP32[30822] | 0); //@line 27819
        $882 = $633 + 20 | 0; //@line 27820
        FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($882 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($882, 0); //@line 27826
        $888 = $633 + 24 | 0; //@line 27827
        FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($888 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($888, 0); //@line 27833
        $$3$i = 0; //@line 27834
        break;
       }
       if (($639 | 0) < 1936875892) {
        switch ($639 | 0) {
        case 1936553332:
         {
          break;
         }
        default:
         {
          label = 51; //@line 27844
          break L69;
         }
        }
        $802 = (__Z22audio_get_speaker_mutev() | 0) & 1; //@line 27849
        FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($633 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($633, $802); //@line 27855
        $$3$i = 0; //@line 27856
        break;
       } else {
        switch ($639 | 0) {
        case 1936875892:
         {
          break;
         }
        default:
         {
          label = 51; //@line 27864
          break L69;
         }
        }
        FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($633 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($633, HEAP32[30817] | 0); //@line 27874
        $$3$i = 0; //@line 27875
        break;
       }
      }
      if (($639 | 0) >= 1936943482) {
       if (($639 | 0) < 1937141612) {
        switch ($639 | 0) {
        case 1936943482:
         {
          break;
         }
        default:
         {
          label = 51; //@line 27888
          break L69;
         }
        }
        FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($633 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($633, HEAP32[30818] | 0); //@line 27898
        $$3$i = 0; //@line 27899
        break;
       } else {
        switch ($639 | 0) {
        case 1937141612:
         {
          break;
         }
        default:
         {
          label = 51; //@line 27907
          break L69;
         }
        }
        $808 = __Z24audio_get_speaker_volumev() | 0; //@line 27911
        FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($633 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($633, $808); //@line 27917
        $$3$i = 0; //@line 27918
        break;
       }
      }
      if (($639 | 0) < 1936941430) {
       switch ($639 | 0) {
       case 1936875894:
        {
         break;
        }
       default:
        {
         label = 51; //@line 27929
         break L69;
        }
       }
       HEAP32[$2 >> 2] = (HEAP32[30809] | 0) - (HEAP32[30808] | 0); //@line 27936
       _Execute68kTrap(-24286, $2); //@line 27937
       $760 = HEAP32[$2 + 32 >> 2] | 0; //@line 27939
       if (!$760) {
        $$3$i = -108; //@line 27942
        break;
       }
       FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($633 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($633, (HEAP32[30809] | 0) - (HEAP32[30808] | 0) >> 2); //@line 27954
       $771 = $633 + 2 | 0; //@line 27955
       FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($771 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($771, $760); //@line 27961
       $781 = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($760 >>> 16 << 2) >> 2] >> 2] & 127]($760) | 0; //@line 27966
       $783 = HEAP32[30808] | 0; //@line 27968
       if ((HEAP32[30809] | 0) == ($783 | 0)) {
        $$3$i = 0; //@line 27971
        break;
       } else {
        $$087$i = 0; //@line 27974
        $$in$i = $783; //@line 27974
       }
       do {
        $787 = ($$087$i << 2) + $781 | 0; //@line 27979
        FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($787 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($787, HEAP32[$$in$i + ($$087$i << 2) >> 2] | 0); //@line 27987
        $$087$i = $$087$i + 1 | 0; //@line 27988
        $$in$i = HEAP32[30808] | 0; //@line 27990
       } while ($$087$i >>> 0 < (HEAP32[30809] | 0) - $$in$i >> 2 >>> 0);
       $$3$i = 0; //@line 27997
      } else {
       switch ($639 | 0) {
       case 1936941430:
        {
         break;
        }
       default:
        {
         label = 51; //@line 28007
         break L69;
        }
       }
       HEAP32[$2 >> 2] = (HEAP32[30812] | 0) - (HEAP32[30811] | 0); //@line 28014
       _Execute68kTrap(-24286, $2); //@line 28015
       $656 = HEAP32[$2 + 32 >> 2] | 0; //@line 28017
       if (!$656) {
        $$3$i = -108; //@line 28020
        break;
       }
       FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($633 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($633, (HEAP32[30812] | 0) - (HEAP32[30811] | 0) >> 1); //@line 28032
       $667 = $633 + 2 | 0; //@line 28033
       FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($667 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($667, $656); //@line 28039
       $677 = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($656 >>> 16 << 2) >> 2] >> 2] & 127]($656) | 0; //@line 28044
       $679 = HEAP32[30811] | 0; //@line 28046
       if ((HEAP32[30812] | 0) == ($679 | 0)) {
        $$3$i = 0; //@line 28049
        break;
       } else {
        $$07984$i = 0; //@line 28052
        $$in92$i = $679; //@line 28052
       }
       do {
        $683 = ($$07984$i << 1) + $677 | 0; //@line 28057
        FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($683 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($683, HEAPU16[$$in92$i + ($$07984$i << 1) >> 1] | 0); //@line 28066
        $$07984$i = $$07984$i + 1 | 0; //@line 28067
        $$in92$i = HEAP32[30811] | 0; //@line 28069
       } while ($$07984$i >>> 0 < (HEAP32[30812] | 0) - $$in92$i >> 1 >>> 0);
       $$3$i = 0; //@line 28076
      }
     }
    } while (0);
    if ((label | 0) == 51) {
     $894 = HEAP32[30820] | 0; //@line 28084
     if (!$894) {
      $$3$i = -2147450878; //@line 28087
     } else {
      HEAP32[$3 + 32 >> 2] = $633; //@line 28090
      HEAP32[$3 >> 2] = $639; //@line 28091
      HEAP32[$3 + 36 >> 2] = $645; //@line 28093
      HEAP32[$3 + 40 >> 2] = $894; //@line 28095
      _Execute68k((HEAP32[30824] | 0) + 54 | 0, $3); //@line 28098
      $$3$i = HEAP32[$3 >> 2] | 0; //@line 28100
     }
    }
    $$0187 = $$3$i; //@line 28103
    STACKTOP = sp; //@line 28104
    return $$0187 | 0; //@line 28104
   }
  case 260:
   {
    $906 = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($5 >>> 16 << 2) >> 2] >> 2] & 127]($5) | 0; //@line 28112
    $907 = $0 + 8 | 0; //@line 28113
    $912 = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($907 >>> 16 << 2) >> 2] >> 2] & 127]($907) | 0; //@line 28118
    $913 = $0 + 12 | 0; //@line 28119
    $918 = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($913 >>> 16 << 2) >> 2] >> 2] & 127]($913) | 0; //@line 28124
    L133 : do {
     if (($912 | 0) < 1936553332) {
      if (($912 | 0) >= 1752003956) {
       if (($912 | 0) < 1752592236) {
        switch ($912 | 0) {
        case 1752003956:
         {
          break;
         }
        default:
         {
          label = 80; //@line 28137
          break L133;
         }
        }
        $$3$i190 = 0; //@line 28141
        break;
       } else {
        switch ($912 | 0) {
        case 1752592236:
         {
          break;
         }
        default:
         {
          label = 80; //@line 28149
          break L133;
         }
        }
        $$3$i190 = 0; //@line 28153
        break;
       }
      }
      switch ($912 | 0) {
      case 1667785070:
       {
        break;
       }
      default:
       {
        label = 80; //@line 28162
        break L133;
       }
      }
      if (!(HEAP32[30821] | 0)) {
       if ((HEAP32[30819] | 0) == ($906 | 0)) {
        $$3$i190 = 0; //@line 28172
       } else {
        $956 = HEAP32[30815] | 0; //@line 28174
        $957 = HEAP32[30814] | 0; //@line 28175
        $959 = $957; //@line 28177
        if (($956 | 0) == ($957 | 0)) {
         $$3$i190 = -205; //@line 28179
        } else {
         $961 = $956 - $957 >> 1; //@line 28182
         $$052$i = 0; //@line 28183
         while (1) {
          if ((HEAPU16[$959 + ($$052$i << 1) >> 1] | 0) == ($906 | 0)) {
           break;
          }
          $966 = $$052$i + 1 | 0; //@line 28192
          if ($966 >>> 0 < $961 >>> 0) {
           $$052$i = $966; //@line 28195
          } else {
           $$3$i190 = -205; //@line 28197
           break L133;
          }
         }
         $968 = __Z18audio_set_channelsi($$052$i) | 0; //@line 28201
         $$3$i190 = $968 ? 0 : -205; //@line 28203
        }
       }
      } else {
       $$3$i190 = -227; //@line 28207
      }
     } else {
      if (($912 | 0) < 1936943482) {
       if (($912 | 0) < 1936875892) {
        switch ($912 | 0) {
        case 1936553332:
         {
          break;
         }
        default:
         {
          label = 80; //@line 28219
          break L133;
         }
        }
        $$3$i190 = 0; //@line 28223
        break;
       }
       switch ($912 | 0) {
       case 1936875892:
        {
         break;
        }
       default:
        {
         label = 80; //@line 28231
         break L133;
        }
       }
       if (HEAP32[30821] | 0) {
        $$3$i190 = -227; //@line 28238
        break;
       }
       if ((HEAP32[30817] | 0) == ($906 | 0)) {
        $$3$i190 = 0; //@line 28244
        break;
       }
       $940 = HEAP32[30809] | 0; //@line 28247
       $941 = HEAP32[30808] | 0; //@line 28248
       $943 = $941; //@line 28250
       if (($940 | 0) == ($941 | 0)) {
        $$3$i190 = -225; //@line 28252
        break;
       }
       $945 = $940 - $941 >> 2; //@line 28256
       $$03450$i = 0; //@line 28257
       while (1) {
        if ((HEAP32[$943 + ($$03450$i << 2) >> 2] | 0) == ($906 | 0)) {
         break;
        }
        $949 = $$03450$i + 1 | 0; //@line 28265
        if ($949 >>> 0 < $945 >>> 0) {
         $$03450$i = $949; //@line 28268
        } else {
         $$3$i190 = -225; //@line 28270
         break L133;
        }
       }
       $951 = __Z21audio_set_sample_ratei($$03450$i) | 0; //@line 28274
       $$3$i190 = $951 ? 0 : -225; //@line 28276
       break;
      } else {
       if (($912 | 0) >= 1937141612) {
        switch ($912 | 0) {
        case 1937141612:
         {
          break;
         }
        default:
         {
          label = 80; //@line 28286
          break L133;
         }
        }
        $$3$i190 = 0; //@line 28290
        break;
       }
       switch ($912 | 0) {
       case 1936943482:
        {
         break;
        }
       default:
        {
         label = 80; //@line 28298
         break L133;
        }
       }
       if (HEAP32[30821] | 0) {
        $$3$i190 = -227; //@line 28305
        break;
       }
       if ((HEAP32[30818] | 0) == ($906 | 0)) {
        $$3$i190 = 0; //@line 28311
        break;
       }
       $923 = HEAP32[30812] | 0; //@line 28314
       $924 = HEAP32[30811] | 0; //@line 28315
       $926 = $924; //@line 28317
       if (($923 | 0) == ($924 | 0)) {
        $$3$i190 = -226; //@line 28319
        break;
       }
       $928 = $923 - $924 >> 1; //@line 28323
       $$03549$i = 0; //@line 28324
       while (1) {
        if ((HEAPU16[$926 + ($$03549$i << 1) >> 1] | 0) == ($906 | 0)) {
         break;
        }
        $933 = $$03549$i + 1 | 0; //@line 28333
        if ($933 >>> 0 < $928 >>> 0) {
         $$03549$i = $933; //@line 28336
        } else {
         $$3$i190 = -226; //@line 28338
         break L133;
        }
       }
       $935 = __Z21audio_set_sample_sizei($$03549$i) | 0; //@line 28342
       $$3$i190 = $935 ? 0 : -226; //@line 28344
       break;
      }
     }
    } while (0);
    if ((label | 0) == 80) {
     $969 = HEAP32[30820] | 0; //@line 28350
     if (!$969) {
      $$3$i190 = -2147450878; //@line 28353
     } else {
      HEAP32[$2 + 32 >> 2] = $906; //@line 28356
      HEAP32[$2 >> 2] = $912; //@line 28357
      HEAP32[$2 + 36 >> 2] = $918; //@line 28359
      HEAP32[$2 + 40 >> 2] = $969; //@line 28361
      _Execute68k((HEAP32[30824] | 0) + 78 | 0, $2); //@line 28364
      $$3$i190 = HEAP32[$2 >> 2] | 0; //@line 28366
     }
    }
    $$0187 = $$3$i190; //@line 28369
    STACKTOP = sp; //@line 28370
    return $$0187 | 0; //@line 28370
   }
  case 261:
   {
    $977 = $0 + 8 | 0; //@line 28374
    HEAP32[$4 >> 2] = FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($977 >>> 16 << 2) >> 2] | 0) + 4 >> 2] & 127]($977) | 0; //@line 28381
    HEAP32[$4 + 32 >> 2] = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($5 >>> 16 << 2) >> 2] >> 2] & 127]($5) | 0; //@line 28388
    HEAP32[$4 + 36 >> 2] = HEAP32[30820]; //@line 28391
    _Execute68k((HEAP32[30824] | 0) + 146 | 0, $4); //@line 28394
    $$0187 = 0; //@line 28395
    STACKTOP = sp; //@line 28396
    return $$0187 | 0; //@line 28396
   }
  case 263:
  case 262:
   {
    break;
   }
  case 264:
   {
    HEAP32[$4 >> 2] = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($5 >>> 16 << 2) >> 2] >> 2] & 127]($5) | 0; //@line 28408
    $1004 = $0 + 8 | 0; //@line 28409
    HEAP32[$4 + 32 >> 2] = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($1004 >>> 16 << 2) >> 2] >> 2] & 127]($1004) | 0; //@line 28416
    $1011 = $0 + 12 | 0; //@line 28417
    HEAP32[$4 + 36 >> 2] = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($1011 >>> 16 << 2) >> 2] >> 2] & 127]($1011) | 0; //@line 28424
    HEAP32[$4 + 40 >> 2] = HEAP32[30820]; //@line 28427
    _Execute68k((HEAP32[30824] | 0) + 102 | 0, $4); //@line 28430
    $$0187 = HEAP32[$4 >> 2] | 0; //@line 28432
    STACKTOP = sp; //@line 28433
    return $$0187 | 0; //@line 28433
   }
  case -4:
   {
    $$0187 = 65539; //@line 28437
    STACKTOP = sp; //@line 28438
    return $$0187 | 0; //@line 28438
   }
  default:
   {
    if (($sext | 0) <= 16711680) {
     $$0187 = -2147450878; //@line 28444
     STACKTOP = sp; //@line 28445
     return $$0187 | 0; //@line 28445
    }
   }
  }
 } while (0);
 HEAP32[$4 + 32 >> 2] = HEAP32[30820]; //@line 28452
 HEAP32[$4 + 36 >> 2] = $0; //@line 28454
 _Execute68k(HEAP32[30824] | 0, $4); //@line 28456
 $$0187 = HEAP32[$4 >> 2] | 0; //@line 28458
 STACKTOP = sp; //@line 28459
 return $$0187 | 0; //@line 28459
}
function __Z6EmulOptP13M68kRegisters($0, $1) {
 $0 = $0 | 0;
 $1 = $1 | 0;
 var $$0$in = 0, $$0313 = 0, $$pre$phi326Z2D = 0, $102 = 0, $11 = 0, $110 = 0, $126 = 0, $127 = 0, $129 = 0, $13 = 0, $135 = 0, $138 = 0, $139 = 0, $143 = 0, $15 = 0, $151 = 0, $152 = 0, $153 = 0, $155 = 0, $157 = 0, $158 = 0, $165 = 0, $166 = 0, $17 = 0, $170 = 0, $175 = 0, $177 = 0, $188 = 0, $19 = 0, $190 = 0, $2 = 0, $200 = 0, $207 = 0, $209 = 0, $21 = 0, $214 = 0, $23 = 0, $230 = 0, $233 = 0, $245 = 0, $247 = 0, $25 = 0, $252 = 0, $27 = 0, $278 = 0, $281 = 0, $287 = 0, $288 = 0, $29 = 0, $31 = 0, $33 = 0, $36 = 0, $40 = 0, $468 = 0, $469 = 0, $47 = 0, $474 = 0, $476 = 0, $482 = 0, $483 = 0, $484 = 0, $486 = 0, $49 = 0, $493 = 0, $499 = 0, $5 = 0, $50 = 0, $508 = 0, $514 = 0, $520 = 0, $522 = 0, $527 = 0, $535 = 0, $541 = 0, $546 = 0, $548 = 0, $553 = 0, $555 = 0, $56 = 0, $562 = 0, $568 = 0, $575 = 0, $581 = 0, $588 = 0, $595 = 0, $601 = 0, $607 = 0, $614 = 0, $63 = 0, $631 = 0, $648 = 0, $652 = 0, $655 = 0, $658 = 0, $671 = 0, $673 = 0, $678 = 0, $685 = 0, $69 = 0, $691 = 0, $696 = 0, $698 = 0, $7 = 0, $703 = 0, $710 = 0, $716 = 0, $722 = 0, $724 = 0, $731 = 0, $733 = 0, $739 = 0, $747 = 0, $759 = 0, $760 = 0, $761 = 0, $762 = 0, $769 = 0, $771 = 0, $776 = 0, $778 = 0, $78 = 0, $785 = 0, $791 = 0, $792 = 0, $793 = 0, $794 = 0, $799 = 0, $801 = 0, $808 = 0, $810 = 0, $815 = 0, $817 = 0, $822 = 0, $824 = 0, $833 = 0, $840 = 0, $859 = 0, $861 = 0, $863 = 0, $865 = 0, $867 = 0, $869 = 0, $87 = 0, $871 = 0, $873 = 0, $875 = 0, $877 = 0, $879 = 0, $881 = 0, $883 = 0, $885 = 0, $887 = 0, $890 = 0, $9 = 0, $97 = 0, $vararg_buffer = 0, $vararg_buffer17 = 0, $vararg_buffer20 = 0, $vararg_buffer23 = 0, sp = 0;
 sp = STACKTOP; //@line 11179
 STACKTOP = STACKTOP + 224 | 0; //@line 11180
 $vararg_buffer23 = sp + 88 | 0; //@line 11181
 $vararg_buffer20 = sp + 80 | 0; //@line 11182
 $vararg_buffer17 = sp + 72 | 0; //@line 11183
 $vararg_buffer = sp; //@line 11184
 $2 = sp + 156 | 0; //@line 11185
 do {
  switch ($0 << 16 >> 16) {
  case 28929:
   {
    _puts(92920) | 0; //@line 11189
    $5 = HEAP32[$1 + 4 >> 2] | 0; //@line 11192
    $7 = HEAP32[$1 + 8 >> 2] | 0; //@line 11194
    $9 = HEAP32[$1 + 12 >> 2] | 0; //@line 11196
    $11 = HEAP32[$1 + 16 >> 2] | 0; //@line 11198
    $13 = HEAP32[$1 + 20 >> 2] | 0; //@line 11200
    $15 = HEAP32[$1 + 24 >> 2] | 0; //@line 11202
    $17 = HEAP32[$1 + 28 >> 2] | 0; //@line 11204
    $19 = HEAP32[$1 + 32 >> 2] | 0; //@line 11206
    $21 = HEAP32[$1 + 36 >> 2] | 0; //@line 11208
    $23 = HEAP32[$1 + 40 >> 2] | 0; //@line 11210
    $25 = HEAP32[$1 + 44 >> 2] | 0; //@line 11212
    $27 = HEAP32[$1 + 48 >> 2] | 0; //@line 11214
    $29 = HEAP32[$1 + 52 >> 2] | 0; //@line 11216
    $31 = HEAP32[$1 + 56 >> 2] | 0; //@line 11218
    $33 = HEAP32[$1 + 60 >> 2] | 0; //@line 11220
    $36 = HEAPU16[$1 + 64 >> 1] | 0; //@line 11223
    HEAP32[$vararg_buffer >> 2] = HEAP32[$1 >> 2]; //@line 11224
    HEAP32[$vararg_buffer + 4 >> 2] = $5; //@line 11226
    HEAP32[$vararg_buffer + 8 >> 2] = $7; //@line 11228
    HEAP32[$vararg_buffer + 12 >> 2] = $9; //@line 11230
    HEAP32[$vararg_buffer + 16 >> 2] = $11; //@line 11232
    HEAP32[$vararg_buffer + 20 >> 2] = $13; //@line 11234
    HEAP32[$vararg_buffer + 24 >> 2] = $15; //@line 11236
    HEAP32[$vararg_buffer + 28 >> 2] = $17; //@line 11238
    HEAP32[$vararg_buffer + 32 >> 2] = $19; //@line 11240
    HEAP32[$vararg_buffer + 36 >> 2] = $21; //@line 11242
    HEAP32[$vararg_buffer + 40 >> 2] = $23; //@line 11244
    HEAP32[$vararg_buffer + 44 >> 2] = $25; //@line 11246
    HEAP32[$vararg_buffer + 48 >> 2] = $27; //@line 11248
    HEAP32[$vararg_buffer + 52 >> 2] = $29; //@line 11250
    HEAP32[$vararg_buffer + 56 >> 2] = $31; //@line 11252
    HEAP32[$vararg_buffer + 60 >> 2] = $33; //@line 11254
    HEAP32[$vararg_buffer + 64 >> 2] = $36; //@line 11256
    _printf(92935, $vararg_buffer) | 0; //@line 11257
    __Z19VideoQuitFullScreenv(); //@line 11258
    __Z12QuitEmulatorv(); //@line 11259
    break;
   }
  case 28930:
   {
    __Z12QuitEmulatorv(); //@line 11264
    break;
   }
  case 28931:
   {
    __Z10TimerResetv(); //@line 11269
    __Z10EtherResetv(); //@line 11270
    __Z10AudioResetv(); //@line 11271
    $40 = (HEAP32[30893] | 0) + -4096 + (HEAP32[30895] | 0) | 0; //@line 11275
    _memset(FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($40 >>> 16 << 2) >> 2] | 0) + 24 >> 2] & 127]($40) | 0, 0, 4096) | 0; //@line 11282
    $47 = HEAP32[30893] | 0; //@line 11283
    $49 = (HEAP32[30895] | 0) + $47 | 0; //@line 11285
    $50 = $49 + -28 | 0; //@line 11286
    FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($50 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($50, $47); //@line 11292
    $56 = $49 + -24 | 0; //@line 11293
    FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($56 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($56, HEAP32[30895] | 0); //@line 11300
    $63 = $49 + -20 | 0; //@line 11301
    FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($63 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($63, -1); //@line 11307
    $69 = $49 + -16 | 0; //@line 11308
    FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($69 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($69, 0); //@line 11314
    $78 = (HEAP32[30896] | 0) + 24 + (HEAP32[28201] | 0) | 0; //@line 11318
    HEAP32[$1 >> 2] = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($78 >>> 16 << 2) >> 2] >> 2] & 127]($78) | 0; //@line 11324
    $87 = (HEAP32[30896] | 0) + 28 + (HEAP32[28201] | 0) | 0; //@line 11328
    HEAP32[$1 + 4 >> 2] = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($87 >>> 16 << 2) >> 2] >> 2] & 127]($87) | 0; //@line 11335
    $97 = (HEAP32[30896] | 0) + 16 + (HEAP32[28201] | 0) | 0; //@line 11339
    $102 = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($97 >>> 16 << 2) >> 2] >> 2] & 127]($97) | 0; //@line 11344
    HEAP32[$1 + 8 >> 2] = HEAP32[30881] | 0 ? $102 | 268435456 : $102 & -268435457; //@line 11351
    $110 = (HEAP32[28201] | 0) + (HEAP32[30896] | 0) | 0; //@line 11354
    HEAP32[$1 + 32 >> 2] = (FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($110 >>> 16 << 2) >> 2] >> 2] & 127]($110) | 0) + $110; //@line 11362
    HEAP32[$1 + 36 >> 2] = (HEAP32[28201] | 0) + (HEAP32[30896] | 0); //@line 11367
    HEAP32[$1 + 56 >> 2] = $50; //@line 11369
    HEAP32[$1 + 60 >> 2] = (HEAP32[30893] | 0) + 65536; //@line 11373
    STACKTOP = sp; //@line 11374
    return;
   }
  case 28932:
   {
    $126 = $1 + 4 | 0; //@line 11378
    $127 = HEAP32[$126 >> 2] | 0; //@line 11379
    $129 = ($127 & 128 | 0) != 0; //@line 11381
    L8 : do {
     if (($127 & 120 | 0) == 56) {
      $135 = $127 << 5 & 224 | $127 >>> 10 & 31; //@line 11390
      if (!$129) {
       $151 = $1 + 8 | 0; //@line 11396
       $152 = HEAP32[$151 >> 2] | 0; //@line 11397
       if (($135 | 0) == 138 & (HEAP8[659480] | 0) == 0) {
        $153 = $152 | 5; //@line 11399
        HEAP32[$151 >> 2] = $153; //@line 11400
        $155 = $153; //@line 11401
       } else {
        $155 = $152; //@line 11403
       }
       HEAP8[656025 + $135 >> 0] = $155; //@line 11407
       $$pre$phi326Z2D = $151; //@line 11408
       break;
      }
      $138 = HEAPU8[656025 + $135 >> 0] | 0; //@line 11413
      $139 = $1 + 8 | 0; //@line 11414
      HEAP32[$139 >> 2] = $138; //@line 11415
      $143 = (HEAP8[656250] | HEAP8[656249]) << 24 >> 24 == 0; //@line 11419
      switch (($135 & 255) << 24 >> 24) {
      case 8:
       {
        if ((HEAP16[325960] | 0) == 1660) {
         $$pre$phi326Z2D = $139; //@line 11426
         break L8;
        }
        HEAP32[$139 >> 2] = $138 & 248; //@line 11430
        $$pre$phi326Z2D = $139; //@line 11431
        break L8;
        break;
       }
      case -118:
       {
        HEAP32[$139 >> 2] = $138 | 5; //@line 11437
        $$pre$phi326Z2D = $139; //@line 11438
        break L8;
        break;
       }
      case -32:
       {
        if (!$143) {
         $$pre$phi326Z2D = $139; //@line 11444
         break L8;
        }
        HEAP32[$139 >> 2] = 0; //@line 11447
        $$pre$phi326Z2D = $139; //@line 11448
        break L8;
        break;
       }
      case -31:
       {
        if (!$143) {
         $$pre$phi326Z2D = $139; //@line 11454
         break L8;
        }
        HEAP32[$139 >> 2] = 241; //@line 11457
        $$pre$phi326Z2D = $139; //@line 11458
        break L8;
        break;
       }
      case -30:
       {
        if (!$143) {
         $$pre$phi326Z2D = $139; //@line 11464
         break L8;
        }
        HEAP32[$139 >> 2] = 0; //@line 11467
        $$pre$phi326Z2D = $139; //@line 11468
        break L8;
        break;
       }
      case -29:
       {
        if (!$143) {
         $$pre$phi326Z2D = $139; //@line 11474
         break L8;
        }
        HEAP32[$139 >> 2] = 10; //@line 11477
        $$pre$phi326Z2D = $139; //@line 11478
        break L8;
        break;
       }
      default:
       {
        $$pre$phi326Z2D = $139; //@line 11483
        break L8;
       }
      }
     } else {
      $157 = $127 >>> 2; //@line 11488
      $158 = $157 & 31; //@line 11489
      if ($158 >>> 0 > 15 | ($157 & 28 | 0) == 8) {
       if ($129) {
        $165 = $1 + 8 | 0; //@line 11499
        HEAP32[$165 >> 2] = HEAPU8[656025 + $158 >> 0]; //@line 11500
        $$pre$phi326Z2D = $165; //@line 11501
        break;
       } else {
        $166 = $1 + 8 | 0; //@line 11504
        HEAP8[656025 + $158 >> 0] = HEAP32[$166 >> 2]; //@line 11508
        $$pre$phi326Z2D = $166; //@line 11509
        break;
       }
      }
      if ($158 >>> 0 > 7 | $129 ^ 1) {
       $$pre$phi326Z2D = $1 + 8 | 0; //@line 11518
       break;
      }
      $170 = __Z13TimerDateTimev() | 0; //@line 11521
      switch ($157 & 3) {
      case 1:
       {
        $$0$in = $170 >>> 8; //@line 11527
        break;
       }
      case 2:
       {
        $$0$in = $170 >>> 16; //@line 11532
        break;
       }
      case 3:
       {
        $$0$in = $170 >>> 24; //@line 11537
        break;
       }
      default:
       {
        $$0$in = $170; //@line 11541
       }
      }
      $175 = $1 + 8 | 0; //@line 11545
      HEAP32[$175 >> 2] = $$0$in & 255; //@line 11546
      $$pre$phi326Z2D = $175; //@line 11547
     }
    } while (0);
    HEAP32[$1 >> 2] = 0; //@line 11550
    HEAP32[$126 >> 2] = HEAP32[$$pre$phi326Z2D >> 2]; //@line 11552
    STACKTOP = sp; //@line 11553
    return;
   }
  case 28933:
   {
    $177 = $1 + 4 | 0; //@line 11557
    HEAP32[$177 >> 2] = HEAPU8[656025 + (HEAP32[$177 >> 2] & 255) >> 0]; //@line 11563
    STACKTOP = sp; //@line 11564
    return;
   }
  case 28934:
   {
    HEAP32[$1 >> 2] = HEAPU8[656025 + (HEAP32[$1 >> 2] & 255) >> 0]; //@line 11573
    STACKTOP = sp; //@line 11574
    return;
   }
  case 28935:
   {
    $188 = $1 + 48 | 0; //@line 11578
    $190 = (HEAP32[$188 >> 2] | 0) + -20 | 0; //@line 11580
    FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($190 >>> 16 << 2) >> 2] | 0) + 12 >> 2] & 31]($190, (HEAP32[30895] | 0) + (HEAP32[30893] | 0) | 0); //@line 11589
    $200 = (HEAP32[$188 >> 2] | 0) + -26 | 0; //@line 11591
    FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($200 >>> 16 << 2) >> 2] | 0) + 20 >> 2] & 31]($200, 0); //@line 11597
    $207 = (HEAP32[$188 >> 2] | 0) + -25 | 0; //@line 11599
    $209 = 123608 + ($207 >>> 16 << 2) | 0; //@line 11601
    $214 = FUNCTION_TABLE_ii[HEAP32[(HEAP32[$209 >> 2] | 0) + 8 >> 2] & 127]($207) | 0 | 1; //@line 11606
    FUNCTION_TABLE_vii[HEAP32[(HEAP32[$209 >> 2] | 0) + 20 >> 2] & 31]($207, $214); //@line 11610
    HEAP32[$1 + 56 >> 2] = (HEAP32[30895] | 0) + (HEAP32[30893] | 0); //@line 11615
    STACKTOP = sp; //@line 11616
    return;
   }
  case 28936:
   {
    HEAP32[$1 + 36 >> 2] = (((HEAP32[30895] | 0) * 3 | 0) >>> 2) + (HEAP32[30893] | 0); //@line 11626
    STACKTOP = sp; //@line 11627
    return;
   }
  case 28937:
   {
    $230 = FUNCTION_TABLE_ii[HEAP32[HEAP32[30902] >> 2] & 127](7928) | 0; //@line 11633
    $233 = FUNCTION_TABLE_ii[HEAP32[HEAP32[30902] >> 2] & 127](7924) | 0; //@line 11636
    FUNCTION_TABLE_vii[HEAP32[(HEAP32[30902] | 0) + 12 >> 2] & 31](7928, HEAP32[30895] | 0); //@line 11641
    FUNCTION_TABLE_vii[HEAP32[(HEAP32[30902] | 0) + 12 >> 2] & 31](7924, $233 - $230 + (HEAP32[30895] | 0) | 0); //@line 11648
    STACKTOP = sp; //@line 11649
    return;
   }
  case 28963:
   {
    $245 = HEAP32[$1 >> 2] & 255; //@line 11654
    $247 = HEAP32[$1 + 32 >> 2] | 0; //@line 11656
    $252 = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($247 >>> 16 << 2) >> 2] >> 2] & 127]($247) | 0; //@line 11661
    __Z5ADBOphPh($245, FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($252 >>> 16 << 2) >> 2] | 0) + 24 >> 2] & 127]($252) | 0); //@line 11668
    STACKTOP = sp; //@line 11669
    return;
   }
  case 28964:
   {
    HEAP32[$1 >> 2] = (__Z7InsTimejt(HEAP32[$1 + 32 >> 2] | 0, HEAP32[$1 + 4 >> 2] & 65535) | 0) << 16 >> 16; //@line 11680
    STACKTOP = sp; //@line 11681
    return;
   }
  case 28965:
   {
    HEAP32[$1 >> 2] = (__Z7RmvTimej(HEAP32[$1 + 32 >> 2] | 0) | 0) << 16 >> 16; //@line 11689
    STACKTOP = sp; //@line 11690
    return;
   }
  case 28966:
   {
    HEAP32[$1 >> 2] = (__Z9PrimeTimeji(HEAP32[$1 + 32 >> 2] | 0, HEAP32[$1 >> 2] | 0) | 0) << 16 >> 16; //@line 11699
    STACKTOP = sp; //@line 11700
    return;
   }
  case 28967:
   {
    __Z12MicrosecondsRjS_($1 + 32 | 0, $1); //@line 11705
    STACKTOP = sp; //@line 11706
    return;
   }
  case 28938:
   {
    __Z14InstallDriversj(HEAP32[$1 + 32 >> 2] | 0); //@line 11712
    $278 = HEAP32[28202] | 0; //@line 11713
    if ($278 | 0) {
     HEAP32[$2 >> 2] = 43518; //@line 11716
     HEAP32[$2 + 32 >> 2] = $278; //@line 11718
     _Execute68kTrap(-22969, $2); //@line 11719
    }
    $281 = HEAP32[28203] | 0; //@line 11721
    if ($281 | 0) {
     HEAP32[$2 >> 2] = 43517; //@line 11724
     HEAP32[$2 + 32 >> 2] = $281; //@line 11726
     _Execute68kTrap(-22969, $2); //@line 11727
    }
    if ((HEAP16[325960] | 0) == 1660) {
     HEAP32[$2 >> 2] = 4096; //@line 11732
     _Execute68kTrap(-22754, $2); //@line 11733
     $287 = HEAP32[$2 + 32 >> 2] | 0; //@line 11735
     $288 = $287 + 2048 | 0; //@line 11736
     FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($288 >>> 16 << 2) >> 2] | 0) + 20 >> 2] & 31]($288, 15); //@line 11742
     FUNCTION_TABLE_vii[HEAP32[(HEAP32[30902] | 0) + 12 >> 2] & 31](3264, $287); //@line 11746
    }
    STACKTOP = sp; //@line 11748
    return;
   }
  case 28939:
   {
    __Z11InstallSERDv(); //@line 11752
    STACKTOP = sp; //@line 11753
    return;
   }
  case 28940:
   {
    HEAP32[$1 >> 2] = (__Z8SonyOpenjj(HEAP32[$1 + 32 >> 2] | 0, HEAP32[$1 + 36 >> 2] | 0) | 0) << 16 >> 16; //@line 11763
    STACKTOP = sp; //@line 11764
    return;
   }
  case 28941:
   {
    HEAP32[$1 >> 2] = (__Z9SonyPrimejj(HEAP32[$1 + 32 >> 2] | 0, HEAP32[$1 + 36 >> 2] | 0) | 0) << 16 >> 16; //@line 11774
    STACKTOP = sp; //@line 11775
    return;
   }
  case 28942:
   {
    HEAP32[$1 >> 2] = (__Z11SonyControljj(HEAP32[$1 + 32 >> 2] | 0, HEAP32[$1 + 36 >> 2] | 0) | 0) << 16 >> 16; //@line 11785
    STACKTOP = sp; //@line 11786
    return;
   }
  case 28943:
   {
    HEAP32[$1 >> 2] = (__Z10SonyStatusjj(HEAP32[$1 + 32 >> 2] | 0, HEAP32[$1 + 36 >> 2] | 0) | 0) << 16 >> 16; //@line 11796
    STACKTOP = sp; //@line 11797
    return;
   }
  case 28944:
   {
    HEAP32[$1 >> 2] = (__Z8DiskOpenjj(HEAP32[$1 + 32 >> 2] | 0, HEAP32[$1 + 36 >> 2] | 0) | 0) << 16 >> 16; //@line 11807
    STACKTOP = sp; //@line 11808
    return;
   }
  case 28945:
   {
    HEAP32[$1 >> 2] = (__Z9DiskPrimejj(HEAP32[$1 + 32 >> 2] | 0, HEAP32[$1 + 36 >> 2] | 0) | 0) << 16 >> 16; //@line 11818
    STACKTOP = sp; //@line 11819
    return;
   }
  case 28946:
   {
    HEAP32[$1 >> 2] = (__Z11DiskControljj(HEAP32[$1 + 32 >> 2] | 0, HEAP32[$1 + 36 >> 2] | 0) | 0) << 16 >> 16; //@line 11829
    STACKTOP = sp; //@line 11830
    return;
   }
  case 28947:
   {
    HEAP32[$1 >> 2] = (__Z10DiskStatusjj(HEAP32[$1 + 32 >> 2] | 0, HEAP32[$1 + 36 >> 2] | 0) | 0) << 16 >> 16; //@line 11840
    STACKTOP = sp; //@line 11841
    return;
   }
  case 28948:
   {
    HEAP32[$1 >> 2] = (__Z9CDROMOpenjj(HEAP32[$1 + 32 >> 2] | 0, HEAP32[$1 + 36 >> 2] | 0) | 0) << 16 >> 16; //@line 11851
    STACKTOP = sp; //@line 11852
    return;
   }
  case 28949:
   {
    HEAP32[$1 >> 2] = (__Z10CDROMPrimejj(HEAP32[$1 + 32 >> 2] | 0, HEAP32[$1 + 36 >> 2] | 0) | 0) << 16 >> 16; //@line 11862
    STACKTOP = sp; //@line 11863
    return;
   }
  case 28950:
   {
    HEAP32[$1 >> 2] = (__Z12CDROMControljj(HEAP32[$1 + 32 >> 2] | 0, HEAP32[$1 + 36 >> 2] | 0) | 0) << 16 >> 16; //@line 11873
    STACKTOP = sp; //@line 11874
    return;
   }
  case 28951:
   {
    HEAP32[$1 >> 2] = (__Z11CDROMStatusjj(HEAP32[$1 + 32 >> 2] | 0, HEAP32[$1 + 36 >> 2] | 0) | 0) << 16 >> 16; //@line 11884
    STACKTOP = sp; //@line 11885
    return;
   }
  case 28952:
   {
    HEAP32[$1 >> 2] = (__Z15VideoDriverOpenjj(HEAP32[$1 + 32 >> 2] | 0, HEAP32[$1 + 36 >> 2] | 0) | 0) << 16 >> 16; //@line 11895
    STACKTOP = sp; //@line 11896
    return;
   }
  case 28953:
   {
    HEAP32[$1 >> 2] = (__Z18VideoDriverControljj(HEAP32[$1 + 32 >> 2] | 0, HEAP32[$1 + 36 >> 2] | 0) | 0) << 16 >> 16; //@line 11906
    STACKTOP = sp; //@line 11907
    return;
   }
  case 28954:
   {
    HEAP32[$1 >> 2] = (__Z17VideoDriverStatusjj(HEAP32[$1 + 32 >> 2] | 0, HEAP32[$1 + 36 >> 2] | 0) | 0) << 16 >> 16; //@line 11917
    STACKTOP = sp; //@line 11918
    return;
   }
  case 28955:
   {
    HEAP32[$1 >> 2] = (__Z10SerialOpenjji(HEAP32[$1 + 32 >> 2] | 0, HEAP32[$1 + 36 >> 2] | 0, HEAP32[$1 >> 2] | 0) | 0) << 16 >> 16; //@line 11929
    STACKTOP = sp; //@line 11930
    return;
   }
  case 28956:
   {
    HEAP32[$1 >> 2] = (__Z11SerialPrimejji(HEAP32[$1 + 32 >> 2] | 0, HEAP32[$1 + 36 >> 2] | 0, HEAP32[$1 >> 2] | 0) | 0) << 16 >> 16; //@line 11941
    STACKTOP = sp; //@line 11942
    return;
   }
  case 28957:
   {
    HEAP32[$1 >> 2] = (__Z13SerialControljji(HEAP32[$1 + 32 >> 2] | 0, HEAP32[$1 + 36 >> 2] | 0, HEAP32[$1 >> 2] | 0) | 0) << 16 >> 16; //@line 11953
    STACKTOP = sp; //@line 11954
    return;
   }
  case 28958:
   {
    HEAP32[$1 >> 2] = (__Z12SerialStatusjji(HEAP32[$1 + 32 >> 2] | 0, HEAP32[$1 + 36 >> 2] | 0, HEAP32[$1 >> 2] | 0) | 0) << 16 >> 16; //@line 11965
    STACKTOP = sp; //@line 11966
    return;
   }
  case 28959:
   {
    HEAP32[$1 >> 2] = (__Z11SerialClosejji(HEAP32[$1 + 32 >> 2] | 0, HEAP32[$1 + 36 >> 2] | 0, HEAP32[$1 >> 2] | 0) | 0) << 16 >> 16; //@line 11977
    STACKTOP = sp; //@line 11978
    return;
   }
  case 28960:
   {
    HEAP32[$1 >> 2] = (__Z9EtherOpenjj(HEAP32[$1 + 32 >> 2] | 0, HEAP32[$1 + 36 >> 2] | 0) | 0) << 16 >> 16; //@line 11988
    STACKTOP = sp; //@line 11989
    return;
   }
  case 28961:
   {
    HEAP32[$1 >> 2] = (__Z12EtherControljj(HEAP32[$1 + 32 >> 2] | 0, HEAP32[$1 + 36 >> 2] | 0) | 0) << 16 >> 16; //@line 11999
    STACKTOP = sp; //@line 12000
    return;
   }
  case 28962:
   {
    __Z15EtherReadPacketRjS_S_S_($1 + 32 | 0, $1 + 44 | 0, $1 + 12 | 0, $1 + 4 | 0); //@line 12008
    STACKTOP = sp; //@line 12009
    return;
   }
  case 28977:
   {
    HEAP32[$1 >> 2] = (__Z11SoundInOpenjj(HEAP32[$1 + 32 >> 2] | 0, HEAP32[$1 + 36 >> 2] | 0) | 0) << 16 >> 16; //@line 12019
    STACKTOP = sp; //@line 12020
    return;
   }
  case 28978:
   {
    HEAP32[$1 >> 2] = (__Z12SoundInPrimejj(HEAP32[$1 + 32 >> 2] | 0, HEAP32[$1 + 36 >> 2] | 0) | 0) << 16 >> 16; //@line 12030
    STACKTOP = sp; //@line 12031
    return;
   }
  case 28979:
   {
    HEAP32[$1 >> 2] = (__Z14SoundInControljj(HEAP32[$1 + 32 >> 2] | 0, HEAP32[$1 + 36 >> 2] | 0) | 0) << 16 >> 16; //@line 12041
    STACKTOP = sp; //@line 12042
    return;
   }
  case 28980:
   {
    HEAP32[$1 >> 2] = (__Z13SoundInStatusjj(HEAP32[$1 + 32 >> 2] | 0, HEAP32[$1 + 36 >> 2] | 0) | 0) << 16 >> 16; //@line 12052
    STACKTOP = sp; //@line 12053
    return;
   }
  case 28981:
   {
    HEAP32[$1 >> 2] = (__Z12SoundInClosejj(HEAP32[$1 + 32 >> 2] | 0, HEAP32[$1 + 36 >> 2] | 0) | 0) << 16 >> 16; //@line 12063
    STACKTOP = sp; //@line 12064
    return;
   }
  case 28968:
   {
    $468 = $1 + 60 | 0; //@line 12068
    $469 = HEAP32[$468 >> 2] | 0; //@line 12069
    $474 = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($469 >>> 16 << 2) >> 2] >> 2] & 127]($469) | 0; //@line 12074
    $476 = (HEAP32[$468 >> 2] | 0) + 4 | 0; //@line 12076
    $482 = FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($476 >>> 16 << 2) >> 2] | 0) + 4 >> 2] & 127]($476) | 0; //@line 12082
    $483 = HEAP32[$468 >> 2] | 0; //@line 12083
    $484 = $483 + 6 | 0; //@line 12084
    HEAP32[$468 >> 2] = $484; //@line 12085
    do {
     switch (($482 & 65535) << 16 >> 16) {
     case 0:
      {
       $486 = (__Z9SCSIResetv() | 0) << 16 >> 16; //@line 12091
       FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($484 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($484, $486); //@line 12097
       $$0313 = 0; //@line 12098
       break;
      }
     case 1:
      {
       $493 = (__Z7SCSIGetv() | 0) << 16 >> 16; //@line 12103
       FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($484 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($484, $493); //@line 12109
       $$0313 = 0; //@line 12110
       break;
      }
     case 11:
     case 2:
      {
       $499 = $483 + 8 | 0; //@line 12114
       $508 = (__Z10SCSISelecti((FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($484 >>> 16 << 2) >> 2] | 0) + 4 >> 2] & 127]($484) | 0) & 255) | 0) << 16 >> 16; //@line 12123
       FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($499 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($499, $508); //@line 12129
       $$0313 = 2; //@line 12130
       break;
      }
     case 3:
      {
       $514 = $483 + 12 | 0; //@line 12134
       $520 = FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($484 >>> 16 << 2) >> 2] | 0) + 4 >> 2] & 127]($484) | 0; //@line 12140
       $522 = (HEAP32[$468 >> 2] | 0) + 2 | 0; //@line 12142
       $527 = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($522 >>> 16 << 2) >> 2] >> 2] & 127]($522) | 0; //@line 12147
       $535 = (__Z7SCSICmdiPh($520, FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($527 >>> 16 << 2) >> 2] | 0) + 24 >> 2] & 127]($527) | 0) | 0) << 16 >> 16; //@line 12155
       FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($514 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($514, $535); //@line 12161
       $$0313 = 6; //@line 12162
       break;
      }
     case 4:
      {
       $541 = $483 + 18 | 0; //@line 12166
       $546 = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($484 >>> 16 << 2) >> 2] >> 2] & 127]($484) | 0; //@line 12171
       $548 = (HEAP32[$468 >> 2] | 0) + 4 | 0; //@line 12173
       $553 = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($548 >>> 16 << 2) >> 2] >> 2] & 127]($548) | 0; //@line 12178
       $555 = (HEAP32[$468 >> 2] | 0) + 8 | 0; //@line 12180
       $562 = (__Z12SCSICompletejjj($546, $553, FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($555 >>> 16 << 2) >> 2] >> 2] & 127]($555) | 0) | 0) << 16 >> 16; //@line 12187
       FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($541 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($541, $562); //@line 12193
       $$0313 = 12; //@line 12194
       break;
      }
     case 8:
     case 5:
      {
       $568 = $483 + 10 | 0; //@line 12198
       $575 = (__Z8SCSIReadj(FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($484 >>> 16 << 2) >> 2] >> 2] & 127]($484) | 0) | 0) << 16 >> 16; //@line 12205
       FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($568 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($568, $575); //@line 12211
       $$0313 = 4; //@line 12212
       break;
      }
     case 9:
     case 6:
      {
       $581 = $483 + 10 | 0; //@line 12216
       $588 = (__Z9SCSIWritej(FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($484 >>> 16 << 2) >> 2] >> 2] & 127]($484) | 0) | 0) << 16 >> 16; //@line 12223
       FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($581 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($581, $588); //@line 12229
       $$0313 = 4; //@line 12230
       break;
      }
     case 10:
      {
       $595 = (__Z8SCSIStatv() | 0) & 65535; //@line 12235
       FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($484 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($484, $595); //@line 12241
       $$0313 = 0; //@line 12242
       break;
      }
     case 12:
      {
       $601 = $483 + 10 | 0; //@line 12246
       FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($601 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($601, 0); //@line 12252
       $$0313 = 4; //@line 12253
       break;
      }
     case 13:
      {
       $607 = $483 + 8 | 0; //@line 12257
       FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($607 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($607, 0); //@line 12263
       $$0313 = 2; //@line 12264
       break;
      }
     case 14:
      {
       $614 = (__Z11SCSIMgrBusyv() | 0) << 16 >> 16; //@line 12269
       FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($484 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($484, $614); //@line 12275
       $$0313 = 0; //@line 12276
       break;
      }
     default:
      {
       HEAP32[$vararg_buffer17 >> 2] = $482 & 65535; //@line 12281
       _printf(93072, $vararg_buffer17) | 0; //@line 12282
       __Z12QuitEmulatorv(); //@line 12283
      }
     }
    } while (0);
    HEAP32[$1 + 32 >> 2] = $474; //@line 12289
    HEAP32[$1 + 36 >> 2] = (HEAP32[$468 >> 2] | 0) + $$0313; //@line 12293
    STACKTOP = sp; //@line 12294
    return;
   }
  case 28969:
   {
    HEAP32[$1 >> 2] = 0; //@line 12298
    if (HEAP32[30882] & 1 | 0) {
     __Z18ClearInterruptFlagj(1); //@line 12303
     $631 = (FUNCTION_TABLE_ii[HEAP32[HEAP32[30902] >> 2] & 127](362) | 0) + 1 | 0; //@line 12307
     FUNCTION_TABLE_vii[HEAP32[(HEAP32[30902] | 0) + 12 >> 2] & 31](362, $631); //@line 12311
     if ((FUNCTION_TABLE_ii[HEAP32[HEAP32[30902] >> 2] & 127](3324) | 0) == 1464619843) {
      __Z14TimerInterruptv(); //@line 12317
      __Z14VideoInterruptv(); //@line 12318
      if ((HEAP16[325960] | 0) == 1660) {
       HEAP32[$2 >> 2] = 0; //@line 12322
       _Execute68kTrap(-24462, $2); //@line 12323
      }
      HEAP32[$1 >> 2] = 1; //@line 12325
     }
    }
    if (HEAP32[30882] & 2 | 0) {
     __Z18ClearInterruptFlagj(2); //@line 12332
     if ((FUNCTION_TABLE_ii[HEAP32[HEAP32[30902] >> 2] & 127](3324) | 0) == 1464619843) {
      __Z13SonyInterruptv(); //@line 12338
      __Z13DiskInterruptv(); //@line 12339
      __Z14CDROMInterruptv(); //@line 12340
     }
    }
    $648 = HEAP32[30882] | 0; //@line 12343
    if (!($648 & 4)) {
     $652 = $648; //@line 12347
    } else {
     __Z18ClearInterruptFlagj(4); //@line 12349
     __Z15SerialInterruptv(); //@line 12350
     $652 = HEAP32[30882] | 0; //@line 12352
    }
    if (!($652 & 8)) {
     $655 = $652; //@line 12357
    } else {
     __Z18ClearInterruptFlagj(8); //@line 12359
     __Z14EtherInterruptv(); //@line 12360
     $655 = HEAP32[30882] | 0; //@line 12362
    }
    if (!($655 & 16)) {
     $658 = $655; //@line 12367
    } else {
     __Z18ClearInterruptFlagj(16); //@line 12369
     __Z14AudioInterruptv(); //@line 12370
     $658 = HEAP32[30882] | 0; //@line 12372
    }
    if ($658 & 64 | 0) {
     __Z18ClearInterruptFlagj(64); //@line 12377
     if ((FUNCTION_TABLE_ii[HEAP32[HEAP32[30902] >> 2] & 127](3324) | 0) == 1464619843) {
      __Z12ADBInterruptv(); //@line 12383
     }
    }
    if (!(HEAP32[30882] & 128)) {
     STACKTOP = sp; //@line 12390
     return;
    }
    __Z18ClearInterruptFlagj(128); //@line 12392
    if ((FUNCTION_TABLE_ii[HEAP32[HEAP32[30902] >> 2] & 127](3324) | 0) != 1464619843) {
     STACKTOP = sp; //@line 12398
     return;
    }
    STACKTOP = sp; //@line 12400
    return;
   }
  case 28970:
   {
    $671 = $1 + 60 | 0; //@line 12404
    $673 = (HEAP32[$671 >> 2] | 0) + 4 | 0; //@line 12406
    $678 = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($673 >>> 16 << 2) >> 2] >> 2] & 127]($673) | 0; //@line 12411
    FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($678 >>> 16 << 2) >> 2] | 0) + 24 >> 2] & 127]($678) | 0; //@line 12417
    $685 = (HEAP32[$671 >> 2] | 0) + 8 | 0; //@line 12419
    FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($685 >>> 16 << 2) >> 2] >> 2] & 127]($685) | 0; //@line 12424
    $691 = (HEAP32[$671 >> 2] | 0) + 12 | 0; //@line 12426
    FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($691 >>> 16 << 2) >> 2] >> 2] & 127]($691) | 0; //@line 12431
    STACKTOP = sp; //@line 12432
    return;
   }
  case 28971:
   {
    $696 = $1 + 60 | 0; //@line 12436
    $698 = (HEAP32[$696 >> 2] | 0) + 4 | 0; //@line 12438
    $703 = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($698 >>> 16 << 2) >> 2] >> 2] & 127]($698) | 0; //@line 12443
    FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($703 >>> 16 << 2) >> 2] | 0) + 24 >> 2] & 127]($703) | 0; //@line 12449
    $710 = (HEAP32[$696 >> 2] | 0) + 8 | 0; //@line 12451
    FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($710 >>> 16 << 2) >> 2] >> 2] & 127]($710) | 0; //@line 12456
    $716 = (HEAP32[$696 >> 2] | 0) + 12 | 0; //@line 12458
    FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($716 >>> 16 << 2) >> 2] >> 2] & 127]($716) | 0; //@line 12463
    STACKTOP = sp; //@line 12464
    return;
   }
  case 28972:
   {
    $722 = HEAP32[$1 + 4 >> 2] | 0; //@line 12469
    $724 = HEAP32[$1 + 40 >> 2] | 0; //@line 12471
    $731 = (FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($724 >>> 16 << 2) >> 2] | 0) + 4 >> 2] & 127]($724) | 0) & 65535; //@line 12478
    $733 = HEAP32[$1 + 32 >> 2] | 0; //@line 12480
    if (!$733) {
     STACKTOP = sp; //@line 12483
     return;
    }
    $739 = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($733 >>> 16 << 2) >> 2] >> 2] & 127]($733) | 0; //@line 12489
    if (!$739) {
     STACKTOP = sp; //@line 12492
     return;
    }
    $747 = $739 + -8 | 0; //@line 12500
    __Z9CheckLoadjsPhj($722, $731, FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($739 >>> 16 << 2) >> 2] | 0) + 24 >> 2] & 127]($739) | 0, (FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($747 >>> 16 << 2) >> 2] >> 2] & 127]($747) | 0) & 16777215); //@line 12507
    STACKTOP = sp; //@line 12508
    return;
   }
  case 28973:
   {
    HEAP32[$1 >> 2] = __Z13AudioDispatchjj(HEAP32[$1 + 44 >> 2] | 0, HEAP32[$1 + 48 >> 2] | 0) | 0; //@line 12517
    STACKTOP = sp; //@line 12518
    return;
   }
  case 28974:
   {
    $759 = $1 + 60 | 0; //@line 12522
    $760 = HEAP32[$759 >> 2] | 0; //@line 12523
    $761 = $760 + 14 | 0; //@line 12524
    $762 = $760 + 12 | 0; //@line 12525
    $769 = (FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($762 >>> 16 << 2) >> 2] | 0) + 4 >> 2] & 127]($762) | 0) & 65535; //@line 12532
    $771 = (HEAP32[$759 >> 2] | 0) + 8 | 0; //@line 12534
    $776 = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($771 >>> 16 << 2) >> 2] >> 2] & 127]($771) | 0; //@line 12539
    $778 = (HEAP32[$759 >> 2] | 0) + 4 | 0; //@line 12541
    $785 = (__Z9ExtFSCommtjj($769, $776, FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($778 >>> 16 << 2) >> 2] >> 2] & 127]($778) | 0) | 0) << 16 >> 16; //@line 12548
    FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($761 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($761, $785); //@line 12554
    STACKTOP = sp; //@line 12555
    return;
   }
  case 28975:
   {
    $791 = $1 + 60 | 0; //@line 12559
    $792 = HEAP32[$791 >> 2] | 0; //@line 12560
    $793 = $792 + 20 | 0; //@line 12561
    $794 = $792 + 16 | 0; //@line 12562
    $799 = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($794 >>> 16 << 2) >> 2] >> 2] & 127]($794) | 0; //@line 12567
    $801 = (HEAP32[$791 >> 2] | 0) + 14 | 0; //@line 12569
    $808 = (FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($801 >>> 16 << 2) >> 2] | 0) + 4 >> 2] & 127]($801) | 0) & 65535; //@line 12576
    $810 = (HEAP32[$791 >> 2] | 0) + 10 | 0; //@line 12578
    $815 = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($810 >>> 16 << 2) >> 2] >> 2] & 127]($810) | 0; //@line 12583
    $817 = (HEAP32[$791 >> 2] | 0) + 6 | 0; //@line 12585
    $822 = FUNCTION_TABLE_ii[HEAP32[HEAP32[123608 + ($817 >>> 16 << 2) >> 2] >> 2] & 127]($817) | 0; //@line 12590
    $824 = (HEAP32[$791 >> 2] | 0) + 4 | 0; //@line 12592
    $833 = (__Z8ExtFSHFSjtjjs($799, $808, $815, $822, (FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($824 >>> 16 << 2) >> 2] | 0) + 4 >> 2] & 127]($824) | 0) & 65535) | 0) << 16 >> 16; //@line 12601
    FUNCTION_TABLE_vii[HEAP32[(HEAP32[123608 + ($793 >>> 16 << 2) >> 2] | 0) + 16 >> 2] & 31]($793, $833); //@line 12607
    STACKTOP = sp; //@line 12608
    return;
   }
  case 28976:
   {
    $840 = HEAP32[$1 + 32 >> 2] | 0; //@line 12613
    FUNCTION_TABLE_ii[HEAP32[(HEAP32[123608 + ($840 >>> 16 << 2) >> 2] | 0) + 24 >> 2] & 127]($840) | 0; //@line 12619
    STACKTOP = sp