// ---------------------------------------------------------------------------
//	M88 - PC-8801 Emulator.
//	Copyright (C) cisc 1998, 1999.
// ---------------------------------------------------------------------------
//	$Id: fdu.cpp,v 1.5 1999/07/29 14:35:31 cisc Exp $

#include "headers.h"

#include "misc.h"
#include "fdc.h"
#include "fdu.h"
#include "diskmgr.h"

using namespace PC8801;

// ---------------------------------------------------------------------------
//	\zEj
//
FDU::FDU()
{
	disk = 0;
	cyrinder = 0;
}

FDU::~FDU()
{
	if (disk)
		Unmount();
}

// ---------------------------------------------------------------------------
//	
//
bool FDU::Init(DiskManager* dm, int dr, Scheduler *s)
{
	scheduler = s;
	diskmgr = dm;
	drive = dr;
	return true;
}

// ---------------------------------------------------------------------------
//	FDU::Mount
//	C[W蓖ĂihCuɃfBXNj
//
bool FDU::Mount(FloppyDisk* fd)
{
	disk = fd;
	return true;
}

// ---------------------------------------------------------------------------
//	FDU::Unmount
//
bool FDU::Unmount()
{
	disk = 0;
	return true;
}

// ---------------------------------------------------------------------------
//	FDU::GetCurrentPos
//
uint FDU::GetCurrentPos()
{
	if (!disk) {
		return 0;
	}

	uint capacity = disk->GetTrackCapacity();
	uint pos = (uint)scheduler->GetTime();
	if (disk->GetType() == FloppyDisk::DiskType::MD2HD) {
		// RPM 360(Pʂ0.01ms)
		pos %= 16600;
		pos *= capacity;
		pos /= 16600;
	} else {
		// RPM 300(Pʂ0.01ms)
		pos %= 20000;
		pos *= capacity;
		pos /= 20000;
	}

	return pos;
}

// ---------------------------------------------------------------------------
//	FDU::Bytes2Time
//
uint FDU::Bytes2Time(uint len)
{
	if (!disk) {
		return 0;
	}

	// 10usPʂɕϊ
	if (disk->GetType() == FloppyDisk::DiskType::MD2HD) {
		// 1000kbps
		len *= 16;
		len *= 1000;
		len /= 10000;
	} else {
		// 500kbps
		len *= 16;
		len *= 1000;
		len /= 5000;
	}

	return len;
}

// ---------------------------------------------------------------------------
//	FDU::SetHead
//	wbh̎w
//
inline void FDU::SetHead(uint hd)
{
	if (!disk) {
		return;
	}

	head = hd & 1;
	track = (cyrinder << 1) + (hd & 1);
	disk->Seek(track);
}

// ---------------------------------------------------------------------------
//	FDU::Seek
//	w肳ꂽV_[ԍփV[N
//
//	cyrinder V[N
//
uint FDU::Seek(uint cy)
{
	cyrinder = cy;
	return 0;
}

// ---------------------------------------------------------------------------
//	FDU::NextSector
//	̃ZN^擾
//
FloppyDisk::Sector* FDU::NextSector(uint flags)
{
	if (!disk) {
		return 0;
	}

	for (int i = disk->GetNumSectors(); i > 0; i--) {
		FloppyDisk::Sector* sector = disk->GetSector();
		if (!sector) {
			disk->IndexHole();
			sector = disk->GetSector();
		}

		if ((flags & 0xc0) == (sector->flags & 0xc0)) {
			return sector;
		}
	}

	disk->IndexHole();
	return 0;
}

// ---------------------------------------------------------------------------
//	FDU::SearchID
//	ZN^
//
uint FDU::SearchID(uint flags, IDR *id, bool idmatch, bool ignorecrc, int *stime)
{
	uint result = 0;
	int num;
	bool find = false;

	if (!disk)
		return FDC::ST0_AT | FDC::ST1_MA;

	// ԏNA
	*stime = 0;

	// ZN^
	sector = 0;
	num = disk->GetNumSectors();
	while (num--) {
		// ̃ZN^擾
		sector = NextSector(flags);
		if (!sector) {
			break;
		}

		if (idmatch) {
			// IDvZN^𔭌
			if (*id == sector->id) {
				find = true;
				result = 0;

				// IDCRCĂZN^Ȃf[^G[Œf(ST2_DD̓lQ[g)
				if (!ignorecrc && sector->flags & FloppyDisk::idcrc) {
					result = FDC::ST0_AT | FDC::ST1_DE;
				}
				break;
			} else {
				if (id->c != sector->id.c) {
					if (sector->id.c == 0xff) {
						result |= FDC::ST2_BC;
					} else {
						result |= FDC::ST2_NC;
					}
				}
			}
		} else {
			// CRCG[ĂȂZN^
			if (ignorecrc || !(sector->flags & FloppyDisk::idcrc)) {
				*id = sector->id;
				find = true;
				result = 0;
				break;
			}
		}
	}

	// ID
	if (find) {
		// ̃ZN^܂ł̎
		if (disk->IsFdx() && !sector->update) {
			uint pos = GetCurrentPos();
			*stime = sector->fdx_ioffset +
				((sector->flags & 0x40) ? 10 * 16 : 7 * 32);
			*stime = (*stime >> 4) - pos;
			if (*stime < 0) { *stime += disk->GetTrackCapacity(); }
			*stime = Bytes2Time(*stime);
		}
		return result;
	}

	// IDȂ
	if (!sector) {
		// xɂZN^Ȃ
		result = FDC::ST0_AT | FDC::ST1_MA;
	} else {
		// m[f[^
		result |= FDC::ST0_AT | FDC::ST1_ND;
	}

	// ZN^΃CfbNXz[2񌟏o܂Ō
	uint pos = GetCurrentPos();
	*stime = Bytes2Time(disk->GetTrackCapacity() * 2 - pos);
	return result;
}

// ---------------------------------------------------------------------------
//	FDU::ReadSector
//	ZN^ǂ
//
//	head	wbhԍ
//	id		ǂݍރZN^̃ZN^ ID
//	data	f[^̓]
//	size	]TCY
//	stime	]Jn܂ł̎(usP)
//	txtime	](usP)
//
uint FDU::ReadSector(uint flags, IDR id, uint8* data, int *size, int *stime, int *txtime)
{
	// ԂNA
	*stime = 0;
	*txtime = 0;

	if (!disk)
		return FDC::ST0_AT | FDC::ST1_MA;
	SetHead(flags);

	// ZN^T
	uint result = SearchID(flags, &id, true, false, stime);
	if (result) {
		return result;
	}

	// MAMĂZN^Ȃ~bVOAhX}[NŒf
	if (sector->flags & FloppyDisk::mam) {
		return FDC::ST0_AT | FDC::ST1_MA | FDC::ST2_MD;
	}

	// FDXg
	// f[^̈ʒu(DAM/DDAMXLbv)
	if (disk->IsFdx() && !sector->update) {
		int offset = sector->fdx_doffset;
		if (sector->flags & 0x40) {
			offset += 64;
		} else {
			offset += 32;
		}

		// f[^̊Jnʒu܂ł̎
		uint pos = GetCurrentPos();
		*stime = (offset >> 4) - pos;
		if (*stime < 0) {
			*stime += disk->GetTrackCapacity();
		}
		*stime = Bytes2Time(*stime);
	}

	// FDXg
	// ZN^XVĂ炸CRCG[Ă
	// NbNĕԂ(sf[^Č)
	if (disk->IsFdx() && !sector->update && (sector->flags & FloppyDisk::datacrc)) {
		FDX::bf_splitclock(
			sector->fdx_encdata, sector->fdx_enclen, 0,
			data, sector->size,
			sector->flags & 0x40 ? true : false, true);

		// ]TCYA]Ԃ͖
		*size = sector->size;
	} else {
		// ]TCYA]Ԃ͖
		*size = Min(0x2000, sector->size);

		// PRs[
		memcpy(data, sector->image, *size);
	}

	// ]ɕKvȎ
	*txtime = *size * ((sector->flags & 0x40) ? 1 : 2);
	*txtime = Bytes2Time(*txtime);

	// FDXg
	// mȓ]ԂԂ(EF[uČ)
	FloppyDisk::Track *track = disk->GetTrack();
	if (disk->IsFdx() && track->fdx_rawdata && !sector->update && !(sector->flags & FloppyDisk::mam)) {
		int start = FDX::encode_to_raw_pos(
			track->fdx_rawdata, track->fdx_rawlen,
			sector->fdx_doffset + FDX_CELL_BITS * ((sector->flags & 0x40) ? 4 : 1));
		int end = FDX::encode_to_raw_pos(
			track->fdx_rawdata, track->fdx_rawlen,
			sector->fdx_eoffset - FDX_CELL_BITS * 2);
		if (start >= 0 && end >= 0) {
			int elapse = end - start;
			if (elapse < 0) {
				elapse += track->fdx_rawlen;
			}
			elapse >>= 7;
			*txtime = Bytes2Time(elapse);
		}
	}

	if (sector->flags & FloppyDisk::datacrc) {
		return FDC::ST0_AT | FDC::ST1_DE | FDC::ST2_DD;
	}

	if (sector->flags & FloppyDisk::deleted) {
		return FDC::ST2_CM;
	}

	return 0;
}

// ---------------------------------------------------------------------------
//	FindID	
//
uint FDU::FindID(uint flags, IDR id, int size, int *stime, int *txtime)
{
	// ԂNA
	*stime = 0;
	*txtime = 0;

	if (!disk)
		return FDC::ST0_AT | FDC::ST1_MA;
	SetHead(flags);

	if (disk->IsReadOnly())
		return FDC::ST0_AT | FDC::ST1_NW;

	// ZN^T
	uint result = SearchID(flags, &id, true, false, stime);
	if (result) {
		return result;
	}

	// f[^]
	*txtime = size * ((sector->flags & 0x40) ? 1 : 2);
	*txtime = Bytes2Time(*txtime);
	return 0;
}

// ---------------------------------------------------------------------------
//	FDU::WriteSector
//	ZN^ɏ
//
uint FDU::WriteSector(uint flags, IDR id, const uint8* data, bool deleted)
{
	if (!disk)
		return 0;

	uint writesize = 0x80 << Min(7, id.n);

	if (writesize > sector->size)
	{
		if (!disk->Resize(sector, writesize))
			return 0;

		// tH[}bgꂽƂɂ
		disk->GetTrack()->formated = true;
	}

	memcpy(sector->image, data, writesize);
	sector->flags = (flags & 0xc0) | (deleted ? FloppyDisk::deleted : 0);
	sector->size = writesize;
	sector->update = true;
	diskmgr->Modified(drive, track);
	return 0;
}

// ---------------------------------------------------------------------------
//	FDU::SenceDeviceStatus
//	foCXEX^[^X𓾂
//
uint FDU::SenceDeviceStatus()
{
	uint result = 0x20 | (cyrinder ? 0 : 0x10) | (head ? 4 : 0);
	if (disk)
		result |= disk->IsReadOnly() ? 0x48 : 8;
	return result;
}

// ---------------------------------------------------------------------------
//	FDU::ReadID
//	ZN^ƂĂ
//
uint FDU::ReadID(uint flags, IDR* id, int *stime)
{
	// ԂNA
	*stime = 0;

	if (!disk)
		return FDC::ST0_AT | FDC::ST1_MA;
	SetHead(flags);

	// ZN^T
	return SearchID(flags, id, false, false, stime);
}

// ---------------------------------------------------------------------------
//	FDU::SearchIndex
//	CfbNXz[
//
uint FDU::SearchIndex(uint flags, int *stime)
{
	// ԂNA
	*stime = 0;

	if (!disk)
		return FDC::ST0_AT | FDC::ST1_MA;
	SetHead(flags);

	// CfbNz[Ɉړ
	disk->IndexHole();

	// ̃CfbNXz[܂ł̎
	uint pos = GetCurrentPos();
	*stime = Bytes2Time(disk->GetTrackCapacity() - pos);
	return 0;
}

// ---------------------------------------------------------------------------
//	ReadDiag
//
uint FDU::ReadDiag(uint flags, uint8* data, uint* size, IDR idr, int *stime, int *txtime)
{
	if (!disk)
		return FDC::ST0_AT | FDC::ST1_MA;

	// FDXg
	if (disk->IsFdx() && !disk->GetTrack()->formated) {
		return ReadDiagFdx(flags, data, size, idr, stime, txtime);
	}

	// ԏNA
	*stime = 0;
	*txtime = 0;

	// IDT
	IDR id;
	uint result = SearchID(flags, &id, false, true, stime);
	if (result) {
		return result;
	}

	// f[^
	if (sector->flags & FloppyDisk::mam) {
		return FDC::ST0_AT | FDC::ST1_MA | FDC::ST2_MD;
	}

	// IDr(̂R=1Ŕr悤)
	if (sector->id.r != 1 || sector->id.n != idr.n) {
		result |= FDC::ST1_ND;
	}

	// IDCRC
	if (sector->flags & FloppyDisk::idcrc) {
		result |= FDC::ST1_DE;
	}

	// ZN^TCYƓȂZN^DATACRCG[𔽉f
	if (sector->id.n == idr.n) {
		if (sector->flags & FloppyDisk::datacrc) {
			result |= FDC::ST1_DE | FDC::ST2_DD;
		}
	}

	// ZN^TCYƈقȂȂDATACRCG[ƂȂ͂
	if (sector->id.n != idr.n) {
		result |= FDC::ST1_DE | FDC::ST2_DD;
	}

	// DELETEDZN^
	if (sector->flags & FloppyDisk::deleted) {
		result |= FDC::ST2_CM;
	}

	uint capacity = disk->GetTrackCapacity();
	if ((flags & 0x40) == 0) capacity >>= 1;

	FloppyDisk::Sector* sec = disk->GetFirstSector(track);
	uint8 buf[0x4000];
	uint8* dest = buf;
	uint8* limit = dest + capacity;
	uint startpos = 0;
	uint endpos = 0;
	FloppyDisk::Sector* lastsec = 0;

	// vAv
	if (flags & 0x40)
	{	// MFM
		memset(dest, 0x4e, 80);				// GAP4a
		memset(dest + 80, 0x00, 12);		// Sync
		dest[92] = 0xc2; dest[93] = 0xc2;	// IAM
		dest[94] = 0xc2; dest[95] = 0xfc;
		memset(dest + 96, 0x4e, 50);		// GAP1
		dest += 146;
	} else
	{	// FM
		memset(dest, 0xff, 40);				// GAP4a
		memset(dest + 40, 0x00, 6);			// Sync
		dest[46] = 0xfc;					// IAM
		memset(dest + 47, 0xff, 26);		// GAP1
		dest += 73;
	}

	for (; sec && dest < limit; sec = sec->next)
	{
		if (((flags ^ sec->flags) & 0xc0) == 0)
		{
			// tracksize = 94/49 + secsize
			// ID
			if (flags & 0x40)
			{	// MFM
				memset(dest, 0, 12);			// SYNC
				dest[12] = 0xa1; dest[13] = 0xa1; // IDAM
				dest[14] = 0xa1; dest[15] = 0xfe;
				dest[16] = sec->id.c; dest[17] = sec->id.h;
				dest[18] = sec->id.r; dest[19] = sec->id.n;
				memset(dest + 20, 0x4e, 22 + 2);	// CRC+GAP2
				memset(dest + 44, 0, 12);			// SYNC
				dest[56] = 0xa1; dest[57] = 0xa1; // IDAM
				dest[58] = 0xa1; dest[59] = sec->flags & FloppyDisk::deleted ? 0xf8 : 0xfb;
				dest += 60;
			} else
			{	// FM
				memset(dest, 0, 6);			// SYNC
				dest[6] = 0xfe;					// IDAM
				dest[7] = sec->id.c; dest[8] = sec->id.h;
				dest[9] = sec->id.r; dest[10] = sec->id.n;
				memset(dest + 11, 0xff, 11 + 2);	// CRC+GAP2
				memset(dest + 24, 0, 6);			// SYNC
				dest[30] = sec->flags & FloppyDisk::deleted ? 0xf8 : 0xfb;
				dest += 31;
			}

			// Diagf[^̊Jn_
			if (sec == sector) {
				startpos = dest - buf;
			}

			memcpy(dest, sec->image, sec->size);
			dest += sec->size;
			if (flags & 0x40)
				memset(dest, 0x4e, 2 + 0x20), dest += 0x22;
			else
				memset(dest, 0xff, 2 + 0x10), dest += 0x12;
		} else
		{
			// Diagf[^̊Jn_
			if (sec == sector) {
				startpos = dest - buf;
			}

			if (flags & 0x40)
			{
				memset(dest, 0, (49 + sec->size) * 2);
				dest += (49 + sec->size) * 2;
			} else
			{
				memset(dest, 0, (94 + sec->size) / 2);
				dest += (94 + sec->size) / 2;
			}
		}

		// ŏIZN^m
		lastsec = sec;
	}

	if (dest < limit)
	{
		memset(dest, (flags & 0x40) ? 0x4e : 0xff, limit - dest);
	}

	// ](ő\)
	uint rest = *size;
	uint offset = startpos;
	dest = data;
	while (rest--) {
		*dest++ = buf[offset++];
		if (offset >= capacity) {
			offset = 0;
		}
	}

	// f[^]
	*txtime = *size * ((sector->flags & 0x40) ? 1 : 2);
	*txtime = Bytes2Time(*txtime);

	// Diagf[^̊_̃ZN^Zo
	// ̎_ōŏIZN^wĂ
	endpos = startpos + *size;
	while (endpos >= capacity) {
		endpos -= capacity;
	}

	offset = (flags & 0x40) ? 146 : 73;
	if (endpos > offset) {
		sec = disk->GetFirstSector(track);
		for (; sec; sec = sec->next) {
			if (((flags ^ sec->flags) & 0xc0) == 0) {
				if (flags & 0x40) {
					offset += 60;
				} else {
					offset += 31;
				}

				offset += sec->size;

				if (flags & 0x40) {
					offset += 0x22;
				} else {
					offset += 0x12;
				}
			} else {
				if (flags & 0x40) {
					offset += (49 + sec->size) * 2;
				} else {
					offset += (94 + sec->size) / 2;
				}
			}

			lastsec = sec;

			if (offset > endpos) {
				break;
			}
		}
	}

	// ݂̃ZN^Ɉړ
	while (true) {
		sector = NextSector(flags);
		if (sector == lastsec) {
			break;
		}
	}

	return result;
}

// ---------------------------------------------------------------------------
//	ReadDiag p̃f[^쐬(FDX`)
//
uint FDU::ReadDiagFdx(uint flags, uint8* data, uint* size, IDR idr, int *stime, int *txtime)
{
	if (!disk)
		return FDC::ST0_AT | FDC::ST1_MA;

	// ԏNA
	*stime = 0;
	*txtime = 0;

	// IDT
	IDR id;
	uint result = SearchID(flags, &id, false, true, stime);
	if (result) {
		return result;
	}

	// IDr(̂R=1Ŕr悤)
	if (sector->id.r != 1 || sector->id.n != idr.n) {
		result |= FDC::ST1_ND;
	}

	// IDCRC
	if (sector->flags & FloppyDisk::idcrc) {
		result |= FDC::ST1_DE;
	}

	// f[^
	if (sector->flags & FloppyDisk::mam) {
		result |= FDC::ST0_AT | FDC::ST1_MA | FDC::ST2_MD;
		return result;
	}

	// ZN^TCYƓȂZN^DATACRCG[𔽉f
	if (sector->id.n == idr.n) {
		if (sector->flags & FloppyDisk::datacrc) {
			result |= FDC::ST1_DE | FDC::ST2_DD;
		}
	}

	// ZN^TCYƈقȂȂDATACRCG[ƂȂ͂
	if (sector->id.n != idr.n) {
		result |= FDC::ST1_DE | FDC::ST2_DD;
	}

	// DELETEDZN^
	if (sector->flags & FloppyDisk::deleted) {
		result |= FDC::ST2_CM;
	}

	// f[^̈ʒu(DAM/DDAMXLbv)
	int offset = sector->fdx_doffset;
	if (sector->flags & 0x40) {
		offset += 64;
	} else {
		offset += 32;
	}

	// f[^̊Jnʒu܂ł̎
	uint pos = GetCurrentPos();
	*stime = (offset >> 4) - pos;
	if (*stime < 0) {
		*stime += disk->GetTrackCapacity();
	}
	*stime = Bytes2Time(*stime);

	// obt@̐L
	if (*size > 0x4000) {
		*size = 0x4000;
	}

	// GR[hf[^ϊ
	FDX::fdxtrack_t *fdxt = disk->GetTrack()->fdx_track;
	FDX::bf_splitclock(
		fdxt->data, fdxt->length, offset, data,
		*size, flags & 0x40 ? true : false, true);

	// f[^]
	*txtime = *size * ((sector->flags & 0x40) ? 1 : 2);
	*txtime = Bytes2Time(*txtime);

	// ϊIʒũZN^ɋIɈړ
	offset += *size * (flags & 0x40 ? 16 : 32);

	// ݈ʒu̎̃ZN^TČ݂̃ZN^߂
	FloppyDisk::Sector* lastsec = disk->GetFirstSector(track);
	FloppyDisk::Sector* nextsec = lastsec->next;
	while (nextsec) {
		if (nextsec->fdx_ioffset > offset) {
			break;
		}
		lastsec = nextsec;
		nextsec = nextsec->next;
	}

	// ݂̃ZN^Ɉړ
	while (true) {
		sector = NextSector(flags);
		if (sector == lastsec) {
			break;
		}
	}

	return result;
}

// ---------------------------------------------------------------------------
//	FDU::WritePreamble
//
uint FDU::WritePreamble(uint flags, int *txtime)
{
	if (!disk)
		return FDC::ST0_AT | FDC::ST1_MA;

	// ԂNA
	*txtime = 0;

	// gbNXVL
	diskmgr->Modified(drive, track);

	// ZN^Sď
	if (!disk->FormatTrack(0, 0))
	{
		return FDC::ST0_AT | FDC::ST0_EC;
	}

	// vAv
	uint len;
	if (flags & 0x40)
	{	// MFM
		len = 80 + 12 + 4 + 50;
	} else
	{	// FM
		len = 40 + 6 + 1 + 26;
	}

	// FM2{
	if (!(flags & 0x40)) {
		len *= 2;
	}

	// ]
	*txtime = Bytes2Time(len);
	return 0;
}

// ---------------------------------------------------------------------------
//	FDU::WriteID
//
uint FDU::WriteID(uint flags, WIDDESC* wid, int *txtime)
{
	if (!disk)
		return 0;

	FloppyDisk::Sector* sec = disk->AddSector(0x80 << Min(7, wid->n));
	sec->id = *wid->idr;
	sec->flags = flags & 0xc0;
	sec->gap3 = wid->gpl;
	sec->update = true;
	memset(sec->image, wid->d, sec->size);

	// ނ܂ł̏v
	uint len;
	if (flags & 0x40)
	{	// MFM
		len = 12 + 4 + 4 + 2 + 22 + 12 + 4 + 2;
	} else
	{	// FM
		len = 6 + 1 + 4 + 2 + 11 + 6 + 1 + 2;
	}
	len += sec->size + sec->gap3;

	// FM2{
	if (!(flags & 0x40)) {
		len *= 2;
	}

	*txtime = Bytes2Time(len);
	return 0;
}

// ---------------------------------------------------------------------------
//	FDU::WritePostamble
//
uint FDU::WritePostamble(uint flags, int *txtime)
{
	if (!disk)
		return 0;

	// vAv
	uint len;
	if (flags & 0x40)
	{	// MFM
		len = 80 + 12 + 4 + 50;	
	} else
	{	// FM
		len = 40 + 6 + 1 + 26;
	}

	// ZN^
	FloppyDisk::Sector* sec = disk->GetFirstSector(track);
	for (; sec; sec = sec->next)
	{
		if (flags & 0x40)
		{	// MFM
			len += 12 + 4 + 4 + 2 + 22 + 12 + 4 + 2;
		} else
		{	// FM
			len += 6 + 1 + 4 + 2 + 11 + 6 + 1 + 2;
		}

		len += sec->size + sec->gap3;
	}

	// FM2{
	if (!(flags & 0x40)) {
		len *= 2;
	}

	// |XgAu̓]
	uint limit = disk->GetTrackCapacity();
	if (len < limit)
	{
		// gbN̗eʂ܂GAPŖ߂鎞
		*txtime = Bytes2Time(limit - len);
	} else {
		// gbN̗eʂ𒴂̂ŃAtH[}bgɂ
		disk->FormatTrack(0, 0);
		*txtime = 0;
	}

	disk->IndexHole();
	return 0;
}
