/*
 *	FM-7 EMULATOR "XM7"
 *
 *	Copyright (C) 1999-2020 ohD(Twitter:@xm6_original)
 *	Copyright (C) 2001-2020 Ryu Takegami (Twitter:@RyuTakegami)
 *	Copyright (C) 2021-2025 GIMONS
 *
 *	[ ⏕c[ ]
 */

#include <string.h>
#include <stdlib.h>
#include <math.h>
#include "xm7.h"
#include "multipag.h"
#include "ttlpalet.h"
#include "apalet.h"
#include "subctrl.h"
#include "display.h"
#include "device.h"
#include "crcgen.h"
#include "fdc.h"
#include "tools.h"


/*
 *	fW^pbge[u
 *	(RGBQUAD)
 */
static BYTE bmp_palet_table[] = {
	0x00, 0x00, 0x00, 0x00,
	0xff, 0x00, 0x00, 0x00,
	0x00, 0x00, 0xff, 0x00,
	0xff, 0x00, 0xff, 0x00,
	0x00, 0xff, 0x00, 0x00,
	0xff, 0xff, 0x00, 0x00,
	0x00, 0xff, 0xff, 0x00,
	0xff, 0xff, 0xff, 0x00,
#if XM7_VER == 1
	0x00, 0x00, 0x00, 0x00,
	0x33, 0x40, 0x1f, 0x00,
	0x5f, 0x78, 0x3a, 0x00,
	0x6a, 0x87, 0x41, 0x00,
	0xa1, 0xcc, 0x63, 0x00,
	0xa7, 0xd5, 0x67, 0x00,
	0xb6, 0xe7, 0x80, 0x00,
	0xbb, 0xee, 0x73, 0x00,
	0x00, 0x00, 0x1e, 0x00,
	0x14, 0x23, 0x56, 0x00,
	0x27, 0x43, 0x89, 0x00,
	0x2b, 0x4a, 0x94, 0x00,
	0x41, 0x71, 0xd1, 0x00,
	0x63, 0x76, 0xd9, 0x00,
	0x49, 0x7f, 0xe8, 0x00,
	0x4b, 0x83, 0xee, 0x00
#endif
};

#if (XM7_VER == 1) && defined(L4CARD)
static BYTE bmp_palet_table_16[] = {
	0x00, 0x00, 0x00, 0x00,
	0xbb, 0x00, 0x00, 0x00,
	0x00, 0x00, 0xbb, 0x00,
	0xbb, 0x00, 0xbb, 0x00,
	0x00, 0xbb, 0x00, 0x00,
	0xbb, 0xbb, 0x00, 0x00,
	0x00, 0xbb, 0xbb, 0x00,
	0xbb, 0xbb, 0xbb, 0x00,
	0x44, 0x44, 0x44, 0x00,
	0xff, 0x44, 0x44, 0x00,
	0x44, 0x44, 0xff, 0x00,
	0xff, 0x44, 0xff, 0x00,
	0x44, 0xff, 0x44, 0x00,
	0xff, 0xff, 0x44, 0x00,
	0x44, 0xff, 0xff, 0x00,
	0xff, 0xff, 0xff, 0x00,
	0x00, 0x00, 0x00, 0x00,
	0x25, 0x30, 0x17, 0x00,
	0x4a, 0x5e, 0x2e, 0x00,
	0x53, 0x69, 0x33, 0x00,
	0x7c, 0x9e, 0x4c, 0x00,
	0x80, 0xa3, 0x4f, 0x00,
	0x8b, 0xb0, 0x55, 0x00,
	0x8e, 0xb5, 0x57, 0x00,
	0x36, 0x45, 0x21, 0x00,
	0x4c, 0x60, 0x2e, 0x00,
	0x6e, 0x8c, 0x44, 0x00,
	0x77, 0x97, 0x49, 0x00,
	0xa6, 0xd3, 0x66, 0x00,
	0xab, 0xd9, 0x69, 0x00,
	0xb7, 0xe8, 0x70, 0x00,
	0xbb, 0xee, 0x73, 0x00,
	0x00, 0x00, 0x17, 0x00,
	0x08, 0x0e, 0x2a, 0x00,
	0x15, 0x24, 0x49, 0x00,
	0x19, 0x2a, 0x51, 0x00,
	0x2d, 0x4c, 0x81, 0x00,
	0x30, 0x50, 0x8b, 0x00,
	0x35, 0x59, 0x94, 0x00,
	0x37, 0x5d, 0x98, 0x00,
	0x0e, 0x17, 0x38, 0x00,
	0x16, 0x25, 0x4b, 0x00,
	0x26, 0x3f, 0x6f, 0x00,
	0x2b, 0x47, 0x7a, 0x00,
	0x45, 0x74, 0xb9, 0x00,
	0x49, 0x7b, 0xc2, 0x00,
	0x50, 0x87, 0xc2, 0x00,
	0x53, 0x8b, 0xc2, 0x00
};
#endif

/* HighColorppbge[u */
#if XM7_VER == 2
static WORD bmp_palet_table_ttl[] = {
	0x0000 | 0x0000 | 0x0000,
	0x0000 | 0x0000 | 0x001f,
	0x7c00 | 0x0000 | 0x0000,
	0x7c00 | 0x0000 | 0x001f,
	0x0000 | 0x03e0 | 0x0000,
	0x0000 | 0x03e0 | 0x001f,
	0x7c00 | 0x03e0 | 0x0000,
	0x7c00 | 0x03e0 | 0x001f
};
#endif

/*
 *	摜kp
 */
#define GAMMA200L	1.483239697419133	/* 摜k̃␳l (200LINE) */
#define GAMMA400L	1.217883285630907	/* 摜k̃␳l (400LINE) */

/*
 *	uNfBXN쐬 Tu
 */
static BOOL FASTCALL make_d77_sub(int fileh, DWORD dat)
{
	BYTE buf[4];

	buf[0] = (BYTE)(dat & 0xff);
	buf[1] = (BYTE)((dat >> 8) & 0xff);
	buf[2] = (BYTE)((dat >> 16) & 0xff);
	buf[3] = (BYTE)((dat >> 24) & 0xff);

	return file_write(fileh, buf, 4);
}

/*
 *	uNfBXN쐬
 */
BOOL FASTCALL make_new_d77(char *fname, char *name, BOOL mediatype)
{
	int fileh;
	BYTE header[0x2b0];
	DWORD offset;
	int i;
	int j;

	/* assert */
	ASSERT(fname);

	/* unused */
#if XM7_VER <= 2
	UNUSED(mediatype);
#endif

	/* t@CI[v */
	fileh = file_open(fname, OPEN_W);
	if (fileh == -1) {
		return FALSE;
	}

	/* wb_쐬 */
	memset(header, 0, sizeof(header));
	if (name != NULL) {
		for (i=0; i<16; i++) {
			if (name[i] == '\0') {
				break;
			}
			header[i] = name[i];
		}
	}
	else {
		strcpy((char*)header, "Default");
	}

	/* x(2D,2DD,2HD) */
#if XM7_VER >= 3
	if (mediatype) {
		header[0x1b] = 0x10;
	}
#endif

	/* wb_ */
	if (!file_write(fileh, header, 0x1c)) {
		file_close(fileh);
		return FALSE;
	}

	/* t@Cg[^TCY */
#if XM7_VER >= 3
	if (mediatype) {
		offset = 0x0e1ab0;
	}
	else {
		offset = 0x073ab0;
	}
#else
	offset = 0x073ab0;
#endif
	if (!make_d77_sub(fileh, offset)) {
		file_close(fileh);
		return FALSE;
	}

	/* gbNItZbg */
	offset = 0x2b0;
	for (i=0; i<84; i++) {
		if (!make_d77_sub(fileh, offset)) {
			file_close(fileh);
			return FALSE;
		}
		offset += 0x1600;
	}
#if XM7_VER >= 3
	if (mediatype) {
		for (i=0; i<80; i++) {
			if (!make_d77_sub(fileh, offset)) {
				file_close(fileh);
				return FALSE;
			}
			offset += 0x1600;
		}
	}
	else {
		/* wb_ */
		if (!file_write(fileh, &header[0x170], 0x2b0 - 0x170)) {
			file_close(fileh);
			return FALSE;
		}
	}
#else
	/* wb_ */
	if (!file_write(fileh, &header[0x170], 0x2b0 - 0x170)) {
		file_close(fileh);
		return FALSE;
	}
#endif

	/* kf[^ */
	memset(header, 0, sizeof(header));
	for (i=0; i<84; i++) {
		for (j=0; j<11; j++) {
			if (!file_write(fileh, header, 0x200)) {
				file_close(fileh);
				return FALSE;
			}
		}
	}
#if XM7_VER >= 3
	if (mediatype) {
		for (i=0; i<80; i++) {
			for (j=0; j<11; j++) {
				if (!file_write(fileh, header, 0x200)) {
					file_close(fileh);
					return FALSE;
				}
			}
		}
	}
#endif

	/* ok */
	file_close(fileh);
	return TRUE;
}

/*
 *	[UfBXN쐬
 */
BOOL FASTCALL make_new_userdisk(char *fname, char *name, BOOL mediatype)
{
	static const BYTE dummyipl[9] = {
		0x1a, 0x50,			/* ORCC	#$50	*/
		0x86, 0x41,			/* LDA	#$41	*/
		0xb7, 0xfd, 0x03,	/* STA	$FD03	*/
		0x20, 0xfe			/* BRA	*		*/
	};

	int fileh;
	BYTE header[0x2b0];
	DWORD offset;
	int i;
	int j;
	int k;

	/* assert */
	ASSERT(fname);

	/* unused */
#if XM7_VER <= 2
	UNUSED(mediatype);
#endif

	/* t@CI[v */
	fileh = file_open(fname, OPEN_W);
	if (fileh == -1) {
		return FALSE;
	}

	/* wb_쐬 */
	memset(header, 0, sizeof(header));
	if (name != NULL) {
		for (i=0; i<16; i++) {
			if (name[i] == '\0') {
				break;
			}
			header[i] = name[i];
		}
	}
	else {
		strcpy((char*)header, "Default");
	}

	/* x(2D,2DD,2HD) */
#if XM7_VER >= 3
	if (mediatype) {
		header[0x1b] = 0x10;
	}
#endif

	/* wb_ */
	if (!file_write(fileh, header, 0x1c)) {
		file_close(fileh);
		return FALSE;
	}

	/* t@Cg[^TCY */
#if XM7_VER >= 3
	if (mediatype) {
		offset = 0x0aa2b0;	/* 0x02b0(header) + (0x0010 + 0x0100) * 16 * 160 */
	}
	else {
		offset = 0x0552b0;	/* 0x02b0(header) + (0x0010 + 0x0100) * 16 * 80 */
	}
#else
	offset = 0x0552b0;		/* 0x02b0(header) + (0x0010 + 0x0100) * 16 * 80 */
#endif
	if (!make_d77_sub(fileh, offset)) {
		file_close(fileh);
		return FALSE;
	}

	/* gbNItZbg */
	offset = 0x2b0;
#if XM7_VER >= 3
	if (mediatype) {
		k = 160;
	}
	else {
		k = 80;
	}
#else
	k = 80;
#endif
	for (i=0; i<164; i++) {
		if (i >= k) {
			offset = 0x0000;
		}
		if (!make_d77_sub(fileh, offset)) {
			file_close(fileh);
			return FALSE;
		}
		if (offset) {
			offset += 0x1100;
		}
	}

	/* kf[^ */
	memset(header, 0, sizeof(header));
	for (i=0; i<k; i++) {
		for (j=1; j<=16; j++) {
			memset(header, 0, 0x10);
			header[0] = (BYTE)(i >> 1);
			header[1] = (BYTE)(i & 1);
			header[2] = (BYTE)j;
			header[3] = 0x01;
			header[4] = 0x10;
			header[14] = 0x00;
			header[15] = 0x01;
			if (!file_write(fileh, header, 0x10)) {
				file_close(fileh);
				return FALSE;
			}

			if ((i == 0) && (j == 1)) {
				/* _~[IPLZN^쐬 */
				memset(header, 0x00, 0x100);
				memcpy(header, dummyipl, sizeof(dummyipl));
			}
			else if ((i == 0) && (j == 3)) {
				/* IDZN^쐬 */
				memset(header, 0x00, 0x100);
#if XM7_VER >= 3
				if (mediatype) {
					header[0] = 0x45;
				}
				else {
					header[0] = 0x53;
				}
#else
				header[0] = 0x53;
#endif
				header[1] = 0x20;
				header[2] = 0x20;
			}
			else if ((i == 2) || (i == 3)) {
				/* FAT/fBNg쐬 */
				memset(header, 0xff, 0x100);
				if ((i == 2) && (j == 1)) {
					header[0] = 0x00;
				}
			}
			else {
				/* ʏZN^쐬 */
				memset(header, 0xe5, 0x100);
			}
			if (!file_write(fileh, header, 0x100)) {
				file_close(fileh);
				return FALSE;
			}
		}
	}

	/* ok */
	file_close(fileh);
	return TRUE;
}

/*
 *	uNe[v쐬
 */
BOOL FASTCALL make_new_t77(char *fname)
{
	int fileh;

	ASSERT(fname);

	/* t@CI[v */
	fileh = file_open(fname, OPEN_W);
	if (fileh == -1) {
		return FALSE;
	}

	/* wb_ */
	if (!file_write(fileh, (BYTE*)"XM7 TAPE IMAGE 0", 16)) {
		file_close(fileh);
		return FALSE;
	}

	/*  */
	file_close(fileh);
	return TRUE;
}

#if XM7_VER == 1 && defined(BUBBLE)
/*
 *	uNouJZbg쐬
 */
BOOL FASTCALL make_new_bubble(char *fname, char *name)
{
	static const char volumelabel[] = "VOL00000";

	int fileh;
	BYTE buffer[0x40];
	BYTE header[0x20];
	DWORD i;

	/* assert */
	ASSERT(fname);

	/* t@CI[v */
	fileh = file_open(fname, OPEN_W);
	if (fileh == -1) {
		return FALSE;
	}

	/* wb_쐬 */
	if (name != NULL) {
		memset(header, 0, sizeof(header));
		for (i=0; i<16; i++) {
			if (name[i] == '\0') {
				break;
			}
			header[i] = name[i];
		}

		/* TCY(32KBŒ) */
		header[0x1b] = 0x80;

		/* wb_ */
		if (!file_write(fileh, header, 0x1c)) {
			file_close(fileh);
			return FALSE;
		}

		/* t@Cg[^TCY */
		if (!make_d77_sub(fileh, 0x008020)) {
			file_close(fileh);
			return FALSE;
		}
	}

	/* kf[^ */
	for (i=0; i<=0x03ff; i++) {
		memset(buffer, 0, sizeof(buffer));
		if (i == 0) {
			/* IDZN^쐬 */
			memcpy((char *)buffer, volumelabel, strlen(volumelabel));
			buffer[8] = 0x08;
			buffer[9] = 0x00;
			buffer[10] = 0x00;
			buffer[11] = 0x01;
		}
		if (!file_write(fileh, buffer, 32)) {
			file_close(fileh);
			return FALSE;
		}
	}

	/* ok */
	file_close(fileh);
	return TRUE;
}
#endif	/* XM7_VER == 1 */

/*
 *	VFDD77ϊ
 */
BOOL FASTCALL conv_vfd_to_d77(char *src, char *dst, char *name)
{
	int files;
	int filed;
	BYTE vfd_h[480];
	BYTE d77_h[0x2b0];
	BYTE *buffer;
	int trk;
	int sec;
	int secs;
	int len;
	int trklen;
	DWORD offset;
	DWORD srclen;
	DWORD wrlen;
	BYTE *header;
	BYTE *ptr;

	/* assert */
	ASSERT(src);
	ASSERT(dst);
	ASSERT(name);

	/* [Nm */
	buffer = (BYTE *)malloc(8192);
	if (buffer == NULL) {
		return FALSE;
	}

	/* VFDt@CI[v */
	files = file_open(src, OPEN_R);
	if (files == -1) {
		free(buffer);
		return FALSE;
	}

	/* ŁAt@CTCY擾Ă */
	srclen = file_getsize(files);

	/* VFDwb_ǂݍ */
	if (!file_read(files, vfd_h, sizeof(vfd_h))) {
		free(buffer);
		file_close(files);
		return FALSE;
	}

	/* D77t@C쐬 */
	filed = file_open(dst, OPEN_W);
	if (filed == -1) {
		free(buffer);
		file_close(files);
		return FALSE;
	}

	/* wb_쐬 */
	memset(d77_h, 0, sizeof(d77_h));
	if (strlen(name) <= 16) {
		strcpy((char*)d77_h, name);
	}
	else {
		memcpy(d77_h, name, 16);
	}

	/* UAwb_ */
	if (!file_write(filed, d77_h, sizeof(d77_h))) {
		free(buffer);
		file_close(files);
		file_close(filed);
		return FALSE;
	}

	/* ݃|C^ */
	wrlen = sizeof(d77_h);

	/* gbN[v */
	header = vfd_h;
	for (trk=0; trk<80; trk++) {
		/* wb_擾 */
		offset = header[3];
		offset *= 256;
		offset |= header[2];
		offset *= 256;
		offset |= header[1];
		offset *= 256;
		offset |= header[0];
		header += 4;
		len = *header++;
		len &= 7;
		if (len >= 4) {
			len = 3;
		}
		secs = *header++;

		/* secs=0ւ̑Ή */
		if (secs == 0) {
			continue;
		}
		else {
			/* ݃|C^L */
			d77_h[trk * 4 + 0x20 + 3] = (BYTE)(wrlen >> 24);
			d77_h[trk * 4 + 0x20 + 2] = (BYTE)((wrlen >> 16) & 255);
			d77_h[trk * 4 + 0x20 + 1] = (BYTE)((wrlen >> 8) & 255);
			d77_h[trk * 4 + 0x20 + 0] = (BYTE)(wrlen & 255);
		}

		/* gbNvZ */
		switch (len) {
			case 0:
				trklen = secs * (128 + 16);
				break;
			case 1:
				trklen = secs * (256 + 16);
				break;
			case 2:
				trklen = secs * (512 + 16);
				break;
			case 3:
				trklen = secs * (1024 + 16);
				break;
		}

		/* wb_ */
		if ((offset > srclen) | (trklen > 8192)) {
			free(buffer);
			file_close(files);
			file_close(filed);
			return FALSE;
		}

		/* V[N */
		if (!file_seek(files, offset)) {
			free(buffer);
			file_close(files);
			file_close(filed);
			return FALSE;
		}

		/* ZN^[v */
		ptr = buffer;
		for (sec=1; sec<=secs; sec++) {
			memset(ptr, 0, 0x10);
			/* C,H,R,N */
			ptr[0] = (BYTE)(trk >> 1);
			ptr[1] = (BYTE)(trk & 1);
			ptr[2] = (BYTE)sec;
			ptr[3] = (BYTE)len;
			/* ZN^ */
			ptr[4] = (BYTE)(secs);
			/* ZN^f[^ǂݍ */
			switch (len) {
				case 0:
					ptr[0x0e] = 0x80;
					ptr += 0x10;
					file_read(files, ptr, 128);
					ptr += 128;
					break;
				case 1:
					ptr[0x0f] = 0x01;
					ptr += 0x10;
					file_read(files, ptr, 256);
					ptr += 256;
					break;
				case 2:
					ptr[0x0f] = 0x02;
					ptr += 0x10;
					file_read(files, ptr, 512);
					ptr += 512;
					break;
				case 3:
					ptr[0x0f] = 0x04;
					ptr += 0x10;
					file_read(files, ptr, 1024);
					ptr += 1024;
					break;
			}
		}

		/* ꊇ */
		if (!file_write(filed, buffer, trklen)) {
			free(buffer);
			file_close(files);
			file_close(filed);
			return FALSE;
		}

		/* ݃|C^i߂ */
		wrlen += trklen;
	}

	/* t@CTCYݒ */
	d77_h[0x1f] = (BYTE)((wrlen >> 24) & 0xff);
	d77_h[0x1e] = (BYTE)((wrlen >> 16) & 0xff);
	d77_h[0x1d] = (BYTE)((wrlen >> 8) & 0xff);
	d77_h[0x1c] = (BYTE)(wrlen & 0xff);

	/* ēxAwb_ */
	if (!file_seek(filed, 0)) {
		free(buffer);
		file_close(files);
		file_close(filed);
		return FALSE;
	}
	if (!file_write(filed, d77_h, sizeof(d77_h))) {
		free(buffer);
		file_close(files);
		file_close(filed);
		return FALSE;
	}

	/* ׂďI */
	free(buffer);
	file_close(files);
	file_close(filed);
	return TRUE;
}

/*
 *	2D/2DD/2HDD77ϊ
 */
BOOL FASTCALL conv_2d_to_d77(char *src, char *dst, char *name)
{
	int files;
	int filed;
	BYTE d77_h[0x2b0];
	BYTE *buffer;
	BYTE *ptr;
	DWORD offset;
	int trk;
	int sec;
	int size;
	int max_track;
#if XM7_VER == 1 && defined(SFDC)
	int max_sector;
	int num;
	int len;
	int den;
#endif	/* XM7_VER == 1 && defined(SFDC) */

	/* assert */
	ASSERT(src);
	ASSERT(dst);
	ASSERT(name);

	/* [Nm */
#if XM7_VER == 1 && defined(SFDC)
	buffer = (BYTE *)malloc(0x1ba0);
#else	/* XM7_VER == 1 && defined(SFDC) */
	buffer = (BYTE *)malloc(0x1100);
#endif	/* XM7_VER == 1 && defined(SFDC) */
	if (buffer == NULL) {
		return FALSE;
	}

	/* 2Dt@CI[vAt@CTCY`FbN */
	files = file_open(src, OPEN_R);
	if (files == -1) {
		free(buffer);
		return FALSE;
	}
	size = file_getsize(files);
#if XM7_VER >= 2
	if ((size != 327680) && (size != 655360)) {
#elif XM7_VER == 1 && defined(SFDC)
	if ((size != 327680) && (size != 1021696)) {
#else
	if (size != 327680) {
#endif
		free(buffer);
		file_close(files);
		return FALSE;
	}

	/* D77t@C쐬 */
	filed = file_open(dst, OPEN_W);
	if (filed == -1) {
		free(buffer);
		file_close(files);
		return FALSE;
	}

	/* wb_쐬 */
	memset(d77_h, 0, sizeof(d77_h));
	if (strlen(name) <= 16) {
		strcpy((char*)d77_h, name);
	}
	else {
		memcpy(d77_h, name, 16);
	}

	/* t@CTCY */
#if XM7_VER >= 2
	if (size == 655360) {
		d77_h[0x1b] = 0x10;
		d77_h[0x1c] = 0xb0;
		d77_h[0x1d] = 0xa2;
		d77_h[0x1e] = 0x0a;
		max_track = 160;
	}
	else {
		d77_h[0x1b] = 0x00;
		d77_h[0x1c] = 0xb0;
		d77_h[0x1d] = 0x52;
		d77_h[0x1e] = 0x05;
		max_track = 80;
	}
#elif XM7_VER == 1 && defined(SFDC)
	if (size == 1021696) {
		d77_h[0x1b] = 0x20;
		d77_h[0x1c] = 0xf0;
		d77_h[0x1d] = 0x93;
		d77_h[0x1e] = 0x10;
		max_track = 154;
		max_sector = 26;
	}
	else {
		d77_h[0x1b] = 0x00;
		d77_h[0x1c] = 0xb0;
		d77_h[0x1d] = 0x52;
		d77_h[0x1e] = 0x05;
		max_track = 80;
		max_sector = 16;
	}
#else
	d77_h[0x1b] = 0x00;
	d77_h[0x1c] = 0xb0;
	d77_h[0x1d] = 0x52;
	d77_h[0x1e] = 0x05;
	max_track = 80;
#endif

	/* gbNItZbg */
	offset = 0x2b0;
	for (trk=0; trk<max_track; trk++) {
		d77_h[0x20 + trk * 4 + 0] = (BYTE)(offset & 0xff);
		d77_h[0x20 + trk * 4 + 1] = (BYTE)((offset >> 8) & 0xff);
		d77_h[0x20 + trk * 4 + 2] = (BYTE)((offset >> 16) & 0xff);
		d77_h[0x20 + trk * 4 + 3] = (BYTE)((offset >> 24) & 0xff);
#if XM7_VER == 1 && defined(SFDC)
		if ((size == 1021696) && (trk == 0)) {
			offset += 0x090 * max_sector;
		}
		else {
			offset += 0x110 * max_sector;
		}
#else	/* XM7_VER == 1 && defined(SFDC) */
		offset += 0x1100;
#endif	/* XM7_VER == 1 && defined(SFDC) */
	}

	/* wb_ */
	if (!file_write(filed, d77_h, sizeof(d77_h))) {
		free(buffer);
		file_close(files);
		file_close(filed);
		return FALSE;
	}

	/* gbN[v */
	for (trk=0; trk<max_track; trk++) {
		ptr = buffer;
#if XM7_VER == 1 && defined(SFDC)
		/* ZN^[v */
		if ((size == 1021696) && (trk == 0)) {
			num = 0;
			len = 128;
			den = 0x40;
		}
		else {
			num = 1;
			len = 256;
			den = 0;
		}
		for (sec=1; sec<=max_sector; sec++) {
			memset(ptr, 0, 0x10);
			/* C,H,R,N */
			ptr[0] = (BYTE)(trk >> 1);
			ptr[1] = (BYTE)(trk & 1);
			ptr[2] = (BYTE)sec;
			ptr[3] = num;

			/* L^x */
			ptr[6] = den;

			/* ZN^AOX */
			ptr[4] = max_sector;
			ptr[0x0e] = (len % 256);
			ptr[0x0f] = (len / 256);
			ptr += 0x10;

			/* f[^ǂݍ */
			file_read(files, ptr, len);
			ptr += len;
		}

		/* ꊇ */
		if (!file_write(filed, buffer, (len + 0x10) * max_sector)) {
			free(buffer);
			file_close(files);
			file_close(filed);
			return FALSE;
		}
#else	/* XM7_VER == 1 && defined(SFDC) */
		/* ZN^[v */
		for (sec=1; sec<=16; sec++) {
			memset(ptr, 0, 0x10);
			/* C,H,R,N */
			ptr[0] = (BYTE)(trk >> 1);
			ptr[1] = (BYTE)(trk & 1);
			ptr[2] = (BYTE)sec;
			ptr[3] = 1;

			/* ZN^AOX */
			ptr[4] = 16;
			ptr[0x0f] = 0x01;
			ptr += 0x10;

			/* f[^ǂݍ */
			file_read(files, ptr, 256);
			ptr += 256;
		}

		/* ꊇ */
		if (!file_write(filed, buffer, 0x1100)) {
			free(buffer);
			file_close(files);
			file_close(filed);
			return FALSE;
		}
#endif	/* XM7_VER == 1 && defined(SFDC) */
	}

	/* ׂďI */
	free(buffer);
	file_close(files);
	file_close(filed);
	return TRUE;
}

/*
 *	VTPT77ϊ
 *	PoCgo
 */
static BOOL FASTCALL vtp_conv_write(int handle, BYTE dat)
{
	int i;
	BYTE buf[44];

	/* X^[grbgݒ */
	buf[0] = 0x00;
	buf[1] = 0x34;
	buf[2] = 0x80;
	buf[3] = 0x1a;
	buf[4] = 0x00;
	buf[5] = 0x1a;

	/* Xgbvrbgݒ */
	buf[38] = 0x80;
	buf[39] = 0x2f;
	buf[40] = 0x00;
	buf[41] = 0x37;
	buf[42] = 0x80;
	buf[43] = 0x2f;

	/* 8rbg */
	for (i=0; i<8; i++) {
		if (dat & 0x01) {
			buf[i * 4 + 6 + 0] = 0x80;
			buf[i * 4 + 6 + 1] = 0x30;
			buf[i * 4 + 6 + 2] = 0x00;
			buf[i * 4 + 6 + 3] = 0x30;
		}
		else {
			buf[i * 4 + 6 + 0] = 0x80;
			buf[i * 4 + 6 + 1] = 0x18;
			buf[i * 4 + 6 + 2] = 0x00;
			buf[i * 4 + 6 + 3] = 0x1a;
		}
		dat >>= 1;
	}

	/* 44oCgɊg債ď */
	if (!file_write(handle, buf, 44)) {
		return FALSE;
	}

	return TRUE;
}

/*
 *	VTPT77ϊ
 */
BOOL FASTCALL conv_vtp_to_t77(char *src, char *dst)
{
	int files;
	int filed;
	int i;
	BYTE buf[44];
	BYTE hdr[4];
	char *header = "XM7 TAPE IMAGE 0";
	BYTE dat;
	int count;

	/* assert */
	ASSERT(src);
	ASSERT(dst);

	/* VTPt@CI[v */
	files = file_open(src, OPEN_R);
	if (files == -1) {
		return FALSE;
	}

	/* T77t@C쐬 */
	filed = file_open(dst, OPEN_W);
	if (filed == -1) {
		file_close(files);
		return FALSE;
	}

	/* wb_ */
	if (!file_write(filed, (BYTE*)header, 16)) {
		file_close(filed);
		file_close(files);
		return FALSE;
	}

	/* T77f[^쐬 */
	while (TRUE) {
		/* JE^ */
		count = 0;

		/* S~f[^XLbv */
		/* 32ȏ̘A 0xFF 𔭌ł܂ŌJԂ */
		do {
			/* rŃt@CIꍇ͐I */
			if (!file_read(files, &dat, 1)) {
				file_close(filed);
				file_close(files);
				return TRUE;
			}

			/* 0xFF𔭌łJE^AȊOȂJE^NA */
			if (dat == 0xff) {
				count ++;
			}
			else {
				count = 0;
			}
		} while (count < 32);

		/* 1t@C̃f[^쐬 */
		do {
			/* wb_ */
			/* rŃt@CIꍇ͐I */
			do {
				do {
					if (!file_read(files, &hdr[0], 1)) {
						file_close(filed);
						file_close(files);
						return TRUE;
					}
					count ++;
				} while (hdr[0] != 0x01);

				if (!file_read(files, &hdr[1], 1)) {
					file_close(filed);
					file_close(files);
					return TRUE;
				}
			} while ((hdr[0] != 0x01) || (hdr[1] != 0x3c));

			/* c̃wb_ǂݍ */
			for (i = 2; i < 4; i++) {
				if (!file_read(files, &hdr[i], 1)) {
					file_close(filed);
					file_close(files);
					return FALSE;
				}
			}

			/* t@CubNꍇGap̑OɃ}[Jݒ */
			if (hdr[2] == 0x00) {
				buf[0] = 0;
				buf[1] = 0;
				buf[2] = 0x7f;
				buf[3] = 0xff;
				if (!file_write(filed, buf, 4)) {
					file_close(filed);
					file_close(files);
					return FALSE;
				}
			}

			/* Gap */
			/* f[^͑S 0xFF ɓꂷ */
			for (i = 0; i < count; i++) {
				if (!vtp_conv_write(filed, 0xFF)) {
					file_close(filed);
					file_close(files);
					return FALSE;
				}
			}

			/* wb_ */
			for (i = 0; i < 4; i++) {
				if (!vtp_conv_write(filed, hdr[i])) {
					file_close(filed);
					file_close(files);
					return FALSE;
				}
			}

			/* f[^E`FbNT */
			for (i = 0; i <= hdr[3]; i++) {
				if (!file_read(files, &dat, 1)) {
					file_close(filed);
					file_close(files);
					return FALSE;
				}
				if (!vtp_conv_write(filed, dat)) {
					file_close(filed);
					file_close(files);
					return FALSE;
				}
			}

			/* JE^ */
			count = 0;
		} while (hdr[2] != 0xFF);
	}
}

#if XM7_VER == 1 && defined(BUBBLE)
/*
 *	BBLB77ϊ
 */
BOOL FASTCALL conv_bbl_to_b77(char *src, char *dst, char *name)
{
	int files;
	int filed;
	BYTE buffer[0x40];
	BYTE b77_h[0x20];
	int size;
	int page;
	int max_page;
	int max_byte;

	/* assert */
	ASSERT(src);
	ASSERT(dst);
	ASSERT(name);

	/* BBLt@CI[vAt@CTCY`FbN */
	files = file_open(src, OPEN_R);
	if (files == -1) {
		return FALSE;
	}
	size = file_getsize(files);
	if ((size != 32768) && (size != 131072)) {
		file_close(files);
		return FALSE;
	}

	/* B77t@C쐬 */
	filed = file_open(dst, OPEN_W);
	if (filed == -1) {
		file_close(files);
		return FALSE;
	}

	/* wb_쐬 */
	memset(b77_h, 0, sizeof(b77_h));
	if (strlen(name) <= 16) {
		strcpy((char*)b77_h, name);
	}
	else {
		memcpy(b77_h, name, 16);
	}

	/* t@CTCY */
	if (size == 131072) {
		b77_h[0x1b] = 0x90;
		b77_h[0x1c] = 0x20;
		b77_h[0x1d] = 0x00;
		b77_h[0x1e] = 0x02;
		max_page = 0x0800;
		max_byte = 0x0040;
	}
	else {
		b77_h[0x1b] = 0x80;
		b77_h[0x1c] = 0x20;
		b77_h[0x1d] = 0x80;
		b77_h[0x1e] = 0x00;
		max_page = 0x0400;
		max_byte = 0x0020;
	}

	/* wb_ */
	if (!file_write(filed, b77_h, sizeof(b77_h))) {
		file_close(files);
		file_close(filed);
		return FALSE;
	}

	/* y[W[v */
	for (page=0; page<max_page; page++) {

		/* f[^ǂݍ */
		file_read(files, buffer, max_byte);

		/* f[^ */
		if (!file_write(filed, buffer, max_byte)) {
			file_close(files);
			file_close(filed);
			return FALSE;
		}
	}

	/* ׂďI */
	file_close(files);
	file_close(filed);
	return TRUE;
}
#endif

/*
 *	BMPwb_
 */
static BOOL FASTCALL bmp_header_sub(int fileh)
{
	BYTE filehdr[14];
	BYTE infohdr[40];

	ASSERT(fileh >= 0);

	/* \̃NA */
	memset(filehdr, 0, sizeof(filehdr));
	memset(infohdr, 0, sizeof(infohdr));

	/* BITMAPFILEHEADER */
	filehdr[0] = 'B';
	filehdr[1] = 'M';

#if XM7_VER >= 3
	switch (screen_mode) {
		case SCR_262144 :	/* 262144F t@CTCY 14+40+768000 */
							filehdr[2] = 0x36;
							filehdr[3] = 0xb8;
							filehdr[4] = 0x0b;
							break;
		case SCR_4096	:	/* 4096F t@CTCY 14+40+512000 */
							filehdr[2] = 0x36;
							filehdr[3] = 0xd0;
							filehdr[4] = 0x07;
							break;
		default		:	/* 640x200/400 8F t@CTCY 14+40+16*4+128000 */
							filehdr[2] = 0x76;
							filehdr[3] = 0xf4;
							filehdr[4] = 0x01;
	}
#elif XM7_VER >= 2
	if (mode320) {
		/* 4096F t@CTCY 14+40+512000 */
		filehdr[2] = 0x36;
		filehdr[3] = 0xd0;
		filehdr[4] = 0x07;
	}
	else {
		/* 640x200 8F t@CTCY 14+40+16*4+128000 */
		filehdr[2] = 0x76;
		filehdr[3] = 0xf4;
		filehdr[4] = 0x01;
	}
#else
	/* 640x200 8F t@CTCY 14+40+16*4+128000 */
	filehdr[2] = 0x76;
	filehdr[3] = 0xf4;
	filehdr[4] = 0x01;
#endif

	/* rbg}bvւ̃ItZbg */
#if XM7_VER >= 2
#if XM7_VER >= 3
	if (screen_mode & SCR_ANALOG) {
#else
	if (mode320) {
#endif
		/* 4096F or 262144F */
		filehdr[10] = 14 + 40;
	}
	else {
		/* 8F */
		filehdr[10] = 14 + 40 + (16 * 4);
	}
#else
	/* 8F */
	filehdr[10] = 14 + 40 + (16 * 4);
#endif

	/* BITMAPFILEHEADER  */
	if (!file_write(fileh, filehdr, sizeof(filehdr))) {
		return FALSE;
	}

	/* BITMAPINFOHEADER */
	infohdr[0] = 40;
	infohdr[4] = 0x80;
	infohdr[5] = 0x02;
	infohdr[8] = 0x90;
	infohdr[9] = 0x01;
	infohdr[12] = 0x01;
	/* BiBitCount */
#if XM7_VER >= 3
	switch (screen_mode) {
		case	SCR_262144	:	/* 262144F */
								infohdr[14] = 24;
								break;
		case	SCR_4096	:	/* 4096F */
								infohdr[14] = 16;
								break;
		default				:	/* 640x200/400 8F */
								infohdr[14] = 4;
								break;
	}
#elif XM7_VER >= 2
	if (mode320) {
		infohdr[14] = 16;
	}
	else {
		infohdr[14] = 4;
	}
#else
	infohdr[14] = 4;
#endif

	/* BITMAPFILEHEADER  */
	if (!file_write(fileh, infohdr, sizeof(infohdr))) {
		return FALSE;
	}

	return TRUE;
}

/*
 *	8F[h摜kp J[
 */
static WORD FASTCALL mix_color(BYTE *palet_table, BYTE palet_count, MONITORTYPE monitortype, double gamma)
{
	DWORD col;
	DWORD colcount;
	DWORD brg;
	double maxbrg;
	int i;
	int j;
#if XM7_VER == 1
	int base;

	if (monitortype == MONITOR_COLOR) {
		base = 0;
	}
	else if (monitortype == MONITOR_GREEN) {
		base = 32;
	}
	else if (monitortype == MONITOR_AMBERYELLOW) {
		base = 64;
	}
#else
	UNUSED(monitortype);
#endif

	maxbrg = pow(255., 1 / gamma);
	col = 0;

	for (i=0; i<3; i++) {
		colcount = 0;
		for (j=0; j<palet_count; j++) {
#if XM7_VER == 1
			colcount += bmp_palet_table[palet_table[j] * 4 + i + base];
#else
			colcount += bmp_palet_table[palet_table[j] * 4 + i];
#endif
		}
		colcount /= palet_count;
		brg = (int)((pow((double)colcount, 1 / gamma) / maxbrg) * 31.) << 15;
		col = (col | brg) >> 5;
	}

	return (WORD)(col & 0x7fff);
}

/*
 *	16F[h摜kp J[
 */
#if XM7_VER == 1 && defined(L4CARD)
static WORD FASTCALL mix_color_16(BYTE *palet_table, MONITORTYPE monitortype)
{
	DWORD col;
	DWORD colcount;
	DWORD brg;
	double maxbrg;
	int i;
	int j;
	int base;

	if (monitortype == MONITOR_COLOR) {
		base = 0;
	}
	else if (monitortype == MONITOR_GREEN) {
		base = 64;
	}
	else if (monitortype == MONITOR_AMBERYELLOW) {
		base = 128;
	}

	maxbrg = pow(255., 1 / GAMMA400L);
	col = 0;

	for (i=0; i<3; i++) {
		colcount = 0;
		for (j=0; j<4; j++) {
			colcount += bmp_palet_table_16[palet_table[j] * 4 + i + base];
		}
		colcount /= 4;
		brg = (int)((pow((double)colcount, 1 / GAMMA400L) / maxbrg) * 31.)
			<< 15;
		col = (col | brg) >> 5;
	}

	return (WORD)(col & 0x7fff);
}
#endif

/*
 *	BMPwb_ (k摜p)
 */
static BOOL FASTCALL bmp_header_sub2(int fileh)
{
	BYTE filehdr[14];
	BYTE infohdr[40];

	ASSERT(fileh >= 0);

	/* \̃NA */
	memset(filehdr, 0, sizeof(filehdr));
	memset(infohdr, 0, sizeof(infohdr));

	/* BITMAPFILEHEADER */
	filehdr[0] = 'B';
	filehdr[1] = 'M';

	/* BiBitCount */
#if XM7_VER >= 3
	if (screen_mode == SCR_262144) {
		/* t@CTCY 14+40+192000 */
		/* 262144F */
		filehdr[2] = 0x36;
		filehdr[3] = 0xee;
		filehdr[4] = 0x02;
	}
	else {
		/* t@CTCY 14+40+128000 */
		/* 8F/4096F */
		filehdr[2] = 0x36;
		filehdr[3] = 0xf4;
		filehdr[4] = 0x01;
	}
#else
	/* t@CTCY 14+40+128000 */
	/* 8F/4096F */
	filehdr[2] = 0x36;
	filehdr[3] = 0xf4;
	filehdr[4] = 0x01;
#endif

	/* rbg}bvւ̃ItZbg */
	filehdr[10] = 14+40;

	/* BITMAPFILEHEADER  */
	if (!file_write(fileh, filehdr, sizeof(filehdr))) {
		return FALSE;
	}

	/* BITMAPINFOHEADER */
	infohdr[0] = 40;
	infohdr[4] = 0x40;
	infohdr[5] = 0x01;
	infohdr[8] = 0xC8;
	infohdr[9] = 0x00;
	infohdr[12] = 0x01;

	/* BiBitCount */
#if XM7_VER >= 3
	if (screen_mode == SCR_262144) {
		/* 262144F */
		infohdr[14] = 24;
	}
	else {
		/* 8F/4096F */
		infohdr[14] = 16;
	}
#else
	/* 8F/4096F */
	infohdr[14] = 16;
#endif

	/* BITMAPFILEHEADER  */
	if (!file_write(fileh, infohdr, sizeof(infohdr))) {
		return FALSE;
	}

	return TRUE;
}

/*
 *	BMPpbg(8F/16F̂)
 */
static BOOL FASTCALL bmp_palette_sub(int fileh, MONITORTYPE monitortype)
{
	int i;
	BYTE *p;
	int vpage;
#if XM7_VER == 1
	int base;
#endif

	ASSERT(fileh >= 0);

	/* \y[Wl */
	vpage = (~(multi_page >> 4)) & 0x07;

#if XM7_VER == 1
	if (monitortype == MONITOR_COLOR) {
		base = 0;
	}
	else if (monitortype == MONITOR_GREEN) {
		base = 64;
	}
	else if (monitortype == MONITOR_AMBERYELLOW) {
		base = 128;
	}

#ifdef L4CARD
	if (enable_400line) {
		if (crt_flag) {
			/* Œpbg16F */
			if (!file_write(fileh, &bmp_palet_table_16[base], 4 * 16)) {
				return FALSE;
			}
		}
		else {
			/* 16F(Cʓ|Ȃ̂ŋʏ) */
			p = &bmp_palet_table_16[base];
			for (i=0; i<16; i++) {
				if (!file_write(fileh, p, 4)) {
					return FALSE;
				}
			}
		}

		return TRUE;
	}
#endif

	if (monitortype == MONITOR_COLOR) {
		base = 0;
	}
	else if (monitortype == MONITOR_GREEN) {
		base = 32;
	}
	else if (monitortype == MONITOR_AMBERYELLOW) {
		base = 64;
	}

	/* pbg8F */
	for (i=0; i<8; i++) {
		if (crt_flag) {
			p = &bmp_palet_table[ (ttl_palet[i & vpage] & 0x07) * 4 + base];
		}
		else {
			p = &bmp_palet_table[base];
		}
		if (!file_write(fileh, p, 4)) {
			return FALSE;
		}
	}

	/* 8F(Cʓ|Ȃ̂ŋʏ) */
	p = &bmp_palet_table[base];
	for (i=0; i<8; i++) {
		if (!file_write(fileh, p, 4)) {
			return FALSE;
		}
	}
#else
	UNUSED(monitortype);

	/* pbg8F */
	for (i=0; i<8; i++) {
		if (crt_flag) {
			/* ʏ탂[h */
			p = &bmp_palet_table[ (ttl_palet[i & vpage] & 0x07) * 4 ];
		}
		else {
			/*  */
			p = bmp_palet_table;
		}
		if (!file_write(fileh, p, 4)) {
			return FALSE;
		}
	}

	/* 8F(Cʓ|Ȃ̂ŋʏ) */
	p = bmp_palet_table;
	for (i=0; i<8; i++) {
		if (!file_write(fileh, p, 4)) {
			return FALSE;
		}
	}
#endif

	return TRUE;
}

/*
 *  BMPf[^(320[h)
 */
#if XM7_VER >= 2
static BOOL FASTCALL bmp_320_sub(int fileh, BOOL fullscan, MONITORTYPE monitortype)
{
	int x, y;
	int offset;
	int i;
	BYTE buffer[2][1280];
	int dat;
	BYTE bit;
	WORD color;
	int mask;
	BYTE *vramptr;
#if XM7_VER >= 3
	WORD dx1,dx2;
	BOOL winy;
#endif

	ASSERT(fileh >= 0);
#if XM7_VER >= 3
	UNUSED(monitortype);
#endif

	/* ItZbgݒ */
	offset = 40 * 199;

#if XM7_VER >= 3
	/* EBhË̃NbsOs */
	window_clip(1);
	dx1 = (WORD)(window_dx1 >> 3);
	dx2 = (WORD)(window_dx2 >> 3);
#endif

	/* }XN擾 */
	mask = 0;
#if XM7_VER == 2
	if (monitortype == MONITOR_FORCETTLCOLOR) {
		mask = ((~multi_page) >> 4) & 0x07;
	}
	else {
		if (!(multi_page & 0x10)) {
			mask |= 0x000f;
		}
		if (!(multi_page & 0x20)) {
			mask |= 0x00f0;
		}
		if (!(multi_page & 0x40)) {
			mask |= 0x0f00;
		}
	}
#else
	if (!(multi_page & 0x10)) {
		mask |= 0x000f;
	}
	if (!(multi_page & 0x20)) {
		mask |= 0x00f0;
	}
	if (!(multi_page & 0x40)) {
		mask |= 0x0f00;
	}
#endif

	/* 0ŏ */
	memset(buffer[0], 0, sizeof(buffer[0]));

	/* y[v */
	for (y=0; y<200; y++) {
#if XM7_VER >= 3
		winy = (((199 - y) >= window_dy1) && ((199 - y) <= window_dy2));

		/* x[v */
		for (x=0; x<40; x++) {
			bit = 0x80;
			if (winy && (x >= dx1) && (x < dx2)) {
				/* EBhE(ubN) */
				vramptr = vram_bdptr;
			}
			else {
				/* EBhEO(\ubN) */
				vramptr = vram_dptr;
			}
#else
		/* x[v */
		for (x=0; x<40; x++) {
			bit = 0x80;
			vramptr = vram_dptr;
#endif

			/* rbg[v */
			for (i=0; i<8; i++) {
				dat = 0;

#if XM7_VER >= 3
				/* G] */
				if (vramptr[offset + 0x10000] & bit) {
					dat |= 0x800;
				}
				if (vramptr[offset + 0x12000] & bit) {
					dat |= 0x400;
				}
				if (vramptr[offset + 0x14000] & bit) {
					dat |= 0x200;
				}
				if (vramptr[offset + 0x16000] & bit) {
					dat |= 0x100;
				}

				/* R] */
				if (vramptr[offset + 0x08000] & bit) {
					dat |= 0x080;
				}
				if (vramptr[offset + 0x0a000] & bit) {
					dat |= 0x040;
				}
				if (vramptr[offset + 0x0c000] & bit) {
					dat |= 0x020;
				}
				if (vramptr[offset + 0x0e000] & bit) {
					dat |= 0x010;
				}

				/* B] */
				if (vramptr[offset + 0x00000] & bit) {
					dat |= 0x008;
				}
				if (vramptr[offset + 0x02000] & bit) {
					dat |= 0x004;
				}
				if (vramptr[offset + 0x04000] & bit) {
					dat |= 0x002;
				}
				if (vramptr[offset + 0x06000] & bit) {
					dat |= 0x001;
				}

				/* AiOpbgf[^擾 */
				dat &= mask;
				color = apalet_r[dat];
				color <<= 1;
				if (apalet_r[dat] > 0) {
					color |= 1;
				}
				color <<= 4;

				color |= apalet_g[dat];
				color <<= 1;
				if (apalet_g[dat] > 0) {
					color |= 1;
				}
				color <<= 4;

				color |= apalet_b[dat];
				color <<= 1;
				if (apalet_b[dat] > 0) {
					color |= 1;
				}
#else
				if (monitortype == MONITOR_FORCETTLCOLOR) {
					/* G] */
					if (vramptr[offset + 0x08000] & bit) {
						dat |= 0x04;
					}

					/* R] */
					if (vramptr[offset + 0x04000] & bit) {
						dat |= 0x02;
					}

					/* B] */
					if (vramptr[offset + 0x00000] & bit) {
						dat |= 0x01;
					}

					/* TTLpbgf[^擾 */
					color = bmp_palet_table_ttl[ttl_palet[dat & mask] & 7];
				}
				else {
					/* G] */
					if (vramptr[offset + 0x08000] & bit) {
						dat |= 0x800;
					}
					if (vramptr[offset + 0x0a000] & bit) {
						dat |= 0x400;
					}
					if (vramptr[offset + 0x14000] & bit) {
						dat |= 0x200;
					}
					if (vramptr[offset + 0x16000] & bit) {
						dat |= 0x100;
					}

					/* R] */
					if (vramptr[offset + 0x04000] & bit) {
						dat |= 0x080;
					}
					if (vramptr[offset + 0x06000] & bit) {
						dat |= 0x040;
					}
					if (vramptr[offset + 0x10000] & bit) {
						dat |= 0x020;
					}
					if (vramptr[offset + 0x12000] & bit) {
						dat |= 0x010;
					}

					/* B] */
					if (vramptr[offset + 0x00000] & bit) {
						dat |= 0x008;
					}
					if (vramptr[offset + 0x02000] & bit) {
						dat |= 0x004;
					}
					if (vramptr[offset + 0x0c000] & bit) {
						dat |= 0x002;
					}
					if (vramptr[offset + 0x0e000] & bit) {
						dat |= 0x001;
					}

					/* AiOpbgf[^擾 */
					dat &= mask;
					color = apalet_r[dat];
					color <<= 1;
					if (apalet_r[dat] > 0) {
						color |= 1;
					}
					color <<= 4;

					color |= apalet_g[dat];
					color <<= 1;
					if (apalet_g[dat] > 0) {
						color |= 1;
					}
					color <<= 4;

					color |= apalet_b[dat];
					color <<= 1;
					if (apalet_b[dat] > 0) {
						color |= 1;
					}
				}
#endif

				/* CRTtO */
				if (!crt_flag) {
					color = 0;
				}

				/* Q񑱂ē̂ */
				buffer[1][x * 32 + i * 4 + 0] = (BYTE)(color & 255);
				buffer[1][x * 32 + i * 4 + 1] = (BYTE)(color >> 8);
				buffer[1][x * 32 + i * 4 + 2] = (BYTE)(color & 255);
				buffer[1][x * 32 + i * 4 + 3] = (BYTE)(color >> 8);

				/* ̃rbg */
				bit >>= 1;
			}
			offset++;
		}

		/* tXL */
		if (fullscan) {
			memcpy(buffer[0], buffer[1], sizeof(buffer[1]));
		}

		/*  */
		if (!file_write(fileh, (BYTE *)buffer, sizeof(buffer))) {
			return FALSE;
		}

		/* y(߂) */
		offset -= (40 * 2);
	}

	return TRUE;
}
#endif

#if XM7_VER >= 3
/*
 *  BMPf[^(26F[h)
 */
static BOOL FASTCALL bmp_256k_sub(int fileh, BOOL fullscan)
{
	int x, y;
	int offset;
	int i;
	BYTE buffer[2][1920];
	BYTE bit;
	BYTE r, g, b;
	
	ASSERT(fileh >= 0);

	/* ItZbgݒ */
	offset = 40 * 199;

	/* Oŏ */
	memset(buffer[0], 0, sizeof(buffer[0]));

	/* y[v */
	for (y=0; y<200; y++) {

		/* x[v */
		for (x=0; x<40; x++) {
			bit = 0x80;
			/* rbg[v */
			for (i=0; i<8; i++) {
				r = g = b = 0;

				if (!(multi_page & 0x40)) {
					/* G] */
					if (vram_c[offset + 0x10000] & bit) {
						g |= 0x20;
					}
					if (vram_c[offset + 0x12000] & bit) {
						g |= 0x10;
					}
					if (vram_c[offset + 0x14000] & bit) {
						g |= 0x08;
					}
					if (vram_c[offset + 0x16000] & bit) {
						g |= 0x04;
					}
					if (vram_c[offset + 0x28000] & bit) {
						g |= 0x02;
					}
					if (vram_c[offset + 0x2a000] & bit) {
						g |= 0x01;
					}
				}

				if (!(multi_page & 0x20)) {
					/* R] */
					if (vram_c[offset + 0x08000] & bit) {
						r |= 0x20;
					}
					if (vram_c[offset + 0x0a000] & bit) {
						r |= 0x10;
					}
					if (vram_c[offset + 0x0c000] & bit) {
						r |= 0x08;
					}
					if (vram_c[offset + 0x0e000] & bit) {
						r |= 0x04;
					}
					if (vram_c[offset + 0x20000] & bit) {
						r |= 0x02;
					}
					if (vram_c[offset + 0x22000] & bit) {
						r |= 0x01;
					}
				}

				if (!(multi_page & 0x10)) {
					/* B] */
					if (vram_c[offset + 0x00000] & bit) {
						b |= 0x20;
					}
					if (vram_c[offset + 0x02000] & bit) {
						b |= 0x10;
					}
					if (vram_c[offset + 0x04000] & bit) {
						b |= 0x08;
					}
					if (vram_c[offset + 0x06000] & bit) {
						b |= 0x04;
					}
					if (vram_c[offset + 0x18000] & bit) {
						b |= 0x02;
					}
					if (vram_c[offset + 0x1a000] & bit) {
						b |= 0x01;
					}
				}

				/* CRTtO */
				if (!crt_flag) {
					r = g = b = 0;
				}

				/* Q񑱂ē̂ */
				buffer[1][x * 48 + i * 6 + 0] = (BYTE)truecolorbrightness[b];
				buffer[1][x * 48 + i * 6 + 1] = (BYTE)truecolorbrightness[g];
				buffer[1][x * 48 + i * 6 + 2] = (BYTE)truecolorbrightness[r];
				buffer[1][x * 48 + i * 6 + 3] = (BYTE)truecolorbrightness[b];
				buffer[1][x * 48 + i * 6 + 4] = (BYTE)truecolorbrightness[g];
				buffer[1][x * 48 + i * 6 + 5] = (BYTE)truecolorbrightness[r];

				/* ̃rbg */
				bit >>= 1;
			}
			offset++;
		}

		/* tXL */
		if (fullscan) {
			memcpy(buffer[0], buffer[1], sizeof(buffer[1]));
		}

		/*  */
		if (!file_write(fileh, (BYTE *)buffer, sizeof(buffer))) {
			return FALSE;
		}

		/* y(߂) */
		offset -= (40 * 2);
	}

	return TRUE;
}
#endif

/*
 *  BMPf[^(640[h)
 */
static BOOL FASTCALL bmp_640_sub(int fileh, BOOL fullscan)
{
	int x, y;
	int i;
	int offset;
	BYTE bit;
	BYTE buffer[2][320];
	BYTE *vramptr;
#if XM7_VER >= 3
	WORD dx1, dx2;
	BOOL winy;
#endif

	ASSERT(fileh >= 0);

	/* ItZbgݒ */
	offset = 80 * 199;

#if XM7_VER >= 3
	/* EBhË̃NbsOs */
	window_clip(0);
	dx1 = (WORD)(window_dx1 >> 3);
	dx2 = (WORD)(window_dx2 >> 3);

	/* J[9ŏ */
	memset(buffer[0], 0x99, sizeof(buffer[0]));

	/* y[v */
	for (y=0; y<200; y++) {
		winy = (((199 - y) >= window_dy1) && ((199 - y) <= window_dy2));

		/* UNA */
		memset(buffer[1], 0, sizeof(buffer[1]));

		/* x[v */
		for (x=0; x<80; x++) {
			bit = 0x80;
			if (winy && (x >= dx1) && (x < dx2)) {
				/* EBhE(ubN) */
				vramptr = vram_bdptr;
			}
			else {
				/* EBhEO(\ubN) */
				vramptr = vram_dptr;
			}
#else
	/* J[9ŏ */
	memset(buffer[0], 0x99, sizeof(buffer[0]));

	/* y[v */
	for (y=0; y<200; y++) {

		/* UNA */
		memset(buffer[1], 0, sizeof(buffer[1]));

		/* x[v */
		for (x=0; x<80; x++) {
			bit = 0x80;
#if XM7_VER >= 2
			vramptr = vram_dptr;
#else
			vramptr = vram_c;
#endif
#endif

			/* bit[v */
			for (i=0; i<4; i++) {
#if XM7_VER >= 3
				if (vramptr[offset + 0x00000] & bit) {
					buffer[1][x * 4 + i] |= 0x10;
				}
				if (vramptr[offset + 0x08000] & bit) {
					buffer[1][x * 4 + i] |= 0x20;
				}
				if (vramptr[offset + 0x10000] & bit) {
					buffer[1][x * 4 + i] |= 0x40;
				}
				bit >>= 1;

				if (vramptr[offset + 0x00000] & bit) {
					buffer[1][x * 4 + i] |= 0x01;
				}
				if (vramptr[offset + 0x08000] & bit) {
					buffer[1][x * 4 + i] |= 0x02;
				}
				if (vramptr[offset + 0x10000] & bit) {
					buffer[1][x * 4 + i] |= 0x04;
				}
				bit >>= 1;
#else
				if (vramptr[offset + 0x0000] & bit) {
					buffer[1][x * 4 + i] |= 0x10;
				}
				if (vramptr[offset + 0x4000] & bit) {
					buffer[1][x * 4 + i] |= 0x20;
				}
				if (vramptr[offset + 0x8000] & bit) {
					buffer[1][x * 4 + i] |= 0x40;
				}
				bit >>= 1;

				if (vramptr[offset + 0x0000] & bit) {
					buffer[1][x * 4 + i] |= 0x01;
				}
				if (vramptr[offset + 0x4000] & bit) {
					buffer[1][x * 4 + i] |= 0x02;
				}
				if (vramptr[offset + 0x8000] & bit) {
					buffer[1][x * 4 + i] |= 0x04;
				}
				bit >>= 1;
#endif
			}
			offset++;
		}

		/* tXL */
		if (fullscan) {
			memcpy(buffer[0], buffer[1], sizeof(buffer[1]));
		}

		/*  */
		if (!file_write(fileh, (BYTE *)buffer, sizeof(buffer))) {
			return FALSE;
		}

		/* y(߂) */
		offset -= (80 * 2);
	}

	return TRUE;
}

/*
 *  BMPf[^(640[h,J[^400C)
 */
#if XM7_VER >= 2
static BOOL FASTCALL bmp_p400c_sub(int fileh)
{
	int x, y;
	int i;
	int offset;
	BYTE bit;
	BYTE buffer[2][320];
	BYTE *vramptr;
#if XM7_VER >= 3
	WORD dx1, dx2;
	BOOL winy;
#endif

	ASSERT(fileh >= 0);

	/* ItZbgݒ */
	offset = 80 * 199;

#if XM7_VER >= 3
	/* EBhË̃NbsOs */
	window_clip(0);
	dx1 = (WORD)(window_dx1 >> 3);
	dx2 = (WORD)(window_dx2 >> 3);

	/* y[v */
	for (y=0; y<200; y++) {
		winy = (((199 - y) >= window_dy1) && ((199 - y) <= window_dy2));

		/* UNA */
		memset(buffer, 0, sizeof(buffer));

		/* x[v */
		for (x=0; x<80; x++) {
			bit = 0x80;
			if (winy && (x >= dx1) && (x < dx2)) {
				/* EBhE(ubN) */
				vramptr = vram_bdblk;
			}
			else {
				/* EBhEO(\ubN) */
				vramptr = vram_dblk;
			}
#else
	/* y[v */
	for (y=0; y<200; y++) {

		/* UNA */
		memset(buffer, 0, sizeof(buffer));

		/* x[v */
		for (x=0; x<80; x++) {
			bit = 0x80;
			vramptr = vram_c;
#endif

			/* bit[v */
			for (i=0; i<4; i++) {
#if XM7_VER >= 3
				if (vramptr[offset + 0x00000] & bit) {
					buffer[1][x * 4 + i] |= 0x10;
				}
				if (vramptr[offset + 0x08000] & bit) {
					buffer[1][x * 4 + i] |= 0x20;
				}
				if (vramptr[offset + 0x10000] & bit) {
					buffer[1][x * 4 + i] |= 0x40;
				}
				if (vramptr[offset + 0x04000] & bit) {
					buffer[0][x * 4 + i] |= 0x10;
				}
				if (vramptr[offset + 0x0c000] & bit) {
					buffer[0][x * 4 + i] |= 0x20;
				}
				if (vramptr[offset + 0x14000] & bit) {
					buffer[0][x * 4 + i] |= 0x40;
				}
				bit >>= 1;

				if (vramptr[offset + 0x00000] & bit) {
					buffer[1][x * 4 + i] |= 0x01;
				}
				if (vramptr[offset + 0x08000] & bit) {
					buffer[1][x * 4 + i] |= 0x02;
				}
				if (vramptr[offset + 0x10000] & bit) {
					buffer[1][x * 4 + i] |= 0x04;
				}
				if (vramptr[offset + 0x04000] & bit) {
					buffer[0][x * 4 + i] |= 0x01;
				}
				if (vramptr[offset + 0x0c000] & bit) {
					buffer[0][x * 4 + i] |= 0x02;
				}
				if (vramptr[offset + 0x14000] & bit) {
					buffer[0][x * 4 + i] |= 0x04;
				}
				bit >>= 1;
#else
				if (vramptr[offset + 0x00000] & bit) {
					buffer[1][x * 4 + i] |= 0x10;
				}
				if (vramptr[offset + 0x04000] & bit) {
					buffer[1][x * 4 + i] |= 0x20;
				}
				if (vramptr[offset + 0x08000] & bit) {
					buffer[1][x * 4 + i] |= 0x40;
				}
				if (vramptr[offset + 0x0c000] & bit) {
					buffer[0][x * 4 + i] |= 0x10;
				}
				if (vramptr[offset + 0x10000] & bit) {
					buffer[0][x * 4 + i] |= 0x20;
				}
				if (vramptr[offset + 0x14000] & bit) {
					buffer[0][x * 4 + i] |= 0x40;
				}
				bit >>= 1;

				if (vramptr[offset + 0x00000] & bit) {
					buffer[1][x * 4 + i] |= 0x01;
				}
				if (vramptr[offset + 0x04000] & bit) {
					buffer[1][x * 4 + i] |= 0x02;
				}
				if (vramptr[offset + 0x08000] & bit) {
					buffer[1][x * 4 + i] |= 0x04;
				}
				if (vramptr[offset + 0x0c000] & bit) {
					buffer[0][x * 4 + i] |= 0x01;
				}
				if (vramptr[offset + 0x10000] & bit) {
					buffer[0][x * 4 + i] |= 0x02;
				}
				if (vramptr[offset + 0x14000] & bit) {
					buffer[0][x * 4 + i] |= 0x04;
				}
				bit >>= 1;
#endif
			}
			offset++;
		}

		/*  */
		if (!file_write(fileh, (BYTE *)buffer, sizeof(buffer))) {
			return FALSE;
		}

		/* y(߂) */
		offset -= (80 * 2);
	}

	return TRUE;
}
#endif

#if XM7_VER == 1
/*
 *  BMPf[^(640[h,mN^400C)
 */
static BOOL FASTCALL bmp_p400m_sub(int fileh)
{
	int x, y;
	int i;
	int offset;
	BYTE bit;
	BYTE buffer[2][320];
	BYTE pal[8];
	BYTE *vramptr;
	BYTE col1, col2, col3, col4;

	ASSERT(fileh >= 0);

	/* Ff[^ */
	for (i=0; i<8; i++) {
		if (crt_flag) {
			pal[i] = (BYTE)(ttl_palet[i & ((~(multi_page >> 4)) & 7)] & 7);
		}
		else {
			pal[i] = 0;
		}
	}

	/* ItZbgݒ */
	offset = 80 * 199;

	/* J[0ŏ */
	memset(buffer[0], 0x00, sizeof(buffer[0]));

	/* y[v */
	for (y=0; y<200; y++) {

		/* UNA */
		memset(buffer[1], 0, sizeof(buffer[1]));

		/* x[v */
		for (x=0; x<80; x++) {
			bit = 0x80;
			vramptr = vram_c;

			/* bit[v */
			for (i=0; i<4; i++) {
				col1 = 0;
				if (vramptr[offset + 0x0000] & bit) {
					col1 |= 0x01;
				}
				if (vramptr[offset + 0x4000] & bit) {
					col1 |= 0x02;
				}
				if (vramptr[offset + 0x8000] & bit) {
					col1 |= 0x04;
				}
				col1 = pal[col1];
				bit >>= 1;

				col2 = 0;
				if (vramptr[offset + 0x0000] & bit) {
					col2 |= 0x01;
				}
				if (vramptr[offset + 0x4000] & bit) {
					col2 |= 0x02;
				}
				if (vramptr[offset + 0x8000] & bit) {
					col2 |= 0x04;
				}
				col2 = pal[col2];
				bit >>= 1;

				if (col1 & 4) {
					col3 = 7;
				}
				else {
					col3 = 0;
				}
				if (col1 & 2) {
					col1 = 7;
				}
				else {
					col1 = 0;
				}
				if (col2 & 4) {
					col4 = 7;
				}
				else {
					col4 = 0;
				}
				if (col2 & 2) {
					col2 = 7;
				}
				else {
					col2 = 0;
				}

				buffer[0][x * 4 + i] = (BYTE)((col1 << 4) | col2);
				buffer[1][x * 4 + i] = (BYTE)((col3 << 4) | col4);
			}

			offset++;
		}

		/*  */
		if (!file_write(fileh, (BYTE *)buffer, sizeof(buffer))) {
			return FALSE;
		}

		/* y(߂) */
		offset -= (80 * 2);
	}

	return TRUE;
}
#endif

#if XM7_VER >= 3
/*
 *  BMPf[^(400C[h)
 */
static BOOL FASTCALL bmp_400l_sub(int fileh)
{
	int x, y;
	int i;
	int offset;
	BYTE bit;
	BYTE buffer[320];
	BYTE *vramptr;
	WORD dx1, dx2;
	BOOL winy;

	ASSERT(fileh >= 0);

	/* ItZbgݒ */
	offset = 80 * 399;

	/* EBhË̃NbsOs */
	window_clip(2);
	dx1 = (WORD)(window_dx1 >> 3);
	dx2 = (WORD)(window_dx2 >> 3);

	/* y[v */
	for (y=0; y<400; y++) {
		winy = (((399 - y) >= window_dy1) && ((399 - y) <= window_dy2));

		/* UNA */
		memset(buffer, 0, sizeof(buffer));

		/* x[v */
		for (x=0; x<80; x++) {
			bit = 0x80;
			if (winy && (x >= dx1) && (x < dx2)) {
				/* EBhE(ubN) */
				vramptr = vram_bdptr;
			}
			else {
				/* EBhEO(\ubN) */
				vramptr = vram_dptr;
			}

			/* bit[v */
			for (i=0; i<4; i++) {
				if (vramptr[offset + 0x00000] & bit) {
					buffer[x * 4 + i] |= 0x10;
				}
				if (vramptr[offset + 0x08000] & bit) {
					buffer[x * 4 + i] |= 0x20;
				}
				if (vramptr[offset + 0x10000] & bit) {
					buffer[x * 4 + i] |= 0x40;
				}
				bit >>= 1;

				if (vramptr[offset + 0x00000] & bit) {
					buffer[x * 4 + i] |= 0x01;
				}
				if (vramptr[offset + 0x08000] & bit) {
					buffer[x * 4 + i] |= 0x02;
				}
				if (vramptr[offset + 0x10000] & bit) {
					buffer[x * 4 + i] |= 0x04;
				}
				bit >>= 1;
			}
			offset++;
		}

		/*  */
		if (!file_write(fileh, buffer, sizeof(buffer))) {
			return FALSE;
		}

		/* y(߂) */
		offset -= (80 * 2);
	}

	return TRUE;
}
#endif

#if XM7_VER == 1 && defined(L4CARD)
/*
 *  BMPf[^(L4 400C[h)
 */
static BOOL FASTCALL bmp_400l4_sub(int fileh, MONITORTYPE monitortype)
{
	int x, y;
	int i;
	int offset;
	BYTE bit, bit2;
	BYTE buffer[320];
	WORD taddr, gaddr, textbase;
	BYTE csr_st, csr_ed, csr_type;
	BYTE raster, lines;
	BYTE col, chr, atr, chr_dat;
	BOOL enable_page;

	ASSERT(fileh >= 0);
	UNUSED(monitortype);

	/* ItZbgݒ */
	offset = 80 * 399;

	/* eLXgWJ ݒ(S) */
	csr_st = (BYTE)(crtc_register[10] & 0x1f);
	csr_ed = (BYTE)(crtc_register[11] & 0x1f);
	csr_type = (BYTE)((crtc_register[10] & 0x60) >> 5);
	lines = (BYTE)((crtc_register[9] & 0x1f) + 1);

	/* y[v */
	for (y=0; y<400; y++) {
		/* UNA */
		memset(buffer, 0, sizeof(buffer));

		/* eLXgWJ ݒ(X^P) */
		textbase = (WORD)text_start_addr;
		textbase += (WORD)(((399 - y) / lines) * (crtc_register[1] << 2));
		textbase &= 0xffe;
		raster = (BYTE)((399 - y) % lines);

		/* x[v */
		for (x=0; x<80; x++) {
			bit = 0x80;
			if (!width40_flag || !(x & 1)) {
				bit2 = 0x80;

				/* LN^R[hEAgr[g擾 */
				if (width40_flag) {
					taddr = (WORD)((textbase + (x & ~1)) & 0xffe);
				}
				else {
					taddr = (WORD)((textbase + (x << 1)) & 0xffe);
				}
				chr = tvram_c[taddr + 0];
				atr = tvram_c[taddr + 1];

				/* Agr[g`Fݒ */
				col = (BYTE)((atr & 0x07) | ((atr & 0x20) >> 2));

				/* tHgf[^擾(Agr[g/J[\܂) */
				if ((!(atr & 0x10) || text_blink) && (raster < 16)) {
					chr_dat = subcg_l4[(WORD)(chr << 4) + raster];
				}
				else {
					chr_dat = 0x00;
				}
				if (atr & 0x08) {
					chr_dat ^= (BYTE)0xff;
				}
				if (csr_type != 1) {
					if (((taddr == cursor_addr) &&
						(cursor_blink || !csr_type)) &&
						((raster >= csr_st) && (raster <= csr_ed))) {
						chr_dat ^= (BYTE)0xff;
					}
				}
			}

			/* GVRAM AhX擾 */
			gaddr = (WORD)((offset + vram_offset[0]) & 0x7fff);
			enable_page = FALSE;
			if (gaddr >= 0x4000) {
				if (!(multi_page & 0x10)) {
					enable_page = TRUE;
				}
			}
			else {
				if (!(multi_page & 0x20)) {
					enable_page = TRUE;
				}
			}

			/* bit[v */
			for (i=0; i<4; i++) {
				if (chr_dat & bit2) {
					buffer[x * 4 + i] |= (BYTE)(col << 4);
				}
				else if ((vram_c[gaddr] & bit) && enable_page) {
					buffer[x * 4 + i] |= (BYTE)(ttl_palet[1] << 4);
				}
				else {
					buffer[x * 4 + i] |= (BYTE)(ttl_palet[0] << 4);
				}
				bit >>= 1;
				if (!width40_flag) {
					bit2 >>= 1;
				}

				if (chr_dat & bit2) {
					buffer[x * 4 + i] |= (BYTE)col;
				}
				else if ((vram_c[gaddr] & bit) && enable_page) {
					buffer[x * 4 + i] |= (BYTE)ttl_palet[1];
				}
				else {
					buffer[x * 4 + i] |= (BYTE)ttl_palet[0];
				}
				bit >>= 1;
				bit2 >>= 1;
			}
			offset++;
		}

		/*  */
		if (!file_write(fileh, buffer, sizeof(buffer))) {
			return FALSE;
		}

		/* y(߂) */
		offset -= (80 * 2);
	}

	return TRUE;
}
#endif

/*
 *  BMPf[^(320[hEk摜)
 */
#if XM7_VER >= 2
static BOOL FASTCALL bmp_320_sub2(int fileh, MONITORTYPE monitortype)
{
	int x, y;
	int offset;
	int i;
	BYTE buffer[640];
	int dat;
	BYTE bit;
	WORD color;
	int mask;
	BYTE *vramptr;
#if XM7_VER >= 3
	WORD dx1,dx2;
	BOOL winy;
#endif

	ASSERT(fileh >= 0);
#if XM7_VER >= 3
	UNUSED(monitortype);
#endif

	/* ItZbgݒ */
	offset = 40 * 199;

#if XM7_VER >= 3
	/* EBhË̃NbsOs */
	window_clip(1);
	dx1 = (WORD)(window_dx1 >> 3);
	dx2 = (WORD)(window_dx2 >> 3);
#endif

	/* }XN擾 */
	mask = 0;
#if XM7_VER == 2
	if (monitortype == MONITOR_FORCETTLCOLOR) {
		mask = ((~multi_page) >> 4) & 0x07;
	}
	else {
		if (!(multi_page & 0x10)) {
			mask |= 0x000f;
		}
		if (!(multi_page & 0x20)) {
			mask |= 0x00f0;
		}
		if (!(multi_page & 0x40)) {
			mask |= 0x0f00;
		}
	}
#else
	if (!(multi_page & 0x10)) {
		mask |= 0x000f;
	}
	if (!(multi_page & 0x20)) {
		mask |= 0x00f0;
	}
	if (!(multi_page & 0x40)) {
		mask |= 0x0f00;
	}
#endif

	/* y[v */
	for (y=0; y<200; y++) {
#if XM7_VER >= 3
		winy = (((199 - y) >= window_dy1) && ((199 - y) <= window_dy2));

		/* x[v */
		for (x=0; x<40; x++) {
			bit = 0x80;
			if (winy && (x >= dx1) && (x < dx2)) {
				/* EBhE(ubN) */
				vramptr = vram_bdptr;
			}
			else {
				/* EBhEO(\ubN) */
				vramptr = vram_dptr;
			}
#else
		/* x[v */
		for (x=0; x<40; x++) {
			bit = 0x80;
			vramptr = vram_dptr;
#endif

			/* rbg[v */
			for (i=0; i<8; i++) {
				dat = 0;

#if XM7_VER >= 3
				/* G] */
				if (vramptr[offset + 0x10000] & bit) {
					dat |= 0x800;
				}
				if (vramptr[offset + 0x12000] & bit) {
					dat |= 0x400;
				}
				if (vramptr[offset + 0x14000] & bit) {
					dat |= 0x200;
				}
				if (vramptr[offset + 0x16000] & bit) {
					dat |= 0x100;
				}

				/* R] */
				if (vramptr[offset + 0x08000] & bit) {
					dat |= 0x080;
				}
				if (vramptr[offset + 0x0a000] & bit) {
					dat |= 0x040;
				}
				if (vramptr[offset + 0x0c000] & bit) {
					dat |= 0x020;
				}
				if (vramptr[offset + 0x0e000] & bit) {
					dat |= 0x010;
				}

				/* B] */
				if (vramptr[offset + 0x00000] & bit) {
					dat |= 0x008;
				}
				if (vramptr[offset + 0x02000] & bit) {
					dat |= 0x004;
				}
				if (vramptr[offset + 0x04000] & bit) {
					dat |= 0x002;
				}
				if (vramptr[offset + 0x06000] & bit) {
					dat |= 0x001;
				}

				/* AiOpbgf[^擾 */
				dat &= mask;
				color = apalet_r[dat];
				color <<= 1;
				if (apalet_r[dat] > 0) {
					color |= 1;
				}
				color <<= 4;

				color |= apalet_g[dat];
				color <<= 1;
				if (apalet_g[dat] > 0) {
					color |= 1;
				}
				color <<= 4;

				color |= apalet_b[dat];
				color <<= 1;
				if (apalet_b[dat] > 0) {
					color |= 1;
				}
#else
				if (monitortype == MONITOR_FORCETTLCOLOR) {
					/* G] */
					if (vramptr[offset + 0x08000] & bit) {
						dat |= 0x04;
					}

					/* R] */
					if (vramptr[offset + 0x04000] & bit) {
						dat |= 0x02;
					}

					/* B] */
					if (vramptr[offset + 0x00000] & bit) {
						dat |= 0x01;
					}

					/* TTLpbgf[^擾 */
					color = bmp_palet_table_ttl[ttl_palet[dat & mask] & 7];
				}
				else {
					/* G] */
					if (vramptr[offset + 0x08000] & bit) {
						dat |= 0x800;
					}
					if (vramptr[offset + 0x0a000] & bit) {
						dat |= 0x400;
					}
					if (vramptr[offset + 0x14000] & bit) {
						dat |= 0x200;
					}
					if (vramptr[offset + 0x16000] & bit) {
						dat |= 0x100;
					}

					/* R] */
					if (vramptr[offset + 0x04000] & bit) {
						dat |= 0x080;
					}
					if (vramptr[offset + 0x06000] & bit) {
						dat |= 0x040;
					}
					if (vramptr[offset + 0x10000] & bit) {
						dat |= 0x020;
					}
					if (vramptr[offset + 0x12000] & bit) {
						dat |= 0x010;
					}

					/* B] */
					if (vramptr[offset + 0x00000] & bit) {
						dat |= 0x008;
					}
					if (vramptr[offset + 0x02000] & bit) {
						dat |= 0x004;
					}
					if (vramptr[offset + 0x0c000] & bit) {
						dat |= 0x002;
					}
					if (vramptr[offset + 0x0e000] & bit) {
						dat |= 0x001;
					}

					/* AiOpbgf[^擾 */
					dat &= mask;
					color = apalet_r[dat];
					color <<= 1;
					if (apalet_r[dat] > 0) {
						color |= 1;
					}
					color <<= 4;

					color |= apalet_g[dat];
					color <<= 1;
					if (apalet_g[dat] > 0) {
						color |= 1;
					}
					color <<= 4;

					color |= apalet_b[dat];
					color <<= 1;
					if (apalet_b[dat] > 0) {
						color |= 1;
					}
				}
#endif

				/* CRTtO */
				if (!crt_flag) {
					color = 0;
				}

				buffer[x * 16 + i * 2 + 0] = (BYTE)(color & 255);
				buffer[x * 16 + i * 2 + 1] = (BYTE)(color >> 8);

				/* ̃rbg */
				bit >>= 1;
			}
			offset++;
		}

		/*  */
		if (!file_write(fileh, buffer, sizeof(buffer))) {
			return FALSE;
		}

		/* y(߂) */
		offset -= (40 * 2);
	}

	return TRUE;
}
#endif

#if XM7_VER >= 3
/*
 *  BMPf[^(26F[hEk摜)
 */
static BOOL FASTCALL bmp_256k_sub2(int fileh)
{
	int x, y;
	int offset;
	int i;
	BYTE buffer[960];
	BYTE bit;
	BYTE r, g, b;

	ASSERT(fileh >= 0);

	/* ItZbgݒ */
	offset = 40 * 199;

	/* y[v */
	for (y=0; y<200; y++) {

		/* x[v */
		for (x=0; x<40; x++) {
			bit = 0x80;
			/* rbg[v */
			for (i=0; i<8; i++) {
				r = g = b = 0;

				if (!(multi_page & 0x40)) {
					/* G] */
					if (vram_c[offset + 0x10000] & bit) {
						g |= 0x20;
					}
					if (vram_c[offset + 0x12000] & bit) {
						g |= 0x10;
					}
					if (vram_c[offset + 0x14000] & bit) {
						g |= 0x08;
					}
					if (vram_c[offset + 0x16000] & bit) {
						g |= 0x04;
					}
					if (vram_c[offset + 0x28000] & bit) {
						g |= 0x02;
					}
					if (vram_c[offset + 0x2a000] & bit) {
						g |= 0x01;
					}
				}

				if (!(multi_page & 0x20)) {
					/* R] */
					if (vram_c[offset + 0x08000] & bit) {
						r |= 0x20;
					}
					if (vram_c[offset + 0x0a000] & bit) {
						r |= 0x10;
					}
					if (vram_c[offset + 0x0c000] & bit) {
						r |= 0x08;
					}
					if (vram_c[offset + 0x0e000] & bit) {
						r |= 0x04;
					}
					if (vram_c[offset + 0x20000] & bit) {
						r |= 0x02;
					}
					if (vram_c[offset + 0x22000] & bit) {
						r |= 0x01;
					}
				}

				if (!(multi_page & 0x10)) {
					/* B] */
					if (vram_c[offset + 0x00000] & bit) {
						b |= 0x20;
					}
					if (vram_c[offset + 0x02000] & bit) {
						b |= 0x10;
					}
					if (vram_c[offset + 0x04000] & bit) {
						b |= 0x08;
					}
					if (vram_c[offset + 0x06000] & bit) {
						b |= 0x04;
					}
					if (vram_c[offset + 0x18000] & bit) {
						b |= 0x02;
					}
					if (vram_c[offset + 0x1a000] & bit) {
						b |= 0x01;
					}
				}

				/* CRTtO */
				if (!crt_flag) {
					r = g = b = 0;
				}

				buffer[x * 24 + i * 3 + 0] = (BYTE)truecolorbrightness[b];
				buffer[x * 24 + i * 3 + 1] = (BYTE)truecolorbrightness[g];
				buffer[x * 24 + i * 3 + 2] = (BYTE)truecolorbrightness[r];

				/* ̃rbg */
				bit >>= 1;
			}
			offset++;
		}

		/*  */
		if (!file_write(fileh, buffer, sizeof(buffer))) {
			return FALSE;
		}

		/* y(߂) */
		offset -= (40 * 2);
	}

	return TRUE;
}
#endif

/*
 *  BMPf[^(640[hEk摜)
 */
static BOOL FASTCALL bmp_640_sub2(int fileh, MONITORTYPE monitortype)
{
	int x, y;
	int i;
	int offset;
	BYTE bit;
	BYTE buffer[640];
	BYTE *vramptr;
	BYTE pal[8];
	BYTE col[2];
	WORD color;
#if XM7_VER >= 3
	WORD dx1, dx2;
	BOOL winy;
#endif

	ASSERT(fileh >= 0);

	/* Ff[^ */
	for (i=0; i<8; i++) {
		if (crt_flag) {
			pal[i] = (BYTE)(ttl_palet[i & ((~(multi_page >> 4)) & 7)] & 7);
		}
		else {
			pal[i] = 0;
		}
	}

	/* ItZbgݒ */
	offset = 80 * 199;

#if XM7_VER >= 3
	/* EBhË̃NbsOs */
	window_clip(0);
	dx1 = (WORD)(window_dx1 >> 3);
	dx2 = (WORD)(window_dx2 >> 3);

	/* y[v */
	for (y=0; y<200; y++) {
		winy = (((199 - y) >= window_dy1) && ((199 - y) <= window_dy2));

		/* x[v */
		for (x=0; x<80; x++) {
			bit = 0x80;
			if (winy && (x >= dx1) && (x < dx2)) {
				/* EBhE(ubN) */
				vramptr = vram_bdptr;
			}
			else {
				/* EBhEO(\ubN) */
				vramptr = vram_dptr;
			}
#else
	/* y[v */
	for (y=0; y<200; y++) {

		/* x[v */
		for (x=0; x<80; x++) {
			bit = 0x80;
#if XM7_VER >= 2
			vramptr = vram_dptr;
#else
			vramptr = vram_c;
#endif
#endif

			/* bit[v */
			for (i=0; i<4; i++) {
				col[0] = 0;
				col[1] = 0;

#if XM7_VER >= 3
				if (vramptr[offset + 0x00000] & bit) {
					col[0] |= 1;
				}
				if (vramptr[offset + 0x08000] & bit) {
					col[0] |= 2;
				}
				if (vramptr[offset + 0x10000] & bit) {
					col[0] |= 4;
				}
				bit >>= 1;

				if (vramptr[offset + 0x00000] & bit) {
					col[1] |= 1;
				}
				if (vramptr[offset + 0x08000] & bit) {
					col[1] |= 2;
				}
				if (vramptr[offset + 0x10000] & bit) {
					col[1] |= 4;
				}
				bit >>= 1;
#else
				if (vramptr[offset + 0x0000] & bit) {
					col[0] |= 1;
				}
				if (vramptr[offset + 0x4000] & bit) {
					col[0] |= 2;
				}
				if (vramptr[offset + 0x8000] & bit) {
					col[0] |= 4;
				}
				bit >>= 1;

				if (vramptr[offset + 0x0000] & bit) {
					col[1] |= 1;
				}
				if (vramptr[offset + 0x4000] & bit) {
					col[1] |= 2;
				}
				if (vramptr[offset + 0x8000] & bit) {
					col[1] |= 4;
				}
				bit >>= 1;
#endif

				col[0] = pal[col[0]];
				col[1] = pal[col[1]];
				color = mix_color(col, 2, monitortype, GAMMA200L);
				buffer[x * 8 + i * 2 + 0] = (BYTE)(color & 255);
				buffer[x * 8 + i * 2 + 1] = (BYTE)(color >> 8);
			}
			offset++;
		}

		/*  */
		if (!file_write(fileh, buffer, sizeof(buffer))) {
			return FALSE;
		}

		/* y(߂) */
		offset -= (80 * 2);
	}

	return TRUE;
}
/*
 *  BMPf[^(640[hEJ[^400CEk摜)
 */
#if XM7_VER >= 2
static BOOL FASTCALL bmp_p400c_sub2(int fileh)
{
	int x, y;
	int i;
	int offset;
	BYTE bit;
	BYTE buffer[640];
	BYTE *vramptr;
	BYTE pal[8];
	BYTE col[4];
	WORD color;
#if XM7_VER >= 3
	WORD dx1, dx2;
	BOOL winy;
#endif

	ASSERT(fileh >= 0);

	/* Ff[^ */
	for (i=0; i<8; i++) {
		if (crt_flag) {
			pal[i] = (BYTE)(ttl_palet[i & ((~(multi_page >> 4)) & 7)] & 7);
		}
		else {
			pal[i] = 0;
		}
	}

	/* ItZbgݒ */
	offset = 80 * 199;

#if XM7_VER >= 3
	/* EBhË̃NbsOs */
	window_clip(0);
	dx1 = (WORD)(window_dx1 >> 3);
	dx2 = (WORD)(window_dx2 >> 3);

	/* y[v */
	for (y=0; y<200; y++) {
		winy = (((199 - y) >= window_dy1) && ((199 - y) <= window_dy2));

		/* x[v */
		for (x=0; x<80; x++) {
			bit = 0x80;
			if (winy && (x >= dx1) && (x < dx2)) {
				/* EBhE(ubN) */
				vramptr = vram_bdblk;
			}
			else {
				/* EBhEO(\ubN) */
				vramptr = vram_dblk;
			}
#else
	/* y[v */
	for (y=0; y<200; y++) {

		/* x[v */
		for (x=0; x<80; x++) {
			bit = 0x80;
			vramptr = vram_c;
#endif

			/* bit[v */
			for (i=0; i<4; i++) {
				col[0] = 0;
				col[1] = 0;
				col[2] = 0;
				col[3] = 0;

#if XM7_VER >= 3
				if (vramptr[offset + 0x00000] & bit) {
					col[0] |= 1;
				}
				if (vramptr[offset + 0x08000] & bit) {
					col[0] |= 2;
				}
				if (vramptr[offset + 0x10000] & bit) {
					col[0] |= 4;
				}
				if (vramptr[offset + 0x04000] & bit) {
					col[2] |= 1;
				}
				if (vramptr[offset + 0x0c000] & bit) {
					col[2] |= 2;
				}
				if (vramptr[offset + 0x14000] & bit) {
					col[2] |= 4;
				}
				bit >>= 1;

				if (vramptr[offset + 0x00000] & bit) {
					col[1] |= 1;
				}
				if (vramptr[offset + 0x08000] & bit) {
					col[1] |= 2;
				}
				if (vramptr[offset + 0x10000] & bit) {
					col[1] |= 4;
				}
				if (vramptr[offset + 0x04000] & bit) {
					col[3] |= 1;
				}
				if (vramptr[offset + 0x0c000] & bit) {
					col[3] |= 2;
				}
				if (vramptr[offset + 0x14000] & bit) {
					col[3] |= 4;
				}
				bit >>= 1;
#else
				if (vramptr[offset + 0x00000] & bit) {
					col[0] |= 1;
				}
				if (vramptr[offset + 0x04000] & bit) {
					col[0] |= 2;
				}
				if (vramptr[offset + 0x08000] & bit) {
					col[0] |= 4;
				}
				if (vramptr[offset + 0x0c000] & bit) {
					col[2] |= 1;
				}
				if (vramptr[offset + 0x10000] & bit) {
					col[2] |= 2;
				}
				if (vramptr[offset + 0x14000] & bit) {
					col[2] |= 4;
				}
				bit >>= 1;

				if (vramptr[offset + 0x0000] & bit) {
					col[1] |= 1;
				}
				if (vramptr[offset + 0x4000] & bit) {
					col[1] |= 2;
				}
				if (vramptr[offset + 0x8000] & bit) {
					col[1] |= 4;
				}
				if (vramptr[offset + 0x0c000] & bit) {
					col[3] |= 1;
				}
				if (vramptr[offset + 0x10000] & bit) {
					col[3] |= 2;
				}
				if (vramptr[offset + 0x14000] & bit) {
					col[3] |= 4;
				}
				bit >>= 1;
#endif

				col[0] = pal[col[0]];
				col[1] = pal[col[1]];
				col[2] = pal[col[2]];
				col[3] = pal[col[3]];
				color = mix_color(col, 4, MONITOR_COLOR, GAMMA400L);
				buffer[x * 8 + i * 2 + 0] = (BYTE)(color & 255);
				buffer[x * 8 + i * 2 + 1] = (BYTE)(color >> 8);
			}
			offset++;
		}

		/*  */
		if (!file_write(fileh, buffer, sizeof(buffer))) {
			return FALSE;
		}

		/* y(߂) */
		offset -= (80 * 2);
	}

	return TRUE;
}
#endif

#if XM7_VER >= 3
/*
 *  BMPf[^(400C[hEk摜)
 */
static BOOL FASTCALL bmp_400l_sub2(int fileh)
{
	int x, y;
	int i;
	int offset;
	BYTE bit;
	BYTE buffer[640];
	BYTE *vramptr;
	BYTE pal[8];
	BYTE lbuf[640];
	BYTE pbuf[4];
	WORD color;
	BYTE col1, col2;
	WORD dx1, dx2;
	BOOL winy;

	ASSERT(fileh >= 0);

	/* Ff[^ */
	for (i=0; i<8; i++) {
		if (crt_flag) {
			pal[i] = (BYTE)(ttl_palet[i & ((~(multi_page >> 4)) & 7)] & 7);
		}
		else {
			pal[i] = 0;
		}
	}

	/* ItZbgݒ */
	offset = 80 * 399;

	/* EBhË̃NbsOs */
	window_clip(2);
	dx1 = (WORD)(window_dx1 >> 3);
	dx2 = (WORD)(window_dx2 >> 3);

	/* y[v */
	for (y=0; y<400; y++) {
		winy = (((399 - y) >= window_dy1) && ((399 - y) <= window_dy2));

		if ((y % 2) == 0) {
			/* pbgobt@ */
			memset(lbuf, 0, sizeof(pbuf));
		}

		/* x[v */
		for (x=0; x<80; x++) {
			bit = 0x80;
			if (winy && (x >= dx1) && (x < dx2)) {
				/* EBhE(ubN) */
				vramptr = vram_bdptr;
			}
			else {
				/* EBhEO(\ubN) */
				vramptr = vram_dptr;
			}

			/* bit[v */
			for (i=0; i<4; i++) {
				col1 = 0;
				col2 = 0;

				if (vramptr[offset + 0x00000] & bit) {
					col1 |= 1;
				}
				if (vramptr[offset + 0x08000] & bit) {
					col1 |= 2;
				}
				if (vramptr[offset + 0x10000] & bit) {
					col1 |= 4;
				}
				bit >>= 1;

				if (vramptr[offset + 0x00000] & bit) {
					col2 |= 1;
				}
				if (vramptr[offset + 0x08000] & bit) {
					col2 |= 2;
				}
				if (vramptr[offset + 0x10000] & bit) {
					col2 |= 4;
				}
				bit >>= 1;

				if ((y % 2) == 0) {
					lbuf[(x * 4 + i) * 2 + 0] = col1;
					lbuf[(x * 4 + i) * 2 + 1] = col2;
				}
				else {
					pbuf[0] = pal[lbuf[(x * 4 + i) * 2 + 0]];
					pbuf[1] = pal[lbuf[(x * 4 + i) * 2 + 1]];
					pbuf[2] = pal[col1];
					pbuf[3] = pal[col2];

					color = mix_color(pbuf, 4, MONITOR_COLOR, GAMMA400L);
					buffer[x * 8 + i * 2 + 0] = (BYTE)(color & 255);
					buffer[x * 8 + i * 2 + 1] = (BYTE)(color >> 8);
				}
			}
			offset++;
		}

		/*  */
		if ((y % 2) == 1) {
			if (!file_write(fileh, buffer, sizeof(buffer))) {
				return FALSE;
			}
		}

		/* y(߂) */
		offset -= (80 * 2);
	}

	return TRUE;
}
#endif

#if XM7_VER == 1 && defined(L4CARD)
/*
 *  BMPf[^(L4 400C[hEk摜)
 */
static BOOL FASTCALL bmp_400l4_sub2(int fileh, MONITORTYPE monitortype)
{
	int x, y;
	int i;
	int offset;
	BYTE bit, bit2;
	BYTE buffer[640];
	BYTE lbuf[640];
	BYTE pbuf[4];
	WORD color;
	BYTE col1, col2;
	WORD taddr, gaddr, textbase;
	BYTE csr_st, csr_ed, csr_type;
	BYTE raster, lines;
	BYTE col, chr, atr, chr_dat;
	BOOL enable_page;

	ASSERT(fileh >= 0);

	/* ItZbgݒ */
	offset = 80 * 399;

	/* eLXgWJ ݒ(S) */
	csr_st = (BYTE)(crtc_register[10] & 0x1f);
	csr_ed = (BYTE)(crtc_register[11] & 0x1f);
	csr_type = (BYTE)((crtc_register[10] & 0x60) >> 5);
	lines = (BYTE)((crtc_register[9] & 0x1f) + 1);

	/* pbgobt@ */
	memset(lbuf, 0, sizeof(pbuf));

	/* y[v */
	for (y=0; y<400; y++) {
		/* eLXgWJ ݒ(X^P) */
		textbase = (WORD)text_start_addr;
		textbase += (WORD)(((399 - y) / lines) * (crtc_register[1] << 2));
		textbase &= 0xffe;
		raster = (BYTE)((399 - y) % lines);

		/* x[v */
		for (x=0; x<80; x++) {
			bit = 0x80;
			if (!width40_flag || !(x & 1)) {
				bit2 = 0x80;

				/* LN^R[hEAgr[g擾 */
				if (width40_flag) {
					taddr = (WORD)((textbase + (x & ~1)) & 0xffe);
				}
				else {
					taddr = (WORD)((textbase + (x << 1)) & 0xffe);
				}
				chr = tvram_c[taddr + 0];
				atr = tvram_c[taddr + 1];

				/* Agr[g`Fݒ */
				col = (BYTE)((atr & 0x07) | ((atr & 0x20) >> 2));

				/* tHgf[^擾(Agr[g/J[\܂) */
				if ((!(atr & 0x10) || text_blink) && (raster < 16)) {
					chr_dat = subcg_l4[(WORD)(chr << 4) + raster];
				}
				else {
					chr_dat = 0x00;
				}
				if (atr & 0x08) {
					chr_dat ^= (BYTE)0xff;
				}
				if (csr_type != 1) {
					if (((taddr == cursor_addr) &&
						(cursor_blink || !csr_type)) &&
						((raster >= csr_st) && (raster <= csr_ed))) {
						chr_dat ^= (BYTE)0xff;
					}
				}
			}

			/* GVRAM AhX擾 */
			gaddr = (WORD)((offset + vram_offset[0]) & 0x7fff);
			enable_page = FALSE;
			if (gaddr >= 0x4000) {
				if (!(multi_page & 0x10)) {
					enable_page = TRUE;
				}
			}
			else {
				if (!(multi_page & 0x20)) {
					enable_page = TRUE;
				}
			}

			/* bit[v */
			for (i=0; i<4; i++) {
				if (chr_dat & bit2) {
					col1 = col;
				}
				else if ((vram_c[gaddr] & bit) && enable_page) {
					col1 = ttl_palet[1];
				}
				else {
					col1 = ttl_palet[0];
				}
				bit >>= 1;
				if (!width40_flag) {
					bit2 >>= 1;
				}

				if (chr_dat & bit2) {
					col2 = col;
				}
				else if ((vram_c[gaddr] & bit) && enable_page) {
					col2 = ttl_palet[1];
				}
				else {
					col2 = ttl_palet[0];
				}
				bit >>= 1;
				bit2 >>= 1;

				if ((y % 2) == 0) {
					lbuf[(x * 4 + i) * 2 + 0] = col1;
					lbuf[(x * 4 + i) * 2 + 1] = col2;
				}
				else {
					if (crt_flag) {
						pbuf[0] = lbuf[(x * 4 + i) * 2 + 0];
						pbuf[1] = lbuf[(x * 4 + i) * 2 + 1];
						pbuf[2] = col1;
						pbuf[3] = col2;

						color = mix_color_16(pbuf, monitortype);
						buffer[x * 8 + i * 2 + 0] = (BYTE)(color & 255);
						buffer[x * 8 + i * 2 + 1] = (BYTE)(color >> 8);
					}
					else {
						buffer[x * 8 + i * 2 + 0] = 0;
						buffer[x * 8 + i * 2 + 1] = 0;
					}
				}
			}
			offset++;
		}

		/* 2CƂɏ */
		if (y & 1) {
			if (!file_write(fileh, buffer, sizeof(buffer))) {
				return FALSE;
			}
		}

		/* y(߂) */
		offset -= (80 * 2);
	}

	return TRUE;
}
#endif

#if XM7_VER == 1
/*
 *  BMPf[^(^400CEk摜)
 */
static BOOL FASTCALL bmp_p400m_sub2(int fileh, MONITORTYPE monitortype)
{
	int x, y;
	int i;
	int offset;
	BYTE bit;
	BYTE buffer[640];
	BYTE pal[8];
	BYTE pbuf[4];
	WORD color;
	BYTE col;

	ASSERT(fileh >= 0);
	UNUSED(monitortype);

	/* Ff[^ */
	for (i=0; i<8; i++) {
		if (crt_flag) {
			pal[i] = (BYTE)(ttl_palet[i & ((~(multi_page >> 4)) & 7)] & 7);
		}
		else {
			pal[i] = 0;
		}
	}

	/* ItZbgݒ */
	offset = 80 * 199;

	/* y[v */
	for (y=0; y<400; y++) {
		/* x[v */
		for (x=0; x<80; x++) {
			bit = 0x80;

			/* bit[v */
			for (i=0; i<4; i++) {
				pbuf[0] = 0;
				pbuf[1] = 0;
				pbuf[2] = 0;
				pbuf[3] = 0;

				col = 0;
				if (vram_c[offset + 0x00000] & bit) {
					col |= 1;
				}
				if (vram_c[offset + 0x04000] & bit) {
					col |= 2;
				}
				if (vram_c[offset + 0x08000] & bit) {
					col |= 4;
				}
				bit >>= 1;

				if (pal[col] & 2) {
					pbuf[0] = 7;
				}
				if (pal[col] & 4) {
					pbuf[1] = 7;
				}

				col = 0;
				if (vram_c[offset + 0x00000] & bit) {
					col |= 1;
				}
				if (vram_c[offset + 0x04000] & bit) {
					col |= 2;
				}
				if (vram_c[offset + 0x08000] & bit) {
					col |= 4;
				}
				bit >>= 1;

				if (pal[col] & 2) {
					pbuf[2] = 7;
				}
				if (pal[col] & 4) {
					pbuf[3] = 7;
				}

				color = mix_color(pbuf, 4, monitortype, GAMMA400L);
				buffer[x * 8 + i * 2 + 0] = (BYTE)(color & 255);
				buffer[x * 8 + i * 2 + 1] = (BYTE)(color >> 8);
			}
			offset++;
		}

		/*  */
		if (!file_write(fileh, buffer, sizeof(buffer))) {
			return FALSE;
		}

		/* y(߂) */
		offset -= 80 * 2;
	}

	return TRUE;
}
#endif

/*
 *	ʃLv`(BMP)
 */
BOOL FASTCALL capture_to_bmp(char *fname, BOOL fullscan, MONITORTYPE monitortype, BOOL p400line)
{
	int fileh;

#if XM7_VER == 1
	UNUSED(fullscan);
#endif
#if XM7_VER >= 3
	UNUSED(monitortype);
#endif
	ASSERT(fname);

	/* t@CI[v */
	fileh = file_open(fname, OPEN_W);
	if (fileh == -1) {
		return FALSE;
	}

	/* wb_ */
	if (!bmp_header_sub(fileh)) {
		file_close(fileh);
		return FALSE;
	}

	/* pbg */
#if XM7_VER >= 2
#if XM7_VER >= 3
	if (!(screen_mode & SCR_ANALOG)) {
#else
	if (!mode320) {
#endif
		if (!bmp_palette_sub(fileh, FALSE)) {
			file_close(fileh);
			return FALSE;
		}
	}
#else
	if (!bmp_palette_sub(fileh, monitortype)) {
		file_close(fileh);
		return FALSE;
	}
#endif

	/* {̏ */
#if XM7_VER >= 3
	switch (screen_mode) {
		case SCR_400LINE	:	/* 640~400 8F[h */
								if (!bmp_400l_sub(fileh)) {
									file_close(fileh);
									return FALSE;
								}
								break;
		case SCR_262144		:	/* 320~200 26F[h */
								if (!bmp_256k_sub(fileh, fullscan)) {
									file_close(fileh);
									return FALSE;
								}
								break;
		case SCR_4096		:	/* 320~200 4096F[h */
								if (!bmp_320_sub(fileh, fullscan, FALSE)) {
									file_close(fileh);
									return FALSE;
								}
								break;
		case SCR_200LINE	:	/* 640~200 8F[h */
								if (p400line) {
									if (!bmp_p400c_sub(fileh)) {
										file_close(fileh);
										return FALSE;
									}
								}
								else {
									if (!bmp_640_sub(fileh, fullscan)) {
										file_close(fileh);
										return FALSE;
									}
								}
								break;
	}
#elif XM7_VER >= 2
	if (mode320) {
		/* 320~200 4096F[h */
		if (!bmp_320_sub(fileh, fullscan, monitortype)) {
			file_close(fileh);
			return FALSE;
		}
	}
	else {
		/* 640~200 8F[h */
			if (p400line) {
				if (!bmp_p400c_sub(fileh)) {
					file_close(fileh);
					return FALSE;
				}
			}
		else {
			if (!bmp_640_sub(fileh, fullscan)) {
				file_close(fileh);
				return FALSE;
			}
		}
	}
#elif XM7_VER == 1 && defined(L4CARD)
	if (enable_400line) {
		/* 640~400 PF[h */
		if (!bmp_400l4_sub(fileh, monitortype)) {
			file_close(fileh);
			return FALSE;
		}
	}
	else {
		if (p400line) {
			/* 640~200 ^400C[h */
			if (!bmp_p400m_sub(fileh)) {
				file_close(fileh);
				return FALSE;
			}
		}
		else {
			/* 640~200 8F[h */
			if (!bmp_640_sub(fileh, fullscan)) {
				file_close(fileh);
				return FALSE;
			}
		}
	}
#else
	if (p400line) {
		/* 640~200 ^400C[h */
		if (!bmp_p400m_sub(fileh)) {
			file_close(fileh);
			return FALSE;
		}
	}
	else {
		/* 640~200 8F[h */
		if (!bmp_640_sub(fileh, fullscan)) {
			file_close(fileh);
			return FALSE;
		}
	}
#endif

	/*  */
	file_close(fileh);
	return TRUE;
}

/*
 *	ʃLv`(BMPEk摜)
 */
BOOL FASTCALL capture_to_bmp2(char *fname, MONITORTYPE monitortype, BOOL p400line)
{
	int fileh;

#if XM7_VER >= 3
	UNUSED(monitortype);
#endif
	ASSERT(fname);

	/* t@CI[v */
	fileh = file_open(fname, OPEN_W);
	if (fileh == -1) {
		return FALSE;
	}

	/* wb_ */
	if (!bmp_header_sub2(fileh)) {
		file_close(fileh);
		return FALSE;
	}

	/* {̏ */
#if XM7_VER >= 3
	switch (screen_mode) {
		case SCR_400LINE	:	/* 640~400 8F[h */
								if (!bmp_400l_sub2(fileh)) {
									file_close(fileh);
									return FALSE;
								}
								break;
		case SCR_262144		:	/* 320~200 26F[h */
								if (!bmp_256k_sub2(fileh)) {
									file_close(fileh);
									return FALSE;
								}
								break;
		case SCR_4096		:	/* 320~200 4096F[h */
								if (!bmp_320_sub2(fileh, FALSE)) {
									file_close(fileh);
									return FALSE;
								}
								break;
		case SCR_200LINE	:	/* 640~200 8F[h */
								if (p400line) {
									if (!bmp_p400c_sub2(fileh)) {
										file_close(fileh);
										return FALSE;
									}
								}
								else {
									if (!bmp_640_sub2(fileh, FALSE)) {
										file_close(fileh);
										return FALSE;
									}
								}
								break;
	}
#elif XM7_VER >= 2
	if (mode320) {
		/* 320~200 4096F[h */
		if (!bmp_320_sub2(fileh, monitortype)) {
			file_close(fileh);
			return FALSE;
		}
	}
	else {
		/* 640~200 8F[h */
			if (p400line) {
				if (!bmp_p400c_sub2(fileh)) {
					file_close(fileh);
					return FALSE;
				}
			}
		else {
			if (!bmp_640_sub2(fileh, FALSE)) {
				file_close(fileh);
				return FALSE;
			}
		}
	}
#elif XM7_VER == 1 && defined(L4CARD)
	if (enable_400line) {
		/* 640~400 PF[h */
		if (!bmp_400l4_sub2(fileh, monitortype)) {
			file_close(fileh);
			return FALSE;
		}
	}
	else {
		if (p400line) {
			/* 640~200 ^400C[h */
			if (!bmp_p400m_sub2(fileh, monitortype)) {
				file_close(fileh);
				return FALSE;
			}
		}
		else {
			/* 640~200 8F[h */
			if (!bmp_640_sub2(fileh, monitortype)) {
				file_close(fileh);
				return FALSE;
			}
		}
	}
#else
	if (p400line) {
		/* 640~200 ^400C[h */
		if (!bmp_p400m_sub2(fileh, monitortype)) {
			file_close(fileh);
			return FALSE;
		}
	}
	else {
		/* 640~200 8F[h */
		if (!bmp_640_sub2(fileh, monitortype)) {
			file_close(fileh);
			return FALSE;
		}
	}
#endif

	/*  */
	file_close(fileh);
	return TRUE;
}

/*
 *	t@CTCY擾
 */
DWORD FASTCALL get_file_size(char *fname)
{
	int fileh;
	DWORD fsize;

	/* assert */
	ASSERT(fname);

	/* t@CI[v */
	fileh = file_open(fname, OPEN_R);
	if (fileh == -1) {
		return 0;
	}

	/* t@CI[vAt@CTCY𒲂ׂ */
	fsize = file_getsize(fileh);

	/* ok */
	file_close(fileh);
	return fsize;
}

/*
 *	fBA^CgύX
 */
BOOL FASTCALL set_title_media(char *fname, int index, char *name)
{
	int fileh;
	BYTE header[0x20];
	DWORD offset;
	DWORD len;
	int i;
	int count;

	/* assert */
	ASSERT(fname);
	ASSERT((index >= 0) && (index < 16));

	/*  */
	offset = 0;

	/* t@CI[v */
	fileh = file_open(fname, OPEN_RW);
	if (fileh == -1) {
		return FALSE;
	}

	/* fBA[v */
	for (count = 0; count < 16; count++) {
		/* V[N */
		if (!file_seek(fileh, offset)) {
			file_close(fileh);
			return FALSE;
		}

		/* ǂݍ */
		if (!file_read(fileh, header, 0x020)) {
			file_close(fileh);
			return FALSE;
		}

		if (count == index) {
			/* wb_쐬 */
			memset(header, 0, 16 + 1);
			if (name != NULL) {
				for (i=0; i<16; i++) {
					if (name[i] == '\0') {
						break;
					}
					header[i] = name[i];
				}
			}

			/* V[N */
			if (!file_seek(fileh, offset)) {
				file_close(fileh);
				return FALSE;
			}

			/*  */
			if (!file_write(fileh, header, 0x020)) {
				file_close(fileh);
				return FALSE;
			}
			break;
		}

		/* next */
		len = 0;
		len |= header[0x1f];
		len *= 256;
		len |= header[0x1e];
		len *= 256;
		len |= header[0x1d];
		len *= 256;
		len |= header[0x1c];
		offset += len;
	}

	/* ő僁fBAɒB */
	file_close(fileh);
	return TRUE;
}

/*
 *	fBA^Cg擾
 */
BOOL FASTCALL get_title_media(char *fname, int index, char *name)
{
	int fileh;
	DWORD fsize;
	BYTE header[0x20];
	DWORD offset;
	DWORD len;
	int i;
	int count;

	/* assert */
	ASSERT(fname);
	ASSERT((index >= 0) && (index < 16));

	/* i[`FbN */
	if (name == NULL) {
		return FALSE;
	}

	/*  */
	offset = 0;

	/* t@CI[vAt@CTCY𒲂ׂ */
	fileh = file_open(fname, OPEN_RW);
	if (fileh == -1) {
		return FALSE;
	}
	fsize = file_getsize(fileh);

	/*
	 * 2Dt@C
	 */
	if ((fsize == 327680) && (index == 0)) {
		name[0] = '\0';
		file_close(fileh);
		return FALSE;
	}

#if XM7_VER >= 2
	/*
	 * 2DDt@C
	 */
	if ((fsize == 655360) && (index == 0)) {
		name[0] = '\0';
		file_close(fileh);
		return FALSE;
	}
#endif

#if XM7_VER == 1 && defined(SFDC)
	/*
	 * 2HDt@C
	 */
	if ((fsize == 1021696) && (index == 0)) {
		name[0] = '\0';
		file_close(fileh);
		return FALSE;
	}
#endif

	/*
	 * VFDt@C
	 */
	if (file_seek(fileh, 0)) {
		if (file_read(fileh, header, 0x0020)) {
			if ((header[0x00] == 0xe0) && (header[0x01] == 0x01) &&
				(header[0x02] == 0x00) && (header[0x03] == 0x00)) {
				name[0] = '\0';
				file_close(fileh);
				return FALSE;
			}
		}
	}

#if XM7_VER == 1 && defined(BUBBLE)
	/*
	 * BBLt@C(32KB)
	 */
	if ((fsize == 32768) && (index == 0)) {
		name[0] = '\0';
		file_close(fileh);
		return FALSE;
	}

	/*
	 * BBLt@C(128KB)
	 */
#ifdef BUBBLE128
	if ((fsize == 131072) && (index == 0)) {
		name[0] = '\0';
		file_close(fileh);
		return FALSE;
	}
#endif
#endif

	/*
	 * D77t@C/B77t@C
	 */
	for (count = 0; count < 16; count++) {
		/* V[N */
		if (!file_seek(fileh, offset)) {
			file_close(fileh);
			return FALSE;
		}

		/* ǂݍ */
		if (!file_read(fileh, header, 0x020)) {
			file_close(fileh);
			return FALSE;
		}

		if (count == index) {
			/* fBA擾 */
			memset(name, 0, 16 + 1);
			for (i=0; i<16; i++) {
				name[i] = header[i];
				if (name[i] == '\0') {
					break;
				}
			}
			break;
		}

		/* next */
		len = 0;
		len |= header[0x1f];
		len *= 256;
		len |= header[0x1e];
		len *= 256;
		len |= header[0x1d];
		len *= 256;
		len |= header[0x1c];
		offset += len;
	}

	/* ő僁fBAɒB */
	file_close(fileh);
	return TRUE;
}

/*
 *	fBA擾
 */
int FASTCALL get_media_num(char *fname)
{
	int fileh;
	DWORD fsize;
	BYTE header[0x20];
	DWORD offset;
	DWORD len;
	int count;

	/* assert */
	ASSERT(fname);

	/*  */
	offset = 0;

	/* t@CI[vAt@CTCY𒲂ׂ */
	fileh = file_open(fname, OPEN_RW);
	if (fileh == -1) {
		return FALSE;
	}
	fsize = file_getsize(fileh);

	/*
	 * 2Dt@C
	 */
	if (fsize == 327680) {
		file_close(fileh);
		return 1;
	}

#if XM7_VER >= 2
	/*
	 * 2DDt@C
	 */
	if (fsize == 655360) {
		file_close(fileh);
		return 1;
	}
#endif

#if XM7_VER == 1 && defined(SFDC)
	/*
	 * 2HDt@C
	 */
	if (fsize == 1021696) {
		file_close(fileh);
		return 1;
	}
#endif

	/*
	 * VFDt@C
	 */
	if (file_seek(fileh, 0)) {
		if (file_read(fileh, header, 0x0020)) {
			if ((header[0x00] == 0xe0) && (header[0x01] == 0x01) &&
				(header[0x02] == 0x00) && (header[0x03] == 0x00)) {
				file_close(fileh);
				return 1;
			}
		}
	}

#if XM7_VER == 1 && defined(BUBBLE)
	/*
	 * BBLt@C(32KB)
	 */
	if (fsize == 32768) {
		file_close(fileh);
		return 1;
	}

	/*
	 * BBLt@C(128KB)
	 */
#ifdef BUBBLE128
	if (fsize == 131072) {
		file_close(fileh);
		return 1;
	}
#endif
#endif

	/*
	 * D77t@C/B77t@C
	 */
	for (count = 0; count < 16; count++) {
		/* V[N */
		if (!file_seek(fileh, offset)) {
			file_close(fileh);
			return count;
		}

		/* ǂݍ */
		if (!file_read(fileh, header, 0x020)) {
			file_close(fileh);
			return count;
		}

		/* next */
		len = 0;
		len |= header[0x1f];
		len *= 256;
		len |= header[0x1e];
		len *= 256;
		len |= header[0x1d];
		len *= 256;
		len |= header[0x1c];
		offset += len;
	}

	/* ő僁fBAɒB */
	file_close(fileh);
	return count;
}

/*
 *	fBXNt@C`FbN
 */
int FASTCALL check_disk(char *fname, int index)
{
	int handle;
	DWORD fsize;
	int count;
	DWORD offset;
	BYTE buf[0x20];
	DWORD len;

	/*  */
	offset = 0;

	/* mbgfBɂꍇ */
	if (fname == NULL) {
		return TYPE_NOSUPPORT;
	}

	/* t@CI[vAt@CTCY𒲂ׂ */
	handle = file_open(fname, OPEN_R);
	if (handle == -1) {
		return TYPE_NOSUPPORT;
	}
	fsize = file_getsize(handle);

	/*
	 * 2Dt@C
	 */
	if ((fsize == 327680) && (index == 0)) {
		file_close(handle);
		return TYPE_2D;
	}

#if XM7_VER >= 2
	/*
	 * 2DDt@C
	 */
	if ((fsize == 655360) && (index == 0)) {
		file_close(handle);
		return TYPE_2DD;
	}
#endif

#if XM7_VER == 1 && defined(SFDC)
	/*
	 * 2HDt@C
	 */
	if ((fsize == 1021696) && (index == 0)) {
		file_close(handle);
		return TYPE_2HD;
	}
#endif

	/*
	 * VFDt@C
	 */
	if (file_seek(handle, 0)) {
		if (file_read(handle, buf, 0x0020)) {
			if ((buf[0x00] == 0xe0) && (buf[0x01] == 0x01) &&
				(buf[0x02] == 0x00) && (buf[0x03] == 0x00)) {
				file_close(handle);
				return TYPE_2D;
			}
		}
	}

	/*
	 * D77t@C
	 */
	for (count = 0; count < 16; count++) {
		/* V[N */
		if (!file_seek(handle, offset)) {
			file_close(handle);
			return TYPE_NOSUPPORT;
		}

		/* ǂݍ */
		if (!file_read(handle, buf, 0x0020)) {
			file_close(handle);
			return TYPE_NOSUPPORT;
		}

		/* next */
		len = 0;
		len |= buf[0x1f];
		len *= 256;
		len |= buf[0x1e];
		len *= 256;
		len |= buf[0x1d];
		len *= 256;
		len |= buf[0x1c];
		offset += len;

		/* ^Cv`FbNB2D,2DD,2HDɂꂼΉ */
		if (count == index) {
			if (buf[0x001b] == 0x00) {
				file_close(handle);
				return TYPE_2D;
			}

#if XM7_VER >= 3
			if (buf[0x001b] == 0x10) {
				file_close(handle);
				return TYPE_2DD;
			}
#endif

#if XM7_VER == 1 && defined(SFDC)
			if (buf[0x001b] == 0x20) {
				file_close(handle);
				return TYPE_2HD;
			}
#endif

			return TYPE_NOSUPPORT;
		}
	}

	/* ő僁fBAɒB */
	file_close(handle);
	return TYPE_NOSUPPORT;
}

#if XM7_VER == 1 && defined(BUBBLE)
/*
 *	out@C`FbN
 */
int FASTCALL check_bubble(char *fname, int index)
{
	int handle;
	DWORD fsize;
	int count;
	DWORD offset;
	BYTE buf[0x20];
	DWORD len;

	/*  */
	offset = 0;

	/* mbgfBɂꍇ */
	if (fname == NULL) {
		return TYPE_NOSUPPORT;
	}

	/* t@CI[vAt@CTCY𒲂ׂ */
	handle = file_open(fname, OPEN_R);
	if (handle == -1) {
		return TYPE_NOSUPPORT;
	}
	fsize = file_getsize(handle);

	/*
	 * BBLt@C(32KB)
	 */
	if ((fsize == 32768) && (index == 0)) {
		file_close(handle);
		return TYPE_B32;
	}

	/*
	 * BBLt@C(128KB)
	 */
#ifdef BUBBLE128
	if ((fsize == 131072) && (index == 0)) {
		file_close(handle);
		return TYPE_B128;
	}
#endif

	/*
	 * B77t@C
	 */
	for (count = 0; count < 16; count++) {
		/* V[N */
		if (!file_seek(handle, offset)) {
			file_close(handle);
			return TYPE_NOSUPPORT;
		}

		/* ǂݍ */
		if (!file_read(handle, buf, 0x0020)) {
			file_close(handle);
			return TYPE_NOSUPPORT;
		}

		/* next */
		len = 0;
		len |= buf[0x1f];
		len *= 256;
		len |= buf[0x1e];
		len *= 256;
		len |= buf[0x1d];
		len *= 256;
		len |= buf[0x1c];
		offset += len;

		/* ^Cv`FbNB32KB,128KBɂꂼΉ */
		if (count == index) {
			if (buf[0x001b] == 0x80) {
				file_close(handle);
				return TYPE_B32;
			}

#ifdef BUBBLE128
			if (buf[0x001b] == 0x90) {
				file_close(handle);
				return TYPE_B128;
			}
#endif

			return TYPE_NOSUPPORT;
		}
	}

	/* ő僁fBAɒB */
	file_close(handle);
	return TYPE_NOSUPPORT;
}
#endif

/*
 *	FDXT|[g
 */
int FASTCALL fill_dat(BYTE *p, WORD dat, int cnt)
{
	int i;

	for (i = 0; i < cnt; i++) {
		*p++ = (BYTE)(dat >> 8);
		*p++ = (BYTE)dat;
	}

	return cnt * 2;
}

/*
 *	uNfBXN쐬(FDX)
 */
BOOL FASTCALL make_new_fdx(char *fname, char *name, BOOL mediatype)
{
	int i;
	int fileh;
	fdxheader_t header;
	int tracklen;
	int c;
	int h;
	fdxtrack_t *t;
	fdxtrack_t *trackbuf[FDX_MAX_CYLINDER][FDX_MAX_HEADS];
	BOOL ret;

	/* assert */
	ASSERT(fname);

	/* unused */
#if XM7_VER <= 2
	UNUSED(mediatype);
#endif

	/* t@CI[v */
	fileh = file_open(fname, OPEN_W);
	if (fileh == -1) {
		return FALSE;
	}

	/* wb_쐬 */
	memset(&header, 0, sizeof(fdxheader_t));
	header.signature[0] = 'F';
	header.signature[1] = 'D';
	header.signature[2] = 'X';
	header.revision = 3;
	if (name != NULL) {
		for (i=0; i<16; i++) {
			if (name[i] == '\0') {
				break;
			}
			header.name[i] = name[i];
		}
	}
	else {
		strcpy((char*)header.name, "Default");
	}

	/* x(2D,2DD) */
#if XM7_VER >= 3
	if (mediatype) {
		header.type = FDX_TYPE_2DD;
		header.cylinders = FDX_MAX_CYLINDER;
		header.heads = FDX_MAX_HEADS;
		header.rate = 500;
		header.rpm = 300;
	} else {
		header.type = FDX_TYPE_2D;
		header.cylinders = FDX_MAX_CYLINDER / 2;
		header.heads = FDX_MAX_HEADS;
		header.rate = 500;
		header.rpm = 300;
	}
#else
	header.type = FDX_TYPE_2D;
	header.cylinders = FDX_MAX_CYLINDER / 2;
	header.heads = FDX_MAX_HEADS;
	header.rate = 500;
	header.rpm = 300;
#endif

	/* gbNubN */
	header.trackblk = (FDX_MAX_CELLS * 2) / 2;

	/* wb_ */
	if (!file_write(fileh, (BYTE*)&header, sizeof(fdxheader_t))) {
		file_close(fileh);
		return FALSE;
	}

	/* gbNf[^ */
	tracklen = header.rate * 1000;
	tracklen *= 60;
	tracklen /=	header.rpm;

	/* gbN񏉊 */
	for (c = 0; c < header.cylinders; c++) {
		for (h = 0; h < header.heads; h++) {
			t = (fdxtrack_t *)malloc(header.trackblk);
			memset(t, 0x00, header.trackblk);
			t->cylinder = c;
			t->head = h;
			t->index = 0;
			t->length = tracklen;
			trackbuf[c][h] = t;
		}
	}

	/* gbN񏑂 */
	ret = TRUE;
	for (c = 0; c < header.cylinders; c++) {
		for (h = 0; h < header.heads; h++) {
			t = trackbuf[c][h];
			if (!file_write(fileh, (BYTE*)t, header.trackblk)) {
				ret = FALSE;
				goto save_exit;
			}
		}
	}

save_exit:
	/* gbN */
	for (c = 0; c < header.cylinders; c++) {
		for (h = 0; h < header.heads; h++) {
			if (trackbuf[c][h]) {
				free(trackbuf[c][h]);
				trackbuf[c][h] = NULL;
			}
		}
	}

	file_close(fileh);
	return ret;
}

/*
 *	[UfBXN쐬(FDX)
 */
BOOL FASTCALL make_new_userdisk_fdx(char *fname, char *name, BOOL mediatype)
{
	static const BYTE dummyipl[9] = {
		0x1a, 0x50,			/* ORCC	#$50	*/
		0x86, 0x41,			/* LDA	#$41	*/
		0xb7, 0xfd, 0x03,	/* STA	$FD03	*/
		0x20, 0xfe			/* BRA	*		*/
	};
	int i;
	int fileh;
	fdxheader_t header;
	int tracklen;
	int c;
	int h;
	fdxtrack_t *t;
	fdxtrack_t *trackbuf[FDX_MAX_CYLINDER][FDX_MAX_HEADS];
	BYTE *p;
	BYTE buf[512];
	WORD crc;
	int last;
	BOOL ret;

	/* assert */
	ASSERT(fname);

	/* unused */
#if XM7_VER <= 2
	UNUSED(mediatype);
#endif

	/* t@CI[v */
	fileh = file_open(fname, OPEN_W);
	if (fileh == -1) {
		return FALSE;
	}

	/* wb_쐬 */
	memset(&header, 0, sizeof(fdxheader_t));
	header.signature[0] = 'F';
	header.signature[1] = 'D';
	header.signature[2] = 'X';
	header.revision = 3;
	if (name != NULL) {
		for (i=0; i<16; i++) {
			if (name[i] == '\0') {
				break;
			}
			header.name[i] = name[i];
		}
	}
	else {
		strcpy((char*)header.name, "Default");
	}

	/* x(2D,2DD) */
#if XM7_VER >= 3
	if (mediatype) {
		header.type = FDX_TYPE_2DD;
		header.cylinders = FDX_MAX_CYLINDER;
		header.heads = FDX_MAX_HEADS;
		header.rate = 500;
		header.rpm = 300;
	} else {
		header.type = FDX_TYPE_2D;
		header.cylinders = FDX_MAX_CYLINDER / 2;
		header.heads = FDX_MAX_HEADS;
		header.rate = 500;
		header.rpm = 300;
	}
#else
	header.type = FDX_TYPE_2D;
	header.cylinders = FDX_MAX_CYLINDER / 2;
	header.heads = FDX_MAX_HEADS;
	header.rate = 500;
	header.rpm = 300;
#endif

	/* gbNubN */
	header.trackblk = (FDX_MAX_CELLS * 2) / 2;

	/* wb_ */
	if (!file_write(fileh, (BYTE*)&header, sizeof(fdxheader_t))) {
		file_close(fileh);
		return FALSE;
	}

	/* gbNf[^ */
	tracklen = header.rate * 1000;
	tracklen *= 60;
	tracklen /=	header.rpm;

	/* gbN񏉊 */
	for (c = 0; c < header.cylinders; c++) {
		for (h = 0; h < header.heads; h++) {
			t = (fdxtrack_t *)malloc(header.trackblk);
			memset(t, 0x00, header.trackblk);
			t->cylinder = c;
			t->head = h;
			t->index = 0;
			t->length = tracklen;
			trackbuf[c][h] = t;
		}
	}

	/* tH[}bg */
	for (c = 0; c < header.cylinders; c++) {
		for (h = 0; h < header.heads; h++) {
			t = trackbuf[c][h];
			p = t->data;
			last = 1;

			p += fdx_fill_mfm(p, 0x4e, 80, &last);			/* GAP4a */

			p += fdx_fill_mfm(p, 0x00, 12, &last);			/* SYNC */
			p += fill_dat(p, 0x5224, 3);					/* IAM */
			p += fill_dat(p, 0x5552, 1);
			last = 0;

			p += fdx_fill_mfm(p, 0x4e, 50, &last);			/* GAP1 */

			/* ZN^[v */
			for (i = 0; i < 16; i++) {
				buf[0] = 0xa1;
				buf[1] = 0xa1;
				buf[2] = 0xa1;
				buf[3] = 0xfe;
				buf[4] = c;
				buf[5] = h;
				buf[6] = i + 1;
				buf[7] = 1;
				crc = calc_crc_buf(buf, 8);
				buf[8] = (BYTE)(crc >> 8);
				buf[9] = (BYTE)crc;

				p += fdx_fill_mfm(p, 0x00, 12, &last);		/* SYNC */
				p += fill_dat(p, 0x4489, 3);				/* IDAM */
				p += fill_dat(p, 0x5554, 1);
				last = 0;
				p += fdx_gen_mfm(p, &buf[4], 4, &last);		/* CHRN */
				p += fdx_gen_mfm(p, &buf[8], 2, &last);		/* CRC */

				p += fdx_fill_mfm(p, 0x4e, 22, &last);		/* GAP1 */

				buf[0] = 0xa1;
				buf[1] = 0xa1;
				buf[2] = 0xa1;
				buf[3] = 0xfb;

				if (c == 0 && h == 0 && i == 0) {
					/* _~[IPLZN^쐬 */
					memset(&buf[4], 0x00, 256);
					memcpy(&buf[4], dummyipl, sizeof(dummyipl));
				}
				else if (c == 0 && h == 0 && i == 2) {
					/* IDZN^쐬 */
					memset(&buf[4], 0x00, 256);
#if XM7_VER >= 3
					if (mediatype) {
						buf[4] = 0x45;
					}
					else {
						buf[4] = 0x53;
					}
#else
					buf[4] = 0x53;
#endif
					buf[5] = 0x20;
					buf[6] = 0x20;
				}
				else if (c == 1) {
					/* FAT/fBNg쐬 */
					memset(&buf[4], 0xff, 256);
					if (h == 0 && i == 0) {
						buf[4] = 0x00;
					}
				}
				else {
					/* ʏZN^쐬 */
					memset(&buf[4], 0xe5, 256);
				}
				crc = calc_crc_buf(buf, 260);
				buf[260] = (BYTE)(crc >> 8);
				buf[261] = (BYTE)crc;

				p += fdx_fill_mfm(p, 0x00, 12, &last);		/* SYNC */
				p += fill_dat(p, 0x4489, 3);				/* DAM */
				p += fill_dat(p, 0x5545, 1);
				last = 1;

				p += fdx_gen_mfm(p, &buf[4], 256, &last);	/* DATA */
				p += fdx_gen_mfm(p, &buf[260], 2, &last);	/* CRC */

				p += fdx_fill_mfm(p, 0x4e, 51, &last);		/* GAP3 */
			}

			p += fdx_fill_mfm(								/* GAP4b */
				p, 0x4e,((tracklen / 8) - (p - t->data)) / 2, &last);
		}
	}

	/* gbN񏑂 */
	ret = TRUE;
	for (c = 0; c < header.cylinders; c++) {
		for (h = 0; h < header.heads; h++) {
			t = trackbuf[c][h];
			if (!file_write(fileh, (BYTE*)t, header.trackblk)) {
				ret = FALSE;
				goto save_exit;
			}
		}
	}

save_exit:
	/* gbN */
	for (c = 0; c < header.cylinders; c++) {
		for (h = 0; h < header.heads; h++) {
			if (trackbuf[c][h]) {
				free(trackbuf[c][h]);
				trackbuf[c][h] = NULL;
			}
		}
	}

	file_close(fileh);
	return ret;
}
