/*
 * Decompiled with CFR 0.152.
 */
package omegadrive.savestate;

import com.google.common.collect.ImmutableSet;
import java.nio.ByteBuffer;
import java.nio.file.Paths;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.IntStream;
import omegadrive.Device;
import omegadrive.SystemLoader;
import omegadrive.bus.z80.SmsBus;
import omegadrive.cpu.z80.Z80Provider;
import omegadrive.memory.IMemoryProvider;
import omegadrive.savestate.BaseStateHandler;
import omegadrive.savestate.MekaSavestateVersion;
import omegadrive.savestate.StateUtil;
import omegadrive.util.ArrayEndianUtil;
import omegadrive.util.FileUtil;
import omegadrive.util.LogHelper;
import omegadrive.util.Util;
import omegadrive.vdp.SmsVdp;
import org.slf4j.Logger;

public class MekaStateHandler
implements BaseStateHandler {
    private static final String MAGIC_WORD_STR = "MEKA";
    private static final byte[] MAGIC_WORD = "MEKA".getBytes();
    private static final int Z80_REG_lEN = 26;
    private static final int Z80_MISC_LEN = 27;
    private static final int VDP_MISC_LEN = 20;
    private static final MekaSavestateVersion DEFAULT_SAVE_VERSION = MekaSavestateVersion.VER_D;
    private static final Function<Integer, String> toCrcStringFn = v -> (v < 16 ? "0" : "") + Util.th(v);
    private static final String fileExtension = "s00";
    private static final Logger LOG = LogHelper.getLogger(MekaStateHandler.class.getSimpleName());
    private static final Set<Class<? extends Device>> deviceClassSet = ImmutableSet.of(Z80Provider.class, SmsVdp.class, IMemoryProvider.class, SmsBus.class);
    final int[] vdpState = new int[3];
    private ByteBuffer buffer;
    private int version;
    private int softwareId;
    private String fileName;
    private BaseStateHandler.Type type;
    private SystemLoader.SystemType systemType;
    private MekaSavestateVersion mekaVersion;
    private List<Device> deviceList = Collections.emptyList();

    private MekaStateHandler() {
    }

    public static BaseStateHandler createInstance(SystemLoader.SystemType systemType, String fileName, BaseStateHandler.Type type, Set<Device> deviceSet) {
        return type == BaseStateHandler.Type.LOAD ? MekaStateHandler.createLoadInstance(fileName, deviceSet) : MekaStateHandler.createSaveInstance(fileName, systemType, deviceSet);
    }

    private static BaseStateHandler createLoadInstance(String fileName, Set<Device> deviceSet) {
        MekaStateHandler h = new MekaStateHandler();
        h.fileName = MekaStateHandler.handleFileExtension(fileName);
        h.buffer = ByteBuffer.wrap(FileUtil.readBinaryFile(Paths.get(h.fileName, new String[0]), new String[0]));
        h.type = BaseStateHandler.Type.LOAD;
        h.setDevicesWithContext(deviceSet);
        return h.detectStateFileType();
    }

    private static MekaStateHandler createSaveInstance(String fileName, SystemLoader.SystemType systemType, Set<Device> deviceSet) {
        MekaStateHandler h = MekaStateHandler.createSaveInstance(fileName, systemType, "0");
        h.setDevicesWithContext(deviceSet);
        return h;
    }

    public static BaseStateHandler createLoadInstance(String fileName, byte[] data, Set<Device> deviceSet) {
        MekaStateHandler h = new MekaStateHandler();
        h.fileName = MekaStateHandler.handleFileExtension(fileName);
        h.buffer = ByteBuffer.wrap(data);
        h.type = BaseStateHandler.Type.LOAD;
        h.setDevicesWithContext(deviceSet);
        return h.detectStateFileType();
    }

    private static MekaStateHandler createSaveInstance(String fileName, SystemLoader.SystemType systemType, String romCrc32) {
        int machineDriverId;
        MekaStateHandler h = new MekaStateHandler();
        int n = systemType == SystemLoader.SystemType.SMS ? 0 : (machineDriverId = systemType == SystemLoader.SystemType.GG ? 1 : -1);
        if (machineDriverId < 0) {
            throw new IllegalArgumentException("Invalid systemType: " + String.valueOf((Object)systemType));
        }
        long crc32 = Long.parseLong(romCrc32, 16);
        int len = DEFAULT_SAVE_VERSION.getMemoryEndPos() + 3 << 1;
        h.buffer = ByteBuffer.allocate(len);
        h.buffer.put(MAGIC_WORD);
        h.buffer.put((byte)26);
        h.buffer.put((byte)DEFAULT_SAVE_VERSION.getVersion());
        h.buffer.put((byte)machineDriverId);
        h.buffer.put((byte)(crc32 & 0xFFL));
        h.buffer.put((byte)(crc32 >> 8 & 0xFFL));
        h.buffer.put((byte)(crc32 >> 16 & 0xFFL));
        h.buffer.put((byte)(crc32 >> 24 & 0xFFL));
        h.buffer.put(len - 3, (byte)69);
        h.buffer.put(len - 2, (byte)79);
        h.buffer.put(len - 1, (byte)70);
        h.mekaVersion = DEFAULT_SAVE_VERSION;
        h.systemType = systemType;
        h.fileName = MekaStateHandler.handleFileExtension(fileName);
        h.type = BaseStateHandler.Type.SAVE;
        return h;
    }

    private static String handleFileExtension(String fileName) {
        return fileName + (!fileName.toLowerCase().contains(".s0") ? ".s00" : "");
    }

    private static String decodeCrc32(MekaSavestateVersion version, ByteBuffer data) {
        int index = data.position();
        data.position(index + 4);
        return toCrcStringFn.apply(data.get(index + 3) & 0xFF) + toCrcStringFn.apply(data.get(index + 2) & 0xFF) + toCrcStringFn.apply(data.get(index + 1) & 0xFF) + toCrcStringFn.apply(data.get(index) & 0xFF);
    }

    @Override
    public void processState() {
        SmsBus bus = StateUtil.getInstanceOrThrow(this.deviceList, SmsBus.class);
        SmsVdp vdp = StateUtil.getInstanceOrThrow(this.deviceList, SmsVdp.class);
        Z80Provider z80 = StateUtil.getInstanceOrThrow(this.deviceList, Z80Provider.class);
        IMemoryProvider mem2 = StateUtil.getInstanceOrThrow(this.deviceList, IMemoryProvider.class);
        if (this.type == BaseStateHandler.Type.LOAD) {
            z80.loadContext(this.buffer);
            StateUtil.skip(this.buffer, 27);
            this.loadVdp(vdp, bus);
            mem2.loadContext(this.buffer);
            this.loadVdpMemory(vdp);
        } else {
            z80.saveContext(this.buffer);
            StateUtil.skip(this.buffer, 27);
            this.saveVdp(vdp, bus);
            mem2.saveContext(this.buffer);
            this.saveVdpMemory(vdp);
        }
    }

    private void setDevicesWithContext(Set<Device> devs) {
        if (!this.deviceList.isEmpty()) {
            LOG.warn("Overwriting device list: {}", (Object)Arrays.toString(this.deviceList.toArray()));
        }
        this.deviceList = StateUtil.getDeviceOrderList(deviceClassSet, devs);
    }

    private BaseStateHandler detectStateFileType() {
        String fileType = Util.toStringValue(this.buffer.get(), this.buffer.get(), this.buffer.get(), this.buffer.get());
        if (!MAGIC_WORD_STR.equalsIgnoreCase(fileType)) {
            LOG.error("Unable to load savestate of type: {}, size: {}", (Object)fileType, (Object)this.buffer.capacity());
            return BaseStateHandler.EMPTY_STATE;
        }
        this.buffer.get();
        this.version = this.buffer.get();
        this.mekaVersion = MekaSavestateVersion.getMekaVersion(this.version);
        byte machineDriverId = this.buffer.get();
        SystemLoader.SystemType systemType = machineDriverId == 0 ? SystemLoader.SystemType.SMS : (this.systemType = machineDriverId == 1 ? SystemLoader.SystemType.GG : null);
        if (this.systemType == null) {
            throw new IllegalArgumentException("Unknown machineDriverId: " + machineDriverId);
        }
        this.crcCheck();
        return this;
    }

    private void crcCheck() {
        if (this.version >= 12) {
            String crc32 = MekaStateHandler.decodeCrc32(this.mekaVersion, this.buffer);
            LOG.info("ROM crc32: {}", (Object)crc32);
        }
    }

    private void loadVdp(SmsVdp vdp, SmsBus bus) {
        int pos = this.buffer.position();
        vdp.loadContext(this.buffer);
        boolean isHeliosFormat = this.buffer.position() > pos + 16 + 3;
        int toSkip = isHeliosFormat ? 20 - (this.vdpState.length * 4 + 3) : 17;
        StateUtil.skip(this.buffer, toSkip);
        bus.loadContext(this.buffer);
        if (this.version >= 13) {
            int vdpLine = ArrayEndianUtil.getUInt32LE(this.buffer.get(), this.buffer.get());
            LOG.info("vdpLine: {}", (Object)vdpLine);
        }
    }

    private void saveVdp(SmsVdp vdp, SmsBus bus) {
        vdp.saveContext(this.buffer);
        StateUtil.skip(this.buffer, 20 - (this.vdpState.length * 4 + 3));
        bus.saveContext(this.buffer);
        this.buffer.put((byte)0);
        this.buffer.put((byte)0);
    }

    private void loadVdpMemory(SmsVdp vdp) {
        byte[] vram = vdp.getVRAM();
        int[] cram = vdp.getCRAM();
        IntStream.range(0, 16384).forEach(i -> {
            vram[i] = this.buffer.get();
        });
        IntStream.range(0, 32).forEach(i -> {
            byte smsCol = this.buffer.get();
            int r = smsCol & 3;
            int g = smsCol >> 2 & 3;
            int b = smsCol >> 4 & 3;
            cram[i] = r * 85 << 16 | g * 85 << 8 | b * 85;
        });
        vdp.forceFullRedraw();
    }

    private void saveVdpMemory(SmsVdp vdp) {
        byte[] vram = vdp.getVRAM();
        int[] cram = vdp.getCRAM();
        IntStream.range(0, 16384).forEach(i -> this.buffer.put((byte)(vram[i] & 0xFF)));
        IntStream.range(0, 32).forEach(i -> {
            int javaColor = cram[i];
            int b = (javaColor & 0xFF) / 85;
            int g = (javaColor >> 8 & 0xFF) / 85;
            int r = (javaColor >> 16 & 0xFF) / 85;
            int smsCol = b << 4 | g << 2 | r;
            this.buffer.put((byte)(smsCol & 0xFF));
        });
    }

    @Override
    public BaseStateHandler.Type getType() {
        return this.type;
    }

    @Override
    public String getFileName() {
        return this.fileName;
    }

    @Override
    public ByteBuffer getDataBuffer() {
        return this.buffer;
    }
}

