/*
 * Decompiled with CFR 0.152.
 */
package omegadrive.sound.msumd;

import com.google.common.io.Files;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.file.Path;
import java.util.Arrays;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.atomic.AtomicReference;
import javax.sound.sampled.AudioInputStream;
import javax.sound.sampled.AudioSystem;
import javax.sound.sampled.Clip;
import javax.sound.sampled.FloatControl;
import javax.sound.sampled.LineEvent;
import javax.sound.sampled.LineListener;
import omegadrive.SystemLoader;
import omegadrive.sound.msumd.CueFileParser;
import omegadrive.sound.msumd.MsuMdHandler;
import omegadrive.util.LogHelper;
import omegadrive.util.Size;
import omegadrive.util.SoundUtil;
import omegadrive.util.Util;
import org.digitalmediaserver.cuelib.CueSheet;
import org.digitalmediaserver.cuelib.FileData;
import org.digitalmediaserver.cuelib.Message;
import org.digitalmediaserver.cuelib.TrackData;
import org.slf4j.Logger;

public class MsuMdHandlerImpl
implements MsuMdHandler {
    private static final Logger LOG = LogHelper.getLogger(MsuMdHandlerImpl.class.getSimpleName());
    private static final boolean verbose = true;
    private final MsuMdHandler.MsuCommandArg commandArg = new MsuMdHandler.MsuCommandArg();
    private int clock = 0;
    private boolean init;
    private volatile Clip clip;
    private volatile byte[] buffer = new byte[0];
    private final AtomicReference<LineListener> lineListenerRef = new AtomicReference();
    private final RandomAccessFile binFile;
    private final MsuMdHandler.TrackDataHolder[] trackDataHolders = new MsuMdHandler.TrackDataHolder[100];
    private final ClipContext clipContext = new ClipContext();
    private volatile boolean busy = false;
    private int lastPlayed = -1;

    private static int sectorsToClipFrames(int val) {
        int mins = val / 75 / 60;
        int sec = val / 75 % 60;
        int frames = val % 75;
        return (int)(44.1 * ((double)(mins * 60 * 1000 + sec * 1000) + (double)frames / 75.0));
    }

    protected void initTrackData(CueSheet cueSheet, long binLen) {
        List trackDataList = cueSheet.getAllTrackData();
        Arrays.fill(this.trackDataHolders, NO_TRACK);
        for (int i = 0; i < trackDataList.size(); ++i) {
            TrackData td = (TrackData)trackDataList.get(i);
            MsuMdHandler.TrackDataHolder h = new MsuMdHandler.TrackDataHolder();
            h.type = MsuMdHandler.CueFileDataType.getFileType(td.getParent().getFileType());
            switch (h.type) {
                case BINARY: {
                    int numBytes = 0;
                    int startFrame = td.getFirstIndex().getPosition().getTotalFrames();
                    if (trackDataList.size() == td.getNumber()) {
                        numBytes = (int)binLen - startFrame * 2352;
                    } else {
                        TrackData trackDataNext = (TrackData)trackDataList.get(td.getNumber());
                        int endFrame = trackDataNext.getFirstIndex().getPosition().getTotalFrames();
                        numBytes = (endFrame - startFrame) * 2352;
                    }
                    h.numBytes = Optional.of(numBytes);
                    h.startFrame = Optional.of(startFrame);
                    break;
                }
                case WAVE: {
                    h.waveFile = Optional.ofNullable(cueSheet.getFile().resolveSibling(td.getParent().getFile()).toFile());
                }
            }
            this.initCueLoopPoints(cueSheet.getMessages(), i, h);
            this.trackDataHolders[td.getNumber()] = h;
        }
    }

    private MsuMdHandler.TrackDataHolder initCueLoopPoints(List<Message> loopInfoList, int index, MsuMdHandler.TrackDataHolder h) {
        if (loopInfoList == null || loopInfoList.size() <= index) {
            return h;
        }
        String str = Optional.ofNullable(loopInfoList.get(index)).map(Message::getInput).orElse("");
        if (str.contains("LOOP")) {
            h.cueLoop = Optional.of(!str.contains("NOLOOP"));
            if (h.cueLoop.get().booleanValue()) {
                try {
                    int loopPoint = Integer.parseInt(str.split("\\s+")[2].trim());
                    h.cueLoopPoint = Optional.of(loopPoint);
                }
                catch (Exception e) {
                    LOG.warn("Unable to parse loop point for pos #{}: {}", (Object)index, (Object)str);
                }
            }
            LOG.info("CueFile has loop info for pos #{}: {}", (Object)index, (Object)str);
        }
        return h;
    }

    private MsuMdHandlerImpl(CueSheet cueSheet, RandomAccessFile binFile) {
        this.binFile = binFile;
        LOG.info("Enabling MSU-MD handling, using cue sheet: {}", (Object)cueSheet.getFile().toAbsolutePath());
    }

    public static MsuMdHandler createInstance(SystemLoader.SystemType systemType, Path romPath) {
        if (romPath == null) {
            return NO_OP_HANDLER;
        }
        if (systemType.isMegaCdAttached()) {
            LOG.info("Disabling MSU-MD handling, {} detected.", (Object)systemType);
            return NO_OP_HANDLER;
        }
        CueSheet cueSheet = MsuMdHandlerImpl.initCueSheet(romPath);
        if (cueSheet == null) {
            LOG.info("Disabling MSU-MD handling, unable to find CUE file.");
            return NO_OP_HANDLER;
        }
        RandomAccessFile binFile = MsuMdHandlerImpl.initBinFile(romPath, cueSheet);
        if (binFile == null) {
            LOG.error("Disabling MSU-MD handling, unable to find BIN file");
            return NO_OP_HANDLER;
        }
        long binLen = 0L;
        try {
            binLen = binFile.length();
            if (binLen == 0L) {
                throw new Exception("Zero length file: " + String.valueOf(romPath));
            }
        }
        catch (Exception e) {
            LOG.error("Disabling MSU-MD handling, unable to find BIN file");
            return NO_OP_HANDLER;
        }
        MsuMdHandlerImpl h = new MsuMdHandlerImpl(cueSheet, binFile);
        h.initTrackData(cueSheet, binLen);
        return h;
    }

    private int handleFirstRead() {
        this.init = true;
        LOG.info("Read MCD_STATUS: {}", (Object)MCD_STATE.INIT);
        return MCD_STATE.INIT.ordinal();
    }

    private static CueSheet initCueSheet(Path romPath) {
        String romName = romPath.getFileName().toString();
        String cueFileName = romName.replace("." + Files.getFileExtension((String)romName), ".cue");
        return CueFileParser.parse(romPath.resolveSibling(cueFileName));
    }

    private static RandomAccessFile initBinFile(Path romPath, CueSheet sheet) {
        RandomAccessFile binFile = null;
        try {
            Path binPath = romPath.resolveSibling(((FileData)sheet.getFileData().get(0)).getFile());
            binFile = new RandomAccessFile(binPath.toFile(), "r");
        }
        catch (Exception e) {
            LOG.error("Error", (Throwable)e);
        }
        return binFile;
    }

    public void setBusy(boolean busy) {
        LOG.info("Busy: {} -> {}", (Object)(this.busy ? 1 : 0), (Object)(busy ? 1 : 0));
        this.busy = busy;
    }

    @Override
    public void handleMsuMdWrite(int address, int data, Size size) {
        if (address >= 0x420000 && address <= 4327423) {
            return;
        }
        this.handleMsuMdWriteByte(address, data, size);
    }

    private void handleMsuMdWriteByte(int address, int data, Size size) {
        switch (address) {
            case 10559519: {
                LOG.info("Cmd clock: {} -> {}", (Object)this.clock, (Object)data);
                this.processCommand(this.commandArg);
                this.clock = data;
                break;
            }
            case 10559504: {
                int cmdData = data >> (size == Size.WORD ? 8 : (size == Size.LONG ? 24 : 0));
                this.commandArg.command = MsuMdHandler.MsuCommand.getMsuCommand(cmdData);
                if (size == Size.WORD) {
                    this.commandArg.arg = data & 0xFF;
                }
                LOG.info("Cmd: {}, arg {}, arg1 {}", new Object[]{this.commandArg.command, this.commandArg.arg, this.commandArg.arg1});
                break;
            }
            case 10559506: {
                this.commandArg.arg1 = data;
                LOG.info("Cmd Arg: {}, arg {}, arg1 {}", new Object[]{this.commandArg.command, this.commandArg.arg, this.commandArg.arg1});
                break;
            }
            default: {
                this.handleIgnoredMcdWrite(address, data);
            }
        }
    }

    @Override
    public int handleMsuMdRead(int address, Size size) {
        switch (address) {
            case 10559520: {
                if (!this.init) {
                    return this.handleFirstRead();
                }
                MCD_STATE m = this.busy ? MCD_STATE.CMD_BUSY : MCD_STATE.READY;
                LOG.info("Read MCD_STATUS: {}, busy: {}", (Object)m, (Object)this.busy);
                return m.ordinal();
            }
            case 10559519: {
                LOG.info("Read CLOCK_ADDR: {}", (Object)this.clock);
                return this.clock;
            }
        }
        return this.handleIgnoredMcdRead(address, size);
    }

    private void handleIgnoredMcdWrite(int address, int data) {
        switch (address) {
            case 10559489: {
                LOG.info("Write MCD_GATE_ARRAY_START: {}", (Object)data);
                break;
            }
            case 10559491: {
                LOG.info("Write MCD_MMOD: {}", (Object)data);
                break;
            }
            case 10559503: {
                LOG.info("Write MCD_COMF: {}", (Object)data);
                break;
            }
            case 10559490: {
                LOG.info("Write MCD_MEMWP: {}", (Object)data);
                break;
            }
            default: {
                LOG.error("Unexpected bus write: {}, data {} {}", new Object[]{Util.th(address), Util.th(data), Size.BYTE});
            }
        }
    }

    private int handleIgnoredMcdRead(int address, Size size) {
        switch (address) {
            case 10559489: {
                LOG.info("Read MCD_GATE_ARRAY_START: {}", (Object)255);
                return size.getMask();
            }
        }
        LOG.warn("Unexpected MegaCD address range read at: {}, {}", (Object)Util.th(address), (Object)size);
        return size.getMask();
    }

    private MsuMdHandler.MsuCommandArg injectCueLoopSetup(MsuMdHandler.MsuCommandArg commandArg) {
        if (commandArg.command == MsuMdHandler.MsuCommand.PLAY && this.trackDataHolders[commandArg.arg].cueLoop.orElse(false).booleanValue()) {
            MsuMdHandler.TrackDataHolder tdh = this.trackDataHolders[commandArg.arg];
            String str = String.format("Using loop info from CUE file to override: %s %X %s", new Object[]{commandArg.command, commandArg.arg, commandArg.arg1});
            if (tdh.cueLoopPoint.orElse(0) > 0) {
                commandArg.command = MsuMdHandler.MsuCommand.PLAY_OFFSET;
                commandArg.arg1 = tdh.cueLoopPoint.get();
            } else {
                commandArg.command = MsuMdHandler.MsuCommand.PLAY_LOOP;
            }
            LOG.info("{} to: {} 0x{} 0x{}", new Object[]{str, commandArg.command, commandArg.arg, commandArg.arg1});
        }
        return commandArg;
    }

    private void processCommand(MsuMdHandler.MsuCommandArg commandArg) {
        int arg = commandArg.arg;
        Runnable r = null;
        LOG.info("{} track: {}", (Object)commandArg.command, (Object)arg);
        commandArg = this.injectCueLoopSetup(commandArg);
        switch (commandArg.command) {
            case PLAY: {
                this.clipContext.track = arg;
                this.clipContext.loop = false;
                this.clipContext.loopOffsetCDDAFrames = 0;
                r = this.playTrack(this.clipContext);
                break;
            }
            case PLAY_LOOP: {
                this.clipContext.track = arg;
                this.clipContext.loop = true;
                this.clipContext.loopOffsetCDDAFrames = 0;
                r = this.playTrack(this.clipContext);
                break;
            }
            case PAUSE: {
                r = this.pauseTrack((double)arg / 75.0);
                break;
            }
            case RESUME: {
                r = this.resumeTrack();
                break;
            }
            case VOL: {
                r = this.volumeTrack(arg);
                break;
            }
            case PLAY_OFFSET: {
                this.clipContext.track = arg;
                this.clipContext.loop = true;
                this.clipContext.loopOffsetCDDAFrames = commandArg.arg1;
                r = this.playTrack(this.clipContext);
                break;
            }
            default: {
                LOG.warn("Unknown command: {}", (Object)commandArg.command);
            }
        }
        if (r != null) {
            Util.executorService.submit(Util.wrapRunnableEx(r));
        }
    }

    private Runnable volumeTrack(int val) {
        return () -> {
            if (this.clip != null) {
                FloatControl gain = (FloatControl)this.clip.getControl(FloatControl.Type.MASTER_GAIN);
                double gainAbs = Math.max((double)val, 0.1) / 255.0;
                float gain_dB = (float)(Math.log(gainAbs) / Math.log(10.0) * 20.0);
                float prevDb = gain.getValue();
                gain.setValue(gain_dB);
                LOG.info("Volume: {}, gain_db changed from: {}, to: {}", new Object[]{val, Float.valueOf(prevDb), Float.valueOf(gain_dB)});
            }
        };
    }

    private void stopTrackInternal(boolean busy) {
        SoundUtil.close(this.clip);
        if (this.clip != null) {
            this.clip.removeLineListener(this.lineListenerRef.getAndSet(null));
        }
        this.setBusy(busy);
        LOG.info("Track stopped");
    }

    private Runnable pauseTrack(double fadeMs) {
        return () -> {
            if (this.clip != null) {
                this.clipContext.positionMicros = this.clip.getMicrosecondPosition();
                this.clip.stop();
                this.clipContext.paused = true;
            }
        };
    }

    private Runnable resumeTrack() {
        return () -> this.startClipInternal(this.clip, this.clipContext);
    }

    private Runnable playTrack(ClipContext context) {
        int track = context.track;
        if (this.lastPlayed == track && track == 20) {
            LOG.warn("Trying to play again track: {}, ignoring", (Object)track);
            return () -> {};
        }
        this.lastPlayed = track;
        this.setBusy(true);
        return () -> {
            try {
                MsuMdHandler.TrackDataHolder h = this.trackDataHolders[track];
                this.stopTrackInternal(true);
                this.clip = AudioSystem.getClip();
                this.lineListenerRef.set(this::handleClipEvent);
                this.clip.addLineListener(this.lineListenerRef.get());
                switch (h.type) {
                    case WAVE: 
                    case OGG: {
                        AudioInputStream ais = AudioSystem.getAudioInputStream(h.waveFile.get());
                        AudioInputStream dataIn = AudioSystem.getAudioInputStream(CDDA_FORMAT, ais);
                        this.clip.open(dataIn);
                        break;
                    }
                    case BINARY: {
                        this.prepareBuffer(h);
                        this.clip.open(CDDA_FORMAT, this.buffer, 0, h.numBytes.get());
                        break;
                    }
                    default: {
                        LOG.error("Unable to parse track {}, type: {}", (Object)track, (Object)h.type);
                        return;
                    }
                }
                if (!this.clipContext.paused) {
                    this.startClipInternal(this.clip, context);
                } else {
                    this.clipContext.positionMicros = 0L;
                }
                LOG.info("Track started: {}", (Object)track);
            }
            catch (Exception e) {
                LOG.error("Error", (Throwable)e);
                e.printStackTrace();
            }
            finally {
                this.setBusy(false);
            }
        };
    }

    @Override
    public void close() {
        this.stopTrackInternal(false);
        LOG.info("Closing");
    }

    private void handleClipEvent(LineEvent event) {
        LOG.info("Clip event: {}", (Object)event.getType());
    }

    private void startClipInternal(Clip clip, ClipContext context) {
        if (clip == null) {
            return;
        }
        LOG.info("Playing clip: {}", (Object)context);
        if (this.clipContext.positionMicros > 0L) {
            clip.setMicrosecondPosition(this.clipContext.positionMicros);
        }
        if (this.clipContext.loopOffsetCDDAFrames > 0) {
            int startLoop = Math.min(clip.getFrameLength(), MsuMdHandlerImpl.sectorsToClipFrames(context.loopOffsetCDDAFrames));
            clip.setLoopPoints(Math.max(startLoop, 0), -1);
        }
        clip.loop(context.loop ? -1 : 0);
        this.clipContext.paused = false;
        this.clipContext.positionMicros = 0L;
    }

    private void prepareBuffer(MsuMdHandler.TrackDataHolder h) {
        int numBytes = h.numBytes.get();
        if (this.buffer.length < numBytes) {
            this.buffer = new byte[numBytes];
        }
        Arrays.fill(this.buffer, (byte)0);
        try {
            this.binFile.seek(h.startFrame.get() * 2352);
            this.binFile.readFully(this.buffer, 0, numBytes);
        }
        catch (IOException e) {
            LOG.error("Error", (Throwable)e);
        }
    }

    static class ClipContext {
        int track;
        int loopOffsetCDDAFrames;
        long positionMicros;
        boolean loop;
        boolean paused;

        ClipContext() {
        }

        public String toString() {
            return "ClipContext{track=" + this.track + ", loopOffsetCDDAFrames=" + this.loopOffsetCDDAFrames + ", positionMicros=" + this.positionMicros + ", loop=" + this.loop + ", paused=" + this.paused + "}";
        }
    }

    static enum MCD_STATE {
        READY,
        INIT,
        CMD_BUSY;

    }
}

