/*
 * Decompiled with CFR 0.152.
 */
package omegadrive.bus.md;

import java.util.Objects;
import java.util.Optional;
import omegadrive.Device;
import omegadrive.SystemLoader;
import omegadrive.bus.DeviceAwareBus;
import omegadrive.bus.md.BusArbiter;
import omegadrive.bus.md.SvpMapper;
import omegadrive.bus.model.MdM68kBusProvider;
import omegadrive.bus.model.MdMainBusProvider;
import omegadrive.bus.model.SvpBus;
import omegadrive.cart.MdCartInfoProvider;
import omegadrive.cart.loader.MdRomDbModel;
import omegadrive.cart.mapper.RomMapper;
import omegadrive.cart.mapper.md.ExSsfMapper;
import omegadrive.cart.mapper.md.MdBackupMemoryMapper;
import omegadrive.joypad.JoypadProvider;
import omegadrive.joypad.MdJoypad;
import omegadrive.sound.fm.FmProvider;
import omegadrive.sound.msumd.MsuMdHandler;
import omegadrive.sound.msumd.MsuMdHandlerImpl;
import omegadrive.sound.psg.PsgProvider;
import omegadrive.system.SystemProvider;
import omegadrive.util.BufferUtil;
import omegadrive.util.LogHelper;
import omegadrive.util.MdRuntimeData;
import omegadrive.util.Size;
import omegadrive.util.Util;
import omegadrive.vdp.model.MdVdpProvider;
import org.slf4j.Logger;

public class MdBus
extends DeviceAwareBus<MdVdpProvider, MdJoypad>
implements MdMainBusProvider,
RomMapper {
    private static final Logger LOG = LogHelper.getLogger(MdBus.class.getSimpleName());
    public static final boolean verbose = false;
    public static final int M68K_CYCLE_PENALTY = 3;
    protected MdCartInfoProvider cartridgeInfoProvider;
    private RomMapper mapper;
    private RomMapper exSsfMapper = RomMapper.NO_OP_MAPPER;
    private RomMapper backupMemMapper = RomMapper.NO_OP_MAPPER;
    private SvpBus svpMapper = SvpBus.NO_OP;
    protected MsuMdHandler msuMdHandler = MsuMdHandler.NO_OP_HANDLER;
    private BusArbiter busArbiter = BusArbiter.NO_OP;
    final MdM68kBusProvider.BusWriteRunnable vdpRunnable = new MdM68kBusProvider.BusWriteRunnable(){

        @Override
        public void run() {
            BufferUtil.CpuDeviceAccess prev = MdRuntimeData.setAccessTypeExt(this.cpu);
            MdBus.this.write(this.address, this.data, this.size);
            MdRuntimeData.setAccessTypeExt(prev);
        }
    };
    private static long ROM_END_ADDRESS;
    private BusState busState = BusState.NOT_READY;
    protected byte[] rom;
    protected byte[] ram;
    protected int romMask;
    private boolean z80BusRequested;
    private boolean z80ResetState;
    private int sramLockValue;
    private final boolean enableTmss;
    private byte[] serialPortData = new byte[20];

    public MdBus() {
        this.mapper = this;
        this.enableTmss = Boolean.parseBoolean(System.getProperty("md.enable.tmss", "false"));
    }

    void initializeRomData() {
        SystemLoader.SystemType st = this.systemProvider.getSystemType();
        this.cartridgeInfoProvider = (MdCartInfoProvider)this.systemProvider.getMediaSpec().getBootableMedia().mediaInfoProvider;
        assert (this.cartridgeInfoProvider != null);
        ROM_END_ADDRESS = Math.min(this.cartridgeInfoProvider.getRomSize(), 0xA00000);
        assert (ROM_END_ADDRESS > 0L);
        if (this.cartridgeInfoProvider.getEntry().hasEeprom()) {
            this.checkBackupMemoryMapper(RomMapper.SramMode.READ_WRITE, this.cartridgeInfoProvider.getEntry());
        } else if (this.cartridgeInfoProvider.isSramEnabled()) {
            this.checkBackupMemoryMapper(RomMapper.SramMode.DISABLE);
        }
        if (this.cartridgeInfoProvider.isSsfMapper()) {
            this.checkExSsfMapper();
        }
        this.msuMdHandler = MsuMdHandlerImpl.createInstance(st, this.systemProvider.getRomPath());
        if (st == SystemLoader.SystemType.MD && !this.cartridgeInfoProvider.isSsfMapper() && ROM_END_ADDRESS > 0x3FFFFFL) {
            LOG.warn("Assuming flat ROM mapper up to address: {}", (Object)ROM_END_ADDRESS);
        }
        if (this.cartridgeInfoProvider.isSvp()) {
            this.checkSvpMapper();
        }
        if (this.cartridgeInfoProvider.getEntry().hasForce3Btn()) {
            this.systemProvider.handleSystemEvent(SystemProvider.SystemEvent.FORCE_PAD_TYPE, JoypadProvider.JoypadType.BUTTON_3.name());
        }
    }

    @Override
    public void resetFrom68k() {
        this.getFm().reset();
        this.z80Provider.reset();
        this.z80ResetState = true;
    }

    @Override
    public boolean is68kRunning() {
        return this.busArbiter.is68kRunning();
    }

    @Override
    public void setVdpBusyState(MdVdpProvider.VdpBusyState state) {
        this.busArbiter.setVdpBusyState(state);
    }

    @Override
    public MdMainBusProvider attachDevice(Device device) {
        if (device instanceof BusArbiter) {
            this.busArbiter = (BusArbiter)device;
        }
        super.attachDevice(device);
        return this;
    }

    private void detectState() {
        boolean ok = Objects.nonNull(this.systemProvider) && Objects.nonNull(this.m68kProvider) && Objects.nonNull(this.joypadProvider) && Objects.nonNull(this.vdpProvider) && Objects.nonNull(this.memoryProvider) && Objects.nonNull(this.z80Provider) && Objects.nonNull(this.soundProvider) && Objects.nonNull(this.cartridgeInfoProvider) && Objects.nonNull(this.busArbiter);
        this.busState = ok ? BusState.READY : BusState.NOT_READY;
    }

    @Override
    public int read(int address, Size size) {
        return this.mapper.readData(address, size);
    }

    @Override
    public void write(int address, int data, Size size) {
        this.mapper.writeData(address, data, size);
    }

    @Override
    public void init() {
        this.initializeRomData();
        if (this.busArbiter == null || this.busArbiter == BusArbiter.NO_OP) {
            this.attachDevice(BusArbiter.createInstance((MdVdpProvider)this.vdpProvider, this.m68kProvider, this.z80Provider));
        } else {
            LOG.warn("BusArbiter already created");
        }
        this.z80BusRequested = false;
        this.z80ResetState = true;
        this.detectState();
        LOG.info("Bus state: {}", (Object)this.busState);
        this.ram = this.memoryProvider.getRamData();
        this.rom = this.memoryProvider.getRomData();
        this.romMask = this.memoryProvider.getRomMask();
    }

    @Override
    public int readData(int address, Size size) {
        int data;
        if ((long)(address &= 0xFFFFFF) < ROM_END_ADDRESS) {
            data = Util.readDataMask(this.rom, address, this.romMask, size);
        } else if (address >= 0xE00000 && address <= 0xFFFFFF) {
            data = Util.readDataMask(this.ram, address, 65535, size);
        } else if (address > 0x3FFFFF && address < 0xA00000) {
            data = this.reservedRead(address, size);
        } else if (address >= 0xA00000 && address <= 0xA0FFFF) {
            data = this.z80MemoryRead(address, size);
        } else if (address >= 0xA10000 && address <= 10555391) {
            data = this.ioRead(address, size);
        } else if (address >= 0xA11000 && address <= 0xBFFFFF) {
            data = this.internalRegRead(address, size);
        } else if (address >= 0xC00000 && address <= 0xDFFFFF) {
            data = this.vdpRead(address, size);
        } else if (this.cartridgeInfoProvider.isSramUsedWithBrokenHeader(address)) {
            this.checkBackupMemoryMapper(RomMapper.SramMode.READ_WRITE);
            data = this.mapper.readData(address, size);
        } else {
            LogHelper.logWarnOnce(LOG, "Unexpected bus read: {}, 68k PC: {}", Util.th(address), Util.th(this.m68kProvider.getPC()));
            data = size.getMask();
        }
        return data & size.getMask();
    }

    @Override
    public void writeData(int address, int data, Size size) {
        data &= size.getMask();
        if ((address &= 0xFFFFFF) >= 0xE00000 && address <= 0xFFFFFF) {
            Util.writeDataMask(this.ram, address, data, 65535, size);
        } else if (address >= 0xA00000 && address <= 0xA0FFFF) {
            this.z80MemoryWrite(address, size, data);
        } else if (address >= 0xA10000 && address <= 10555391) {
            this.ioWrite(address, size, data);
        } else if (address >= 0xA11000 && address <= 0xBFFFFF) {
            this.internalRegWrite(address, size, data);
        } else if (address >= 0xC00000 && address < 0xDFFFFF) {
            this.vdpWrite(address, size, data);
        } else if ((long)address < ROM_END_ADDRESS) {
            this.cartWrite(address, data, size);
        } else if (this.cartridgeInfoProvider.isSramUsedWithBrokenHeader(address)) {
            this.checkBackupMemoryMapper(RomMapper.SramMode.READ_WRITE);
            this.mapper.writeData(address, data, size);
        } else {
            this.reservedWrite(address, data, size);
        }
    }

    private void cartWrite(int addressL, int data, Size size) {
        if (this.cartridgeInfoProvider.isSramUsedWithBrokenHeader(addressL)) {
            LOG.info("Unexpected Sram write: {}, value : {}", (Object)Util.th(addressL), (Object)data);
            boolean adjust = this.cartridgeInfoProvider.adjustSramLimits(addressL);
            this.checkBackupMemoryMapper(RomMapper.SramMode.READ_WRITE, adjust);
            this.mapper.writeData(addressL, data, size);
            return;
        }
        LogHelper.logWarnOnce(LOG, "Unexpected write to ROM address {} {}", new Object[]{Util.th(addressL), size});
    }

    private void reservedWrite(int addressL, int data, Size size) {
        if (this.msuMdHandler == MsuMdHandler.NO_OP_HANDLER) {
            LogHelper.logWarnOnce(LOG, "Unexpected bus write: {}, 68k PC: {}", Util.th(addressL), Util.th(this.m68kProvider.getPC()));
        } else {
            this.msuMdHandler.handleMsuMdWrite(addressL, data, size);
        }
    }

    private int reservedRead(int address, Size size) {
        if (this.msuMdHandler == MsuMdHandler.NO_OP_HANDLER) {
            LogHelper.logWarnOnce(LOG, "Read on reserved address: {}, {}", new Object[]{Util.th(address), size});
            return size.getMax();
        }
        assert (false);
        return Util.readDataMask(this.rom, address, 0x3FFFFF, size);
    }

    private void logVdpCounter(int v, int h) {
    }

    private int internalRegRead(int address, Size size) {
        if (address >= 0xA11100 && address <= 0xA111FF) {
            return this.z80BusReqRead(size);
        }
        if (address >= 10555904 && address <= 10556159) {
            LogHelper.logWarnOnce(LOG, "Unexpected Z80 read at: {}, {}", new Object[]{Util.th(address), size});
        } else {
            if (address >= 10559488 && address <= 10559743) {
                return this.msuMdHandler.handleMsuMdRead(address, size);
            }
            if (address >= 10563584 && address <= 10563839) {
                return this.timeLineControlRead(address, size);
            }
            if (address >= 10567680 && address <= 10567683) {
                LOG.warn("TMSS read enable cart");
            } else if (address >= 10567936 && address <= 10567937) {
                LOG.warn("TMSS read enable cart");
            } else if (address >= 0xA11000 && address <= 10555647) {
                LOG.warn("Memory mode reg read");
            } else {
                if (address >= 10571776 && address <= 10571784) {
                    this.checkSvpMapper();
                    return this.svpMapper.m68kSvpRegRead(address, size);
                }
                LOG.error("Unexpected internalRegRead: {} , {}", (Object)Util.th(address), (Object)size);
            }
        }
        return 255;
    }

    private void internalRegWrite(int address, Size size, int data) {
        if (address >= 0xA11000 && address <= 10555647) {
            LOG.info("Setting memory mode to: {}", (Object)data);
        } else if (address >= 0xA11100 && address <= 0xA111FF) {
            this.z80BusReqWrite(data, size);
        } else if (address >= 10555904 && address <= 10556159) {
            this.z80ResetControlWrite(data, size);
        } else if (address >= 10563584 && address <= 10563839) {
            assert (size != Size.LONG);
            this.timeLineControlWrite(address, data, size);
        } else if (address >= 10567680 && address <= 10567683) {
            LOG.warn("TMSS write, vdp lock: {}", (Object)Util.th(data));
        } else if (address == 10567936 || address == 10567937) {
            LOG.warn("TMSS write enable cart: {}", (Object)(data == 1 ? 1 : 0));
        } else if (address >= 10571776 && address <= 10571784) {
            this.checkSvpMapper();
            this.svpMapper.m68kSvpRegWrite(address, data, size);
        } else if (address >= 10559488 && address <= 10559743) {
            this.msuMdHandler.handleMsuMdWrite(address, data, size);
        } else {
            LOG.warn("Unexpected internalRegWrite: {}, {}, {}", new Object[]{Util.th(address), Util.th(data), size});
        }
    }

    private int z80BusReqRead(Size size) {
        int prefetch = this.m68kProvider.getPrefetchWord();
        int value = prefetch & 0xFE | (this.z80BusRequested && !this.z80ResetState ? 0 : 1);
        if (size == Size.WORD) {
            value = value << 8 | prefetch & 0xFEFF;
        }
        return value;
    }

    private void timeLineControlWrite(int addressL, int data, Size size) {
        if (size == Size.WORD) {
            LogHelper.logWarnOnce(LOG, "timeLineControlWrite should be byte-wide {} {}", new Object[]{Util.th(addressL), size});
            addressL |= 1;
        }
        if (addressL >= 10563827 && addressL <= 10563839) {
            LOG.info("Mapper bank set, address: {} , data: {}", (Object)Util.th(addressL), (Object)Util.th(data));
            this.checkExSsfMapper();
            this.mapper.writeBankData(addressL, data);
        } else if (addressL == 10563825) {
            boolean writable;
            this.sramLockValue = data;
            boolean rom = (data & 1) == 0;
            boolean bl = writable = (data & 2) > 0;
            if (rom) {
                this.checkExSsfMapper();
            } else {
                this.checkBackupMemoryMapper(writable ? RomMapper.SramMode.READ_WRITE : RomMapper.SramMode.READ_ONLY);
            }
            if (this.backupMemMapper != NO_OP_MAPPER) {
                this.backupMemMapper.setSramMode(writable ? RomMapper.SramMode.READ_WRITE : RomMapper.SramMode.READ_ONLY);
            }
        } else {
            LOG.warn("Unexpected mapper set, address: {}, data: {} {}", new Object[]{Util.th(addressL), Util.th(data), size});
        }
    }

    private int timeLineControlRead(int addressL, Size size) {
        if (addressL == 10563825 && size == Size.BYTE) {
            return this.sramLockValue;
        }
        LOG.warn("Unexpected /TIME or mapper read at: {} {}", (Object)Util.th(addressL), (Object)size);
        return size.getMask();
    }

    private void z80ResetControlWrite(int data, Size size) {
        boolean reset;
        if (size == Size.WORD) {
            data >>>= 8;
        }
        boolean bl = reset = (data & 1) == 0;
        if (reset) {
            if (this.z80BusRequested) {
                this.z80ResetState = true;
            }
        } else if (this.z80ResetState) {
            this.doZ80Reset();
            this.z80ResetState = false;
        }
    }

    private void doZ80Reset() {
        this.z80Provider.reset();
        this.getFm().reset();
    }

    private void z80BusReqWrite(int data, Size size) {
        boolean busReq;
        if (size == Size.WORD) {
            data >>>= 8;
        }
        boolean bl = busReq = (data & 1) > 0;
        if (busReq) {
            if (!this.z80BusRequested) {
                this.z80BusRequested = true;
            }
        } else if (this.z80BusRequested) {
            this.z80BusRequested = false;
        }
    }

    private int ioRead(int address, Size size) {
        int data = 0;
        if ((address & 0xFFF) <= 1) {
            int expUnit = this.systemProvider.getSystemType().isMegaCdAttached() ? 0 : 32;
            data = expUnit | this.systemProvider.getRegionCode() | (this.enableTmss ? 1 : 0);
            data = size == Size.WORD ? data << 8 | data : data;
            return data;
        }
        switch (size) {
            case BYTE: 
            case WORD: {
                data = this.ioReadInternal(address, size);
                break;
            }
            case LONG: {
                data = this.ioReadInternal(address, size);
                data = data << 16 | this.ioReadInternal(address + 2, size);
            }
        }
        return data;
    }

    private int ioReadInternal(int addressL, Size size) {
        int data = 255;
        int address = addressL & 0xFFE;
        switch (address) {
            case 2: {
                data = ((MdJoypad)this.joypadProvider).readDataRegister1();
                break;
            }
            case 4: {
                data = ((MdJoypad)this.joypadProvider).readDataRegister2();
                break;
            }
            case 6: {
                data = ((MdJoypad)this.joypadProvider).readDataRegister3();
                break;
            }
            case 8: {
                data = ((MdJoypad)this.joypadProvider).readControlRegister1();
                break;
            }
            case 10: {
                data = ((MdJoypad)this.joypadProvider).readControlRegister2();
                break;
            }
            case 12: {
                data = ((MdJoypad)this.joypadProvider).readControlRegister3();
                break;
            }
            default: {
                if (address >= 14 && address < 32) {
                    data = Util.readData(this.serialPortData, address - 14, size);
                    LogHelper.logWarnOnce(LOG, "Reading serial control: {} {}, data: {}", new Object[]{Util.th(address), size, Util.th(data)});
                    break;
                }
                LogHelper.logWarnOnce(LOG, "Unexpected ioRead: {}", Util.th(addressL));
            }
        }
        if (size == Size.WORD) {
            data |= data << 8 & 0xFF00;
        }
        return data;
    }

    private void ioWrite(int address, Size size, int data) {
        switch (size) {
            case BYTE: {
                if ((address & 1) == 0) {
                    LogHelper.logWarnOnce(LOG, "Ignore byte-wide writes on even addr: {} {}", new Object[]{Util.th(address), size});
                    return;
                }
            }
            case WORD: {
                this.ioWriteInternal(address, data, size);
                break;
            }
            case LONG: {
                this.ioWriteInternal(address, data >>> 16, size);
                this.ioWriteInternal(address + 2, data & 0xFFFF, size);
            }
        }
    }

    private void ioWriteInternal(int addressL, int dataL, Size size) {
        int address = addressL & 0xFFE;
        int data = dataL & 0xFF;
        switch (address) {
            case 2: {
                ((MdJoypad)this.joypadProvider).writeDataRegister1(data);
                break;
            }
            case 4: {
                ((MdJoypad)this.joypadProvider).writeDataRegister2(data);
                break;
            }
            case 6: {
                break;
            }
            case 8: {
                ((MdJoypad)this.joypadProvider).writeControlRegister1(data);
                break;
            }
            case 10: {
                ((MdJoypad)this.joypadProvider).writeControlRegister2(data);
                break;
            }
            case 12: {
                ((MdJoypad)this.joypadProvider).writeControlRegister3(data);
                break;
            }
            default: {
                if (address >= 14 && address < 32) {
                    LogHelper.logWarnOnceWhenEn(LOG, "Write serial control: {}", Util.th(addressL));
                    Util.writeData(this.serialPortData, address - 14, data, size);
                    break;
                }
                LOG.error("Unexpected ioWrite {}, data {}", (Object)Util.th(addressL), (Object)Util.th(data));
            }
        }
    }

    private int z80MemoryRead(int address, Size size) {
        this.busArbiter.addCyclePenalty(BusArbiter.CpuType.M68K, 3);
        if (!this.z80BusRequested || this.z80ResetState) {
            LOG.warn("68k read access to Z80 bus with busreq: {}, z80reset: {}", (Object)this.z80BusRequested, (Object)this.z80ResetState);
            return 0;
        }
        int addressZ = address & Short.MAX_VALUE;
        if (size == Size.BYTE) {
            return this.z80Provider.readMemory(addressZ);
        }
        if (size == Size.WORD) {
            return this.z80MemoryReadWord(addressZ);
        }
        this.busArbiter.addCyclePenalty(BusArbiter.CpuType.M68K, 3);
        int dataHigh = this.z80MemoryReadWord(addressZ);
        int dataLow = this.z80MemoryReadWord(addressZ + 2);
        return dataHigh << 16 | dataLow;
    }

    private void z80MemoryWrite(int address, Size size, int dataL) {
        this.busArbiter.addCyclePenalty(BusArbiter.CpuType.M68K, 3);
        if (!this.z80BusRequested || this.z80ResetState) {
            LogHelper.logWarnOnce(LOG, "68k write access to Z80 bus with busreq: {}, z80reset: {}", this.z80BusRequested, this.z80ResetState);
            return;
        }
        address &= Short.MAX_VALUE;
        if (size == Size.BYTE) {
            this.z80Provider.writeMemory(address, dataL);
        } else if (size == Size.WORD) {
            this.z80MemoryWriteWord(address, dataL);
        } else {
            this.busArbiter.addCyclePenalty(BusArbiter.CpuType.M68K, 3);
            this.z80MemoryWriteWord(address, dataL >>> 16);
            this.z80MemoryWriteWord(address + 2, dataL & 0xFFFF);
        }
    }

    private void z80MemoryWriteWord(int address, int data) {
        this.z80Provider.writeMemory(address, data >> 8 & 0xFF);
    }

    private int z80MemoryReadWord(int address) {
        int data = this.z80Provider.readMemory(address);
        return data << 8 | data;
    }

    private int vdpRead(int address, Size size) {
        switch (size) {
            case BYTE: 
            case WORD: {
                return this.vdpReadInternal(address, size);
            }
            case LONG: {
                int res = this.vdpReadInternal(address, Size.WORD) << 16;
                return res | this.vdpReadInternal(address + 2, Size.WORD);
            }
        }
        return 255;
    }

    private void vdpWrite(int address, Size size, int data) {
        switch (size) {
            case BYTE: 
            case WORD: {
                this.vdpWriteWord(address, size, data);
                break;
            }
            case LONG: {
                int address1 = address & 0x1F;
                if (address1 < 8 && this.checkVdpBusy(address, size, data)) {
                    return;
                }
                this.vdpWriteWord(address, Size.WORD, data >>> 16);
                this.vdpWriteWord(address + 2, Size.WORD, data & 0xFFFF);
            }
        }
    }

    private int vdpReadInternal(int addressL, Size size) {
        boolean valid;
        boolean bl = valid = (addressL & 0xE700E0) == 0xC00000;
        if (!valid) {
            LogHelper.logWarnOnce(LOG, "Illegal VDP read at {} {}", new Object[]{Util.th(addressL), size});
            return size.getMask();
        }
        int address = addressL & 0x1F;
        if (address <= 7) {
            int vdpData = this.readVdpPortWord(address);
            switch (size) {
                case WORD: {
                    return vdpData;
                }
                case BYTE: {
                    return (address & 1) == 0 ? vdpData >>> 8 : vdpData & 0xFF;
                }
            }
            LOG.error("Unexpected {} vdp port read: {}", (Object)size, (Object)Util.th(addressL));
            return 255;
        }
        if (address <= 14) {
            int v = ((MdVdpProvider)this.vdpProvider).getVCounter();
            int h = ((MdVdpProvider)this.vdpProvider).getHCounter();
            this.logVdpCounter(v, h);
            if (size == Size.WORD) {
                return v << 8 | h;
            }
            if (size == Size.BYTE) {
                return (address & 1) == 0 ? v : h;
            }
        } else if (address == 28) {
            LOG.warn("Ignoring VDP debug register read, address : {}", (Object)Util.th(addressL));
        } else if (address > 23) {
            LOG.info("vdpRead on unused address: {}", (Object)Util.th(addressL));
        } else {
            LOG.warn("Unexpected vdpRead, address: {}", (Object)Util.th(addressL));
        }
        return 255;
    }

    private int readVdpPortWord(int address) {
        MdVdpProvider.VdpPortType portType = address < 4 ? MdVdpProvider.VdpPortType.DATA : MdVdpProvider.VdpPortType.CONTROL;
        return ((MdVdpProvider)this.vdpProvider).readVdpPortWord(portType);
    }

    private void vdpWriteWord(int addressLong, Size size, int data) {
        boolean valid;
        boolean bl = valid = (addressLong & 0xE700E0) == 0xC00000;
        if (!valid) {
            LOG.error("Illegal VDP write, address {}, data {}, size {}", new Object[]{Util.th(addressLong), Util.th(data), size});
            throw new RuntimeException("Illegal VDP write");
        }
        int address = addressLong & 0x1F;
        if (address < 8) {
            MdVdpProvider.VdpPortType portType;
            MdVdpProvider.VdpPortType vdpPortType = portType = address < 4 ? MdVdpProvider.VdpPortType.DATA : MdVdpProvider.VdpPortType.CONTROL;
            if (this.checkVdpBusy(addressLong, size, data)) {
                return;
            }
            if (size == Size.BYTE) {
                data = data << 8 | data;
            }
            ((MdVdpProvider)this.vdpProvider).writeVdpPortWord(portType, data);
        } else if (address < 15) {
            LOG.warn("HV counter write");
        } else if (address >= 16 && address < 24) {
            int psgData = data & 0xFF;
            if (size == Size.WORD) {
                LOG.warn("PSG word write, address: {}, data: {}", (Object)Util.th(address), (Object)data);
            }
            this.soundProvider.getPsg().write(psgData);
        } else {
            LOG.warn("Unexpected vdpWrite, address: {}, data: {}", (Object)Util.th(address), (Object)data);
        }
    }

    private boolean checkVdpBusy(int address, Size size, int data) {
        if (this.busArbiter.getVdpBusyState() == MdVdpProvider.VdpBusyState.FIFO_FULL || this.busArbiter.getVdpBusyState() == MdVdpProvider.VdpBusyState.MEM_TO_VRAM) {
            this.vdpRunnable.cpu = MdRuntimeData.getAccessTypeExt();
            this.vdpRunnable.address = address;
            this.vdpRunnable.size = size;
            this.vdpRunnable.data = data;
            this.busArbiter.runLater68k(this.vdpRunnable);
            return true;
        }
        return false;
    }

    private void checkExSsfMapper() {
        if (this.exSsfMapper == RomMapper.NO_OP_MAPPER) {
            this.exSsfMapper = ExSsfMapper.createInstance(this, this.memoryProvider);
        }
        this.mapper = this.exSsfMapper;
    }

    private void checkSvpMapper() {
        if (this.svpMapper == SvpBus.NO_OP) {
            this.svpMapper = SvpMapper.createInstance((RomMapper)this, this.memoryProvider);
            this.mapper = this.svpMapper;
            LOG.info("Enabling mapper: {}", (Object)this.mapper.getClass().getSimpleName());
        }
    }

    private void checkBackupMemoryMapper(RomMapper.SramMode sramMode) {
        this.checkBackupMemoryMapper(sramMode, false);
    }

    private void checkBackupMemoryMapper(RomMapper.SramMode sramMode, boolean forceCreateSram) {
        this.checkBackupMemoryMapper(sramMode, forceCreateSram ? this.cartridgeInfoProvider.getEntry() : MdRomDbModel.NO_ENTRY);
    }

    private void checkBackupMemoryMapper(RomMapper.SramMode sramMode, MdRomDbModel.RomDbEntry entry) {
        if (this.backupMemMapper == RomMapper.NO_OP_MAPPER) {
            this.backupMemMapper = MdBackupMemoryMapper.createInstance(this, this.cartridgeInfoProvider, sramMode, entry);
        }
        this.backupMemMapper.setSramMode(sramMode);
        this.mapper = this.backupMemMapper;
    }

    @Override
    public boolean isSvp() {
        return this.svpMapper != SvpBus.NO_OP;
    }

    @Override
    public FmProvider getFm() {
        return this.soundProvider.getFm();
    }

    @Override
    public PsgProvider getPsg() {
        return this.soundProvider.getPsg();
    }

    @Override
    public SystemProvider getSystem() {
        return this.systemProvider;
    }

    @Override
    public MdVdpProvider getVdp() {
        return (MdVdpProvider)this.vdpProvider;
    }

    @Override
    public boolean isZ80Running() {
        return !this.z80ResetState && !this.z80BusRequested;
    }

    @Override
    public void setZ80BusRequested(boolean z80BusRequested) {
        this.z80BusRequested = z80BusRequested;
    }

    @Override
    public void setZ80ResetState(boolean z80ResetState) {
        this.z80ResetState = z80ResetState;
    }

    @Override
    public boolean isZ80BusRequested() {
        return this.z80BusRequested;
    }

    @Override
    public boolean isZ80ResetState() {
        return this.z80ResetState;
    }

    @Override
    public void handleVdpInterrupts68k() {
        this.busArbiter.handleInterrupts(BufferUtil.CpuDeviceAccess.M68K);
    }

    @Override
    public void handleVdpInterruptsZ80() {
        this.busArbiter.handleInterrupts(BufferUtil.CpuDeviceAccess.Z80);
    }

    @Override
    public void ackInterrupt68k(int level) {
        this.busArbiter.ackInterrupt68k(level);
    }

    @Override
    public int[] getMapperData() {
        return this.getStateAwareMapper().getState();
    }

    @Override
    public void setMapperData(int[] data) {
        this.getStateAwareMapper().setState(data);
    }

    private RomMapper.StateAwareMapper getStateAwareMapper() {
        RomMapper.StateAwareMapper m = NO_STATE;
        if (this.exSsfMapper instanceof RomMapper.StateAwareMapper) {
            m = (RomMapper.StateAwareMapper)((Object)this.exSsfMapper);
        } else if (this.mapper instanceof RomMapper.StateAwareMapper) {
            m = (RomMapper.StateAwareMapper)((Object)this.mapper);
        }
        return m;
    }

    @Override
    public MdCartInfoProvider getCartridgeInfoProvider() {
        return this.cartridgeInfoProvider;
    }

    private static void logInfo(String str, Object ... args) {
    }

    @Override
    public void closeRom() {
        if (this.mapper != this) {
            this.mapper.closeRom();
        }
        this.exSsfMapper.closeRom();
        if (this.mapper != this.backupMemMapper) {
            this.backupMemMapper.closeRom();
        }
        this.msuMdHandler.close();
    }

    @Override
    public void onNewFrame() {
        Optional.ofNullable((MdJoypad)this.joypadProvider).ifPresent(JoypadProvider::newFrame);
    }

    static enum BusState {
        READY,
        NOT_READY;

    }
}

