/*
 * Decompiled with CFR 0.152.
 */
package s32x.sh2.cache;

import java.io.Serializable;
import java.nio.ByteBuffer;
import java.util.Optional;
import omegadrive.util.BufferUtil;
import omegadrive.util.LogHelper;
import omegadrive.util.MdRuntimeData;
import omegadrive.util.Size;
import omegadrive.util.Util;
import org.slf4j.Logger;
import s32x.bus.Sh2Bus;
import s32x.savestate.Gs32xStateHandler;
import s32x.sh2.cache.Sh2Cache;

public class Sh2CacheImpl
implements Sh2Cache {
    private static final Logger LOG = LogHelper.getLogger(Sh2CacheImpl.class.getSimpleName());
    private static final boolean verbose = false;
    public static final boolean PARANOID_ON_CACHE_ENABLED_TOGGLE = false;
    protected final ByteBuffer data_array = ByteBuffer.allocate(4096);
    protected Sh2Cache.Sh2CacheContext ctx;
    private Sh2Cache.CacheRegContext cacheRegCtx;
    protected Sh2Cache.Sh2CacheEntry ca;
    private final BufferUtil.CpuDeviceAccess cpu;
    private final Sh2Bus memory;
    private final Sh2Cache.CacheInvalidateContext invalidCtx;

    protected Sh2CacheImpl(BufferUtil.CpuDeviceAccess cpu, Sh2Bus memory) {
        this.memory = memory;
        this.cpu = cpu;
        this.ctx = new Sh2Cache.Sh2CacheContext();
        this.ca = this.ctx.ca = new Sh2Cache.Sh2CacheEntry();
        this.cacheRegCtx = this.ctx.cacheContext = new Sh2Cache.CacheRegContext();
        this.cacheRegCtx.cpu = cpu;
        this.invalidCtx = new Sh2Cache.CacheInvalidateContext();
        this.invalidCtx.cpu = cpu;
        for (int i = 0; i < this.ca.way.length; ++i) {
            for (int j = 0; j < this.ca.way[i].length; ++j) {
                this.ca.way[i][j] = new Sh2Cache.Sh2CacheLine();
            }
        }
        Gs32xStateHandler.addDevice(this);
    }

    @Override
    public void cacheClear() {
        for (int entry = 0; entry < 64; ++entry) {
            this.ca.lru[entry] = 0;
            for (int way = 0; way < 4; ++way) {
                this.invalidatePrefetcher(this.ca.way[way][entry], entry, -1);
                this.ca.way[way][entry].v = 0;
            }
        }
    }

    @Override
    public int readDirect(int addr, Size size) {
        switch (addr & 0xE0000000) {
            case 0: {
                if (this.ca.enable > 0) {
                    int tagaddr = addr & 0x1FFFFC00;
                    int entry = (addr & 0x3F0) >> 4;
                    for (int i = 0; i < 4; ++i) {
                        Sh2Cache.Sh2CacheLine line = this.ca.way[i][entry];
                        if (line.v <= 0 || line.tag != tagaddr) continue;
                        return Sh2CacheImpl.getCachedData(line.data, addr & 0xF, size) & size.getMask();
                    }
                }
                assert (this.cpu == MdRuntimeData.getAccessTypeExt());
                return this.memory.readMemoryUncachedNoDelay(addr, size);
            }
            case -1073741824: {
                return this.readDataArray(addr, size);
            }
        }
        LOG.error("{} Unexpected cache read: {}, {}", new Object[]{this.cpu, Util.th(addr), size});
        throw new RuntimeException();
    }

    @Override
    public int cacheMemoryRead(int addr, Size size) {
        switch (addr & 0xE0000000) {
            case 0: {
                if (this.ca.enable == 0) {
                    assert (this.cpu == MdRuntimeData.getAccessTypeExt());
                    return this.readMemoryUncached(this.memory, addr, size);
                }
                return this.readCache(addr, size);
            }
            case -1073741824: {
                return this.readDataArray(addr, size);
            }
            case 0x40000000: {
                this.purgeCache(addr);
                break;
            }
            case 0x60000000: {
                LOG.warn("{} CACHE_ADDRESS_ARRAY read: {}, {}", new Object[]{this.cpu, Util.th(addr), size});
                return this.readAddressArray(addr);
            }
            default: {
                LOG.error("{} Unexpected cache read: {}, {}", new Object[]{this.cpu, Util.th(addr), size});
                throw new RuntimeException();
            }
        }
        return size.getMask();
    }

    @Override
    public boolean cacheMemoryWrite(int addr, int val, Size size) {
        boolean change = false;
        switch (addr & 0xE0000000) {
            case 0: {
                if (this.ca.enable == 0) {
                    this.writeMemoryUncached(this.memory, addr, val, size);
                    return true;
                }
                change = this.writeCache(addr, val, size);
                break;
            }
            case -1073741824: {
                change = this.writeDataArray(addr, val, size);
                break;
            }
            case 0x40000000: {
                this.purgeCache(addr);
                break;
            }
            case 0x60000000: {
                assert (size == Size.LONG);
                this.writeAddressArray(addr, val);
                break;
            }
            default: {
                LOG.error("{} Unexpected cache write: {}, {} {}", new Object[]{this.cpu, Util.th(addr), Util.th(val), size});
                throw new RuntimeException();
            }
        }
        return change;
    }

    private int readCache(int addr, Size size) {
        Sh2Cache.Sh2CacheLine line;
        int tagaddr = addr & 0x1FFFFC00;
        int entry = (addr & 0x3F0) >> 4;
        for (int i = 0; i < 4; ++i) {
            line = this.ca.way[i][entry];
            if (line.v <= 0 || line.tag != tagaddr) continue;
            Sh2CacheImpl.updateLru(i, this.ca.lru, entry);
            assert (this.cacheRegCtx.twoWay == 0 || this.cacheRegCtx.twoWay == 1 && i > 1);
            return Sh2CacheImpl.getCachedData(line.data, addr & 0xF, size);
        }
        int lruway = Sh2CacheImpl.selectWayToReplace(this.cacheRegCtx.twoWay, this.ca.lru[entry]);
        assert (this.cacheRegCtx.twoWay == 0 || this.cacheRegCtx.twoWay == 1 && lruway > 1);
        line = this.ca.way[lruway][entry];
        this.invalidatePrefetcher(line, entry, addr);
        Sh2CacheImpl.updateLru(lruway, this.ca.lru, entry);
        line.tag = tagaddr;
        this.refillCache(line.data, addr);
        line.v = 1;
        return Sh2CacheImpl.getCachedData(line.data, addr & 0xF, size);
    }

    private boolean writeCache(int addr, int val, Size size) {
        int tagaddr = addr & 0x1FFFFC00;
        int entry = (addr & 0x3F0) >> 4;
        boolean change = false;
        for (int i = 0; i < 4; ++i) {
            Sh2Cache.Sh2CacheLine line = this.ca.way[i][entry];
            if (line.v <= 0 || line.tag != tagaddr) continue;
            assert (this.cacheRegCtx.twoWay == 0 || this.cacheRegCtx.twoWay == 1 && i > 1);
            int prev = Sh2CacheImpl.getCachedData(line.data, addr & 0xF, size);
            if (prev != val) {
                this.setCachedData(line.data, addr & 0xF, val, size);
                change = true;
            }
            Sh2CacheImpl.updateLru(i, this.ca.lru, entry);
            break;
        }
        this.writeMemoryUncached(this.memory, addr, val, size);
        return change;
    }

    private boolean writeDataArray(int addr, int val, Size size) {
        assert (this.cacheRegCtx.cacheEn == 0 || this.cacheRegCtx.twoWay == 1);
        int dataArrayMask = 4095 >> (this.cacheRegCtx.cacheEn & this.cacheRegCtx.twoWay);
        int address = addr & dataArrayMask;
        boolean change = false;
        if (address == (addr & 0xFFF)) {
            change = BufferUtil.writeBufferRaw(this.data_array, address, val, size);
        } else {
            LOG.error("{} Error Cache data array write: {}({}) {}, val: {}", new Object[]{this.cpu, Util.th(addr), Util.th(address), size, Util.th(val)});
        }
        return change;
    }

    private int readDataArray(int addr, Size size) {
        assert (this.cacheRegCtx.cacheEn == 0 || this.cacheRegCtx.twoWay == 1);
        int dataArrayMask = 4095 >> (this.cacheRegCtx.cacheEn & this.cacheRegCtx.twoWay);
        int address = addr & dataArrayMask;
        if (address != (addr & 0xFFF)) {
            LOG.error("{} Error Cache data array read: {}({}) {}, val: {}", new Object[]{this.cpu, Util.th(addr), Util.th(addr & dataArrayMask), size, Util.th(BufferUtil.readBuffer(this.data_array, address, size))});
            return size.getMask();
        }
        BufferUtil.readBuffer(this.data_array, address, size);
        return BufferUtil.readBuffer(this.data_array, address, size);
    }

    private void writeAddressArray(int addr, int data) {
        int tagaddr = addr & 0x1FFFFC00;
        int entry = (addr & 0x3F0) >> 4;
        this.ca.lru[entry] = data >> 6 & 0x3F;
        Sh2Cache.Sh2CacheLine line = this.ca.way[this.cacheRegCtx.way][entry];
        line.v = addr >> 2 & 1;
        line.tag = tagaddr;
    }

    private int readAddressArray(int addr) {
        int entry = (addr & 0x3F0) >> 4;
        int tagaddr = this.ca.way[this.cacheRegCtx.way][entry].tag;
        return tagaddr & 0x1FFFFC00 | this.ca.lru[entry] << 4 | this.cacheRegCtx.cacheEn;
    }

    private void purgeCache(int addr) {
        int tagaddr = addr & 0x1FFFFC00;
        int entry = (addr & 0x3F0) >> 4;
        for (int i = 0; i < 4; ++i) {
            if (this.ca.way[i][entry].tag != tagaddr) continue;
            assert (this.cacheRegCtx.twoWay == 0 || this.cacheRegCtx.twoWay == 1 && i > 1);
            this.ca.way[i][entry].v = 0;
            MdRuntimeData.addCpuDelayExt(2);
            this.invalidatePrefetcher(this.ca.way[i][entry], entry, addr & 0x3FFFFFFF);
        }
        assert (addr < 0x48000000);
    }

    @Override
    public Sh2Cache.CacheRegContext updateState(int value) {
        Sh2Cache.CacheRegContext cacheCtx = this.ctx.cacheContext;
        cacheCtx.way = value >> 6 & 3;
        cacheCtx.cachePurge = value >> 4 & 1;
        cacheCtx.twoWay = value >> 3 & 1;
        cacheCtx.dataReplaceDis = value >> 2 & 1;
        cacheCtx.instReplaceDis = value >> 1 & 1;
        cacheCtx.cacheEn = value & 1;
        this.handleCacheEnabled(cacheCtx);
        if (cacheCtx.cachePurge > 0) {
            this.cacheClear();
            cacheCtx.cachePurge = 0;
            value &= 0xEF;
        }
        cacheCtx.ccr = value;
        return cacheCtx;
    }

    private void handleCacheEnabled(Sh2Cache.CacheRegContext cacheCtx) {
        int prevCaEn = this.ca.enable;
        this.ca.enable = cacheCtx.cacheEn;
        if (prevCaEn != cacheCtx.cacheEn) {
            // empty if block
        }
    }

    @Override
    public ByteBuffer getDataArray() {
        return this.data_array;
    }

    public static Optional<Integer> getCachedValueIfAny(Sh2CacheImpl cache, int addr, Size size) {
        int tagaddr = addr & 0x1FFFFC00;
        int entry = (addr & 0x3F0) >> 4;
        for (int i = 0; i < 4; ++i) {
            Sh2Cache.Sh2CacheLine line = cache.ca.way[i][entry];
            if (line.v <= 0 || line.tag != tagaddr) continue;
            return Optional.of(Sh2CacheImpl.getCachedData(line.data, addr & 0xF, size));
        }
        return Optional.empty();
    }

    private static void updateLru(int way, int[] lruArr, int lruPos) {
        int lru = lruArr[lruPos];
        if (way == 3) {
            lru |= 0xB;
        } else if (way == 2) {
            lru &= 0x3E;
            lru |= 0x14;
        } else if (way == 1) {
            lru |= 0x20;
            lru &= 0x39;
        } else {
            lru &= 7;
        }
        lruArr[lruPos] = lru;
    }

    private static int selectWayToReplace(int twoWay, int lru) {
        if (twoWay > 0) {
            if ((lru & 1) == 1) {
                return 2;
            }
            return 3;
        }
        if ((lru & 0x38) == 56) {
            return 0;
        }
        if ((lru & 0x26) == 6) {
            return 1;
        }
        if ((lru & 0x15) == 1) {
            return 2;
        }
        if ((lru & 0xB) == 0) {
            return 3;
        }
        throw new RuntimeException();
    }

    private void refillCache(byte[] data, int addr) {
        MdRuntimeData.addCpuDelayExt(4);
        assert (this.cpu == MdRuntimeData.getAccessTypeExt());
        for (int i = 0; i < 16; i += 4) {
            int val = this.memory.readMemoryUncachedNoDelay((addr & 0xFFFFFFF0) + i, Size.LONG);
            this.setCachedData(data, i & 0xF, val, Size.LONG);
        }
    }

    private void invalidatePrefetcher(Sh2Cache.Sh2CacheLine line, int entry, int addr) {
        if (line.v > 0) {
            boolean force = addr < 0;
            this.invalidCtx.line = line;
            this.invalidCtx.prevCacheAddr = line.tag | entry << 4;
            boolean invalidate = true;
            int n = this.invalidCtx.cacheReadAddr = force ? this.invalidCtx.prevCacheAddr : addr;
            if (invalidate) {
                this.memory.invalidateCachePrefetch(this.invalidCtx);
            }
        }
    }

    @Override
    public void saveContext(ByteBuffer buffer) {
        Sh2Cache.super.saveContext(buffer);
        this.data_array.rewind().get(this.ctx.dataArray);
        this.ctx.cacheContext = this.cacheRegCtx;
        this.ctx.ca = this.ca;
        buffer.put(Util.serializeObject(this.ctx));
    }

    @Override
    public void loadContext(ByteBuffer buffer) {
        Sh2Cache.super.loadContext(buffer);
        Serializable s = Util.deserializeObject(buffer);
        assert (s instanceof Sh2Cache.Sh2CacheContext);
        this.ctx = (Sh2Cache.Sh2CacheContext)s;
        this.data_array.rewind().put(this.ctx.dataArray);
        this.ca = this.ctx.ca;
        this.cacheRegCtx = this.ctx.cacheContext;
    }

    @Override
    public Sh2Cache.CacheRegContext getCacheContext() {
        return this.cacheRegCtx;
    }

    @Override
    public Sh2Cache.Sh2CacheContext getSh2CacheContext() {
        return this.ctx;
    }

    private void setCachedData(byte[] data, int addr, int val, Size size) {
        Util.writeDataMask(data, addr, val, 15, size);
    }

    private static int getCachedData(byte[] data, int addr, Size size) {
        return Util.readDataMask(data, addr, 15, size);
    }
}

