/*
 * Decompiled with CFR 0.152.
 */
package omegadrive.system;

import java.nio.file.Path;
import java.time.Duration;
import java.util.Optional;
import java.util.concurrent.CyclicBarrier;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;
import omegadrive.Device;
import omegadrive.SystemLoader;
import omegadrive.UserConfigHolder;
import omegadrive.bus.model.BaseBusProvider;
import omegadrive.input.InputProvider;
import omegadrive.input.KeyboardInput;
import omegadrive.joypad.JoypadProvider;
import omegadrive.memory.IMemoryProvider;
import omegadrive.savestate.BaseStateHandler;
import omegadrive.sound.SoundProvider;
import omegadrive.sound.javasound.AbstractSoundManager;
import omegadrive.system.MediaSpecHolder;
import omegadrive.system.SystemProvider;
import omegadrive.system.perf.Telemetry;
import omegadrive.ui.DisplayWindow;
import omegadrive.ui.PrefStore;
import omegadrive.util.FileUtil;
import omegadrive.util.LogHelper;
import omegadrive.util.MdRuntimeData;
import omegadrive.util.RegionDetector;
import omegadrive.util.RomHolder;
import omegadrive.util.Sleeper;
import omegadrive.util.Util;
import omegadrive.util.VideoMode;
import omegadrive.vdp.model.BaseVdpAdapterEventSupport;
import omegadrive.vdp.model.BaseVdpProvider;
import org.slf4j.Logger;

public abstract class BaseSystem<BUS extends BaseBusProvider>
implements SystemProvider,
SystemProvider.NewFrameListener,
SystemProvider.SystemClock {
    private static final Logger LOG = LogHelper.getLogger(BaseSystem.class.getSimpleName());
    static final long MAX_DRIFT_NS = Duration.ofMillis(10L).toNanos();
    private static final long DRIFT_THRESHOLD_NS = Util.MILLI_IN_NS / 10L;
    protected IMemoryProvider memory;
    protected BaseVdpProvider vdp;
    protected JoypadProvider joypad;
    protected SoundProvider sound;
    protected InputProvider inputProvider;
    protected BUS bus;
    protected SystemLoader.SystemType systemType;
    protected MediaSpecHolder mediaSpec = MediaSpecHolder.NO_ROM;
    protected Future<Void> runningRomFuture;
    protected DisplayWindow display;
    protected DisplayWindow.DisplayContext displayContext;
    private final ExecutorService executorService = Executors.newSingleThreadExecutor();
    protected volatile boolean saveStateFlag = false;
    protected volatile BaseStateHandler stateHandler;
    protected MdRuntimeData rt;
    private boolean vdpDumpScreenData = false;
    private volatile boolean pauseFlag = false;
    protected volatile boolean futureDoneFlag = false;
    protected volatile boolean softResetPending = false;
    private boolean soundEnFlag = true;
    protected int cycleCounter = 1;
    protected Telemetry telemetry;
    public static final boolean fullThrottle = Boolean.parseBoolean(System.getProperty("helios.fullSpeed", "false"));
    protected long elapsedWaitNs;
    protected long frameProcessingDelayNs;
    protected long targetNs;
    protected long startNs = 0L;
    private long driftNs = 0L;
    private final CyclicBarrier pauseBarrier = new CyclicBarrier(2);
    final Consumer<String> statsConsumer = st -> {
        this.displayContext.label = Optional.of(st);
    };

    protected abstract void loop();

    @Override
    public void init() {
        MdRuntimeData.releaseInstance();
        this.rt = MdRuntimeData.newInstance(this.systemType, this);
        this.sound = AbstractSoundManager.createSoundProvider(this.systemType);
        this.mediaSpec.region = RegionDetector.selectRegion(this.display, this.mediaSpec.getBootableMedia().mediaInfoProvider);
        this.sound.init(this.mediaSpec.getRegion());
        this.displayContext = new DisplayWindow.DisplayContext();
        this.displayContext.megaCdLedState = Optional.empty();
        this.displayContext.videoMode = VideoMode.PAL_H40_V30;
        this.telemetry = Telemetry.resetClock(this);
        this.display.setRomData(this.mediaSpec);
        LOG.info("Region set as: {}", (Object)this.mediaSpec.region);
    }

    protected abstract void resetCycleCounters(int var1);

    protected abstract void updateVideoMode(boolean var1);

    protected BaseSystem(DisplayWindow emuFrame) {
        this.display = emuFrame;
    }

    @Override
    public void handleSystemEvent(SystemProvider.SystemEvent event, Object parameter) {
        LOG.info("Event: {}, with parameter: {}", (Object)event, parameter);
        switch (event) {
            case NEW_ROM: {
                this.handleNewRom((MediaSpecHolder)parameter);
                break;
            }
            case CLOSE_ROM: {
                this.handleCloseRom();
                break;
            }
            case LOAD_STATE: 
            case QUICK_LOAD: {
                this.handleLoadState((Path)parameter);
                break;
            }
            case SAVE_STATE: 
            case QUICK_SAVE: {
                this.handleSaveState((Path)parameter);
                break;
            }
            case TOGGLE_FULL_SCREEN: {
                this.display.setFullScreen((Boolean)parameter);
                break;
            }
            case TOGGLE_PAUSE: {
                this.handlePause();
                break;
            }
            case SOUND_ENABLED: {
                this.soundEnFlag = (Boolean)parameter;
                Optional.ofNullable(this.sound).ifPresent(s -> s.setEnabled(this.soundEnFlag));
                break;
            }
            case TOGGLE_SOUND_RECORD: {
                this.sound.setRecording(!this.sound.isRecording());
                break;
            }
            case CLOSE_APP: {
                this.handleCloseApp();
                break;
            }
            case CONTROLLER_CHANGE: {
                String[] s2 = parameter.toString().split(":");
                this.inputProvider.setPlayerController(InputProvider.PlayerNumber.valueOf(s2[0]), s2[1]);
                break;
            }
            case SOFT_RESET: {
                this.softResetPending = true;
                break;
            }
            case PAD_SETUP_CHANGE: 
            case FORCE_PAD_TYPE: {
                UserConfigHolder.addUserConfig(event, parameter);
                break;
            }
            default: {
                LOG.warn("Unable to handle event: {}, with parameter: {}", (Object)event, parameter);
            }
        }
    }

    protected void handleSoftReset() {
        if (this.softResetPending) {
            LOG.info("Soft Reset");
        }
        this.softResetPending = false;
    }

    protected void reloadWindowState() {
        this.display.addKeyListener(KeyboardInput.createKeyAdapter(this.systemType, this.joypad));
        this.display.reloadControllers(this.inputProvider.getAvailableControllers());
    }

    public void handleNewRom(MediaSpecHolder romSpec) {
        this.mediaSpec = romSpec;
        this.init();
        this.postInit();
        RomRunnable runnable = new RomRunnable(romSpec);
        PrefStore.addRecentFile(romSpec.toString());
        this.runningRomFuture = this.executorService.submit(Util.wrapRunnableEx(runnable), null);
    }

    private void handleCloseApp() {
        try {
            this.handleCloseRom();
            this.display.close();
            this.sound.close();
            Util.executorService.shutdown();
            Util.executorService.awaitTermination(1L, TimeUnit.SECONDS);
            PrefStore.close();
        }
        catch (Exception e) {
            LOG.error("Error while closing app", (Throwable)e);
        }
    }

    protected BaseStateHandler createStateHandler(Path file, BaseStateHandler.Type type) {
        return BaseStateHandler.createInstance(this.systemType, file, type, this.bus.getAllDevices(Device.class));
    }

    private void handleLoadState(Path file) {
        this.stateHandler = this.createStateHandler(file, BaseStateHandler.Type.LOAD);
        LOG.info("Savestate action detected: {} , using file: {}", (Object)this.stateHandler.getType(), (Object)this.stateHandler.getFileName());
        this.saveStateFlag = true;
    }

    private void handleSaveState(Path file) {
        this.stateHandler = this.createStateHandler(file, BaseStateHandler.Type.SAVE);
        LOG.info("Savestate action detected: {} , using file: {}", (Object)this.stateHandler.getType(), (Object)this.stateHandler.getFileName());
        this.saveStateFlag = true;
    }

    protected void processSaveState() {
        if (this.saveStateFlag) {
            this.stateHandler.processState();
            if (this.stateHandler.getType() == BaseStateHandler.Type.SAVE) {
                this.stateHandler.storeData();
            } else {
                this.sound.getPsg().reset();
            }
            this.stateHandler = BaseStateHandler.EMPTY_STATE;
            this.saveStateFlag = false;
        }
    }

    protected void handleCloseRom() {
        this.handleRomInternal();
    }

    @Override
    public boolean isRomRunning() {
        return this.runningRomFuture != null && !this.runningRomFuture.isDone();
    }

    protected void pauseAndWait() {
        if (!this.pauseFlag) {
            return;
        }
        LOG.info("Pause start: {}", (Object)this.pauseFlag);
        try {
            Util.waitOnBarrier(this.pauseBarrier);
            LOG.info("Pause end: {}", (Object)this.pauseFlag);
        }
        finally {
            this.pauseBarrier.reset();
        }
    }

    protected void getStats(long nowNs, long prevStartNs) {
        long fc = this.telemetry.getFrameCounter();
        Optional<String> statsOpt = this.telemetry.newFrame(nowNs - prevStartNs, this.driftNs);
        statsOpt.ifPresent(st -> {
            if (SystemLoader.showFps) {
                this.statsConsumer.accept((String)st);
            }
            this.displayContext.fps = Optional.of(this.telemetry.getAvgFps(fc));
        });
    }

    protected long syncCycle(long startCycle) {
        long baseRemainingNs;
        long remainingNs;
        long now = System.nanoTime();
        if (fullThrottle) {
            return now;
        }
        long driftDeltaNs = 0L;
        if (Math.abs(this.driftNs) > DRIFT_THRESHOLD_NS) {
            driftDeltaNs = this.driftNs > 0L ? DRIFT_THRESHOLD_NS : -DRIFT_THRESHOLD_NS;
            this.driftNs -= driftDeltaNs;
        }
        if ((remainingNs = (baseRemainingNs = startCycle + this.targetNs + driftDeltaNs) - now) > 0L) {
            Sleeper.parkFuzzy(remainingNs);
            remainingNs = baseRemainingNs - System.nanoTime();
        }
        this.driftNs += remainingNs;
        this.driftNs = Math.min(MAX_DRIFT_NS, this.driftNs);
        this.driftNs = Math.max(-MAX_DRIFT_NS, this.driftNs);
        return System.nanoTime();
    }

    private void handleRomInternal() {
        if (this.pauseFlag) {
            this.handlePause();
        }
        if (this.isRomRunning()) {
            this.futureDoneFlag = true;
            this.runningRomFuture.cancel(true);
            while (this.isRomRunning()) {
                Util.sleep(100L);
            }
            LOG.info("Rom thread cancel: {}", (Object)this.mediaSpec);
            this.display.resetScreen();
            this.sound.reset();
            this.bus.closeRom();
            this.telemetry.reset();
            Optional.ofNullable(this.vdp).ifPresent(Device::reset);
            this.cycleCounter = 1;
            MdRuntimeData.releaseInstance();
        }
    }

    protected void createAndAddVdpEventListener() {
        this.vdp.addVdpEventListener(new BaseVdpAdapterEventSupport.VdpEventListener(){

            @Override
            public void onNewFrame() {
                BaseSystem.this.newFrame();
            }

            @Override
            public int order() {
                return 100;
            }
        });
    }

    @Override
    public void newFrame() {
        long startWaitNs = System.nanoTime();
        long prevStartNs = this.startNs;
        this.elapsedWaitNs = this.syncCycle(this.startNs) - startWaitNs;
        this.startNs = System.nanoTime();
        this.updateVideoMode(false);
        this.getStats(this.startNs, prevStartNs);
        this.doRendering(this.vdp.getScreenDataLinear());
        this.frameProcessingDelayNs = this.startNs - startWaitNs - this.elapsedWaitNs;
        this.handleVdpDumpScreenData();
        this.processSaveState();
        this.pauseAndWait();
        this.resetCycleCounters(this.cycleCounter);
        this.cycleCounter = 0;
        this.futureDoneFlag = this.runningRomFuture.isDone();
        this.handleSoftReset();
        this.inputProvider.handleEvents();
    }

    protected void loadRomDataIfEmpty(MediaSpecHolder mediaSpec, IMemoryProvider memoryProvider) {
        boolean isRomCart;
        assert (memoryProvider != null);
        assert (mediaSpec.systemType == SystemLoader.SystemType.NONE || mediaSpec.systemType == this.getSystemType());
        boolean bl = isRomCart = !mediaSpec.getBootableMedia().type.isDiscImage();
        if (isRomCart && memoryProvider.getRomHolder() == RomHolder.EMPTY_ROM) {
            byte[] d = FileUtil.readBinaryFile(mediaSpec.getBootableMedia().romFile, this.getSystemType());
            if (d.length == 0) {
                LOG.error("Unable to open/access file: {}", (Object)mediaSpec);
            }
            memoryProvider.setRomData(d);
        }
    }

    protected void handleVdpDumpScreenData() {
        if (this.vdpDumpScreenData) {
            this.vdp.dumpScreenData();
            this.vdpDumpScreenData = false;
        }
    }

    protected void doRendering(int[] data) {
        this.displayContext.data = data;
        this.display.renderScreenLinear(this.displayContext);
    }

    private void handlePause() {
        boolean wasPausing = this.pauseFlag;
        this.pauseFlag = !wasPausing;
        this.sound.setEnabled(wasPausing);
        LOG.info("Pausing: {}, soundEn: {}", (Object)this.pauseFlag, (Object)wasPausing);
        if (wasPausing) {
            Util.waitOnBarrier(this.pauseBarrier);
        }
    }

    @Override
    public void reset() {
        this.handleCloseRom();
        this.handleNewRom(this.mediaSpec);
    }

    protected void postInit() {
        assert (this.vdp != null && this.joypad != null && this.bus != null && this.memory != null);
        this.loadRomDataIfEmpty(this.mediaSpec, this.memory);
        this.vdp.setRegion(this.mediaSpec.getRegion());
        this.joypad.init();
        this.vdp.init();
        this.bus.init();
        this.futureDoneFlag = false;
        this.sound.setEnabled(this.soundEnFlag);
    }

    @Override
    public final SystemLoader.SystemType getSystemType() {
        return this.systemType;
    }

    @Override
    public MediaSpecHolder getMediaSpec() {
        assert (this.mediaSpec != null);
        return this.mediaSpec;
    }

    @Override
    public int getCycleCounter() {
        return this.cycleCounter;
    }

    @Override
    public long getFrameCounter() {
        return this.telemetry.getFrameCounter();
    }

    class RomRunnable
    implements Runnable {
        private final MediaSpecHolder romSpec;
        private static final String threadNamePrefix = "cycle-";

        public RomRunnable(MediaSpecHolder romSpec) {
            assert (romSpec != MediaSpecHolder.NO_ROM);
            this.romSpec = romSpec;
        }

        @Override
        public void run() {
            try {
                if (BaseSystem.this.memory.getRomData().length == 0) {
                    return;
                }
                String romName = FileUtil.getFileName(this.romSpec.getBootableMedia().romFile);
                LOG.info("Running rom: {},\n{}", (Object)romName, (Object)this.romSpec);
                Thread.currentThread().setName(threadNamePrefix + romName);
                Thread.currentThread().setPriority(6);
                LOG.info("Starting game loop");
                BaseSystem.this.loop();
                LOG.info("Exiting rom thread loop");
            }
            catch (Error | Exception e) {
                e.printStackTrace();
                LOG.error("Error", e);
            }
            BaseSystem.this.handleCloseRom();
        }
    }
}

