/*
 * Decompiled with CFR 0.152.
 */
package omegadrive.util;

import com.google.common.base.Strings;
import java.util.Arrays;
import java.util.Comparator;
import java.util.EnumSet;
import java.util.Objects;
import java.util.Optional;
import mcd.cart.MegaCdCartInfoProvider;
import omegadrive.cart.MdCartInfoProvider;
import omegadrive.cart.MediaInfoProvider;
import omegadrive.util.LogHelper;
import org.slf4j.Logger;

public class RegionDetector {
    private static final Logger LOG = LogHelper.getLogger(RegionDetector.class.getSimpleName());
    private static final Comparator<Region> REGION_COMPARATOR = Comparator.comparingInt(r -> r.order);
    public static final int PAL_FPS = 50;
    public static final int NTSC_FPS = 60;

    public static Region detectHeaderRegion(MdCartInfoProvider cartInfoProvider, boolean verbose) {
        String s = cartInfoProvider.getRegion();
        s = Strings.padEnd((String)s, (int)3, (char)' ');
        assert (s.length() == 3);
        Region[] regions = new Region[]{Region.getRegion(s.charAt(0)), Region.getRegion(s.charAt(1)), Region.getRegion(s.charAt(2))};
        Optional<Region> optRegion = Arrays.stream(regions).filter(Objects::nonNull).min(REGION_COMPARATOR);
        Region res = optRegion.orElse(RegionDetector.detectRegionFallBack(s).orElse(null));
        if (res == null) {
            LOG.warn("Unable to find a region, defaulting to USA");
            res = Region.USA;
        }
        if (verbose) {
            LOG.info("{} ({})", (Object)res.name(), (Object)s);
        }
        return res;
    }

    public static Region detectHeaderRegion(MdCartInfoProvider cartInfo) {
        return RegionDetector.detectHeaderRegion(cartInfo, false);
    }

    public static Region selectRegion(RegionOverrideSupplier w, MediaInfoProvider cip) {
        assert (cip != MdCartInfoProvider.NO_PROVIDER);
        Region regionFileName = RegionDetector.getRegionFileName(cip.getRomName());
        Region regionOvrUi = RegionDetector.getRegion(w.getRegionOverride());
        Region regionOvrConfig = null;
        Region securityCodeRegion = null;
        Region romHeaderRegion = RegionDetector.getRegion(cip.getRegion());
        if (cip instanceof MdCartInfoProvider) {
            MdCartInfoProvider mcip = (MdCartInfoProvider)cip;
            regionOvrConfig = RegionDetector.getRegion(Optional.ofNullable(mcip.getEntry().forceRegion).orElse(null));
            romHeaderRegion = RegionDetector.detectHeaderRegion(mcip, false);
        }
        if (cip instanceof MegaCdCartInfoProvider) {
            MegaCdCartInfoProvider mcdip = (MegaCdCartInfoProvider)cip;
            securityCodeRegion = mcdip.securityCodeRegion;
        }
        if (regionOvrUi != null && regionOvrUi != romHeaderRegion) {
            LOG.info("Setting region override from UI: {} to {}", (Object)romHeaderRegion, (Object)regionOvrUi);
            return regionOvrUi;
        }
        if (regionOvrConfig != null && regionOvrConfig != romHeaderRegion) {
            LOG.info("Setting region override from Config: {} to {}", (Object)romHeaderRegion, (Object)regionOvrConfig);
            return regionOvrConfig;
        }
        if (securityCodeRegion != null && securityCodeRegion != romHeaderRegion) {
            LOG.info("Setting region from securityCode: {} to {}", (Object)romHeaderRegion, (Object)securityCodeRegion);
            return securityCodeRegion;
        }
        if (regionFileName != null && regionFileName != romHeaderRegion) {
            LOG.info("Setting region from fileName: {} to {}", (Object)romHeaderRegion, (Object)regionFileName);
            return regionFileName;
        }
        LOG.info("Region: {}", (Object)romHeaderRegion);
        return romHeaderRegion;
    }

    public static Region getRegionFileName(String fileName) {
        boolean eu = fileName.toUpperCase().contains("(EUROPE)");
        boolean us = fileName.toUpperCase().contains("(US)");
        boolean jp = fileName.toUpperCase().contains("(JAPAN)");
        Region r = eu ? Region.EUROPE : null;
        r = jp ? Region.JAPAN : r;
        r = us ? Region.USA : r;
        return r;
    }

    private static Optional<Region> detectRegionFallBack(String regionStr) {
        int val = Character.getNumericValue(regionStr.charAt(0));
        Region region = null;
        if (val < 16) {
            Region region2 = region = (val & 4) > 0 ? Region.USA : region;
            Region region3 = region == null ? ((val & 1) > 0 ? Region.JAPAN : region) : (region = region);
            region = region == null ? ((val & 8) > 0 ? Region.EUROPE : region) : region;
        }
        return Optional.ofNullable(region);
    }

    public static Region getRegion(String regionName) {
        if (Objects.isNull(regionName) || regionName.length() < 1) {
            return null;
        }
        return Region.getRegion(regionName.charAt(0));
    }

    public static enum Region {
        JAPAN('J', 2, 0, 60),
        USA('U', 0, 128, 60),
        EUROPE('E', 1, 192, 50);

        private static final EnumSet<Region> values;
        private final char region;
        private final int versionCode;
        private final int fps;
        private final int order;
        private final double frameIntervalMs;

        private Region(char region, int order, int versionCode, int fps) {
            this.region = region;
            this.versionCode = versionCode;
            this.fps = fps;
            this.order = order;
            this.frameIntervalMs = 1000.0 / (double)fps;
        }

        public static Region getRegion(char region) {
            Region res = null;
            for (Region r : values) {
                res = r.region == region ? r : res;
            }
            return res;
        }

        public int getFps() {
            return this.fps;
        }

        public double getFrameIntervalMs() {
            return this.frameIntervalMs;
        }

        public int getVersionCode() {
            return this.versionCode;
        }

        static {
            values = EnumSet.allOf(Region.class);
        }
    }

    public static interface RegionOverrideSupplier {
        public String getRegionOverride();
    }
}

