from sys import argv, exit
from json import load
from subsetsum import solutions
special_char = {
    0x24: '$',
    0x25: '⚱',
    0x26: '♕',
    0x27: '🧍',
    0x28: '♥',
    0x29: '⛏',
    0x2a: '🧠',
    0x2b: '×',
    0x2c: '!',
    0x30: '・',
    0x31: '.',
    0x32: '-',
    0x33: '↵',
    0x34: '↑',
    0x35: '↓',
    0x36: '←',
    0x37: '→',
    0x38: '␛',
    0x39: '⇧',
    0x3a: '␠',
    0x3b: '↖',
    0x3c: '↗',
    0x3d: '↘',
    0x3e: '↙',
    0x40: '┌',
    0x41: '┐',
    0x42: '└',
    0x43: '┘',
    0x44: '─',
    0x45: '│',
    0x46: '┼',
    0x47: '├',
    0x48: '┤',
    0x49: '┬',
    0x4a: '┴',
    0x4b: '[',
    0x4c: ']',
    0x9f: ' ',
    0x3f: ' '
}
back_char = {v: k for k, v in special_char.items()}

formats = {
    "Dragon Slayer, Level 1.1": {
        "chain": [b"\xcd\x2c\x96"],
        "posi": [b"\xcd\x09\x9a"],
        "arg": [b"\x11"],
        "program": range(12,14),
        "data": [3],
        "offset": 0x0,
        "sectors": (0, 5),
        "inside": [b"\xed\x53\xdf\xb9"]
    },
    "Dragon Slayer, Level 2.0": {
        "chain": [b"\xcd\x0b\x13", b"\xc3\x0b\x13"],
        "posi": None,
        "arg": [b"\xdd\x21",b"\xfd\x21"],
        "program": range(3, 7),
        "data": [8, 11],
        "offset": 0x5000,
        "sectors": (1, 5),
        "inside": None
    },
    "Dragon Slayer, LOGiN version": {
        "chain": [b"\xcd\x79\x0d"],
        "posi": None,
        "arg": [b"\xdd\x21",b"\xfd\x21", b"\x32", b"\x3a"],
        "program": range(3, 4),
        "data": [5,6],
        "offset": 0x2000,
        "sectors": (0, 5),
        "inside": [b"\xcd\x21\x0e"]
    }
}

def to_bytes(text: str):
    b_chain = b''
    while len(text) > 0:
        char = text[0]
        if len(text) > 7 and text[:7] == '<color:':
            pos = text.index('>')
            char = b'\xfe' + int(text[7:pos]).to_bytes(1,'little')
            text = text[pos:]
        elif char in back_char.keys():
            char = back_char[char].to_bytes(1,'little')
        elif char.isnumeric() or char.isalpha():
            char = int(char, 36).to_bytes(1,'little')
        else:
            print("error")
        if type(char) is str:
            print(text)
        b_chain += char
        text = text[1:]
    b_chain += b'\xff'
    return b_chain

def chain_to_bytes(chain: list):
    b_chain = b''
    for text in chain:
        b_chain += int(text[0]).to_bytes(2,'little')
        b_chain += to_bytes(text[1])
    return b_chain + b'\xff\xff'

def read_string(text: str):
    if b'\xff' in text:
        text = text[:text.index(b'\xff')]
    chain = ''
    in_tag = False
    for b in text:
        if b == 0xfe:
            b = '<color:'
            in_tag = True
        elif b < 10 or in_tag:
            b = str(b)
            if in_tag:
                b += '>'
                in_tag = False
        elif b < 36: 
            b = chr(b+0x37)
        elif b in special_char.keys():
            b = special_char[b]
        else:
            return None
        chain += b
    return chain
    
def read_chain(start: int, binary: bytes):
    strings = binary[start:]
    end = strings.index(b'\xff\xff')
    strings = strings[:end]
    length = [start, start + len(strings) + 2]
    strings = strings.split(b'\xff')
    decoded = []
    for text in strings:
        if text == b'':
            continue
        coords = int.from_bytes(text[:2], 'little')
        decoded.append([coords, read_string(text[2:])])
        if not decoded[-1][1]:
            return None, -1
    return decoded, length
    
def read_movable(start: int, binary: bytes):
    text = binary[start:]
    end = text.index(b'\xff')
    text = text[:end]
    length = [start, start + len(text) + 1]
    return read_string(text), length

def gettrackoffsets(filef):
    adresslist = []
    for i in range(0x20,0x160,4):
        filef.seek(i)
        adress = filef.read(4)
        adresslist.append(int.from_bytes(adress,'little'))
    return adresslist

def parse_header(header: bytes):
    cylinder = header[0]
    side = header[1]
    sector = header[2]
    size = 128 << header[3]
    return cylinder, side, sector, size

def detrack(filename: str):
    with open(filename,'rb') as filef:
        adresslist = gettrackoffsets(filef)
        tracklist = [None]*len(adresslist)
        for adress in adresslist:
            current = None
            filef.seek(adress)
            while True:
                header = filef.read(0x10)
                if header == b'':
                    break
                c, s, sct, size = parse_header(header)
                if current is None:
                    current = 2*c + s
                    if current > len(tracklist):
                        print("Invalid track:",hex(adress))
                        break
                    tracklist[current] = []
                if current != 2*c + s:
                    #print(current, len(tracklist[current]), hex(size), end=" / ")
                    break
                tracklist[current].append(filef.read(size))
    return tracklist

def retrack(track_content: list, data: bytes, form: dict, part: str):
    start, end = form["sectors"]
    for track in form[part]:
        for sector in range(start, end):
            dim = len(track_content[track][sector])
            track_content[track][sector] = data[:dim]
            data = data[dim:]
    if len(data) != 0:
        print("Erreur: données surdimensionnées")
    return track_content

def rewrite(orig_file: str, filename: str, track_content: list):
    with open(orig_file, 'rb') as original:
        adresslist = gettrackoffsets(original)
        if adresslist != sorted(adresslist):
            print("Erreur: pistes désordonnées")
            exit()
        if len(adresslist) != len(track_content):
            print("Erreur: non correspondance")
            exit()
        original.seek(0)
        step = 0
        with open(filename, 'wb') as new:
            for i in range(len(track_content)):
                if not track_content[i]:
                    continue
                new.write(original.read(adresslist[i] - step))
                step = adresslist[i]
                for sector in track_content[i]:
                    new.write(original.read(0x10))
                    new.write(sector)
                    step += 0x10 + len(sector)
                    original.seek(step)
    
def parse_content(track_content: list, form: dict, part: str):
    tracks = list(track_content[t] for t in form[part])
    sectors = []
    start, end = form["sectors"]
    for track in tracks:
        sectors += track[start:end]
    data = b''.join(sectors)
    return data
    
def check_format(form: dict, track_content: list):
    code = parse_content(track_content, form, "program")
    if form["chain"][0] in code:
        return code
    else:
        return None

def get_args(cut: bytes, args: list):
    are_in = []
    for arg in args:
        if arg in cut:
            are_in.append(cut.rfind(arg) - 0x20 + len(arg))
    return are_in

def get_txt(code: bytes, data: bytes, read_fun, form: dict, signatures: list):
    txt = []
    freedom = []
    references = []
    for signature in form[signatures]:
        step = 0
        while (offset:= code.find(signature, step)) != -1:
            args = get_args(code[offset-0x20:offset], form["arg"])
            for arg in args:
                ref = offset + arg
                adress = int.from_bytes(code[ref:ref + 2], 'little') - form["offset"]
                if adress < 0 or len(data) < adress:
                    continue
                temp = read_fun(adress, data)
                if not temp or not temp[0]:
                    continue
                if type(temp) is str:
                    temp = (temp, (adress, adress + len(temp)))
                if not temp[0] in txt:
                    txt.append(temp[0])
                    freedom.append(temp[1])
                    references.append([ref])
                else:
                    references[txt.index(temp[0])].append(ref)
            step = offset + 1
    return txt, freedom, references

def read_json(name: str):
    with open(name, 'r', encoding = "UTF-8") as f:
        return load(f)

def txt_to_bytes(txt: list):
    orig = read_json("dragon_slayer_orig.json")
    trad = read_json("dragon_slayer_fr.json")
    check = False
    for i in range(len(txt)):
        if txt[i] in orig:
            txt[i] = trad[orig.index(txt[i])]
            check = True
        if type(txt[i]) == list:
            txt[i] = chain_to_bytes(txt[i])
        else:
            txt[i] = to_bytes(txt[i])
    if not check:
        print("Erreur: Textes non reconnus")
        exit()
    return txt

def modify_code(code: bytes, ref: int, step: int):
    return code[:ref] + (step).to_bytes(2, 'little') + code[ref+2:]

def modify_data(code: bytes, data: bytes, cur: bytes, step: int, space_end: int, reference: list, offset: int):
    plus = step + len(cur)
    if plus > space_end:
        return code, data, step
    data = data[:step] + cur + data[plus:]
    for ref in reference:
        code = modify_code(code, ref, step + offset)
    return code, data, plus
            
def change(code: bytes, data: dict, form: dict):
    txt, freedom, references = get_txt(code, data, read_chain, form, "chain")
    
    if form["posi"]:
        a, b, c = get_txt(code, data, read_movable, form, "posi")
        txt += a
        freedom += b
        references += c
        
    if form["inside"]:
        inside_ids = []
        _, inside_space, inside_refs = get_txt(code, data, read_movable, form, "inside")
        for i in range(len(inside_space)):
            for j in range(len(freedom)):
                if inside_space[i][0] > freedom[j][0] and inside_space[i][0] < freedom[j][1]:
                    inside_ids.append(j)
                    inside_space[i] = inside_space[i][0] - freedom[j][0]
                    break
        
    freedom.sort()
    for i in range(1, len(freedom)):
        while i < len(freedom) and (freedom[i][0] - freedom[i-1][1] < 4):
            freedom[i-1] = (freedom[i-1][0], freedom.pop(i)[1])
            
    txt = txt_to_bytes(txt)
            
    lengths = [len(part) for part in txt]
    for space in freedom:
        step = space[0]
        target = space[1] - step
        result = []
        while (result == [] or 0) and target != 0:
            result = list(solutions(lengths, target))
            target -= 1
        for i in result[0][::-1]:
            cur, cur_step = txt.pop(i), step
            code, data, step = modify_data(code, data, cur, step, space[1], references.pop(i), form["offset"])
            if form["inside"]:
              if (i in inside_ids):
                cur_inside = inside_ids.index(i)
                inside_ids.pop(cur_inside)
                cur_space = inside_space.pop(cur_inside)
                for ref in inside_refs.pop(cur_inside):
                    code = modify_code(code, ref, cur_step + form["offset"] + cur_space)
              inside_ids = list(map(lambda present : present - (present > i), inside_ids))
            lengths.pop(i)
    
    if len(txt) != 0:
        print("Erreur: pas assez d'espace libre pour inclure les traductions")
        exit()
    return code, data
    
if __name__ == "__main__":
    if len(argv) < 2 or len((filename:= argv[1])) < 4:
        print("Fichier introuvable")
        exit()
    track_content = detrack(filename)
    for form in formats.items():
        if (code:= check_format(form[1], track_content)):
            print(form[0])
            break
    if code:
        title = form[0]
        form = formats[title]
        data = parse_content(track_content, form, "data")
        code, data = change(code, data, form)
        track_content = retrack(track_content, data, form, "data")
        track_content = retrack(track_content, code, form, "program")
        rewrite(filename, filename[:-4] + " [FR].d88", track_content)
    else:
        print("Format non reconnu")