////////////////////////////////////////////////////////////////////////
//
// Attract-Mode Frontend - Emulator detection script
//
// Copyright (c) 2017 Andrew Mickelson
//
// This program comes with ABSOLUTELY NO WARRANTY.  It is licensed under
// the terms of the GNU General Public License, version 3 or later.
//
////////////////////////////////////////////////////////////////////////

fe.load_module( "file" );



function console_report( name, found )
{
	console_message( "Executable", found.len() ? found : "-" )
}

function console_message( name = "", message = "" )
{
	if ( !name.len() && !message.len() ) return fe.log("");
	fe.log( " - " + format( "%-15s", name + " " ) + message );
}


////////////////////////////////////////////////////////////////////////
//
// This is the array of emulators we have generated that we think are
// actually in use by the user.  The plan is for Attract-Mode to read
// this array after running this script, and prompt the user for some
// automatic setup (when teh frontend is first run)
//
////////////////////////////////////////////////////////////////////////
emulators_to_generate <- [];

// Replace path slashes with OS slashes
function format_path( path ) {
	local windows = ( OS == "Windows" );
	local s = windows ? '/' : '\\';
	local r = windows ? '\\' : '/';
	local out = "";
	foreach ( c in path ) out += (c == s ? r : c).tochar();
	return out;
}

// Append value with separator
function format_sep( value ) {
	if ( !value.len() ) return value;
	if ( value[0] == ';' ) value = value.slice( 1 );
	if ( value[value.len()-1] == ';' ) value = value.slice( 0, value.len() - 1 );
	return value;
}

////////////////////////////////////////////////////////////////////////
//
// Write contents of "emu" to specified "filename".  Don't overwrite
// existing files unless force=true.
//
// "emu" is a table in the form of one of the entries in db.nut
//
// Returns true if file written, false otherwise
//
////////////////////////////////////////////////////////////////////////
function write_config( emu, filename, force=false )
{
	console_message( "Template", filename + (force ? " (Overwrite)" : "") );

	if ( !force && fe.path_test( filename, PathTest.IsFile ) )
		return false;

	local new_cfg = WriteTextFile( filename );
	local fmt = "%-24s";

	new_cfg.write_line( "# Generated by Attract-Mode Plus " + FeVersion + "\n" );
	new_cfg.write_line( "#\n" );
	new_cfg.write_line( format( fmt, "executable" ) + emu["exe"] + "\n" );

	if ( emu.rawin( "args" ) )
		new_cfg.write_line( format( fmt, "args" ) + emu["args"] + "\n" );

	if ( emu.rawin( "workdir" ) )
		new_cfg.write_line( format( fmt, "workdir" ) + format_path( emu["workdir"] ) + "\n" );

	if ( emu.rawin( "rompath" ) && emu["rompath"].len() )
		new_cfg.write_line( format( fmt, "rompath" ) + format_sep( format_path( emu["rompath"] ) ) + "\n" );

	if ( emu.rawin( "exts" ) )
		new_cfg.write_line( format( fmt, "romext" ) + format_sep( emu["exts"] ) + "\n" );

	local is_arcade = false;
	if ( emu.rawin( "system" ) )
	{
		new_cfg.write_line( format( fmt, "system" ) + format_sep( emu["system"] ) + "\n" );
		is_arcade = ( emu["system"] == "Arcade" )
	}

	if ( emu.rawin( "source" ) )
		new_cfg.write_line( format( fmt, "info_source" ) + emu["source"] + "\n" );

	if ( emu.rawin( "import_extras" ) )
		new_cfg.write_line( format( fmt, "import_extras" ) + format_sep( format_path( emu["import_extras"] ) ) + "\n" );

	if ( emu.rawin( "min_run" ) )
		new_cfg.write_line( format( fmt, "minimum_run_time" ) + emu["min_run"] + "\n" );

	if ( emu.rawin( "hotkey" ) )
		new_cfg.write_line( format( fmt, "exit_hotkey" ) + emu["hotkey"] + "\n" );

	// The resources to look for
	local homedirs = [ "$HOME/" + emu["name"] + "/" ];
	if ( emu.rawin( "workdir" ) )
		homedirs.push( emu["workdir"] );

	local resources = {
		snap = ["snap", "video"],
		screenshot = ["screenshot"],
		marquee = ["marquee"],
		banner = ["banner"],
		flyer = ["flyer"],
		boxart = ["boxart"],
		wheel = ["wheel"],
		clearlogo = ["clearlogo"],
		fanart = ["fanart"]
	};

	// Add each resource if the dir exists
	local found = false;
	foreach ( name, paths in resources ) {
		local respath = "";
		foreach (path in paths) {
			foreach (homedir in homedirs) {
				local dir = fe.path_expand( homedir + path + "/" );
				if ( fe.path_test( dir, PathTest.IsDirectory )) {
					respath += dir + ";";
				}
			}
		}
		if (respath.len()) {
			found = true;
			new_cfg.write_line( format( fmt, "artwork " + name ) + format_sep( format_path( respath ) ) + "\n" );
		}
	}

	// Write some defaults if no resources found
	if ( !found ) {
		if ( is_arcade )
			new_cfg.write_line( format( fmt, "artwork marquee" ) + format_path( homedirs[0] + "marquee/" ) + "\n" );
		else
			new_cfg.write_line( format( fmt, "artwork flyer" ) + format_path( homedirs[0] + "boxart/" ) + "\n" );

		new_cfg.write_line( format( fmt, "artwork snap") + format_path( homedirs[0] + "video/" ) + ";" + format_path( homedirs[0] + "snap/" ) + "\n" );
	}

	return true;
}

////////////////////////////////////////////////////////////////////////
//
// Search for the first file with a name in the "names" array, within
// the specified paths in the "paths" array, returning the first match
//
// return value:  an array [ "path", "name" ] if file found, null otherwise
//
////////////////////////////////////////////////////////////////////////
function search_for_file( paths, names )
{
	foreach ( p in paths )
	{
		local ep = fe.path_expand( p );
		foreach ( e in names )
		{
			local temp = ep + e;
			if ( fe.path_test( temp, PathTest.IsFile ) )
				return [ ep, e ];
		}
	}

	return null;
}

////////////////////////////////////////////////////////////////////////
// return true if the specified paths (array) doesn't contain any files with
// the specified extension(s) (array)
////////////////////////////////////////////////////////////////////////
function path_is_empty( paths, exts )
{
	if (( paths == null ) || ( exts == null ))
		return true;

	foreach ( p in paths )
	{
		local dl = DirectoryListing( fe.path_expand( p ) );

		foreach ( f in dl.results )
		{
			foreach ( e in exts )
			{
				local tlen = f.len() - e.len();
				if ( tlen < 0 )
					continue;

				if ( f.slice( tlen ) == e )
					return false;
			}
		}
	}
	return true;
}

////////////////////////////////////////////////////////////////////////
//
// Routine for generating emulator config and template from
// a db.nut entry
//
// General idea:
//   1. Try to find the specified emulator
//   2. If found, create a config for it in "<config_dir>/emulators/"
//   3. Separately generate a config template in "<config_dir>/templates/emulators"
//      This template will be available to the user when they create a new emulator
//      in the frontend.  The idea is to provide sensible default settings.
//
////////////////////////////////////////////////////////////////////////
function generate_emulator_config( emu )
{
	console_message();
	console_message( "Generate", emu["name"] );

	local my_OS = OS;
	if ( OS != "Windows" )
		my_OS = "Linux";

	// Nothing to do if this is an emulator for another OS
	//
	if ( emu.rawin( "OS" ) && ( my_OS != emu["OS"] ) )
		return;

	// Put all our OS-specific values in the base emu table now (if any)
	//
	if ( emu.rawin( my_OS ) )
	{
		foreach ( i,v in emu[ my_OS ] )
			emu[i] <- v;
	}

	// If a custom setup script is specified, use it instead
	// of our generic routine
	//
	if ( emu.rawin( "custom" ) )
		return dofile( fe.script_dir + emu["custom"], true );

	////////////////////////////////////////////////////////////////
	//
	// Generic routine for generating emulator config:
	//
	////////////////////////////////////////////////////////////////

	// Default paths to check for the executable
	//
	local default_search_paths = {

		"Linux" : [
			"/usr/games",
			"/usr/bin",
			"/usr/local/games",
			"/usr/local/bin",
			"$HOME/" + emu["name"],
			"./" + emu["name"],
			"../" + emu["name"],
			],

		"Windows" : [
			"$HOME/" + emu["name"] + "/",
			"./" + emu["name"] + "/",
			"../" + emu["name"] + "/",
			],
	};

	local search_paths;

	// Override search_paths if emu["path"] already provided
	//
	if ( emu.rawin( "path" ) )
	{
		search_paths = [];

		if ( type( emu[ "path" ] ) == "array" )
			foreach ( p in emu[ "path" ] )
				search_paths.push( p );
		else
			search_paths.push( emu["path"] );
	}
	else
		search_paths = default_search_paths[ my_OS ];


	local search_names = [];

	// Override search_names if emu["exe"] already provided
	//
	if ( emu.rawin( "exe" ) )
	{
		if ( type( emu[ "exe" ] ) == "array" )
			foreach ( p in emu[ "exe" ] )
				search_names.push( p );
		else
			search_names.push( emu["exe"] );
	}
	else
		search_names = [ emu["name"] ];

	if ( my_OS == "Windows" )
	{
		for ( local i=0; i<search_names.len(); i++ )
			search_names[i] += ".exe";
	}

	local res = search_for_file( search_paths, search_names );

	// Fill in emu rompath with sensible default value (if not provided)
	//
	if ( !emu.rawin( "rompath" ) )
		emu["rompath"] <- "$HOME/" + emu["name"] + "/roms/";

	if ( res == null )
	{
		// Did not find emulator executable
		// Set emu["exe"] to a string so there is something there for the template
		//
		emu["exe"] <- search_paths[0] + search_names[0];
		emu["workdir"] <- search_paths[0];
	}
	else
	{
		// Found emulator, write config to emulators/
		//
		emu["exe"] <- res[0] + res[1];
		emu["workdir"] <- res[0];

		//
		// Add emulator to our list of emulators
		//
		emulators_to_generate.push( emu["name"] );
	}

	// Write template to templates/emulators/
	//
	console_report( emu["name"], ( res ) ? emu["exe"] : "" );
	write_config( emu, FeConfigDirectory + "templates/emulators/" + emu["name"] + ".cfg", true );
}

////////////////////////////////////////////////////////////////////////
//
// Load db.nut and generate emulator configs and templates based on its
// contents...
//
////////////////////////////////////////////////////////////////////////
emus <- dofile( fe.script_dir + "db.nut", true );

foreach ( e in emus )
	generate_emulator_config( e );

console_message()