/*
    System 16 Sound Player 
    by Charles Mac Donald
    E-mail: cgfm2@hooked.net
    WWW: http://www.emucamp.com/cgfm2/
*/

#include "main.h"

#define SAMPLE_RATE (44100)             /* Sample rate */
#define SAMPLE_BITS (16)                /* Bits per sample */

int sample_rate = SAMPLE_RATE;          /* To keep sound.c happy */
int buffer_size = (SAMPLE_RATE / 60);   /* Sound buffer size */
static SAMP *sound_buffer[2];           /* Sound buffers for the L and R channel */
static unsigned char command = 0x90;    /* Initial sound command value */
FILE *error_log = NULL;                 /* Error log for debug messages */

char rom[0x8000];                       /* 32k ROM */
char ram[0x800];                        /* 2K RAM */

int main (int argc, char **argv)
{
    FILE *handle = NULL;

    /* Make sure a file was specified */
    if(argc < 2)
    {
        printf("System 16 Sound Player\n");
        printf("Usage: %s <file.bin>\n", argv[0]);
        exit(1);
    }

    /* Try to open file */
    handle = fopen(argv[1],"rb");
    if(!handle)
    {
        printf("File '%s' not found.\n", argv[1]);
        exit(1);
    }

    /* Read in contents */
    fread(rom, 0x8000, 1, handle);
    fclose(handle);

    /* Open debug log */
    error_log = fopen("error.log","w");

    /* Set up RAZE */
    z80_init_memmap();
    z80_map_fetch (0x0000, 0x7FFF, rom);
    z80_map_read  (0x0000, 0x7FFF, rom);

    z80_map_fetch (0xF800, 0xFFFF, ram);
    z80_map_read  (0xF800, 0xFFFF, ram);
    z80_map_write (0xF800, 0xFFFF, ram);

    z80_add_write (0x8000, 0xF7FF, Z80_MAP_HANDLED, &write_handler);
    z80_add_read  (0x8000, 0xF7FF, Z80_MAP_HANDLED, &read_handler);

    z80_set_in    (port_in_handler);
    z80_set_out   (port_out_handler);
    z80_end_memmap();
    z80_reset();

    allegro_init();

    /* Set up sound */
    msdos_init_sound();
    sound_buffer[0] = malloc(buffer_size * 2);
    sound_buffer[1] = malloc(buffer_size * 2);
    YM2151Init(1, (double)3579580, SAMPLE_RATE, SAMPLE_BITS);

    install_keyboard();
    printf("Press SPACE to send commands, Z to dec cmd., X to inc cmd.\n");

    /* Loop until ESC is presssed */
    while(!key[KEY_ESC])
    {
        /* Emulate Z80 for one frame (228 cycles per line, 262 lines per frame) */
        z80_emulate(228 * 262);

        /* Update and play sound buffers */
        YM2151UpdateOne(0, sound_buffer, buffer_size);
        osd_play_streamed_sample_16(0, (signed short *)sound_buffer[0], buffer_size * 2, SAMPLE_RATE, 50, -100);
        osd_play_streamed_sample_16(1, (signed short *)sound_buffer[1], buffer_size * 2, SAMPLE_RATE, 50,  100);

        /* Check user input */
        if(check_key(KEY_Z))
        {
            command = (command - 1) & 0xFF;
            printf("Command %02X\r", command);
        }

        if(check_key(KEY_X))
        {
            command = (command + 1) & 0xFF;
            printf("Command %02X\r", command);
        }

        if(check_key(KEY_SPACE))
        {
            error("*** Sent command %02X\n", command);
            z80_raise_IRQ(0xFF);
            z80_emulate(32);
            z80_lower_IRQ();
        }
    }

    /* Shut down sound emulation */
    YM2151Shutdown();
    msdos_shutdown_sound();

    /* Print state of Z80 and exit */
    error("PC:%04X SP:%04X\n", z80_get_reg(Z80_REG_PC), z80_get_reg(Z80_REG_SP));
    if(error_log) fclose(error_log);
    return (0);
}


/****************************************************************************/

/* Memory access handlers */
void write_handler(UWORD address, UBYTE data)
{
    error("Write %02X to %04X (@%04X)\n", data, address, z80_get_reg(Z80_REG_PC));
}


UBYTE read_handler(UWORD address)
{
    error("Read %04X (@%04X)\n", address, z80_get_reg(Z80_REG_PC));
}


/* Port access handlers */
void port_out_handler(UWORD port, UBYTE data)
{
    static int r;
    static int d;
    port &= 0xFF;

    /* YM2151 register */
    if(port == 0x00)
    {
        r = data;
        return;
    }

    /* YM2151 data */
    if(port == 0x01)
    {
        d = data;
        YM2151WriteReg(0, r, d);
        return;
    }

    error("IO write %02X to %02X (@%04X)\n", data, port & 0xFF, z80_get_reg(Z80_REG_PC));
}


UBYTE port_in_handler(UWORD port)
{
    port &= 0xFF;

    /* YM2151 status */
    if(port == 0x01)
    {
        return (YM2151ReadStatus(0));
    }

    /* Sound latch - retrives sound command byte */
    if(port == 0xC0)
    {
        return (command);
    }

    error("IO read %02X (@%04X)\n", port, z80_get_reg(Z80_REG_PC));
    return (0x00);
}


void error(char *format, ...)
{
    va_list arg;
    va_start(arg, format);
    vfprintf(error_log, format, arg);
    va_end(arg);
}


int check_key(int code)
{
    static int scan = 0x00;
    if((!key[code]) && (scan == code)) scan = 0;
    if((key[code]) && (scan != code))
    {
        scan = code;
        return (1);
    }

    return (0);
}

/* Timer handlers - not implemented yet. */

void *timer_set(double duration, int param, void (*callback)(int))
{
    error("set timer %f msec.\n", duration);
    return (NULL);
}


void timer_remove(void *which)
{
}


