package jape.cpc;

import jemu.core.*;
import jemu.core.device.*;
import jemu.core.device.crtc.*;

/**
 * Title:        JAPE Version 1.0
 * Description:  Java Amstrad CPC Plus Emulator
 * Copyright:    Copyright (c) 2002
 * Company:
 * @author
 * @version 1.0
 */

public class GateArray extends Device implements CRTCListener {

  protected CPC cpc;
  protected Z80 z80;
  protected Basic6845 crtc;
  protected int r52;
  protected int interruptMask;
  protected int hSyncCount;
  protected int screenMode = -1;
  protected int newMode = 0;
  protected boolean inHSync = false;
  protected boolean outHSync = false;
  protected int monitorLine = 0;
  protected int[] pixels;
  protected int[] inks = new int[32];
  protected int[] modeMap;
  protected int offset = 0;
  protected int scanStart = 0;
  protected Renderer borderRenderer = new BorderRenderer();
  protected Renderer renderer = borderRenderer;
  protected int selInk = 0;

  protected static final int[] maTranslate = new int[65536];

  protected static final int[] inkTranslate = {
    0x666, 0x666, 0xF06, 0xFF6, 0x006, 0x0F6, 0x606, 0x6F6,
    0x0F6, 0xFF6, 0xFF0, 0xFFF, 0x0F0, 0x0FF, 0x6F0, 0x6FF,
    0x006, 0xF06, 0xF00, 0xF0F, 0x000, 0x00F, 0x600, 0x60F,
    0x066, 0xF66, 0xF60, 0xF6F, 0x060, 0x06F, 0x660, 0x66F
  };

  protected static final int[][] modeMaps = new int[4][];
  static {
    for (int mode = 0; mode < 4; mode++) {
      int[] map = new int[65536 * 8];
      modeMaps[mode] = map;
      for (int i = 0; i < 65535 * 8;) {
        int b1 = i & 0xff;
        int b2 = (i >> 8) & 0xff;
        decodeMode(map,i,mode,b1);
        i += 4;
        decodeMode(map,i,mode,b2);
        i += 4;
      }
    }
    for (int i = 0; i < maTranslate.length; i++) {
      int j = i << 1;
      maTranslate[i] = (j & 0x7fe) | ((j & 0x6000) << 1);
    }
  }

  protected static final void decodeMode(int[] map, int offs, int mode, int b) {
    switch(mode) {
      case 0:
        map[offs++] = map[offs++] =
          ((b >> 7) & 0x01) | ((b >> 2) & 0x02) | ((b >> 3) & 0x04) | ((b << 2) & 0x08);
        map[offs++] = map[offs++] =
          ((b >> 6) & 0x01) | ((b >> 1) & 0x02) | ((b >> 2) & 0x04) | ((b << 1) & 0x08);
        break;

      case 1:
        map[offs++] = ((b >> 7) & 0x01) | ((b >> 2) & 0x02);
        map[offs++] = ((b >> 6) & 0x01) | ((b >> 1) & 0x02);
        map[offs++] = ((b >> 5) & 0x01) | (b & 0x02);
        map[offs++] = ((b >> 4) & 0x01) | ((b << 1) & 0x02);
        break;

      case 2:
        break;

      case 3:
        break;
    }
  }

  public GateArray(CPC cpc) {
    super("Amstrad Gate Array");

    this.cpc = cpc;
    z80 = cpc.z80;
    crtc = cpc.crtc;
    reset();
  }

  public void setPixels(int[] value) {
    System.out.println("Pixels set " + (value == null));
    pixels = value;
  }

  public void reset() {
    r52 = 0;
    setScreenMode(0);
    inks[0x10] = 0xff808080;
  }

  public void writePort(int port, int value) {
    if ((value & 0x80) == 0) {
      if ((value & 0x40) == 0) {
        value &= 0x1f;
        selInk = value < 0x10 ? value : 0x10;
      }
      else {
        // This is a CPC Plus ink
        int ink = inkTranslate[value & 0x1f];
        inks[selInk] = 0xff000000 + ((ink & 0xf0) << 16) + ((ink & 0xf00) << 4) +
          ((ink & 0x0f) << 4);
      }
    }
    else {
      CPCMemory memory = (CPCMemory)cpc.getMemory();
      if ((value & 0x40) == 0) {
        memory.setLowerEnabled((value & 0x04) == 0);
        memory.setUpperEnabled((value & 0x08) == 0);
        if ((value & 0x10) != 0) {
          r52 = 0;
          setInterruptMask(interruptMask & 0x70);
        }
        newMode = value & 0x03;
      }
      else
        memory.setRAMBank(value);
    }
  }

  protected void setScreenMode(int mode) {
    screenMode = mode;
  }

  public void setInterruptMask(int value) {
    interruptMask = value;
    z80.setInterruptPending(value != 0);
  }

  public void interrupt() {
    r52 &= 0x1f;
    setInterruptMask(interruptMask & 0x70);
  }

  public void cycle() {
    hSyncCount++;
    if (hSyncCount == 2)
      outHSync = true;
    else if (hSyncCount == 7)
      endHSync();
    crtc.cycle();
  }

  // TODO: WinAPE does some weird stuff here to suit Overflow Preview 3
  protected void endHSync() {
    z80.setCycleDevice(crtc);
//    if (outHSync) endMonitorHSync();
    outHSync = false;
    if (screenMode != newMode)
      setScreenMode(newMode);
  }

  public void hSyncStart() {
    hSyncCount = 0;
    inHSync = true;
    outHSync = false;
    z80.setCycleDevice(this);
  }

  public void hSyncEnd() {
    endHSync();
    if (++r52 == 52) {
      r52 = 0;
      cpc.interrupt();
      setInterruptMask(interruptMask | 0x80);
    }
    if (monitorLine >= 0 && monitorLine < 270) {
      for (int i = 0; i < 48; i++)
        renderer.render();
      offset = scanStart += 384;
    }
    monitorLine++;
  }

  public void hDispEnd() { }

  public void scanStart() { }

  public void vSyncStart() {
    // TODO: This doesn't necessarily start a VSYNC
    monitorLine = -32;
    scanStart = offset = 0;
  }

  public void vSyncEnd() { }

  protected abstract class Renderer {
    public void render() { }
  }

  protected class BorderRenderer extends Renderer {
    public void render() {
      for (int i = 0; i < 8; i++)
        pixels[offset++] = inks[0x10];
    }
  }

}