package javzed.vz;

import java.applet.*;
import java.awt.*;
import java.awt.event.*;
import java.io.*;
import jemu.core.cpu.*;
import jemu.core.device.*;
import jemu.core.device.keyboard.*;
import jemu.core.device.memory.*;
import jemu.core.device.sound.*;
import jemu.ui.*;
import jemu.util.diss.*;

/**
 * Title:        JAPE Version 1.0
 * Description:  Java Amstrad CPC Plus Emulator
 * Copyright:    Copyright (c) 2002
 * Company:
 * @author
 * @version 1.0
 */

public class VZ extends Computer {

  protected static final int CYCLES_PER_SCAN = 228;
  protected static final int SCANS_PER_FRAME = 312;
  protected static final int SCANS_OF_FLYBACK = 57;
  protected static final int CYCLES_PER_FRAME = CYCLES_PER_SCAN * SCANS_PER_FRAME;

  protected static final int AUDIO_PER_FRAME = 8000 / 50;
  protected static final double AUDIO_CYCLE_RATIO =
    (double)AUDIO_PER_FRAME / CYCLES_PER_FRAME;
  protected static final int AUDIO_TEST = 0x40000000;
  protected static final int AUDIO_ADD =
    (int)Math.round(AUDIO_CYCLE_RATIO * AUDIO_TEST);
  protected static final int AUDIO_RESYNC_FRAMES = 50;

  protected Z80 z80 = new Z80();
  protected VZMemory memory = new VZMemory(this);
  protected int cycles = 0;
  protected int frameFlyback = 0;
  protected int vdcLatch = 0x00;
  protected SimpleRenderer renderer = new SimpleRenderer();
  protected int frameSkip = 0;
  protected Keyboard keyboard = new Keyboard();
  protected Disassembler disassembler = new DissZ80();
  protected SunAudio audio = new SunAudio(8000);
  protected byte soundByte = 0;
  protected int soundUpdate = 0;
  protected int syncCnt = AUDIO_RESYNC_FRAMES;

  public VZ(Applet applet, String name) {
    super(applet,name);
    z80.setMemoryDevice(this);
    z80.setCycleDevice(this);
    z80.setInterrupDevice(this);
  }

  public void initialise(String romPath) {
    memory.setMemory(0,getFile(romPath + "vz/VZROM.V20",16384));
    renderer.setFontData(getFile(romPath + "vz/VZ.CHR",768));
    audio.setWriteAhead(600);
  }

  public Memory getMemory() {
    return memory;
  }

  public Processor getProcessor() {
    return z80;
  }

  public void cycle() {
    if (++cycles == (SCANS_PER_FRAME - SCANS_OF_FLYBACK) * CYCLES_PER_SCAN) {
      if (--syncCnt == 0) {
        audio.sync();
        syncCnt = AUDIO_RESYNC_FRAMES;
      }
      frameFlyback = 0x80;
      z80.setInterruptPending(true);
      if (frameSkip == 0)
        renderer.renderScreen(memory);
      startTime += 20;
      long time = System.currentTimeMillis();
      if (time > startTime) {
        frameSkip = (frameSkip + 1) % 20;
        startTime = time;
      }
      else {
        frameSkip = 0;
        while(System.currentTimeMillis() < startTime) {
          try {
            Thread.sleep(1);
          } catch(Exception e) {
            e.printStackTrace();
          }
        }
      }
    }
    else if (cycles == SCANS_PER_FRAME * CYCLES_PER_SCAN) {
      cycles = frameFlyback = 0;
    }
    soundUpdate += AUDIO_ADD;
    if ((soundUpdate & AUDIO_TEST) != 0) {
      soundUpdate -= AUDIO_TEST;
      audio.writeulaw(soundByte);
    }
  }

  public void interrupt() {
    z80.setInterruptPending(false);
  }

  public int readByte(int address) {
    return (address >= 0x7000 || address < 0x6800) ? memory.readByte(address) :
      frameFlyback | keyboard.readByte(address);
  }

  public void writeByte(int address, int value) {
    if (address >= 0x7000)
      memory.writeByte(address,value);
    else if (address >= 0x6800) {
      if (((vdcLatch ^ value) & 0x41) != 0)
        soundByte = (byte)(soundByte ^ 0x80);
      vdcLatch = value;
      renderer.setVDCLatch(value);
    }
  }

  public void processKeyEvent(KeyEvent e) {
    if (e.getID() == KeyEvent.KEY_PRESSED)
      keyboard.keyPressed(e.getKeyCode());
    else if (e.getID() == KeyEvent.KEY_RELEASED)
      keyboard.keyReleased(e.getKeyCode());
  }

  protected void vzFileException(String error) throws Exception {
    if (error == null)
      error = "Bad VZ File format";
    throw new Exception(error);
  }

  public void loadFile(String name) throws Exception {
    InputStream in = openFile(name);
    try {
      byte[] header = new byte[24];
      int len = in.read(header);
      if (len != 24)                /*|| !"VZF0".equals(new String(header,0,4))) - Doe*/
        vzFileException(null);

      int type = header[21] & 0xff;
      int start = (header[22] & 0xff) + 256 * (header[23] & 0xff);
      int address = start;

      int read;
      do {
        read = in.read();
        if (read != -1) {
          memory.writeByte(address,read);
          address = (address + 1) & 0xffff;
        }
      } while(read != -1);
      in.close();

      if (type == 0xf0) {
        memory.writeByte(0x78a4,header[22]);
        memory.writeByte(0x78a5,header[23]);
        memory.writeByte(0x78f9,address);
        memory.writeByte(0x78fa,address >> 8);
      }
      else if (type == 0xf1)
        z80.setPC(start);
    } finally {
      in.close();
    }
  }

  public void setDisplay(Display value) {
    renderer.setDisplay(value);
  }

  public Dimension getDisplaySize() {
    return new Dimension(256,192);
  }

  public Disassembler getDisassembler() {
    return disassembler;
  }

  public void emulate(int mode) {
    audio.play();
    super.emulate(mode);
    audio.stop();
  }

}