package jemu.core.cpu;

import java.util.*;
import jemu.core.*;
import jemu.core.device.*;

/**
 * Title:        JAPE Version 1.0
 * Description:  Java Amstrad CPC Plus Emulator
 * Copyright:    Copyright (c) 2002
 * Company:
 * @author
 * @version 1.0
 */

public abstract class Processor {

  // Memory for processor
  protected Device memory;

  // Input Devices
  protected DeviceMapping[] inputDevice = new DeviceMapping[0];

  // Output Devices
  protected DeviceMapping[] outputDevice = new DeviceMapping[0];

  // Cycle devices
  protected Device cycleDevice = null;

  // Interrupt device
  protected Device interruptDevice = null;

  // Total number of cycles executed
  protected long cycles = 0;

  // Processor stopped
  protected boolean stopped = false;

  protected void cycle(int count) {
    if (cycleDevice == null)
      cycles += count;
    else
      for (; count > 0; count--) {
        cycles++;
        cycleDevice.cycle();
      }
  }

  public void reset() {
    cycles = 0;
  }

  public long getCycles() {
    return cycles;
  }

  public abstract void step();

  public abstract void stepOver();

  public abstract void runTo(int address);

  public void run() {

  }

  public synchronized void stop() {
    stopped = true;
  }

  public final int readWord(int addr) {
    return readByte(addr) + (readByte((addr + 1) & 0xffff) << 8);
  }

  public final void writeWord(int addr, int value) {
    writeByte(addr,value);
    writeByte((addr + 1) & 0xffff, value >> 8);
  }

  public final int readByte(int address) {
    return memory.readByte(address);
  }

  public final int writeByte(int address, int value) {
    memory.writeByte(address,value);
    return value;
  }

  public final int in(int port) {
    for (int i = 0; i < inputDevice.length; i++) {
      int result = inputDevice[i].readPort(port);
      if (result != -1)
        return result & 0xff;
    }
    return 0xff;
  }

  public final void out(int port, int value) {
    for (int i = 0; i < outputDevice.length; i++)
      outputDevice[i].writePort(port,value);
  }

  public final void setMemoryDevice(Device value) {
    memory = value;
  }

  public final Device getMemoryDevice() {
    return memory;
  }

  public final void addInputDeviceMapping(DeviceMapping value) {
    inputDevice = (DeviceMapping[])Util.arrayInsert(inputDevice,inputDevice.length,1,
      value);
  }

  public final void removeInputDeviceMapping(DeviceMapping value) {
    inputDevice = (DeviceMapping[])Util.arrayDeleteElement(inputDevice,value);
  }

  public final void addOutputDeviceMapping(DeviceMapping value) {
    outputDevice = (DeviceMapping[])Util.arrayInsert(outputDevice,outputDevice.length,1,
      value);
  }

  public final void removeOutputDeviceMapping(DeviceMapping value) {
    outputDevice = (DeviceMapping[])Util.arrayDeleteElement(outputDevice,value);
  }

  public final void setCycleDevice(Device value) {
    cycleDevice = value;
  }

  public final void setInterrupDevice(Device value) {
    interruptDevice = value;
  }

  public abstract String getState();

  public abstract String[] getRegisterNames();

  public abstract int getRegisterBits(int index);

  public abstract int getRegisterValue(int index);

  public abstract int getProgramCounter();

}