package jemu.core.device;

import java.applet.*;
import java.awt.*;
import java.awt.event.*;
import java.io.*;
import java.lang.reflect.*;
import java.net.*;
import java.util.*;
import jemu.core.*;
import jemu.core.cpu.*;
import jemu.core.device.memory.*;
import jemu.ui.*;
import jemu.util.diss.*;

/**
 * Title:        JAPE Version 1.0
 * Description:  Java Amstrad CPC Plus Emulator
 * Copyright:    Copyright (c) 2002
 * Company:
 * @author
 * @version 1.0
 */

public abstract class Computer extends Device implements Runnable {

  protected static Hashtable computers = new Hashtable();
  static {
    computers.put("VZ200","javzed.vz.VZ");
    computers.put("VZ300","javzed.vz.VZ");
    computers.put("CPC464","jape.cpc.CPC");
    computers.put("CPC664","jape.cpc.CPC");
    computers.put("CPC6128","jape.cpc.CPC");
  };

  public static final int STOP      = 0;
  public static final int STEP      = 1;
  public static final int STEP_OVER = 2;
  public static final int RUN       = 3;

  public Applet applet;
  protected Thread thread = new Thread(this);
  protected boolean stopped = false;
  protected int action = STOP;
  protected boolean running = false;
  protected boolean waiting = false;
  protected long startTime;

  // Listeners for stopped emulation
  protected Vector listeners = new Vector(1);

  public static Computer createComputer(Applet applet, String name) throws Exception {
    Enumeration enum = computers.keys();
    while (enum.hasMoreElements()) {
      String key = (String)enum.nextElement();
      if (key.equalsIgnoreCase(name)) {
        Class cl = Util.findClass(null,(String)computers.get(key));
        Constructor con = cl.getConstructor(new Class[] { Applet.class, String.class });
        return (Computer)con.newInstance(new Object[] { applet, name });
      }
    }
    throw new Exception("Computer " + name + " not found");
  }

  public Computer(Applet applet, String name) {
    super("Computer");
    this.applet = applet;
//    thread.setPriority(Thread.MAX_PRIORITY);
    thread.start();
  }

  public void initialise(String romPath) { }

  public InputStream openFile(String name) throws Exception {
    InputStream result;
    try {
      result = new URL(applet.getCodeBase(),name).openStream();
    } catch(Exception e) {
      e.printStackTrace();
      result = new FileInputStream(name);
    }
    return result;
  }

  protected byte[] getFile(String name, int size) {
    byte[] buffer = new byte[size];
    try {
      InputStream stream = null;
      try {
        stream = openFile(name);
        stream.read(buffer,0,size);
      } finally {
        if (stream != null)
          stream.close();
      }
    } catch (Exception e) {
      e.printStackTrace();
    }
    return buffer;
  }

  public void setDisplay(Display display) { }

  // For now, only supporting a single Processor
  public Disassembler getDisassembler() {
    return null;
  }

  public abstract Processor getProcessor();

  public abstract Memory getMemory();

  public void processKeyEvent(KeyEvent e) { }

  public void loadFile(String name) throws Exception { }

  public abstract Dimension getDisplaySize();

  public void start() {
    setAction(RUN);
  }

  public void stop() {
    setAction(STOP);
  }

  public void step() {
    setAction(STEP);
  }

  public void stepOver() {
    setAction(STEP_OVER);
  }

  public synchronized void setAction(int value) {
    if (running && value != RUN) {
      action = STOP;
      getProcessor().stop();
      while(running) {
        try {
          Thread.sleep(10);
        } catch(Exception e) {
          e.printStackTrace();
        }
      }
    }
    synchronized(thread) {
      action = value;
      thread.notify();
    }
  }

  public void dispose() {
    stopped = true;
    stop();
    try {
      thread.join();
    } catch(Exception e) {
      e.printStackTrace();
    }
  }

  protected void emulate(int mode) {
    switch(mode) {
      case STEP:      getProcessor().step();     break;
      case STEP_OVER: getProcessor().stepOver(); break;
      case RUN:       getProcessor().run();      break;
    }
  }

  public void addActionListener(ActionListener listener) {
    listeners.addElement(listener);
  }

  public void removeActionListener(ActionListener listener) {
    listeners.removeElement(listener);
  }

  protected void fireActionEvent() {
    ActionEvent e = new ActionEvent(this,0,null);
    for (int i = 0; i < listeners.size(); i++)
      ((ActionListener)listeners.elementAt(i)).actionPerformed(e);
  }

  public void run() {
    while(!stopped) {
      try {
        if (action == STOP) {
          synchronized(thread) {
            thread.wait();
          }
        }
        if (action != STOP) {
          try {
            running = true;
            int mode;
            synchronized(thread) {
              mode = action;
              action = STOP;
            }
            startTime = System.currentTimeMillis();
            emulate(mode);
          } finally {
            running = false;
            fireActionEvent();
          }
        }
      } catch(Exception e) {
        e.printStackTrace();
      }
    }
  }

}