package jemu.core.device.crtc;

import jemu.core.*;
import jemu.core.device.*;

/**
 * Title:        JAPE Version 1.0
 * Description:  Java Amstrad CPC Plus Emulator
 * Copyright:    Copyright (c) 2002
 * Company:
 * @author
 * @version 1.0
 */

public class Basic6845 extends CRTC {

  protected final int EVENT_HSYNC_START = 0x01;
  protected final int EVENT_HDISP_END   = 0x02;

  protected int hCC;
  protected int vLC;
  protected int vCC;
  protected int hCCMask = 0x7f;

  protected int[] reg = new int[32];
  protected int[] orig = new int[32];
  protected int[] rdMask;
  protected int[] wrMask;

  protected int[] eventMask = new int[256];

  protected int registerSelectMask = 0x01;
  protected int registerSelectTest = 0x00;

  protected int selReg;

  protected int hChars = 1;
  protected int hSyncStart, hDisplayed, hSyncWidth, hSyncCount, vSyncWidth, vSyncCount;
  protected boolean inHSync = false;
  protected boolean inVSync = false;

  public Basic6845() {
    super("Basic 6845");
    rdMask = wrMask = new int[32];
    for (int i = 0; i < 32; i++)
      rdMask[i] = wrMask[i] = 0xff;
    reset();
  }

  public void reset() {
    selReg = hCC = 0;
    hSyncWidth = hSyncCount = 0;
    inHSync = false;
    for (int i = 0; i < eventMask.length; i++)
      eventMask[i] = 0;
    reg[0] = 0x3f & wrMask[0];
    setEvents();
  }

  public void cycle() {
    if (hCC == reg[0]) {
      hCC = 0;
      listener.scanStart();
      scanStart();
    }
    else
      hCC = (hCC + 1) & hCCMask;
    if (inHSync) {
      hSyncCount = (hSyncCount + 1) & 0x0f;
      if (hSyncCount == hSyncWidth) {
        inHSync = false;
        listener.hSyncEnd();
      }
    }
    int mask = eventMask[hCC];
    if (mask != 0) {
      if ((mask & EVENT_HSYNC_START) != 0) {
        inHSync = true;
        listener.hSyncStart();
      }
      if ((mask & EVENT_HDISP_END) != 0)
        listener.hDispEnd();
    }
  }

  protected void scanStart() {
    if (++vLC == reg[9]) {
      vLC = 0;
      vCC = (vCC + 1) & 0x7f;
      if (inVSync && ++vSyncCount == vSyncWidth) {
        inVSync = false;
        listener.vSyncEnd();
      }
      if (vCC == reg[7] && !inVSync) {
        inVSync = true;
        listener.vSyncStart();
        vSyncCount = 0;
      }
    }
  }

  public void writePort(int port, int value) {
    if ((port & registerSelectMask) == registerSelectTest)
      selReg = value & 0x1f;
    else
      setReg(selReg,value);
  }

  protected void setReg(int index, int value) {
    orig[index] = value;
    value &= wrMask[index];
    if (reg[index] != value) {
      reg[index] = value;
      switch(index) {
        case 0:
        case 1:
        case 2: setEvents(); break;
        case 3: setReg3(value); setEvents(); break;
      }
    }
  }

  protected void setEvents() {
    eventMask[hSyncStart] = eventMask[hDisplayed] = 0;
    hChars = reg[0] + 1;
    hSyncStart = reg[2];
    hDisplayed = reg[1];
    eventMask[hSyncStart] = EVENT_HSYNC_START;
    eventMask[hDisplayed] |= EVENT_HDISP_END;
  }

  protected void setReg3(int value) {
    hSyncWidth = value & 0x0f;
    vSyncWidth = (value >> 4) & 0x0f;
  }

  public void setSelectMask(int mask, int test) {
    registerSelectMask = mask;
    registerSelectTest = test;
  }

  public int getHCC() {
    return hCC;
  }

}