package jemu.core.device.sound;

import java.io.*;
import jemu.core.device.*;
import sun.audio.*;

/**
 * Title:        JAPE Version 1.0
 * Description:  Java Amstrad CPC Plus Emulator
 * Copyright:    Copyright (c) 2002
 * Company:
 * @author
 * @version 1.0
 */

public class SunAudio extends Device {

  protected AudioStream stream;
  protected boolean playing = false;
  protected int writeAhead = 600;
  protected AudioPlayer player = AudioPlayer.player;

  public SunAudio(int samples) {
    this("Sun Audio Device",samples);
  }

  public SunAudio(String type, int samples) {
    super(type);
    stream = new AudioStream(samples);
  }

  public void play() {
    if (!playing) {
      sync();
      player.start(stream);
      playing = true;
    }
  }

  public void stop() {
    if (playing) {
      player.stop(stream);
      playing = false;
    }
  }

  public void sync() {
    stream.sync();
  }

  public void writePCM(int value) {
    stream.writePCM(value);
  }

  public void writeulaw(byte value) {
    stream.writeulaw(value);
  }

  public void setWriteAhead(int value) {
    writeAhead = value;
  }

  protected static byte[] PCM_TO_ULAW = new byte[8160];
  protected static int[] PCM_RANGE = { 0, 32, 96, 224, 480, 992, 2016, 4064, 8160 };
  protected static int[] ULAW_PREFIX = { 0xf0, 0xe0, 0xd0, 0xc0, 0xb0, 0xa0, 0x90, 0x80 };
  static {
    for (int i = 0; i < 8160; i++) {
      for (int j = 0; j < PCM_RANGE.length - 1; j++) {
        if (i < PCM_RANGE[j + 1]) {
          PCM_TO_ULAW[i] = (byte)(ULAW_PREFIX[j] | 15 - (i - PCM_RANGE[j]) / (2 << j));
          break;
        }
      }
    }
  };

  protected class AudioStream extends InputStream {

    byte[] buffer;
    int pos = 0;
    int wrPos = 0;

    protected AudioStream(int samples) {
      buffer = new byte[samples];
    }

    int cnt = 0;

    public int read() {
      int result = (int)(buffer[pos] & 0xff);
      pos = (pos + 1) % buffer.length;
      return result;
    }

    public int read(byte[] buff, int offs, int len) {
      int end = len + offs;
      for (; offs < end; offs++) {
        buff[offs] = buffer[pos];
        pos = (pos + 1) % buffer.length;
      }
      System.out.println("pos=" + pos + ", wrPos=" + wrPos);
      return len;
    }

    public void writeulaw(byte value) {
      buffer[wrPos] = value;
      wrPos = (wrPos + 1) % buffer.length;
    }

    public void writePCM(int value) {
      int mask;
      if (value < 0) {
        value = -value;
        mask = 0xff;
      }
      else
        mask = 0x7f;

      if (value >= 8160)
        value = 8159;

      writeulaw((byte)(PCM_TO_ULAW[value] & mask));
    }

    public int available() {
      return 8000;
    }

    public void close() {
      Thread.dumpStack();
      playing = false;
    }

    public void sync() {
      int newPos = wrPos - writeAhead;
      while (newPos < 0)
        newPos += buffer.length;
      pos = newPos;
    }
  }

}