package jemu.ui;

import java.applet.*;
import java.awt.*;
import java.awt.event.*;
import java.awt.image.*;
import java.net.*;
import java.text.*;
import java.util.*;
import jemu.core.*;

/**
 * Title:        JAPE Version 1.0
 * Description:  Java Amstrad CPC Plus Emulator
 * Copyright:    Copyright (c) 2002
 * Company:
 * @author
 * @version 1.0
 */

public class BaseControl extends Container {

  // ======================================
  // Static field definitions
  // ======================================

  // ======================================
  // Constants for alignment
  // ======================================

  /**
   * Left alignment and left direction constant.
   */
  public static final int LEFT    = 0;

  /**
   * Right alignment and right direction constant.
   */
  public static final int RIGHT   = 1;

  /**
   * Centre alignment constant (both horizontal and vertical).
   */
  public static final int CENTRE  = 2;

  /**
   * Top alignment constant.
   */
  public static final int TOP     = 3;

  /**
   * Bottom alignment constant.
   */
  public static final int BOTTOM  = 4;

  /**
   * Constant for tiling images.
   */
  public static final int TILE    = 5;

  /**
   * Constant for stretching images.
   */
  public static final int STRETCH = 6;

  // ======================================
  // Constants for arrows/directions
  // ======================================

  /**
   * Constant for Up scroll arrow.
   */
  public static final int UP      = TOP;

  /**
   * Constant for Down scroll arrow.
   */
  public static final int DOWN    = BOTTOM;

  // ======================================
  // Constants for orientation
  // ======================================

  /**
   * Constant for Horizontal orientation.
   */
  public static final int HORIZONTAL = Adjustable.HORIZONTAL;

  /**
   * Constant for Vertical orientation.
   */
  public static final int VERTICAL   = Adjustable.VERTICAL;

  // ======================================
  // Double buffer handling
  // ======================================

  /**
   * Double-buffer image used for painting.
   */
  protected static Image doubleBuffer = null;

  /**
   * Width of double buffer image.
   */
  protected static int dbWidth = 0;

  /**
   * Height of double buffer image.
   */
  protected static int dbHeight = 0;

  // ======================================
  // Image Cache
  // ======================================

  /**
   * Cache of used Images.
   */
  protected static Hashtable imageCache = new Hashtable();

  /**
   * Base URL. Path in which to look for images.
   */
  protected static URL baseImageURL;

  /**
   * Used to load images.
   */
  protected static MediaTracker mediaTracker;

  // Zero Dimension

  /**
   * Used internally to provide dimensions with width 0 and height 0.
   */
  public static final Dimension zeroDimension = new Dimension();

  // ======================================
  // Static colour definitions
  // ======================================

  /**
   * Foreground of the control.
   */
  public static final Color FOREGROUND = new Color(0,0,0);

  /**
   * Background of the control.
   */
  public static final Color BACKGROUND = new Color(0,0,0);

  // ======================================
  // Current drop down/ popup object
  // ======================================

  /**
   * Current drop-down or pop-up window.
   */
  protected static Window dropWindow = null;

  /**
   * Owner control of the current drop-down or pop-up window.
   */
  protected static BaseControl dropWindowOwner = null;

  /**
   * Allows mouse events to be ignored for other components when a drop-down or
   * pop-up window is displayed.
   */
  protected static boolean ignoreTillUp = false;

  // ======================================
  // Date, Time and number formatting
  // ======================================

  /**
   * Default formatting instance for decimal numbers.
   */
  protected static DecimalFormat decimalFormat = new DecimalFormat();

  /**
   * Default formatting symbol instance for decimal numbers.
   */
  protected static DecimalFormatSymbols decimalFormatSymbols =
    new DecimalFormatSymbols();

  /**
   * Default formatting instance for dates.
   */
  protected static DateFormat dateFormat =
    DateFormat.getDateInstance(DateFormat.SHORT);

  /**
   * Default formatting instance for times.
   */
  protected static DateFormat timeFormat =
    DateFormat.getTimeInstance(DateFormat.MEDIUM);

  //MSBUG: The "new GregorianCalendar().getTimeZone()" is used because the
  //MS JVM had a bug with the default time zone in builds < 3167.
  static {
    TimeZone zone = new GregorianCalendar().getTimeZone();
    dateFormat.setTimeZone(zone);
    timeFormat.setTimeZone(zone);
  };

  // ======================================
  // Default Font
  // ======================================

  /**
   * Default Font for all controls.
   */
  protected static Font defaultFont = new Font(
    Util.determineJVM() == Util.JVM_MS ? "Arial" : "Dialog",Font.PLAIN,11);

  // ======================================
  // Focused Control
  // ======================================

  /**
   * Current focused BaseControl instance.
   */
  protected static BaseControl focusControl = null;



  // ======================================
  // Instance field definitions
  // ======================================

  /**
   * The DoubleBuffered property. This should only be set to true on the main
   * parent BaseControl since a single static Image is used.
   */
  protected boolean doubleBuffered = false;

  /**
   * The Border property.
   */
  protected Border border = Border.empty;

  /**
   * The Transparent property.
   */
  protected boolean transparent = false;

  /**
   * Used to cache the Text property.
   */
  protected String text = null;

  /**
   * Horizontal alignment property for text and image.
   */
  protected int hAlign = CENTRE;

  /**
   * Vertical alignment property for text and image.
   */
  protected int vAlign = CENTRE;

  /**
   * Horizontal alignment of the text.
   */
  protected int textAlign = CENTRE;

  /**
   * The PreferredSize property. Allows a preset size to be defined.
   */
  protected Dimension prefSize = null;

  /**
   * The MinimumSize property. Allows a preset size to be defined.
   */
  protected Dimension minSize = null;

  /**
   * The MaximumSize property. Allows a preset size to be defined.
   */
  protected Dimension maxSize = null;

  /**
   * The Image property.
   */
  protected Image image = null;

  /**
   * The gap between the text and image.
   */
  protected int imageGap = 4;

  /**
   * The position of the image relative to the text.
   */
  protected int imagePos = LEFT;

  /**
   * The Tag property. Can be used to attach any object to the control.
   */
  protected Object tag = null;

  /**
   * The Preferred Height of the control in characters (of the current Font).
   */
  protected int prefHeightChars = 0;

  /**
   * The Preferred Width of the control in characters (of the current Font).
   */
  protected int prefWidthChars = 0;

  /**
   * The CanFocus property. If true allows the control to accept the keyboard
   * focus.
   */
  protected boolean canFocus = false;

  /**
   * true if the control has the keyboard focus.
   */
  protected boolean hasFocus = false;

  /**
   * Allows controls to contain another BaseControl which will always fill the
   * contents, when the control has no layout. This allows InternalFrame
   * controls to be used inside the BaseControl.
   */
  protected BaseControl content = null;

  /**
   * The display Rectangle of the Image of the control.
   */
  protected Rectangle imageRect;

  /**
   * The display Rectangle of the Text of the control.
   */
  protected Rectangle textRect;

  /**
   * Constructs a BaseControl with no text or image.
   */
  public BaseControl() {
    this(null,null);
  }

  /**
   * Constructs a BaseControl with the given text, but no image.
   *
   * @param text The text for the control.
   */
  public BaseControl(String text) {
    this(text,null);
  }

  /**
   * Constructs a BaseControl with the given image, but no text.
   *
   * @param image The image for the control.
   */
  public BaseControl(Image image) {
    this(null,image);
  }

  /**
   * Constructs a BaseControl with both text and an image.
   *
   * @param text The text for the control.
   * @param image The image for the control.
   */
  public BaseControl(String text, Image image) {
    enableEvents(MouseEvent.MOUSE_EVENT_MASK |
      MouseEvent.MOUSE_MOTION_EVENT_MASK |
      FocusEvent.FOCUS_EVENT_MASK |
      KeyEvent.KEY_EVENT_MASK);
    setText(text);
    setImage(image);
  }

  /**
   * Sets if the control is transparent. If set to true the background will not
   * be painted.
   *
   * @param value true for transparent, false for opaque.
   */
  public void setTransparent(boolean value) {
    transparent = value;
  }

  /**
   * Determines if the control is transparent. If set to true the background
   * will not be painted.
   *
   * @return true if transparent, false if opaque.
   */
  public boolean isTransparent() {
    return transparent;
  }

  /**
   * Sets the Border for the control.
   *
   * @param value The Border for the control.
   */
  public void setBorder(Border value) {
    border = value == null ? Border.empty : value;
  }

  /**
   * Gets the Border for the control.
   *
   * @return The Border of the control.
   */
  public Border getBorder() {
    return border;
  }

  /**
   * Sets if the control is the main double-buffer provider.
   *
   * @param value true if this control provides double-buffering for all child
   *              controls.
   */
  public void setDoubleBuffered(boolean value) {
    doubleBuffered = value;
  }

  /**
   * Determines if the control is the main double-buffer provider.
   *
   * @return true if this control provides double-buffering for all child
   *         controls.
   */
  public boolean isDoubleBuffered() {
    return doubleBuffered;
  }

  /**
   * Sets the Text for the control.
   *
   * @param value The Text for the control.
   */
  public void setText(String value) {
    text = value;
  }

  /**
   * Gets the Text for the control.
   *
   * @return The Text for the control.
   */
  public String getText() {
    return text;
  }

  /**
   * Sets the horizontal alignment of the contents of the control (text and
   * image).
   *
   * @param value The horizontal alignment for the contents.
   */
  public void setHorizontalAlignment(int value) {
    hAlign = value;
  }

  /**
   * Gets the horizontal alignment of the contents of the control (text and
   * image).
   *
   * @return The horizontal alignment for the contents.
   */
  public int getHorizontalAlignment() {
    return hAlign;
  }

  /**
   * Sets the vertical alignment of the contents of the control (text and
   * image).
   *
   * @param value The vertical alignment for the contents.
   */
  public void setVerticalAlignment(int value) {
    vAlign = value;
  }

  /**
   * Gets the vertical alignment of the contents of the control (text and
   * image).
   *
   * @return The vertical alignment for the contents.
   */
  public int getVerticalAlignment() {
    return vAlign;
  }

  /**
   * Sets the horizontal alignment for the text of the control.
   *
   * @param value The horizontal alignment for the text.
   */
  public void setTextAlignment(int value) {
    textAlign = value;
  }

  /**
   * Gets the horizontal alignment for the text of the control.
   *
   * @return The horizontal alignment for the text.
   */
  public int getTextAlignment() {
    return textAlign;
  }

  /**
   * Sets the gap in pixels between the text and image.
   *
   * @param value The gap in pixels betweeen the text and image.
   */
  public void setImageGap(int value) {
    imageGap = value;
  }

  /**
   * Gets the gap in pixels between the text and image.
   *
   * @return The gap in pixels betweeen the text and image.
   */
  public int getImageGap() {
    return imageGap;
  }

  /**
   * Sets the Image for the control.
   *
   * @param value The Image for the control.
   */
  public void setImage(Image value) {
    image = value;
  }

  /**
   * Gets the Image for the control.
   *
   * @return The Image for the control.
   */
  public Image getImage() {
    return image;
  }

  /**
   * Sets the Image for the control using the given image file name.
   *
   * @param value the file name for the image.
   */
  public void setImage(String value) {
    setImage(getImage(value));
  }

  /**
   * Sets the position of the Image relative to the text (or controls stretching
   * and tiling of the image).
   *
   * @param value The position of the Image.
   */
  public void setImagePos(int value) {
    imagePos = value;
  }

  /**
   * Gets the position of the Image.
   *
   * @return The position of the Image.
   */
  public int getImagePos() {
    return imagePos;
  }

  /**
   * Sets the preferred size of the control.
   *
   * @param value The preferred size for the control.
   */
  public void setPreferredSize(Dimension value) {
    prefSize = value;
  }

  /**
   * Gets the preferred size of the control. If this has been set using
   * <code>setPreferredSize()</code> then the set value is returned, otherwise
   * the preferred size is calculated using the maximum preferred size for the
   * layout of the control, and the preferred size for the text and image of the
   * control.
   *
   * @return The preferred size of the control.
   */
  public Dimension getPreferredSize() {
    Dimension result;
    if (prefSize != null)
      result = new Dimension(prefSize);
    else {
      result = getPreferredInnerSize(getFontMetrics());
      Insets insets = getInsets();
      result.width += insets.left + insets.right;
      result.height += insets.top + insets.bottom;
    }
    return result;
  }

  /**
   * Gets the preferred Inner size of the control.
   */
  public Dimension getPreferredInnerSize(FontMetrics fm) {
    Dimension result = getPreferredContentSize(fm);
    Dimension lPrefSize = getPreferredLayoutSize();
    result.width = Math.max(result.width,lPrefSize.width);
    result.height = Math.max(result.height,lPrefSize.height);
    return result;
  }

  /**
   * Calculates the preferred size of the layout for the control.
   *
   * @return The preferred size of the layout for the control.
   */
  protected Dimension getPreferredLayoutSize() {
    LayoutManager layout = getLayout();
    Dimension result = zeroDimension;
    if (layout != null) {
      if (layout instanceof LayoutManager2)
        ((LayoutManager2)layout).invalidateLayout(this);
      result = layout.preferredLayoutSize(this);
      Insets insets = getInsets();
      result.width -= insets.left + insets.right;
      result.height -= insets.top + insets.bottom;
    }
    return result;
  }

  /**
   * Calculates the preferred size of the contents (text and image) of the
   * control.
   *
   * @return The preferred size of the contents of the control.
   */
  protected Dimension getPreferredContentSize(FontMetrics fm) {
    Dimension result;
    Font font = getFont();
    if (font != null && text != null) {
      result = new Dimension(fm.stringWidth(text),fm.getHeight());
    }
    else if (image != null) {
      result = new Dimension(image.getWidth(null),image.getHeight(null));
    }
    else
      result = new Dimension();

    if (fm != null) {
      if (prefWidthChars != 0)
        result.width = Math.max(result.width,
          fm.charWidth('X') * prefWidthChars);
      if (prefHeightChars != 0)
        result.height = Math.max(result.height,
          fm.getHeight() * prefHeightChars);
    }

    return result;
  }

  /**
   * Sets the minimum size of the control.
   *
   * @param value The minimum size for the control.
   */
  public void setMinimumSize(Dimension value) {
    minSize = value;
  }

  /**
   * Gets the minimum size of the control.
   *
   * @return The minimum size of the control.
   */
  public Dimension getMinimumSize() {
    return minSize != null ? minSize : getMinimumLayoutSize();
  }

  /**
   * Calculates the minimum layout size of the control.
   *
   * @return The minimum layout size of the control.
   */
  protected Dimension getMinimumLayoutSize() {
    LayoutManager layout = getLayout();
    return layout == null ? zeroDimension : layout.minimumLayoutSize(this);
  }

  /**
   * Sets the maximum size of the control.
   *
   * @param value The maximum size for the control.
   */
  public void setMaximumSize(Dimension value) {
    maxSize = value;
  }

  /**
   * Gets the maximum size of the control.
   *
   * @return The maximum size of the control.
   */
  public Dimension getMaximumSize() {
    return maxSize != null ? maxSize : getMaximumLayoutSize();
  }

  /**
   * Calculates the maximum layout size of the control.
   *
   * @return The maximum layout size of the control.
   */
  protected Dimension getMaximumLayoutSize() {
    LayoutManager layout = getLayout();
    return layout instanceof LayoutManager2 ?
      ((LayoutManager2)layout).maximumLayoutSize(this) : zeroDimension;
  }

  /**
   * Gets the Insets for the control. This is usually determined by the Border.
   *
   * @return The Insets for the control
   */
  public Insets getInsets() {
    return border.getInsets();
  }

  /**
   * Gets the bounding Rectangle for the contents of the control. This is the
   * Rectangle for the controls bounds adjusted to remove the Insets.
   *
   * @return The content Rectangle of the control.
   */
  public Rectangle getContentBounds() {
    Dimension size = getSize();
    Insets insets = getInsets();
    return new Rectangle(insets.left,insets.top,
      size.width - insets.left - insets.right,
      size.height - insets.top - insets.bottom);
  }

  /**
   * Paints a Rectangle of the control.
   *
   * @param g The Graphics context on which to paint.
   * @param rect The Rectangle to be painted.
   */
  public void paintRect(Graphics g, Rectangle rect) {
    Rectangle oldClip = clipRect(g,rect);
    paint(g);
    g.setClip(oldClip);
  }

  /**
   * Overridden to provide smoother graphics display.
   *
   * @param g The Graphics context to update.
   */
  public void update(Graphics g) {
    paint(g);
  }

  /**
   * Paints the control. This provides double-buffering and grey-scale
   * functionality where appropriate, and paints the border, background, text
   * and image of the control, along with any child controls clipped to their
   * bounds.
   *
   * @param g The Graphics context on which to paint.
   */
  public final void paint(Graphics g) {
    Dimension size = getSize();

    int width = size.width;
    int height = size.height;

    if (width > 0 && height > 0) {
      Rectangle oldClip = clipRect(g,0,0,width,height);
      Rectangle bounds = new Rectangle(0,0,width,height);
      Graphics gUse = g;
      Graphics gBuffer = null;
      Image buffer = null;
      if (doubleBuffered) {
        buffer = getDoubleBufferImage(this,width,height);
        gUse = gBuffer = buffer.getGraphics();
        gBuffer.setClip(getClipBounds(g));
        gUse = gBuffer;
      }

      Color b = getBackground();
      gUse.setColor(b);
      paintBackground(gUse,bounds);
      paintBorder(gUse,bounds);
      clipRect(gUse,bounds);
      Rectangle childBounds = new Rectangle(bounds);
      paintContents(gUse,bounds);
      paintChildren(gUse,childBounds);

      if (buffer != null) {
        gBuffer.dispose();
        g.drawImage(buffer,0,0,this);
      }

      g.setClip(oldClip);
    }
  }

  /**
   * Paints the background of the control. Modifying the bounds will affect the
   * painting of the border, and clipping of contents and child controls.
   *
   * @param g The Graphics context on which to paint.
   * @param rect The bounding rectangle of the control.
   */
  protected void paintBackground(Graphics g, Rectangle rect) {
    if (!transparent)
      g.fillRect(rect.x,rect.y,rect.width,rect.height);
  }

  /**
   * Paints the Border of the control. Modifying the bounds will affect clipping
   * of contents and child controls.
   *
   * @param g The Graphics context on which to paint the Border.
   * @param rect The bounding rectangle for the border.
   */
  protected void paintBorder(Graphics g, Rectangle rect) {
    border.paintBorder(this,g,rect);
  }

  /**
   * Paints the contents (text and image) of the control.
   *
   * @param g The Graphics context on which to paint.
   * @param rect The bounding rectangle for the contents.
   */
  protected void paintContents(Graphics g, Rectangle rect) {
    paintTextAndImage(g,rect);
  }

  /**
   * Paints the text and image of the control.
   *
   * @param g The Graphics context on which to paint.
   * @param rect The bounding rectangle for the text and image.
   */
  protected void paintTextAndImage(Graphics g, Rectangle rect) {
    Font font = getFont();
    g.setFont(font);
    FontMetrics fm = g.getFontMetrics();
    Dimension m = text != null ? new Dimension(fm.stringWidth(text),fm.getHeight()) :
      (image != null ? new Dimension(image.getWidth(null),image.getHeight(null)) :
       zeroDimension);

    int x = rect.x;
    int y = rect.y;

    if (hAlign == RIGHT)
      x += rect.width - m.width;
    else if (hAlign == CENTRE)
      x += (rect.width - m.width) / 2;

    if (vAlign == BOTTOM)
      y += rect.height - m.height;
    else if (vAlign == CENTRE)
      y += (rect.height - m.height) / 2;

    if (image != null) {
      paintImage(g,image,x,y,rect,imagePos,this);
      if (imageRect == null)
        imageRect = new Rectangle();
      if (imagePos == STRETCH || imagePos == TILE) {
        imageRect.x = rect.x;
        imageRect.y = rect.y;
        imageRect.width = rect.width;
        imageRect.height = rect.height;
      }
      else {
        imageRect.x = x;
        imageRect.y = y;
        imageRect.width = image.getWidth(null);
        imageRect.height = image.getWidth(null);
      }
    }
    else
      imageRect = null;

    if (text != null) {
      paintText(g,fm,x,y);
      if (textRect == null)
        textRect = new Rectangle();
      textRect.x = x;
      textRect.y = y;
      textRect.width = fm.stringWidth(text);
      textRect.height = fm.getHeight();
    }
    else
      textRect = null;
  }

  protected void paintText(Graphics g, FontMetrics fm, int x, int y) {
    g.setColor(getForeground());
    g.drawString(text,x,y + fm.getAscent());
  }

  /**
   * Paints the child components of the component.
   *
   * @param g The Graphics context on which to paint
   * @param rect The clipping rectangle in which to paint children
   */
  protected void paintChildren(Graphics g, Rectangle rect) {
    Component[] components = getComponents();
    if (components.length > 0) {
      Rectangle oldClip = clipRect(g,rect);
      for (int i = components.length - 1; i >= 0; i--)
        paintChild(g,components[i]);
      g.setClip(oldClip);
    }
  }

  /**
   * Gets an Image from the Image cache. Loads the new Image if required, and
   * returns the Image from the cache.
   *
   * @param name The name of the Image file
   * @return The Image, or null if the image cannot be loaded
   */
  public Image getImage(String name) {
    if (baseImageURL == null)
      throw new RuntimeException("No BaseImageURL defined for image cache.");
    Image result = (Image)imageCache.get(name);
    if (result == null) {
      if (mediaTracker == null)
        mediaTracker = new MediaTracker(this);
      try {
        URL url = new URL(baseImageURL,name);
        result = getToolkit().getImage(url);
        mediaTracker.addImage(result,0);
        mediaTracker.waitForAll();
        mediaTracker.removeImage(result);
      } catch (Exception e) {
        throw new RuntimeException(e.getMessage());
      }
      imageCache.put(name,result);
    }
    return result;
  }

  /**
   * Notifies a list of ActionListeners of an action.
   *
   * @param listeners A Vector containing ActionListeners
   * @param command The command String for the ActionEvent
   */
  public void fireActionEvent(Vector listeners, String command) {
    fireActionEvent(listeners,this,0,command,0);
  }

  /**
   * Notifies a list of ActionListeners of an action.
   *
   * @param listeners A Vector containing ActionListeners
   * @param command The command String for the ActionEvent
   */
  public void fireActionEvent(Vector listeners, int id, String command,
    int modifiers)
  {
    fireActionEvent(listeners,this,id,command,modifiers);
  }

  /**
   * Shows a Drop-down or Pop-up window for the control.
   *
   * @param value The Window to show
   */
  protected void showDropWindow(Window value) {
    showDropWindow(this,value);
  }

  /**
   * Notification that a Drop-down or Pop-up window for this control has been
   * hidden.
   */
  protected void dropWindowHidden() { }

  /**
   * Overridden to support Drop-down/Pop-up windows, Drag-and-drop processing,
   * enhanced focus processing, and child MouseEvent processing.
   *
   * @param e The AWTEvent
   */
  protected void processEvent(AWTEvent e) {
    if (e instanceof MouseEvent) {
      if (e.getID() == MouseEvent.MOUSE_PRESSED) {
        if (dropWindow != null &&
          findParentWindow((Component)e.getSource()) != dropWindow)
        {
          hideDropWindow();
          ignoreTillUp = true;
          return;
        }
        else if (isEnabled()) {
          if (canFocus)
            requestFocus();
        }
      }
      else if (ignoreTillUp) {
        if (e.getID() == MouseEvent.MOUSE_RELEASED)
          ignoreTillUp = false;
        return;
      }
      // Parent mouse processing
      childMouseEvent((MouseEvent)e);
    }
    else if (e.getID() == FocusEvent.FOCUS_GAINED) {
      focusControl = this;
      focusChanged(true);
    }
    else if (e.getID() == FocusEvent.FOCUS_LOST) {
      if (focusControl == this)
        focusControl = null;
      focusChanged(false);
    }
    super.processEvent(e);
  }

  /**
   * Overridden to support child KeyEvent processing, and to support clipboard
   * functionality. ie. Cut, Copy and Paste with CTRL-X, CTRL-C and CTRL-V.
   *
   * @param e The KeyEvent
   */
  protected void processKeyEvent(KeyEvent e) {
    super.processKeyEvent(e);
    if (isEnabled() && !e.isConsumed()) {
      childKeyEvent(e);
      if (e.getID() == KeyEvent.KEY_PRESSED && hasFocus && e.isControlDown()) {
        switch(e.getKeyCode()) {
          case KeyEvent.VK_X: doCut(); break;
          case KeyEvent.VK_C: doCopy(); break;
          case KeyEvent.VK_V: doPaste(); break;
        }
      }
    }
  }

  /**
   * Gets a control colour. Supports standard colours, and the ability to get
   * the FOREGROUND and BACKGROUND colour of the control.
   *
   * @param value The colour to get
   * @return The colour
   */
  public Color getColour(Color value) {
    if (value == FOREGROUND)
      return getForeground();
    else if (value == BACKGROUND)
      return getBackground();
    else
      return value;
  }

  /**
   * Sets the preferred width of the control in characters. A value of zero is
   * used to indicate no preferred width in characters.
   *
   * @param value The preferred width in characters
   */
  public void setPreferredWidthChars(int value) {
    prefWidthChars = value;
  }

  /**
   * Gets the preferred width of the control in characters.
   *
   * @return The preferred width in characters
   */
  public int getPreferredWidthChars() {
    return prefWidthChars;
  }

  /**
   * Sets the preferred height of the control in characters. A value of zero is
   * used to indicate no preferred height in characters.
   *
   * @param value The preferred height in characters
   */
  public void setPreferredHeightChars(int value) {
    prefHeightChars = value;
  }

  /**
   * Gets the preferred height of the control in characters.e
   *
   * @return The preferred height in characters
   */
  public int getPreferredHeightChars() {
    return prefHeightChars;
  }

  /**
   * Gets the FontMetrics for the Font of the control.
   *
   * @return The FontMetrics for the control
   */
  public FontMetrics getFontMetrics() {
    return getFontMetrics(getFont());
  }

  /**
   * Gets the FontMetrics for a given font.
   *
   * @param font The Font for which to retrieve the FontMetrics
   * @return The FontMetrics for the given Font
   */
  public FontMetrics getFontMetrics(Font font) {
    return font == null ? null : getToolkit().getFontMetrics(font);
  }

  /**
   * Sets if the control can receive the keyboard focus.
   *
   * @param value true to allow the control to receive the focus
   */
  public void setFocusTraversable(boolean value) {
    canFocus = value;
  }

  /**
   * Overridden to allow the control to receive focus when the enabled and
   * it can receive the focus.
   *
   * @return true if the control can receive the keyboard focus
   */
  public boolean isFocusTraversable() {
    return isEnabled() && canFocus;
  }

  /**
   * Called when the keyboard focus of the control changes.
   *
   * @param value true if receiving focus, false if losing focus
   */
  protected void focusChanged(boolean value) {
    hasFocus = value;
  }

  /**
   * Empty method to be overridden by controls implementing Cut clipboard
   * functionality.
   */
  public void doCut() { }

  /**
   * Empty method to be overridden by controls implementing Copy clipboard
   * functionality.
   */
  public void doCopy() { }

  /**
   * Empty method to be overridden by controls implementing Paste clipboard
   * functionality.
   */
  public void doPaste() { }

  /**
   * Overridden to ensure that the content control if any is resized to the
   * content bounds when no layout manager is defined for the control.
   */
  public void doLayout() {
    LayoutManager layout = getLayout();
    if (layout instanceof LayoutManager2)
      ((LayoutManager2)layout).invalidateLayout(this);
    super.doLayout();
    if (getLayout() == null && content != null)
      content.setBounds(getContentBounds());
  }

  /**
   * Overridden to report as disabled if any parent control is disabled.
   *
   * @return true only if enabled and all parent controls are enabled
   */
  public boolean isEnabled() {
    boolean result = super.isEnabled();
    if (result) {
      Container parent = getParent();
      result &= parent == null || parent.isEnabled();
    }
    return result;
  }

  /**
   * Converts the given coordinates in the control to a Point on the screen.
   *
   * @param x The x coordinate of the point
   * @param y The y coordinate of the point
   * @return The Point on the screen represented by the given coordinates
   */
  public Point getLocationOnScreen(int x, int y) {
    Point where = getLocationOnScreen();
    where.x += x;
    where.y += y;
    return where;
  }

  /**
   * Converts a Point in the control to a Point on the screen.
   *
   * @param location The Point determining the location within the control
   * @return The Point on the screen represented by the Point in the control
   */
  public Point convertPointToScreen(Point location) {
    Point where = getLocationOnScreen();
    where.x += location.x;
    where.y += location.y;
    return where;
  }

  /**
   * Converts a Point on the screen to a Point within the control.
   *
   * @param location The Point determining the location on the screen
   * @return The Point in the control represented by the Point on the screen
   */
  public Point convertPointFromScreen(Point location) {
    Point where = getLocationOnScreen();
    where.x = location.x - where.x;
    where.y = location.y - where.y;
    return where;
  }

  /**
   * Gets the Point on the screen for a MouseEvent for the control.
   *
   * @param e The MouseEvent
   * @return The Point on the screen for the given MouseEvent
   */
  public Point convertEventToScreen(MouseEvent e) {
    return convertPointToScreen(e.getPoint());
  }

  /**
   * Converts a Point within the control to a Point within a parent control.
   *
   * @param parent The parent to convert the Point to
   * @param p The Point to be converted
   * @return The input Point p converted to parent coordinates
   */
  public Point convertToParent(Container parent, Point p) {
    return convertToParent(this,parent,p);
  }

  /**
   * Converts a Point within a parent control to a Point within the control.
   *
   * @param parent The parent to convert the Point from
   * @param p The Point to be converted
   * @return The input Point p converted to child coordinates
   */
  public Point convertFromParent(Container parent, Point p) {
    return convertFromParent(this,parent,p);
  }

  /**
   * Brings the control to the front of all siblings within the parent
   * container control.
   */
  public void toFront() {
    Container parent = getParent();
    if (parent != null && this != parent.getComponent(0)) {
      parent.remove(this);
      parent.add(this,0);
      repaint();
    }
  }

  /**
   * Overridden to allow for DefaultFont functionality.
   *
   * @return The Font of the control, or the default Font if null
   */
  public Font getFont() {
    Font result = super.getFont();
    return result == null ? defaultFont : result;
  }

  /**
   * Gets the Rectangle in which the Image for the control is rendered.
   *
   * @return The Image Rectangle, or null if no Image has been rendered
   */
  public Rectangle getImageRect() {
    return imageRect == null ? null : new Rectangle(imageRect);
  }

  /**
   * Gets the Rectangle in which the Text for the control is rendered.
   *
   * @return The Text Rectangle, or null if no Text has been rendered
   */
  public Rectangle getTextRect() {
    return textRect == null ? null : new Rectangle(textRect);
  }

  /**
   * Sets the content component for the control. This can be used to create a
   * control with no layout manager to be used as a parent for InternalFrame
   * controls, but with another control used for the background.
   *
   * @param value The control to be used for the content
   */
  public void setContentComponent(BaseControl value) {
    content = value;
    add(content,0);
  }

  /**
   * Gets the content component for the control.
   *
   * @return The content component for the control
   */
  public BaseControl getContentComponent() {
    return content;
  }

  /**
   * Overridden to invalidate the correct rectangle when moved. (Since JDK 1.1.x has a bug in
   * Component.reshape().
   *
   * @param x The new x position
   * @param y The new y position
   * @param width The new width
   * @param height The new height
   */
  public void setBounds(int x, int y, int width, int height) {
    Rectangle bounds = getBounds();
    Component parent = getParent();
    if (parent != null) {
      if (bounds.x != x || bounds.y != y || bounds.width != width || bounds.height != height)
        parent.repaint(bounds.x,bounds.y,bounds.width,bounds.height);
    }
    super.setBounds(x,y,width,height);
    Rectangle newBounds = getBounds();
    boolean resized = newBounds.width != bounds.width || newBounds.height != bounds.height;
    if (resized)
      doLayout();
  }

  /**
   * Overridden to get around a bug in the MS JVM (JDK 1.1.5).
   *
   * @param size The new size for the control
   */
  public void setSize(Dimension size) {
    setSize(size.width,size.height);
  }

  /**
   * Checks that the given Rectangle is within the bounds of the parent control.
   * The Rectangle is modified to ensure it is within the parent content bounds.
   *
   * @param bounds The bounding rectangle to check
   */
  public void checkBounds(Rectangle bounds) {
    Container parent = getParent();
    if (parent != null) {
      Dimension size = parent.getSize();
      Insets insets = parent.getInsets();
      bounds.width = Math.min(bounds.width,size.width - insets.left -
        insets.right);
      bounds.height = Math.min(bounds.height,size.height - insets.top -
        insets.bottom);
      if (bounds.x < insets.left)
        bounds.x = insets.left;
      if (bounds.x + bounds.width > size.width - insets.right)
        bounds.x = (size.width - insets.right) - bounds.width;
      if (bounds.y < insets.top)
        bounds.y = insets.top;
      if (bounds.y + bounds.height > size.height - insets.bottom)
        bounds.y = (size.height - insets.bottom) - bounds.height;
    }
  }

  /**
   * Finds the component at a given Point on the screen within the parent
   * window of this control.
   *
   * @param where The Point on the screen at which to find the Component
   * @return The Component at the given Point, or null if none found
   */
  public Component findComponentAt(Point where) {
    Component result = findComponentAt(findParentWindow(this),where);
    if (result != null) {
      Point pos = result.getLocationOnScreen();
      where.x -= pos.x;
      where.y -= pos.y;
    }
    return result;
  }

  /**
   * Sets the Tag of the control. The Tag can be used to attach user data to a\
   * control.
   *
   * @param value The Tag for the control
   */
  public void setTag(Object value) {
    tag = value;
  }

  /**
   * Gets the Tag for the control.
   *
   * @return The Tag for the control
   */
  public Object getTag() {
    return tag;
  }

  /**
   * Gets the index of a child Component of a control.
   *
   * @param value The child Component for which to find the index
   * @return The index of the child Component or -1 if not found
   */
  public int getComponentIndex(Component value) {
    Component[] components = getComponents();
    for (int i = 0; i < components.length; i++)
      if (components[i] == value)
        return i;
    return -1;
  }

  /**
   * Provides the ability to process MouseEvents for this control and all child
   * controls.
   *
   * @param e The MouseEvent
   */
  protected void childMouseEvent(MouseEvent e) {
    Component parent = getParent();
    if (parent instanceof BaseControl)
      ((BaseControl)parent).childMouseEvent(e);
  }

  /**
   * Provides the ability to process KeyEvents for this control and all child
   * controls.
   *
   * @param e The KeyEvent
   */
  protected void childKeyEvent(KeyEvent e) {
    Component parent = getParent();
    if (parent instanceof BaseControl)
      ((BaseControl)parent).childKeyEvent(e);
  }

  /**
   * Sets the Data for the control. By default this sets the Text for the
   * control, but can be overridden in descendants to provide other data.
   *
   * @param value The Data value for the control
   */
  public void setData(Object value) {
    setText((String)value);
  }

  /**
   * Gets the Data for the control. By default this gets the Text for the
   * control.
   *
   * @return The Data for the control
   */
  public Object getData() {
    return getText();
  }

  /**
   * Gets the Applet for the current running applet.
   *
   * @return The current running Applet instance
   */
  public Applet getApplet() {
    Container parent = getParent();
    while (parent != null && !(parent instanceof Applet))
      parent = parent.getParent();
    return (Applet)parent;
  }

  /**
   * Paints an individual child component, clipping and translating the Graphics
   * to suit. If the resulting clipping rectangle is empty, the component will
   * not be painted at all.
   *
   * @param g The Graphics object on which to paint
   * @param component The Component to paint
   */
  protected void paintChild(Graphics g, Component component) {
    paintComponent(g,component);
  }

  /**
   * Centres the control to another Component.
   *
   * @param component The Component to centre to
   */
  public void centre(Component component) {
    Point where = component.getLocationOnScreen();
    Dimension pSize = component.getSize();
    Dimension size = getSize();
    where.x += (pSize.width - size.width) / 2;
    where.y += (pSize.height - size.height) / 2;
    setLocation(convertPointFromScreen(where));
  }

  // ======================================
  // Static methods
  // ======================================

  /**
   * Paints the image for a control. This supports stretching and tiling.
   *
   * @param g The Graphics context on which to paint.
   * @param image The image to be painted.
   * @param x The left coordinate at which to paint the image.
   * @param y The top coordinate at which to paint the image.
   * @param rect The bounding rectangle in which to paint the image.
   * @param imagePos The position of the image, or TILED or STRETCHED.
   * @param component Used for Image rendering methods. Should be the control
   *                  requesting the painting.
   */
  protected static void paintImage(Graphics g, Image image, int x, int y,
    Rectangle rect, int imagePos, Component component)
  {
    if (imagePos == STRETCH)
      g.drawImage(image,rect.x,rect.y,rect.width,rect.height,component);
    else if (imagePos == TILE) {
      int iw = image.getWidth(null);
      int ih = image.getHeight(null);
      if (iw > 0 && ih > 0)
        for (y = 0; y < rect.height; y += ih)
          for (x = 0; x < rect.width; x += iw)
            g.drawImage(image,rect.x + x,rect.y + y,component);
    }
    else
      g.drawImage(image,x,y,component);
  }

  /**
   * Maintains an Image to be used by double-buffering components.
   *
   * @param c The component for which to get the buffer
   * @param width The required width of the buffer
   * @param height The required height of the buffer
   * @return An Image on which to render controls for double-buffering
   */
  public static Image getDoubleBufferImage(Component c, int width, int height) {
    if (doubleBuffer == null || dbWidth < width || dbHeight < height) {
      dbWidth = Math.max(dbWidth,width);
      dbHeight = Math.max(dbHeight,height);
      if (doubleBuffer != null) {
        doubleBuffer.flush();
        doubleBuffer = null;
        System.gc();
        System.runFinalization();
        System.gc();
      }
      doubleBuffer = c.createImage(dbWidth,dbHeight);
    }
    return doubleBuffer;
  }

  /**
   * Clip a region of a Graphics object. This should be used rather than
   * Graphics.clipRect() since the Microsoft JVM does not always clip correctly.
   *
   * @param g The Graphics object to clip
   * @param x The left of the clipping rectangle
   * @param y The top of the clipping rectangle
   * @param width The width of the clipping rectangle
   * @param height The height of the clipping rectangle
   * @return The previous clipping rectangle of the Graphics
   */
  public static Rectangle clipRect(Graphics g, int x, int y, int width,
    int height)
  {
    return clipRect(g,new Rectangle(x,y,width,height));
  }

  /**
   * Clip a region of a Graphics object. This should be used rather than
   * Graphics.clipRect() since the Microsoft JVM does not always clip correctly.
   *
   * @param g The Graphics object to clip
   * @param rect The clipping rectangle
   * @return The previous clipping rectangle of the Graphics
   */
  public static Rectangle clipRect(Graphics g, Rectangle rect) {
    Rectangle oldClip = g.getClipBounds();
    if (oldClip != null)
      rect = oldClip.intersection(rect);
    g.setClip(rect);
    return oldClip;
  }

  /**
   * Return the clipping region of a given Graphics object. This prevents a bug
   * with the Microsoft JVM where an empty clipping region returns null.
   * BEWARE: If the Graphics has not had a clipping rectangle defined this
   * method could return null for no (not empty) clipping rectangle. This is the
   * standard behaviour for a Java Graphics object, but MS JVM may return null
   * for an empty rectangle.
   *
   * @param g The Graphics object
   * @return The clipping rectangle of the Graphics
   */
  public static Rectangle getClipBounds(Graphics g) {
    Rectangle result = g.getClipBounds();
    if (result == null)
      result = new Rectangle();
    return result;
  }

  /**
   * Paints an individual child component, clipping and translating the Graphics
   * to suit. If the resulting clipping rectangle is empty, the component will
   * not be painted at all.
   *
   * @param g The Graphics object on which to paint
   * @param component The Component to paint
   * @return true if the Component was painted
   */
  public static boolean paintComponent(Graphics g, Component component) {
    boolean result = false;
    if (component != null && component.isShowing()) {
      Rectangle bounds = component.getBounds();
      Rectangle oldClip = clipRect(g,bounds);
      if (!getClipBounds(g).isEmpty()) {
        result = true;
        g.translate(bounds.x,bounds.y);
        component.paint(g);
        g.translate(-bounds.x,-bounds.y);
      }
      g.setClip(oldClip);
    }
    return result;
  }

  /**
   * Sets the URL for the path to cached Images.
   *
   * @param value The base URL for the cached Image path
   */
  public static void setBaseImageURL(URL value) {
    baseImageURL = value;
  }

  /**
   * Creates and ActionEvent for a component and sends the event to a list of
   * ActionListeners.
   *
   * @param listeners A Vector of ActionListeners
   * @param source The source Component for the event
   * @param command The ActionCommand for the event
   */
  public static void fireActionEvent(Vector listeners, Component source,
    int id, String command, int modifiers)
  {
    fireActionEvent(listeners,new ActionEvent(source,id,command,modifiers));
  }

  /**
   * Sends an ActionEvent to a list of ActionListeners.
   *
   * @param listeners A Vector of ActionListeners
   * @param e The ActionEvent to be sent
   */
  public static void fireActionEvent(Vector listeners, ActionEvent e) {
    if (listeners != null)
      for (int i = 0; i < listeners.size(); i++)
        ((ActionListener)listeners.elementAt(i)).actionPerformed(e);
  }

  /**
   * Draws an arrow using the current colour.
   *
   * @param g The Graphics context on which to paint the arrow
   * @param rect The bounding Rectangle for the arrow
   * @param direction The direction of the arrow (UP,DOWN,LEFT or RIGHT)
   */
  public static void drawArrow(Graphics g, Rectangle rect, int direction) {
    if (direction == LEFT || direction == RIGHT) {
      int y = rect.y + rect.height / 2;
      int max = (rect.height - 1) / 2;
      int x, step;
      if (direction == LEFT) {
        x = rect.x + (rect.width - max) / 2;
        step = 1;
      }
      else {
        x = rect.x + rect.width - (rect.width - max) / 2 - 1;
        step = -1;
      }
      for (int hgt = 0; hgt < max; hgt++) {
        g.drawLine(x,y,x,y + hgt * 2);
        y--;
        x += step;
      }
    }
    else {
      int x = rect.x + rect.width / 2;
      int max = (rect.width - 1) / 2;
      int y, step;
      if (direction == UP) {
        y = rect.y + (rect.height - max) / 2;
        step = 1;
      }
      else {
        y = rect.y + rect.height - (rect.height - max) / 2 - 1;
        step = -1;
      }
      for (int wid = 0; wid < max; wid++) {
        g.drawLine(x,y,x + wid * 2,y);
        x--;
        y += step;
      }
    }
  }

  /**
   * Draws a rectangle using a dotted line in the current colour.
   *
   * @param g The Graphics context on which to paint the rectangle
   * @param r The rectangle to be painted
   */
  public static void drawDottedRect(Graphics g, Rectangle r) {
    int maxX = r.x + r.width - 1;
    int maxY = r.y + r.height - 1;
    int i = (r.height & 1) ^ 1;
    for (int x = r.x; x <= maxX; x+=2) {
      g.drawLine(x,r.y,x,r.y);
      if (x+i <= maxX) g.drawLine(x+i,maxY,x+i,maxY);
    }
    i = (r.width & 1) ^ 1;
    for (int y = r.y + 2; y <= maxY; y+= 2) {
      if (y < maxY) g.drawLine(r.x,y,r.x,y);
      if (y-i < maxY) g.drawLine(maxX,y-i,maxX,y-i);
    }
  }

  /**
   * Draws a dotted horizontal line in the current color.
   *
   * @param g The graphics context on which to draw the line
   * @param x The x coordinate of the start of the line
   * @param y The y coordinate of the start of the line
   * @param width The width of the line
   */
  public static void drawDottedHorizontal(Graphics g, int x, int y, int width) {
    width += x;
    for (; x < width; x += 2)
      g.drawLine(x,y,x,y);
  }

  /**
   * Draws a dotted vertical line in the current color.
   *
   * @param g The graphics context on which to draw the line
   * @param x The x coordinate of the start of the line
   * @param y The y coordinate of the start of the line
   * @param height The height of the line
   */
  public static void drawDottedVertical(Graphics g, int x, int y, int height) {
    height += y;
    for (; y < height; y += 2)
      g.drawLine(x,y,x,y);
  }

  /**
   * Shows a Drop-down or Pop-up window for a control.
   *
   * @param owner The owner control of the window
   * @param window The drop-down or pop-up window
   */
  protected static final void showDropWindow(BaseControl owner,
    Window value)
  {
    if (dropWindow != value || dropWindowOwner != owner) {
      hideDropWindow();
      dropWindow = value;
      dropWindowOwner = owner;
      dropWindow.setVisible(true);
    }
  }

  /**
   * Hides the currently displayed drop-down or pop-up window if any.
   */
  protected static final void hideDropWindow() {
    if (dropWindowOwner != null) {
      dropWindowOwner.dropWindowHidden();
      dropWindow.setVisible(false);
      dropWindow = null;
      dropWindowOwner = null;
    }
  }

  /**
   * Finds the parent Window of a component.
   *
   * @param component The component for which to find the parent Window
   * @return The parent Window or null if none found
   */
  public static Window findParentWindow(Component component) {
    return component == null | (component instanceof Window) ?
      (Window)component : findParentWindow(component.getParent());
  }

  /**
   * Finds the parent Frame of a component.
   *
   * @param component The component for which to find the parent Frame
   * @return The parent Frame or null if none found
   */
  public static Frame findParentFrame(Component component) {
    return component == null | (component instanceof Frame) ?
      (Frame)component : findParentFrame(component.getParent());
  }

  /**
   * Finds a parent of a Component of the specified class.
   *
   * @param cl The class of the parent
   * @param component The component for which to find the parent
   * @return The parent or null
   */
  public static Component findParent(Class cl, Component component) {
    return component == null | cl.isInstance(component) ?
      component : findParent(cl,component.getParent());
  }

  /**
   * Gets the default DecimalFormat instance used to format decimal numbers.
   *
   * @return The default DecimalFormat instance
   */
  public static DecimalFormat getDecimalformat() {
    return decimalFormat;
  }

  /**
   * Gets the default DateFormat instance used to format Date instances as a
   * Date.
   *
   * @return The default DateFormat instance for formatting a Date
   */
  public static DateFormat getDateFormat() {
    return dateFormat;
  }

  /**
   * Gets the default DateFormat instance used to format Date instances as a
   * time.
   *
   * @return The default DateFormat instance for formatting a time
   */
  public static DateFormat getTimeFormat() {
    return timeFormat;
  }

  /**
   * Gets the default DecimalFormatSymbols instance.
   *
   * @return The default DecimalFormatSymbols instance
   */
  public static DecimalFormatSymbols getDecimalFormatSymbols() {
    return decimalFormatSymbols;
  }

  /**
   * Creates a GreyScale version of an Image for a Component.
   *
   * @param source The source Image
   * @param component The component to use to create the new Image
   * @return A GreyScale version of the source Image
   */
  public static Image greyImage(Image source, Component component) {
    return greyImage(source,component,-1,0);
  }

  /**
   * Hashtable used to cache GreyScale versions of Image instances.
   */
  protected static Hashtable greyCache = new Hashtable();

  /**
   * Creates a GreyScale version of an Image for a Component.
   *
   * @param source The source Image
   * @param component The component to use to create the new Image
   * @param baseGrey The base RGB value for the grey shade to use. If this is -1
   *                 the result will be a black and white version of the Image
   *                 using all possible shades of grey
   * @param percent The percentage deviation allowed from the base grey shade
   * @return A GreyScale version of the source Image
   */
  public static Image greyImage(Image source, Component component, int baseGrey,
    int percent)
  {
    if (source == null)
      return null;

    Image result = (Image)greyCache.get(source);
    if (result != null)
      return result;

    int iw = source.getWidth(null);
    int ih = source.getHeight(null);
    if (iw <= 0 || ih <= 0)
      return null;

    int sz = iw * ih;
    int[] pixels = new int[sz];
    PixelGrabber grabber = new PixelGrabber(source,0,0,iw,ih,pixels,0,iw);
    try {
      grabber.grabPixels();
    } catch (Exception e) {
      e.printStackTrace();
    }
    for (int i = 0; i < sz; i++) {
      int rgb = pixels[i];
      int alpha = rgb & 0xFF000000;
      int rv = (rgb >> 16) & 0xFF;
      int gv = (rgb >> 8) & 0xFF;
      int bv = rgb & 0xFF;
      int max = Math.max(bv,Math.max(rv,gv));
      int min = Math.min(bv,Math.min(rv,gv));
      rgb = (min + max) / 2;
      if (baseGrey != -1)
        rgb = Math.min(255,Math.max(0,baseGrey - (baseGrey - rgb) * percent / 100));
      pixels[i] = alpha | (rgb << 16) | (rgb << 8) | rgb;
    }
    result = component.createImage(new MemoryImageSource(iw,ih,pixels,0,iw));
    greyCache.put(source,result);
    return result;
  }

  /**
   * Converts a Point in a child Component to parent coordinates.
   *
   * @param source The source Component to convert the coordinates from
   * @param parent The parent Container to convert the coordinates to
   * @param p The Point to be converted. This Point is modified.
   * @return The converted Point in parent coordinates
   */
  public static Point convertToParent(Component source, Container parent,
    Point p)
  {
    if (source != parent) {
      Point where = source.getLocation();
      p.x += where.x;
      p.y += where.y;
      convertToParent(source.getParent(),parent,p);
    }
    return p;
  }

  /**
   * Converts a Point in a parent Container to child coordinates.
   *
   * @param source The source Component to convert the coordinates to
   * @param parent The parent Container to convert the coordinates from
   * @param p The Point to be converted. This Point is modified.
   * @return The converted Point in child coordinates
   */
  public static Point convertFromParent(Component source, Container parent,
    Point p)
  {
    if (source != parent) {
      Point where = source.getLocation();
      p.x -= where.x;
      p.y -= where.y;
      convertFromParent(source.getParent(),parent,p);
    }
    return p;
  }

  /**
   * Sets the default Font for all controls in the application or applet.
   *
   * @param value The default Font
   */
  public static void setDefaultFont(Font value) {
    defaultFont = value;
  }

  /**
   * Gets the default Font for all controls.
   *
   * @return The default Font
   */
  public static Font getDefaultFont() {
    return defaultFont;
  }

  /**
   * Finds the top-level parent of a control which is a BaseControl instance.
   * This control is used for double-buffering, and for drag and drop rendering.
   *
   * @param source The source control for which to find the parent
   * @return The top-level BaseControl parent instance of the source
   */
  public static BaseControl findTopBaseControl(BaseControl source) {
    Container parent = source.getParent();
    return parent instanceof BaseControl ?
      findTopBaseControl((BaseControl)parent) : source;
  }

  /**
   * Finds a component at the given screen co-ordinates within the given parent.
   *
   * @param source The parent component in which to find the component
   * @param where The Point on the screen at which to find the child
   * @return The child component at the given Point on the screen or null if
   *         none found
   */
  public static Component findComponentAt(Component source, Point where) {
    Component result = null;
    if (source.isVisible()) {
      Point pos = source.getLocationOnScreen();
      if (where.x >= pos.x && where.y >= pos.y) {
        Dimension size = source.getSize();
        if (where.x < pos.x + size.width && where.y < pos.y + size.height) {
          if (source instanceof Container) {
            Component[] components = ((Container)source).getComponents();
            for (int i = 0; i < components.length; i++) {
              result = findComponentAt(components[i],where);
              if (result != null)
                break;
            }
          }
          if (result == null)
            result = source;
        }
      }
    }
    return result;
  }

  /**
   * Invalidates a Component and all child Components, validates and repaints.
   *
   * @param parent The top-level Component to invalidate
   */
  public static void revalidate(Component parent) {
    if (parent != null) {
      invalidateChildren(parent);
      validateChildren(parent);
      parent.repaint();
    }
  }

  /**
   * Invalidates a Component and all child Components.
   *
   * @param parent The top-level Component to invalidate
   */
  protected static void invalidateChildren(Component parent) {
    parent.invalidate();
    if (parent instanceof Container) {
      Component[] children = ((Container)parent).getComponents();
      for (int i = 0; i < children.length; i++)
        invalidateChildren(children[i]);
    }
  }

  /**
   * Invalidates a Component and all child Components.
   *
   * @param parent The top-level Component to invalidate
   */
  protected static void validateChildren(Component parent) {
    if (parent instanceof Container) {
      Component[] children = ((Container)parent).getComponents();
      for (int i = 0; i < children.length; i++)
        validateChildren(children[i]);
    }
    parent.validate();
  }

  public static void drawPolyline(Graphics g, int[] xPoints, int[] yPoints,
    int nPoints)
  {
    if (nPoints == 1)
      g.fillRect(xPoints[0],yPoints[0],1,1);
    else
      for (int i = 0; i < nPoints - 1; i++)
        g.drawLine(xPoints[i],yPoints[i],xPoints[i + 1],yPoints[i + 1]);
  }

}
