#include <stdint.h>
#include <string.h>

/*
一、录音磁带的格式
波形图见
数据磁带机.png
LASER310_cas_0xAA_Figure.png

读入的字节依次为：
同步和前导：128个 0x80 5个 0xFE
程序类型：1字节 TYPE
程序文件名：最多17字节 FILENAME，空文件为1个0x00
静音：约0.003s。程序在此期间显示文件名
程序开始： 2字节 START
程序结束：2字节 END
程序数据：
求和校验值：2字节
结尾：0x00 1个字节, 没有该字节不影响读取

在读入文件名后，约有0.003秒的静音。这段时间应该是留给程序显示文件名用的。
长度约 555us x 5.5


录音末尾是求和校验值
计算方法是 0xFF00 或 0x0000 、文件开始（高低字节）、文件结尾（高低字节）、数据，求和后取低16位。
LASER310 是 0x0000
另外 LASER310 尾部加了 20个 0x00
尾部 0x00 没有也能正确读入。

二、vz 文件格式
VZF_MAGIC : 4个字节 0x20 0x20 0x00 0x00 或者是 0x56 0x5A 0x46 0x30
vzf_filename : 17个字节 最后一个字节必须为零，末尾不足的字符用空格0x20代替
vzf_type : 1个字节 同磁带
vzf_startaddr : 2个字节 程序开始地址
之后是数据
*/

extern uint8_t cas_buf[1024*64];
extern int cas_dat_len;
extern int cas_fn_len;

extern uint64_t cas_ticks;
extern int cas_step;

int cas_bit_0[6] = {1,0,1,1,0,0};
int cas_bit_1[6] = {1,0,1,0,1,0};

void cas_set_data(uint8_t* cas_name, uint8_t cas_type, uint16_t cas_start, uint16_t cas_len, uint8_t *cas_dat)
{
	int c=0;
	uint8_t ch;
	uint16_t sum;

	memset(cas_buf, 0, 1024*64);
	// 同步
	memset(cas_buf+c,	0x80, 128);
	c+=128;
	memset(cas_buf+c,	0xFE, 5);
	c+=5;

	// 类型
	cas_buf[c] = cas_type;
	c+=1;

	// 名称
	cas_fn_len=0;
	memset(cas_buf+c,	0x20, 17);
	for(int i=0; i<17; i++) {
		ch = cas_name[i];
		cas_buf[c+i] = ch;
		cas_fn_len++;
		if(ch==0) break;
	}
	c+=cas_fn_len;

	sum = 0;
	// start
	*(uint16_t*)(cas_buf+c) = cas_start;
	sum+=cas_buf[c];
	sum+=cas_buf[c+1];
	c+=2;

	// end
	*(uint16_t*)(cas_buf+c) = cas_start+cas_len;
	sum+=cas_buf[c];
	sum+=cas_buf[c+1];
	c+=2;
	
	// data
	memcpy(cas_buf+c, cas_dat, cas_len);
	for(int i=0;i<cas_len;i++)
		sum += cas_buf[c+i];
	c+=cas_len;
	
	// sum
	*(uint16_t*)(cas_buf+c) = sum;
	c+=2;

	// 结尾
	cas_buf[c] = 0;
	c+=1;

	cas_dat_len = cas_len+2;
}

// 时钟 17745000/5 = 3549*1000
// 277us 555us 时钟数
// 277*10^-6 * 3549*1000 = 277*3549 * 10*-3 = 983
// 每秒 3549*1000 / 983 = 3610
// 每个位 555*3 = 277*6
// 每个字节 6 * 8

// CLK 20*14*2/1000 us
// 一个方波 277 us
// 50MHZ / 28
// 277 / (20*14*2/1000) = 277 * 1000 / 560 = 494.6

int cas_read(uint64_t ticks)
{
	if(ticks==0) return 0;
	int t = ticks/983;
	int pos, b_pos;
	uint8_t ch, b;

	cas_step=0;

	// 空白2秒钟左右
	if( t<7000 ) return 0;
	t-=7000;
	// 同步 类型 文件名 128 + 5 + 1 + cas_fn_len
	if(t<48*(128+5+1+cas_fn_len)) {
		pos = t/48;
		b_pos = t%48;
		goto read_cas_buf;
	}
	t-=48*(128+5+1+cas_fn_len);
	// 静音
	if(t<11) return 0;
	t-=11;
	// 开始 结束 数据
	if(t<48*(2+2+cas_dat_len)) {
		pos = t/48+(128+5+1+cas_fn_len);
		b_pos = t%48;
		goto read_cas_buf;
	}

	// 播放结束
	cas_step=100;
	cas_ticks = 0;
	return 0;

read_cas_buf:
	cas_step=pos*100/(128+5+1+cas_fn_len+cas_dat_len);

	ch = cas_buf[pos];
	b = ch<<(b_pos/6);
	//b = ch>>(b_pos/6);

	//if(b)
	if(b&0x80)
		return cas_bit_1[b_pos%6];
	else
		return cas_bit_0[b_pos%6];
}
