#include <stdbool.h>
#include <stdint.h>
#include <string.h>
#include <stdio.h>

#include <SDL2/SDL.h>

#ifdef _WIN32
#include "stb/stb_ime.h"

// https://github.com/Davipb/utf8-utf16-converter
//#include "libs/converter.h"

#include <stdlib.h>
#include <wchar.h>
#include "charlib/wchar2char.h"

#endif

#include "stb/bithacks.h"


//#define SDL_RDR_SOFTWARE

//#define ENABLE_DBG
//#include "stb/dbg.h"
//#define ENABLE_LOG
#include "stb/log.h"

#include "vzcontext.h"
#include "vkey.h"

uint64_t ticks = 0;  // accumulated number of clock cycles

// 模拟磁带计数
uint8_t cas_buf[1024*64];
int cas_dat_len=0;
int cas_fn_len=0;

uint64_t cas_ticks = 0;
int cas_step=0;

// 写入磁带时，加速执行标志
int skip_cycles=0;


// 光笔 LIGHT PEN
// 光笔笔尖开关压下
int light_pen_btn=0;


#ifdef ENABLE_LOG
#define DASM_Z80
#endif

int debug=0;

#include "stb/stb_file.h"
#include "stb/filehelp.h"

#include "zel/z80_instructions.h"

#define SCREEN_RES_W	256
#define SCREEN_RES_H	192
#define SCREEN_BPP		8

// memory layout
#define ROMSIZE			0x4000
#define DOSROMSIZE		0x2000
#define FONTROMSIZE		0x0C00

VZCONTEXT	vzcontext;

uint32_t vkey_delay, vkey_enter_delay;	// 中断计时，虚拟按键计时
int vkey_keydown;

uint8_t		fn[512];
uint8_t		vzbuf[0x10000];
uint8_t		vzfn[18];

uint8_t		vz_name[18];
uint8_t		vz_type;
uint16_t	vz_start;
uint16_t	vz_len;
uint8_t		vz_dat[0x10000];

#ifdef SYS_LASER310
#include "rom/laser310.h"
#endif

#ifdef SYS_LASER200
#include "rom/laser200.h"
#endif

#ifdef SYS_VZ200
#include "rom/vz200.h"
#endif

#include "emu.h"

#include "vz.h"
#include "cas.h"

char title[1000];// UPDATE WINDOW TITLE

uint8_t inst_pc[0x10000];

#define	CLIP_MAXLEN	(32*1024)

#ifdef WIN32

#include <windows.h>
// 从剪切板中取得数据
BOOL GetTextFromClipboard(uint8_t* buf)
{
	buf[0] = 0;

	if(OpenClipboard(NULL)) {
		// 获得剪贴板数据
		HGLOBAL hMem = GetClipboardData(CF_TEXT);
		if(NULL != hMem) {
			char* lpStr = (char*)GlobalLock(hMem); 
			if(NULL != lpStr) {
				//MessageBoxA(0, lpStr, "Clipboard", MB_OK);
				strcpy_s((void*)buf, CLIP_MAXLEN, lpStr);
				//strcpy(buf, lpStr);
				GlobalUnlock(hMem);
			}
		}
		CloseClipboard();
		return TRUE;
	}
	return FALSE;
}

#endif	// WIN32

//================================================================ SOFT SWITCHES


//====================================================================== SPEAKER

#define audioBufferSize 4096													// found to be large enought
Sint8 audioBuffer[2][audioBufferSize] = { 0 };									// see in main() for more details
SDL_AudioDeviceID audioDevice;
bool muted = false;// mute/unmute switch

void playSound() {
	static uint64_t lastTick = 0LL;
	static bool SPKR = false;													// $C030 Speaker toggle

	if (!muted) {
		SPKR = !SPKR;// toggle speaker state
		Uint32 length = (int)((double)(ticks - lastTick) / 36.96875f);			// 17745000Hz/5/96000Hz = 36.96875
		lastTick = ticks;
		if (length > audioBufferSize) length = audioBufferSize;
		SDL_QueueAudio(audioDevice, audioBuffer[SPKR], length | 1);					// | 1 TO HEAR HIGH FREQ SOUNDS
	}
}

//====================================================================== DISK ][

void sys_reset()
{
	memset(vzcontext.memory+0x7000, 0x7f, 0x9000);
	memset(vzcontext.vram, 0x7f, 1024*8);
	memset(vzcontext.scancode, 0x00, sizeof(vzcontext.scancode)/sizeof(uint8_t));
	memset(vzcontext.vscancode, 0x00, sizeof(vzcontext.vscancode)/sizeof(uint8_t));

	vzcontext.latched_ga = 0xff;
	vzcontext.latched_shrg = 0x08;
	//vzcontext.latched_shrg = 0x00;

	/* Emulate. */
	Z80Reset(&vzcontext.state);
	vzcontext.state.pc = 0x0000;

	memset(inst_pc, 0x01, 0x10000);
}

int emu_vkey_keyup()
{
	memset(vzcontext.vscancode, 0x00, sizeof(vzcontext.vscancode)/sizeof(uint8_t));

	if(vzcontext.vkey_len==0) return 0;
	vzcontext.vkey_cur++;
	if(vzcontext.vkey_cur>=vzcontext.vkey_len) {
		 vzcontext.vkey_len=0;
		 vzcontext.vkey_cur=0;
	}

	return 0;
}

int emu_vkey_keydown()
{
	if(vzcontext.vkey_len==0) return 0;

	uint8_t ch = vzcontext.vkey[vzcontext.vkey_cur];
	if(vkey[ch].i==-1) {
		vzcontext.vkey_cur++;
		return 0;
	}

	if(vkey[ch].shift)
		B_SET(vzcontext.vscancode[2],2);
	B_SET(vzcontext.vscancode[vkey[ch].i],vkey[ch].j);

	return 1;
}

//========================================== MEMORY MAPPED SOFT SWITCHES HANDLER

//======================================================================= MEMORY

uint8_t read_byte_NoAct(uint16_t adr, void *context)
{
	uint8_t x;
	x = ((VZCONTEXT *)context)->memory[adr&0xffff];
	return x;
}

int Z80_Disassemble( uint16_t address, char *buffer, void *data )
{
	Instruction inst;
	int length = IF_ID( &inst, address, read_byte_NoAct, data );
	if( buffer != NULL )
		DisassembleInstruction( &inst, buffer );
	return length;
}

unsigned int cycles_count;

void CpuExec(uint64_t cycleCount)
{
	//unsigned int cycles_count=0;
	unsigned int cycles=0;

#ifdef DASM_Z80
	char disasm[25];
	char disasm_bytes[16];
	int disasm_n;
#endif
	uint16_t PC;
	int flen;

	cycles_count=0;

	while(cycles_count<cycleCount) {
		PC = vzcontext.state.pc;

#ifdef DASM_Z80
		//if(!debug && PC==0x7B80) debug = 1;	//
		//if(!debug && PC==0x7AE9) debug = 1;	//
		if(!debug && PC==0x7B12) debug = 1;		// START OF LIGHT PEN ROUTINE
		//if(!debug && PC==0x7B7C) debug = 1;
		//if(!debug && PC==0x3656) debug = 1;	// CLOAD
		//if(!debug && PC==0x34A9) debug = 1;	// CSAVE

		//if(debug && PC>=0x4000) {
		if(debug) {
			if(inst_pc[PC]) {
			//if(1) {
				inst_pc[PC]=0;
				disasm_n = Z80_Disassemble(PC, disasm, &vzcontext);
				LOG("%5d %04X: [%04X %04X %04X %04X] %s\n", cycles_count, PC,
				vzcontext.state.registers.word[Z80_AF], vzcontext.state.registers.word[Z80_BC],
				vzcontext.state.registers.word[Z80_DE], vzcontext.state.registers.word[Z80_HL],
				disasm);
/*
				LOG("%04X: [%04X %04X %04X %04X] %s\n", PC,
				vzcontext.state.registers.word[Z80_AF], vzcontext.state.registers.word[Z80_BC],
				vzcontext.state.registers.word[Z80_DE], vzcontext.state.registers.word[Z80_HL],
				disasm);
*/
			}
		}
#endif

		// 把写入磁带的内容直接写为VZ文件
		// 地址3575 是 CSAVE 里写文件类型的位置。
		if(PC==0x3575) {
			memset(vzbuf, 0, 0x10000);
			// LASER310
			vz_type = vzcontext.state.registers.byte[Z80_C];
			flen = vz_save(&vzcontext, vz_type, vzfn, vzbuf);
			if(flen>0) {
				if(*vzfn) {
					sprintf((char*)fn, "%s.VZ", vzfn);
				} else {
					for(int i=1;i<1000;i++) {
						sprintf((char*)fn, "%03d.VZ", i);
						if(!fexist((const char*)fn)) break;
					}
				}
				fwrite_buf_bin((const char*)fn, vzbuf, flen);
			}
		}

		cycles = z80emulate(&vzcontext.state, 0, &vzcontext);

		// 加速程序执行 34C2 到 3508 磁带写入，不计入程序执行时间
		if(PC==0x34C2) skip_cycles=1;	// 开始加速，停止计时
		if(PC==0x3508) skip_cycles=0;	// 结束加速，开始计时
		if(skip_cycles) continue;

		cycles_count += cycles;
		ticks += cycles;
	}

	// Z80_INTERRUPT_MODE_1 : RTS 38H
	cycles = Z80Interrupt(&vzcontext.state, 0, &vzcontext);
}

void SysInit()
{
	skip_cycles=0;
}

void SysReset()
{
	skip_cycles=0;

	sys_reset();

	// reset the CPU
}

//=======================================================================
/*
int readBinFile(uint8_t *buf, uint32_t sz, char *fn, char* workdir, int wd_len)
{
	int r = 0;
	FILE *f;
	char msg[512];
	workdir[wd_len] = 0;

	f = fopen(strncat(workdir, fn, 256), "rb");				// load the file

	if (f) {
		if (fread(buf, 1, sz, f) != sz) {							// the file is too small
			sprintf(msg, "%s should be exactly %d %s", fn, (sz>=1024)?sz/1024:sz, (sz>=1024)?"KB":"bytes");
			r=1;
		}
		fclose(f);
	} else {
		sprintf(msg, "Could not locate %s in the rom folder", fn);
		r=2;
	}

	if(r)
		SDL_ShowSimpleMessageBox(SDL_MESSAGEBOX_ERROR, "Fatal error", msg, NULL);

	return r;
}
*/

//========================================================== PROGRAM ENTRY POINT

int main(int argc, char *argv[]) {
	int i, j;
	//========================================================= SDL INITIALIZATION
	int zoom = 2;
	int fullscreen = 0;

	size_t fileSize;

	SDL_Event event;
	SDL_bool running = true, paused = false, ctrl = false, shift = false, alt = false;

#ifdef LOADDSK
	fullscreen = 1;
#endif
	if (SDL_Init(SDL_INIT_VIDEO | SDL_INIT_AUDIO) < 0) {
		printf("failed to initialize SDL2 : %s", SDL_GetError());
		return -1;
	}

	//SDL_Window *wdo = SDL_CreateWindow("Reinette ][+", SDL_WINDOWPOS_CENTERED, SDL_WINDOWPOS_CENTERED, SCREEN_RES_W * zoom, SCREEN_RES_H * zoom, SDL_WINDOW_OPENGL);
	SDL_Window *wdo = SDL_CreateWindow("LASER-310", SDL_WINDOWPOS_CENTERED, SDL_WINDOWPOS_CENTERED, SCREEN_RES_W * zoom, SCREEN_RES_H * zoom, SDL_WINDOW_RESIZABLE);

#ifdef _WIN32
	SDL_IME_Disable(wdo);
#endif

#ifdef SDL_RDR_SOFTWARE
	SDL_Renderer *rdr = SDL_CreateRenderer(wdo, -1, SDL_RENDERER_SOFTWARE);
	//SDL_Renderer *rdr = SDL_CreateRenderer(wdo, -1, SDL_RENDERER_SOFTWARE | SDL_RENDERER_PRESENTVSYNC);	// SDL_RENDERER_PRESENTVSYNC 无效
#else
	SDL_Renderer *rdr = SDL_CreateRenderer(wdo, -1, SDL_RENDERER_ACCELERATED);
	//SDL_Renderer *rdr = SDL_CreateRenderer(wdo, -1, SDL_RENDERER_ACCELERATED | SDL_RENDERER_PRESENTVSYNC);
#endif
	SDL_EventState(SDL_DROPFILE, SDL_ENABLE);									// ask SDL2 to read dropfile events
	SDL_SetWindowMinimumSize(wdo, SCREEN_RES_W, SCREEN_RES_H);
	if(fullscreen) SDL_SetWindowFullscreen(wdo, SDL_WINDOW_FULLSCREEN_DESKTOP);

	uint8_t screenData[SCREEN_RES_W*SCREEN_RES_H];
	SDL_Surface *sdlSurface = SDL_CreateRGBSurface(0, SCREEN_RES_W, SCREEN_RES_H, 32, 0,0,0,0);
	SDL_Surface *sdlScreen = SDL_CreateRGBSurfaceFrom((void*)screenData, SCREEN_RES_W, SCREEN_RES_H, SCREEN_BPP, SCREEN_RES_W*1, 0, 0, 0, 0);

	SDL_Color colors[16];

/*
3'b000	24'h07ff00 : // GREEN
3'b001	24'hffff00 : // YELLOW
3'b010	24'h3b08ff : // BLUE
3'b011	24'hcc003b : // RED
3'b100	24'hffffff : // BUFF
3'b101	24'h07e399 : // CYAN
3'b110	24'hff1cff : // MAGENTA
3'b111	24'hff8100 ; // ORANGE

border		24'h000000
background	24'h07ff00
*/

	colors[0].r = 0x07;	colors[0].g = 0xff;	colors[0].b = 0x00; colors[0].a = 0xff;
	colors[1].r = 0xff;	colors[1].g = 0xff;	colors[1].b = 0x00; colors[1].a = 0xff;
	colors[2].r = 0x3b;	colors[2].g = 0x08;	colors[2].b = 0xff; colors[2].a = 0xff;
	colors[3].r = 0xcc;	colors[3].g = 0x00;	colors[3].b = 0x3b; colors[3].a = 0xff;
	colors[4].r = 0xff;	colors[4].g = 0xff;	colors[4].b = 0xff; colors[4].a = 0xff;
	colors[5].r = 0x07;	colors[5].g = 0xe3;	colors[5].b = 0x99; colors[5].a = 0xff;
	colors[6].r = 0xff;	colors[6].g = 0x1c;	colors[6].b = 0xff; colors[6].a = 0xff;
	colors[7].r = 0xff;	colors[7].g = 0x81;	colors[7].b = 0x00; colors[7].a = 0xff;

	colors[8].r = 0x00;	colors[8].g = 0x00;	colors[8].b = 0x00; colors[8].a = 0xff;
	colors[9].r = 0x07;	colors[9].g = 0xff;	colors[9].b = 0x00; colors[9].a = 0xff;

	SDL_SetPaletteColors(sdlScreen->format->palette, colors, 0, 10);

	//=================================================== SDL AUDIO INITIALIZATION

	SDL_AudioSpec desired = { 96000, AUDIO_S8, 1, 0, 4096, 0, 0, NULL, NULL };
	audioDevice = SDL_OpenAudioDevice(NULL, 0, &desired, NULL, SDL_FALSE);		// get the audio device ID
	SDL_PauseAudioDevice(audioDevice, muted);									// unmute it (muted is false)
	uint8_t volume = 4;

	for (int i = 0; i < audioBufferSize; i++) {									// two audio buffers,
		audioBuffer[true][i] = volume;											// one used when SPKR is true
		audioBuffer[false][i] = -volume;										// the other when SPKR is false
	}

	//===================================== VARIABLES USED IN THE VIDEO PRODUCTION
	uint8_t flashCycle = 0;
	//================================= LOAD NORMAL AND REVERSE CHARACTERS BITMAPS
/*
	char workDir[1000];															// find the working directory
	int workDirSize = 0;
	i = 0;
	while (argv[0][i] != '\0') {
		workDir[i] = argv[0][i];
		if (argv[0][++i] == '\\') workDirSize = i + 1;							// find the last '/' if any
	}
*/
	//================================================================== LOAD ROMS

/*
	if(readBinFile(rom,		ROMSIZE,	 "roms/apple2/appleII+.rom", workDir, workDirSize)) return 1;
*/

#ifdef SYS_LASER310
	emu_init(fontrom, laser310_rom_2_1, laser310_dosrom_1_2patch);
#endif

#ifdef SYS_LASER200
	emu_init(fontrom, laser200_rom, laser310_dosrom_1_2patch);
#endif

#ifdef SYS_VZ200
	emu_init(fontrom, vz200_rom, laser310_dosrom_1_2patch);
#endif

	memset(inst_pc, 0x01, 0x10000);

	SysInit();

	//========================================================== VM INITIALIZATION

	//================================================================== MAIN LOOP
	vkey_delay=0;
	vkey_enter_delay=0;
	vkey_keydown=0;

	uint64_t ticks_step=1;
	uint64_t last_ticks = SDL_GetTicks64();
	uint64_t current_ticks;

	uint32_t screen_refresh_cnt =0;

	while (running) {

		// 逐行PAL制式 逐行NTSC制式
		// 隔行NTSC制 帧速29.97fps（帧/秒），每帧525行262线 分辨率为720×480
		// 隔行PAL制 帧速25fps，每帧625行312线 分辨率为720×576
		// 逐行PAL制 312行、50帧/秒的PAL制式

		// 50*312*227.5*5 = 17745000
		// 17745000/5/1000 = 3549
		//const uint64_t cycles_1ms = 3549; // 每毫秒执行CPU时钟数
		// 17745000/5/1000*20 = 3549
	
		if (!paused) {
			// execute instructions for 1/50 of a second
			for(int i=0;i<10;i++) {
				CpuExec(70980);
				//emu_video_retrace(&vzcontext);

				// 处理虚拟按键 约每秒 7 个字符
				// 回车键间隔稍长些，便于LASER310系统处理输入
				if(vkey_enter_delay>0) {
					vkey_enter_delay--;
				} else if(vkey_delay>0) {
					vkey_delay--;
					if(vkey_delay==0&&vkey_keydown) {
						vkey_keydown = emu_vkey_keyup();
						if(vzcontext.vkey_cur==vzcontext.vkey_len) {
							vzcontext.vkey_len=0;
							vzcontext.vkey_cur=0;
						}
					}
				} else {
					if((vkey_keydown==0) && vzcontext.vkey_len>0 ) {
						if(vzcontext.vkey[vzcontext.vkey_cur-1]=='\n') {
							vkey_enter_delay = 10;
							vkey_delay = 7;
						} else {
							vkey_enter_delay = 0;
							vkey_delay = 10;
						}

						vkey_keydown = emu_vkey_keydown();
						//LOG("vkey down %d\n",vzcontext.vkey_cur);
					}
				}

				if(!cas_ticks) break;	// 装载磁带时速度提高10倍
			}
		}

		//=============================================================== USER INPUT

		while(1) {
			while (SDL_PollEvent(&event)) {
				alt	  = SDL_GetModState() & KMOD_ALT   ? true : false;
				ctrl  = SDL_GetModState() & KMOD_CTRL  ? true : false;
				shift = SDL_GetModState() & KMOD_SHIFT ? true : false;

				if (event.type == SDL_QUIT) running = false;					// WM sent TERM signal

#ifdef _WIN32
				if (event.type == SDL_WINDOW_INPUT_FOCUS) SDL_IME_Disable(wdo);
#endif

				if (event.type == SDL_DROPFILE) {								// user dropped a file
					char *filename = event.drop.file;							// get full pathname (UTF-8)
					//LOG("SDL_DROPFILE %s\n",filename);

#ifdef _WIN32
					char* fn = Utf8ToLoc(filename);
					fileSize = fn_readfile(fn,vzbuf);
					free(fn);
#else
					fileSize = fn_readfile(filename,vzbuf);
#endif

					if(fileSize&&vz_parse(vzbuf, fileSize, vz_name, &vz_type, &vz_start, &vz_len, vz_dat)) {
						// 0x1A33 0x040C
						//while(vzcontext.state.pc!=0x040C)
						//	z80emulate(&vzcontext.state, 0, &vzcontext);

						strcpy((char*)vzcontext.vkey,"CLOAD\n");
						vzcontext.vkey_cur = 0;
						vzcontext.vkey_len = 6;

						// 开始播放磁带数据
						cas_set_data(vz_name, vz_type, vz_start, vz_len, vz_dat);
						cas_ticks = ticks;
						//debug = 1;
						//vz_load(&vzcontext, vz_type, vz_start, vz_len, vz_dat);
					}

					SDL_free(filename);											// free filename memory
					paused = false;												// might already be the case

					if (!(alt || ctrl)) {										// if ALT or CTRL were not pressed
						//SysReset();											// do a cold reset
					}
				}

#ifdef WIN32
				if (event.type == SDL_MOUSEBUTTONDOWN) {
					if(event.button.button==SDL_BUTTON_RIGHT) {
						if(vzcontext.vkey_len==0) {
							GetTextFromClipboard(vzcontext.vkey);
							vzcontext.vkey_cur = 0;
							vzcontext.vkey_len = strlen((char*)vzcontext.vkey);
						}
					}
				}
#endif	// WIN32


				if (event.type == SDL_MOUSEBUTTONDOWN) {
					if(event.button.button==SDL_BUTTON_LEFT) {
						light_pen_btn = 1;
					}
				}

				if (event.type == SDL_MOUSEBUTTONUP) {
					if(event.button.button==SDL_BUTTON_LEFT) {
						light_pen_btn = 0;
					}
				}
/*
				if (event.type == SDL_MOUSEMOTION) {
					Sint32 x = event.motion.x;
					Sint32 y = event.motion.y;

					// 转换坐标到
					//#define SCREEN_RES_W	256
					//#define SCREEN_RES_H	192
				}
*/
				if (event.type == SDL_KEYDOWN) {								// a key has been pressed
					if(key_map(&event.key.keysym, &i,&j)) B_SET(vzcontext.scancode[i],j);

					switch (event.key.keysym.sym) {

					// EMULATOR CONTROLS :

					case SDLK_F1:												// help box
						if(fullscreen) {SDL_SetWindowFullscreen(wdo, 0); fullscreen=0;}
						SDL_ShowSimpleMessageBox(SDL_MESSAGEBOX_INFORMATION, "Help",
							"\tReinette ][ plus v0.4.8\n"
							"\n"
							"F1\tthis help\n"
							"\n"
							"ctrl F12\treset\n"
							"\n"
							"More information at github.com/ArthurFerreira2\n", NULL);
							{ticks_step=1;last_ticks=SDL_GetTicks64();}
					break;

/*
					case SDLK_F2:												// help box
						strcpy(vzcontext.vkey,"RUN\n");
						vzcontext.vkey_cur = 0;
						vzcontext.vkey_len = 4;
						//LOG("vkey %d %d\n",vzcontext.vkey_cur, vzcontext.vkey_len);
					break;
*/

					case SDLK_F4:												// VOLUME
						if (shift && (volume < 120)) volume++;					// increase volume
						if (ctrl && (volume > 0)) volume--;						// decrease volume
						if (!ctrl && !shift) muted = !muted;					// toggle mute / unmute
						for (int i = 0; i < audioBufferSize; i++) {				// update the audio buffers,
							audioBuffer[true][i] = volume;						// one used when SPKR is true
							audioBuffer[false][i] = -volume;					// the other when SPKR is false
						}
					break;

					case SDLK_F7:												// ZOOM
						if (!ctrl && !shift) {
							fullscreen = fullscreen?0:1;
							SDL_SetWindowFullscreen(wdo, fullscreen?SDL_WINDOW_FULLSCREEN_DESKTOP:0);
						}
						if (!fullscreen) {
							if(ctrl && (zoom>1)) {	// zoom out
								zoom--;
								SDL_SetWindowSize(wdo, SCREEN_RES_W * zoom, SCREEN_RES_H * zoom);
							}
							if(shift && (zoom<4)) {	// zoom in
								zoom++;
								SDL_SetWindowSize(wdo, SCREEN_RES_W * zoom, SCREEN_RES_H * zoom);
							}
						}
						break;


					case SDLK_F10: debug = debug?0:1;break;

					case SDLK_F11: paused = !paused; if(!paused){ticks_step=1;last_ticks=SDL_GetTicks64();} break;									// toggle pause

					case SDLK_F12: if (ctrl) SysReset(); break;					// simulate a reset

					// EMULATED KEYS :
					}
				}
				
				if (event.type == SDL_KEYUP) {
					if(key_map(&event.key.keysym, &i,&j)) B_UNSET(vzcontext.scancode[i],j);
				}

			}

			current_ticks = SDL_GetTicks64();
			if( current_ticks-last_ticks > ticks_step*20 ) {		// ticks_step*1000/50 == ticks_step*20
				ticks_step++;
				break;	// 跳出循环更新窗口
			}

		}	// while

		screen_refresh_cnt++;

		// 更新 title
		if(cas_step&&(screen_refresh_cnt%50==0)) {
			if(cas_step==100) {
				sprintf(title, "LASER310 cass %02X %04X %04X",
				cas_buf[128+5], *(uint16_t*)(cas_buf+128+5+1+cas_fn_len), *(uint16_t*)(cas_buf+128+5+1+2+cas_fn_len));
				SDL_SetWindowTitle(wdo, title);
				cas_step=0;
			} else {
				sprintf(title, "LASER310 cass %02X %04X %04X load: %d %%",
				cas_buf[128+5], *(uint16_t*)(cas_buf+128+5+1+cas_fn_len), *(uint16_t*)(cas_buf+128+5+1+2+cas_fn_len),
				cas_step);
				SDL_SetWindowTitle(wdo, title);
			}
		}

		//============================================================= VIDEO OUTPUT
		emu_drawscreen(screenData);

		//========================================================= SDL RENDER FRAME

		if (++flashCycle == 30) {													// increase cursor flash cycle
			flashCycle = 0;														// reset to zero every half second
		}

		Uint32 fmt = sdlSurface->format->format;
		SDL_Surface *surf = SDL_ConvertSurfaceFormat(sdlScreen, fmt, 0);
		SDL_BlitScaled(surf,NULL,sdlSurface,NULL);
		SDL_FreeSurface(surf);

		SDL_Texture *sdlTex;
		sdlTex = SDL_CreateTextureFromSurface(rdr, sdlSurface);
		SDL_RenderCopy(rdr, sdlTex, 0, 0);
		SDL_DestroyTexture(sdlTex);

		SDL_RenderPresent(rdr);													// swap buffers
	}				// while (running)


	//================================================ RELEASE RESSOURSES AND EXIT

	SDL_FreeSurface(sdlScreen);
	SDL_FreeSurface(sdlSurface);

	SDL_AudioQuit();
	SDL_Quit();
	return 0;
}
