unit BinaryFile;

interface

uses
  SysUtils;

type
  EBinaryFile = class(Exception);
  TBinaryFile = class(TObject)
  private
    FHandle: Cardinal;
  public
    constructor Create;
    destructor Destroy; override;
    procedure CreateFile(Filename: string; Overwrite: Boolean = True);
    procedure OpenReadWrite(Filename: string; CanCreate: Boolean = False);
    procedure OpenReadOnly(Filename: string);
    procedure Close;
    procedure Write(Buffer: PChar; Count: Integer);
    function Read(Buffer: PChar; Count: Integer): Integer;
    procedure ReadCount(Buffer: PChar; Count: Integer);
    procedure SetSize;
    function GetSize: Integer;
    function GetPosition: Integer;
    function SetPositionEnd(Position: Integer): Integer;
    function SetPositionBegin(Position: Integer): Integer;
    function SetPositionCurrent(Position: Integer): Integer;
    procedure Flush;
    function isOpen: Boolean;
  end;

function Win32WriteFile(Handle: Cardinal; Buffer: PChar; Count: Integer;
  var BytesWritten: Integer; Dummy: Integer = 0): LongBool; stdcall;
function Win32ReadFile(Handle: Cardinal; Buffer: PChar; Count: Integer;
  var BytesWritten: Integer; Dummy: Integer = 0): LongBool; stdcall;
function Win32CreateFile(Filename: PChar; Access: Cardinal;
  Share: Cardinal; Dummy: Integer; Creation: Cardinal;
  FlagsAttribs: Cardinal; Dummy2: Integer): Cardinal; stdcall;
function Win32SetEndOfFile(Handle: Cardinal): LongBool; stdcall;
function Win32CloseHandle(Handle: Cardinal): LongBool; stdcall;
function Win32GetFileSize(Handle: Cardinal; Dummy: Integer = 0): Integer; stdcall;
function Win32SetFilePointer(Handle: Cardinal; Position: Integer;
  Dummy: Integer; Method: Cardinal): Cardinal; stdcall;
function Win32FlushFileBuffers(Handle: Cardinal): LongBool; stdcall;

implementation

const
  INVALID_HANDLE = $FFFFFFFF;
  kernel32 = 'kernel32.dll';

function Win32WriteFile; external kernel32 name 'WriteFile';
function Win32ReadFile; external kernel32 name 'ReadFile';
function Win32CreateFile; external kernel32 name 'CreateFileA';
function Win32SetEndOfFile; external kernel32 name 'SetEndOfFile';
function Win32CloseHandle; external kernel32 name 'CloseHandle';
function Win32GetFileSize; external kernel32 name 'GetFileSize';
function Win32SetFilePointer; external kernel32 name 'SetFilePointer';
function Win32FlushFileBuffers; external kernel32 name 'FlushFileBuffers';

constructor TBinaryFile.Create;
begin
  inherited Create;
  FHandle := INVALID_HANDLE;
end;

function TBinaryFile.isOpen: Boolean;
begin
  Result := FHandle <> INVALID_HANDLE;
end;

procedure TBinaryFile.CreateFile(Filename: string; Overwrite: Boolean);
begin
  if FHandle <> INVALID_HANDLE then Close;
  FHandle := Win32CreateFile(PChar(Filename), $40000000, $1, 0,
    1 + Ord(Overwrite), $80, 0);
  if FHandle = INVALID_HANDLE then
    raise EBinaryFile.Create('Cannot Create File: ' + Filename);
end;

procedure TBinaryFile.OpenReadWrite(Filename: string; CanCreate: Boolean);
begin
  if FHandle <> INVALID_HANDLE then Close;
  FHandle := Win32CreateFile(PChar(Filename), $C0000000, $1, 0,
    3 + Ord(CanCreate), $80, 0);
  if FHandle = INVALID_HANDLE then
    raise EBinaryFile.Create('Cannot Open File for Read/Write' + Filename);
end;

procedure TBinaryFile.OpenReadOnly(Filename: string);
begin
  if FHandle <> INVALID_HANDLE then Close;
  FHandle := Win32CreateFile(PChar(Filename), $80000000, $1, 0, 3, $80, 0);
  if FHandle = INVALID_HANDLE then
    raise EBinaryFile.Create('Cannot Open File for Reading: ' + Filename);
end;

destructor TBinaryFile.Destroy;
begin
  if FHandle <> INVALID_HANDLE then Close;
  inherited Destroy;
end;

procedure TBinaryFile.Close;
begin
  if not Win32CloseHandle(FHandle) then
    raise EBinaryFile.Create('Cannot Close File');
  FHandle := INVALID_HANDLE;
end;

procedure TBinaryFile.Write(Buffer: PChar; Count: Integer);
var
  LRet: Integer;
begin
  while Count > 0 do
  begin
    if not Win32WriteFile(FHandle, Buffer, Count, LRet)  then
      raise EBinaryFile.Create('Cannot Write to File');
    Count := Count - LRet; Buffer := Buffer + LRet;
  end;
end;

function TBinaryFile.Read(Buffer: PChar; Count: Integer): Integer;
begin
  if not Win32ReadFile(FHandle, Buffer, Count, Result)  then
    raise EBinaryFile.Create('Cannot Read from File');
end;

procedure TBinaryFile.ReadCount(Buffer: PChar; Count: Integer);
var
  BytesRead: Integer;
begin
  BytesRead := Read(Buffer, Count);
  if BytesRead <> Count then raise EBinaryFile.Create(
    Format('Can Read Only %d Out Of %d Bytes', [BytesRead, Count]));
end;

procedure TBinaryFile.SetSize;
begin
  if not Win32SetEndOfFile(FHandle) then
    raise EBinaryFile.Create('Cannot Set File Size');
end;

function TBinaryFile.GetSize: Integer;
begin
  Result := Win32GetFileSize(FHandle);
  if Result = -1 then EBinaryFile.Create('Cannot Get File Size');
end;

function TBinaryFile.GetPosition: Integer;
begin
  Result := Win32SetFilePointer(FHandle, 0, 0, 1);
  if Result = -1 then raise EBinaryFile.Create('Cannot Get File Position');
end;

function TBinaryFile.SetPositionEnd(Position: Integer): Integer;
begin
  Result := Win32SetFilePointer(FHandle, Position, 0, 2);
  if Result = -1 then raise EBinaryFile.Create('Cannot Set File Position');
end;

function TBinaryFile.SetPositionBegin(Position: Integer): Integer;
begin
  Result := Win32SetFilePointer(FHandle, Position, 0, 0);
  if Result = -1 then raise EBinaryFile.Create('Cannot Set File Position');
end;

function TBinaryFile.SetPositionCurrent(Position: Integer): Integer;
begin
  Result := Win32SetFilePointer(FHandle, Position, 0, 1);
  if Result = -1 then raise EBinaryFile.Create('Cannot Set File Position');
end;

procedure TBinaryFile.Flush;
begin
  if not Win32FlushFileBuffers(Fhandle) then
    raise EBinaryFile.Create('Cannot Flush File Buffers');
end;

end.
