unit CPU;

interface

uses
  BinaryFile;

type
  TIOWriteHandler = procedure(Value: Byte);
  TIOReadHandler = function: Byte;
  TCPUState = record
    CyclesDone, BadOpcode: Integer;
    IRQs: Byte;
  end;

const
  RET_OK          = 0;
  RET_BAD_OPCODE  = -1;
  RET_NOT_MEMORY  = -2;
  RET_BAD_MMR     = -3;
  IRQ_TIMER       = $04;
  IRQ_VIDEO       = $02;

var
  State: TCPUState;
  IOWriteHandler: array[0..$1fff] of TIOWriteHandler;
  IOReadHandler: array[0..$1fff] of TIOReadHandler;

procedure SetRom(ARom: Pointer; RomPages: Integer);
procedure Reset;
function Run(Cycles: Cardinal): Integer;
function SaveState(F: TBinaryFile): Boolean;
function LoadState(F: TBinaryFile): Boolean;

implementation

const
  BAD_OPCODE_CYCLES = 2;
  FLAG_I        = $04;  FLAG_D        = $08;  FLAG_O        = $40;
  FLAG_B        = $10;  FLAG_Z        = $02;  FLAG_C        = $01;
  FLAG_N        = $80;  FLAG_T        = $20;  EMU_FLAG_Z    = $40;
  MMR_IS_RAM    = 0;    MMR_IS_ROM    = 1;    MMR_IS_IO     = 2;
  MMR_IS_BAD    = 3;
  VEC_BREAK     = $FFF6; // irq2
  VEC_RESET     = $FFFE;
  VEC_TIMER     = $FFFA;
  VEC_VIDEO     = $FFF8; // irq1
  IRQ_ALL       = IRQ_VIDEO + IRQ_TIMER;
  RT_EXEC_IRQS  = 0;    RT_RESUME_E3  = 1;    RT_RESUME_F3  = 2;
  RT_RESUME_73  = 3;    RT_RESUME_D3  = 4;    RT_RESUME_C3  = 5;

type
  TPrivateState = record
    MMRType: array[0..7] of Byte;
    FlagOverflow, FlagsxxTBDIxx: Byte;
    RunTo, CurrentPage, CurrentPageStart, Backup_PC: Cardinal;
    RA, RX, RY, RSP, RF: Byte;
    RMMR: array[0..7] of Byte;
    SaveRAM, RAM: array[0..$2000] of Byte;
    MMRBase: array[0..7] of Cardinal;
    RomAddr, RamAddr: Cardinal;
    TransferCount, TransferSource, TransferDest: Cardinal;
    Backup_PageLeft: Cardinal;
  end;

var
  PS: TPrivateState;
  Rom8kPages: Cardinal;
  Backup_A: Byte;
  RomType: array[0..$ff] of Byte;
  RomMap: array[0..$ff] of Pointer;
  ROM: Cardinal;

procedure ExitBadOpcode; forward;
procedure ExitBadMMR; forward;
procedure NoMoreCycles; forward;
procedure NextOpcode; forward;
procedure NextOpcodeT; forward;
procedure ExecuteIRQs; forward;
procedure PackFlags; forward;
procedure UnpackFlags; forward;
procedure _E3_Resume; forward;
procedure _F3_Resume; forward;
procedure _D3_Resume; forward;
procedure _C3_Resume; forward;
procedure _73_Resume; forward;

const
  RunAddr: array[0..5] of Pointer = (
    @ExecuteIRQs, @_E3_Resume,  @_F3_Resume,
    @_73_Resume,  @_D3_Resume,  @_C3_Resume);

procedure SetRom(ARom: Pointer; RomPages: Integer);
var
  i, RomMask: Integer;
begin
  Rom := Cardinal(ARom);
  Rom8kPages := RomPages;
  RomMask := 1;
  while RomMask < RomPages do RomMask := RomMask shl 1;
  Dec(RomMask);
  for i := 0 to $F6 do
    begin
      if RomPages = $30 then
        case i and $70 of
          $00, $10, $50:
            RomMap[i] := PChar(Rom) + (i and RomMask) * $2008;
          $30, $70:
            RomMap[i] := PChar(Rom) + ((i - $10) and RomMask) * $2008;
          else // $20, $40, $60:
            RomMap[i] := PChar(Rom) + ((i - $20) and RomMask) * $2008;
        end
      else
        RomMap[i] := PChar(Rom) + (i and RomMask) * $2008;
      RomType[i] := MMR_IS_ROM;
    end;
    RomMap[$f7] := @PS.SaveRam;
    RomType[$f7] := MMR_IS_RAM;
    RomMap[$f8] := @PS.Ram;
    RomType[$f8] := MMR_IS_RAM;
    for i := $f9 to $ff do
      begin
        RomType[i] := MMR_IS_BAD;
        RomMap[i] := PChar(0);
      end;
    RomType[$ff] := MMR_IS_IO;
end;

procedure SetMMR; // [edi] = edx
asm
  push  eax
  push  esi
  push  ebx
  mov   byte ptr [PS.RMMR + edi], dl
  mov   eax, dword ptr [RomMap + edx * 4]
  shl   edi, 13
  mov   esi, eax
  sub   eax, edi
  shr   edi, 13
  mov   bl, byte ptr [RomType + edx]

  cmp   bl, MMR_IS_BAD
  jnz   @not_bad

  pop   ebx
  pop   esi
  pop   eax
  pop   edx
  jmp   ExitBadMMR

@not_bad:

  mov   dword ptr [PS.MMRBase + edi * 4], eax
  cmp   bl, MMR_IS_ROM
  mov   byte ptr [PS.MMRType + edi], bl
  jnz   @no
  mov   eax, edi
  dec   eax
  and   eax, 7
  cmp   byte ptr [PS.MMRType + eax], MMR_IS_ROM
  jnz   @test_next
  mov   al, byte ptr [PS.RMMR + eax]
  mov   ebx, [esi]
  mov   eax, dword ptr [RomMap + 4 * eax]
  mov   [eax + $2000], ebx
  mov   ebx, [esi + 4]
  mov   [eax + $2004], ebx
@test_next:
  mov   eax, edi
  inc   eax
  and   eax, 7
  cmp   byte ptr [PS.MMRType + eax], MMR_IS_ROM
  jnz   @no
  mov   al, byte ptr [PS.RMMR + eax]
  mov   eax, dword ptr [RomMap + 4 * eax]
  mov   ebx, [eax]
  mov   eax, [eax + 4]
  mov   [esi + $2000], ebx
  mov   [esi + $2004], eax
@no:
  pop   ebx
  pop   esi
  pop   eax
end;

procedure NewPC; // edx = new PC
asm
  push  eax
  and   ebp, $ffff0000
  mov   edi, edx
  mov   esi, edx
  and   edx, $e000
  mov   [PS.CurrentPageStart], edx
  mov   dword ptr [PS.CurrentPage], edx
  shr   edx, 13
  mov   eax, dword ptr [PS.MMRBase + 4 * edx]
  add   esi, eax
  or    ebp, $9fff
  and   edi, $1fff
  sub   ebp, edi
  add   [PS.CurrentPageStart], eax
  pop   eax
end;

procedure NewPC_J;
asm
  push  offset NextOpcode
  jmp  NewPC
end;

procedure AdjustPC; // check ebp
asm
  test  ebp, $6000
  jz    @ok
  sub   esi, [PS.CurrentPageStart]
  mov   edx, [PS.CurrentPage]
  add   edx, esi
  jmp   NewPC
@ok:
end;

procedure GetPC; // -> edx = real PC
asm
  mov   edx, [PS.CurrentPage]
  add   edx, esi
  sub   edx, [PS.CurrentPageStart]
end;

{$I cpu_asm.pas}

procedure PackFlags; // from ah/... to dl/6280
asm
  mov   dl, [PS.FlagsxxTBDIxx]
  and   dl, FLAG_B + FLAG_D + FLAG_I
  mov   dh, ah
  and   dh, FLAG_N + FLAG_C
  or    dl, dh
  mov   dh, ah
  shr   dh, 5
  and   dh, FLAG_Z
  or    dl, dh
  mov   dh, [PS.FlagOverflow]
  shl   dh, 6
  or    dl, dh
end;

procedure UnpackFlags; // from cl/6280 to ah/...
asm
  push  edx
  mov   [PS.FlagsxxTBDIxx], cl
  mov   ah, cl
  and   ah, FLAG_N + FLAG_C
  mov   dh, cl
  and   dh, FLAG_Z
  shl   dh, 5
  or    ah, dh
  mov   dh, cl
  and   dh, FLAG_O
  shr   dh, 6
  mov   [PS.FlagOverflow], dh
  pop   edx
end;

procedure Dump;
asm
  mov   [PS.RA], al
  mov		[PS.RX], bl
  mov		[PS.RY], bh
  mov   [PS.RSP], ch
  mov   [PS.Backup_PC], esi
  call  PackFlags
  mov   [PS.RF], dl

  mov   eax, ebp
  shr   eax, 20
  test  eax, $800
  jnz   @ok
  or    eax, $fffff800
  sub   [State.CyclesDone], eax
  xor   eax, eax
@ok:
  and   eax, $3ff
  sub   [State.CyclesDone], eax
  and   ebp, $ffff
  mov   dword ptr [PS.Backup_PageLeft], ebp
end;

procedure Undump;
asm
  mov   al, [PS.RA]
  mov		bl, [PS.RX]
  mov		bh, [PS.RY]
  mov   ch, [PS.RSP]
  mov   esi, [PS.Backup_PC]
  mov   cl, [PS.RF]
  call  UnpackFlags
  or    ebp, [PS.Backup_PageLeft]
end;

procedure NoMoreCycles;
asm
  call  Dump
  pop   ebx
  pop   ebp
  pop   esi
  pop   edi
  mov   eax, RET_OK
end;

procedure ExitBadOpcode;
asm
  mov   [State.BadOpcode], edx
  call  Dump
  pop   ebx
  pop   ebp
  pop   esi
  pop   edi
  mov   eax, RET_BAD_OPCODE
end;

procedure ExitIsIO;
asm
  call  Dump
  pop   ebx
  pop   ebp
  pop   esi
  pop   edi
  mov   eax, RET_NOT_MEMORY
end;

procedure ExitBadMMR;
asm
  call  Dump
  pop   ebx
  pop   ebp
  pop   esi
  pop   edi
  mov   eax, RET_BAD_MMR
end;

procedure Reset;
asm
  push  ebp
  push  esi
  push  edi
  xor   eax, eax
  mov   dword ptr [PS.RunTo], RT_EXEC_IRQS
  mov   [State.IRQs], al
  mov   [State.CyclesDone], eax
  mov   [PS.RA], al
  mov   [PS.RX], al
  mov   [PS.RY], al
  mov   [PS.RSP], $FF
  mov   [PS.RF], FLAG_I

  mov   edi, 7
  xor   edx, edx
  call  SetMMR
  dec   edi
  mov   edx, 5
  call  SetMMR
  dec   edi
  dec   edx
  call  SetMMR
  dec   edi
  dec   edx
  call  SetMMR
  dec   edi
  dec   edx
  call  SetMMR
  dec   edi
  dec   edx
  call  SetMMR
  dec   edi
  mov   edx, $F8
  call  SetMMR
  dec   edi
  mov   edx, $FF
  call  SetMMR

  mov   edi, dword ptr [PS.MMRBase + 4 * (VEC_RESET shr 13)]
  movzx edx, word ptr [edi + VEC_RESET]
  call  NewPC
  mov   [PS.Backup_PC], esi
  and   ebp, $ffff
  mov   [PS.Backup_PageLeft], ebp

  mov   edi, offset PS.RAM
  mov   ecx, $2000 shr 2
  cld
  xor   eax, eax
  rep   stosd
  mov   edi, offset PS.SaveRAM
  mov   ecx, $2000 shr 2
  rep   stosd

  mov   eax, [ROM]
  mov   [PS.RomAddr], eax
  mov   [PS.RamAddr], offset PS.RAM

  pop   edi
  pop   esi
  pop   ebp
end;

function Run(Cycles: Cardinal): Integer;
asm
  and   eax, $3ff
  push  edi
  push  esi
  push  ebp
  push  ebx
  add   [State.CyclesDone], eax
  mov   ebp, eax
  shl   ebp, 20
  or    ebp, $80000000
  call  Undump
  mov   edx, [PS.RunTo]
  jmp   dword ptr [RunAddr + 4 * edx]
end;

const Opcodes: array[0..255] of procedure = (
  _00, _01, _02, _03, _04, _05, _06, _07, _08, _09, _0A, BAD, _0C, _0D, _0E, _0F,
  _10, _11, _12, _13, _14, _15, _16, _17, _18, _19, _1A, BAD, _1C, _1D, _1E, _1F,
  _20, _21, _22, _23, _24, _25, _26, _27, _28, _29, _2A, BAD, _2C, _2D, _2E, _2F,
  _30, _31, _32, BAD, _34, _35, _36, _37, _38, _39, _3A, BAD, _3C, _3D, _3E, _3F,
  _40, _41, _42, _43, _44, _45, _46, _47, _48, _49, _4A, BAD, _4C, _4D, _4E, _4F,
  _50, _51, _52, _53, NOP, _55, _56, _57, _58, _59, _5A, BAD, BAD, _5D, _5E, _5F,
  _60, _61, _62, BAD, _64, _65, _66, _67, _68, _69, _6A, BAD, _6C, _6D, _6E, _6F,
  _70, _71, _72, _73, _74, _75, _76, _77, _78, _79, _7A, BAD, _7C, _7D, _7E, _7F,
  _80, _81, _82, _83, _84, _85, _86, _87, _88, _89, _8A, BAD, _8C, _8D, _8E, _8F,
  _90, _91, _92, _93, _94, _95, _96, _97, _98, _99, _9A, BAD, _9C, _9D, _9E, _9F,
  _A0, _A1, _A2, _A3, _A4, _A5, _A6, _A7, _A8, _A9, _AA, BAD, _AC, _AD, _AE, _AF,
  _B0, _B1, _B2, _B3, _B4, _B5, _B6, _B7, _B8, _B9, _BA, BAD, _BC, _BD, _BE, _BF,
  _C0, _C1, _C2, _C3, _C4, _C5, _C6, _C7, _C8, _C9, _CA, BAD, _CC, _CD, _CE, _CF,
  _D0, _D1, _D2, _D3, NOP, _D5, _D6, _D7, _D8, _D9, _DA, BAD, BAD, _DD, _DE, _DF,
  _E0, _E1, BAD, _E3, _E4, _E5, _E6, _E7, _E8, _E9, _EA, BAD, _EC, _ED, _EE, _EF,
  _F0, _F1, _F2, _F3, _F4, _F5, _F6, _F7, _F8, _F9, _FA, BAD, BAD, _FD, _FE, _FF
);

const OpcodesT: array[0..127] of procedure = (
    BAD, _T_01,   BAD,   BAD,   BAD, _T_05,   BAD,   BAD,
    BAD, _T_09,   BAD,   BAD,   BAD, _T_0D,   BAD,   BAD,
    BAD, _T_11, _T_12,   BAD,   BAD, _T_15,   BAD,   BAD,
    BAD, _T_19,   BAD,   BAD,   BAD, _T_1D,   BAD,   BAD,
    BAD, _T_21,   BAD,   BAD,   BAD, _T_25,   BAD,   BAD,
    BAD, _T_29,   BAD,   BAD,   BAD, _T_2D,   BAD,   BAD,
    BAD, _T_31, _T_32,   BAD,   BAD, _T_35,   BAD,   BAD,
    BAD, _T_39,   BAD,   BAD,   BAD, _T_3D,   BAD,   BAD,
    BAD, _T_41,   BAD,   BAD,   BAD, _T_45,   BAD,   BAD,
    BAD, _T_49,   BAD,   BAD,   BAD, _T_4D,   BAD,   BAD,
    BAD, _T_51, _T_52,   BAD,   BAD, _T_55,   BAD,   BAD,
    BAD, _T_59,   BAD,   BAD,   BAD, _T_5D,   BAD,   BAD,
    BAD, _T_61,   BAD,   BAD,   BAD, _T_65,   BAD,   BAD,
    BAD, _T_69,   BAD,   BAD,   BAD, _T_6D,   BAD,   BAD,
    BAD, _T_71, _T_72,   BAD,   BAD, _T_75,   BAD,   BAD,
    BAD, _T_79,   BAD,   BAD,   BAD, _T_7D,   BAD,   BAD
);

{procedure Check;
asm
  mov   edx, $1fff
  sub   edx, esi
  add   edx, [PS.CurrentPageStart]
  push  eax
  mov   eax, ebp
  and   eax, $1fff
  //cmp   eax, 5
  //jbe   @bad
  cmp   eax, edx
  pop   eax
  jnz   @bad
  ret
@bad:
  pop   eax
  pop   edx
  jmp   ExitBadMMR
end;}

procedure NextOpcode;
asm
  test  ebp, $40006000
  jz    @ok
  test  ebp, $6000
  jnz   @change_pc
  jmp   NoMoreCycles
@change_pc:
  call  AdjustPC
  jmp   NextOpcode
@ok:
  //call  Check
  movzx edx, byte ptr [esi]
  inc   esi
  jmp   dword ptr [Opcodes + 4 * edx]
end;

procedure NextOpcodeT;
asm
  test  ebp, $6000
  jz    @ok
  call  AdjustPC
@ok:
  //call  Check
  movzx edx, byte ptr [esi]
  inc   esi
  test  dl, $80
  jnz   ExitBadOpcode
  jmp   dword ptr [OpcodesT + 4 * edx]
end;

procedure ExecuteIRQs;
asm
  test  [PS.FlagsxxTBDIxx], FLAG_I
  jnz   NextOpcode
  test  [State.IRQs], IRQ_VIDEO + IRQ_TIMER
  jz    NextOpcode

  call  GetPC

  movzx edi, ch
  mov   byte ptr [edi + PS.RAM + $100], dh
  dec   ch

  movzx edi, ch
  mov   byte ptr [edi + PS.RAM + $100], dl
  dec   ch

  call  PackFlags

  movzx edi, ch
  and   dl, $FF - FLAG_B
  mov   byte ptr [edi + PS.RAM + $100], dl
  dec   ch

  and   [PS.FlagsxxTBDIxx], $FF - FLAG_D
  or    [PS.FlagsxxTBDIxx], FLAG_I
  sub   ebp, 7 shl 20

  test  [State.IRQs], IRQ_VIDEO
  jnz   @video
@timer:
  mov   edi, dword ptr [PS.MMRBase + 4 * (VEC_TIMER shr 13)]
  and   [State.IRQs], $FF - IRQ_TIMER
  movzx edx, word ptr [edi + VEC_TIMER]
  jmp   NewPC_J
@video:
  mov   edi, dword ptr [PS.MMRBase + 4 * (VEC_VIDEO shr 13)]
  and   [State.IRQs], $FF - IRQ_VIDEO
  movzx edx, word ptr [edi + VEC_VIDEO]
  jmp   NewPC_J
end;

function LoadState(F: TBinaryFile): Boolean;
var
  Sig: array[0..4] of Char;
  RomDif, RamDif: Cardinal;
  i: Integer;
begin
  Result := False;
  try
    F.ReadCount(@Sig[0], 5);
    if Sig <> 'SFCPU' then Exit;
    F.ReadCount(@State, SizeOf(State));
    F.ReadCount(@PS, SizeOf(PS));
    with PS do
      begin
        RomDif := Cardinal(RomAddr) - Cardinal(Rom);
        RamDif := Cardinal(RamAddr) - Cardinal(@Ram);
        RomAddr := Rom;
        RamAddr := Cardinal(@Ram);
        for i := 0 to 7 do
          begin
            if MMRType[i] = MMR_IS_ROM then Dec(MMRBase[i], RomDif)
            else if MMRType[i] = MMR_IS_RAM then Dec(MMRBase[i], RamDif);
          end;
        if MMRType[CurrentPage shr 13] = MMR_IS_ROM then
          begin
            Dec(CurrentPageStart, RomDif);
            Dec(Backup_PC, RomDif);
          end
        else
          begin
            Dec(CurrentPageStart, RamDif);
            Dec(Backup_PC, RamDif);
          end;
      end;
    Result := True;
  except
  end;
end;

function SaveState(F: TBinaryFile): Boolean;
begin
  Result := False;
  try
    F.Write('SFCPU', 5);
    F.Write(@State, SizeOf(State));
    F.Write(@PS, SizeOf(PS));
    Result := True;
  except
  end;
end;

end.

