unit IO_Video;

interface

uses
  DXDraws, DXInput;

const
  SURFACE_WIDTH = 352;
  SURFACE_HEIGHT = 228 + 14;
  MAX_SKIP_FRAME = 10;

var
  DXDraw: TDXDraw;
  DInput: TDXInput;

procedure PowerOn;
procedure DrawScanline;

implementation

uses
  CPU, IO_Shared, LogUnit, IO_IRQ, IO_Joystick, SysUtils, Graphics,
  Windows, BinaryFile, DirectX, ExtCtrls, TextFiles;

type
  Sprite = packed record
    Y, X, No, Attr: Word;
  end;
  PSprite = ^Sprite;
  PCardinal = ^Cardinal;

  TPS = record
    Status, LowVWR: Byte;
    Increment, VR: Integer;
    Regs: array[0..19] of WordBytes;
    SPRAM: array[0..63] of Sprite;
    BGWidthMask, BGWidthNMask, BGHeightMask, BGWidthShift: Integer;
    HorizontalTiles, ScreenWidth, ScreenHeight: Integer;
    SATBLine, PrevLine, FirstLine, LastLine, MaxLine: Integer;
    PendingVSync, Scroll, HasSATB: Boolean;
    ScrollYDiff: Integer;
    ScrollYDiffOld, ScrollXOld, ScrollYOld: Integer;
    Scanline: Integer;
    Frame: Integer;
    FramesSkipped, Ticks, FrameCount: Cardinal;
    BGWidth, BGHeight, FPS: Integer;
    SkipFrame: Boolean;
  end;

const
  R_MAWR  = 0;  R_MARR  = 1;  R_VWR   = 2;  R_VRR   = 2;
  R_VDC3  = 3;  R_VDC4  = 4;  R_CR    = 5;  R_RCR   = 6;
  R_BXR   = 7;  R_BYR   = 8;  R_MWR   = 9;  R_HSR   = 10;
  R_HDR   = 11; R_VPR   = 12; R_VDW   = 13; R_VCR   = 14;
  R_DCR   = 15; R_SOUR  = 16; R_DESR  = 17; R_LENR  = 18;
  R_SATB  = 19;

const
  STATUS_SATB_FINISH    = $08;
  MAX_WIDTH = 352 + 64;
  MAX_HEIGHT = 256;
  MAX_DISPLAY_LINES = 227;
  SCANLINES_PER_FRAME = 263;
  CR_BG_ON = $80;

const
  TILE_PALETTE_CHANGED  = $08;
  TILE_IMAGE_CHANGED    = $04;
  VRAM_BYTES = $10000;
  VRAM_WORDS = VRAM_BYTES div 2;
  VRAM_DWORDS = VRAM_BYTES div 4;
  VRAM_TILES = VRAM_BYTES div 32;
  VRAM_SPRITES = VRAM_BYTES div 128;
  VRAM_TILECACHE_DELTA = $10000;
  VRAM_SPRITECACHE_DELTA = $20000;

var
  PS: TPS;
  // these two must be in this order
  VRAM: array[0..VRAM_WORDS - 1] of WordBytes;
  TileCache: array[0..VRAM_DWORDS - 1] of Cardinal;
  SpriteCache: array[0..VRAM_DWORDS - 1] of Cardinal; // this one not yet!
  //////////
  Bitmap: array[0..MAX_WIDTH * MAX_HEIGHT  - 1] of Word;
  TileState: array[0..VRAM_TILES - 1] of Byte;
  SpriteState: array[0..VRAM_SPRITES - 1] of Byte;
  SkipFrameCount: Integer;

procedure SetMaxLine(Line: Integer);
begin
  with PS do begin
    MaxLine := Line;
    FirstLine := 0;
    SATBLine := MaxLine + 5;
    if (SCANLINES_PER_FRAME - 1) < SATBLine then
      SATBLine := SCANLINES_PER_FRAME - 1;
    if MaxLine > MAX_DISPLAY_LINES then
      FirstLine := (MaxLine - MAX_DISPLAY_LINES + 1) div 2;
    LastLine := MaxLine - FirstLine;
  end;
end;

{$I v2.pas}
{$I v1.pas}
{$I v3.pas}
{$I v0.pas}

function Read_0: Byte;
begin
  Result := PS.Status;
  PS.Status := 0;
end;

function Read_1: Byte;
begin
  Result := 0;
end;

function Read_2: Byte;
begin
  if PS.VR = R_VRR then
    Result := VRAM[PS.Regs[R_MARR].W].L
  else
    Result := PS.Regs[PS.VR].L;
end;

function Read_3: Byte;
begin
  if PS.VR = R_VRR then
    begin
      Result := VRAM[PS.Regs[R_MARR].W].H;
      Inc(PS.Regs[R_MARR].W, PS.Increment);
    end
  else
    Result := PS.Regs[PS.VR].H;
end;

procedure SetScroll;
begin
  with PS do begin
  if not Scroll then
    begin
      ScrollXOld := Regs[R_BXR].W;
      ScrollYOld := Regs[R_BYR].W;
      ScrollYDiffOld := ScrollYDiff;
      Scroll := True;
    end;
  end;
end;

procedure Write_2_02(Value: Byte); // VWR
begin
  PS.LowVWR := Value;
end;

procedure Write_2_11(Value: Byte); // HDR
begin
  with PS do begin
  ScreenWidth := (Value + 1) * 8;
  Assert(ScreenWidth <= SURFACE_WIDTH);
  HorizontalTiles := Value + 2;
  Regs[R_HDR].L := Value;
  end;
end;

procedure Write_2_09(Value: Byte); // MWR
const
  BG_WIDTH: array[0..3] of Byte = (32, 64, 128, 128);
  BG_WIDTH_SHIFT: array[0..3] of Byte = (5, 6, 7, 7);
begin
  with PS do begin
  BGWidth := BG_WIDTH[(Value div 16) and $03];
  BGWidthShift := BG_WIDTH_SHIFT[(Value div 16) and $03];
  BGWidthMask := BGWidth - 1;
  BGWidthNMask := not BGWidthMask;
  BGHeight := Value and $40;
  BGHeight := BGHeight or (BGHeight xor $40) shr 1;
  BGHeightMask := BGHeight - 1;
  FillChar(TileState[0], SizeOf(TileState), $ff);
  FillChar(SpriteState[0], SizeOf(SpriteState), $ff);
  Regs[R_MWR].L := Value;
  end;
end;

procedure Write_2_08(Value: Byte); // BYR
begin
  with PS do begin
  SetScroll;
  ScrollYDiff := Scanline - 1;
  Regs[R_BYR].L := Value;
  end;
end;

procedure Write_2_07(Value: Byte); // BXR
begin
  SetScroll;
  PS.Regs[R_BXR].L := Value;
end;

procedure Write_2_xx(Value: Byte);
begin
    PS.Regs[PS.VR].L := Value;
end;

const
  Write_2_Reg: array[0..19] of TIOWriteHandler = (
    Write_2_xx, Write_2_xx, Write_2_02, Write_2_xx,
    Write_2_xx, Write_2_xx, Write_2_xx, Write_2_07,
    Write_2_08, Write_2_09, Write_2_xx, Write_2_11,
    Write_2_xx, Write_2_xx, Write_2_xx, Write_2_xx,
    Write_2_xx, Write_2_xx, Write_2_xx, Write_2_xx );

procedure Write_3_02(Value: Byte); // VWR
begin
  with PS, VRAM[Regs[R_MAWR].W] do
    begin
      L := LowVWR;
      H := Value;
      LowVWR := 0;
      TileState[Regs[R_MAWR].W shr 4] := $ff;
      SpriteState[Regs[R_MAWR].W shr 6] := $ff;
      Regs[R_MAWR].W := (Regs[R_MAWR].W + Increment) and $7fff;
    end;
end;

procedure Write_3_13(Value: Byte); // VDW
begin
  with PS do
    begin
      Regs[R_VDW].H := Value;
      ScreenHeight := Regs[R_VDW].W and $1ff + 1;
      SetMaxLine(ScreenHeight - 1);
    end;
end;

procedure Write_3_05(Value: Byte); // CR
const
  INC_SIZE: array[0..3] of Byte = (1, 32, 64, 128);
begin
  PS.Regs[R_CR].H := Value;
  PS.Increment := INC_SIZE[(Value shr 3) and $03];
end;

procedure Write_3_19(Value: Byte); // SATB
const
  INC_SIZE: array[0..3] of Byte = (1, 32, 64, 128);
begin
  Ps.Regs[R_SATB].H := Value;
  PS.HasSATB := True;
  PS.Status := PS.Status and not STATUS_SATB_FINISH;
end;

procedure Write_3_08(Value: Byte); // BYR
begin
  SetScroll;
  PS.Regs[R_BYR].H := Value and $01;
end;

procedure Write_3_07(Value: Byte); // BXR
begin
  SetScroll;
  PS.Regs[R_BXR].H := Value and $03;
end;

procedure Write_3_18(Value: Byte); // LENR
const
  DMA_FINISH    = $10;
var
  Source, Dest, Count: Integer;
begin
  with PS do
    begin
      Source := Regs[R_SOUR].W;
      Dest := Regs[R_DESR].W;
      Regs[R_LENR].H := Value;
      Count := Regs[R_LENR].W + 1;
      Inc(Regs[R_DESR].W, Count);
      Inc(Regs[R_SOUR].W, Count);
      Count := Min(Count, $8000 - Dest);
      Count := Min(Count, $8000 - Source);
      Move(VRAM[Source], VRAM[Dest], Count shl 1);
      FillChar(TileState[Dest shr 4], Count shr 4, $ff);
      FillChar(SpriteState[Dest shr 6], Count shr 6, $ff);
      Status := Status or DMA_FINISH;
    end;
end;

procedure Write_3_xx(Value: Byte);
begin
    PS.Regs[PS.VR].H := Value;
end;

const
  Write_3_Reg: array[0..19] of TIOWriteHandler = (
    Write_3_xx, Write_3_xx, Write_3_02, Write_3_xx,
    Write_3_xx, Write_3_05, Write_3_xx, Write_3_07,
    Write_3_08, Write_3_xx, Write_3_xx, Write_3_xx,
    Write_3_xx, Write_3_13, Write_3_xx, Write_3_xx,
    Write_3_xx, Write_3_xx, Write_3_18, Write_3_19 );

procedure Write_0(Value: Byte);
begin
  Value := Value and $1f;
  if Value < 20 then
    begin
      PS.VR := Value;
      CPU.IOWriteHandler[2] := Write_2_Reg[Value];
      CPU.IOWriteHandler[3] := Write_3_Reg[Value];
    end;
end;

procedure Reset;
var
  i: Integer;
begin
  with PS do
    begin
      for i := 0 to 19 do Regs[i].W := 0;
      VR := 0;
      Increment := 1;
      Status := 0;
      HasSATB := False;
      LowVWR := 0;
      FillChar(VRAM[0], SizeOf(VRAM), 0);
      FillChar(SPRAM[0], SizeOf(SPRAM), 0);
      FillChar(TileState[0], SizeOf(TileState), $ff);
      FillChar(SpriteState[0], SizeOf(SpriteState), $ff);
    	PrevLine := 0;
      SetMaxLine(255);
      Scanline := 0;
      Frame := 0;
      FrameCount := 0;
      ScreenHeight := 0;

      Ticks := GetTickCount;
      FPS := 0;

      SkipFrame := False;
      FramesSkipped := 0;
      SkipFrameCount := 0;
      DXDraw.Surface.Fill(0);
      DXDraw.Flip;
    end;
end;

function LoadState(F: TBinaryFile): Boolean;
begin
  Result := False;
  try
    F.ReadCount(@VRAM, SizeOf(VRAM));
    F.ReadCount(@PS, SizeOf(PS));
    FillChar(TileState[0], SizeOf(TileState), $ff);
    FillChar(SpriteState[0], SizeOf(SpriteState), $ff);
    CPU.IOWriteHandler[2] := Write_2_Reg[PS.VR];
    CPU.IOWriteHandler[3] := Write_3_Reg[PS.VR];
    Result := True;
  except
  end;
end;

function SaveState(F: TBinaryFile): Boolean;
begin
  Result := False;
  try
    F.Write(@VRAM, SizeOf(VRAM));
    F.Write(@PS, SizeOf(PS));
    Result := True;
  except
  end;
end;

procedure PowerOn;
begin
  IOReadHandler[$0] := Read_0;
  IOReadHandler[$1] := Read_1;
  IOReadHandler[$2] := Read_2;
  IOReadHandler[$3] := Read_3;
  IOWriteHandler[$0] := Write_0;
  IOWriteHandler[$2] := Write_2_xx;
  IOWriteHandler[$3] := Write_3_xx;
  Assert(Integer(@TileCache) - Integer(@VRAM) = VRAM_TILECACHE_DELTA);
  Assert(Integer(@SpriteCache) - Integer(@VRAM) = VRAM_SPRITECACHE_DELTA);

  ResetList.Add(@Reset);
  ResumeList.Add(@Resume);
  LoadSaveStateList.Add(@LoadState);
  LoadSaveStateList.Add(@SaveState);
  Reset;
end;

end.
