unit IO_Palette;

interface

procedure PowerOn;
procedure SetPixelFormat(Red, Green, Blue: Cardinal);

implementation

uses
  IO_Shared, LogUnit, CPU, BinaryFile;


type
  TPS = record
    CR: Byte;
    Palette: array[0..$1ff] of WordBytes;
    Index: WordBytes;
  end;

var
  PS: TPS;
  ToRGB: array[0..$1ff] of Word;

procedure SetPalette(Index, Color: Integer);
var
  i: Integer;
  RGB: Word;
begin
  RGB := ToRGB[Color];
  if (Index = 0) then
    for i := 0 to 15 do RGBPalette[i * 16] := RGB
  else
    if Index and $0f <> 0 then RGBPalette[Index] := RGB;
end;

procedure SetPixelFormat(Red, Green, Blue: Cardinal);
var
  i, RS, GS, BS: Integer;
begin
  Assert(Blue <> 0); Assert(Green <> 0); Assert(Red <> 0);
  RS := 31; GS := 31; BS := 31;
  while Red and (1 shl RS) = 0 do Dec(RS);
  while Green and (1 shl GS) = 0 do Dec(GS);
  while Blue and (1 shl BS) = 0 do Dec(BS);
  Dec(RS, 2); Dec(GS, 2); Dec(BS, 2);
  for i := 0 to $1ff do
    begin
      ToRGB[i] :=
        (i and $7) shl BS or
        ((i shr 3) and $7) shl RS or
        ((i shr 6) and $7) shl GS;
      SetPalette(i, PS.Palette[i].W);
    end;
end;

procedure PaletteReset;
var
  i: Integer;
begin
  for i := 0 to $1ff do
    begin
      PS.Palette[i].W := 0;
      RGBPalette[i] := 0;
    end;
end;

function Read_402: Byte;
begin
  Result := PS.Index.L
end;

function Read_403: Byte;
begin
  Result := PS.Index.H;
end;

function Read_404: Byte;
begin
  Result := PS.Palette[PS.Index.W].L;
end;

function Read_405: Byte;
begin
  Result := PS.Palette[PS.Index.W].H;
  PS.Index.W := (PS.Index.W + 1) and $1FF;
end;

procedure Write_400(Value: Byte);
begin
  PS.CR := Value;
  if Value = 0 then PaletteReset;
end;

procedure Write_402(Value: Byte);
begin
  PS.Index.L := Value;
end;

procedure Write_403(Value: Byte);
begin
  PS.Index.H := Value and $1;
end;

procedure Write_404(Value: Byte);
begin
  PS.Palette[PS.Index.W].L := Value;
  SetPalette(PS.Index.W, PS.Palette[PS.Index.W].W);
end;

procedure Write_405(Value: Byte);
begin
  PS.Palette[PS.Index.W].H := Value and $1;
  SetPalette(PS.Index.W, PS.Palette[PS.Index.W].W);
  PS.Index.W := (PS.Index.W + 1) and $1FF;
end;

procedure Reset;
begin
  PaletteReset;
  PS.Index.W := 0;
end;

function LoadState(F: TBinaryFile): Boolean;
var
  i: Integer;
begin
  Result := False;
  try
    F.ReadCount(@PS, SizeOf(PS));
    for i := 0 to $1ff do SetPalette(i, PS.Palette[i].W);
    Result := True;
  except
  end;
end;

function SaveState(F: TBinaryFile): Boolean;
begin
  Result := False;
  try
    F.Write(@PS, SizeOf(PS));
    Result := True;
  except
  end;
end;

procedure PowerOn;
begin
  IOReadHandler[$402] := Read_402;
  IOReadHandler[$403] := Read_403;
  IOReadHandler[$404] := Read_404;
  IOReadHandler[$405] := Read_405;

  IOWriteHandler[$400] := Write_400;
  IOWriteHandler[$402] := Write_402;
  IOWriteHandler[$403] := Write_403;
  IOWriteHandler[$404] := Write_404;
  IOWriteHandler[$405] := Write_405;
  ResetList.Add(@Reset);
  LoadSaveStateList.Add(@LoadState);
  LoadSaveStateList.Add(@SaveState);
  Reset;
end;

end.
