unit IO_Sound;

interface

uses
  DXSounds;

var
  DXSound: TDXSound;

procedure PowerOn;

implementation

uses
  CPU, IO_Shared, LogUnit, SysUtils, MMSystem, Wave, PentiumTimer,
  DirectX, BinaryFile, Config;

type
  VoiceData = record
    Frequency: WordBytes;
    Status, Balance, Buffer, Noise: Byte;
    SetSamples: Boolean;
  end;
  PByte = ^Byte;
  PCardinal = ^Cardinal;
  TPS = record
    V, Volume: Byte;
    Voices: array[0..5] of VoiceData;
    Samples: array[0..5, 0..31] of Byte;
    LFOFrequency, LFOControl: Byte;
  end;

const
  NOISE_ON = $80;
  VOICE_ON = $80;
  DDA_ON = $40;
  LFO_OFF = $80;
  BASE_CLOCK = 3580000;
  SAMPLES_REPEAT_SHIFT = 5;
  SAMPLES_REPEAT = 1 shl SAMPLES_REPEAT_SHIFT;

var
  PS: TPS;
  SoundBuffer: array[0..5] of TDirectSoundBuffer;
  NoiseSamples: array[0..SAMPLES_REPEAT * 32 - 1] of Byte;
  SoundOn: Boolean;

procedure SetSamplesSound(V: Byte);
var
  i, S1, S2: Integer;
  P1, P2: PChar;
begin
  if not SoundOn then Exit;
  if not SoundBuffer[V].Lock(0, 32 * SAMPLES_REPEAT,
    Pointer(P1), S1, Pointer(P2), S2) then Exit;
  Assert(S2 = 0);
  P2 := @PS.Samples[V];
  for i := 0 to SAMPLES_REPEAT - 1 do
    begin
      PCardinal(P1 + $00)^ := PCardinal(P2 + $00)^;
      PCardinal(P1 + $04)^ := PCardinal(P2 + $04)^;
      PCardinal(P1 + $08)^ := PCardinal(P2 + $08)^;
      PCardinal(P1 + $0c)^ := PCardinal(P2 + $0c)^;
      PCardinal(P1 + $10)^ := PCardinal(P2 + $10)^;
      PCardinal(P1 + $14)^ := PCardinal(P2 + $14)^;
      PCardinal(P1 + $18)^ := PCardinal(P2 + $18)^;
      PCardinal(P1 + $1c)^ := PCardinal(P2 + $1c)^;
      P1 := PChar(P1) + 32;
    end;
  SoundBuffer[V].UnLock;
end;

procedure SetSamplesNoise(V: Byte);
var
  S1, S2: Integer;
  P1, P2: Pointer;
begin
  if not SoundOn then Exit;
  if not SoundBuffer[V].Lock(0, 32 * SAMPLES_REPEAT, P1, S1, P2, S2) then Exit;
  Assert(S2 = 0);
  Assert(S1 = SizeOf(NoiseSamples));
  Move(NoiseSamples, P1^, S1);
  SoundBuffer[V].UnLock;
end;

procedure PlayOn(V: Byte);
begin
  if not SoundOn then Exit;
  with PS do
    begin
      if (V <= 1) and (LFOControl and LFO_OFF <> 0) then
        begin
          if LFOControl and $3 <> 0 then Exit;
          if V = 1 then Exit;
        end;
      if not SoundBuffer[V].Playing then
        begin
          if Voices[V].SetSamples then
            begin
              if Voices[V].Noise and NOISE_ON = 0 then
                SetSamplesSound(V)
              else
                SetSamplesNoise(V);
              SoundBuffer[V].Position := 0;
              Voices[V].SetSamples := False;
            end;
          SoundBuffer[V].Play(True);
        end;
    end;
end;

procedure PlayOff(V: Byte);
begin
  if not SoundOn then Exit;
  if SoundBuffer[V].Playing then SoundBuffer[V].Stop;
end;

procedure SetSamplesDDA(V, Value: Byte);
var
  S1, S2: Integer;
  P1, P2: Pointer;
begin
  if not SoundOn then Exit;
  if not SoundBuffer[V].Lock(0, 32 * SAMPLES_REPEAT, P1, S1, P2, S2) then Exit;
  Assert(S2 = 0);
  Assert(S1 = SizeOf(NoiseSamples));
  FillChar(P1^, S1, Value);
  SoundBuffer[V].UnLock;
end;

procedure ChangeFrequency(V: Byte; Freq: Integer);
var
  F: Integer;
begin
  if not SoundOn then Exit;
  if Freq = 0 then
    begin
      SoundBuffer[V].Frequency := 0;
      Exit;
    end;
  F := BASE_CLOCK div Freq;
  //F := Max(Min(F, 100000), 100);
  SoundBuffer[V].Frequency := F;
end;

procedure ChangeVolume(V: Byte);
const
  PC_VOL: array[0..31] of Integer = (
    0,      2000,   3169,   4000,   4643,   5169,   5614,   6000,
    6339,   6643,   6918,   7169,   7400,   7614,   7813,   8000,
    8174,   8339,   8495,   8643,   8784,   8918,   9047,   9169,
    9287,   9400,   9509,   9614,   9715,   9813,   9908,   10000);
var
  Left, Right, MaxChannel, Vol: Integer;
begin
  if not SoundOn then Exit;
  with PS do
    begin
      Left := Min(Voices[V].Balance shr 4, Volume shr 4) shl 1;
      Right := Min(Voices[V].Balance and $f, Volume and $f) shl 1;
      if Right >= Left then
        begin
          Vol := 10000 - PC_VOL[31 - (Right - Left)];
          MaxChannel := Right;
        end
      else
        begin
          Vol := PC_VOL[31 - (Left - Right)] - 10000;
          MaxChannel := Right;
        end;
      SoundBuffer[V].Pan := Vol;

      Vol := Min(Voices[V].Status and $1f, MaxChannel);
      Vol := PC_VOL[Vol];

  {Left := Min(Voices[V].Balance shr 4, Volume shr 4);
  Right := Min(Voices[V].Balance and $f, Volume and $f);
  SoundBuffer[V].Pan := ((Right - Left) * 10000) div 15;
  Vol := Min(Voices[V].Status and $1f, Max(Left, Right) shl 1);
  Vol := (Vol * (DSBVOLUME_MAX - DSBVOLUME_MIN)) shr 5;}

  {Left := (Voices[V].Balance shr 4) * (Volume shr 4);
  Right := (Voices[V].Balance and $f) * (Volume and $f);
  SoundBuffer[V].Pan := ((Right - Left) * 10000) div (15 * 15);
  Vol := ((Voices[V].Status and $1f) * (DSBVOLUME_MAX - DSBVOLUME_MIN)) shr 5;
  Vol := (Vol * Max(Left, Right)) div (15 * 15);}

      SoundBuffer[V].Volume := DSBVOLUME_MIN + Vol;
    end;
end;

function Read_800: Byte;
begin
  Result := PS.V;
end;

function Read_801: Byte;
begin
  Result := PS.Volume;
end;

function Read_802: Byte;
begin
  Result := PS.Voices[PS.V].Frequency.L;
end;

function Read_803: Byte;
begin
  Result := PS.Voices[PS.V].Frequency.H;
end;

function Read_804: Byte;
begin
  Result := PS.Voices[PS.V].Status;
end;

function Read_805: Byte;
begin
  Result := PS.Voices[PS.V].Balance;
end;

function Read_806: Byte;
begin
  with PS do begin
    Result := Samples[V, Voices[V].Buffer] shr 3;
    Voices[V].Buffer := (Voices[V].Buffer + 1) and $1f;
  end;
end;

function Read_807: Byte;
begin
  Result := PS.Voices[PS.V].Noise;
end;

function Read_808: Byte;
begin
  Result := PS.LFOFrequency;
end;

function Read_809: Byte;
begin
  Result := PS.LFOControl;
end;

procedure Write_800(Value: Byte);
begin
  PS.V := (Value and $07) mod 6;
end;

procedure Write_801(Value: Byte);
var
  i: Integer;
begin
  with PS do begin
    if Volume = Value then Exit;
    Volume := Value;
    for i := 0 to 5 do ChangeVolume(i);
  end;
end;

procedure Write_802(Value: Byte);
begin
  with PS do begin
    if Voices[V].Frequency.L = Value then Exit;
    Voices[V].Frequency.L := Value;
    if PS.Voices[V].Noise and NOISE_ON = 0 then
      ChangeFrequency(V, Voices[V].Frequency.W);
  end;
end;

procedure Write_803(Value: Byte);
begin
  with PS do begin
    if Voices[V].Frequency.H = Value then Exit;
    Voices[V].Frequency.H := Value and $f;
    if PS.Voices[V].Noise and NOISE_ON = 0 then
      ChangeFrequency(V, Voices[V].Frequency.W);
  end;
end;

procedure Write_804(Value: Byte);
begin
  with PS do begin
    if Voices[V].Status = Value then Exit;
    if Value and VOICE_ON <> 0 then
      begin
        if (Value xor Voices[V].Status) and DDA_ON <> 0 then
          if Value and DDA_ON = 0 then SetSamplesSound(V);
        PlayOn(V);
      end
    else
      begin
        PlayOff(V);
        if Value and DDA_ON <> 0 then Voices[V].Buffer := 0;
      end;
    if (Voices[V].Status xor Value) and $1f <> 0 then
      begin
        Voices[V].Status := Value;
        ChangeVolume(V);
      end;
    Voices[V].Status := Value;
  end;
end;

procedure Write_805(Value: Byte);
begin
  with PS do begin
    if Voices[V].Balance = Value then Exit;
    Voices[V].Balance := Value;
    ChangeVolume(V);
  end;
end;

procedure Write_806(Value: Byte);
begin
  with PS do begin
    Value := Value shl 3;
    if Voices[V].Status and DDA_ON = DDA_ON then
      SetSamplesDDA(V, Value)
    else
      if Voices[V].Status and (VOICE_ON + DDA_ON) = 0 then
      begin
        if Samples[V, Voices[V].Buffer] <> Value then
          begin
            Samples[V, Voices[V].Buffer] := Value;
            Voices[V].SetSamples := True;
          end;
        Voices[V].Buffer := (Voices[V].Buffer + 1) and $1f;
      end;
  end;
end;

procedure Write_807(Value: Byte);
begin
  with PS do begin
    if Value = Voices[V].Noise then Exit;
    if (V < 4) or (V > 5) then Exit;
    if Value and NOISE_ON <> 0 then
      begin
        ChangeFrequency(V, Value and $1f);
        SetSamplesNoise(V);
      end
    else
      begin
        SetSamplesSound(V);
        ChangeFrequency(V, PS.Voices[V].Frequency.W);
      end;
    Voices[V].Noise := Value;
  end;
end;

procedure Write_808(Value: Byte);
begin
  PS.LFOFrequency := Value;
end;

procedure Write_809(Value: Byte);
begin
  with PS do begin
    LFOControl := Value;
    if Value and LFO_OFF = 0 then
      begin
        if Value and $3 <> 0 then PlayOff(0);
        PlayOff(1);
      end
    else
      begin
        if Voices[0].Status and VOICE_ON <> 0 then PlayOn(0);
        if Voices[1].Status and VOICE_ON <> 0 then PlayOn(1);
      end;
  end;
end;

procedure CreateBuffers;
var
  i: Integer;
  SoundFormat: TWaveFormatEx;
  Desc: TDSBufferDesc;
begin
  Assert(SoundOn);
  MakePCMWaveFormatEx(SoundFormat, 8192, 8, 1);
  FillChar(Desc, SizeOf(Desc), 0);
  with Desc do
    begin
      dwSize := SizeOf(Desc);
      dwFlags := DSBCAPS_CTRLDEFAULT or DSBCAPS_STATIC;
      dwBufferBytes := 32 * SAMPLES_REPEAT;
      lpwfxFormat := @SoundFormat;
    end;
  try
    FillChar(SoundBuffer[0], SizeOf(SoundBuffer), 0);
    for i := 0 to 5 do
      begin
        SoundBuffer[i] := TDirectSoundBuffer.Create(DXSound.DSound);
        if not SoundBuffer[i].CreateBuffer(Desc) then
          raise EDirectSoundError.Create('Cannot Create DXSound Buffer');
        SoundBuffer[i].Volume := -10000;
        SoundBuffer[i].Pan := 0;
      end;
  except
    for i := 0 to 5 do FreeAndNil(SoundBuffer[i]);
    SoundOn := False;
  end;
end;

procedure PowerOff;
var
  i: Integer;
begin
  if SoundOn then
    for i := 0 to 5 do
      FreeAndNil(SoundBuffer[i]);
end;

procedure Pause;
var
  i: Integer;
begin
  Assert(SoundOn);
  for i := 0 to 5 do SoundBuffer[i].Stop;
end;

procedure Resume;
var
  i: Integer;
begin
  Assert(SoundOn);
  for i := 0 to 5 do
    if PS.Voices[i].Status and VOICE_ON <> 0 then PlayOn(i);
end;

procedure ResetBuffers;
var
  i: Integer;
begin
  Assert(SoundOn);
  for i := 0 to 5 do
    begin
      SoundBuffer[i].Stop;
      SoundBuffer[i].Frequency := 0;
      SoundBuffer[i].Volume := -10000;
      SoundBuffer[i].Pan := 0;
      SetSamplesDDA(i, 0);
    end;
end;

procedure Reset;
var
  i: Integer;
begin
  with PS do begin
    Volume := 0;
    V := 0;
    LFOFrequency := 0;
    LFOControl := LFO_OFF;
    for i := 0 to 5 do
      begin
        Voices[i].Frequency.W := 0;
        Voices[i].Balance := 0;
        Voices[i].Buffer := 0;
        Voices[i].Noise := 0;
        Voices[i].Status := 0;
        FillChar(Samples[i][0], 32, 0);
      end;
    if SoundOn then ResetBuffers;
  end;
end;

function LoadState(F: TBinaryFile): Boolean;
var
  i: Integer;
begin
  Result := False;
  try
    F.ReadCount(@PS, SizeOf(PS));
    if SoundOn then
      for i := 0 to 5 do
        with PS do
          begin
            Voices[i].SetSamples := True;
            if Voices[i].Noise and NOISE_ON <> 0 then
              ChangeFrequency(i, Voices[i].Frequency.W)
            else
              ChangeFrequency(i, Voices[i].Noise and $1f);
            ChangeVolume(i);
          end;
    Result := True;
  except
  end;
end;

function SaveState(F: TBinaryFile): Boolean;
begin
  Result := False;
  try
    F.Write(@PS, SizeOf(PS));
    Result := True;
  except
  end;
end;

procedure PowerOn;
var
  i: Integer;
begin
  PowerOffList.Add(@PowerOff);
  ResetList.Add(@Reset);
  LoadSaveStateList.Add(@LoadState);
  LoadSaveStateList.Add(@SaveState);
  IOReadHandler[$800] := Read_800;
  IOReadHandler[$801] := Read_801;
  IOReadHandler[$802] := Read_802;
  IOReadHandler[$803] := Read_803;
  IOReadHandler[$804] := Read_804;
  IOReadHandler[$805] := Read_805;
  IOReadHandler[$806] := Read_806;
  IOReadHandler[$807] := Read_807;
  IOReadHandler[$808] := Read_808;
  IOReadHandler[$809] := Read_809;
  IOWriteHandler[$800] := Write_800;
  IOWriteHandler[$801] := Write_801;
  IOWriteHandler[$802] := Write_802;
  IOWriteHandler[$803] := Write_803;
  IOWriteHandler[$804] := Write_804;
  IOWriteHandler[$805] := Write_805;
  IOWriteHandler[$806] := Write_806;
  IOWriteHandler[$807] := Write_807;
  IOWriteHandler[$808] := Write_808;
  IOWriteHandler[$809] := Write_809;
  SoundOn := DXSound.Initialized;
  if SoundOn then CreateBuffers;
  if SoundOn then
    begin
      PauseList.Add(@Pause);
      ResumeList.Add(@Resume);
      Randomize;
      for i := 0 to SAMPLES_REPEAT * 32 - 1 do
        NoiseSamples[i] := Random(2) - 1;
    end;
  Reset;
end;

end.
