unit IO;

interface

uses
  BinaryFile;

procedure PowerOn;
procedure PowerOff;
procedure Pause;
procedure Resume;
procedure Reset;
function LoadState(F: TBinaryFile): Boolean;
function SaveState(F: TBinaryFile): Boolean;

implementation

uses
  CPU, SysUtils, LogUnit, IO_Shared,
  IO_Palette, IO_Timer, IO_IRQ, IO_Arcade, IO_Joystick,
  IO_CDROM, IO_Video, IO_Sound, Classes;

type
  TProc = procedure;
  TFuncState = function(F: TBinaryFile): Boolean;

const
  PowerOnDevices: array[0..7] of procedure = (
    IO_Palette.PowerOn, IO_Timer.PowerOn, IO_IRQ.PowerOn,
    IO_Arcade.PowerOn,  IO_CDROM.PowerOn, IO_Video.PowerOn,
    IO_Sound.PowerOn,   IO_Joystick.PowerOn);

procedure Write_Invalid(Value: Byte);
begin
  //Log(9999, 'invalid i/o write');
  Inc(InvalidWrites);
end;

function Read_Invalid: Byte;
begin
  Inc(InvalidReads);
  //Log(9999, 'invalid i/o read');
  Result := $ff;
end;

procedure PowerOn;
var
  i: Integer;
begin
  if IsOn then
    begin
      Reset;
      Exit;
    end;
  for i := 0 to $1fff do
    begin
      IOReadHandler[i] := Read_Invalid;
      IOWriteHandler[i] := Write_Invalid;
    end;
  PowerOffList := TList.Create;
  PauseList := TList.Create;
  ResumeList := TList.Create;
  ResetList := TList.Create;
  LoadSaveStateList := TList.Create;
  for i := Low(PowerOnDevices) to High(PowerOnDevices) do PowerOnDevices[i];
  IsOn := True;
  IsPaused := False;
  InvalidWrites := 0;
  InvalidReads := 0;
end;

procedure PowerOff;
var
  i: Integer;
begin
  if not IsOn then Exit;
  for i := 0 to PowerOffList.Count - 1 do TProc(PowerOffList[i]);
  LoadSaveStateList.Free;
  PowerOffList.Free;
  PauseList.Free;
  ResumeList.Free;
  ResetList.Free;
  IsOn := False;
end;

procedure Pause;
var
  i: Integer;
begin
  if IsPaused then Exit;
  for i := 0 to PauseList.Count - 1 do TProc(PauseList[i]);
  IsPaused := True;
end;

procedure Resume;
var
  i: Integer;
begin
  if not IsPaused then Exit;
  for i := 0 to ResumeList.Count - 1 do TProc(ResumeList[i]);
  IsPaused := False;
end;

procedure Reset;
var
  i: Integer;
begin
  for i := 0 to ResetList.Count - 1 do TProc(ResetList[i]);
end;

function LoadState(F: TBinaryFile): Boolean;
var
  Sig: array[0..3] of Char;
  i: Integer;
begin
  Result := False;
  F.ReadCount(@Sig[0], 4);
  if Sig <> 'SFIO' then Exit;
  for i := 0 to LoadSaveStateList.Count shr 1 - 1 do
    if not TFuncState(LoadSaveStateList[i shl 1])(F) then Exit;
  Result := True;
end;

function SaveState(F: TBinaryFile): Boolean;
var
  i: Integer;
begin
  Result := False;
  F.Write('SFIO', 4);
  for i := 0 to LoadSaveStateList.Count shr 1 - 1 do
    if not TFuncState(LoadSaveStateList[i shl 1 + 1])(F) then Exit;
  Result := True;
end;

end.
