unit TextFiles;

interface

uses
  SysUtils, BinaryFile;

type
  ETextFile = class(Exception);
  TTextFileRead = class
  private
    FFile: TBinaryFile;
    FBuffer: PChar;
    FStringPos, FBufferCount: Integer;
    procedure FillBuffer;
    function FindString: Boolean;
    function GetString: string;
  public
    constructor Create;
    destructor Destroy; override;
    procedure Open(Filename: string);
    procedure Close;
    function Eof: Boolean;
    function Read: string;
    function ReadInt: Integer;
    function ReadBool: Boolean;
    function isOpen: Boolean;
  end;
  TTextFileWrite = class
  private
    FFile: TBinaryFile;
    FBuffer: PChar;
    FBufferCount: Integer;
    procedure FlushBuffer;
    procedure AddToBuffer(Text: PChar; Size: Integer);
  public
    constructor Create;
    destructor Destroy; override;
    procedure CreateFile(Filename: string);
    procedure AppendFile(Filename: string);
    procedure Close;
    procedure Flush;
    function isOpen: Boolean;
    procedure Write(Text: string); overload;
    procedure Write(Text: Integer); overload;
    procedure Write(Text: Boolean); overload;
  end;

implementation

const
  CR_LF = #13 + #10;
  BUFFER_SIZE = 32 * 1024;

function TTextFileRead.isOpen: Boolean;
begin
  Result := FFile.isOpen;
end;

constructor TTextFileRead.Create;
begin
  inherited Create;
  GetMem(FBuffer, BUFFER_SIZE);
  FFile := TBinaryFile.Create;
  FStringPos := 0;
  FBufferCount := 0;
end;

destructor TTextFileRead.Destroy;
begin
  FFile.Free;
  FreeMem(FBuffer);
  inherited Destroy;
end;

procedure TTextFileRead.Open(Filename: string);
begin
  FFile.OpenReadOnly(Filename);
end;

procedure TTextFileRead.Close;
begin
  FFile.Close;
end;

procedure TTextFileRead.FillBuffer;
begin
  FBufferCount := FFile.Read(FBuffer, BUFFER_SIZE - 1);
  FBuffer[FBufferCount] := #0;
  FStringPos := 0;
end;

function TTextFileRead.FindString: Boolean;
begin
  Result := True;
  if FStringPos < FBufferCount then Exit;
  FillBuffer;
  Result := FBufferCount > 0;
end;

function TTextFileRead.GetString: string;
var
  LEnd: PChar;
  LStr: string;
  LSize: Integer;
begin
  Result := '';
  repeat
    LEnd := StrPos(FBuffer + FStringPos, CR_LF);
    if LEnd <> nil then
    begin
      LSize := LEnd - FBuffer - FStringPos;
    end else
    begin
      LSize := FBufferCount - FStringPos;
      FillBuffer;
    end;
    SetString(LStr, FBuffer + FStringPos, LSize);
    Inc(FStringPos, LSize + 2);
    Result := Result + LStr;
  until (LEnd <> nil) or (FBufferCount = 0);
end;

function TTextFileRead.Eof: Boolean;
begin
  Result := not FindString;
end;

function TTextFileRead.Read: string;
begin
  if not FindString then raise ETextFile.Create('EOF');
  Result := GetString;
end;

function TTextFileRead.ReadInt: Integer;
begin
  Result := StrToInt(Read);
end;

function TTextFileRead.ReadBool: Boolean;
begin
  Result := Boolean(ReadInt);
end;

constructor TTextFileWrite.Create;
begin
  FFile := TBinaryFile.Create;
  GetMem(FBuffer, BUFFER_SIZE);
  FBufferCount := 0;
end;

destructor TTextFileWrite.Destroy;
begin
  Flush;
  FreeMem(FBuffer);
  FFile.Free;
end;

procedure TTextFileWrite.CreateFile(Filename: string);
begin
  FFile.CreateFile(Filename);
end;

procedure TTextFileWrite.AppendFile(Filename: string);
begin
  FFile.OpenReadWrite(Filename, True);
  FFile.SetPositionEnd(0);
end;

procedure TTextFileWrite.Close;
begin
  Flush;
  FFile.Close;
end;

procedure TTextFileWrite.FlushBuffer;
begin
  if FBufferCount > 0 then
    begin
      FFile.Write(FBuffer, FBufferCount);
      FBufferCount := 0;
    end;
end;

procedure TTextFileWrite.Flush;
begin
  if FBufferCount > 0 then FlushBuffer;
  FFile.Flush;
end;

function TTextFileWrite.isOpen: Boolean;
begin
  Result := FFile.isOpen;
end;

procedure TTextFileWrite.AddToBuffer(Text: PChar; Size: Integer);
var
  i, n: Integer;
  Dest: PChar;
begin
  repeat
    Dest := FBuffer + FBufferCount;
    n := BUFFER_SIZE - FBufferCount;
    if Size < n then n := Size;
    for i := 0 to n - 1 do
      (Dest + i)^ := (Text + i)^;
    Inc(Text, n);
    Dec(Size, n);
    Inc(FBufferCount, n);
    if Size = 0 then Break;
    FlushBuffer;
  until False;
end;

procedure TTextFileWrite.Write(Text: string);
begin
  AddToBuffer(PChar(Text), Length(Text));
  AddToBuffer(PChar(CR_LF), 2);
end;

procedure TTextFileWrite.Write(Text: Integer);
begin
  Write(IntToStr(Text));
end;

procedure TTextFileWrite.Write(Text: Boolean);
begin
  Write(IntToStr(Ord(Text)));
end;

end.
