unit PCE;

interface

procedure PowerOn;
function LoadRom(Filename: string): Boolean;
procedure DiscardRom;
procedure PowerOff;
procedure Run;
procedure Pause;
procedure Reset;
function Active: Boolean;
function Paused: Boolean;
function Runnable: Boolean;
function RomName: string;
function SaveState(Filename: string): Boolean;
function LoadState(Filename: string): Boolean;

implementation

uses
  CPU, BinaryFile, SysUtils, LogUnit, IO, IO_Video, IO_Timer, TextFiles,
  Classes, SyncObjs, Windows, PentiumTimer, Dialogs, IO_Shared;

type
  TPCEThread = class(TThread)
  protected
    procedure Execute; override;
  end;

const
  BASE_CLOCK = 3580000 * 2;
  SCANLINES_PER_FRAME = 263;
  FRAMES_PER_SECOND = 60;
  SCANLINE_CYCLES = BASE_CLOCK div SCANLINES_PER_FRAME div FRAMES_PER_SECOND;
  TIMER_TICKS = BASE_CLOCK div 1000 * 3 * 1024 div 21480;
  STATE_SIG = 'SFPCE03';
  STATE_SIG_SIZE = Length(STATE_SIG);

var
  IsActive: Boolean = False;
  HasRom, IsPaused: Boolean;
  ScanlineCycles, Timer: Integer;
  Rom: PChar = nil;

  PCEThread: TPCEThread;
  ToPCE, FromPCE: TEvent;
  ActiveRomName: string;

function RomName: string;
begin
  Result := ActiveRomName;
end;

procedure Reset;
begin
  if not IsActive then Exit;
  if not HasRom then Exit;
  Pause;
  CPU.Reset;
  IO.Reset;
  Run;
end;

procedure PowerOn;
begin
  if IsActive then Exit;
  IsActive := True;
  IsPaused := True;
  HasRom := False;
  ToPCE := TEvent.Create(nil, False, False, '');
  FromPCE := TEvent.Create(nil, False, False, '');
  PCEThread := TPCEThread.Create(False);
  FromPCE.WaitFor(INFINITE);
  IO.PowerOn;
  ActiveRomName := '';
end;

procedure Run;
begin
  if not IsActive then Exit;
  if not HasRom then Exit;
  if not IsPaused then Exit;
  IsPaused := False;
  IO.Resume;
  ToPCE.SetEvent;
  FromPCE.WaitFor(INFINITE);
  //IO.Resume;
end;

procedure Pause;
begin
  if not IsActive then Exit;
  if not HasRom then Exit;
  if IsPaused then Exit;
  IsPaused := True;
  FromPCE.WaitFor(INFINITE);
  IO.Pause;
end;

function Schedule(Cycles: Integer): Integer;
var
  Done, Todo, Ret: Integer;
begin
  Result := 0;
  //if IsPaused or not IsActive then Exit;
  Todo := Cycles;
  if ScanlineCycles < Todo then Todo := ScanlineCycles;
  if Timer < Todo then Todo := Timer;
  while (Result < Cycles) and not IsPaused do
    begin
      Done := CPU.State.CyclesDone;

      Ret := CPU.Run(Todo);
      Done := CPU.State.CyclesDone - Done;
      Inc(Result, Done);

      if Ret <> RET_OK then
        begin
          IsPaused := True;
          IO.Pause;
          Result := -1;
        end;

      Dec(ScanlineCycles, Done);
      while ScanlineCycles <= 0 do
        begin
          IO_Video.DrawScanline;
          Inc(ScanlineCycles, SCANLINE_CYCLES);
        end;

      Dec(Timer, Done);
      while Timer <= 0 do
        begin
          IO_Timer.Tick;
          Inc(Timer, TIMER_TICKS);
        end;
    end;
    if (Result < Cycles) and not IsPaused then
      Result := Result + Schedule(Cycles - Result);
end;

function Active: Boolean;
begin
  Result := IsActive;
end;

function Runnable: Boolean;
begin
  Result := HasRom;
end;

function Paused: Boolean;
begin
  Result := IsPaused;
end;

procedure DiscardRom;
begin
  if not IsActive then Exit;
  if not HasRom then Exit;
  if not IsPaused then
    begin
      IsPaused := True;
      FromPCE.WaitFor(INFINITE);
      //CPU.Reset;
      IO.Reset;
    end;
  FreeMem(Rom);
  Rom := nil;
  HasRom := False;
  ActiveRomName := '';
end;

procedure PowerOff;
begin
  if not IsActive then Exit;
  if HasRom then DiscardRom;
  IO.PowerOff;
  IsActive := False;
  ToPCE.SetEvent;
  FromPCE.WaitFor(INFINITE);
  PCEThread := nil;
  FromPCE.Free;
  ToPCE.Free;
end;

function LoadRom(Filename: string): Boolean;
var
  RFile: TBinaryFile;
  i, RFileSize, RSize: Integer;
begin
  Result := False;
  if not IsActive then Exit;
  if HasRom then DiscardRom;
  RFile := nil;
  try
    RFile := TBinaryFile.Create;
    RFile.OpenReadOnly(Filename);
    RFileSize := RFile.GetSize;
    RSize := RFileSize shr 13;
    GetMem(Rom, RSize * $2008);
    RFile.SetPositionBegin(RFileSize and $1fff);
    for i := 0 to RSize - 1 do
      RFile.Read(Rom + i * $2008, $2000);
    RFile.Close;
    RFile.Free;
  except
    on E: Exception do
      begin
        RFile.Free;
        Log(999999, 'Unable to Load ROM: %s (%s)', [Filename, E.Message]);
        Exit;
      end;
  end;
  CPU.SetRom(Rom, RSize);
  CPU.Reset;
  //IO.Reset;
  HasRom := True;
  ScanlineCycles := SCANLINE_CYCLES;
  Timer := TIMER_TICKS;
  Result := True;
  ActiveRomName := Filename;
end;

procedure TPCEThread.Execute;
label
  WaitRun, ExecSlice, ExecPaused;
begin
  FreeOnTerminate := True;
  Priority := tpHigher;
  FromPCE.SetEvent;
  if IsPaused then goto WaitRun else goto ExecSlice;
ExecPaused:
  FromPCE.SetEvent;
WaitRun:
  if not IsActive then Exit;
  ToPCE.WaitFor(INFINITE);
  FromPCE.SetEvent;
  if IsPaused then goto WaitRun;
ExecSlice:
  if Schedule(BASE_CLOCK) < 0 then goto WaitRun; // bad opcode - paused
  if IsPaused then
    goto ExecPaused
  else
    goto ExecSlice;
end;

function SaveState(Filename: string): Boolean;
var
  WasPaused: Boolean;
  F: TBinaryFile;
begin
  Result := False;
  if not IsActive then Exit;
  if not HasRom then Exit;
  WasPaused := IsPaused;
  Pause;
  F := nil;
  try
    F := TBinaryFile.Create;
    F.CreateFile(Filename);
    F.Write(STATE_SIG, STATE_SIG_SIZE);
    if CPU.SaveState(F) then if IO.SaveState(F) then Result := True;
    F.Close;
  except
  end;
  F.Free;
  if not WasPaused then Run;
end;

function LoadState(Filename: string): Boolean;
var
  WasPaused: Boolean;
  F: TBinaryFile;
  Sig: array[0..STATE_SIG_SIZE - 1] of Char;
label
  Discard;
begin
  Result := False;
  if not IsActive then Exit;
  if not HasRom then Exit;
  WasPaused := IsPaused;
  Pause;
  F := nil;
  try
    F := TBinaryFile.Create;
    F.OpenReadOnly(Filename);
    F.ReadCount(@Sig[0], STATE_SIG_SIZE);
    if Sig = STATE_SIG then
      begin
        if not CPU.LoadState(F) then
          DiscardRom
        else
          if not IO.LoadState(F) then
            DiscardRom
          else
            Result := True;
      end;
    F.Close;
  except
  end;
  F.Free;
  if HasRom and not WasPaused then Run;
end;

initialization
finalization
  //if IsActive then Terminate;
  //if Rom <> nil then FreeMem(Rom);
end.

