/*****************************************************************************
 *                                                                           *
 * Module : UPD.C                                                            *
 *                                                                           *
 * Gestion du circuit UPD 765 (Controlleur disquette)                       *
 *                                                                           *
 *****************************************************************************/


#include  <exec/types.h>

#include  <string.h>
#include  <fcntl.h>
#include  <io.h>


#include  <dos.h>
#include  "vga.h"


//
// Constantes gnrales...
//
#define SECTSIZE   512
#define NBSECT     9


// Bits de Status
#define DIO         0x40
#define NDM         0x20
#define CB          0x10

#define N_DIO       0xBF
#define N_NDM       0xDF
#define N_CB        0xEF

#define T0          0x10
#define N_T0        0xE0


static SHORT handle = 0;


typedef struct
    {
    char  debut[ 0x30 ]; // "MV - CPCEMU Disk-File\r\nDisk-Info\r\n"
    UBYTE NbTracks;
    UBYTE NbHeads;
    SHORT DataSize; // 0x1300 = 256 + ( 512 * nbsecteurs )
    UBYTE Unused[ 0xCC ];
    } CPCEMUEnt;


typedef struct
    {
    UBYTE C;                 // track
    UBYTE H;                 // head
    UBYTE R;                 // sect
    UBYTE N;                 // size
    SHORT Un1;
    SHORT Un2;
    } CPCEMUSect;


typedef struct
    {
    char        ID[ 0x10 ];   // "Track-Info\r\n"
    UBYTE       Track;
    UBYTE       Head;
    SHORT       Unused;
    UBYTE       SectSize; // 2
    UBYTE       NbSect;   // 9
    UBYTE       Gap3;     // 0x4E
    UBYTE       OctRemp;  // 0xE5
    CPCEMUSect  Sect[ 29 ];
    } CPCEMUTrack;


typedef UBYTE ( * pfctUPD )( UBYTE );

typedef struct
    {
    UBYTE C;
    UBYTE H;
    UBYTE R;
    UBYTE N;
    UBYTE datas[ SECTSIZE ];
    } StSect;

typedef struct
    {
    StSect Sect[ NBSECT ];
    } StTrack;


static UBYTE Rien( UBYTE );

static pfctUPD fct = Rien;

static UBYTE etat = 0;

static StTrack CurrTrackDatasCPC;

static CPCEMUTrack CurrTrackDatasDSK;

static CPCEMUEnt Infos;

static UBYTE far Datas[ 0x4000 ];


//
// Registres du uPD 765
//
static UBYTE Status = 0x80;  // status
static UBYTE ST0    = 0x01;  // status interne 0
static UBYTE ST1    = 0x00;  // status interne 1
static UBYTE ST2    = 0x00;  // status interne 2
static UBYTE ST3    = 0x29;  // status interne 3
static UBYTE C      = 0x00;  // Cylindre (n piste)
static UBYTE H      = 0x00;  // Head     (n tte)
static UBYTE R      = 0xC1;  // Record   (n secteur)
static UBYTE N      = 0x02;  // Number   (nbre d'octet poids forts)
static UBYTE Drive  = 0x00;  // Drive    ( 0=A, 1=B)
static UBYTE EOT    = 0xC1;  // Secteur final;


static void ( * FctWriteUPD )( UBYTE );


inline void ChangeCurrTrack_CPC( UBYTE newTrack )
{
    if ( ! newTrack )
        ST0 |= T0;
    else
        ST0 &= N_T0;

    if ( handle != -1 )
        {
        lseek( handle, ( ULONG )newTrack * sizeof( CurrTrackDatasCPC ), SEEK_SET );
        read( handle, &CurrTrackDatasCPC, sizeof( CurrTrackDatasCPC ) );
        }
    C = CurrTrackDatasCPC.Sect[ 0 ].C;
    H = CurrTrackDatasCPC.Sect[ 0 ].H;
    R = CurrTrackDatasCPC.Sect[ 0 ].R;
    N = CurrTrackDatasCPC.Sect[ 0 ].N;
}


inline void WriteCurrTrack_CPC( UBYTE Track )
{
    if ( handle != -1 )
        {
        lseek( handle, ( ULONG )Track * sizeof( CurrTrackDatasCPC ), SEEK_SET );
        write( handle, &CurrTrackDatasCPC, sizeof( CurrTrackDatasCPC ) );
        }
}


static UBYTE RechercheSecteur_CPC( UBYTE newSect )
{
    USHORT i;

    for ( i = 0; i < NBSECT; i++ )
        if ( CurrTrackDatasCPC.Sect[ i ].R == newSect )
            return i;

    return 0xFF;
}


static UBYTE Rien( UBYTE val )
{
    Status &= N_CB & N_DIO;
    etat = 0;
    val = 0;
    return val;
}


static UBYTE MoveTrack_CPC( UBYTE val )
{
    switch( etat++ )
        {
        case 1 :
            Drive = val;
            Status |= NDM;
            break;

        case 2 :
            ChangeCurrTrack_CPC( C = val );
            etat = 0;
            Status &= N_CB & N_DIO & N_NDM;
            break;
        }
    return 0;
}


static UBYTE MoveTrack0_CPC( UBYTE val )
{
    Drive = val;
    ChangeCurrTrack_CPC( C = 0 );
    etat = 0;
    Status &= N_CB & N_DIO & N_NDM;
    return 0;
}


static UBYTE ReadST0( UBYTE val )
{
    if ( etat++ == 1)
        return ST0;

    etat = 0;
    Status &= N_CB & N_DIO;
    return C;
}


static UBYTE ReadST3( UBYTE val )
{
    if ( etat++ == 1 )
        {
        Drive = val;
        Status |= DIO;
        return 0;
        }
    etat = 0;
    Status &= N_CB & N_DIO;
    return ST3;
}


//
// Commande specify, non utilise
//
static UBYTE Specify( UBYTE val )
{
    if ( etat++ == 1 )
        return 0;

    etat = 0;
    Status &= N_CB & N_DIO;
    return 0;
}


//
// Lecture de n secteurs
//
static UBYTE ReadData_CPC( UBYTE val )
{
    static USHORT sect = 0, cntdata = 0;
    static BOOL cond = TRUE;
    UBYTE fin, donnee;

    switch( etat++ )
        {
        case 1 :
            Drive = val;
            break;

        case 2 :
            C = val;
            break;

        case 3 :
            H = val;
            break;

        case 4 :
            cond = TRUE;
            sect = RechercheSecteur_CPC( R = val );
            if ( sect == 0xFF )
                cond = FALSE;
            break;

        case 5 :
            N = val;
            break;

        case 6 :
            EOT = val;
            break;

        case 7 :
            if ( cond == TRUE )
                {
                if ( ( C != CurrTrackDatasCPC.Sect[ sect ].C ) ||
                     ( H != CurrTrackDatasCPC.Sect[ sect ].H ) ||
                     ( R != CurrTrackDatasCPC.Sect[ sect ].R ) ||
                     ( N != CurrTrackDatasCPC.Sect[ sect ].N )
                   )
                    cond = FALSE;
                }
            cntdata = 0;
            break;

        case 8 :
            Status |= DIO | NDM;
            break;

        case 9 :
            if ( cond )
                {
                fin = 0;
                donnee = CurrTrackDatasCPC.Sect[ sect ].datas[ cntdata++ ];
                if ( cntdata == SECTSIZE )
                    {
                    fin = 1;
                    if ( R++ < EOT )
                        {
                        sect = RechercheSecteur_CPC( R );
                        if ( sect == 0xFF )
                             cond = FALSE;
                        else
                            if ( R != CurrTrackDatasCPC.Sect[ sect ].R )
                                cond = FALSE;
                        fin = 0;
                        }
                    cntdata = 0;
                    }
                if ( ! fin )
                    etat--;
                else
                    Status &= N_NDM;
                return donnee;
                }
            Status &= N_NDM;
            return 0;

        case 10 :
            return ST0;

        case 11 :
            return ST1;

        case 12 :
            return ST2;

        case 13 :
            return C;

        case 14 :
            return H;

        case 15 :
            return R;

        case 16 :
            etat = 0;
            Status &= N_CB & N_DIO;
            return N;
        }
    return 0;
}


//
// Ecriture de n secteurs
//
static UBYTE WriteData_CPC( UBYTE val )
{
    static USHORT sect = 0, cntdata = 0;
    static BOOL cond = TRUE;
    UBYTE fin;

    switch( etat++ )
        {
        case 1 :
            Drive = val;
            break;

        case 2 :
            C = val;
            break;

        case 3 :
            H = val;
            break;

        case 4 :
            cond = TRUE;
            sect = RechercheSecteur_CPC( R = val );
            if ( sect == 0xFF )
                cond = FALSE;
            break;

        case 5 :
            N = val;
            break;

        case 6 :
            EOT = val;
            break;

        case 7 :
            if ( cond == TRUE )
                {
                if ( ( C != CurrTrackDatasCPC.Sect[ sect ].C ) ||
                     ( H != CurrTrackDatasCPC.Sect[ sect ].H ) ||
                     ( R != CurrTrackDatasCPC.Sect[ sect ].R ) ||
                     ( N != CurrTrackDatasCPC.Sect[ sect ].N )
                   )
                    cond = FALSE;
                }
            cntdata = 0;
            break;

        case 8 :
            Status |= DIO | NDM;
            break;

        case 9 :
            if ( cond )
                {
                fin = 0;
                CurrTrackDatasCPC.Sect[ sect ].datas[ cntdata++ ] = val;
                if ( cntdata == SECTSIZE )
                    {
                    fin = 1;
                    if ( R++ < EOT )
                        {
                        sect = RechercheSecteur_CPC( R );
                        if ( sect == 0xFF )
                             cond = FALSE;
                        else
                            if ( R != CurrTrackDatasCPC.Sect[ sect ].R )
                                cond = FALSE;
                        fin = 0;
                        }
                    cntdata = 0;
                    }
                if ( ! fin )
                    etat--;
                else
                    Status &= N_NDM;
                return 0;
                }
            Status &= N_NDM;
            return 0;

        case 10 :
            WriteCurrTrack_CPC( C );
            return ST0;

        case 11 :
            return ST1;

        case 12 :
            return ST2;

        case 13 :
            return C;

        case 14 :
            return H;

        case 15 :
            return R;

        case 16 :
            etat = 0;
            Status &= N_CB & N_DIO;
            return N;
        }
    return 0;
}


static UBYTE ReadID( UBYTE val )
{
    switch( etat++ )
        {
        case 1 :
            Drive = val;
            Status |= DIO;
            break;

        case 2 :
            return ST0;

        case 3 :
            return ST1;

        case 4 :
            return ST2;

        case 5 :
            return C;

        case 6 :
            return H;

        case 7 :
            return R;

        case 8 :
            etat = 0;
            Status &= N_CB & N_DIO;
            return N;
        }
    return 0;
}

/*
static UBYTE FormatTrack( UBYTE val )
{
    etat = 0;
    Status &= N_CB & N_DIO;
    return 0;
}
*/

UBYTE ReadUPD( USHORT port )
{
    if ( port == 0xFB7E )
        return( Status );

    return( fct( 0 ) );
}


void WriteUPD_CPC( UBYTE val )
{
    if ( etat )
        fct( val );
    else
        {
        Status |= CB;
        etat = 1;
        switch( val & 0x5F )
            {
            case 0x03 :
                // Specify
                fct = Specify;
                break;

            case 0x04 :
                // Lecture ST3
                fct = ReadST3;
                break;

            case 0x07 :
                // Dplacement tte piste 0
                Status |= NDM;
                fct = MoveTrack0_CPC;
                break;

            case 0x08 :
                // Lecture ST0, track
                Status |= DIO;
                fct = ReadST0;
                break;

            case 0x0F :
                // Dplacement tte
                fct = MoveTrack_CPC;
                break;

            case 0x45 :
            case 0x49 :
                // Ecriture donnes
                fct = WriteData_CPC;
                break;

            case 0x46 :
            case 0x4C :
                // Lecture donnes
                fct = ReadData_CPC;
                break;

            case 0x4A :
                // Lecture champ ID
                fct = ReadID;
                break;

//            case 0x4D :
//                // Formattage piste
//                fct = FormatTrack;
//                break;

//            case 0x51 :
//            case 0x71 :
//                // Scanning...
//                break;

            default :
                fct = Rien;
            }
        }
}


void InitUPD_CPC( void )
{
    ChangeCurrTrack_CPC( 0 );
}


inline void ChangeCurrTrack_DSK( UBYTE newTrack )
{
    if ( ! newTrack )
        ST0 |= T0;
    else
        ST0 &= N_T0;

    if ( handle != -1 )
        {
        ULONG Pos = ( ULONG )( sizeof( Infos ) + ( ULONG )Infos.DataSize * ( ULONG )newTrack );
        lseek( handle
             , Pos
             , SEEK_SET
             );
        read( handle, &CurrTrackDatasDSK, sizeof( CurrTrackDatasDSK ) );
        read( handle, Datas, Infos.DataSize );
        }
    C = CurrTrackDatasDSK.Sect[ 0 ].C;
    H = CurrTrackDatasDSK.Sect[ 0 ].H;
    R = CurrTrackDatasDSK.Sect[ 0 ].R;
    N = CurrTrackDatasDSK.Sect[ 0 ].N;
}


inline void WriteCurrTrack_DSK( UBYTE Track )
{
    if ( handle != -1 )
        {
        ULONG Pos = ( ULONG )( sizeof( Infos ) + ( ULONG )Infos.DataSize * ( ULONG )Track );
        lseek( handle
             , Pos
             , SEEK_SET
             );
        write( handle, &CurrTrackDatasDSK, sizeof( CurrTrackDatasDSK ) );
        write( handle, Datas, Infos.DataSize );
        }
}


static UBYTE RechercheSecteur_DSK( UBYTE newSect )
{
    USHORT i;

    for ( i = 0; i < CurrTrackDatasDSK.NbSect; i++ )
        if ( CurrTrackDatasDSK.Sect[ i ].R == newSect )
            return( i );

    return( 0xFF );
}


static UBYTE MoveTrack_DSK( UBYTE val )
{
    switch( etat++ )
        {
        case 1 :
            Drive = val;
            Status |= NDM;
            break;

        case 2 :
            ChangeCurrTrack_DSK( C = val );
            etat = 0;
            Status &= N_CB & N_DIO & N_NDM;
            break;
        }
    return( 0 );
}


static UBYTE MoveTrack0_DSK( UBYTE val )
{
    Drive = val;
    ChangeCurrTrack_DSK( C = 0 );
    etat = 0;
    Status &= N_CB & N_DIO & N_NDM;
    return( 0 );
}


//
// Lecture de n secteurs
//
static UBYTE ReadData_DSK( UBYTE val )
{
    static USHORT sect = 0, cntdata = 0, fin = 0;
    static BOOL cond = TRUE;
    UBYTE donnee;

    switch( etat++ )
        {
        case 1 :
            Drive = val;
            break;

        case 2 :
            C = val;
            break;

        case 3 :
            H = val;
            break;

        case 4 :
            cond = TRUE;
            sect = RechercheSecteur_DSK( R = val );
            if ( sect == 0xFF )
                cond = FALSE;
            break;

        case 5 :
            N = val;
            break;

        case 6 :
            EOT = val;
            fin = ( EOT - R + 1 ) * CurrTrackDatasDSK.SectSize * 256;
            break;

        case 7 :
            if ( cond == TRUE )
                {
                if ( ( C != CurrTrackDatasDSK.Sect[ sect ].C ) ||
                     ( H != CurrTrackDatasDSK.Sect[ sect ].H ) ||
                     ( R != CurrTrackDatasDSK.Sect[ sect ].R ) ||
                     ( N != CurrTrackDatasDSK.Sect[ sect ].N )
                   )
                    cond = FALSE;
                }
            cntdata = sect * CurrTrackDatasDSK.SectSize * 256;
            break;

        case 8 :
            Status |= DIO | NDM;
            break;

        case 9 :
            if ( cond )
                {
                donnee = Datas[ cntdata++ ];
                if ( --fin )
                    etat--;
                else
                    Status &= N_NDM;

                return( donnee );
                }
            Status &= N_NDM;
            return( 0 );

        case 10 :
            return( ST0 );

        case 11 :
            return( ST1 );

        case 12 :
            return( ST2 );

        case 13 :
            return( C );

        case 14 :
            return( H );

        case 15 :
            return( R );

        case 16 :
            etat = 0;
            Status &= N_CB & N_DIO;
            return( N );
        }
    return( 0 );
}


//
// Ecriture de n secteurs
//
static UBYTE WriteData_DSK( UBYTE val )
{
    static USHORT sect = 0, cntdata = 0, fin = 0;
    static BOOL cond = TRUE;

    switch( etat++ )
        {
        case 1 :
            Drive = val;
            break;

        case 2 :
            C = val;
            break;

        case 3 :
            H = val;
            break;

        case 4 :
            cond = TRUE;
            sect = RechercheSecteur_DSK( R = val );
            if ( sect == 0xFF )
                cond = FALSE;
            break;

        case 5 :
            N = val;
            break;

        case 6 :
            EOT = val;
            fin = ( EOT - R + 1 ) * CurrTrackDatasDSK.SectSize * 256;
            break;

        case 7 :
            if ( cond == TRUE )
                {
                if ( ( C != CurrTrackDatasDSK.Sect[ sect ].C ) ||
                     ( H != CurrTrackDatasDSK.Sect[ sect ].H ) ||
                     ( R != CurrTrackDatasDSK.Sect[ sect ].R ) ||
                     ( N != CurrTrackDatasDSK.Sect[ sect ].N )
                   )
                    cond = FALSE;
                }
            cntdata = sect * CurrTrackDatasDSK.SectSize * 256;
            break;

        case 8 :
            Status |= DIO | NDM;
            break;

        case 9 :
            if ( cond )
                {
                Datas[ cntdata++ ] = val;
                if ( --fin )
                    etat--;
                else
                    Status &= N_NDM;

                return( 0 );
                }
            Status &= N_NDM;
            return( 0 );

        case 10 :
            WriteCurrTrack_DSK( C );
            return( ST0 );

        case 11 :
            return( ST1 );

        case 12 :
            return( ST2 );

        case 13 :
            return( C );

        case 14 :
            return( H );

        case 15 :
            return( R );

        case 16 :
            etat = 0;
            Status &= N_CB & N_DIO;
            return( N );
        }
    return( 0 );
}


void WriteUPD_DSK( UBYTE val )
{
    if ( etat )
        fct( val );
    else
        {
        Status |= CB;
        etat = 1;
        switch( val & 0x5F )
            {
            case 0x03 :
                // Specify
                fct = Specify;
                break;

            case 0x04 :
                // Lecture ST3
                fct = ReadST3;
                break;

            case 0x07 :
                // Dplacement tte piste 0
                Status |= NDM;
                fct = MoveTrack0_DSK;
                break;

            case 0x08 :
                // Lecture ST0, track
                Status |= DIO;
                fct = ReadST0;
                break;

            case 0x0F :
                // Dplacement tte
                fct = MoveTrack_DSK;
                break;

            case 0x45 :
            case 0x49 :
                // Ecriture donnes
                fct = WriteData_DSK;
                break;

            case 0x46 :
            case 0x4C :
                // Lecture donnes
                fct = ReadData_DSK;
                break;

            case 0x4A :
                // Lecture champ ID
                fct = ReadID;
                break;

//            case 0x4D :
//                // Formattage piste
//                fct = FormatTrack;
//                break;

//            case 0x51 :
//            case 0x71 :
//                // Scanning...
//                break;

            default :
                fct = Rien;
            }
        }
}


void InitUPD_DSK( void )
{
    read( handle, &Infos, sizeof( Infos ) );
    ChangeCurrTrack_DSK( 0 );
}


void WriteUPD( UBYTE val )
{
    FctWriteUPD( val );
}


void InitUPD( char * nomFic )
{
    handle = open( nomFic, O_RDWR | O_BINARY );
    if ( strstr( nomFic, ".CPC" ) )
        {
        FctWriteUPD = WriteUPD_CPC;
        InitUPD_CPC();
        }
    else
        {
        FctWriteUPD = WriteUPD_DSK;
        InitUPD_DSK();
        }
}


void EndUPD( void )
{
    if ( handle != -1 )
        close( handle );
}
