/*****************************************************************************
 *                                                                           *
 * Module : VGA.C                                                            *
 *                                                                           *
 * Gestion du Video Gate Array                                               *
 *                                                                           *
 *****************************************************************************/


#include  <string.h>
#include  <alloc.h>
#include  <fcntl.h>
#include  <dos.h>
#include  <io.h>

#include  "vga.h"
#include  "tools.h"


#define   OFFSET_1  0x0400
#define   OFFSET_2  0x0A00
#define   OFFSET_3  0x1900
#define   OFFSET_4  0x2A80
#define   OFFSET_5  0x3930
#define   OFFSET_6  0x4000


typedef struct
    {
    USHORT Debut;
    USHORT Fin;
    } StRedraw;


typedef struct
    {
    UBYTE r;
    UBYTE v;
    UBYTE b;
    } stCouleur;


extern UBYTE DmdChgFic;
extern UBYTE Reset;
extern UBYTE VBL;

extern volatile int IRQ;

extern USHORT volatile FreqInt;


UBYTE * segAdrCPC = 0; // Segment Mmoire

UBYTE TabCoul[ 32 ];

UBYTE TabDac[ 32 ] =
    {
#ifdef DEBUG
    0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x14, 0x07,
    0x38, 0x39, 0x3A, 0x3B, 0x3C, 0x3D, 0x3E, 0x3F,
#else
    0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
    0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
#endif
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF
    };

USHORT DecodeurAdresse = ROMINF_SEL;

USHORT segdosAdrCPC;

UBYTE * TabPOKE[ 4 ];
UBYTE * TabPEEK[ 4 ];


static stCouleur tabCouleursCPC[ 32 ] =
    {
        { 0x1F, 0x1F, 0x1F } ,               // Blanc            (13)
        { 0x1F, 0x1F, 0x1F } ,               // Blanc            (13)
        { 0x00, 0x3F, 0x1F } ,               // Vert Marin       (19)
        { 0x3F, 0x3F, 0x1F } ,               // Jaune Pastel     (25)
        { 0x00, 0x00, 0x1F } ,               // Bleu              (1)
        { 0x3F, 0x00, 0x1F } ,               // Pourpre           (7)
        { 0x00, 0x1F, 0x1F } ,               // Turquoise        (10)
        { 0x3F, 0x1F, 0x1F } ,               // Rose             (16)
        { 0x3F, 0x00, 0x1F } ,               // Pourpre           (7)
        { 0x3F, 0x3F, 0x00 } ,               // Jaune vif        (24)
        { 0x3F, 0x3F, 0x00 } ,               // Jaune vif        (24)
        { 0x3F, 0x3F, 0x3F } ,               // Blanc Brillant   (26)
        { 0x3F, 0x00, 0x00 } ,               // Rouge vif         (6)
        { 0x3F, 0x00, 0x3F } ,               // Magenta vif       (8)
        { 0x3F, 0x1F, 0x00 } ,               // Orange           (15)
        { 0x3F, 0x1F, 0x3F } ,               // Magenta pastel   (17)
        { 0x00, 0x00, 0x1F } ,               // Bleu              (1)
        { 0x00, 0x3F, 0x1F } ,               // Vert Marin       (19)
        { 0x00, 0x3F, 0x00 } ,               // Vert vif         (18)
        { 0x00, 0x3F, 0x3F } ,               // Turquoise vif    (20)
        { 0x00, 0x00, 0x00 } ,               // Noir              (0)
        { 0x00, 0x00, 0x3F } ,               // Bleu vif          (2)
        { 0x00, 0x1F, 0x00 } ,               // Vert              (9)
        { 0x00, 0x1F, 0x3F } ,               // Bleu ciel        (11)
        { 0x1F, 0x00, 0x1F } ,               // Magenta           (4)
        { 0x1F, 0x3F, 0x1F } ,               // Vert pastel      (22)
        { 0x1F, 0x3F, 0x00 } ,               // Vert citron      (21)
        { 0x1F, 0x3F, 0x3F } ,               // Turquoise pastel (23)
        { 0x1F, 0x00, 0x00 } ,               // Rouge             (3)
        { 0x1F, 0x00, 0x3F } ,               // Mauve             (5)
        { 0x1F, 0x1F, 0x00 } ,               // Jaune            (12)
        { 0x1F, 0x1F, 0x3F }                 // Bleu pastel      (14)
    };


static UBYTE tabOctMode0P1[ 256 ] =
    {
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F,
    0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F,
    0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F,
    0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F,
    0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F,
    0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F,
    0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F,
    0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F,
    0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0,
    0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0,
    0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0,
    0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0,
    0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0,
    0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0,
    0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0,
    0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF
    };


static UBYTE tabOctMode0P2[ 256 ] =
    {
    0x00, 0x00, 0x00, 0x00, 0x0F, 0x0F, 0x0F, 0x0F,
    0xF0, 0xF0, 0xF0, 0xF0, 0xFF, 0xFF, 0xFF, 0xFF,
    0x00, 0x00, 0x00, 0x00, 0x0F, 0x0F, 0x0F, 0x0F,
    0xF0, 0xF0, 0xF0, 0xF0, 0xFF, 0xFF, 0xFF, 0xFF,
    0x00, 0x00, 0x00, 0x00, 0x0F, 0x0F, 0x0F, 0x0F,
    0xF0, 0xF0, 0xF0, 0xF0, 0xFF, 0xFF, 0xFF, 0xFF,
    0x00, 0x00, 0x00, 0x00, 0x0F, 0x0F, 0x0F, 0x0F,
    0xF0, 0xF0, 0xF0, 0xF0, 0xFF, 0xFF, 0xFF, 0xFF,
    0x00, 0x00, 0x00, 0x00, 0x0F, 0x0F, 0x0F, 0x0F,
    0xF0, 0xF0, 0xF0, 0xF0, 0xFF, 0xFF, 0xFF, 0xFF,
    0x00, 0x00, 0x00, 0x00, 0x0F, 0x0F, 0x0F, 0x0F,
    0xF0, 0xF0, 0xF0, 0xF0, 0xFF, 0xFF, 0xFF, 0xFF,
    0x00, 0x00, 0x00, 0x00, 0x0F, 0x0F, 0x0F, 0x0F,
    0xF0, 0xF0, 0xF0, 0xF0, 0xFF, 0xFF, 0xFF, 0xFF,
    0x00, 0x00, 0x00, 0x00, 0x0F, 0x0F, 0x0F, 0x0F,
    0xF0, 0xF0, 0xF0, 0xF0, 0xFF, 0xFF, 0xFF, 0xFF,
    0x00, 0x00, 0x00, 0x00, 0x0F, 0x0F, 0x0F, 0x0F,
    0xF0, 0xF0, 0xF0, 0xF0, 0xFF, 0xFF, 0xFF, 0xFF,
    0x00, 0x00, 0x00, 0x00, 0x0F, 0x0F, 0x0F, 0x0F,
    0xF0, 0xF0, 0xF0, 0xF0, 0xFF, 0xFF, 0xFF, 0xFF,
    0x00, 0x00, 0x00, 0x00, 0x0F, 0x0F, 0x0F, 0x0F,
    0xF0, 0xF0, 0xF0, 0xF0, 0xFF, 0xFF, 0xFF, 0xFF,
    0x00, 0x00, 0x00, 0x00, 0x0F, 0x0F, 0x0F, 0x0F,
    0xF0, 0xF0, 0xF0, 0xF0, 0xFF, 0xFF, 0xFF, 0xFF,
    0x00, 0x00, 0x00, 0x00, 0x0F, 0x0F, 0x0F, 0x0F,
    0xF0, 0xF0, 0xF0, 0xF0, 0xFF, 0xFF, 0xFF, 0xFF,
    0x00, 0x00, 0x00, 0x00, 0x0F, 0x0F, 0x0F, 0x0F,
    0xF0, 0xF0, 0xF0, 0xF0, 0xFF, 0xFF, 0xFF, 0xFF,
    0x00, 0x00, 0x00, 0x00, 0x0F, 0x0F, 0x0F, 0x0F,
    0xF0, 0xF0, 0xF0, 0xF0, 0xFF, 0xFF, 0xFF, 0xFF,
    0x00, 0x00, 0x00, 0x00, 0x0F, 0x0F, 0x0F, 0x0F,
    0xF0, 0xF0, 0xF0, 0xF0, 0xFF, 0xFF, 0xFF, 0xFF,
    };


static UBYTE tabOctMode0P3[ 256 ] =
    {
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F,
    0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F,
    0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0,
    0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F,
    0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F,
    0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0,
    0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F,
    0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F,
    0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0,
    0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F,
    0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F,
    0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0,
    0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    };


static UBYTE tabOctMode0P4[ 256 ] =
    {
    0x00, 0x0F, 0xF0, 0xFF, 0x00, 0x0F, 0xF0, 0xFF,
    0x00, 0x0F, 0xF0, 0xFF, 0x00, 0x0F, 0xF0, 0xFF,
    0x00, 0x0F, 0xF0, 0xFF, 0x00, 0x0F, 0xF0, 0xFF,
    0x00, 0x0F, 0xF0, 0xFF, 0x00, 0x0F, 0xF0, 0xFF,
    0x00, 0x0F, 0xF0, 0xFF, 0x00, 0x0F, 0xF0, 0xFF,
    0x00, 0x0F, 0xF0, 0xFF, 0x00, 0x0F, 0xF0, 0xFF,
    0x00, 0x0F, 0xF0, 0xFF, 0x00, 0x0F, 0xF0, 0xFF,
    0x00, 0x0F, 0xF0, 0xFF, 0x00, 0x0F, 0xF0, 0xFF,
    0x00, 0x0F, 0xF0, 0xFF, 0x00, 0x0F, 0xF0, 0xFF,
    0x00, 0x0F, 0xF0, 0xFF, 0x00, 0x0F, 0xF0, 0xFF,
    0x00, 0x0F, 0xF0, 0xFF, 0x00, 0x0F, 0xF0, 0xFF,
    0x00, 0x0F, 0xF0, 0xFF, 0x00, 0x0F, 0xF0, 0xFF,
    0x00, 0x0F, 0xF0, 0xFF, 0x00, 0x0F, 0xF0, 0xFF,
    0x00, 0x0F, 0xF0, 0xFF, 0x00, 0x0F, 0xF0, 0xFF,
    0x00, 0x0F, 0xF0, 0xFF, 0x00, 0x0F, 0xF0, 0xFF,
    0x00, 0x0F, 0xF0, 0xFF, 0x00, 0x0F, 0xF0, 0xFF,
    0x00, 0x0F, 0xF0, 0xFF, 0x00, 0x0F, 0xF0, 0xFF,
    0x00, 0x0F, 0xF0, 0xFF, 0x00, 0x0F, 0xF0, 0xFF,
    0x00, 0x0F, 0xF0, 0xFF, 0x00, 0x0F, 0xF0, 0xFF,
    0x00, 0x0F, 0xF0, 0xFF, 0x00, 0x0F, 0xF0, 0xFF,
    0x00, 0x0F, 0xF0, 0xFF, 0x00, 0x0F, 0xF0, 0xFF,
    0x00, 0x0F, 0xF0, 0xFF, 0x00, 0x0F, 0xF0, 0xFF,
    0x00, 0x0F, 0xF0, 0xFF, 0x00, 0x0F, 0xF0, 0xFF,
    0x00, 0x0F, 0xF0, 0xFF, 0x00, 0x0F, 0xF0, 0xFF,
    0x00, 0x0F, 0xF0, 0xFF, 0x00, 0x0F, 0xF0, 0xFF,
    0x00, 0x0F, 0xF0, 0xFF, 0x00, 0x0F, 0xF0, 0xFF,
    0x00, 0x0F, 0xF0, 0xFF, 0x00, 0x0F, 0xF0, 0xFF,
    0x00, 0x0F, 0xF0, 0xFF, 0x00, 0x0F, 0xF0, 0xFF,
    0x00, 0x0F, 0xF0, 0xFF, 0x00, 0x0F, 0xF0, 0xFF,
    0x00, 0x0F, 0xF0, 0xFF, 0x00, 0x0F, 0xF0, 0xFF,
    0x00, 0x0F, 0xF0, 0xFF, 0x00, 0x0F, 0xF0, 0xFF,
    0x00, 0x0F, 0xF0, 0xFF, 0x00, 0x0F, 0xF0, 0xFF,
    };


static UBYTE tabOctMode1P1[ 256 ] =
    {
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03,
    0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03,
    0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C,
    0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C,
    0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F,
    0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F,
    0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30,
    0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30,
    0x33, 0x33, 0x33, 0x33, 0x33, 0x33, 0x33, 0x33,
    0x33, 0x33, 0x33, 0x33, 0x33, 0x33, 0x33, 0x33,
    0x3C, 0x3C, 0x3C, 0x3C, 0x3C, 0x3C, 0x3C, 0x3C,
    0x3C, 0x3C, 0x3C, 0x3C, 0x3C, 0x3C, 0x3C, 0x3C,
    0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F,
    0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F,
    0xC0, 0xC0, 0xC0, 0xC0, 0xC0, 0xC0, 0xC0, 0xC0,
    0xC0, 0xC0, 0xC0, 0xC0, 0xC0, 0xC0, 0xC0, 0xC0,
    0xC3, 0xC3, 0xC3, 0xC3, 0xC3, 0xC3, 0xC3, 0xC3,
    0xC3, 0xC3, 0xC3, 0xC3, 0xC3, 0xC3, 0xC3, 0xC3,
    0xCC, 0xCC, 0xCC, 0xCC, 0xCC, 0xCC, 0xCC, 0xCC,
    0xCC, 0xCC, 0xCC, 0xCC, 0xCC, 0xCC, 0xCC, 0xCC,
    0xCF, 0xCF, 0xCF, 0xCF, 0xCF, 0xCF, 0xCF, 0xCF,
    0xCF, 0xCF, 0xCF, 0xCF, 0xCF, 0xCF, 0xCF, 0xCF,
    0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0,
    0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0,
    0xF3, 0xF3, 0xF3, 0xF3, 0xF3, 0xF3, 0xF3, 0xF3,
    0xF3, 0xF3, 0xF3, 0xF3, 0xF3, 0xF3, 0xF3, 0xF3,
    0xFC, 0xFC, 0xFC, 0xFC, 0xFC, 0xFC, 0xFC, 0xFC,
    0xFC, 0xFC, 0xFC, 0xFC, 0xFC, 0xFC, 0xFC, 0xFC,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    };


static UBYTE tabOctMode1P2[ 256 ] =
    {
    0x00, 0x03, 0x0C, 0x0F, 0x30, 0x33, 0x3C, 0x3F,
    0xC0, 0xC3, 0xCC, 0xCF, 0xF0, 0xF3, 0xFC, 0xFF,
    0x00, 0x03, 0x0C, 0x0F, 0x30, 0x33, 0x3C, 0x3F,
    0xC0, 0xC3, 0xCC, 0xCF, 0xF0, 0xF3, 0xFC, 0xFF,
    0x00, 0x03, 0x0C, 0x0F, 0x30, 0x33, 0x3C, 0x3F,
    0xC0, 0xC3, 0xCC, 0xCF, 0xF0, 0xF3, 0xFC, 0xFF,
    0x00, 0x03, 0x0C, 0x0F, 0x30, 0x33, 0x3C, 0x3F,
    0xC0, 0xC3, 0xCC, 0xCF, 0xF0, 0xF3, 0xFC, 0xFF,
    0x00, 0x03, 0x0C, 0x0F, 0x30, 0x33, 0x3C, 0x3F,
    0xC0, 0xC3, 0xCC, 0xCF, 0xF0, 0xF3, 0xFC, 0xFF,
    0x00, 0x03, 0x0C, 0x0F, 0x30, 0x33, 0x3C, 0x3F,
    0xC0, 0xC3, 0xCC, 0xCF, 0xF0, 0xF3, 0xFC, 0xFF,
    0x00, 0x03, 0x0C, 0x0F, 0x30, 0x33, 0x3C, 0x3F,
    0xC0, 0xC3, 0xCC, 0xCF, 0xF0, 0xF3, 0xFC, 0xFF,
    0x00, 0x03, 0x0C, 0x0F, 0x30, 0x33, 0x3C, 0x3F,
    0xC0, 0xC3, 0xCC, 0xCF, 0xF0, 0xF3, 0xFC, 0xFF,
    0x00, 0x03, 0x0C, 0x0F, 0x30, 0x33, 0x3C, 0x3F,
    0xC0, 0xC3, 0xCC, 0xCF, 0xF0, 0xF3, 0xFC, 0xFF,
    0x00, 0x03, 0x0C, 0x0F, 0x30, 0x33, 0x3C, 0x3F,
    0xC0, 0xC3, 0xCC, 0xCF, 0xF0, 0xF3, 0xFC, 0xFF,
    0x00, 0x03, 0x0C, 0x0F, 0x30, 0x33, 0x3C, 0x3F,
    0xC0, 0xC3, 0xCC, 0xCF, 0xF0, 0xF3, 0xFC, 0xFF,
    0x00, 0x03, 0x0C, 0x0F, 0x30, 0x33, 0x3C, 0x3F,
    0xC0, 0xC3, 0xCC, 0xCF, 0xF0, 0xF3, 0xFC, 0xFF,
    0x00, 0x03, 0x0C, 0x0F, 0x30, 0x33, 0x3C, 0x3F,
    0xC0, 0xC3, 0xCC, 0xCF, 0xF0, 0xF3, 0xFC, 0xFF,
    0x00, 0x03, 0x0C, 0x0F, 0x30, 0x33, 0x3C, 0x3F,
    0xC0, 0xC3, 0xCC, 0xCF, 0xF0, 0xF3, 0xFC, 0xFF,
    0x00, 0x03, 0x0C, 0x0F, 0x30, 0x33, 0x3C, 0x3F,
    0xC0, 0xC3, 0xCC, 0xCF, 0xF0, 0xF3, 0xFC, 0xFF,
    0x00, 0x03, 0x0C, 0x0F, 0x30, 0x33, 0x3C, 0x3F,
    0xC0, 0xC3, 0xCC, 0xCF, 0xF0, 0xF3, 0xFC, 0xFF,
    };


static UBYTE tabOctMode2P1[ 256 ] =
    {
    0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
    0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F,
    0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
    0x18, 0x19, 0x1A, 0x1B, 0x1C, 0x1D, 0x1E, 0x1F, 
    0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 
    0x28, 0x29, 0x2A, 0x2B, 0x2C, 0x2D, 0x2E, 0x2F, 
    0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 
    0x38, 0x39, 0x3A, 0x3B, 0x3C, 0x3D, 0x3E, 0x3F, 
    0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47, 
    0x48, 0x49, 0x4A, 0x4B, 0x4C, 0x4D, 0x4E, 0x4F, 
    0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57, 
    0x58, 0x59, 0x5A, 0x5B, 0x5C, 0x5D, 0x5E, 0x5F, 
    0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67,
    0x68, 0x69, 0x6A, 0x6B, 0x6C, 0x6D, 0x6E, 0x6F, 
    0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77,
    0x78, 0x79, 0x7A, 0x7B, 0x7C, 0x7D, 0x7E, 0x7F, 
    0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87,
    0x88, 0x89, 0x8A, 0x8B, 0x8C, 0x8D, 0x8E, 0x8F,
    0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97,
    0x98, 0x99, 0x9A, 0x9B, 0x9C, 0x9D, 0x9E, 0x9F,
    0xA0, 0xA1, 0xA2, 0xA3, 0xA4, 0xA5, 0xA6, 0xA7,
    0xA8, 0xA9, 0xAA, 0xAB, 0xAC, 0xAD, 0xAE, 0xAF,
    0xB0, 0xB1, 0xB2, 0xB3, 0xB4, 0xB5, 0xB6, 0xB7,
    0xB8, 0xB9, 0xBA, 0xBB, 0xBC, 0xBD, 0xBE, 0xBF,
    0xC0, 0xC1, 0xC2, 0xC3, 0xC4, 0xC5, 0xC6, 0xC7,
    0xC8, 0xC9, 0xCA, 0xCB, 0xCC, 0xCD, 0xCE, 0xCF,
    0xD0, 0xD1, 0xD2, 0xD3, 0xD4, 0xD5, 0xD6, 0xD7,
    0xD8, 0xD9, 0xDA, 0xDB, 0xDC, 0xDD, 0xDE, 0xDF,
    0xE0, 0xE1, 0xE2, 0xE3, 0xE4, 0xE5, 0xE6, 0xE7,
    0xE8, 0xE9, 0xEA, 0xEB, 0xEC, 0xED, 0xEE, 0xEF,
    0xF0, 0xF1, 0xF2, 0xF3, 0xF4, 0xF5, 0xF6, 0xF7,
    0xF8, 0xF9, 0xFA, 0xFB, 0xFC, 0xFD, 0xFE, 0xFF
    };


static UBYTE Oct1[ 0x100 ];
static UBYTE Oct2[ 0x100 ];
static UBYTE Oct3[ 0x100 ];
static UBYTE Oct4[ 0x100 ];


//
// Nouvelle interruption 08
//
void interrupt NewInt08( __CPPARGS )
{
    static UBYTE r = 6;
    static StRedraw TabRedraw[] =
        {
            { OFFSET_5, OFFSET_6 },
            { 0,        OFFSET_1 },
            { OFFSET_1, OFFSET_2 },
            { OFFSET_2, OFFSET_3 },
            { OFFSET_3, OFFSET_4 },
            { OFFSET_4, OFFSET_5 },
        };

    if ( ++r > 5 )
        {
        r = 0;
        VBL = 1;
        }
    else
        VBL = 0;

    if ( ! DmdChgFic )
        {
        USHORT Debut = TabRedraw[ r ].Debut;
        USHORT Fin = TabRedraw[ r ].Fin;
        UBYTE * q = TabPOKE[ adrEcr ];
        outportb( 0x3C5, 1 );
        for ( USHORT i = Debut; i < Fin; i++ )
            * ( UBYTE * )MK_FP( 0xA000, i ) = Oct1[ q[ tabAdrMemEcrCPC[ i ] ] ];

        outportb( 0x3C5, 2 );
        for ( i = Debut; i < Fin; i++ )
            * ( UBYTE * )MK_FP( 0xA000, i ) = Oct2[ q[ tabAdrMemEcrCPC[ i ] ] ];

        outportb( 0x3C5, 4 );
        for ( i = Debut; i < Fin; i++ )
            * ( UBYTE * )MK_FP( 0xA000, i ) = Oct3[ q[ tabAdrMemEcrCPC[ i ] ] ];

        outportb( 0x3C5, 8 );
        for ( i = Debut; i < Fin; i++ )
            * ( UBYTE * )MK_FP( 0xA000, i ) = Oct4[ q[ tabAdrMemEcrCPC[ i ] ] ];
        }

    IRQ = 1;
    SetFreq( FreqInt );
    outportb( 0x20, 0x20 );
}


static void ChangeRAM( UBYTE val )
{
    static USHORT Adj[ 8 ] =
        {
        0x0400, 0x0400, 0x0400, 0x0400, 0x1000, 0x1400, 0x1800, 0x1C00
        };

    TabPOKE[ 1 ] =
    TabPEEK[ 1 ] = ( UBYTE * )MK_FP( segdosAdrCPC + Adj[ val & 7 ], 0 );
}


void WriteVGA( UBYTE val )
{
    static UBYTE couleur = 0;
    static UBYTE lastMode = 0;
    register newVal = val & 0x1F;

    switch( val & 0xC0 )
        {
        case 0x00 :
            couleur = newVal;
            break;

        case 0x40 :
            TabCoul[ couleur ] = newVal;
            SetDac( couleur
                  , tabCouleursCPC[ newVal ].r
                  , tabCouleursCPC[ newVal ].v
                  , tabCouleursCPC[ newVal ].b
                  );
            break;

        case 0x80 :
            DecodeurAdresse &= N_ROMINF_SEL & N_ROMSUP_SEL;

            if ( ! ( val & 4 ) )
                DecodeurAdresse |= ROMINF_SEL;

            if ( ! ( val & 8 ) )
                DecodeurAdresse |= ROMSUP_SEL;

            if ( lastMode != ( val & 3 ) )
                {
                lastMode = val & 3;
                switch( lastMode )
                    {
                    case 0 :
                    case 3 :
                        memcpy( Oct1, tabOctMode0P1, 0x100 );
                        memcpy( Oct2, tabOctMode0P2, 0x100 );
                        memcpy( Oct3, tabOctMode0P3, 0x100 );
                        memcpy( Oct4, tabOctMode0P4, 0x100 );
                        break;

                    case 1 :
                        memcpy( Oct1, tabOctMode1P1, 0x100 );
                        memcpy( Oct2, tabOctMode1P2, 0x100 );
                        memset( Oct3, 0, 0x100 );
                        memset( Oct4, 0, 0x100 );
                        break;

                    case 2 :
                        memcpy( Oct1, tabOctMode2P1, 0x100 );
                        memset( Oct2, 0, 0x100 );
                        memset( Oct3, 0, 0x100 );
                        memset( Oct4, 0, 0x100 );
                        break;
                    }
                }
            ChangeROMInf();
            ChangeROMSup();
            break;

        case 0xC0 :
            ChangeRAM( val );
            break;
        }
}


BOOL GetEcran( void )
{
    struct REGPACK r;

#ifdef DEBUG
    r.r_ax = 0x10;
#else
    r.r_ax = 0x0E;
#endif
    intr( 0x10, & r);
    outportb( 0x3C0, 0x31 );
    outportb( 0x3C0, 0xFF );
    outportb( 0x3C4, 2 );
    return TRUE;
}


void CloseEcran( void )
{
    struct REGPACK r;

    r.r_ax = 3;
    intr( 0x10, &r );
}


BOOL AllocMemCPC( void )
{
    SHORT i = allocmem( 0x2000, &segdosAdrCPC );
    segAdrCPC = ( UBYTE * )MK_FP( segdosAdrCPC, 0 );
    if ( i == -1 )
        {
        ChangeROMInf();
        ChangeROMSup();
        ChangeRAM( 0xC0 );
        TabPOKE[ 0 ] = ( UBYTE * )MK_FP( segdosAdrCPC, 0 );
        TabPOKE[ 2 ] =
        TabPEEK[ 2 ] = ( UBYTE * )MK_FP( segdosAdrCPC + 0x800, 0 );
        TabPOKE[ 3 ] = ( UBYTE * )MK_FP( segdosAdrCPC + 0xC00, 0 );
        return TRUE;
        }
    return FALSE;
}


void FreeMemCPC( void )
{
    freemem( segdosAdrCPC );
}


#ifdef DEBUG
void AfficheTexte( char * Texte )
{
    static UBYTE far * Fonte = NULL;

    USHORT y, c;
    USHORT Posx = 0, Posy = 210;
    if ( ! Fonte )
        {
        struct REGPACK r;

        r.r_ax = 0x1130;
        r.r_bx = 0x0300;        /* Jeux 8x8 */
        intr( 0x10, &r );

        Fonte = ( UBYTE far * )MK_FP( r.r_es, r.r_bp );
        }

    outportb( 0x3C5, 1 );
    while( * Texte )
        {
        c = ( * Texte++ ) << 3;
        for ( y = 0; y < 8; y++ )
            * ( UBYTE far * )MK_FP( 0xA000, Posx + 80 * ( Posy + y ) ) = Fonte[ c++ ];

        if ( ++Posx > 76 )
            {
            Posx = 0;
            Posy += 8;
            }
        }

}
#endif

