/*
	Portable PC-Engine Emulator
	1998 by BERO bero@geocities.co.jp
*/

#include	<stdio.h>
#include	<stdlib.h>
#include	"m6502.h"

#ifndef TRUE
#define	TRUE	1
#define	FALSE	0
#endif

byte RAM[0x8000];
#define	ZW	64
byte ZBuf[ZW*256];
byte *Page[8],*ROMMap[256];
byte *VRAM,*ROM,*vchange,*PCM,*WRAM,*DMYROM;
short SPRAM[64*4];
unsigned long *VRAM2;
int ROM_size;
extern int Debug,vmode;
int UPeriod,Country;
static int IPeriod,BaseClock;
static int scanlines_per_frame  = 280;
static int scanline;
static int MinLine = 0,MaxLine = 255;

byte Pal[512];

struct {
	pair VDC[32];
	pair VCE[0x200];
	pair vce_reg;
	/* VDC */
	word vdc_inc,vdc_raster_count;
	byte vdc_reg,vdc_status,vdc_ratch,vce_ratch;
	int bg_h,bg_w;
	int screen_w,screen_h;
	/* joypad */
	byte JOY[16];
	byte joy_select,joy_counter;
	/* PSG */
	byte PSG[6][8],wave[6][32],wavofs[6];
	byte psg_ch,psg_volume,psg_lfo_freq,psg_lfo_ctrl;
	/* TIMER */
	byte timer_reload,timer_start,timer_counter;
	/* IRQ */
	byte irq_mask,irq_status;
	/* CDROM extention */
	int backup,adpcm_firstread;
	pair adpcm_ptr;
	word adpcm_rptr,adpcm_wptr;
} io;

#define	VRR	2
enum _VDC_REG {
	MAWR,MARR,VWR,vdc3,vdc4,CR,RCR,BXR,
	BYR,MWR,HSR,HDR,VPR,VDW,VCR,DCR,
	SOUR,DISTR,LENR,SATB};
#define	NODATA	0xff
#define	ENABLE	1
#define	DISABLE	0

#define	VDC_CR	0x01
#define	VDC_OR	0x02
#define	VDC_RR	0x04
#define	VDC_DS	0x08
#define	VDC_DV	0x10
#define	VDC_VD	0x20
#define	VDC_BSY	0x40
#define	VDC_SpHit	VDC_CR
#define	VDC_Over	VDC_OR
#define	VDC_RasHit	VDC_RR
#define	VDC_InVBlank VDC_VD
#define	VDC_DMAfinish	VDC_DS
#define	VDC_SATBfinish	VDC_VD

#define	SpHitON		(io.VDC[CR].W&0x01)
#define	OverON		(io.VDC[CR].W&0x02)
#define	RasHitON	(io.VDC[CR].W&0x04)
#define	VBlankON	(io.VDC[CR].W&0x08)
#define	SpriteON	(io.VDC[CR].W&0x40)
#define	ScreenON	(io.VDC[CR].W&0x80)

#define	IRQ2	1
#define	IRQ1	2
#define	TIRQ	4

#define	ScrollX	io.VDC[BXR].W
#define	ScrollY	io.VDC[BYR].W

typedef struct {
	short y,x,no,atr;
} SPR;


void  IO_write(word A,byte V);
byte  IO_read(word A);
void RefreshLine(int Y1,int Y2);
void RefreshSprite(int Y1,int Y2);
void RefreshScreen(void);

void bank_set(byte P,byte V)
{
	if (V>=ROM_size && V<0xF7) {
	//	printf("ignore bank set %x:%02x",P,V);
	//	TrashMachine();
	//	exit(-1);
	}
	if (ROMMap[V]==NULL) Page[P]=NULL;
	else Page[P]=ROMMap[V]-P*0x2000;
}

byte _Rd6502(word A)
{
	if (Page[A>>13]) return Page[A>>13][A];
	else return IO_read(A);
}

void _Wr6502(word A,byte V)
{
	if (Page[A>>13]) Page[A>>13][A]=V;
	else IO_write(A,V);
}

/* write */

void  IO_write(word A,byte V)
{
	//printf("w%04x,%02x ",A&0x3FFF,V);
  switch(A&0x3C00) {
  case 0x0000:	/* VDC */
	switch(A&3){
	case 0: io.vdc_reg = V&31; return;
	case 1: return;
	case 2:
		//printf("vdc_l%d,%02x ",io.vdc_reg,V);
		switch(io.vdc_reg){
		case VWR:
			/*VRAM[io.VDC[MAWR].W*2]=*/ io.vdc_ratch = V;
			return;
		case HDR:
			io.screen_w = (V+1)*8;
			//printf("screen:%dx%d %d\n",io.screen_w,256,V);
			break;
		case MWR:
			io.bg_h=(V&0x40)?64:32;
			io.bg_w=32<<((V>>4)&3);
			//printf("bg:%dx%d\n",io.bg_w,io.bg_h);
			break;
		}
		io.VDC[io.vdc_reg].B.l = V;
		if (io.vdc_reg>19) {
			printf("ignore write lo vdc%d,%02x\n",io.vdc_reg,V);
		}
		return;
	case 3:
		//printf("vdc_h%d,%02x ",io.vdc_reg,V);
		switch(io.vdc_reg){
		case VWR:
			//printf("v%04x\n",io.VDC[MAWR].W);
			VRAM[io.VDC[MAWR].W*2]=io.vdc_ratch;
			VRAM[io.VDC[MAWR].W*2+1]=V;
			vchange[io.VDC[MAWR].W/16]=1;
			io.VDC[MAWR].W+=io.vdc_inc;
			io.vdc_ratch=0;
			return;
		case LENR:
			io.VDC[LENR].B.h = V;
			//printf("DMA:%04x %04x %04x\n",io.VDC[DISTR].W,io.VDC[SOUR].W,io.VDC[LENR].W);
			/* VRAM to VRAM DMA */
			memcpy(VRAM+io.VDC[DISTR].W*2,VRAM+io.VDC[SOUR].W*2,
				io.VDC[LENR].W*2);
			memset(vchange+io.VDC[DISTR].W/16,1,io.VDC[LENR].W/16);
			io.vdc_status|=VDC_DMAfinish;
			return;
		case CR :
			{static byte incsize[]={1,32,64,128};
			io.vdc_inc = incsize[(V>>3)&3];
			}
			break;
		}
		io.VDC[io.vdc_reg].B.h = V;
		if (io.vdc_reg>19) {
			printf("ignore write hi vdc%d,%02x\n",io.vdc_reg,V);
		}
		return;
	}
	break;

  case 0x0400:	/* VCE */
	switch(A&7) {
	case 0: /*io.vce_reg.W = 0; io.vce_ratch=0;*/ /*??*/ return;
	case 2: io.vce_reg.B.l = V; return;
	case 3: io.vce_reg.B.h = V&1; return;
	case 4: io.VCE[io.vce_reg.W].B.l= V; return;
	case 5:
		/*io.VCE[io.vce_reg.W].B.l = io.vce_ratch;*/
		io.VCE[io.vce_reg.W].B.h = V;
		{byte c; int i,n;
		n = io.vce_reg.W;
		c = io.VCE[n].W>>1;
		if (n==0) {
			for(i=0;i<256;i+=16) Pal[i]=c;
		}else if (n&15) Pal[n] = c;
		}
		io.vce_reg.W=(io.vce_reg.W+1)&0x1FF;
		return;
	//case 6: /* ?? */ return;
	}
	break;

  case 0x0800:	/* PSG */
  	switch(A&15){
	case 0: io.psg_ch = V&7; return;
	case 1: io.psg_volume = V; return;
	case 2: io.PSG[io.psg_ch][2] = V; return;
	case 3: io.PSG[io.psg_ch][3] = V; return;
	case 4: io.PSG[io.psg_ch][4] = V; return;
	case 5: io.PSG[io.psg_ch][5] = V; return;
	case 6: io.wave[io.psg_ch][(io.wavofs[io.psg_ch]++)&31]=V&31;return;
	case 7: io.PSG[io.psg_ch][7] = V; return;
	case 8: io.psg_lfo_freq = V; return;
	case 9: io.psg_lfo_ctrl = V; return;
    }
    break;

  case 0x0c00:	/* timer */
	//printf("w%04x,%02x ",A,V);
	switch(A&1){
	case 0: io.timer_reload = V&127; return;
	case 1: 
		V&=1;
		if (V && !io.timer_start)
			io.timer_counter = io.timer_reload;
		io.timer_start  = V; 
		return;
	}
	break;

  case 0x1000:	/* joypad */
		io.joy_select = V;
		if (V&2) io.joy_counter = 0;
		return;

  case 0x1400:	/* IRQ */
	switch(A&15){
	case 2: io.irq_mask = V; return;
	case 3: /* ?? */ return;
	}
	break;

  case 0x1800:	/* CD-ROM extention */
	switch(A&15){
	case 7: io.backup = ENABLE; return;
	case 8: io.adpcm_ptr.B.l = V; return;
	case 9: io.adpcm_ptr.B.h = V; return;
	case 0xa: PCM[io.adpcm_wptr++] = V; return;
	case 0xd:
		if (V&4) io.adpcm_wptr = io.adpcm_ptr.W;
		else { io.adpcm_rptr = io.adpcm_ptr.W; io.adpcm_firstread = TRUE; }
		return;
	}
	break;
  }
	printf("ignore I/O write %04x,%02x\n",A,V);
}

/* read */
byte IO_read(word A)
{
	byte ret;
	//printf("r%04x ",A&0x3FFF);
  switch(A&0x3C00){
  case 0x0000: /* VDC */
  	switch(A&3){
	case 0:
		ret = io.vdc_status;
		io.vdc_status&=~VDC_BSY;
		return ret;
	case 1:
		return 0;
	case 2:
		if (io.vdc_reg==VRR)
			return VRAM[io.VDC[MARR].W*2];
		else return io.VDC[io.vdc_reg].B.l;
	case 3:
		if (io.vdc_reg==VRR) {
			ret = VRAM[io.VDC[MARR].W*2+1];
			io.VDC[MARR].W+=1; //io.vdc_inc;
			return ret;
		} else return io.VDC[io.vdc_reg].B.h;
	}
	break;

  case 0x0400:/* VCE */
  	switch(A&7){
	case 4: return io.VCE[io.vce_reg.W].B.l;
	case 5: return io.VCE[io.vce_reg.W].B.h;
	}
	break;

  case 0x0800:	/* PSG */
	switch(A&15){
	case 0: return io.psg_ch;
	case 1: return io.psg_volume;
	case 2: return io.PSG[io.psg_ch][2];
	case 3: return io.PSG[io.psg_ch][3];
	case 4: return io.PSG[io.psg_ch][4];
	case 5: return io.PSG[io.psg_ch][5];
	case 6: return io.PSG[io.psg_ch][6];
	case 7: return io.PSG[io.psg_ch][7];
	case 8: return io.psg_lfo_freq;
	case 9: return io.psg_lfo_ctrl;
	}
	break;

  case 0x0c00:	/* timer */
	return io.timer_counter;

  case 0x1000:	/* joypad */
		ret = io.JOY[io.joy_counter]^0xff;
		if (io.joy_select&1) ret>>=4;
		else ret&=15;
		return ret|Country; /* country 0:JPN 1=US */

  case 0x1400:	/* IRQ */
	switch(A&15){
	case 2: return io.irq_mask;
	case 3: ret = io.irq_status;io.irq_status=0;return ret;
	}
	break;

  case 0x1800:	/* CD-ROM extention */
	switch(A&15){
	case 3: return io.backup = DISABLE;
	case 0xa: 
		if (!io.adpcm_firstread) return PCM[io.adpcm_rptr++];
		else {io.adpcm_firstread=FALSE; return NODATA;}
	}
	break;
  }
	printf("ignore I/O read %04x\n",A);
	return NODATA;
}

byte Loop6502(M6502 *R)
{
	static int UCount = 0;
	static int ACount = 0;
	static int prevline;
	int ret;
	//printf("PC:%04x ",R->PC.W);

	scanline=(scanline+1)%scanlines_per_frame;
	//printf("scan %d\n",scanline);
	ret = INT_NONE;
	if (scanline==MinLine) {
		io.vdc_status&=~VDC_InVBlank;
		prevline=MinLine;
	}else
	if (scanline==MaxLine) {
		if (CheckSprites()) io.vdc_status|=VDC_SpHit;
		else io.vdc_status&=~VDC_SpHit;
		if (UCount) UCount--;
		else {
			RefreshLine(prevline,scanline);
			prevline=scanline+1;
			UCount=UPeriod;
			RefreshScreen();
		}
	}
	if (scanline>=MinLine && scanline<=MaxLine) {
	 if (scanline==(io.VDC[RCR].W&1023)-64) {
		if (RasHitON && !UCount) {
			RefreshLine(prevline,scanline);
			prevline=scanline+1;
		}
		io.vdc_status|=VDC_RasHit;
		if (RasHitON) ret = INT_IRQ;
	 }
	}else
	if (scanline==MaxLine+1) {
		int J=JoyStick();
		if (J&0x10000) return INT_QUIT;
		io.JOY[0]=J;
		io.vdc_status=(io.vdc_status&~VDC_RasHit)|VDC_InVBlank;
		/* VRAM to SATB DMA */
		memcpy(SPRAM,VRAM+io.VDC[SATB].W*2,64*8);
		io.vdc_status|=VDC_SATBfinish;
		if (VBlankON) ret = INT_IRQ;
	}
/*	else 
	if (scanline==MaxLine+2) {
		io.vdc_status|=VDC_SATBfinish;
	}*/
	if (ret) {
		io.irq_status|=IRQ1;
		if (!(io.irq_mask&IRQ1)) {
			//printf("irq:scan %d\n ",scanline);
			return ret;
		}
	}
	return INT_NONE;
}

/*
	Hit Chesk Sprite#0 and others
*/
int CheckSprites(void)
{
	int i,x0,y0,w0,h0,x,y,w,h;
	SPR *spr;
	spr = (SPR*)SPRAM;
	x0 = spr->x;
	y0 = spr->y;
	w0 = (((spr->atr>>8 )&1)+1)*16;
	h0 = (((spr->atr>>12)&3)+1)*16;
	spr++;
	for(i=1;i<64;i++,spr++) {
		x = spr->x;
		y = spr->y;
		w = (((spr->atr>>8 )&1)+1)*16;
		h = (((spr->atr>>12)&3)+1)*16;
		if ((x<x0+w0)&&(x+w>x0)&&(y<y0+h0)&&(y+h>y0)) return TRUE;
	}
	return FALSE;
}

static void plane2pixel(int no)
{
	byte M,*C=VRAM+no*32;
	unsigned long L,*C2 = VRAM2+no*8;
	int j;
  for(j=0;j<8;j++,C+=2,C2++) {
    M=C[0];
    L =((M&0x88)>>3)|((M&0x44)<<6)|((M&0x22)<<15)|((M&0x11)<<24);
    M=C[1];
    L|=((M&0x88)>>2)|((M&0x44)<<7)|((M&0x22)<<16)|((M&0x11)<<25);
    M=C[16];
    L|=((M&0x88)>>1)|((M&0x44)<<8)|((M&0x22)<<17)|((M&0x11)<<26);
    M=C[17];
    L|=((M&0x88))|((M&0x44)<<9)|((M&0x22)<<18)|((M&0x11)<<27);
    C2[0] = L; //37261504
  }
}

#if 0
static void sprite2pixel(void *dst,void *src,int h)
{
  int i;
  long *C2 = dst;
  byte *C = src;
  for(i=0;i<h*2;i++,C++,C2++){
	long L;
	byte M;
    M=C[0];
    L =((M&0x88)>>3)|((M&0x44)<<6)|((M&0x22)<<15)|((M&0x11)<<24);
    M=C[32];
    L|=((M&0x88)>>2)|((M&0x44)<<7)|((M&0x22)<<16)|((M&0x11)<<25);
    M=C[64];
    L|=((M&0x88)>>1)|((M&0x44)<<8)|((M&0x22)<<17)|((M&0x11)<<26);
    M=C[96];
    L|=((M&0x88))|((M&0x44)<<9)|((M&0x22)<<18)|((M&0x11)<<27);
	C2[0]=L;
/*
	M = C[0];
    L  =((M&0x8888)>>3)|((M&0x4444)<<14);
    LL =((M&0x2222)>>1)|((M&0x1111)<<16);
	M = C[16];
    L |=((M&0x8888)>>2)|((M&0x4444)<<15);
    LL|=((M&0x2222))|((M&0x1111)<<17);
	M = C[32];
    L |=((M&0x8888)>>1)|((M&0x4444)<<16);
    LL|=((M&0x2222)<<1)|((M&0x1111)<<18);
	M = C[64];
    L |=((M&0x8888))|((M&0x4444)<<17);
    LL|=((M&0x2222)<<2)|((M&0x1111)<<19);
	C2[0]=L;   //159D048C
	C2[1]=LL;  //37BF26AE
	C++;
	C2+=2;
*/
  }
}
#endif
static long sp2pixel(byte *C)
{
	long L;
	byte M;
    M=C[0];
    L =((M&0x88)>>3)|((M&0x44)<<6)|((M&0x22)<<15)|((M&0x11)<<24);
    M=C[32];
    L|=((M&0x88)>>2)|((M&0x44)<<7)|((M&0x22)<<16)|((M&0x11)<<25);
    M=C[64];
    L|=((M&0x88)>>1)|((M&0x44)<<8)|((M&0x22)<<17)|((M&0x11)<<26);
    M=C[96];
    L|=((M&0x88))|((M&0x44)<<9)|((M&0x22)<<18)|((M&0x11)<<27);
	return L;
}

#define	PAL(c)	R[c]
#define	SPal	(Pal+256)
extern byte *XBuf;
#define	WIDTH	(320+64)
#define	HEIGHT	256
byte Black = 0;
#define	FC_W	io.screen_w
#define	FC_H	256

void RefreshLine(int Y1,int Y2)
{
	int X1,X2,XW,Line;
	int x,y,h,offset,Shift;

	byte *PP,*ZP;
	//printf("%d-%d ",Y1,Y2);
	Y2++;
	PP=XBuf+WIDTH*(HEIGHT-FC_H)/2+(WIDTH-FC_W)/2+WIDTH*Y1;
	if(!ScreenON) memset(PP,Black,(Y2-Y1)*WIDTH);
	else {
	y = Y1+ScrollY;
	offset = y&7;
	h = 8-offset;
	if (h>Y2-Y1) h=Y2-Y1;
	y/=8;
	PP-=ScrollX&7;
	XW=io.screen_w/8+1;
	Shift = ScrollX&7;
	{byte *Z=ZBuf+Y1*ZW;
	for(Line=Y1;Line<Y2;Line++,Z+=ZW) Z[0]=0;
	}

  for(Line=Y1;Line<Y2;y++) {
    ZP = ZBuf+Line*ZW;
	x = ScrollX/8;
	y &= io.bg_h-1;
	for(X1=0;X1<XW;X1++,x++,PP+=8,ZP++){
		byte *R,*P,*C,*Z;
		unsigned long *C2;
		int no,i;
		x&=io.bg_w-1;
		no = ((word*)VRAM)[x+y*io.bg_w];
		R = &Pal[(no>>12)*16];
		no&=0xFFF;
		if (vchange[no]) { vchange[no]=0; plane2pixel(no);}
		C2 = &VRAM2[no*8+offset];
		C = &VRAM[no*32+offset*2];
		P = PP;
		Z = ZP;
		for(i=0;i<h;i++,P+=WIDTH,C2++,C+=2,Z+=ZW) {
			unsigned long L;

			L=C2[0];
			P[0]=PAL((L>>4)&15);
			P[1]=PAL((L>>12)&15);
			P[2]=PAL((L>>20)&15);
			P[3]=PAL((L>>28));
			P[4]=PAL((L)&15);
			P[5]=PAL((L>>8)&15);
			P[6]=PAL((L>>16)&15);
			P[7]=PAL((L>>24)&15);
			L = (C[0]|C[1]|C[16]|C[17])<<Shift;
			Z[0]|=L>>8;Z[1]=L;
		}
	}
	Line+=h;
	PP+=WIDTH*h-XW*8;
	offset = 0;
	h = Y2-Line;
	if (h>8) h=8;
  }
  	}
	/* Refresh Sprite */
//	if (SpriteON) RefreshSprite(Y1,Y2);
}

#define	V_FLIP	0x8000
#define	H_FLIP	0x0800
#define	SPBG	0x80
#define	CGX		0x100

static void PutSprite(byte *P,byte *C,byte *R,int h,int inc)
{
	int i,J;
	unsigned long L;
	for(i=0;i<h;i++,C+=inc,P+=WIDTH) {
		J = ((word*)C)[0]|((word*)C)[16]|((word*)C)[32]|((word*)C)[48];
		if (!J) continue;
		L = sp2pixel(C+1);
		if (J&0x8000) P[0]=PAL((L>>4)&15);
		if (J&0x4000) P[1]=PAL((L>>12)&15);
		if (J&0x2000) P[2]=PAL((L>>20)&15);
		if (J&0x1000) P[3]=PAL((L>>28));
		if (J&0x0800) P[4]=PAL((L)&15);
		if (J&0x0400) P[5]=PAL((L>>8)&15);
		if (J&0x0200) P[6]=PAL((L>>16)&15);
		if (J&0x0100) P[7]=PAL((L>>24)&15);
		L = sp2pixel(C);
		if (J&0x80) P[8 ]=PAL((L>>4)&15);
		if (J&0x40) P[9 ]=PAL((L>>12)&15);
		if (J&0x20) P[10]=PAL((L>>20)&15);
		if (J&0x10) P[11]=PAL((L>>28));
		if (J&0x08) P[12]=PAL((L)&15);
		if (J&0x04) P[13]=PAL((L>>8)&15);
		if (J&0x02) P[14]=PAL((L>>16)&15);
		if (J&0x01) P[15]=PAL((L>>24)&15);
	}
}

static void PutSpriteHflip(byte *P,byte *C,byte *R,int h,int inc)
{
	int i,J;
	unsigned long L;
	for(i=0;i<h;i++,C+=inc,P+=WIDTH) {
		J = ((word*)C)[0]|((word*)C)[16]|((word*)C)[32]|((word*)C)[48];
		if (!J) continue;
		L = sp2pixel(C+1);
		if (J&0x8000) P[15]=PAL((L>>4)&15);
		if (J&0x4000) P[14]=PAL((L>>12)&15);
		if (J&0x2000) P[13]=PAL((L>>20)&15);
		if (J&0x1000) P[12]=PAL((L>>28));
		if (J&0x0800) P[11]=PAL((L)&15);
		if (J&0x0400) P[10]=PAL((L>>8)&15);
		if (J&0x0200) P[9]=PAL((L>>16)&15);
		if (J&0x0100) P[8]=PAL((L>>24)&15);
		L = sp2pixel(C);
		if (J&0x80) P[7]=PAL((L>>4)&15);
		if (J&0x40) P[6]=PAL((L>>12)&15);
		if (J&0x20) P[5]=PAL((L>>20)&15);
		if (J&0x10) P[4]=PAL((L>>28));
		if (J&0x08) P[3]=PAL((L)&15);
		if (J&0x04) P[2]=PAL((L>>8)&15);
		if (J&0x02) P[1]=PAL((L>>16)&15);
		if (J&0x01) P[0]=PAL((L>>24)&15);
	}
}

void RefreshSprite(int Y1,int Y2)
{
	int n;
	SPR *spr;

	spr = (SPR*)SPRAM + 63;

	for(n=0;n<64;n++,spr--){
		int x,y,no,atr,inc,cgx,cgy;
		byte *R,*P,*C;
		int h,t,i,j;
		y = (spr->y&1023)-64;
		x = (spr->x&1023)-32;
		no= spr->no&2047;
		atr=spr->atr;
		cgx = (atr>>8)&1;
		cgy = (atr>>12)&3;
		no &= ~((cgy*4+cgx*2));
		if (y>=Y2 || y+(cgy+1)*16<Y1 || x>=FC_W || x+(cgx+1)*16<0) continue;

		R = &SPal[(atr&15)*16];
		C = &VRAM[no*64];
		P = XBuf+WIDTH*(HEIGHT-FC_H)/2+(WIDTH-FC_W)/2+WIDTH*y+x;
		inc = 2;
		if (atr&V_FLIP) { inc=-2; C+=15*2+cgy*256;}
		t = Y1-y;
		h = 16;
		if (t>0) {
			C+=t*inc;
			h-=t;
			P+=t*WIDTH;
		}
		//printf("(%d,%d,%d,%d,%d)",x,y,cgx,cgy,h);
		for(i=0;i<=cgy;i++) {
			if (h>Y2-y) h = Y2-y;
			for(j=0;j<=cgx;j++){
			  if (atr&H_FLIP)
				PutSpriteHflip(P+(cgx-j)*16,C+j*128,R,h,inc);
			  else
				PutSprite(P+j*16,C+j*128,R,h,inc);
			}
			P+=h*WIDTH;
			C+=h*inc+32*7;
			h=16;
		}
	}
}

void RefreshScreen(void)
{
	int i;
//	memset(XBuf,255,WIDTH*HEIGHT/2);
	if (SpriteON) RefreshSprite(0,256);
	PutImage((WIDTH-FC_W)/2,(HEIGHT-FC_H)/2+MinLine,FC_W,MaxLine-MinLine+1);
}

int CartLoad(char *name)
{
	FILE *fp;
	int fsize,i;
	fp=fopen(name,"rb");
	if (fp==NULL) {
		printf("%s not found.\n",name);
		return -1;
	}
	fseek(fp,0,SEEK_END);
	fsize = ftell(fp);
	fseek(fp,fsize&0x1fff,SEEK_SET);
	//printf("seekptr %x",ftell(fp));
	fsize&=~0x1fff;
	ROM = malloc(fsize);
	ROM_size = fsize/0x2000;
	//printf("ROM size:%x\n",fsize);
	fread(ROM,1,fsize,fp);
	fclose(fp);
	return 0;
}


void ResetPCE(M6502 *M)
{
	memset(M,0,sizeof(*M));
	M->IPeriod = IPeriod;
	M->TrapBadOps = 1;
	if (Debug) {
		M->Trace = 1;
	}
	scanline = 0;
	io.vdc_inc = 1;
	Reset6502(M);
	//printf("reset PC=%04x ",M->PC.W);
}

int InitPCE(char *name)
{
	int i,ROMmask;
	if (CartLoad(name)) return -1;
#define	VRAMSIZE	0x10000

	DMYROM=malloc(0x2000);
	memset(DMYROM,NODATA,0x2000);
	WRAM=malloc(0x2000);
	memset(WRAM,0,0x2000);
	VRAM=malloc(VRAMSIZE);
	VRAM2=malloc(VRAMSIZE);
	//memset(VRAM,0,VRAMSIZE);
	vchange = malloc(VRAMSIZE/32);
	memset(vchange,1,VRAMSIZE/32);
	ROMmask = 1;
	while(ROMmask<ROM_size) ROMmask<<=1;
	ROMmask--;
	for(i=0;i<0xF7;i++)
		ROMMap[i]=ROM+(i&ROMmask)*0x2000;
	ROMMap[0xF7]=WRAM;
	ROMMap[0xF8]=RAM;
	ROMMap[0xF9]=RAM+0x2000;
	ROMMap[0xFA]=RAM+0x4000;
	ROMMap[0xFB]=RAM+0x6000;
	ROMMap[0xFF]=NULL; /* NULL = I/O area */
	return 0;
}

int RunPCE(void)
{
	M6502 M;
	ResetPCE(&M);
	Run6502(&M);
}

void TrashPCE(void)
{
	if (vchange) free(vchange);
	if (DMYROM) free(DMYROM);
	if (WRAM) free(WRAM);
	if (VRAM) free(VRAM);
	if (VRAM2) free(VRAM2);
	if (ROM) free(ROM);
}

int JoyStick(void)
{
	return Joysticks();
}

int main(int argc,char *argv[])
{
	BaseClock = 7160000; //3.58-21.48;
	IPeriod = BaseClock/(scanlines_per_frame*60);
	UPeriod = 2;
	vmode = 0;
	Debug = 0;
	if (argc<2) return -1;
	if (!InitMachine()) return -1;
	if (!InitPCE(argv[1])) {
		RunPCE();
		TrashPCE();
	}
	TrashMachine();
}
