/********************************************************************/
/*																	*/
/*							Make a Pack File						*/
/*																	*/
/********************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

void BuildPack(char *PackDesc);
void Error(char *ErrorMsg);
char *GetLine(FILE *f,char *Buffer);
char *GetLineProcess(FILE *f,char *Buffer);
void Header(FILE *f,char *Name,int Type,int Rec);
void AddFile(FILE *fPack,char *FileName,char *Title,int Type);
void ConvertFile(char *TxtName,char *ObName);
void CalculateChecksum(char *Filename);

/********************************************************************/
/*							Main Program							*/
/********************************************************************/

void main(int argc,char *argv[])
{
int i;
for (i = 1;i < argc;i++)				/* Build all packs in the list */
				BuildPack(argv[i]);
}

/********************************************************************/
/*							Report an error							*/
/********************************************************************/

void Error(char *ErrorMsg)
{
fprintf(stderr,"ERROR: [%s]\n",ErrorMsg);
exit(0);
}

/********************************************************************/
/*			  Build a pack from a descriptor file					*/
/********************************************************************/

void BuildPack(char *PackDesc)
{
FILE *f,*fPack;
char Target[128],Line[256],Type[32],FileName[256],Packname[256];
unsigned int PackSize,n,Processed;
long lSize;
printf("Processing descriptor file \"%s\".\n",PackDesc);
f = fopen(PackDesc,"r");				/* Open pack for reading */
if (f == NULL) Error("Can't open descriptor file");

if (GetLineProcess(f,Line) == NULL)		/* Read the first line */
					Error("No Filename/Size line.\n");
if (sscanf(Line,"%s %u",Target,&PackSize) != 2)
					Error("Bad Format for Filename/Size line");

strcat(Target,".opk");strlwr(Target);	/* Process target */
strcpy(Packname,Target);

if (PackSize != 8 &&					/* Check Pack is Ok Size */
		PackSize != 16 && PackSize != 32)
						Error("Illegal Size for Pack");

printf("  Creating Pack \"%s\" size %uk bytes.\n",Target,PackSize);

fPack = fopen(Target,"wb");				/* Create blank Pack */
if (fPack == NULL) Error("Can't open target");

putc('O',fPack);fputc('P',fPack);		/* 6 bytes are OPK 00 00 00 */
fputc('K',fPack);fputc(0,fPack);
fputc(0,fPack);fputc(0,fPack);

fputc(0x72,fPack);						/* Type */
fputc(PackSize/8,fPack);				/* Size */
fputc(98,fPack);						/* Date */
fputc(1,fPack);
fputc(1,fPack);
fputc(1,fPack);

fputc(random(256),fPack);				/* Cycle thingy */
fputc(random(256),fPack);
fputc(0x44,fPack);fputc(0x55,fPack);	/* Checksum (not !) */

Header(fPack,"MAIN",0x81,0x90);			/* Add "MAIN" */

while (GetLineProcess(f,Line) != NULL)	/* Work through the files */
	{
	n=sscanf(Line,"%s %s",Target,Type);	/* Get type information */
	if (n != 2) Error("Bad pack component");
	strupr(Target);strupr(Type);
	printf("    Adding %s [type %s] to the ROMPack.\n",Target,Type);
	strcpy(FileName,Target);
	strcat(FileName,".");
	strcat(FileName,Type);

	Processed = 0;
	if (strcmp(Type,"OB3") == 0)		/* xxxx.OB3 */
		{
		Processed = 1;
		AddFile(fPack,FileName,Target,0x83);
		}
	if (strcmp(Type,"TXT") == 0 ||
		strcmp(Type,"OPL") == 0)			/* xxxx.TXT */
		{
		Processed = 1;
		ConvertFile(FileName,"_temp.ob3");
		AddFile(fPack,"_temp.ob3",Target,0x83);
		}
	if (Processed == 0)
		Error("Component Type unknown.");
	}

fputc(0xFF,fPack);
fputc(0xFF,fPack);

lSize = ftell(fPack);
fclose(fPack);
fclose(f);

fPack = fopen(Packname,"rb+");			/* Patch the size word */
fseek(f,4L,SEEK_SET);
fputc(lSize >> 8,fPack);
fputc(lSize,fPack);
fclose(fPack);

CalculateChecksum(Packname);			/* Calculate the checksum */

printf("\n");
}

/********************************************************************/
/*							Output a file							*/
/********************************************************************/

void AddFile(FILE *fPack,char *FileName,char *Title,int Type)
{
unsigned int Size = 0;
FILE *fSrc;
int ch;

printf("    Now adding file \"%s\".\n",FileName);
fSrc = fopen(FileName,"rb");			/* Open the file */
if (fSrc == NULL) Error("Can't open File to add");
fseek(fSrc,0L,SEEK_END);        		/* Get the size */
Size = (unsigned int)ftell(fSrc)-6;
fseek(fSrc,6L,SEEK_SET);				/* Start of data */

Header(fPack,Title,Type,0);				/* Do the header for the file */
fputc(0x02,fPack);						/* 02 80 hh ll */
fputc(0x80,fPack);
fputc(Size >> 8,fPack);
fputc(Size & 255,fPack);

while (ch = fgetc(fSrc),ch != EOF)		/* Copy the actual data */
							fputc(ch,fPack);
fclose(fSrc);
}

/********************************************************************/
/*						   Read & Strip a line						*/
/********************************************************************/

char *GetLine(FILE *f,char *Buffer)
{
if (fgets(Buffer,255,f) == NULL)		/* End of file */
							return(NULL);
while (*Buffer != '\0' &&				/* Remove trailing spaces etc */
		Buffer[strlen(Buffer)-1] <= ' ')
			Buffer[strlen(Buffer)-1] = '\0';
return(Buffer);
}

/********************************************************************/
/*	  Read a line , ignore empty,comment,control,leading blanks		*/
/********************************************************************/

char *GetLineProcess(FILE *f,char *Buffer)
{
int Done = 0;
char *p;
while (!Done)							/* While incomplete */
	{
	if (GetLine(f,Buffer) == NULL)		/* Get line */
						return(NULL);	/* end of file */
	p = Buffer;
	while (*p != '\0')					/* Control characters */
		{
		if (*p < ' ') *p = ' ';
		p++;
		}
	while (Buffer[0] == ' ')			/* Leading spaces */
				strcpy(Buffer,Buffer+1);
	if (Buffer[0] != '\0' &&       		/* Not empty & not comment */
		Buffer[0] != ';') Done = 1;		/* finished.... */
	}
return(Buffer);
}

/********************************************************************/
/*							 Output a header						*/
/********************************************************************/

void Header(FILE *f,char *Name,int Type,int Rec)
{
char Temp[32];
fputc(9,f);								/* 09 */
fputc(Type,f);							/* 81 or 83 usually */
sprintf(Temp,"%s      ",Name);			/* make 'filename' */
strupr(Temp);
Temp[8] = '\0';
fputs(Temp,f);							/* put out 8 characters space padded */
fputc(Rec,f);							/* and the extra byte */
}

/********************************************************************/
/*		 Convert a Text file to an object-less OB3 file				*/
/********************************************************************/

void ConvertFile(char *TxtName,char *ObName)
{
FILE *fIn,*fOut;
char Buffer[256];
unsigned int Size;

printf("    Converting \"%s\" to \"%s\".\n",TxtName,ObName);
fIn = fopen(TxtName,"r");				/* Open both files */
fOut = fopen(ObName,"wb");
if (fIn == NULL || fOut == NULL)
					Error("Couldn't open file");

fputs("123456",fOut);					/* 6 char header (junk) */
fputc(0,fOut);fputc(0,fOut);			/* 0 bytes QCODE */
fputc(0,fOut);fputc(0,fOut);			/* Size goes here (0 initially) */

while (GetLine(fIn,Buffer) != NULL)		/* Copy the lines over */
	{
	fputs(Buffer,fOut);					/* Terminate each line with $00 */
	fputc(0,fOut);
	}

Size = (unsigned int)(ftell(fOut)-6-4);	/* Get size,allow for header */
fclose(fIn);							/* (the size of the text bit) */
fclose(fOut);

fOut = fopen(ObName,"rb+");				/* Patch Size from earlier */
fseek(fOut,2L+6L,SEEK_SET);				/* File position */
fputc(Size >> 8,fOut);
fputc(Size & 255,fOut);
fclose(fOut);
}

/********************************************************************/
/*						Calculate the checksum						*/
/********************************************************************/

void CalculateChecksum(char *Filename)
{
FILE *f;
unsigned int n,Checksum,w;

f = fopen(Filename,"rb+");
fseek(f,6L,SEEK_SET);

Checksum = 0;
for (n = 0;n < 4;n++)
	{
	w = (fgetc(f) << 8);
	w = w + fgetc(f);
	Checksum = Checksum+w;
	}
fclose(f);

f = fopen(Filename,"rb+");
fseek(f,14L,SEEK_SET);
printf("The Checksum for this pack is %04X",Checksum);
n = fputc(Checksum >> 8,f);
n = fputc(Checksum,f);
fclose(f);
}